!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   Semaphore, OrderedCollection, SequenceableCollection, Collection, Object.
!
!=========================================================================

! Forward references needed
run
Globals at: #ProcessorScheduler ifAbsent: [Globals at: #ProcessorScheduler put: nil].
true
%

! note, Semaphore is made NP in bomlastconv.gs 
!
! Gs64 v3.0 Semaphore created/changed in bom.c , superClass is now OrderedCollection
!   so that (aSemaphore _isArray)==false  in ProcessorScheduler and GsSocket code . 

! Remove existing behavior from Semaphore
set class Semaphore
removeallmethods 
removeallclassmethods 

! ------------------- Class methods for Semaphore
category: 'For Documentation Installation only'
classmethod
installDocumentation

self comment:
'Instances of Semaphore are used to define critical regions in
 Smalltalk processes.  Only one instance of a GsProcess will
 may execute within a critical region at a time.  Instances of
 Semaphore may not be committed to disk; they exist only for the
 life of a session, or until garbage collected.

Constraints:
	signalCount: SmallInteger' .
%

category: 'Instance Creation'
classmethod:
forMutualExclusion
  "Answer a new semaphore that has an initial pending
   signal, that can be used for mutual exclusion processing."

  ^ self _basicNew _initializeMutex
%
category: 'Instance Creation'
classmethod:
new
  "Answer a new semaphore"
  ^ self _basicNew initialize
%
category: 'Instance Creation'
classmethod:
new: size
  "Answer a new semaphore"
  ^ self _basicNew initialize 
%
! ------------------- Instance methods for Semaphore
category: 'Private
method: 
_initializeMutex
  "initialize a new instance that has an initial pending
   signal, that can be used for mutual exclusion processing.
   Returns receiver."

  signalCount := 1
%

category: 'Mutual Exclusion'
method:
critical: aBlock
  "execute aBlock when no other critical blocks are executing"
  
  | value |
  self wait.
  value := aBlock ensure: [self signal].
  ^value
%
category: 'Initialization'
method:
initialize

  signalCount := 0.
%
category: 'Private'
method:
excessSignals

  ^ signalCount
%
category: 'Formatting'
method:
printOn: aStream
  aStream
    nextPutAll: self class name;
    nextPut: $(.
  self asOop printOn: aStream.
  self size ~~ 0 ifTrue: [
    aStream nextPutAll: ',p='.
    1 to: self size do: [:i | (self at: i) printOn: aStream] ].
  aStream nextPut: $)
%

! fixed 42384 , 42403
category: 'Communication'
method:
signal
  "Send a signal through the receiver. If one or more GsProcesses have
   been suspended trying to receive a signal, make the one with the
   highest priority that has been waiting the longest ready to run.
   If no GsProcess is waiting, remember the excess signal.
   Returns the receiver."
   
  <primitive: 736> "enter critical region, prim always fails"
    "ProcessorScheduler scheduler dbglog: [ 'signalling semaphore ', self printString ]." 
  (self size ~~ 0) ifTrue: [ | waitingProc sched |
    (waitingProc := self removeFirst )  _reapSignal: self . "puts waiter on a queue"
    waitingProc priority >  (sched := self _scheduler) activeProcess priority ifTrue:[
      sched yield   "scheduler will exit critical region"
    ] ifFalse:[
      self _exitCritical .
    ]
  ] ifFalse: [
    signalCount := signalCount + 1  .
    self _exitCritical .
  ]
%

method:
_exitCritical
  "exit critical region. see also prim 737 in scheduler"
<primitive: 955>

self _primitiveFailed: #_exitCritical
%

category: 'Communication'
method:
signalAll
  "Just like signal except all of the GsProcesses waiting on the receiver
   are made ready to run.
   Returns the receiver."

  <primitive: 736> "prim always fails, sets OM.schedulerInReapEvents flag"
  | highPrio sched |
    "ProcessorScheduler scheduler dbglog: [ 'signalling all semaphore ', self printString ]."
  sched := self _scheduler .
  highPrio := sched lowestPriority .
  1 to: self size do: [:j | | aProc  aPrio |
    (aProc := self at:j)  _reapSignal: self.
    (aPrio := aProc priority) > highPrio ifTrue:[  highPrio := aPrio ].
  ].
  self size: 0.
  highPrio > sched activeProcess priority ifTrue:[
    sched yield  "scheduler will exit critical region"
  ] ifFalse:[
    self _exitCritical
  ]
%
category: 'Private'
method:
_scheduler

"Returns   ProcessorScheduler scheduler "
<primitive: 457>

self _primitiveFailed: #_scheduler
%

! fixed 42384 
category: 'Communication'
method:
wait
  "if signalCount > 0, this merely decrements the count.  Otherwise
   this will suspend the active process and add it to the end of
   the receiver's list of suspended processes"
  
  <primitive: 736> "prim always fails, sets OM.schedulerInReapEvents flag"
  | count |
    "ProcessorScheduler scheduler dbglog: [ ('wait for ', self printString) ]."
  (count := signalCount) > 0 ifTrue: [
    signalCount := count - 1 .
    self _exitCritical .
  ] ifFalse: [ | sched |
    "sched dbglog: [ '  semaphore suspending process' ]."
    self add: (sched := self _scheduler) activeProcess .
    sched _waitOnSema: self   "scheduler exits critical region"
  ].
%

category: 'Communication'
method:
tryLock
  "If signalCount > 0, decrements the count and returns true,
   otherwise returns false"

  | count |
  ^ (count := signalCount) > 0
        ifTrue:[ signalCount := count - 1 . true ]
       ifFalse:[ false ]
%

category: 'Communication'
method:
isLocked
   ^ signalCount == 0
%


category: 'Communication'
method:
waitForMilliseconds: millisecondCount
  "if signalCount > 0, this merely decrements the count.  Otherwise
   this will suspend the active process and add it to the end of
   the receiver's list of suspended processes.
   It will wait for at least millisecondCount milliseconds for the receiver to
   be signalled.
   Returns true if the semaphore was signalled for the caller.
   Returns false if the timeout expires without a signal for the caller."

   | count |
  (count := signalCount) > 0 ifTrue: [
    signalCount := count - 1.
    ^true
  ] ifFalse: [ | currProc ofs sched |
    sched := self _scheduler _enterCritical .
    currProc := sched activeProcess .
    self add: currProc .
    sched _waitForMilliseconds:  millisecondCount .
    sched _enterCritical.
    ofs := self indexOfIdentical: currProc .
    ofs == 0 ifTrue:[
      "not in the semaphore so must have been signalled"
      sched _exitCritical .
      ^ true
    ].
    self removeFrom: ofs to: ofs .
    sched _exitCritical .
    ^ false
  ].
%

category: 'Communication'
method:
highPriorityWaitForSeconds: secondCount
  "If signalCount > 0, this merely decrements the count.  Otherwise
   this will suspend the active process and add it to the end of
   the receiver's list of suspended processes.
   It will wait for at least secondCount seconds for the receiver to
   be signalled.  
   The active process' priority is raised for the duration of the 
   wait so that expiration of the wait will interrupt any infinite loop
   or infinite wait for socket activity in other processes.
   Returns true if the semaphore was signalled for the caller.
   Returns false if the timeout expires without a signal for the caller."

  ^ self highPriorityWaitForMilliseconds: secondCount * 1000  
%

category: 'Communication'
method:
highPriorityWaitForMilliseconds: millisecondCount

  "If signalCount > 0, this merely decrements the count.  Otherwise
   this will suspend the active process and add it to the end of
   the receiver's list of suspended processes.
   It will wait for at least millisecondCount milliseconds for the receiver to
   be signalled.
   The active process' priority is raised for the duration of the 
   wait so that expiration of the wait will interrupt any infinite loop
   or infinite wait for socket activity in other processes.
   Returns true if the semaphore was signalled for the caller.
   Returns false if the timeout expires without a signal for the caller."

  | count |
  (count := signalCount) > 0 ifTrue: [
    signalCount := count - 1.
    ^true
  ] ifFalse: [ | currProc ofs sched oldPrio |
    sched := self _scheduler _enterCritical .
    currProc := sched activeProcess .
    oldPrio :=  currProc _raisePriority .
    self add: currProc . 
    sched _waitForMilliseconds:  millisecondCount .
    sched activeProcess == currProc ifFalse:[
      Error signal:'current process changed in highPriorityWaitForMilliseconds:'
    ].
    sched _enterCritical .
    currProc priority: oldPrio .
    ofs := self indexOfIdentical: currProc .
    ofs == 0 ifTrue:[
      "not in the semaphore so must have been signalled"
      sched _exitCritical .
      ^ true
    ].
    self removeFrom: ofs to: ofs .
    sched _exitCritical .
    ^ false 
  ].
%

category: 'Communication'
method:
waitForSeconds: secondCount
  "If signalCount > 0, this merely decrements the count.  Otherwise
   this will suspend the active process and add it to the end of
   the receiver's list of suspended processes.
   It will wait for at least secondCount seconds for the receiver to
   be signalled.
   Returns true if the semaphore was signalled for the caller.
   Returns false if the timeout expires without a signal for the caller."

  ^ self waitForMilliseconds: secondCount * 1000 
%

category: 'Private'
method:
_reapSignal: signalSource
  "Signal the receiver "

  self signal .
%

category: 'Private'
method:
_canWaitOnSocket

  ^ true
%

! deleted _setupReapSignalLocation:

category: 'Private'
method:
_signalAll
  "Wake up the receiver"

  self signalAll.
%
category: 'Private'
method:
_wait

  self wait.
%

category: 'Adding'
method:
add: newObject

" Makes newObject one of the receiver's elements and returns newObject.
  Reimplemented so that it will add based on the priority of newObject."

  | count |
  count := self size.
  (count == 0) ifTrue: [
    super add: newObject.
  ] ifFalse: [
    | newPriority |
    newPriority := newObject priority.
    count _downTo: 1 do: [ :i |
      (newPriority <= ((self at: i) priority)) ifTrue: [
        self insertObject: newObject at: (i + 1).
        ^ newObject.
      ].
    ].
    self insertObject: newObject at: 1.
  ].
  ^newObject.
%

category: 'Private'
method:
_unscheduleProcess: aGsProcess
  "Used by ProcessorScheduler"
  self removeIdentical: aGsProcess otherwise: nil .
%

category: 'Private'
method:
_changePriority: aGsProcess from: oldPriority
  "Used by GsProcess to change the priority of a GsProcess in the receiver."

  | removed |
  removed := self removeIdentical: aGsProcess otherwise: nil .
  removed == nil ifFalse:[
    self add: aGsProcess.
  ]
%

category: 'Copying'
method:
replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex

"Disallowed."
^ self shouldNotImplement: #replaceFrom:to:with:startingAt:
%

method:
replaceFrom: startIndex to: stopIndex with: aSeqCollection

"Disallowed."
^ self shouldNotImplement: #replaceFrom:to:with:
%

method:
replaceFrom: startIndex to: stopIndex withObject: anObject

"Disallowed."
^ self shouldNotImplement: #replaceFrom:to:withObject:
%

