!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   PathEvaluator, Array, SequenceableCollection, Collection, Object.
!
! class created in idxclasses.topaz
!=========================================================================

removeallmethods PathEvaluator
removeallclassmethods PathEvaluator

category: 'For Documentation Installation only'
classmethod: PathEvaluator
installDocumentation

self comment:
'The class PathEvaluator implements only GemStone internals.  That is, it 
provides only functionality required by GemStone itself.  It is not intended 
for customer use, by creating instances or by subclassing.
 
Constraints:
	nsc: Collection
	ivOffsets: Array
	constraintClasses: (obsolete) Array
'
%

! ------------------- Class methods for PathEvaluator
category: 'Instance Creation'
classmethod: PathEvaluator
new

"Create a new initialized instance of the receiver."

^ self basicNew initialize
%

! ------------------- Instance methods for PathEvaluator
category: 'Initialization'
method: PathEvaluator
initialize
  "Initialize a new instance of the receiver."

  ivOffsets := {}
%

category: 'Query Select'
method: PathEvaluator
findAllValuesEqualTo: aValue
  "Returns a set of all values that satisfy the query."

  | enumerator |
  enumerator := self bagEnumerator compareEqual.
  ^ enumerator findAllValuesFor: aValue
%

category: 'Query Select'
method: PathEvaluator
findAllValuesGreaterThanKey: aValue andEquals: aBoolean
  "Returns a set of all values that satisfy the query."

  | enumerator |
  enumerator := self bagEnumerator.
  aBoolean
    ifTrue: [ enumerator compareGreaterThanOrEqual ]
    ifFalse: [ enumerator compareGreaterThan ].
  ^ enumerator findAllValuesFor: aValue
%

category: 'Query Select'
method: PathEvaluator
findAllValuesIdenticalTo: aValue
  "Returns a set of all values that satisfy the query."

  | enumerator |
  self isIndexOnRootNsc
    ifTrue: [ 
      doBlock 
        ifNil: [
          " optimize query for objects in the NSC itself "
          ^ self _allIdenticalTo: aValue in: nsc ].
      ^ self _allIdenticalTo: aValue in: nsc do: doBlock].
  enumerator := self bagIdentityEnumerator compareIdentical.
  ^ enumerator findAllValuesFor: aValue
%

category: 'Query Select'
method: PathEvaluator
findAllValuesLessThanKey: aValue andEquals: aBoolean
  "Returns a set of all values that satisfy the query."

  | enumerator |
  enumerator := self bagEnumerator.
  aBoolean
    ifTrue: [ enumerator compareLessThanOrEqual ]
    ifFalse: [ enumerator compareLessThan ].
  ^ enumerator findAllValuesFor: aValue
%

category: 'Query Select'
method: PathEvaluator
findAllValuesNotEqualTo: aValue
  "Returns a set of all values that satisfy the query."

  | enumerator |
  enumerator := self bagEnumerator compareNotEqual.
  ^ enumerator findAllValuesFor: aValue
%

category: 'Query Select'
method: PathEvaluator
findAllValuesNotIdenticalTo: aValue
  "Returns a set of all values that satisfy the query."

  | enumerator |
  self isIndexOnRootNsc
    ifTrue: [
      doBlock 
        ifNil: [
          " optimize query for objects in the NSC itself "
          ^ self _allNotIdenticalTo: aValue in: nsc ].
      ^ self _allNotIdenticalTo: aValue in: nsc do: doBlock ].
  enumerator := self bagIdentityEnumerator compareNotIdentical.
  ^ enumerator findAllValuesFor: aValue
%

category: 'Query Detect'
method: PathEvaluator
findFirstValueEqualTo: aValue
  "Returns the first value that satisfies the query."

  | enumerator |
  (enumerator := self bagEnumerator)
    compareEqual;
    doBlock: [ :obj | ^ obj ] .
  enumerator findAllValuesFor: aValue.
  ^ #'_incompletePathTraversal'
%

category: 'Query Detect'
method: PathEvaluator
findFirstValueGreaterThanKey: aValue andEquals: aBoolean
  "Returns the first value that satisfies the query."

  | enumerator |
  (enumerator := self bagEnumerator)
    doBlock: [ :obj | ^ obj ] .
  aBoolean
    ifTrue: [ enumerator compareGreaterThanOrEqual ]
    ifFalse: [ enumerator compareGreaterThan ].
  enumerator findAllValuesFor: aValue.
  ^ #'_incompletePathTraversal'
%

category: 'Query Detect'
method: PathEvaluator
findFirstValueIdenticalTo: aValue
  "Returns the first value that satisfies the query."

  | enumerator |
  self isIndexOnRootNsc
    ifTrue: [ 
      " optimize query for objects in the NSC itself "
      ^ (nsc includesIdentical: aValue)
        ifTrue: [ aValue ]
        ifFalse: [ #'_incompletePathTraversal' ] ].
  (enumerator := self bagIdentityEnumerator compareIdentical)
    doBlock: [ :obj | ^ obj ] .
  enumerator findAllValuesFor: aValue.
  ^ #'_incompletePathTraversal'
%

category: 'Query Detect'
method: PathEvaluator
findFirstValueLessThanKey: aValue andEquals: aBoolean
  "Returns the first value that satisfies the query."

  | enumerator |
  (enumerator := self bagEnumerator)
    doBlock: [ :obj | ^ obj ] .
  aBoolean
    ifTrue: [ enumerator compareLessThanOrEqual ]
    ifFalse: [ enumerator compareLessThan ].
  enumerator findAllValuesFor: aValue.
  ^ #'_incompletePathTraversal'
%

category: 'Query Detect'
method: PathEvaluator
findFirstValueNotEqualTo: aValue
  "Returns the first value that satisfies the query."

  | enumerator |
  (enumerator := self bagEnumerator compareNotEqual)
    doBlock: [ :obj | ^ obj ] .
  enumerator findAllValuesFor: aValue.
  ^ #'_incompletePathTraversal'
%

category: 'Query Detect'
method: PathEvaluator
findFirstValueNotIdenticalTo: aValue
  "Returns the first value that satisfies the query."

  | enumerator |
  (enumerator := self bagIdentityEnumerator compareNotIdentical)
    doBlock: [ :obj | ^ obj ] .
  enumerator findAllValuesFor: aValue.
  ^ #'_incompletePathTraversal'
%

category: 'Testing'
method: PathEvaluator
isIdentityIndex

"Returns false."

^ false
%

category: 'Testing'
method: PathEvaluator
isPathEvaluator

"Returns true."

^ true
%

category: 'Testing'
method: PathEvaluator
isRangeEqualityIndex

"Returns true.  A path evaluator can perform the searches of an equality
 index."

^ true
%

category: 'Accessing'
method: PathEvaluator
ivOffsets

"Returns the value of the instance variable 'ivOffsets'."

^ivOffsets
%

category: 'Updating'
method: PathEvaluator
ivOffsets: newValue

"Modify the value of the instance variable 'ivOffsets'."

ivOffsets := newValue
%

category: 'Accessing'
method: PathEvaluator
nsc

"Returns the value of the instance variable 'nsc'."

^nsc
%

category: 'Accessing'
method: PathEvaluator
nscRoot

"Returns the value of the instance variable 'nsc'."

^nsc
%

category: 'Updating'
method: PathEvaluator
nsc: newValue

"Modify the value of the instance variable 'nsc'."

nsc := newValue
%

category: 'Traversing'
method: PathEvaluator
traverseObject: anObject
  "Traverse the sub-objects of the given object using the path names of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, returns the
 incompletePathTraversalObject."

  "This method is intended for support of QueryExecutor only"

  self isIndexOnNscElements
    ifTrue: [ ^ anObject ].
  nil == anObject
    ifTrue: [ ^ #'_incompletePathTraversal' ].
  ^ self _traverseObject: anObject
%

category: 'Testing'
method: PathEvaluator
isIndexOnNscElements

"Returns whether the receiver is a path directly on the elements of an NSC
 (either the root NSC or a set-valued instance variable along the path)."

^ (self at: self size) ==  #'' 
%

category: 'Testing'
method: PathEvaluator
isIndexOnRootNsc

"Returns whether the receiver is a path directly on the elements of the root
 NSC."

^ (self at: self size) ==  #'' 
%

! deleted _isPathConstrained v2.0

! deleted isPathConstrained v2.0

category: 'Traversing'
method: PathEvaluator
_traverseObject: anObject
  "Traverse the sub-objects of the given object using the path names of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, returns the
 incompletePathTraversalObject."

  "This method is intended for support of QueryExecutor only"

  ^ self
    _traverse: anObject
    cachedOffsets: nil
    cachedClasses: nil
    incomplete: #'_incompletePathTraversal'
%

category: 'Traversing'
method: PathEvaluator
_createIvOffsetCache
  | ivOffsetCache sz |
sz := self size .
ivOffsetCache := { Array new: sz . Array new: sz }.
^ivOffsetCache
%

category: 'Traversing'
method: PathEvaluator
_traverseObject: anObject cache: ivOffsetCache
  "Traverse the sub-objects of the given object using the path names of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, returns the
 incompletePathTraversalObject."

  "This method is intended for support of QueryExecutor only"

  ^ self
    _traverse: anObject
    cachedOffsets: (ivOffsetCache at: 2)
    cachedClasses: (ivOffsetCache at: 1)
    incomplete: #'_incompletePathTraversal'
%

category: 'Traversing'
method: PathEvaluator
_traverse: anObject
cachedOffsets: offsets
cachedClasses: classes
incomplete: incompletePathTraversal

<primitive: 593>

| nextObj sz |
nextObj := anObject.
sz := self size.
1 to: sz do: [ :i |
  (offsets == nil)
     ifTrue: [  nextObj := self _nextObjectFor: nextObj at: i ]
     ifFalse: [  nextObj := self _nextObjectFor: nextObj at: i offsets: offsets classes: classes ].

  ( nil == nextObj and: [ i ~~ sz ] )
    ifTrue: [ ^ #_incompletePathTraversal ].

  (nextObj == #_incompletePathTraversal)
    ifTrue: [ ^ nextObj ].

].
^ nextObj
%

category: 'Traversing'
method: PathEvaluator
_nextObjectFor: anObject at: offset offsets: offsets classes: classes

"Returns the next object along the path in anObject for the named instance
 variable at the given offset in the receiver."

| ivOffset  |
(classes at: offset) ~~ anObject class
  ifTrue: [
    ivOffset := anObject class _ivOffsetOf: (self at: offset).
    ivOffset == nil
      ifTrue: [ anObject _errorInvalidOffset: (self at: offset) ].
    classes at: offset put: anObject class.
    offsets at: offset put: ivOffset.
  ]
  ifFalse: [
    ivOffset := offsets at: offset.
  ].

( ivOffset > anObject _primitiveSize )
    ifTrue: [ ^ #_incompletePathTraversal ].

^ anObject instVarAt: ivOffset
%

category: 'Query Select'
method: PathEvaluator
findAllValuesWithComparison: selector to: aValue
  "Returns a set of all values that satisfy the query.  Must iterate through all
 elements in the NSC, traversing down the path for each object."

  "Should not be used ... does not handle unicode string comparisons correctly"

  | enumerator |
  enumerator := self bagEnumerator selector: selector.
  ^ enumerator findAllValuesFor: aValue
%

# deleted PathEvaluator>>_findAllValuesWithComparison:to: in version  3.2

category: 'Query Detect'
method: PathEvaluator
findFirstValueWithComparison: selector to: aValue
  "Returns the first value that satisfies the query.  Must iterate through the
 elements in the NSC, traversing down the path for each object."

  "Should not be used ... does not handle unicode string comparisons correctly"

  | enumerator |
  enumerator := (selector == #'==' or: [ selector == #'~~' ])
    ifTrue: [ self bagIdentityEnumerator ]
    ifFalse: [ self bagEnumerator ].
  enumerator
    selector: selector;
    doBlock: [ :obj | ^ obj ] .
  enumerator findAllValuesFor: aValue.
  ^ #'_incompletePathTraversal'
%

# deleted PathEvaluator>>_findFirstValueWithComparison:to: in version 3.2

category: 'Sorting'
method: PathEvaluator
_sortOn: indexObjs using: sorter persistentRoot: persistentArrayOrNil

"Returns an Array of objects from the receiver's NSC that is sorted according
 to the paths traversed by indexObjs (with the direction of the sort specified
 in theBooleans)."

|  obj aColl firstIncompleteArray sortNodes anArray
  hasSecondarySort incompletesArrays incomplete indexManager |

sortNodes := sorter sortNodes.
hasSecondarySort := indexObjs size > 1.
incomplete := #_incompletePathTraversal.

incompletesArrays := sorter incompletes.
firstIncompleteArray := incompletesArrays at: 1.

aColl := nsc _asCollectionForSorting.

persistentArrayOrNil ~~ nil ifTrue: [ persistentArrayOrNil add: aColl ].
" optimize when no traversal is needed "
indexManager := IndexManager current.
self isIndexOnRootNsc
  ifTrue: [
    1 to: aColl size do: [ :i |
      obj := aColl _at: i.
      obj == nil
        ifTrue: [ firstIncompleteArray add: nil ]
        ifFalse: [
          sorter _addObject: obj
            inNodes: sortNodes
            hasSecondarySort: hasSecondarySort
        ].
      (i \\ 100) == 0 ifTrue:[ indexManager commitIndexMaintenance: self at: i ].
    ].
  ]
  ifFalse: [
    1 to: aColl size do: [ :i |
      obj := aColl _at: i.
      obj == nil
        ifTrue: [ firstIncompleteArray add: nil ]
        ifFalse: [
          sorter _addObject: obj
            inNodes: sortNodes
            hasSecondarySort: hasSecondarySort
            incompletes: incompletesArrays
            incomplete: incomplete
        ].
      (i \\ 100) == 0 ifTrue:[ indexManager commitIndexMaintenance: self at: i ].
    ]
  ].

anArray := Array new: nsc size.
persistentArrayOrNil ~~ nil 
  ifTrue: [ 
    persistentArrayOrNil add: anArray.
    indexManager commitIndexMaintenance: self at: aColl size.
 ].
^ sorter sortInto: anArray startingAt: 1
%

category: 'Sorting'
method: PathEvaluator
_sortOn: indexObjs directions: theBooleans persistentRoot: persistentArrayOrNil

"Returns an Array of objects from the receiver's NSC that is sorted according
 to the paths traversed by indexObjs (with the direction of the sort specified
 in theBooleans)."

  | sorter |

sorter := PathSorter on: indexObjs directions: theBooleans.
(IndexManager autoCommit and: [ persistentArrayOrNil ~~ nil ])
  ifTrue: [ | mgr result |
    (mgr := IndexManager current) executeStartingIndexMaintenance: [
      persistentArrayOrNil add: sorter.
      mgr  _doCommit.
      result := self _sortOn: indexObjs using: sorter persistentRoot: persistentArrayOrNil.
      persistentArrayOrNil size: 0 ].
    ^result ]
  ifFalse: [
    ^ self _sortOn: indexObjs using: sorter persistentRoot: nil
  ]
%

category: 'Accessing'
method: PathEvaluator
lastElementClass
	"Returns the class of the last object along the path (can only be determined if
 the path is fully constrained).  Returns nil if the path is not fully
 constrained."

	^nil	"constraints are no longer supported, so no path will be fully constrained."
%

category: 'Accessing'
method: PathEvaluator
progress: anInteger

"ignored - required for autoCommit functionality"
%

category: 'Accessing'
method: PathEvaluator
pathComponentsString

"Returns a string of the path components."

| str sz |
sz := self size.
str := String new.
1 to: sz do: [ :i |
    str addAll: (self at: i).
    i == sz
        ifFalse: [ str add: $.  ]
].
^ str

%

category: 'Query Support'
method: PathEvaluator
_allIdenticalTo: anObject in: anNsc
  "Returns a new NSC containing the given object with the number of occurrences
 as in the given NSC."

  | result tmp |
  result := self _primAllIdenticalTo: anObject in: anNsc _asIdentityBag.
  result class = anNsc species
    ifFalse: [ 
      tmp := anNsc species new.
      tmp addAll: result.
      result := tmp ].
  ^ result
%

category: 'Query Support'
method: PathEvaluator
_primAllIdenticalTo: anObject in: aBag

"Returns a new NSC containing the given object with the number of occurrences as
 in the given NSC."

<primitive: 340>
self _primitiveFailed: #allIdenticalTo:in: args: { anObject . aBag } .
self _uncontinuableError
%

category: 'Query Support'
method: PathEvaluator
hasSamePathAs: aPathEvaluator

"Returns whether the given path evaluator is on the same path as the receiver."

" make sure it is a path evaluator "
aPathEvaluator isPathEvaluator
    ifFalse: [ ^ false ].

" make sure they have the same number of terms "
self size == aPathEvaluator size
    ifFalse: [ ^ false ].

" check each term "
1 to: self size do: [ :i |
    (self at: i) == (aPathEvaluator at: i)
        ifFalse: [ ^ false ]
].
^ true
%

category: 'Traversing'
method: PathEvaluator
_traverseObject: anObject incompletesInto: incompleteArray incomplete: incomplete
  "Traverse the sub-objects of the given object using the path names of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, place the object
 into the appropriate Array in the incompleteArray of Arrays."

  "Intended for support of PathSort ... no collection-based terms"

  | nextObj sz |
  nextObj := anObject.
  sz := self size.
  1 to: sz do: [ :i | 
    nextObj := self _nextObjectFor: nextObj at: i.
    ((nil == nextObj and: [ i ~~ sz ])
      or: [ nextObj == #'_incompletePathTraversal' ])
      ifTrue: [ 
        incompleteArray == nil
          ifTrue: [ ^ nil ].
        (incompleteArray at: i + 1) add: anObject.
        ^ incomplete ] ].
  ^ nextObj
%

category: 'Traversing'
method: PathEvaluator
traverseObject: anObject
incompletesInto: incompleteArray
incomplete: incomplete

"Traverse the sub-objects of the given object using the path names of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, place the object
 into the appropriate Array in the incompleteArray of Arrays."

self isIndexOnNscElements
    ifTrue: [ ^ anObject ].

nil == anObject
    ifTrue: [ ^ incomplete ].

^ self _traverseObject: anObject
  incompletesInto: incompleteArray
  incomplete: incomplete
%

category: 'Query Select'
method: PathEvaluator
findAllValuesGreaterThan: val1 andEquals: bool1 andLessThan: val2 andEquals: bool2
  "Returns a set of all values that satisfy the range query."

  | enumerator |
  enumerator := self
    bagRangeEnumeratorGreaterThanAndEquals: bool1
    andLessThanAndEquals: bool2.
  ^ enumerator findAllValuesFor: val1 and: val2
%

category: 'Query Detect'
method: PathEvaluator
findFirstValueGreaterThan: val1 andEquals: bool1 andLessThan: val2 andEquals: bool2
  ""

  | enumerator |
  (enumerator := self bagRangeEnumeratorGreaterThanAndEquals: bool1
                       andLessThanAndEquals: bool2)
    doBlock: [ :obj | ^ obj ] .
  enumerator findAllValuesFor: val1 and: val2.
  ^ #'_incompletePathTraversal'
%

category: 'Converting'
method: PathEvaluator
asPathEvaluator

""

^ self
%
method: PathEvaluator
_kind
  "Returns 2 for set valued term, 1 for enumerated term, 0 for selector term,
   nil otherwise. "
  | sv enum sel |
  1 to: self size do: [ :i | | elem | 
    elem := self at: i .
    elem size ~~ 0 ifTrue:[
      elem == #'*' 
        ifTrue:[ sv := true ] 
        ifFalse:[ 
          (elem includesValue: $| ) "enumerated" ifTrue: [ enum := true ].
          (elem at: 1) == $#  ifTrue:[ sel := true ].
        ]
      ]
  ].
  sv ifNotNil:[ ^ 2 ].
  enum ifNotNil:[ ^ 1 ].
  sel ifNotNil:[ ^ 0 ].
  ^ nil
% 
  
category: 'Converting'
classmethod: PathEvaluator
asMostSpecificType: anArray
  "Returns a new instance of a subclass, or self, depending upon the path.
   A class method to avoid legacyChangeClassTo failing with   self on stack error."
  | evaluator knd |
  evaluator := anArray at: 1 .
  knd := evaluator _kind .
  knd ifNotNil:[
    knd == 2
      ifTrue: [ 
        evaluator := evaluator legacyChangeClassTo: SetValuedPathEvaluator.
        evaluator _pathTermsRequired: evaluator termsRequired.
        ^ evaluator ].
    knd == 1
      ifTrue: [ 
        evaluator := evaluator legacyChangeClassTo: EnumeratedPathEvaluator.
        evaluator _pathTermsRequired: evaluator termsRequired.
        ^ evaluator ].
    ^ evaluator legacyChangeClassTo: SelectorPathEvaluator .
  ].
  ^ evaluator
%


category: 'Query Support'
method: PathEvaluator
traverse: anObject andCompare: selector with: aValue
  "Traverse the given object and perform the comparison operation
 on the object at the end of the path and the given value."

  "This method assumes that the index is not over a set-valued instance variable."

  "This method is intended for support of QueryExecutor only"

  | endObj |
  ^ (endObj := self _traverseObject: anObject) ~~ #'_incompletePathTraversal'
    and: [ endObj perform: selector with: aValue ]
%

category: 'Query Support'
method: PathEvaluator
traverse: anObject andCompare: selector withTraverser: aTraverser
  "Traverse the given object and perform the comparison operation on the object
 at the end of the path and the value obtained by using the traverser."

  "This method assumes that the index is not over a set-valued instance variable."

  "This method is intended for support of QueryExecutor only"

  | endObj1 endObj2 |
  ^ (endObj1 := self _traverseObject: anObject) ~~ #'_incompletePathTraversal'
    and: [ 
      (endObj2 := aTraverser _traverseObject: anObject) ~~ #'_incompletePathTraversal'
        and: [ endObj1 perform: selector with: endObj2 ] ]
%

category: 'Accessing'
method: PathEvaluator
sizeForNscBuilder
  "Returns the size to be used when an NscBuilder is collecting objects
 based upon the receiver's path."

  ^ self hasSetValuedTerm
    ifTrue: [ NscBuilder maxSize ]
    ifFalse: [ nsc size ]
%

category: 'Traversing'
method: PathEvaluator
_nextObjectFor: anObject at: offset

"Returns the next object along the path in anObject for the named instance
 variable at the given offset in the receiver."

| ivOffset |
ivOffset := anObject class _ivOffsetOf: (self at: offset).

ivOffset == nil
  ifTrue: [ anObject _errorInvalidOffset: (self at: offset) ].

( ivOffset > anObject _primitiveSize )
    ifTrue: [ ^ #_incompletePathTraversal ].

^ anObject instVarAt: ivOffset
%

category: 'Traversing'
method: PathEvaluator
_initTraverseCache

"Do nothing."

%

# deleted PathEvaluator>>_findAllValuesGreaterThan:andEquals:andLessThan:andEquals: in version 3.2

category: 'Converting'
method: PathEvaluator
asQueryEvaluator

  ^self copy
%
category: 'Testing'
set compile_env: 0
method: PathEvaluator
hasEnumeratedTerm
  "Returns true if the path has a term that indicates an enumerated path term."

  ^ false
%
category: 'Testing'
set compile_env: 0
method: PathEvaluator
hasSetValuedTerm
  "Returns true if the path has a term that indicates a set-valued instance
 variable."

  ^ false
%
category: 'Traversing'
set compile_env: 0
method: PathEvaluator
traverse: anObject do: aBlock
  "Traverse the sub-objects of anObject, using the path names of the receiver starting at the path name at the given offset.  For each object at the end of
 the traversal, evaluate the given block."

  ^ self traverse: anObject startingAt: 1 do: aBlock
%
category: 'Testing'
set compile_env: 0
method: PathEvaluator
hasSelectorTerm
  "Returns true if the path has a term that indicates a selector."

  ^ false
%
category: 'Accessing'
set compile_env: 0
method: PathEvaluator
collator
  "Returns IcuCollator to be used when comparing Unicode strings ... nil indicates that default collator should be used."

  ^ collator
%
category: 'Accessing'
set compile_env: 0
method: PathEvaluator
collator: anIcuCollator
  "Set the receiver's collator. Use a copy of anIcuCollator to disallow 
   changes to strength, etc, that might affect the sort ordering. "

  collator := anIcuCollator copy immediateInvariant
%
category: 'Accessing'
set compile_env: 0
method: PathEvaluator
bagEnumerator
  | res |
  (res := self collator
      ifNil: [ BagEnumerator new ]
      ifNotNil: [ BagUnicodeEnumerator new ])
      evaluator: self;
    doBlock: self doBlock .
  ^ res
%
category: 'Accessing'
set compile_env: 0
method: PathEvaluator
bagRangeEnumeratorGreaterThanAndEquals: bool1 andLessThanAndEquals: bool2
  | enumerator |
  (enumerator := self collator
    ifNil: [ BagRangeEnumerator new ]
    ifNotNil: [ BagUnicodeRangeEnumerator new ])
    evaluator: self;
    doBlock: self doBlock .
  bool1
    ifTrue: [ 
      bool2
        ifTrue: [ enumerator compareGreaterThanOrEqualAndLessThanOrEqual ]
        ifFalse: [ enumerator compareGreaterThanOrEqualAndLessThan ] ]
    ifFalse: [ 
      " bool1 is false "
      bool2
        ifTrue: [ enumerator compareGreaterThanAndLessThanOrEqual ]
        ifFalse: [ enumerator compareGreaterThanAndLessThan ] ].
  ^ enumerator
%
category: 'Testing'
set compile_env: 0
method: PathEvaluator
isCollectionBased
  "Returns false."

  ^ false
%
category: 'Query Select'
set compile_env: 0
method: PathEvaluator
tmpHolder
  ^ NscBuilder for: nsc species new max: self sizeForNscBuilder
%
category: 'Traversing'
set compile_env: 0
method: PathEvaluator
traverse: anObject startingAt: offset do: aBlock
  "Traverse the sub-objects of anObject, using the path names of the receiver starting at the path name at the given offset.  For each object at the end of
 the traversal, evaluate the given block."

  | nextObj ivOffset pathName sz |
  (nil == anObject _and: [ self isIndexOnNscElements not ])
    ifTrue: [ ^ self ].
  nextObj := anObject.
  sz := self size.
  offset to: sz do: [ :i | 
    " get the next object along the path "
    nil == nextObj
      ifTrue: [ ^ self ].
    pathName := self at: i.
    ivOffset := nextObj class _ivOffsetOf: pathName.
    ivOffset == nil
      ifTrue: [ 
        nextObj _errorInvalidOffset: pathName.
        nextObj := nil ]
      ifFalse: [ nextObj := nextObj instVarAt: ivOffset ].
    (nil == nextObj _and: [ i ~= sz ])
      ifTrue: [ ^ self ] ].
  ^ aBlock value: nextObj
%
category: 'Traversing'
set compile_env: 0
method: PathEvaluator
traverse: anObject startingAt: offset into: resultSet
  "Traverse the sub-objects of all the objects in the NSC, using the path names
 of the receiver starting at the path term at the given offset.  Add each
 object at the end of the traversal to the result set.  Returns the result
 set."

  self
    traverse: anObject
    startingAt: offset
    do: [ :nextObj | resultSet add: nextObj ].
  ^ resultSet
%
category: 'Query Support'
set compile_env: 0
method: PathEvaluator
_allNotIdenticalTo: anObject in: anNsc
  "Returns a new NSC containing the given object with the number of occurrences
 as in the given NSC."

  | result anIdentityBag tmp |
  anIdentityBag := anNsc _asIdentityBag.
  result := anIdentityBag copy.
  result removeAll: (self _allIdenticalTo: anObject in: anIdentityBag).
  result class = anNsc species
    ifFalse: [ 
      tmp := anNsc species new.
      tmp addAll: result.
      result := tmp ].
  ^ result
%
category: 'Query Support'
set compile_env: 0
method: PathEvaluator
_allNotIdenticalTo: anObject in: anNsc do: aBlock
  "Returns a new NSC containing the given object with the number of occurrences
 as in the given NSC."

  | result anIdentityBag |
  anIdentityBag := anNsc _asIdentityBag.
  result := anIdentityBag copy.
  result removeAll: (self _allIdenticalTo: anObject in: anIdentityBag).
  ^ result do: aBlock
%
category: 'Accessing'
set compile_env: 0
method: PathEvaluator
bagIdentityEnumerator
  | res |
  (res := BagEnumerator new)
    evaluator: self;
    doBlock: self doBlock.
  ^ res
%
category: 'Accessing'
set compile_env: 0
method: PathEvaluator
doBlock
  ^ doBlock
%
category: 'Accessing'
set compile_env: 0
method: PathEvaluator
doBlock: aBlock
  doBlock := aBlock
%
category: 'Sorting'
set compile_env: 0
method: PathEvaluator
sortNodeClass
  "Returns the class of SortNode to use."

  ^ SortNode
%
category: 'Testing'
set compile_env: 0
method: PathEvaluator
hasCollectionBasedTerm
  "Returns true if the path has an enumerated path term or a term that indicates a 
   set-valued instance variable."

  ^ self hasEnumeratedTerm or: [ self hasSetValuedTerm ]
%
category: 'Testing'
set compile_env: 0
method: PathEvaluator
_commonPathTermsWith: anEvaluatorOrIndex
  ^ anEvaluatorOrIndex _commonPathTermsWithPathEvaluator: self
%
category: 'Testing'
set compile_env: 0
method: PathEvaluator
_commonPathTermsWithIndex: anIndex
  | commonSize commonTerms |
  commonSize := self size min: anIndex size.
  commonTerms := {}.
  1 to: commonSize do: [ :index | 
    | term |
    (term := self at: index) = (anIndex at: index) name
      ifFalse: [ ^ commonTerms ].
    commonTerms add: term ].
  ^ commonTerms
%
category: 'Testing'
set compile_env: 0
method: PathEvaluator
_commonPathTermsWithPathEvaluator: anPathEvaluator
  | commonSize commonTerms |
  commonSize := self size min: anPathEvaluator size.
  commonTerms := {}.
  1 to: commonSize do: [ :index | 
    | term |
    (term := self at: index) = (anPathEvaluator at: index)
      ifFalse: [ ^ commonTerms ].
    commonTerms add: term ].
  ^ commonTerms
%
category: 'Testing'
method: PathEvaluator
termsRequired
  "Answer true if receiver requires that instance variables of indexed objects are present "

  ^ true
%
category: 'Testing'
method: PathEvaluator
isStreamable

"Returns false."

^ false
%
category: 'Testing'
method: PathEvaluator
optimizingComparison
  "Answer true if comparison operations can be optimized in Btree operations and elsewhere. 
   Controlled by optimizedComparison index option."
  
  "this type of optimization can only be done in concert with an index."

  ^ false
%
category: 'Traversing'
method: PathEvaluator
traverse: anObject cachedOffsets: cachedOffsets cachedClasses: cachedClasses do: aBlock
  | nextObj |
  nextObj := self
    _traverse: anObject
    cachedOffsets: cachedOffsets
    cachedClasses: cachedClasses
    incomplete: #'_incompletePathTraversal'.
  nextObj ~~ #'_incompletePathTraversal'
    ifTrue: [ aBlock value: nextObj ]
%
category: 'Traversing'
method: PathEvaluator
traverseUsingAccessors: anObject do: aBlock
  "Traverse the sub-objects of anObject, using the path names of the receiver as accessor 
   selectors.  For each object at the end of the traversal, evaluate the given block."

  | nextObj sz accessor |
  (nil == anObject _and: [ self isIndexOnNscElements not ])
    ifTrue: [ ^ self ].
  nextObj := anObject.
  sz := self size.
  1 to: sz do: [ :i |
    " get the next object along the path using the inst var name as an accessor message "
    nil == nextObj
      ifTrue: [ ^ self ].
    accessor := self at: i.
    nextObj := nextObj perform: accessor asSymbol.
    (nil == nextObj _and: [ i ~= sz ])
      ifTrue: [ ^ self ] ].
  ^ aBlock value: nextObj
%
category: 'Traversing'
method: PathEvaluator
traverseUsingAccessors: anObject to: sz do: aBlock
  "Traverse the sub-objects of anObject, using the path names of the receiver as accessor 
   selectors.  For each object at the end of the traversal, evaluate the given block."

  | nextObj  accessor |
  (nil == anObject _and: [ self isIndexOnNscElements not ])
    ifTrue: [ ^ self ].
  nextObj := anObject.
  1 to: sz do: [ :i |
    " get the next object along the path using the inst var name as an accessor message "
    nil == nextObj
      ifTrue: [ ^ self ].
    accessor := self at: i.
    nextObj := nextObj perform: accessor asSymbol.
    (nil == nextObj _and: [ i ~= sz ])
      ifTrue: [ ^ self ] ].
  ^ aBlock value: nextObj
%
category: 'Accessing'
method: PathEvaluator
btreeComparisonQuerySpec

  ^ BtreeComparisonForCompare newForComparison: nil
%
category: 'Query Support'
method: PathEvaluator
_allIdenticalTo: anObject in: anNsc do: aBlock
  "Returns a new NSC containing the given object with the number of occurrences
 as in the given NSC."

  | result |
  result := self _primAllIdenticalTo: anObject in: anNsc _asIdentityBag.
  ^ result do: aBlock
%
