!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   Object, GsObjectInventory
!
!=========================================================================

expectvalue %String
run
^Object _newKernelSubclass: 'GsObjectInventory'
        instVarNames: #( entriesByCount entriesByBytes includeHiddenClasses)
        classVars: #()
        classInstVars: #()
        poolDictionaries: { }
        inDictionary: Globals
        options: #()
        reservedOop: 1137
%

removeallmethods GsObjectInventory
removeallclassmethods GsObjectInventory

! ------------------- Class methods for GsObjectInventory
category: 'Documentation'
classmethod: 
comment
^'GsObjectInventory provides methods for counting instances of
classes in a repository.'
%

category: 'Repository Analysis'
classmethod: GsObjectInventory
profileRepository

"Scans the entire repository and creates a profile by class.  Returns a new
 instance of the receiver.

 Counts the number of instances and total physical bytes occupied of each
 class in the repository with one or more instances.

 Objects which are pending garbage collection reclamation are not included
 in the result.

 Acquires the garbage collection lock which prevents running garbage
 collection or other scans which require the GC lock while this method
 is running.  

 Raises error #rtErrGetGcLockFailed if the GC lock could not be acquired
 within 3 minutes.

 If this session contains uncommitted changes to the repository, 
 the method  signals a error: #rtErrAbortWouldLoseData, 
 to indicate that data could be lost. Otherwise
 it puts the session into auto-begin transaction mode and aborts."

System needsCommit
    ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^self _profileRepositoryAndShowHiddenClasses: true
%

category: 'Repository Analysis'
classmethod: GsObjectInventory
fastProfileRepository

"Same as the #profileRepository method except the scan is performed 
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

System needsCommit
    ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^self _fastProfileRepositoryAndShowHiddenClasses: true
%

category: 'Repository Analysis'
classmethod: GsObjectInventory
profileRepositoryAndSkipHiddenClasses

"Performs the same function as the #profileRepository method except that 
 statistics for GemStone private classes (e.g. LargeObjectNode,
 NscSetLeaf, etc) are not shown.  In effect, instances of private classes
 are ignored during the scan.  However the approximate physical space 
 occupied by private objects is included in the physical size of the
 private object's root object, which will always be a public class.
 For example, the approximate physical bytes consumed by a NscSetLeaf object
 will be included in the physical size reported for root object, which will be
 an IdentitySet (or a subclass of IdentitySet)."

^self _profileRepositoryAndShowHiddenClasses: false
%

category: 'Repository Analysis'
classmethod: GsObjectInventory
fastProfileRepositoryAndSkipHiddenClasses

"Same as the #profileRepositoryAndSkipHiddenClasses method except the scan is performed 
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

^self _fastProfileRepositoryAndShowHiddenClasses: false
%

category: 'Garbage Analysis'
classmethod: GsObjectInventory
profileGarbageFromFile: aFilename includeHiddenObjects: aBoolean

"Analyze the disconnected (garbage) objects in the repository using
 aFilename (an instance of String), which was produced by one of the 
 Repository>>findDisconnectedObjectsAndWriteToFile: methods.  Only objects
 which still exist in the repository are analyzed.  Does not create
 references to any object in the file.

 Scans the entire repository and creates a profile by class of the objects
 in the file.  Returns a new instance of the receiver.

 DeadNotReclaimed objects (those pending garbage collection reclamation)
 are not included in the result.

 IMPORTANT:  This method stays in a transaction for its entire duration and
             is therefore not recommended for use on production systems.

 Requires the #GarbageCollection privilege.

 Acquires the garbage collection lock which prevents running garbage
 collection or other scans which requires the GC lock while this method
 is running.  

 Raises error #rtErrGetGcLockFailed if the GC lock could not be acquired
 within 3 minutes.

 If this session contains uncommitted changes to the repository, 
 the method  signals a error: #rtErrAbortWouldLoseData, 
 to indicate that data could be lost. Otherwise
 it puts the session into auto-begin transaction mode and aborts."

System needsCommit
    ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .
(GsBitmap newForHiddenSet: #GcCandidates) removeAll.
SystemRepository loadGcCandidatesFromFile: aFilename intoHiddenSet: #GcCandidates .
^ self _objInventory: 2 waitForLock: 3 pageBufSize: 8 
       percentCpuActiveLimit: 90 showHiddenClasses: aBoolean
       aHiddenSet: (GsBitmap newForHiddenSet: #GcCandidates)
       listInstances: nil toFile: nil inMemoryOnly: false
%

category: 'Garbage Analysis'
classmethod: GsObjectInventory
fastProfileGarbageFromFile: aFilename includeHiddenObjects: aBoolean

"Same as the #profileGarbageFromFile:includeHiddenObjects: method 
 except the scan is performed  aggressively in order to complete 
 in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

System needsCommit
    ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .
(GsBitmap newForHiddenSet: #GcCandidates) removeAll.
SystemRepository loadGcCandidatesFromFile: aFilename intoHiddenSet: #GcCandidates .
^ self _objInventory: SystemRepository _aggressiveMaxThreadCount waitForLock: 3 pageBufSize: 8 
       percentCpuActiveLimit: 95 showHiddenClasses: aBoolean
       aHiddenSet: (GsBitmap newForHiddenSet: #GcCandidates)
       listInstances: nil toFile: nil inMemoryOnly: false
%

category: 'Private'
classmethod: GsObjectInventory
_profileRepositoryAndShowHiddenClasses: aBoolean

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ self _objInventory: 2 waitForLock: 3 pageBufSize: 8 
       percentCpuActiveLimit: 90 showHiddenClasses: aBoolean
       aHiddenSet: (GsBitmap new)
       listInstances: nil toFile: nil inMemoryOnly: false
%

category: 'Repository Analysis'
classmethod: GsObjectInventory
profileMemory

"Scans this session's temporary object memory and creates a profile by class.  
 Returns a new instance of the receiver.

 Counts the number of instances and total physical bytes occupied 
 for objects in memory.
"

^ self _objInventory: 2 waitForLock: 3 pageBufSize: 8
       percentCpuActiveLimit: 90 showHiddenClasses: true
       aHiddenSet: (GsBitmap new)
       listInstances: nil toFile: nil inMemoryOnly: true
%

category: 'Private'
classmethod: GsObjectInventory
_fastProfileRepositoryAndShowHiddenClasses: aBoolean

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ self _objInventory: SystemRepository _aggressiveMaxThreadCount waitForLock: 3 pageBufSize: 8 
       percentCpuActiveLimit: 95 showHiddenClasses: aBoolean
       aHiddenSet: (GsBitmap new)
       listInstances: nil toFile: nil inMemoryOnly: false
%

! fixed 45930
category: 'Private'
classmethod: GsObjectInventory
_objInventory: maxSessions waitForLock: lockWaitTime 
      pageBufSize: aBufSize percentCpuActiveLimit: percentCpu
      showHiddenClasses: aBoolean aHiddenSet: hiddenSetSpecifier
      listInstances: anIDSetOfClasses toFile: aFileName     
      inMemoryOnly: inMemBoolean 

" This primitive method performs a scan of the Repository.
  If inMemBoolean==true the scan only looks at temporary object memory,
  otherwise the scan only looks at committed objects .

  This primitive uses a multi-threaded algoritm to sweep the active
  data pages in the repository to gather the information requested.

  The maxSessions argument specifies the maximum number of slave sessions
  (threads) that will be used during the operation.  The actual number of 
  active threads can be adjusted to a lower value automatically at runtime 
  by the main thread based on the percentCpuActiveLimit and by the user 
  by interactively setting the mtThreadLimit (see details below).

  The lockWaitTime argument is used to specify how many seconds method should
  wait while attempting to acquire the gcLock.  No other garbage collection 
  operations may be started or in progress while this method is running.  
  Objects in the possibleDead or deadNotReclaimed sets at the start of 
  the scan are ignored by the scan.
 
  The pageBufSize, which must be a power of two, specifies the number 
  of pages to buffer per thread. This parameter in conjunction with the 
  maxSessions largely determines the memory footprint needed to perform 
  this operation.  The pageBufSize doesn't have much impact on the 
  performance of the scan, so a default size of 8 is probably sufficient.

  The percentCpu specifies a level of total cpu activity at which the 
  algorithm automatically inactivates threads to prevent overload 
  of system resources.

  This algorithm makes use of additional sessions (threads) to achieve 
  significant performance improvements.  It also makes space/time trade offs
  so that heap memory in addition to the TemporaryObjectCache (TOC)
  resources are used.  In fact, this algorithm doesn't require much TOC, 
  except for the operations that return array results, so configuring this 
  process for a smaller TOC space is advantageous. 

  The memory space that is needed is variable and depends upon number of
  instances being searched for, the number found, the number of sessions 
  requested and the pageBufSize specified for each.

  The showHiddenClasses argument indicates whether instances of hidden 
  classes are included in the results.

  If the hiddenSetSpecifier is not an empty GsBitmap, then only objects in the hidden set
  are included in the results.

  If the hiddenSetSpecifier is a valid hidden set that is NOT 34 
  (GcCandidates), then the primitive assumes that it is analyzing 
  the contents of a collection and performs the scan in a transaction 
  which can cause a large commit record backlog if the set is large 
  and there is a lot of commit activity.  Otherwise, the operation can
  cause the session to abort to prevent a commit record backlog during
  the scan.

  If inMemBoolean == true, ignores arguments   
     maxSessions, lockWaitTime, aBufSize, percentCpu
     hiddenSetSpecifier, aBoolean ,  aFileName  
  and reports on the in-memory objects of this session .

  The anIDSetOfClasses and aFileName arguments are used for the combined 
  functions of profiling the repository and listing specified instances.

  See the profileRepository* methods below.
"

<primitive: 897>
  | maxInt |
  maxInt := SmallInteger maximum32bitInteger .
  maxSessions _validateClass: SmallInteger; _validateMin: 1 max: maxInt .
  lockWaitTime _validateClass: SmallInteger ; _validateMin: -1 max: maxInt .
  aBufSize _validateClass: SmallInteger; _validateIsPowerOf2 .
  Repository _validatePercentage: percentCpu  .
  aBoolean _validateClass: Boolean.
  hiddenSetSpecifier _validateClass: GsBitmap.
  anIDSetOfClasses _validateInstanceOf: IdentitySet.
  aFileName _validateKindOfClass: String.
^ self _primitiveFailed: #_objInventory:waitForLock:pageBufSize:percentCpuActiveLimit:showHiddenClasses:aHiddenSet:listInstances:toFile:inMemoryOnly:
       args: { maxSessions . lockWaitTime . aBufSize . 
               percentCpu . aBoolean . hiddenSetSpecifier .
               anIDSetOfClasses . aFileName . inMemBoolean }
%

category: 'Profiling'
classmethod: GsObjectInventory
profileObjectsInHiddenSet: hiddenSetSpecifier showHiddenClasses: aBool

"Generate a profile of the objects in the given hidden set which
 exist and for which the session has permission to read.  Objects
 which do not meet these criteria are silently omitted from the result.

 This method does not alter the contents of the hidden set.

 Returns an instance of the receiver.

 If this session contains uncommitted changes to the repository, 
 the method  signals a error: #rtErrAbortWouldLoseData, 
 to indicate that data could be lost. Otherwise
 it puts the session into auto-begin transaction mode and aborts."

System needsCommit
    ifTrue: [ self _error: #rtErrAbortWouldLoseData ].

^ self _objInventory: 2 waitForLock: 3 pageBufSize: 8 
       percentCpuActiveLimit: 90 showHiddenClasses: aBool
       aHiddenSet: (GsBitmap newForHiddenSet: hiddenSetSpecifier)
       listInstances: nil toFile: nil inMemoryOnly: false
%

category: 'Profiling'
classmethod: GsObjectInventory
fastProfileObjectsInHiddenSet: hiddenSetSpecifier showHiddenClasses: aBool

"Same as the #profileObjectsInHiddenSet:showHiddenClasses: method 
 except the scan is performed  aggressively in order to complete 
 in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

System needsCommit
    ifTrue: [ self _error: #rtErrAbortWouldLoseData ].

^ self _objInventory: SystemRepository _aggressiveMaxThreadCount waitForLock: 3 pageBufSize: 8 
       percentCpuActiveLimit: 95 showHiddenClasses: aBool
       aHiddenSet: (GsBitmap newForHiddenSet: hiddenSetSpecifier)
       listInstances: nil toFile: nil inMemoryOnly: false
%


category: 'Profiling'
classmethod: GsObjectInventory
profileObjectsIn: aCollection

"Generate a profile of the objects in the given collection.
 The collection must not contain any special objects, otherwise
 an error will be raised.

 Returns an instance of the receiver."

| anArray objInvBm |
System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

anArray := aCollection asArray .
objInvBm := GsBitmap newForHiddenSet: #ObjInventory.
objInvBm removeAll.
objInvBm addAll: anArray.

^ self _objInventory: 2 waitForLock: 3 pageBufSize: 8 
       percentCpuActiveLimit: 90 showHiddenClasses: true
       aHiddenSet: objInvBm
       listInstances: nil toFile: nil inMemoryOnly: false
%


category: 'Profiling'
classmethod: GsObjectInventory
fastProfileObjectsIn: aCollection

"Same as the #profileObjectsIn: method except the scan is performed 
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

| anArray objInvBm |
System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .
anArray := aCollection asArray .
objInvBm := GsBitmap newForHiddenSet: #ObjInventory.
objInvBm removeAll.
objInvBm addAll: anArray.
^ self _objInventory: SystemRepository _aggressiveMaxThreadCount waitForLock: 3 pageBufSize: 8 
       percentCpuActiveLimit: 95 showHiddenClasses: true
       aHiddenSet: objInvBm
       listInstances: nil toFile: nil inMemoryOnly: false
%


! ------------------- Instance methods for GsObjectInventory
category: 'Formatting'
method: GsObjectInventory
asString

| sz result|
sz := self entriesByCount size.
result := String new.
result addAll: 'a'; addAll: self class name; addAll: ' of '; addAll: sz asString; addAll: ' classes.'.
^result
%
category: 'Reporting'
method: GsObjectInventory
byteCountReport
	^self byteCountReportDownTo: 0

%
category: 'Reporting'
method: GsObjectInventory
byteCountReportDownTo: minCountToReport

"Return a String showing the receiver in tabular form sorted by instance count."

| totalBytes totalCount result tmp lf array line space|
lf := Character lf.
space := Character space.
result := String new.
tmp := String withAll: '*** GsObjectInventory byteCountReport printed at: '.
tmp addAll: DateTime now asString.
tmp addAll: ' ***'.
result addLineWith: tmp centeredToWidth: 80 ; 
       add: lf.
result addAll: 'Hidden classes are '.
self includeHiddenClasses
  ifFalse:[ result addAll: 'not '.].
result addAll: 'included in this report.'; add: lf.
line := String new.
80 timesRepeat:[line add: $_].
line add: lf.
result addAll: line.

result 	addAll: ((String withAll: 'Class') width: 50; yourself); add: space;
		addAll: ((String withAll: 'Instances') width: -14; yourself); add: space;
		addAll: ((String withAll: 'Bytes') width: -14; yourself); add: lf.
result addAll: line.
totalBytes := 0.
totalCount := 0.
array := self entriesByBytes.
1 to: array size do:[:n| |entry|
  entry := array at: n.
  (entry byteCount < minCountToReport)
	ifTrue:[result addAll: line.
		^result ].
  result addAll: (entry theClass name asString width: 50); add: space;
		addAll: (entry instanceCount asString width: -14); add: space;
		addAll: (entry byteCount asString width: -14); add: lf.
  totalCount := totalCount + entry instanceCount.
  totalBytes := totalBytes + entry byteCount.
].
result addAll: line.
result 	addAll: ((String withAll: 'Totals') width: 50; yourself); add: space;
        addAll: (totalCount asString width: -14) ; add: space;
        addAll: (totalBytes asString width: -14); add: lf.
result addAll: line.
^result
%
category: 'Accessing'
method: GsObjectInventory
entriesByBytes

^entriesByBytes
%
category: 'Updating'
method: GsObjectInventory
entriesByBytes: newValue

entriesByBytes := newValue
%
category: 'Accessing'
method: GsObjectInventory
entriesByCount

^entriesByCount
%
category: 'Accessing'
method: GsObjectInventory
includeHiddenClasses

^includeHiddenClasses
%
category: 'Updating'
method: GsObjectInventory
entriesByCount: newValue

entriesByCount := newValue
%
category: 'Updating'
method: GsObjectInventory
includeHiddenClasses: newValue

includeHiddenClasses := newValue
%
category: 'Reporting'
method: GsObjectInventory
instanceCountReport

"Return a String showing the receiver in tabular form sorted by instance count."
^self instanceCountReportDownTo: 0
%
category: 'Reporting'
method: GsObjectInventory
instanceCountReportDownTo: minCountToReport

"Return a String showing the receiver in tabular form sorted by instance count."
| totalCount totalBytes result tmp lf array line space|
lf := Character lf.
space := Character space.
result := String new.
tmp := String withAll: '*** GsObjectInventory instanceCountReport printed at: '.
tmp addAll: DateTime now asString.
tmp addAll: ' ***'.
result addLineWith: tmp centeredToWidth: 80 ; 
       add: lf.
result addAll: 'Hidden classes are '.
self includeHiddenClasses
  ifFalse:[ result addAll: 'not '.].
result addAll: 'included in this report.'; add: lf.

line := String new.
80 timesRepeat:[line add: $_].
line add: lf.
result addAll: line.
result 	addAll: ((String withAll: 'Class') width: 50; yourself); add: space;
		addAll: ((String withAll: 'Instances') width: -14; yourself); add: space;
		addAll: ((String withAll: 'Bytes') width: -14; yourself); add: lf.
result addAll: line.
array := self entriesByCount.
totalCount := 0.
totalBytes := 0.
1 to: array size do:[:n| |entry|
  entry := array at: n.
  (entry instanceCount < minCountToReport)
	ifTrue:[	result addAll: line.
			^result
	].
  result addAll: (entry theClass name asString width: 50); add: space;
		addAll: (entry instanceCount asString width: -14); add: space;
		addAll: (entry byteCount asString width: -14); add: lf.
  totalCount := totalCount + entry instanceCount.
  totalBytes := totalBytes + entry byteCount.
].
result addAll: line.
result 	addAll: ((String withAll: 'Totals') width: 50; yourself); add: space;
        addAll: (totalCount asString width: -14) ; add: space;
        addAll: (totalBytes asString width: -14); add: lf.
result addAll: line.
^result
%

! Start request 43179
! 45292: fix return value

category: 'Repository Analysis'
classmethod: GsObjectInventory
profileRepositoryAndListInstancesInPageOrder: anArrayOfClasses toFile: aString

"Combines the functions of the following methods in a single scan of the 
 repository:

   GsObjectInventory>>profileRepository
   Repository>>listInstancesInPageOrder: toFile:

 Refer to the comments in these methods for more information.

 Requires the #GarbageCollection privilege.

 Acquires the garbage collection lock which prevents running garbage
 collection or other scans which requires the GC lock while this method
 is running.  

 Raises error #rtErrGetGcLockFailed if the GC lock could not be acquired
 within 3 minutes.

 Aborts the current transaction.   If this session contains uncommitted
 changes to the repository, the method  signals a error: 
 #rtErrAbortWouldLoseData, to indicate that data could be lost. Otherwise
 it puts the session into auto-begin transaction mode and aborts.

 Returns an Array containing 2 elements:
  [1] - An new instance of the receiver.
  [2] - An Integer indicating the number of object identifiers written 
        to the file."

| idSet objinv count |
System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

idSet := IdentitySet withAll: anArrayOfClasses .
objinv :=  self _objInventory: 2 waitForLock: 3 pageBufSize: 8 
       percentCpuActiveLimit: 90 showHiddenClasses: true
       aHiddenSet: (GsBitmap new)
       listInstances: idSet toFile: aString inMemoryOnly: false .
count := 0.
objinv entriesByCount do: [:entry |
  (idSet includes: (entry theClass)) 
     ifTrue: [ count := count + (entry instanceCount)]].
^ Array with: objinv with: count
%

category: 'Repository Analysis'
classmethod: GsObjectInventory
profileRepositoryAndSkipHiddenClassesAndListInstancesInPageOrder: anArrayOfClasses toFile: aString

"Combines the functions of the following methods in a single scan of the 
 repository:

   GsObjectInventory>>profileRepositoryAndSkipHiddenClasses
   Repository>>listInstancesInPageOrder: toFile:

 Refer to the comments in these methods for more information.

 Requires the #GarbageCollection privilege.

 Acquires the garbage collection lock which prevents running garbage
 collection or other scans which requires the GC lock while this method
 is running.  

 Raises error #rtErrGetGcLockFailed if the GC lock could not be acquired
 within 3 minutes.

 Aborts the current transaction.   If this session contains uncommitted
 changes to the repository, the method  signals a error: 
 #rtErrAbortWouldLoseData, to indicate that data could be lost. Otherwise
 it puts the session into auto-begin transaction mode and aborts.

 Returns an Array containing 2 elements:
  [1] - An new instance of the receiver.
  [2] - An Integer indicating the number of object identifiers written 
        to the file."

| idSet objinv count |
System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

idSet := IdentitySet withAll: anArrayOfClasses .
objinv :=  self _objInventory: 2 waitForLock: 3 pageBufSize: 8 
       percentCpuActiveLimit: 90 showHiddenClasses: false
       aHiddenSet: (GsBitmap new)
       listInstances: idSet toFile: aString inMemoryOnly: false .
count := 0.
objinv entriesByCount do: [:entry |
  (idSet includes: (entry theClass)) 
     ifTrue: [ count := count + (entry instanceCount)]].
^ Array with: objinv with: count
%

