!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! gsmethoddictionary.gs
! 
! Superclass Hierarchy:
!   GsMethodDictionary, IdentityKeyValueDictionary, KeyValueDictionary,
!   Collection, Object.
!
!=========================================================================

! remove existing behavior from GsMethodDictionary
removeallmethods GsMethodDictionary
removeallclassmethods GsMethodDictionary

category: 'For Documentation Installation only'
classmethod: GsMethodDictionary
installDocumentation

self comment:
'GsMethodDictionary optimizes IdentityKeyValueDictionary for use as method
 dictionaries in classes.  It employs a different internal structure that is
 well-suited for efficient execution in smaller dictionaries.  Changes to
 GsMethodDictionies are protected by the #CodeModification privilege, and
 therefore should not be used in customer applications.  For this purpose,
 you should use the subclass FastIdentityKeyValueDictionary.

 The keys of method dictionaries must be canonical symbols 
 (Symbols, DoubleByteSymbols, or QuadByteSymbols).

 Implementation details:
 Within the hash table, entries are of size two and contain key/value pairs:
    aSymbol, aValue
    nil,     SmallInteger - one-based offset of start of collision chain
    nil,     nil          - empty hash slot
 
 Collisions chains are linked lists within the root object itself, and
 are stored in the area after the hash table.  
 Collision list entries are triples (key, value, nextOffset):
    aSymbol, aValue, nextOffset - entry in chain, nextOffset is one-based
    aSymbol, aValue, nil        - end of chain
    nil,     nil,    nextOffset - empty element in collision chain, only
                                  removed by rebuildTable 

Constraints:
	numElements: SmallInteger
	numCollisions: SmallInteger
	collisionLimit: SmallInteger
	tableSize: SmallInteger
	valueConstraint: Behavior
	keyConstraint: Behavior

instVar valueConstraint -- The Class that specifies a constraint on the GsMethodDictionary''s values.
 If nil, there is no constraint.

instVar keyConstraint -- The Class that specifies a constraint on the GsMethodDictionary''s keys.
 If nil, there is no constraint.' .
%


! ------------------- Class methods for GsMethodDictionary

! instance creation inherited from KeyValueDictionary

! ------------------- Instance methods for GsMethodDictionary

category: 'Private'
method: GsMethodDictionary
collisionBucketClass

"Returns the class of object to create when keys collide.
 GsMethodDictionary does not use collision buckets."

^ nil
%

! initialize: inherited from KeyValueDictionary 

category: 'Hashing'
method: GsMethodDictionary
hashFunction: aKey

"The hash function performs an operation on the value of the key aKey and
 returns some Integer between 1 and tableSize, inclusive."

^(aKey basicIdentityHash \\ self tableSize) + 1
%

category: 'Accessing'
method: GsMethodDictionary
at: aKey ifAbsent: aBlock

"Returns the value whose key is identical to aKey.  If no such key/value pair
 exists, returns the result of evaluating the zero-argument block aBlock."

| offset hashKey |

nil == aKey ifTrue:[ ^self _reportKeyNotFound: aKey with: aBlock ].

(hashKey := self _basicAt: 
       ( offset := (self hashFunction: aKey) * 2 - 1 )
  ) == aKey 
ifTrue:[
  ^ self _basicAt: offset + 1 . "the value"  
  ]
ifFalse:[
  nil == hashKey ifTrue:[ 
    "search collision chain"
    offset := self _basicAt: (offset + 1). "get one based offset to start of chain"
    [ offset == nil ] whileFalse:[
      (self _basicAt: offset )  == aKey ifTrue:[ 
        ^ self _basicAt: offset + 1 
        ].
      offset := self _basicAt: offset + 2 .
      ] .
    ].  
  ].
^ self _reportKeyNotFound: aKey with: aBlock .
%

category 'Accessing'
method: GsMethodDictionary
valueConstraint

"Returns the value constraint of the receiver."

^ valueConstraint
%

category 'Accessing'
method: GsMethodDictionary
keyConstraint

"Returns the key constraint of the receiver."

^ keyConstraint
%

category 'Updating'
method: GsMethodDictionary
valueConstraint: aClass

"Sets the value constraint of the receiver to aClass.  Generates an error if
 the receiver is not empty."

self _validatePrivilege ifTrue:[
  numElements ~~ 0 ifTrue:[ ^ self _error: #rtErrCannotChgConstraint ].
  valueConstraint := aClass
]
%

category 'Updating'
method: GsMethodDictionary
keyConstraint: aClass

"Sets the key constraint of the receiver to aClass.  Generates an error if
 the receiver is not empty."

self _validatePrivilege ifTrue:[
  numElements ~~ 0 ifTrue:[ ^ self _error: #rtErrCannotChgConstraint ].
  keyConstraint := aClass
]
%

category: 'Updating'
method: GsMethodDictionary
at: aKey put: aValue

"Stores the aKey/aValue pair in the hash dictionary.  Rebuilds the hash table
 if the addition caused the number of collisions to exceed the limit allowed.

 If aKey is not compatible with the key constraint of the receiver, or aValue is
 not compatible with the value constraint of the receiver, an error is
 generated."

self _validatePrivilege ifTrue:[
  self _atKey: aKey put: aValue .
].
^ aValue
%

category: 'Private'
method: GsMethodDictionary
atHash: hashIndex putValue: aValue

"Disallowed."

self shouldNotImplement: #atHash:putValue:
%

category: 'Private'
method: GsMethodDictionary
atHash: hashIndex putKey: aValue

"Disallowed."

self shouldNotImplement: #atHash:putKey:
%
method: GsMethodDictionary
atHash: hashIndex putKey: aKey value: aValue

"Disallowed."

self shouldNotImplement: #atHash:putKey:value:
%

category: 'Private'
method: GsMethodDictionary
_atKey: aKey put: aValue

"aValue == nil means remove KV pair , and result is
 the value removed, or nil if not found. 

 aValue ~~ nil means add KV pair, and return aValue.
 aValue must not be nil."

<primitive: 721>
self _primitiveFailed: #_atKey:put: args: { aKey . aValue }
%

category: 'Accessing'
method: GsMethodDictionary
at: aKey otherwise: aValue

"Returns the value whose key is identical to aKey.  If no such key/value pair
 exists, returns the given alternate value."

<primitive: 857>
keyConstraint ifNotNil:[ aKey _validateClass: keyConstraint ].
valueConstraint ifNotNil:[ aValue _validateClass: valueConstraint ].
self _primitiveFailed: #at:otherwise: args: { aKey . aValue }
%

category: 'Removing'
method: GsMethodDictionary
removeKey: aKey ifAbsent: aBlock

"Removes the key/value pair whose key is identical to aKey.  If no such
 key/value pair exists, returns the result of evaluating the zero-argument
 block aBlock."

 | oldVal |

 self _validatePrivilege ifTrue:[
   nil == aKey ifTrue:[ ^self _error: #rtErrNilKey ].
   oldVal := self _atKey: aKey put: nil .
   oldVal == nil ifTrue:[
     ^ self _reportKeyNotFound: aKey with: aBlock
   ].
   ^ oldVal
 ]
%

! added for 36675
category: 'Removing'
method: GsMethodDictionary
removeKey: aKey otherwise: notFoundValue

"Removes the key/value pair whose key is identical to aKey.  If no such
 key/value pair exists, returns notFoundValue . "

 | oldVal |

 self _validatePrivilege ifTrue:[
   nil == aKey ifTrue:[ ^self _error: #rtErrNilKey ].
   oldVal := self _atKey: aKey put: nil .
   oldVal == nil ifTrue:[
     ^ notFoundValue
   ].
   ^ oldVal
 ]
%

category: 'Removing'
method: GsMethodDictionary
removeAll

"Remove all key/value pairs from the receiver."

| tSize |
self _validatePrivilege ifTrue:[
  self _basicSize: 0 . "dereference all keys and values"
  tableSize :=  (tSize := 29) .
  self _basicSize: tSize + tSize  .  "reinitialize hash table to all nils"
  numElements := 0 .
  numCollisions := 0 
]
%

! tableSize: inherited from KeyValueDictionary
! rebuildTable:  inherited from KeyValueDictionary

category: 'Copying'
method: GsMethodDictionary
postCopy

"Because there are no collision buckets to copy, we can ignore inherited behavior."

^self
%

category: 'Accessing'
method: GsMethodDictionary
keyAtValue: anObject ifAbsent: aBlock

"Returns the key of the first value identical to anObject.  If no
 match is found, this method evaluates the block aBlock and returns its
 result."

self keysAndValuesDo:[ :aKey :aValue |
  anObject == aValue ifTrue:[ ^ aKey ]
  ].
^ aBlock value
%

category: 'Enumerating'
method: GsMethodDictionary
keysAndValuesDo: aBlock

"Evaluates aBlock with each of the receiver's key/value pairs as the
 arguments.  The argument aBlock must be a two-argument block.  The
 first argument is the key and the second argument is the value of
 each key/value pair.  Returns the receiver."

| aKey tSize |

"process hash slots"
tSize := tableSize * 2 .
1 to: tSize by: 2 do: [ :offset |
  (aKey := self _basicAt: offset) ifNotNil:[
    aBlock value: aKey value: (self _basicAt: offset + 1) .
    ].
  ].
"process collision chains"
tSize + 1 to: self _basicSize by: 3 do:[:offset |
  (aKey := self _basicAt: offset) ifNotNil:[ 
    aBlock value: aKey value: (self _basicAt: offset + 1) .
    ].
  ].
%

! fix 45043
category: 'Enumerating'
method: GsMethodDictionary
keysDo: aBlock

"Evaluates aBlock with each of the receiver's keys as the
 argument.  The argument aBlock must be a one-argument block.  
 Returns the receiver."

| aKey tSize |

"process hash slots"
tSize := tableSize * 2 .
1 to: tSize by: 2 do: [ :offset |
  (aKey := self _basicAt: offset) ifNotNil:[
    aBlock value: aKey .
    ].
  ].
"process collision chains"
tSize + 1 to: self _basicSize by: 3 do:[:offset |
  (aKey := self _basicAt: offset) ifNotNil:[ 
    aBlock value: aKey 
    ].
  ].
%

! added for 36675, renamed for 39898
category: 'Enumerating'
method: GsMethodDictionary
accompaniedBy: anObj keysAndValuesDo: aBlock

"Evaluates aBlock with each of the receiver's key/value pairs as the
 2nd and 3rd arguments.
 aBlock must be a 3 argument block, with arguments anObj, key value ."

| aKey tSize |
"process hash slots"
tSize := tableSize * 2 .
1 to: tSize by: 2 do: [ :offset |
  nil == (aKey := self _basicAt: offset) ifFalse:[
    aBlock value: anObj value: aKey value: (self _basicAt: offset + 1) .
    ].
  ].
"process collision chains"
tSize + 1 to: self _basicSize by: 3 do:[:offset |
  (aKey := self _basicAt: offset) == nil ifFalse:[
    aBlock value: anObj value: aKey value: (self _basicAt: offset + 1) .
    ].
  ].
%

category: 'Enumerating'
method: GsMethodDictionary
associationsDo: aBlock

"Evaluates aBlock with each of the receiver's key/value pairs as the argument by
 creating a SymbolAssociation for each key/value pair.  The argument aBlock must
 be a one-argument block.  Returns the receiver."

self keysAndValuesDo: [:aKey :aValue |
  aBlock value: (SymbolAssociation newWithKey: aKey value: aValue)
  ].
%

category: 'Statistics'
method: GsMethodDictionary
statistics

"A GsMethodDictionary has no collision buckets, so the statistics defined
 for KeyValueDictionary have no meaning."

^ nil
%

! detectValues:ifNone: inherited
! selectValues: inherited

! inherited  rejectValues: aBlock
! inherited selectValuesAsArray:
! inherited rejectValuesAsArray:

category: 'Formatting'
method: GsMethodDictionary
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

"copy the implementation from Object."

aStream nextPutAll: self asString
%

! deleted _canonicalizeSymbolAt: offset oldSymbol: oldSym newSymbol: newSym

category: 'Canonical Symbol Support'
method: GsMethodDictionary
_selectiveAbort

"Performs an abort operation on the receiver."

"Receiver is just an ordinary small or large object as far as selective abort is
 concerned, so execute the implementation in Object.  Must avoid the
 reimplementation in IdentityKeyValueDictionary, so can't use super."
  
^ self _primitiveSelectiveAbort
%

category: 'Clustering'
method: GsMethodDictionary
clusterDepthFirst

"This method clusters the receiver and its values in depth-first order.  The
 keys are not clustered because they are Symbols.

 Has no effect and returns true if the receiver was previously clustered in the
 current transaction."

self cluster
  ifTrue:[ ^ true ]
  ifFalse: [ 
      "none of the named instance variables should be clustered"

      self valuesDo:[:aMethod | aMethod clusterDepthFirst ].
      ^ false
    ]
%


category: 'Hashing'
method: GsMethodDictionary
rebuildTable: newSize

"Rebuilds the method dictionary by populating a larger method dictionary
 first and doing a (primitive) become:"

"NOTE: This method is reimplemented and reinstalled in bomlast.gs to handle
instances that have a dependency list, used for modification tracking."

<primitive: 2001>  "enter protected mode"
| prot |
prot := System _protectedMode .
[
  | sx newGsMethodDict cx |

  self _validatePrivilege ifFalse:[ ^ self ].
  tableSize == newSize ifTrue:[ ^ self "no change in table size" ].
  collisionLimit == 536870911 ifTrue:[
    ^ self  "avoid recursive rebuild"
  ].
  sx := self .
  newGsMethodDict := (cx := self class) new: (newSize + newSize).
  newGsMethodDict valueConstraint: valueConstraint.
  newGsMethodDict keyConstraint: keyConstraint.

  self keysAndValuesDo: [ :aKey :aValue |
    newGsMethodDict at: aKey put: aValue.
    ].

  newGsMethodDict _becomeDictionary: self.
] ensure:[
  prot _leaveProtectedMode
]
%

category: 'Initializing'
method: GsMethodDictionary
initialize: newSize

"Initializes the instance variables of the receiver to be an empty
 IdentityKeyValueDictionary of the specified size."

self _validatePrivilege ifTrue:[
  super initialize: newSize .
]
%

category: 'Private'
method: GsMethodDictionary
_initializeWithoutClear: newSize

"Private. Initializes the instance variables of the receiver to be an empty
 KeyValueDictionary of the specified size. Does not clear the contents
 of the receiver - assumes they are all nil."

tableSize := newSize.
numElements := 0.
numCollisions := 0.
collisionLimit := newSize // 4 .
^self
%

category: 'Private'
method: GsMethodDictionary
_resetParentRef

"Private. After a become:, the parent refs of the collisionBuckets must 
 be reset to point to the correct parent."

"GsMethodDictionarys don't use collision buckets"

^self

%

category: 'CodeModification Override'
method: GsMethodDictionary
instVarAt: anIndex put: aValue

self _validatePrivilege ifTrue:[
  ^ super instVarAt: anIndex put: aValue
]
%

category: 'CodeModification Override'
method: GsMethodDictionary
_basicAt: anIndex put: aValue

self _validatePrivilege ifTrue:[
  ^ super _basicAt: anIndex put: aValue
]
%

category: 'CodeModification Override'
method: GsMethodDictionary
squeakBasicAt: anIndex put: aValue

  ^ self _basicAt: anIndex put: aValue
%

category: 'CodeModification Override'
method: GsMethodDictionary
_basicSize: anInteger

self _validatePrivilege ifTrue:[
  ^ super _basicSize: anInteger
]
%

category: 'CodeModification Override'
method: GsMethodDictionary
_primitiveAt: anIndex put: aValue

self _validatePrivilege ifTrue:[
  ^ super _primitiveAt: anIndex put: aValue
]
%

category: 'CodeModification Override'
method: GsMethodDictionary
_unsafeAt: anIndex put: aValue

self _validatePrivilege ifTrue:[
  ^ super _unsafeAt: anIndex put: aValue
]
%

category: 'CodeModification Override'
method: GsMethodDictionary
_validatePrivilege

( self isMemberOf: GsMethodDictionary ) ifTrue: [
  ^   System myUserProfile _validateCodeModificationPrivilege 
].
^ true
%

category: 'Private'
method: GsMethodDictionary
_collisionBucketsDo: aBlock
  "a GsMethodDictionary has no collision buckets"
  ^ self
%

! objectSecurityPolicy: inherited:

category: 'Private'
method: GsMethodDictionary
_nodesObjectSecurityPolicy: anObjectSecurityPolicy
  "a GsMethodDictionary has no collision buckets"
  ^ self
%

category: 'Testing'
method: 
_isLarge

"Returns true if the object is implemented as a tree of private smaller objects.
 Note that GsMethodDictionary does not use collision buckets.
"
^ ((self _status: false) bitShift: -5) ~~ 0
%


!-------------------------------------------------------------------------
! NOTE:
! The method GsMethodDictionary>>rebuildTable: is reimplemented and
! reinstalled in bomlast.gs to handle instances that have a dependency list
!-------------------------------------------------------------------------


