! ========================================================================
! KerberosPrincipal.gs
!
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
! ========================================================================
set compile_env: 0

expectvalue %String
run
^ Object _newKernelSubclass: 'KerberosPrincipal'
        instVarNames: #( 'name' 'loginUserProfile' 'loginUserProfileGroups' 'loginAsAnyoneEnabled' )
        classVars: #( )
        classInstVars: #()
        poolDictionaries: { }
        inDictionary: Globals
        options: #( ) 
        reservedOop: 1233
%

! Remove existing behavior from KerberosPrincipal
doit
KerberosPrincipal removeAllMethods.
KerberosPrincipal class removeAllMethods.
true
%

! ------------------- Class comment for KerberosPrincipal
category: 'For Documentation Installation only'
classmethod: KerberosPrincipal
installDocumentation

self comment:
'This class is the logical representation of a Kerberos principal.
 It contains the following instance variables:
   name - A symbol representing the Kerberos name of the principal, including the realm.
          Example: #''alice@GEMTALKSYSTEMS.COM''
			    
   loginUserProfile - a UserProfile that corresponds to the Kerberos principal, 
             or nil if the principal does not map to a single UserProfile.  
             The named Kerberos user principal may login to this UserProfile without 
             specifying a password.
			    
   loginUserProfileGroups - anIdentitySet of UserProfileGroups.  The named Kerberos user 
             principal may login as any UserProfile in any of these groups without 
             specifying a password.

   loginAsAnyoneEnabled - a Boolean indicating if the named Kerberos principal has 
             permission to login as any UserProfile except for SystemUser.
'.
%

! ------------------- Class methods for KerberosPrincipal
category: 'Accessing'
classmethod: KerberosPrincipal
allPrincipals
  ^ Globals at: #AllKerberosPrincipals otherwise: nil
%

category: 'Accessing'
classmethod: KerberosPrincipal
principalWithName: aStringOrSymbol
"Answer the KerberosPrincipal that has the given name or nil if no
 such KerberosPrincipal exists."
 
  ^ self allPrincipals at: aStringOrSymbol otherwise: nil
%

category: 'Illegal Operations'
classmethod: KerberosPrincipal
new

"Disallowed.  To create a new KerberosPrincipal, use newWithName:loginUserProfile: instead "

self shouldNotImplement: #new
%
category: 'Illegal Operations'
classmethod: KerberosPrincipal
new: anInt

"Disallowed.  To create a new KerberosPrincipal, use newWithName:loginUserProfile: instead."
self shouldNotImplement: #new:
%

category: 'Private'
classmethod: KerberosPrincipal
basicNewWithName: aStringOrSymbol loginUserProfile: aUserProfile
"Creates a new instance of the receiver, but does not add the new instance
 to the list to AllKerberosPrincipals.  It is an error if a principal with the same name 
 already exists.  It is allowed to use nil for aUserProfile, in the case the 
 KerberosPrincipal does not map to a single UserProfile."
 
| result |
((self principalWithName: aStringOrSymbol) notNil)
  ifTrue:[ ^ self _error: #rtErrPrincipalAlreadyExists args: { aStringOrSymbol } ] .
  
result := super new .
result _initializeForName: aStringOrSymbol ; loginUserProfile: aUserProfile .
^ result
%

category: 'Updating'
classmethod: KerberosPrincipal
newPrincipalWithName: aStringOrSymbol loginUserProfile: aUserProfile
"Creates a new instance of KerberosPrincipal and adds it to the AllKerberosPrincipals 
 collection. It is an error if a principal with the same name already exits. It is allowed to 
 use nil for aUserProfile, in the case the KerberosPrincipal does not map to a single 
 UserProfile.

 Requires write access to the GsObjectSecurityPolicy for DataCurator."
 
| result |
result := self basicNewWithName: aStringOrSymbol loginUserProfile: aUserProfile .
self allPrincipals at: aStringOrSymbol put: result .
^ result
%

! ------------------- Instance methods for KerberosPrincipal
category: 'Accessing'
method: KerberosPrincipal
name
  ^ name
%
category: 'Accessing'
method: KerberosPrincipal
loginUserProfile
  ^ loginUserProfile
%
category: 'Accessing'
method: KerberosPrincipal
loginUserProfileGroups
  ^ loginUserProfileGroups
%  
category: 'Accessing'
method: KerberosPrincipal
loginAsAnyoneEnabled
  ^ loginAsAnyoneEnabled
%  

category: 'Private'
method: KerberosPrincipal
_initializeForName: aStringOrSymbol
"Private.  Initializes the receiver. Used only at instance creation time."
name := aStringOrSymbol asSymbol .
loginUserProfileGroups := IdentitySet new .
loginAsAnyoneEnabled := false .
^ self

%
category: 'Updating'
method: KerberosPrincipal
loginUserProfile: aUserProfile
  loginUserProfile := aUserProfile
%

category: 'Accessing'
method: KerberosPrincipal
addGroup: aUserProfileGroup
  aUserProfileGroup _validateClass: UserProfileGroup .
  loginUserProfileGroups add: aUserProfileGroup .
  ^ self
%

category: 'Accessing'
method: KerberosPrincipal
loginAsAnyoneEnabled
  ^ loginAsAnyoneEnabled
%

category: 'Testing'
method: KerberosPrincipal
validateLoginFor: aUserProfile

aUserProfile _validateClass: UserProfile .
(aUserProfile userId = 'SystemUser')
  ifTrue:[ ^ false ]. "Passwordless login may not be used for SystemUser"

(aUserProfile == loginUserProfile)
  ifTrue:[ ^ true ].

loginAsAnyoneEnabled == true
  ifTrue:[ ^ true ].

^ (loginUserProfileGroups detect:[:e| e includesIdentical: aUserProfile] ifNone:[nil]) notNil
%

category: 'Removing'
method: KerberosPrincipal
removeGroup: aUserProfileGroup
  loginUserProfileGroups removeIdentical: aUserProfileGroup ifAbsent:[] .
  ^ self
%

category: 'Removing'
method: KerberosPrincipal
removeGroup: aUserProfileGroup ifAbsent: aBlock
  aUserProfileGroup _validateClass: UserProfileGroup .
  loginUserProfileGroups removeIdentical: aUserProfileGroup ifAbsent: aBlock .
  ^ self
%

category: 'Group Management'
classmethod: KerberosPrincipal
removeGroup: aUserProfileGroup

"Remove the given UserProfileGroup object from every instance of KerberosPrincipal contained
 in the global collection AllKerberosPrincipals.  Returns the receiver.

 Requires write access to the DataCurator segment."
 
aUserProfileGroup _validateClass: UserProfileGroup .
self allPrincipals valuesDo:[:eachPrincipal| eachPrincipal removeGroup: aUserProfileGroup ].
^ self
%

category: 'Group Management'
classmethod: KerberosPrincipal
removeGroupWithName: aStringOrSymbol

"Remove the UserProfileGroup with name aStringOrSymbol from every instance of 
 KerberosPrincipal contained in the global collection AllKerberosPrincipals.
 Returns the receiver.

 Requires write access to the DataCurator segment."

| group |
group := UserProfileGroup groupWithName: aStringOrSymbol ifAbsent:[^ self ].
self allPrincipals valuesDo:[:eachPrincipal| eachPrincipal removeGroup: group ].
^ self
%

category: 'Updating'
method: KerberosPrincipal
loginAsAnyoneEnabled: aBoolean
  loginAsAnyoneEnabled := aBoolean
%  

commit
