! ------------------- Class definition for GsFileIn
expectvalue /Class
doit
Object subclass: 'GsFileIn'
  instVarNames: #( session fileStream line
                    currentClass category compileEnvironment)
  classVars: #()
  classInstVars: #()
  poolDictionaries: #()
  inDictionary: Globals
  options: #()

%
expectvalue /Class
doit
GsFileIn comment: 
'GsFileIn supports filing in from topaz-format GemStone source files into the image, without the use of topaz.
	
	To use, send one of the from* methods.  This also performs the file in.
	  for example,
	     GsFileIn fromServerPath: ''mySourceCode.gs''
	
	Only a subset of topaz commands is supported:
	   doit , printit , run , nbrun, send
	   input
	   category:
	   classmethod , classmethod:
	   method , method:
	   removeallmethods , removeallclassmethods
	   env N
	   set compile_env N , set class
	     (other set commands are ignored)
Following can be read but are ignored:
	   expectvalue, expecterror, iferr, iferr_clear, iferr_list, errorcount
	   fileformat, fileout, output
	   display, omit, level, limit
	   time, remark, status'
%
expectvalue /Class
doit
GsFileIn category: 'Kernel'
%
! ------------------- Remove existing behavior from GsFileIn
expectvalue /Metaclass3
doit
GsFileIn removeAllMethods .
GsFileIn class  removeAllMethods .
%
set compile_env: 0
! ------------------- Class methods for GsFileIn
category: 'other'
classmethod: GsFileIn
fromClientPath: aString
"file in from path on session client machine into current session"

	self
		fromClientPath: aString
		to: nil.
%
category: 'other'
classmethod: GsFileIn
fromClientPath: aString to: anExternalSession

	self
		fromPath: aString
		on: #clientText
		to: anExternalSession.
%
category: 'other'
classmethod: GsFileIn
fromGciHostPath: aString

	self
		fromGciHostPath: aString
		to: nil.
%
category: 'other'
classmethod: GsFileIn
fromGciHostPath: aString to: anExternalSession

	self
		fromPath: aString
		on: #clientText
		to: anExternalSession.
%
category: 'other'
classmethod: GsFileIn
fromGemHostPath: aString

	self
		fromGemHostPath: aString
		to: nil.
%
category: 'other'
classmethod: GsFileIn
fromGemHostPath: aString to: anExternalSession

	self
		fromPath: aString
		on: #serverText
		to: anExternalSession.
%
category: 'other'
classmethod: GsFileIn
fromPath: aString on: aFileType to: anExternalSession
"aString is file path to file containg topaz file-out format of smalltalk code.
type is either #clientFile or #serverFile to denote relative user action to use.
anExternal session is needed if file-in happens into that given session"

	| fileStream  |
	
	fileStream := FileStreamPortable read: aString type: aFileType. 
	[ (self _fromStream: fileStream to: anExternalSession) doFileIn	] ensure: [fileStream close].
%
category: 'other'
classmethod: GsFileIn
fromServerPath: aString

	self
		fromServerPath: aString
		to: nil.
%
category: 'other'
classmethod: GsFileIn
fromServerPath: aString to: anExternalSession

	self
		fromPath: aString
		on: #serverText
		to: anExternalSession.
%
category: 'other'
classmethod: GsFileIn
new

	^super new 
		initialize;
		yourself.
%
category: 'other'
classmethod: GsFileIn
_fromStream: aFileStream
"file in from a stream to current session"

	^self _fromStream: aFileStream to: nil
%
category: 'other'
classmethod: GsFileIn
_fromStream: aFileStream to: anExternalSessionOrNil
"anExternalSessionOrNil can be nil meaning file in will happen in current session"

	^self new 
		fileStream: aFileStream;
		setSession: anExternalSessionOrNil;
		yourself
%
! ------------------- Instance methods for GsFileIn
category: 'processing'
method: GsFileIn
abort
	
	^System abortTransaction
%
category: 'processing'
method: GsFileIn
abortTransaction

	^session abort
%
category: 'processing'
method: GsFileIn
category

	category := (line copyFrom: 10 to: line size) trimSeparators.
	category isEmpty ifTrue: [self error: 'category is empty'].
	category first = $' ifTrue: [
		category last = $' ifFalse: [self error: 'category begins with quote but does not end with quote'].
		category := category copyFrom: 2 to: category size - 1.
	].
	(category includes: $') ifTrue: [self error: 'quote not supported in category name'].
%
category: 'processing'
method: GsFileIn
classMethod

	currentClass isNil ifTrue: [self error: 'current class not defined'].
	self compileMethodIn: currentClass , ' class'.
%
category: 'processing'
method: GsFileIn
classMethod: aString

	aString ifNotNil:[ currentClass := aString ].
	self classMethod.
%
category: 'processing'
method: GsFileIn
commit
	
	^System commitTransaction
%
category: 'processing'
method: GsFileIn
commitTransaction

	^session commit
%
category: 'processing'
method: GsFileIn
compileEnvironment: arg

	arg isDigits ifFalse: [self error: 'ENV  only accepts integers'].
  compileEnvironment := arg .
%
category: 'processing'
method: GsFileIn
compileMethodIn: aClassContex
"compile method from stream in context of current class"

	| string |
	string := aClassContex , '
compileMethod: ' , self nextChunk printString , '
dictionaries: System myUserProfile symbolList
category: ' , category printString,
' environmentId: ', compileEnvironment.
	self execute: string.
%
category: 'processing'
method: GsFileIn
currentClass

	^currentClass
%
category: 'processing'
method: GsFileIn
currentClass: aClassName

	currentClass:= aClassName
%
category: 'processing'
method: GsFileIn
doFileIn
  | lineNum |
  lineNum := 0 .
	session isNil ifTrue: [self session: self].
	[
		fileStream atEnd not.
	] whileTrue: [
		line := fileStream nextLine trimSeparators.
    lineNum := lineNum + 1 .
		[ 
      self processLine 
    ] on: Error do:[:ex | | ln |
      ln := ', at line ', lineNum asString .
      ex messageText ifNil:[ ex messageText: ln ]
                  ifNotNil:[:str | str addAll: ln ].
      ex pass
    ]
	].
%
category: 'processing'
method: GsFileIn
doit

	self execute: self nextChunk.
%
category: 'processing'
method: GsFileIn
execute: aString

	| string result |
  session == self ifFalse:[
	  string := '[' , aString , '] on: Error do: [:ex | ^ex description]. nil'.
	  (result := session executeString: string) ifNotNil:[:aDescription | 
      self error: aDescription .
      ^ Error signal: aDescription "do not expect to be here"
    ].
  ] ifTrue:[
    result := aString evaluate
  ].
  ^ result 
%
category: 'processing'
method: GsFileIn
executeString:  string

	string evaluate.
	^nil
%
category: 'accessors'
method: GsFileIn
fileStream: aFileStream
 
	fileStream := aFileStream.
%
category: 'processing'
method: GsFileIn
ignoreList

	^#('EXPECTVALUE' 
		'EXPECTERROR' 
		'ERRORCOUNT' 
		'FILEOUT'
		'FILEFORMAT' 
		'DISPLAY' 
		'LEVEL' 
		'LIMIT' 
		'LIST' 
		'IFERR' 
		'IFERR_LIST'
		'IFERR_CLEAR'
		'OMIT' 
		'OUTPUT' 
		'REMARK' 
		'STATUS' 
		'TIME')
%
category: 'initialize'
method: GsFileIn
initialize

	session := self.
	category := 'as yet unspecified'.
	compileEnvironment := '0' .
%
category: 'processing'
method: GsFileIn
inputNestedFile: aPath
"input nested file"

	| aSession |
	session == self ifFalse: [aSession := session].
	fileStream gsfile isClient 
		ifTrue: [GsFileIn fromClientPath: aPath to: aSession]
		ifFalse: [GsFileIn fromServerPath: aPath to: aSession]
%
category: 'processing'
method: GsFileIn
method

	currentClass isNil ifTrue: [self error: 'current class not defined'].
	self compileMethodIn: currentClass.
%
category: 'processing'
method: GsFileIn
method: aString

	aString ifNotNil:[ currentClass := aString ].
	self method.
%
category: 'processing'
method: GsFileIn
nextChunk

	| stream |
	stream := WriteStream on: String new.
	[
		fileStream atEnd not.
	] whileTrue: [
		((line := fileStream nextLine trimTrailingSeparators) notEmpty and: [line first = $%]) ifTrue: [
			^stream contents.
		].
		stream nextPutAll: line; lf.
	].
	self error: 'ran off end of file!'.
%
category: 'processing'
method: GsFileIn
processLine
	| words command firstChar |
	line isEmpty ifTrue: [^self].
	(#('DOIT' 'PRINTIT' 'RUN' 'NBRUN') includes: line asUppercase) ifTrue: [^self doit].
	words := line subStrings.
	command := words first asUppercase.
	(firstChar := command at:1) == $!  ifTrue: [^nil].
  
	firstChar == $S ifTrue:[
	  command = 'SET' ifTrue: [
		  ((words at: 2) equalsNoCase: 'compile_env:') 
			ifTrue:[
				words size == 3 ifFalse:[
					self error:'wrong number of arguments to SET COMPILE_ENV:'
				].
				^self compileEnvironment: (words at: 3)
			].
		(((words at: 2) equalsNoCase: 'class') 
			or: [(words at: 2) equalsNoCase: 'class:']) ifTrue: [
				words size == 3 ifFalse:[
					self error:'wrong number of arguments to SET CLASS'
				].
				^self currentClass: (words at: 3)
			].
		^nil ].
		command = 'SEND' ifTrue: [^self execute: (words at: 2), ' ', (words at: 3)]
  ].
  firstChar == $C ifTrue:[
	  (command = 'CATEGORY:' or: [command = 'CATEGORY']) ifTrue: [^self category].
	  command = 'CLASSMETHOD' ifTrue: [^self classMethod].
	  command = 'CLASSMETHOD:' ifTrue: [^self classMethod: (words atOrNil: 2)].
	  command = 'COMMIT' ifTrue: [^self commitTransaction].
  ].
  firstChar == $M ifTrue:[
	  command = 'METHOD' ifTrue: [^self method].
	  command = 'METHOD:' ifTrue: [^self method: (words atOrNil: 2)].
  ].
  firstChar == $E ifTrue:[
    command = 'ENV' ifTrue:[
      words size == 2 ifTrue:[ ^ self compileEnvironment: (words at: 2)].
      words size == 1 ifTrue:[ ^ nil "no change to env"].
      self error:'wrong number of arguments to ENV'.
    ].
   ].
	firstChar == $R ifTrue: [
		command = 'REMOVEALLMETHODS' ifTrue: [^self removeAllMethods: (words atOrNil: 2)]. 
		command = 'REMOVEALLCLASSMETHODS' ifTrue: [^self removeAllClassMethods: (words atOrNil: 2)].
	].	
	firstChar == $A ifTrue: [
		command = 'ABORT' ifTrue: [^self abortTransaction]. 
	].
	firstChar == $I ifTrue: [
		command = 'INPUT' ifTrue: [words size == 1 ifTrue: [self error:'wrong number of arguments to input'].
		^self inputNestedFile: (words at: 2)]. 
	].
  ^(self ignoreList includes: command)  
		ifTrue: [nil ] 
		ifFalse: [self error: 'unrecognized command: ' , command printString].
%
category: 'processing'
method: GsFileIn
removeAllClassMethods

	currentClass isNil ifTrue: [self error: 'current class not defined'].
	self execute: currentClass ,' class removeAllMethods'.
%
category: 'processing'
method: GsFileIn
removeAllClassMethods: aClassName
"removes all  class methods for supplied class. Supplied class becomes current class"

	aClassName notNil ifTrue: [currentClass := aClassName].
	currentClass isNil ifTrue: [self error: 'current class not defined'].
	self removeAllClassMethods
%
category: 'processing'
method: GsFileIn
removeAllMethods

	currentClass isNil ifTrue: [self error: 'current class not defined'].
	self execute: currentClass , ' removeAllMethods'.
%
category: 'processing'
method: GsFileIn
removeAllMethods: aClassName
"remove all methods for supplied class. Supplied class becomes current class"

	aClassName notNil ifTrue: [currentClass := aClassName].
	currentClass isNil ifTrue: [self error: 'current class not defined'].
	self removeAllMethods
%
category: 'accessors'
method: GsFileIn
session: anExternalSession

	session := anExternalSession.
%
category: 'accessors'
method: GsFileIn
setCurrentClass: aClassName

	self currentClass: aClassName
%
category: 'accessors'
method: GsFileIn
setSession: anExternalSession

	self session: (anExternalSession ifNil: [self])
%
