!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
!  MessageNotUnderstood    methods are in MessageNotUnderstood.gs
!  ZeroDivide              methods in ZeroDivide.gs
!  Warning is created in image filein, methods in Warning.gs 
!
! See file AstractException.gs for  documentation of the exception hierarchy,
! and for methods defined in AstractException 
!=========================================================================

set class Exception
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
Exception is the top-level ANSI class in the exception framework. It is
an abstract class that includes Error and Notification as its subclasses.
It would generally not be appropriate to signal or catch an Exception.
In GemStone/S, Exception is a subclass of AbstractException so that some
common code can be shared with the Ruby implementation.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_Exception.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class RubyBreakException
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
RubyBreakException is used in Ruby and is not used in Smalltalk.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_RubyBreakException.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class RubyThrowException
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
RubyThrowException is used in Ruby and is not used in Smalltalk.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	value: Object
	signalNum: SmallInteger
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_RubyThrowException.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class ControlInterrupt
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
ControlInterrupt is the abstract superclass for various exceptions that
generally pass control from the server to the GCI client.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_ControlInterrupt.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class Break
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
Break is a ControlInterrupt that is generated by the virtual machine based on
a Gci*Break() from the client. In Topaz, this will be generated by <Ctrl>+<C>

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object'.
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_Break.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class Breakpoint
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
Breakpoint is a ControlInterrupt generated by the virtual machine based on
a debugging event, such as GsNMethod>>#setBreakAtStepPoint: or a 
GsProcess>>#_step* call.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	context: GsProcess
	stepPoint: SmallInteger
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_Breakpoint.
  gsResumable := true .
  gsTrappable := true .
%
method: 
resume
  "Resume execution at the step point where this Breakpoint was signalled
   and attempt to execute the bytecode at that step point."

  ^ self resume: nil 
%
method: 
resume: aValue
  "Resume execution at the step point where this Breakpoint was signalled
   and attempt to execute the bytecode at that step point."

  ^ super resume: aValue
%
method:
return 
  "Return from the on:do: which installed a handler for this Breakpoint.
   Set the current GsProcess' state so that the breakpoint is reenabled."

  GsProcess _current clearLastBreakpointHistory .
  ^ super return
%
method:
return: aValue
  "Return from the on:do: which installed a handler for this Breakpoint.
   Set the current GsProcess' state so that the breakpoint is reenabled."

  GsProcess _current clearLastBreakpointHistory .
  ^ super return: aValue 
%
method:
resignalAs: anException
  "not working yet."
  ^ super resignalAs: anException
%
classmethod: 
trappable: aBoolean
  "Default is that Breakpoints are signaled by the VM
   direct to the GCI, and they are not trappable in Smalltalk,
   and Breakpoint>>initialize is not run.

   To override the default, execute
     Breakpoint trappable: true
   and they will be signalled as normal trappable exceptions.
   Returns a Boolean, previous state from the VM .
  "
  <primitive: 829>
  ^ self _primitiveFailed: #trappable:
%

! =========================
set class ClientForwarderSend
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
ClientForwarderSend is ControlInterrupt generated by the implementation of
#doesNotUnderstand:* in ClientForwarder. Generally this Exception would not
be signaled or caught explicitly in GemStone Smalltalk code but would be
caught in the GCI client (typically GBS) and after being handled appropriately
the GCI client would resume with some return value. This effectively allows
message sends from the server to the client.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	receiver: Object
	clientObj: Symbol
	selector: Symbol
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_ClientForwarderSend.
  gsResumable := true .
  gsTrappable := false . "should always go to the client"
%


method:
receiver: aSelf clientObj: aClientObj selector: aSymbol args: anArray
  receiver := aSelf .
  clientObj := aClientObj .
  selector := aSymbol .
  gsArgs := anArray 
%

category: 'Handling'
method:
defaultAction

"Return an error to the controlling GCI client. Stack is saved
 and available as an argument to the GCI error struct.

 instVars of the receiver go into the GCI error struct as follows
 self.receiver -->  GciErrSType.args[0]
 self.clientObj --> GciErrSType.args[1]
 self.selector -->  GciErrSType.args[2]
 self.gsArgs -->    GciErrSType.args[3]   (instVar defined in AbstractException)
"

<primitive: 33>  "this frame removed from stack before saving stack."
self _uncontinuableError
%

! =========================
set class Halt
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
Halt is a ControlInterrupt signaled in Object>>#halt, #halt:, and #pause.
It is typically not caught on the server and is sent to the GCI client
to be handled by a debugger (typically GBS or Topaz).

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_Halt.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class Error
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
Error is an Exception that is the superclass for a variety of errors.
It is typically used as the selector for the Exec*Block>>#on:do: method.
While an Error can be signaled directly, it is more common to see 
UserDefinedError signaled from the Object>>#error: method.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_Error.
  gsResumable := false .
  gsTrappable := true .
%

category: 'Signaling'
classmethod: 
signalFatalSigTerm

  "Terminates the session with a Fatal error 4052.  Allows
   Smalltalk or Ruby handler to implement SIGTERM semantics." 

  ^ self new _number: GS_ERR_SIGTERM ; signalNotTrappable
%

! Error >> asString in AbstractException2.gs 

! =========================
set class CompileError
removeallmethods
removeallclassmethods

! additional methods in AbstractException2.gs

category: 'Documentation'
classMethod:
comment
^ '
CompileError is an Error generated by the virtual machine when a
syntax error is found during a method compile. The method #errorDetails
will return an Array of Array instances, one for each compile error.
Each compile error has the following details:
	(1) an Integer error number; 
	(2) an Integer offset into the source string; and
	(3) a String describing the error.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_CompileError.
  gsResumable := true .
  gsTrappable := true .
%

category: 'Formatting'
method:
asString
  "Final implementation installed in AbstractException2.gs"
  ^ super asString .
%

method:
errorDetails
  "Final implementation in AbstractException2.gs .
   Return nil or the array of compiler error details"
| arr |
arr := gsArgs ifNotNil:[:gsa | gsa atOrNil: 1 ] .
^ arr ifNil:[ { { 1075 . 1 . 'malformed CompilerError' } } ]. "STDB_MALFORMED_COMPILER_ERROR"
%

method
sourceString
^ gsArgs ifNotNil:[:gsa | gsa atOrNil: 2 ]
%

method
_firstCompilerErrorNumber

^ (self errorDetails at: 1) at: 1
%

! =========================
set class EndOfStream
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
The EndOfStream Error is generated when an attempt is made to read
beyond the end of a stream. For example:
	(ReadStream on: String new) next.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_EndOfStream.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class ExternalError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
ExternalError is an abstract superclass for errors that happen during
a call to some non-Smalltalk code, such as the file system or a socket.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_ExternalError.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class IOError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
IOError is an ExternalError signaled by the virtual machine when certain
operating system calls fail. Typically, it represents a problem with some
file operation.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_IOError.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class SocketError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
Most GsSocket methods that fail will return nil (or false; see #connectTo:)
and the caller is then expected to call one of the #lastError* methods.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Formatting'
method:
asString
^ self _description:(
  [ | str |
    "get the details from error 2362 thrown from a primitive"
    str := 'socket error ' copy .
    1 to: gsArgs size  do:[:n | | elem |
      elem := gsArgs atOrNil: n .
      (elem _isSymbol or:[ elem _isOneByteString]) ifTrue:[ str add: elem ]
         ifFalse:[  elem _isSmallInteger ifTrue:[ str add: elem asString ]
            ifFalse:[ str add: 'a ' ; add: elem class name ]].
      n < gsArgs size ifTrue:[ str add: ', ' ]
    ].
    gsDetails ifNotNil:[:d| str add:', '; add: d  asString ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_SocketError.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class SecureSocketError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
An exception signaled when certain GsSecureSocket methods fail.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_SecureSocketError.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class SystemCallError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
SystemCallError is signaled when an external system call fails. 
At present, it is primarily used by Ruby.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	errno: SmallInteger
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_SystemCallError.
  gsResumable := true .
  gsTrappable := true .
%
method:
_legacyHandlerArgs
  | arr |
  (arr := { errno } ) addAll: super _legacyHandlerArgs .
  ^ arr
%
method:
args: anArray
  anArray _isArray ifTrue:[ | sz |
    errno := anArray atOrNil: 1 .
    (sz := anArray size) > 1 ifTrue:[
      gsArgs := anArray copyFrom:2 to: sz
    ].
  ]
%
method:
errno: anInt
  errno := anInt
% 

category: 'Formatting'
method:
asString
^ self _description:(
  [ | str |
    errno ifNotNil:[:e| (str := String withAll:'errno=') add: errno asString].
    gsDetails ifNotNil:[:d| str add:', '; add: d  asString ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

! =========================
set class ImproperOperation
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
ImproperOperation is a superclass for a variety of Smalltalk coding
errors, such as passing an invalid or out of range argument to a method.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	object: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_ImproperOperation.
  gsResumable := true .
  gsTrappable := true .
%

method:
object: anObject
  object := anObject
%
method:
_legacyHandlerArgs
  | arr |
  (arr := { object } ) addAll: super _legacyHandlerArgs .
  ^ arr
%
method:
args: anArray
  anArray _isArray ifTrue:[ | sz |
    object := anArray atOrNil: 1 .
    (sz := anArray size) > 1 ifTrue:[
      gsArgs := anArray copyFrom:2 to: sz
    ].
  ]
%

! =========================
set class ArgumentError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
ArgumentError identifies a situation in which an argument to a method is invalid.
The type of the argument may be valid (else it would be an ArgumentTypeError)
but there is something else incorrect about an argument.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	object: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_ArgumentError.
  gsResumable := true .
  gsTrappable := true .
%

method:
name: aName
  object := aName
%

category: 'Formatting'
method:
asString
^ self _description:(
  [ | str | 
    gsDetails ifNotNil:[:det | 
      (str := String new ) add: det asString . 
      object ifNotNil:[:o| 
        str add:', '; add: o  asString 
      ] ifNil:[
        gsArgs ifNotNil:[:ga | (ga atOrNil: 1) ifNotNil:[:d | str add: d asString]]
      ]
    ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

! =========================
set class ArgumentTypeError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
ArgumentTypeError identifies a situation in which an argument to a method 
is the wrong type (or class). For example, an Interval must be initialize
to have instances of Number as the start, stop, and step.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	object: Object
	expectedClass: Behavior
	actualArg: Behavior
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_ArgumentTypeError.
  gsResumable := true .
  gsTrappable := true .
%

method:
name: aName expectedClass: aClass actualArg: anObject
  object := aName .
  expectedClass := aClass "may be an array of Classes" .
  actualArg := anObject
%
method:
constrainedIv: aName expectedClass: aClass actualArg: anObject
  gsNumber := 2107 .  "legacy error info"
  gsReason := #objErrConstraintViolation . 
  object := aName .
  expectedClass := aClass "may be an array of Classes" .
  actualArg := anObject
%


method:
_legacyHandlerArgs
  | arr |
  arr := { object . expectedClass . actualArg } .
  gsArgs ifNotNil:[:a |  arr addAll: a ].
  ^ arr
%
method:
args: anArray
  anArray _isArray ifTrue:[ | sz |
    object := anArray atOrNil: 1 .
    expectedClass := anArray atOrNil: 2 .
    actualArg := anArray atOrNil: 3  .
    (sz := anArray size) > 3 ifTrue:[
      gsArgs := anArray copyFrom: 4 to: sz
    ].
  ]
%
category: 'Formatting'
method:
asString
^ self _description:(
  [ | str |
    gsDetails ifNotNil:[ :d |  "example use: attempt to modify invariant object"
      str := d asString copy .
      object ifNotNil:[:a | str add: ', ' ; add: a asString ].
    ] ifNil:[ | exp act |
      exp := expectedClass . act := actualArg . 
      (exp ~~ nil or:[ act ~~ nil]) ifTrue:[
        str := String new .
        object ifNotNil:[:o | str add: 'for '; add: o asString ; add:'  '].
        exp ifNotNil:[ str add: 'expected a ' ; add: exp name ].
        act ifNotNil:[ str add:', arg is a '; add: act class name ].
      ].
    ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

! =========================
set class CannotReturn
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
CannotReturn is an ImproperOperation generated by the virtual machine
when a return is attempted from a block that is no longer on the stack.
If a method returns a block, then that block should not have an explicit
return in it.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	object: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_CannotReturn.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class LookupError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
LookupError is an ImproperOperation that is signaled when a lookup is
made and not found. The key instance variable holds the missing key.
For example, evaluate the following code:
	[Dictionary new at: 42] on: LookupError do: [:ex | ex halt].

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	object: Object
	key: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_LookupError.
  gsResumable := true .
  gsTrappable := true .
%
method:
_legacyHandlerArgs
  | arr |
  arr := { object . key } .
  gsArgs ifNotNil:[:a |  arr addAll: a ].
  ^ arr
%
method:
args: anArray
  anArray _isArray ifTrue:[ | sz |
    object := anArray atOrNil: 1 .
    key := anArray atOrNil: 2 .
    (sz := anArray size) > 2 ifTrue:[
      gsArgs := anArray copyFrom: 3 to: sz
    ].
  ]
%

method
key: anObject
  key := anObject
%
category: 'Formatting'
method:
asString
^ self _description:(
  [ | str |
    gsDetails ifNotNil:[ :gsd |
      str := String new .
      object ifNotNil:[:o | str add: ' accessing a '; add: o class name ].
      key ifNotNil:[:k| 
        (str := String withAll:', ') add: k asString
      ].
      str add: ', ' ; add: gsd  asString .
    ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

! =========================
set class OffsetError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
OffsetError is an ImproperOperation signaled when an invalid offset 
is used to attempt a lookup. For example:
	[Array new at: 2] on: OffsetError do: [:ex | ex halt].

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	object: Object
	maximum: SmallInteger
	actual: SmallInteger
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_OffsetError.
  gsResumable := true .
  gsTrappable := true .
%

method:
offset: anOffset maximum: aMax
  actual := anOffset .
  maximum := aMax
% 
method:
_legacyHandlerArgs
  | arr |
  arr := { object . maximum . actual } .
  gsArgs ifNotNil:[:a |  arr addAll: a ].
  ^ arr
%
method:
args: anArray
  anArray _isArray ifTrue:[ | sz o |
    object := (o := anArray atOrNil: 1) .
    (actual := anArray atOrNil: 2 ) ifNotNil:[ 
      maximum := (anArray atOrNil: 3) ifNil:[ o ifNotNil:[ o size ]].
      (sz := anArray size) > 3 ifTrue:[
        gsArgs := anArray copyFrom: 4 to: sz
      ].
    ].
  ]
%

category: 'Formatting'
method:
asString
^ self _description:(
  [ | str |
    str := String new .
    object ifNotNil:[:o | str add: 'accessing a '; add: o class name ].
    maximum ifNotNil:[:m |
      str := String new .
      str add:'max:'; add: m asString .
      actual ifNotNil:[:a| str add: ' actual:'; add: a asString].
    ].
    gsDetails ifNotNil:[:d| str add:', '; add: d  asString ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

! =========================
set class OutOfRange
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
OutOfRange is signaled for various conditions when an invalid argument
is passed to a method. For example:
	[ScaledDecimal with: 42 scale: -2] on: OutOfRange do: [:ex | ex halt].

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	object: Object
	minimum: Object
	maximum: Object
	actual: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_OutOfRange.
  gsResumable := true .
  gsTrappable := true .
%
method:
name: aName min: aMin actual: aValue
  gsArgs := aName .
  minimum := aMin .
  actual := aValue
%
method:
name: aName max: aMax actual: aValue
  gsArgs := aName .
  maximum := aMax .
  actual := aValue
%
method:
name: aName min: aMin max: aMax actual: aValue
  gsArgs := aName .
  minimum := aMin .
  maximum := aMax .
  actual := aValue
%
method:
expectedNumArgs: aNumber actual: aValue
  minimum := aNumber .
  maximum := aNumber .
  actual := aValue
%
method:
_legacyHandlerArgs
  | arr |
  arr := { object . minimum . maximum . actual } .
  gsArgs ifNotNil:[:a |  arr add: a ].
  ^ arr
%
method:
args: anArray
  anArray _isArray ifTrue:[ | sz |
    object := anArray atOrNil: 1 .
    minimum := anArray atOrNil: 2 .
    maximum := anArray atOrNil: 3 .
    actual := anArray atOrNil: 4 .
    (sz := anArray size) > 4 ifTrue:[
      gsArgs := anArray at: 5 .
    ].
  ]
%

! fixed 45798
category: 'Formatting'
method:
asString
^ self _description:(
  [
    | str |
    str := String new .
    object ifNotNil:[:o | 
       o _isOneByteString ifTrue:[ str add: 'for '; add: o quoted]
         ifFalse:[ str add: 'for a '; add: o class name].
    ].
    gsArgs "argName" ifNotNil:[:a| 
          str size > 0 ifTrue:[ str add:', ' ].
          str add: ' the value for '; add: a asString ; 
		      add: ' is ' ; add: actual asString ] 
       ifNil:[ actual ifNotNil: [:a | str add: a asString ] ].
    minimum ifNotNil: [
	   maximum ifNotNil:[ str add: ' should be between '; 
                   add: minimum asString; add: ' and '; add: maximum asString ]
		      ifNil: [str add: ' minimum is '; add: minimum asString]
    ] ifNil: [
	     maximum ifNotNil: [str add: ' maximum is '; add: maximum asString]
    ].
    gsDetails ifNotNil: [:arg | str add: ', '; add: arg asString] .
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

! =========================
set class FloatingPointError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
FloatingPointError is signaled for various error conditions during
numerical operations.  By default floating point operations are silent
and return a NaN or Infinity in some cases.   

After
  FloatingPointError enableExceptions: #divideByZero
a floating point divide by zero will signal an instance of FloatingPointError.
Note that a divide by integer zero will always signal an instance of ZeroDivide.

FloatingPointError enableAllExceptions  
will cause a FloatingPointError to be signaled whenever a floating point 
arithmetic operation would produce a NaN an Infinity as the result. 

See documentation in FloatingPointError class >> enabledExceptions
for the list of floating point exception conditions which can
signal a FloatingPointError .

Some operations that do not involve floating point numbers can 
signal a FloatingPointError, such as LargeInteger overflow.
For example, evaluate the following expression:
	[10 raisedTo: 100000] on: FloatingPointError do: [:ex | ex halt].
Since there is no appropriate exceptional number to return, these exceptions 
are not managed by exception enabling and are always signaled.
'
%
category: 'Private'
classmethod:
_initializeConstants
   "VM changes needed if you change any of these bit definitions.
    Also fix sends of _enableExceptions: in image."
   self _addInvariantClassVar: #divideByZero value: 1 ;
         _addInvariantClassVar: #overflow value: 2  ;
         _addInvariantClassVar: #underflow value: 4  ;
         _addInvariantClassVar: #invalidOperation  value: 8  ;
         _addInvariantClassVar: #inexactResult  value: 16r10 .
%
run
FloatingPointError _initializeConstants .
true
%

category: 'Private
classmethod:
_enabledBitsAsArray: bits
  "The result is in sorted order."
  | res |
  res := { } .
  bits ifNotNil:[ 
    (bits bitAnd: 1) ~~ 0 ifTrue:[ res add: #divideByZero ].
    (bits bitAnd: 16r10) ~~ 0 ifTrue:[ res add: #inexactResult ].
    (bits bitAnd: 8) ~~ 0 ifTrue:[ res add: #invalidOperation ].
    (bits bitAnd: 2) ~~ 0 ifTrue:[ res add: #overflow ].
    (bits bitAnd: 4) ~~ 0 ifTrue:[ res add: #underflow ].
  ].
  ^ res
%

category: 'Formatting'
method:
asString
  | str arr cls |
  str := 'a ' copy .
  str add: (cls := self class) name .
  arr := cls _enabledBitsAsArray: actual .
  arr size == 0 ifTrue:[
    gsNumber == ERR_LargeIntegerOverflow 
      ifTrue:[ str add:' overflow during LargeInteger arithmetic'].
  ] ifFalse:[
    1 to: arr size do:[:n | str add: $  ; add: (arr at: n) ].
  ].
  ^ str .
%
category: 'Accessing'
method:
exceptionList 
 "Return an Array of Symbols , the list of exceptions 
  which produced this instance.  The result will contain one or more of
     #divideByZero #overflow #underflow #invalidOperation #inexactResult  .
  and is sorted alphabetically.
 "
  ^ self class _enabledBitsAsArray: actual
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_FloatingPointError.
  gsResumable := true .
  gsTrappable := true .
  actual := 0 .
%

category: 'Private'
classmethod:
_getExceptions: aBoolean

"Fetch and clear bits representing floating point exceptions that have occurred
 since last execution of this primitive.  Result is a SmallInteger.

 aBoolean == false  means get bits enabled by the last call to 
 FloatingPointError(C)>>enableExceptions: , and clear all bits.
 aBoolean == true means get and clear all bits .

 The bits in the result are in the range 16r1F and 
 are defined by the constants in FloatingPointError(C)>>_initializeConstants ."

<primitive: 129>

aBoolean _validateClass: Boolean .
self _primitiveFailed: #_getExceptions: args: { aBoolean }
%

category: 'Private'
classmethod:
_enableExceptions: aSmallInteger

"aSmallInteger may have one or more bits in the range 16r1F,
 per the constants in FloatingPointError(C)>>_initializeConstants .
 other bits are ignored.  The specified exceptions will be checked for
 after each arithmetic primitive in Float and SmallDouble and if 
 any of those execptions occurs, a FloatingPointError will be signaled.
 Returns the value which was passed in the previous call to this primitive.

 If aSmallInteger == -1 , then returns the positive value 
 which was passed in the previous call to this primitive without changing
 the state of the enabled exceptions.
"

<primitive: 122>

aSmallInteger _validateClass: SmallInteger .
self _primitiveFailed: #enableExceptions: args: { aSmallInteger }
%

category: 'Private'
classmethod:
_symbolToBit: aSymbol

  | bit |
  bit := classVars at: aSymbol otherwise: nil .
  bit ifNil:[ Error signal:'invalid name of a floating point exception bit, ' , aSymbol].
  ^ bit
%

category: 'Accessing'
classmethod:
enableExceptions: argument

"Argument may be one of the symbols 
   #divideByZero #overflow #underflow #invalidOperation #inexactResult
 or an Array containing zero or more of those Symbols ,
 or nil which means enable none of the exceptions.

 The specified exceptions will be checked for
 after each arithmetic primitive in Float and SmallDouble and if
 any of those execptions occurs, a FloatingPointError will be signaled.
 Overrides the settings of a previous call to enableExceptions: . 
 Returns a SmallInteger, the previously enabled exception bits."

 | bits |
 argument ifNil:[ ^ self _enableExceptions: 0 ].
 argument _isSymbol ifTrue:[
   ^ self _enableExceptions: (self _symbolToBit: argument)
 ].
 argument _isArray ifTrue:[
   bits := 0 .
   argument do:[:aSym | 
     bits := bits bitOr:( self _symbolToBit: aSym)
   ].
   ^ self _enableExceptions: bits
 ].
 argument _validateClasses: { Array . Symbol }.
% 

category: 'Private'
classmethod:
_exceptionList

^  #( #divideByZero #overflow #underflow #invalidOperation #inexactResult)
%

classmethod:
enableAllExceptions

^ self enableExceptions: self _exceptionList
%

classmethod:
enabledExceptions
  "Returns an Array containing zero or more of the Symbols
    #divideByZero #overflow #underflow #invalidOperation #inexactResult
  reflecting the most recent call to either of
     FloatingPointError class>>enableAllExceptions 
     FloatingPointError class>>enableExceptions:
  "
  ^ self _enabledBitsAsArray: (self _enableExceptions: -1)  .
%

method: 
actual: aValue
  actual := aValue
%

category: 'Private'
classmethod:
_checkFpStatus
  "If any of the currently enabled exceptions have occurred,  
   clear those bits from the VM state and signal an error.
   Currently enabled exceptions are returned by 
     FloatingPointError(C) >> enabledExceptions .
   This method is called from primitive failure paths in in each floating 
   point primitive."

| flags |
flags := self _getExceptions: false .
flags ~~ 0 ifTrue:[
  self new actual: flags ; signal
].
%


! =========================
set class IndexingErrorPreventingCommit
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
IndexingErrorPreventingCommit is signaled when an error occurs during 
an indexing operation and the internal indexing structures are in an
inconsistent state. The virtual machine will prevent a commit so as to
preserve the correctness of the indexing structures. 

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_IndexingErrorPreventingCommit.
  gsResumable := true .
  gsTrappable := true .
%

category: 'Private'
method:
_signalWith: inCextensionArg

  self _preventCommits .
  ^ super _signalWith: inCextensionArg
%
method:
_preventCommits
  | details |
  (details := gsDetails) ifNil:[ 
    details := 'anIndexingErrorPreventingCommit'.
    gsDetails := details.
  ].
  System disableCommitsUntilAbortWithReason: details . 
%

classmethod: IndexingErrorPreventingCommit
resignal: anException objWithIndex: anObject
  " check for recursive signal "

  (anException isKindOf: IndexingErrorPreventingCommit)
    ifTrue: [ 
      | mgr |
      (mgr := IndexManager) current autoCommit
        ifTrue: [ 
          anObject
            ifNotNil: [ anObject removeIncompleteIndex ]
            ifNil: [ mgr removeAllIncompleteIndexes ] ].
      anException pass ]
    ifFalse: [ 
      | epcEx |
      (epcEx := IndexingErrorPreventingCommit new) args: {anException}.
      epcEx messageText: anException description.
      epcEx _preventCommits.
      anException resignalAs: epcEx ]
%

! =========================
set class InternalError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
An InternalError indicates that an error was detected in the core
GemStone/S code. Typically this does not represent an application 
error and the appropriate action is probably to file a help request
with support.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_InternalError.
  gsResumable := false .
  gsTrappable := true .
%

! =========================
set class GciTransportError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
GciTransportError is an InternalError generated by the virtual machine
when it receives invalid data from the GCI client. The most likely cause
of this error is some sort of network problem between the client and the
server. Otherwise, it suggests that there is a problem with GemStone/S
and opening a help request with support may be appropriate.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_GciTransportError.
  gsResumable := false .
  gsTrappable := true .
%

! =========================
set class LockError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
A LockError is signaled when an attempt to get a read lock or a write lock
on an object fails. Calls to System class>>#readLock: or #writeLock: should
be coded to handle this Error.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	object: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_LockError.
  gsResumable := true .
  gsTrappable := true .
%

method: 
object: anObj 

  object := anObj
%


method:
_legacyHandlerArgs
  | arr |
  (arr := { object } ) addAll: super _legacyHandlerArgs .
  ^ arr
%
method:
args: anArray
  anArray _isArray ifTrue:[ | sz |
    object := anArray atOrNil: 1 .
    (sz := anArray size) > 1 ifTrue:[
      gsArgs := anArray copyFrom:2 to: sz
    ].
  ]
%
category: 'Accessing'
method:
object
  ^ object
%
category: 'Formatting'
method:
asString
^ self _description:(
  [
    gsDetails ifNotNil:[:s| s asString ]
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%


! =========================
set class NameError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
NameError is the superclass for MessageNotUnderstood. It is signaled in Ruby
when code attempts to reference an instance variable by name but the name
does not exist. NameError is not used in Smalltalk.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	selector: String
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_NameError.
  gsResumable := true .
  gsTrappable := true .
%

method:
_legacyHandlerArgs
  | arr |
  (arr := { selector } ) addAll: super _legacyHandlerArgs .
  ^ arr
%
method:
args: anArray
  anArray _isArray ifTrue:[ | sz |
    selector := anArray atOrNil: 1 .
    (sz := anArray size) > 1 ifTrue:[
      gsArgs := anArray copyFrom:2 to: sz
    ].
  ]
%
category: 'Formatting'
method:
asString
^ self _description:(
  [ | str |
    str := String new .
    selector ifNotNil:[:s| str add: s asString ] .
    gsDetails ifNotNil:[:d| str add:', '; add: d  asString ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

! =========================
set class MessageNotUnderstood
removeallmethods
removeallclassmethods

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_MessageNotUnderstood.
  gsResumable := true .
  gsTrappable := true .
  envId := 0 .
%

! more methods in MessageNotUnderstood.gs 

! =========================
set class NumericError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
NumericError is an abstract superclass for ZeroDivide. 
It is not used in Smalltalk.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_NumericError.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class ZeroDivide
removeallmethods
removeallclassmethods

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_ZeroDivide.
  gsResumable := true .
  gsTrappable := true .
%
! more methods in ZeroDivide.gs

! =========================
set class RepositoryError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
RepositoryError is signaled by the virtual machine for various
repository-related problems. Some of these are programming errors
(such as an attempt to create more than the allowed number of
extents) and some are I/O errors (such as disk full or disk error).

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_RepositoryError.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class SecurityError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
SecurityError is signaled by the system for various errors where an
action is attempted without proper security. For example:
	[System myUserProfile oldPassword: ''foo'' newPassword: ''bar''] 
		on: SecurityError 
		do: [:ex | ex halt].

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_SecurityError.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class SignalBufferFull
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
SignalBufferFull is sent by the virtual machine when a sessions signal 
buffer is full. For example:
	1 to: 1000 do: [:i | 
		System sendSignal: i to: System session withMessage: String new.
	].

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_SignalBufferFull.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class ThreadError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
ThreadError is signaled when certain scheduling errors occur, including
not having a process to run. For example:
	[Semaphore new wait] on: ThreadError do: [:ex | ex halt].

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_ThreadError.
  gsResumable := true .
  gsTrappable := true .
%

method:
object: anObject
  gsArgs := { anObject }
%

! =========================
set class TransactionError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
TransactionError is signaled in a variety of situations where the session
is in an invalid transaction state for the operation attempted. For example,
Repository>>#listInstances: will fail if there are modification to persistent
objects in the current view. To see a TransactionError, evaluate the following:
	UserGlobals at: #foo put: 42.
	[SystemRepository listInstances: (Array with: Array)]
		on: TransactionError
		do: [:ex | ex halt].

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%
category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_TransactionError.
  gsResumable := true .
  gsTrappable := true .
%

category: 'Formatting'
method: 
asString
  gsNumber == 2407 ifTrue:[
    gsArgs ifNotNil:[ | obj txt str |
      obj := gsArgs atOrNil: 1 .
      txt := (gsArgs atOrNil: 2) ifNil:[ ' cannot be committed'].
      (str := 'a' copy) add: (obj class name); 
          add: ' (oop ' ; add: obj asOop asString; add: ') ' ;
          add: txt .
      ^ str
    ].
  ].
  ^ super asString "default computed in superclass"
%

! =========================
set class UncontinuableError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
UncontinuableError is signaled when an attempt is made to continue
execution after a handler from which continuing execution is not allowed.
For example, it is generally fine to return from a dynamic (stack-based) 
handler:
	[2 / 0] on: ZeroDivide do: [:ex | ex return: 3].
On the other hand, it is not permitted to return from a default (static)
handler:
	ZeroDivide addDefaultHandler: [:ex | ex return: 3].
	[2 / 0] on: UncontinuableError do: [:ex | ex halt].

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_UncontinuableError.
  gsResumable := false .
  gsTrappable := true .
%

! =========================
set class UserDefinedError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
UserDefinedError is signaled by Object>>#error: and related methods.
It would be better for applications to create an explicit subclass 
of Error that describes the actual error. 

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_UserDefinedError.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class Notification
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
According to ANSI Smalltalk, Notification is "used to represent exceptional
conditions that may occur but which are not considered errors. Actual notification
exceptions used by an application may be subclasses of this class."

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

classmethod:
enableSignalling
  "enable signalling of Notifications to the GCI client."
  System __sessionStateAt: 22 put: true .
%

classmethod:
disableSignalling
  "revert to the default behavior of Notification which is to ignore them."
  System __sessionStateAt: 22 put:  nil
%
classmethod:
signallingEnabled

  "returns true if Notifications will be signalled to the GCI client,
   false otherwise. "
  ^ (System __sessionStateAt: 22) ~~ nil 
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_Notification.
  gsResumable := true .
  gsTrappable := true .
%
category: 'Handling'
method:
defaultAction
 
  (System __sessionStateAt: 22) ifNotNil:[
    self _signalGciError "deliver error to GCI, does not return"
  ].
  ^ nil "take no action,  the value 'nil' is returned as the value
      of the message that signaled the exception."
%
category: 'Private'
method:
_errNumPrefix

^ '(notification '
%

! =========================
set class Admonition
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
An Admonition is a type of Notification that it would be best not to ignore. 
While not strictly an Error, it generally indicate that an error is imminent.
For example, AlmostOutOfMemory is an Admonition.

If one has established a generic Error handler, it might be appropriate to
add Admonition to it since the result of an ignored Admonition might make it
impossible to handle the resulting Error. For example:

	[self doSomething] on: Error, Admonition do: [:ex | self recordError: ex].

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_Admonition.
  gsResumable := true .
  gsTrappable := true .
%

! Admonition inherits defaultAction from Notification


! =========================
set class AlmostOutOfStack
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
AlmostOutOfStack is an Admonition (Notification) that the process is about 
to run out of stack space. The typical handler will log a copy of the stack
(see GsProcess class>>#stackReportToLevel:) and then unwind the stack.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_AlmostOutOfStack.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class AlmostOutOfMemory
removeallmethods
removeallclassmethods

! comments edited for 45155
category: 'Documentation'
classmethod:
comment
^ '
AlmostOutOfMemory is an Admonition (Notification) that the virtual machine''s
temporary object space is nearly exhausted. Because this can happen in any
process, it may be useful to set up a default handler for this situation.

It is enabled by default at login, with threshold 90% .
However you must install a handler for AlmostOutOfMemory to take action
or execute
    Notification enableSignalling
to have AlmostOutOfMemory and other Notifications signalled to the
application to avoid having AlmostOutOfMemory silently ignored.

After it is signalled, it  must be explicitly reenabled before another
notification will be delivered.  Reenabling should usually be done after
taking action to allow in-memory garbage collection to free some objects,
or after raising the threshold.
See AbstractException class>>#addDefaultHandler:.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%
category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_AlmostOutOfMemory.
  gsResumable := true .
  gsTrappable := true .
%

! comments edited for 42999
category: 'Control'
classmethod:
enable
  "Enable signaling of AlmostOutOfMemory with the current threshold.
  If no in-memory garbage collection has run since the last delivery
  of an AlmostOutOfMemory exception, then the actual enable will
  be deferred until after in-memory garbage collection .

  You must install a handler for AlmostOutOfMemory to take action
  or execute
    Notification enableSignalling
  to have AlmostOutOfMemory and other Notifications signalled to the
  application to avoid having AlmostOutOfMemory silently ignored."

  System _updateSignalErrorStatus: 5 toState: 0 .
  ^ self
%
classmethod:
disable
  "Disables signalling of AlmostOutOfMemory and resets threshold to 90%"
  System _updateSignalErrorStatus: 5 toState: -1 .
  ^ self
%
classmethod:
enabled
  "Returns true if signaling of AlmostOutOfMemory is enabled.

   See also AlmostOutOfMemory >> enable .
   If the effect of the previous enable was deferred, will return false."
  ^ (System _signalErrorStatus: 5) > 0
%

classmethod:
threshold
  "Returns the current threshold without changing enabled status
   of the AlmostOutOfMemory notification."
  ^ System _updateSignalErrorStatus: 7 toState: -1 .
%

classmethod:
_validateThreshold: anInteger

  "anInteger represents a percentage and must be between 1 and 125."

  anInteger _isSmallInteger ifFalse:[ anInteger _validateClass: SmallInteger ].
  (anInteger < 1 or:[ anInteger > 125]) ifTrue:[
    OutOfRange new name: 'threshold' min: 1 max: 125 actual: anInteger ;
	signal 
  ].
%

classmethod:
threshold: anInteger
  "anInteger represents a percentage and must be between 1 and 125.  
  Sets the threshold for signalling AlmostOutOfMemory notification
  without changing the enabled status .
  Returns the receiver."

  self _validateThreshold: anInteger .
  System _updateSignalErrorStatus: 7 toState: anInteger .
  ^ self
%

classmethod
enableAtThreshold: anInteger

  "anInteger represents a percentage and must be between 1 and 125.
  Sets the threshold for signalling AlmostOutOfMemory notification
  and enables the notification.
  Returns the receiver."

  self _validateThreshold: anInteger .
  System _updateSignalErrorStatus: 5 toState: anInteger .
  ^ self
%

! =========================
set class RepositoryViewLost
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
RepositoryViewLost is an Admonition signaled by the virtual machine if
the session is not in an transaction and fails to respond to a
TransactionBacklog notification within a specified time. The appropriate
action to take is to abort or begin a transaction to get a new view.

TransactionBacklog represents a ''sigAbort'' which was sent to the session
by stone.  

RepositoryViewLost represents a ''lostOt'' sent to the session by stone.
When the ''lostOt'' is received, the VM executes the equivalent of
System abortTransaction, and then signals the RepositoryViewLost .

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_RepositoryViewLost.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class Deprecated
removeallmethods
removeallclassmethods

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_Deprecated.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class FloatingPointException
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
FloatingPointException might be used some time in the future
with hardware floating point operations.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_FloatingPointException.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class InterSessionSignal
removeallmethods
removeallclassmethods

category: 'Documentation'
classmethod:
comment
^ '
InterSessionSignal is a Notification generated by the virtual machine
when another VM sends a signal to the session.
For example
    InterSessionSignal addDefaultHandler: [:ex | ex halt] ; 
              enableSignalling ;
              sendSignal: 0 to: System session withMessage: ''hello!''.
    (Delay forSeconds: 1) wait.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	sendingSession: SmallInteger
	signal: SmallInteger
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_InterSessionSignal.
  gsResumable := true .
  gsTrappable := true .
%

category: 'Handling'
method:
defaultAction

"Return an error to the controlling GCI client .
 See AbstractException>>defaultAction for documentation."
<primitive: 33>
self _uncontinuableError
%

category: 'Instance initialization'
method:
fromSession: sessionIdInt signal: signalInt message: msgString
  gsNumber := ERR_InterSessionSignal .
  sendingSession := sessionIdInt .
  signal := signalInt .
  gsArgs := { msgString } .
%
method:
fromSession: sessionIdInt signal: signalInt message: msgString
  gsNumber := ERR_InterSessionSignal .
  sendingSession := sessionIdInt .
  signal := signalInt .
  gsArgs := { msgString . 0} .
%
method:
fromSession: sessionIdInt signal: signalInt message: msgString numPending: anInt
  gsNumber := ERR_InterSessionSignal .
  sendingSession := sessionIdInt .
  signal := signalInt .
  gsArgs := { msgString . anInt } .
%

! methods added for 41580 , 41579
category: 'Signalling'
classmethod:
poll
"Returns a new instance of InterSessionSignal if a signal is
 available from another session, or nil ."

| sigArray ex |
sigArray := System _signalFromGemStoneSession .
sigArray ifNil:[ ^ nil ].
(ex := self new) fromSession:  (sigArray at: 1)
              signal: (sigArray at: 2) message: (sigArray at: 3) 
              numPending: ( sigArray at: 4).
^ ex
%

classmethod:
enableSignalling

"Enable instances of InterSessionSignal to be signalled by the VM .
 must be resent by exception handling code after handling each
 instance of InterSessionSignal that was signalled by the VM."

 System _updateSignalErrorStatus: 3 toState: true
%

classmethod:
disableSignalling

"Prevent instances of InterSessionSignal from being signalled by the VM"

 System _updateSignalErrorStatus: 3 toState: false
%

classmethod:
signallingEnabled

"Return true if signaling enabled, return false otherwise  .
 See InterSessionSignal(C)>>enableSignalling "

^ System _signalErrorStatus: 3
%

classmethod:
pollAndSignal
  "Signal a new instance of InterSessionSignal if a signal is
    available from another session, otherwise do nothing."

  self poll ifNotNil:[ :ex | ex signal ].
%

classmethod: 
sendSignal: aSmallInt to: aSessionId withMessage: aString

"Send a signal to another VM which can result in an InterSessionSignal
 being signalled within that VM."

^ System sendSignal: aSmallInt to: aSessionId withMessage: aString
%

method: 
replyToSenderWithSignal: aSmallInteger withString: aString

"Reply to the VM which caused this instance "

^ System sendSignal: aSmallInteger to: self sendingSessionId withMessage: aString
%

category: 'Deprecated'
method:
sendingSession
  self deprecated: 'Exception>>sendingSession deprecatedi v3.0, use #sendingSessionId or #sendingSessionSerial'.
  ^ sendingSession
%

category: 'Accessing'
method:
sendingSessionId
  ^ GsSession sessionIdOfSerial: sendingSession
%

method:
sendingSessionSerial
  ^ sendingSession
%

method:
sentInt
  ^ signal
%

method:
sentMessage
  ^ gsArgs ifNotNil:[:a | a atOrNil: 1 ].
%
! fix 47973
method:
numPending
  "Returns a SmallInteger the number pending in stone beyond this one
   at the point this one was obtained from stone."
  | res |
  res := gsArgs ifNotNil:[:a | a atOrNil: 2 ].
  res ifNil:[ res := 0 ].
  ^ res
%

category: 'Formatting'
method:
asString
^ self _description:(
  [ | str |
    (str := String withAll:'fromSession=') add: sendingSession asString;
      add: ' signal:'; add: signal asString ;
      add: ' message: ' ; add: (gsArgs atOrNil: 1) asString .
    gsDetails ifNotNil:[:d| str add:', '; add: d  asString ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

method:
_legacyHandlerArgs
  | arr |
  (arr := { sendingSession . signal } ) addAll: super _legacyHandlerArgs .
  ^ arr
%

! =========================
set class ObjectsCommittedNotification
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
ObjectsCommittedNotification is signaled by the virtual machine when an
object in your notify set has been modified. For example:
	| assoc |
	UserGlobals at: #temp put: (assoc := Association new).
	System commitTransaction; enableSignaledObjectsError; addToNotifySet: assoc.
	ObjectsCommittedNotification addDefaultHandler: [:ex | ex halt].
	assoc key: 42.
	System commitTransaction.
	(Delay forSeconds: 1) wait.
	

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_ObjectsCommittedNotification.
  gsResumable := true .
  gsTrappable := true .
%
category: 'Handling'
method:
defaultAction

"Return an error to the controlling GCI client .
 See AbstractException>>defaultAction for documentation."
<primitive: 33>
self _uncontinuableError
%

! fix 41579
category: 'Signalling'
classmethod:
enableSignalling

"Enable instances of ObjectsCommittedNotification to be signalled by the VM .
 must be resent by exception handling code after handling each
 instance of ObjectsCommittedNotification that was signalled by the VM."

 System _updateSignalErrorStatus: 1 toState: true
%

classmethod:
disableSignalling

"Prevent instances of ObjectsCommittedNotification from being signalled by the VM."

 System _updateSignalErrorStatus: 1 toState: false
%

classmethod:
signallingEnabled

"Return true if signaling enabled, return false otherwise  .
 See ObjectsCommittedNotification(C)>>enableSignalling "

^ System _signalErrorStatus: 1
%


! =========================
set class TerminateProcess
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
If a non-forked process is terminated, then after the process
termination is completed a TerminateProcess error will be 
signaled and sent to the GCI. This Error is not-trappable.
You can see the error by evaluating the following expression:
	Processor activeProcess terminate.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_TerminateProcess.
  gsResumable := false .
  gsTrappable := false .
%
category: 'Handling'
method:
defaultAction

"Return an error to the controlling GCI client .
 See AbstractException>>defaultAction for documentation."
<primitive: 33>
self _uncontinuableError
%

! =========================
set class TransactionBacklog
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod: TransactionBacklog
comment
^ '
TransactionBacklog is a Notification that your session is holding an
old commit record and this is a request to do an abort or commit (as
appropriate). You will only get this Notification if you have enabled
it (see System class>>#enableSignaledAbortError and 
#enableSignaledFinishTransactionError).
TransactionBacklog represents a ''sigAbort'' which was sent to the session.
RepositoryViewLost which represents a ''lostOt'' is a separate 
subclass of Notification.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	inTransaction: Boolean
'
%

!edits for 41578
category: 'Signalling'
classmethod:
enableSignalling

"Enables a TransactionBacklog Notification if this session
 should commit or abort.  If you want to only get the should commit or
 should abort Notification , 
 use System class>>enableSignaledFinishTransactionError
 or System class>>enableSignaledAbortError. 

 If sent within a handlerBlock which is handling an instance of
 TransactionBacklog, the enable takes effect when execution
 leaves the handlerBlock via a return or resume .
"

"enable both SignaledAbort and SignaledFinishTransaction"
System _updateSignalErrorStatus: 8 toState: true .
%

classmethod:
disableSignalling
  
"disable both SignaledAbort and SignaledFinishTransaction"
System _updateSignalErrorStatus: 8 toState: false .

%

classmethod:
signallingEnabled

"Return true if signaling enabled, return false otherwise.
 See TransactionBacklog(C)>>enableSignalling "

^ System _signalErrorStatus: 4
%

category: 'Handling'
method:
defaultAction

"Return an error to the controlling GCI client .
 See AbstractException>>defaultAction for documentation."
<primitive: 33>
self _uncontinuableError
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_TransactionBacklog.
  gsResumable := true .
  gsTrappable := true .
%

!  inTransaction: not need; usually signaled from C

method:
_legacyHandlerArgs
  | arr |
  (arr := { inTransaction } ) addAll: super _legacyHandlerArgs .
  ^ arr
%
method:
args: anArray
  anArray _isArray ifTrue:[ | sz |
    inTransaction := anArray atOrNil: 1 .
    (sz := anArray size) > 1 ifTrue:[
      gsArgs := anArray copyFrom: 2 to: sz
    ].
  ]
%

category: 'Accessing'
method:
inTransaction
  ^ inTransaction  "aBoolean"
%

category: 'Formatting'
method:
asString
^ self _description:(
  [ | str |
    str := 'inTransaction=', inTransaction asString .
    gsDetails ifNotNil:[:d| str add:', '; add: d  asString ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

! =========================
set class Warning
removeallmethods
removeallclassmethods

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_Warning.
  gsResumable := true .
  gsTrappable := true .
%
! other methods in Warning.gs

! =========================
set class TestFailure
removeallmethods
removeallclassmethods

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_TestFailure .
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class CryptoError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
An exception signaled when certain encrypt/decrypt methods fail.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_CryptoError.
  gsResumable := true .
  gsTrappable := true .
%

! =========================
set class MigrationError
removeallmethods
removeallclassmethods

category: 'Documentation'
classMethod:
comment
^ '
An exception signaled when an object cannot be migrated from
being an instance of a class to an instance of some other class.
  Instance variables
     object - the object for which migration failed
'
%

category: 'Accessing'
classmethod:
errorNumber 
  ^ ERR_MigrationError
%

category: 'Instance initialization'
method:
initialize
  gsNumber := ERR_MigrationError .
  gsResumable := false .
  gsTrappable := true .
%

method:
args: anArray
  anArray _isArray ifTrue:[ | sz |
    object := anArray atOrNil: 1 .
    (sz := anArray size) > 1 ifTrue:[
      gsArgs := anArray copyFrom:2 to: sz
    ].
  ]
%

method:
object: anObject
  object := anObject
%

category: 'Accessing'
method:
object
  ^ object
%

category: 'Formatting'
method:
asString
^ self _description:(
  [ | str |
    (str := String withAll:'object=a ') add: object class versionedName .
    gsDetails ifNotNil:[:d| str add:', '; add: d  asString ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

