!=========================================================================
! Copyright (C) GemTalk Systems 2017-2020.  All Rights Reserved.
!
! $Id: Btree-Evaluator-Core.gs 38384 2016-01-08 18:22:36Z lalmarod $
!
! Btree-Evaluator-Core.gs  -  source code for the gs evaluator classes
!
!========================================================================

! class created in btreeplusclasses.gs

! Class Implementation for GsIndexedQueryEvaluator

! Remove existing behavior from GsIndexedQueryEvaluator
removeallmethods GsIndexedQueryEvaluator
removeallclassmethods GsIndexedQueryEvaluator

! ------------------- Class methods for GsIndexedQueryEvaluator
category: 'Instance Creation'
classmethod: GsIndexedQueryEvaluator
on: anIndex

  ^ self new index: anIndex
%
! ------------------- Instance methods for GsIndexedQueryEvaluator
category: 'Support'
method: GsIndexedQueryEvaluator
_allIdenticalTo: anObject in: nsc

"Returns a new NSC containing the given object with the number of occurrences
 as in the root NSC."

| result tmp |
result := self _primAllIdenticalTo: anObject in: nsc _asIdentityBag.
result class = nsc species
  ifFalse: [
    tmp := nsc species new.
    tmp addAll: result.
    result := tmp
  ].
^ result 
%
category: 'Support'
method: GsIndexedQueryEvaluator
_allIdenticalTo: anObject in: nsc do: aBlock
  "Returns a new NSC containing the given object with the number of occurrences
 as in the root NSC."

  | result |
  result := self _primAllIdenticalTo: anObject in: nsc _asIdentityBag.
  result do: aBlock
%
category: 'Error Handling'
method: GsIndexedQueryEvaluator
_errorCannotInvokeRangeOperationOnIdentityIndex

"An attempt was made to use a range operation (< > = <= >=) on a path
 expression only supported by an identity index."

^ self index _errorCannotInvokeRangeOperationOnIdentityIndex
%
category: 'Searching'
method: GsIndexedQueryEvaluator
_findAllValuesThroughSetsNotIdenticalTo: aKey
  "Returns a set of all values that satisfy the query.  This method works
 when the path is through a set-valued instance variable."

  |  tmpHolder idx |
  tmpHolder := NscBuilder
    for: self nscRoot species new
    max: self sizeForNscBuilder.
  idx := self index.
  idx btreeRoot _lastRootObjectsNotIdenticalTo: aKey into: tmpHolder.
  ^ tmpHolder completeBag
%
category: 'Searching'
method: GsIndexedQueryEvaluator
_findFirstKeyNotIdenticalTo: aValue atTerm: pathTerm

  ^IndexedQueryEvaluator subclassResponsibility: #_findFirstKeyNotIdenticalTo:atTerm:
%
category: 'Support'
method: GsIndexedQueryEvaluator
_primAllIdenticalTo: anObject in: nsc

"Returns a new NSC containing the given object with the number of occurrences
 as in the root NSC."

<primitive: 340>

self _primitiveFailed: #_primAllIdenticalTo:in: args: { anObject . nsc } .
self _uncontinuableError
%
category: 'Accessing'
method: GsIndexedQueryEvaluator
btreeComparisonQuerySpec

  ^self index btreeComparisonQuerySpec
%
category: 'Accessing'
method: GsIndexedQueryEvaluator
collator
  "Return the IcuCollator to be used in comparing Unicode strings"

  ^ self index collator
%
category: 'Accessing'
method: GsIndexedQueryEvaluator
doBlock
  ^ doBlock
%
category: 'Accessing'
method: GsIndexedQueryEvaluator
doBlock: aBlock
  doBlock := aBlock
%
category: 'Query Select'
method: GsIndexedQueryEvaluator
findAllValuesIdenticalTo: aValue
  "Returns a set of all values that satisfy the query.  Since this is an identity
 comparison on a range index (which supports equality comparison), this
 method creates a B-tree read stream for equal values, then iterates through the
 values making the identity comparison."

  | tmpHolder stream |
  " optimize query for objects in the NSC itself "
  self index isIndexOnRootNsc
    ifTrue: [
      doBlock 
        ifNil: [
          " optimize query for objects in the NSC itself "
          ^ self _allIdenticalTo: aValue in: self nscRoot ].
      ^ self _allIdenticalTo: aValue in: self nscRoot do: doBlock ].
  stream := self _findAllValuesIdenticalTo: aValue.
  self doBlock
    ifNotNil: [ 
      stream atEnd
        ifTrue: [ ^ self ].
      ^ [ stream atEnd ] 
           whileFalse: [ 
             | nextKey next |
             nextKey := stream _peekKey.
             next := stream next.
             nextKey == aValue 
             ifTrue: [ self doBlock value: next ] ] ].
  stream atEnd
    ifTrue: [ ^ self nscRoot species new ].
  tmpHolder := NscBuilder
    for: self nscRoot species new
    max: self sizeForNscBuilder.
  stream _valuesAddInto: tmpHolder identicalTo: aValue.
  ^ tmpHolder completeBag
%
category: 'Query Select'
method: GsIndexedQueryEvaluator
findAllValuesNotIdenticalTo: aValue
  "Returns a set of all values that satisfy the query."

  | tmpHolder res |
  self index isIndexOnRootNsc
    ifTrue: [ 
      tmpHolder := self nscRoot species new.
      tmpHolder addAll: self nscRoot.
      tmpHolder removeAll: (self _allIdenticalTo: aValue in: self nscRoot).
      doBlock ifNotNil: [ tmpHolder do: doBlock ].
      ^ tmpHolder ].
  self hasCollectionBasedTerm
    ifTrue: [ 
      res := self _findAllValuesThroughSetsNotIdenticalTo: aValue.
      doBlock ifNotNil: [ res do: doBlock ].
      ^ res ].
  tmpHolder := NscBuilder
    for: self nscRoot species new
    max: self sizeForNscBuilder.
  tmpHolder _addAllNsc: (self _findAllValuesIdenticalTo: aValue) makeNsc.	" get all objects identical to aValue and identical to nil "
  tmpHolder _addAllNsc: (self _allIdenticalTo: nil in: self nscRoot).
  " get all objects with nil along the path "
  self index _getObjectsWithNilOnPathInto: tmpHolder.
  res := self nscRoot _asIdentityBag - tmpHolder completeBag _asIdentityBag.
  doBlock ifNotNil: [ res do: doBlock ].
  res class == self nscRoot species
    ifFalse: [ 
      | finalRes |
      finalRes := self nscRoot species new.
      finalRes addAll: res.
      ^ finalRes ].
  ^ res
%
category: 'Query Select'
method: GsIndexedQueryEvaluator
findAllValuesNotIdenticalTo: aValue do: aBlock
  "Iterates over all values that satisfy the query."

  | result |
  result := self findAllValuesNotIdenticalTo: aValue.
  result do: aBlock.
  ^ result
%
category: 'Query Detect'
method: GsIndexedQueryEvaluator
findFirstValueIdenticalTo: aValue

  ^self subclassResponsibility: #findFirstValueIdenticalTo:
%
category: 'Query Detect'
method: GsIndexedQueryEvaluator
findFirstValueNotIdenticalTo: aValue

"Returns the first value that satisfies the query."

| key lastPathTerm |

lastPathTerm := self index lastPathTerm.
" optimize query for objects in the NSC itself "
self index isIndexOnRootNsc
  ifTrue: [
    index nscRoot do: [ :element |
      element == aValue
        ifFalse: [ ^ element ] ].
    ^ #_incompletePathTraversal ].

key := self _findFirstKeyNotIdenticalTo: aValue atTerm: lastPathTerm.
(key == #_incompletePathTraversal)
  ifTrue: [ ^ #_incompletePathTraversal ].

^ self findFirstValueIdenticalTo: key
%
category: 'Testing'
method: GsIndexedQueryEvaluator
hasCollectionBasedTerm
  ^ self index hasCollectionBasedTerm
%
category: 'Testing'
method: GsIndexedQueryEvaluator
hasSetValuedTerm
  ^ self index hasSetValuedTerm
%
category: 'Accessing'
method: GsIndexedQueryEvaluator
index

  ^index
%
category: 'Updating'
method: GsIndexedQueryEvaluator
index: anIndex

  index := anIndex
%
category: 'Accessing'
method: GsIndexedQueryEvaluator
nscRoot

^self index nscRoot
%
category: 'Accessing'
method: GsIndexedQueryEvaluator
sizeForNscBuilder

"Returns the size to be used when an NscBuilder is collecting objects."

^ self nscRoot size
%

! Class Implementation for GsEqualityIndexQueryEvaluator

! Remove existing behavior from GsEqualityIndexQueryEvaluator
removeallmethods GsEqualityIndexQueryEvaluator
removeallclassmethods GsEqualityIndexQueryEvaluator

! ------------------- Instance methods for GsEqualityIndexQueryEvaluator

category: 'Searching'
method: GsEqualityIndexQueryEvaluator
_findAllValuesEqualTo: val1 using: aComparison
  "Returns a RangeIndexReadStream that iterates over all objects in the index
 that satisfy the query."

  | stream querySpec |
  stream := self
    _findAllValuesGreaterThan: val1
    andEquals: true
    andLessThan: val1
    andEquals: true
    using: aComparison.
  querySpec := self index btreeComparisonQuerySpec
    key: val1 selector: #'=';
    yourself.
  stream streamQuerySpec: querySpec.
  ^ stream
%
category: 'Searching'
method: GsEqualityIndexQueryEvaluator
_findAllValuesGreaterThan: val1 andEquals: bool1 andLessThan: val2 andEquals: bool2 using: aComparison
  "Returns a RangeIndexReadStream that iterates over all objects in the index
 that satisfy the query."

  | array1 array2 stream doNext |
  val1 ifNotNil: [
    "nil allowed in all queries, disallowed as a key in optimized comparison indexes."
    (self index _canCompareWith: val1)
      ifFalse: [ self _error: #'assocErrBadComparison' ] ].
  (self index _canCompareWith: val2 )
    ifFalse: [ self _error: #'assocErrBadComparison' ].
  stream := self readStreamClass new rangeIndex: self index.	" check to see if the query is even satisfiable "
  (aComparison
    satisfiableQueryGreaterThan: val1
    andEquals: bool1
    andLessThan: val2
    andEquals: bool2)
    ifFalse: [ 
      stream currentStack: {0}.
      ^ stream ].
  doNext := false.
  array1 := {}.
  (self index optimizingComparison and: [ val1 == nil ])
    ifTrue: [ 
      " all were greater than or equal to "
      self btreeRoot
        _putFirstIndexOfFirstChildInto: array1
        ifGreaterThanOrEqualTo: val1
        using: aComparison ]
    ifFalse: [ 
      " see if querying > or >= "
      bool1
        ifTrue: [ 
          " if >=, perform < and do an '_advance' operation later "
          (self btreeRoot
            _findAllValuesLessThanKey: val1
            andEquals: false
            into: array1
            using: aComparison)
              ifTrue: [ doNext := true ]
              ifFalse: [ 
                " all were greater than or equal to "
                self btreeRoot
                  _putFirstIndexOfFirstChildInto: array1
                  ifGreaterThanOrEqualTo: val1
                  using: aComparison ] ]
        ifFalse: [ 
          " if >, ask the B-tree specifically for > "
          (self btreeRoot
            _findAllValuesGreaterThanKey: val1
            into: array1
            using: aComparison)
              ifFalse: [ 
                " none were greater than "
                stream currentStack: {0}.
                ^ stream ] ] ].
  array2 := {}.	" ask the B-tree for the second boundary of the query result "
  (self btreeRoot
    _findAllValuesLessThanKey: val2
    andEquals: bool2
    into: array2
    using: aComparison)
    ifFalse: [ 
      " none were found less than "
      stream currentStack: {0}.
      ^ stream ].
  stream endIndex: (array2 at: 1).
  stream endNode: (array2 at: 2).
  (stream endNode == (array1 at: 2) and: [ stream endIndex < (array1 at: 1) ])
    ifTrue: [ 
      array1 at: 1 put: 0.
      stream currentStack: array1.
      ^ stream ].
  stream currentStack: array1.
  doNext
    ifTrue: [ 
      " advance the B-tree positioning "
      stream _advance ].
  ^ stream
%
category: 'Searching'
method: GsEqualityIndexQueryEvaluator
_findAllValuesGreaterThanKey: aKey andEquals: aBoolean
  "Returns a RangeIndexReadStream that iterates over all objects in the index
 that satisfy the query."

  ^ self
    _findAllValuesGreaterThanKey: aKey
    andEquals: aBoolean
    using: self comparisonForCompare
%
category: 'Searching'
method: GsEqualityIndexQueryEvaluator
_findAllValuesGreaterThanKey: aKey andEquals: aBoolean using: aComparison
  "Returns a RangeIndexReadStream that iterates over all objects in the index
 that satisfy the query."

  | array1 array2 stream doNext querySpec |
  (self index _canCompareWith: aKey)
    ifFalse: [ self _error: #'assocErrBadComparison' ].
  stream := self readStreamClass new rangeIndex: self index.
  doNext := false.
  array1 := {}.	" see if querying > or >= "
  aBoolean
    ifTrue: [ 
      " if >=, perform < and do a 'next' operation later "
      (self btreeRoot
        _findAllValuesLessThanKey: aKey
        andEquals: false
        into: array1
        using: aComparison)
        ifTrue: [ doNext := true ]
        ifFalse: [ 
          " all were greater than or equal to "
          self btreeRoot
            _putFirstIndexOfFirstChildInto: array1
            ifGreaterThanOrEqualTo: aKey
            using: aComparison ] ]
    ifFalse: [ 
      " if >, ask the B-tree specifically for > "
      (self btreeRoot
        _findAllValuesGreaterThanKey: aKey
        into: array1
        using: aComparison)
        ifFalse: [ 
          " none were greater than "
          stream currentStack: {0}.
          ^ stream ] ].
  array2 := {}.
  self btreeRoot _putLastIndexOfLastChildInto: array2.
  stream currentStack: array1.
  stream endIndex: (array2 at: 1).
  stream endNode: (array2 at: 2).
  doNext
    ifTrue: [ stream _advance ].
  querySpec := self index btreeComparisonQuerySpec
    key: aKey
    selector:
      (aBoolean
        ifTrue: [ #'>=' ]
        ifFalse: [ #'>' ]);
    yourself.
  stream streamQuerySpec: querySpec.
  ^ stream
%
category: 'Searching'
method: GsEqualityIndexQueryEvaluator
_findAllValuesIdenticalTo: aValue
  "Returns a set of all values that satisfy the query.  Since this is an identity
 comparison on a range index (which supports equality comparison), this
 method creates a B-tree read stream for equal values, then iterates through the
 values making the identity comparison."

  |  stream  |
  stream := super _findAllValuesIdenticalTo: aValue.
  stream
    streamQuerySpec: (stream btreeComparisonQuerySpec key: aValue selector: #'==').
  ^ stream 
%
category: 'Searching'
method: GsEqualityIndexQueryEvaluator
_findAllValuesLessThanKey: aKey andEquals: aBoolean using: aComparison
  "Returns a RangeIndexReadStream that iterates over all objects in the index
 that satisfy the query."

  | array1 array2 stream querySpec |
  (self index _canCompareWith: aKey)
    ifFalse: [ self _error: #'assocErrBadComparison' ].
  stream := self readStreamClass new rangeIndex: self index.
  array2 := {}.	" ask the B-tree to make the search and put the result path into array2 "
  (self btreeRoot
    _findAllValuesLessThanKey: aKey
    andEquals: aBoolean
    into: array2
    using: aComparison)
    ifTrue: [ 
      array1 := {}.	" put path to first entry of the B-tree into array1 "
      self btreeRoot _putFirstIndexOfFirstChildInto: array1.
      stream currentStack: array1.
      stream endIndex: (array2 at: 1).
      stream endNode: (array2 at: 2) ]
    ifFalse: [ 
      " none were found "
      stream currentStack: {0} ].
  querySpec := self index btreeComparisonQuerySpec
    key: aKey
    selector:
      (aBoolean
        ifTrue: [ #'<=' ]
        ifFalse: [ #'<' ]);
    yourself.
  stream streamQuerySpec: querySpec.
  ^ stream
%
category: 'Searching'
method: GsEqualityIndexQueryEvaluator
_findFirstKeyNotIdenticalTo: aKey atTerm: pathTerm
  "Returns the first key in the receiver's B-tree that is not identical to the
 given key."

  | stream key |
  stream := self btreeReadStreamClass on: self btreeRoot.
  [ stream _atEnd ] whileFalse: [ key := stream _peekKey.
      key == aKey
        ifFalse: [ ^ key ].
      stream _btreeNext ].
  ^ #'_incompletePathTraversal'
%
category: 'Accessing'
method: GsEqualityIndexQueryEvaluator
btreeReadStreamClass
  "Returns the class of btree read stream to create for query results."

  ^ index btreeReadStreamClass
%
category: 'Accessing'
method: GsEqualityIndexQueryEvaluator
btreeRoot

  ^self index btreeRoot
%
category: 'Accessing'
method: GsEqualityIndexQueryEvaluator
comparisonForCompare
  ^ self index comparisonForCompare
%
category: 'Accessing'
method: GsEqualityIndexQueryEvaluator
comparisonForSort
  ^ self index comparisonForSort
%
category: 'Query Select'
method: GsEqualityIndexQueryEvaluator
findAllValuesEqualTo: aValue
  "Returns a set of all values that satisfy the query."

  | stream |
  stream := self _findAllValuesEqualTo: aValue using: self comparisonForCompare.
  ^ self doBlock
    ifNotNil: [ [ stream atEnd ] whileFalse: [ self doBlock value: stream next ] ]
    ifNil: [ stream makeNscComparing: #'=' key: aValue ]
%
category: 'Query Select'
method: GsEqualityIndexQueryEvaluator
findAllValuesGreaterThan: val1 andEquals: bool1 andLessThan: val2 andEquals: bool2
  "Returns a set of all values that satisfy the query."

  | selector1 selector2 stream |
  selector1 := bool1
    ifTrue: [ #'>=' ]
    ifFalse: [ #'>' ].
  selector2 := bool2
    ifTrue: [ #'<=' ]
    ifFalse: [ #'<' ].
  stream := self
    _findAllValuesGreaterThan: val1
    andEquals: bool1
    andLessThan: val2
    andEquals: bool2
    using: self comparisonForCompare.
  ^ self doBlock
    ifNotNil: [ stream streamQuerySpec: (stream btreeRangeComparisonQuerySpec
            key: val1 selector: selector1;
            key2: val2 selector2: selector2;
            yourself).
      [ stream atEnd ] whileFalse: [ self doBlock value: stream next ] ]
    ifNil: [ stream
        makeNscComparing: selector1
        key: val1
        and: selector2
        key: val2 ]
%
category: 'Query Select'
method: GsEqualityIndexQueryEvaluator
findAllValuesGreaterThanKey: aValue andEquals: aBoolean
  "Returns a set of all values that satisfy the query."

  | selector stream |
  selector := aBoolean
    ifTrue: [ #'>=' ]
    ifFalse: [ #'>' ].
  stream := self
    _findAllValuesGreaterThanKey: aValue
    andEquals: aBoolean
    using: self comparisonForCompare.
  ^ self doBlock
    ifNotNil: [ [ stream atEnd ] whileFalse: [ self doBlock value: stream next ] ]
    ifNil: [ stream makeNscComparing: selector key: aValue ]
%
category: 'Query Select'
method: GsEqualityIndexQueryEvaluator
findAllValuesLessThanKey: aValue andEquals: aBoolean
  "Returns a set of all values that satisfy the query."

  | selector stream |
  selector := aBoolean
    ifTrue: [ #'<=' ]
    ifFalse: [ #'<' ].
  stream := self
    _findAllValuesGreaterThan: nil
    andEquals: true
    andLessThan: aValue
    andEquals: aBoolean
    using: self comparisonForCompare.
  ^ self doBlock
    ifNotNil: [ | querySpec |
      querySpec := self index btreeComparisonQuerySpec
        key: aValue selector: selector;
        yourself.
      stream streamQuerySpec: querySpec.
      [ stream atEnd ] whileFalse: [ self doBlock value: stream next ] ]
    ifNil: [ stream makeNscComparing: selector key: aValue ]
%
category: 'Query Select'
method: GsEqualityIndexQueryEvaluator
findAllValuesNotEqualTo: aValue
  "Returns a set of all values that satisfy the query."

  | preStream eqStream postStream querySpec |
  preStream := self
    _findAllValuesGreaterThan: nil
    andEquals: true
    andLessThan: aValue
    andEquals: false
    using: self comparisonForSort.
  eqStream := self
    _findAllValuesGreaterThan: aValue
    andEquals: true
    andLessThan: aValue
    andEquals: true
    using: self comparisonForSort.
  querySpec := self index btreeComparisonQuerySpec
    key: aValue selector: #'~=';
    yourself.
  eqStream streamQuerySpec: querySpec.
  postStream := self
    _findAllValuesGreaterThanKey: aValue
    andEquals: false
    using: self comparisonForSort.
  postStream streamQuerySpec: nil.
  ^ self doBlock
    ifNotNil: [ :aBlock | 
      [ preStream atEnd ] whileFalse: [ aBlock value: preStream next ].
      [ eqStream atEnd ] whileFalse: [ aBlock value: eqStream next ].
      [ postStream atEnd ] whileFalse: [ aBlock value: postStream next ] ]
    ifNil: [ 
      | tmpHolder |
      tmpHolder := NscBuilder
        for: self nscRoot species new
        max: self sizeForNscBuilder.
      preStream _valuesAddInto: tmpHolder.
      eqStream _valuesAddInto: tmpHolder notEqualTo: aValue.
      postStream _valuesAddInto: tmpHolder.
      tmpHolder completeBag ]
%
category: 'Query Detect'
method: GsEqualityIndexQueryEvaluator
findFirstValueEqualTo: aValue

"Returns the first value that satisfies the query."

(index _canCompareWith: aValue)
    ifFalse: [ self _error: #assocErrBadComparison ].

^ self btreeRoot _findFirstRootForKey: aValue using: self comparisonForCompare
%
category: 'Query Detect'
method: GsEqualityIndexQueryEvaluator
findFirstValueGreaterThan: val1
andEquals: bool1
andLessThan: val2
andEquals: bool2

"Returns the first value that satisfies the query."

| stream selector1 selector2 |
stream := self
    _findAllValuesGreaterThan: val1
    andEquals: bool1
    andLessThan: val2
    andEquals: bool2 
    using: self comparisonForCompare.

stream == nil 
  ifTrue: [ ^ #_incompletePathTraversal ].
selector1 := bool1
  ifTrue: [ #'_idxForCompareGreaterThanOrEqualTo:' ]
  ifFalse: [ #'_idxForCompareGreaterThan:' ].
selector2 := bool2
  ifTrue: [ #'_idxForCompareLessThanOrEqualTo:' ]
  ifFalse: [ #'_idxForCompareLessThan:' ].
^ stream nextRootComparing: selector1 key: val1 and: selector2 key: val2
%
category: 'Query Detect'
method: GsEqualityIndexQueryEvaluator
findFirstValueGreaterThanKey: aValue
andEquals: aBoolean

"Returns the first value that satisfies the query."

| stream selector |
stream := self  _findAllValuesGreaterThanKey: aValue 
                    andEquals: aBoolean
                    using: self comparisonForCompare.
stream == nil 
  ifTrue: [ ^ #_incompletePathTraversal ].
selector := aBoolean
  ifTrue: [ #'_idxForCompareGreaterThanOrEqualTo:' ]
  ifFalse: [ #'_idxForCompareGreaterThan:' ].
^ stream nextRootComparing: selector key: aValue
%
category: 'Query Detect'
method: GsEqualityIndexQueryEvaluator
findFirstValueIdenticalTo: aValue
  "Returns the first value that satisfies the query.  Since this is an identity
 comparison on a range index (which supports equality comparison), create a
 BtreeReadStream for equal values, then iterate through the values making
 the identity comparison."

  | stream |
  stream := self _findAllValuesIdenticalTo: aValue.
  stream == nil
    ifTrue: [ ^ #'_incompletePathTraversal' ].	" iterate through the stream, checking for an identical value "
  [ stream _atEnd not ]
    whileTrue: [ 
      stream _peekKey == aValue
        ifTrue: [ 
          ^ stream _peekRoot ].
      stream _btreeNext ].
  ^ #'_incompletePathTraversal'
%
category: 'Query Detect'
method: GsEqualityIndexQueryEvaluator
findFirstValueLessThanKey: aValue
andEquals: aBoolean

"Returns the first value that satisfies the query."

| stream selector |
stream := self 
      _findAllValuesGreaterThan: nil
      andEquals: true
      andLessThan: aValue
      andEquals: aBoolean
      using: self comparisonForCompare.
stream == nil 
  ifTrue: [ ^ #_incompletePathTraversal ].
selector := aBoolean
  ifTrue: [ #'_idxForCompareLessThanOrEqualTo:' ]
  ifFalse: [ #'_idxForCompareLessThan:' ].
^ stream nextRootComparing: selector key: aValue
%
category: 'Query Detect'
method: GsEqualityIndexQueryEvaluator
findFirstValueNotEqualTo: aValue
  "Returns the first value that satisfies the query."

  | stream totalStream obj bag incomplete |
  " find the ones that are equal "
  stream := self
    _findAllValuesGreaterThan: aValue
    andEquals: true
    andLessThan: aValue
    andEquals: true
    using: self comparisonForCompare.
  stream _atEnd
    ifTrue: [ " if none are equal, pick the first one "
      self hasSetValuedTerm
        ifTrue: [ bag := self nscRoot _asIdentityBag.
          1 to: bag size do: [ :i | nil ~~ (bag _at: i)
              ifTrue: [ ^ bag _at: i ] ].
          ^ #'_incompletePathTraversal' ]
        ifFalse: [ incomplete := #'_incompletePathTraversal'.
          bag := self nscRoot _asIdentityBag.
          1 to: bag size do: [ :i | obj := bag _at: i.
            (self index traverseObject: obj) == incomplete
              ifFalse: [ ^ obj ] ].
          ^ #'_incompletePathTraversal' ] ].
  totalStream := self readStreamClass on: self index.	" get a stream over the entire NSC "
  (totalStream _peekKey _idxForCompareNotEqualTo: stream _peekKey)
    ifTrue: [ " see if very first object is not equal "
      ^ totalStream next ].
  (totalStream _hasSameEndAs: stream)
    ifTrue: [ " see if end of original stream is the very last object "
      ^ #'_incompletePathTraversal' ]
    ifFalse: [ " advance the stream to the last object "
      totalStream _positionToEnd.
      ^ totalStream next ]
%
category: 'Accessing'
method: GsEqualityIndexQueryEvaluator
readStreamClass
  "Returns the class of read stream to create for query results."

  ^ index readStreamClass
%

! Class Implementation for GsIdentityIndexQueryEvaluator

! Remove existing behavior from GsIdentityIndexQueryEvaluator
removeallmethods GsIdentityIndexQueryEvaluator
removeallclassmethods GsIdentityIndexQueryEvaluator

! ------------------- Instance methods for GsIdentityIndexQueryEvaluator

category: 'Searching'
method: GsIdentityIndexQueryEvaluator
_findAllValuesGreaterThanKey: aKey andEquals: aBoolean
  "Returns a BtreePlusRangeIndexReadStream that iterates over all objects in the index
 that satisfy the query."

  ^ self
    _findAllValuesGreaterThanKey: aKey
    andEquals: aBoolean
    using: self comparisonForSort
%

category: 'Searching'
method: GsIdentityIndexQueryEvaluator
_findAllValuesGreaterThanKey: aKey andEquals: aBoolean using: aComparison
  "Returns a RangeIndexReadStream that iterates over all objects in the index
 that satisfy the query."

  | array1 array2 stream doNext querySpec |
  (self index _canCompareWith: aKey)
    ifFalse: [ self _error: #'assocErrBadComparison' ].
  stream := self readStreamClass new rangeIndex: self index.
  doNext := false.
  array1 := {}.	" see if querying > or >= "
  aBoolean
    ifTrue: [ 
      " if >=, perform < and do a 'next' operation later "
      (self btreeRoot
        _findAllValuesLessThanKey: aKey
        andEquals: false
        into: array1
        using: aComparison)
        ifTrue: [ doNext := true ]
        ifFalse: [ 
          " all were greater than or equal to "
          self btreeRoot
            _putFirstIndexOfFirstChildInto: array1
            ifGreaterThanOrEqualTo: aKey
            using: aComparison ] ]
    ifFalse: [ 
      " if >, ask the B-tree specifically for > "
      (self btreeRoot
        _findAllValuesGreaterThanKey: aKey
        into: array1
        using: aComparison)
        ifFalse: [ 
          " none were greater than "
          stream currentStack: {0}.
          ^ stream ] ].
  array2 := {}.
  self btreeRoot _putLastIndexOfLastChildInto: array2.
  stream currentStack: array1.
  stream endIndex: (array2 at: 1).
  stream endNode: (array2 at: 2).
  doNext
    ifTrue: [ stream _advance ].
  querySpec := self index btreeComparisonQuerySpec
    key: aKey selector: (aBoolean
            ifTrue: [ #'>=' ]
            ifFalse: [ #'>' ]);
    yourself.
  stream streamQuerySpec: querySpec.
  ^ stream
%

category: 'Searching'
method: GsIdentityIndexQueryEvaluator
_findFirstKeyNotIdenticalTo: aKey atTerm: pathTerm
  "Returns the first key in the receiver's B-tree that is not identical to the
 given key."

  | stream key |
  stream := self btreeReadStreamClass on: self btreeRoot.
  [ stream _atEnd ] whileFalse: [ key := stream _peekKey.
      key == aKey
        ifFalse: [ ^ key ].
      stream _btreeNext ].
  ^ #'_incompletePathTraversal'
%

category: 'Accessing'
method: GsIdentityIndexQueryEvaluator
btreeReadStreamClass
  "Returns the class of btree read stream to create for query results."

  ^ index btreeReadStreamClass
%

category: 'Accessing'
method: GsIdentityIndexQueryEvaluator
btreeRoot

  ^self index btreeRoot
%

category: 'Accessing'
method: GsIdentityIndexQueryEvaluator
comparisonForCompare
  ^ self index comparisonForCompare
%

category: 'Accessing'
method: GsIdentityIndexQueryEvaluator
comparisonForSort
  ^ self index comparisonForSort
%

category: 'Query Select'
method: GsIdentityIndexQueryEvaluator
findAllValuesEqualTo: aValue

"Identity indexes do not support equality comparison."

^ self _errorCannotInvokeRangeOperationOnIdentityIndex
%

category: 'Query Select'
method: GsIdentityIndexQueryEvaluator
findAllValuesGreaterThan: val1
andEquals: bool1
andLessThan: val2
andEquals: bool2

"Identity indexes do not support range comparison."

^ self _errorCannotInvokeRangeOperationOnIdentityIndex
%

category: 'Query Select'
method: GsIdentityIndexQueryEvaluator
findAllValuesGreaterThanKey: aValue andEquals: aBoolean

"Identity indexes do not support range comparison."

^ self _errorCannotInvokeRangeOperationOnIdentityIndex
%

category: 'Query Select'
method: GsIdentityIndexQueryEvaluator
findAllValuesIdenticalTo: aValue do: aBlock
  "Iterates over all values that satisfy the query.  "

  ^ (self findAllValuesIdenticalTo: aValue) do: aBlock
%

category: 'Query Select'
method: GsIdentityIndexQueryEvaluator
findAllValuesLessThanKey: aValue andEquals: aBoolean

"Identity indexes do not support range comparison."

^ self _errorCannotInvokeRangeOperationOnIdentityIndex
%

category: 'Query Select'
method: GsIdentityIndexQueryEvaluator
findAllValuesNotEqualTo: aValue

"Identity indexes do not support equality comparison."

^ self _errorCannotInvokeRangeOperationOnIdentityIndex
%

category: 'Query Detect'
method: GsIdentityIndexQueryEvaluator
findFirstValueEqualTo: aValue

"Identity indexes do not support range comparison."

^ self _errorCannotInvokeRangeOperationOnIdentityIndex
%

category: 'Query Detect'
method: GsIdentityIndexQueryEvaluator
findFirstValueGreaterThan: val1
andEquals: bool1
andLessThan: val2
andEquals: bool2

"Identity indexes do not support range comparison."

^ self _errorCannotInvokeRangeOperationOnIdentityIndex
%

category: 'Query Detect'
method: GsIdentityIndexQueryEvaluator
findFirstValueGreaterThanKey: aValue
andEquals: aBoolean

"Identity indexes do not support range comparison."

^ self _errorCannotInvokeRangeOperationOnIdentityIndex
%

category: 'Query Detect'
method: GsIdentityIndexQueryEvaluator
findFirstValueIdenticalTo: aValue
  "Returns the first value that satisfies the query.  Since this is an identity
 comparison on a range index (which supports equality comparison), create a
 BtreeReadStream for equal values, then iterate through the values making
 the identity comparison."

  | stream |
  stream := self _findAllValuesIdenticalTo: aValue.
  stream == nil
    ifTrue: [ ^ #'_incompletePathTraversal' ].	" iterate through the stream, checking for an identical value "
  [ stream _atEnd not ]
    whileTrue: [ 
      stream _peekKey == aValue
        ifTrue: [ ^ stream _peekRoot ].
      stream _btreeNext ].
  ^ #'_incompletePathTraversal'
%

category: 'Query Detect'
method: GsIdentityIndexQueryEvaluator
findFirstValueLessThanKey: aValue
andEquals: aBoolean

"Identity indexes do not support range comparison."

^ self _errorCannotInvokeRangeOperationOnIdentityIndex
%

category: 'Query Detect'
method: GsIdentityIndexQueryEvaluator
findFirstValueNotEqualTo: aValue

"Identity indexes do not support range comparison."

^ self _errorCannotInvokeRangeOperationOnIdentityIndex
%

category: 'Accessing'
method: GsIdentityIndexQueryEvaluator
readStreamClass
  "Returns the class of read stream to create for query results."

  ^ index readStreamClass
%

! Class Extensions

! Class initializers 

doit
true.
%

category: 'Searching'
method: GsIndexedQueryEvaluator
_findAllValuesIdenticalTo: aValue
  "Returns a set of all values that satisfy the query."

  | array1 array2 stream doNext |
  " if can't make the comparison, then none are identical "
  (self index _canCompareWith: aValue)
    ifFalse: [ self _error: #'assocErrBadComparison' ].
  " get boundary entries in the B-tree "
  array1 := {}.
  " >=, perform < and do an 'next' operation later "
  doNext := false.
  (self btreeRoot
    _findAllValuesLessThanKey: aValue
    andEquals: false
    into: array1
    using: self comparisonForCompare)
    ifTrue: [ doNext := true ]
    ifFalse: [ " all were greater than or equal to "
      self btreeRoot
        _putFirstIndexOfFirstChildInto: array1
        ifGreaterThanOrEqualTo: aValue
        using: self comparisonForCompare ].
  array2 := {}.
  stream := self readStreamClass on: self index.
  (self btreeRoot
    _findAllValuesLessThanKey: aValue
    andEquals: true
    into: array2
    using: self comparisonForCompare)
    ifFalse: [ " none were found less than "
       stream currentStack: {0}.
       ^ stream ].
  " create the read stream "
  stream currentStack: array1.
  stream endIndex: (array2 at: 1).
  stream endNode: (array2 at: 2).
  doNext
    ifTrue: [ " advance the B-tree positioning "
      stream _advance ].
  ^ stream
%
