! ========================================================================
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
!
! $Id$
!
! Asymmetric signature/verify methods for ByteArray and CharacterCollection
! ========================================================================
!
category: 'Private'
method:
_validateInteger: anInt inRangeFrom: lowest to: highest
anInt _validateClass: SmallInteger .
((anInt < lowest) or:[anInt > highest])
  ifTrue:[ anInt _error: #rtErrArgOutOfRange args:{ lowest . highest } ] .
^ true
%

category: 'Private'
method:
_validateKey: aKey withDigestKind: opCode
|alg|
alg := aKey algorithm . "#RSA, #DSA, #DH, #EC, etc"
alg == #RSA "SHA1,2,3 allowed"
  ifTrue:[ ^ self _validateInteger: opCode abs inRangeFrom: 2 to: 8 ].
alg == #DSA "SHA1 and SHA2 only"
  ifTrue:[ ^ self _validateInteger: opCode inRangeFrom: 2 to: 4 ].
alg == #EC "No digest required/allowed"
  ifTrue:[ ^ self _validateInteger: opCode inRangeFrom: 0 to: 0 ].

^ CryptoError signal: 'Invalid key kind for signing or verifying'
%

category: 'Private'
method:
_primAsymSignVerifyWithKey: aKey digestKind: opCode signature: aByteArray

"Signs or verifies the message contained in the receiver using public key
 encryption. aKey must be an instance of GsTlsPrivateKey (which indicates a
 signing operation) or GsTlsPublicKey (which indicates a verify operation).
 For signing keys that require a message digest, anInt indicates one of the
 supported message digests for the specified key type. For signing keys 
 that do not require a message digest, anInt must be zero.  See the tables
 below. The same message digest must be used for both signing and 
 verifying.

 For signing operations, aByteArray must be a variant instance of ByteArray
 or nil, in which case a new instance of ByteArray will be created to
 contain the signature. For verifying operations, aByteArray must be a 
 non-empty instance of ByteArray containing the signature from the signing.

 For secure signing and verifying with RSA keys only, a padding scheme must
 be used. The default RSA padding scheme is RSA_PKCS1_PADDING 
 (PKCS #1 v1.5 padding), which is the most common type. However the newer
 PSS (Probabilistic Signature Scheme) is more secure and is recommended
 whenever possible. See RFC 3447 for additional information on PSS padding.
 Other RSA padding schemes, including no padding, are not supported due
 to known security vulnerabilities. The padding scheme selected must be the
 same for the signature and verification else the verification will fail.

 To sign or verify with RSA_PKCS1_PSS_PADDING, negate the digest opCode.
 For example, a message signed with an RSA private key using opCode 2 uses
 SHA1 message digest and (the default) RSA_PKCS1_PADDING. Signing a message
 with an RSA private key using opCode -2 uses SHA1 message digest and 
 RSA_PKCS1_PSS_PADDING padding. 

 RSA keys of type RSA-PSS may only use RSA_PKCS1_PSS_PADDING. PSS padding
 will be used for such keys in all cases, even if RSA_PKCS1_PADDING is
 requested. RSA-PSS private keys will answer #EVP_PKEY_RSA_PSS when sent
 the #sslAlgorithm message.

 Signing operations return aByteArray containing the signature on success 
 and raise a CryptoError exception on error. Verify operations return true
 if the verification succeeds and raise a CryptoError exception if 
 verification fails or an error occurs.

 Note that not all private/public key pairs support digital signatures.

 Key Type  Supported Digest Types
 ================================
 DSA       SHA1,SHA2
 ECDSA	   SHA1,SHA2
 RSA	   SHA1,SHA2,SHA3
 EC        None
 Ed25519   None
 Ed448     None
 ==============================


 Digest Type	OpCode
 =====================
 None           0
 SHA1		2
 SHA2-256	3
 SHA2-512	4
 SHA3-224	5
 SHA3-256	6
 SHA3-384	7
 SHA3-512	8
 =====================
"

<primitive: 1089>
opCode _validateClass: SmallInteger .
aKey _validateClasses: { GsTlsPrivateKey . GsTlsPublicKey } .
(aKey isKindOf: GsTlsPrivateKey)
  ifTrue:[ "signing, arg may be nil"
    aByteArray ifNotNil:[ aByteArray _validateClass: ByteArray ].
  ]
  ifFalse:[
     "verifying, arg must be a ByteArray"
    aByteArray _validateClass: ByteArray
].
self _validateKey: aKey withDigestKind: opCode .
^ self _primitiveFailed: #_primAsymSignVerifyWithKey:digestKind:signature:
%

!*********** RSA Signing Methods
category: 'Digital Signature Creation - RSA'
method:
signWithSha1AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA1 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the 
 resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 2 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha1AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA1 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting 
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -2 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha256AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA 256 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 3 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha256AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA2 256 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting 
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -3 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha512AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA2 512 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 4 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha512AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA2 512 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting 
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -4 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha3_224AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 224 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 5 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha3_224AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 224 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting 
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -5 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha3_256AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 256 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 6 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha3_256AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 256 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting 
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -6 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha3_384AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 384 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 7 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha3_384AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 384 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting 
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -7 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha3_512AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 512 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 8 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method:
signWithSha3_512AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 512 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting 
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -8 signature: aByteArrayOrNil
%

!*********** DSA Signing Methods
category: 'Digital Signature Creation - DSA'
method:
signWithSha1AndDsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA1 and signs the resulting hash with the given
 DSA private key. Returns a ByteArray containing the resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #DSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 2 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - DSA'
method:
signWithSha256AndDsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA2 256 and signs the resulting hash with the given
 DSA private key. Returns a ByteArray containing the resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #DSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 3 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - DSA'
method:
signWithSha512AndDsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA2 512 and signs the resulting hash with the given
 DSA private key. Returns a ByteArray containing the resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #DSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 4 signature: aByteArrayOrNil
%

!*********** RSA Verification Methods
category: 'Digital Signature Verification - RSA'
method:
verifyWithSha1AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA1 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 2 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method:
verifyWithSha1AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA1 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -2 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method:
verifyWithSha256AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA2 256 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 3 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method:
verifyWithSha256AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA2 256 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -3 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method:
verifyWithSha512AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA2 512 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 4 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method:
verifyWithSha512AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA2 512 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."
 
aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -4 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method:
verifyWithSha3_224AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 224 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 5 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method:
verifyWithSha3_224AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 224 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."
 
aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -5 signature: aByteArray
%


category: 'Digital Signature Verification - RSA'
method:
verifyWithSha3_256AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 256 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 6 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method:
verifyWithSha3_256AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 256 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."
 
aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -6 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method:
verifyWithSha3_384AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 384 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 7 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method:
verifyWithSha3_384AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 384 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -7 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method:
verifyWithSha3_512AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 512 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 8 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method:
verifyWithSha3_512AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 512 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -8 signature: aByteArray
%

!*********** DSA Verification Methods

category: 'Digital Signature Verification - DSA'
method:
verifyWithSha1AndDsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA1 and verifies the resulting hash using the 
 given DSA public key and signature. Returns true if the signature is correct.
 Otherwise raises an exception.

 See the method:
  # _primAsymSignVerifyWithKey:digestKind:signature: 
 for more information."

aGsTlsPublicKey _validateAlgorithm: #DSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 2 signature: aByteArray
%

category: 'Digital Signature Verification - DSA'
method:
verifyWithSha256AndDsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA256 and verifies the resulting hash using the 
 given DSA public key and signature. Returns true if the signature is correct.
 Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #DSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 3 signature: aByteArray
%

category: 'Digital Signature Verification - DSA'
method:
verifyWithSha512AndDsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA512 and verifies the resulting hash using the 
 given DSA public key and signature. Returns true if the signature is correct.
 Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #DSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 4 signature: aByteArray
%

!*********** EC Signature Methods
category: 'Digital Signature Creation - EC'
method:
signWithEcPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Signs the receiver with the given elliptic curve private key. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #EC ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 0 signature: aByteArrayOrNil
%

!*********** EC Verification Methods
category: 'Digital Signature Creation - EC'
method:
verifyWithEcPublicKey: aGsTlsPublicKey signature: aByteArray
"Verifies the receiver using the given elliptic curve key and signature.  Returns true 
 if the signature is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #EC ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 0 signature: aByteArray
%
