!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: reposit.gs 47826 2020-02-20 19:23:23Z bretlb $
! Superclass Hierarchy:
!   Repository, Collection, Object.
!
!=========================================================================

removeallmethods Repository
removeallclassmethods Repository

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'A Repository is an object that represents a virtual storage into which users
 can place their data.  Each Repository includes an Array of up to 65534
 GsObjectSecurityPolicies, which represent authorization regions for the data.  
 Repositories are described in the GemStone Programming Guide.

Constraints:
	name: Symbol
	dataDictionary: AbstractDictionary
	[elements]: GsObjectSecurityPolicy.

instance variables
name -- The name of the Repository.
dataDictionary -- not used
'
%

expectvalue %String
run
| reposCls |
reposCls := Globals at:#Repository .
reposCls superClass == Array ifTrue:[
  reposCls changeNameTo: #OldRepository .
  Globals at: #Repository put: nil .
  Globals at:#OldRepository put: reposCls .
  ^  Collection _newKernelIndexableSubclass: 'Repository' 
    instVarNames:#( name dataDictionary )
    classVars: #()
    classInstVars: #()
    poolDictionaries: { }
    inDictionary: Globals 
    options: #( disallowGciStore ) 
    reservedOop: 973  
] ifFalse:[
  reposCls superClass == Collection ifTrue:[
    ^ 'class already exists as subclass of Collection'
  ] ifFalse:[
    ^ -1 "error, invalid super class"
  ].
]
%
run
GsObjectSecurityPolicy _disallowGciCreateStore .
^ true
%

category: 'Accessing'
method:
dataDictionary

"Accesses the user-defined data dictionary. (Reserved)"

^ dataDictionary
%

category: 'Accessing'
method:
name

"Returns the logical name of the receiver (a Symbol)."

^ name
%

category: 'Accessing'
method:
fileNames

  <primitive: 429>

"Returns an Array containing the filenames for the extents
 of the receiver.

 Each element within the returned Array contains a String representing
 the filename of the Nth extent (where N is the index into the
 returned Array)."

  self _primitiveFailed: #fileNames .
  self _uncontinuableError
%

! fix 41673, shouldn't rename SystemRepository, removed Repository >> name:

category: 'Updating'
method:
_setVersion

"This method writes the Stone executable version string to the receiver."

<primitive: 289>

self _primitiveFailed: #_setVersion .
self _uncontinuableError
%

! fix 33706
category: 'Clustering'
method:
extentForPage: aPageId

"Returns a SmallInteger specifying an offset into the result from the
 fileNames method.  

 The argument aPageId is an Integer, such as the result from the Object | page
 method, specifying a page in the receiver."

^ (aPageId bitShift: -31 ) + 1
%


category: 'Repository Usage Reporting'
method:
_dbfStatistics: extentFilename

"Returns an Array of size 2 containing the number of bytes the given extent
 requires for physical size and the number of bytes of free space in the
 extent."

<primitive: 283>

extentFilename _validateKindOfClass: String .
^ self _primitiveFailed: #_dbfStatistics: args: { extentFilename }
%

! v2.3: edits to obtain stats for multiple extents in one round trip to stone
category: 'Repository Usage Reporting'
method:
_extentStatistics: extentIndex

"If extentIndex > 0 , 
 Returns an Array of size 2 containing the number of bytes the
 given extent requires for physical size and the number of bytes of
 free space in the extent.

 If extentIndex == -1,
 Returns an Array of Arrays. The outer Array has size equal to the
 number of extents; the inner Arrays each have size 2 and contain
 extent size and extent free space.
"

<primitive: 286>

(extentIndex _isSmallInteger)
  ifTrue:[ ^ self _errorIndexOutOfRange: extentIndex]
  ifFalse:[ extentIndex _validateClass: SmallInteger ].
^ self _primitiveFailed: #_extentStatistics: args: { extentIndex }
%

category: 'Transaction Logging'
method:
_logInfo: logDirId

"If logDirId is within the range of 1 to number of log directories, returns an
 Array containing the following instance variables.  Otherwise returns nil.

 1: logDirId (SmallInteger)
 2: numLogDirs (SmallInteger)
 3: isActiveLog (Boolean)
 4: fileId (SmallInteger) (if active log )
 5: fileSize (SmallInteger) in units of MBytes (if active log)
 6: maxFileSize (SmallInteger) in units of MBytes
 7: logOriginTime (Integer)  (if active log)
 8: fileName (String) (fileOrDevice if active log, else directoryOrDevice)
 9: replicateName (no longer used, always nil)
10: fileId of begin record of oldest transaction in current checkpoint,
	 if active log, otherwise nil
11: fileSize (SmallInteger) in units of 512byte blocks for active log"

<primitive: 397>

logDirId _validateClass: SmallInteger.
^ self _primitiveFailed: #_logInfo: args: { logDirId }
%

category: 'Transaction Logging'
method:
_currentLogInfo

"Returns the result specified for Repository | _logInfo: for the
 currently active transaction log."

| numLogs logInfo |

numLogs := (System configurationAt: #STN_TRAN_LOG_DIRECTORIES) size .
2 timesRepeat:[ "cope with changing to the next log"
  1 to: numLogs do:[ :j |
    logInfo := self _logInfo: j .
    (logInfo at: 3) == true ifTrue:[
      ^ logInfo 
      ].
    ].
  ].
self _halt:'Unable to determine the current log'.
^ { }
%

! fix bug 11283
category: 'Transaction Logging'
method:
currentTranlogSizeMB

"Returns an Integer that is the size of the currently active transaction log in
 units of Megabytes."

^ self _currentLogInfo at: 5
%

category: 'Repository Usage Reporting'
method:
freeSpaceInExtent: extentFilename

"Returns the number of bytes of free space in the extent with the given name.
 This quantity is equal to the number of free pages in the extent times the
 size of a page.

 If the given file is not an active member of the logical Repository
 represented by the receiver, then this method generates an error."

^(self _dbfStatistics: extentFilename) at: 2
%

category: 'Repository Usage Reporting'
method:
fileSizeOfExtent: extentFilename

"Returns the physical size, in bytes, of the extent with the given name.

 If the given file is not an active member of the logical Repository
 represented by the receiver, then this method generates an error."

^(self _dbfStatistics: extentFilename) at: 1
%

category: 'Repository Usage Reporting'
method:
fileSizeReport
"Returns a string which reports on the name, size, and amount of free space
 for each extent and the size and free space of the entire logical Repository.
 Report is in units of GBytes or MBytes, depending on repository size."

^ self _fileSizeReport: nil
%
category: 'Repository Usage Reporting'
method:
fileSizeReportMB

"Returns a string which reports on the name, size, and amount of free space
 for each extent and the size and free space of the entire logical Repository.
 Report is in units of MBytes. "

^ self _fileSizeReport: 1048576.0
%

category: 'Repository Usage Reporting'
method:
_fileSizeReport: unitsArg

"Returns a string which reports on the name, size, and amount of free space
 for each extent and the size and free space of the entire logical Repository.
 unitsArg should a Float power of two that is either MB or GB."

| result stats repositoryFiles extentFile totalSize totalFree units |
result := String new.
stats := self _extentStatistics: -1 .
totalSize := 0.0 . totalFree := 0.0 .
1 to: stats size do:[:j | 
  totalSize := totalSize + ((stats at: j) at: 1) .
  totalFree := totalFree + ((stats at: j) at: 2) .
].
repositoryFiles := self fileNames.
units := unitsArg .
units ifNil:[ | gb |
  gb := 1073741824.0 .
  units := totalSize > (gb * 2) ifTrue:[ gb ] ifFalse:[ 1048576.0 ].
].
1 to: stats size do: [:i |
    extentFile := repositoryFiles at: i.
    i > 1 ifTrue:[ result addAll: '-----------'; lf. ].
    result addAll: 'Extent #'; addAll: i asString; lf.
    result addAll: '   Filename = '; addAll: extentFile; lf ;
           addAll: '   File size =       ';
           addAll: (self _numToString:((stats at:i) at: 1) units: units) ; lf;
           addAll: '   Space available = ';
           addAll: (self _numToString:((stats at:i) at: 2) units: units) ; lf.
    ].
result addAll: '------'; lf.
result addAll: 'Totals'; lf.
result addAll: '   Repository size = ';
       addAll: (self _numToString: totalSize units: units) ; lf;
       addAll: '   Free Space =      ';
       addAll: (self _numToString: totalFree units: units) ; lf .
^ result
%

category: 'Repository Usage Reporting'
method:
extentsReport

"Returns a string which reports on the name, size, and amount of free space
 for each extent and the size and free space of the entire logical Repository.
 Report is in units of GBytes or MBytes, depending on repository size."

^ self _extentsReport: nil 
%

category: 'Repository Usage Reporting'
method:
extentsReportMB

"Returns a string which reports on the name, size, and amount of free space
 for each extent and the size and free space of the entire logical Repository.
 Report is in units of MBytes. "

^ self _extentsReport: 1048576.0
%


category: 'Repository Usage Reporting'
method:
_extentsReport: unitsArg

"Returns a string which reports on the name, size, and amount of free space
 for each extent and the size and free space of the entire logical Repository.
 unitsArg should a Float power of two that is either MB or GB."

| result stats fileNames totalSize totalFree units |
result := String new.
stats := self _extentStatistics: -1 .  
totalSize := 0.0 . totalFree := 0.0 .
1 to: stats size do:[:j | 
  totalSize := totalSize + ((stats at: j) at: 1) .
  totalFree := totalFree + ((stats at: j) at: 2) .
].
units := unitsArg .
units ifNil:[ | gb |
  gb := 1073741824.0 .
  units := totalSize > (gb * 2) ifTrue:[ gb ] ifFalse:[ 1048576.0 ].
].
fileNames := self fileNames.
1 to: stats size do: [:j | 
    result addAll: (fileNames at: j) ; lf ;
           addAll: '   File size   ';
           addAll: (self _numToString:((stats at:j) at: 1) units: units) ; 
           addAll: '   free space  ';
           addAll: (self _numToString:((stats at:j) at: 2) units: units) ; lf.
    ].
result addAll: 'Total'; lf ;
           addAll: '   File size   ';
       addAll: (self _numToString: totalSize units: units) ; 
           addAll: '   free space  ';
       addAll: (self _numToString: totalFree units: units) ; lf .
^ result
%

category: 'Repository Usage Reporting'
method: 
numToMByteString: aNumber

^ self _numToString: aNumber units: 1048576.0
%

! fixed 48613
category: 'Repository Usage Reporting'
method:
_numToString: aNumber units: units 

"Convert a number representing a file size in bytes to a formatted
 string reporting the size in megabytes."

| val fmt |
val := aNumber asFloat / units .
units == 1048576.0 ifTrue:[
  ^ (val asStringUsingFormat: #( -5 0 false )), ' MB'.
].
units = 1073741824.0 ifFalse:[ Error signal:'expected divisor of G or M'].
fmt := #( -6 0 false ).
val < 10.0 ifTrue:[  fmt := #( -8 2 false) ].
val < 100.0 ifTrue:[ fmt := #( -7 1 false) ].
^ (val asStringUsingFormat: fmt) , ' GB'
%

category: 'Repository Usage Reporting'
method:
freeSpace

"Returns an integer that gives the number of bytes of free space in the logical
 Repository.  This quantity is equal to the number of free pages in the
 Repository times the size of a page."

| total stats |

total := 0.
stats := self _extentStatistics: -1.
1 to: stats size do:[:j |
  total := total + ((stats at: j) at: 2)
].
^ total
%

category: 'Repository Usage Reporting'
method:
fileSize

"Returns an integer giving the total physical size, in bytes, of all
 of the physical extents that compose the logical Repository."

| total stats |

total := 0.
stats := self _extentStatistics: -1.
1 to: stats size do:[:j |
  total := total + ((stats at: j) at: 1)
].
^ total
%

! fixed 33546
category: 'Repository Usage Reporting'
method:
pageSize

"Returns size in bytes of a disk page in the Repository."

^ System _zeroArgPrim:50
%

category: 'Transaction Logging'
method:
startNewLog

"Causes the most current transaction log to be closed and a new transaction log
 to be opened for writing.  The location of the new log is controlled by the
 STN_TRAN_LOG_DIRECTORIES configuration file parameter.

 If GemStone is running in partial logging mode, then a preceding transaction
 log may be deleted.  See documentation on the STN_TRAN_FULL_LOGGING
 configuration parameter for more details.

 If a checkpoint is in progress, or the repository is in restore mode,
 the operation is NOT performed and a -1 is returned.  
 If it is successful the SmallInteger fileId of the new log is returned.

 This method requires the FileControl privilege."

<primitive: 432>

self _primitiveFailed: #startNewLog .
self _uncontinuableError
%

! deleted replicate argument from addTransactionLog
category: 'Deprecated'
method:
addTransactionLog: deviceOrDirectory  replicate: replicateSpec size: aSize

self deprecated: 'Repository>>addTransactionLog:replicate:size: is Obsolete, replicate arguments is ignored'.
^ self addTransactionLog: deviceOrDirectory size: aSize
%

category: 'Transaction Logging'
method:
addTransactionLog: deviceOrDirectory  size: aSizeMB
 
"Add deviceOrDirectory to the configuration parameter STN_TRAN_LOG_DIRECTORIES.
 
 The aSizeMB argument must be a positive SmallInteger; it is added to the value
 of the STN_TRAN_LOG_SIZES configuration parameter. Units are M bytes.

 Any environment variables in deviceOrDirectory are expanded using the environment
 of the stone process.  An error is signalled if deviceOrDirectory does not exist
 on the stone's host, or if deviceOrDirectory specifies /dev/null .
 /dev/null can only be specifed as a single entry in STN_TRAN_LOG_DIRECTORIES 
 read by stone at stone startup .

 This method requires the FileControl privilege."

<primitive: 337>

deviceOrDirectory _validateKindOfClass: String .
aSizeMB _validateClass: SmallInteger .
aSizeMB > 0 ifFalse:[ aSizeMB _error: #errArgTooSmall args:{ 1 } ] .
^ self _primitiveFailed: #addTransactionLog:size: 
       args: { deviceOrDirectory . aSizeMB }
%

category: 'Transaction Logging'
method:
currentLogFile

"Returns a String containing the file name of the transaction log file to which
 log records are being appended.  If the result is of size 0, then a 
 log file creation has failed and stone is waiting for more tranlog space
 to be made available."

 ^ System stoneConfigurationAt:#StnCurrentTranLogNames
%

category: 'Transaction Logging'
method:
allTranlogDirectories

"Returns an Array of Strings, each of which represents 1 element of the 
 STN_TRAN_LOG_DIRECTORIES configuration parameter."

| result |
result := System stoneConfigurationAt: #STN_TRAN_LOG_DIRECTORIES .
^ result class == Array ifTrue:[ result ] ifFalse:[ { result } ].
%

category: 'Transaction Logging'
method:
allTranlogSizes

"Returns an Array of SmallIntegers, each of which represents 1 element of the 
 STN_TRAN_LOG_SIZES  configuration parameter.  Each element has units of M bytes"

| result |
result := System stoneConfigurationAt: #STN_TRAN_LOG_SIZES .
^ result class == Array ifTrue:[ result ] ifFalse:[ { result } ].
%

category: 'Transaction Logging'
method:
currentLogDirectoryId

"Returns a positive SmallInteger, which is the one-based offset specifying the
 element of the configuration list STN_TRAN_LOG_DIRECTORIES for the current
 transaction log.  (See also the currentLogFile method.)"

^ (System stoneConfigurationAt:#StnCurrentTranLogDirId) + 1 
%

category: 'Transaction Logging'
method:
currentLogFileId

"Returns a positive SmallInteger, which is the internal fileId of 
 the current transaction log."

^ (self _logInfo: (self currentLogDirectoryId)) at: 4
%

category: 'Transaction Logging'
method:
oldestLogFileIdForRecovery

"Returns a positive SmallInteger, which is the internal fileId of the oldest
 transaction log needed to recover from the most recent checkpoint, if the
 Stone were to fail right now."

^ self _currentLogInfo at: 10
%

category: 'Transaction Logging'
method:
logOriginTime

"Returns the log origin time of the receiver. This is the time when a Stone
 started a new sequence of log files for the receiver. A new sequence of
 logs is started if one of the following occurs:

 * Stone is started using extents that were cleanly shutdown, and without
   any log files being present.

 * Stone is started using extents and no existing logs 
   using the 'startstone -N' command.

 * The commitRestore method is executed, and during preceding 
   restore operations we restored a log file with fileId greater
   than the fileId of the log file being written to during the restore."

^ System stoneConfigurationAt:#StnTranLogOriginTime 
%


category: 'Private'
method:
_writeFdcArray: anArray toFile: aFileNameString
"Given an array of objects produced by the one of the findDisconnectedObjects*
 methods, write the array of objects to a file which can be loaded by 
   System (C) >> readHiddenSet:fromSortedFile:
 for later analysis of disconnected objects.

 anArray must be an array object and contain objects sorted by object ID.
 aFileNameString must be an instance of String that refers to a file which
 does not yet exist."

<primitive: 557>
anArray _validateInstanceOf: Array.
anArray isCommitted ifFalse:[ ArgumentError signal: 'anArray is not a committed object'].
aFileNameString _validateKindOfClass: String.
self _primitiveFailed: #_writeFdcArray:toFile: 
     args: { anArray . aFileNameString } .
self _uncontinuableError.
%

category: 'Disconnected Objects'
method:
writeFdcArrayToFile: aFileNameString
"Write a bitmap with the array of dead objects found by the last findDisconnectedObjects
 (FDC) to the specified file. This can be loaded into a hidden set and used to analyze 
 garbage.

 The file is created by this method and must not exist when it is called.

 anArray must be an array object and contain objects sorted by object ID.
 The primitive will fail if the objects are not in order.

 aFileNameString must be an instance of String that refers to a file which
 does not yet exist.

 Returns true if successful, false if the entire file could not be written,
 or nil if the results of the last FDC could not be found."

 |array theSize result|
 array := Globals at: #FdcResults otherwise: nil.
 (array == nil)
  ifTrue:[^nil].
 (array size < 3)
  ifTrue:[^nil].
 array := array at: 3.
 (array class == Array)
    ifFalse:[^nil].
 theSize := array size.
 (theSize == 0)
   ifTrue:[^nil].
 result := self _writeFdcArray: array toFile: aFileNameString.
 ^ result == theSize.
%

category: 'Disconnected Objects'
method:
findDisconnectedObjectsAndWriteToFile: aFileNameString pageBufferSize: anInt
 saveToRepository: saveToRep
"Perform an FDC operation on the system and write the list of dead objects 
 to the given file, and if saveToRep is true, also store the results in the 
 array under the key #FdcResults in Globals.

 The file or (Globals at:#FdcResults) can be used to analyze the objects
 that would be collectable by a markForCollection, or to manually reconnect
 objects that have been dereferenced by application coding errors.

 To save the dead objects to an array but not write them to a file, set
 aFileNameString to nil and saveToRep to true.  It is illegal for 
 aFileNameString to be nil and for saveToRep to be false.

 The markGcCandidatesFromFile operation is no longer supported, use
 markForCollection to actually perform garbage collection.
 If a previous FDC created FdcResults , you must execute
    Globals at:#FdcResults put: nil . System commit
 before the markForCollection , to make the garbage collectable.
 
 The pageBufSize, which must be a power of two, specifies the number 
 of pages to buffer per thread.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.         

 This method will return an error if the repository is in restore mode.

 *** WARNING *** WARNING *** WARNING *** WARNING *** WARNING ***
 This method is intended to complete the FDC operation as quickly as possible
 and will start 2 threads per CPU core.  Therefore it may consume all available 
 CPU and disk I/O host system resources.

 The FDC operation can be run less aggressively by calling the
 #basicFindDisconnectedObjectsAndWriteToFile: method with a smaller number
 of threads.
 *** WARNING *** WARNING *** WARNING *** WARNING *** WARNING ***

 See the comments in the method:
   Repository>>_fdcWithMaxThreads:waitForLock:pageBufSize:
               percentCpuActiveLimit:toFile:resultArray:

 for a desciption of the result array and a more detailed description of
 this method."

^ self basicFindDisconnectedObjectsAndWriteToFile: aFileNameString 
       pageBufferSize: anInt 
       saveToRepository: saveToRep 
       withMaxThreads: self _aggressiveMaxThreadCount
       maxCpuUsage: 95
%

category: 'Private'
classmethod:
_validatePercentage: anInt
anInt _validateClass: SmallInteger .
((anInt < 1) _or:[ anInt > 100 ])
  ifTrue:[anInt _error: #rtErrArgOutOfRange args:{ 1 . 100 } ] .
%

method:
_validatePercentage: anInt
  ^ self class _validatePercentage: anInt
%

category: 'Disconnected Objects'
method:
findDisconnectedObjectsAndWriteToFile: aFileNameString withMaxThreads: maxThreads
 maxCpuUsage: aPercentage

"Start an FDC operation where the results will be save to the given file but 
 not stored in the repository.  

 The file can be loaded and used to analyze the objects that would be collectable 
 by a markForCollection, or to manually reconnect objects that have been dereferenced 
 by application coding errors.

 Starts maxThreads on the host system and allows the host to run up to aPercentage 
 percent CPU usage.  A page buffer of 128 pages (2 MB) is allocated per thread.

 See the comments in the method:
   Repository>>_fdcWithMaxThreads:waitForLock:pageBufSize:
               percentCpuActiveLimit:toFile:resultArray:

 for a desciption of the result array and a more detailed description of
 this method."

^ self basicFindDisconnectedObjectsAndWriteToFile: aFileNameString 
       pageBufferSize: 128 "must be a power of 2"
       saveToRepository: false 
       withMaxThreads: maxThreads
       maxCpuUsage: aPercentage
%

! removed _markGcCandidatesFromFile... , fix 45606 in Gs64 v3.3

category: 'Private'
method:
cleanupFdcResultsForMgc: forMgc
 "Deprecated.
  Delete the results produced by a previous findDisconnectedObjects operation.
  The forMgc argument is ignored."

 self deprecated: 'Repository>>cleanupFdcResultsForMgc:: deprecated in v3.3. Use cleanupFdcResults .'.
 ^ self cleanupFdcResults
%

method:
cleanupFdcResults
"Delete the results produced by a previous findDisconnectedObjects operation."

(Globals at: #FdcResults otherwise: nil) ifNotNil:[:t |
  t at: 1 put: 0.
  t at: 2 put: 0.
  (t at: 3) ifNotNil:[:a | a size: 0 ].
  t at: 4 put: false.
].
^true
%

method:
_createFdcResults

  (Globals at: #FdcResults otherwise: nil) ifNil:[
    Globals at: #FdcResults put: { 0 . 0 . { } . false }.
  ].
  self cleanupFdcResults .
  ^ (Globals at: #FdcResults) 
%

! cleanupMgcResultsForMgc: deleted
! basicMarkGcCandidatesFromFile ... deleted
! markGcCandidatesFromFile ... deleted

category: 'Repository Usage Reporting'
method:
pagesWithPercentFree: aPercent

"This method returns an Array containing the following statistics:

 1.  The total number of data pages processed.
 2.  The sum in bytes of unused space in all data pages.  This quantity is a
     measure of data fragmentation in the receiver.
 3.  The number of bytes in a page.
 4.  The number of data pages that have at least the specified percentage of
     unused space.
 5.  The number of data pages that have at least the specified percentage of
     unused space and contain only 1 object.
 6.  The number of data pages that contain only 1 object.
 7.  The number of pages that should be in the scavengablePages that are not.

 Do not confuse unused space on a page with free space (unused pages) in a
 Repository or Extent.  See the freeSpace and freeSpaceInExtent: methods
 for more information.

 This method requires the GarbageCollection privilege."

^ self _pagesWithPercentFree: aPercent withMaxThreads:
    self getDefaultNumThreads maxCpuUsage: 90 doScavenge: false
%

category: 'Deprecated'
method:
_pagesWithPercentFree: aPercent doScavenge: aBoolean

" Deprecated, use pagesWithPercentFree:. Also see the comments for the primitive:
    _pagesWithPercentFree:withMaxThreads:maxCpuUsage:doScavenge:"

self deprecated: 'Repository>>_pagesWithPercentFree:doScavenge: deprecated v3.4 or earlier ',
 ' use #pagesWithPercentFree:'.

^ self pagesWithPercentFree: aPercent 
%

category: 'Repository Usage Reporting'
method:
fastPagesWithPercentFree: aPercent

"Same as pagesWithPercentFree: except that the scan is performed aggressively 
 in order to complete in as little time as possible."

^ self _pagesWithPercentFree: aPercent withMaxThreads: self _aggressiveMaxThreadCount 
                              maxCpuUsage: 95 doScavenge: false
%

category: 'Repository Usage Reporting'
method:
_pagesWithPercentFree: aPercent withMaxThreads: maxThreads maxCpuUsage: percentCpu doScavenge: aBoolean

"This primitive implements pagesWithPercentFree.

 In order to execute this method you must have the GarbageCollection
 privilege.  

 If the doScavenge argument is true, you must also have System Control 
 privilege and the method resets the stone's scavengable pages list
 to the results computed, possibly repairing any corruption in the 
 scavengable pages (see stat 7 above).

 Returns an array of size 7 with the following contents:
   1. numDataPagesProcessed
   2. totalBytesFree
   3. CFG_PAGE_SIZE_BYTES
   4. numPagesWithPercentFree
   5. pages containing oneObjWithPercentFree
   6. pages containing only one object
   7. numPagesNeedScavenging

 Use of the doScavenge option is discouraged without explicit recomendations 
 from GemTalk Systems support."

<primitive: 390>
self _validatePercentage: aPercent .
maxThreads _validateClass: SmallInteger ; 
  _validateMin: 1 max: SmallInteger maximum32bitInteger .
self _validatePercentage: percentCpu .
aBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_pagesWithPercentFree:withMaxThreads:maxCpuUsage:doScavenge:
       args: { aPercent . maxThreads . percentCpu . aBoolean }
%

category: 'Garbage Collection'
method:
_setGcConfigAt: aSymbol put: aValue

"try 10 times to change the value of the specified GC configuration
 parameter.  If cannot successfully commit within 10 tries, raises an error. 
 Aborts current transaction . Returns the previous
 value of the parameter.  aSymbol must be resolvable in the UserGlobals
 for GcUser. "

| triesLeft commitResult oldVal gcUserUg |

triesLeft := 10 .
commitResult := false .
gcUserUg := ((AllUsers userWithId:'GcUser' ) resolveSymbol:#UserGlobals ) value.
[triesLeft > 0 and:[commitResult == false] ] whileTrue:[
  System beginTransaction .
  oldVal := gcUserUg at: aSymbol .
  gcUserUg at: aSymbol put: aValue .

  commitResult := System commitTransaction  .
  triesLeft := triesLeft - 1.
  commitResult 
    ifFalse:[ System abortTransaction .  System sleep: 2 . ]
    ifTrue:[ GsFile gciLogServer:'--setGcConfig: set ' , aSymbol , ' to ' , 
		aValue asString 
    ].
].
commitResult ifFalse:[ ^ self error:'Unable to modify Gc configuration'].
^ oldVal
%

! fixed 44321
category: 'Audit and Repair'
method:
objectAuditWithMaxThreads: maxThreads percentCpuActiveLimit: aPercent

"Checks all objects in GemStone for consistency.

 This method should be executed from topaz -l (the linked version of Topaz).

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.         

 This method requires SystemControl privileges.

 A GciHardBreak during this method will terminate the session.

 This method raises the error #3022 if audit detects problems, otherwise
 returns true."

| errCount |

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

errCount := self _objectAuditWithMaxThreads: maxThreads waitForLock: 60 pageBufSize: 8 
                      percentCpuActiveLimit: aPercent
                      csvFile: nil repair: false.
errCount == 0 ifFalse:[
  RepositoryError new 
     details: 'Object Audit Errors , ' , errCount asString , ' errors found' ;
     signalNotTrappable
].
^ true.
%

category: 'Audit and Repair'
method:
objectAudit
 "See comments in objectAuditWithMaxThreads:percentCpuActiveLimit:"

^ self objectAuditWithMaxThreads: self getDefaultNumThreads percentCpuActiveLimit: 90
%

category: 'Audit and Repair'
method:
objectAuditToCsvFile: aFileName
 "Similar to the basic objectAudit except that if a valid file name is provided for the 
  csvFile (comma separated values) argument then a line is written to the file for each 
  error encountered.  The lines contain the following information: 
    1. ErrorKind - the name associated with each of the errors defined above.
    2. ObjectId  - the objectId for the object with an error.
    3. ClassName  - the name of the object's class.
    4. Offset    - the offset (or other integer value, e.g. size)
    5. Reference - the reference that does not exist.
  If no errors are found, the file is deleted. "

| errCount |

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

errCount := self _objectAuditWithMaxThreads: self getDefaultNumThreads
                      waitForLock: 60 pageBufSize: 8 
                      percentCpuActiveLimit: 90
                      csvFile: aFileName repair: false.
errCount == 0 ifFalse:[
  RepositoryError new 
     details: 'Object Audit Errors , ' , errCount asString , ' errors found' ;
     signalNotTrappable
].
^ true.
%

category: 'Audit and Repair'
method:
fastObjectAudit
"Same as the #objectAudit method except the audit is performed 
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

^ self objectAuditWithMaxThreads: self _aggressiveMaxThreadCount
  percentCpuActiveLimit: 95
%

category: 'Deprecated'
method:
auditWithLimit: sizeLimit

" Deprecated, use #objectAudit instead of this method "

self deprecated: 'Repository>>auditWithLimit: deprecated in v3.2. Use objectAudit instead.'.
^ self objectAudit
%

category: 'Deprecated'
method:
repairWithLimit: sizeLimit

" Deprecated, use #repair instead of this method."
self deprecated: 'Repository>>repairWithLimit: deprecated in v3.2. Use repair instead.'. 
^self repair
%

category: 'Audit and Repair'
method:
repair
" This method requires SystemControl privileges.

  See the comments in: 
    _objectAuditWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:csvFile:repair: 
  for the kinds of repository errors that can be repaired."


System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^self _objectAuditWithMaxThreads: self getDefaultNumThreads 
                      waitForLock: 60 pageBufSize: 8 
                      percentCpuActiveLimit: 90
                      csvFile: nil repair: true.
%

! fix 41773
category: 'Garbage Collection'
method:
markForCollection
"Performs a garbage collection analysis of all permanent objects on disk.  
 Every object in the receiver that cannot be reached from AllUsers is marked 
 for subsequent reclaiming of storage space. 

 This method aborts the current transaction, empties the GcCandidates queue 
 and commits, runs the analysis while outside of a transaction and then reenters
 a transaction if the session was in a transaction at the start of this method.
 If an abort would cause unsaved changes to be lost, it does not execute and 
 signals an error, #rtErrAbortWouldLoseData.

 When this method completes successfully, signals a Warning and then
 returns the Warning.  The Warning has error text of the form
   Successful completion of Garbage Collection
     found <anInt> live objects,
     found <anInt> dead objects, occupying <anInt> bytes
 This specific Warning will satisfy a topaz 
     expectvalue true
 expected result.

 After this method completes, the GcGem process automatically reclaims the space
 occupied by the dead objects.  Space is not reclaimed until other sessions have
 either committed or aborted the transactions that were concurrent with this
 method.

 This method may fail with an error if another garbage collection is
 in progress.  This method waits up to 2 minutes for the other garbage
 collection to finish before issuing the error.
 Another garbage collection can be in progress because:
   1) an epoch garbage collection is in progress in a gc gem.
   2) a markForCollection is in progress in another session
   3) a previous epoch, or markForCollection completed the
     mark phase, but the voting on the possibly dead objects has
     not completed. Note that an admin GcGem must be running for the
     voting to complete.  Also, long running sessions that are idle and
     never aborting will prevent voting from completing.

 This method requires the GarbageCollection privilege.
 A GciHardBreak during this method terminates the session.

 This method performs the MFC in a non-aggressive manner and uses only 2 
 threads.  To complete the operation in less time, see the
 #fastMarkForCollection method.

 See also additional comments in the method:
 #_mfcWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:"

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ self markForCollectionWithMaxThreads: self getDefaultNumThreads waitForLock: 120
%

category: 'Garbage Collection'
method:
markForCollectionWait: waitTimeSeconds

"Perform a markForCollection, waiting up to waitTimeSeconds for any
 garbage collection in progress in another session to complete.

 To wait as long as necessary for the other gc to finish, pass the argument -1.
 This should be done with caution, however, as under certain conditions
 the session could appear to wait forever.  To avoid this you need to:

 1.  Make sure that other sessions are committing/aborting to allow
     voting on possible dead to complete.

 2.  Make sure that the Admin Gc gem and at least one Reclaim Gc gem are
     is running to complete processing of dead objects once the vote is
     completed.

 This method performs the MFC in a non-aggressive manner and uses only 2 
 threads.  To complete the operation in less time, see the
 #fastMarkForCollection method.

 For further documentation see markForCollection. "

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ self markForCollectionWithMaxThreads: self getDefaultNumThreads waitForLock: waitTimeSeconds
%


! fix 31566
category: 'Extent Operations
method:
shrinkExtents

"Deprecated in 3.0.
 Truncate all Extents of the Repository to remove internal free space between
 the last used page in each extent and the end of the file containing the
 extent.  Has no effect for extents on a raw disk partition.

 WARNING, you should take a full backup of your extents before 
 executing shrinkExtents.  If the machine crashes (such as a power failure)
 while shrinkExtents is executing, the repository may be unusable.

 Requires that no other user sessions be logged in.
 Stops and restarts the symbol creation session and all gc sessions .
 Because it stops the symbol creation session, this method can only
 be run as SystemUser .

 If DBF_PRE_GROW is enabled in the configuration file then this the extents
 will be grown again the next time Stone is restarted, thus cancelling the
 effect of this method. 

 Returns receiver if successful, otherwise returns a String describing
 sessions that prevented the shrink."

<primitive: 2001> "enter protected mode"
| result prot |
prot := System _protectedMode .
[
  | symStopped gcStopped errCount finallyBlock oldRemoveThresh 
    pagesNeedRemoveOfs removalThreshSym finallyDone |
  self deprecated: 'Repository>>shrinkExtents deprecated in v3.0'.
  symStopped := false .
  gcStopped := false .
  errCount := 0 .
  removalThreshSym := #StnPageMgrRemoveMinPages .
  oldRemoveThresh := System stoneConfigurationAt: removalThreshSym .
  System stoneConfigurationAt: removalThreshSym put: 0 .
  pagesNeedRemoveOfs := System cacheStatisticsDescription indexOf:
   'PagesWaitingForRemovalInStoneCount'.
  finallyDone := false .
  finallyBlock := [
      finallyDone ifFalse:[
        finallyDone := true .
        System stoneConfigurationAt: removalThreshSym put: oldRemoveThresh .    
        symStopped ifTrue:[ 					" fix 31630"
	  GsFile gciLogClient:'--shrink: restarting symbol creation session'.
	  System startSymbolCreationSession .
	  symStopped := false .
        ].
        gcStopped ifTrue:[ 
	  GsFile gciLogClient:'--shrink: restarting gc sessions'.
	  System startAllGcSessions .
	  System waitForAllGcGemsToStartForUpToSeconds:10 .
	  gcStopped := false .
        ].
        self __inProtectedMode .  "gciLog operations will have cleared it"
      ].
  ].
  [
    [ | done sesList count otherSes |
      GsFile gciLogClient:'--shrink: stopping symbol creation session'.
      symStopped := true .
      System stopSymbolCreationSession .   "stop first to fix 31606"
      GsFile gciLogClient:'--shrink: stopping gc sessions'.
      gcStopped := true .
      System stopAllGcGems.
      done := false .
      count := 1 .
      [ done ] whileFalse:[
	sesList := System currentSessions .
	sesList size == 1 ifTrue:[ 
	  done := true .
	] ifFalse:[
	  count <= 10 ifTrue:[
	    count := count + 1 .
	    System sleep: 1 .
	  ] ifFalse:[
	    otherSes := System otherSessionNames .
	    done := true
	  ].
	].
      ].
      otherSes == nil ifTrue:[ | needRemove |
	GsFile gciLogClient:'--shrink: doing simple commits'.
	3 timesRepeat:[
	  System _simpleCommitForReclaim: false .
	  System abortTransaction .
	  System sleep: 3 .
	  System abortTransaction .
	  System sleep: 3 .
	].
	GsFile gciLogClient:'--shrink: attempting startCheckpointSync '.
	System startCheckpointSync ifFalse:[
	  result := 'ERROR, false from startCheckpointSync. ' .
	].  
	System abortTransaction .
	System sleep: 15 . 
	needRemove := 1 .
	count := 0 .
	GsFile gciLogClient:'--shrink: waiting for pagesNeedRemove=0 ' .
	[ needRemove > 0 and:[count < 10]] whileTrue:[ 
	  System sleep: 3 .
	  System abortTransaction .
	  needRemove := (System cacheStatistics: 1) at: pagesNeedRemoveOfs .
	  count := count + 1
	].
	needRemove > 0 ifTrue:[
	  result == nil ifTrue:[ result := String new ].
	  result addAll: ' ERROR, pagesNeedRemove = ' , needRemove asString .
	].
	self __inProtectedMode .  "gciLog operations will have cleared it"
	self _shrinkExtents .
	result == nil ifTrue:[ result := self ].
      ] ifFalse:[
	result := 'ERROR, sessions preventing shrink:' , otherSes .
      ].
    ] ensure: finallyBlock .
  ] onSynchronous: Error do:[:ex |
    errCount := errCount + 1 .
    " GsFile gciLogClient:'--shrink: error count ' , errCount asString . "
    errCount == 1 ifTrue:[ finallyBlock value ] .
    symStopped := false .
    gcStopped := false .
    ex outer
  ].
] ensure: [
  prot _leaveProtectedMode
].
^ result
%

category: 'Private'
method:
_shrinkExtents

<protected primitive: 427>

self _primitiveFailed: #shrinkExtents .
self _uncontinuableError
%

category: 'Extent Operations'
method:
createExtent: extentFilename

"Creates a new Extent with a file named extentFilename (a String).  The new
 Extent has no maximum size.

 This method updates the DBF_EXTENT_NAMES Stone option.  It does not require
 the system to be in single-user mode.

 If the given file already exists, then this method generates an error and the
 given Extent is not added to the logical Repository.

 Creating an extent with this method bypasses any setting the user may have
 specified for the DBF_PRE_GROW option at system startup.  As extents created
 with this method have no maximum size, their files cannot be grown to an
 initial size.

 If GemStone is being run using weighted disk resource allocation, then
 the new Extent will be given a weight equal to the average weight of all
 other extents.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser."

^ self createExtent: extentFilename withMaxSize: 0
%

! fix 33802
! fix 40859
category: 'Private'
method:
_primCreateExtent: extentFilename withMaxSize: aSize startNewReclaimGem: aBool

"Creates a new Extent with the given extentFilename (aString) and sets the
 maximum size of that Extent to the given size. 
 The parameter for startNewReclaimGem is obsolete in v3.2."

<primitive: 284>

extentFilename _validateKindOfClass: String.
aSize _validateClass: SmallInteger.
aBool _validateClass: Boolean .
" aSize out of range errors generated in the primitive "
^ self _primitiveFailed: #createExtent:withMaxSize:startNewReclaimGem:
       args: { extentFilename . aSize . aBool }
%

! Fix 44670 - min extent size is now 16 MB
category: 'Deprecated'
method:
createExtent: extentFilename withMaxSize: aSize startNewReclaimGem: aBool

"Creates a new Extent with the given extentFilename (aString) and sets the
 maximum size of that Extent to the given size. 

 If aSize == 0 , then the maximum physical size of the extent is limited
 to disk space or 33554G bytes , whichever comes first . If aSize > 0,
 it specifies the maximum physical size of the file in megabytes.  The 
 minimum size of a new extent is 16 MB.  Attempts to create an extent of
 a smaller size will raise an error.  The actual maximum size of the new
 extent will be equal to aSize if and only if aSize is a multiple of 16.
 Otherwise the actual maximum size will be the next lowest multiple of 16.
 For example, an extent created with a maximum size of 260 MB really has a
 maximum size of only 256 MB.

 This method updates the DBF_EXTENT_NAMES and DBF_EXTENT_SIZES Stone options.
 It does not require the system to be in single-user mode.

                                  Note:
   The extent is created with the default ownership and permissions of the
   Stone process.  If this is not the same as the ownership and permissions of
   the other extents, then Unix utilities must be used to change the ownership
   or permissions of the new file; such changes may be made without stopping
   the Stone, and should be made a soon as possible, to avoid other sessions
    encountering authorization errors.

 If the given file already exists, then this method generates an error and the
 given Extent is not added to the logical Repository.

 If the Stone option DBF_PRE_GROW is set to true, then this method will cause
 the newly created extent's file to be grown to the given size.  If the grow
 fails, then this method returns an error and the new Extent is not added to the
 logical Repository.

 If GemStone is being run using weighted disk resource allocation, then
 the new Extent will be given a weight equal to the average weight of all
 other Extents.

 If aBool is true, then a new reclaim gem is started to reclaim pages from the
 newly created extent.  If aBool is false, then no reclaim gem is started.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser."

self deprecated: 'Repository>>createExtent:withMaxSize:startNewReclaimGem: 
    deprecated in v3.2, reclaimgems no longer extent-specific; use createExtent:withMaxSize:'.
 ^self _primCreateExtent: extentFilename withMaxSize: aSize startNewReclaimGem: aBool
%

! fix 33802
! Fix 44670 - min extent size is now 16 MB
category: 'Extent Operations'
method:
createExtent: extentFilename withMaxSize: aSize

"Creates a new Extent with the given extentFilename (aString) and sets the
 maximum size of that Extent to the given size.

 If aSize == 0 , then the maximum physical size of the extent is limited
 to disk space or 33554G bytes , whichever comes first . If aSize > 0 ,
 it specifies the maximum physical size of the file in megabytes.  The 
 minimum size of a new extent is 16 MB.  Attempts to create an extent of
 a smaller size will raise an error.  The actual maximum size of the new
 extent will be equal to aSize if and only if aSize is a multiple of 16.
 Otherwise the actual maximum size will be the next lowest multiple of 16.
 For example, an extent created with a maximum size of 260 MB really has a
 maximum size of only 256 MB.

 This method updates the DBF_EXTENT_NAMES and DBF_EXTENT_SIZES Stone options.
 It does not require the system to be in single-user mode.

                                  Note:
   The extent is created with the default ownership and permissions of the
   Stone process.  If this is not the same as the ownership and permissions of
   the other extents, then Unix utilities must be used to change the ownership
   or permissions of the new file; such changes may be made without stopping
   the Stone, and should be made a soon as possible, to avoid other sessions
    encountering authorization errors.

 If the given file already exists, then this method generates an error and the
 given Extent is not added to the logical Repository.

 If the Stone option DBF_PRE_GROW is set to true, then this method will cause
 the newly created extent's file to be grown to the given size.  If the grow
 fails, then this method returns an error and the new Extent is not added to the
 logical Repository.

 If GemStone is being run using weighted disk resource allocation, then
 the new Extent will be given a weight equal to the average weight of all
 other Extents.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser."

 ^self _primCreateExtent: extentFilename withMaxSize: aSize startNewReclaimGem: false 
%

category: 'Extent Operations'
method:
numberOfExtents

  <primitive: 428>

"Returns the number of active extents."

  self _primitiveFailed: #numberOfExtents .
  self _uncontinuableError
%

category: 'Extent Operations'
method:
validateExtentId: anExtentId

"Returns the argument if it is valid, otherwise generates an error."

| maxExtentId |
anExtentId == nil ifTrue:[ ^ anExtentId ] .
anExtentId _validateClass: SmallInteger.
maxExtentId := self numberOfExtents .
(anExtentId < 1 or:[ anExtentId > maxExtentId]) ifTrue:[
   anExtentId _error: #rtErrArgOutOfRange args:{ 1 . maxExtentId }.
   anExtentId < 1 ifTrue:[ ^ 1 ].
   anExtentId > maxExtentId ifTrue:[ ^ maxExtentId].
].
^ anExtentId .
%

category: 'Listing Instances'
method:
listInstancesToHiddenSet: aClassOrArray withMaxThreads: maxThreads maxCpuUsage: aPercentage

"Scans the entire Repository and generates a list of instances into hidden set 1.
 The previous state of the hidden set 1 is cleared.

 aClassOrArray is expected to be either an Array of classes or a Class.
 Does not include objects in VM's temporary object memory.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData. The entire
 repository is scanned once.

 If aClassOrArray is a Class, this method returns the number of instances of
 aClassOrArray which were added to hidden set 1.

 If aClassOrArray is an Array of classes, this method returns an array of pairs with
 a Class at the odd numbered elements and a count at the even numbered elements.
 The count is the number of instances of the class that were added to hidden set 1.
 All instances of all classes present in aClassOrArray are combined and added to
 hidden set 1.

 Note that the order in which classes appear in the result array may differ
 from the order which they appear in aClassOrArray.
 
 This method is kept for compatibility, ignores withMaxThreads: and maxCpuUsage: args
 and executes fastListInstancesToHiddenSet: .
"

^ self fastListInstancesToHiddenSet: aClassOrArray.
%

category: 'Private'
method:
_aggressiveMaxThreadCount

"Answer the number of threads to start for an aggressive, 
 multithreaded repository scan.  The calculation below is really just a
 guess.  We used twice the number of CPU cores on the host since it is expected
 that threads will often be in I/O wait when scanning large repositories."

^ 2 max: ((System maxSessionId - 7) min: ((System hostCpuCount * 2) min: 64))
%

category: 'Listing Instances'
method:
listInstancesToHiddenSet: aClassOrArray

"Scans the entire Repository and generates a list of instances into hidden set 1.
 The previous state of the hidden set 1 is cleared.

 aClassOrArray is expected to be either an Array of classes or a Class.
 Does not include objects in VM's temporary object memory.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData. The entire
 Repository is scanned once.

 If aClassOrArray is a Class, this method returns the number of instances of
 aClassOrArray which were added to hidden set 1.

 If aClassOrArray is an Array of classes, this method returns an array of pairs with
 a Class at the odd numbered elements and a count at the even numbered elements.
 The count is the number of instances of the class that were added to hidden set 1.
 Note that the order in which classes appear in the result array may differ
 from the order which they appear in aClassOrArray.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To 
 complete the operation in less time, see the #fastListInstancesToHiddenSet:
 method."

| arr |
arr := aClassOrArray.
arr isBehavior ifTrue: [ arr := { arr } ].

^ self _buildListInstResult: (self allInstances: arr)
%


category: 'Listing Instances'
method:
fastListInstancesToHiddenSet: aClassOrArray

"Same as the #listInstancesToHiddenSet: method except the scan is performed 
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

| arr |
arr := aClassOrArray.
arr isBehavior ifTrue: [ arr := { arr } ].

^ self _buildListInstResult: (self fastAllInstances: arr)
%

category: 'Private'
method:
_allInstances: anArray toDirectory: aString fast: isFast
| allInst result |

anArray _validateInstanceOf: Array.
aString _validateKindOfClass: String.

isFast ifTrue: [ allInst := self fastAllInstances: anArray ]
       ifFalse: [ allInst := self allInstances: anArray ].

result := Array new.
1 to: allInst size do: [ :i |  | each fileName eachClass |
  each := allInst at: i.
  eachClass := each at: 1.
  aString isNil ifFalse: [
    fileName := aString , '/' , eachClass asString , '-' , eachClass asOop asString, '-instances.bm'.
    GsFile removeServerFile: fileName.
    (each at: 2) writeToFile: fileName.
  ].
  result add: eachClass.
  result add: (each at: 2) size.
].

^ result
%

category: 'Listing Instances'
method:
listInstances: anArray toDirectory: aString

"Scan the repository for the instances of classes in anArray and write
 the results to binary bitmap files in the directory specified by
 aString.  Binary bitmap files have an extension of .bm and may be
 loaded into hidden sets using class methods in class System or
 GsBitmap readFromFile: method.
 
 aString must specify a path to a writable directory.

 Does not include objects in VM's temporary object memory.

 Bitmap files are named as follows:

 <ClassName>-<classOop>-instances.bm

 where className is the name of the class and classOop is the object
 ID of the class.

 The result is an Array of pairs.  For each element of the argument anArray,
 the result array contains  <aClass>, <numberOfInstancesFound>.
 The numberOfInstances is the total number written to the output bitmap file
 
 Note: the order of classes in the result array may differ from that of 
 the argument anArray.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.  The entire
 Repository is scanned at least once.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To 
 complete the operation in less time, see the #fastListInstances:toDirectory:
 method."

^ self _allInstances: anArray toDirectory: aString fast: false
%

category: 'Listing Instances'
method:
fastListInstances: anArray toDirectory: aString

"Same as the #listInstances:toDirectory: method except the scan is performed 
 aggressively in order to complete in as little time as possible,
 and does not include in-memory objects.
 
 This method may consume most or all host system resources while it
 is in progress."

^ self _allInstances: anArray toDirectory: aString fast: true
%

category: 'Listing Instances'
method:
listInstances: anArray limit: aSmallInt

"Returns a list of instances on the receiver that belong to one of the
 classes listed in anArray. .  Includes in-memory objects.

 aSmallInt is the maximum number of instances to report for each class,
 or 0 for unlimited . 

 The result is an Array of pairs, for each element of the argument anArray,
 the result array contains  <numberOfInstances>, <arrayOfInstances> .
 The numberOfInstances is the total number in the repository .
 The arrayOfInstances will be limited by the argument aSmallInt .

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.  The entire        
 Repository is scanned at least once.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To 
 complete the operation in less time, see the #fastListInstances:limit:
 method."

^ self listInstances: anArray limit: aSmallInt toDirectory: nil 
     withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90 memoryOnly: false
%

category: 'Listing Instances'
method:
fastListInstances: anArray limit: aSmallInt

"Same as the #listInstances:limit: method except the scan is performed 
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is in progress."

^ self listInstances: anArray limit: aSmallInt toDirectory: nil 
      withMaxThreads: self _aggressiveMaxThreadCount maxCpuUsage: 95
	memoryOnly: false 
%


category: 'Listing Instances'
method:
listInstances: anArray limit: aSmallInt toDirectory: directoryString
 withMaxThreads: maxThreads maxCpuUsage: aPercentage 

 ^ self listInstances: anArray limit: aSmallInt toDirectory: directoryString
   withMaxThreads: maxThreads maxCpuUsage: aPercentage memoryOnly: false
%

method:
listInstances: anArray limit: aSmallInt toDirectory: directoryString
 withMaxThreads: maxThreads maxCpuUsage: aPercentage memoryOnly: memOnlyBool

"If directoryString == nil, result includes in-memory objects.

 If memBool == true, result contains only the in-memory objects,
  and maxThreads and aPercentage  are ignored."

| inputSet resultInSetOrder result code scanBlk |

memOnlyBool ifFalse:[ 
  System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ]
]. 
inputSet := self _arrayOfClassesAsSet: anArray .
inputSet size < 1 ifTrue:[ ^ { } ] .

memOnlyBool ifFalse:[
 scanBlk := [ :scanSetThisTime | | scanKind |
     scanKind := directoryString ifNotNil:[ 2"OP_LIST_INST_TO_FILES - OP_FIRST_SCAN_KIND" ] 
                                    ifNil:[ 0"OP_LIST_INSTANCES - OP_FIRST_SCAN_KIND" ] .
     self _scanPomWithMaxThreads: maxThreads waitForLock: 60 pageBufSize: 8
                             percentCpuActiveLimit: aPercentage  
                             identSet: scanSetThisTime limit: aSmallInt
                             scanKind: scanKind 
                             toDirectory: directoryString 
  ].
].
code := directoryString ifNotNil:[ 2 ] ifNil:[ 0 ].
inputSet := IdentitySet withAll: anArray .

resultInSetOrder:= self _doListInstancesFrom: inputSet with: scanBlk 
			includeMemory:  directoryString == nil limit: aSmallInt.

directoryString ifNotNil:[
  result := resultInSetOrder "primitive wrote the files, so we are done"
] ifNil:[ | inputArraySize |
  inputArraySize := anArray size .
  result := Array new: inputArraySize * 2.
  1 to: inputArraySize do:[:j| | soIdx resIdx anObj |
      anObj := anArray at: j .
      soIdx := (inputSet _offsetOf: anObj) * 2 .
      resIdx := j * 2 .
      result at: resIdx - 1 put: ( resultInSetOrder at: soIdx - 1 ) ."totalCount"
      result at: resIdx put: ( resultInSetOrder at: soIdx) . "array of instances"
  ].
].
^ result
%

category: 'Private'
method:
_arrayOfClassesAsSet: anArray

1 to: anArray size do:[:n| | each |
  each := anArray at: n.
  each _validateClass: Behavior.
].
^ IdentitySet withAll: anArray .
%

! New code for 42245
category: 'Private'
method:
_doListInstancesFrom: inputSet with: diskBlock includeMemory: inMemoryBool limit: aSmallInt
| inputSetSize result limit |
limit := aSmallInt.
limit == 0 ifTrue: [ limit := SmallInteger maximum32bitInteger ].
inputSetSize := inputSet size .
result := { }  .  "pairs  size, array of instances, in order of input set"
diskBlock ifNotNil:[
  result addAll: (diskBlock value: inputSet ).
 ].

inMemoryBool ifTrue:[ | memResult idx |
  idx := 1.
  memResult := self _listInstancesInMemory: (Array withAll: inputSet) limit: aSmallInt.
  1 to: memResult size do:[:j | | elem memRes |
    memRes := memResult at: j .
    (elem := result atOrNil: idx) ifNil:[
       result at: idx put: memRes size ;
              at: idx + 1 put: memRes .
    ] ifNotNil:[ 
       elem := result at: idx + 1. 
       (elem := IdentitySet withAll: elem) addAll: memRes .
       elem := elem asArray.
       elem size > limit ifTrue: [ elem size: limit ].
       result at: idx put: elem size ;
              at: idx + 1 put: elem.
       
    ].
    idx := idx + 2
  ].
].
^ result
%

category: 'Listing Instances'
method:
listInstances: anArray

"Returns a list of instances on the receiver that belong to one of the
 classes listed in anArray.  The result of this method is an Array of Arrays,
 where the contents of each inner array consists of all instances 
 whose class is equal to the corresponding element in anArray.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.         

 If anArray contains multiple occurrences of a class, then the result
 will contain corresponding multiple occurrences of the same Array that lists
 the instances of that class.

 If anArray contains an element that is not a kind of Behavior, an error
 is generated.

 Scans the entire Repository at least once.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To 
 complete the operation in less time, see the #fastListInstances:  method."

| reportArr res |
reportArr := self listInstances: anArray limit: 0 .
res := Array new: anArray size .
1 to: anArray size do:[:j |
  res at:j put: (reportArr at: j * 2 ) . "array of instances"
].
^ res
%

category: 'Listing Instances'
method:
fastListInstances: anArray

"Same as the #listInstances: method except the scan is performed 
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

| reportArr res |
reportArr := self fastListInstances: anArray limit: 0 .
res := Array new: anArray size .
1 to: anArray size do:[:j |
  res at:j put: (reportArr at: j * 2 ) . "array of instances"
].
^ res
%

method:
listInstancesInMemory: anArray

"Returns a list of instances of the classes specified in anArray
 that are found in temporary object memory.

 The result of this method is an Array of Arrays, where the 
 contents of each inner array consists of all instances
 of the corresponding element in anArray.

 The result contains in-memory objects only.  Objects which are not
 currently in memory are not analyzed.

 Does not abort the current transaction."

| pairs res |
pairs := self listInstances: anArray limit: 0 toDirectory: nil 
      withMaxThreads: 1 maxCpuUsage: 95 memoryOnly: true .
res := Array new: anArray size .
1 to: anArray size do:[:j |
  res at:j put: (pairs at: j * 2 ) . "array of instances"
].
^ res
%

category: 'Private'
method:
_listInstancesInMemory: anArray limit: aSmallInt

"anArray may not contain duplicates.
 Returns an Array, each element is an Array of instances for
 corresponding element of anArray."

<primitive: 389>
anArray _validateInstanceOf: Array.
aSmallInt _validateClass: SmallInteger; _validateMin: 0 max: SmallInteger maximum32bitInteger .
^ self _primitiveFailed: #_listInstancesInMemory: args: { anArray . aSmallInt } 
%

! Move primitive to a new method for 43315 ; fixed 46397
category: 'Private'
method:
_listReferencesInMemory: anArray nonStubbedOnly: aBoolean

"An error is signaled if anArray contains duplicates.
 Returns an Array, each element is an Array of instances for
 corresponding element of anArray."

<primitive: 393>
anArray _validateInstanceOf: Array.
aBoolean _validateClass: Boolean .
^ self _primitiveFailed: #listReferencesInMemory:nonStubbedOnly: args: { anArray . aBoolean } 
%

! Fix 43315 - disallow duplicates in the list
category: 'Listing References'
method:
listReferencesInMemory: anArray

"Returns a list of instances in temporary object memory that 
 have a reference to one of the objects specified in anArray.  
 The result of this method is an Array of Arrays, where the 
 contents of each inner array consists of all instances
 that have a reference to the corresponding element in anArray.
 
 The result contains in-memory objects only.  Objects which are not
 currently in memory are not analyzed.

 An error is raised if anArray contains the same object more than once.

 Does not abort the current transaction."

| resultArray |

resultArray := self _listReferencesInMemory: anArray nonStubbedOnly: false .

"nil means the primitive detected at least one duplicate entry.  Raise an error and bail."
resultArray ifNil:[  ArgumentError new object: anArray ;
    signal:'Array of references to search for contains 1 or more duplicates'
].
^ resultArray
%

category: 'Class Management'
method:
_loadedClasses: includeModulesBoolean

"Returns an Array of all Classes currently loaded in memory.
 If includeModulesBoolean==true, the result also includes instances of Module.
 Result does not include meta classes, meta modules, or Ruby virtual classes."

<primitive: 869> 
includeModulesBoolean _validateClass: Boolean .
self _primitiveFailed: #_loadedClasses: args: { includeModulesBoolean } 
%

category: 'Listing References'
method:
listReferences: anArray 

"Returns a list of instances in the Repository that have a reference to one of
 the objects specified in anArray.  The result of this method is an Array of
 Arrays, where the contents of each inner array contains a target object at offset 1 and
 a GsBitmap that contains the references to that object at offset 2.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.         

 The result contains only committed objects. 

 The scan is done in an non-aggressive manner and uses only 2 threads.  To 
 complete the operation in less time, see the #fastListReferences
 method."

^ self _buildAllRefsResult: (self allReferences: anArray) forInput: anArray
%

category: 'Listing References'
method:
listReferences: anArray withLimit: aSmallInt

"Returns a list of instances in the Repository that have a reference to one of
 the objects specified in anArray.  The result of this method is an Array of
 Arrays, where the contents of each inner array contains a target object at offset 1 and
 a GsBitmap that contains the references to that object at offset 2.
 The number in each inner array is limited to approximately aSmallInt.  

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.         

 The result contains only committed objects. 

 The scan is done in an non-aggressive manner and uses only 2 threads.  To 
 complete the operation in less time, see the #fastListReferences
 method."

^ self _buildAllRefsResult: (self allReferences: anArray) forInput: anArray withLimit: aSmallInt
%

category: 'Listing References'
method:
fastListReferences: anArray 

"Same as the #listReferences: method except the scan is performed 
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is in progress."

^ self _buildAllRefsResult: (self fastAllReferences: anArray)  forInput: anArray
%
      
! Bug 42710 - must use a SortedCollection instead of IdentitySet to handle 
!             a nil element in anArray.
category: 'Listing References'
method:
listReferences: anArray withLimit: aSmallInt withMaxThreads: maxThreads
 maxCpuUsage: aPercentage

"Returns a list of instances in the Repository that have a reference to one of
 the objects specified in anArray.  The result of this method is an Array of
 Arrays, where the contents of each inner array contains a target object at offset 1 and
 a GsBitmap that contains the references to that object at offset 2.
 The number in each inner array is limited to approximately aSmallInt.  

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.         

 The result contains both permanent and temporary objects.  The temporary
 objects found may vary from run to run.

 anArray is expected to contain no duplicates.  An ArgumentError is raised if
 duplicate entries are detected.

 This method is very expensive.  It scans the entire Repository and looks
 at every instance variable of every object."

| outerArr sortedArray resultArray sortedCollection |

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

"Sort by oop order."
sortedCollection := SortedCollection sortBlock:[:a :b| a asOop < b asOop].
sortedCollection addAll: anArray.
sortedArray := Array withAll: sortedCollection .

outerArr := self _listReferencesWithMaxThreads: maxThreads waitForLock: 60
      pageBufSize: 8 percentCpuActiveLimit: aPercentage
      targets: sortedArray withLimit: aSmallInt.

"nil means the primitive detected a duplicate entry.  Raise an error and bail."
outerArr ifNil:[  ArgumentError new object: anArray ;
    signal:'Array of references to search for contains 1 or more duplicates'
].

"Now truncate the inner result arrays as needed and build the outer result array
 to match the order of the argument array."
resultArray := Array new: outerArr size.
1 to: outerArr size  do:[ :j| | innerArr index |
    innerArr := outerArr at: j .
    (aSmallInt > 0 and:[ innerArr size > aSmallInt ])
      ifTrue:[ innerArr size: aSmallInt ].
    index := anArray indexOfIdentical: (sortedArray at: j).
    resultArray at: index put: innerArr.
].
^ resultArray
%
      
category: 'Private'
method:
_listReferencesWithMaxThreads: maxThreads waitForLock: lockWaitTime 
      pageBufSize: aBufSize percentCpuActiveLimit: percentCpu
      targets: anArray withLimit: aSmallInt

"Private. Returns a list of instances in the Repository that have a 
 reference to one of the objects specified in the targets array.  The 
 result of this method is an  Array of Arrays, where the contents of 
 each inner array consists of instances that have a reference to the 
 corresponding element in anArray.  

 This method aborts the current transaction.

 The number in inner array is limited to approximately aSmallInt,
 and may exceed aSmallInt by the maximum number of objects in a data page.
 If aSmallInt is <= 0, the result size is unlimited. 

 The result contains only permanent objects.

 This primitive method performs a multithreaded scan of the Repository.

 See _scanPomWithMaxThreads primitive for a description of the multithreaded args.

 anArray must be sorted in OOP order and not contain any duplicate entries.
 
 The maximum size of anArray is no longer restricted to 2034 elements.
"

<primitive: 899>
 | maxInt |
 maxInt := SmallInteger maximum32bitInteger .
 maxThreads _validateClass: SmallInteger; _validateMin: 1 max: maxInt .
 lockWaitTime _validateClass: SmallInteger; _validateMin: -1 max: maxInt .
 aBufSize _validateClass: SmallInteger ; _validateIsPowerOf2 .
 self _validatePercentage: percentCpu .
 aSmallInt _validateClass: SmallInteger ; _validateMin: 0 max: SmallInteger maximumValue . 
 anArray _validateInstanceOf: Array.
 ^ self _primitiveFailed: 
    #_listReferencesWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:targets:withLimit:
    args: {  maxThreads . lockWaitTime . aBufSize . 
             percentCpu . anArray . aSmallInt }
%

category: 'Deprecated'
method:
listObjectsInSegments: anArray 
"Deprecated"

self deprecated: 'Repository>>listObjectsInSegments: deprecated v3.0. 
Use #listObjectsInObjectSecurityPolicies: instead.'.
^ self _buildListInstResult: (self allObjectsInObjectSecurityPolicies: anArray)
%

category: 'Listing By Security Policy'
method:
listObjectsInObjectSecurityPolicies: anArray 

"Returns a list of objects in the receiver that have the specified 
 objectSecurityPolicyIds.  The result of this method is an Array
 of Arrays, where the contents of each inner array consists of
 objects whose objectSecurityPolicyId is the objectSecurityPolicyId of
 of the corresponding element in anArray.

 anArray must be an Array of objectSecurityPolicyIds.  The maximum size
 of anArray is no longer restricted to 2034 elements.

 The value zero can be used to indicate you wish to get the objects that
 have no objectSecurityPolicyId.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.         

 The result contains both permanent and temporary objects.  The temporary
 objects found may vary from run to run.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To 
 complete the operation in less time, see the #fastListInstances:toDirectory:
 method."

^ self listObjectsInObjectSecurityPolicies: anArray withLimit: 0
%

category: 'Listing By Security Policy'
method:
fastListObjectsInObjectSecurityPolicies: anArray 

"Same as the #listObjectsInObjectSecurityPolicies: method except the scan
 is performed aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is
 in progress."

^ self fastListObjectsInObjectSecurityPolicies: anArray withLimit: 0
%

category: 'Deprecated'
method:
_validateSegmentId: anObjectSecurityPolicyId 
"Deprecrated, use #_validateObjectSecurityPolicyId: instead of this method."

self deprecated: 'Repository>>_validateSegmentId: deprecated v3.4 or earlier ',
 ' use #_validateObjectSecurityPolicyId:'.
^ self _validateObjectSecurityPolicyId: anObjectSecurityPolicyId 
%

category: 'Private'
method:
_validateObjectSecurityPolicyId: anObjectSecurityPolicyId 

"Generates an error if anObjectSecurityPolicyId is not a SmallInteger
 >= 0 and <= self size .  
 Note that objectSecurityPolicyId 0 specifies world write,
 and does not have an associated GsObjectSecurityPolicy . "
| sz |
anObjectSecurityPolicyId _validateClass: SmallInteger .
(anObjectSecurityPolicyId < 0 or:[ anObjectSecurityPolicyId > (sz := self size) ]) 
  ifTrue:[anObjectSecurityPolicyId _error: #rtErrArgOutOfRange args:{ 0 . sz }
].
% 

category: 'Deprecated'
method:
listObjectsInSegmentToHiddenSet: anObjectSecurityPolicyId
"Deprecated"

self deprecated: 'Repository>>listObjectsInSegmentToHiddenSet: deprecated v3.0. 
Use #listObjectsInObjectSecurityPolicyToHiddenSet: instead.'.

^ self listObjectsInObjectSecurityPolicyToHiddenSet: anObjectSecurityPolicyId
%

category: 'Listing By Security Policy'
method:
listObjectsInObjectSecurityPolicyToHiddenSet: anObjectSecurityPolicyId

"Scans the entire Repository and generates a list of objects that have specified 
 objectSecurityPolicyId into hidden set 1 .  The previous state of hidden set 1
 is cleared.

 The value zero can be used to indicate you wish to get the objects that
 have no objectSecurityPolicyId.

 Returns the number of instances found.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData. The entire
 repository is scanned once.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To 
 complete the operation in less time, see the 
 #fastListObjectsInObjectSecurityPolicyToHiddenSet: method."

self _validateObjectSecurityPolicyId: anObjectSecurityPolicyId .
^ self _listObjectsInObjectSecurityPolicies: { anObjectSecurityPolicyId }  
       limit: 0 
       scanKind: 5 "OP_LIST_OBJS_IN_SEC_POLICY_TO_HIDDEN-OP_FIRST_SCAN_KIND"
       toDirectory: nil withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90
%

category: 'Listing By Security Policy'
method:
fastListObjectsInObjectSecurityPolicyToHiddenSet: anObjectSecurityPolicyId

"Same as the #listObjectsInObjectSecurityPolicyToHiddenSet: method except the scan is
 performed aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is in progress."

self _validateObjectSecurityPolicyId: anObjectSecurityPolicyId .
^ self _listObjectsInObjectSecurityPolicies: { anObjectSecurityPolicyId }  
       limit: 0 
       scanKind: 5 "OP_LIST_OBJS_IN_SEC_POLICY_TO_HIDDEN-OP_FIRST_SCAN_KIND"
       toDirectory: nil 
       withMaxThreads: self _aggressiveMaxThreadCount maxCpuUsage: 95
%

category: 'Deprecated'
method:
listObjectsInSegments: anArray toDirectory: aString
"Deprecated"

self deprecated: 'Repository>>listObjectsInSegment:toDirectory: deprecated v3.0. 
Use #listObjectsInObjectSecurityPolicies:toDirectory: instead.'.

^ self listObjectsInObjectSecurityPolicies: anArray toDirectory: aString.
%

category: 'Listing By Security Policy'
method:
listObjectsInObjectSecurityPolicies: anArray toDirectory: aString

"Scan the repository for objects that have the specified objectSecurityPolicyIds
 and writes the results to binary bitmap files in the directory specified by
 aString.  Binary bitmap files have an extension of .bm and may be
 loaded into hidden sets using class methods in class System.
 aString must specify a path to a writable directory.

 anArray must be an Array of objectSecurityPolicyIds.  The maximum size
 of anArray is no longer restricted to 2034 elements.

 The value zero can be used to indicate you wish to get the objects that
 have no objectSecurityPolicyId.

 Bitmap files are named as follows:  
   segment<objectSecurityPolicyId>-objects.bm
 where objectSecurityPolicyId is an element of anArray .

 The result is an Array of pairs.  For each element of the argument anArray,
 the result array contains  <objectSecurityPolicyId>, <numberOfInstancesFound>.
 The numberOfInstances is the total number written to the output bitmap file.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData."

aString _validateKindOfClass: String .
^ self _listObjectsInObjectSecurityPolicies: anArray limit: 0 
       scanKind: 6"OP_LIST_OBJS_IN_SEC_POLICY_TO_FILES-OP_FIRST_SCAN_KIND"
       toDirectory: aString withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90
%

category: 'Listing By Security Policy'
method:
fastListObjectsInObjectSecurityPolicies: anArray toDirectory: aString

"Same as the #listObjectsInObjectSecurityPolicies:toDirectory: method
 except the scan is performed aggressively in order to complete in as
 little time as possible.

 This method may consume most or all host system resources while it is in progress."

aString _validateKindOfClass: String .
^ self _listObjectsInObjectSecurityPolicies: anArray limit: 0 
       scanKind: 6"OP_LIST_OBJS_IN_SEC_POLICY_TO_FILES-OP_FIRST_SCAN_KIND"
       toDirectory: aString withMaxThreads: self _aggressiveMaxThreadCount
       maxCpuUsage: 95
%

category: 'Deprecated'
method:
listObjectsInSegments: anArray withLimit:  aSmallInt
"Deprecrated"

  self deprecated: 'Repository>>listObjectsInSegments:withLimit: deprecated v3.0. 
Use #listObjectsInObjectSecurityPolicies:withLimit: instead.'.
^ self listObjectsInObjectSecurityPolicies: anArray withLimit:  aSmallInt
%

category: 'Listing By Security Policy'
method:
listObjectsInObjectSecurityPolicies: anArray withLimit:  aSmallInt

"Returns a list of objects in the receiver that have the specified
 objectSecurityPolicyIds.  The result of this method is an Array
 of Arrays, where the contents of each inner array consists of
 objects whose objectSecurityPolicyId is the objectSecurityPolicyId of
 of the corresponding element in anArray.  The number in each inner
 array is limited to approximately aSmallInt.  

 anArray must be an Array of objectSecurityPolicyIds.  The maximum size
 of anArray is no longer restricted to 2034 elements.

 The value zero can be used to indicate you wish to get the objects that
 have no objectSecurityPolicyId.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.         

 The result contains both permanent and temporary objects.  The temporary
 objects found may vary from run to run.

 This method is very expensive.  It scans the entire Repository and looks
 at the object header of every object."

^ self _listObjectsInObjectSecurityPolicies: anArray limit: aSmallInt 
      scanKind: 4 "OP_LIST_OBJS_IN_SEC_POLICY-OP_FIRST_SCAN_KIND"
      toDirectory: nil withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90
%


category: 'Listing By Security Policy'
method:
fastListObjectsInObjectSecurityPolicies: anArray withLimit:  aSmallInt

"Same as the #listObjectsInObjectSecurityPolicies:withLimit: method except
 the scan is performed aggressively in order to complete in as little time
 as possible.

 This method may consume most or all host system resources while it is in progress."

^ self _listObjectsInObjectSecurityPolicies: anArray limit: aSmallInt 
      scanKind: 4 "OP_LIST_OBJS_IN_SEC_POLICY-OP_FIRST_SCAN_KIND"
      toDirectory: nil withMaxThreads: self _aggressiveMaxThreadCount
      maxCpuUsage: 95
%

category: 'Private'
method:
_listObjectsInObjectSecurityPolicies: anArray limit: aLimit scanKind: kind
 toDirectory: aStringOrNil withMaxThreads: maxThreads maxCpuUsage: aPercentage

 "aLimit is a SmallInteger, max number of objects to report, or 0 for unlimited.

 anArray must be an Array of objectSecurityPolicyIds.  The maximum size
 of anArray is no longer restricted to 2034 elements."

| sortedSegIds inputSegIds primResult primResultOfs result |

System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

sortedSegIds := SortedCollection new .
inputSegIds := { }  .
anArray _validateInstanceOf: Array .
1 to: anArray size do:[ :k | |segId |
  segId := anArray at: k .
  self _validateObjectSecurityPolicyId:  segId .
  inputSegIds add: segId .
  sortedSegIds add: segId
].
primResult := self _scanPomWithMaxThreads: maxThreads waitForLock: 90 pageBufSize: 8 
                percentCpuActiveLimit: aPercentage
                identSet: (IdentitySet withAll: sortedSegIds)
                limit: aLimit scanKind: kind toDirectory: aStringOrNil .

kind == 4 ifFalse:[ ^ primResult ].
result := Array new: anArray size * 2 .

primResultOfs := 1 .
1 to: sortedSegIds size do:[:segIdOfs | | inOfs |
  inOfs := inputSegIds indexOfIdentical: (sortedSegIds at: segIdOfs) .
  result at: (inOfs * 2) -1  put: (primResult at: primResultOfs ) .
  result at: (inOfs * 2)   put: (primResult at: primResultOfs + 1) .
  primResultOfs := primResultOfs + 2 .
].
^ result
%


! _replaceAllObsSym deleted

category: 'Storing and Loading'
method:
writeTo: aPassiveObject

"Instances of Repository cannot be converted to passive form.  This method
 writes nil to aPassiveObject and stops GemStone Smalltalk execution with a
 notifier."

aPassiveObject writeObject: nil.
self _error: #rtErrAttemptToPassivateInvalidObject
%

category: 'Formatting'
method:
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

"Copy the implementation from Object so we don't inherit it from Collection."

aStream nextPutAll: self asString
%

category: 'Private'
method:
_contentsOfPage: aPage
 
"If aPage is an invalid page ID, then the primitive fails.  Returns an Array of
 size at least 5 for all kinds of pages.  For data pages, the Array size is at
 least 15 and varies depending upon the number of objects in the page.

 The aPage argument should be a positive Integer specifying a valid page ID.
  
 The contents of the returned Array are as follows:

    index       description
     1.         pageKind
     2.         sessionId for user that wrote the page.
     3.         low beginId  (transaction begin ID)
     4.         high beginId
     5.         bytesUsed
                     (elements 6 and above are only present for data pages )
     6.         nil  (was approx page creation time (a DateTime) in gs64v1.0)
     7.         clusterId
     8.         freeBytes
     9.         numFreeChunks  
    10.         sizeLargestFree
    11.         numUsedBytes   
    12.         numberOfObjects
    13.         maxObjSize
    14.         minObjSize
    15.         numberOfValidObjects  
    16.         numberOfInvalidObjects
    (17.. 17 + numberOfValidObjects - 1) list of valid objects  
    (17 + numberOfValidObjects ...)  list of OOPs of invalid objects.  "

<primitive: 399>
self _primitiveFailed: #_contentsOfPage: args: { aPage }
%

category: 'Private'
method:
_shadowPagesByExtent

"This method aborts the current transaction.
 Returns an Array ; each element is a SmallInteger , the number of pages needing
 reclaim in that extent."

<primitive: 526>
self _primitiveFailed: #_shadowPagesByExtent
%

category: 'Private'
method:
findObjsConnectedTo: anArray
"Return an array of all objects reachable from the objects in the given 
 array.  Objects implicitly referenced from the header like the class are 
 not included in the sweep.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData."         

self deprecated: 'Repository>>findObjsConnectedTo: private method deprecated in v3.2, use public findObjectsConnectedTo:'.

^self findObjectsConnectedTo: anArray
%

category: 'Repository Analysis'
method:
findObjectsConnectedTo: anArray
"Return an Array of all objects reachable from the objects in the given
 array.  Objects implicitly referenced from the header like the class are
 not included in the sweep.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData."

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ self _findConnectedObjs: anArray.
%

category: 'Private'
method:
_findConnectedObjs: anArray

"Returns an Array of all objects reachable from the oops in anArray.
 This method aborts the current transaction."

<primitive: 560>
anArray _validateInstanceOf: Array .
(anArray size) < 1 ifTrue:[ Error signal:'anArray is empty'].
^ self _primitiveFailed: #_findConnectedObjs
%

category: 'Private'
method:
_findOopsOnPages: anArrayOfPageIds
"Scan the object table and return an array containing all OOPs that reference any page
 in the input array of page IDs."
<primitive: 561>
self _primitiveFailed: #_findOopsOnPages: args: { anArrayOfPageIds }
%

category: 'Private'
method:
_pageForOop: anInteger
"Returns the page ID for the given object ID by looking in the
 object table.  A result of -1 indicates the object ID is a 
 free oop, or the object ID is not in the object table."
<primitive: 562>
self _primitiveFailed: #_pageForOop: args: { anInteger }
%
category: 'Private'
method:
_findPagesContainingOops: anArrayOfInts
"Returns an Array of page ID's that contain the oops in the input
 array.  The input array must contain the oop as a SmallInteger or
 LargeInteger, not the object itself.  

 This method does not look at the object table, rather is scans
 the data pages directly from the extents.  Note that the object
 may appear on more than 1 page if it is shadowed."
<primitive: 563>
anArrayOfInts _validateInstanceOf: Array .
anArrayOfInts size == 0 ifTrue:[ anArrayOfInts _error: #errArgTooSmall args:{ 1 } ].
self _primitiveFailed: #_findPagesContainingOops: args: { anArrayOfInts }
%

category: 'Private'
method:
_oopIsDead: anInteger
"Returns a boolean indicating if the given oop is present in the
 stone's dead object set.  Primitive fails if the integer is not
 a valid disk oop value."

<primitive: 564>
self _primitiveFailed: #_oopIsDead: args: { anInteger } 
%

category: 'Deprecated'
method:
quickObjectAuditWithLevel: anInt

" Deprecated, use #fastObjectAudit instead of this method."

self deprecated: 'quickObjectAuditWithLevel: deprecated in v3.2, use fastObjectAudit instead'.
^ self fastObjectAudit
%


! changes to fix 32188
! Changes to fix 45317
category: 'Reference Paths'
method:
buildLimitSetForRefPathScan

"Build a list of objects to serve as the limit set for a reference path scan.
 The reference path scan ends when a reference from any objects
 returned by this method are found during the scan.
 Returns an Array containing all of the objectSecurityPolicies, 
 SharedDependencyLists, and the AllSymbols dictionary."

|set symUser dictBlock symListBlock|

set := IdentitySet new.
set addAll: SystemRepository. "Add all objectSecurityPolicies"
"Get all symbol list dictionaries and all versions of all classes from all user profiles.
 Using a set prevents duplicates."

"Handles SymbolDictionary and SymbolListDictionary"
dictBlock := [:k :v|
      (set includes: v)
	ifFalse:[
          v isBehavior ifTrue:[
            set add: v .
            set add: v class .
	    1 to: v classHistory size do:[:p| |class|
	      class := v classHistory at: p.
              class ~~ v ifTrue:[
	        set add: class.
	        set add: class class
            ]
          ]
        ]
	ifFalse:[
	  (v class == SymbolDictionary)
	    ifTrue:[ set add: v.  v keysAndValuesDo: dictBlock ]
	    ifFalse:[
	      (v class == SymbolList)
	        ifTrue:[ set add: v.  v do: symListBlock].
	].
    ].
  ].
].
  
symListBlock := [:eachDict|
    (set includes: eachDict) ifFalse:[ 
      set add: eachDict.
      eachDict keysAndValuesDo: dictBlock .
    ].
].

AllUsers do:[:aUserPro|
  set add: aUserPro .
  aUserPro symbolList do: symListBlock
].

set add: (Globals at: #SharedDependencyLists).

	"gemstone64, AllSymbols is now in SymbolUser's UserGlobals . "
symUser := AllUsers userWithId:'SymbolUser' .
set add: (symUser resolveSymbol: #AllSymbols) value .

^set asArray
%

category: 'Class Membership'
method:
species

"Returns a class similar to, or the same as, the receiver's class which
 can be used for containing derived copies of the receiver."

^ Array 
%

category: 'Repository Conversion'
method:
_migrateGroups

"Migrates group collections in all the ObjectSecurityPolicies of the receiver."

self do:[ :anObjectSecurityPolicy | anObjectSecurityPolicy ~~ nil 
  ifTrue:[ anObjectSecurityPolicy _migrateGroups] ]
%

category: 'Searching'
method:
indexOf: anObject

"Returns the index of the first element in the receiver that is equivalent to
 the argument. If the receiver does not have any elements equivalent to the
 argument, returns, zero.

 Since ObjectSecurityPolicies are canonicalized by the GsObjectSecurityPolicy creation methods, and
 only referenced from SystemRepository, this is implemented with
 indexOfIdentical:
"

^ self indexOfIdentical: anObject
%

category: 'Searching'
method:
indexOfIdentical: anObject

"Returns the index of the first element in the receiver that is identical to
 the argument. If the receiver does not have any elements identical to the
 argument, returns, zero."

<primitive: 494>
self _primitiveFailed: #indexOfIdentical: args: { anObject } 
%

category: 'Searching'
method:
indexOf: anObject ifAbsent: anExceptionBlock

"Returns the index of the first element in the receiver that is equivalent to
 the argument. If the receiver does not have any elements equivalent to the
 argument, returns the value of evaluating anExceptionBlock."

| ofs |
ofs := self indexOfIdentical: anObject .
ofs > 0 ifTrue:[ ^ ofs ].

^ anExceptionBlock value.
%

! ------------ conversion of pre-v2.2  SystemRepository
category: 'Conversion'
method: GsObjectSecurityPolicy
_setObjectSecurityPolicyId: anInt

self _unsafeAt: 7 put: anInt
%
set class Repository

expectvalue %SmallInteger
run
 "Check number of old objectSecurityPolicies needing conversion"
 | oldRep |
 oldRep := Globals at:#SystemRepository .
 oldRep class superClass == Array ifTrue:[ | oldSize maxSegId |
   oldSize := oldRep size .
   maxSegId := 16rFFFE .
   oldSize <= maxSegId ifTrue:[
      ^ oldSize 
   ] ifFalse:[
     ^ 'old SystemRepository too big, only the first ' , maxSegId asString ,
	' out of ' , oldSize asString , ' will be converted.' 
   ]
 ] ifFalse:[
   ^ 0 "no conversion needed"
 ]
%
expectvalue %String
run
  "Create aliases for changed names in Globals"
| report |
report := String new .
#(
  #(#'Segment'				#'GsObjectSecurityPolicy')
  #(#'DataCuratorSegment'		#'DataCuratorObjectSecurityPolicy')
  #(#'SystemSegment'			#'SystemObjectSecurityPolicy')
  #(#'SecurityDataSegment'	#'SecurityDataObjectSecurityPolicy')
  #(#'GsIndexingSegment'		#'GsIndexingObjectSecurityPolicy')
) do: [:pair |
  | oldName newName oldValue newValue |
  oldName := pair at: 1.
  newName := pair at: 2.
  oldValue := Globals at: oldName otherwise: nil.
  newValue := Globals at: newName otherwise: nil.
  (oldValue == nil and:[ newValue ~~ nil]) ifTrue: [
    Globals at: oldName put: newValue.
    report add: 'added '; add: oldName ; lf .
  ].
  (newValue == nil and:[ oldValue ~~ nil]) ifTrue: [
     Globals at: newName put: oldValue.
     report add: 'added '; add: newName ; lf .
  ].
].
^ report
%

! fix 41803 merged from 2.4.5 ; fix 47441
expectvalue %String 
run
 "Now convert or initialize the objectSecurityPolicies in the repository."
|  oldRep inConversion 
   rep count report seg sysUser installSegBlock segsCreated |
report := String new .
oldRep := Globals at:#SystemRepository .
oldRep class superClass == Array ifTrue:[ | newSize newRep |
   inConversion := true .
   newSize := oldRep size min: 16rFFFE .  "note message above"
   newSize < 2 ifTrue:[ | dcSeg |
     dcSeg := Globals at:#DataCuratorObjectSecurityPolicy otherwise: nil  .
     dcSeg ~~ nil ifTrue:[
       "looks like a 2.0 or 2.1 repository, need to fixup DataCuratorObjectSecurityPolicy"
       newSize := 2.
       oldRep at:2 put: DataCuratorObjectSecurityPolicy .
       { #GsIndexingObjectSecurityPolicy . #PublishedObjectSecurityPolicy . 
	 #GsTimeZoneObjectSecurityPolicy . #SecurityDataObjectSecurityPolicy } 
		do:[ :aKey | | aSeg |
          aSeg := Globals at: aKey otherwise: nil .
          aSeg == (oldRep at: 1) ifTrue:[
            "delete this ref to SystemObjectSecurityPolicy"  
            Globals removeKey: aKey . 
          ]. 
        ]
     ].
   ].
   newRep := Repository new: newSize .   "new: not yet disallowed"
   newRep instVarAt:1 put: (oldRep instVarAt:1) "name"  .
   newRep instVarAt:2 put: (oldRep instVarAt:2) "dataDictionary" .
   newRep _unsafeSetOop: 240641 .

   GsObjectSecurityPolicy _clearConstraints .

   1 to: newSize do:[:segId| | anObjectSecurityPolicy |
     anObjectSecurityPolicy := oldRep at: segId .
     anObjectSecurityPolicy _unsafeAt:1 put: newRep .
     anObjectSecurityPolicy _unsafeAt:7 put: segId  "objectSecurityPolicyId" .
   ].
   1 to: newSize do:[:segId| | anObjectSecurityPolicy |
     anObjectSecurityPolicy := oldRep at: segId .
     newRep _basicAt:segId put: anObjectSecurityPolicy .
     anObjectSecurityPolicy objectSecurityPolicy: DataCuratorObjectSecurityPolicy .
   ].
   newRep objectSecurityPolicy:  DataCuratorObjectSecurityPolicy .
   Globals at:#SystemRepository put: newRep .
   Globals at:#OldSystemRepository put: oldRep .
   report addAll: 'converted ' ; addAll: newSize asString ;
     addAll: ' objectSecurityPolicies. '
] ifFalse:[
   inConversion := false .
   report addAll: 'conversion of SystemRepository not needed. '
].

"Ensure all ObjectSecurityPolicies from a virgin 2.2 exist . 
 Gs64 v2.2 bom.c only creates the first 2 . "

rep := Globals at:#SystemRepository .
count := 0 .
segsCreated := { false . false . false . false .  false . false . false . false } .
rep size < 8 ifTrue:[
  [ rep size < 8 ] whileTrue:[
    count := count + 1 .
    GsObjectSecurityPolicy newInRepository: rep . 
    segsCreated at: rep size put: true .
  ].
  report addAll:'created '; addAll: count asString ; addAll: ' objectSecurityPolicies. '
].
sysUser := AllUsers userWithId:'SystemUser' .
rep := Globals at:#SystemRepository .

installSegBlock := [ :aKey :aSeg | | oldSeg |
  oldSeg := Globals at: aKey otherwise: nil .
  oldSeg == nil ifTrue:[
    Globals at: aKey put: aSeg
  ] ifFalse:[
    oldSeg == aSeg ifFalse:[
      nil error:'invalid GsObjectSecurityPolicy reference in Globals'
    ]
  ]
].

seg := rep at: 1 .		"repeats bom work, needed for conversion"
installSegBlock value:#SystemObjectSecurityPolicy value: seg .
((segsCreated at: 1) or:[ inConversion]) ifTrue:[
  seg  owner: sysUser ; ownerAuthorization: #write  ;
    worldAuthorization: #read  .
].

seg := rep at: 2 .		"repeats bom work, needed for conversion"
installSegBlock value:#DataCuratorObjectSecurityPolicy value: seg .
((segsCreated at: 2) or:[ inConversion]) ifTrue:[
  seg owner: (AllUsers userWithId:'DataCurator') ;
    ownerAuthorization: #write  ;
    worldAuthorization: #read  .
].

				"remaining not done in bom"
seg := rep at: 3 .	"GsTimeZoneObjectSecurityPolicy no longer used"	

seg := rep at: 4 .
installSegBlock value:#GsIndexingObjectSecurityPolicy value: seg .
((segsCreated at: 4) or:[ inConversion]) ifTrue:[
  seg isInvariant ifFalse:[
    seg name: #GsIndexingObjectSecurityPolicy ;
      owner: sysUser ; ownerAuthorization: #write  ;
      worldAuthorization: #write  ; immediateInvariant .
].
].

seg := rep at: 5 .
installSegBlock value:#SecurityDataObjectSecurityPolicy value: seg .
seg isInvariant ifFalse:[
  seg name: #SecurityDataObjectSecurityPolicy ;
    owner: sysUser ; ownerAuthorization: #write  ;
    worldAuthorization: #none  ;  immediateInvariant; _unsafeSetOop: 235777 .
].

seg := rep at: 6 .
installSegBlock value:#PublishedObjectSecurityPolicy value: seg .
((segsCreated at: 6) or:[ inConversion]) ifTrue:[
  seg isInvariant ifFalse:[
    seg owner: sysUser ;   "group authorizations done later in bomlast"
      ownerAuthorization: #write ;
    worldAuthorization: #none .
  ].
].

"GsObjectSecurityPolicy 7 , owned by GcUser , fixed up later"
"GsObjectSecurityPolicy 8, owned by Nameless, fixed up later"

inConversion ifTrue:[
  "converting from a pre-Gs64 v2.2 repository.
   Fix 35966 , repositories from Gs64 v2.0.x and v2.1.x  may 
   have objectSecurityPolicyId == OOP_NIL (10r20) in disk object headers ,
   so preallocate a GsObjectSecurityPolicy for objectSecurityPolicyId 20 that is world read-write "
   rep size < 20 ifTrue:[ | newSeg |
     "Possibly a Gs64 v2.0.x or v2.1.x  repository "
     newSeg := GsObjectSecurityPolicy newInRepository: rep .  
     newSeg owner: sysUser; ownerAuthorization: #write;
            worldAuthorization:#write ; immediateInvariant  .
     newSeg objectSecurityPolicyId >= 20 ifTrue:[
        nil error:'invalid objectSecurityPolicyId' 
     ].
     newSeg _setObjectSecurityPolicyId: 20. 		"fix 38084"
     rep size: 20 .
     rep at: 20 put: newSeg .
     report addAll: ' created GsObjectSecurityPolicy 20 .'.
   ]
].
^ report 
%

expectvalue %String
run
  "Create aliases for changed names in Globals"
| report list |
report := String new .
list := #(
  #(#'Segment'				#'GsObjectSecurityPolicy')
  #(#'DataCuratorSegment'		#'DataCuratorObjectSecurityPolicy')
  #(#'SystemSegment'			#'SystemObjectSecurityPolicy')
  #(#'SecurityDataSegment'	#'SecurityDataObjectSecurityPolicy')
  #(#'GsIndexingSegment'		#'GsIndexingObjectSecurityPolicy')).
list do: [:pair |
  | oldName newName oldValue newValue |
  oldName := pair at: 1.
  newName := pair at: 2.
  oldValue := Globals at: oldName otherwise: nil.
  newValue := Globals at: newName otherwise: nil.
  (oldValue == nil and:[ newValue ~~ nil]) ifTrue: [
    Globals at: oldName put: newValue.
    report add: 'added '; add: oldName ; lf .
  ].
  (newValue == nil and:[ oldValue ~~ nil]) ifTrue: [
     Globals at: newName put: oldValue.
     report add: 'added '; add: newName ; lf .
  ].
].
list do:[:pair | pair do:[:name | (Globals associationAt: name) immediateInvariant ]].
^ report
%


expectvalue true
run
GsObjectSecurityPolicy removeSelector: #_setObjectSecurityPolicyId: .
^ true
%

! ------- instance creation methods disallowed
category: 'Instance Creation'
classmethod:
new

"Disallowed."
self shouldNotImplement: #new
%

category: 'Instance Creation'
classmethod:
new: anInteger

"Disallowed."

self shouldNotImplement: #new:
%


! with:...  disallowed because new: disallowed

!--------------- methods inherited from Object that are disallowed
set class Repository

category: 'Updating'
method
at: offset put: aValue
"Disallowed"
self shouldNotImplement: #at:put:
%

method
basicAt: offset put: aValue
"Disallowed"
self shouldNotImplement: #basicAt:put:
%

method
squeakBasicAt: anIndex put: aValue
  "Disallowed"
  self shouldNotImplement: #squeakBasicAt:put:
%

method
instVarAt: offset put: aValue 
"Disallowed"
self shouldNotImplement: #instVarAt:put:
%

method
nilFields
"Disallowed"
self shouldNotImplement: #nilFields
%

method
size: aSize
"Disallowed"
self shouldNotImplement: #size:
%

method
_basicAt: offset put: aValue
"Disallowed"
self shouldNotImplement: #_basicAt:put:
%

method
_basicSize: aSize
"Disallowed"
self shouldNotImplement: #_basicSize:
%

method
_primitiveAt: offset put: aValue
"Disallowed"
self shouldNotImplement: #_primitiveAt:put:
%

method
_unsafeAt: offset put: aValue
"Disallowed"
self shouldNotImplement: #_unsafeAt:put:
%

method
objectSecurityPolicy: aSeg
"Disallowed"
self shouldNotImplement: #objectSecurityPolicy:
%

method
_objectSecurityPolicy: aSeg
"Disallowed"
self shouldNotImplement: #_objectSecurityPolicy:
%

category: 'Storing and Loading'

method
basicLoadFrom: passiveObj
"Disallowed."
self shouldNotImplement: #basicLoadFrom:
%

method
basicLoadFrom: passiveObj size: aSize
"Disallowed"
self shouldNotImplement: #basicLoadFrom:size:
%

method
basicLoadFromNoRead: passiveObj
"Disallowed"
self shouldNotImplement: #basicLoadFromNoRead:
%

method
basicLoadFromNoRead: passiveObj size: aSize
"Disallowed"
self shouldNotImplement: #basicLoadFromNoRead:size:
%

method
basicLoadFromOld: passiveObj
"Disallowed"
self shouldNotImplement: #basicLoadFromOld:
%

method
loadFrom: passiveObj
"Disallowed"
self shouldNotImplement: #loadFrom:
%

method
loadNamedIVsFrom: passiveObj
"Disallowed"
self shouldNotImplement: #loadNamedIVsFrom:
%

method
loadVaryingFrom: passiveObj
"Disallowed"
self shouldNotImplement: #loadVaryingFrom:
%

method
loadVaryingFrom: passiveObj size: aSize
"Disallowed"
self shouldNotImplement: #loadVaryingFrom:size:
%

category: 'Instance migration'

method
become: anObj
"Disallowed"
self shouldNotImplement: #become:
%

method
_primitiveBecome: anObject forDict: aBoolean
"Disallowed"
self shouldNotImplement: #_primitiveBecome:forDict:
%

method
migrate
"Disallowed"
self shouldNotImplement: #migrate
%

method
migrateFrom: anObj
"Disallowed"
self shouldNotImplement: #migrateFrom:
%

method
migrateFrom: anObj instVarMap: aDict
"Disallowed"
self shouldNotImplement: #migrateFrom:instVarMap:
%

method
migrateIndexable: anObj myClass: aCls otherClass: secondCls
"Disallowed"
self shouldNotImplement: #migrateIndexable:myClass:otherClass:
%

method
changeClassTo: aCls
"Disallowed"
self shouldNotImplement: #changeClassTo:
%

category: 'Copying
method
copy
"Disallowed"
self shouldNotImplement: #copy
%

method
deepCopy
"Disallowed"
self shouldNotImplement: #deepCopy
%

method
shallowCopy
"Disallowed"
self shouldNotImplement: #shallowCopy
%


! --------------  methods inherited from Collection  that are disallowed 
method
add: anObj
"Disallowed"
self shouldNotImplement: #add:
%

method
addAll: aCollection
"Disallowed"
self shouldNotImplement: #addAll:
%

method
remove: anObj
"Disallowed"
self shouldNotImplement: #remove:
%

method
remove: anObj ifAbsent: aBlock
"Disallowed"
self shouldNotImplement: #remove:ifAbsent:
%

method
removeAll: aCollection
"Disallowed"
self shouldNotImplement: #removeAll:
%

method
removeAllIdentical: aCollection
"Disallowed"
self shouldNotImplement: #removeAllIdentical:
%

method
removeIdentical: anObj
"Disallowed"
self shouldNotImplement: #removeIdentical:
%

method
removeIdentical: anObj ifAbsent: aBlock
"Disallowed"
self shouldNotImplement: #removeIdentical:ifAbsent:
%


category: 'Instance Counting'
method:
countInstances: anArray withMaxThreads: maxThreads maxCpuUsage: aPercentage

"Scans the entire Repository and generates an array of instance counts of the 
 the classes contained in anArray. Does not include in-memory objects.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData. The entire
 Repository is scanned once."

| inputSet resultInSetOrder result inputArraySize anObj |

System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

inputSet := self _arrayOfClassesAsSet: anArray .
inputSet size < 1 ifTrue:[ ^ { } ] .

resultInSetOrder := self _scanPomWithMaxThreads: maxThreads 
       waitForLock: 90 
       pageBufSize: 8
       percentCpuActiveLimit: aPercentage  
       identSet: inputSet 
       limit: 0 
       scanKind: 8 "OP_COUNT_INSTANCES - OP_FIRST_SCAN_KIND"
       toDirectory: nil .

inputArraySize := anArray size .
result := Array new: inputArraySize .
1 to: inputArraySize do:[:j|
  anObj := anArray at: j .
  result at: j put: ( resultInSetOrder at: (inputSet _offsetOf: anObj) * 2)
  ].
^ result
%

category: 'Instance Counting'
method:
countInstances: anArray

"Scans the entire Repository and generates an array of instance counts of the 
 the classes contained in anArray.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To 
 complete the operation in less time, see the #fastCountInstances:
 method."

^ self countInstances: anArray withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90
%

category: 'Instance Counting'
method:
fastCountInstances: anArray

"Same as the #countInstances: method except the scan is performed 
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is in progress."

^ self countInstances: anArray withMaxThreads: self _aggressiveMaxThreadCount
       maxCpuUsage: 95
%

category: 'Page-Order Operations'
method:
listInstancesInPageOrder: anArrayOfClasses toFile: aString 
 withMaxThreads: maxThreads maxCpuUsage: aPercentage

"Scans the repository for all committed instances of the classes
 contained in anArray and stores the results into a new file
 specified in aString.
 Does not include objects in VM's temporary object memory.

 anArrayOfClasses must be an instance of Array which contains at
 least 1 and not more than 2034 classes.

 aString must be the full path to a new file on the gem's host system 
 which will be created by this method.  It is an error if the file
 already exists.

 All results of the scan (all instances of the classes in anArray) are
 stored in the file in page order (i.e., the same order as the  objects
 are stored on disk) as of the time of the scan.   Note that unlike other
 #listInstances methods,  the file contains the complete list of all
 instances of all classes contained in anArray.

 The ultimate file size will be approximately 5 bytes for each object
 found by the scan, plus 24 bytes.  For example, a scan which locates 10 
 million objects will generate a file approximately 50 megabytes in size.

 Objects created and committed after starting this method are not included
 in the result file.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.

 This method runs in a transaction and is therefore not recommended for 
 production systems.

 Operations which process a large volume of objects (e.g.: instance migration,
 etc) often run significantly faster if the objects are processed in page order
 rather than in object ID order.  Other #listInstances: methods always return
 results in object ID order.

 The objects written to the file are left in hidden set 1 when this method
 completes successfully.

 Objects are routinely moved from old pages to new pages whenever modified.
 The Reclaim GC gems also move objects to new pages.  Therefore the actual
 order the objects appear on disk will diverge over time from the order of
 the objects in the result file.

 Equivalent to executing the following methods:

   SystemRepository listInstancesToHiddenSet: anArrayOfClasses
                    withMaxThreads: maxThreads maxCpuUsage: aPercentage .
   System writeHiddenSet: 1 toPageOrderFile: aString maxBufferSizeMb: 0 .

 except that this method creates the result file before starting the scan.

 Returns a SmallInteger representing the total number of objects written to the
 file."
 
System needsCommit
  ifTrue: [ ^ self _error: #rtErrAbortWouldLoseData ] .

^ self _scanPomWithMaxThreads: maxThreads waitForLock: 90 pageBufSize: 8 
                  percentCpuActiveLimit: aPercentage
                  identSet: (IdentitySet withAll: anArrayOfClasses)
                  limit: 0 
          scanKind: 3  "OP_LIST_INST_PAGE_ORDER-OP_FIRST_SCAN_KIND"
          toDirectory: aString
%


category: 'Page-Order Operations'
method:
listInstancesInPageOrder: anArrayOfClasses toFile: aString
"See the method #listInstancesInPageOrder:toFile:withMaxThreads:maxCpuUsage: 
 for more information on this method.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To 
 complete the operation in less time, see the #fastListInstancesInPageOrder:toFile:
 method."

^ self listInstancesInPageOrder: anArrayOfClasses toFile: aString 
       withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90
%

category: 'Page-Order Operations'
method:
fastListInstancesInPageOrder: anArrayOfClasses toFile: aString
"Same as the #listInstancesInPageOrder:toFile: method except the scan is performed 
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is in progress."

^ self listInstancesInPageOrder: anArrayOfClasses toFile: aString 
       withMaxThreads: self _aggressiveMaxThreadCount maxCpuUsage: 95

%

category: 'Deprecated Page-Order Operations'
method:
openPageOrderOopFile: aString

"Deprecated, See GsBitmap category 'File Operations', these methods 
    don't keep the file open, so no need to open/close with an Id.

 Opens a binary page-ordered file of object ID's created by the
 Repository>>listInstancesInPageOrder:toFile: method.

 Raises an error if the file could not be opened for reading, the file
 is corrupt, or aString is not an instance of String.

 Returns a SmallInteger which is the ID of the opened file.  This ID must
 be used to read the contents of the file (see the method
 readObjectsFromFileWithId:startingAt:upTo:into: in this class).

 This method acquires and retains the garbage collection lock.   No garbage
 collection operations can be started until the
 Repository>>closePageOrderOopFileWithId: method is executed or the session
 logs out."

<primitive: 830>
aString _validateKindOfClass: String .
aString size == 0 ifTrue:[ aString _error: #errArgTooSmall args:{ 1 } ].
^ self _primitiveFailed: #openPageOrderOopFile: args: { aString }
%

category: 'Deprecated Page-Order Operations'
method:
readObjectsFromFileWithId: aSmallInt startingAt: startIndex upTo: endIndex into: anArray

"Deprecated: use GsBitmap>>readFromFile:withLimit:startingAt:

 Reads and validates objects from a page-ordered file which was previously
 opened with the Repository>>openPageOrderOopFile: method.

 aSmallInt must be the result of the Repository>>openPageOrderOopFile: method.

 startIndex is the index of the first object in the file to read.  The first 
 object in the file has an index of 1.  It is an error if startIndex is less than 1
 or greater than endIndex.

 endIndex is the index of the last object to read from the file.  endIndex 
 must be greater than or equal to startIndex.  endIndex may exceed the index of
 the last object in the file.  In that case, all objects from startIndex to the
 end of the file will be returned.

 anArray must be an instance of Array when the resulting objects will be 
 stored.  It is an error if anArray is not empty.

 Returns a SmallInteger which is the size of anArray.  This will be the lesser
 of the number (endingIndex - startIndex + 1) or, all remaining objects in the
 file.

 It is possible that one or more object identifiers contained in the file
 are no longer valid due to garbage collection.  Objects which are no longer
 valid have nil stored in their place in anArray.  Objects that have been 
 garbage collected and are in the free oop list or the dead not reclaimed set
 are considered to be invalid.

 The session must be in a transaction when this method is invoked.  If the
 session is not in a transaction, an #rtErrPrimOutsideTrans error is raised.

 For best performance, it is recommended that no more than 2034 objects
 be returned by any single invocation of this method.  Returning a high
 number of objects will cause anArray to grow very large and may result in
 a fatal out-of-memory error."

 self deprecated: 'Repository>>readObjectsFromFileWithId:startingAt:upTo:into: ',
  'deprecated v3.4. Use GsBitmap>>readFromFile:withLimit:startingAt:'.

^ self _primReadObjectsFromFileWithId: aSmallInt startingAt: startIndex upTo: endIndex into: anArray 
%


category: 'Deprecated Page-Order Operations'
method:
numberOfObjectsInPageOrderOopFileWithId: aSmallInt

"Deprecated, use (GsBitmap fileInfo: <bitmapFilePath>) at: 1

 Answer the number of objects in a page order oop file previously opened 
 with the Repository>>openPageOrderOopFile: method.
 aSmallInt must be the result of the Repository>>openPageOrderOopFile: method.
 Does not close or otherwise change the state of the file."

  self deprecated: 'Repository>>numberOfObjectsInPageOrderOopFileWithId: deprecated v3.4. ',
     'Use (GsBitmap fileInfo: <bitmapFilePath>) at: 1'.

^ self _primReadObjectsFromFileWithId: aSmallInt startingAt: nil upTo: nil into: nil
%

category: 'Private'
method:
_primReadObjectsFromFileWithId: aSmallInt startingAt: startIndex upTo: endIndex into: anArray

<primitive: 831>
aSmallInt _validateClass: SmallInteger .
startIndex ifNil:[ "all args nil means return the number of oops in the file"
 endIndex _validateClass: UndefinedObject .
 anArray _validateClass: UndefinedObject .
] ifNotNil:[ "normal case: request to read the file failed."
  startIndex _validateClass: SmallInteger .
  endIndex _validateClass: SmallInteger .
  anArray _validateInstanceOf: Array .
  anArray size == 0 ifFalse:[ anArray _error: #errArgTooSmall args:{ 1 } ] .
  (startIndex < 1 or:[ startIndex > endIndex]) ifTrue:[ 
     startIndex _error: #rtErrArgOutOfRange args:{ 1 . endIndex } ] .
].
^ self _primitiveFailed: 
       #_primReadObjectsFromFileWithId:startingAt:upTo:into: 
       args: { aSmallInt . startIndex . endIndex . anArray }
%
category: 'Deprecated Page-Order Operations'
method:
closePageOrderOopFileWithId: aSmallInt

"Deprecated. Page order operations done using GsBitmap don't require close, 
 so there is no replacement. 
 Closes a file previously opened with the Repository>>openPageOrderOopFile: 
 method and releases the garbage collection lock.

 Returns true if successful; false if the file was not open or was already 
 closed."

<primitive: 832>
aSmallInt _validateClass: SmallInteger .
^ self _primitiveFailed: #closePageOrderOopFileWithId: args: { aSmallInt }
%

category: 'Deprecated Page-Order Operations'
method:
auditPageOrderOopFileWithId: aSmallInt

"Deprecated, use 
   | arr fileInfo | 
   fileInfo := GsBitmap fileInfo: <bitmapFilePath>. 
   arr := Array new.
   arr add: fileInfo at: 1.
   arr add: fileInfo at: 4.
   ^ arr

 Reads the page order oop file with the ID returned by the 
 Repository>>openPageOrderOopFile: method and counts the number of
 objects in the file which are no longer valid.  Objects are 
 considered invalid if they no longer exist (i.e., are not 
 present in the shared object table) or present in the dead objects
 set maintained by stone.

 The session must be in a transaction when this method is invoked.  If the
 session is not in a transaction, an #rtErrPrimOutsideTrans error is raised.

 Returns an array containing 2 SmallIntegers:
    array[1] - total number of oops in the file.
    array[2] - number of invalid oops in the file.

 Returns an array containing 2 nils if aSmallInt is not a valid file
 ID returned by the Repository>>openPageOrderOopFile: method."

<primitive: 833>
aSmallInt _validateClass: SmallInteger .
^ self _primitiveFailed: #auditPageOrderOopFileWithId: args: { aSmallInt }
%

category: 'Private'
method:
_pageForOopIsTagged: anInt
"Lookup anInt in the shared object table and return a Boolean indicating
 if the page ID for the object is tagged, indicating it is in the shared
 dependency map.  Returns nil if anInt is not in the object table or it is
 a free oop."

<primitive: 570>
self _primitiveFailed: #_pageForOopIsTagged: args: { anInt }
%

category: 'Private'
method:
_sharedDepMapLookupForOop: anInt
"Lookup anInt in the shared dependency map and return an integer which indicates
 the value in the map for the given OOP (which is the DependencyList for the
 given OOP).

 Returns nil if anInt is not present in the shared dependency map."

<primitive: 571>
self _primitiveFailed: #_sharedDepMapLookupForOop: args: { anInt }
%

category: 'Disconnected Objects'
method:
basicFindDisconnectedObjectsAndWriteToFile: aFileNameString
 pageBufferSize: pageBufSize saveToRepository: saveToRep
 withMaxThreads: maxThreads maxCpuUsage: aPercentage

"Perform an FDC on the system and write the list of
 dead objects to given file.  If saveToRep is true, also store the results
 in the array under the key #FdcResults in Globals.

 To save the dead objects to an array but not write them to a file, set
 aFileNameString to nil and saveToRep to true.  It is illegal for 
 aFileNameString to be nil and for saveToRep to be false.

 This method aborts the current transaction.

 pageBufSize must be a power of two and specifies the number of pages to buffer per thread.

 This method will return an error if the repository is in restore mode.

 See also the comments in the method
  Repository>> _fdcWithMaxThreads:... 

 for a desciption of the result array and a more detailed description of
 this method."

| fdcResults arrayOfDead |

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

(System transactionMode == #autoBegin)
  ifFalse:[System transactionMode: #autoBegin].

(SystemRepository restoreActive)
  ifTrue:[^ self _error: #rtErrCommitDbInRestore].

System abortTransaction.
System commitTransaction
  ifFalse:[^ self _error: #rtErrGcCommitFailure].
fdcResults := self _createFdcResults .

arrayOfDead := fdcResults at: 3.
saveToRep ifTrue:[ arrayOfDead ifNil:[ fdcResults at: 3 put: Array new ]]
         ifFalse:[ fdcResults at: 3 put: nil ].

System commitTransaction
  ifFalse:[^ self _error: #rtErrGcCommitFailure].

^ self _fdcWithMaxThreads: maxThreads
      waitForLock: 90 pageBufSize: pageBufSize percentCpuActiveLimit: aPercentage
      toFile: aFileNameString resultArray: fdcResults
%

category 'Garbage Collection'
method:
markForCollectionWithMaxThreads: maxThreads waitForLock: waitTimeSeconds
"See the markForCollection method for comments."

^ self markForCollectionWithMaxThreads: maxThreads 
                           waitForLock: waitTimeSeconds 
                           pageBufSize: 128
                 percentCpuActiveLimit: 90
%

! fixed 41773
category 'Garbage Collection'
method:
markForCollectionWithMaxThreads: maxThreads waitForLock: waitTimeSeconds 
 percentCpuActiveLimit: aPercent

"See the markForCollection method for comments."

| arr ex msg |
arr := self _mfcWithMaxThreads: maxThreads waitForLock: waitTimeSeconds 
    pageBufSize: 128 percentCpuActiveLimit: aPercent.
(ex := Warning new) _number: 2515 "WARNING_FINISHED_MFC" ; args: arr .
msg := 'markForCollection found ' ,
        (arr atOrNil: 1) asString , ' live objects, ',
        (arr atOrNil: 2) asString , ' dead objects(occupying approx ' ,
        (arr atOrNil: 3) asString , ' bytes)' .
ex signal: msg .
^ ex
%

category 'Garbage Collection'
method:
_fastMarkForCollection

"Returns an Array { numLiveObjects . numDeadObjects . approxSizeOfDeadObjects .
                     numPossibleDeadSymbols } .
"
^ self _mfcWithMaxThreads: self _aggressiveMaxThreadCount 
    waitForLock: 90"seconds" 
    pageBufSize: 128 percentCpuActiveLimit: 95.
%

category 'Garbage Collection'
method:
markForCollectionWithMaxThreads: maxThreads waitForLock: waitTimeSeconds 
 pageBufSize: pageBufSize percentCpuActiveLimit: aPercent

"See the markForCollection method for comments."

| arr ex msg |
arr := self _mfcWithMaxThreads: maxThreads waitForLock: waitTimeSeconds 
    pageBufSize: pageBufSize percentCpuActiveLimit: aPercent.
(ex := Warning new) _number: 2515"WARNING_FINISHED_MFC" ; args: arr .
msg := 'markForCollection found ' , 
        (arr atOrNil: 1) asString , ' live objects, ',
        (arr atOrNil: 2) asString , ' dead objects(occupying approx ' ,
        (arr atOrNil: 3) asString , ' bytes)' .
(arr atOrNil: 4) ifNotNil:[:nPd |
  msg addAll: ', ', nPd asString , ' possibleDeadSymbols' 
].
ex signal: msg .
^ ex
%

category 'Garbage Collection'
method:
fastMarkForCollection

"Same as the #markForCollection method except the scan is performed 
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

^ self markForCollectionWithMaxThreads: self _aggressiveMaxThreadCount 
  waitForLock: 90 pageBufSize: 128 percentCpuActiveLimit: 95
%

category 'Garbage Collection'
method:
fastMarkForCollectionWithPageBufSize: pageBufSize

"Same as the #fastMarkForCollection method except you can specify 
 a larger pageBufSize than the default 128.

 This method may consume most or all host system resources while it
 is in progress."

^ self markForCollectionWithMaxThreads: self _aggressiveMaxThreadCount 
  waitForLock: 90 pageBufSize: pageBufSize percentCpuActiveLimit: 95
%

! fixed 41773
category: 'Private'
method:
_mfcWithMaxThreads: maxThreads waitForLock: lockWaitTime 
      pageBufSize: aBufSize percentCpuActiveLimit: cpuPercent
 "Perform a multithreaded markForCollection operation using at most maxThreads.

  Returns an Array { numLiveObjects . numDeadObjects . approxSizeOfDeadObjects .
                     numPossibleDeadSymbols } .
  The numPossibleDeadSymbols element of the result will be nil if
  STN_SYMBOL_GC_ENABLED is false in the stone configuartion.

  The maxThreads argument specifies the maximum number of threads
  that can be used during the operation.  The actual number of active threads
  can be adjusted to a lower value at runtime automatically by the main 
  thread based on the percentCpuActiveLimit and dynamically by the setting 
  the mtThreadLimit (see details below).

  The lockWaitTime argument is used to specify how many seconds method should
  wait while attempting to acquire the gcLock.  No other garbage collection 
  operations may be started or in progress while this method is running.  
  There also must be no outstanding possible dead objects in the system for 
  the GC lock to be granted.
 
  The pageBufSize, which must be a power of two, specifies the number 
  of pages to buffer per thread. This parameter in conjunction with the 
  maxThreads largely determines the memory footprint needed to perform 
  this operation.

  The cpuActiveLimit specifies a level of total cpu activity at which the algorithm
  automatically inactivates threads to prevent overload of system resources.

  This algorithm makes use of additional sessions (threads) to achieve 
  significant performance improvements.  It also makes space/time trade offs
  so that heap memory in addition to the TemporaryObjectCache (TOC)
  resources are used.  In fact, this algorithm doesn't require any TOC, so
  configuring this process for the smallest TOC space is advantageous. 
  The memory space that is needed is variable and depends upon the current 
  oopHighWater value, the number of sessions and the pageBufSize specified.

  The overhead associated with the oopHighWater value can be computed:
    oopHighWater in bytes = (stnOopHighWater + 10M) / 2

  The memory cost per thread is 50K + (180K * pageBufSize)

  To give some idea of how this scales, a system that had an oopHighWaterMark
  of 500M running 8 sessions with a pageBufSize of 512 would require about 1GB 
  of free memory to start up.

  Other tuning considerations:

  During the operation the repository is scanned repeatedly from low pageId 
  to high pageId within each extent.  Performance can be optimized by arranging 
  the extents so that the scan causes the fewest head movements on each drive 
  and the threads can simultaneously initiate I/O operations on multiple extents.

  For example:   With a single disk controller and one extent per drive the 
    following would provide an optimal configuration: 
       Disk1: extent0
      Disk2: extent1
       ... 
  With multiple controllers:
     Controller1:  Disk 1 extent0
                   Disk 2 extent2
                   Disk 3 extent4
     Controller2:  Disk 4 extent1
                   Disk 5 extent3
                   Disk 6 extent5
  
  For use with a RAID or SAN system, the device should be configured for 16K read
  operations from sequentially increasing blocks within an extent.

  This operation requires almost continuous access to the entire object table.
  Running on a system that can hold all of the object table in the cache is will
  allow it to perform optimally.  The object table size is approximately 
  stnOopHighWater * 12 bytes.

  If the machine can be dedicated to performing this operation, a good rule of thumb
  is to set the maxThreads to numCpus + numControllers and the percentCpuActiveLimit
  should be increased to 100.

  If the operation is being performed on a system that is running a live application
  the maxThreads can still be set to a high value and the cpuActiveLimit can be
  used to control the cpu resources used by this method.  If this operation pushes the
  cpu utilization above the specified limit the main thread automatically deactivates 
  slave threads until the load is again below the specified limit.  While threads are 
  not actively working on the operation, they still respond to the commit record 
  backlog and abort as needed to prevent the backlog from exceeding its specified limit.

  The pageBufSize specified must be a power of 2. Larger values can improve 
  performance.  For example, in a test case where the machine was dedicated to 
  performing this operation, increasing the pageBufSize from 64 to 512 reduced 
  the elapsed time by about 20 percent.  On the other hand, if the operation is being 
  run on a live system then smaller values may be better depending on how often the algorithm 
  must abort (and clear the buffers) to prevent the commit record backlog from growing.

  Caution should be taken when setting the numSessions and pageBufSize to prevent 
  the memory footprint for this process from being so large that the system starts
  swapping.

  The values of the arguments percentCpuActiveLimit and maxThreads are used as initial
  values for two statistics associated with this process, the mtPercentCpuActiveLimit 
  and mtThreadLimit.  The main thread monitors these statistics and uses them to control
  the number of threads that are actively working, which is displayed in the mtActiveThreads
  statistic.  Thus the operation can be tuned dynamically by setting either the 
  mtPercentCpuActiveLimit or mtThreadLimit for the process.  When the system is not performing
  adequately because the operation is consuming too much of the cpu resources, the 
  mtPercentCpuActiveLimit: method can be used to decrease the value of the corresponding
  statistic and thus remove some of the load on the system caused by this operation.
  In situations where I/O is the limiting factor on the system and the operation is 
  using too much I/O bandwidth, the number of active threads can be changed by using 
  the mtThreadsLimit: method.  The mtPercentCpuActiveLimit can be set to any value 
  between 0 and 100, mtThreadLimit can be set to values between 0 and maxThreads."
  
<primitive: 876>
| maxInt |
maxInt := SmallInteger maximum32bitInteger .
maxThreads _validateClass: SmallInteger; _validateMin: 1 max: maxInt .
lockWaitTime _validateClass: SmallInteger; _validateMin: -1 max: maxInt .
aBufSize _validateClass: SmallInteger ; _validateIsPowerOf2 .
self _validatePercentage: cpuPercent .
^ self _primitiveFailed: #_mfcWithMaxThreads:
       args: { maxThreads . lockWaitTime . aBufSize . cpuPercent}
%

category: 'Private'
method:
_fdcWithMaxThreads: maxThreads waitForLock: lockWaitTime 
    pageBufSize: aBufSize percentCpuActiveLimit: cpuPercent
    toFile: aFileNameString resultArray: anArray

 "Run a multithreaded algorithm to determine the disconnected (dead) objects
  in the repository.  The dead objects identifiers (oops) are written
  to a file, stored in anArray, or both.  

  Returns anArray .

  This is method never results in the garbage collection of any objects.

  See _mfcWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:
  for documentation of the common arguments.

  If the toFile argument is not nil, a file is created and a list of the
  dead object identifiers found is written to the file in a binary format.
  The file must not exist when the method is called and the argument
  must be an instance or subclass of String.  
  The size of the file in bytes is approximately 8 times the number 
  of dead objects found. 

  Be careful to ensure there is sufficient disk space to write the 
  entire list of dead objects when using this method with large repositories.

  The resultArray is expected to be a committed array object of size 4.
  If successful, this method stores the following values in the resultArray:

    1 - anInteger representing the number of live objects found.
    2 - anInteger representing the number of dead objects found.
    3 - anArray containing the dead objects, false if a commit failed,
        or nil if the list of dead objects is not stored in the repository.
    4 - nil if aFileNameString was nil, otherwise a boolean indicating 
        whether the dead object ids were successfully written to the file.

  This method requires the GarbageCollection privilege and aborts the
  current transaction.

  Pre-existing dead objects, i.e., objects which reside in the stone's
  deadNotReclaimed set, are ignored and are not returned by this method.

  The output file is deleted if this method encounters an error.

  See the description of 
    #_mfcWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:
  for a description of the memory costs and performance optimizations.

  See the following methods in class System for details of accessing 
  and updating the mt statistics:
     mtThreadsLimit: sessionId
     mtThreadsLimit: sessionId setValue: newVal
     mtPercentCpuActiveLimit: sessionId
     mtPercentCpuActiveLimit: sessionId setValue: newVal
     mtMaxThreads: sessionId"

<primitive: 877>
| maxInt |
maxInt := SmallInteger maximum32bitInteger .
maxThreads _validateClass: SmallInteger ; _validateMin: 1 max: maxInt .
lockWaitTime _validateClass: SmallInteger; _validateMin: -1 max: maxInt .
aBufSize _validateClass: SmallInteger ; _validateIsPowerOf2 .
self _validatePercentage: cpuPercent .
aFileNameString ifNil:[ (anArray at: 3) _validateInstanceOf: Array ]
             ifNotNil:[ aFileNameString _validateKindOfClass: String ].
anArray _validateInstanceOf: Array.
(anArray size == 4) ifFalse:[anArray _error: #rtErrArgOutOfRange ].
(anArray at: 3) ifNil:[ aFileNameString _validateKindOfClass: String].
anArray isCommitted ifFalse:[ ArgumentError signal:'result array anArray is not a committed object'].
self _primitiveFailed: 
     #_fdcWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:toFile:resultArray:
     args: {  maxThreads . lockWaitTime . aBufSize . 
              cpuPercent . aFileNameString . anArray} .
^ self _uncontinuableError
%

category: 'Private'
method:
_check: anInt min: minValue maxInt32: aName
  | aMax |
  anInt < minValue ifTrue:[ 
    OutOfRange new name: aName min: minValue actual: anInt ; signal
  ].
  anInt > (aMax := 16r7FFFFFFF) ifTrue:[ 
    OutOfRange new name: aName max: aMax actual: anInt ; signal
  ]
%

category: 'Private'
method:
_scanPomWithMaxThreads: maxThreads waitForLock: lockWaitTime 
      pageBufSize: aBufSize percentCpuActiveLimit: percentCpu 
      identSet: anIdentitySet limit: aLimit
      scanKind: kind toDirectory: aStringOrNil

" This primitive method performs various scans of the Repository.
  The scans only take into account committed objects; i.e., this method 
  does not operate on objects in the Temporary Object Cache (TOC).

  This primitive uses a multithreaded algorithm to sweep the active
  data pages in the repository to gather the information requested.

  The maxThreads argument specifies the maximum number of threads
  that will be used during the operation.  The actual number of 
  active threads can be adjusted to a lower value automatically at runtime 
  by the main thread based on the percentCpuActiveLimit and by the user 
  by interactively setting the mtThreadLimit (see details below).

  The lockWaitTime argument is used to specify how many seconds method should
  wait while attempting to acquire the garbage collection (GC) lock.
  No other garbage collection operations may be started or in progress while
  this method is running.  Objects in the possibleDead or deadNotReclaimed 
  sets at the start of the scan are ignored by the scan.
  A lockWaitTime of -1 or 0 means wait forever.
 
  The pageBufSize, which must be a power of two, specifies the number 
  of pages to buffer per thread. This parameter in conjunction with the 
  maxThreads largely determines the memory footprint needed to perform 
  this operation.  The pageBufSize doesn't have much impact on the 
  performance of the scan, so a default size of 8 is probably sufficient.

  The percentCpu specifies a level of total cpu activity at which the 
  algorithm automatically inactivates threads to prevent overload 
  of system resources.

  This algorithm makes use of additional sessions (threads) to achieve 
  significant performance improvements.  It also makes space/time trade offs
  so that both heap memory and TemporaryObjectCache (TOC)
  resources are used.  In fact, this algorithm doesn't require much TOC, 
  except for the operations that return array results, so configuring this 
  process for a smaller TOC space is advantageous. 

  The memory space that is needed is variable and depends upon number of
  instances being searched for, the number found, the number of sessions 
  requested and the pageBufSize specified for each.

  The anIdentitySet argument should be initialized to contain the 
  classes whose instances are to be located.

  The aLimit argument specifies the maximum number of instances to report 
  for any class or for the objsLargerThan scan.  A value of 0 specifies
  that all instances are returned (warning - unlimited searches on a large
  database can run out of TOC space).

  The behavior of this method is dependent on the scanKind as follows:

   scanKind  Function       Result
   ===========================================================================
     0     listInstances  Returns an Array.  For each element of 
                          anIdentitySet, the result array contains 2 
                          elements: the total number of instances and 
                          an array of the instances (possibly limited in size).
                          (OP_LIST_INSTANCES in C code)
   
     1     listInstances  The input set must contain a single class.
                          The instances found are stored into the hidden
                          set: ListInstancesResult, and the number of 
                          elements found is the primitive return value.
                          (OP_LIST_INST_TO_HIDDEN in C code)

     2     listInstances  Returns an Array of pairs where the odd numbered 
                          elements are the classes specified in the IdentitySet
                          (in oop order) and the even numbered elements are 
                          the number of instances of the preceding class 
                          that were found.  (OP_LIST_INST_TO_FILES in C code)

                          If the toDirectory argument is nil, then only 
                          array described above is returned.  If not nil, 
                          then  Bitmaps containing the object ID's of 
                          the instances are written to binary bitmap 
                          files in the specified directory.  The resulting
                          bitmap files are named:
                             <ClassName>-<classOop>-instances.bm
                          where className is the name of the class and 
                          classOop is the object ID of the class.

     3    listInPageOrder This operation requires a result file specification
                          (passed in the toDirOrFile argument).  It is an 
                          error if the file already exists.  The oops of the 
                          objects in the repository that are instances
                          of the classes specified are written to the file
                          in page order and take 5 bytes each.   Returns a 
                          SmallInteger representing the total number of 
                          objects found.  (OP_LIST_INST_PAGE_ORDER)
 
     4    objsInSecPolicy Returns an Array.  For each element of 
                          anIdentitySet, the result array contains 2 
                          elements: the total number of instances and 
                          an array of the instances (possibly limited in size)
                          that have the corresponding objectSecurityPolicyId.
                          (OP_LIST_OBJS_IN_SEC_POLICY)

     5    objsInSecPolicy The input set must contain a single securityPolicy.
                          The instances found are stored into the hidden
                          set: ListInstancesResult, and the number of 
                          elements found is the primitive return value.
                          (OP_LIST_OBJS_IN_SEC_POLICY_TO_HIDDEN)

     6    objsInSecPolicy Returns an Array of pairs where the odd numbered 
                          elements are the security policy ids specified 
                          in the IdentitySet (in order) and the even numbered 
                          elements are the number of instances in the preceding 
                          security policy that were found. (OP_LIST_OBJS_IN_SEC_POLICY_TO_FILES)

                          If the toDirectory argument is nil, then only 
                          array described above is returned.  If not nil, 
                          then  Bitmaps containing the object ID's of 
                          the instances are written to binary bitmap 
                          files in the specified directory.  The resulting
                          bitmap files are named: 
                           segment<objectSecurityPolicyId>-objects.bm
                          where objectSecurityPolicyId is an element of anArray.

     7    objsLargerThan  The input set must contain a single SmallInteger
                          which is the target size for the scan. An array
                          of the objects which are larger than the target
                          size is returned. Result is limited by the
                          aLimit argument. (OP_OBJECTS_LARGER)

     8    countInstances  Returns an Array of pairs where the odd numbered 
                          elements are the classes specified in the IdentitySet
                          (in oop order) and the even numbered elements are 
                          the number of instances of the preceding class 
                          that were found. (OP_COUNT_INSTANCES)

   ===========================================================================

   This method aborts the current transaction.  Because the view can change,
   the results of the scan can vary depending on which view was used when 
   a given object was scanned.
   
   Scans the entire Repository once."

<primitive: 896>
| maxInt |
maxInt := SmallInteger maximum32bitInteger .
maxThreads _validateClass: SmallInteger ; _validateMin: 1 max: maxInt .
lockWaitTime _validateClass: SmallInteger ; _validateMin: -1 max: maxInt .
aBufSize _validateClass: SmallInteger;  _validateIsPowerOf2 .
self _validatePercentage: percentCpu  .
anIdentitySet _validateClass: IdentitySet .
aLimit _validateClass: SmallInteger ; _validateMin: 0 max: SmallInteger maximumValue .
kind _validateClass: SmallInteger ; _validateMin: 0 max: 14 .
aStringOrNil ifNotNil:[ aStringOrNil _validateKindOfClass: String ].
^ self _primitiveFailed: 
       #_scanPomWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:identSet:limit:scanKind:toDirectory:
       args: { maxThreads . lockWaitTime . aBufSize . percentCpu .
               anIdentitySet . aLimit . kind . aStringOrNil }
%

category: 'Multithreaded Scan Tuning'
method:
setMultiThreadedScanThreads: numThreads maxCpu: aPercent

"Change the number of threads active for a multithreaded scan. 
 Also changes the maximum CPU usage parameter if aPercent is not nil.

 Returns true if successful, false if the session running the multithreaded
 scan could not be determined.

 Raises an error if the session holding the GC lock is not running on this
 host.

 Raises an error if numThreads is greater than the maximum number of threads for
 which the scan was configured (i.e., the value of the maxThreads argument
 passed to the multithreaded scan method).

 Requires the SessionAccess and SystemControl privileges."

| sesId |
sesId := System sessionIdHoldingGcLock .
sesId == 0
  ifTrue:[ RepositoryError signal: 'No scan in progress' ]. 

self getMultiThreadedScanMaxThreads == 0
  ifTrue:[ RepositoryError signal: 'No scan in progress' ]. 

  self mtThreadsLimit: sesId setValue: numThreads .
aPercent == nil
  ifFalse:[ self mtPercentCpuActiveLimit: sesId setValue: aPercent ] .
^ true "success if we get here"
%

category: 'Multithreaded Scan Tuning'
method:
getMultiThreadedScanMaxThreads

"Answer the maximum number of threads that the current multithreaded scan
 is configured to use.  Answers 0 if no multithreaded scan is in progress.

 Raises an error if the session holding the GC lock is not running on this
 host or if the GC lock session is not performing a multithreaded repository
 scan.

 Requires the SessionAccess privilege."

| sesId |
sesId := System sessionIdHoldingGcLock .
sesId == 0
  ifTrue:[ RepositoryError signal: 'No scan in progress' ]. 

^ self mtMaxThreads: sesId
%

category: 'Multithreaded Scan Tuning'
method:
pauseMultiThreadedScan

"Suspends any multithreaded scan in progress by setting the number of threads 
 available to the scan to 0.  The session running the scan is not stopped or 
 killed, it is in a suspended state waiting to be resumed.

 Returns true if successful, false if the session running the multithreaded
 scan could not be determined.

 Raises an error if the session holding the GC lock is not running on this
 host or if the GC lock session is not performing a multithreaded repository
 scan.

 Examples of multithreaded scan operations include:
   markForCollection
   findDisconnectedObjects
   listInstances
   listReferences
   objectAudit

 Multithreaded scan operations always hold the GC lock.  However not all 
 operations which hold the GC lock are multithreaded scans.

 The scan may be resumed by executing one of the setMultiThreadedScan* methods
 in this class.

 Requires the SessionAccess and SystemControl privileges."

^ self setMultiThreadedScanThreads: 0  maxCpu: nil "nil means no change"
%

category: 'Multithreaded Scan Tuning'
method:
setMultiThreadedScanMaxAggressive

"Makes any multithreaded scan in progress as aggressive as possible.  This
 will cause the scan to complete as quickly as possible at the expense of
 consuming more system resources.

 Returns true on success, false if the multithreaded scan session could not
 be found on this host.

 Raises an error if the session holding the GC lock is not running on this
 host."

^ self setMultiThreadedScanThreads: 
       self getMultiThreadedScanMaxThreads maxCpu: 95 .
%

category: 'Multithreaded Scan Tuning'
method:
setMultiThreadedScanMinAggressive

"Makes any multithreaded scan in progress run as slowly as possible without 
 suspending the scan.  This will cause the scan to take much more time to 
 complete and will greatly reduce the system resources used by the scan.

 Requires the SessionAccess and SystemControl privileges.

 Returns true on success, false if the multithreaded scan session could not
 be found on this host.

 Raises an error if the session holding the GC lock is not running on this
 host."

self getMultiThreadedScanMaxThreads == 0
  ifTrue:[ RepositoryError signal: 'No scan in progress' ]. 
^ self setMultiThreadedScanThreads: 1 maxCpu: 60 .
%

category: 'Multithreaded Scan Tuning'
method:
setMultiThreadedScanMediumAggressive

"Makes any multithreaded scan in progress moderately aggressive.
 This is a compromise between completing as quickly as possible and the
 overall impact on the system.

 Returns true on success, false if the multithreaded scan session could not
 be found on this host.

 Raises an error if the session holding the GC lock is not running on this
 host."

| maxThds |
maxThds := self getMultiThreadedScanMaxThreads .

"Use 50% of the maximum number of threads configured."
maxThds := 1 min: (maxThds // 2) .

^ self setMultiThreadedScanThreads: maxThds maxCpu: 80 .
%

category 'Private'
method:
_mtStat: sesId at: statCode

"Returns the value of the specified stat.
 statCode values are:
    0 - mtMaxThreads
    1 - mtThreadsLimit
    2 - mtPercentCpuActiveLimit
 
 Requires the SessionAccess and SystemControl privileges.

 Raises an error if no session with the specified ID was found on this
 host."

<primitive: 878>
sesId _validateClass: SmallInteger .
statCode _validateClass: SmallInteger ; _validateMin: 0 max: 2 .
self _primitiveFailed: #_mtStat:at: args: { sesId . statCode }
%

category 'Private'
method:
_mtStat: sesId at: statCode put: newValue

"Sets the value of the specified stat.
 statCode values are:
    1 - mtThreadsLimit
    2 - mtPercentCpuActiveLimit

 Requires the SessionAccess and SystemControl privileges.

 Raises an error if no session with the specified ID was found on this
 host or if the session is not performing a multithreaded repository
 scan."

<primitive: 879>

sesId _validateClass: SmallInteger .
statCode _validateClass: SmallInteger ; _validateMin: 0 max: 2 .
newValue _validateClass: SmallInteger .
self _primitiveFailed: #_mtStat:at:put: args: { sesId . statCode . newValue }
%

category: 'Multithreaded Scan Tuning'
method:
mtMaxThreads: sessionId

"Returns the current value of the mtMaxThreads for the specified sessionId.
 The process executing this method must be on the same machine as the specified session.

 Requires the SessionAccess and SystemControl privileges."

^ self _mtStat: sessionId at: 0
%

category: 'Multithreaded Scan Tuning'
method:
mtThreadsLimit: sessionId

"Returns the current value of the mtThreadsLimit for the specified sessionId.
 The process executing this method must be on the same machine as the specified session.

 Requires the SessionAccess and SystemControl privileges."

^ self _mtStat: sessionId at: 1
%

category: 'Multithreaded Scan Tuning'
method:
mtThreadsLimit: sessionId setValue: newVal

"Sets a new value for the mtThreadsLimit for the specified sessionId.
 The process executing this method must be on the same machine as the specified session.
 The newVal may be 0, which can be used to temporarily pause the operation in progress.
 The maximum newVal allowed is determined by the current value of mtMaxSessions.
 Requires the SessionAccess and SystemControl privileges."

^ self _mtStat: sessionId at: 1 put: newVal
%

category: 'Multithreaded Scan Tuning'
method:
mtPercentCpuActiveLimit: sessionId

"Returns the current value of the mtPercentCpuActiveLimit for the specified sessionId.
 The process executing this method must be on the same machine as the specified session.

 Requires the SessionAccess and SystemControl privileges."

^ self _mtStat: sessionId at: 2
%

category: 'Multithreaded Scan Tuning'
method:
mtPercentCpuActiveLimit: sessionId setValue: newVal

"Sets a new value for the mtThreadsLimit for the specified sessionId.
 The process executing this method must be on the same machine as the specified session.
 The newVal may be in the range from 0 to 100.

 Requires the SessionAccess and SystemControl privileges."

^ self _mtStat: sessionId at: 2 put: newVal
%

category: 'Audit and Repair'
method:
_objectAuditWithMaxThreads: maxThreads waitForLock: waitTimeSeconds 
                       pageBufSize: aBufSize 
                       percentCpuActiveLimit: percentCpu
                       csvFile: aFileName
                       repair: doRepair 

"Performs a multi-threaded object audit on all objects in the repository.

 Requires SystemControl privileges.

 See _scanPomWithMaxThreads for definitions of maxThreads, waitTimeSeconds, 
 aBufSize, cpuPersent.

 The following errors can be generated:
   BadClassId     - object references classId which does not exist.
   BadClass       - object class reference is not a class.
   BadFormat      - the object format disagrees with the class format.
   BadSecPolicy   - the object security policy is invalid.
   BadPhysSize    - the object physical size is invalid.
   BadLogicalSize - the object logical size is greater than the physical size.
   BadLength      - the object length > bodySize.
   BadBody        - the object body contains a reference to oop illegal.
   BadReference   - the object references a non existent object at offset.
   BadOt          - is in page but has an invalid OT entry.
   InFreeOops     - the object is an a page and in the FreeOop list.
   FreeOop        - does not exist but is neither in dead nor in the FreeOop list.
   BadObjId       - invalid objectId found in object.
   BadDependency  - the OT indicates the object has a dependency but it is not in depMap.
   BadDepMapKey   - the objectId is in the depMapKeys but is not a valid object.
   BadDepMapValue - the objectId is in the depMapValues but is not a valid object.
   BadDepTag      - the object is in the depMap, but the OT entry is not tagged
 
 These are always printed to the GciErrorLog.  If a valid file name is provided for the 
 csvFile (comma separated values) argument then a line is written to the file for each 
 error encountered.  The lines contain the following information: 
   1. ErrorKind - the name associated with each of the errors defined above.
   2. ObjectId  - the objectId for the object with an error.
   3. ClassName  - the name of the object's class.
   4. Offset    - the offset (or other integer value, e.g. size)
   5. Reference - the reference that does not exist.
 If no errors are found, the file is deleted.

 If doRepair is true, the repairs are performed after the entire scan of the repository
 is complete.  It is possible that if a very large number of errors are found the 
 system may run out of memory to record the information needed to fix the errors.
 In this case the following warning message will be generated:
    WARNING: Ran out of memory to record errors, remaining errors will not be fixed.
 If this is found in the output, then the errors before this message will be fixed, but
 any that are printed after this message will not be fixed and the audit will need to be
 rerun to correct any additional errors.  In addion an additional warning is generated at 
 the end of the repair output.

 During the repair phase, each error will be printed out along with a description of the
 action taken to repair the problem.  The system will commit after every 2000 repair 
 operations to avoid overflowing temporary memory space.

 The only errors that cannot be corrected by the audit code are the InFreeOops and FreeOop
 error kinds.  To fix these you will need to rebuild the FreeOop list - see admin guide for 
 instructions on how to rebuild the FreeOop list."

<primitive: 392>
| maxInt |
maxInt := SmallInteger maximum32bitInteger .
maxThreads _validateClass: SmallInteger; _validateMin: 1 max: maxInt .
waitTimeSeconds _validateClass: SmallInteger ; _validateMin: -1 max: maxInt .
aBufSize _validateClass: SmallInteger ; _validateIsPowerOf2 .
self _validatePercentage: percentCpu .
aFileName _validateKindOfClass: String.
doRepair _validateClass: Boolean.
^ self _primitiveFailed:
       #_objectAuditWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:csvFile:repair:
       args: { maxThreads . waitTimeSeconds . aBufSize .
               percentCpu . aFileName . doRepair }
%

category: 'Backup and Restore'
method:
fullBackupTo: fileNameOrArrayOfNames

"Writes a full backup to an individual file or an array of files containing 
 the most recently committed version of the receiver as of the time the 
 method is executed.  If an array of files is specified, the backup
 will partition the data equally across the files.

 This method always writes the file without compression. To compress
 the backup files use fullBackupGzCompressedTo: or fullBackupLz4CompressedTo

 The fileName argument may use GemStone Network Resource String syntax.
 For example, this may be used to access a file on another machine, provided
 a GemStone NetLDI process is running on the remote machine. fileName cannot
 specify a raw partition. 

 If the device containing the file runs out of space, then the backup 
 terminates with a system I/O error and the partially written backup 
 file is deleted.

 If the session contains uncommitted changes to the repository, the method 
 signals a error: #rtErrAbortWouldLoseData, to indicate that data could 
 be lost.  Otherwise, it puts the session in autoBegin mode and performs 
 the backup operation.  The session may be aborted a number of times during
 the backup to prevent a commit record backlog.  

 When the backup completes, the session is set to manualBegin mode
 mode so that it does not reference a commit record  that would cause 
 the repository to grow.

 Returns true if the backup was completed.

 This method requires the FileControl privilege.

 This method performs the backup using multiple threads.  
 The number of threads is automatically set to 2 times the 
 number of extents in the repository.  This can be overridden by
 executing:
     SessionTemps current at: #GsOverrideNumThreads put: numThreads.
 where numThreads can be any value between 1 and 4 * numCpus before
 executing the backup method.
 The performance can be modified during the run by 
 updating the Multithreaded Scan Tuning methods.

 A GciHardBreak during this method will terminate the session."

 ^ self fullBackupTo: fileNameOrArrayOfNames MBytes: 0
%

category 'Private'
method:
_checkMbyteLimit: aVal

  ^ aVal class == SmallInteger and:[ aVal >= 0 and:[ aVal <= (4000 * 1024) ]]
%

category 'Private'
method:
_validateFileNames: fileNames limits: mByteLimit
  | limitArray |
  System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ].
  fileNames _validateInstanceOf: Array.
  fileNames size == 0 ifTrue: [ fileNames _error: #rtErrArgOutOfRange args:{ 1 . 256 }].
  1 to: fileNames size do: [ :i | (fileNames at: i) _validateKindOfClass: String].
  (mByteLimit isKindOfClass: SmallInteger)  ifTrue:[
     ( self _checkMbyteLimit: mByteLimit ) ifFalse:[
           mByteLimit _error: #rtErrArgOutOfRange args: {0 . 4000 * 1024} ].
      limitArray := Array new: fileNames size.
      1 to: limitArray size do: [ :i |
        limitArray at: i put: mByteLimit 
      ] 
   ] ifFalse: [
      (mByteLimit isKindOfClass: Array)
         ifTrue:  [ limitArray := mByteLimit copy ]
         ifFalse: [ ( ArgumentTypeError new name: 'mByteLimit' expectedClass: 
                    { Array . SmallInteger } actualArg: mByteLimit ) signal].
      1 to: (fileNames size min: limitArray size) do:[:j | | v |
        (self _checkMbyteLimit: (v := limitArray at: j)) ifFalse:[ 
          OutOfRange signal: 'element ', j asString , ' of limits Array is invalid'
        ].
      ].
      limitArray size + 1 to: fileNames size do:[:k|
         limitArray at: k put: 0.
      ].
   ].
  ^limitArray
%


category: 'Backup and Restore'
method:
fullBackupTo: fileNames MBytes: mByteLimit 

"The same as fullBackupTo: except that it accepts an array of filenames 
 as the first argument so that the backup may be spread across multiple
 files on different devices.  In addition, the mByteLimit argument 
 specifies the maximum size of the files.  The mByteLimit may be a single 
 integer which is applied to each file in the fileNamesArray or
 an array of integer values, where the array must be the same size 
 as the fileNamesArray. 
 
 The size of the last file in the array may be larger than the specified
 limit. In order to avoid backup failures, the last file does not limit
 the size, so backup will complete unless the device used for the last
 backup file becomes full. 
 
 A zero value for the megabyte limit means that there is no limit on the 
 size of the resulting file(s) and is an easy way to specify that the backup
 should partition the data equally across the files.  The maximum value
 of a backup file even with no limit is 4TB. A mByteLimit value less 
 than zero generates an error."

^ self fullBackupTo: fileNames MBytes: mByteLimit compressKind: 0 bufSize: 8
%

category: 'Deprecated'
method:
fullBackupCompressedTo: fileName

"The same as fullBackupTo: except that the backup file is compressed
 using gzip as it is written.

 If the fileName is does not end in '.gz' then a '.gz' suffix is added.

 Deprecated.  Use the method:
    fullBackupGzCompressedTo: fileName
 instead of this one."

 self deprecated: 'Repository>>fullBackupCompressedTo: deprecated in v3.4.'.
 ^ self fullBackupGzCompressedTo: fileName
%

category: 'Backup and Restore'
method:
fullBackupGzCompressedTo: fileNameOrArrayOfNames

"The same as fullBackupTo: except that the backup file is compressed
 using gzip as it is written.

 If the fileName is does not end in '.gz' then a '.gz' suffix is added."

 ^ self fullBackupGzCompressedTo: fileNameOrArrayOfNames MBytes: 0
%

category: 'Backup and Restore'
method:
fullBackupLz4CompressedTo: fileNameOrArrayOfNames

"The same as fullBackupTo: except that the backup file is compressed
 using LZ4 as it is written.

 If the fileName is does not end in '.lz4' then a '.lz4' suffix is added."

 ^ self fullBackupLz4CompressedTo: fileNameOrArrayOfNames MBytes: 0
%

category: 'Deprecated'
method:
fullBackupCompressedTo: fileNames MBytes: mByteLimit

"The same as fullBackupTo:MBytes: except that the backup file 
 is compressed using gzip as it is written.

 If the fileName is does not end in '.gz' then a '.gz' suffix is added.
 
 Deprecated.  Use the method:
   fullBackupGzCompressedTo:MBytes:
 instead."

  self deprecated: 'Repository>>fullBackupCompressedTo:MBytes: deprecated in v3.4.' .
  ^ self fullBackupGzCompressedTo: fileNames MBytes: mByteLimit
%

category: 'Backup and Restore'
method:
fullBackupGzCompressedTo: fileNames MBytes: mByteLimit

"The same as fullBackupTo:MBytes: except that the backup file 
 is compressed using gzip as it is written.

 NOTE: gz compression only allows a buffer size of one, so the bufSize argument
 does not get used.

 If the fileName is does not end in '.gz' then a '.gz' suffix is added."

^ self fullBackupTo: fileNames MBytes: mByteLimit compressKind: 1 bufSize: 1
%

category: 'Backup and Restore'
method:
fullBackupLz4CompressedTo: fileNames MBytes: mByteLimit

"The same as fullBackupTo:MBytes: except that the backup file 
 is compressed using lz4 compression as it is written.

 If the fileName is does not end in '.lz4' then an '.lz4' suffix is added."

^ self fullBackupTo: fileNames MBytes: mByteLimit compressKind: 2 bufSize: 16
%

category: 'Backup and Restore'
method:
fullBackupTo: fileNames MBytes: mByteLimit compressKind: anInt bufSize: count

"The same as fullBackupTo:MBytes: except that the backup file is compressed 
 based on the value of anInt (see primitive for definition of compressionKinds).

 
"

  | limitArray fnArray |
  (fileNames isKindOfClass: Array) 
    ifTrue: [ fnArray := fileNames ]
    ifFalse: [ fnArray := { fileNames } ].
  limitArray := self _validateFileNames: fnArray limits: mByteLimit.
  ^ self _fullBackupTo: fnArray MBytes: limitArray compressKind: anInt 
                        bufSize: count numThreads: self _getNumThreads
%

category: 'Backup and Restore'
method:
_fullBackupTo: fileNames MBytes: limitArray compressKind: anInt 
                         bufSize: count numThreads: threadCount

"Private.  Provides implementation of backup methods.

 Valid compressKinds:
   0 - no compression
   1 - zlib (aka gzip) compression.
   2 - lz4 compression.

 The bufSize argument controls the size of the buffer used to write records to the file.
 The count argument specifies the number of 128KB backup records contained in the buffer.
 The values allowed are 1 (128KB) through 8196 (1GB).

 Returns true if the backup was completed."

  <primitive: 394>
  | a b |
  fileNames _validateInstanceOf: Array.
  limitArray _validateInstanceOf: Array.
  anInt _validateClass: SmallInteger .
  threadCount _validateClass: SmallInteger .
  ((anInt < 0) or:[ anInt > 2])
    ifTrue:[ anInt _error: #rtErrArgOutOfRange args:{ 0 . 2 } ] .
  count _validateClass: SmallInteger .
  (a := fileNames size) == (b := limitArray size) ifFalse:[
    ArgumentError signal: 'number of fileNames (' , a asString, 
') not equal to number of limits (', b asString , ')' 
  ].
  self _primitiveFailed: #_fullBackupTo:MBytes:compressKind:bufSize: .
%

category: 'Backup and Restore'
method:
restoreStatusInfo

"Returns an Array describing the current restore status of the Repository,  
 including the next transaction log file or backup file required to continue
 the restore.  The Array contains the following elements:
   
   1: a String describing the restore status.
   2: a SmallInteger,
          0 = restore not active,
	  1 = restoreFromBackup withErrors
          2 = active from log
	  3 = restoreFromLog withErrors
   3: a SmallInteger, next fileId to restore 
   4: a SmallInteger, -1 (unused element)
   5: an Integer,  firstStartupTime of the repository   
   6: an Integer,  commitSequence.high of the repository
   7: an Integer,  commitSequence.low of the repository
   8: a SmallInteger, fileId of oldest tranlog that must be present for restore.
   9: a SmallInteger, fileId of current restore position
   10: a SmallInteger, blockId of current restore position
   11: a String, the time of the last checkpoint restored to.
   12: a SmallInteger, 0 no log receiver, 1 log receiver is connected to stone.
   13: a SmallInteger, 0 or the posix time in seconds of the failover timestamp
				for which continous restore was stopped.
   14: a SmallInteger, 0 or the posix time in seconds of the time returned
                    from the last execution of suspendCommitsForFailover.
                    

 If restore is not active, the elements 3 to 11 are nil.
 Element 11 may be nil if restore has not yet replayed any checkpoints
 from tranlogs.

 Restore status is an attribute of the Repository, not of the session.
 The restore position in the tranlogs persists across logout/login 
 and stopstone/startstone.

 Element 13 of the result persists across logout/login only"

^ System _zeroArgPrim: 143
%

category: 'Backup and Restore'
method:
restoreStatus
  
"Returns a String describing the current restore status of the Repository,
 including the next transaction log file or backup file required to continue
 the restore, and the oldest transaction log file that must be present
 for a restore from log."

^ self restoreStatusInfo at: 1
%
     
category: 'Backup and Restore'
method:
restoreActive

"Returns true if the restore is currently active, false if not."

^(self restoreStatusInfo at: 2) ~= 0
%

category: 'Backup and Restore'
method:
restoreStatusNextFileId
    
"Returns a SmallInteger, the fileId of the next transaction log or backup that
 should be restored, or nil if restore not active."
  
^ self restoreStatusInfo at: 3
%

category: 'Backup and Restore'
method:
restoreStatusOldestFileId
 
"Returns a SmallInteger, the fileId of the oldest transaction log
 that needs to be present for the next restore from log operation,
 or nil if restore not active."
 
^ self restoreStatusInfo at: 8
%   

category: 'Backup and Restore'
method:
restoreStatusTimeRestoredTo
 
"Returns a String which represents time of the last checkpoint 
 that the repository was restored to or nil if restore not active,
 or the restored-to time is not available. 

 The result will be nil if restore is active but has not yet 
 replayed any checkpoints from tranlogs."
 
^ self restoreStatusInfo at: 11 
%   

category: 'Backup and Restore'
method:
commitRestore

"Terminates a restore operation and puts the system back into a normal 
 operating mode in which commits are allowed and reenables logins.

 The repository must be in the restoreFromLogs state, otherwise this 
 method generates an error.

 If the logs have not been restored to the end of the known logs, i.e., 
 the last restore operation was a restoreToEndOfLog: or restoreToPointInTime:
 then a warning is issued and the restore is stopped at the current 
 location.  This use of commitRestore can result in failure to restore 
 all previously committed transactions.  However, this allows a Repository 
 to be restored as far as practical when some log files are lost or corrupted.

 Upon successful completion, the session is automatically logged out and
 the RestoreLogSuccess error (4048) is generated.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser.

 Because any session that logs in or performs a restore operation 
 is prevented from doing normal commits, this session is automatically
 logged out when the restore operation completes."

self restoreActive ifFalse: [
  ImproperOperation signal: 
     'You cannot commitRestore without first executing restoreFromBackup'].
^ System _zeroArgPrim: 141
%

category: 'Backup and Restore'
method: Repository
commitRestoreWithLostData
"A special case of commitRestore that allows the commit to occur when
 the backup file was corrupted and there was lost data.

 Terminates a restore operation and puts the system back into a normal 
 operating mode in which commits are allowed and reenables logins.

 The repository must be in the restoreFromLogLostData state, otherwise this 
 method generates an error.

 If the logs have not been restored to the end of the known logs, i.e., 
 the last restore operation was a restoreToEndOfLog: or restoreToPointInTime:
 then the commitRestoreWithLostData should be used to allow the repository
 to be restored as far as practical when some log files are lost or corrupted.

 Upon successful completion, the session is automatically logged out and
 the RestoreLogSuccess error (4048) is generated.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser.

 Because any session that logs in or performs a restore operation 
 is prevented from doing normal commits, this session is automatically
 logged out when the restore operation completes."

self restoreActive ifFalse: [
  ImproperOperation signal: 
     'You cannot commitRestore without first executing restoreFromBackup'].
^ System _zeroArgPrim: 144
%

category: 'Backup and Restore'
method:
restoreFromBackup: fileNameOrArrayOfNames

"Disables logins and starts a full restore of the repository based 
 on the contents of the specified backup file.

 Restored objects are clustered in a manner that is similar, but not
 necessarily identical to the clustering at the time the  backup file 
 was created.  If the Repository being restored into has the same 
 number of extents as the system had when the backup file was created, 
 then distribution of objects within extents is preserved unless one 
 of the extents becomes full.  If the number of extents is different 
 than the number when the backup was created, then the current 
 DBF_ALLOCATION_MODE configuration controls the distribution of 
 objects across the extents.  

 If the backup file was made when in partial-logging mode, then logins
 are reenabled and the system is fully operational when the restore 
 completes.  If the backup was made when in full-logging mode, then 
 the system is not fully operational until tranlogs have been 
 restored and the commitRestore method is executed.

 If the file has a .gz or .lz4 suffix, then the file can be specified
 with or without the suffix.  The restore checks for files that match
 the filename.  If it doesn't find a match and the filename doesn't
 contain a suffix, then the restore attempts to find a match first
 with the .gz suffix and then with the .lz4 suffix.
 
 To minimize the size of the resulting repository the stone should be
 restarted on a copy of the initial repository (copied from 
 $GEMSTONE/bin/extent0.dbf).

 To optimize the time to restore the backup, the exents in the new
 repository should be pregrown to the minimum expected size of 
 the restored repository.

 Upon successful completion, the session is automatically logged out and
 the RestoreBackupSuccess error (4046) is generated.

 A GciHardBreak during this method will terminate the session.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser.

 This method performs the restore using multiple threads.  
 The number of threads is automatically set to 2 times the 
 number of extents in the repository.  This can be overridden by
 executing:
     SessionTemps current at: #GsOverrideNumThreads put: numThreads.
 where numThreads can be any value between 1 and 4 * numCpus before
 executing the restore method.
 The performance can be modified during the run by 
 updating the Multithreaded Scan Tuning methods.

 The restore must be run from a process on the same machine as the stone.
 If it is attempted from a remote machine, a repository error (2734) is
 reported.
"
| arr | 
arr := fileNameOrArrayOfNames.
fileNameOrArrayOfNames _isArray ifFalse: [ arr := { fileNameOrArrayOfNames } ].
^ self restoreFromBackups: arr
%

category: 'Backup and Restore'
method:
restoreFromBackups: arrayOfFileNames

"Similar to restoreFromBackup: except that it accepts an array of fileNames."

^ self _restoreBackups: arrayOfFileNames scavPercentFree: 100 
                        bufSize: 8 numThreads: self _getNumThreads
%

category: 'Backup and Restore'
method:
restoreFromBackups: arrayOfFileNames scavengePagesWithPercentFree: aPercent

"Same as restoreFromBackups, except that restored pages that have greater
 than aPercent free space are added to the scavengable pages at the end
 of the restore.  WARNING: a small percentage, less than 5 percent can cause
 the reclaim gems to be very busy after the restore."

^ self _restoreBackups: arrayOfFileNames scavPercentFree: aPercent 
                        bufSize: 8 numThreads: self _getNumThreads
%

category: 'Backup and Restore'
method:
_restoreBackups: arrayOfFileNames scavPercentFree: aPercent 
                 bufSize: count numThreads: threadCount

"Private.  Provides implementation of restoreBackups."

<primitive: 822>
arrayOfFileNames _validateInstanceOf: Array.
count _validateClass: SmallInteger .
threadCount _validateClass: SmallInteger .
self _validatePercentage: aPercent .
self _primitiveFailed: #restoreBackups:scavPercentFree:bufSize: .
self _uncontinuableError
%

category 'Private'
method:
_checkRestoreActive

self restoreActive ifFalse: [
  ImproperOperation signal: 
     'You cannot restoreLogs without first executing restoreFromBackup'].
%

category: 'Backup and Restore'
method:
restoreFromCurrentLogs

"After a restoreFromBackup: successfully completes this method may be 
 executed to replay transactions which occurred since the backup was made.
 If the restorestatus of the repository is not activeFromLog then an error 
 is generated.
 
 This method attempts to sequentially replay all transaction logs from 
 the first transaction log required to the end of the current logs. 

 If previous restore operation was a variant of restoreFromArchiveLogs,
 the last archive log will be reopened and any data written since
 the last EOF postion restored, before restoring the first current log.

 The restoreStatus or restoreStatusNextFileId methods can be used to determine 
 the next file required for a restoreLogs operation.

 If some of the tranlogs have been archived and are not in the locations
 configured with the STN_TRAN_LOG_DIRECTORIES, then the restoreFromArchiveLogs:
 methods should be used to restore up until the next files are in the 
 current tranlog directories.

 When opening a log file on a file system, if the filename ends in '.gz',
 the file is expected to be in gzip compressed format.  If the filename
 ends in '.lz4', the file is expected to be in lz4 compressed format.

 Restoring a compressed tranlog from a raw partition is not supported.

 Upon successful completion, the session is automatically logged out and
 the RestoreLogSuccess error (4048) is generated.

 If an error occurs, such as a corrupted tranlog or a missing tranlog in 
 the sequence then an error is generated and the session is not loggged out.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser."

self _checkRestoreActive.
^ self _restoreLogs: -1 time: 0 archiveLogs: 0
%


category: 'Backup and Restore'
method:
restoreToEndOfLog: fileId

"Similar to restoreFromCurrentLogs except that the restore stops when it has 
 completed replaying the transactions in the file specified by fileId.

 The fileId argument must be a positive SmallInteger.

 A subsequent restore operation may be used to continue restoring logs
 past the log location at which this operation stopped."

self _checkRestoreActive.
fileId _validateClass: SmallInteger.
fileId > 0 ifFalse:[ fileId _error: #rtErrArgNotPositive ].
^ self _restoreLogs: fileId time: 0 archiveLogs: 0
%

! edited for 43026
category: 'Private'
method:
_checkPointInTime: aDateTime

| checkpointTime aTimeT |
self restoreStatusTimeRestoredTo ifNotNil:[ :tStr |
  checkpointTime := DateTime fromString: ((tStr) copyFrom: 1 to: 17) usingFormat: #( 1 2 3 $/ 1 2 $: true true false ).
  aDateTime < checkpointTime  ifTrue: [ 
    ImproperOperation signal: ( 'aDateTime ' , aDateTime asString ,  
      ' is less than restoreStatusTimeRestoredTo ' ,
     self restoreStatusTimeRestoredTo)
  ].
].
aTimeT := aDateTime asSecondsGmt - 2177452800.
(aTimeT < 0 or:[ aTimeT > 16r7fffffff]) ifTrue:[
  aDateTime _error: #rtErrArgOutOfRange args:{ 0 . 16r7fffffff }
  ].
^ aTimeT
%

category: 'Backup and Restore'
method:
restoreToPointInTime: aDateTime

"Similar to restoreToEndOfLog except that the restore stops the restore at 
 the first checkpoint which originally occurred at or after aDateTime.

 An error is generated if aDateTime precedes the time of the last restored
 checkpoint."

| aTimeT |
self _checkRestoreActive.
aTimeT := self _checkPointInTime: aDateTime.
self _restoreLogs: -1 time: aTimeT archiveLogs: 0
%

category: 'Backup and Restore'
method:
_restoreLogs: fileId time: aTime archiveLogs: useArchiveInt

"Private.  Provides implementation of restoreFromLogs.
 If _primSetArchiveLogDirectories was called and
   useArchiveInt == 1, archive logs will be used.
   useArchiveInt == 2, continuous restore entered using archive logs.
   useArchiveInt == 3, stop continuous restore.
 "

<primitive: 395>
fileId _validateClass: SmallInteger; _validateMin: 1 max: SmallInteger maximum32bitInteger .
aTime _validateClass: SmallInteger; _validateMin: 0 max: SmallInteger maximumValue .
useArchiveInt _validateClass: SmallInteger; _validateMin: 0 max: 3 .
useArchiveInt >= 2 ifTrue:[
  fileId == -1 ifFalse:[  
    ArgumentError signal: 'fileId must be -1 for continuous restore'.
  ].
  aTime == 0 ifFalse:[
    ArgumentError signal: 'aTime must be 0 for continuous restore'.
  ].
].
self _primitiveFailed: #_restoreLogs:time:archiveLogs:.
self _uncontinuableError
%

category: 'Backup and Restore'
method:
restoreFromArchiveLogs: archiveLogDir

"Similar to restoreFromLogs except that the restore operation looks
 for the logs to restore in the directory or array of directories 
 specified instead of the currently configured STN_TRAN_LOG_DIRECTORIES.

 archiveLogDir may be a single directory specification, 
 or an Array  of directory specifications.
 
 Each directory specification must be a String that names a file system 
 directory.  It is an error if any of the directories specified does
 not exist.  The directory specified may be empty.

 When opening a log file on a file system, if the filename ends in '.gz',
 the file is expected to be in gzip compressed format.  If the filename
 ends in '.lz4', the file is expected to be in lz4 compressed format.
 
 Environment variables , if any, in the directory specifications 
 are expanded using the environment of the gem  or topaz -l) process
 running this session, before the directory specifications are passed to 
 stone for execution of the restore operation.  

 Terminates when all of the logs in the specified directories have
 been restored or if an error is discovered while processing the logs."

self restoreFromArchiveLogs: archiveLogDir withPrefix: nil
%

category: 'Backup and Restore'
method:
restoreFromArchiveLogs: archiveLogDir withPrefix: aPrefix

"Similar to restoreFromLogs except that the restore operation looks
 for the logs to restore in the directory or array of directories 
 specified instead of the currently configured STN_TRAN_LOG_DIRECTORIES.
 Each directory specification must be a String that names a file system 
 directory.  It is an error if any of the directories specified does
 not exist.  The directory specified may be empty.
 
 aPrefix specifies file name pattern in sequence instead of standard tranlog.
 aPrefix must be astring if nil default tranlog<n> is used
 
 Terminates when all of the logs in the specified directories have
 been restored or if an error is discovered while processing the logs."

self _checkRestoreActive.
self _setArchiveLogs: archiveLogDir withPrefix: aPrefix.
self _restoreLogs: -1 time: 0 archiveLogs: 1 .
%

method:
_setupContinuousRestore: archiveLogDir

  | ready printed count |
  ready := false .
  printed := false .
  count := 0 .
  [ ready ] whileFalse:[ | info |
    info := self restoreStatusInfo .
    (info at: 2) == 0 ifTrue:[ self _checkRestoreActive ].
    (info at: 12) == 1 ifTrue:[ 
      ready := true 
    ] ifFalse:[
      printed ifFalse:[ printed := true .
         GsFile gciLogServer:'  [Info:] Waiting for a log receiver to connect to stone.' ].
      count := count + 1 .
      count > 15 ifTrue:[ ImproperOperation signal:
  	'continuousRestore cannot start, no log receiver connected to stone.' ].
      System sleep: 1 .
    ]
  ].
  self _setArchiveLogs: archiveLogDir withPrefix: nil.
%

! edited comments for fix 47567
method:
continuousRestoreFromArchiveLogs: archiveLogDir

"Similar to restoreFromArchiveLogs: except that stone enters
 continuous restore mode, and control is returned to this session
 executing this method. Intended for use on a hot standby stone 
 where the archive logs are being written by a log receiver process
 which is receiving data from the master stone's log feeder process.
 
 Requires at least one tranlog already exist in directory(s) specified
 in archiveLogDir. It is recommended to start both logsender and 
 logreceiver before executing this method.

 Waits up to 15 seconds for a log receiver to connect to stone, after
 which if no log recever is detected, signals an error.

 The restore can be terminated by  Repository >> stopContinuousRestore.

 Repository >> restoreStatus is usable while the continuous restore
 is running.
"

  self _setupContinuousRestore: archiveLogDir.
  self _restoreLogs: -1 time: 0 archiveLogs: 2 .
%

method:
continuousRestoreFromArchiveLogs: archiveLogDir withDelay: seconds

"Similar to continuousRestoreFromArchiveLogs except that stone enters
 continuous restore mode with the specified delay.

 Records are transferred immediatly to the standby system but the
 processing of commits  and checkpoints are delayed by the specifed 
 number of seconds to allow the system to be stopped to avoid 
 restoring an operation on the database that is not desired.
"
  self _setupContinuousRestore: archiveLogDir.
  self _restoreLogs: -1 time: seconds archiveLogs: 2 .
%

method:
stopContinuousRestore

"Terminate a previous continuousRestoreFromArchiveLogs:.
 If a delay was specified, then stop immediately, otherwise
 processes the tranlogs found in the archive logs directory
 to be restored, but does not wait for more logs to be
 received by the logreceiver process.
 Has no effect if no such restore is active."

  self _checkRestoreActive.
  self _restoreLogs: -1 time: 0 archiveLogs: 3
%


category: 'Backup and Restore'
method:
restoreFromArchiveLogs: archiveLogDir toEndOfLog: fileId

"Similar to restoreFromArchiveLogs: except that the restore stops 
 when the end of the log with fileId is reached."

self restoreFromArchiveLogs: archiveLogDir toEndOfLog: fileId withPrefix: nil
%

category: 'Backup and Restore'
method:
restoreFromArchiveLogs: archiveLogDir toPointInTime: aDateTime

"Similar to restoreFromArchiveLogs: except that the restore stops at 
 the first checkpoint which originally occurred at or after aDateTime.

 An error is generated if aDateTime precedes the time of the last restored
 checkpoint."

 self restoreFromArchiveLogs: archiveLogDir toPointInTime: aDateTime withPrefix: nil
%

category: 'Backup and Restore'
method:
restoreFromArchiveLogs: archiveLogDir toEndOfLog: fileId withPrefix: prefixString

"Similar to restoreFromArchiveLogs:toEndOfLog: except that the specified
 prefixString is used instead of the value of the configuration parameter
 specified by STN_TRAN_LOG_PREFIX."

self _checkRestoreActive.
self _setArchiveLogs: archiveLogDir withPrefix: prefixString.
self _restoreLogs: fileId time: 0 archiveLogs: 1
%

category: 'Backup and Restore'
method:
restoreFromArchiveLogs: archiveLogDir toPointInTime: aDateTime withPrefix: prefixString

"Similar to restoreFromArchiveLogs:toPointInTime: except that the specified
 prefixString is used instead of the value of the configuration parameter
 specified by STN_TRAN_LOG_PREFIX."

| aTime |
self _checkRestoreActive.
aTime := self _checkPointInTime: aDateTime.
self _setArchiveLogs: archiveLogDir withPrefix: prefixString.
^ self _restoreLogs: -1 time: aTime archiveLogs: 1
%

category: 'Private'
method:
_validateString: aString

aString == nil ifFalse: [
  (aString class isSubclassOf: String) ifFalse:[ 
    aString _errorExpectedClass: String ].
  aString size > 1023 ifTrue:[ aString _error: #errArgTooLarge args:{ 1023 } ].
  ].
%

category: 'Private'
method:
_setArchiveLogs: dirArg withPrefix: prefixString
  "dirArg may be a String or an Array of Strings"
| logDirs |
dirArg _isOneByteString
   ifTrue: [ logDirs := { dirArg } ]
   ifFalse: [ logDirs := dirArg ].
logDirs _validateClass: Array.
logDirs size < 1 ifTrue:[
  logDirs _error: #errArgTooSmall args:{ 1 }.
].
logDirs do:[ :aString | aString _validateKindOfClass: String ].
prefixString ifNotNil:[  prefixString _validateKindOfClass: String ].
^ self _primSetArchiveLogDirectories: logDirs withPrefix: prefixString
%

category: 'Private'
method:
_primSetArchiveLogDirectories: logDirs withPrefix: prefixString

"Send archive log directories specification to Stone.  Sender has
 done class kind checks of all the arguments.  Stone checks for
 existence of each directory.  arrayOfDirectorySpecs should be non-empty.
 If the prefixString is nil, the STN_TRAN_LOG_PREFIX configuration 
 parameter is used."

<primitive: 469>
logDirs _validateClass: Array .
self _primitiveFailed: #_primSetArchiveLogDirectories:withPrefix:
     args: { logDirs . prefixString }
%

! deleted setArchiveLogDirectory:
! deleted setArchiveLogDirectories:

category: 'Backup and Restore'
method:
suspendCommitsForFailover
  
  ^ System suspendCommitsForFailover
%
category: 'Backup and Restore'
method:
resumeCommits
  
  ^ System resumeCommits
%


!
! Feature 42921
!
category: 'Deprecated (Multi Ref Path)'
method:
_findAllReferencePathsToObjects: anArray limitObjArray: limitArray printToLog: aBoolean maxPaths: anInt

<primitive: 275>
anArray _validateInstanceOf: Array.
limitArray _validateInstanceOf: Array.
aBoolean _validateClass: Boolean.
anInt _validateClass: SmallInteger .
(anInt < 0)
  ifTrue:[ ^ anInt _error: #rtErrArgOutOfRange  ] .
(anArray size ==  0)
  ifTrue:[ ^ anArray  _error: #rtErrArgOutOfRange  ] .
(limitArray size ==  0)
  ifTrue:[ ^ limitArray _error: #rtErrArgOutOfRange  ] .
^self _primitiveFailed: #_findAllReferencePathsToObjects:limitObjArray:printToLog:maxPaths:
%

category: 'Deprecated (Multi Ref Path)'
method:
findAllReferencePathsToObjects: anArray limitObjArray: limitArray printToLog: printToLog maxPaths: anInt

"Finds multiple reference paths to an array of objects.

 The number of reference paths returned for a given element of anArray is governed
 by anInt. For each element of anArray, at most anInt paths will be returned.  Setting
 anInt to 0 indicates all reference paths for all objects should be returned.
 
 NOTE: this method is not intended to work for classes or metaclasses.

 Objects in the stone's list of dead objects are excluded from the scan.  It is
 an error if any input object is in the stone's dead objects set, in which case
 this method will return nil.

 It is an error if any input object is not committed.  It is also an error if
 any object is in both the search array and the limit set.  nil will be returned
 in both of these cases.

 It is possible for an object to be disconnected from the repository and 
 therefore have no reference paths.   In this case, the second element of the
 result array will be false, indicating the reference path was not found.

 If printToLog is true, messages showing progress of the scan will be printed to
 the stdout of the gem calling this method.  

 The progressCount statistic indicates how many objects have been processed for
 each scan and is reset to zero at the end of each scan.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData.         

 This method operates in a transaction and will begin one if necessary.  This
 method may run for long periods of time and should not be called in production
 repositories.

 A reference path scan for an object will be terminated and considered completed
 if any class object is found in the reference path, even if the class is not
 present in the limit set.  Normally all classes are connected to the repository
 and will be included in the limit set.  However, for garbage collection 
 purposes, disconnected classes are considered live objects if any other object
 references them.  In other words, a class is a live object if any live
 instances of that class exist in the repository.

 Returns an array of arrays, or nil on error.  Each sub-array corresponds to a
 search object in the input array and has the following format:
 1 - the search object
 2 - aBoolean - true if the reference path was found, false if not.
 3 - an Array of reference paths, or an empty array if element 2 is false.
     Each element of this array represents one complete reference path from an
     object in the limit set to the search object.

 NOTE - This method uses a single thread and may be very time consuming on large
        repositories.
 
 WARNING - This method may consume excessive amounts of memory when anInt is set to zero 
           because a given object may be reachable via millions of reference paths.  For
           this reason, setting anInt to zero is NOT recommended."

self deprecated: 'Repository>>findAllReferencePathsToObjects:limitObjArray:printToLog:maxPaths: ',
      'unsupported as of v3.4, unreliable for large operations. Use multiple single ref path queries'.

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ self _findAllReferencePathsToObjects: anArray 
       limitObjArray: limitArray
       printToLog: printToLog
       maxPaths: anInt
%

category: 'Deprecated (Multi Ref Path)'
method:
findAllReferencePathsToObject: anObject maxPaths: anInt

"Finds reference paths to a single object.  See the comments in method
 Repository>>findAllReferencePathsToObjects:limitObjArray:printToLog:maxPaths: for more
 information."

|result limitArray|

self deprecated: 'Repository>>findAllReferencePathsToObject:maxPaths: ',
      'unsupported as of v3.4, unreliable for large operations. Use multiple single ref path queries'.

limitArray := self buildLimitSetForRefPathScan.
result := self findAllReferencePathsToObjects: (Array with: anObject)
                limitObjArray: limitArray
                printToLog: false maxPaths: anInt .
result == nil
  ifTrue:[^ nil] .
^ result first .
%

category: 'Deprecated (Multi Ref Path)'
method:
findAllReferencePathsToObjects: anArray printToLog: printToLog maxPaths: anInt

 "Finds an array of reference paths to objects.  See the comments in method
 Repository>>findAllReferencePathsToObjects:limitObjArray:printToLog:maxPaths:
 for more information."

| limitArray |

self deprecated: 'Repository>>findAllReferencePathsToObjects:printToLog:maxPaths: ',
      'unsupported as of v3.4, unreliable for large operations. Use multiple single ref path queries'.

limitArray := self buildLimitSetForRefPathScan .
^ self findAllReferencePathsToObjects: anArray limitObjArray: limitArray printToLog: printToLog maxPaths: anInt
%

category: 'Deprecated (Multi Ref Path)'
method:
findAllReferencePathsToObjects: anArray maxPaths: anInt

 "Finds an array of reference paths to objects.  See the comments in method
 Repository>>findAllReferencePathsToObjects:limitObjArray:printToLog:maxPaths:
 for more information."

| limitArray |

self deprecated: 'Repository>>findAllReferencePathsToObjects:maxPaths: ',
      'unsupported as of v3.4, unreliable for large operations. Use multiple single ref path queries'.

limitArray := self buildLimitSetForRefPathScan .
^ self findAllReferencePathsToObjects: anArray limitObjArray: limitArray printToLog: false maxPaths: anInt
%

category: 'Deprecated (Multi Ref Path)'
method:
findAllReferencePathsToObjects: anArray limitObjArray: limitArray printToLog: printToLog

self deprecated: 'Repository>>findAllReferencePathsToObjects:limitArray:printToLog: ',
      'unsupported as of v3.4, unreliable for large operations. Use multiple single ref path queries'.

^ self findAllReferencePathsToObjects: anArray limitObjArray: limitArray printToLog: printToLog maxPaths: 32
%

category: 'Deprecated (Multi Ref Path)'
method:
findAllReferencePathsToObject: anObject

"Finds an array of reference paths to objects.  The number of references paths 
 returned is limited to 32.  See the comments in method 
 Repository>>findAllReferencePathsToObjects:limitObjArray:printToLog:maxPaths:
 for more information. "

self deprecated: 'Repository>>findAllReferencePathsToObject: unsupported as of v3.4, unreliable for large operations. ',
   'Use multiple single ref path queries'.

^ self findAllReferencePathsToObject: anObject  maxPaths: 32
%

category: 'Deprecated (Multi Ref Path)'
method:
findAllReferencePathsToObjects: anArray printToLog: printToLog
"Finds an array of reference paths to objects.  The number of references paths 
 returned is limited to 32 per search object.  See the comments in method
 Repository>>findAllReferencePathsToObjects:limitObjArray:printToLog:maxPaths:
 for more information."

self deprecated: 'Repository>>findAllReferencePathsToObjects:printToLog: ',
      'unsupported as of v3.4, unreliable for large operations. Use multiple single ref path queries'.

^ self findAllReferencePathsToObjects: anArray printToLog: printToLog maxPaths: 32
%

category: 'Deprecated (Multi Ref Path)'
method:
findAllReferencePathsToObjects: anArray

"Finds an array of reference paths to objects.  The number of references paths 
 returned is limited to 32 per search object.  See the comments in method
 Repository>>findAllReferencePathsToObjects:limitObjArray:printToLog:maxPaths:
 for more information."

self deprecated: 'Repository>>findAllReferencePathsToObjects: ',
      'unsupported as of v3.4, unreliable for large operations. Use multiple single ref path queries'.

^ self findAllReferencePathsToObjects: anArray maxPaths: 32
%

category: 'Listing References'
method:
listReferencesToInstancesOfClasses: anArray toDirectory: aString

"First scan the repository for the instances of classes in anArray,
 then rescan to produce the sets of references to these instances.

 If anArray contains multiple occurrences of a class or an object that
 is not a kind of Behavior, then an error is generated.

 The value returned from this method is an array of triplets which 
 contains meta information about the results of the scan.  
 For each element of the argument anArray, the result array contains:
    <aClass>, <numberOfInstancesFound>  <numberOfReferencesFound>

 The special classes Boolean, Character, SmallInteger, SmallDouble,
 UndefinedObject and JISCharacter may be used as inputs in the array.
 The second value in the triplet for these classes is always zero.
 
 The instances which contain references to the classes are written to
 bitmap files in the directory specified by aString, which must specify
 a path to a writable directory.

 For each class in the input array, a bitmap file is created with the
 filename <className>-<seqNum>-references.bm and this bitmap contains
 the objects that have references to instances of this class.
 The seqNum is incremented for each file written by this method so that multiple runs
 can be made without deleting the files or getting errors for trying to overwrite a file.

 The binary bitmap files may be loaded into GsBitmap (see GsBitmap>>readFromFile:) to analyze the results.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.         

 The scan is done in an non-aggressive manner and uses only 2 threads.  To 
 complete the operation in less time, see the #fastListReferencesToInstancesOfClasses
 method."

^ self _listReferencesToInstancesOfClasses: anArray toDirectory: aString 
       withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90
%

category: 'Listing References'
method:
fastListReferencesToInstancesOfClasses: anArray toDirectory: aString

"Similar to listReferencesToInstancesOfClasses:toDirectory: except that 
 the scan is performed aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is in progress."

^ self _listReferencesToInstancesOfClasses: anArray  toDirectory: aString 
       withMaxThreads: self _aggressiveMaxThreadCount maxCpuUsage: 95
%
      
category: 'Private'
method:
_listReferencesToInstancesOfClasses: anArray toDirectory: aString
   withMaxThreads: maxThreads maxCpuUsage: aPercent

"Similar to fastListReferencesToInstancesOfClasses:toDirectory: except that
 the user can also specify the maxThreads and cpuUseage."

| primResult sortedArray result sortedCollection |
System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

"Sort by oop order."
sortedCollection := SortedCollection sortBlock:[:a :b| a asOop < b asOop].
sortedCollection addAll: anArray.
sortedArray := Array withAll: sortedCollection .

primResult :=  self _listRefsToInstOfClasses: sortedArray  toDirectory: aString
                       withMaxThreads: maxThreads maxCpuUsage: aPercent.

"nil means the primitive detected a duplicate entry.  Raise an error and bail."
primResult ifNil:[  ArgumentError new object: anArray ;
    signal:'Array of references to search for contains invalid class'
].

"build the result array to match the order of the argument array."
result := Array new.
1 to: anArray size do: [:j| | cls i |
  cls := anArray at: j .
  i := primResult indexOfIdentical: cls .
  i == 0 ifTrue:[
    ArgumentError signal:'class ' , cls name , 
	 ' not found in results from primitive _listRefsToInstOfClasses' .
    result add: cls ; add: 0; add: 0 .   
  ] ifFalse:[
    result add: (primResult at: i); add: (primResult at: i + 1);
         add: (primResult at: i + 2).
  ]
].
^ result
%

category: 'Private'
method:
_listRefsToInstOfClasses: anArray  toDirectory: aString
   withMaxThreads: maxThreads maxCpuUsage: cpuPercent

"Primitive implementation of listReferencesToInstancesOfClasses" 

<primitive: 504>
 anArray _validateInstanceOf: Array.
 anArray size == 0 ifTrue:[ anArray _error: #errArgTooSmall args:{ 1 } ].
 aString _validateKindOfClass: String.
 maxThreads _validateClass: SmallInteger; _validateMin: 1 max: SmallInteger maximum32bitInteger .
 self _validatePercentage: cpuPercent .
 ^ self _primitiveFailed: 
    #_listReferencesToInstancesOfClasses:toDirectory:withMaxThreads:maxCpuUsage:
    args: {  anArray . aString . maxThreads . cpuPercent }
%

category: 'Repository Conversion'
method:
_migrateGroups: aBlock

"Migrates group collections in all the ObjectSecurityPolicies of the receiver."

self do:[ :anObjectSecurityPolicy | anObjectSecurityPolicy ~~ nil 
  ifTrue:[ anObjectSecurityPolicy _migrateGroups: aBlock ] ]
%

category: 'Listing Instances'
method:
allInstances: inputArg

" This method scans the repository for instances of classes.

  If the argument is a single class this method returns a GsBitmap containing 
  the instances of the class.  If the input argument is an array or GsBitmap
  of classes then this method returns an array of subArrays, where each 
  subArray contains:
     <aClass> , <aGsBitmap with the instances of the class>
  Note that the order in which classes appear in the result array may differ
  from the order which they appear in the inputArg if it is an array.

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData. 

  The entire repository is scanned once and the result contains only committed
  objects that were present in the view of the data at that time and for 
  which the user executing this method has read authorization to the object.

  The scan is done in an non-aggressive manner and uses only 2 threads.  To 
  complete the operation in less time, see the version of this method with 
  the fast prefix.

  If the argument is a class or an array of classes, then an error is raised
  if the objects being passed in are specials, not committed or not a class.
  If the argument is a GsBitmap, it cannot contain special objects so that 
  check is not necessary.  In order to not fault in all of the instances
  represented in the bitmap, checks are not done on the contents, but you 
  may get back empty results for entries in the GsBitmap that are not committed
  or not a class.
"
^ self _doScan: 0 "OP_ALL_INSTANCES-OP_ALL_INSTANCES"
        fast: false with: inputArg
%

category: 'Listing Instances'
method:
fastAllInstances: inputArg

" Same as allInstances: except that the scan is performed more agressively,
  using more of the system resouces for both cpu and I/O.
"

^ self _doScan: 0 "OP_ALL_INSTANCES-OP_ALL_INSTANCES"
    fast: true with: inputArg
%

category: 'Listing References'
method:
allReferences: inputArg

" This method scans the repository for references to specific objects.

  If the argument is a single object this method returns a GsBitmap containing
  the objects that reference it.  If the input argument is an array or GsBitmap 
  of objects then this method returns an array of subarrays, where each 
  subArray contains:
     <anInstance> , <aGsBitmap with the objects that reference it>
  Note that the order in which classes appear in the result array may differ
  from the order which they appear in the inputArg if it is an array.

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData. 

  The entire repository is scanned once and the result contains only committed
  objects that were present in the view of the data at that time and for 
  which the user executing this method has read authorization to the object.

  The scan is done in an non-aggressive manner and uses only 2 threads.  To 
  complete the operation in less time, see the version of this method with 
  the fast prefix.
"

^ self _doScan: 1"OP_ALL_REFERENCES-OP_ALL_INSTANCES" 
      fast: false with: inputArg
%

category: 'Listing References'
method:
fastAllReferences: inputArg

" Same as allreferences: except that the scan is performed more agressively,
  using more of the system resouces for both cpu and I/O.
"

^ self _doScan: 1 "OP_ALL_REFERENCES-OP_ALL_INSTANCES" 
   fast: true with: inputArg
%

category: 'Private'
method:
_allReferencesByParentClass: inputArg

" This method scans the repository for references to objects in the inputArg.

  The result is an array of subArrays, where each subArray contains:
     <aClass> , <aGsBitmap> containing the objects that reference any of the objects in the inputArg
                            whose parent object is an instance of <aClass>

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData. 

  The entire repository is scanned once and the result contains only committed
  objects that were present in the view of the data at that time.

  The scan is done in an non-aggressive manner and uses only 2 threads.  To 
  complete the operation in less time, see the version of this method with 
  the fast prefix.
"

^ self _doScan: 3 "OP_ALL_REFS_BY_PARENT_CLASS-OP_ALL_INSTANCES"
    fast: false with: inputArg
%

category: 'Private'
method:
_fastAllReferencesByParentClass: inputArg

" Same as allReferencesByParentClass: except that the scan is performed more agressively,
  using more of the system resouces for both cpu and I/O.
"

^ self _doScan: 3"OP_ALL_REFS_BY_PARENT_CLASS-OP_ALL_INSTANCES" 
  fast: true with: inputArg
%

category: 'Listing References'
method:
allReferencesToInstancesOfClasses: inputArg

" This method scans the repository for objects that reference instances of a class.

  If the argument is a single class this method returns a GsBitmap containing 
  objects that reference instances of that class.  If the input argument is 
  an array or GsBitmap of classes then this method returns an array of subArrays, 
  where each subArray contains:
     <aClass>, <aGsBitmap containing the objects that reference instances of the class>
  Note that the order in which classes appear in the result array may differ
  from the order which they appear in the inputArg if it is an array.

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData. 

  This operation requires two scans of the repository, the first scan finds
  instances of the class and the second scan finds the references to these instances.
  The result contains only committed objects that were present in the view of 
  the data at that time and for which the user executing this method has read 
  authorization to the object.

  The scan is done in an non-aggressive manner and uses only 2 threads.  To 
  complete the operation in less time, see the version of this method with 
  the fast prefix.
"

^ self _doScan: 2"OP_ALL_REFS_TO_INSTANCES-OP_ALL_INSTANCES"
     fast: false with: inputArg
%

category: 'Listing References'
method:
fastAllReferencesToInstancesOfClasses: inputArg

" Same as allReferencesToInstancesOfClasses: except that the scan is performed more agressively,
  using more of the system resouces for both cpu and I/O.
"

^ self _doScan: 2 "OP_ALL_REFS_TO_INSTANCES-OP_ALL_INSTANCES"
fast: true with: inputArg
%

category: 'Listing By Security Policy'
method:
allObjectsInObjectSecurityPolicies: inputArg

" This method scans the repository for objects that have the specified 
  ObjectSecurityPolicyId.

  If the argument is a single ObjectSecurityPolicyId this method returns a
  GsBitmap with the objects that have the specified ObjectSecurityPolicyId.
  If the input argument is an array or GsBitmap of ObjectSecurityPolicyIds 
  then this method returns an array of subArrays, where each subArray contains:
    <anObjectSecurityPolicyId> <aGsBitmap with the objects whose SecurityPolicyId
                                matches anObjectSecurityPolicyId>
  Note that the order in which classes appear in the result array may differ
  from the order which they appear in the inputArg if it is an array.

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData. 

  The entire repository is scanned once and the result contains only committed
  objects that were present in the view of the data at that time and for 
  which the user executing this method has read authorization to the object.

  The scan is done in an non-aggressive manner and uses only 2 threads.  To 
  complete the operation in less time, see the version of this method with 
  the fast prefix.
"
self _checkObjectSecurityPolicyArg: inputArg.
^ self _doScan: 4"OP_ALL_OBJS_IN_SEC_POLICY-OP_ALL_INSTANCES" 
   fast: false with: inputArg
%

category: 'Listing By Security Policy'
method:
fastAllObjectsInObjectSecurityPolicies: inputArg


" Same as allObjectsInObjectSecurityPolicies: except that the scan is performed more agressively,
  using more of the system resouces for both cpu and I/O.
"

self _checkObjectSecurityPolicyArg: inputArg.
^ self _doScan: 4 "OP_ALL_OBJS_IN_SEC_POLICY-OP_ALL_INSTANCES"
  fast: true with: inputArg
%

category: 'Listing By Size'
method:
allObjectsLargerThan: aSize

" Returns a GsBitmap with the objects that are larger than aSize.
  The argument aSize must be a SmallInteger.

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData. 

  The entire repository is scanned once and the result contains only committed
  objects that were present in the view of the data at that time and for 
  which the user executing this method has read authorization to the object.

  The scan is done in an non-aggressive manner and uses only 2 threads.  To 
  complete the operation in less time, see the version of this method with 
  the fast prefix.
"

^ self _doScan: 5 "OP_ALL_OBJS_LARGER_THAN-OP_ALL_INSTANCES"
    fast: false with: aSize
%

category: 'Listing By Size'
method:
fastAllObjectsLargerThan: aSize

" Same as allObjectsLargerThan: except that the scan is performed more agressively,
  using more of the system resouces for both cpu and I/O.
"

^ self _doScan: 5 "OP_ALL_OBJS_LARGER_THAN-OP_ALL_INSTANCES"
    fast: true with: aSize
%

category: 'Private'
method:
_checkObjectSecurityPolicyArg: inputArg

(inputArg class == Array) 
  ifTrue: [
      1 to: inputArg size do:[ :k |
      self _validateObjectSecurityPolicyId: (inputArg at: k)] ]
  ifFalse: [self _validateObjectSecurityPolicyId: inputArg]
%

category: 'Private'
method:
_scanWithMaxThreads: maxThreads waitForLock: lockWaitTime pageBufSize: aBufSize
               percentCpuActiveLimit: percentCpu scanKind: kind with: inputArg

" This primitive method performs various scans of the Repository.
  The scans only take into account committed objects; i.e., this method 
  does not operate on objects in the Temporary Object Cache (TOC).

  This primitive uses a multithreaded algorithm to sweep the active
  data pages in the repository to gather the information requested.

  The maxThreads argument specifies the maximum number of threads
  that will be used during the operation.  The actual number of 
  active threads can be adjusted to a lower value automatically at runtime 
  by the main thread based on the percentCpuActiveLimit and by the user 
  by interactively setting the mtThreadLimit (see details below).

  The lockWaitTime argument is used to specify how many seconds method should
  wait while attempting to acquire the garbage collection (GC) lock.
  No other garbage collection operations may be started or in progress while
  this method is running.  Objects in the possibleDead or deadNotReclaimed 
  sets at the start of the scan are ignored by the scan.
  A lockWaitTime of -1 or 0 means wait forever.
 
  The pageBufSize, which must be a power of two, specifies the number 
  of pages to buffer per thread. This parameter in conjunction with the 
  maxThreads largely determines the memory footprint needed to perform 
  this operation.  The pageBufSize doesn't have much impact on the 
  performance of the scan, so a default size of 8 is probably sufficient.

  The percentCpu specifies a level of total cpu activity at which the 
  algorithm automatically inactivates threads to prevent overload 
  of system resources.

  This algorithm makes use of additional sessions (threads) to achieve 
  significant performance improvements.  It also makes space/time trade offs
  so that both heap memory and TemporaryObjectCache (TOC)
  resources are used.  In fact, this algorithm doesn't require much TOC, 
  except for the operations that return array results, so configuring this 
  process for a smaller TOC space is advantageous. 

  The memory space that is needed is variable and depends upon number of
  objects being searched for, the number found, the number of sessions 
  requested and the pageBufSize specified for each.

  The behavior of this method is dependent on the scanKind as follows:

   scanKind  Function     
   =======================
     0     allInstances           (OP_ALL_INSTANCES - OP_ALL_INSTANCES)
     1     allReferences          (OP_ALL_REFERENCES - OP_ALL_INSTANCES)
     2     allReferencesToInstancesOfClasses (OP_ALL_REFS_TO_INSTANCES-OP_ALL_INSTANCES)
     3     allReferencesByParentClass        (OP_ALL_REFS_BY_PARENT_CLASS-OP_ALL_INSTANCES)
     4     allObjectsInObjectSecurityPolicies (OP_ALL_OBJS_IN_SEC_POLICY-OP_ALL_INSTANCES)
     5     allObjectsLargerThan              (OP_ALL_OBJS_LARGER_THAN-OP_ALL_INSTANCES)
"

<primitive: 1030>
 | maxInt |
  maxInt := SmallInteger maximum32bitInteger.
  maxThreads
    _validateClass: SmallInteger;
    _validateMin: 1 max: maxInt.
  lockWaitTime
    _validateClass: SmallInteger;
    _validateMin: -1 max: maxInt.
  aBufSize
    _validateClass: SmallInteger;
    _validateIsPowerOf2.
  self _validatePercentage: percentCpu.
  kind
    _validateClass: SmallInteger;
    _validateMin: 0 max: 4.
^ self _primitiveFailed: 
  #_scanWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:scanKind:with:
    args: { maxThreads . lockWaitTime . aBufSize . percentCpu . kind . inputArg }
%

category: 'Secure Backup and Restore'
method:
secureFullBackupTo: fileNames MBytes: mByteLimit compressKind: compKind 
bufSize: count encryptKind: encKind publicKeyCerts: anArrayOrString 
signatureHashKind: hashKind signingKey: signingKeyFn 
signingKeyPassphrase: aPassphrase 

"Writes a secure full backup file containing the most recently committed 
 version of  the receiver as of the time the method is executed.

 Secure backups support compression, encryption and digital signatures.
 Compression and encryption are optional.  Digital signatures are mandatory.

 fileNames must either be a string which is the backup file name or an array of
 strings containing a list of backup file names.
 
 Secure backups have a file extension of .sdbf, which will be appended to all
 backup file names if necessary.

 The following compression kinds are supported.  LZ4 is recommended since it is
 faster than zlib.
 
 compKind argument meanings:
   0 - no compression
   1 - zlib (aka gzip) compression.
   2 - lz4 compression.

 encKind argument meanings:
   An encKind of zero indicates the backup is not encrypted.
   An encKind greater than zero indicates the backup is encrypted using AES-CTR
   mode using the key size shown below:
   
   0 - no encryption
   1 - AES-CTR-128
   2 - AES-CTR-192
   3 - AES-CTR-256

 hashKind argument meanings:
   hashKind specifies what message digest (hash) algorithm will be used to
   hash the signature before it is encrypted and stored in the backup
   file.  The following message digests are supported:

   1 - SHA256
   2 - SHA384
   3 - SHA512

 Unless otherwise noted, all certificates and keys must be in one
 of the directories specified in the key ring, which is the list
 of directories specified in the GEM_KEYRING_DIRS configuration option.
 Certificate and key file names must be specified without the path.
 GemStone will automatically search the key ring for the matching file.
 
 If encKind is greater than zero, the publicKeyCerts argument anArrayOrString must
 either be a String or an Array of Strings where each element is the filename
 of an X509 certificate in PEM format that contains an RSA public key
 without a passphrase. The RSA public key(s) may be any valid length
 (1024, 2048, 3072, bits etc). A key length of at least 2048 bits
 is recommended. anArrayOrString must be nil if encKind is zero. A maximum
 of eight (8) public key certificates may be specified. The corresponding
 private key(s) for these certificate(s) need not be present in the key ring
 at the time the secure backup is created.  However at least one private key
 will be required in order to restore the backup.

 When encryption is enabled, each backup file is encrypted with a randomly
 generated session key, which is then encrypted with the provided public
 key(s) and stored in the backup file.  One of the corresponding private keys
 must be available in order to restore the backup, but are not required when
 the backup is created.

 signingKeyFn is the name of a private key file in PEM format used to sign the
 backup.  The file must be in the key ring. aPassphrase is a string which is
 the passphrase for the private key. If the private key does not use a
 passphrase, then aPassphrase must be nil. A certificate containing the public
 key which matches the private key specified in privateKeyFn is stored in the
 backup and must also be present in the key ring.  GemStone automatically
 searches the keyring for the matching certificate.

 For encrypting backups, only RSA keys are supported.
 For signing backups, both RSA and DSA private keys are supported.
 
 The bufSize argument controls the size of the buffer used to write records to
 the file. The count argument specifies the number of 128KB backup records are
 contained in the buffer. The values allowed are 1 (128KB) through 8192 (1GB).

 This method performs the backup using multiple threads.
 The number of threads is automatically set to 2 times the 
 number of extents in the repository.  This can be overridden by
 executing:
     SessionTemps current at: #GsOverrideNumThreads put: numThreads.
 where numThreads can be any value between 1 and 4 * numCpus before
 executing the backup method.
 The performance can be modified during the run by 
 updating the Multithreaded Scan Tuning methods.

Returns true if the backup was completed."

| limitArray fnArray |
(fileNames isKindOfClass: Array) 
  ifTrue: [ fnArray := fileNames ]
  ifFalse: [ fnArray := { fileNames } ] .
limitArray := self _validateFileNames: fnArray limits: mByteLimit.
^ self _primSecureFullBackupTo: fnArray MBytes: limitArray compressKind: compKind 
       bufSize: count encryptKind: encKind publicKeyCerts: anArrayOrString 
       signatureHashKind: hashKind signingKey: signingKeyFn 
       signingKeyPassphrase: aPassphrase numThreads: self _getNumThreads
%

category: 'Secure Backup and Restore'
method:
_primSecureFullBackupTo: fileNames MBytes: limitArray compressKind: compKind 
bufSize: count encryptKind: encKind publicKeyCerts: anArrayOrString 
signatureHashKind: hashKind signingKey: signingKeyFn 
signingKeyPassphrase: aPassphrase numThreads: threadCount

"Private.  Provides the implementation of secure full backups."

<primitive: 1039>
| a b |
fileNames _validateInstanceOf: Array.
limitArray _validateInstanceOf: Array.
compKind _validateClass: SmallInteger .
hashKind _validateClass: SmallInteger .
encKind _validateClass: SmallInteger .
threadCount _validateClass: SmallInteger .
((compKind < 0) or:[ compKind > 2])
  ifTrue:[ compKind _error: #rtErrArgOutOfRange args:{ 0 . 2 } ] .
((encKind < 0) or:[ encKind > 3])
  ifTrue:[ encKind _error: #rtErrArgOutOfRange args:{ 0 . 3 } ] .
encKind == 0  ifTrue:[ anArrayOrString _validateClass: UndefinedObject 
  ] ifFalse:[ 
    anArrayOrString _validateKindOfClasses: { String . Array } .
    anArrayOrString _isArray ifTrue:[ | sz |
      sz := anArrayOrString size .
      (sz > 0 and:[ sz < 8 ]) ifFalse:[
        OutOfRange new name: 'number of public keys' min: 1 max: 8 actual: sz ;
          signal.
      ]
    ]
  ] .
((hashKind < 1) or:[hashKind > 3])
  ifTrue:[ hashKind _error: #rtErrArgOutOfRange args:{ 1 . 3 } ] .
signingKeyFn _validateKindOfClass: String .
aPassphrase _validateKindOfClasses: { String . UndefinedObject } .
count _validateClass: SmallInteger .
(a := fileNames size) == (b := limitArray size) ifFalse:[
  ArgumentError signal: 'number of fileNames (' , a asString, 
  ') not equal to number of limits (', b asString , ')' 
].
^ self _primitiveFailed: 
  #_primSecureFullBackupTo:compressKind:bufSize:encryptKind:publicKeyCerts:signatureHashKind:signingKey:signingKeyPassphrase: .
%

category: 'Secure Backup and Restore'
method:
restoreFromSecureBackups: arrayOfFileNames scavengePagesWithPercentFree: aPercent
privateDecryptionKey: aKey passphrase: aPassphrase

"Disables logins and starts a full restore of the repository based 
 on the contents of the specified secure backup file(s).  Behaves the
 same as the restoreFromBackup: method except for extra security
 and integrity checks as described below.

 Secure backup files must have a file extension of .sdbf. All
 file names in arrayOfFileNames will have the .sdbf suffix appended
 if it is not present.
 
 All secure backups have been signed with a private signing key.
 Before the backup is restored, the integrity of each backup file
 is checked by reading the file and computing the signature.  If the
 computed signature does not match the signature stored in the file,
 then an error is raised indicating the backup file has been either
 been corrupted or tampered with. If the signature is correct, the
 restore proceeds.

 All certificates and keys must be in one of the directories specified in
 the key ring, which is the list of directories specified in the
 GEM_KEYRING_DIRS configuration option.  Certificate and key file names
 must be specified without the path.  GemStone will automatically search
 the key ring for the matching file.

 A certificate containing the public key which matches the private
 key used to sign the backup must be present in the key ring. GemStone
 automatically searches all files in the key ring for a certificate which
 contains a matching public key. The restore will fail and an error will be
 raised if the key ring does not contain the appropriate certificate.

 If the backup was encrypted, aKey is a String representing the file name
 containing a private key in PEM format.  The private key must match one
 of the public keys specified when creating the backup. The file must be
 present in the key ring as described above. If the backup is not encrypted
 then aKey must be nil.

 aPassphrase is the passphrase to access the private key. If the backup is
 not encrypted or aKey is not protected by a passphrase (not recommended)
 then aPassphrase must be nil.

 Upon successful completion, the session is automatically logged out and
 the RestoreBackupSuccess error (4046) is generated.

 A GciHardBreak during this method will terminate the session.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser.

 This method performs the restore using multiple threads.
 The number of threads is automatically set to 2 times the 
 number of extents in the repository.  This can be overridden by
 executing:
     SessionTemps current at: #GsOverrideNumThreads put: numThreads.
 where numThreads can be any value between 1 and 4 * numCpus before
 executing the restore method.
 The performance can be modified during the run by 
 updating the Multithreaded Scan Tuning methods.

 WARNING: a small percentage, less than 5 percent can cause
 the reclaim gems to be very busy after the restore."

^ self _primRestoreSecureBackups: arrayOfFileNames scavPercentFree: aPercent bufSize: 8
privateDecryptionKey: aKey passphrase: aPassphrase numThreads: self _getNumThreads
%

category: 'Secure Backup and Restore'
method:
restoreFromSecureBackup: aFileName scavengePagesWithPercentFree: aPercent
privateDecryptionKey: aKey passphrase: aPassphrase

"Same as restoreFromSecureBackups except this method accepts a single file name."

^ self restoreFromSecureBackups: { aFileName } scavengePagesWithPercentFree: aPercent
privateDecryptionKey: aKey passphrase: aPassphrase
%


category: 'Secure Backup and Restore'
method:
restoreFromSecureBackup: aFileName privateDecryptionKey: aKey passphrase: aPassphrase

^ self restoreFromSecureBackups: { aFileName } scavengePagesWithPercentFree: 100
privateDecryptionKey: aKey passphrase: aPassphrase
%

category: 'Secure Backup and Restore'
method:
restoreFromSecureBackups: anArray privateDecryptionKey: aKey passphrase: aPassphrase

^ self restoreFromSecureBackups: anArray scavengePagesWithPercentFree: 100
privateDecryptionKey: aKey passphrase: aPassphrase
%


category: 'Backup and Restore'
method:
_primRestoreSecureBackups: arrayOfFileNames scavPercentFree: aPercent bufSize: count
privateDecryptionKey: aKey passphrase: aPassphrase numThreads: threadCount

"Private.  Provides implementation of restoreSecureBackups."

<primitive: 1040>
arrayOfFileNames _validateInstanceOf: Array.
count _validateClass: SmallInteger .
threadCount _validateClass: SmallInteger .
self  _validatePercentage: aPercent .
aKey ifNotNil:[ aKey _validateClass: String ] .
aPassphrase ifNotNil:[ aPassphrase _validateClass: String ] .
self _primitiveFailed: #_primRestoreBackups:scavPercentFree:bufSize:privateDecryptionKey:passphrase: .
self _uncontinuableError
%

!------------------- GemStone 64 v3.5 and above
expectvalue %String
run
 "install the name for each ObjectSecurityPolicy as a dynamic instVar"
SystemObjectSecurityPolicy _name ifNil:[
  | nameBlk upBlk |
  #( #DataCuratorObjectSecurityPolicy
     #GsIndexingObjectSecurityPolicy
     #PublishedObjectSecurityPolicy
     #SecurityDataObjectSecurityPolicy
     #SystemObjectSecurityPolicy 
     ) do:[ :aSym |
       (Globals at: aSym otherwise: nil) ifNotNil:[:policy |
          (policy _name == nil and:[ policy isInvariant not]) ifTrue:[ 
              policy name: aSym asString . 
              GsFile gciLogServer: 'named ' , aSym 
           ].
       ] ifNil:[ GsFile gciLogServer: ' not found: ' , aSym ].
  ]. 
  nameBlk := [:id :sym |
   SystemRepository size >= id ifTrue:[
     (SystemRepository at: id) ifNotNil:[:pol |
       (pol _name == nil and:[ pol isInvariant not]) ifTrue:[
          pol name: sym .  GsFile gciLogServer:'named ' , id asString, ' as ', sym].
      ].
    ].
  ].
  upBlk := [:id :uid :sym | |up |
    up := AllUsers userWithId: uid ifAbsent:[nil] .
    up ifNotNil:[ (SystemRepository at: id) == up defaultObjectSecurityPolicy 
                       ifTrue:[ nameBlk value: id value: sym ]].
  ].
  nameBlk value: 3 value: #GsTimeZoneObjectSecurityPolicy . "policy no longer used"

  upBlk value: 7 value: 'GcUser' value: #GcUserObjectSecurityPolicy .   
  upBlk value: 8 value: 'Nameless' value: #NamelessObjectSecurityPolicy .   
  ^' added names'
].
^ 'no change'
%

! New convenience methods which load the passphrase from a file
category: 'Secure Backup and Restore'
method:
secureFullBackupTo: fileNames MBytes: mByteLimit compressKind: compKind 
bufSize: count encryptKind: encKind publicKeyCerts: anArrayOrString 
signatureHashKind: hashKind signingKey: signingKeyFn 
signingKeyPassphraseFile: aFileName

"Same as the method
  secureFullBackupTo:MBytes:compressKind:bufSize:encryptKind:publicKeyCerts:signatureHashKind:signingKey:signingKeyPassphrase:
except this method specifies a file name on the server which contains the
passphrase for the signing key.

Raises an IOError if the could not be opened or read.

If aFileName is nil, then the signing key is assumed to have no passphrase.

Returns true if the secure backup was completed."

|pf|
aFileName ifNotNil:[ pf := (GsFile getContentsOfServerFile: aFileName) trimWhiteSpace ] .
^ self secureFullBackupTo: fileNames 
       MBytes: mByteLimit 
       compressKind: compKind 
       bufSize: count 
       encryptKind: encKind 
       publicKeyCerts: anArrayOrString 
       signatureHashKind: hashKind
       signingKey: signingKeyFn 
       signingKeyPassphrase: pf
%
category: 'Secure Backup and Restore'
method:
restoreFromSecureBackups: arrayOfFileNames scavengePagesWithPercentFree: aPercent
privateDecryptionKey: aKey passphraseFile: aFileName

"Same as the method
  restoreFromSecureBackups:scavengePagesWithPercentFree:privateDecryptionKey:passphrase:
except this method specifies a file name on the server which contains the
passphrase for the decryption key.

Raises an IOError if the could not be opened or read.

If aFileName is nil, then the decryption key is assumed to have no passphrase.

 Upon successful completion, the session is automatically logged out and
 the RestoreBackupSuccess error (4046) is generated."


|pf|
aFileName ifNotNil:[ pf := (GsFile getContentsOfServerFile: aFileName) trimWhiteSpace ] .
^ self restoreFromSecureBackups: arrayOfFileNames 
       scavengePagesWithPercentFree: aPercent
       privateDecryptionKey: aKey 
       passphrase: pf
%
category: 'Secure Backup and Restore'
method:
restoreFromSecureBackup: aFileName scavengePagesWithPercentFree: aPercent
privateDecryptionKey: aKey passphraseFile: pfFileName

"Same as restoreFromSecureBackups except this method accepts a single file name
 and a file name on the server which contains the
passphrase for the decryption key.

Raises an IOError if the could not be opened or read.

If aFileName is nil, then the decryption key is assumed to have no passphrase."

|pf|
pfFileName ifNotNil:[ pf := (GsFile getContentsOfServerFile: pfFileName) trimWhiteSpace ] .
^ self restoreFromSecureBackups: { aFileName } 
       scavengePagesWithPercentFree: aPercent
       privateDecryptionKey: aKey 
       passphrase: pf
%
category: 'Secure Backup and Restore'
method:
restoreFromSecureBackup: aFileName privateDecryptionKey: aKey passphraseFile: pfFileName

|pf|
pfFileName ifNotNil:[ pf := (GsFile getContentsOfServerFile: pfFileName) trimWhiteSpace ] .
^ self restoreFromSecureBackups: { aFileName } 
       scavengePagesWithPercentFree: 100
       privateDecryptionKey: aKey 
       passphrase: pf
%

expectvalue /String
run
"Patch SystemRepository for mismatched policyId seen in some 3.3.x repositories, fix 48089"
| newPol oldPol newPolId |
SystemRepository size >= 15 ifTrue:[
  (oldPol := SystemRepository at: 15) objectSecurityPolicyId == 20 ifTrue:[
	  newPol := GsObjectSecurityPolicy new .
    newPolId := newPol objectSecurityPolicyId .
	  1 to: GsObjectSecurityPolicy instSize do:[:n|
		    newPol _unsafeAt: n put:(oldPol instVarAt: n) "copy state of policy 15"
	  ].
	  newPol _unsafeAt: 7 put: 15 . "set itsId instVar"
	  SystemRepository _at: 15 put: newPol .
	  SystemRepository _at: newPolId put: nil .
	  ^ 'Recreated policy 15'
  ]. 
].
^ 'No changes'
%


category: 'Updating'
method
_at: offset put: aValue
"Disallowed"
self shouldNotImplement: #_at:put:
%

