!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: rcidentityset.gs 41207 2017-03-03 18:54:21Z bretlb $
!
! Superclass Hierarchy:
!   RcIdentitySet, IdentitySet, UnorderedCollection, Collection, Object.
!
!=========================================================================

expectvalue %String
run
| cls |
cls := Globals at: #RcIdentitySet otherwise: nil.
^ cls == nil ifTrue:[
   IdentitySet subclass: 'RcIdentitySet'
   instVarNames: #()
   classVars: #()
   classInstVars: #()
   poolDictionaries: #()
   inDictionary: Globals
   options: #().
          'Created class RcIdentitySet'
   ]
   ifFalse:[ 'RcIdentitySet class already exists' ].
%
expectvalue true

removeallmethods RcIdentitySet
removeallclassmethods RcIdentitySet
set class RcIdentitySet

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment: 
'An RcIdentitySet is a special kind of IdentitySet that provides for handling
 modifications to an individual instance by multiple concurrent sessions.  
 When concurrent sessions update an RcIdentitySet and an rc conflict is 
 detected during commit the conflict is resolved by replaying the operation.

 There is no commit conflict with multiple sessions adding and a single session 
 removing objects from the RcIdentitySet.

 In an application with high rates of concurrency, the conflict resolution must
 be serialized, which may create delays.  In this case an RcQueue may be more 
 efficient.
'
%


category: 'Instance Creation'
classmethod:
new

"Returns a new RcIdentitySet."

^super new
%

! ------------------- Instance methods for RcIdentitySet

category: 'Converting'
method:
asIdentitySet

^ IdentitySet withAll: self
%

category: 'Class Membership'
method:
species

"Returns the class to use to select and reject queries."

^ IdentitySet
%

category: 'Adding'
method:
add: anObject

"Adds anObject to the RcIdentitySet and returns anObject.
 Nils are ignored, i.e., not added to the set."

anObject == nil ifTrue:[ ^ anObject "ignore nils" ].
(self includes: anObject) ifFalse:[
  self _rcIncludes: anObject .
  self addRedoLogEntryFor: #_redoAdd:  withArgs: { anObject } .
  super add: anObject.
].
^ anObject
%

category: 'Adding'
method:
addAll: aCollection

"Adds all of the elements of aCollection to the receiver and returns
 aCollection."

self addRedoLogEntryFor: #_redoAddAll:  withArgs: { aCollection copy } .
System _addToRcReadSet: self includingAllNodes: true.

^ super addAll: aCollection
%

category: 'Removing'
method:
remove: anObject

"Removes anObject if present from the receiver.  
 Generates an error if anObject is not in the receiver.  Returns anObject. 
"

anObject ifNotNil:[
  (self includes: anObject) ifTrue:[
    self _rcIncludes: anObject .
    self addRedoLogEntryFor: #_redoRemove:  withArgs: { anObject } .
    ^ super remove: anObject .
  ] ifFalse:[
    ^ self _errorNotFound: anObject
  ]
].
^ anObject
%

category: 'Removing'
method:
remove: anObject ifAbsent: aBlock

"Removes from the receiver an object that is identical to anObject and
 returns anObject.  If anObject is not present in the receiver, 
 evaluates aBlock and returns the result of the evaluation."

(self includes: anObject) ifTrue:[
  self _rcIncludes: anObject .
  self addRedoLogEntryFor: #_redoRemove: withArgs: { anObject } .
  ^ super remove: anObject 
]
ifFalse: [ ^ aBlock value ]
%

category: 'Removing'
method:
remove: anObject otherwise: notFoundValue

"Removes from the receiver an object that is identical to anObject and returns
 anObject.  If anObject is not present in the receiver, returns notFoundValue."

(self includes: anObject) ifTrue:[
  self _rcIncludes: anObject .
  self addRedoLogEntryFor: #_redoRemove: withArgs: { anObject } .
  ^ super remove: anObject 
]
ifFalse: [ ^ notFoundValue ]
%

category: 'Removing'
method:
removeAll: aCollection
  "Removes all of the elements of aCollection from the receiver and returns aCollection.
 Generates an error if any object in the collection is not in the receiver.
"

  self addRedoLogEntryFor: #_redoRemoveAll: withArgs: {aCollection copy}.
  System _addToRcReadSet: self includingAllNodes: true.
  ^ super removeAll: aCollection
%

category: 'Removing'
method:
removeAllPresent: aCollection

"Removes from the receiver each element of aCollection that is also an
 element of the receiver.  Differs from removeAll: in that, if some
 elements of aCollection are not present in the receiver, no error is generated.
 Returns aCollection."

self addRedoLogEntryFor: #_redoRemoveAllPresent: withArgs: { aCollection } .
System _addToRcReadSet: self includingAllNodes: true.

^super removeAllPresent: aCollection
%

category: 'Removing'
method:
removeIdentical: anObject

"Same as remove:."

^self remove: anObject
%

category: 'Removing'
method:
removeIdentical: anObject ifAbsent: aBlock

"Same as remove:ifAbsent:."

^self remove: anObject ifAbsent: aBlock
%

category: 'Removing'
method:
removeIdentical: anObject otherwise: notFoundValue

"Same as remove:otherwise:."

^ self remove: anObject otherwise: notFoundValue
%

category: 'Removing'
method:
removeIfPresent: anObject

"Removes from the receiver an object that is identical to anObject and
 returns anObject.  Returns nil if anObject is not present in the receiver."

(self includes: anObject) ifTrue:[
  self _rcIncludes: anObject .
  self addRedoLogEntryFor: #_redoRemove: withArgs: { anObject } .
  ^ super remove: anObject 
]
ifFalse: [ ^ nil ]
%


! ------------------- Methods to support Rc replay

category: 'Reduced Conflict Support'
method:
_abortAndReplay: conflictObjects
 
"Abort the receiver and replay operations on the receiver from the redo log."
 
| redoLog logEntries |
_indexedPaths 
  ifNotNil: [ 
    _indexedPaths _anyIndexIsLegacy 
      ifTrue: [ 
        "abort and replay not supported when legacy index is involved"
        ^false ] ].

redoLog := System _redoLog.
 
" if no log entries to replay, then we're done "
redoLog == nil ifTrue: [ ^ false ].
logEntries := redoLog getLogEntriesFor: self .
logEntries == nil ifTrue:[ ^ false ].

" cannot perform selective abort if receiver has a dependency tag "
self _hasDependencyList ifTrue: [ ^ false ].
 
" Refresh the state of the receiver."

self _selectiveAbort.

" tell the redo log to replay any operations on the receiver "
^ redoLog _redoOperationsForEntries: logEntries 
%

category: 'Private'
method:
addRedoLogEntryFor: aSelector withArgs: args

"Creates a redo log entry for the selector with the specified argument array,
 adds it to the redolog for this session.
 Sender responsible for adding to rcReadSet .  "

| redo logEntry | 

redo := System redoLog.
logEntry := LogEntry new.
logEntry receiver: self;
         selector: aSelector ;
         argArray: args.
redo addLogEntry: logEntry.
redo addConflictObject: self for: self.
%

category: 'Private'
method:
_redoAdd: anObject

"Performs the replay of adding anObject to the RcIdentitySet and returns true."

self _rcIncludes: anObject .
self _addForReplay: anObject.
^ true
%

category: 'Private'
method:
_redoAddAll: aCollection
  "Performs the replay of adding aCollection to the receiver and returns true."

  System _addToRcReadSet: self includingAllNodes: true.
  aCollection _isRcIdentityBag
    ifTrue: [ self _addAll: aCollection _asIdentityBag forReplay: true ]
    ifFalse: [ self _addAll: aCollection forReplay: true ].
  ^ true
%

category: 'Private'
method:
_redoRemove: anObject

"Performs the replay of removing anObject from the receiver and returns true."

self _rcIncludes: anObject .
^ self _removeForReplay: anObject.
%

category: 'Private'
method:
_redoRemoveAll: aCollection

"Performs the replay of removing aCollection elements from the receiver and returns true."

System _addToRcReadSet: self includingAllNodes: true.
aCollection _isRcIdentityBag
  ifTrue: [ ^ self _removeAll: aCollection _asIdentityBag errIfAbsent: true forReplay: true ]
  ifFalse: [ ^ self _removeAll: aCollection errIfAbsent: true forReplay: true ]
%

category: 'Private'
method:
_redoRemoveAllPresent: aCollection

"Performs the replay of removing the elements in aCollection from the receiver and returns true."

System _addToRcReadSet: self includingAllNodes: true.
aCollection _isRcIdentityBag
  ifTrue: [ ^ self _removeAll: aCollection _asIdentityBag errIfAbsent: false forReplay: true ]
  ifFalse: [ ^ self _removeAll: aCollection errIfAbsent: false forReplay: true ].
%

category: 'Reduced Conflict Support'
method:
_selectiveAbort
  "Performs an abort operation on the receiver. That is, if the object is
 committed, it removes any changes made by the current transaction and allows
 access to the committed state of the object."

  self _checkForRcIndexesBeforeRcSelectiveAbort.
  ^ self _primitiveSelectiveAbort
%
category: 'Reduced Conflict Support'
method: RcIdentitySet
_validateLegacyRcIndexSupport
  "legacy indexes are not supported on RcIndentitySet or RcLowMaintenanceIdentityBag (bug47179)"

  self error: 'Creating a legacy index on an RcIdentitySet or RcLowMaintenanceBag is not supported.'
%
