! file: gsnmethod2.gs 

set class GsNMethod
category: 'Converting ArrayBuilders'

classmethod: GsNMethod
convertArrayBuildersIn: directoryName list: listName
  "given a directoryName and a file named 'listName' within that
   directory, assuming listName has one fileName per line,
   convert each file in the list.  within the list,
   leading/trailing whitespace on each line is ignored,
   and lines starting with # are ignored." 
  
| dPath listF nC nNotC fName line fPath |
dPath := directoryName copy .
dPath last == $/ ifFalse:[ dPath add: $/ ].
listF := GsFile openRead:  dPath , listName  .
listF ifNil:[ ArgumentError signal:'file open failed for: ', (dPath , listName)].
nC := 0 . nNotC := 0 .
[ true ] whileTrue:[ 
  (line := listF nextLine) ifNil:[
    listF close .
    ^ nC asString, ' changed ' , nNotC asString , ' not changed '
  ].
  fName := line trimWhiteSpace .
  (fName at: 1) == $# ifFalse:[ 
    fPath := dPath , fName .
    [
      (self convertArrayBuildersInTopazScript: fPath) 
         ifTrue:[ nC := nC + 1 . GsFile gciLogClient:'  changed ', fName ] 
        ifFalse:[ nNotC := nNotC + 1 ].
    ] onException: Error do:[:ex |
      GsFile gciLogClient:'Error during ' , fName , ' ; ', ex description . 
      nil pause .
    ]
  ].
].
%


classmethod: GsNMethod
convertArrayBuildersInTopazScript: aFileName
  "Convert any ArrayBuilder productions in source String within
   specified file to CurlyArrayBuilder per $GEMSTONE/doc/bnf.txt.
   The file is overwritten in place. Return true if file changed,
   false otherwise."
  | f origContents result cmd lineNum line src startLnum |
  f := GsFile openRead: aFileName . 
  f ifNil:[ ArgumentError signal:'file open failed for: ', aFileName ].
  origContents := f contents .
  f close .
  f := GsFile openRead: aFileName . 
  f ifNil:[ ArgumentError signal:'file open failed for: ', aFileName ].
  result :=  String new .
  lineNum := 0 .
  [ true ] whileTrue:[ 
    (line := f nextLine) ifNil:[
      f close .
      result = origContents ifFalse:[
        f := GsFile openWrite: aFileName . 
        f ifNil:[ ArgumentError signal:'open for write failed for: ', aFileName ].
        (f nextPutAll: result) = result size ifFalse:[
          Error signal:'write failed to ' , aFileName
        ]. 
        f close .
        ^ true
      ].
      ^ false 
    ].
    lineNum := lineNum + 1 .
    result addAll: line .
    (cmd := line trimWhiteSpace) size > 0 ifTrue:[
      "per abbreviations allowed in topaz command decoding in src/tpaux.c"
      (      (cmd at: 1 equalsNoCase:'m') 
        or:[ (cmd at: 1 equalsNoCase:'cl')
        or:[ (cmd at: 1 equalsNoCase:'doi')
        or:[ (cmd at: 1 equalsNoCase:'interp')
        or:[ (cmd at: 1 equalsNoCase:'pri')
        or:[ (cmd at: 1 equalsNoCase:'run')
        or:[ (cmd at: 1 equalsNoCase:'rubyc')
        or:[ (cmd at: 1 equalsNoCase:'rubyr')
        or:[ (cmd at: 1 equalsNoCase:'rubym')
        or:[ (cmd at: 1 equalsNoCase:'ru') and:[ (cmd at: 1 equalsNoCase:'rub') not ]
           ]]]]]]]]] ) ifTrue:[ 
	 src := String new .   startLnum := lineNum .
	 [ line := f nextLine .
           lineNum := lineNum + 1 .
	   line == nil or:[ (line at: 1) == $% ] 
	 ] whileFalse:[
	   src addAll: line .
	 ].
         [
	   result addAll:( self convertArrayBuildersInString: src ).
         ] onException: Error do:[:ex |
           GsFile gciLogClient:'Error for source starting at line ', startLnum asString .
           ex pass .
         ].
	 line ifNotNil:[ result addAll: line  "the % line "]. 
      ].
    ].
  ].
%

classmethod: 
convertArrayBuildersInString: aString
  "Convert any ArrayBuilder productions in aString to
   CurlyArrayBuilder per $GEMSTONE/doc/bnf.txt . 
   Returns aString if no ArrayBuilder found, 
   otherwise returns a modified copy of aString."
  
  ^ self _convertArrayBuilders: aString stream: (ReadStream on: aString)
%

category: 'Converting ArrayBuilders - Private'
classmethod:
_convertArrayBuilders: aString stream: stream
  | result ch |
  result := aString class new .
  [ true ] whileTrue:[
    ch := stream nextOrNil .
    ch ifNil:[ ^ aString = result ifTrue:[ aString ] ifFalse:[ result]].
    self _parseChar: ch stream: stream to: result
  ]. 
%

classmethod: GsNMethod
_convertArrayBuildersIfNeeded: aString
 "called from within VM when parsing a doit, or
  when compiling a method and session methods are not activated. 
  Invoked from code in Behavior when session methods in use.
  VM has already found GemConvertArrayBuilder==true  "

  | stream |
^ [ 
    stream := ReadStream on: aString .
    self _convertArrayBuilders: aString stream: stream .
  ] onException: Error do:[:ex | | pos msg comErrInfo |
    pos := 1 . 
    msg := 'unknown ArrayBuilder conversion error' .
    [ | txt |
      pos := stream ifNotNil:[ stream position ].
      (txt := ex messageText) ifNotNil:[ msg := 'ArrayBuilder conversion, ', txt ].
    ] onException: Error do:[:exx |  "ignore"].
    comErrInfo := { { 1071 . pos . msg } } .
    { nil . comErrInfo . nil } 
  ].
%

classmethod:
_parseChar: ch stream: stream to: result
    ch == $# ifTrue:[
      stream peek == $[  ifTrue:[  
        "don't put $# in result"
        stream next .  "  throw away  $[ "
        self _parseOldArrayBuilder: stream to: result
      ] ifFalse:[
        result add: ch .
        self _parseSymbolLiteral: stream to: result
      ]
    ] ifFalse:[
      result add: ch .
      ch == $$ ifTrue:[ | nxt |  "character literal"
        (nxt := stream nextOrNil) ifNotNil:[ result add: nxt ]. ] ifFalse:[
      ch == $( ifTrue:[  "parenthesized expression"
        self _parseExpression: stream to: result end: $) ] ifFalse:[
      ch == $[ ifTrue:[  "block"
        self _parseExpression: stream to: result end: $] ] ifFalse:[
      ch == ${ ifTrue:[  "select block or CurlyArrayBuilder "
        self _parseExpression: stream to: result end: $} ] ifFalse:[
      ch == $'  ifTrue:[  
        self _parseStringLiteral: stream to: result   ] ifFalse: [
      ch == $"  ifTrue:[
        self _parseComment: stream to: result
      ]]]]]]
    ].
%

classmethod:
_parseOldArrayBuilder: stream to: result
  | ch |
  result add: ${  .
  stream peek == $  ifFalse:[ result add: $  "space after { for readability"].
  [ true ] whileTrue:[
    ch := stream nextOrNil .
    ch ifNil:[ ArgumentError signal:'premature end of stream in _parseOldArrayBuilder' ].
    ch == $,  ifTrue:[  "ensure space before/after .  per syntax"
       result last == $  ifFalse:[ result add: $  ].
       result add: $.  .  
       stream peek == $  ifFalse:[ result add: $  ].
    ] ifFalse:[
      ch == $]  ifTrue:[ 
        result last == $  ifFalse:[ result add: $  "space before } for readability"].
        result add: $} . 
        ^ self
      ] ifFalse:[
        self _parseChar: ch stream: stream to: result
      ] 
    ]
  ] 
%

classmethod: 
_parseStringLiteral: stream to: result
  | ch |
  [ true ] whileTrue:[
    ch := stream nextOrNil .
    ch ifNil:[ ArgumentError signal:'premature end of stream in _parseStringLiteral' ].
    result add: ch .
    ch == $' ifTrue:[
      stream peek == $' ifFalse:[ ^ self  "end of literal"].
      stream next .  "handle  adjacent single quotes"
      result add: $' .
    ]. 
  ].
%     

classmethod:
_parseSymbolLiteral: stream to: result
  "Caller has just consumed a # .
   Just parse enough to distinguish commas within a binary selector Symbol
    literal from commas that are separators or selectors on their own"
  | ch selectorChars |
  ch := stream peek .
  ch ifNil:[ ^ self ].
  selectorChars := '+-\*~<>=|/&@%,?!' .
  (selectorChars indexOf: ch startingAt:1) ~~ 0 ifTrue:[
    "we have a binary selector Symbol literal"
    ch := stream next .
    result add: ch .
    ch := stream peek .
    ch ifNil:[ ^ self ].
    (selectorChars indexOf: ch startingAt:1) ~~ 0 ifTrue:[
      "ch is part of the binary selector"
      ch := stream next .
      result add: ch .
    ].
  ].
%


classmethod:
_parseComment: stream to: result
  | ch |
  [ true ] whileTrue:[
    ch := stream nextOrNil .
    ch ifNil:[ ArgumentError signal:'premature end of stream in _parseComment' ].
    result add: ch .
    ch == $"  ifTrue:[ ^ self "end of comment" ]
  ]
%

classmethod:
_parseExpression: stream to: result end: endCh
  | ch |
  [ true ] whileTrue:[ 
    ch := stream nextOrNil .
    ch ifNil:[ ArgumentError signal:'premature end of stream in _parseExpression' ].
    ch == endCh ifTrue:[
      result add: ch .
      ^ self
    ].
    self _parseChar: ch stream: stream to: result
  ] 
%
classmethod: GsNMethod
_convertArrayBuildersIn: directoryName list: listName refDir: refDirPath
  "refDirPath is to a separate checkout previously converted ;
   its listName files control the operation.
   Compare results of conversion of files in directoryName  
   with files in refDirPath , and raise an error if files differ.
"
  
| dPath listF nC nNotC line fName fPath |
dPath := directoryName copy .
dPath last == $/ ifFalse:[ dPath add: $/ ].
listF := GsFile openRead:  refDirPath , listName  .
listF ifNil:[ ArgumentError signal:'file open failed for: ', (dPath , listName)].
nC := 0 . nNotC := 0 .
[ true ] whileTrue:[ 
  (line := listF nextLine) ifNil:[
    listF close .
    ^ nC asString, ' changed ' , nNotC asString , ' not changed '
  ].
  fName := line trimWhiteSpace .
  (fName at: 1) == $# ifFalse:[ 
    fPath := dPath , fName .
    [
      (self convertArrayBuildersInTopazScript: fPath) ifTrue:[  | refFil newFil |
        nC := nC + 1 . GsFile gciLogClient:'  changed ', fName  .
        refFil := GsFile openRead: refDirPath , fName .
        newFil := GsFile openRead: fPath .
        refFil contents = newFil contents ifFalse:[
          GsFile gciLogClient:'difference in ', fPath 
        ].
        refFil close .  newFil close .
      ] ifFalse:[ 
         nNotC := nNotC + 1 
      ].
    ] onException: Error do:[:ex |
      GsFile gciLogClient:'Error during ' , fName , ' ; ', ex description . 
      nil pause .
    ]
  ].
].
%

category: 'Method Lookup Cache Statistics'
classmethod:
_mluCachesReport: minSize

"Returns an Array of the form 
   { { String, value , ... }     (various stats)
     { Class , lookupCacheSize , ... )   
     { GsNMethod, IP , selector, envId, send_site_cache_size , ... }
   }"

^ self _oneArgPrim: 1 with:  minSize
%

classmethod: 
methodLookupCachesReport: minSize 

| raw arr rpt lf lines line overflows |
(minSize _isSmallInteger not or:[ minSize < 0 ]) ifTrue:[
  ArgumentError signal:'minSize must be a SmallInteger >= 0'. 
].
raw := self _mluCachesReport: minSize .
lf := Character lf .
(rpt := '--- selected session stats' copy) add: lf .

arr := raw at: 1 . "selected method lookup stats ( name, value pairs)"
lines := { } .
1 to: arr size by: 2 do:[:j | | val |
  val := arr at: j + 1 .
  (line := '  ' copy) add: (arr at: j); add: '  ' ; add: val asString .
  lines add: line .
].
(SortedCollection withAll: lines)  do:[:s | rpt add: s ; add: lf . ].

rpt add: '--- Class , descending size of per-class method lookup cache '; add: lf .
arr := raw at: 3 . "size of classes' method lookup cache"
lines := { } .
1 to: arr size by: 2 do:[:m | | cls sz |
  sz := arr at: (m + 1 ).
  sz >= minSize ifTrue:[
    line := String new .
    (cls := arr at: m ) isMeta ifTrue:[ line add:'meta' ].
    "ifFalse:[ cls isRubySingletonClass ifTrue:[ line add:'singleton ' ]]."
    line add: cls thisClass name ; add: '  ' ; add: sz asString .
    line := Association newWithKey: line value: sz .
    lines add: line .
  ].
].
(lines sortDescending:'value') do:[:assoc | rpt add: '  '; add: assoc key; add: lf ].

arr := raw at: 2 . "send-site caches details"
lines := { } .
overflows := { } .
arr := raw at: 2 . 
1 to: arr size by: 5 do:[:k | | cacheSize |
  cacheSize := arr at: k + 4 .
  (cacheSize < 0 or:[ cacheSize >= minSize]) ifTrue:[ 
    | cls selEnvId selector ipOfs meth isBlk |
    meth := arr at: k .
    ipOfs := arr at: k + 1 .
    selector := arr at: k + 2 .
    selEnvId := arr at: k + 3 .
    meth isMethodForBlock ifTrue:[  meth := meth homeMethod . isBlk := true ].
    line := String withAll: (cls := meth inClass) thisClass name .
    line add:( meth environmentId > 0 
               ifTrue:[ cls isMeta ifTrue:[ $. ] ifFalse:[ $# ] ]
               ifFalse:[ cls isMeta ifTrue:[ '(C) >> '] ifFalse:[ ' >> ']]) .
    line add: meth selector ; add: '  ' .
    isBlk ifNotNil:[ line add: ' (in block) ' ].
    line add:'  IP:'; add: ipOfs asString ; add: '  ';
        add:' '; add: selector ; add: ' , ' .
    cacheSize < 0 ifTrue:[ 
      line add: 'polymorphic overflow'. overflows add: line 
    ] ifFalse:[ 
      line add: 'send-site size: '; add: cacheSize asString . lines add: line
    ]
  ].
].
rpt add: '--- Overflowed send sites '; add: lf .
(overflows sortAscending:'') do:[:s | rpt add: '  '; add: s ; add: lf ].
rpt add: '--- Send sites with size >= ' ; add: minSize asString ; add: lf .
(lines sortAscending:'') do:[:s | rpt add: '  '; add: s ; add: lf  ].

^ rpt
%

classmethod: GsNMethod
_findImplementation: aSelector dir: directoryName list: listName 
  "Return a report String describing where in topaz scripts 
   aSelector is implemented. aSelector may be either a Symbol
   or a message pattern String like   'kw1: arg1 kw2: arg2' .
   Both directoryName and listName are Strings.
   directoryName is a path to the file listName which
   contains a list of files to search.  Lines in the file beginning
   with $# are comment lines."
| dPath listF rpt line fName fPath|
rpt := String new .
dPath := directoryName copy .
dPath last == $/ ifFalse:[ dPath add: $/ ].
listF := GsFile openRead:  dPath , listName  .
listF ifNil:[ ArgumentError signal:'file open failed for: ', (dPath , listName)].
[ true ] whileTrue:[ 
  (line := listF nextLine) ifNil:[
    listF close .
    ^ rpt .
  ].
  fName := line trimWhiteSpace .
  (fName at: 1) == $# ifFalse:[ 
    fPath := dPath , fName .
    [
      rpt addAll:(self _findImplementation: aSelector inTopazScript: fPath fileName: fName) 
    ] onException: Error do:[:ex |
      GsFile gciLogServer: 'ERROR during ' , fName  .
      ex pass
    ].
  ].
].
%

classmethod: GsNMethod
_findImplementation: aSelector inTopazScript: aPath fileName: fName 
  "Return a report String describing where in topaz script aPath 
   aSelector is implemented. aSelector may be either a Symbol
   or a message pattern String like   'kw1: arg1 kw2: arg2'  "
  | f keyWords numColons result lineNum selSym lf tab prevSrc |
  f := GsFile openRead: aPath .   lf := Character lf . tab := Character tab .
  f ifNil:[ ArgumentError signal:'file open failed for: ', fName ].
  aSelector _isSymbol ifFalse:[ | pat idx nxt lim sel |
    pat := String new .  "extract selector from message pattern"
    sel := aSelector trimWhiteSpace .
    lim := sel size .
    idx := 1 .
    [ (nxt := sel indexOf: $:  startingAt: idx)  ~~ 0 ] whileTrue:[
      pat addAll:(sel copyFrom: idx to: nxt) .
      idx := nxt + 1 .
      [ idx <= lim and:[ (sel at: idx) isSeparator] ] whileTrue:[
         idx := idx + 1   "skip whitespace after colon"
      ].
      [ idx <= lim and:[ (sel at: idx) isSeparator not ] ] whileTrue:[
         idx := idx + 1   "skip until end of argument name"
      ].
      [ idx <= lim and:[ (sel at: idx) isSeparator] ] whileTrue:[
         idx := idx + 1   "skip whitespace after argumentName"
      ].
    ].
    selSym := pat asSymbol .
  ] ifTrue:[
    selSym := aSelector
  ].
  (selSym includesValue: $: ) 
       ifTrue:[ keyWords := selSym keywords . numColons := keyWords size ]
      ifFalse:[ keyWords := { selSym } . numColons := 0 ].
  result :=  String new . 
  lineNum := 0 .
  [ true ] whileTrue:[ | line methSrc cmd |
    (line := f nextLine) ifNil:[ f close .  ^ result ].
    lineNum := lineNum + 1 .
    cmd := line trimWhiteSpace . 
    cmd size > 0 ifTrue:[
      "per abbreviations allowed in topaz command decoding in src/tpaux.c"
      ((     cmd at: 1 equalsNoCase:'m') 
       or:[ (cmd at: 1 equalsNoCase:'cl')
       or:[ (cmd at: 1 equalsNoCase:'doi')
       or:[ (cmd at: 1 equalsNoCase:'interp')
       or:[ (cmd at: 1 equalsNoCase:'pri')
       or:[ (cmd at: 1 equalsNoCase:'run')
       or:[ (cmd at: 1 equalsNoCase:'rubyc')
       or:[ (cmd at: 1 equalsNoCase:'rubyr')
       or:[ (cmd at: 1 equalsNoCase:'rubym')
       or:[ (cmd at: 1 equalsNoCase:'ru') and:[ (cmd at: 1 equalsNoCase:'rub') not ]
           ]]]]]]]]] ) ifTrue:[  "have doit or method of some kind"

         ((cmd at: 1 equalsNoCase:'m') 
         or:[ cmd at: 1 equalsNoCase:'cl']) ifTrue:[ "have method: or classmethod:" | keywd |
           [
             (line := f nextLine) ifNil:[ f close .  ^ result ].
	     lineNum := lineNum + 1 .   
             (line at: 1) == $% ifFalse:[ line := line trimWhiteSpace ].
             line size == 0
           ] whileTrue .
           keywd := keyWords at: 1 .
	   (line at:1 equals: keywd) ifTrue:[  | found methStartLine |
             methSrc := line copy .   found := false .
	     methStartLine := lineNum . 
             numColons == 0 ifTrue:[
               found :=  line = keywd asString .
             ] ifFalse:[ | ofs |
	       ofs := keywd size + 1 .
	       [ | pos endOfSel nextL |
		 pos := f position .
		 (nextL := f nextLine) ifNil:[ f close .  ^ result ].
		 methSrc addAll: nextL .
		 lineNum := lineNum + 1 .
		 (nextL at: 1) == $% ifTrue:[ f position: pos ].
		 (endOfSel := (nextL occurrencesOf: $: ) == 0) ifFalse:[
		   line addAll: nextL trimWhiteSpace . 
		 ]. 
		 endOfSel    
	       ] whileFalse .
	       (line occurrencesOf: $: ) == numColons ifTrue:[ | j kwlim |
		 found := true .
		 j := 2 . kwlim := numColons .
		 [ j <= kwlim ] whileTrue:[ | idx |
		   keywd := keyWords at: j .
		   (idx := line findString: keywd startingAt: ofs) == 0 ifTrue:[
		     kwlim := 0 . found := false 
		   ] ifFalse:[  "line contains keywd"
		     j := j + 1 .  ofs := idx + keywd size .
		   ]
		 ].
               ].
             ].
	     found ifTrue:[
	       result size == 0 ifTrue:[ result add: lf ].
	       result add: fName , ':' , methStartLine asString , '  ', selSym , lf .
	     ] ifFalse:[
               methSrc := nil .
             ].
	   ].
         ].
         [(line at:1) == $% ] whileFalse:[
           (line := f nextLine) ifNil:[ f close . ^ result ].
           methSrc ifNotNil:[ methSrc addAll: line ].
           lineNum := lineNum + 1 .
         ].
         methSrc ifNotNil:[
           (prevSrc ~~ nil and:[ (methSrc = prevSrc) not]) ifTrue:[ 
	      result addAll:'     <different source>', lf 
           ].
           prevSrc := methSrc .
           methSrc := nil .
         ].
      ].
    ].
  ]
%

category: 'Repository Conversion'
method:  
recompile

"Recompiles the method for execution in a Gs64 v3.0 or later system.
 Returns a GsNMethod, or signals a CompileError or CompileWarning .
 
 See also  recompileIntoMethodDict:intoCategories: "

self methodCompilerVersion >= 4 ifTrue:[ ^ self "recompile not needed"].
^ self recompileIntoMethodDict: nil intoCategories: nil
%

! fix 46519
category: 'Repository Conversion'
method:  
_literalVariablesForRecompile
  "Returns an Array of the literal variables referenced in the receiver.
   The result has key,value pairs of  the form  aSymbol,  aSymbolAssociation"
| litVars obsDict |
litVars := { }  .   
obsDict := self class _classVars at: #ObsoleteClassesDict 
              ifAbsent:[ "handle recompile of default block for SortedCollection in upgrade"
                 SymbolDictionary new 
              ].
self literals do: [:aLit |
  | litName newLit srcName lookupKey obsArr clsName litVal |
  aLit _isSymbol ifFalse:[ 
    (aLit isKindOf: SymbolAssociation ) ifTrue:[ 
      litName := aLit key . litVal := aLit value 
    ] ifFalse:[
      (aLit isClass and:[ aLit isMeta not ]) ifTrue:[ "was optimized ref to a class"
         litName := aLit name . clsName := litName . litVal := aLit 
      ].  
    ].  
    litName ifNotNil:[
      (obsArr := obsDict at: litName otherwise: nil) ifNotNil:[ 
         (litVal ~~ nil and:[ litVal == (obsArr at:1)]) ifTrue:[ 
            srcName := obsArr at: 2 . 
            lookupKey := obsArr at: 3 . 
            newLit := Globals associationAt: lookupKey otherwise: nil .
         ].
       ].
       newLit ifNotNil:[ litVars add: srcName ; add: newLit ]
         ifNil:[ 
            clsName ifNotNil:[ 
              (Globals associationAt: clsName otherwise: nil) ifNotNil:[ :assoc|
                assoc _value == aLit ifTrue:[ litVars add: clsName ; add: assoc ]
              ].
            ] ifNil:[ litVars add: litName ; add: aLit "use old assoc"]
        ].
    ] ifNil:[  "an optimized literal that was not a class"
      | assoc |
      assoc := self _literalValToAssocations: aLit .
      assoc _isArray ifTrue:[ assoc do:[:a | litVars add: a key ; add: a ]]
                    ifFalse:[ litVars add: assoc key ; add: assoc ].
    ].  
  ].  
].
^ litVars
%


! edited for 42285 , 46027
category: 'Repository Conversion'
method: 
recompileIntoMethodDict: aMethodDict intoCategories: aCategDict

"Recompiles the method for execution in a Gs64 v3.0 or later system.

 Literal variables whose key is in ObsoleteClasses
 are replaced by the appropriate association from the ObsoleteClassesDict .
 Other literal variables are looked up in the literal pool of the receiver,
 before searching class variables, class pool dictionaries, or 
 the current symbolList.  Thus recompilation should work without knowing
 what symbolList was used when the receiver was created.   

 The result is a GsNMethod if compilation succeeds, otherwise
 an error is generated.  environmentId zero is used for all compilations.

 If aMethodDict is not nil, and the compilation succeeds,
 the resulting method is added to aMethodDict instead of to
 the receiver's method dictionary.  This is used to add methods
 to per-session dictionaries.

 If aMethodDict is not nil and aCategDict is not nil and
 the compilation succeeds, the resulting method is added aCategDict
 instead of the receiver's categories.

 If the receiver is an anonymous method, the sender of this method is
 reponsible for saving the result."

| cls litVars newSrc |
litVars := self _literalVariablesForRecompile .
newSrc := self sourceString .
(System gemConfigurationAt:#GemConvertArrayBuilder) ifTrue:[
  newSrc := GsNMethod convertArrayBuildersInString: newSrc .
].
cls := self inClass .
cls ifNil:[ 
  ^ newSrc _compileInContext: nil symbolList: nil oldLitVars: litVars
		environmentId: 0 flags: 0 
] ifNotNil:[ | categ |
 categ := cls categoryOfSelector: self selector . 
  ^ cls _checkCompileResult: ( cls _primitiveCompileMethod: newSrc symbolList: nil 
         category: categ oldLitVars: litVars 
         intoMethodDict: aMethodDict intoCategories: aCategDict
         environmentId: 0 )
       source: newSrc .
]
%

