!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: ReadWriteStream.gs 22222 2009-09-02 00:36:20Z stever $
!
! Superclass Hierarchy:
!   ReadWriteStream, WriteStream, PositionableStream, Stream, Object.
!
!=========================================================================

expectvalue %String
doit
WriteStream 
	_newKernelSubclass: 'ReadWriteStream'
	instVarNames: #()
	classVars: #()
	classInstVars: #()
	poolDictionaries: { }
	inDictionary: Globals
        options: #()
	reservedOop: nil.
%

! set category to match other Streams
doit
ReadWriteStream category: 'Streams'.
true
%

! Remove existing behavior from ReadWriteStream
doit
ReadWriteStream removeAllMethods.
ReadWriteStream class removeAllMethods.
true.
%

category: 'For Documentation Installation only'
classmethod: ReadWriteStream
installDocumentation

self comment:
'A ReadWriteStream is a PositionableStream that allows 
its objects to be read and written. 

See ANSI 5.9.7'.
%

! ------------------- Class methods for ReadWriteStream
! ------------------- Instance methods for ReadWriteStream
category: 'collectionStream'
method: ReadWriteStream
contents
	"It is unspecified whether or not the returned collection is the same object as the backing store
	collection. However, if the returned collection is not the same object as the stream backing store
	collection then the class of the returned collection is the same class as would be returned if the
	message #select: was sent to the backing store collection.

	ANSI 5.9.3.1"

	^itsCollection.
%
category: 'gettableStream'
method: ReadWriteStream
next

"Returns the next object that the receiver can access for reading.  Generates
 an error if an attempt is made to read beyond the end of the stream."

(self atEnd)
   ifTrue: [ ^ EndOfStream signal ].
position := position + 1.
^ itsCollection at: (position - 1)
%
category: 'gettableStream'
method: ReadWriteStream
next: anInteger

	"A number of objects equal to amount are removed from the receiver's future sequence values and
	appended, in order, to the end of the receiver's past sequence values. A collection whose elements
	consist of those objects, in the same order, is returned. If amount is equal to 0 an empty collection
	is returned.
	The result is undefined if amount is larger than the number of objects in the receiver's future
	sequence values.

	ANSI 5.9.2.4"

anInteger < 0 ifTrue: [^self _error: #rtErrBadStreamPosition].
^super next: anInteger.
%
category: 'gettableStream'
method: ReadWriteStream
nextLine

	"Each object in the receiver's future sequence values up to and including the first occurrence of the
	objects that constitute an implementation defined end-of-line sequence is removed from the future
	sequence values and appended to the receiver's past sequence values. All of the transfered
	objects, except the end-of-line sequence objects, are collected, in order, as the elements of a string
	that is the return value. The result is undefined if there are no future sequence values in the
	receiver or if the future sequence values do not include the end-of-line sequence.

	ANSI 5.9.2.5"

	| string |
	string := super nextLine.
	(string isKindOf: String) ifTrue: [^string].
	self _error: #rtErrBadStreamColl.
%
category: 'gettableStream'
method: ReadWriteStream
nextMatchFor: anObject

	"The first object is removed from the receiver's future sequence value and appended to the end of
	the receiver's past sequence values. The value that would result from sending #= to the object with
	anObject as the argument is returned.
	The results are undefined if there are no future sequence values in the receiver.

	ANSI 5.9.2.6"

	^self next = anObject.
%
category: 'sequencedStream'
method: ReadWriteStream
position
	"Returns the number of sequence values in the receiver's 
	past sequence values. ANSI 5.9.1.4"

	^super positionA.
%
category: 'sequencedStream'
method: ReadWriteStream
position: anInteger
	"If the number of sequence values in the receiver's past sequence values is smaller than amount,
	move objects in sequence from the front of the receiver's future sequence values to the back of th
	receiver's past sequence values until the number of sequence values in the receiver's past
	sequence values is equal to amount.
	If the number of sequence values in the receiver's past sequence values is greater than amount,
	move objects in sequence from the back of the receiver's past sequence values to the front of th
	receiver's future sequence values until the number of sequence values in the receiver's past
	sequence values is equal to amount.
	If the number of sequence values in the receiver's past sequence values is equal to amount no
	action is taken. 
	
	ANSI 5.9.1.5"

	^super positionA: anInteger.
%
category: 'gettableStream'
method: ReadWriteStream
skipTo: anObject

	"Each object in the receiver's future sequence values up to and including the first occurrence of an
	object that is equivalent to anObject is removed from the future sequence values and appended
	to the receiver's past sequence values. If an object that is equivalent to anObject is not found in
	the receiver's future sequence values, all of the objects in future sequence values are removed
	from future sequence values and appended to past sequence values. If an object equivalent to
	anObject is not found false is returned. Otherwise return true.

	ANSI 5.9.2.10"

	self position + 1 to: itsCollection size do: [:i | 
		(itsCollection at: i) = anObject ifTrue: [
			self position: i.
			^true.
		].
	].
	self setToEnd.
	^false.
%

category: 'Passivation - Support'
set compile_env: 0
method: ReadWriteStream
_fastNext
  "Returns the next item on the input stream without end of stream checks.
 This may result in a bad access error instead of an end of stream error."

  | res |
  res := itsCollection at: position.
  position := position + 1.
  ^ res
%
category: 'Passivation - Support'
set compile_env: 0
method: ReadWriteStream
nextElements: count into: anObject

"Stores the next count elements that the receiver can access for reading
 into anObject.  The receiver's collection and anObject must be compatible
 SequenceableCollections.  Returns the count of elements read.

 Generates an error if an attempt is made to read beyond the end of the stream."

"Used by PassiveObject."

count <= 0 ifTrue:[ ^ 0 ].
(self atEnd)
   ifTrue: [ ^ EndOfStream signal ].

anObject replaceFrom: 1 to: 1 + count - 1 with: itsCollection startingAt: position .
position := position + count.
^ count
%
category: 'Passivation - Support'
set compile_env: 0
method: ReadWriteStream
_fastPosition
  "Returns the receiver's current position reference for accessing the sequence
 of objects.  The position is actually the next element of the collection to be
 read or written; the position is incremented by each read or write.  In
 general, to start reading or writing at the nth element of a collection, the
 position must be set to n."

  "See bug 42596: Compatible with Legacy Stream positioning"

  ^ position
%
category: 'Passivation - Support'
set compile_env: 0
method: ReadWriteStream
_fastPosition: pos
  "Sets the position of the receiver without checking the validity of the
 argument 'pos'.  This may result in bad index errors instead of end
 of stream errors in subsequent access of the stream."

  "See bug 42596: Compatible with Legacy Stream positioning"

  position := pos
%

