! ========================================================================
! LdapDirectoryServer.gs
!
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
! ========================================================================

expectvalue %String
run
^ Object _newKernelSubclass: 'LdapDirectoryServer'
        instVarNames: #( 'uri' 'bindDN' 'bindPW' )
        classVars: #( 'AllLdapDirectoryServers' )
        classInstVars: #()
        poolDictionaries: { }
        inDictionary: Globals
        options: #( ) 
        reservedOop: 1225
%

! Remove existing behavior from LdapDirectoryServer
doit
LdapDirectoryServer removeAllMethods.
LdapDirectoryServer class removeAllMethods.
true
%

! ------------------- Class comment for LdapDirectoryServer
category: 'For Documentation Installation only'
classmethod: LdapDirectoryServer
installDocumentation

self comment:
'This class is the logical representation of an LDAP Directory Server to
 GemStone.  It contains the following instance variables:
   uri               - A Uniform Resource Identifier for LDAP conforming to
                       RFC 4516.
   bindDN            - A string which is the distinguished name (DN) used to
                       bind to the server to perform lookups, or nil if anonymous
		       binds are to be used (no bindDN required).
   bindPW            - a ByteArray, which is the encrypted password for bindDN,
                       or nil if anonymous binds are to be used.
'.
%

! ------------------- Class methods for LdapDirectoryServer
category: 'Private - Initializing'
set compile_env: 0
classmethod: LdapDirectoryServer
_smartInitialize
"This method should only be called once during filein or image upgrade."
  (AllLdapDirectoryServers == nil)
    ifTrue:[ | newArray |
      newArray := Array new .
      newArray _unsafeSetOop: 156929  .
      newArray objectSecurityPolicy: (Globals at:#DataCuratorObjectSecurityPolicy) .
      AllLdapDirectoryServers := newArray .
  ] .
  ^ true
%
category: 'Accessing'
set compile_env: 0
classmethod: LdapDirectoryServer
allServers
  ^ AllLdapDirectoryServers
%
category: 'Illegal Operations'
classmethod: LdapDirectoryServer
new

"Disallowed.  To create a new LdapDirectoryServer, use newWithUri:... instead."

self shouldNotImplement: #new
%
category: 'Illegal Operations'
classmethod: LdapDirectoryServer
new: anInt

"Disallowed.  To create a new LdapDirectoryServer, use newWithUri:... instead."
self shouldNotImplement: #new:
%

category: 'Instance Creation'
classmethod: LdapDirectoryServer
basicNewWithUri: uri bindDN: aBindDn password: password
"Creates a new instance of the receiver, but does not add the new instance
 to the list of LDAP servers used to process login requests."
 
| result |
result := super new.
result uri: uri copy ;
       bindDN: aBindDn copy ;
       password: password .

^ result
%

category: 'Instance Creation'
classmethod: LdapDirectoryServer
newWithUri: uri bindDN: aBindDn password: password
"Creates a new instance of the receiver and adds the resulting object to
 the list of LdapDirectoryServer objects used to authorize logins."
 
| result |
result := self basicNewWithUri: uri
               bindDN: aBindDn
	       password: password .
self allServers add: result .	       
^ result
%

category: 'Connection Testing'
classmethod: LdapDirectoryServer
testConnectionToServer: uri bindDN: aBindDn password: password
"Attempts to perform a bind using aBindDn and password to the LDAP server specified by
 uri.  Returns true if the connection was successful, otherwise returns false."
 
 | result |
 "Create a temporary instance."
result := self basicNewWithUri: uri
               bindDN: aBindDn
	       password: password .
^ result testBind
%


category: 'Searching'
set compile_env: 0
classmethod: LdapDirectoryServer
findServerWithUri: aUriString
"Searches for an instance of LdapDirectoryServer with the given URI.
 Returns nil if the directory server was not found."
 
^ self allServers detect:[:e| e uri = aUriString ] ifNone:[ nil ]
%

category: 'Removing'
set compile_env: 0
classmethod: LdapDirectoryServer
removeServerWithUri: aUriString
"Removes an instance of LdapDirectoryServer with the given URI.
 Returns true if the directory server was removed, false if it 
 could not be found."

| server |
server := self findServerWithUri: aUriString .
server isNil
  ifTrue:[ ^ false "not found" ] .

self allServers removeIdentical: server .
^ true
%	    

! ------------------- Instance methods for LdapDirectoryServer
category: 'Accessing'
set compile_env: 0
method: LdapDirectoryServer
uri
  ^ uri
%
category: 'Accessing'
set compile_env: 0
method: LdapDirectoryServer
bindDN
  ^ bindDN
%
category: 'Accessing'
set compile_env: 0
method: LdapDirectoryServer
bindPW
  ^ bindPW
%

category: 'Updating'
set compile_env: 0
method: LdapDirectoryServer
uri: aString
  uri := aString
%
category: 'Updating'
set compile_env: 0
method: LdapDirectoryServer
bindDN: aString
  bindDN := aString
%

category: 'Private'
set compile_env: 0
method: LdapDirectoryServer
_encryptPassword: aString
"Encrypt aString into a ByteArray. Store the ByteArray in the bindPW inst var.
 Returns the receiver."

<primitive: 1008>
self _validateClass: LdapDirectoryServer .
aString _validateClass: String .
self _primitiveFailed: #_encryptPassword: args: { aString } . 
self _uncontinuableError
%

category: 'Updating'
set compile_env: 0
method: LdapDirectoryServer
password: aString
^ aString == nil
    ifTrue:[ nil ]
   ifFalse:[ self _encryptPassword: aString ]
%


category: 'Private'
set compile_env: 0
method: LdapDirectoryServer
_testBind

<primitive: 1009>
self _validateClass: LdapDirectoryServer .
self _primitiveFailed: #_testBind: args: { } . 
self _uncontinuableError
%

category: 'Private'
set compile_env: 0
method: LdapDirectoryServer
testBind
"Attempt to bind to the specified LDAP server. Returns true if successful, false if
 the password is invalid.  Raises an error if an LDAP error occurs."

| result |
result := self _testBind .
result class == String
  ifTrue:[ IOError signal: result ].
^ result
%

expectvalue true
run
^ LdapDirectoryServer _smartInitialize .
%

expectvalue true
run
LdapDirectoryServer class removeSelector: #_smartInitialize .
^ true
%
commit
