!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!  IndexManager, Object.
!
!=========================================================================

! class created in idxclasses.topaz

removeallmethods IndexManager
removeallclassmethods IndexManager

category: 'For Documentation Installation only'
classmethod: IndexManager
installDocumentation

self comment:
'IndexManager is provides protocol for managing and maintaining indexed collections.
 The single instance of IndexManager should be accessed via the class message #current.
 In normal use, you will use IndexManager to turn autoCommit for index maintenance
 operations off and on and to define the dirtyObjectCommitThreshold and 
 percentTempObjSpaceCommitThreshold, which are control parameters for autoCommit.

 Note that autoCommit will also make the system respond to TransactionBacklog 
 notifications by commiting.

 IndexManager current tracks all instances of UnorderedCollection that have had indexes
 created for them. You may use IndexManager current to #removeAllIndexes, which removes
 indexes from all indexed collections in the system or you may use IndexManager 
 current to #removeAllIncompleteIndexes, which removes partial indexes from collections
 when index creation failed without completing.

 Automatic garbage collection of unreferenced UnorderedCollections is _not_ preformed, so
 one must explicitly remove all indexes from an UnorderedCollection that is no longer used.

 IndexManager may be subclassed.'.
%
! ------------------- Class methods for IndexManager
category: 'Accessing'
classmethod: IndexManager
current

  ^ Current ifNil:[ 
     Current := IndexManager _basicNew initialize .
     Current .
  ].
%
classmethod: IndexManager
new
   "use  
      IndexManager current 
    to get the current instance"

   self shouldNotImplement: #new
%

category: 'Accessing'
classmethod: IndexManager
autoCommit

^self current autoCommit
%

classmethod: IndexManager
sessionAutoCommit

^self current sessionAutoCommit
%

category: 'Index Maintenance'
classmethod: IndexManager
removeAllIndexes

^ self current removeAllIndexes.
%

category: 'Index Maintenance'
classmethod: IndexManager
removeAllIncompleteIndexes

^ self current removeAllIncompleteIndexes.
%

category: 'Updating'
classmethod: IndexManager
dirtyObjectCommitThreshold: anInteger

self current dirtyObjectCommitThreshold: anInteger
%

category: 'Updating'
classmethod: IndexManager
percentTempObjSpaceCommitThreshold: anInteger

self current percentTempObjSpaceCommitThreshold: anInteger
%

category: 'Updating'
classmethod: IndexManager
autoCommit: aBool
"Set autoCommit status; if true, this enabled commits to be performed without 
specific commands, including due to TransactionBacklog"

self current autoCommit: aBool
%

category: 'Updating'
classmethod: IndexManager
sessionAutoCommit: aBool

self current sessionAutoCommit: aBool
%

! ------------------- Instance methods for IndexManager

! deleted IndexManager>>addIndex: renamed to IndexManager>>_addIndex: v2.1

category: 'Accessing'
method: IndexManager
_addIndex: anIndex
  "  Add the given index into the collection of all indexes."

  anIndex objectSecurityPolicy: GsIndexingObjectSecurityPolicy.
  self getAllIndexes add: anIndex
%
category: 'Index Creation'
set compile_env: 0
method: IndexManager
createRcEqualityIndexFor: anNSC on: aPathString withLastElementClass: aClass
  "Replaces UnorderedCollection>>_createEqualityIndexOn:withLastElementClass:commitInterval:, modulo the error handler"

  ^ (RcEqualityIndexSpecification path: aPathString lastElementClass: aClass)
    createIndexOn: anNSC
%
category: 'Index Creation'
set compile_env: 0
method: IndexManager
createEqualityIndexFor: anNSC on: aPathString withLastElementClass: aClass
  ^ (EqualityIndexSpecification path: aPathString lastElementClass: aClass)
    createIndexOn: anNSC
%
category: 'Index Creation'
set compile_env: 0
method: IndexManager
createIdentityIndexFor: anNSC on: aPathString
  ^ (IdentityIndexSpecification path: aPathString) createIndexOn: anNSC
%
category: '(as yet unclassified)'
method: IndexManager
getAllIndexes
  "Return a collection of all indexes in the system."

  allIndexes == nil
    ifTrue: [ 
      allIndexes := RcIdentitySet new.
      allIndexes objectSecurityPolicy: GsIndexingObjectSecurityPolicy ].
  ^ allIndexes
%
category: 'Accessing'
method: IndexManager
getAllNSCRoots

  "Return a collection of all nsc's with indexes in the system."

  | set |
  set := IdentitySet new.
  self getAllIndexes do: [:index |
    set add: index nscRoot
  ].
  ^set
%

category: 'Index Maintenance'
method: IndexManager
removeAllIndexes
  "This algorithm ignores the collections and traverses the dependent objects 
   directly removing the dependent for each object that does not have a 
   modification tracking dependency. The objects with modification tracking 
   dependencies are tracked and the SharedDependencyLists table is rebuilt 
   for only those objects ... Without any modification tracking going on, the 
   SharedDependencyLists table is simply reinitialized. The bookkeeping required 
   for handling modification tracking dependencies is the extra cost that makes 
   this one a bit more expensive than _oldRemoveAllIndexes in the simple 'flat' 
   case.

   IndexManagerAutoCommitPolicy is used to periodically commit if
   temp object space gets low and to perform a final commit at end
   of operation.

   The removeAllIndexes progress count is used to track progress, but
   now tracks the number of indexed objects left that have path terms
   to remove."

  | sys |
  sys := System.
  sys
    setIndexProgressCountTo: UnorderedCollection statValueForRemovingAllIndexes.	"Setup progress counter"
  self
    executeStartingIndexMaintenance: [ 
      [ 
      | sharedDepLists trackedObjects |
      "Perform using index manager auto commit.."
      sharedDepLists := SharedDependencyLists allEntries.
      trackedObjects := IdentitySet new.
      DependencyList depMapKeysToHiddenSet: 2.
      sys setProgressCountTo: (sys hiddenSetSize: 2).
      System
        hiddenSet: 2
        do: [ :dependentObject | 
          | depList hasTracker terms |
          depList := DependencyList for: dependentObject.
          hasTracker := false.
          terms := {}.
          1 to: depList size by: 2 do: [ :i | 
            (depList at: i + 1) = 0
              ifTrue: [ hasTracker := true ]
              ifFalse: [ 
                | pathTerm |
                pathTerm := depList at: i.
                terms add: pathTerm ] ].
          hasTracker
            ifTrue: [ 
              "modification trackers present, so remove the individual pathTerms
               from dependencyList"
              terms
                do: [ :pathTerm | DependencyList removePathTerm: pathTerm for: dependentObject ].
              trackedObjects add: dependentObject ]
            ifFalse: [ 
              sharedDepLists remove: depList ifAbsent: [  ].
              DependencyList set: nil for: dependentObject ].
          sys decrementProgressCountBy: 1 ].
      self getAllNSCRoots
        do: [ :nsc | 
          | iList |
          "Clear _indexedPaths field from all indexed collections"
          iList := nsc _indexedPaths.
          iList
            ifNotNil: [ 
              iList hasTrackingObjects
                ifTrue: [ iList removeAllIndexesFor: nsc ]
                ifFalse: [ nsc _indexedPaths: nil ] ] ].
      sharedDepLists isEmpty
        ifTrue: [ 
          "no modification trackers present, so simply initialize SharedDependencyLists"
          SharedDependencyLists initialize ]
        ifFalse: [ 
          | trackedDepLists |
          "rebuild SharedDependencyLists for modification trackers"
          trackedDepLists := IdentitySet new.
          trackedObjects
            do: [ :dependentObject | 
              | depList |
              depList := DependencyList for: dependentObject.
              trackedDepLists add: depList ].
          SharedDependencyLists initialize.
          trackedDepLists
            do: [ :depList | SharedDependencyLists at: depList logging: false ] ].
      self resetAllIndexes	"Initialize IndexManager allIndexes reference" ]
        ensure: [ sys setIndexProgressCountTo: 0; setProgressCountTo: 0 ] ]
%

category: 'Index Maintenance'
method: IndexManager
removeAllIncompleteIndexes

  self getAllNSCRoots do: [:nsc | 
    self removeAllIncompleteIndexesOn: nsc
  ].
%
category: 'Index Updating'
method: IndexManager
removeAllIndexesOn: anNSC

  | result |
result := self removeAllCompleteIndexesOn: anNSC.
result == anNSC
  ifTrue: [
    self removeAllIncompleteIndexesOn: anNSC.
  ].
^result
%
category: 'Index Updating'
method: IndexManager
removeAllCompleteIndexesOn: anNSC
  "Remove all complete indexes for anNSC. If all of the anNSC's indexes can be removed, 
   this method returns the receiver.
   If an error occurs during index removal, it may not be possible to 
   commit the current transaction later.
  "
  | iList rootIndexes hasNonRootIndex |
  (iList := anNSC _indexedPaths) ifNil: [ ^ anNSC ].

  hasNonRootIndex := false.
  rootIndexes := { } .
  " scan each entry to see if a non-root index is present "
  1 to: iList size by: 2 do: [ :i |
    " if it is a root index ... "
    (((iList at: i + 1) == 1) and: [ (iList at: i) isComplete])
        " add it to a list "
        ifTrue: [ rootIndexes add: (iList at: i) ]
        " otherwise flag that we're really not removing ALL indexes "
        ifFalse: [ hasNonRootIndex := true ]
  ].
  ^ [  | result |
	  IndexManager current executeStartingIndexMaintenance: [
		result := anNSC _removeAllRootIndexes: rootIndexes 
                		hasNonRootIndex: hasNonRootIndex.
		result == anNSC ifTrue: [
		  rootIndexes do: [:each | self _removeIndex: each.  ].
		].
      ].
      result
	] onSynchronous: Error do:[ :ex | 
    IndexingErrorPreventingCommit resignal: ex objWithIndex: anNSC .
	]
%

category: 'Index Updating'
method: IndexManager
removeAllIncompleteIndexesOn: anNSC
  "Remove all incomplete indexes for anNSC. 
   Return the number of incomplete indexes found."

  | iList incompleRootIndexes |
  (iList := anNSC _indexedPaths) == nil
    ifTrue: [ ^ 0 ].
  incompleRootIndexes := Array new.
  1 to: iList size by: 2 do: [ :i | 
    " scan each entry to see if a non-root index is present "
    ((iList at: i + 1) == 1 _and: [ (iList at: i) isComplete not ])
      ifTrue: [ 
        " if it is an incomplete root index ... add it to a list "
        incompleRootIndexes add: (iList at: i) ] ].
  IndexManager current
    executeStartingIndexMaintenance: [ 
      incompleRootIndexes
        do: [ :indexObj | 
          indexObj size > 0
            ifTrue: [ 
              | pathTerm |
              pathTerm := indexObj _findFirstUnsharedPathTerm.
              indexObj nscRoot _undoIndexCreation: indexObj pathTerm: pathTerm ].
          self _removeIndex: indexObj ] ].
  ^ incompleRootIndexes size
%

category: 'Index Updating'
method: IndexManager
removeEqualityIndexFor: anNSC on: aPathString

  self _removeIndex: self _equalityIndex for: anNSC on: aPathString
%
category: 'Index Updating'
method: IndexManager
removeIdentityIndexFor: anNSC on: aPathString

  self _removeIndex: self _identityIndex for: anNSC on: aPathString
%

! deleted IndexManager>>removeIndex: renamed to IndexManager>>_removeIndex: v2.1

category: 'Accessing'
method: IndexManager
_removeIndex: anIndex

  "Remove the given index from the collection of all indexes.
   Make sure that IVs are niled in anIndex, since it will
   survive (in rc redo log) until abort or commit."

  self getAllIndexes remove: anIndex ifAbsent: [ nil ].
  anIndex _clear.
%
category: '(as yet unclassified)'
method: IndexManager
resetAllIndexes
  allIndexes
    ifNil: [ 
      "force lazy init to fresh RcIdentityBag to avoid conflicts (bug43804)"
      self getAllIndexes ]
    ifNotNil: [
      allIndexes _isRcIdentityBag
        ifTrue: [ 
          "switch to using RcIdentitySet"
          allIndexes := nil.
          self getAllIndexes ]
        ifFalse: [ 
          "allow for concurrent index creation (bug43803)" 
          allIndexes removeAll: allIndexes ] ]
%

! deleted IndexManager _createIndex:for:on:withLastElementClass:
! deleted IndexManager _createIndex:for:on:withLastElementClass:equalityIndexClass:

category: 'Private'
method: IndexManager
_doCommit
  | systm |
  systm := System.
  systm _commitPrintingDiagnostics
    ifFalse: [ self _errorCouldNotCommitDuringIndexCreation ].
  systm transactionMode == #'manualBegin'
    ifTrue: [ systm beginTransaction ]
%
category: 'Private'
method: IndexManager
_equalityIndex

  ^ #equality
%
category: 'Private'
method: IndexManager
_identityIndex

  ^ #identity
%
category: 'Index Updating - Private'
method: IndexManager
_removeIndex: indexType for: anNSC on: aPathString
  | iList indexes index pathArray |
  anNSC _checkIndexPathExpression: aPathString.
  iList := anNSC _indexedPaths.
  iList ifNil: [ self _error: #'rtErrNoIndexForPath' args: {aPathString} ].
  pathArray := aPathString asArrayOfPathTerms.	" check if an index exists for the path string "
  indexes := anNSC _findIndexesWithPath: pathArray.
  indexes isEmpty
    ifFalse: [
      " make sure it is an identity index or equality index"
      indexes
        do: [ :iObj | 
          indexType == self _identityIndex
            ifTrue: [ 
              iObj isIdentityIndex
                ifTrue: [ index := iObj ] ]
            ifFalse: [ 
              iObj isRangeEqualityIndex
                ifTrue: [ index := iObj ] ] ] ].
  index
    ifNil: [
      " no index was found with the given path "
      ^ self _error: #'rtErrNoIndexForPath' args: {aPathString} ].
  [
  IndexManager current
    executeStartingIndexMaintenance: [
      index preIndexRemoval.
      anNSC _removeIndex: index.
      self _removeIndex: index ] ]
    onSynchronous: Error do:[ :ex | IndexingErrorPreventingCommit resignal: ex objWithIndex: anNSC ] 
%

category: 'Testing'
method: IndexManager
shouldCommit
"Answer true if a commit should be performed during index maintenance.
   Note - that this method should only be executed from within a
   executeStartingIndexMaintenance: block, since _autoCommitPolicy
   may not have been initialized correctly otherwise."

| policy |
self autoCommit ifFalse: [ ^ false ].

(policy := self _autoCommitPolicy ) ifNil:[ ^ false ].
^ policy shouldCommit
%

category: 'Testing'
method: IndexManager
_autoCommitPolicy

^ (System __sessionStateAt: 11)
%

category: 'Testing'
method: IndexManager
_clearAutoCommitPolicy

System __sessionStateAt: 11 put: nil
%

category: 'Testing'
method: IndexManager
_initializeAutoCommitPolicy

System __sessionStateAt: 11 put: (IndexManagerAutoCommitPolicy on: self)
%

category: 'Initializing'
method: IndexManager
initialize

  "When creating initial instance, make sure all IVs are set, since we don't
   want to dirty the object during normal operation."

  self autoCommit: false .       
  self getAllIndexes.
  self useKeyEncryption: true.
  self dirtyObjectCommitThreshold: SmallInteger maximumValue . "disabled by default"
  "was 75% ,  Gs64 v3.0 change to 60 for new AlmostOutOfMemory handling. "
  self percentTempObjSpaceCommitThreshold: 60.
  self objectSecurityPolicy: GsIndexingObjectSecurityPolicy.
%

category: 'Accessing'
method: IndexManager
autoCommit
  "Use session-specific value if sessionAutoCommit has been set (bug40602)"

| sac |
(sac := self sessionAutoCommit) ~~ nil
  ifTrue: [ ^sac ].
autoCommit == nil ifTrue:[ autoCommit := false ].
^autoCommit
%

category: 'Accessing'
method: IndexManager
useKeyEncryption
  
  ^ useKeyEncryption
%

category: 'Updating'
method: IndexManager
useKeyEncryption: aBool
  
  useKeyEncryption := aBool

%

category: 'Updating'
method: IndexManager
autoCommit: aBool
"Set autoCommit status; if true, this enabled commits to be performed without 
specific commands, including due to TransactionBacklog"

autoCommit := aBool
%

category: 'Accessing'
method: IndexManager
sessionAutoCommit

^SessionTemps current at: #INDEX_MANAGER_AUTO_COMMIT ifAbsent: []
%

category: 'Updating'
method: IndexManager
sessionAutoCommit: aBoolOrNil
"Override persistent autoCommit on a session by session basis.
 If <aBoolOrNil> is nil, remove override.
 If true, this enabled commits to be performed without specific commands, including 
 due to TransactionBacklog"

aBoolOrNil == nil
  ifTrue: [SessionTemps current removeKey: #INDEX_MANAGER_AUTO_COMMIT ifAbsent: []]
  ifFalse: [ SessionTemps current at: #INDEX_MANAGER_AUTO_COMMIT put: aBoolOrNil ]
%

category: 'Accessing'
method: IndexManager
dirtyObjectCommitThreshold

^ dirtyObjectCommitThreshold
%

category: 'Accessing'
method: IndexManager
rcBtreeLeafNodeClass

  ^ RcBtreeLeafNode
%

category: 'Accessing'
method: IndexManager
rcBtreeBasicLeafNodeClass

  self useKeyEncryption ifTrue: [ ^ RcBtreeBasicLeafNode ].

  ^ RcBtreeLeafNode
%

category: 'Accessing'
method: IndexManager
btreeBasicLeafNodeClass

  self useKeyEncryption ifTrue: [ ^ BtreeBasicLeafNode ].

  ^ BtreeLeafNode
%

category: 'Accessing'
method: IndexManager
btreeLeafNodeClass

  ^ BtreeLeafNode
%

category: 'Accessing'
method: IndexManager
sortNodeClass

  self useKeyEncryption ifTrue: [ ^ BasicSortNode ].

  ^ SortNode
%

category: 'Updating'
method: IndexManager
dirtyObjectCommitThreshold: anInteger

dirtyObjectCommitThreshold := anInteger
%

category: 'Accessing'
method: IndexManager
percentTempObjSpaceCommitThreshold

^ percentTempObjSpaceCommitThreshold
%

category: 'Updating'
method: IndexManager
percentTempObjSpaceCommitThreshold: anInteger

  anInteger _isSmallInteger ifFalse:[ anInteger _validateClass: SmallInteger ].
  (anInteger < 10 or:[ anInteger > 100]) ifTrue:[
     OutOfRange new name:'commit threshold' min: 10 max: 100 actual: anInteger;
       details: 'invalid commit threshold' ;  signal
  ].
  percentTempObjSpaceCommitThreshold := anInteger
%

category: 'Updating'
method: IndexManager
objectSecurityPolicy: anObjectSecurityPolicy

"Assigns the receiver and the allIndexes collection to anObjectSecurityPolicy."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  anObjectSecurityPolicy == GsIndexingObjectSecurityPolicy
    ifFalse: [ self _error: #objectSecurityPolicyNotSharedDepListObjectSecurityPolicy ].

  self _objectSecurityPolicy: anObjectSecurityPolicy.
  self getAllIndexes objectSecurityPolicy: anObjectSecurityPolicy.
] ensure:[
  prot _leaveProtectedMode
]
%

category: 'Index Maintenance'
method: IndexManager
commitIndexMaintenance: indexObjOrNil at: progressCount
  self shouldCommit
    ifTrue: [ 
      indexObjOrNil ifNotNil: [ indexObjOrNil progress: progressCount ].
      self _doCommit ]
%

category: 'Private'
method: IndexManager
execStartingIndexMaintenance: aBlock
  "Behavior of index maintenance with respect to transaction state:

When in manual mode:
    When IndexManager current autoCommit is true:
        When originally outside of a transaction:
            Begin a transaction.
            Commit the final transaction of index creation.
            End outside of a transaction.
        When originally inside of a transaction:
            Commit the final transaction of index creation.
            End outside of a transaction.
When in auto mode:
    When IndexManager current autoCommit is true:
        Commit the final transaction of index creation.
"

  | inAutoCommit autoCommitPolicy systm |
  systm := System.
  (inAutoCommit := self autoCommit)
    ifTrue: [ 
      systm transactionMode == #'manualBegin'
        ifTrue: [ 
          systm inTransaction
            ifFalse: [ 
              systm needsCommit
                ifTrue: [ self _error: #'rtErrAbortWouldLoseData' ].
              systm beginTransaction ] ].
      self _initializeAutoCommitPolicy ].
  (autoCommitPolicy := self _autoCommitPolicy)
    ifNil: [ aBlock value ]
    ifNotNil: [ 
      [ autoCommitPolicy evaluate: aBlock for: self ]
        ensure: [ 
          inAutoCommit
            ifTrue: [ self _clearAutoCommitPolicy ] ] ].
  inAutoCommit
    ifTrue: [ 
      systm inTransaction
        ifTrue: [ 
          systm _commitPrintingDiagnostics
            ifFalse: [ self _errorCouldNotCommitDuringIndexCreation ] ] ]
%

category: 'Index Maintenance'
method: IndexManager
executeStartingIndexMaintenance: aBlock
"Disable STN_GEM_TIMEOUT, then perform index maintenance.

Behavior of index maintenance with respect to transaction state:

When in manual mode:
    When IndexManager current autoCommit is true:
        When originally outside of a transaction:
            Begin a transaction.
            Commit the final transaction of index creation.
            End outside of a transaction.
        When originally inside of a transaction:
            Commit the final transaction of index creation.
            End outside of a transaction.
When in auto mode:
    When IndexManager current autoCommit is true:
        Commit the final transaction of index creation."
| sys |
sys := System .
^ [ 
    sys disableStoneGemTimeout .
    self execStartingIndexMaintenance: aBlock
  ] ensure:[ sys enableStoneGemTimeout ] .
%


category: 'Updating Indexes - Private'
method: IndexManager
_addMappingsFor: anNSC on: indexObj
  "Adds the mappings in the index dictionary, dependency lists, and so on for all
 elements in the receiver."

  | pathTerm setOffset hasSet |
  " since this index may use path terms from other indexes,
find the first one that is not shared by another index (only
need to add mappings from that point onward) "
  setOffset := indexObj _findSetValuedPathTermOffset.
  pathTerm := indexObj _findFirstUnsharedPathTerm.
  pathTerm
    ifNil: [ 
      " index is subsumed by an existing index "
      pathTerm := indexObj lastPathTerm.
      ^ pathTerm _doAddSubsumedIndex: indexObj nsc: anNSC ].
  setOffset == 0
    ifTrue: [ hasSet := false ]
    ifFalse: [ 
      "Determine if there is a set-valued instance variable before the unshared path
 term."
      hasSet := setOffset < pathTerm offset ].
  ^ pathTerm _doAddMappingsFor: anNSC on: indexObj indexManager: self hasSet: hasSet
%

category: 'Querying'
method: IndexManager
nscsWithBadIndexes
  "IndexManager current nscsWithBadIndexes"

  ^ self getAllNSCRoots
    reject: [ :each | 
      | audit |
      audit := each auditIndexes.
      audit = 'Indexes are OK'
        or: [ 
          audit
            =
              'Indexes are OK and the receiver participates in one or more indexes with collection-valued path terms' ] ]
%

category: 'Querying'
method: IndexManager
nscsWithIncompleteIndexes
        "IndexManager current nscsWithIncompleteIndexes"

	^ self getAllNSCRoots select: [:each | each _hasIncompleteIndexes].
%

category: 'Querying'
method: IndexManager
usageReport
"
	Returns a string describing all indexes, the collection sizes, and 
	the (visible) methods that could create these indexes.

	IndexManager current usageReport.
"

	| indexes methods keys stream |
	(stream := (AppendStream on: String new))
		lf; nextPutAll: 'Index Type - path - lastElementClass';
		lf; tab; nextPutAll: 'Collection size (collection OOP)';
		lf; tab; tab; nextPutAll: 'Index creation methods';
		lf.
	indexes := self _reportDataForIndexes.
	methods := self _reportDataForMethods.
	keys := (indexes collect: [:each | each key]) asSet asSortedCollection.
	keys do: [:eachKey | 
		stream lf; nextPutAll: eachKey.
		(indexes select: [:each | each key = eachKey]) do: [:each | 
			stream lf; tab; nextPutAll: each value.
		].
		(methods select: [:each | each key = eachKey]) do: [:each | 
			stream lf; tab; tab; nextPutAll: each value.
		].
		stream lf.
	].
	^stream contents.
%

category: 'Querying'
method: IndexManager
_reportDataForIndexes
  "private"

  | list |
  list := OrderedCollection new.
  self getAllIndexes do: [ :each | | firstStr string |
      string := each class name copyFrom: 1 to: each class name size - 5.
      (5 < string size and: [ (string copyFrom: 1 to: 5) = 'Range' ])
        ifTrue: [ string := string copyFrom: 6 to: string size ].
      firstStr := string.
      string := each
        inject: ''
        into: [ :str :pathTerm | str , '.' , pathTerm name ].
      string := string copyFrom: 2 to: string size.
      list add: {firstStr.
          string.
          (each isIdentityIndex
            ifTrue: [ '' ]
            ifFalse: [ each lastElementClass name ]).
          (each nscRoot).
          (each isComplete)} ].
  list := list collect: [ :each | | nsc |
      nsc := each at: 4.
      Association new
        key: (each at: 1) , '-' , (each at: 2) , ((each at: 3) notEmpty
                ifTrue: [ '-' , (each at: 3) ]
                ifFalse: [ '' ])
        value:
          nsc size printString , ' (' , nsc asOop printString , ')' , ((each at: 5)
                ifTrue: [ '' ]
                ifFalse: [ ' - INCOMPLETE' ]) ].
  ^ list
%

category: 'Querying'
method: IndexManager
_reportDataForMethods
		"private"

	| indexCreationSelectors classOrganizer smallList bigList |
	classOrganizer := ClassOrganizer new.
	smallList := OrderedCollection new.
	indexCreationSelectors := #(
		#'createEqualityIndexOn:'
		#'createEqualityIndexOn:withLastElementClass:'
		#'createIdentityIndexOn:'
		#'createRcEqualityIndexOn:'
		#'createRcEqualityIndexOn:withLastElementClass:'
	).
	indexCreationSelectors do: [:each | | arrays |
		arrays := classOrganizer sendersOf: each.
		1 to: arrays first size do: [:i | 
			smallList add: (Array
				with: (arrays first at: i)
				with: each
				with: (arrays last at: i)).
		].
	].
	smallList := smallList reject: [:each | each first inClass == UnorderedCollection].
	smallList := smallList reject: [:each | each first inClass name == #'IndexedQueryExtensionsTestCase'].
	bigList := OrderedCollection new.
	smallList do: [:each | 
		| selector pieces firstPiece lastPiece source index |
		selector := each at: 2.
		pieces := selector subStrings: $:.
		pieces := pieces copyFrom: 1 to: pieces size - 1.
		firstPiece := pieces first.
		lastPiece := pieces last.
		source := each first sourceString.
		index := (each at: 3) - 1.
		[
			0 < (index := source indexOfSubCollection: firstPiece startingAt: index + 1).
		] whileTrue: [
			| readStream array word |
			readStream := ReadStream on: (source copyFrom: index to: source size).
			array := { ((word := self _nextWordFrom: readStream) copyFrom: 7 to: word size - 8) .
						nil . nil . nil . nil } .
			word := self _nextWordFrom: readStream.
			word := word copyFrom: 2 to: word size - 1.
			[''';.' includes: word last] whileTrue: [word := word copyFrom: 1 to: word size - 1].
			array at: 2 put: word.
			2 <= pieces size ifTrue: [
				self _nextWordFrom: readStream.
                                word := self _nextWordFrom: readStream.
				word := word reject: [:char | ''';.' includes: char].
			] ifFalse: [
				word := ''.
			].
			array 
				at: 3 put: word;
				at: 4 put: each first inClass name;
				at: 5 put: each first selector  .
			bigList add: array.
		].
	].
	bigList := bigList collect: [:each | 
		Association new
			key: ((each at: 1) , '-' , (each at: 2) , 
					((each at: 3) notEmpty ifTrue: ['-' , (each at: 3)] ifFalse: [''])) 
			value: ((each at: 4) , '>>#' , (each at: 5)).
	].
	^ bigList
%

! make sure default values are initialized
run
IndexManager current .
^ true
%

! delete IndexManager createEqualityIndexFor:on:withCollator:
! delete IndexManager createRcEqualityIndexFor:on:withCollator:
! delete IndexManager _buildIndex:for:on:

category: 'Index Creation'
set compile_env: 0
method: IndexManager
createIndexFor: anNsc fromSpec: anIndexSpec
  | index |
  anIndexSpec _validateSpecificationOn: anNsc.
  " see if incomplete indexes "
  anNsc _hasIncompleteIndexes
    ifTrue: [ ^ self _error: #'rtErrCollectionWithIncompleteIndex' ].
  index := anIndexSpec _createIndex.
  self
    _checkExistingIndexesFor: anNsc
    against: index
    withSpec: anIndexSpec
    ifPresent: [ ^ self ].
  self
    executeStartingIndexMaintenance: [ self _buildIndex: index for: anNsc fromSpec: anIndexSpec ]
%
category: 'Index Updating - Private'
set compile_env: 0
method: IndexManager
_buildIndex: index for: anNsc fromSpec: anIndexSpec
  [ | pathArray original |
  pathArray := anIndexSpec path asArrayOfPathTerms.
  index _preIndexCreationBuildIndexDictionaryFor: pathArray for: anNsc.
  index nscRoot: anNsc.
  self _addIndex: index.
  anNsc _getIndexList
    buildPathTermsFor: index
    with: pathArray
    fromSpec: anIndexSpec.
  anNsc _putInWriteSet.
  original := index preIndexCreation.
  index isIndexOnRootNsc
    ifTrue: [ " optimize creation of indexes on elements of the NSC itself "
      index
        addDirectMappingsFor: anNsc _asIdentityBag
        indexList: anNsc _indexedPaths ]
    ifFalse: [ self _addMappingsFor: anNsc on: index ].
  index postIndexCreation: original ]
    onSynchronous: Error
    do: [ :ex | IndexingErrorPreventingCommit resignal: ex objWithIndex: anNsc ]
%
category: 'Index Updating - Private'
set compile_env: 0
method: IndexManager
_checkExistingIndexesFor: anNsc against: index withSpec: anIndexSpec ifPresent: presentBlock
  "evaluate presentBlock if an existing index matches the specified index"

  | pathArray indexes pathString indexOptions |
  indexOptions := index options.
  indexOptions legacyIndex
    ifTrue: [
      "legacy indexes are not supported on RcIndentitySet or RcLowMaintenanceIdentityBag (bug47179)"
      anNsc _validateLegacyRcIndexSupport ].
  (anNsc _indexedPaths ) 
    ifNotNil: [ :ilist |
      ilist indexObjectsDo: [:idxObj |
        "All indexes on anNsc must all be either legacy or btree plus ... no mixing"
        | idxObjOptions  |
        idxObjOptions := idxObj options.
        ((idxObjOptions legacyIndex == indexOptions legacyIndex) 
          and: [idxObjOptions btreePlusIndex == indexOptions btreePlusIndex])
          ifFalse: [
            self error: 'A ', 
              (idxObjOptions legacyIndex ifTrue: ['legacy'] ifFalse: ['btree plus']),
              ' index exists for this collection and it conflicts with the requested ',
              (idxObjOptions btreePlusIndex ifTrue: ['legacy'] ifFalse: ['btree plus']),
              ' index.'] ] ].
  pathString := anIndexSpec path.
  pathArray := pathString asArrayOfPathTerms.
  indexes := anNsc _findIndexesWithPath: pathArray.
  indexes
    do: [ :idxObj | 
      idxObj indexType == index indexType
        ifTrue: [ 
          "index already exists on the given path for this type"
          (idxObj _checkSameIndexOptionsAs: anIndexSpec)
            ifFalse: [ "Bug44038: If index options are different, then signal an error"
              ^ self error: 'An ' , index indexType asString printString
                    , ' index already exists on the path '
                    , pathString printString , ' for this collection with '
                    , (idxObj termsRequired
                        ifTrue: [ 'REQUIRED' ]
                        ifFalse: [ 'OPTIONAL' ])
                    , ' path terms, which conflicts with the requested '
                    , (index termsRequired
                        ifTrue: [ 'REQUIRED' ]
                        ifFalse: [ 'OPTIONAL' ]) , ' path terms.' ].
          index indexType == #'identity'
            ifTrue: [ "no chance for conflicting index types for identity index"
              ^ presentBlock value ].
          (index _checkSameLastElementClassAs: idxObj)
            ifTrue: [ "If path and lastElementClass are the same, we'll let it slide"
              ^ presentBlock value ].	"Bug 43035: signal error if attempt was made to create a different
           index with the same path"
          ^ self error: 'An ' , index indexType asString printString
                , ' index already exists on the path ' , pathString printString
                , ' for this collection with lastElementClass of '
                , idxObj lastElementClassDescription printString
                , ' which conflicts with the requested lastElementClass of '
                , index lastElementClassDescription printString , '.' ] ]
%
category: 'Index Maintenance'
set compile_env: 0
method: IndexManager
_oldRemoveAllIndexes
  "The _oldRemoveAllIndexes algorithm is actually pretty spiffy as long as there is
   a single index per nsc and each object participates in only one index. In fact 
   this algorithm will (slightly) out-perform the new removeAllIndexes algorithm. 
   When an object has more than one entry in the dependency list, this algorithm 
   quickly blows up as the shared dependency list is recalculated for each element 
   "

  | sys |
  sys := System.
  ^ [ 
  | roots sz |
  sys
    setIndexProgressCountTo: UnorderedCollection statValueForRemovingAllIndexes.
  roots := self getAllNSCRoots.
  sz := roots size.
  sys setProgressCountTo: sz.
  1 to: sz do: [ :n | 
    | nsc result |
    nsc := roots at: n.
    result := self removeAllIndexesOn: nsc.
    nsc == result
      ifFalse: [ 
        self
          _error: #'rtErrRemoveAllIndexesFailed'
          args:
            {nsc.
            result} ].
    sys decrementProgressCountBy: 1 ] ]
    ensure: [ 
      sys
        setIndexProgressCountTo: 0;
        setProgressCountTo: 0 ]
%
category: 'Index Maintenance'
set compile_env: 0
method: IndexManager
removeAllTracking
  "This method removes all indexes and all modification tracking meta data. This is
   the fastest algorithm of all. It unconditionally removes the dependencs for each
   object.

   IndexManagerAutoCommitPolicy is used to periodically commit if
   temp object space gets low and to perform a final commit at end
   of operation.

   The removeAllIndexes progress count is used to track progress, but
   now tracks the number of indexed objects left that have path terms
   to remove."

  | sys |
  "Setup progress counter"
  sys := System.
  sys
    setIndexProgressCountTo: UnorderedCollection statValueForRemovingAllIndexes.
  "Perform using index manager auto commit.."
  self
    executeStartingIndexMaintenance: [ 
      [ 
      | nscs |
      nscs := self getAllNSCRoots.
      sys setProgressCountTo: nscs size.
      DependencyList depMapKeysToHiddenSet: 2.
      DependencyList clearForHiddenSet: 2.	"Clear dependencyList for the dependent objects"
      1 to: nscs size do: [ :k | 
        | uc |
        uc := nscs _at: k.
        uc _indexedPaths: nil.	"Clear the indexed structures from UnorderedCollections"
        sys decrementProgressCountBy: 1 ].
      SharedDependencyLists initialize.	"Initialize the SharedDependencyLists structure"
      self resetAllIndexes	"Initialize IndexManager allIndexes reference" ]
        ensure: [ sys setIndexProgressCountTo: 0 ; setProgressCountTo: 0 ] ]
%

category: 'Private'
method: IndexManager
_nextWordFrom: aReadStream
  "Assume that the aReadStream's collection is a kind of String.  Returns the
 next word in the string or nil if there is no next word."

  | result |
  result := String new.
  aReadStream skipSeparators.
  [ aReadStream atEnd not and: [ aReadStream peek isSeparator not ] ]
    whileTrue: [ result add: aReadStream next ].
  result size == 0
    ifTrue: [ ^ nil ]
    ifFalse: [ ^ result ]
%
category: 'Repository Conversion'
classmethod: IndexManager
_loadBitMap: bm
  | files |
  bm removeAll.
  files := GsFile contentsOfDirectory: self getNscIndexDir onClient: false.
  files := files
    select: [ :e | (e findPattern: (Array with: '.bm') startingAt: 1) ~~ 0 ].
  1 to: files size do: [ :n | 
    | fn |
    fn := files at: n.
    bm readFromFile: fn ].
  ^ bm size
%
category: 'Repository Conversion'
classmethod: IndexManager
_loadBitmap: bm fromFile: fn
  ^ SortedCollection _loadBitmap: bm fromFile: fn
%
category: 'Repository Conversion'
classmethod: IndexManager
buildFileNameForGem: anInt
  | result |
  result := String withAll: 'failedCharCollNscs_'.
  result
    addAll: anInt asString;
    addAll: '.bm'.
  ^ result
%
category: 'Repository Conversion'
classmethod: IndexManager
getNscIndexDir
  ^ SortedCollection getUpgradeLogDir , '/ccnscbm'
%

category: 'Repository Conversion'
classmethod: IndexManager
rebuildAllCharCollIndexes: anArray
  "Rebuild indexes for an Array of nscs with CharacterColletion indexes 
     by recording the indexes aassociated with the nsc, removing all indexes 
     on the nsc, and then rebuilding all indexes.

   Caller must handle and catch AlmostOutOfMemory errors and handle commits
 and aborts."

  | sys commitBlock oldAutoCommit |
  sys := System.
  commitBlock := [ 
  sys commitTransaction
    ifFalse: [ 
      sys abortTransaction.
      TransactionError signal: 'Unexpected commit failure in index rebuild' ] ].
  oldAutoCommit := self autoCommit.
  [ 
  self autoCommit: true.
  1 to: anArray size do: [ :n | 
    | nsc indexSpec |
    nsc := anArray at: n.
    indexSpec := nsc indexSpec.
    GsFile gciLogServer: 'Rebuilding indexes for nsc: ', nsc asOop printString.
    GsFile gciLogServer: indexSpec printString.
    nsc removeAllIndexes.
    indexSpec createIndexesOn: nsc.
    indexSpec := nsc indexSpec.
    GsFile gciLogServer: 'Rebuilt indexes for nsc: ', nsc asOop printString.
    GsFile gciLogServer: indexSpec printString ] ]
    ensure: [ 
      self autoCommit: oldAutoCommit.
      commitBlock value ]
%

category: 'Repository Conversion'
classmethod: IndexManager
writeTotalsFiles
  SortedCollection writeTotalsFile: 'ccidx_ok.txt' counterIndex: 0.
  SortedCollection writeTotalsFile: 'ccidx_failed.txt' counterIndex: 1.
  SortedCollection writeTotalsFile: 'ccidx_done.txt' counterIndex: 2.
  ^ true
%
