!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
! Superclass Hierarchy:
!   GsSingleRefPathFinder, Object.
!
!=========================================================================

! ------------------- Class definition for GsSingleRefPathFinder

expectvalue /Class
run
GsSingleRefPathFinder comment: 
'GsSingleRefPathFinder is a class used to determine how each object in a list
are connected to the repository.  It is typically used to determine why objects
expected to be collected by global garbage collection (markForCollection) were
not removed.  The list of objects connecting an object to the repository is
called a reference path.  An object may have more than one reference path,
however this class only determines one reference path for every search object.

Instance Variables:

searchObjects - a GsBitmap containing all objects for which reference paths will
be searched.

limitObjects - an array of GsBitmaps.  The first element contains the base limit
set of the search, that is, a group of objects known to be
connected to the repository.  The base limit set is returned by
the method SystemRepository buildLimitSetForRefPathScan.  The
next element in the array are the children of the based limit
set, and the next are the grandchildren, etc.

allSearches - anArray containing instances of GsSingleRefPathFinderForObject.  One
instance for each search object.

numSearchesActive - number of searches not yet completed.

otScanDone - Boolean indicating if the object table scan has been completed.
This is the first phase of the scan and only runs once.

printToLog - Boolean indicating if messages and results are to be written to
stdout of the process.

scanStartTime - A SmallInteger representing the timestamp of when the scan
started.

opStartTime - A SmallInteger - representing the timestamp of when an operation
started.  Used to print elapsed times to the log.

maxThreads - number of threads to use to perform the scan.

lockWaitTime - number of seconds to wait when acquiring the garbage collection
lock before starting the scan.

pageBufferSize - number of 16 KB pages each thread will buffer.  Must be a power
of 2.

percentCpuLimit - percentage of total system CPU which the scan may consume.

maxLimitSetDescendantObjs - maximum size of a generation of objects reachable
from the limit set.  At the beginning of the scan, the default limit set is
traversed until the size of the generation exceeds this value.  Default is
one million objects.

maxLimitSetDescendantLevels - maximum number of generations traversed to build
the limit set.  At the beginning of the scan, the default limit set is
traversed until the number of generations searched this value.  Default is 
unlimited.

Steps to find a reference path:

1) Create an instance with default settings:
| inst |
inst := GsSingleRefPathFinder newForSearchObjects: (Array with: mySearchObject)

2) Run the scan
inst runScan

3) Build the result objects (An Array of GsSingleRefPathResult objects)
inst buildResultObjects

The GsSingleRefPathResult object is the equivalent of the structured Array 
returned by findReferencePathToObj* methods in earlier releases.

4) For display, collect the results as strings:
<resultObjs> collect: [:e | e resultString]

For example:

| inst |
inst := GsSingleRefPathFinder newForSearchObjects: { myObj }.
inst runScan.
(inst buildResultObjects) 
   collect: [:e | e resultString]

Steps 2-4 can be done using inst scanAndReport. For example:

(GsSingleRefPathFinder newForSearchObjects: { myObj }) scanAndReport
'
%


! ------------------- Remove existing behavior from GsSingleRefPathFinder
removeallmethods GsSingleRefPathFinder
removeallclassmethods GsSingleRefPathFinder
! ------------------- Class methods for GsSingleRefPathFinder
set compile_env: 0
category: 'Defaults'
classmethod: GsSingleRefPathFinder
defaultLockWaitTime

"Default number of seconds to wait for the garbage collection lock expressed in seconds."
^ 120
%
category: 'Defaults'
classmethod: GsSingleRefPathFinder
defaultMaxThreads

"Default number of threads to use to scan the repository."
^ SystemRepository _aggressiveMaxThreadCount
%
category: 'Defaults'
classmethod: GsSingleRefPathFinder
defaultPageBufferSize

"Default number of 16 KB disk pages for each thread to buffer.  Must be a power of 2."
^ 8
%
category: 'Defaults'
classmethod: GsSingleRefPathFinder
defaultPercentCpuLimit

"Default maximum percentage of total CPU that the all threads in the scan may consume."
^ 95 "percent of total CPU"
%
category: 'Defaults'
classmethod: GsSingleRefPathFinder
defaultLimitSetDescendantsMaxLevels

"Default number of levels that may be traversed to seed the limit set."
 ^SmallInteger maximumValue "effectively infinite"
%
category: 'Defaults'
classmethod: GsSingleRefPathFinder
defaultLimitSetDescendantsMaxObjs

"Maximum size of a limit set descendant generation in number of objects."
^ 1000000
%
category: 'Defaults'
classmethod: GsSingleRefPathFinder
defaultPrintToLog

"Default boolean value for printing output to stdout of the process."
^ true
%


category: 'Instance Creation'
classmethod: GsSingleRefPathFinder
newForSearchObjects: anArray
"Creates a new instance with default settings."
^self
  newForSearchObjects: anArray
  withMaxThreads: nil
  waitForLock: nil
  pageBufSize: nil
  percentCpuLimit: nil
  maxLimitSetDescendantObjs: nil
  maxLimitSetDescendantLevels: nil
  printToLog: nil
%
category: 'Instance Creation'
classmethod: GsSingleRefPathFinder
newForSearchObjects: anArray withMaxThreads: threads waitForLock: seconds pageBufSize: pages percentCpuLimit: percent
maxLimitSetDescendantObjs: maxChildObjs maxLimitSetDescendantLevels: maxChildLevels printToLog: logBoolean

"Creates and configures a new instance of the receiver.  The first argument must be an array of committed disk objects.
 The logBoolean argument must be a Boolean or nil.   All other arguments must be a SmallInteger or nil.  
 A nil argument indicates the default value for the argument is to be used.

 The pages argument must be nil or a SmallInteger which is a power of 2."
 
| result |
result := self new.
result
  maxThreads: (threads ifNil:[self defaultMaxThreads] ifNotNil:[threads]) ;
  lockWaitTime: (seconds ifNil:[self defaultLockWaitTime] ifNotNil:[seconds]) ;
  pageBufferSize: (pages ifNil:[self defaultPageBufferSize] ifNotNil:[pages]) ;
  percentCpuLimit: (percent ifNil:[self defaultPercentCpuLimit] ifNotNil:[percent]) ;
  maxLimitSetDescendantObjs: (maxChildObjs ifNil:[self defaultLimitSetDescendantsMaxObjs] ifNotNil:[maxChildObjs]) ;
  maxLimitSetDescendantLevels: (maxChildLevels ifNil:[self defaultLimitSetDescendantsMaxLevels] ifNotNil:[maxChildLevels]) ;
  printToLog: (logBoolean ifNil:[self defaultPrintToLog] ifNotNil:[logBoolean]) ;
  initializeForSearchObjects: anArray.
^ result
%

category: 'Logging'
classmethod: GsSingleRefPathFinder
printMessageToLog: aString

"Print a message to stdout and include a timestamp."
^ self printMessageToLog: aString includeTime: true
%
category: 'Logging'
classmethod: GsSingleRefPathFinder
printMessageToLog: aString includeTime: aBoolean

aBoolean ifTrue:[ | msg |
  msg := String withAll: '['.
  msg
    addAll: DateTime now asString;
    addAll: '] ';
    addAll: aString;
    add: Character lf.
    GsFile gciLogServer: msg]
ifFalse: [GsFile gciLogServer: aString].
^ self
%

category: 'Primitive'
classmethod: GsSingleRefPathFinder
_refPathSetupScanWithMaxThreads: maxThreads waitForLock: lockWaitTime 
    pageBufSize: aBufSize percentCpuLimit: percentCpu

"Sets up the current session for performing multi threaded scans of the 
 repository for finding reference paths to objects.  Performs an abort
 and leaves the session in a transaction.

 See _scanPomWithMaxThreads for definitions of maxThreads, waitTimeSeconds, 
 aBufSize, cpuPersent.
 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.         

 A GciHardBreak during this method will terminate the session."

<primitive: 527>
| maxInt |
maxInt := SmallInteger maximum32bitInteger .
maxThreads _validateClass: SmallInteger ; _validateMin: 1 max: maxInt .
lockWaitTime _validateClass: SmallInteger ; _validateMin: -1 max: maxInt .
aBufSize _validateClass: SmallInteger;  _validateIsPowerOf2 .
self _validatePercentage: percentCpu .
^ self _primitiveFailed: 
    #_refPathSetupScanWithMaxThreads:waitForLock:pageBufSize:percentCpuLimit:
    args: {  maxThreads . lockWaitTime . aBufSize . percentCpu }
%

category: 'Primitive'
classmethod: GsSingleRefPathFinder
scanForParents: anArrayOfBitmaps

"Scans the data pages found in the refPathSetup method to find the parent oops
 for each group of child oops.

 anArrayOfBitmaps is an array of non-empty GsBitmap instances containing child
 oops to scan for. The argument array and its GsBitmap elements are not 
 modified by this method.

 This method may be called multiple times.  The size of the anArrayOfBitmaps 
 argument may be different each time.  It is an error if any of the GsBitmap
 elements are empty.

 Returns an Array of GsBitmap instances containing the parent oops of the 
 objects contained in the argument bitmaps. The result array will be the
 same size and in the same order as the argument array."

<primitive: 597>
^ self _primitiveFailed: #scanForParents: args: { anArrayOfBitmaps }
%

category: 'Primitive'
classmethod: GsSingleRefPathFinder
_refPathDoScanForParents: searchObjs excludeParentRefs: excludeOops onlySearchObjs: aBoolean

"Scans the data pages found in the refPathSetup method to find the parent oops.
 If onlySearchObjs is true, then data for only the search objects is collected,
 otherwise data for all of the objects in the repository is collected and any 
 object may be queried and the oop of a single parent and whether it has 
 additional parents is returned. For each of the objects in the searchObjs this
 method saves all of the parent references. If a parent object is in the 
 excludeOops, it is not included in the parent set.

 The searchObjs and excludeOops arguments may be passed an Array or a GsBitmap.

 Elements of both the searchObjs and the excludeOops must be either committed 
 non-special objects, or  SmallIntegers;  SmallIntegers must be objectIds of 
 committed non-special objects and can be resolved as 
 (Object _objectForOop: anObjectId).

 When the scan is complete the parentsOf or findReferencePath methods can be 
 used to get the information captured for any object. "

<primitive: 894>
^ self _primitiveFailed: 
    #_refPathDoScanForParents:excludeParentRefs:onlySearchObjs:
    args: {  searchObjs . excludeOops . aBoolean }
%

category: 'Primitive'
classmethod: GsSingleRefPathFinder 
refPathCleanup

"Logs out the slave sessions and cleans up the saved state for the 
 multi threaded reference path scans."

System _zeroArgPrim: 179.
%

! ------------------- Instance methods for GsSingleRefPathFinder

category: 'Accessing'
method: GsSingleRefPathFinder
allSearches
^allSearches
%
category: 'Accessing'
method: GsSingleRefPathFinder
limitObjects
^limitObjects
%
category: 'Accessing'
method: GsSingleRefPathFinder
lockWaitTime
^lockWaitTime
%
category: 'Accessing'
method: GsSingleRefPathFinder
maxLimitSetDescendantLevels
^ maxLimitSetDescendantLevels
%
category: 'Accessing'
method: GsSingleRefPathFinder
maxLimitSetDescendantObjs
^ maxLimitSetDescendantObjs
%
category: 'Accessing'
method: GsSingleRefPathFinder
maxThreads
^maxThreads
%
category: 'Accessing'
method: GsSingleRefPathFinder
numPassesDone
^numPassesDone
%
category: 'Accessing'
method: GsSingleRefPathFinder
numSearchesActive
^numSearchesActive
%
category: 'Accessing'
method: GsSingleRefPathFinder
opStartTime
^opStartTime
%
category: 'Accessing'
method: GsSingleRefPathFinder
otScanDone
^otScanDone
%
category: 'Accessing'
method: GsSingleRefPathFinder
pageBufferSize
^pageBufferSize
%
category: 'Accessing'
method: GsSingleRefPathFinder
percentCpuLimit
^percentCpuLimit
%
category: 'Accessing'
method: GsSingleRefPathFinder
printToLog
^printToLog
%
category: 'Accessing'
method: GsSingleRefPathFinder
scanStartTime
^scanStartTime
%
category: 'Accessing'
method: GsSingleRefPathFinder
searchObjects
^searchObjects
%

category: 'Adding'
method: GsSingleRefPathFinder
addSearchObject: anObject

"Adds anObject to list of objects for which reference paths are to be found."

|newSearch|
newSearch := GsSingleRefPathFinderForObject newForSearchObject: anObject
                                            refPathFinder: self .
allSearches add: newSearch .
numSearchesActive := numSearchesActive + 1 .
^ self
%

category: 'Initialization'
method: GsSingleRefPathFinder
basicInit

allSearches := Array new .
numSearchesActive := 0 .
numPassesDone := 0 .
otScanDone := false .
^ self
%

category: 'Initialization'
method: GsSingleRefPathFinder
initializeForSearchObjects: anArray

self basicInit .
self validateSearchObjects: anArray .
searchObjects := anArray asGsBitmap .
anArray do:[:e| self addSearchObject: e] .
^ self
%

category: 'Limit Set'
method: GsSingleRefPathFinder
buildLimitSet

"Collect descendants of the default limit set and add them to the limitObjects 
 array.  Stop collecting descendants when the size of a generation exceeds 
 maxLimitSetDescendantObjs or the depth of the traversal exceeds 
 maxLimitSetDescendantLevels."

self printTimedOpStartMessageToLog: 'Starting build of limit set and descendants'.
limitObjects := Array with:
  (GsBitmap withAll: SystemRepository buildLimitSetForRefPathScan) .
self buildLimitSetDescendants .
self printTimedOpEndMessageToLog: 'Finished build of limit set and descendants'.
self logLimitSetSizes .
^ self
%
category: 'Limit Set'
method: GsSingleRefPathFinder
buildLimitSetDescendants

"Collect descendants of the default limit set and add them to the limitObjects 
 array.  Stop collecting descendants when the size of a generation exceeds 
 maxLimitSetDescendantObjs or the depth of the traversal exceeds 
 maxLimitSetDescendantLevels."

| parents alreadySeen done levels |
(maxLimitSetDescendantObjs <= 0 or:[maxLimitSetDescendantLevels <= 0])
  ifTrue:[ ^ self ]. "Limit set descendants is disabled"

parents := self defaultLimitSet .
alreadySeen := parents copy.
done := false.
levels := 0.
[done] whileFalse:[ |children|
  levels := levels + 1.
  children := parents primReferencedObjects .
  children removeAll: alreadySeen.
  children isEmpty
    ifTrue:[ done := true]
    ifFalse:[
      limitObjects add: children.
      alreadySeen addAll: children.		
      done := (children size > maxLimitSetDescendantObjs) or:[ levels >= maxLimitSetDescendantLevels ].
      parents := children.
    ].
].
alreadySeen removeAll.
^ self
%
category: 'Limit Set'
method: GsSingleRefPathFinder
defaultLimitSet

"Answer the limit set returned by SystemRepository buildLimitSetForRefPathScan,
 which is the first element in the limitObjects array."

^ limitObjects first
%
category: 'Limit Set'
method: GsSingleRefPathFinder
handleLimitSetDescendantsWithSearchObject: anObj

| aSearch|
aSearch := self searchForObject: anObj.
aSearch handleLimitSetDescendantsWithSearchObject.
^ self
%
category: 'Limit Set'
method: GsSingleRefPathFinder
handleSearchObjectsInLimitSetDescendants
| intersection |

intersection := self searchObjectsInLimitSetDescendants.
intersection do:[:eachObj|
  self handleLimitSetDescendantsWithSearchObject: eachObj ].
intersection removeAll .
^ self
%
category: 'Limit Set'
method: GsSingleRefPathFinder
limitSetDescendants
"Answer a GsBitmap containing all descendants of the base limit set, but not 
 the base limit set itself."

| result |
result := GsBitmap new.
"First element is the base limit set.  Skip that one."
2 to: limitObjects size do: [:n | result addAll: (limitObjects at: n)].
^result
%
category: 'Limit Set'
method: GsSingleRefPathFinder
searchObjectsInLimitSetDescendants
"Answer a GsBitmap containing any search objects that are present in the 
 descendants of the limit set."

^ self limitSetDescendants * searchObjects
%

category: 'Logging'
method: GsSingleRefPathFinder
logEndOfScan

printToLog
ifTrue: 
[| msg |
msg := String withAll: 'Finished scan '.
msg addAll: numPassesDone asString.
self printTimedOpEndMessageToLog: msg]
%
category: 'Logging'
method: GsSingleRefPathFinder
logLimitSetSizes

printToLog
  ifTrue:[ | ws |
  ws := AppendStream on: String new.
  ws
    nextPutAll: 'Sizes of limit set descendants by generation:';
    lf.
  1 to: limitObjects size do:[:n |
    ws tab;
      nextPutAll: n asString;
      space: 2;
      nextPutAll: (limitObjects at: n) size asString;
      lf].
  self class printMessageToLog: ws contents includeTime: false].
^self
%
category: 'Logging'
method: GsSingleRefPathFinder
logStartOfScan

printToLog
ifTrue: 
[| msg numCompleted numSearchObjs |
numSearchObjs := searchObjects size .
numCompleted := numSearchObjs - numSearchesActive .
msg := String withAll: 'Starting scan '.
msg addAll: numPassesDone asString ;
    addAll: '. Search object summary: ';
    addAll: numSearchObjs asString ;
    addAll: ' total,  ' ;
    addAll: numSearchesActive asString ;
    addAll: ' active,  ';
    addAll: numCompleted asString ;
    addAll: ' completed'.
self printTimedOpStartMessageToLog: msg]
%
category: 'Logging'
method: GsSingleRefPathFinder
logStartup

printToLog
  ifTrue:[
    self printMessageToLog:
      'Starting find one reference path scan for the following objects:';
    printSearchObjectsToLog]
%
category: 'Logging'
method: GsSingleRefPathFinder
printMessageToLog: aString

printToLog ifTrue: [self class printMessageToLog: aString].
^ self
%
category: 'Logging'
method: GsSingleRefPathFinder
printSearchObjectsToLog

printToLog ifTrue:[| oopList classNameList msg |
  msg := String new.
  oopList := allSearches collect: [:e | e searchOop asOop asString].
  classNameList := allSearches
  collect: [:e | e searchOop class name asString].
  1 to: oopList size do:[:n |
    msg
    addAll: '   ';
    addAll: n asString;
    addAll: '  ';
    addAll: (oopList at: n);
    addAll: ' (';
    addAll: (classNameList at: n);
    addAll: ')';
    add: Character lf].
  GsFile gciLogServer: msg]
%
category: 'Logging'
method: GsSingleRefPathFinder
printTimedOpEndMessageToLog: aString


printToLog ifTrue: [ |seconds msg|
  seconds := (System timeGmt2005 - opStartTime) asString .
  msg := String withAll: aString .
  msg addAll: ' in '; addAll: seconds ; addAll: ' seconds'.
  self class printMessageToLog: msg].
^ self
%
category: 'Logging'
method: GsSingleRefPathFinder
printTimedOpStartMessageToLog: aString

printToLog ifTrue: [
  opStartTime := System timeGmt2005 .
  self class printMessageToLog: aString].
^ self
%
category: 'Logging'
method: GsSingleRefPathFinder
printTimestampToLog

printToLog ifTrue:[ | msg |
  msg := DateTime now asString.
  msg add: Character lf.
  GsFile gciLogServer: msg]
%
category: 'Results'
method: GsSingleRefPathFinder
scanAndReport
"Run the scan, build the results, and collect the path strings
 for each result.  Return a string."

| str |
str := String new.
self runScan buildResultObjects
   do:[:e| str add: e resultString].
^str
%
category: 'Results'
method: GsSingleRefPathFinder
buildResultObjects
"Returns an Array of GsSingleRefPathResult objects"

^ allSearches collect:[:e| e buildResultObject]
%
category: 'Scanning'
method: GsSingleRefPathFinder
completedOneSearch
numSearchesActive := numSearchesActive - 1.
%
category: 'Scanning'
method: GsSingleRefPathFinder
runOneScan

| searches childrenBitmaps parentBitmaps |
self scanObjectTable .
searches := self allActiveSearches.
searches do:[:e| e updateSearchOopsUnion].
childrenBitmaps := searches collect:[:e| e childrenToFind ].
numPassesDone := numPassesDone + 1.
self logStartOfScan.
parentBitmaps := GsSingleRefPathFinder scanForParents: childrenBitmaps.
self logEndOfScan.
1 to: searches size do:[:n|
  (searches at: n) processResultsOfScan: (parentBitmaps at: n)
].
%
category: 'Scanning'
method: GsSingleRefPathFinder
runScan

self logStartup.
self buildLimitSet .
self validateSearchObjectsAreNotLimitObjects .
self handleSearchObjectsInLimitSetDescendants .
scanStartTime := System timeGmt2005 .
[numSearchesActive > 0] whileTrue:[ self runOneScan].
GsSingleRefPathFinder refPathCleanup
%
category: 'Scanning'
method: GsSingleRefPathFinder
scanObjectTable

otScanDone ifFalse:[	
  self printTimedOpStartMessageToLog: 'Starting object table scan'.
  GsSingleRefPathFinder
    _refPathSetupScanWithMaxThreads: maxThreads
    waitForLock: lockWaitTime
    pageBufSize: pageBufferSize
    percentCpuLimit: percentCpuLimit.
  self printTimedOpEndMessageToLog: 'Finished object table scan'.
  otScanDone := true].
^self
%

category: 'Searching'
method: GsSingleRefPathFinder
allActiveSearches
"Answer an array of GsSingleRefPathFinderForObject that are not finished."
^ allSearches select:[:e| e completed not]
%
category: 'Searching'
method: GsSingleRefPathFinder
searchForObject: anObject
"Find the GsSingleRefPathFinderForObject instance for anObject."
^ allSearches detect:[:e| e searchOop == anObject ]
%

category: 'Updating'
method: GsSingleRefPathFinder
allSearches: newValue
allSearches := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
limitObjects: newValue
limitObjects := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
lockWaitTime: newValue
lockWaitTime := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
maxLimitSetDescendantLevels: newValue
maxLimitSetDescendantLevels := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
maxLimitSetDescendantObjs: newValue
maxLimitSetDescendantObjs := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
maxThreads: newValue
maxThreads := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
numPassesDone: newValue
numPassesDone := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
numSearchesActive: newValue
numSearchesActive := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
opStartTime: newValue
opStartTime := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
otScanDone: newValue
otScanDone := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
pageBufferSize: newValue

"Value must be a power of 2"
newValue _validateIsPowerOf2 .
pageBufferSize := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
percentCpuLimit: newValue
percentCpuLimit := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
printToLog: newValue
printToLog := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
scanStartTime: newValue
scanStartTime := newValue
%
category: 'Updating'
method: GsSingleRefPathFinder
searchObjects: newValue
searchObjects := newValue
%

category: 'Validation'
method: GsSingleRefPathFinder
validateSearchObjectsAreNotLimitObjects
"Checks to ensure that none of the search objects appear in the default limit 
 set.  Raises an exception if any objects satisfy that condition."

| searchOopsInLimitSet|
searchOopsInLimitSet := self defaultLimitSet * searchObjects .
searchOopsInLimitSet isEmpty
  ifFalse:[
    ArgumentError signal: 'One or more search objects also appears in the limit set'].
^ self
%

category: 'Validation'
method: GsSingleRefPathFinder
validateSearchObjects: anArray

"Check input array to ensure all objects are committed disk objects.
 Special objects and uncommitted objects are not allowed."
 
1 to: anArray size do: [:i| | obj |
  obj := anArray at: i .
  (obj isSpecial or: [ obj isCommitted not ]) ifTrue: 
    [ obj _error: #rtErrSpecialOrNotCommitted ]
].
^ self
%
