!========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: GsPipe.gs 40333 2016-10-04 18:17:04Z otisa $
!
! definitions for GsPipeElement, GsPipe
!
!========================================================================


!=========================================================================
! Superclass Hierarchy:
!   GsPipeElement, Object.
!
!=========================================================================

! ------------------- Class definition for GsPipeElement
expectvalue %String
run
| cls |
cls := Globals at: #GsPipeElement otherwise: nil.
^ cls == nil ifTrue:[
    Object subclass: 'GsPipeElement'
      instVarNames: #( next value  )
      classVars: #()
      classInstVars: #()
      poolDictionaries: #()
      inDictionary: Globals
      options: #().
          'Created class GsPipeElement'
   ] 
   ifFalse:[ 'GsPipeElement class already exists' ].
%

removeallmethods GsPipeElement
removeallclassmethods GsPipeElement

category: 'For Documentation Installation only'
classmethod: GsPipeElement
installDocumentation

self comment:
'GsPipeElement describes an individual entry in a GsPipe.

--- instVars
  next    - references the next element in the list for this pipe
  value   - references the value of this element
'
%

! ------------------- Class methods for GsPipeElement

category: 'Instance Creation'
classmethod: GsPipeElement
newWithNext: nextElement value: aValue

"Returns a new GsPipeElement instance with the nextElement and aValue"

| element |
element := super new next: nextElement.
element value: aValue.
^element
%

! ------------------- Instance methods for GsPipeElement

category: 'Accessing'
method: GsPipeElement
next

"Returns the current value of the next instance variable"
^ next
%

category: 'Accessing'
method: GsPipeElement
value

"Returns the current value of the GsPipeElement"
^ value
%

category: 'Updating'
method: GsPipeElement
next: anElement

next := anElement
%

category: 'Updating'
method: GsPipeElement
value: anObj

value := anObj
%

category: 'Reduced Conflict Support'
method: GsPipeElement
_abortAndReplay: conflictObjects
 
"Abort the receiver, no replay necessary."
 
" refresh the state of the receiver "
self _selectiveAbort.
 
^ true
%

!=========================================================================
! Superclass Hierarchy:
!   GsPipe, Collection, Object.
!
!=========================================================================

expectvalue %String

! ------------------- Class definition for GsPipe
expectvalue %String
run
| cls |
cls := Globals at: #GsPipe otherwise: nil.
^ cls == nil ifTrue:[
    Collection subclass: 'GsPipe'
      instVarNames: #( head  tail )
      classVars: #()
      classInstVars: #()
      poolDictionaries: #()
      inDictionary: Globals
      options: #().
          'Created class GsPipe'
   ] 
   ifFalse:[ 'GsPipe class already exists' ].
%

removeallmethods GsPipe
removeallclassmethods GsPipe

! ------------------- Class methods for GsPipe
category: 'For Documentation Installation only'
classmethod: GsPipe
installDocumentation

self comment:
'A GsPipe is an implementation of a FIFO queue that has no concurrency
conflicts when used in an environment with a single producer (a user who 
adds elements to the pipe) and a single consumer (a user who may remove 
items from the pipe).  In this environment, the producer and the consumer 
are guaranteed not to conflict with each other.

This implemementation is based on US Patent Number 6,360,219 
         "Object queues with concurrent updating".

The "head" and "tail" objects described in the patent are implemented as a
GsPipeElement where the next field references another GsPipeElement and the 
value is a SmallInteger.  For the head GsPipe element, the value indicates 
the number of removes performed and in the tail the number of additions 
performed.  

When the GsPipe is empty the head and tail next fields both refer to the same 
GsPipeElement.  A removal, if the GsPipe is not empty only modifies the head
GsPipeElement.  An add creates a new GsPipeElement object with the value specified
and links it into the list by storing the new GsPipeElement into the next field
of the GsPipeElement referenced by the tail and then updates the 
tail to reference the new element.  Since adds and removes never update the
same objects, there can be NO concurrency conflicts with these operations.

Multiple concurrent producers (sessions adding to the GsPipe) will experience
concurrency conflicts.  Users needing to have concurrent producers run without
conflict should consider using an RcPipe or an RcQueue.

Values in the GsPipe are always removed in the order that they were committed.
'
%

category: 'Instance Creation'
classmethod: GsPipe
new

"Returns a new GsPipe."

^super new _initialize
%

! ------------------- Instance methods for GsPipe

category: 'Accessing'
method: GsPipe
head

^head
%

category: 'Accessing'
method: GsPipe
tail

^tail
%

category: 'Accessing'
method: GsPipe
size

"Returns the number of valid entries in the GsPipe."

^(tail value) - (head value)
%

category: 'Adding'
method: GsPipe
add: aValue

"Adds aValue to the GsPipe and returns aValue."

| element |
element := GsPipeElement newWithNext: nil value: aValue.

tail next next: element. 
tail next: element.
tail value: (tail value + 1).
^ aValue
%

category: 'Converting'
method: GsPipe
asArray

"Returns an Array with the contents of the receiver."

| result element |

result := Array new.
element := head next next.
[element == nil] whileFalse: [
  result add: (element value).
  element := element next
].
^ result
%

category: 'Copying'
method: GsPipe
copy

"Returns a new instance of GsPipe with the elements that are in the receiver."

| contents |

contents := self asArray.
^ (GsPipe withAll: contents)
%

category: 'Enumerating'
method: GsPipe
do: aBlock

"Evaluates aBlock with each of the current elements of the GsPipe as
 the argument. The argument aBlock must be a one-argument block.  This
 method traverses the pipe elements in order.  Returns the receiver."

| element |

element := head next next.
[element == nil] whileFalse: [
  aBlock value: (element value).
  element := element next
].
^ self
%

category: 'Private'
method: GsPipe
_initialize

"initialize the sub-components"
| element |

element := GsPipeElement newWithNext: nil value: nil.
head := GsPipeElement newWithNext: element value: 0.
tail := GsPipeElement newWithNext: element value: 0.
%

category: 'Removing'
method: GsPipe
remove

"Removes the first element from the receiver and returns that element.
 If the receiver is empty, returns nil."

| element aValue | 
self size == 0 ifTrue: [ ^ nil ].

element := head next next.
aValue := element value.
head next: element.
head value: (head value + 1).
^aValue
%

category: 'Removing'
method: GsPipe
peek

"Returns the leading element from the receiver without removing it.
 If the receiver is empty, returns nil."

| element | 
self size == 0 ifTrue: [ ^ nil ].

element := head next next.
^ element value
%

category: 'Removing'
method: GsPipe
removeAll

"Removes all entries from the GsPipe, and returns an Array that contains
 those entries, in order."

| anArray |

anArray := self asArray.
head next: (tail next).
head value: (tail value).
^ anArray
%

category: 'Testing'
method: GsPipe
isEmpty

"Returns true if the queue is empty, and false otherwise."

^self size == 0
%

category: 'Updating'
method: GsPipe
objectSecurityPolicy: anObjectSecurityPolicy

"Assigns the receiver and subcomponents to the given security policy."

| element |
super objectSecurityPolicy: anObjectSecurityPolicy.
head objectSecurityPolicy: anObjectSecurityPolicy.
tail objectSecurityPolicy: anObjectSecurityPolicy.

element := head next.
[element == nil] whileFalse: [
  element objectSecurityPolicy: anObjectSecurityPolicy.
  element := element next
].
^ self
%

