! ========================================================================
! GsNetworkResourceString.gs
!
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
! ========================================================================


expectvalue %String
doit
(Object subclass: 'GsNetworkResourceString'
	instVarNames: #( protocol node authorization encrypted netldi dir log resource body)
	classVars: #()
	classInstVars: #()
	poolDictionaries: #()
	inDictionary: Globals
	options: #()) definition
%

! ------------------- Class comment for GsNetworkResourceString
doit
GsNetworkResourceString comment: 
'This class is part of the External Sessions implementation and 
represents a Network Resource String (NRS). It is typically used 
to construct an NRS from parts, helping ensure correct syntax.

Class methods provide a variety of "sample" ways to create instances.
Instance methods provide getters and setters for the various NRS attributes,
as well as formatting methods for providing a valid string representation.
'.
true
%

doit
GsNetworkResourceString category: 'External Sessions'.
true
%

! Remove existing behavior from GsNetworkResourceString
doit
GsNetworkResourceString removeAllMethods.
GsNetworkResourceString class removeAllMethods.
true
%
! ------------------- Class methods for GsNetworkResourceString
category: 'Accessing'
classmethod: GsNetworkResourceString
currentGemHostName

	^System gemVersionReport at: #nodeName.
%
category: 'Accessing'
classmethod: GsNetworkResourceString
currentStoneHostName

  GsSession currentSession isSolo ifTrue:[ ^ 'localhost' ].

  ^System stoneVersionReport at: #nodeName
%
category: 'Accessing'
classmethod: GsNetworkResourceString 
currentStoneName
	"The name of the current running stone, excluding host and other NRS information"

	^(System stoneName subStrings: $!) last
%
category: 'Defaults'
classmethod: GsNetworkResourceString
defaultGemService
	^'gemnetobject'
%
! fixed 44708
category: 'Defaults'
classmethod: GsNetworkResourceString
defaultNetLDIName
	^'gs64ldi'
%
category: 'Defaults'
classmethod: GsNetworkResourceString
defaultStoneName
	^'gs64stone'
%
category: 'Instance Creation'
classmethod: GsNetworkResourceString
new

	^super new
		initialize;
		yourself.
%
category: 'Instance Creation'
classmethod: GsNetworkResourceString
defaultGemNRSFromCurrent
	"Return a new gem NRS with the gem's host being the same as the current login.
      Use default gem service, and default netldi, or as set by environment variable GEMSTONE_NRS_ALL.
	Ignore any other parameters in the NRS of the current login"

	| netldiEnv |
	netldiEnv := self systemNetLDIName.
	^ self new
		node: self currentGemHostName;
		netldi: netldiEnv;
		resource: 'task';
		body: self defaultGemService;
    yourself 
%
category: 'Instance Creation'
classmethod: GsNetworkResourceString
defaultStoneNRSFromCurrent
	"Return a new stone NRS with the stone name and stone's host being the same as the current login.
	Ignore any other login parameters in the NRS of the current login"

	| stoneNameParts resourceParts protocolParts body resource node res |
	stoneNameParts := System stoneName subStrings: $!.
  stoneNameParts size >= 3 ifTrue:[
	  body := stoneNameParts at: 3.
	  resourceParts := (stoneNameParts at: 2) subStrings: $#.
	  resource := resourceParts last.
	  protocolParts := (resourceParts at: 1) subStrings: $@.
  ] ifFalse:[
    body := stoneNameParts at: 1.
    resource := 'server' .
  ].
	protocolParts size < 2 ifTrue:["handle default stone name of the form   '!#server!gs64stone' "
		node := self currentGemHostName	"if stone does not have host specified, uses gem host"
  ] ifFalse:[
    node := protocolParts at: 2
  ].
	(res := self new)
		node: node;
		resource: resource;
		body: body.
  ^ res
%
category: 'Instance Creation'
classmethod: GsNetworkResourceString
gemNRS

"Return a new basic gem NRS, which is only the gem service"

	^self new
		body: self defaultGemService;
		yourself.
%
category: 'Instance Creation'
classmethod: GsNetworkResourceString
gemNRSForNetLDI: nameOrPort

"Return a new gem NRS for a gem with the given netldi."

	^self new
		netldi: nameOrPort;
		body: self defaultGemService;
		yourself.
%
category: 'Instance Creation'
classmethod: GsNetworkResourceString
gemNRSForNetLDI: nameOrPort onHost: gemhostname

"Return a new gem NRS for a gem that will run on the given host, using the given netldi"

	^self new
		node: gemhostname;
		netldi: nameOrPort;
		body: self defaultGemService;
		yourself.
%
category: 'Instance Creation'
classmethod: GsNetworkResourceString
gemNRSForNetLDI: nameOrPort onHost: gemhostname gemService: customGemService

"Return a new gem NRS for a gem that will run on the given host, using the given netldi,
 and specifying the customGemService.  Options for <customGemService> are 
 gemnetobject, gemnetdebug, or a custom script that starts up a gem."

	^self new
		node: gemhostname;
		netldi: nameOrPort;
		body: customGemService;
		yourself.
%
category: 'Instance Creation'
classmethod: GsNetworkResourceString
stoneNRS
	"Return a new stone NRS for stone with default name."

	^self new
		body: self defaultStoneName;
		yourself
%
category: 'Instance Creation'
classmethod: GsNetworkResourceString
stoneNRSForStoneName: aStoneName
	"Return a new stone NRS with stone with the given name."

	^self new
		body: aStoneName;
		yourself
%
category: 'Instance Creation'
classmethod: GsNetworkResourceString
stoneNRSForStoneName: aStoneName onHost: stoneHostName
	"Return a new stone NRS with stone with the given name."

	^self new
		node: stoneHostName;
		body: aStoneName;
		yourself
%
category: 'Instance Creation'
classmethod: GsNetworkResourceString
systemNetLDIName

	| list ev |
  ev := System gemEnvironmentVariable: 'GEMSTONE_NRS_ALL' .	
  ev ifNil:[ ^ self defaultNetLDIName ].
	list := (ev subStrings: $#) collect: [:nrs | nrs subStrings: $: ].
	^(list detect: [:nrs | nrs first matchPattern: #( 'netldi' ) ] 
         ifNone: [ { self defaultNetLDIName } ]) last
%
! ------------------- Instance methods for GsNetworkResourceString
category: 'Accessing'
method: GsNetworkResourceString
authorization
	"username [@password]
	 A valid user on the target network (default is the current user). 
	 A valid password is needed only if the resource type requires authentication.
	 If no authentication information is specified, the system will try to get it from the .netrc file
		(this type of authorization is the default)."

	^authorization
%
category: 'Accessing'
method: GsNetworkResourceString
body
	"The <body> is interpreted according to the context established by the <resource>.
	 No extended identifier expansion is done in the <body> and no special escapes are needed."

	^body.
%
category: 'Accessing'
method: GsNetworkResourceString
dir
	"<dir> sets the default directory of the network resource.
	It has no effect if the resource already exists.
	If a directory is not set, the pattern %H is used.

	The string can contain patterns that are expanded in the context of the created resource.
	The following patterns are supported:
		%H	home directory
		%M	machine's network node name
		%N	executable's base name
		%P	process pid
		%U	user name
		%%	%"

	^dir
%
category: 'Accessing'
method: GsNetworkResourceString
encrypted
	"Answer whether the authorization credentials are encrypted."

	^encrypted
%
category: 'Accessing'
method: GsNetworkResourceString
log
	"<log> sets the name of the log file of the network resource.
	It has no effect if the resource already exists.
	If the log name is a relative path, it is relative to the working directory.
	If a log name is not set, the pattern '%N%P%M.log' is used.

	The string can contain patterns that are expanded in the context of the created resource.
	The following patterns are supported:
		%H	home directory
		%M	machine's network node name
		%N	executable's base name
		%P	process pid
		%U	user name
		%%	%"

	^log
%
category: 'Accessing'
method: GsNetworkResourceString
netldi
	"Use the named GsNetLDI to service the request."

	^netldi
%
category: 'Accessing'
method: GsNetworkResourceString
node
	"Specifies the host on which the resource exists.
	If no node is specified, the current machine's network node name is used.
	The identifier may be an Internet-style numeric address."

	^node
%
category: 'Accessing'
method: GsNetworkResourceString
protocol
	"Only tcp is supported"

	^ 'tcp'
%
category: 'Accessing'
method: GsNetworkResourceString
resource
	"Identifies the intended purpose of the <body>. 
	An NRS can contain only one resource modifier.
	The default resource modifier is context sensitive.
	For instance, if the system expects an NRS for a database file
	(say, in the context of copydbf), then the default is #dbf.

	'server' directs the GsNetLDI to search for the network address of a server,
	such as a GsStone or another GsNetLDI. If successful, it returns the address.
	The <body> is a network server name. A successful lookup means only
	that the service has been defined; it does not indicate whether the service
	is currently running. A new process will not be started.
	(Authorization is needed only if the GsNetLDI is on a remote node
	and is running in secure mode.)

	'task' starts a new Gem. The <body> is a GsNetLDI service name
	(such as 'gemnetobject'), followed by arguments to the command line.
	The GsNetLDI creates the named service by looking first for an entry in
	$GEMSTONE/bin/services.dat, and then in the user's home directory
	for an executable having that name. 
	The GsNetLDI returns the network address of the service.
	(Authorization is needed to create a new process unless the GsNetLDI
	is in guest mode.)
	The 'task' resource modifier is also used internally to create page servers.

	'dbf' is used to access a database file. The <body> is the file spec 
	of a GemStone dataabse file. 
	The GsNetLDI creates a page server on the given node to access
	the database and returns the network address of the page server.
	(Authorization is needed unless the GsNetLDI is in guest mode.)

	'spawn' is used internally to start the garbage-collection Gem process.

	'monitor' is used internally to start up a shared page cache monitor.

	'file' means the nrs-body is the file spec of a file on the given host (not
	currently implemented)."

	^resource
%
category: 'Formatting'
method: GsNetworkResourceString
asString

	^self printString
%
! fixed 47222
category: 'Formatting'
method: GsNetworkResourceString
printOn: aStream

	| bang |
	bang := self hasNrsHeader ifTrue: ['!'] ifFalse: [''].
	aStream nextPutAll: bang.
	self
		printSegment: node designated: '@' on: aStream;
		printSegment: authorization
			designated: (encrypted ifTrue: ['#encrypted:'] ifFalse: ['#auth:'])
			on: aStream;
		printSegment: netldi designated: '#netldi:' on: aStream;
		printSegment: dir designated: '#dir:' on: aStream;
		printSegment: log designated: '#log:' on: aStream;
		printSegment: resource designated: '#' on: aStream.
	aStream
		nextPutAll: bang;
		nextPutAll: body
%
category: 'Private'
method: GsNetworkResourceString
printSegment: aString designated: designationString on: aStream

	aString isEmpty ifTrue: [^self].
	aStream
		nextPutAll: designationString;
		nextPutAll: aString
%
category: 'Private'
method: GsNetworkResourceString
initialize

	protocol := ''.
	node := ''.
	authorization := ''.
	encrypted := false.
	netldi := ''.
	dir := ''.
	log := ''.
	resource := ''.
	body := ''.
%
category: 'Private'
method: GsNetworkResourceString
hasNrsHeader

	^(protocol , node , authorization , netldi , dir , log , resource)
		notEmpty
%
category: 'Updating'
method: GsNetworkResourceString
authorization: aString
	"username [@password]
	 A valid user on the target network (default is the current user). 
	 A valid password is needed only if the resource type requires authentication.
	 If no authentication information is specified, the system will try to get it from the .netrc file
		(this type of authorization is the default)."

	(aString isKindOf: String) ifFalse: [self error: 'parameter must be a String'].
	authorization := aString.
%
category: 'Updating'
method: GsNetworkResourceString
beEncrypted

	encrypted := true.
%
category: 'Updating'
method: GsNetworkResourceString
beNotEncrypted

	encrypted := false.
%
category: 'Updating'
method: GsNetworkResourceString
body: aString
	"The <body> is interpreted according to the context established by the <resource>.
	 No extended identifier expansion is done in the body and no special escapes are needed."

	(aString isKindOf: String) ifFalse: [self error: 'parameter must be a String'].
	body := aString.
%
category: 'Updating'
method: GsNetworkResourceString
dir: aString
	"<dir> sets the default directory of the network resource.
	It has no effect if the resource already exists.
	If a directory is not set, the pattern %H is used.

	The string can contain patterns that are expanded in the context of the created resource.
	The following patterns are supported:
		%H	home directory
		%M	machine's network node name
		%N	executable's base name
		%P	process pid
		%U	user name
		%%	%"

	(aString isKindOf: String) ifFalse: [self error: 'parameter must be a String'].
	dir := aString.
%
category: 'Updating'
method: GsNetworkResourceString
log: aString
	"<log> sets the name of the log file of the network resource.
	It has no effect if the resource already exists.
	If the log name is a relative path, it is relative to the working directory.
	If a log name is not set, the pattern '%N%P%M.log' is used.

	The string can contain patterns that are expanded in the context of the created resource.
	The following patterns are supported:
		%H	home directory
		%M	machine's network node name
		%N	executable's base name
		%P	process pid
		%U	user name
		%%	%"


	(aString isKindOf: String) ifFalse: [self error: 'parameter must be a String'].
	log := aString.
%
category: 'Updating'
method: GsNetworkResourceString
netldi: aString
	"Use the named GsNetLDI to service the request."

	(aString isKindOf: String) ifFalse: [self error: 'parameter must be a String'].
	netldi := aString.
%
category: 'Updating'
method: GsNetworkResourceString
node: aString
	"Specifies the host on which the resource exists.
	If no node is specified, the current machine's network node name is used.
	The identifier may be an Internet-style numeric address."

	(aString isKindOf: String) ifFalse: [self error: 'parameter must be a String'].
	node := aString.
%
category: 'Updating'
method: GsNetworkResourceString
protocol: aString
   "The only supported protocol is TCP"

	(aString isKindOf: String)
		ifFalse: [self error: 'parameter must be a String'].
	((aString isEquivalent: 'TCP') or:[ aString = '' 
	   or:[ aString isEquivalent: 'DEFAULT']])
		ifFalse: [self error: 'Invalid protocol: ' , aString].
	protocol := 'tcp'.
%
category: 'Updating'
method: GsNetworkResourceString
resource: aString
	"Identifies the intended purpose of the <body>. 
	An NRS can contain only one resource modifier.
	The default resource modifier is context sensitive.
	For instance, if the system expects an NRS for a database file
	(say, in the context of copydbf), then the default is #dbf.

	'server' directs the GsNetLDI to search for the network address of a server,
	such as a GsStone or another GsNetLDI. If successful, it returns the address.
	The <body> is a network server name. A successful lookup means only
	that the service has been defined; it does not indicate whether the service
	is currently running. A new process will not be started.
	(Authorization is needed only if the GsNetLDI is on a remote node
	and is running in secure mode.)

	'task' starts a new Gem. The <body> is a GsNetLDI service name
	(such as 'gemnetobject'), followed by arguments to the command line.
	The GsNetLDI creates the named service by looking first for an entry in
	$GEMSTONE/bin/services.dat, and then in the user's home directory
	for an executable having that name. 
	The GsNetLDI returns the network address of the service.
	(Authorization is needed to create a new process unless the GsNetLDI
	is in guest mode.)
	The 'task' resource modifier is also used internally to create page servers.

	'dbf' is used to access a database file. The <body> is the file spec 
	of a GemStone dataabse file. 
	The GsNetLDI creates a page server on the given node to access
	the database and returns the network address of the page server.
	(Authorization is needed unless the GsNetLDI is in guest mode.)

	'spawn' is used internally to start the garbage-collection Gem process.

	'monitor' is used internally to start up a shared page cache monitor.

	'file' means the nrs-body is the file spec of a file on the given host (not
	currently implemented)."

	(aString isKindOf: String) ifFalse: [self error: 'parameter must be a String'].
	(#('server' 'task' 'dbf' 'spawn' 'monitor' 'file') includes: aString) ifFalse: [self error: 'Invalid resource: ', aString].
	resource := aString.
%
category: 'Updating'
method: GsNetworkResourceString
temporaryObjectCacheSize: sizeKB

  self body: (self body , ' -T ', sizeKB asString)
%
category: 'Updating'
method: GsNetworkResourceString
nativeCodeConfig: anInt
 "Pass in the setting for GEM_NATIVE_CODE_ENABLED to use for the 
  session, may be 0, 1, or 2."

  self body: (self body , ' -N ', anInt asString)
%

