!=========================================================================
! File GsHostProcess.gs
!
! Copyright (C) GemTalk Systems 2013-2020.  All Rights Reserved.
!
!=========================================================================

expectvalue %String
run
Globals at: #GsHostProcess ifAbsent:[
  Object _newKernelSubclass: 'GsHostProcess'
    subclassOf: Object
    instVarNames: #( cmd in out err childPid childStatus )
             " String, 3xGsSocket  , SmallInteger, SmallInteger "
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
    options:  #( #instancesNonPersistent )
    reservedOop: 1213 .
  ^ ' created GsHostProcess'
].
^ 'exists'
%
run
| o |
(o := GsHostProcess asOop) == 155137 ifFalse:[ ^ o ].
^ true
%
set class GsHostProcess
removeallmethods
removeallclassmethods

category: 'Documentation'
classmethod:
comment
^ 'GsHostProcess represents a forked child process whose stdin, stdout, and
stderr are accessable via instances of GsSocket.

Lookup in the PATH environment variable is not performed. The argument
to fork:  must specify a complete path to an executable or script.
Instances may not be committed.

The GsSocket instance variables stdin, stdout, and stderr hold GsSockets
representing  the parent process''s ends of the pipe.   The forked child
process has the other end of the pipe.  The child accepts input on the
stdin and puts output on  stdout and stderr.  For the parent end, stdin
is write-only and  stdout and stderr are read only.

When an instance is garbage collected, and the child has not yet been
reaped,  the child process represented by that instance is killed with
kill -TERM, and waitpid is called to reap the child.

For stdout or stderr of a GsHostProcess, if the child writes data while
it is running then the corresponding GsSocket can read that data. Use
GsSocket>>readWillNotBlock or GsSocket>>readWillNotBlockWithin:
to determine whether data is ready to read.

GsSocket>>read:into:startingAt: will yield to other GsProcess(s) as for
reading from a GsSocket represending an AF_INET or AF_INET6 socket .
If the child has exited and there is no more data to read, this read
method will wait forever.

When GsHostProcess>>status returns non-nil, then any data not yet read
from stdout/stderr is still available to read. The parent end of the
pipe is not closed until the corresponding GsSocket is garbage collected
or explicitly sent GsSocket>>close

If the child tries to read from stdin it will block until data is
written by the parent to the stdin of the GsHostProcess. If the child 
tries to write a lot of data to stdout or stderr, the child may block
until the parent reads enough data from the respective GsSocket to 
unblock the pipe. 

For example,
  GsHostProcess fork: ''$GEMSTONE/bin/topaz -r''
will fork a topaz process which will do blocking reads on stdin and
execute each command that is written to the stdin of the GsHostProcess 
instance.
'
%

category: 'Instance creation'
classmethod: 
basicNew

"disallowed"

self shouldNotImplement: #basicNew
%

classmethod:
fork: commandLineString
" Create a new instance of the receiver, fork a child process,
 and exec the commandLineString in the child.
 Utf16 command lines are not yet supported by the _fork: primitive.
 Lookup in the PATH environment variable is not performed, 
 the commandLineString must specify a complete path to an executable 
 or script to be exec'ed .

 Signals an error if the privilege NoPerformOnServer is true.
 If running in a topaz -l process and (GsFile stdin isTerminal == true)
 The child process will ignore SIGINT.
 "

  | inst |
  (inst := self _basicNew) _fork: commandLineString .
  ^ inst
%
classmethod:
execute: commandLineString input: stdinString
 "Executes commandLineString in a child process, 
  writing stdinString to the stdin of the child.
  stdinString may be nil, in which case nothing is written to stdin of child
  If child returns non-zero exit status, signals an Error using
  contents of stderr from the child.  Otherwise returns a String
  containing stdout from the child. 
  If child produces exit status
  zero and non-empty stderr the result is prefixed with
  'WARNING: <contents of stderr>' . 
  Lookup in the PATH environment variable is not performed, 
  the commandLineString must specify a complete path to an executable 
  or script.
  Signals an error if the privilege NoPerformOnServer is true."

  | arr out err result |
  arr := self _execute: commandLineString input: stdinString .
  out := arr at: 1 .
  err := arr at: 2 .
  err size > 0 ifTrue:[
    (result := 'WARNING: <' copy) addAll: err asString; add: '> ' ; lf;
      addAll: out .
  ] ifFalse:[
    result := out 
  ]. 
  ^ result
%

classmethod:
execute: commandLineString
 ^ self execute: commandLineString input: nil
%

classmethod:
_execute: commandLineString input: stdinString
 "Executes commandLineString in a child process .
  If stdinString ~~ nil, writes stdinString to the child's stdin.
  If child returns non-zero exit status, signals an Error using
  contents of stderr from the child.  Otherwise returns 
  an Array of the form { stdout . stderr } .
  containing stdout and stderr from the child. 

  Lookup in the PATH environment variable is not performed,
  the commandLineString must specify a complete path to an executable
  or script.
  Signals an error if the privilege NoPerformOnServer is true."

  | inst out err status readDataBlk writeOfs nToWrite |
  writeOfs := 1 .
  nToWrite := stdinString size .
  (inst := self _basicNew) _fork: commandLineString .
  out := String new .
  err := String new .
  readDataBlk := [
    [ inst stdout readWillNotBlock] whileTrue:[
      out addAll: (inst stdout read: 16384 )
    ].
    [ inst stderr readWillNotBlock] whileTrue:[
      err addAll: (inst stderr read: 16384 )
    ].
  ].
  [ status := inst childStatus .  
    status == nil
  ] whileTrue:[ 
    (nToWrite > 0 and:[ inst stdin writeWillNotBlock]) ifTrue:[ | nWrote |
      nWrote := inst stdin write: nToWrite from: stdinString startingAt: writeOfs . 
      nWrote ifNil:[ Error signal:'lost stdin' , inst stdin lastErrorString ].
      writeOfs := writeOfs + nWrote .
      nToWrite := nToWrite - nWrote .
      nToWrite == 0 ifTrue:[ inst stdin close ].
    ].   
    readDataBlk value 
  ].
  readDataBlk value .
  status ~~ 0 ifTrue:[ Error signal:'Error , ' , err asString ].
  ^ { out . err } .
%

category: 'Private'
classmethod:
_basicNew

"creates an instance registered with VM for finalization of cData"

<primitive: 674>
^ self _primitiveFailed: #_basicNew
%

method:
_fork: commandLineString 

"Signals an error if the privilege NoPerformOnServer is true.
 If running in a topaz -l process and (GsFile stdin isTerminal == true)
 The child process will ignore SIGINT.
"
 
<primitive: 956>
commandLineString _validateClass: String .
self _primitiveFailed: #_fork: args: { commandLineString } 
%

category: 'Accessing'
method:
commandLine
  "After fork: , returns the command line that was used to
   exec the child process."
  ^ cmd
%

method:
stdin
  "After fork: , returns a non-blocking write-only GsSocket 
   representing the parent's end of a pipe (see   man 2 pipe )  
   connected to stdin of the child.  The child's end of the pipe
   is blocking and a read by the child on stdin will block until
   Smalltalk code writes sufficient data to this GsSocket 
   to satisfy the read."
  ^ in
%

method:
stdout 
  "After fork: , returns a non-blocking read-only GsSocket 
  representing the parent's end of a pipe (see   man 2 pipe )  
  connected to stdout of the child.  If the GsSocket is not read
  from sufficiently to consume data, the child process may block 
  waiting to write more data to its stdout ."
  ^ out
%

method:
stderr
  "After fork: , returns a non-blocking read-only GsSocket 
  representing the parent's end of a pipe (see   man 2 pipe )  
  connected to stderr of the child.  If the GsSocket is not read
  from sufficiently to consume data, the child process may block 
  waiting to write more data to its stderr ."
  ^ err
%

method:
processId
  "After fork: , returns a SmallInteger , the operating system
   process id of the child."
  ^ childPid
%

category: 'Private'
method:
_waitChild: killBoolean

"calls waitpid() .
 If child is still running and killBoolean == true, attempts to kill
 the child with a SIGTERM.
 The return status is computed before attempting any kill of the child.

 Returns nil if child still running, otherwise returns a SmallInteger
 representing child status from waitpid() .

  result == 0 means child exited with exit code of 0 .

  ((result bitAnd: 16r100) ~~ 0)  means child exited due to a signal
     and  (result bitAnd: 16rFF) is the number of the signal .

  ((result bitAnd: 16r200) ~~ 0) means child was stopped by a signal
     and  (result bitAnd: 16rFF) is the number of the signal .

  otherwize non-zero result means child exited with that error status.
"
<primitive: 957>
killBoolean _validateClass: Boolean .
self _primitiveFailed: #_waitChild: args: { killBoolean }
%

category: 'Status'
method:
childStatus
  "Returns nil if the child has not been reaped, 
   or returns a SmallInteger as follows

  result == 0 means child exited with exit code of 0 .

  ((result bitAnd: 16r100) ~~ 0)  means child exited due to a signal
     and  (result bitAnd: 16rFF) is the number of the signal .

  ((result bitAnd: 16r200) ~~ 0) means child was stopped by a signal 
     and  (result bitAnd: 16rFF) is the number of the signal .

  otherwize non-zero result means child exited with that error status."
 
  | s |
  (s := childStatus) ifNil:[ 
    s := self _waitChild: false  .
    s ifNotNil:[
      (s bitAnd: 16r200) == 0 ifTrue:[  "child exited"
        childStatus := s
        "in close . out close . err close ???" "close sockets to child"  
      ] 
    ].
  ].
  ^ s
%

method:
childHasExited
| s |
 s := self childStatus .
 s ifNil:[ ^ false "still running"].
 ^ (s  bitAnd: 16r200) == 0   "if not stopped with a signal it has exited"
%

method:
_killChild: timeoutSeconds
  "Attempts to kill the child. Waits specified time for child to go way.
   Returns child status."
  | s endMs |
  endMs := DateTime now asMillisecondsGmt + (1000 * timeoutSeconds).
  (s := childStatus) ifNil:[
     self _waitChild: true . "kill with SIGTERM"
     [ 
       Delay waitForMilliseconds:10.
       (s := self childStatus) ifNotNil:[ ^ s ].
       DateTime now asMillisecondsGmt < endMs 
     ] whileTrue .
  ].
  ^ s
%

method:
killChild: timeoutSeconds
  "Attempts to kill the child. Waits specified time for child to go way.
   Returns child status. Reading from childs stdout or sterr after 
   a kill may block due to no data available. (use GsSocket>>readWillNotBlock)"

  ^ self  _killChild: timeoutSeconds
%

method:
killChild
  "Attempts to kill the child. Waits one second for child to go way.
   Returns child status. Reading from childs stdout or sterr after 
   a kill may block due to no data available. (use GsSocket>>readWillNotBlock)"

^ self  _killChild: 1
%
