!=========================================================================
! Copyright (C) GemTalk Systems 2017-2020.  All Rights Reserved.
!
! $Id: GsEventLog.gs $
!
! GsEventLog, GsEventLogEntry, GsEventLogSystemEntry
!
!=========================================================================


! ------------------- Class definition for GsEventLog
expectvalue /Class
doit
Object subclass: 'GsEventLog'
  instVarNames: #( entryArray newEntriesInvariant)
  classVars: #( )
  classInstVars: #( CurrentLog)
  poolDictionaries: #()
  inDictionary: Globals
  options: #()

%
expectvalue /Class
doit
GsEventLog comment: 
'Instances of GsEventLog hold collections of log entries, each representing an item recorded 
in a logical log file.  There is a shared, system-wide instance in the class instance variable. 
All users are allowed to add and remove entries from this log.  

The entries are stored in an instance of RcArray, allowing concurrent writes of log entries.  
Note that the order of elements is based on the order in which the commits occurred, 
while entry timestamps reflect the time at which the entry was created.

GsEventLog may hold both application (user) events and system events.  
User entires can be added in two ways: class convenience methods such as logError:, logInfo, etc., 
or by creating an instance of GsEventLogEntry and sending addToLog.  
System events should be added only by GemStone code (no events are added at this time).
To restrict modifying or removing events, execute GsEventLog entriesUnmodifiable.  
After this is executed, new entries to the log are made invariant and the standard delete methods 
will not delete them.  However, they are not protected from delete using private delete protocol. 
System events are also protected from modification or delete, other than using private delete 
protocol.

Removing entries is subject to commit conflict, it is recommended to lock the event log prior to 
delete.'
%
! ------------------- Class definition for GsEventLogEntry
expectvalue /Class
doit
Object subclass: 'GsEventLogEntry'
  instVarNames: #( pid timestamp label priority object tag)
  classVars: #()
  classInstVars: #()
  poolDictionaries: #()
  inDictionary: Globals 
  options: #()

%
expectvalue /Class
doit
GsEventLogEntry comment: 
'GsEventLogEntry represents a single entry in a GsEventLog. Instances of GsEventLogEntry are user 
or application-specific events. 
By default, GsEventLogEntries are modifiable.  They can be made invariant in which case they are 
not deleted by default protocol.
It is supported to create custom subclasses with additional fields or different behavior.

GsEventLogEntries have the following fields:

 pid - SmallInteger, the processId of the session that created this entry
 timestamp - DateAndTime, the timestamp that this event was created.
 label - String, a string describing this entry
 priority - SmallInteger (required), the priority, see Class methods errorPrioity, warningPriority, etc.
 object - any object
 tag - any symbol'
%

! ------------------- Class definition for GsEventLogSystemEntry
expectvalue /Class
doit
GsEventLogEntry subclass: 'GsEventLogSystemEntry'
  instVarNames: #( subsystem)
  classVars: #()
  classInstVars: #()
  poolDictionaries: #()
  inDictionary: Globals
  options: #()

%
expectvalue /Class
doit
GsEventLogSystemEntry comment: 
'GsEventLogSystemEntry represents a single, system-generated entry in a GsEventLog. 
Instances of GsEventLogSystemEntry should only be generated by GemStone code. 
They are not deleted from a GsEventLog using default protocol.
'
%

! ------------------- Remove existing behavior from GsEventLog
expectvalue /Metaclass3       
doit
GsEventLog removeAllMethods.
GsEventLog class removeAllMethods.
%
! ------------------- Class methods for GsEventLog
category: 'Instance Creation'
classmethod: GsEventLog
current
  ^ CurrentLog ifNil: [ 
     		CurrentLog := self new .
     		CurrentLog.
 		].
%
category: 'Immutability'
classmethod: GsEventLog
entriesModifiable
	self current entriesModifiable
%
category: 'Immutability'
classmethod: GsEventLog
entriesUnmodifiable
	self current entriesUnmodifiable
%
category: 'Add Log Entries'
classmethod: GsEventLog
logDebug

	^self current addEntry: GsEventLogEntry debug
%
category: 'Add Log Entries'
classmethod: GsEventLog
logDebug: aString

	^self current addEntry: 
		(GsEventLogEntry debug
			label: aString; 
			yourself)
%
category: 'Add Log Entries'
classmethod: GsEventLog
logError

	^self current addEntry: GsEventLogEntry error
%
category: 'Add Log Entries'
classmethod: GsEventLog
logError: aString

	^self current addEntry: 
		(GsEventLogEntry error
			label: aString; 
			yourself)
%
category: 'Add Log Entries'
classmethod: GsEventLog
logFatal

	^self current addEntry: GsEventLogEntry fatal
%
category: 'Add Log Entries'
classmethod: GsEventLog
logFatal: aString

	^self current addEntry: 
		(GsEventLogEntry fatal
			label: aString; 
			yourself)
%
category: 'Add Log Entries'
classmethod: GsEventLog
logInfo

	^self current addEntry: GsEventLogEntry info
%
category: 'Add Log Entries'
classmethod: GsEventLog
logInfo: aString

	^self current addEntry: 
		(GsEventLogEntry info
			label: aString; 
			yourself)
%
category: 'Add Log Entries'
classmethod: GsEventLog
logObject: anObject

	^GsEventLogEntry trace 
		object: anObject;
		label: anObject printString;
		addToLog;
		yourself.
%
category: 'Add Log Entries'
classmethod: GsEventLog
logTrace

	^self current addEntry: GsEventLogEntry trace
%
category: 'Add Log Entries'
classmethod: GsEventLog
logTrace: aString

	^self current addEntry: 
		(GsEventLogEntry trace
			label: aString; 
			yourself)
%
category: 'Add Log Entries'
classmethod: GsEventLog
logWarning

	^self current addEntry: GsEventLogEntry warning
%
category: 'Add Log Entries'
classmethod: GsEventLog
logWarning: aString

	^self current addEntry: 
		(GsEventLogEntry warning
			label: aString; 
			yourself)
%
category: 'Instance Creation'
classmethod: GsEventLog
new
	^super new initialize
	
%
category: 'Private'
classmethod: GsEventLog
_logSystemInfoEventFor: subsystem label: aString object: obj

	GsEventLogSystemEntry info 
		subsystem: subsystem; 
		label: aString; 
		object: obj;
		_addSystemEntryToLog.
%
category: 'Instance Creation'
classmethod: GsEventLog
_newWithEntries: anRcArray
	
	^self new 
		_entryArray: anRcArray; 
		yourself
%
! ------------------- Instance methods for GsEventLog
category: 'Deleting'
method: GsEventLog
acquireGsEventLogLock
	"The lock is not needed for add, only for delete"

	| arr |
	arr := self _entryArray.
	System writeLock: arr
		ifDenied: [ ^false ]
		ifChanged: [
			System addToCommitOrAbortReleaseLocksSet: arr. 
			^false ].
	System addToCommitOrAbortReleaseLocksSet: arr.
	^true
%
category: 'Actions'
method: GsEventLog
addEntry: anEntry

	self _entryArray add: anEntry.
	self _newEntriesInvariant ifTrue: [anEntry makeUnmodifiable].
	^anEntry
%
category: 'Deleting'
method: GsEventLog
deleteAllEntries
	"Delete all entries in the recevier, except for system events or events that cannot be deleted. "

	self _entryArray removeAllSuchThat: [:entry | entry canBeDeleted]
%
category: 'Deleting'
method: GsEventLog
deleteEntries: anotherEventLog
	"delete all entries in the argument from the receiver.
	Quiety ignore entries that are not present or are that cannot be deleted.
	This method does not delete system events."

	self _entryArray == anotherEventLog _entryArray ifTrue: [
		"remove all"
		^self deleteAllEntries].		

	anotherEventLog _entryArray do: [:anEvent | self deleteEntry: anEvent]
%
category: 'Deleting'
method: GsEventLog
deleteEntry: anEvent
	"Remove the given event from the receiver, if allowed"

	anEvent canBeDeleted
		ifTrue: [self _basicDeleteEntry: anEvent].
%
category: 'Queries'
method: GsEventLog
entriesDo: aBlock
	"Execute aBlock for each entry"

	self _entryArray do: aBlock.
%
category: 'Actions'
method: GsEventLog
entriesModifiable 
	self _newEntriesInvariant: false
%
category: 'Queries'
method: GsEventLog
entriesSatisfying: aBlock
	"return a new instance of the receiver containing the entries from the receiver that satisfy the block"

	| results |
	results := self _entryArray select: aBlock.
	^self class _newWithEntries: results
%
category: 'Actions'
method: GsEventLog
entriesUnmodifiable 
	self _newEntriesInvariant: true
%
category: 'Initialization'
method: GsEventLog
initialize
	self objectSecurityPolicy: nil.
	self _entryArray: 
		(RcArray new
			objectSecurityPolicy: nil;
			yourself).
	self _newEntriesInvariant: false.
%
category: 'Queries'
method: GsEventLog
last
	^entryArray last
%
category: 'Reporting'
method: GsEventLog
report
	| str |
	str := AppendStream on: String new.
	self _entryArray do: [:entry | entry printOn: str.  str lf].
	^str contents
%
category: 'Queries'
method: GsEventLog
size
	^entryArray size
%
category: 'Queries'
method: GsEventLog
systemEvents
	^self _entryArray entriesSatisfying: [:ev | ev isSystemEvent]
%
category: 'Queries'
method: GsEventLog
userEvents
	^self entriesSatisfying: [:ev | ev isUserEvent]
%
category: 'Actions'
method: GsEventLog
_addSystemEntry: anEntry

	self _entryArray add: anEntry.
	anEntry makeUnmodifiable.
%
category: 'Private-Deleting system entries'
method: GsEventLog
_basicDeleteAllEntries
	"Delete all entries in the recevier, including system events and events that are 
	configured to not be deleted. "

	self _entryArray size: 0.
%
category: 'Private-Deleting system entries'
method: GsEventLog
_basicDeleteEntry: anEvent
	"Remove the given event from the receiver"

	self _entryArray remove: anEvent ifAbsent: [].
%
category: 'Private-Deleting system entries'
method: GsEventLog
_deleteEntriesIncludingSystem: anotherEventLog
	"delete all entries in the argument from the receiver, including system events
	and events marked do not delete."

	self _entryArray == anotherEventLog _entryArray ifTrue: [
		"remove all"
		^self _basicDeleteAllEntries].		

	anotherEventLog _entryArray do: [:anEvent | self _basicDeleteEntry: anEvent]	
%
category: 'Accessing'
method: GsEventLog
_entryArray
	^entryArray
%
category: 'Updating'
method: GsEventLog
_entryArray: newValue
	entryArray := newValue
%
category: 'Accessing'
method: GsEventLog
_newEntriesInvariant
	^newEntriesInvariant
%
category: 'Updating'
method: GsEventLog
_newEntriesInvariant: newValue
	newEntriesInvariant := newValue
%

! ------------------- Remove existing behavior from GsEventLogEntry
expectvalue /Metaclass3       
doit
GsEventLogEntry removeAllMethods.
GsEventLogEntry class removeAllMethods.
%
! ------------------- Class methods for GsEventLogEntry
category: 'Constants'
classmethod: GsEventLogEntry
debugPriority
	^60
%
category: 'Constants'
classmethod: GsEventLogEntry
errorPriority
	^20
%
category: 'Constants'
classmethod: GsEventLogEntry
fatalPriority
	^10
%
category: 'Constants'
classmethod: GsEventLogEntry
infoPriority
	^40
%
category: 'Constants'
classmethod: GsEventLogEntry
tracePriority
	^50
%
category: 'Constants'
classmethod: GsEventLogEntry
warningPriority
	^30
%
category: 'Instance Creation'
classmethod: GsEventLogEntry
debug

	^self _new 
		_priority: self debugPriority;
		yourself
%
category: 'Instance Creation'
classmethod: GsEventLogEntry
error

	^self _new 
		_priority: self errorPriority;
		yourself
%
category: 'Instance Creation'
classmethod: GsEventLogEntry
fatal

	^self _new 
		_priority: self fatalPriority;
		yourself
%
category: 'Instance Creation'
classmethod: GsEventLogEntry
info

	^self _new 
		_priority: self infoPriority;
		yourself
%
category: 'Instance Creation'
classmethod: GsEventLogEntry
trace

	^self _new 
		_priority: self tracePriority;
		yourself
%
category: 'Instance Creation'
classmethod: GsEventLogEntry
warning

	^self _new
		_priority: self warningPriority;
		yourself
%
category: 'Instance Creation'
classmethod: GsEventLogEntry
_new
	^super new initialize 
%
! ------------------- Instance methods for GsEventLogEntry
category: 'Accessing'
method: GsEventLogEntry
label
	^label
%
category: 'Accessing'
method: GsEventLogEntry
object
	^object
%
category: 'Accessing'
method: GsEventLogEntry
pid
	^pid
%
category: 'Accessing'
method: GsEventLogEntry
priority
	^priority
%
category: 'Accessing'
method: GsEventLogEntry
tag
	^tag
%
category: 'Accessing'
method: GsEventLogEntry
timestamp
	^timestamp
%
category: 'Actions'
method: GsEventLogEntry
addToLog
	
	GsEventLog current addEntry: self
%
category: 'Actions'
method: GsEventLogEntry
deleteEntry
	"Delete the receiver from the current GsEventLog"
	
	GsEventLog current deleteEntry: self
%
category: 'Actions'
method: GsEventLogEntry
makeUnmodifiable
	self immediateInvariant.
%
category: 'Displaying'
method: GsEventLogEntry
printOn: aStream

	aStream 
		nextPutAll: self timestampString;
		tab;
		nextPutAll: self processIdentification;
		tab;
		nextPutAll: self priorityAsLabel;
		tab;
		nextPutAll: self label.
	self object notNil ifTrue:
		[aStream tab; nextPutAll: self object asString].

%
category: 'Displaying'
method: GsEventLogEntry
processIdentification
	^self pid asString
%
category: 'Displaying'
method: GsEventLogEntry
priorityAsLabel
	self priority = self class errorPriority ifTrue: [^'Error'].
	self priority = self class warningPriority ifTrue: [^'Warn'].
	self priority = self class infoPriority ifTrue: [^'Info'].
	self priority = self class tracePriority ifTrue: [^'Trace'].
	self priority = self class debugPriority ifTrue: [^'Debug'].
	self priority = self class fatalPriority ifTrue: [^'Fatal'].
	^'Unknown'
%
category: 'Displaying'
method: GsEventLogEntry
report
	| str |
	str := AppendStream on: String new.
	self printOn: str.
	^str contents
%
category: 'Displaying'
method: GsEventLogEntry
timestampString
	| s |
	s := AppendStream on: String new.
	timestamp printLocalOn: s.
	^s contents
%
category: Displaying'
method: GsEventLogEntry
subsecondDigits
	"The number of digits of subsecond resolution for timestamps"
	^4
%
category: 'Initialization'
method: GsEventLogEntry
initialize

	pid := System gemVersionReport at: 'processId'.
	timestamp := DateAndTime nowWithScale: self subsecondDigits.
	object := nil.
	label := String new.
%
category: 'Private'
method: GsEventLogEntry
_addSystemEntryToLog
	
	GsEventLog current _addSystemEntry: self
%
category: 'Testing'
method: GsEventLogEntry
canBeDeleted
	^self isInvariant not
%
category: 'Testing'
method: GsEventLogEntry
hasTag

	^tag ~~ nil
%
category: 'Testing'
method: GsEventLogEntry
isSystemEvent
	^false
%
category: 'Testing'
method: GsEventLogEntry
isUserEvent
	^true
%
category: 'Updating'
method: GsEventLogEntry
label: newValue
	label := newValue
%
category: 'Updating'
method: GsEventLogEntry
object: newValue
	object := newValue
%
category: 'Updating'
method: GsEventLogEntry
tag: newValue
	tag := newValue
%
category: 'Updating'
method: GsEventLogEntry
_priority: anInteger
	priority := anInteger
%

! ------------------- Remove existing behavior from GsEventLogSystemEntry
expectvalue /Metaclass3       
doit
GsEventLogSystemEntry removeAllMethods.
GsEventLogSystemEntry class removeAllMethods.
%
category: 'Instance creation'
classmethod: GsEventLogSystemEntry
new
	self shouldNotImplement: #new
%
category: 'Accessing'
method: GsEventLogSystemEntry
subsystem
	^subsystem
%
category: 'Actions'
method: GsEventLogSystemEntry
deleteEntry
	"This protocol does not delete system entries"
	
	^self
%
category: 'Displaying'
method: GsEventLogSystemEntry
printOn: aStream

	aStream 
		nextPutAll: self timestampString;
		tab;
		nextPutAll: self priorityAsLabel;
		tab;
		nextPutAll: '(System) ';
		nextPutAll: self label.

	self object notNil ifTrue:
		[aStream tab; nextPutAll: self object asString].
%
category: 'Testing'
method: GsEventLogSystemEntry
canBeDeleted
	^false
%
category: 'Testing'
method: GsEventLogSystemEntry
isSystemEvent
	^true
%
category: 'Testing'
method: GsEventLogSystemEntry
isUserEvent
	^false
%
category: 'Updating'
method: GsEventLogSystemEntry
subsystem: newValue
	subsystem := newValue
%

! make sure the log is initialized
run
GsEventLog current.
^ true
%

