!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: FileStream.gs 25549 2011-03-24 21:51:57Z jfoster $
!
! Based on ANSI spec
!
! Superclass Hierarchy:
!   FileStream, ReadWriteStream, WriteStream, PositionableStream, Stream, Object.
!
! Portable implementation
!
! See installStreamHierarchy.topaz for filein information
!
!=========================================================================

! class created by installStreamHierarchy.topaz

removeallmethods FileStream
removeallclassmethods FileStream

! ------------------- Class methods for FileStream
doit
FileStream comment: '
FileStream / FileStreamPortable is an ANSI-stream compliant stream that is specifically on a file. 

Instance variables:
gsfile - an instance of GsFile
streamType - a Symbol, one of: #binary, #text, #serverBinary, #serverText, #clientBinary, #clientText
'.
true
%

category: 'compatability'
classmethod: FileStream
fileNamed: fileName 
	^ self read: fileName
%
category: 'instance creation'
classmethod: FileStream
on: aCollection

	self shouldNotImplement: #on:
%
category: 'instance creation'
classmethod: FileStream
on: aCollection from: firstIndex to: lastIndex 

	self shouldNotImplement: #on:from:to:
%
category: 'instance creation'
classmethod: FileStream
read: path

	^self read: path type: #text
%
category: 'instance creation'
classmethod: FileStream
read: path type: type
	 " type: #binary, #text, #serverBinary, #serverText, #clientBinary, #clientText"

	^(self basicNew) openForRead: path type: type
%
category: 'instance creation'
classmethod: FileStream
with: aCollection 

	self shouldNotImplement: #with:
%
category: 'instance creation'
classmethod: FileStream
write: path

	^self write: path mode: #create check: false type: #text
%
category: 'instance creation'
classmethod: FileStream
write: path mode: mode

	^self write: path mode: mode check: false type: #text
%
category: 'instance creation'
classmethod: FileStream
write: path mode: mode check: checkBool type: type
	"mode: #create, #append, #truncate
	 check: 
	  mode == #create and check == false and the file exists, then the file is used
	  mode == #create and check == true and the file exists, an error is thrown
	  mode == #append and check == false and the file does not exist then it is created
	  mode == #append and check == true and the file does not exist an error is thrown
	  mode == #truncate and check == false and file does not exist then it is created
	  mode == #truncate and check == true and the file does not exist an error is thrown
	 type: #binary, #text, #serverBinary, #serverText, #clientBinary, #clientText
	"
	
	^(self basicNew) openForWrite: path mode: mode check: checkBool type: type
%
! ------------------- Instance methods for FileStream
category: 'Accessing'
method: FileStream
atBeginning
"Answer true if the stream is positioned at the beginning"

^self position == 0
%
category: 'Accessing'
method: FileStream
atEnd

"Returns true if the receiver cannot access any more objects, false if it can."

	^self gsfile atEnd ifNil: [true].
%
category: 'Accessing'
method: FileStream
binary
	streamType := #binary
%
category: 'Accessing'
method: FileStream
close

	self gsfile close
%
category: 'Accessing'
method: FileStream
collectionSpecies
	"Answer the species of collection into which the receiver can stream"
	
	^streamType == #binary
		ifTrue: [ ByteArray ]
		ifFalse: [ String ]
%
category: 'Accessing'
method: FileStream
contents
	"Return the contents of the receiver. Do not close or otherwise touch the receiver. Return data in whatever mode the receiver is in (e.g., binary or text)."
	| s savePos |
	savePos := self position.
	self position: 0.
	s := self next: self size.
	self position: savePos.
	^s
%
category: 'Adding'
method: FileStream
cr
"Adds a newline to the output stream."

| cr |
cr := self isBinary
	ifTrue: [ Character cr codePoint ]
	ifFalse: [ Character cr ].
self nextPut: cr.
%
category: 'Accessing'
method: FileStream
externalType
	"Return a symbol that identifies the external stream type of the receiver."
	
	^self gsfile _isBinary
		ifTrue: [ #binary ]
		ifFalse: [ #text ]
%
category: 'Accessing'
method: FileStream
flush
	"Update a stream's backing store."

	self gsfile flush
%
category: 'Accessing'
method: FileStream
gsfile

	^gsfile
%
category: 'Testing'
method: FileStream
isBinary

	^streamType == #binary
%
category: 'Testing'
method: FileStream
isEmpty
"Returns true if the collection that the receiver accesses contains
 no elements; otherwise returns false."

^ self size = 0
%
category: 'Testing'
method: FileStream
isText

	^self isBinary not
%
category: 'Adding'
method: FileStream
lf
"Adds a linefeed to the output stream."

| lf |
lf := self isBinary
	ifTrue: [ Character lf codePoint ]
	ifFalse: [ Character lf ].
self nextPut: lf.
%
category: 'Accessing'
method: FileStream
next

	| res |
	(res := self next: 1) ifNil: [ ^nil ].
	^res at: 1
%
category: 'Accessing'
method: FileStream
next: n
	"Return a the appropriate collection species with the next n characters/bytes of the filestream in it."

	| result charSize |
	result := self collectionSpecies new.
	n = 0 ifTrue: [ ^result ]. "avoid 0 size check in #next:ofSize:into:"
	charSize := self isBinary
		ifTrue: [ 1 ]
		ifFalse: [ result charSize ].
        charSize == 1 ifTrue:[ 
	  (self gsfile next: n into: result ) ifNil:[ ^ nil ].
	  ^ result
        ].
	(self gsfile next: n ofSize: charSize into: result ) ifNil:[ ^ nil ].
	^ result
%
category: 'Accessing'
method: FileStream
next: anInteger putAll: aCollection startingAt: startIndex
"Store the next anInteger elements from the given collection."
(startIndex = 1 and:[anInteger = aCollection size])
	ifTrue:[^self nextPutAll: aCollection].
^self nextPutAll: (aCollection copyFrom: startIndex to: startIndex+anInteger-1)
%
category: 'Accessing'
method: FileStream
nextLine
"Answer next line (may be empty) without line end delimiters, or nil if at end.
Let the stream positioned after the line delimiter(s).
Handle a zoo of line delimiters CR, LF, or CR-LF pair"

self atEnd ifTrue: [ ^self collectionSpecies new ].
self isBinary
	ifTrue: [self error: '#nextLine not appropriate for a binary stream' ]. 
^super nextLine
%

category: 'Adding'
method: FileStream
nextPut: anObject
  "Inserts anObject as the next element that the receiver can access for writing.
 Returns anObject."

  | status |
  self isBinary
    ifTrue: 
      [(anObject isKindOf: Integer) ifFalse: [^self error: 'Expected an Integer']]
    ifFalse: 
      [(anObject isKindOf: AbstractCharacter)
        ifFalse: [^self error: 'Expected a Character']].
  status := self gsfile nextPut: anObject.
  "Status should be either true or nil."
  status == true ifFalse: [^self error: 'Unknown error writing to file'].
  ^anObject
%
category: 'Accessing'
method: FileStream
nextPutAll: aCollection

"Inserts the elements of aCollection as the next elements that the receiver can
 access.  Returns aCollection."

	aCollection do: [:each | self nextPut: each ]
%
! fixed 47155
category: 'initialization'
method: FileStream
openForRead: path type: type
	 " type: #binary, #text, #serverBinary, #serverText, #clientBinary, #clientText"

	| mode onClient |
	streamType := #text.
	(type == #binary or: [ type == #serverBinary ])
		ifTrue: [ mode := 'rb'. onClient := false. streamType := #binary ].
	(type == #text or: [ type == #serverText ] )
		ifTrue: [ mode := 'r'. onClient := false ].
	type == #clientBinary
		ifTrue: [ mode := 'rb'. onClient := true. streamType := #binary ].
	type == #clientText
		ifTrue: [ mode := 'r'. onClient := true ].
	gsfile := GsFile open: path mode: mode onClient: onClient.
	gsfile == nil ifTrue: [ ^self error: 'Error opening file: ', (GsFile classUserAction: #GsfClassError onClient: onClient with: nil) ].
%
category: 'initialization'
method: FileStream
openForWrite: path mode: modeSymbol check: check type: type
	"mode: #create, #append, #truncate
	 check: 
	  mode == #create and check == false and the file exists, then the file is used
	  mode == #create and check == true and the file exists, an error is thrown
	  mode == #append and check == false and the file does not exist then it is created
	  mode == #append and check == true and the file does not exist an error is thrown
	  mode == #truncate and check == false and file does not exist then it is created
	  mode == #truncate and check == true and the file does not exist an error is thrown
	 type: #binary, #text, #serverBinary, #serverText, #clientBinary, #clientText
	"
	
	| prefix mode onClient |
	prefix := 'w+'.
	streamType := #text.
	modeSymbol == #append
		ifTrue: [ prefix := 'a+' ].
	(type == #binary or: [ type == #serverBinary ])
		ifTrue: [ mode := prefix, 'b'. onClient := false. streamType := #binary ].
	(type == #text or: [ type == #serverText ] )
		ifTrue: [ mode := prefix. onClient := false ].
	type == #clientBinary
		ifTrue: [ mode := prefix, 'b'. onClient := true. streamType := #binary ].
	type == #clientText
		ifTrue: [ mode := prefix. onClient := true ].
	(modeSymbol == #create and: [ check and: [ GsFile _exists: path onClient: onClient ]])
		ifTrue: [ ^self error: 'The file already exists.' ].
	((modeSymbol == #truncate or: [ modeSymbol == #append]) and: [ check and: [ (GsFile _exists: path onClient: onClient) not ]])
		ifTrue: [ ^self error: 'The file does not exist.' ].
	gsfile := GsFile open: path mode: mode onClient: onClient.
	gsfile == nil ifTrue: [ ^self error: 'Error opening file: ', (GsFile classUserAction: #GsfClassError onClient: onClient with: nil) ].
	modeSymbol == #truncate
		ifTrue: [ self truncate: 0 ].
	modeSymbol == #append
		ifTrue: [ gsfile seekFromEnd: 0 ].
%
category: 'Accessing'
method: FileStream
peek
"Returns the next element in the collection, but does not alter the current
 position reference.  If the receiver is at the end of the collection, returns
 nil."
	| pos result |

	pos := self position.
	result := self next.
	self position: pos.
	^result
%
category: 'Accessing'
method: FileStream
peek2
"Peeks at the second incoming object."

^self gsfile peek2
%

! fix 47156
category: 'Accessing'
method: FileStream
peekTwice

  "Returns an array containing the two elements that would would be returned 
  if the message #next were sent to the receiver twice. If the receiver is at 
  or reaches the end, the array will include one or two nils."

 ^ { self peek . self peek2 }
%

category: 'Accessing'
method: FileStream
peekFor: anObject
"Answer false and do not move over the next element if it is not equal to 
the argument, anObject, or if the receiver is at the end. Answer true 
and increment the position for accessing elements, if the next element is 
equal to anObject."

| nextObject |
self atEnd ifTrue: [^false].
nextObject := self next.
"peek for matching element"
anObject = nextObject ifTrue: [^true].
"gobble it if found"
self position: self position - 1.
^false
%

category: 'Accessing'
method: FileStream
position
	"Return the receiver's current file position."

	^self gsfile position
%
category: 'Accessing'
method: FileStream
position: pos
	"Set the receiver's position as indicated."

	(pos > self size or: [ pos < 0 ]) ifTrue: [ ^self error: 'Attempt to position file beyond bounds of file' ].
	self gsfile position: pos
%
category: 'Accessing'
method: FileStream
reset
"Sets the receiver's position to the beginning of the sequence of objects."

	self position: 0
%
category: 'Accessing'
method: FileStream
setToEnd
	"Set the position of the receiver to the end of the sequence of objects."

	self position: self size
%
category: 'Accessing'
method: FileStream
size

	^self gsfile fileSize
%
category: 'Accessing'
method: FileStream
skip: amount
	"Sets the receiver's position to position+amount."

	self gsfile skip: amount
%
category: 'Adding'
method: FileStream
space
"Adds a space to the output stream."

| space |
space := self isBinary
	ifTrue: [ Character space codePoint ]
	ifFalse: [ Character space ].
self nextPut: space.
%
category: 'Adding'
method: FileStream
tab
"Adds a tab to the output stream."

| tab |
tab := self isBinary
	ifTrue: [ Character tab codePoint ]
	ifFalse: [ Character tab ].
self nextPut: tab.
%
category: 'Accessing'
method: FileStream
throughAll: matchCollection

"Returns a collection of objects from the receiver up to and including the
sequence of objects in the argument 'matchCollection', leaving the stream
positioned after the sequence.  If the sequence of objects is not found, this
returns the remaining contents of the receiver and leaves me positioned
at my end."

| numMatched numToMatch result  |

numMatched := 0.
result := self collectionSpecies new.
numToMatch := matchCollection size.
[self atEnd _or: [numMatched = numToMatch]]
     whileFalse:
           [self next = (matchCollection at: numMatched + 1)
                ifTrue: [numMatched := numMatched + 1]
          ifFalse: [self position: self position - numMatched - 1.
                        result add: self next.
                        numMatched := 0]
].

"add matched or partially matched chars"
self position: self position - numMatched.
numMatched timesRepeat: [result add: self next].

^ result.
%
category: 'Accessing'
method: FileStream
truncate: pos
	"Truncate to this position"

	self position: pos.
	self gsfile flush.
	self close.
	self gsfile open.
	self position: pos
%
category: 'Accessing'
method: FileStream
upTo: anObject 
"Answer a subcollection from the current access position to the 
occurrence (if any, but not inclusive) of anObject in the receiver. If 
anObject is not in the collection, answer the entire rest of the receiver."
| newStream element |
newStream := AppendStream on: self collectionSpecies new.
[self atEnd or: [(element := self next) = anObject]]
	whileFalse: [newStream nextPut: element].
^newStream contents
%
category: 'Accessing'
method: FileStream
upToAny: aCollection
"Answer a subcollection from the current access position to the 
occurrence (if any, but not inclusive) of any objects in the given collection in the receiver. If 
any of these is not in the collection, answer the entire rest of the receiver."
	
| newStream element |
newStream := AppendStream on: self collectionSpecies new.
[self atEnd or: [aCollection includes: (element := self next)]]
	whileFalse: [newStream nextPut: element].
^newStream contents
%
category: 'Accessing'
method: FileStream
upToAnyOf: subcollection do: aBlock
"Answer a subcollection from the current access position to the occurrence (if any, but not inclusive) of any object in the collection.
Evaluate aBlock with this occurence as argument.
If no matching object is found, don't evaluate aBlock and answer the entire rest of the receiver."

| stream ch |
stream := AppendStream on: self collectionSpecies new.	
[ self atEnd or: [ (subcollection includes: (ch := self next)) and: [aBlock value: ch. true] ] ] 
	whileFalse: [ stream nextPut: ch ].
^ stream contents  "private stream, no need to copy"
%
category: 'Accessing'
method: FileStream
upToEnd
	"Answer a subcollection from the current access position through the last element of the receiver."

	| stream next |
	stream := AppendStream on: self collectionSpecies.
	[ (next := self next) == nil ] whileFalse: [
			stream nextPut: next ].
	^stream contents
%
