!=========================================================================
! Copyright (C) GemTalk Systems 2017-2020.  All Rights Reserved.
!
! $Id: Btree-Node-Core.gs 38384 2016-01-08 18:22:36Z lalmarod $
!
! Btree-Node-Core.gs  -  source code for the gs btree plus node classes
!
!========================================================================

! class created in btreeplusclasses.gs

! Class Implementation for BtreePlusNode

! Remove existing behavior from BtreePlusNode
removeallmethods BtreePlusNode
removeallclassmethods BtreePlusNode

! ------------------- Class methods for BtreePlusNode

category: 'Constants'
classmethod: BtreePlusNode
entrySize
  "Returns the size of an entry."

  "in-lined on instance-side for performance"

  self subclassResponsibility: #'entrySize'
%

category: 'Instance Creation'
classmethod: BtreePlusNode
new

"Returns a new initialized instance with the correct size."

^ (super new: self entrySize * self maxNumberOfElements) numElements: 0.
%

! ------------------- Instance methods for BtreePlusNode

category: 'Query Support'
method: BtreePlusNode
_addRootObjects: aQuerySpec
start: startIndex
end: endIndex 
into: collection

^aQuerySpec 
    _addRootObjectsFrom: self
    entrySize: self entrySize
    start: startIndex
    end: endIndex 
    into: collection
%

category: 'Testing'
method: BtreePlusNode
_basicCanCompare: aKey withClass: aClass
  "Returns true if the receiver can make comparisons with the given key, and
false otherwise.  Used for basic nodes."

  ^ aKey _idxBasicPlusCanCompareWithClass: aClass
%

category: 'Testing'
method: BtreePlusNode
_canCompare: aKey withClass: aClass
  "Returns whether the receiver can make comparisons with the given key."

  ^ self indexObject optimizingComparison
    ifTrue: [ self _canCompareOptimized: aKey withClass: aClass ]
    ifFalse: [ aKey == nil
        ifTrue: [ ^ true ].
      self _canCompareStandard: aKey withClass: aClass ]
%

category: 'Testing'
method: BtreePlusNode
_canCompareOptimized: aKey withClass: aClass
  "Returns whether the receiver can make comparisons with the given key.
If optimized compare okay, apply standard rules."

  (aKey _idxOptimizedCompareWithClass: aClass) ifFalse: [ ^ false ].
  ^ self _canCompareStandard: aKey withClass: aClass
%

category: 'Testing'
method: BtreePlusNode
_canCompareStandard: aKey withClass: aClass
  "Returns whether the receiver can make comparisons with the given key."

  (GsRangeEqualityIndex isBasicClass: aClass)
    ifTrue: [ 
      "preserve legacy index constraints for basic index classes"
      ^ self _basicCanCompare: aKey withClass: aClass ].
  ^ true
%

category: 'Updating'
method: BtreePlusNode
_createSibling
  "Creates a new sibling node and move the second half of the receiver's entries into the
sibling.  Returns the new sibling."

  | newSibling sz splitIndex halfNumElements |
  newSibling := self class new.
  newSibling
    objectSecurityPolicy: self objectSecurityPolicy;
    rootNode: self rootNode.
  sz := numElements * self entrySize.
  halfNumElements := numElements quo: 2.
  splitIndex := halfNumElements * self entrySize + 1.	" splitIndex is where to split the receiver "
  newSibling
    _insertAt: 1
    from: self
    fromStart: splitIndex
    fromEnd: sz
    numToMoveDown: 0.	" copy the last half of entries into the new sibling "
  self _deleteNoShrinkFrom: splitIndex to: sz.	" nil out last half of receiver "
  newSibling numElements: numElements - halfNumElements.	" update the element counts in the receiver..."
  numElements := halfNumElements.	"... and the sibling "
  ^ newSibling
%

category: 'Searching'
method: BtreePlusNode
_findAllValuesForIdenticalKey: aKey into: aCollection
  "Finds all the values for the given identical key, placing them in the
 collection.  Return whether the last key was equal or not (so the caller
 knows it must check the next leaf node)."

  ^ self
    _findAllValuesForIdenticalKey: aKey
    into: aCollection
    using: self comparisonForSort
%

category: 'Searching'
method: BtreePlusNode
_findCoveringIndexForKey: aKey totalOrder: bool1
  "Returns the index for the first entry in which aKey is found.  This is the
first entry whose key >= aKey.  If the receiver's last key is greater than
aKey, or if the receiver is empty, returns 0.  Uses a binary search."

  ^ self _findCoveringIndexForKey: aKey totalOrder: bool1 using: self comparisonForSort
%

category: 'Searching'
method: BtreePlusNode
_findCoveringIndexForKey: aKey totalOrder: bool1 using: aComparison
  "Returns the index for the first entry in which aKey is found.  This is the
first entry whose key >= aKey.  If the receiver's last key is greater than
aKey, or if the receiver is empty, returns 0.  Uses a binary search."

  | index |
  numElements == 0
    ifTrue: [ ^ 0 ].
  index := self binarySearchCoveringKey: aKey totalOrder: bool1  using: aComparison.	" check if not found "
  ^ index > self _lastKeyIndex
    ifTrue: [ 0 ]
    ifFalse: [ index ]
%

category: 'Accessing'
method: BtreePlusNode
_lastEntryIndex
  "Returns the index of the last entry in the node, zero if the node is empty."

  numElements == 0
    ifTrue: [ ^ 0 ].
  ^ (numElements - 1) * self entrySize + 1
%

category: 'Accessing'
method: BtreePlusNode
_lastIndex

"Returns the index of the last used slot in the node."

^ numElements * self entrySize
%

category: 'Accessing'
method: BtreePlusNode
_lastKeyIndex
  "Returns the index of the key of the last entry, or zero if the node is empty."

  ^ numElements == 0
    ifTrue: [ 0 ]
    ifFalse: [ self _lastEntryIndex + self keyIndexOffset ]
%

category: 'Accessing'
method: BtreePlusNode
_lastRootIndex
  "Returns the index of the root object of the last entry, or zero if the node is empty."

  ^ numElements == 0
    ifTrue: [ 0 ]
    ifFalse: [ self _lastEntryIndex + self rootIndexOffset ]
%

category: 'Audit'
method: BtreePlusNode
_primEncodingFor: aKey 

"Get the encryption for the given value and return it
 in a 2 slot Array."

<primitive: 871>
self _primitiveFailed: #'_encodingFor:' args: { aKey }
%

category: 'Updating'
method: BtreePlusNode
add: anObject
  self shouldNotImplement: #add:
%

category: 'Updating'
method: BtreePlusNode
addAll: aCollection
  self shouldNotImplement: #addAll:
%

category: 'Updating'
method: BtreePlusNode
addLast: anObject
  self shouldNotImplement: #addLast:
%

category: 'Updating'
method: BtreePlusNode
at: aKey put: aValue
  self shouldNotImplement: #at:put:
%

category: 'Updating'
method: BtreePlusNode
at: aKey put: aValue root: rootObject
  "Adds the key/value/root tuple to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  self subclassResponsibility: #'at:put:root:'
%

category: 'Rc Updating'
method: BtreePlusNode
at: aKey put: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil
  "Adds the key/value/root tuple to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  self subclassResponsibility: #'at:put:root:selectiveAbortSet:'
%

category: 'Audit'
method: BtreePlusNode
auditDepListForLastElements: setOfKeys bagOfKeys: bagOfKeys for: pathTerm offset: offset using: auditor
  self rootNode isIdentityIndex
    ifTrue: [ "Identity indexes do not use dependency lists"
      ^ self ].
  1 to: setOfKeys size do: [ :n | | aKey |
    aKey := setOfKeys _at: n.
    (DependencyList for: aKey) ~~ nil
      ifTrue: [ pathTerm
          auditDepListForLastElementObject: aKey
          occurrences: (bagOfKeys occurrencesOf: aKey)
          index: (pathTerm at: offset)
          using: auditor ] ]
%

category: 'Audit'
method: BtreePlusNode
auditForPathTerm: pathTerm using: auditor do: aBlock
  "Audit the receiver's entries and each child node"

  self auditForPathTerm: pathTerm using: auditor parentNode: nil previousChild: nil nextChild: nil do: aBlock
%

category: 'Audit'
method: BtreePlusNode
auditForPathTerm: pathTerm using: auditor parentNode: parentNode previousChild: previousChild nextChild: nextChild do: aBlock
  self subclassResponsibility: #'auditForPathTerm:using:parentNode:previousChild:nextChild:do:'
%

category: 'Audit'
method: BtreePlusNode
auditNsc: nsc for: pathTerm offset: offset using: auditor
  "Verifies that the btree structure consistent relative to the given pathTerm."

  | indicatesIndexOnNscElements indicatesMultiValue count key value rootObject bag ivOffset bagOfKeys setOfKeys comparisonForSort previousKey previousValue sentinel enumBag previousRoot enumValue setValBag setValValue lastChildNode  bagOfRoots |
  indicatesIndexOnNscElements := pathTerm indicatesIndexOnNscElements.
  indicatesMultiValue := pathTerm indicatesMultiValue.
  count := 0.
  bag := IdentityBag new.
  enumBag := IdentityBag new.
  setValBag := IdentityBag new.
  bagOfKeys := IdentityBag new.
  bagOfRoots := IdentityBag new.
  setOfKeys := IdentitySet new.
  comparisonForSort := self comparisonForSort.
  sentinel := Object new.
  previousRoot := previousKey := sentinel.
  lastChildNode := sentinel.
  self
    auditForPathTerm: pathTerm
    using: auditor
    do: [ :childNode :k :v :r | setValValue := enumValue := false.
        lastChildNode := childNode.
        previousKey == sentinel
        ifFalse: [ (comparisonForSort compareKey: previousKey lessThanOrEqualTo: k)
            ifTrue: [ (comparisonForSort compareKey: previousKey equalTo: k)
                ifTrue: [ previousValue asOop <= v asOop
                    ifFalse: [ auditor
                        btree: childNode
                        outOfOrderKeyValuePairs: pathTerm
                        key: k
                        previous: previousKey
                        value: v
                        previous: previousValue ].
                  previousValue asOop = v asOop
                    ifTrue: [ previousRoot asOop <= r asOop
                        ifFalse: [ auditor
                            btree: childNode
                            outOfOrderKeyValueRootPairs: pathTerm
                            key: k
                            previous: previousKey
                            value: v
                            previous: previousValue
                            root: r
                            previous: previousRoot ] ] ] ]
            ifFalse: [ auditor btree: childNode outOfOrderKeys: pathTerm key: k previous: previousKey ] ].
      previousKey := k.
      previousValue := v.
      previousRoot := r.
      count := count + 1.
      indicatesIndexOnNscElements
        ifTrue: [ pathTerm isSetValuedTerm ifFalse: [v == k
            ifFalse: [ auditor
                btree: childNode
                mismatchedKeyAndValueForIndexOnNscElements: pathTerm
                key: k
                value: v ] ].
          indicatesMultiValue
            ifFalse: [
              v == r
                ifFalse: [ 
                  auditor
                    btree: childNode
                    mismatchedRootAndValueForIndexOnNscElements: pathTerm
                    key: k
                    value: v
                    root: r ] ] .
pathTerm isSetValuedTerm ifTrue: [ setValValue := true ]]
        ifFalse: [ pathTerm isEnumeratedTerm
            ifTrue: [ | ivOffsets res  |
              ivOffsets := pathTerm _ivOffsetsFor: v.
              res := {}.
              ivOffsets
                do: [ :off | | expected |
                 off ifNotNil: [ 
                  (expected := v instVarAt: off) == k
                    ifTrue: [ res add: {true} ]
                    ifFalse: [ res add: {false. expected. off} ] ] ].
              res 
                detect: [:ar | ar at: 1 ]
                ifNone: [
                  res do: [:ar |
                    auditor
                            btree: childNode
                            incorrectKey: pathTerm
                            key: k
                            value: v
                            expectedKey: (ar at: 2)
                            ivOffset: (ar at: 3) ] ].
              enumValue := true ]
            ifFalse: [ | expected |
              pathTerm isSelectorTerm
                ifTrue: [ expected := v perform: pathTerm termSelector.
                  (expected _idxForSortEqualTo: k)
                    ifFalse: [ auditor
                        btree: childNode
                        incorrectKeyForEnumeratedPathTerm: pathTerm
                        key: k
                        value: v
                        expectedKey: expected
                        selector: pathTerm termSelector ] ]
                ifFalse: [ 
                  pathTerm isSetValuedTerm
                    ifTrue: [ setValValue := true ]
                    ifFalse: [
                      ivOffset := pathTerm _ivOffsetFor: v.
                      (expected := v instVarAt: ivOffset) == k
                        ifFalse: [ auditor
                          btree: childNode
                          incorrectKey: pathTerm
                          key: k
                          value: v
                          expectedKey: expected
                          ivOffset: ivOffset ] ] ] ] ].
      count > 1
        ifTrue: [
          (key == k and: [ value == v and: [ rootObject == r ] ])
            ifFalse: [ childNode
                auditKey: key
                value: value
                root: rootObject
                bag: bag
                nsc: nsc
                for: pathTerm
                offset: offset
                bagOfRoots: bagOfRoots
                enumBag: enumBag
                setValBag: setValBag
                using: auditor ].
          (comparisonForSort compareKey: key equalTo: k)
            ifTrue: [ bagOfKeys add: k.
              setOfKeys add: k ]
            ifFalse: [ 
              self
                auditDepListForLastElements: setOfKeys
                bagOfKeys: bagOfKeys
                for: pathTerm
                offset: offset
                using: auditor.
              bagOfKeys removeAll: bagOfKeys.
              setOfKeys removeAll: setOfKeys.
              bagOfKeys add: k.
              setOfKeys add: k ] ]
        ifFalse: [ bagOfKeys add: k.
          setOfKeys add: k ].
      bagOfRoots add: r.
      value := v.
      key := k.
      rootObject := r.
      bag add: v.
      enumValue ifTrue: [ enumBag add: v ].
      setValValue ifTrue: [ setValBag add: v ] ].
  (count > 0 and: [ lastChildNode ~~ sentinel ])
    ifTrue: [ lastChildNode
        auditKey: key
        value: value
        root: rootObject
        bag: bag
        nsc: nsc
        for: pathTerm
        offset: offset
        bagOfRoots: bagOfRoots
        enumBag: enumBag
        setValBag: setValBag
        using: auditor ].
  self
    auditDepListForLastElements: setOfKeys
    bagOfKeys: bagOfKeys
    for: pathTerm
    offset: offset
    using: auditor.
  ^ count
%

category: 'Searching'
method: BtreePlusNode
binarySearchCompareLessThanOrEqualTo: aKey forEntryAt: index using: aComparisonForSort
  | entry |
  entry := self at: index.
  (aComparisonForSort compareKey: aKey lessThan: entry)
    ifTrue: [ ^ true ]
    ifFalse: [ (aComparisonForSort compareKey: aKey greaterThan: entry)
        ifTrue: [ ^ false ]
        ifFalse: [ ^ true ] ]
%

category: 'Searching'
method: BtreePlusNode
binarySearchCoveringKey: aKey
  "Returns the index for the entry in which aKey would be inserted.  This
   is the first entry which has its key >= aKey. If the receiver is empty, returns 
   1.  Uses a binary search."

  | lowPt midPt highPt entrySize index entryGreaterThan entryEqualTo comparisonForSort keyIndexOffset |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ] whileTrue: [ | objAtIndex |
      entryGreaterThan := false.
      entryEqualTo := false.
      midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      objAtIndex := self at: index.
      (comparisonForSort compareKey: aKey lessThan: objAtIndex)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ (comparisonForSort compareKey: aKey greaterThan: objAtIndex)
            ifTrue: [ lowPt := midPt + 1.
              entryGreaterThan := true ]
            ifFalse: [ "If keys are equal, then secondary sort key is oop value, tertiary sort key os oop of rootObject"
              (comparisonForSort compareKey: aKey equalTo: objAtIndex)
                ifTrue: [ (self
                    compareValueOop: aKey
                    lessThanOrEqualToEntryValueOopAt: index)
                    ifTrue: [ highPt := midPt - 1 ]
                    ifFalse: [ lowPt := midPt + 1 ].
                  entryEqualTo := true ]
                ifFalse: [ lowPt := midPt + 1 ] ] ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  (entryEqualTo
    and: [ "use secondary sort key (oop of value), tertiary sort key os oop of rootObject"
      self
        compareValueOop: aKey
        greaterThanEntryValueOopAt: index ])
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  ^ index - keyIndexOffset
%

category: 'Searching'
method: BtreePlusNode
binarySearchCoveringKey: aKey totalOrder: aBoolean
  "Returns the index for the first entry in which aKey is found utilizing a
binary search.  This is the first entry whose key >= aKey."

  ^ self binarySearchCoveringKey: aKey totalOrder: aBoolean using: self comparisonForSort
%

category: 'Searching'
method: BtreePlusNode
binarySearchCoveringKey: aKey totalOrder: aBoolean using: aComparisonForSort
  "Returns the index for the first entry in which aKey is found utilizing a
binary search.  This is the first entry whose key >= aKey."

  | lowPt midPt highPt entrySize index entryGreaterThan keyIndexOffset |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ] whileTrue: [ midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      entryGreaterThan := false.
      (self
        binarySearchCompareLessThanOrEqualTo: aKey
        forEntryAt: index
        using: aComparisonForSort)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ lowPt := midPt + 1.
          entryGreaterThan := true ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  ^ index - keyIndexOffset
%

category: 'Searching'
method: BtreePlusNode
binarySearchCoveringKey: aKey value: aValue
  "Returns the index for the entry in which aKey/aValue would be inserted.  This
 is the first entry which has its key >= aKey.  If multiple entries have the
 same key (key = aKey), then aValue is used (since entries with duplicate keys 
 are inserted by the value's OOP ordering). If the receiver is empty, returns 
 1.  Uses a binary search."

  | lowPt midPt highPt entrySize index entryGreaterThan entryEqualTo comparisonForSort keyIndexOffset |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ] whileTrue: [ | objAtIndex |
      entryGreaterThan := false.
      entryEqualTo := false.
      midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      objAtIndex := self at: index.
      (comparisonForSort compareKey: aKey lessThan: objAtIndex)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ (comparisonForSort compareKey: aKey greaterThan: objAtIndex)
            ifTrue: [ lowPt := midPt + 1.
              entryGreaterThan := true ]
            ifFalse: [ "If keys are equal, then secondary sort key is oop value, tertiary sort key os oop of rootObject"
              (comparisonForSort compareKey: aKey equalTo: objAtIndex)
                ifTrue: [ (self
                    compareValueOop: aValue
                    lessThanOrEqualToEntryValueOopAt: index)
                    ifTrue: [ highPt := midPt - 1 ]
                    ifFalse: [ lowPt := midPt + 1 ].
                  entryEqualTo := true ]
                ifFalse: [ lowPt := midPt + 1 ] ] ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  (entryEqualTo
    and: [ "use secondary sort key (oop of value), tertiary sort key os oop of rootObject"
      self
        compareValueOop: aValue
        greaterThanEntryValueOopAt: index ])
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  ^ index - keyIndexOffset
%

category: 'Searching'
method: BtreePlusNode
binarySearchCoveringKey: aKey value: aValue root: rootObject
  "Returns the index for the entry in which aKey/aValue would be inserted.  This
 is the first entry which has its key >= aKey.  If multiple entries have the
 same key (key = aKey), then aValue is used (since entries with duplicate keys 
 are inserted by the value's OOP ordering).  If multiple entries have the same value 
 (value = aValue), then rootObject is used (since entries with duplicate values are
 inserted by the rootObject's OOP ordering). If the receiver is empty, returns 
 1.  Uses a binary search."

  | lowPt midPt highPt entrySize index entryGreaterThan entryEqualTo comparisonForSort keyIndexOffset |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ]
    whileTrue: [ 
      | objAtIndex |
      entryGreaterThan := false.
      entryEqualTo := false.
      midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      objAtIndex := self at: index.
      (comparisonForSort compareKey: aKey lessThan: objAtIndex)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ 
          (comparisonForSort compareKey: aKey greaterThan: objAtIndex)
            ifTrue: [ 
              lowPt := midPt + 1.
              entryGreaterThan := true ]
            ifFalse: [ 
              "If keys are equal, then secondary sort key is oop value, tertiary sort key os oop of rootObject"
              (comparisonForSort compareKey: aKey equalTo: objAtIndex)
                ifTrue: [ 
                  (self
                    compareValueOop: aValue
                    lessThanOrEqualToEntryValueOopAt: index
                    root: rootObject)
                    ifTrue: [ highPt := midPt - 1 ]
                    ifFalse: [ lowPt := midPt + 1 ].
                  entryEqualTo := true ]
                ifFalse: [ lowPt := midPt + 1 ] ] ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  (entryEqualTo
    and: [ 
      "use secondary sort key (oop of value), tertiary sort key os oop of rootObject"
      self
        compareValueOop: aValue
        greaterThanEntryValueOopAt: index
        root: rootObject ])
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  ^ index - keyIndexOffset
%

category: 'Searching'
method: BtreePlusNode
binarySearchForKey: aKey value: aValue root: rootObject
  "Returns the 'value' index whose key equality matches aKey and whose root matches rootObject.  
   If no match is found, returns zero.  Uses a binary search."

  | lowPt midPt highPt entrySize index comparisonForSort keyIndexOffset |
  numElements == 0
    ifTrue: [ ^ 0 ].
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ] whileTrue: [ midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      (self
        compare: comparisonForSort
        key: aKey
        value: aValue
        equalToEntryAt: index
        root: rootObject)
        ifTrue: [ ^ index - keyIndexOffset ].
      (self
        compare: comparisonForSort
        key: aKey
        value: aValue
        lessThanOrEqualToEntryAt: index
        root: rootObject)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ lowPt := midPt + 1 ] ].
  ^ 0
%

category: 'Accessing'
method: BtreePlusNode
collator
  "Answer the collator for the receiver"

  ^ rootNode collator
%

category: 'Comparison Operators'
method: BtreePlusNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  self subclassResponsibility: #'compareValueOop:greaterThanEntryValueAt:'
%

category: 'Comparison Operators'
method: BtreePlusNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex root: aRootObjectOrNil
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index. If the value oops are equal, compare the oop of aRootObjectOrNil and the 
   oop of the root object at the given index."

  self subclassResponsibility: #'compareValueOop:greaterThanEntryValueAt:root:'
%

category: 'Comparison Operators'
method: BtreePlusNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  self
    subclassResponsibility: #'compareValueOop:lessThanOrEqualToEntryValueOopAt:'
%

category: 'Comparison Operators'
method: BtreePlusNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex root: aRootObjectOrNil
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  self
    subclassResponsibility: #'compareValueOop:lessThanOrEqualToEntryValueOopAt:root:'
%

category: 'Comparison Operators'
method: BtreePlusNode
comparisonForSort
  "Answer the comparison object to be used for this index"

  ^ rootNode comparisonForSort
%

category: 'Constants'
method: BtreePlusNode
entrySize
  "Returns the size of an entry."

  "in-line of `self class entrySize` for performance"

  self subclassResponsibility: #'entrySize'
%

category: 'Accessing'
method: BtreePlusNode
indexObject

  ^ self rootNode indexObject
%

category: 'Testing'
method: BtreePlusNode
isBtreePlusNode
  ^ true
%

category: 'Testing'
method: BtreePlusNode
isLeaf

"Returns whether the B-tree node is a leaf node."

^ false
%

category: 'Accessing'
method: BtreePlusNode
keyIndexOffset
  "Answer the offset from entry index for the key. Typical order is value/key/root."

  ^ 1
%

category: 'Accessing'
method: BtreePlusNode
maxNumberOfElements
  ^ self class maxNumberOfElements
%

category: 'Constants'
method: BtreePlusNode
mergeThreshold

"Returns the threshold such that if an entry has less than this number, it must
 have elements added from its sibling or be merged with its sibling."

^ self maxNumberOfElements quo: 2.
%

category: 'Accessing'
method: BtreePlusNode
numElements

"Returns the value of the instance variable 'numElements'."

^ numElements
%

category: 'Updating'
method: BtreePlusNode
numElements: newValue

"Modify the value of the instance variable 'numElements'."

numElements := newValue
%

category: 'Constants'
method: BtreePlusNode
parentNodeClass
  "Returns the class of node to be created as the parent when a split occurs."

  ^ self subclassResponsibility: #'parentNodeClass'
%

category: 'Accessing'
method: BtreePlusNode
rootIndexOffset
  "Answer the offset from entry index for the root object. Typical order is value/key/root."

  ^ 2
%

category: 'Accessing'
method: BtreePlusNode
rootNode
  "Answer the BtreePlusRoot of the receiver"

  ^ rootNode
%

category: 'Accessing'
method: BtreePlusNode
rootNode: aBtreePlusRoot
  "set the BtreePlusRoot of the receiver"

  rootNode := aBtreePlusRoot
%

category: 'Accessing'
method: BtreePlusNode
valueIndexOffset
  "Answer the offset from entry index for the value. Typical order is value/key/root."

  ^ 0
%

! Class Implementation for BtreePlusInteriorNode

! Remove existing behavior from BtreePlusInteriorNode
removeallmethods BtreePlusInteriorNode
removeallclassmethods BtreePlusInteriorNode

! ------------------- Class methods for BtreePlusInteriorNode

! ------------------- Instance methods for BtreePlusInteriorNode

category: 'Rc Updating'
method: BtreePlusInteriorNode
_at: aKey put: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil
  "Adds the key/value/root pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  "self has already been selectively aborted"

  | index node returnNode |
  index := self binarySearchInteriorNodeCoveringKey: aKey put: aValue root: rootObject  selectiveAbortSet: selectiveAbortSetOrNil.
  node := self at: index.
  self _selectiveAbort: node ifNotIn: selectiveAbortSetOrNil.
  returnNode := node
    at: aKey
    put: aValue
    root: rootObject
    selectiveAbortSet: selectiveAbortSetOrNil. "node is selectively aborted and added to rcReadSet"
  ^ self updateAfterAddAt: index child: node returnNode: returnNode selectiveAbortSet: selectiveAbortSetOrNil
%

category: 'Removing'
method: BtreePlusInteriorNode
_attemptTakingFromNeighborForNodeAt: nodeIndex

"The node at the given index has too few entries, so try to take entries from
 one of its neighboring siblings.  First try the next (right) sibling, then
 the previous (left) sibling."

| node sibling siblingEntrySize siblingLoss numSiblingElements copyStart copyEnd
fromNextSibling fromPreviousSibling targetStart lastChildKey  numToMove keyIndexOffset entrySize lastChildRoot |

node := self at: nodeIndex.
fromNextSibling := false.  " is true if taken from next sibling "
fromPreviousSibling := false.  " is true if taken from previous sibling "

siblingEntrySize := node entrySize.
entrySize := self entrySize.
keyIndexOffset := self keyIndexOffset.
" check if the node is the last entry "
nodeIndex == (self _lastKeyIndex - keyIndexOffset)
    ifFalse: [ " this is not the last entry, check the next sibling "
        sibling := self at: (nodeIndex + self entrySize).
        numSiblingElements := sibling numElements.
        numSiblingElements < (node mergeThreshold + 1)
            ifFalse: [ " can take entries from next sibling "
                fromNextSibling := true.
                siblingLoss := (numSiblingElements - node mergeThreshold) quo: 2.
                siblingLoss == 0 ifTrue: [ siblingLoss := 1 ].
                copyStart := 1.
                copyEnd := siblingLoss * siblingEntrySize.
                numToMove := 0 .
                targetStart := node _lastIndex + 1
            ]
    ].

" if could not get entries from next sibling and this is not the first entry "
(fromNextSibling not and: [ nodeIndex > 1 ] )
    ifTrue: [ " try previous entry "
        sibling := self at: (nodeIndex - self entrySize).
        numSiblingElements := sibling numElements.
        numSiblingElements < (node mergeThreshold + 1)
            ifFalse: [ " can take entries from previous sibling "
                fromPreviousSibling := true.
                siblingLoss := (numSiblingElements - node mergeThreshold) quo: 2.
                siblingLoss == 0 ifTrue: [ siblingLoss := 1 ].
                copyStart := (numSiblingElements - siblingLoss) * siblingEntrySize + 1.
                copyEnd := numSiblingElements * siblingEntrySize.
                numToMove := (node numElements) * siblingEntrySize . 
                targetStart := 1
            ]
    ].

" if cannot take from previous or next sibling "
(fromPreviousSibling or:[ fromNextSibling ])
    ifFalse: [ ^ false ].

" copy from sibling and insert into node "
node _insertAt: targetStart 
     from: sibling fromStart: copyStart fromEnd: copyEnd
     numToMoveDown: numToMove .

node numElements: (node numElements + siblingLoss).

fromNextSibling
    ifTrue: [ 
        "entries added to end of node from beginning of sibling"
        sibling _deleteNoShrinkFrom: copyStart to: copyEnd .

        sibling numElements: (numSiblingElements - siblingLoss).

        " update the receiver's key for this entry "
        lastChildKey := node at: (node _lastKeyIndex).
        self primitiveAt: (nodeIndex + keyIndexOffset) put: lastChildKey.
        " see if last root in child node has changed "
        lastChildRoot := node at: node _lastRootIndex.
        self updateLastChildRoot: lastChildRoot at: nodeIndex.
        node _updateLastValue
    ].

fromPreviousSibling
    ifTrue: [
        "entries added to beginning of node from end of sibling"
        sibling numElements: (numSiblingElements - siblingLoss).

        " update the receiver's key for previous entry "
        lastChildKey := sibling at: sibling _lastKeyIndex .
        self  primitiveAt: (nodeIndex + keyIndexOffset - entrySize) put: lastChildKey.
        " see if last root in child node has changed "
        lastChildRoot := sibling at: sibling _lastRootIndex.
        self updateLastChildRoot: lastChildRoot at: nodeIndex - entrySize.
        sibling _updateLastValue .

       " nil out sibling entries at end "
        sibling _deleteNoShrinkFrom: copyStart to: copyEnd .
    ].

^ true
%

category: 'Rc Removing'
method: BtreePlusInteriorNode
_attemptTakingFromNeighborForNodeAt: nodeIndex selectiveAbortSet: selectiveAbortSetOrNil

"The node at the given index has too few entries, so try to take entries from
 one of its neighboring siblings.  First try the next (right) sibling, then
 the previous (left) sibling."

| node sibling siblingEntrySize siblingLoss numSiblingElements copyStart copyEnd
fromNextSibling fromPreviousSibling targetStart lastChildKey numToMove keyIndexOffset entrySize lastChildRoot |

node := self at: nodeIndex.
self _selectiveAbort: node ifNotIn: selectiveAbortSetOrNil.
fromNextSibling := false.  " is true if taken from next sibling "
fromPreviousSibling := false.  " is true if taken from previous sibling "

siblingEntrySize := node entrySize.
entrySize := self entrySize.
keyIndexOffset := self keyIndexOffset.
" check if the node is the last entry "
nodeIndex == (self _lastKeyIndex - keyIndexOffset)
    ifFalse: [ " this is not the last entry, check the next sibling "
        sibling := self at: (nodeIndex + self entrySize).
        self _selectiveAbort: sibling ifNotIn: selectiveAbortSetOrNil.
        numSiblingElements := sibling numElements.
        numSiblingElements < (node mergeThreshold + 1)
            ifFalse: [ " can take entries from next sibling "
                fromNextSibling := true.
                siblingLoss := (numSiblingElements - node mergeThreshold) quo: 2.
                siblingLoss == 0 ifTrue: [ siblingLoss := 1 ].
                copyStart := 1.
                copyEnd := siblingLoss * siblingEntrySize.
                numToMove := 0 .
                targetStart := node _lastIndex + 1
            ]
    ].

" if could not get entries from next sibling and this is not the first entry "
(fromNextSibling not and: [ nodeIndex > 1 ] )
    ifTrue: [ " try previous entry "
        sibling := self at: (nodeIndex - self entrySize).
        self _selectiveAbort: sibling ifNotIn: selectiveAbortSetOrNil.
        numSiblingElements := sibling numElements.
        numSiblingElements < (node mergeThreshold + 1)
            ifFalse: [ " can take entries from previous sibling "
                fromPreviousSibling := true.
                siblingLoss := (numSiblingElements - node mergeThreshold) quo: 2.
                siblingLoss == 0 ifTrue: [ siblingLoss := 1 ].
                copyStart := (numSiblingElements - siblingLoss) * siblingEntrySize + 1.
                copyEnd := numSiblingElements * siblingEntrySize.
                numToMove := (node numElements) * siblingEntrySize . 
                targetStart := 1
            ]
    ].

" if cannot take from previous or next sibling "
(fromPreviousSibling or:[ fromNextSibling ])
    ifFalse: [ ^ false ].

" copy from sibling and insert into node "
node _insertAt: targetStart 
     from: sibling fromStart: copyStart fromEnd: copyEnd
     numToMoveDown: numToMove .

node numElements: (node numElements + siblingLoss).

fromNextSibling
    ifTrue: [ 
        "entries added to end of node from beginning of sibling"
        sibling _deleteNoShrinkFrom: copyStart to: copyEnd .

        sibling numElements: (numSiblingElements - siblingLoss).

        " update the receiver's key for this entry "
        lastChildKey := node at: (node _lastKeyIndex).
        self primitiveAt: (nodeIndex + keyIndexOffset) put: lastChildKey.
        " see if last root in child node has changed "
        lastChildRoot := node at: node _lastRootIndex.
        self updateLastChildRoot: lastChildRoot at: nodeIndex.
        node _updateLastValue: selectiveAbortSetOrNil
    ].

fromPreviousSibling
    ifTrue: [
        "entries added to beginning of node from end of sibling"
        sibling numElements: (numSiblingElements - siblingLoss).

        " update the receiver's key for previous entry "
        lastChildKey := sibling at: sibling _lastKeyIndex .
        self  primitiveAt: (nodeIndex + keyIndexOffset - entrySize) put: lastChildKey.
        " see if last root in child node has changed "
        lastChildRoot := sibling at: sibling _lastRootIndex.
        self updateLastChildRoot: lastChildRoot at: nodeIndex - entrySize.
        sibling _updateLastValue: selectiveAbortSetOrNil .

       " nil out sibling entries at end "
        sibling _deleteNoShrinkFrom: copyStart to: copyEnd .
    ].

self rootNode 
  _addBtreePlusNodeToRcReadSet: node;
  _addBtreePlusNodeToRcReadSet: sibling.
^ true
%

category: 'Searching Support'
method: BtreePlusInteriorNode
_findAllValuesGreaterThanKey: aKey into: anArray using: aComparison
  "Puts into anArray the index and the leaf node of the entry that is greater
 than the given key, plus the index and interior node of the path taken to
 reach the leaf.  Returns true if a value was found, false otherwise.  Do this
 by invoking the same operation on the appropriate child node."

  | index eSize lastKeyIndex keyIndexOffset comparisonForSort |
  lastKeyIndex := self _lastKeyIndex.
  lastKeyIndex == 0
    ifTrue: [ ^ false ].
  " find index where key would be inserted "
  comparisonForSort := self comparisonForSort.
  index := self _findCoveringIndexForKey: aKey totalOrder: false  using: comparisonForSort.
  index == 0
    ifTrue: [ ^ false ].
  "scan across all entries that sort ="
  eSize := self entrySize.
  keyIndexOffset := self keyIndexOffset.
  [ comparisonForSort compareKey: aKey equalTo: (self at: index + keyIndexOffset) ]
    whileTrue: [ index := index + eSize.
      index > lastKeyIndex
        ifTrue: [ ^ false ] ].
  " see if this child has any greater than the key "
  ((self at: index)
    _findAllValuesGreaterThanKey: aKey
    into: anArray
    using: aComparison)
    ifFalse: [ ^ false ].
  ^ true
%

category: 'Searching Support'
method: BtreePlusInteriorNode
_findAllValuesLessThanKey: aKey andEquals: aBoolean into: anArray using: aComparison
  "Puts into anArray the index and the leaf node of the entry that is less than
 (or less than and equal to if aBoolean is true) the given key, plus the index
 and interior node of the path taken to reach the leaf.  Returns true if a
 value was found, false otherwise.  Does this by invoking the same operation
 on the appropriate child node."

  | index eSize lastIndex lastKeyIndex comparisonForSort keyIndexOffset | 
  lastIndex := self _lastIndex.
  lastIndex == 0
    ifTrue: [ ^ false ].
  aBoolean
    ifFalse: [ ^ self _findAllValuesLessThanKey: aKey into: anArray using: aComparison ].
  " first see if all entries are > aKey "
  lastKeyIndex := self _lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (comparisonForSort
    compareKey: aKey
    lessThanOrEqualTo: (self at: lastKeyIndex))
    ifFalse: [ "bugfix 33805 - verify that aKey is > last entry"
      (aKey == nil
        or: [ comparisonForSort compareKey: aKey greaterThan: (self at: lastKeyIndex) ])
        ifTrue: [ self _putLastIndexOfLastChildInto: anArray.
          ^ true ] ].
  "find index of first entry >= aKey"
  index := self _findCoveringIndexForKey: aKey totalOrder: false using: comparisonForSort.
  index == 0
    ifTrue: [ ^ false ].
  "scan across all of the values that sort ="
  eSize := self entrySize.
  keyIndexOffset := self keyIndexOffset.
  (comparisonForSort compareKey: aKey equalTo: (self at: index + keyIndexOffset))
    ifTrue: [
      [ index < lastIndex
        and: [ comparisonForSort compareKey: aKey equalTo: (self at: index + keyIndexOffset) ] ]
        whileTrue: [ index := index + eSize ].
      " if we scanned past the end, subtract an entry "
      index > lastIndex
        ifTrue: [ index := index - eSize ] ].
  " see if child has any entries less than or equal to aKey "
  ((self at: index)
    _findAllValuesLessThanKey: aKey
    andEquals: true
    into: anArray
    using: comparisonForSort)
    ifFalse: [ " this child did not find any "
      " if it is not the first child "
      index > 1
        ifTrue: [ " then the entry is the rightmost leaf entry of the previous child "
          index := index - eSize.
          (self at: index) _putLastIndexOfLastChildInto: anArray ]
        ifFalse: [ ^ false ] ].
  ^ true
%

category: 'Searching Support'
method: BtreePlusInteriorNode
_findAllValuesLessThanKey: aKey into: anArray using: aComparison
  "Puts into anArray the index and the leaf node of the entry that is less than
 (or less than and equal to if aBoolean is true) the given key, plus the index
 and interior node of the path taken to reach the leaf.  Returns true if a
 value was found, false otherwise.  Does this by invoking the same operation
 on the appropriate child node."

  | index eSize lastIndex lastKeyIndex comparisonForSort |
  lastIndex := self _lastIndex.
  lastIndex == 0
    ifTrue: [ ^ false ].
  " first see if all entries are > aKey "
  lastKeyIndex := self _lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (comparisonForSort
    compareKey: aKey
    lessThanOrEqualTo: (self at: lastKeyIndex))
    ifFalse: [ "bugfix 33805 - verify that aKey is > last entry"
      (aKey == nil
        or: [ aComparison compareKey: aKey greaterThan: (self at: lastKeyIndex) ])
        ifTrue: [ self _putLastIndexOfLastChildInto: anArray.
          ^ true ] ].
  " find index where key would be inserted "
  index := self _findCoveringIndexForKey: aKey totalOrder: false using: comparisonForSort.
  index == 0
    ifTrue: [ ^ false ].
  eSize := self entrySize.
  " see if this child has any less than the key "
  ((self at: index)
    _findAllValuesLessThanKey: aKey
    into: anArray
    using: aComparison)
    ifFalse: [ " this child did not find any "
      " if it is not the first child "
      index > 1
        ifTrue: [ " then the entry is the rightmost leaf entry of the previous child "
          index := index - eSize.
          (super at: index) _putLastIndexOfLastChildInto: anArray ]
        ifFalse: [ ^ false ] ].
  ^ true
%

category: 'Removing'
method: BtreePlusInteriorNode
_mergeFromNeighborForNodeAt: nodeIndex

"The node at the given index has too few entries and neighboring siblings had
 too few entries to take from, so merge with one of its neighboring siblings.
 First tries the next (right) sibling, then the previous (left) sibling."

| node sibling eSize myESize numSiblingElements copyStart lastChildKey myLastIndex keyIndexOffset lastChildRoot |
node := self at: nodeIndex.
eSize := node entrySize.
myESize := self entrySize.
keyIndexOffset := self keyIndexOffset.

" check if the node is the last entry "
nodeIndex == self _lastEntryIndex
    ifTrue: [ " this is the last entry, try the previous sibling "
        nodeIndex > 1
            ifTrue: [ " keep sibling, throw away node "
                sibling := self at: (nodeIndex - myESize).
                numSiblingElements := sibling numElements.

                " move node's entries onto end of sibling "
                sibling _insertAt: (sibling _lastIndex + keyIndexOffset)
                   from: node fromStart: 1 fromEnd: (node numElements * eSize)
                   numToMoveDown: 0 . 

                sibling numElements: (node numElements + numSiblingElements).

                " update the key for the sibling's entry in the receiver "
                lastChildKey := sibling at: (sibling _lastKeyIndex).
                self primitiveAt: (nodeIndex - myESize + keyIndexOffset) put: lastChildKey.
                " see if last root in child node has changed "
                lastChildRoot := sibling at: sibling _lastRootIndex.
                self updateLastChildRoot: lastChildRoot at: nodeIndex - myESize.
                sibling _updateLastValue.
                "update nextLeaf of sibling and update previousLeaf of node nextLeaf"
                sibling nextLeaf: node nextLeaf.
                node nextLeaf ifNotNil: [:nLeaf | nLeaf previousLeaf: sibling ] ]
            ifFalse:[ self _error: #rtErrBtreeInvalidMerge.
                      self _uncontinuableError ].
        numElements := numElements - 1. "update here -- when losing node (next clause), need to update numElements mid-stream"
    ]
    " this is not the last entry, get the next sibling "
    ifFalse: [ " keep node, throw away sibling "
        sibling := self at: (nodeIndex + myESize).
        numSiblingElements := sibling numElements.

        "copy from sibling onto end of node "
        node _insertAt: (node _lastIndex + keyIndexOffset) 
             from: sibling fromStart: 1 fromEnd: (numSiblingElements * eSize)
             numToMoveDown: 0 .

        copyStart := nodeIndex + myESize + myESize.
        " if sibling was not the last entry in the receiver "
        myLastIndex := self _lastIndex .
        copyStart < myLastIndex
            ifTrue: [
                " move entries in receiver to the left "
                self _deleteNoShrinkFrom: (nodeIndex + myESize) to: copyStart - 1 
            ].

        numElements := numElements - 1. "subsequent _last*Index calculation needs to be correct"
        node numElements: (node numElements + numSiblingElements).

        " update the key for the node's entry in the receiver "
        lastChildKey := node at: (node _lastKeyIndex).
        self primitiveAt: (nodeIndex + keyIndexOffset) put: lastChildKey.
        " see if last root in child node has changed "
        lastChildRoot := node at: node _lastRootIndex.
        self updateLastChildRoot: lastChildRoot at: nodeIndex.
        node _updateLastValue.
        "update nextLeaf of node using the next element in the receiver"
        nodeIndex == self _lastEntryIndex
          ifTrue: [ "update nextLeaf of node and update previousLeaf of sibling nextLeaf"
            node nextLeaf: sibling nextLeaf.
            sibling nextLeaf ifNotNil: [:nLeaf | nLeaf previousLeaf: node ] ]
          ifFalse: [ "sibling is not the last entry so splice in sibling's sibling"
            | newNextLeaf |
            newNextLeaf := self at: (nodeIndex + myESize).
            node nextLeaf: newNextLeaf.
            newNextLeaf previousLeaf: node ] ].

" nil out the last entry of the receiver, we've either explicitly dropped last entry, or
  we slid the entries down in receiver and need to delete duplicate last entry"
myLastIndex := self _lastIndex .
self _deleteNoShrinkFrom: myLastIndex + 1 to: myLastIndex + myESize.
self _updateLastValue.
%

category: 'Rc Removing'
method: BtreePlusInteriorNode
_mergeFromNeighborForNodeAt: nodeIndex selectiveAbortSet: selectiveAbortSetOrNil

"The node at the given index has too few entries and neighboring siblings had
 too few entries to take from, so merge with one of its neighboring siblings.
 First tries the next (right) sibling, then the previous (left) sibling."

| node sibling eSize myESize numSiblingElements copyStart lastChildKey myLastIndex keyIndexOffset lastChildRoot |

node := self at: nodeIndex.
self _selectiveAbort: node ifNotIn: selectiveAbortSetOrNil.
eSize := node entrySize.
myESize := self entrySize.
keyIndexOffset := self keyIndexOffset.
" check if the node is the last entry "
nodeIndex == self _lastEntryIndex
    ifTrue: [ " this is the last entry, try the previous sibling "
        nodeIndex > 1
            ifTrue: [ " keep sibling, throw away node "
                sibling := self at: (nodeIndex - myESize).
                self _selectiveAbort: sibling ifNotIn: selectiveAbortSetOrNil.
                numSiblingElements := sibling numElements.

                " move node's entries onto end of sibling "
                sibling _insertAt: (sibling _lastIndex + keyIndexOffset)
                   from: node fromStart: 1 fromEnd: (node numElements * eSize)
                   numToMoveDown: 0 . 

                sibling numElements: (node numElements + numSiblingElements).

                " update the key for the sibling's entry in the receiver "
                lastChildKey := sibling at: (sibling _lastKeyIndex).
                self primitiveAt: (nodeIndex - myESize + keyIndexOffset) put: lastChildKey.
                " see if last root in child node has changed "
                lastChildRoot := sibling at: sibling _lastRootIndex.
                self updateLastChildRoot: lastChildRoot at: nodeIndex - myESize.
                sibling _updateLastValue: selectiveAbortSetOrNil.
                "update nextLeaf of sibling and update previousLeaf of node nextLeaf"
                sibling nextLeaf: node nextLeaf.
                node nextLeaf ifNotNil: [:nLeaf | 
                  self _selectiveAbort: nLeaf ifNotIn: selectiveAbortSetOrNil.
                  nLeaf previousLeaf: sibling.
                  self rootNode _addBtreePlusNodeToRcReadSet: nLeaf ].
                self rootNode _addBtreePlusNodeToRcReadSet: sibling.
                "write dropped node to force a commit conflict and force replay, in case
                 another session makes mods to the node"
                node numElements: node numElements.
                self rootNode _addBtreePlusNodeToRcReadSet: node ]
            ifFalse:[ self _error: #rtErrBtreeInvalidMerge.
                      self _uncontinuableError ].
        numElements := numElements - 1. "update here -- when losing node (next clause), need to update numElements mid-stream"
    ]
    " this is not the last entry, get the next sibling "
    ifFalse: [ " keep node, throw away sibling "
        sibling := self at: (nodeIndex + myESize).
        self _selectiveAbort: sibling ifNotIn: selectiveAbortSetOrNil.
        numSiblingElements := sibling numElements.

        "copy from sibling onto end of node "
        node _insertAt: (node _lastIndex + keyIndexOffset) 
             from: sibling fromStart: 1 fromEnd: (numSiblingElements * eSize)
             numToMoveDown: 0 .

        copyStart := nodeIndex + myESize + myESize.
        " if sibling was not the last entry in the receiver "
        myLastIndex := self _lastIndex .
        copyStart < myLastIndex
            ifTrue: [
                " move entries in receiver to the left "
                self _deleteNoShrinkFrom: (nodeIndex + myESize) to: copyStart - 1 
            ].

        numElements := numElements - 1. "subsequent _last*Index calculation needs to be correct"
        node numElements: (node numElements + numSiblingElements).

        " update the key for the node's entry in the receiver "
        lastChildKey := node at: (node _lastKeyIndex).
        self primitiveAt: (nodeIndex + keyIndexOffset) put: lastChildKey.
        " see if last root in child node has changed "
        lastChildRoot := node at: node _lastRootIndex.
        self updateLastChildRoot: lastChildRoot at: nodeIndex.
        node _updateLastValue: selectiveAbortSetOrNil.
        "update nextLeaf of node using the next element in the receiver"
        nodeIndex == self _lastEntryIndex
          ifTrue: [ "update nextLeaf of node and update previousLeaf of sibling nextLeaf"
            node nextLeaf: sibling nextLeaf.
            sibling nextLeaf ifNotNil: [:nLeaf | 
              self _selectiveAbort: nLeaf ifNotIn: selectiveAbortSetOrNil.
              nLeaf previousLeaf: node.
              self rootNode _addBtreePlusNodeToRcReadSet: nLeaf ] ]
          ifFalse: [ "sibling is not the last entry so splice in sibling's sibling"
            | newNextLeaf |
            newNextLeaf := self at: (nodeIndex + myESize).
            self _selectiveAbort: newNextLeaf ifNotIn: selectiveAbortSetOrNil.
            node nextLeaf: newNextLeaf.
            newNextLeaf previousLeaf: node.
            self rootNode _addBtreePlusNodeToRcReadSet: newNextLeaf ].
        self rootNode _addBtreePlusNodeToRcReadSet: node.
        "write dropped sibling to force a commit conflict and force replay, in case
         another session makes mods to the node"
        sibling numElements: sibling numElements.
        self rootNode _addBtreePlusNodeToRcReadSet: sibling  ].

" nil out the last entry of the receiver, we've either explicitly dropped last entry, or
  we slid the entries down in receiver and need to delete duplicate last entry"
myLastIndex := self _lastIndex .
self _deleteNoShrinkFrom: myLastIndex + 1 to: myLastIndex + myESize.
self _updateLastValue: selectiveAbortSetOrNil.
%

category: 'Searching Support'
method: BtreePlusInteriorNode
_putFirstIndexOfFirstChildInto: anArray
  "Ask the first child in the receiver to
 putFirstIndexOfFirstChildInto: anArray."

  (self at: 1) _putFirstIndexOfFirstChildInto: anArray
%

category: 'Searching Support'
method: BtreePlusInteriorNode
_putFirstIndexOfFirstChildInto: anArray ifGreaterThanOrEqualTo: aKey using: aComparison
  "Ask the first child in the receiver to putFirstIndexOfFirstChildInto:ifGreaterThanOrEqualTo:.

 Because of NANs we need to validate the assumption of the caller 
 (i.e., that all entries are greater than aKey). See bug33805."

  "make sure that key is less than or equal to first entry"

  (aKey == nil
    or: [ aComparison compareKey: aKey lessThanOrEqualTo: (self at: 1 + self keyIndexOffset) ])
    ifTrue: [ (self at: 1)
        _putFirstIndexOfFirstChildInto: anArray
        ifGreaterThanOrEqualTo: aKey
        using: aComparison ]
    ifFalse: [ anArray addLast: 0 ]
%

category: 'Searching Support'
method: BtreePlusInteriorNode
_putLastIndexOfLastChildInto: anArray
  "Asks the last child in the receiver to
 putLastIndexOfLastChildInto: anArray ."

  | index |
  index := self _lastKeyIndex - self keyIndexOffset.
  (self at: index)
    _putLastIndexOfLastChildInto: anArray.
%

category: 'Updating'
method: BtreePlusInteriorNode
_updateLastValue
  "Update the last value of the interior node by asking the last entry what its
 last value is.  Do not change the value unless there has been a change."

  | nodeLastValue nodeLastRoot   lastChildNode |
  lastChildNode := self at: self _lastEntryIndex.
  nodeLastValue := lastChildNode lastValue.
  nodeLastValue == self lastValue
    ifFalse: [ self lastValue: nodeLastValue ].
  nodeLastRoot := lastChildNode lastRoot.
  nodeLastRoot == self lastRoot
    ifFalse: [ self lastRoot: nodeLastRoot ]
%

category: 'Updating'
method: BtreePlusInteriorNode
at: aKey put: aValue root: rootObject

  "Adds the key/value/root tuple to one of the leaf nodes.  If a split occurs, returns
   the new sibling, otherwise returns the receiver."

  | index node returnNode |
  index := self binarySearchInteriorNodeCoveringKey: aKey put: aValue root: rootObject.
  node := self at: index.
  returnNode := node at: aKey put: aValue root: rootObject.
  ^ self updateAfterAddAt: index child: node returnNode: returnNode
%

category: 'Rc Updating'
method: BtreePlusInteriorNode
at: aKey put: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil
  "Adds the key/value/root pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  | returnNode |
  self _selectiveAbort: self ifNotIn: selectiveAbortSetOrNil.
  returnNode := self _at: aKey put: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil.
  returnNode ~~ self 
    ifTrue: [
      self rootNode _addBtreePlusNodeToRcReadSet: returnNode ].
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  ^ returnNode
%

category: 'Audit'
method: BtreePlusInteriorNode
auditForPathTerm: pathTerm using: auditor parentNode: parentNode previousChild: previousSibling nextChild: nextSibling do: aBlock
  "Audit the receiver's entries and each child node"

  | sentinel previousKey comparisonForSort keyIndexOffset rootIndexOffset lastEntryIndex lastChildNode previousChild nextChild eSize child |
  sentinel := Object new.
  previousKey := sentinel.
  child := previousChild := nextChild := nil.
  previousSibling ifNotNil: [ child := previousSibling at: previousSibling _lastEntryIndex ].
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  rootIndexOffset := self rootIndexOffset.
  lastEntryIndex := self _lastEntryIndex.
  lastChildNode := self at: lastEntryIndex.
  eSize := self entrySize.
  lastChildNode lastValue == self lastValue
    ifFalse: [ auditor
        btreeInteriorNode: self
        childConsistencyError: pathTerm
        lastValue: self lastValue
        lastChildValue: lastChildNode lastValue
        childNode: lastChildNode ].
  lastChildNode lastRoot == self lastRoot
    ifFalse: [ auditor
        btreeInteriorNode: self
        childConsistencyError: pathTerm
        lastRoot: self lastRoot
        lastChildRoot: lastChildNode lastRoot
        childNode: lastChildNode].
  1 to: numElements * self entrySize by: self entrySize do: [ :i | | k r lastChildKey lastChildRoot |
    k := self at: i + keyIndexOffset.
    r := self at: i + rootIndexOffset.
    previousKey == sentinel
      ifFalse: [ (comparisonForSort compareKey: previousKey lessThanOrEqualTo: k)
          ifFalse: [ auditor
              btreeInteriorNode: self
              keyOrderError: pathTerm
              key: k
              previousKey: previousKey 
              index: i ] ].
    previousKey := k.
    previousChild := child.
    (i + eSize) <= lastEntryIndex
      ifTrue: [ nextChild := self at: i + eSize ]
      ifFalse: [ 
        nextChild := nil.
        nextSibling ifNotNil: [ nextChild := nextSibling at: 1 ] ]. 
    child := self at: i.
    lastChildKey := child at: child _lastKeyIndex.
    lastChildKey == k
      ifFalse: [ auditor
          btreeInteriorNode: self
          childConsistencyError: pathTerm
          key: k
          lastChildKey: lastChildKey
          childNode: child
          index: i ].
    lastChildRoot := child at: child _lastRootIndex.
    lastChildRoot == r
      ifFalse: [ auditor
          btreeInteriorNode: self
          childConsistencyError: pathTerm
          key: k
          root: r
          lastChildRoot: lastChildRoot
          childNode: child
          index: i ].
    child auditForPathTerm: pathTerm using: auditor parentNode: self previousChild: previousChild nextChild: nextChild do: aBlock ]
%

category: 'Updating'
method: BtreePlusInteriorNode
binarySearchInteriorNodeCoveringKey: aKey put: aValue root: rootObject

  "Find the index of the first child node in which the insertion would occur"

  | index lastKeyIndex |
  index := self binarySearchCoveringKey: aKey value: aValue root: rootObject.
  lastKeyIndex := self _lastKeyIndex.	
  " if insertion would occur past the last entry, use the last entry "
  index > lastKeyIndex
    ifTrue: [ index := lastKeyIndex - self keyIndexOffset ].
  ^ index.
%

category: 'Comparison Operators'
method: BtreePlusInteriorNode
compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex
  "Performs a = comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, use the OOP of the value as the next basis for
 comparison."

  | entryIndex |
  entryIndex := keyIndex - self keyIndexOffset.
  ^ (comparisonForSort compareKey: aKey equalTo: (self at: keyIndex))
    and: [ 
      "aValue asOop = (self at: entryIndex) lastValue asOop"
      aValue == (self at: entryIndex) lastValue ]
%

category: 'Comparison Operators'
method: BtreePlusInteriorNode
compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex root: rootObject
  "Performs a = comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, use the OOP of the value as the next basis for
 comparison. If the value oops are equeal, use the oop of the rootObject as the
 next basis for comparison."

  ^ (comparisonForSort compareKey: aKey equalTo: (self at: keyIndex))
    and: [ | entryIndex |
      entryIndex := keyIndex - self keyIndexOffset.
      "aValue asOop = (self at: entryIndex) lastValue asOop"
      aValue == (self at: entryIndex) lastValue
        and: [ 
          "rootObject asOop = (self at: entryIndex) lastRoot asOop"
          rootObject == (self at: entryIndex) lastRoot ] ]
%

category: 'Comparison Operators'
method: BtreePlusInteriorNode
compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index. If the keys are equal, then use the OOP of the value for the
 comparison."

  | keyIndexEntry |
  keyIndexEntry := self at: keyIndex.
  (comparisonForSort compareKey: aKey lessThan: keyIndexEntry)
    ifTrue: [ ^ true ].
  (comparisonForSort compareKey: aKey equalTo: keyIndexEntry)
    ifTrue: [ | entryIndex |
      entryIndex := keyIndex - self keyIndexOffset.
      ^ aValue asOop <= (self at: entryIndex) lastValue asOop ].
  ^ false
%

category: 'Comparison Operators'
method: BtreePlusInteriorNode
compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: rootObject
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, then use the OOP of the value for the
 comparison. If the value oops are equal, use the OOP of the rootObject for the 
 comparison."

  | keyIndexEntry |
  keyIndexEntry := self at: keyIndex.
  (comparisonForSort compareKey: aKey lessThan: keyIndexEntry)
    ifTrue: [ ^ true ].
  (comparisonForSort compareKey: aKey equalTo: keyIndexEntry)
    ifTrue: [ | entryIndexEntry entryIndexEntryLastEntry |
      entryIndexEntry := self at: keyIndex - self keyIndexOffset.
      "aValue asOop = entryIndexEntry lastValue asOop"
      entryIndexEntryLastEntry := entryIndexEntry lastValue.
      aValue == entryIndexEntryLastEntry
        ifTrue: [ ^ rootObject asOop <= entryIndexEntry lastRoot asOop ].
      ^ aValue asOop < entryIndexEntryLastEntry asOop ].
  ^ false
%

category: 'Comparison Operators'
method: BtreePlusInteriorNode
compareLastKeyLessThanOrEqualToKey: aKey value: aValue root: rootObject
  "Returns whether the receivers last key/value pair is greater than or equal to
the given key/value pair."

  numElements == 0
    ifTrue: [ ^ false ].
  " return aKey <= last key "
  ^ self
    compare: self comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: self _lastKeyIndex
    root: rootObject
%

category: 'Comparison Operators'
method: BtreePlusInteriorNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  ^ aValue asOop > (self at: keyIndex - self keyIndexOffset) lastValue asOop
%

category: 'Comparison Operators'
method: BtreePlusInteriorNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  ^ aValue asOop <= (self at: keyIndex - self keyIndexOffset) lastValue asOop
%

category: 'Accessing'
method: BtreePlusInteriorNode
lastRoot
  ^ lastRoot
%

category: 'Updating'
method: BtreePlusInteriorNode
lastRoot: anObject
  lastRoot := anObject
%

category: 'Accessing'
method: BtreePlusInteriorNode
lastValue

"Returns the value of the instance variable 'lastValue'."

^ lastValue
%

category: 'Updating'
method: BtreePlusInteriorNode
lastValue: newValue

"Modifies the value of the instance variable 'lastValue'."

lastValue := newValue
%

category: 'Constants'
method: BtreePlusInteriorNode
parentNodeClass
  "Returns the class of node to be created as the parent when a split occurs."

  ^ self class
%

category: 'Updating'
method: BtreePlusInteriorNode
primitiveAt: anIndex put: aValue

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds, if the receiver is not indexable, or if the
 receiver is not of the right class to store the given value.

 The primitive is equivalent to GciStoreIdxOop or GciStoreByte,
 depending on implementation of the receiver."

<primitive: 268>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
(self class isIndexable) "not an indexable object"
  ifFalse: [ ^ self _errorNotIndexable].
(anIndex < 1 or:[ anIndex > (self size + 1)]) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].
(self class isBytes)
ifTrue: [ ((aValue class ~~ SmallInteger) or: [ (aValue < 0) | (aValue > 255) ])
    ifTrue: [^ aValue _error: #rtErrExpectedByteValue].
  ].

self _primitiveFailed: #at:put: args: { anIndex . aValue }.
self _uncontinuableError
%

category: 'Removing'
method: BtreePlusInteriorNode
removedEntryFromNode: node atIndex: nodeIndex
  "Remove occurred, check if a merge or coalesce is necessary."

  | keyIndexOffset lastChildKeyIndex lastChildKey lastChildRoot |
  keyIndexOffset := self keyIndexOffset.
  " see if last key in child node has changed "
  lastChildKeyIndex := node _lastKeyIndex.
  lastChildKey := node at: lastChildKeyIndex.
  lastChildKey == (self at: nodeIndex + keyIndexOffset)
    ifFalse: [ " update key and encryption in the receiver "
      self primitiveAt: nodeIndex + keyIndexOffset put: lastChildKey ].
  " see if the node's number of entries is below the threshold "
  node numElements < node mergeThreshold
    ifTrue: [ " try to steal entries from a neighboring sibling "
      (self _attemptTakingFromNeighborForNodeAt: nodeIndex)
        ifFalse: [ " could not take from neighbor, merge with a neighbor instead. lastChildRoot and lastValue updated"
          self _mergeFromNeighborForNodeAt: nodeIndex.
          ^ self  ] ].
  " see if last root in child node has changed "
  lastChildRoot := node at: node _lastRootIndex.
  self updateLastChildRoot: lastChildRoot at: nodeIndex.
  self _updateLastValue.
%

category: 'Rc Removing'
method: BtreePlusInteriorNode
removedEntryFromNode: node atIndex: nodeIndex selectiveAbortSet: selectiveAbortSetOrNil
  "Remove occurred, check if a merge or coalesce is necessary."

  | keyIndexOffset lastChildKeyIndex lastChildKey lastChildRoot |
  keyIndexOffset := self keyIndexOffset.
  " see if last key in child node has changed "
  lastChildKeyIndex := node _lastKeyIndex.
  lastChildKey := node at: lastChildKeyIndex.
  lastChildKey == (self at: nodeIndex + keyIndexOffset)
    ifFalse: [ " update key and encryption in the receiver "
      self primitiveAt: nodeIndex + keyIndexOffset put: lastChildKey ].
  " see if the node's number of entries is below the threshold "
  node numElements < node mergeThreshold
    ifTrue: [ " try to steal entries from a neighboring sibling "
      (self _attemptTakingFromNeighborForNodeAt: nodeIndex selectiveAbortSet: selectiveAbortSetOrNil)
        ifFalse: [ " could not take from neighbor, merge with a neighbor instead. lastChildRoot and lastValue updated"
          self _mergeFromNeighborForNodeAt: nodeIndex selectiveAbortSet: selectiveAbortSetOrNil.
          ^ self  ] ].
  " see if last root in child node has changed "
  lastChildRoot := node at: node _lastRootIndex.
  self updateLastChildRoot: lastChildRoot at: nodeIndex.
  self _updateLastValue: selectiveAbortSetOrNil.
%

category: 'Removing'
method: BtreePlusInteriorNode
removeKey: aKey value: aValue root: rootObject
  "Removes the key/value pair.  Find the appropriate child node and invoke the
 remove operation on it.  If the remove occurred, check if a merge or coalesce
 is necessary."

  | nodeIndex node |
  (self compareLastKeyLessThanOrEqualToKey: aKey value: aValue root: rootObject)
    ifFalse: [ ^ false ].
  nodeIndex := self
    binarySearchCoveringKey: aKey
    value: aValue
    root: rootObject.
  node := self at: nodeIndex.
  ^ (node removeKey: aKey value: aValue root: rootObject)
    ifTrue: [ " remove occurred - merge or coalesce if necessary"
      self removedEntryFromNode: node atIndex: nodeIndex.
      true ]
    ifFalse: [ false ]
%

category: 'Rc Removing'
method: BtreePlusInteriorNode
removeKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil
  "Removes the key/value pair.  Find the appropriate child node and invoke the
 remove operation on it.  If the remove occurred, check if a merge or coalesce
 is necessary."

  | res nodeIndex node |
  self _selectiveAbort: self ifNotIn: selectiveAbortSetOrNil.
   (self compareLastKeyLessThanOrEqualToKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil)
    ifFalse: [ ^ false ].
  nodeIndex := self
    binarySearchCoveringKey: aKey
    value: aValue
    root: rootObject
    selectiveAbortSet: selectiveAbortSetOrNil.
  node := self at: nodeIndex.
  self _selectiveAbort: node ifNotIn: selectiveAbortSetOrNil.
  res := (node removeKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil)
    ifTrue: [ " remove occurred - merge or coalesce if necessary"
      self removedEntryFromNode: node atIndex: nodeIndex selectiveAbortSet: selectiveAbortSetOrNil.
      true ]
    ifFalse: [ false ].
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  ^ res
%

category: 'Updating'
method: BtreePlusInteriorNode
updateAfterAddAt: index child: node returnNode: returnNode

  "Adds the key/value/root tuple to one of the leaf nodes.  If a split occurs, returns
   the new sibling, otherwise returns the receiver."

  | lastChildKey keyIndexOffset lastChildRoot |

  " see if last key in child node has changed "
  lastChildKey := node at: node _lastKeyIndex.
  keyIndexOffset := self keyIndexOffset.
  lastChildKey == (self at: index + keyIndexOffset)
    ifFalse: [ " update key and encryption in the receiver "
      self primitiveAt: index + keyIndexOffset put: lastChildKey ].
  " see if last root in child node has changed "
  lastChildRoot := node at: node _lastRootIndex.
  self updateLastChildRoot: lastChildRoot at: index.


  " see if a split occurred "
  returnNode == node
    ifTrue: [ 
      " if inserted into the last node, update the last value "
      index == self _lastEntryIndex
        ifTrue: [ self _updateLastValue ] ]
    ifFalse: [ " returnNode is the second half of the split "
      " determine if a split is needed in the receiver "
      numElements == self maxNumberOfElements
        ifTrue: [ | newSibling |
          newSibling := self
            _splitUsingKey: (node at: node _lastKeyIndex)
            node: node 
            newSibling: returnNode.

          " update the last value and last root of the new sibling and the receiver "
          newSibling _updateLastValue.
          self _updateLastValue.
          ^ newSibling ]
        ifFalse: [ " entry will fit in the receiver "

          " if child node was the last node "
          index == self _lastEntryIndex
            ifTrue: [
              self lastValue: returnNode lastValue.
              self lastRoot: returnNode lastRoot ].
          self
            _insertKey: (returnNode at: returnNode _lastKeyIndex)
            value: returnNode
            root: (returnNode at: returnNode _lastRootIndex)
            atIndex: index + self entrySize.
          " see if last root in child node has changed "
          lastChildRoot := returnNode at: returnNode _lastRootIndex.
          self updateLastChildRoot: lastChildRoot at: index + self entrySize ] ]
%

! Class Implementation for BtreePlusInteriorKeyValueNode

! Remove existing behavior from BtreePlusInteriorKeyValueNode
removeallmethods BtreePlusInteriorKeyValueNode
removeallclassmethods BtreePlusInteriorKeyValueNode

! ------------------- Class methods for BtreePlusInteriorKeyValueNode

category: 'Constants'
classmethod: BtreePlusInteriorKeyValueNode
entrySize
  "Returns the size of an entry."

  "in-lined on instance-side for performance"

  ^ 2
%

category: 'Constants'
classmethod: BtreePlusInteriorKeyValueNode
maxNumberOfElements
  "Returns the number of entries that are allowed in an interior node."

  ^ 1014
%

! ------------------- Instance methods for BtreePlusInteriorKeyValueNode

category: 'Updating'
method: BtreePlusInteriorKeyValueNode
_insertKey: aKey value: newPredecessorNode atIndex: insertionIndex
  "Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

  | endIndex entrySize |
  entrySize := self entrySize.
  endIndex := numElements * entrySize.

  " newPredecessorNode node is being inserted in front of node at insertionIndex, update the previous/next leaf values for both nodes"
  (self at: insertionIndex) 
    ifNil: [
      insertionIndex > 1 
        ifTrue: [ 
          | previousNode |
          previousNode := self at: insertionIndex - entrySize.
          newPredecessorNode 
            _updateLeafNodeLinksForNewPreviousLeaf:  previousNode
            newNextLeaf: previousNode nextLeaf ] ]
    ifNotNil: [:formerPredecessorNode | 
      formerPredecessorNode 
        _updateLeafNodeLinksForNewPreviousLeaf: newPredecessorNode 
        newNextLeaf: formerPredecessorNode nextLeaf ].

  self _insertAt: insertionIndex 
       value: newPredecessorNode value: aKey 
       numToMoveDown: endIndex - insertionIndex + 1 .

  numElements := numElements + 1


%

category: 'Updating'
method: BtreePlusInteriorKeyValueNode
_insertKey: aKey value: aValue root: ignored atIndex: insertionIndex

"Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

self _insertKey: aKey value: aValue atIndex: insertionIndex
%

category: 'Updating'
method: BtreePlusInteriorKeyValueNode
at: aKey put: aValue root: ignored
  "Adds the key/value/root tuple to one of the leaf nodes.  If a split occurs, returns
   the new sibling, otherwise returns the receiver."

  ^ super at: aKey put: aValue root: nil
%

category: 'Rc Updating'
method: BtreePlusInteriorKeyValueNode
at: aKey put: aValue root: ignored selectiveAbortSet: selectiveAbortSetOrNil
  "Adds the key/value/root pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  ^ super at: aKey put: aValue root: nil selectiveAbortSet: selectiveAbortSetOrNil
%

category: 'Searching'
method: BtreePlusInteriorKeyValueNode
binarySearchCoveringKey: aKey value: aValue root: ignored
  "Returns the index for the entry in which aKey/aValue would be inserted.  This
 is the first entry which has its key >= aKey.  If multiple entries have the
 same key (key = aKey), then aValue is used (since entries with duplicate keys 
 are inserted by the value's OOP ordering).  If multiple entries have the same value 
 (value = aValue), then rootObject is used (since entries with duplicate values are
 inserted by the rootObject's OOP ordering). If the receiver is empty, returns 
 1.  Uses a binary search."

  ^ self binarySearchCoveringKey: aKey value: aValue
%

category: 'Comparison Operators'
method: BtreePlusInteriorKeyValueNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex root: ignored
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  ^ self compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex
%

category: 'Comparison Operators'
method: BtreePlusInteriorKeyValueNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex root: ignored
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  ^ self compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex
%

category: 'Constants'
method: BtreePlusInteriorKeyValueNode
entrySize
  "Returns the size of an entry."

  "in-line of `self class entrySize` for performance"

  ^ 2
%

category: 'Accessing'
method: BtreePlusInteriorKeyValueNode
rootIndexOffset
  "Answer the offset from entry index for the root object. Typical order is value/key/root.
   The key and root object is the same for the receiver. The root object is not updated."

  ^ 1
%

! Class Implementation for BtreePlusInteriorKeyValueRootObjectNode

! Remove existing behavior from BtreePlusInteriorKeyValueRootObjectNode
removeallmethods BtreePlusInteriorKeyValueRootObjectNode
removeallclassmethods BtreePlusInteriorKeyValueRootObjectNode

! ------------------- Class methods for BtreePlusInteriorKeyValueRootObjectNode

category: 'Constants'
classmethod: BtreePlusInteriorKeyValueRootObjectNode
entrySize
  "Returns the size of an entry."

  "in-lined on instance-side for performance"

  ^ 3
%

category: 'Constants'
classmethod: BtreePlusInteriorKeyValueRootObjectNode
maxNumberOfElements
  "Returns the number of entries that are allowed in an interior node."

  ^ 676
%

! ------------------- Instance methods for BtreePlusInteriorKeyValueRootObjectNode

category: 'Updating'
method: BtreePlusInteriorKeyValueRootObjectNode
_insertKey: aKey value: newPredecessorNode root: rootObject atIndex: insertionIndex
  "Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

  | endIndex entrySize |
  entrySize := self entrySize.
  endIndex := numElements * entrySize.

  " newPredecessorNode node is being inserted in front of node at insertionIndex, 
    update the previous/next leaf values for both nodes --- only applies to leaf nodes"
  newPredecessorNode isLeaf
    ifTrue: [ 
      (self at: insertionIndex ) 
        ifNil: [
          insertionIndex > 1 
            ifTrue: [ 
              | previousNode |
              previousNode := self at: insertionIndex - entrySize.
              newPredecessorNode 
                _updateLeafNodeLinksForNewPreviousLeaf:   previousNode
                newNextLeaf: previousNode nextLeaf ] ]
        ifNotNil: [:formerPredecessorNode | 
          formerPredecessorNode 
            _updateLeafNodeLinksForNewPreviousLeaf: newPredecessorNode 
            newNextLeaf: formerPredecessorNode nextLeaf ] ].

  self _insertAt: insertionIndex 
       value: newPredecessorNode value: aKey value: rootObject
       numToMoveDown: endIndex - insertionIndex + 1.       

  numElements := numElements + 1

%

category: 'Comparison Operators'
method: BtreePlusInteriorKeyValueRootObjectNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex root: aRootObject
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index. If the value oops are equal, compare the oop of aRootObject and the 
   oop of the root object at the given index."

  | entryValue entryValueLastValue |
  entryValue := self at: keyIndex - self keyIndexOffset.
  entryValueLastValue := entryValue lastValue.
  "aValue asOop = entryValueLastValue asOop"
  aValue == entryValueLastValue
    ifTrue: [ ^ aRootObject asOop > entryValue lastRoot asOop ].
  ^ aValue asOop > entryValueLastValue asOop
%

category: 'Comparison Operators'
method: BtreePlusInteriorKeyValueRootObjectNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex root: aRootObject
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index. If the value oops are equal, compare the oop of aRootObject and the 
   oop of the root object at the given index."

  | entryValue entryValueLastValue |
  entryValue := self at: keyIndex - self keyIndexOffset.
  entryValueLastValue := entryValue lastValue.
  "aValue asOop = entryValueLastValue asOop"
  aValue == entryValueLastValue
    ifTrue: [ ^ aRootObject asOop <= entryValue lastRoot asOop ].
  ^ aValue asOop < entryValueLastValue asOop
%

category: 'Constants'
method: BtreePlusInteriorKeyValueRootObjectNode
entrySize
  "Returns the size of an entry."

  "in-line of `self class entrySize` for performance"

  ^ 3
%

! Class Implementation for BtreePlusLeafNode

! Remove existing behavior from BtreePlusLeafNode
removeallmethods BtreePlusLeafNode
removeallclassmethods BtreePlusLeafNode

! ------------------- Class methods for BtreePlusLeafNode

! ------------------- Instance methods for BtreePlusLeafNode

category: 'Query Support'
method: BtreePlusLeafNode
_addRootObjects: aQuerySpec start: startIndex end: endIndex into: collection
  ^ (aQuerySpec canUseOptimizedComparison and: [ self rootNode indexObject optimizingComparison ])
    ifTrue: [ 
      self
        _primAddRootObjects: aQuerySpec
        start: startIndex + self rootIndexOffset
        end: endIndex + self rootIndexOffset
        into: collection
        entrySize: self entrySize ]
    ifFalse: [ 
      super
        _addRootObjects: aQuerySpec
        start: startIndex
        end: endIndex
        into: collection ]
%

category: 'Updating'
method: BtreePlusLeafNode
_createSibling
  "Creates a new sibling node and move the second half of the receiver's entries into the
sibling.  Returns the new sibling."

  | newSibling nLeaf |
  newSibling := super _createSibling.
  nLeaf := self nextLeaf.
  newSibling 
    previousLeaf: self;
    nextLeaf: nLeaf.
  nLeaf ifNotNil: [ nLeaf previousLeaf: newSibling ].
  self nextLeaf: newSibling.
  ^ newSibling
%

category: 'Searching'
method: BtreePlusLeafNode
_findAllValuesForIdenticalKey: aKey into: aCollection using: aComparison

"Finds all the values for the given identical key, placing them in the
 collection.  For leaf nodes, must first find the appropriate entry, then
 iterate through successive entries until we reach the end or a key does
 not match.  Return whether the last key was equal or not (so the caller
 knows it must check the next leaf node)."

| index eSize maxIndex keyIndexOffset |
" find first entry that might contain the key "
index := self _findCoveringIndexForKey: aKey totalOrder: false.
index == 0 ifTrue: [ ^ false ].

eSize := self entrySize.
maxIndex := self _lastIndex.

" now scan entries until reach the end or no match using sort ="
keyIndexOffset := self keyIndexOffset.
[ (index < maxIndex) and:
[ aComparison compareKey: aKey equalTo: (self at: index + keyIndexOffset) ] ] whileTrue: [
  (aKey == (self at: index + keyIndexOffset))
    ifTrue: [ aCollection add: (self at: index) ].
  index := index + eSize
].
^ index >= maxIndex
%

category: 'Searching Support'
method: BtreePlusLeafNode
_findAllValuesGreaterThanKey: aKey into: anArray using: aComparison
  "Puts into anArray the index and the leaf node of the entry that is greater
 than the given key.  Returns true if a value was found, false otherwise."

  | entryIndex eSize lastIndex keyIndexOffset comparisonForSort |
  numElements == 0
    ifTrue: [ ^ false ].
  lastIndex := self _lastKeyIndex.	" find index where the key would be inserted "
  comparisonForSort := self comparisonForSort.
  entryIndex := self _findCoveringIndexForKey: aKey totalOrder: false using: comparisonForSort.
  entryIndex == 0
    ifTrue: [ ^ false ].	"skip keys that sort ="
  eSize := self entrySize.
  keyIndexOffset := self keyIndexOffset.
  [ comparisonForSort
    compareKey: aKey
    equalTo: (self at: entryIndex + keyIndexOffset) ]
    whileTrue: [ entryIndex := entryIndex + eSize.
      entryIndex > lastIndex
        ifTrue: [ ^ false ] ].
  aKey ~~ nil
    ifTrue: [ "bugfix 33805 - verify that aKey < index entry"
      (comparisonForSort
        compareKey: aKey
        lessThan: (self at: entryIndex + keyIndexOffset))
        ifFalse: [ ^ false ] ].
  anArray addLast: entryIndex.
  anArray addLast: self.
  ^ true
%

category: 'Searching Support'
method: BtreePlusLeafNode
_findAllValuesLessThanKey: aKey andEquals: aBoolean into: anArray using: aComparison
  "Puts into anArray the index and the leaf node of the entry that is less than
 (or less than and equal to if aBoolean is true) the given key.  Returns true
 if a value was found, false otherwise."

  numElements == 0
    ifTrue: [ ^ false ].
  aBoolean
    ifFalse: [ ^ self _findAllValuesLessThanKey: aKey into: anArray using: aComparison ].
  (self _findAllValuesGreaterThanKey: aKey into: anArray using: aComparison)
    ifTrue: [ | eSize index |
      eSize := self entrySize.
      index := anArray at: 1.
      index > eSize
        ifTrue: [ "move index pointer for first greater than entry to first less than or equal to entry"
          index := index - eSize.
          anArray at: 1 put: index ]
        ifFalse: [ "first entry is greater than, so no entries are less than or equal"
          anArray size: 0.
          ^ false ].
      ^ true ].
  (aKey == nil or: [ aComparison
        compareKey: aKey
        greaterThanOrEqualTo: (self at: self _lastKeyIndex) ])
    ifTrue: [ " if so, put the rightmost leaf entry in the Array "
      self _putLastIndexOfLastChildInto: anArray.
      ^ true ].
  ^ false
%

category: 'Searching Support'
method: BtreePlusLeafNode
_findAllValuesLessThanKey: aKey into: anArray using: aComparison
  "Puts into anArray the index and the leaf node of the entry that is less than
 the given key.  Returns true if a value was found, false otherwise."

  | entryIndex eSize lastIndex lastKeyIndex keyIndex comparisonForSort |
  numElements == 0
    ifTrue: [ ^ false ].
  lastIndex := self _lastIndex.
  " first see if all entries are > aKey "
  lastKeyIndex := self _lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (comparisonForSort
    compareKey: aKey
    lessThanOrEqualTo: (self at: lastKeyIndex))
    ifFalse: [ "bugfix 33805 - verify that aKey is > index entry"
      (aKey == nil
        or: [ aComparison compareKey: aKey greaterThan: (self at: lastKeyIndex) ])
        ifTrue: [ " if so, put the rightmost leaf entry in the Array "
          self _putLastIndexOfLastChildInto: anArray.
          ^ true ] ].
  " find index where the key would be inserted "
  entryIndex := self _findCoveringIndexForKey: aKey totalOrder: false  using: comparisonForSort.	" if index == 0, found no entryes"
  entryIndex == 0
    ifTrue: [ ^ false ].
  " if index == 1, check if any are < aKey "
  entryIndex == 1
    ifTrue: [ " if the first entry is greater than or equal to aKey, none satisfy "
      | keyIndexOffset |
      keyIndexOffset := self keyIndexOffset.
      ((comparisonForSort compareKey: aKey lessThan: (self at: entryIndex + keyIndexOffset))
        or: [ comparisonForSort compareKey: aKey equalTo: (self at: entryIndex + keyIndexOffset) ])
        ifTrue: [ ^ false ] ].
  eSize := self entrySize.
  entryIndex > 1
    ifTrue: [ " subtract an entry to point to a valid 'less than' entry "
      entryIndex := entryIndex - eSize ].
  keyIndex := entryIndex + self keyIndexOffset.
  aKey ~~ nil
    ifTrue: [ "bugfix 33805 - verify that index aKey is > entry"
      (comparisonForSort compareKey: aKey greaterThan: (self at: keyIndex))
        ifFalse: [ (aComparison compareKey: aKey equalTo: (self at: keyIndex))
            ifTrue: [ ^ false ].
          (aComparison compareKey: aKey lessThan: (self at: keyIndex))
            ifFalse: [ ^ false ] ] ].
  anArray addLast: entryIndex.
  anArray addLast: self.
  ^ true
%

category: 'Query Support'
method: BtreePlusLeafNode
_primAddRootObjects: aQuerySpec start: startIndex end: endIndex into: collection entrySize: entrySize
  <primitive: 1038>
  ^ super
    _addRootObjects: aQuerySpec
    start: startIndex
    end: endIndex
    into: collection
%

category: 'Searching Support'
method: BtreePlusLeafNode
_putFirstIndexOfFirstChildInto: anArray

"Puts the receiver and the first index of the receiver into the Array."

numElements == 0
    ifTrue: [ anArray addLast: 0]
    ifFalse: [ anArray addLast: 1 ].
anArray addLast: self
%

category: 'Searching Support'
method: BtreePlusLeafNode
_putFirstIndexOfFirstChildInto: anArray ifGreaterThanOrEqualTo: aKey using: aComparison

"If first entry is greater than or equal to aKey, Puts the receiver 
 and the first index of the receiver into the Array.

 Because of NANs we need to validate the assumption of the caller 
 (i.e., that all entries are greater than aKey). See bug33805."

"make sure that key is less than or equal to first entry"
(aKey == nil or: [ numElements == 0 
   or: [ aComparison compareKey: aKey lessThanOrEqualTo: (self at: 1 + self keyIndexOffset) ] ])
  ifTrue: [
    numElements == 0
      ifTrue: [ anArray addLast: 0]
      ifFalse: [ anArray addLast: 1 ].
  ]
  ifFalse: [
    anArray addLast: 0.
  ].
  anArray addLast: self
%

category: 'Searching Support'
method: BtreePlusLeafNode
_putLastIndexOfLastChildInto: anArray
  "Puts the receiver and the last index of the receiver into the Array."

  anArray addLast: self _lastKeyIndex - self keyIndexOffset.
  anArray addLast: self
%

category: 'Updating'
method: BtreePlusLeafNode
_splitUsingKey: aKey value: aValue
  "Creates a new sibling node, moving the second half of the receiver's entries into the
 sibling.  Then inserts a new entry with the given key/value into either the
 receiver or sibling.  Returns the new sibling."

  | index newSibling lastKeyIndex lastKey comparisonForSort |
  newSibling := self _createSibling.
  lastKeyIndex := self _lastKeyIndex.
  lastKey := super at: lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (self
    compare: comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: lastKeyIndex
    root: nil)
    ifTrue: [ " adding to the first half "
      index := self
        binarySearchCoveringKey: aKey
        value: aValue.
      self
        _insertKey: aKey
        value: aValue
        atIndex: index ]
    ifFalse: [ " adding to the second half "
      index := newSibling
        binarySearchCoveringKey: aKey
        value: aValue.
      newSibling
        _insertKey: aKey
        value: aValue
        atIndex: index ].
  ^ newSibling
%

category: 'Updating'
method: BtreePlusLeafNode
_splitUsingKey: aKey value: aValue root: rootObject
  "Creates a new sibling node, moving the second half of the receiver's entries into the
 sibling.  Then inserts a new entry with the given key/value/rootObject into either the
 receiver or sibling.  Returns the new sibling."

  | index newSibling lastKeyIndex lastKey comparisonForSort |
  newSibling := self _createSibling.
  lastKeyIndex := self _lastKeyIndex.
  lastKey := super at: lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (self
    compare: comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: lastKeyIndex
    root: rootObject)
    ifTrue: [ " adding to the first half "
      index := self
        binarySearchCoveringKey: aKey
        value: aValue
        root: rootObject.
      self
        _insertKey: aKey
        value: aValue
        root: rootObject
        atIndex: index ]
    ifFalse: [ " adding to the second half "
      index := newSibling
        binarySearchCoveringKey: aKey
        value: aValue
        root: rootObject.
      newSibling
        _insertKey: aKey
        value: aValue
        root: rootObject
        atIndex: index ].
  ^ newSibling
%

category: 'Updating'
method: BtreePlusLeafNode
_updateLastValue

"For leaf nodes, the last value is derived, so nothing is required."

%

category: 'Audit'
method: BtreePlusLeafNode
auditForPathTerm: pathTerm using: auditor parentNode: parentNode previousChild: previousChild nextChild: nextChild do: aBlock
  "Execute the three-argument block with the key, value, root of each entry in the receiver."

  | keyIndexOffset rootIndexOffset |
  keyIndexOffset := self keyIndexOffset.
  rootIndexOffset := self rootIndexOffset.
  previousChild == previousLeaf
    ifFalse: [
      auditor 
        btree: self 
        parentNode: parentNode
        previousLeafMismatch: pathTerm 
        previousLeaf: previousLeaf 
        parentPreviousNode: previousChild ].
  nextChild == nextLeaf
    ifFalse: [
      auditor 
        btree: self 
        parentNode: parentNode
        nextLeafMismatch: pathTerm 
        nextLeaf: nextLeaf 
        parentNextNode: nextChild ].
  1 to: numElements * self entrySize by: self entrySize do: [ :i | | v k r |
    v := self at: i.
    k := self at: i + keyIndexOffset.
    r := self at: i + rootIndexOffset.
    aBlock value: self value: k value: v value: r ]
%

category: 'Comparison Operators'
method: BtreePlusLeafNode
compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex
  "Performs a = comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, use the OOP of the value as the next basis for
 comparison."

  | entryIndex |
  entryIndex := keyIndex - self keyIndexOffset.
  ^ (comparisonForSort compareKey: aKey equalTo: (self at: keyIndex))
    and: [ aValue == (self at: entryIndex) ]
%

category: 'Comparison Operators'
method: BtreePlusLeafNode
compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex root: rootObject
  "Performs a = comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, use the OOP of the value as the next basis for
 comparison. If the value oops are equeal, use the oop of the rootObject as the
 next basis for comparison."

  ^ (comparisonForSort compareKey: aKey equalTo: (self at: keyIndex))
    and: [ | entryIndex |
      entryIndex := keyIndex - self keyIndexOffset.
      aValue == (self at: entryIndex)
        and: [ rootObject == (self at: entryIndex + self rootIndexOffset) ] ]
%

category: 'Comparison Operators'
method: BtreePlusLeafNode
compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index. If the keys are equal, then use the OOP of the value for the
 comparison."

  (comparisonForSort compareKey: aKey lessThan: (self at: keyIndex))
    ifTrue: [ ^ true ].
  (comparisonForSort compareKey: aKey equalTo: (self at: keyIndex))
    ifTrue: [ | entryIndex |
      entryIndex := keyIndex - self keyIndexOffset.
      ^aValue asOop <= (self at: entryIndex) asOop ].
  ^ false
%

category: 'Comparison Operators'
method: BtreePlusLeafNode
compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: rootObject
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, then use the OOP of the value for the
 comparison. If the value oops are equal, use the OOP of the rootObject for the 
 comparison."

  (comparisonForSort compareKey: aKey lessThan: (self at: keyIndex))
    ifTrue: [ ^ true ].
  (comparisonForSort compareKey: aKey equalTo: (self at: keyIndex))
    ifTrue: [ | entryIndex entryIndexValue |
      entryIndex := keyIndex - self keyIndexOffset.
      entryIndexValue := self at: entryIndex.
      "aValue asOop = (entryIndexValue) asOop"
      aValue == entryIndexValue
        ifTrue: [ ^ rootObject asOop <= (self at: entryIndex + self rootIndexOffset) asOop ].
      ^ aValue asOop < entryIndexValue asOop ].
  ^ false
%

category: 'Comparison Operators'
method: BtreePlusLeafNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."


  ^ aValue asOop > (self at: keyIndex - self keyIndexOffset) asOop
%

category: 'Comparison Operators'
method: BtreePlusLeafNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  ^ aValue asOop <= (self at: keyIndex - self keyIndexOffset) asOop
%

category: 'Testing'
method: BtreePlusLeafNode
isLeaf

"Returns whether the B-tree node is a leaf node."

^ true
%

category: 'Accessing'
method: BtreePlusLeafNode
keyAt: anIndex
  "answer the key for the given index"

  ^ self at: anIndex + self keyIndexOffset
%

category: 'Accessing'
method: BtreePlusLeafNode
lastKey

"Returns the key for the last entry in the receiver."

^ self at: (self _lastEntryIndex + self keyIndexOffset)
%

category: 'Accessing'
method: BtreePlusLeafNode
lastRoot

"Returns the root object for the last entry in the receiver."

^ self at: (self _lastEntryIndex + self rootIndexOffset)
%

category: 'Accessing'
method: BtreePlusLeafNode
lastValue

"Returns the value for the last entry in the receiver."

^ self at: self _lastEntryIndex
%

category: 'Accessing'
method: BtreePlusLeafNode
nextLeaf
  "Returns the value of the instance variable 'nextLeaf'."

  ^ nextLeaf
%

category: 'Accessing'
method: BtreePlusLeafNode
nextLeaf: anObject
  "Modify the value of the instance variable 'nextLeaf'."

  nextLeaf := anObject
%

category: 'Accessing'
method: BtreePlusLeafNode
previousLeaf
  "Returns the value of the instance variable 'previousLeaf'."

  ^ previousLeaf
%

category: 'Accessing'
method: BtreePlusLeafNode
previousLeaf: anObject
  "Modify the value of the instance variable 'previousLeaf'."

  previousLeaf := anObject
%

category: 'Removing'
method: BtreePlusLeafNode
removeKey: aKey value: aValue root: rootObject
  "Removes the key/value pair.  Finds the index for the given key and then shifts
 any following entries to the left."

  | valueIndex eSize |
  valueIndex := self binarySearchForKey: aKey value: aValue root: rootObject.
  valueIndex == 0
    ifTrue: [ ^ false ].
  eSize := self entrySize.
  self _deleteNoShrinkFrom: valueIndex to: valueIndex + eSize - 1.
  numElements := numElements - 1.
  ^ true
%

category: 'Rc Removing'
method: BtreePlusLeafNode
removeKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil
  "Removes the key/value pair.  Finds the index for the given key and then shifts
 any following entries to the left."

  | wasRemoved |
  self _selectiveAbort: self ifNotIn: selectiveAbortSetOrNil.
  wasRemoved := self removeKey: aKey value: aValue root: rootObject.
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  ^ wasRemoved
%

category: 'Accessing'
method: BtreePlusLeafNode
rootObjectAt: anIndex
  "answer the root object for the given index"

  ^ self at: anIndex + self rootIndexOffset
%

! Class Implementation for BtreePlusLeafKeyNode

! Remove existing behavior from BtreePlusLeafKeyNode
removeallmethods BtreePlusLeafKeyNode
removeallclassmethods BtreePlusLeafKeyNode

! ------------------- Class methods for BtreePlusLeafKeyNode

category: 'Constants'
classmethod: BtreePlusLeafKeyNode
entrySize
  "Returns the size of an entry with no encryption."

  "in-lined on instance-side for performance"

  ^ 1
%

category: 'Constants'
classmethod: BtreePlusLeafKeyNode
maxNumberOfElements
  "Returns the number of entries that are allowed in an interior node."

  ^ 2028
%

! ------------------- Instance methods for BtreePlusLeafKeyNode

category: 'Updating'
method: BtreePlusLeafKeyNode
_insertKey: aKey atIndex: insertionIndex

"Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

| endIndex entrySize |

" see if any existing entries have to be moved "
entrySize := self entrySize .
endIndex := (numElements * entrySize) .

self _insertAt: insertionIndex 
     value: aKey 
     numToMoveDown: (endIndex - insertionIndex + 1) .

"no prefix encoding"

numElements := numElements + 1
%

category: 'Updating'
method: BtreePlusLeafKeyNode
_insertKey: aKey value: ignoredValue root: ignoredRoot atIndex: insertionIndex

"Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

self _insertKey: aKey atIndex: insertionIndex
%

category: 'Updating'
method: BtreePlusLeafKeyNode
at: aKey put: aValue root: ignoredRoot
  "Adds the key/value pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  | index |
  " determine if a split is needed "
  numElements == self maxNumberOfElements
    ifTrue: [ " performs the split and returns the new sibling "
      ^ self _splitUsingKey: aKey value: aValue  ]
    ifFalse: [ " entry will fit in leaf node "
      numElements == 0
        ifTrue: [ index := 1 ]
        ifFalse: [ index := self binarySearchCoveringKey: aKey ].
      self _insertKey: aKey atIndex: index ]
%

category: 'Comparison Operators'
method: BtreePlusLeafKeyNode
compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex root: ignoredRootObject
  "Performs a = comparison between aKey and the entry whose key is at the given
 index. If the keys are equal, use the OOP of the value as the next basis for
 comparison."

  ^ self compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex
%

category: 'Comparison Operators'
method: BtreePlusLeafKeyNode
compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: ignoredRootObject
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index. If the keys are equal, then use the OOP of the value for the comparison."

  ^self compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex
%

category: 'Comparison Operators'
method: BtreePlusLeafKeyNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  "value/key/root use same index"

  ^ aValue asOop > (self at: keyIndex) asOop
%

category: 'Comparison Operators'
method: BtreePlusLeafKeyNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex root: ignored
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index. "

  ^ self compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex
%

category: 'Comparison Operators'
method: BtreePlusLeafKeyNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  "value/key/root use same index"

  ^ aValue asOop <= (self at: keyIndex) asOop
%

category: 'Comparison Operators'
method: BtreePlusLeafKeyNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex root: ignored
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  ^ self compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex
%

category: 'Constants'
method: BtreePlusLeafKeyNode
entrySize
  "Returns the size of an entry."

  "in-line of `self class entrySize` for performance"

  ^ 1
%

category: 'Accessing'
method: BtreePlusLeafKeyNode
keyIndexOffset
  "Answer the offset from entry index for the key. Typical order is value/key/root.
   The value, key and root objects are the same for the receiver."

  ^ 0
%

category: 'Constants'
method: BtreePlusLeafKeyNode
parentNodeClass
  "Returns the class of node to be created as the parent when a split occurs."

  ^ BtreePlusInteriorKeyValueNode
%

category: 'Accessing'
method: BtreePlusLeafKeyNode
rootIndexOffset
  "Answer the offset from entry index for the root object. Typical order is value/key/root.
   The value, key and root objects are the same for the receiver."

  ^ 0
%

! Class Implementation for BtreePlusLeafKeyValueNode

! Remove existing behavior from BtreePlusLeafKeyValueNode
removeallmethods BtreePlusLeafKeyValueNode
removeallclassmethods BtreePlusLeafKeyValueNode

! ------------------- Class methods for BtreePlusLeafKeyValueNode

category: 'Constants'
classmethod: BtreePlusLeafKeyValueNode
entrySize
  "Returns the size of an entry with no encryption."

  "in-lined on instance-side for performance"

  ^ 2
%

category: 'Constants'
classmethod: BtreePlusLeafKeyValueNode
maxNumberOfElements
  "Returns the number of entries that are allowed in an interior node."

  ^ 1014
%

! ------------------- Instance methods for BtreePlusLeafKeyValueNode

category: 'Updating'
method: BtreePlusLeafKeyValueNode
_insertKey: aKey value: aValue atIndex: insertionIndex
  "Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

  | endIndex entrySize |
  entrySize := self entrySize.
  endIndex := numElements * entrySize.

  self _insertAt: insertionIndex 
       value: aValue value: aKey 
       numToMoveDown: endIndex - insertionIndex + 1 .

  "no prefix encoding"

  numElements := numElements + 1


%

category: 'Updating'
method: BtreePlusLeafKeyValueNode
_insertKey: aKey value: aValue root: ignored atIndex: insertionIndex

"Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

self _insertKey: aKey value: aValue atIndex: insertionIndex
%

category: 'Updating'
method: BtreePlusLeafKeyValueNode
at: aKey put: aValue root: ignored
  "Adds the key/value pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  | index |
  " determine if a split is needed "
  numElements == self maxNumberOfElements
    ifTrue: [ " performs the split and returns the new sibling "
      ^ self _splitUsingKey: aKey value: aValue ]
    ifFalse: [ " entry will fit in leaf node "
      numElements == 0
        ifTrue: [ index := 1 ]
        ifFalse: [ index := self binarySearchCoveringKey: aKey value: aValue ].
      self _insertKey: aKey value: aValue atIndex: index ]
%

category: 'Comparison Operators'
method: BtreePlusLeafKeyValueNode
compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex root: rootObject
  "Performs a = comparison between aKey and the entry whose key is at the given
 index. If the keys are equal, use the OOP of the value as the next basis for
 comparison."

  ^ self compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex
%

category: 'Comparison Operators'
method: BtreePlusLeafKeyValueNode
compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: ignoredRootObject
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index. If the keys are equal, then use the OOP of the value for the comparison."

  ^self compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex
%

category: 'Comparison Operators'
method: BtreePlusLeafKeyValueNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex root: ignored
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index. "

  ^ self compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex
%

category: 'Comparison Operators'
method: BtreePlusLeafKeyValueNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex root: ignored
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  ^ self compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex
%

category: 'Constants'
method: BtreePlusLeafKeyValueNode
entrySize
  "Returns the size of an entry."

  "in-line of `self class entrySize` for performance"

  ^ 2
%

category: 'Constants'
method: BtreePlusLeafKeyValueNode
parentNodeClass
  "Returns the class of node to be created as the parent when a split occurs."

  ^ BtreePlusInteriorKeyValueRootObjectNode
%

category: 'Accessing'
method: BtreePlusLeafKeyValueNode
rootIndexOffset
  "Answer the offset from entry index for the root object. Typical order is value/key/root.
   The value and root object is the same for the receiver."

  ^ 0
%

! Class Implementation for BtreePlusLeafKeyValueRootObjectNode

! Remove existing behavior from BtreePlusLeafKeyValueRootObjectNode
removeallmethods BtreePlusLeafKeyValueRootObjectNode
removeallclassmethods BtreePlusLeafKeyValueRootObjectNode

! ------------------- Class methods for BtreePlusLeafKeyValueRootObjectNode

category: 'Constants'
classmethod: BtreePlusLeafKeyValueRootObjectNode
entrySize
  "Returns the size of an entry with no encryption."

  "in-lined on instance-side for performance"

  ^ 3
%

category: 'Constants'
classmethod: BtreePlusLeafKeyValueRootObjectNode
maxNumberOfElements
  "Returns the number of entries that are allowed in an interior node."

  ^ 676
%

! ------------------- Instance methods for BtreePlusLeafKeyValueRootObjectNode

category: 'Updating'
method: BtreePlusLeafKeyValueRootObjectNode
_insertKey: aKey value: aValue root: rootObject atIndex: insertionIndex
  "Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

  | endIndex entrySize |
  entrySize := self entrySize.
  endIndex := numElements * entrySize.  
  self _insertAt: insertionIndex 
       value: aValue value: aKey value: rootObject
       numToMoveDown: endIndex - insertionIndex + 1.       

  "no prefix encoding."

  numElements := numElements + 1

%

category: 'Updating'
method: BtreePlusLeafKeyValueRootObjectNode
at: aKey put: aValue root: rootObject
  "Adds the key/value/root pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  | index |
  " determine if a split is needed "
  numElements == self maxNumberOfElements
    ifTrue: [ " performs the split and returns the new sibling "
      ^ self _splitUsingKey: aKey value: aValue root: rootObject ]
    ifFalse: [ " entry will fit in leaf node "
      numElements == 0
        ifTrue: [ index := 1 ]
        ifFalse: [ index := self binarySearchCoveringKey: aKey value: aValue root: rootObject ].
      self
        _insertKey: aKey
        value: aValue
        root: rootObject
        atIndex: index ]
%

category: 'Comparison Operators'
method: BtreePlusLeafKeyValueRootObjectNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex root: aRootObject
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index. If the value oops are equal, compare the oop of aRootObject and the 
   oop of the root object at the given index."

  | keyIndexOffset entryValue |
  keyIndexOffset := self keyIndexOffset.
  entryValue := self at: keyIndex - keyIndexOffset.
  "aValue asOop = entryValue asOop"
  aValue == entryValue
    ifTrue: [ ^ aRootObject asOop > (self at: keyIndex - keyIndexOffset + self rootIndexOffset) asOop ].
  ^ aValue asOop > entryValue asOop
%

category: 'Comparison Operators'
method: BtreePlusLeafKeyValueRootObjectNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex root: aRootObject
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index. If the value oops are equal, compare the oop of aRootObject and the 
   oop of the root object at the given index."

  | keyIndexOffset entryValue |
  keyIndexOffset := self keyIndexOffset.
  entryValue := self at: keyIndex - keyIndexOffset.
  "aValue asOop = entryValue asOop"
  aValue == entryValue
    ifTrue: [ ^ aRootObject asOop <= (self at: keyIndex - keyIndexOffset + self rootIndexOffset) asOop ].
  ^ aValue asOop < entryValue asOop
%

category: 'Constants'
method: BtreePlusLeafKeyValueRootObjectNode
entrySize
  "Returns the size of an entry."

  "in-line of `self class entrySize` for performance"

  ^ 3
%

category: 'Constants'
method: BtreePlusLeafKeyValueRootObjectNode
parentNodeClass
  "Returns the class of node to be created as the parent when a split occurs."

  ^ BtreePlusInteriorKeyValueRootObjectNode
%

! Class Implementation for BtreePlusNodeAuditor

! Remove existing behavior from BtreePlusNodeAuditor
removeallmethods BtreePlusNodeAuditor
removeallclassmethods BtreePlusNodeAuditor

! ------------------- Instance methods for BtreePlusNodeAuditor

category: 'pathterm audit failures'
method: BtreePlusNodeAuditor
auditInfoFor: pathTerm info: array
  | obj indexObj depList |
  pathTerm size == 0
    ifTrue: [ ^ self ].
  obj := array at: 1.
  indexObj := array at: 2.
  depList := array at: 3.
  auditResultString add: 'object'.
  (obj isKindOf: CharacterCollection)
    ifTrue: [ auditResultString
        add: Character space;
        add: obj ].
  auditResultString
    add: ' [';
    add: obj asOop asString;
    add: '] of class ';
    add: obj class asString.
  indexObj ~~ nil
    ifTrue: [ auditResultString
        add: ', ';
        add: (indexObj isRangeEqualityIndex
                ifTrue: [ 'equality' ]
                ifFalse: [ 'identity' ]);
        add: ' index <';
        add: indexObj pathComponentsString;
        add: '> [';
        add: indexObj asOop asString;
        add: ']' ].
  auditResultString
    add: ', path term <';
    add: pathTerm name.
  pathTerm name == #'*'
    ifTrue: [ auditResultString add: ' offset: ' , pathTerm offset asString ].
  auditResultString
    add: '> [';
    add: pathTerm asOop asString;
    add: ']'.
  depList ~~ nil
    ifTrue: [ auditResultString
        add: ', depList [';
        add: depList asOop asString;
        add: ']' ].
  auditResultString
    add:
      '
    -- '
%

category: 'accessing'
method: BtreePlusNodeAuditor
auditResultString

   ^auditResultString
%

category: 'accessing'
method: BtreePlusNodeAuditor
auditResultString: anObject

   auditResultString := anObject
%

category: 'btree audit failures'
method: BtreePlusNodeAuditor
btree: btree incorrectKey: pathTerm key: k value: v expectedKey: expectedKey ivOffset: ivOffset
  pathTerm auditInfo: {k.
      nil.
      nil} using: self.
  auditResultString
    add: 'Btree [';
    add: btree asOop asString;
    add: '] has entry for [';
    add: v asOop asString;
    add: '] of class ';
    add: v class asString;
    add: ' associated with incorrect key [';
    add: k asOop asString;
    add: '] expected value is ( ';
    add: expectedKey asOop asString;
    add: '). The key should be identical to the instance variable at offset ( ';
    add: ivOffset asString;
    add: ') in [';
    add: v asOop asString;
    add: ']';
    add: Character lf
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btree: btree incorrectKeyForEnumeratedPathTerm: pathTerm key: k value: v expectedKey: expectedKey selector: selector
  pathTerm auditInfo: {k.
      nil.
      nil} using: self.
  auditResultString
    add: 'Btree [';
    add: btree asOop asString;
    add: '] has entry for [';
    add: v asOop asString;
    add: '] of class ';
    add: v class asString;
    add: ' associated with incorrect key [';
    add: k asOop asString;
    add: '] expected value is ( ';
    add: expectedKey asOop asString;
    add: '). The key should sort equal to the value returned by the selector #';
    add: selector asString;
    add: ' when sent to [';
    add: v asOop asString;
    add: ']';
    add: Character lf
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btree: btree incorrectNumberOfEntries: pathTerm key: k value: v bag: bag numExpected: numExpected
  pathTerm auditInfo: {k.
      nil.
      nil} using: self.
  auditResultString
    add: 'Btree [';
    add: btree asOop asString;
    add: '] has incorrect number of entries (';
    add: bag size asString;
    add: ') for [';
    add: v asOop asString;
    add: '] of class ';
    add: v class asString;
    add: ' (should be ';
    add: numExpected asString;
    add: ')';
    add: Character lf
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btree: btree mismatchedKeyAndValueForIndexOnNscElements: pathTerm key: k value: v
  pathTerm auditInfo: {k.
      nil.
      nil} using: self.
  auditResultString
    add: 'Btree [';
    add: btree asOop asString;
    add: '] has entry for [';
    add: v asOop asString;
    add: '] of class ';
    add: v class asString;
    add: ' associated with incorrect key [';
    add: k asOop asString;
    add: '] should be ( ';
    add: v asOop asString;
    add: '). Key and value should be identical.';
    add: Character lf
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btree: btree mismatchedRootAndValueForIndexOnNscElements: pathTerm key: k value: v root: r
  pathTerm auditInfo: {k.
      nil.
      nil} using: self.
  auditResultString
    add: 'Btree [';
    add: btree asOop asString;
    add: '] has entry for [';
    add: v asOop asString;
    add: '] of class ';
    add: v class asString;
    add: ' associated with incorrect root object [';
    add: r asOop asString;
    add: '] should be ( ';
    add: v asOop asString;
    add: ') ';
    add: Character lf
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btree: btree outOfOrderKeys: pathTerm key: k previous: previousKey
  pathTerm auditInfo: {k.
      nil.
      nil} using: self.
  auditResultString
    add: 'Btree [';
    add: btree asOop asString;
    add: '] has key ';
    add: k printString printString;
    add: ' out of order [';
    add: k asOop asString;
    add: '] that sorts greater than previous key';
    add: previousKey printString printString;
    add: ' [';
    add: previousKey asOop asString;
    add: ']';
    add: Character lf
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btree: btree outOfOrderKeyValuePairs: pathTerm key: k previous: previousKey value: v previous: previousValue
  pathTerm auditInfo: {k.
      nil.
      nil} using: self.
  auditResultString
    add: 'Btree [';
    add: btree asOop asString;
    add: '] has out of oop order key/value pairs where the key ';
    add: k printString printString;
    add: ' [';
    add: k asOop asString;
    add: '] is equal to previous key ';
    add: previousKey printString printString;
    add: ' [';
    add: previousKey asOop asString;
    add: '] but the oop of the value ';
    add: v printString printString;
    add: ' [';
    add: v asOop asString;
    add: '] is less than the oop of the previous value ';
    add: previousValue printString printString;
    add: ' [';
    add: previousValue asOop asString;
    add: ']';
    add: Character lf
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btreeInteriorNode: btree childConsistencyError: pathTerm key: k lastChildKey: lastChildKey childNode: childNode index: index
  pathTerm auditInfo: {k.
      nil.
      nil} using: self.
  auditResultString
    add: 'Btree interior node [';
    add: btree asOop asString;
    add: '] at index: ';
    add: index asString;
    add: ' has key ';
    add: k printString printString;
    add: ' [';
    add: k asOop asString;
    add: '] that does not match the last key ';
    add: lastChildKey printString printString;
    add: ' [';
    add: lastChildKey asOop asString;
    add: ' ] of the child node [';
    add: childNode asOop asString;
    add: ']';
    add: Character lf
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btreeInteriorNode: btree childConsistencyError: pathTerm key: k root: r lastChildRoot: lastChildRoot childNode: childNode index: index
  pathTerm auditInfo: {k.
      nil.
      nil} using: self.
  auditResultString
    add: 'Btree interior node [';
    add: btree asOop asString;
    add: '] at index: ';
    add: index asString;
    add: ' has root ';
    add: r printString printString;
    add: ' [';
    add: r asOop asString;
    add: '] that does not match the last root ';
    add: lastChildRoot printString printString;
    add: ' [';
    add: lastChildRoot asOop asString;
    add: ' ] of the child node [';
    add: childNode asOop asString;
    add: ']';
    add: Character lf
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btreeInteriorNode: btree childConsistencyError: pathTerm lastRoot: lastRoot lastChildRoot: lastChildRoot childNode: childNode
  auditResultString
    add: 'Btree interior node [';
    add: btree asOop asString;
    add: '] has lastRoot ';
    add: lastRoot printString printString;
    add: ' [';
    add: lastRoot asOop asString;
    add: '] that does not match the last child root ';
    add: lastChildRoot printString printString;
    add: ' [';
    add: lastChildRoot asOop asString;
    add: ' ] of the child node [';
    add: childNode asOop asString;
    add: ']';
    add: Character lf
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btreeInteriorNode: btree childConsistencyError: pathTerm lastValue: lastValue lastChildValue: lastChildValue childNode: childNode
  auditResultString
    add: 'Btree interior node [';
    add: btree asOop asString;
    add: ']  has lastValue ';
    add: lastValue printString printString;
    add: ' [';
    add: lastValue asOop asString;
    add: '] that does not match the last child value ';
    add: lastChildValue printString printString;
    add: ' [';
    add: lastChildValue asOop asString;
    add: '] of the child node [';
    add: childNode asOop asString;
    add: ']';
    add: Character lf
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btreeInteriorNode: btree keyOrderError: pathTerm key: k previousKey: previousKey index: index
  pathTerm
    auditInfo:
      {k.
      nil.
      nil}
    using: self.
  auditResultString
    add: 'Btree interior node [';
    add: btree asOop asString;
    add: '] at index: ';
    add: index asString;
    add: ' has key ';
    add: k printString printString;
    add: ' out of order [';
    add: k asOop asString;
    add: '] that sorts greater than previous key';
    add: previousKey printString printString;
    add: ' [';
    add: previousKey asOop asString;
    add: ']';
    add: Character lf
%


category: 'btree audit failures'
method: BtreePlusNodeAuditor
btreeInteriorNodeKeyOrderError: pathTerm key: k previousKey: previousKey index: index
  pathTerm
    auditInfo:
      {k.
      nil.
      nil}
    using: self.
  auditResultString
    add: 'Btree interior node [';
    add: self asOop asString;
    add: '] at index: ';
    add: index asString;
    add: ' has key ';
    add: k printString printString;
    add: ' out of order [';
    add: k asOop asString;
    add: '] that sorts greater than previous key';
    add: previousKey printString printString;
    add: ' [';
    add: previousKey asOop asString;
    add: ']';
    add: Character lf
%

category: 'pathterm audit failures'
method: BtreePlusNodeAuditor
dependencyListHasIncorrectIvOffset: pathTerm object: obj indexObj: indexObj depList: depList ivOffset: ivOffset expectedIvOffset: expectedIvOffset
  pathTerm auditInfo: {obj.
      indexObj.
      depList} using: self.
  auditResultString
    add: 'DependencyList has incorrect iv offset ';
    add: ivOffset asString;
    add: ' instead of ';
    add: expectedIvOffset asString;
    add: Character lf
%

category: 'pathterm audit failures'
method: BtreePlusNodeAuditor
dependencyListHasIncorrectRefCount: pathTerm object: obj indexObj: indexObj depList: depList referenceCount: referenceCount numOccurrences: num
  pathTerm auditInfo: {obj.
      indexObj.
      depList} using: self.
  auditResultString
    add: ' DependencyList has incorrect ref count ';
    add: referenceCount asString;
    add: ' for occurrences ';
    add: num asString;
    add: Character lf
%

category: 'pathterm audit failures'
method: BtreePlusNodeAuditor
dependencyListHasNoEntryForPathTerm: pathTerm object: obj indexObj: indexObj depList: depList
  pathTerm auditInfo: {obj.
      indexObj.
      depList} using: self.
  auditResultString
    add: 'DependencyList has no entry for path term';
    add: Character lf
%

category: 'pathterm audit failures'
method: BtreePlusNodeAuditor
dependencyListUnexpectedLastElementLocation: pathTerm object: obj indexObj: indexObj depList: depList ivOffset: ivOffset 
  pathTerm auditInfo: {obj.
      indexObj.
      depList} using: self.
  auditResultString
    add: ' DependencyList on last element in unexepected location ';
    add: ivOffset asString;
    add: Character lf
%

category: 'pathterm audit failures'
method: BtreePlusNodeAuditor
pathTermIncorrectNumberOfBtreeEntries: pathTerm index: i offset: offset
  auditResultString
    add: Character lf;
    add: ' -- The number of entries in Btree [';
    add: (pathTerm at: i) btreeRoot asOop asString;
    add: '] for pathTerm [';
    add: pathTerm asOop asString;
    add: '] offset ';
    add: offset asString;
    add: ' does not match the number of entries in the base collection (extra elements in either the base collection or btree).';
    add: Character lf
%

category: 'pathterm audit failures'
method: BtreePlusNodeAuditor
pathTermObjectHasNoDependencyBitSet: pathTerm object: obj indexObj: indexObj depList: depList
  pathTerm auditInfo: {obj.
      indexObj.
      depList} using: self.
  auditResultString
    add: ' Object has DependencyList but no dependency bit is set';
    add: Character lf
%

category: 'pathterm audit failures'
method: BtreePlusNodeAuditor
pathTermObjectHasNoDependencyList: pathTerm object: obj indexObj: indexObj depList: depList
  pathTerm auditInfo: {obj.
      indexObj.
      depList} using: self.
  auditResultString
    add: ' Object has no dependency list';
    add: Character lf
%

! Class Implementation for BtreePlusRoot

! Remove existing behavior from BtreePlusRoot
removeallmethods BtreePlusRoot
removeallclassmethods BtreePlusRoot

! ------------------- Instance methods for BtreePlusRoot

category: 'Testing'
method: BtreePlusRoot
_canCompare: aKey withClass: aClass
  "Returns whether the receiver can make comparisons with the given key."

  ^ btreeRootNode _canCompare: aKey withClass: aClass
%

category: 'Accessing'
method: BtreePlusRoot
_createNewRootNodeForSplit: returnNode
  "Create a new root node and put the old root and the new split node into the new root."
  " returnNode is the second half of the split "

  |  node |
  " create the new parent node "
  node := btreeRootNode parentNodeClass new
    rootNode: self;
    objectSecurityPolicy: self objectSecurityPolicy;
    lastValue: returnNode lastValue;
    lastRoot: returnNode lastRoot.
  " insert the first half (the original root) "
  node
     _insertKey: (btreeRootNode _at: btreeRootNode _lastKeyIndex)
     value: btreeRootNode
     root: (btreeRootNode _at: btreeRootNode _lastRootIndex)
     atIndex: 1.
  " insert the second half "
  node
    _insertKey: (returnNode _at: returnNode _lastKeyIndex)
    value: returnNode
    root: (returnNode _at: returnNode _lastRootIndex)
    atIndex: node entrySize + 1. 
  ^ node
%

category: 'Searching Support'
method: BtreePlusRoot
_findAllValuesForIdenticalKey: aKey into: aCollection
  "Finds all the values for the given identical key, placing them in the
 collection.  Return whether the last key was equal or not (so the caller
 knows it must check the next leaf node)."

  ^ self btreeRootNode _findAllValuesForIdenticalKey: aKey into: aCollection
%

category: 'Searching Support'
method: BtreePlusRoot
_findAllValuesLessThanKey: aKey andEquals: aBoolean into: anArray using: aComparison
  "Puts into anArray the index and the leaf node of the entry that is less than
 (or less than and equal to if aBoolean is true) the given key.  Returns true
 if a value was found, false otherwise."

  ^ self btreeRootNode
    _findAllValuesLessThanKey: aKey
    andEquals: aBoolean
    into: anArray
    using: aComparison
%

category: 'Searching Support'
method: BtreePlusRoot
_putFirstIndexOfFirstChildInto: anArray
  "Puts the receiver and the first index of the receiver into the Array."

  self btreeRootNode _putFirstIndexOfFirstChildInto: anArray
%

category: 'Searching Support'
method: BtreePlusRoot
_putFirstIndexOfFirstChildInto: anArray ifGreaterThanOrEqualTo: aKey using: aComparison

"If first entry is greater than or equal to aKey, Puts the receiver 
 and the first index of the receiver into the Array.

 Because of NANs we need to validate the assumption of the caller 
 (i.e., that all entries are greater than aKey). See bug33805."

self btreeRootNode _putFirstIndexOfFirstChildInto: anArray ifGreaterThanOrEqualTo: aKey using: aComparison
%

category: 'Searching Support'
method: BtreePlusRoot
_putLastIndexOfLastChildInto: anArray
  "Asks the last child in the receiver to
 putLastIndexOfLastChildInto: anArray ."

  self btreeRootNode _putLastIndexOfLastChildInto: anArray
%

category: 'Audit'
method: BtreePlusRoot
auditNsc: nsc for: pathTerm offset: offset on: aString
  "Verifies that the btree structure consistent relative to the given pathTerm."

  | auditor |
  auditor := BtreePlusNodeAuditor new auditResultString: aString; yourself.
  ^self btreeRootNode auditNsc: nsc for: pathTerm offset: offset using: auditor
%

category: 'Accessing'
method: BtreePlusRoot
btreeAt: aKey put: aValue root: rootObject
  "Insert the key/value/rootObject tuple into the root B-tree node.  Must check to see if a
 split occurred.  If so, create a new root node and put the old root and the
 new split node into the new root."

  | returnNode |
  returnNode := self btreeRootNode at: aKey put: aValue root: rootObject.	
  returnNode == btreeRootNode
    ifFalse: [
      " create the new root node "
      btreeRootNode := self _createNewRootNodeForSplit: returnNode ]
%

category: 'Accessing'
method: BtreePlusRoot
btreeRootNode
  ^ btreeRootNode
    ifNil: [ btreeRootNode := indexObject btreePlusLeafNodeClass new
        rootNode: self;
        objectSecurityPolicy: self objectSecurityPolicy;
        yourself ]
%

category: 'Accessing'
method: BtreePlusRoot
btreeRootNode: aBtreePlusNode
  aBtreePlusNode rootNode: self.
  btreeRootNode := aBtreePlusNode
%

category: 'Accessing'
method: BtreePlusRoot
collator
  ^ self indexObject collator
%

category: 'Comparison Operators'
method: BtreePlusRoot
comparisonForSort
  "Answer the comparison object to be used for this index"

  ^ self indexObject comparisonForSort
%

category: 'Accessing'
method: BtreePlusRoot
indexObject
  "Answer the GsAbstractIndex that associated with this btree"

  ^ indexObject
%

category: 'Accessing'
method: BtreePlusRoot
indexObject: aGsAbstractIndex
  "Set the GsAbstractIndex that associated with this btree"

  indexObject := aGsAbstractIndex
%

category: 'Testing'
method: BtreePlusRoot
isIdentityIndex
  ^ self indexObject isIdentityIndex
%

category: 'Accessing'
method: BtreePlusRoot
maxNumberOfElements
  ^ self btreeRootNode maxNumberOfElements
%
category: 'Removing'
method: BtreePlusRoot
removeKey: aKey value: aValue root: rootObject
  "Removes the key and value from the B-tree root node.  Must check to see if
 a merge occurred in the root such that the root only contains a single
 entry.  If so, make the single entry the new root.  Returns whether the
 removal occurred."

  (btreeRootNode removeKey: aKey value: aValue root: rootObject)
    ifFalse: [ ^ false ].
  self updateBtreeRootNodeOnRemoval.
  ^ true
%

category: 'Statistics'
method: BtreePlusRoot
report
  | strm info leafCount totalElements |
  strm := AppendStream on: String new.
  info := {}.
  self btreeRootNode _putFirstIndexOfFirstChildInto: info.
  totalElements := leafCount := 0.
  (info at: 1) > 0
    ifTrue: [ | leaf |
      leaf := info at: 2.
      [ leaf notNil ] whileTrue: [ leafCount := leafCount + 1.
          totalElements := totalElements + leaf numElements.
          leaf := leaf nextLeaf ] ].
  leafCount == 0
    ifTrue: [ strm nextPutAll: 'The btree is empty' ]
    ifFalse: [ strm nextPutAll: 'There are ' , leafCount printString , ' leaf nodes with '
            , totalElements printString , ' elements ('
            , (totalElements / leafCount asScaledDecimal: 2) printString
            , ' elements/node).' ].
  ^ strm contents
%

category: 'Removing'
method: BtreePlusRoot
updateBtreeRootNodeOnRemoval
  "If the current root node is an interior node of size one, replace the current root node with its
   single child"

  (btreeRootNode numElements == 1 and: [ btreeRootNode isLeaf not ])
    ifTrue: [ btreeRootNode := btreeRootNode
        at: btreeRootNode _lastKeyIndex - btreeRootNode keyIndexOffset ].
%

! Class Implementation for RcBtreePlusRoot

! Remove existing behavior from RcBtreePlusRoot
removeallmethods RcBtreePlusRoot
removeallclassmethods RcBtreePlusRoot

! ------------------- Instance methods for RcBtreePlusRoot

category: 'Rc Accessing'
method: RcBtreePlusRoot
_btreeAt: aKey put: aValue root: rootObject logging: aBoolean
  "Insert the key/value/rootObject tuple into the root B-tree node.  Must check to see if a
 split occurred.  If so, create a new root node and put the old root and the
 new split node into the new root."

  ^ self
    _btreeAt: aKey
    put: aValue
    root: rootObject
    logging: aBoolean
    selectiveAbortSet: nil
%

category: 'Rc Accessing'
method: RcBtreePlusRoot
_btreeAt: aKey put: aValue root: rootObject logging: aBoolean selectiveAbortSet: selectiveAbortSetOrNil
  "Insert the key/value/rootObject tuple into the root B-tree node.  Must check to see if a
 split occurred.  If so, create a new root node and put the old root and the
 new split node into the new root."

  | returnNode node oldBTreeRootNode |
  returnNode := self btreeRootNode
    at: aKey
    put: aValue
    root: rootObject
    selectiveAbortSet: selectiveAbortSetOrNil.
  " see if a split occurred "
  returnNode == btreeRootNode
    ifFalse: [ 
      " returnNode is the second half of the split "
      self _addBtreePlusNodeToRcReadSet: returnNode.
      " create the new root node "
      node := self _createNewRootNodeForSplit: returnNode selectiveAbortSet: selectiveAbortSetOrNil.
     self _addBtreePlusNodeToRcReadSet: node.
     btreeRootNode _selectiveAbort: btreeRootNode ifNotIn: selectiveAbortSetOrNil.
     oldBTreeRootNode := btreeRootNode.
     btreeRootNode := node.
     "write dropped node to force a commit conflict and force replay, in case
      another session makes mods to the node"
     oldBTreeRootNode numElements: oldBTreeRootNode numElements ].
  System _addRootObjectToRcReadSet: self.
  aBoolean
    ifTrue: [ self _logBtreeAt: aKey put: aValue root: rootObject ].
  ^ true
%

category: 'Rc Logging'
method: RcBtreePlusRoot
_logBtreeAt: aKey put: aValue root: rootObject
  "Create a log entry for adding the given key/value/root."

  | logEntry |
  logEntry := LogEntry new.
  logEntry
    receiver: self;
    selector: #'_btreeAt:put:root:logging:selectiveAbortSet:';
    argArray:
        {aKey.
          aValue.
          rootObject.
          false.
          nil}.
  System redoLog addLogEntry: logEntry
%

category: 'Rc Logging'
method: RcBtreePlusRoot
_logBtreeRemoveKey: aKey value: aValue root: rootObject
  "Create a log entry for removing the given key/value/root."

  | logEntry |
  logEntry := LogEntry new.
  logEntry
    receiver: self;
    selector: #'_removeKey:value:root:logging:selectiveAbortSet:';
    argArray:
        {aKey.
          aValue.
          rootObject.
          false.
          nil}.
  System redoLog addLogEntry: logEntry
%

category: 'Rc Removing'
method: RcBtreePlusRoot
_removeKey: aKey value: aValue root: rootObject logging: aBoolean
  "Removes the key and value from the B-tree root node.  Must check to see if
 a merge occurred in the root such that the root only contains a single
 entry.  If so, make the single entry the new root.  Returns whether the
 removal occurred."

  ^self _removeKey: aKey value: aValue root: rootObject logging: aBoolean selectiveAbortSet: nil
%

category: 'Rc Removing'
method: RcBtreePlusRoot
_removeKey: aKey value: aValue root: rootObject logging: aBoolean selectiveAbortSet: selectiveAbortSetOrNil
  "Removes the key and value from the B-tree root node.  Must check to see if
 a merge occurred in the root such that the root only contains a single
 entry.  If so, make the single entry the new root.  Returns whether the
 removal occurred."

  (btreeRootNode removeKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil)
    ifFalse: [ ^ false ].
  self updateBtreeRootNodeOnRemoval: selectiveAbortSetOrNil.
  aBoolean 
    ifTrue: [ self _logBtreeRemoveKey: aKey value: aValue root: rootObject ].
   ^ true
%

category: 'Private'
method: RcBtreePlusRoot
_resolveRcConflictsWith: conflictObjects

"A logical write-write conflict has occurred on the receiver.  The objects that
 had the actual physical write-write conflicts are in the conflictObjects
 Array.  Selectively abort the receiver and then attempt to replay the
 operations maintained in the System redo log.  Returns true if all the
 operations could be replayed; otherwise returns false."

"inline implementation of _abortAndReplay: to abort old btrees before
 replaying log entries."

| redoLog logEntries selectiveAbortSet |
redoLog := System _redoLog.
    
" if no log entries to replay, then we're done "
redoLog == nil ifTrue: [ ^ false ].
logEntries := redoLog getLogEntriesFor: self .
logEntries == nil ifTrue:[ ^ false ].
   
" cannot perform selective abort if receiver has a dependency tag "
self _hasDependencyList ifTrue: [ ^ false ].

" explicitly abort the btrees that made up original changes, since we need to abort entire tree, before replaying, since
  nodes may split differently."
selectiveAbortSet := IdentitySet new.
1 to: conflictObjects size do:[ :j | | co |
  co := conflictObjects at: j.
  co _selectiveAbort.
  selectiveAbortSet add: co.
].
    
"abort all of the btree nodes involved in the transaction"
redoLog conflictObjects keysAndValuesDo:[ :conflictObject :redoObject |
  redoObject == self 
        ifTrue: [ 
                conflictObject _selectiveAbort.
                selectiveAbortSet add: conflictObject ].
].

" refresh the state of the receiver "
self _selectiveAbort.
    
" replay all operations on the receiver"
1 to: logEntries size do:[:j | | logEnt argArray |
  logEnt := logEntries at: j .
  argArray := logEnt argArray.
  argArray at:  argArray size put: selectiveAbortSet. "install the selectiveAbortSet into the #selectiveAbortSet arg slot (last slot)"
  logEnt redo ifFalse:[
    ^ false  "redo failed, cannot commit"
  ].
].
^ true
%

category: 'Accessing'
method: RcBtreePlusRoot
btreeAt: aKey put: aValue root: rootObject
  "Insert the key/value pair into the root B-tree node.  Must check to see if a
 split occurred.  If so, create a new root node and put the old root and the
 new split node into the new root."

  ^ self _btreeAt: aKey put: aValue root: rootObject logging: true
%

category: 'Removing'
method: RcBtreePlusRoot
removeKey: aKey value: aValue root: rootObject
  "Removes the key and value from the B-tree root node.  Must check to see if
 a merge occurred in the root such that the root only contains a single
 entry.  If so, make the single entry the new root.  Returns whether the
 removal occurred."

^ self _removeKey: aKey value: aValue root: rootObject logging: true
%

! Class Extensions

! Class initializers 

doit
true.
%

category: 'Searching Support'
method: BtreePlusRoot
_findAllValuesGreaterThanKey: aKey into: anArray using: aComparison
  "Puts into anArray the index and the leaf node of the entry that is greater
 than the given key.  Returns true if a value was found, false otherwise."

  ^ self btreeRootNode _findAllValuesGreaterThanKey: aKey into: anArray using: aComparison
%
category: 'pathterm audit failures'
method: BtreePlusNodeAuditor
pathTermIncorrectNilOnPathCount: pathTerm root: root got: actual  expected: expected
  auditResultString
    add: Character lf;
    add: ' -- Incorrect nil on path count for pathTerm [';
    add: pathTerm asOop asString;
    add: '] expected ';
    add: expected asString;
    add: ' got ';
    add: actual asString;
    add: ' for root object [';
    add: root asOop asString;
    add: '].';
    add: Character lf
%
category: 'pathterm audit failures'
method: BtreePlusNodeAuditor
pathTermInvalidIvOffset: pathTerm ivOffset: ivOffset object: obj
  auditResultString
    add: Character lf;
    add: ' -- Invalid ivOffset for pathTerm [';
    add: pathTerm asOop asString;
    add: ' an instance variable name [';
    add: pathTerm name asString;
    add: '] at ivOffset ';
    add: ivOffset asString;
    add: ' in the object [';
    add: obj asOop asString;
    add: '] does not exist.';
    add: Character lf
%
category: 'Updating'
method: BtreePlusInteriorKeyValueNode
updateLastChildRoot: lastChildRoot at: index 

  " Update the root object for receiver, if last child root has changed "

  "the root object is not updated ... as it is not recorded in the receiver "
%
category: 'Printing'
method: BtreePlusNode
prettyPrint

""

| aStream |
aStream := PrintStream printingOn: String new.
self printOn: aStream withIndentationLevel: 0 includingValues: false.
^ aStream contents
%
category: 'Printing'
method: BtreePlusLeafNode
printOn: aStream withIndentationLevel: anInteger

"Puts a displayable representation of the keys of the receiver on the given
 stream, indenting the given number of spaces."

anInteger timesRepeat: [ aStream nextPut: Character space ].

"aStream nextPutAll: '  { '; nextPutAll: self _alias asString; nextPutAll: ' } '."

2 to: (numElements * self entrySize) by: self entrySize do: [ :i |
    aStream nextPutAll: (super at: i) asString;
        nextPut: $(;
        nextPutAll: (super at: i - 1) asString;
        nextPutAll: ') '
].
aStream nextPut: Character lf
%
category: 'Printing'
method: BtreePlusLeafNode
printOn: aStream withIndentationLevel: anInteger includingValues: aBoolean

"Puts a displayable representation of the keys of the receiver on the given
 stream, indenting the given number of spaces."

| valueIndexOffset |
valueIndexOffset := self valueIndexOffset.

anInteger timesRepeat: [ aStream nextPut: Character space ].

2 to: (numElements * self entrySize) by: self entrySize do: [ :i |
    aStream nextPutAll: (self at: i) asString.

    aBoolean
        ifTrue: [
            aStream nextPut: $(;
                nextPutAll: (self at: i + valueIndexOffset) asString;
                nextPutAll: ') '
        ]
        ifFalse: [ aStream nextPutAll: ' ' ]
].
aStream nextPut: Character lf
%
category: 'Printing'
method: BtreePlusInteriorNode
printOn: aStream withIndentationLevel: anInteger

"Puts a displayable representation of the keys of the receiver on the given
 stream, indenting the given number of spaces."

anInteger timesRepeat: [ aStream nextPut: Character space ].
aStream nextPutAll: '[ ';
    nextPutAll: self lastValue asString;
    nextPutAll: ' ';
    nextPutAll: self lastRoot asString;
    nextPutAll: ' ]';

    nextPut: Character lf.

1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
    anInteger timesRepeat: [ aStream nextPut: Character space ].
    aStream nextPutAll: (self at: (i + self keyIndexOffset)) asString; nextPut: Character lf.
    (self at: i) printOn: aStream withIndentationLevel: (anInteger + 4)
]
%
category: 'Printing'
method: BtreePlusInteriorNode
printOn: aStream withIndentationLevel: anInteger includingValues: aBoolean

"Puts a displayable representation of the keys of the receiver on the given
 stream, indenting the given number of spaces."

aBoolean
    ifTrue: [
        anInteger timesRepeat: [ aStream nextPut: Character space ].
        aStream nextPutAll: '[ ';
            nextPutAll: self lastValue asString;
            nextPutAll: ' ';
            nextPutAll: self lastRoot asString;
            nextPutAll: ' ]';
            "nextPutAll: '  { '; nextPutAll: self _alias asString; nextPutAll: ' }';"
            nextPut: Character lf.
    ].

1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
    anInteger timesRepeat: [ aStream nextPut: Character space ].
    aStream nextPutAll: (self at: (i + 1)) asString; nextPut: Character lf.
    (self at: i) printOn: aStream withIndentationLevel: (anInteger + 4) includingValues: aBoolean
]
%
category: 'Updating'
method: BtreePlusInteriorNode
updateLastChildRoot: lastChildRoot at: index 

  " Update the root object for receiver, if last child root has changed "

  | rootIndexOffset |
  rootIndexOffset := self rootIndexOffset.
  lastChildRoot == (self at: index + rootIndexOffset)
    ifFalse: [  self primitiveAt: index + rootIndexOffset put: lastChildRoot ].
%

category: 'btree audit failures'
method: BtreePlusNodeAuditor
btree: btree parentNode: parentNode nextLeafMismatch: pathTerm nextLeaf: nextLeaf parentNextNode: nextNode
  auditResultString
    add: 'Btree [';
    add: btree asOop asString;
    add: '] has a next leaf node [';
    add: nextLeaf asOop asString;
    add: '] of class ';
    add: nextLeaf class asString;
    add: ' that does not match the next leaf node [';
    add: nextNode asOop asString;
    add: '] that is expected by the parent node [';
    add: parentNode asOop asString;
    add: ']';
    add: Character lf
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btree: btree parentNode: parentNode previousLeafMismatch: pathTerm previousLeaf: previousLeaf parentPreviousNode: previousNode
  auditResultString
    add: 'Btree [';
    add: btree asOop asString;
    add: '] has a previous leaf node [';
    add: previousLeaf asOop asString;
    add: '] of class ';
    add: previousLeaf class asString;
    add: ' that does not match the previous leaf node [';
    add: previousNode asOop asString;
    add: '] that is expected by parent node [';
    add: parentNode asOop asString;
    add: ']';
    add: Character lf
%

category: 'Accessing'
method: BtreePlusInteriorNode
nextLeaf
  "BtreePlusLeafNode compat"

  ^ nil
%
category: 'Accessing'
method: BtreePlusInteriorNode
nextLeaf: ignored
  "BtreePlusLeafNode compat - noop"
%
category: 'Accessing'
method: BtreePlusInteriorNode
previousLeaf
  "BtreePlusLeafNode compat"

  ^ nil
%
category: 'Accessing'
method: BtreePlusInteriorNode
previousLeaf:  ignored
  "BtreePlusLeafNode compat - noop"
%
category: 'Comparison Operators'
method: BtreePlusLeafNode
_compare: aComparison key: aKey lessThanOrEqualToEntryAt: index
                                                                                                           
"Perform a <= comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

^ aComparison compareKey: aKey lessThanOrEqualTo: (self at: index)
%
category: 'Comparison Operators'
method: BtreePlusLeafNode
_compare: aComparison key: aKey equalToEntryAt: index 

"Performs a = comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

^ aComparison compareKey: aKey equalTo: (self _at: index)
%
category: 'Searching'
method: BtreePlusRoot
_findFirstRootForKey: aKey using: aComparison
"Returns the first root object associated with the given key.  For interior nodes,
 find the appropriate child node and ask it to find the first root."

^ self btreeRootNode _findFirstRootForKey: aKey using: aComparison
%
category: 'Searching'
method: BtreePlusLeafNode
_findFirstRootForKey: aKey using: aComparison
  "Returns the first root object associated with the given key.  For leaf nodes, finds
 the appropriate entry and makes the comparison.  Follow nextLeaf if we run off end of
 recevier before finding a match. If one is not found, returns the incomplete path 
 traversal object."

  | index eSize maxIndex comparisonForSort keyIndexOffset |
  comparisonForSort := self comparisonForSort.
  index := self _findCoveringIndexForKey: aKey totalOrder: false using: comparisonForSort.
  index == 0
    ifTrue: [ ^ #'_incompletePathTraversal' ].
  eSize := self entrySize.
  maxIndex := self _lastEntryIndex.
  keyIndexOffset := self keyIndexOffset.
  " now scan entries until reach the end or no match "
  [ index <= maxIndex
    and: [ self _compare: comparisonForSort key: aKey equalToEntryAt: index + keyIndexOffset ] ]
    whileTrue: [ (self _compare: aComparison key: aKey equalToEntryAt: index + keyIndexOffset)
        ifTrue: [ ^ self rootObjectAt: index ].
      index := index + eSize ].
  (index > maxIndex and: [ self nextLeaf notNil ])
    ifTrue: [
      "continue the search with the next leaf node .. if nextLeaf not nil"
      ^ self nextLeaf _findFirstRootForKey: aKey using: aComparison ].
  ^ #'_incompletePathTraversal'
%
category: 'Searching'
method: BtreePlusInteriorNode
_findFirstRootForKey: aKey using: aComparison

"Returns the first root object associated with the given key.  For interior nodes,
 find the appropriate child node and ask it to find the first root."

| index |
" find first child node that might contain the key "
index := self _findCoveringIndexForKey: aKey totalOrder: false.
index == 0 ifTrue: [ ^ #_incompletePathTraversal ].

^ (self at: index) _findFirstRootForKey: aKey using: aComparison
%
category: 'Enumerating'
method: BtreePlusRoot
_leafKeysValuesAndRootsDo: aBlock

"Execute the two-argument block for each child node."

btreeRootNode _leafKeysValuesAndRootsDo: aBlock
%
category: 'Accessing'
method: BtreePlusRoot
totalElements

"Return the number of elements."

^ btreeRootNode totalElements
%
category: 'Enumerating'
method: BtreePlusLeafNode
_leafKeysValuesAndRootsDo: aBlock
  "Execute the two-argument block with the key and value of each entry in the
 receiver."

  | keyIndexOffset rootIndexOffset |
  keyIndexOffset := self keyIndexOffset.
  rootIndexOffset := self rootIndexOffset.
  1 to: numElements * self entrySize by: self entrySize do: [ :i | 
    aBlock
      value: (self at: i + keyIndexOffset)
      value: (self at: i)
      value: (self at: i + rootIndexOffset) ]
%
category: 'Accessing'
method: BtreePlusLeafNode
totalElements

"Return the number of elements."

^ numElements
%
category: 'Enumerating'
method: BtreePlusInteriorNode
_leafKeysValuesAndRootsDo: aBlock

"Execute the two-argument block for each child node."

1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
    (self at: i) _leafKeysValuesAndRootsDo: aBlock
]
%
category: 'Accessing'
method: BtreePlusInteriorNode
totalElements

"Return the total number of elements in the leaf nodes."

| sum |

sum := 0.
1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
    sum := sum + (self at: i) totalElements
].
^ sum
%
category: 'Accessing'
method: BtreePlusLeafNode
_totalElementsFrom: start toEndNode: endNode endIndex: endIndex 
  "Determine the number of entries in the receiver counting from start.
   If the receiver is the endNode, then count to endIndex, otherwise count
   to the last index of the receiver."

  | end delta |
  end := self == endNode
    ifTrue: [ endIndex ]
    ifFalse: [ end := self _lastIndex ].
  delta := end - start.
  delta < 0
    ifTrue: [ delta := delta negated ].
  ^ (delta // self entrySize) + 1
%
category: 'Accessing'
method: BtreePlusLeafNode
_totalElementsReversedFrom: start toEndNode: endNode endIndex: endIndex
  "Determine the number of entries in the receiver counting backwards from start.
   If the receiver is the endNode, then count backwards to endIndex, otherwise count
   backwards to 1."

  | end delta |
  end := self == endNode
    ifTrue: [ endIndex ]
    ifFalse: [ 1 ].
  delta := start - end.
  delta < 0
    ifTrue: [ delta := delta negated ].
  ^ (delta // self entrySize) + 1
%
category: 'Sorting Support'
method: BtreePlusRoot
_putAscendingRootObjectsInto: array startingAt: offset

"Returns the number of elements."

^ self btreeRootNode _putAscendingRootObjectsInto: array startingAt: offset
%
category: 'Sorting Support'
method: BtreePlusRoot
_putDescendingRootObjectsInto: array startingAt: offset

"Returns the number of elements."

^ self btreeRootNode _putDescendingRootObjectsInto: array startingAt: offset
%
category: 'Sorting Support'
method: BtreePlusLeafNode
_putAscendingRootObjectsInto: array startingAt: offset

"Returns the number of elements."

| j |
j := offset.
1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
  array at: j put: (self rootObjectAt: i).
  j := j + 1
].
^ numElements
%
category: 'Sorting Support'
method: BtreePlusLeafNode
_putDescendingRootObjectsInto: array startingAt: offset

"Returns the number of elements."

| j |
j := offset.
1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
  array at: j put: (self rootObjectAt: i).
  j := j - 1
].
^ numElements
%
category: 'Sorting Support'
method: BtreePlusInteriorNode
_putAscendingRootObjectsInto: array startingAt: offset

"Returns the number of elements."

| j |
j := offset.
1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
  j := j + ((self at: i) _putAscendingRootObjectsInto: array startingAt: j)
].
^ j - offset
%
category: 'Sorting Support'
method: BtreePlusInteriorNode
_putDescendingRootObjectsInto: array startingAt: offset

"Returns the number of elements."

| j |
j := offset.
1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
  j := j - ((self at: i) _putDescendingRootObjectsInto: array startingAt: j)
].
^ offset - j
%

category: 'Searching Support'
method: BtreePlusInteriorNode
_lastRootObjectsNotIdenticalTo: aKey into: array
  "Send this message to all the receiver's children."

  1 to: numElements * self entrySize by: self entrySize do: [ :i | (self at: i) _lastRootObjectsNotIdenticalTo: aKey into: array ].
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btree: btree outOfOrderKeyValueRootPairs: pathTerm key: k previous: previousKey value: v previous: previousValue root: r previous: previousRoot
  pathTerm auditInfo: {k.
      nil.
      nil} using: self.
  auditResultString
    add: 'Btree [';
    add: btree asOop asString;
    add: '] has out of oop order key/value/root where the key ';
    add: k printString printString;
    add: ' [';
    add: k asOop asString;
    add: '] is equal to previous key ';
    add: previousKey printString printString;
    add: ' [';
    add: previousKey asOop asString;
    add: '] and oop of the value ';
    add: v printString printString;
    add: ' [';
    add: v asOop asString;
    add: '] is equal to the oop of the previous value ';
    add: previousValue printString printString;
    add: ' [';
    add: previousValue asOop asString;
    add: '] but the oop of the root ';
    add: r printString printString;
    add: ' [';
    add: r asOop asString;
    add: '] is less than the oop of the previous root ';
    add: previousRoot printString printString;
    add: ' [';
    add: previousRoot asOop asString;
    add: ']';
    add: Character lf
%
category: 'Searching Support'
method: BtreePlusRoot
_lastRootObjectsNotIdenticalTo: aKey into: array

  self btreeRootNode _lastRootObjectsNotIdenticalTo: aKey into: array
%
category: 'Searching Support'
method: BtreePlusLeafNode
_lastRootObjectsNotIdenticalTo: aKey into: array
  ""

  1 to: numElements * self entrySize by: self entrySize do: [ :i | 
    (self keyAt: i) == aKey
      ifFalse: [ array add: (self rootObjectAt: i) ] ].
 
%
category: 'Searching'
method: BtreePlusInteriorNode
_findAllValuesForIdenticalKey: aKey into: aCollection using: aComparison

"Finds all the values for the given identical key, placing them in the
 collection.  For interior nodes, must first find the appropriate child
 node.  Return whether the last key was equal or not (so the caller
 knows it must check the next node)."

| index eSize maxIndex |
" find first child node that might contain the key "
index := self _findCoveringIndexForKey: aKey totalOrder: false.
index == 0 ifTrue: [ ^ false ].

((self at: index) _findAllValuesForIdenticalKey: aKey into: aCollection using: aComparison)
  ifFalse: [ ^ false ].

eSize := self entrySize.
maxIndex := self _lastIndex.
index := index + eSize.

" now scan child nodes until no values are added to the collection "
[ (index < maxIndex) and:
[ (self at: index) _findAllValuesForIdenticalKey: aKey into: aCollection using: aComparison ] ]
  whileTrue: [ index := index + eSize ].
^ index >= maxIndex
%
category: 'Printing'
method: BtreePlusRoot
prettyPrint

""

^self btreeRootNode prettyPrint
%
category: 'Audit'
method: BtreePlusNode
auditKey: key value: value root: rootObject bag: bag nsc: nsc for: pathTerm offset: offset bagOfRoots: bagOfRoots enumBag: enumBag  setValBag: setValBag using: auditor
  | num bagSize |
  num := bagOfRoots occurrencesOf: rootObject.
  pathTerm isEnumeratedTerm
    ifTrue: [ (num := enumBag occurrencesOf: value) * num ].
  pathTerm isSetValuedTerm
    ifTrue: [ (num := setValBag occurrencesOf: value) ].
  bagSize := bag size.
  bagSize == num
    ifFalse: [ auditor
        btree: self 
        incorrectNumberOfEntries: pathTerm
        key: key
        value: value
        bag: bag
        numExpected: num ].
  bag removeAll: bag.
  enumBag removeAll: enumBag.
  setValBag removeAll: setValBag.
  bagOfRoots removeAll: bagOfRoots.
%
category: 'Updating'
method: BtreePlusLeafKeyNode
_insertKey: aKey value: ignoredValue atIndex: insertionIndex
  "Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

self _insertKey: aKey atIndex: insertionIndex

%
category: 'Updating'
method: BtreePlusInteriorNode
_updateLastValue: selectiveAbortSetOrNil
  "Update the last value of the interior node by asking the last entry what its
 last value is.  Do not change the value unless there has been a change."

  | nodeLastValue nodeLastRoot   lastChildNode |
  lastChildNode := self at: self _lastEntryIndex.
  self _selectiveAbort: lastChildNode ifNotIn: selectiveAbortSetOrNil.
  nodeLastValue := lastChildNode lastValue.
  nodeLastValue == self lastValue
    ifFalse: [ self lastValue: nodeLastValue ].
  nodeLastRoot := lastChildNode lastRoot.
  nodeLastRoot == self lastRoot
    ifFalse: [ self lastRoot: nodeLastRoot ]
%
category: 'Enumerating'
method: BtreePlusLeafNode
_scanForRootObjectMapsStartingAt: startIndex end: endIndex key: aKey value: aValue valueOop: valueOop using: comparisonForCompare do: aBlock
  "Given the fact that the keys from startIndex through endIndex are all #=, we know that 
   all equal keys are sorted by the oop of the value.

     If the key at endIndex is equal to aKey
       If the oop of the value at endIndex is less than the oop of aValue we can skip 
         the elements of the entire leafNode.
     If the key at startIndex is equal to aKey
       If the oop of the value at startIndex is greater than the oop of aValue we can skip
         the elements of the entire leafNode.
"

  | index eSize keyIndexOffset rootIndexOffset firstKey lastKey |
  keyIndexOffset := self keyIndexOffset.
  firstKey := self at: startIndex + keyIndexOffset.
  (comparisonForCompare compareKey: aKey equalTo: firstKey)
    ifTrue: [ (self at: startIndex) asOop > valueOop ifTrue: [ ^ self ] ].
  lastKey := self at: endIndex + keyIndexOffset.
  (comparisonForCompare compareKey: aKey equalTo: lastKey)
    ifTrue: [ (self at: endIndex) asOop < valueOop ifTrue: [ ^ self ] ].
  index := startIndex.
  eSize := self entrySize.
  rootIndexOffset := self rootIndexOffset.
  [ index <= endIndex ] 
    whileTrue: [ 
      ((self at: index) == aValue and: [ (self at: index + keyIndexOffset) == aKey ])
        ifTrue: [ aBlock value: (self at: index + rootIndexOffset) ].
      index := index + eSize ]
%
category: 'Enumerating'
method: BtreePlusInteriorNode
_preOrderDo: aBlock
  aBlock value: self.
  1 to: numElements * self entrySize by: self entrySize do: [ :i | (self at: i) _preOrderDo: aBlock ]
%
category: 'Enumerating'
method: BtreePlusRoot
_preOrderDo: aBlock
  btreeRootNode _preOrderDo: aBlock
%
category: 'Enumerating'
method: BtreePlusLeafNode
_preOrderDo: aBlock
  aBlock value: self
%
category: 'Rc compatibility'
method: RcBtreePlusRoot
_addBtreePlusNodeToRcReadSet: aBtreePlusNode
  "Use from hard-to-reach methods where objects need to be added to rc read set."

  System _addRootObjectToRcReadSet: aBtreePlusNode.
  System redoLog addConflictObject: aBtreePlusNode for: self.
%
category: 'Rc compatibility'
method: BtreePlusRoot
_addBtreePlusNodeToRcReadSet: aBtreePlusNode
  "noop for non-rc btrees"
%
category: 'Updating'
method: BtreePlusInteriorNode
updateAfterAddAt: index child: node returnNode: returnNode selectiveAbortSet: selectiveAbortSetOrNil

  "Adds the key/value/root tuple to one of the leaf nodes.  If a split occurs, returns
   the new sibling, otherwise returns the receiver."

  "node already selectively aborted"

  | lastChildKey keyIndexOffset lastChildRoot |

  " see if last key in child node has changed "
  lastChildKey := node at: node _lastKeyIndex.
  keyIndexOffset := self keyIndexOffset.
  lastChildKey == (self at: index + keyIndexOffset)
    ifFalse: [ " update key and encryption in the receiver "
      self primitiveAt: index + keyIndexOffset put: lastChildKey ].
  " see if last root in child node has changed "
  lastChildRoot := node at: node _lastRootIndex.
  self updateLastChildRoot: lastChildRoot at: index.


  " see if a split occurred "
  returnNode == node
    ifTrue: [ 
      " if inserted into the last node, update the last value "
      index == self _lastEntryIndex
        ifTrue: [ self _updateLastValue: selectiveAbortSetOrNil ] ]
    ifFalse: [ " returnNode is the second half of the split "
      " determine if a split is needed in the receiver "
      numElements == self maxNumberOfElements
        ifTrue: [ | newSibling |
          newSibling := self
            _splitUsingKey: (node at: node _lastKeyIndex)
            node: node 
            newSibling: returnNode
            selectiveAbortSet: selectiveAbortSetOrNil.

          " update the last value and last root of the new sibling and the receiver "
          newSibling _updateLastValue: selectiveAbortSetOrNil.
          self _updateLastValue: selectiveAbortSetOrNil.
          self rootNode _addBtreePlusNodeToRcReadSet: newSibling.
          ^ newSibling ]
        ifFalse: [ " entry will fit in the receiver "

          " if child node was the last node "
          index == self _lastEntryIndex
            ifTrue: [
              self lastValue: returnNode lastValue.
              self lastRoot: returnNode lastRoot ].
          self
            _insertKey: (returnNode at: returnNode _lastKeyIndex)
            value: returnNode
            root: (returnNode at: returnNode _lastRootIndex)
            atIndex: index + self entrySize
            selectiveAbortSet: selectiveAbortSetOrNil.
          " see if last root in child node has changed "
          lastChildRoot := returnNode at: returnNode _lastRootIndex.
          self updateLastChildRoot: lastChildRoot at: index + self entrySize ] ]
%
category: 'Rc Updating'
method: BtreePlusNode
_selectiveAbort: aNode ifNotIn: selectiveAbortSetOrNil

  (selectiveAbortSetOrNil ~~ nil
    and: [ (selectiveAbortSetOrNil includes: aNode) not ])
    ifTrue: [ 
      selectiveAbortSetOrNil add: aNode.
      aNode _selectiveAbort ].
%
category: 'Comparison Operators'
method: BtreePlusInteriorKeyValueNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex root: ignored selectiveAbortSet: selectiveAbortSetOrNil 
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  ^ self compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex selectiveAbortSet: selectiveAbortSetOrNil 
%
category: 'Comparison Operators'
method: BtreePlusInteriorKeyValueNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex root: ignored selectiveAbortSet: selectiveAbortSetOrNil
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  ^ self compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex selectiveAbortSet: selectiveAbortSetOrNil
%
category: 'Updating'
method: BtreePlusLeafNode
_updateLastValue: selectiveAbortSetOrNil

"For leaf nodes, the last value is derived, so nothing is required."

%
category: 'Searching'
method: BtreePlusInteriorNode
binarySearchCoveringKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil
  "Returns the index for the entry in which aKey/aValue would be inserted.  This
 is the first entry which has its key >= aKey.  If multiple entries have the
 same key (key = aKey), then aValue is used (since entries with duplicate keys 
 are inserted by the value's OOP ordering).  If multiple entries have the same value 
 (value = aValue), then rootObject is used (since entries with duplicate values are
 inserted by the rootObject's OOP ordering). If the receiver is empty, returns 
 1.  Uses a binary search."

  | lowPt midPt highPt entrySize index entryGreaterThan entryEqualTo comparisonForSort keyIndexOffset |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ]
    whileTrue: [ 
      | objAtIndex |
      entryGreaterThan := false.
      entryEqualTo := false.
      midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      objAtIndex := self at: index.
      (comparisonForSort compareKey: aKey lessThan: objAtIndex)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ 
          (comparisonForSort compareKey: aKey greaterThan: objAtIndex)
            ifTrue: [ 
              lowPt := midPt + 1.
              entryGreaterThan := true ]
            ifFalse: [ 
              "If keys are equal, then secondary sort key is oop value, tertiary sort key os oop of rootObject"
              (comparisonForSort compareKey: aKey equalTo: objAtIndex)
                ifTrue: [ 
                  (self
                    compareValueOop: aValue
                    lessThanOrEqualToEntryValueOopAt: index
                    root: rootObject
                    selectiveAbortSet: selectiveAbortSetOrNil)
                    ifTrue: [ highPt := midPt - 1 ]
                    ifFalse: [ lowPt := midPt + 1 ].
                  entryEqualTo := true ]
                ifFalse: [ lowPt := midPt + 1 ] ] ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  (entryEqualTo
    and: [ 
      "use secondary sort key (oop of value), tertiary sort key os oop of rootObject"
      self
        compareValueOop: aValue
        greaterThanEntryValueOopAt: index
        root: rootObject
        selectiveAbortSet: selectiveAbortSetOrNil ])
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  ^ index - keyIndexOffset
%
category: 'Searching'
method: BtreePlusInteriorNode
binarySearchCoveringKey: aKey value: aValue selectiveAbortSet: selectiveAbortSetOrNil
  "Returns the index for the entry in which aKey/aValue would be inserted.  This
 is the first entry which has its key >= aKey.  If multiple entries have the
 same key (key = aKey), then aValue is used (since entries with duplicate keys 
 are inserted by the value's OOP ordering). If the receiver is empty, returns 
 1.  Uses a binary search."

  | lowPt midPt highPt entrySize index entryGreaterThan entryEqualTo comparisonForSort keyIndexOffset |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ] whileTrue: [ | objAtIndex |
      entryGreaterThan := false.
      entryEqualTo := false.
      midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      objAtIndex := self at: index.
      (comparisonForSort compareKey: aKey lessThan: objAtIndex)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ (comparisonForSort compareKey: aKey greaterThan: objAtIndex)
            ifTrue: [ lowPt := midPt + 1.
              entryGreaterThan := true ]
            ifFalse: [ "If keys are equal, then secondary sort key is oop value, tertiary sort key os oop of rootObject"
              (comparisonForSort compareKey: aKey equalTo: objAtIndex)
                ifTrue: [ (self
                    compareValueOop: aValue
                    lessThanOrEqualToEntryValueOopAt: index
                    selectiveAbortSet: selectiveAbortSetOrNil)
                    ifTrue: [ highPt := midPt - 1 ]
                    ifFalse: [ lowPt := midPt + 1 ].
                  entryEqualTo := true ]
                ifFalse: [ lowPt := midPt + 1 ] ] ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  (entryEqualTo
    and: [ "use secondary sort key (oop of value), tertiary sort key os oop of rootObject"
      self
        compareValueOop: aValue
        greaterThanEntryValueOopAt: index
        selectiveAbortSet: selectiveAbortSetOrNil ])
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  ^ index - keyIndexOffset
%
category: 'Updating'
method: BtreePlusInteriorNode
binarySearchInteriorNodeCoveringKey: aKey put: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil

  "Find the index of the first child node in which the insertion would occur"

  | index lastKeyIndex |
  index := self binarySearchCoveringKey: aKey value: aValue root: rootObject  selectiveAbortSet: selectiveAbortSetOrNil.
  lastKeyIndex := self _lastKeyIndex.	
  " if insertion would occur past the last entry, use the last entry "
  index > lastKeyIndex
    ifTrue: [ index := lastKeyIndex - self keyIndexOffset ].
  ^ index.
%
category: 'Comparison Operators'
method: BtreePlusInteriorNode
compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: rootObject selectiveAbortSet: selectiveAbortSetOrNil
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, then use the OOP of the value for the
 comparison. If the value oops are equal, use the OOP of the rootObject for the 
 comparison."

  | keyIndexEntry |
  keyIndexEntry := self at: keyIndex.
  (comparisonForSort compareKey: aKey lessThan: keyIndexEntry)
    ifTrue: [ ^ true ].
  (comparisonForSort compareKey: aKey equalTo: keyIndexEntry)
    ifTrue: [ | entryIndexEntry entryIndexEntryLastEntry |
      entryIndexEntry := self at: keyIndex - self keyIndexOffset.
      self _selectiveAbort: entryIndexEntry ifNotIn: selectiveAbortSetOrNil.
      "aValue asOop = entryIndexEntry lastValue asOop"
      entryIndexEntryLastEntry := entryIndexEntry lastValue.
      aValue == entryIndexEntryLastEntry
        ifTrue: [ ^ rootObject asOop <= entryIndexEntry lastRoot asOop ].
      ^ aValue asOop < entryIndexEntryLastEntry asOop ].
  ^ false
%
category: 'Comparison Operators'
method: BtreePlusInteriorNode
compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: rootObject selectiveAbortSetOrNil: selectiveAbortSetOrNil
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, then use the OOP of the value for the
 comparison. If the value oops are equal, use the OOP of the rootObject for the 
 comparison."

  | keyIndexEntry |
  keyIndexEntry := self at: keyIndex.
  (comparisonForSort compareKey: aKey lessThan: keyIndexEntry)
    ifTrue: [ ^ true ].
  (comparisonForSort compareKey: aKey equalTo: keyIndexEntry)
    ifTrue: [ | entryIndexEntry entryIndexEntryLastEntry |
      entryIndexEntry := self at: keyIndex - self keyIndexOffset.
      self _selectiveAbort: entryIndexEntry ifNotIn: selectiveAbortSetOrNil.
      "aValue asOop = entryIndexEntry lastValue asOop"
      entryIndexEntryLastEntry := entryIndexEntry lastValue.
      aValue == entryIndexEntryLastEntry
        ifTrue: [ ^ rootObject asOop <= entryIndexEntry lastRoot asOop ].
      ^ aValue asOop < entryIndexEntryLastEntry asOop ].
  ^ false
%
category: 'Comparison Operators'
method: BtreePlusInteriorNode
compareLastKeyLessThanOrEqualToKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil
  "Returns whether the receivers last key/value pair is greater than or equal to
the given key/value pair."

  numElements == 0
    ifTrue: [ ^ false ].
  " return aKey <= last key "
  ^ self
    compare: self comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: self _lastKeyIndex
    root: rootObject
    selectiveAbortSet: selectiveAbortSetOrNil
%
category: 'Comparison Operators'
method: BtreePlusInteriorNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex selectiveAbortSet: selectiveAbortSetOrNil 
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  | entryNode |
  entryNode := self at: keyIndex - self keyIndexOffset.
  self _selectiveAbort: entryNode ifNotIn: selectiveAbortSetOrNil.
  ^ aValue asOop > entryNode lastValue asOop
%
category: 'Comparison Operators'
method: BtreePlusInteriorNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex selectiveAbortSet: selectiveAbortSetOrNil
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index."

  | entryNode |
  entryNode := self at: keyIndex - self keyIndexOffset.
  self _selectiveAbort: entryNode ifNotIn: selectiveAbortSetOrNil.
  ^ aValue asOop <= entryNode lastValue asOop
%
category: 'Comparison Operators'
method: BtreePlusInteriorKeyValueRootObjectNode
compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex root: aRootObject selectiveAbortSet: selectiveAbortSetOrNil 
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is 
   at the given index. If the value oops are equal, compare the oop of aRootObject and the 
   oop of the root object at the given index."

  | entryValue entryValueLastValue |
  entryValue := self at: keyIndex - self keyIndexOffset.
  self _selectiveAbort: entryValue ifNotIn: selectiveAbortSetOrNil.
  entryValueLastValue := entryValue lastValue.
  "aValue asOop = entryValueLastValue asOop"
  aValue == entryValueLastValue
    ifTrue: [ ^ aRootObject asOop > entryValue lastRoot asOop ].
  ^ aValue asOop > entryValueLastValue asOop
%
category: 'Comparison Operators'
method: BtreePlusInteriorKeyValueRootObjectNode
compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex root: aRootObject selectiveAbortSet: selectiveAbortSetOrNil
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is 
   at the given index. If the value oops are equal, compare the oop of aRootObject and the 
   oop of the root object at the given index."

  | entryValue entryValueLastValue |
  entryValue := self at: keyIndex - self keyIndexOffset.
  self _selectiveAbort: entryValue ifNotIn: selectiveAbortSetOrNil.
  entryValueLastValue := entryValue lastValue.
  "aValue asOop = entryValueLastValue asOop"
  aValue == entryValueLastValue
    ifTrue: [ ^ aRootObject asOop <= entryValue lastRoot asOop ].
  ^ aValue asOop < entryValueLastValue asOop
%
category: 'Accessing'
method: RcBtreePlusRoot
_createNewRootNodeForSplit: returnNode selectiveAbortSet: selectiveAbortSetOrNil
  "Create a new root node and put the old root and the new split node into the new root."
  " returnNode is the second half of the split "

  |  node |
  " create the new parent node "
  node := btreeRootNode parentNodeClass new
    rootNode: self;
    objectSecurityPolicy: self objectSecurityPolicy;
    lastValue: returnNode lastValue;
    lastRoot: returnNode lastRoot.
  " insert the first half (the original root) "
  node
     _insertKey: (btreeRootNode at: btreeRootNode _lastKeyIndex)
     value: btreeRootNode
     root: (btreeRootNode at: btreeRootNode _lastRootIndex)
     atIndex: 1
     selectiveAbortSet: selectiveAbortSetOrNil.
  " insert the second half "
  node
    _insertKey: (returnNode at: returnNode _lastKeyIndex)
    value: returnNode
    root: (returnNode at: returnNode _lastRootIndex)
    atIndex: node entrySize + 1
    selectiveAbortSet: selectiveAbortSetOrNil. 
  ^ node
%
category: 'Updating'
method: BtreePlusLeafKeyNode
_insertKey: aKey value: ignoredValue root: ignoredRoot atIndex: insertionIndex selectiveAbortSet: ignored

"Compat with BtreePlustInteriorNode api ... inserts do not require selective abort logic"

self _insertKey: aKey value: ignoredValue root: ignoredRoot atIndex: insertionIndex 
%
category: 'Updating'
method: BtreePlusNode
_createSibling: selectiveAbortSetOrNil
  "Creates a new sibling node and move the second half half of the receiver's entries into the
sibling.  Returns the new sibling."

  ^ self _createSibling
%
category: 'Updating'
method: BtreePlusLeafKeyValueRootObjectNode
_insertKey: aKey value: aValue root: rootObject atIndex: insertionIndex selectiveAbortSet: ignored
"Compat with BtreePlustInteriorNode api ... inserts do not require selective abort logic"

  self _insertKey: aKey value: aValue root: rootObject atIndex: insertionIndex 

%
category: 'Updating'
method: BtreePlusInteriorKeyValueNode
_insertKey: aKey value: newPredecessorNode atIndex: insertionIndex  selectiveAbortSet: selectiveAbortSetOrNil
  "Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

  | endIndex entrySize |
  entrySize := self entrySize.
  endIndex := numElements * entrySize.

  " newPredecessorNode node is being inserted in front of node at insertionIndex, update the previous/next leaf values for both nodes"
  (self at: insertionIndex) 
    ifNil: [
      insertionIndex > 1 
        ifTrue: [ 
          | previousNode |
          previousNode := self at: insertionIndex - entrySize.
          self _selectiveAbort: previousNode ifNotIn: selectiveAbortSetOrNil.
          newPredecessorNode 
            _updateLeafNodeLinksForNewPreviousLeaf:  previousNode
            newNextLeaf: previousNode nextLeaf
            selectiveAbortSet: selectiveAbortSetOrNil ] ]
    ifNotNil: [:formerPredecessorNode | 
      self _selectiveAbort: formerPredecessorNode ifNotIn: selectiveAbortSetOrNil.
      formerPredecessorNode 
        _updateLeafNodeLinksForNewPreviousLeaf: newPredecessorNode 
        newNextLeaf: formerPredecessorNode nextLeaf
        selectiveAbortSet: selectiveAbortSetOrNil ].

  self _insertAt: insertionIndex 
       value: newPredecessorNode value: aKey 
       numToMoveDown: endIndex - insertionIndex + 1 .

  numElements := numElements + 1


%
category: 'Updating'
method: BtreePlusInteriorKeyValueNode
_insertKey: aKey value: aValue root: ignored atIndex: insertionIndex selectiveAbortSet: selectiveAbortSetOrNil

"Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

self _insertKey: aKey value: aValue atIndex: insertionIndex selectiveAbortSet: selectiveAbortSetOrNil
%
category: 'Updating'
method: BtreePlusLeafNode
_createSibling: selectiveAbortSetOrNil
  "Creates a new sibling node and moves half of the receiver's entries into the
sibling.  Returns the new sibling."

  | newSibling nLeaf |
  newSibling := super _createSibling.
  nLeaf := self nextLeaf.
  newSibling 
    previousLeaf: self;
    nextLeaf: nLeaf.
  nLeaf 
    ifNotNil: [ 
      self _selectiveAbort: nLeaf ifNotIn: selectiveAbortSetOrNil.
      nLeaf previousLeaf: newSibling.
      self rootNode _addBtreePlusNodeToRcReadSet: nLeaf ].
  self nextLeaf: newSibling.
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  ^ newSibling
%
category: 'Searching'
method: BtreePlusLeafNode
_findAndCountKey: aKey value: aValue root: rootObj into: col

  | index eSize maxIndex keyIndexOffset comparisonForSort rootIndexOffset |
  comparisonForSort := self comparisonForSort.
  index := self _findCoveringIndexForKey: aKey totalOrder: false.
  index == 0
    ifTrue: [ ^ false].
  eSize := self entrySize.
  maxIndex := self _lastEntryIndex.
  keyIndexOffset := self keyIndexOffset.
  rootIndexOffset := self rootIndexOffset.
  " now scan entries until reach the end or no match "
  [ index <= maxIndex
    and: [ self _compare: comparisonForSort key: aKey equalToEntryAt: index + keyIndexOffset] ]
    whileTrue: [ 
      (((self at: index + keyIndexOffset) == aKey) and: [((self at: index) == aValue) and: [(self at: index + rootIndexOffset) == rootObj]]) ifTrue: [ col add: rootObj ]. 
      index := index + eSize ].
  ^ index >= maxIndex
%
category: 'Updating'
method: BtreePlusLeafNode
_updateLeafNodeLinksForNewPreviousLeaf: newPreviousLeaf newNextLeaf: newNextLeaf
  " newPreviousNode is being inserted in before of receiver, update the previous/next leaf values for both nodes"

  | originalPrevious  |
  originalPrevious := self previousLeaf.
  originalPrevious ~~ newPreviousLeaf
    ifTrue: [ 
      newPreviousLeaf 
        previousLeaf: originalPrevious; 
        nextLeaf: self.
      originalPrevious ifNotNil: [  originalPrevious nextLeaf: newPreviousLeaf ].
      self 
        previousLeaf: newPreviousLeaf; 
        nextLeaf: newNextLeaf.
      newNextLeaf 
        ifNotNil: [ newNextLeaf previousLeaf: self ] ].
%
category: 'Updating'
method: BtreePlusLeafNode
_updateLeafNodeLinksForNewPreviousLeaf: newPreviousLeaf newNextLeaf: newNextLeaf  selectiveAbortSet: selectiveAbortSetOrNil
  " newPreviousNode is being inserted in before of receiver, update the previous/next leaf values for both nodes"

  | originalPrevious  |
  originalPrevious := self previousLeaf.
  originalPrevious ~~ newPreviousLeaf
    ifTrue: [ 
      originalPrevious ifNotNil: [ 
        self _selectiveAbort: originalPrevious ifNotIn: selectiveAbortSetOrNil ].
      newPreviousLeaf ifNotNil: [ 
        self _selectiveAbort: newPreviousLeaf ifNotIn: selectiveAbortSetOrNil ].
      newNextLeaf  ifNotNil: [ 
        self _selectiveAbort: newNextLeaf ifNotIn: selectiveAbortSetOrNil ].
      newPreviousLeaf 
        previousLeaf: originalPrevious; 
        nextLeaf: self.
      self rootNode _addBtreePlusNodeToRcReadSet: newPreviousLeaf.
      originalPrevious ifNotNil: [ 
        originalPrevious nextLeaf: newPreviousLeaf.
        self rootNode _addBtreePlusNodeToRcReadSet: originalPrevious ].
      self 
        previousLeaf: newPreviousLeaf; 
        nextLeaf: newNextLeaf.
      self rootNode _addBtreePlusNodeToRcReadSet: self.
      newNextLeaf 
        ifNotNil: [ 
          newNextLeaf previousLeaf: self.
          self rootNode _addBtreePlusNodeToRcReadSet: newNextLeaf ] ].
%
category: 'Updating'
method: BtreePlusLeafKeyValueNode
_insertKey: aKey value: aValue root: ignored atIndex: insertionIndex selectiveAbortSet: ignoredSet

"Compat with BtreePlustInteriorNode api ... inserts do not require selective abort logic"

self _insertKey: aKey value: aValue root: ignored atIndex: insertionIndex
%
category: 'Searching'
method: BtreePlusInteriorNode
_findAndCountKey: aKey value: aValue root: rootObj into: col
"return whether the last key was equal or not (so the caller knows to check next node"

| index eSize maxIndex |
index := self _findCoveringIndexForKey: aKey totalOrder: false.
index == 0 ifTrue: [ ^ false ].

((self at: index) _findAndCountKey: aKey value: aValue root: rootObj into: col)
  ifFalse: [ ^ false ].

eSize := self entrySize.
maxIndex := self _lastIndex.
index := index + eSize.

[ (index < maxIndex) and:
[ (self at: index) _findAndCountKey: aKey value: aValue root: rootObj into: col ] ]
  whileTrue: [ index := index + eSize ].
^ index >= maxIndex
%
category: 'Updating'
method: BtreePlusInteriorNode
_updateLeafNodeLinksForNewPreviousLeaf: newPreviousLeaf newNextLeaf: newNextLeaf
  " newPreviousNode is being inserted in before of receiver, update the previous/next leaf values for both nodes"

  "currently a noop"
%
category: 'Updating'
method: BtreePlusInteriorNode
_updateLeafNodeLinksForNewPreviousLeaf: newPreviousLeaf newNextLeaf: newNextLeaf  selectiveAbortSet: selectiveAbortSetOrNil
  " newPreviousNode is being inserted in before of receiver, update the previous/next leaf values for both nodes"

  "currently a noop"
%
category: 'btree audit failures'
method: BtreePlusNodeAuditor
btree: btree incorrectCountForKeyValueRootPairs: pathTerm key: k value: v  root: r found: foundCount expected: expectedCount
  pathTerm auditInfo: {k.
      nil.
      nil} using: self.
  auditResultString
    add: 'Btree [';
    add: btree asOop asString;
    add: '] has incorrect count for key/value/root tuple; key ';
    add: k printString printString;
    add: ' [';
    add: k asOop asString;
    add: '] value ';
    add: v printString printString;
    add: ' [';
    add: v asOop asString;
    add: '] root ';
    add: r printString printString;
    add: ' [';
    add: r asOop asString;
    add: ']. Found ';
    add: foundCount printString ;
    add: ' occurences and expected ';
    add: expectedCount printString;
    add: Character lf
%
category: 'Updating'
method: BtreePlusInteriorKeyValueRootObjectNode
_insertKey: aKey value: newPredecessorNode root: rootObject atIndex: insertionIndex selectiveAbortSet: selectiveAbortSetOrNil
  "Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

  | endIndex entrySize |
  entrySize := self entrySize.
  endIndex := numElements * entrySize.

  " newPredecessorNode node is being inserted in front of node at insertionIndex, update the previous/next leaf values for both nodes"
  (self at: insertionIndex) 
    ifNil: [
      insertionIndex > 1 
        ifTrue: [
          | previousNode |
          previousNode := self at: insertionIndex - entrySize.
          self _selectiveAbort: previousNode ifNotIn: selectiveAbortSetOrNil.
          newPredecessorNode 
            _updateLeafNodeLinksForNewPreviousLeaf: previousNode 
            newNextLeaf: previousNode nextLeaf
            selectiveAbortSet: selectiveAbortSetOrNil ] ]
    ifNotNil: [:formerPredecessorNode | 
      self _selectiveAbort: formerPredecessorNode ifNotIn: selectiveAbortSetOrNil.
      formerPredecessorNode 
        _updateLeafNodeLinksForNewPreviousLeaf: newPredecessorNode 
        newNextLeaf: formerPredecessorNode nextLeaf
        selectiveAbortSet: selectiveAbortSetOrNil ].

  self _insertAt: insertionIndex 
       value: newPredecessorNode value: aKey value: rootObject
       numToMoveDown: endIndex - insertionIndex + 1.       

  numElements := numElements + 1

%
category: 'Comparison Operators'
method: BtreePlusLeafNode
compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: rootObject selectiveAbortSet: selectiveAbortSetOrNil

  "Performs a <= comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, then use the OOP of the value for the
 comparison. If the value oops are equal, use the OOP of the rootObject for the 
 comparison."

 ""

^ self compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: rootObject
%
category: 'Updating'
method: BtreePlusLeafKeyValueNode
at: aKey put: aValue root: ignored selectiveAbortSet: selectiveAbortSetOrNil
  "Adds the key/value pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  | index |
  " determine if a split is needed "
  self _selectiveAbort: self ifNotIn: selectiveAbortSetOrNil.
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  numElements == self maxNumberOfElements
    ifTrue: [ " performs the split and returns the new sibling "
      ^ self _splitUsingKey: aKey value: aValue selectiveAbortSet: selectiveAbortSetOrNil ]
    ifFalse: [ " entry will fit in leaf node "
      numElements == 0
        ifTrue: [ index := 1 ]
        ifFalse: [ index := self binarySearchCoveringKey: aKey value: aValue ].
      self _insertKey: aKey value: aValue atIndex: index ]
%
category: 'Updating'
method: BtreePlusLeafKeyNode
at: aKey put: aValue root: ignoredRoot selectiveAbortSet: selectiveAbortSetOrNil
  "Adds the key/value pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  | index |
  " determine if a split is needed "
  self _selectiveAbort: self ifNotIn: selectiveAbortSetOrNil.
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  numElements == self maxNumberOfElements
    ifTrue: [ " performs the split and returns the new sibling "
      ^ self _splitUsingKey: aKey value: aValue selectiveAbortSet: selectiveAbortSetOrNil ]
    ifFalse: [ " entry will fit in leaf node "
      numElements == 0
        ifTrue: [ index := 1 ]
        ifFalse: [ index := self binarySearchCoveringKey: aKey ].
      self _insertKey: aKey atIndex: index ]
%
category: 'Updating'
method: BtreePlusLeafKeyValueRootObjectNode
at: aKey put: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil
  "Adds the key/value/root pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  | index |
  " determine if a split is needed "
  self _selectiveAbort: self ifNotIn: selectiveAbortSetOrNil.
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  numElements == self maxNumberOfElements
    ifTrue: [ " performs the split and returns the new sibling "
      ^ self _splitUsingKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil ]
    ifFalse: [ " entry will fit in leaf node "
      numElements == 0
        ifTrue: [ index := 1 ]
        ifFalse: [ index := self binarySearchCoveringKey: aKey value: aValue root: rootObject ].
      self
        _insertKey: aKey
        value: aValue
        root: rootObject
        atIndex: index ]
%
category: 'Updating'
method: BtreePlusLeafNode
_splitUsingKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil
  "Creates a new sibling node, moving the second half of the receiver's entries into the
 sibling.  Then inserts a new entry with the given key/value/rootObject into either the
 receiver or sibling.  Returns the new sibling."

  | index newSibling lastKeyIndex lastKey comparisonForSort |
  newSibling := self _createSibling: selectiveAbortSetOrNil.
  lastKeyIndex := self _lastKeyIndex.
  lastKey := self at: lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (self
    compare: comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: lastKeyIndex
    root: rootObject)
    ifTrue: [ " adding to the first half "
      index := self
        binarySearchCoveringKey: aKey
        value: aValue
        root: rootObject.
      self
        _insertKey: aKey
        value: aValue
        root: rootObject
        atIndex: index ]
    ifFalse: [ " adding to the second half "
      index := newSibling
        binarySearchCoveringKey: aKey
        value: aValue
        root: rootObject.
      newSibling
        _insertKey: aKey
        value: aValue
        root: rootObject
        atIndex: index ].
  ^ newSibling
%
category: 'Updating'
method: BtreePlusLeafNode
_splitUsingKey: aKey value: aValue selectiveAbortSet: selectiveAbortSetOrNil
  "Creates a new sibling node, moving the second half of the receiver's entries into the
 sibling.  Then inserts a new entry with the given key/value into either the
 receiver or sibling.  Returns the new sibling."

  | index newSibling lastKeyIndex lastKey comparisonForSort |
  newSibling := self _createSibling: selectiveAbortSetOrNil.
  lastKeyIndex := self _lastKeyIndex.
  lastKey := super at: lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (self
    compare: comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: lastKeyIndex
    root: nil)
    ifTrue: [ " adding to the first half "
      index := self
        binarySearchCoveringKey: aKey
        value: aValue.
      self
        _insertKey: aKey
        value: aValue
        atIndex: index ]
    ifFalse: [ " adding to the second half "
      index := newSibling
        binarySearchCoveringKey: aKey
        value: aValue.
      newSibling
        _insertKey: aKey
        value: aValue
        atIndex: index ].
  ^ newSibling
%
category: 'Removing'
method: RcBtreePlusRoot
updateBtreeRootNodeOnRemoval: selectiveAbortSetOrNil
  "If the current root node is an interior node of size one, replace the current root node with its
   single child"

  | oldBTreeRootNode |
  (btreeRootNode numElements == 1 and: [ btreeRootNode isLeaf not ])
    ifTrue: [ 
      btreeRootNode _selectiveAbort: btreeRootNode ifNotIn: selectiveAbortSetOrNil.
     oldBTreeRootNode := btreeRootNode.
      btreeRootNode := btreeRootNode
        at: btreeRootNode _lastKeyIndex - btreeRootNode keyIndexOffset.
     "write dropped node to force a commit conflict and force replay, in case
      another session makes mods to the node"
     oldBTreeRootNode numElements: oldBTreeRootNode numElements ].
  System _addRootObjectToRcReadSet: self
%
category: 'Updating'
method: BtreePlusInteriorNode
_splitUsingKey: aChildKey node: aChildNode newSibling: newChildSibling
  "Called when aChileNode itself was split and there wasn't room in the receiver for the 
   newChildSibling. Creates a new sibling node of the receiver; moving the second half of the 
   receiver's entries into the newSibling; finds the sibling where aChildNode ended up and inserts 
   newChildSibling immediately after aChildNode. "

  | insertionIndex newSibling lastKeyIndex lastKey lastVal lastRootObject comparisonForSort |
  newSibling := self _createSibling.
  " get the last key in the receiver "
  lastKeyIndex := self _lastKeyIndex.
  lastKey := self at: lastKeyIndex.
  " get the last value and last root in the new entry's node (this is what insertion is based upon) "
  lastVal := aChildNode lastValue.
  lastRootObject := aChildNode lastRoot.
  " determine if childNode end up in the first half (self) or second half (newSibling)"
  comparisonForSort := self comparisonForSort.
  (self
    compare: comparisonForSort
    key: aChildKey
    value: lastVal
    lessThanOrEqualToEntryAt: lastKeyIndex
    root: lastRootObject)
    ifTrue: [ 
      "aChildNode ended up in first half"
      insertionIndex := self binarySearchCoveringKey: aChildKey value: lastVal root: lastRootObject.
      "insert newChildSibling after aChildNode"
      self 
        _insertKey: (newChildSibling at: newChildSibling _lastKeyIndex) 
        value: newChildSibling 
        root: newChildSibling lastRoot
        atIndex: insertionIndex + self entrySize ]
    ifFalse: [ 
      "aChildNode ended up in second half"
       insertionIndex := newSibling binarySearchCoveringKey: aChildKey value: lastVal root: lastRootObject .
      "insert newChildSibling after aChildNode"
      newSibling 
        _insertKey: (newChildSibling at: newChildSibling _lastKeyIndex) 
        value: newChildSibling 
        root: newChildSibling lastRoot
        atIndex: insertionIndex + self entrySize ].
  ^ newSibling
%
category: 'Updating'
method: BtreePlusInteriorNode
_splitUsingKey: aChildKey node: aChildNode newSibling: newChildSibling selectiveAbortSet: selectiveAbortSetOrNil
  "Called when aChileNode itself was split and there wasn't room in the receiver for the 
   newChildSibling. Creates a new sibling node of the receiver; moving the second half of the 
   receiver's entries into the newSibling; finds the sibling where aChildNode ended up and inserts 
   newChildSibling immediately after aChildNode. "

  | insertionIndex newSibling lastKeyIndex lastKey lastVal lastRootObject comparisonForSort |
  newSibling := self _createSibling: selectiveAbortSetOrNil.
  " get the last key in the receiver "
  lastKeyIndex := self _lastKeyIndex.
  lastKey := self at: lastKeyIndex.
  " get the last value and last root in the new entry's node (this is what insertion is based upon) "
  lastVal := aChildNode lastValue.
  lastRootObject := aChildNode lastRoot.
  " determine if childNode end up in the first half (self) or second half (newSibling)"
  comparisonForSort := self comparisonForSort.
  (self
    compare: comparisonForSort
    key: aChildKey
    value: lastVal
    lessThanOrEqualToEntryAt: lastKeyIndex
    root: lastRootObject
    selectiveAbortSet: selectiveAbortSetOrNil)
    ifTrue: [ 
      "aChildNode ended up in first half"
      insertionIndex := self binarySearchCoveringKey: aChildKey value: lastVal root: lastRootObject selectiveAbortSet: selectiveAbortSetOrNil.
      "insert newChildSibling after aChildNode"
      self 
        _insertKey: (newChildSibling at: newChildSibling _lastKeyIndex) 
        value: newChildSibling 
        root: newChildSibling lastRoot
        atIndex: insertionIndex + self entrySize
        selectiveAbortSet: selectiveAbortSetOrNil ]
    ifFalse: [ 
      "aChildNode ended up in second half"
       insertionIndex := newSibling binarySearchCoveringKey: aChildKey value: lastVal root: lastRootObject selectiveAbortSet: selectiveAbortSetOrNil.
      "insert newChildSibling after aChildNode"
      newSibling 
        _insertKey: (newChildSibling at: newChildSibling _lastKeyIndex) 
        value: newChildSibling 
        root: newChildSibling lastRoot
        atIndex: insertionIndex + self entrySize
        selectiveAbortSet: selectiveAbortSetOrNil ].
  ^ newSibling
%

category: 'pathterm audit failures'
method: BtreePlusNodeAuditor
setValuedPathTermExtraNilOnPathCount: pathTerm root: root excessCount: excessNilOnPathCount   auditResultString
    add: Character lf;
    add: ' -- Nil on path count too high for pathTerm [';
    add: pathTerm asOop asString;
    add: '] excess ';
    add: excessNilOnPathCount asString;
    add: ' for root object [';
    add: root asOop asString;
    add: '].';
    add: Character lf
%
