#!/bin/bash
#set -xv

my_name=`basename $0`
usage(){   
    echo "Usage: $my_name certName [client | server]"
    echo "where:"
    echo "   certName is a unique name for the certificate"
    echo "      Usually this will be the name of the person or entity owning the certificate."
    echo "      Spaces are not allowed and will be replaced by underscore characters."
    echo "   client or server is type of certificate to create"
}

if [ $# -ne 2 ]; then
    usage
    exit 1
fi

. sslcommon.sh
if [ $OK -ne 1 ]; then
    exit 1
fi

CN=$1
CN=${CN// /_}
KIND=$2

if [ "$KIND" = "client" ]; then
    EXTENSION=client_cert
elif [ "$KIND" = "server" ]; then
    EXTENSION=server_cert
else
    usage
    exit 1
fi

# Do NOT delete CA files here!
CAPASSWD=$SSL_BASE_DIR/private/ca_key_password.txt
ensureFileExists $CAPASSWD 

CACERT=$SSL_BASE_DIR/certs/cacert.pem
ensureFileExists $CACERT

CAKEY=$SSL_BASE_DIR/private/cacertkey.pem
ensureFileExists $CAKEY

# Begin files we create.  Delete on error.
KEY=$SSL_BASE_DIR/private/${CN}_${KIND}key.pem
ensureFileDoesNotExist $KEY

PASSWD=$SSL_BASE_DIR/private/${CN}_${KIND}_passwd.txt
ensureFileDoesNotExist $PASSWD

CSR=$SSL_BASE_DIR/certs/${CN}_${KIND}req.pem
ensureFileDoesNotExist $CSR

CERT=$SSL_BASE_DIR/certs/${CN}_${KIND}cert.pem
ensureFileDoesNotExist $CERT

CLEANUP_FILES="${KEY} ${PASSWD} ${CSR} ${CERT}"

cleanup(){
    echo "Removing ${CLEANUP_FILES}"
    rm -f ${CLEANUP_FILES} >/dev/null 2>&1
}

# Now do the work

# Create random strong password
$OPENSSL rand -base64 40 >${PASSWD}
checkForError $CLEANUP_FILES
ensureFileExists $PASSWD $CLEANUP_FILES

# Create new private key with password from above and cert request
$OPENSSL req -newkey rsa:4096 -sha256 -keyout ${KEY} \
	 -out ${CSR} -extensions $EXTENSION \
	 -passout file:${PASSWD} \
	 -subj "/C=US/ST=OR/L=Beaverton/O=GemTalk\ Systems/CN=${CN}"
checkForError $CLEANUP_FILES

# Sign the cert
$OPENSSL ca -in $CSR  -out $CERT \
-cert $CACERT -keyfile $CAKEY \
-extensions $EXTENSION \
-passin file:${CAPASSWD} -preserveDN -batch -notext 

checkForError $CLEANUP_FILES

echo "Success! ${KIND} certificate and private key created"
echo "Private key password located in file ${PASSWD}"

exit 0
