!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   SmallInteger, Integer, Number, Magnitude, Object.
!
!=========================================================================

removeallmethods SmallInteger
removeallclassmethods SmallInteger

category: 'For Documentation Installation only'
classmethod: SmallInteger
installDocumentation

self comment:
'Instances of SmallInteger are an optimization for commonly occurring integers
 (between -(2 to the 60 power) and ((2 to the 60 power) - 1) ).  You may not
 create subclasses of class SmallInteger.  You may not create any new
 SmallIntegers.  Note that all instances of a given SmallInteger refer to a
 single, unique GemStone object.  That is, they are all both equal (=) and
 identical (==).'.
%

category: 'Converting'
method: SmallInteger
_coerce: aNumber

"Reimplemented from Integer."

^aNumber truncated
%

category: 'Converting'
method: SmallInteger
_digitsAsString

"Returns self as a String of at least two digits."

| aString |

(self < 10)
  ifTrue: [aString := '0' , self asString]
  ifFalse: [aString := self asString].
^ aString
%

category: 'Converting'
method: SmallInteger
_digitsAsString3

"Returns self as a String of at least three digits."

self < 100 ifTrue:[
  (self < 10) ifTrue: [
    ^ '00' , self asString
  ] ifFalse:[ 
    ^ '0', self asString 
  ].
] ifFalse:[
  ^ self asString
] 
%

category: 'Converting'
method: SmallInteger
asString

"Returns a String that indicates the numeric value of the
 receiver.  Positive values do not include a leading + ."

<primitive: 54>
self _primitiveFailed: #asString .
self _uncontinuableError
%

! asFloat inherited from Integer

category: 'Converting'
method: SmallInteger
asDecimalFloat

"Returns a DecimalFloat representing the receiver."

<primitive: 123>

self _primitiveFailed: #asDecimalFloat .
self _uncontinuableError
%

category: 'Converting'
method: SmallInteger
_generality

"Returns the generality of a SmallInteger"

^20
%

category: 'Clustering'
method: SmallInteger
clusterDepthFirst

"Returns true.  (Because SmallIntegers are self-defining objects,
 this method has no effect.)"

^ true
%

category: 'Comparing'
method: SmallInteger
< aNumber

"< is a special selector in SmallInteger, this implementation is used only
 by perform:  .
 Returns true if the receiver is less than aNumber;
 returns false otherwise. 

 Should not be reimplemented.  The native code generator generates
 in-line implementations of SmallInteger >> < . Reimplementations
 with different semantics than primitive 3 will not take effect 
 unless native code is disabled."

<primitive: 3>
aNumber _isInteger ifFalse:[
  aNumber _isFloat ifTrue:[
    ^ self asFloat < aNumber
  ].
  ^ self _retry: #< coercing: aNumber
].
^ super < aNumber
%

category: 'Comparing'
method: SmallInteger
> aNumber

"Returns true if the receiver is greater than aNumber;
 returns false otherwise.

 Should not be reimplemented.  The native code generator generates
 in-line implementations of SmallInteger >> > . Reimplementations
 with different semantics than primitive 4 will not take effect 
 unless native code is disabled."

<primitive: 4>
aNumber _isInteger ifFalse:[
  aNumber _isFloat ifTrue:[
    ^ self asFloat > aNumber
  ].
  ^ self _retry: #> coercing: aNumber
].
^ super > aNumber
%

category: 'Comparing'
method: SmallInteger
<= aNumber

"<= is a special selector in SmallInteger, this implementation is used only
 by perform:  .
 Returns true if the receiver is less than or equal to
 aNumber; returns false otherwise.

 Should not be reimplemented.  The native code generator generates
 in-line implementations of SmallInteger >> <= . Reimplementations
 with different semantics than primitive 5 will not take effect 
 unless native code is disabled."

<primitive: 5>
aNumber _isInteger ifFalse:[
  aNumber _isFloat ifTrue:[
    ^ self asFloat <= aNumber
  ].
  ^ self _retry: #<= coercing: aNumber
].
^ super <= aNumber
%

category: 'Comparing'
method: SmallInteger
>= aNumber

">= is a special selector in SmallInteger, this implementation is used only
 by perform:  .
 Returns true if the receiver is greater than or equal to
 aNumber; returns false otherwise.

 Should not be reimplemented.  The native code generator generates
 in-line implementations of SmallInteger >> >= . Reimplementations
 with different semantics than primitive 6 will not take effect 
 unless native code is disabled."

<primitive: 6>
aNumber _isInteger ifFalse:[
  aNumber _isFloat ifTrue:[
    ^ self asFloat >= aNumber
  ].
  ^ self _retry: #>= coercing: aNumber
].
^ super >= aNumber
%

category: 'Comparing'
method: SmallInteger
= aNumber

"Returns true if the receiver is equal to
 aNumber; returns false otherwise.  

 Should not be reimplemented.  The native code generator generates
 in-line implementations of SmallInteger>>= . Reimplementations
 with different semantics than primitive 7 will not take effect 
 unless native code is disabled."

<primitive: 7>
aNumber _isNumber ifTrue:[ 
  aNumber _isFloat ifTrue:[ ^ self asFloat = aNumber].
  aNumber _isInteger ifTrue:[ ^ super = aNumber ].
  ^ self _retry: #= coercing: aNumber
].
^ false 
%

category: 'Comparing'
method: SmallInteger
~= aNumber

"Returns true if the receiver is not equal to aNumber;
 returns false otherwise.

 Should not be reimplemented.  The native code generator generates
 in-line implementations of SmallInteger>>~= . Reimplementations
 with different semantics than primitive 8 will not take effect 
 unless native code is disabled."

<primitive: 8>
aNumber _isNumber ifTrue:[ 
  aNumber _isFloat ifTrue:[ ^ self asFloat ~= aNumber ].
  aNumber _isInteger ifTrue:[ ^ super ~= aNumber ].
   ^ self _retry: #~= coercing: aNumber
].
^ true
%

! hash is now inherited from Integer

category: 'Comparing'
method: SmallInteger
identityHash

"Returns a numeric hash index.  For a SmallInteger, returns the receiver."

^self
%

category: 'Copying'
method: SmallInteger
copy

"Overrides the inherited method; you cannot create any new SmallIntegers.
 Returns the receiver."

^ self
%

! the selectors #+, #-, #*   are optimized selectors and don't need
!  implementations here.  perform of these selectors is handled by
!  the implementations in Integer .

category: 'Arithmetic'
method: SmallInteger
/ aNumber

"Returns the result of dividing the receiver by aNumber."

<primitive: 10>
(aNumber _isInteger) ifTrue:[ 
  (aNumber == 0) ifTrue: [^ self _errorDivideByZero].
  ^ Fraction numerator: self denominator: aNumber 
].
^ super / aNumber
%

category: 'Arithmetic'
method: SmallInteger
// aNumber

"Divides the receiver by aNumber.
 Returns the integer quotient, with truncation toward negative infinity.
 For example,

 9//4 = 2
 -9//4 = -3

 The selector \\ returns the remainder from this division."

<primitive: 12>
(aNumber == 0) ifTrue: [^ self _errorDivideByZero].
^ super // aNumber
%


category: 'Arithmetic'
method: SmallInteger
quo: aNumber

"Divides the receiver by aNumber.  Returns the integer quotient, with
 truncation toward zero.  For example,

 -9 quo: 4 = -2

 The selector rem: returns the remainder from this division."

<primitive: 13>
(aNumber == 0)
  ifTrue: [^ self _errorDivideByZero]
  ifFalse: [^ super quo: aNumber]
%

! fixed 42513
category: 'Arithmetic'
method: SmallInteger
rem: aNumber

"Returns the integer remainder defined in terms of quo: (division
 of the receiver by aNumber, with truncation toward zero)."

<primitive: 188>
(aNumber == 0)
  ifTrue: [^ self _errorDivideByZero]
  ifFalse: [^ super rem: aNumber]
%

method: SmallInteger
quoRem: aNumber into: anArray

"anArray should be an Array of size 2.
 Performs
   anArray at: 1 put: (self quo: aNumber) .
   anArray at: 2 put: (self rem: aNumber) .
 Returns anArray . 
"
<primitive: 950>
(aNumber == 0) ifTrue: [^ self _errorDivideByZero].
anArray _isArray ifFalse:[ anArray _validateClass: Array ].
^ self _retry: #quoRem:into: coercing: aNumber into: anArray
%
method: SmallInteger
divMod: aNumber into: anArray

"anArray should be an Array of size 2.
 Performs
   anArray at: 1 put: (self // aNumber) .
   anArray at: 2 put: (self \\ aNumber) .
 Returns anArray .
"
<primitive: 951>
(aNumber == 0) ifTrue: [^ self _errorDivideByZero].
anArray _isArray ifFalse:[ anArray _validateClass: Array ].
^ self _retry: #divMod:into: coercing: aNumber into: anArray
%

category: 'Arithmetic'
method: SmallInteger
\\ aNumber

"Returns the modulus defined in terms of //.  Returns a Number with the same
 sign as the argument aNumber.  For example,

 9\\4 = 1
 -9\\4 = 3
 9\\-4 = -3"

<primitive: 11>
(aNumber == 0)
  ifTrue: [^ self _errorDivideByZero]
  ifFalse: [^ super \\ aNumber]
%

category: 'Arithmetic'
method: SmallInteger
abs

"Returns a Number that is the absolute value of the receiver."

self < 0 ifTrue:[ ^ 0 - self ].
^ self
%

category: 'Bit Manipulation'
method: SmallInteger
bitAnd: aNumber

"SmallInteger >> bitAnd: is a special send if both receiver and arg
 are SmallIntegers .
 Returns an Integer whose bits are the logical and of the receiver's bits and
 the bits of aNumber."

<primitive: 14>
^aNumber bitAnd: self
%

category: 'Bit Manipulation'
method: SmallInteger
bitOr: aNumber

"Returns an Integer whose bits are the logical or of the receiver's bits and
 the bits of aNumber."

<primitive: 15>
^aNumber bitOr: self
%

! fixed 33631
category: 'Bit Manipulation'
method: SmallInteger
bitShift: shift

"Returns an Integer whose value (in two's-complement representation) is the
 receiver's value (also in two's-complement representation) shifted left by
 'shift' bits.

 Negative arguments shift right.  Zeros are shifted in from the right in left
 shifts.  The sign bit is extended in right shifts."

<primitive: 17>
shift _validateClass: SmallInteger.
^ super bitShift: shift
%

category: 'Bit Manipulation'
method: SmallInteger
bitXor: aNumber

"SmallInteger >> bitXor: is a special send if both receiver and arg
 are SmallIntegers .
 Returns an Integer whose bits are the logical xor of the receiver's bits and
 the bits of aNumber."

<primitive: 16>
^aNumber bitXor: self
%

category: 'Bit Manipulation'
method: SmallInteger
bitAt: bitNumber

"Returns the bit at the ith position of the receiver, where 1 is the least
 significant bit.

 Gs64 v3.0 changed argument from 0 based to 1 based. "

| num |
bitNumber _isSmallInteger ifFalse:[ bitNumber _validateClass: SmallInteger ].
(bitNumber < 1) ifTrue: [^ bitNumber _error: #rtErrArgNotPositive].
num := bitNumber . 
(bitNumber > 61) ifTrue:[  num := 61 "sign extension semantics"].
^ (self bitShift: (1 - num)) bitAnd: 1 .
%

! fixed 36141
! fixed 42163
category: 'Bit Manipulation'
method: SmallInteger
highBit

"Returns the index of the high-order bit that is set
 in the binary representation of the receiver.  (If the receiver is a negative
 integer, takes its absolute value first.)  If the receiver is zero, this
 returns nil.  
 As of Gs64 v3.2  the least significant bit has index 1 ,
 in previous versions the least significant bit had index 0 .

 Example:   4 highBit == 3  "

<primitive: 853>
self _primitiveFailed: #highBit 
%

category: 'Bit Manipulation'
method: SmallInteger
asBitString

"Returns a string of 1 and 0 Characters representing the bits in the receiver."

| bit num bits str |

bits := { }.
bit := 1.
num := self.
[
  bits add: (num bitAnd: bit).
  num := num bitAnd: bit bitInvert.
  bit := bit bitShift: 1.
  num = 0
] untilTrue.

str := String new.
bits size _downTo: 1 do: [ :i | 
  str add: ((bits at: i) > 0 ifTrue: [$1] ifFalse: [$0]) 
].

^str.
%

category: 'Formatting'
method: SmallInteger
printString

"Returns a String whose contents are a displayable representation of the
 receiver."

"Reimplemented for efficiency.  GemStone does not allow the creation of new
 kinds of Number, so there is no point in creating a stream and sending
 printOn:"

^self asString
%

category: 'Queries'
classmethod: SmallInteger
maximumValue

"Returns the maximum allowable SmallInteger value."

^ 1152921504606846975
%
classmethod: SmallInteger
maximum32bitInteger
  "Returns the maximum value representable by a 32bit signed integer"
  ^ 16r7FFFFFFF
%

category: 'Queries'
classmethod: SmallInteger
minimumValue

"Returns the minimum allowable SmallInteger value."

^ -1152921504606846976
%

category: 'Storing and Loading'
method: SmallInteger
containsIdentity

"Private."

^true
%
category: 'Testing'
method: SmallInteger
isSpecial

"Returns true if the receiver is a special object."

^ true
%
category: 'Private'
method: SmallInteger
_floatParts

" Returns a 2 element Array describing receiver;
    first element is a Float , 
    second element is an Integer .
  The Float represents the most significant 53 bits
  of the receiver, rounded the same as  for Integer>>asFloat .
  The second element is the power of 2 describing the
  remaining bits of the receiver."

^ { self asFloat . 0 }
%

category: 'Arithmetic'
method: SmallInteger
raisedTo: aNumber

"Returns the receiver raised to the power of the argument."
<primitive: 662>
FloatingPointError _checkFpStatus .

"primitive handles SmallInteger and Float args. "

aNumber _isInteger ifTrue:[ 
  ^ super raisedToInteger: aNumber
].
^ super raisedTo: aNumber  
%

category: 'Arithmetic'
method: SmallInteger
raisedToInteger: aNumber

aNumber _isInteger ifFalse:[ aNumber _validateClass: Integer ].
^ self raisedTo: aNumber
%

category: 'Testing'
method: SmallInteger
_bottomBit
  "used by raisedTo methods"
  ^ self bitAnd: 1
%

category: 'Private'
method: SmallInteger
_leaveProtectedMode
  "(Reserved selector) a special selector compiled to a bytecode.
   This method is for documentation only and should not be used.
   See System(C) >> _protectedMode for more information."

 ^ self _primitiveFailed:#_leaveProtectedMode 
%

category: 'Unicode'
method: SmallInteger
asUnicodeDigitRadix: radixSmallInt
  "Returns a Character representing the receiver as a digit
   in radix radixSmallInt, or signals an Error .
   Result is computed by calling u_forDigit(self, radix) in libicu .
  "

<primitive: 967>
radixSmallInt _validateClass: SmallInteger .
(radixSmallInt < 0 or:[ radixSmallInt > 16rFF]) ifTrue:[
   OutOfRange new 
     name: 'radixSmallInt' min: 0 max: 16rFF actual: radixSmallInt ;
     signal
].
ArgumentError signal:'no valid digit translation'.
 ^ self _primitiveFailed: #asUnicodeDigitRadix: args: { radixSmallInt }
%

category: 'Private'
method: SmallInteger
_validateMin: aMin max: aMax 
  (self < aMin or:[ self > aMax]) ifTrue:[
    self _error: #rtErrArgOutOfRange args:{ aMin . aMax } 
  ]
%

