!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: rclowmaintidentitybag.gs 41207 2017-03-03 18:54:21Z bretlb $
!
! Superclass Hierarchy:
!   RcLowMaintenanceIdentityBag, IdentityBag, UnorderedCollection, Collection, Object.
!
!=========================================================================

expectvalue %String
run
| cls |
cls := Globals at: #RcLowMaintenanceIdentityBag otherwise: nil.
^ cls ifNil:[
   IdentityBag subclass: 'RcLowMaintenanceIdentityBag'
   instVarNames: #()
   classVars: #()
   classInstVars: #()
   poolDictionaries: #()
   inDictionary: Globals
   options: #().
          'Created class RcLowMaintenanceIdentityBag'
   ]
   ifNotNil:[ 'RcLowMaintenanceIdentityBag class already exists' ].
%
expectvalue true

set class RcLowMaintenanceIdentityBag
removeallmethods 
removeallclassmethods 

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment: 
'RcLowMaintenanceIdentityBag is a special kind of IdentityBag that provides 
IdentityBag functionality, but with no transaction conflicts between multiple 
sessions that add objects to the bag, and a single session that removes objects. 

When concurrent sessions update an RcLowMantenanceIdentityBag and a conflict
is detected during commit, the conflict is resolved by replaying the operation.  

RcLowMaintenanceIdentityBag provides the same collection and conflict handling 
behavior as RcIdentityBag, but the internal implemention is different. 
Unlike RcIdentityBag, an RcLowMaintenanceIdentityBag does not require cleanup.  
RcLowMaintenanceIdentityBag is expected to have better performance since
fewer objects must be faulted into memory, and should require less disk space.

GemStone recommends migrating instances of RcIdentityBag to 
RcLowMaintenanceIdentityBag. The components-based implementation of RcIdentityBag 
may be deprecated and replaced by the replay-based (renamed) 
RcLowMaintenanceIdentityBag in a future release.  
'
%

category: 'Instance Creation'
classmethod:
new

"Returns a new RcLowMaintenanceIdentityBag."

^super new
%

! ------------------- Instance methods for RcLowMaintenanceIdentityBag

category: 'Converting'
method:
asIdentityBag

^ IdentityBag withAll: self
%

category: 'Class Membership'
method:
species

"Returns the class to use to select and reject queries."

^ IdentityBag
%

category: 'Adding'
method:
add: anObject

"Adds anObject to the RcLowMaintenanceIdentityBag and returns anObject.
 Nils are ignored, i.e., not added to the bag."

anObject ifNil:[ ^ anObject "ignore nils" ].
self _rcIncludes: anObject .  "add path to leaf node to rcReadSet"
self addRedoLogEntryFor: #_redoAdd:  withArgs: { anObject } .

^ super add: anObject
%

category: 'Adding'
method:
addAll: aCollection

"Adds all of the elements of aCollection to the receiver and returns
 aCollection."

System _addToRcReadSet: self includingAllNodes: true. 
self addRedoLogEntryFor: #_redoAddAll:  withArgs: { aCollection } .

^ super addAll: aCollection
%

category: 'Adding'
method:
add: anObject withOccurrences: aSmallInteger

"Includes anObject as an element of the receiver aSmallInteger number of times."

anObject ifNil:[ ^ anObject "ignore nils" ].
self _rcIncludes: anObject .  "add path to leaf node to rcReadSet"
self addRedoLogEntryFor: #_redoAddWithOccurrences:  withArgs: { { anObject . aSmallInteger } }.
^ super add: anObject withOccurrences: aSmallInteger
%

category: 'Removing'
method:
remove: anObject

"Removes anObject if present from the receiver.  
 Generates an error if anObject is not in the receiver.  Returns anObject. 
"

anObject ifNil:[ ^ anObject "ignore nils" ].
(self includesIdentical: anObject)  ifTrue:[
  self _rcIncludes: anObject .  "add path to leaf node to rcReadSet"
  self addRedoLogEntryFor: #_redoRemove:  withArgs: { anObject } .
  ^ super remove: anObject
] ifFalse:[
  ^ self _errorNotFound: anObject
]
%

category: 'Removing'
method:
remove: anObject ifAbsent: aBlock

"Removes from the receiver an object that is identical to anObject and
 returns anObject.  If anObject is not present in the receiver, 
 evaluates aBlock and returns the result of the evaluation."

(self includesIdentical: anObject) ifTrue: [
  self _rcIncludes: anObject .  "add path to leaf node to rcReadSet"
  self addRedoLogEntryFor: #_redoRemove: withArgs: { anObject } .
  ^ super remove: anObject 
]
ifFalse: [ ^ aBlock value ]
%

category: 'Removing'
method:
remove: anObject otherwise: notFoundValue

"Removes from the receiver an object that is identical to anObject and returns
 anObject.  If anObject is not present in the receiver, returns notFoundValue."

(self includesIdentical: anObject) ifTrue: [
   self _rcIncludes: anObject .  "add path to leaf node to rcReadSet"
   self addRedoLogEntryFor: #_redoRemove: withArgs: { anObject } .
   ^ super remove: anObject 
]
ifFalse: [ ^ notFoundValue ]
%

category: 'Removing'
method:
removeAll: aCollection

"Removes all of the elements of aCollection from the receiver and returns aCollection.
 Generates an error if any object in the collection is not in the receiver.
"

System _addToRcReadSet: self includingAllNodes: true. 
self addRedoLogEntryFor: #_redoRemoveAll: withArgs: { aCollection copy } .

^ super removeAll: aCollection
%

category: 'Removing'
method:
removeAllPresent: aCollection

"Removes from the receiver each element of aCollection that is also an
 element of the receiver.  Differs from removeAll: in that, if some
 elements of aCollection are not present in the receiver, no error is generated.
 Returns aCollection."

System _addToRcReadSet: self includingAllNodes: true. 
self addRedoLogEntryFor: #_redoRemoveAllPresent: withArgs: { aCollection } .

^super removeAllPresent: aCollection
%

category: 'Removing'
method:
removeIdentical: anObject

"Same as remove:."

^self remove: anObject
%

category: 'Removing'
method:
removeIdentical: anObject ifAbsent: aBlock

"Same as remove:ifAbsent:."

^self remove: anObject ifAbsent: aBlock
%

category: 'Removing'
method:
removeIdentical: anObject otherwise: notFoundValue

"Same as remove:otherwise:."

^ self remove: anObject otherwise: notFoundValue
%

category: 'Removing'
method:
removeIfPresent: anObject

"Removes from the receiver an object that is identical to anObject and
 returns anObject.  Returns nil if anObject is not present in the receiver."

(self includesIdentical: anObject) ifTrue: [
  self _rcIncludes: anObject .  "add path to leaf node to rcReadSet"
  self addRedoLogEntryFor: #_redoRemove: withArgs: { anObject } .
  ^ super remove: anObject 
] ifFalse: [ ^ nil ]
%


! ------------------- Methods to support Rc replay

category: 'Reduced Conflict Support'
method:
_abortAndReplay: conflictObjects
 
"Abort the receiver and replay operations on the receiver from the redo log."
 
| redoLog logEntries |
_indexedPaths 
  ifNotNil: [ 
    _indexedPaths _anyIndexIsLegacy 
      ifTrue: [ 
        "abort and replay not supported when legacy index is involved"
        ^false ] ].

redoLog := System _redoLog.
 
" if no log entries to replay, then we're done "
redoLog ifNil: [ ^ false ].
logEntries := redoLog getLogEntriesFor: self .
logEntries ifNil:[ ^ false ].

" cannot perform selective abort if receiver has a dependency tag "
self _hasDependencyList ifTrue: [ ^ false ].
 
" Refresh the state of the receiver."

self _selectiveAbort.

" tell the redo log to replay any operations on the receiver "
^ redoLog _redoOperationsForEntries: logEntries 
%

category: 'Private'
method:
addRedoLogEntryFor: aSelector withArgs: args

"Creates a redo log entry for the selector with the specified argument array,
 adds it to the redolog for this session.
 Sender responsible for rcReadSet additions.
"

| redo logEntry | 

redo := System redoLog.
logEntry := LogEntry new.
logEntry receiver: self;
         selector: aSelector ;
         argArray: args.
redo addLogEntry: logEntry.
redo addConflictObject: self for: self.
%

category: 'Private'
method:
_redoAdd: anObject

"Performs the replay of adding anObject to the RcLowMaintenanceIdentityBag and returns true."

self _rcIncludes: anObject .
self _addForReplay: anObject.
^ true
%

category: 'Private'
method:
_redoAddAll: aCollection
  "Performs the replay of adding aCollection to the receiver and returns true."

  System _addToRcReadSet: self includingAllNodes: true. 
  aCollection _isRcIdentityBag
    ifTrue: [ self _addAll: aCollection _asIdentityBag forReplay: true ]
    ifFalse: [ self _addAll: aCollection forReplay: true ].
  ^ true
%

category: 'Private'
method:
_redoAddWithOccurrences: args
  "Performs the replay of adding arg1 to the receiver arg2 times and returns true."
  | obj |
  self _rcIncludes: (obj := args at: 1 ) .
  super add: obj  withOccurrences: (args at: 2).
  ^ true
%

category: 'Private'
method:
_redoRemove: anObject

"Performs the replay of removing anObject from the receiver and returns true."

self _rcIncludes: anObject .
^ self _removeForReplay: anObject.
%

category: 'Private'
method:
_redoRemoveAll: aCollection

"Performs the replay of removing aCollection elements from the receiver and returns true."

System _addToRcReadSet: self includingAllNodes: true.
aCollection _isRcIdentityBag
  ifTrue: [ ^ self _removeAll: aCollection _asIdentityBag errIfAbsent: true forReplay: true ]
  ifFalse: [ ^ self _removeAll: aCollection errIfAbsent: true forReplay: true ].
%

category: 'Private'
method:
_redoRemoveAllPresent: aCollection

"Performs the replay of removing the elements in aCollection from the receiver and returns true."

System _addToRcReadSet: self includingAllNodes: true.
aCollection _isRcIdentityBag
  ifTrue: [ ^ self _removeAll: aCollection _asIdentityBag errIfAbsent: false forReplay: true ]
  ifFalse: [ ^ self _removeAll: aCollection errIfAbsent: false forReplay: true ].
%
category: 'Reduced Conflict Support'
method: 
_selectiveAbort
  "Performs an abort operation on the receiver. That is, if the object is
 committed, it removes any changes made by the current transaction and allows
 access to the committed state of the object."

  ^ self _primitiveSelectiveAbort
%

category: 'Private'
method: RcLowMaintenanceIdentityBag
_redoAdd: anObject withOccurrences: aSmallInteger

"Performs the replay of adding aCollection to the receiver and returns true."

self _rcIncludes: anObject .
self _add: anObject withOccurrences: aSmallInteger forReplay: true.
^ true
%
category: 'Reduced Conflict Support'
method: RcLowMaintenanceIdentityBag
_selectiveAbort
  "Performs an abort operation on the receiver. That is, if the object is
 committed, it removes any changes made by the current transaction and allows
 access to the committed state of the object."

  self _checkForRcIndexesBeforeRcSelectiveAbort.
  ^ self _primitiveSelectiveAbort
%
category: 'Reduced Conflict Support'
method: RcLowMaintenanceIdentityBag
_validateLegacyRcIndexSupport
  "legacy indexes are not supported on RcIndentitySet or RcLowMaintenanceIdentityBag (bug47179)"

  self error: 'Creating a legacy index on an RcIdentitySet or RcLowMaintenanceBag is not supported.'
%
