!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   IdentityKeyValueDictionary, KeyValueDictionary, AbstractDictionary,
!   Collection, Object.
!
!=========================================================================

! 5.0: IdentityKeyValueDictionary created in bom.c

! remove existing behavior from IdentityKeyValueDictionary
removeallmethods IdentityKeyValueDictionary
removeallclassmethods IdentityKeyValueDictionary
set class IdentityKeyValueDictionary

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'An IdentityKeyValueDictionary is a KeyValueDictionary that is an identity-based
 collection instead of equality-based.  That is, two keys or two values are
 considered to be the same only if they are identical; equivalent objects are
 not the same.  Thus, if you add two key-value pairs to an
 IdentityKeyValueDictionary and the keys are equivalent but not identical, then
 the result is that you have two pairs in the dictionary because the keys are
 not the same.

 An IdentityKeyValueDictionary sends #identityHash to keys to obtain the 
 hash value, and sends #==  to compare two keys.  
 It does not use  #hashFunction: and #compareKey:with:  methods. 

 IdentityKeyValueDictionary exhibits better performance than KeyValueDictionary
 and is to be preferred where it is appropriate.

 For multiuser applications that involve a lot of concurrent use of
 dictionaries, use RcKeyValueDictionary.

Constraints:
	numElements: SmallInteger
	numCollisions: SmallInteger
	collisionLimit: SmallInteger
	tableSize: SmallInteger' .
%

! ------------------- Class methods for IdentityKeyValueDictionary
! ------------------- Instance methods for IdentityKeyValueDictionary
category: 'Hashing'
method:
hashFunction: aKey

"No longer used by implementation of IdentityKeyValueDictionary"

self shouldNotImplement: #hashFunction:
%
set compile_env: 0
category: 'Hashing'
method: IdentityKeyValueDictionary
rehash
	"Re-establish any hash invariants of the receiver.
	 Identity hashes cannot change."
%
method:
compareKey: key1 with: key2

"No longer used by implementation of IdentityKeyValueDictionary"

self shouldNotImplement: #compareKey:with:
%

category: 'Private'
method:
collisionBucketClass

"Returns the class of object to create when keys collide."

^ IdentityCollisionBucket
%


category: 'Private'
method:
_initializeWithoutClear: newSize

"Private. Initializes the instance variables of the receiver to be an empty
 KeyValueDictionary of the specified size. Does not clear the contents
 of the receiver - assumes they are all nil."

tableSize := newSize.
numElements := 0.
numCollisions := 0.
collisionLimit := newSize  * 500 .
  "This is where it differs from KeySoftValueDictionary. By setting the default
   collision limit to be high in comparison to the size of the dictionary,
   the dictionary will only be rebuilt when the collision buckets have
   become big."
^self
%

! added reimplementation of rebuildTable: in gemstone64 v1.1 
category: 'Hashing'
method:
rebuildTable: newSize

super rebuildTable: newSize .
self collisionLimit: newSize * 500 . "differs from KeyValueDictionary"
%

category: 'Canonical Symbol Support'
method:
_selectiveAbort

""

| aKey collisionBkt |

1 to: tableSize * 2 by: 2 do: [ :tableIndex |
  nil == (aKey := self _at: tableIndex) 
    ifTrue: [ nil == (collisionBkt := self _at: (tableIndex + 1)) 
      ifFalse: [collisionBkt _selectiveAbort]].
  ].
  
super _selectiveAbort
%

category: 'Comparing'
method:
hash

"Returns a numeric hash key for the receiver."

| hashValue |

hashValue := 97633 bitXor: (self size).
"For large dictionaries, the hash value is just a function of its size"
(self size > 64) ifTrue: [ ^ hashValue abs ].
self keysDo: [ :aKey |
   "Skip if the key is a dictionary."
   (aKey isKindOf: AbstractDictionary)
     ifFalse: [
       hashValue := hashValue bitXor: aKey identityHash
       ]
     ].
^ hashValue abs
%

! Globals at: #ProcessorScheduler put:nil is now done in bom.c

! deleted  _removeValues:, it was used by old ProcessorScheduler implementation

category: 'Private'
method:
_at: aKey otherwise: defaultValue

"Returns the value that corresponds to aKey.  If no such key/value pair
 exists, returns the given alternate value."

| hofs hashKey collisionBkt |

aKey ifNotNil:[ 
  hofs := (aKey identityHash \\ tableSize) . hofs := hofs + hofs .
  hashKey := self _atZ: hofs  .
  hashKey ifNil: [ 
    collisionBkt := self _atZ: hofs + 1 .
    collisionBkt ifNotNil: [ ^collisionBkt at: aKey otherwise: defaultValue ]
  ] ifNotNil: [ 
    (aKey == hashKey) ifTrue: [ ^ self _atZ: hofs + 1 ] 
  ].
].
^ defaultValue
%

category: 'Accessing'
method:
at: aKey otherwise: defaultValue

"Returns the value that corresponds to aKey.  If no such key/value pair
 exists, returns the given alternate value."

| hofs hashKey collisionBkt |

aKey ifNotNil:[ 
  hofs := (aKey identityHash \\ tableSize) . hofs := hofs + hofs .
  hashKey := self _atZ: hofs  .
  hashKey ifNil: [
    collisionBkt := self _atZ: hofs + 1 .
    collisionBkt ifNotNil: [ ^collisionBkt at: aKey otherwise: defaultValue ]
  ] ifNotNil: [ 
    (aKey == hashKey) ifTrue: [ ^ self _atZ: hofs + 1 ]
  ].
].
^ defaultValue
%

category: 'Accessing'
method:
at: aKey ifAbsent: aBlock

"Returns the value that corresponds to aKey.  If no such key/value pair
 exists, returns the given alternate value."

| hofs hashKey collisionBkt |

aKey ifNotNil:[ 
  hofs := (aKey identityHash \\ tableSize) . hofs := hofs + hofs .
  hashKey := self _atZ: hofs  .
  hashKey ifNil: [ 
      collisionBkt := self _atZ: hofs + 1 .
      collisionBkt ifNotNil: [ ^ collisionBkt at: aKey ifAbsent: aBlock ]
  ] ifNotNil: [ 
    (aKey == hashKey) ifTrue: [ ^ self _atZ: hofs + 1 ] 
  ].
].
aBlock ifNil:[^ self _errorKeyNotFound: aKey ] .
^ aBlock value
%

category: 'Private'
method:
_at: aKey put: aValue

"Stores the aKey/aValue pair in the hash dictionary.  Rebuilds the hash table
 if the addition caused the number of collisions to exceed the limit allowed."

| hofs hashKey hashValue collisionBkt cbStatus |
aKey ifNil: [ ^ self _error: #rtErrNilKey ].
hofs := (aKey identityHash \\ tableSize) . hofs := hofs + hofs .
hashKey := self _atZ: hofs  .
hashKey ifNil: [
  collisionBkt := self _atZ: hofs + 1 .
  collisionBkt ifNil: [ 
    self _atZ: hofs putKey: aKey value: aValue .
    numElements := numElements + 1 
  ] ifNotNil: [  
    cbStatus := collisionBkt at: aKey put: aValue keyValDict_coll: self.
    cbStatus ~~ 0 ifTrue:[
      numElements := numElements + 1.
      cbStatus ~~ 1 ifTrue:[ 
        numCollisions := numCollisions + 1
      ].
    ].
  ]
] ifNotNil: [ 
  aKey == hashKey ifTrue: [ 
    self _atZ: hofs  putKey: aKey value: aValue .
  ] ifFalse: [ 
    hashValue := self _atZ: hofs + 1 .
    collisionBkt := self collisionBucketClass new.
    collisionBkt objectSecurityPolicy: self objectSecurityPolicy .
    collisionBkt keyValueDictionary: self.
    self _atZ: hofs putKey: nil value: collisionBkt .
    collisionBkt at: hashKey put: hashValue keyValDict_coll: self.
    numElements := numElements + 1.
    cbStatus := collisionBkt at: aKey put: aValue keyValDict_coll: self.
    cbStatus > 1 ifTrue:[
      numCollisions := numCollisions + 1 
    ]
  ] 
].
(numCollisions > collisionLimit) ifTrue: [
  self rebuildTable: (Integer _selectedPrimeGreaterThan: tableSize * 2)
].
^aValue
%

category: 'Updating'
method:
at: aKey put: aValue

^ self _at: aKey put: aValue
%

category: 'Private'
method:
_removeKey: aKey ifAbsent: aBlock

"Removes the key/value pair with key aKey from the receiver and returns
 the value.  If no key/value pair is present with key aKey, evaluates
 the zero-argument block aBlock and returns the result of that evaluation."

| hofs thisKey collisionBkt aValue lastPair |

aKey ifNil:[ ^ self _error: #rtErrNilKey ].
hofs := (aKey identityHash \\ tableSize) . hofs := hofs + hofs .
(thisKey := (self _atZ: hofs )) ifNil:[ "It might be in a collision bucket"
      collisionBkt := self _atZ: hofs + 1.
      collisionBkt ifNil:[ 
        aBlock ifNil:[^ self _errorKeyNotFound: aKey ] .
        ^ aBlock value
      ].
      aValue := collisionBkt removeKey: aKey ifAbsent: [ 
        aBlock ifNil:[^ self _errorKeyNotFound: aKey ] .
        ^ aBlock value
      ].
      "If we got this far, a key/value pair was removed from collisionBkt"
      numCollisions := numCollisions - 1.
      (collisionBkt size <= 1) ifTrue: [
         "We don't need the collision bucket anymore.
          Put its last key/value pair in our table."
         lastPair := collisionBkt _firstPair.
         self _atZ: hofs  putKey: (lastPair at:1) value: (lastPair at:2)
      ]
] ifNotNil: [ "There is a key/value pair in this slot"
      (aKey == thisKey) ifFalse: [ ^ self _reportKeyNotFound: aKey with: aBlock ] .
      "The key matches - remove it"
      aValue := self _atZ: hofs + 1.
      self _atZ: hofs putKey: nil value: nil .
].
numElements := numElements - 1.
^aValue
%

category: 'Removing'
method:
removeKey: aKey ifAbsent: aBlock
  ^ self _removeKey: aKey ifAbsent: aBlock
%
method:
removeKey: aKey otherwise: defaultValue

  ^ self _removeKey: aKey ifAbsent:[ ^ defaultValue ]
%
