!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   GsPackagePolicy, Object
!
!=========================================================================

expectvalue /String
run
| oldCls |
oldCls := Globals at:#GsPackagePolicy otherwise: nil .
(oldCls isNil or: [ oldCls instVarNames includes: #sessionMethodDictionary ])
  ifTrue:[
   "gs64 v3.0  deleted unused instVar  sessionMethodDictionary"
  Object subclass: 'GsPackagePolicy'
    instVarNames: #( enabled homeSymbolDict externalSymbolList authorInitials )
  classVars: #()
  classInstVars: #( restrictedClasses)
  poolDictionaries: { }
  inDictionary: Globals .
  ^ 'created class'
] ifFalse:[
  ^ 'existing class definition:  ' , oldCls definition
].
%

! Remove existing behavior from GsPackagePolicy
removeallmethods GsPackagePolicy
removeallclassmethods GsPackagePolicy
set class GsPackagePolicy

! ------------------- Class methods for GsPackagePolicy
category: 'Accessing'
classmethod:
current
  "note that with the fix for bug 41433, the logic for initializing the sessionMethodPolicy 
   has been moved to bom.gs and userpro.gs ... lazy initialization left to handle upgraded repos."

  | userGlobals sessionMethodPolicy |
  userGlobals := GsSession currentSession objectNamed: #UserGlobals.
  sessionMethodPolicy := userGlobals at: self globalName otherwise: nil.
  sessionMethodPolicy == nil
    ifTrue: [
      "self pause ."
      sessionMethodPolicy := self new.
      userGlobals at: self globalName put: sessionMethodPolicy
    ].
  ^sessionMethodPolicy
%
category: 'Accessing'
classmethod:
currentOrNil

 "Returns nil or the previously installed and enabled GsPackagePolicy."
  | pp |
  pp := (( GsSession currentSession objectNamed: #UserGlobals ) ifNil: [^nil]) at: self globalName otherwise: nil.
  pp ifNotNil:[ pp enabled ifTrue:[ ^ pp ]].
  ^ nil 
%

category: 'Accessing'
classmethod:
enabled
  "As an extended fix to bug 41433, it is necessary to avoid lazy initialization
   of GsPackagePolicy class>>current, until after the initial login, thus the
   necessity of in-lining GsPackagePolicy class>>current"

  | userGlobals sessionMethodPolicy |
  userGlobals := GsSession currentSession objectNamed: #'UserGlobals'.
  userGlobals ifNil: [^false].
  sessionMethodPolicy := userGlobals at: self globalName otherwise: nil.
  sessionMethodPolicy == nil
    ifTrue: [ ^ false ].
  ^ sessionMethodPolicy enabled
%
category: 'Accessing'
classmethod:
globalName

  ^#GsPackagePolicy_Current
%
category: 'Initialize'
classmethod:
deinitialize

  (GsSession currentSession objectNamed: #UserGlobals) removeKey: self globalName 
       ifAbsent: [ nil ].
%
category: 'Initialize'
classmethod:
initialize
  ^ self deinitialize
%
category: 'Instance Creation'
classmethod:
new

  ^self basicNew initialize
%
category: 'Accessing'
classmethod:
restrictedClasses
  " restrictedClasses is an IdentitySet of class names"

  ^ restrictedClasses ifNil:[
      restrictedClasses := IdentitySet withAll: #( BasicSortNode
        BtreeBasicInteriorNode
        BtreeBasicLeafNode
        BtreeComparisonForCompare
        BtreeComparisonForSort
        BtreeComparisonQuerySpec
        BtreeInteriorNode
        BtreeLeafNode
        BtreeNode
        BtreeReadStream
        BucketValueBag
        DependencyList
        DepListBucket
        DepListTable
        GciInterface
        GsCurrentSession
        GsNMethod
        GsMethodDictionary
        GsSession
        GsSessionMethodDictionary
        GsPackagePolicy
        GsPackage
        IdentityIndex
        IndexList
        IndexManager
        MappingInfo
        NscBuilder
        RangeEqualityIndex
        RcBtreeBasicInteriorNode
        RcBtreeBasicLeafNode
        RcBtreeInteriorNode
        RcBtreeLeafNode
        RcCollisionBucket
        RcIndexBucket
        RcIndexBucketWithCache
        RcIndexDictionary
        RcRangeEqualityIndex
        Repository
        ObjectSecurityPolicy
        GsObjectSecurityPolicySet
        SymbolAssociation
        SymbolDictionary
        SymbolKeyValueDictionary
        SymbolList
        SymbolSet
        System
        UserProfile
        UserProfileSet
        UserSecurityData
      ).
      restrictedClasses
    ].
%
category: 'Method lookup control'
classmethod:
loadSessionMethodDictionary
  "Install a SessionMethodDictionary,should only be called at session login"

  | statusArray policy |
  (statusArray := Globals at: #ConversionStatus otherwise: nil ) 
    ifNotNil: [
      statusArray size > 3
        ifTrue: [ 
          (statusArray at: 4) ifTrue: [
            "in sessionMethod conversion don't install sessionMethods" 
            ^self ]]].
  policy := (GsSession currentSession objectNamed: #UserGlobals) at: self globalName otherwise: nil.
  "Avoid installing anything if the policy is not enabled or does not exist"
  (policy ~~ nil and: [ policy enabled ]) ifTrue: [ policy refreshSessionMethodDictionary ].

    "install a policy for handling TransactionBoundary notification"
    TransactionBoundaryDefaultPolicy isActive
        ifTrue: [ TransactionBoundaryDefaultPolicy installCurrent ].

    "login notification"
    SystemLoginNotification sessionStart.
%

! deleted unused  GsPackagePolicy(C)>>systemNotificationGlobalName

! ------------------- Instance methods for GsPackagePolicy
category: 'Accessing'
classmethod:
authorInitials
    "authorInitials stored in SessionTemps _not_ the IV. Don't want 
     authorInitials persisted"

  | ai |
  ai := SessionTemps current at: self authorInitialsGlobalName otherwise:nil.
  ai == nil 
    ifTrue:[
      ai := GsSession currentSession userProfile userId asString.
      SessionTemps current at: self authorInitialsGlobalName put: ai.
    ].
  ^ai
%
category: 'Accessing'
method: 
authorInitials
  ^ self class authorInitials
%

category: 'Accessing'
classmethod:
authorInitials: aString
    "authorInitials stored in SessionTemps _not_ the IV. Don't want 
     authorInitials persisted"

  SessionTemps current at: self authorInitialsGlobalName put: aString.
%
category: 'Accessing'
method:
authorInitials: aString
  self class authorInitials: aString
%
category: 'Categories'
method:
addCategory: aSymbol for: aBehavior

  ^ self homeSessionMethods addCategory: aSymbol for: aBehavior
%
category: 'Categories'
method:
removeCategory: aSymbol for: aBehavior
  self packages_Do: [ :package |
    package removeCategory: aSymbol for: aBehavior.
  ].
%
category: 'Categories'
method:
categoryNamesFor: aBehavior into: anArray

  self packages_Do: [ :package |
    package categoryNamesFor: aBehavior into: anArray.
  ].
%
category: 'Methods'
method:
categoryOfSelector: aSymbol for: aBehavior
  self packages_Do: [ :package |  | aKey |
    aKey := package categoryOfSelector: aSymbol for: aBehavior.
    aKey ifNotNil: [ ^aKey ].
  ].
  ^ nil
%
category: 'Methods'
method:
compiledMethodAt: aSymbol for: aBehavior
  self packages_Do: [ :package | |meth|
    meth := package compiledMethodAt: aSymbol for: aBehavior.
    meth ifNotNil: [ ^meth ].
  ].
  ^ nil
%
method:
findMethods: aSymbol for: aBehavior
  "Return array of package description , package, methods triples"
  | arr |
  arr := { } .
  self packages_Do: [ :package | | meth |
    meth := package compiledMethodAt: aSymbol for: aBehavior.
    meth ifNotNil: [ arr add: package printString; add: package; add: meth ].
  ].
  ^ arr
%

category: 'Methods'
method:
copyCategoryDictFor: aBehavior into: aGsMethodDictionary

  self packages_Do: [ :package |
    package copyCategoryDictFor: aBehavior into: aGsMethodDictionary.
  ].
%
category: 'Methods'
method:
copyMethodDictFor: aBehavior into: aGsMethodDictionary

  self packageReverse_Do: [ :package |
    package copyMethodDictFor: aBehavior into: aGsMethodDictionary.
  ].
%
category: 'Private'
method:
_sessionMethodsSet: aValue
    "aValue is an IdentitySet of classes having session methods installed, or nil
     returns previous value"
  <protected>
  | tmps old key |
  aValue ifNotNil:[ aValue _validateClass: IdentitySet ].
  tmps := SessionTemps current .
  key := self sessionMethodDictionaryGlobalName .
  old := tmps at: key otherwise: nil .
  tmps at: key put: aValue .
  ^ old
%

method:
sessionMethodsSet
  ^ SessionTemps current at: self sessionMethodDictionaryGlobalName otherwise: nil 
%

! deleted  sessionMethodDictionary

category: 'Accessing'
method:
disable

  enabled := false.
  self refreshSessionMethodDictionary .

%

category: 'Private'
method:
_disableNoRefresh
  "For use only by SystemUser during image upgrade."
  enabled := false.
%

category: 'Accessing'
method:
enable

  enabled := true.
  self refreshSessionMethodDictionary .
%
category: 'Accessing'
method:
enabled

  ^ enabled
%
category: 'Accessing'
method:
externalSymbolList

  ^externalSymbolList
%
category: 'Accessing'
method:
externalSymbolList: anArray

  externalSymbolList := anArray
%
category: 'Accessing'
method:
homeSessionMethods

  | package |
  package := self homeSymbolDict  at: GsPackage globalName otherwise: nil.
  package ifNil: [ package := GsPackage installIn: self homeSymbolDict ].
  ^package
%
category: 'Accessing'
method:
homeSymbolDict

  ^homeSymbolDict
%
category: 'Accessing'
method:
homeSymbolDict: aSymDict

  homeSymbolDict := aSymDict
%
category: 'Methods'
method:
includesSelector: aSymbol for: aBehavior

  <primitive: 2001>  "enter protected mode"
  | ans prot |
  prot := System _protectedMode .
  ans := false .
  [
    | mDict |
    mDict := aBehavior transientMethodDictForEnv: 0 .
    mDict ifNotNil:[
      ans := mDict includesKey: aSymbol.
    ].
  ] ensure:[
    prot _leaveProtectedMode
  ].
  ^ans
%
category: 'Initialize-Release'
method:
initialize

      | currentSession |
  enabled := false.
  currentSession := GsSession currentSession.
  homeSymbolDict := currentSession objectNamed: #UserGlobals.
  externalSymbolList := { } .
%
category: 'Method lookup control'
method:
refreshSessionMethodDictionary

  self buildSessionMethodDictionary 
%
! deleted installSessionMethodDictionary

category: 'Compiling'
method:
updateMethodLookupCacheFor: aGsMethod in: aBehavior

  self updateMethodLookupCacheForSelector: aGsMethod selector 
				method: aGsMethod in: aBehavior
%
          
category: 'Compiling'
method:
updateMethodLookupCacheForSelector: selector method: aGsMethod in: aBehavior

  <primitive: 2001>  "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [
    | mDict oldMeth |
    mDict := aBehavior transientMethodDictForEnv: 0 .
    mDict ifNil: [
      mDict := GsSessionMethodDictionary new.
      self sessionMethodsSet add: aBehavior .
      aBehavior transientMethodDictForEnv:0 put: mDict .
    ] ifNotNil:[
       oldMeth := mDict at: selector otherwise: nil .
    ].
    oldMeth ifNil:[
       "need to find oldMeth if possible to be sure breakpoints are cleared"
       oldMeth := aBehavior compiledMethodAt: selector environmentId: 0 otherwise: nil
    ].
    mDict at: selector put: aGsMethod.
    aBehavior _refreshLookupCache: selector oldMethod: oldMeth env: 0.
    self sessionMethodChanged .
  ] ensure:[
    prot _leaveProtectedMode
  ].
%

category: 'Compiling'
method:
methodAndCategoryDictionaryFor: aBehavior 
source: sourceString 
dictionaries: aSymbolList 
category: categorySymbol 
  "Returns a 2 element Array, or signals a CompileError"

  self enabled ifTrue: [ | selector |
      selector := self
        extractSelectorFor: aBehavior 
        source: sourceString 
        dictionaries: aSymbolList 
        category: categorySymbol.
      (self permitSessionMethodFor: aBehavior selector: selector)
        ifTrue: [ 
           ^self homeSessionMethods methodAndCategoryDictionaryFor: aBehavior 
        ].
    ].
  ^ { nil . nil }
%
category: 'Compiling'
method:
pragmasForMethod: selector in: aBehavior

  | pragmas |
  self packages_Do: [ :package | | pragmaDict |
    pragmaDict := package methodPragmaDictFor: aBehavior.
    pragmaDict ifNotNil: [ 
      (Symbol _existingWithAll: selector) ifNotNil:[ :sel |
        pragmas := pragmaDict at: sel otherwise: nil .
      ].
      pragmas ifNotNil: [ ^pragmas ].
    ].
  ].
  ^ nil
%
category: 'Compiling'
method:
stampForMethod: selector in: aBehavior

  self packages_Do: [ :package | |stampDict |
    stampDict := package methodStampDictFor: aBehavior.
    stampDict ifNotNil: [  | stamp |
      (Symbol _existingWithAll: selector) ifNotNil:[ :sel |
        stamp := stampDict at: sel otherwise: nil 
      ].
      stamp ifNotNil: [ ^stamp ].
    ].
  ].
  ^ nil
%
category: 'Private'
classmethod:
authorInitialsGlobalName

  ^#GsPackagePolicy_AuthorInitials
%
category: 'Private'
method:
sessionMethodDictionaryGlobalName

  ^#GsPackagePolicy_SessionMethodDictionary
%
category: 'Private'
method:
buildSessionMethodDictionary
 "returns receiver"
 <primitive: 2001>  "enter protected mode"
 | prot |
 prot := System _protectedMode .
 [ | smSet clsDict clsList oldSet oldList envId |
    envId := 0 .
    smSet := IdentitySet _basicNew .
    "transientMethodDictForEnv:put: will  keep classes in memory"
    clsDict := IdentityDictionary new .
    (Unicode16 _unicodeCompareEnabled) ifTrue:[
      | mapping |
      mapping := Unicode16 _unicodeCompareMapping .
      1 to: mapping size by: 2 do:[:index | 
        | cls |
        cls := mapping at: index .
        clsDict at: cls put: (Unicode16 _unicodeCompareTmdForClass: cls selectors: (mapping at: index + 1)) .
        smSet add: cls .
      ].
    ].
    self enabled ifTrue:[ | rejected |
      rejected := { } .
      self _packageReverse_Do: [:package |
        package behaviorAndMethodDictDo: [:behavior :methodDict | | tmd |
	  (tmd := clsDict at: behavior otherwise: nil ) ifNil:[
             tmd := GsSessionMethodDictionary new .
             clsDict at: behavior put: tmd .
             smSet add: behavior.
          ].
	  methodDict keysAndValuesDo: [:k :v | 
            v class == GsNMethod ifTrue:[ tmd at: k put: v ]
                    ifFalse:[ rejected add: { package . behavior . k . v } ].
          ].
        ].
      ].
      rejected size ~~ 0 ifTrue:[ 
        "rejected is of the form { { package . behavior . key . value } .  ... }"
        ImproperOperation new object: rejected ; reason: 'buildSessionMethodDictionaryFail';
          signal: 'one or more values in package method dictionaries is not a GsNMethod'.
      ].
      clsList := { } .
      clsDict keysAndValuesDo:[ :cls :dict | clsList add: cls ; add: dict ].
      oldSet := self _sessionMethodsSet: smSet .  
      "following 2 loops must not use any methods implemented in session methods"
      oldSet ifNotNil:[ 
        oldList := oldSet asArray .
        1 to: oldList size do:[:n | 
          (oldList at: n) transientMethodDictForEnv: envId put: nil .
        ]. 
      ].
      1 to: clsList size by: 2 do:[:n | 
        (clsList at: n) transientMethodDictForEnv: envId put: (clsList at: n + 1) 
      ]. 
      GsCurrentSession currentSession enableSessionMethods: true env: envId. "clears lookup caches"
    ] ifFalse:[
      oldSet := self _sessionMethodsSet: nil .
      oldSet ifNotNil:[ 
        oldList := oldSet asArray .
        "following loop must not use any methods implemented in session methods"
        1 to: oldList size do:[:n | | cls tmd |
          cls := oldList at: n .
          tmd := (smSet includes: cls) ifTrue:[ clsDict at: cls ]
                                      ifFalse:[ nil ].
          cls transientMethodDictForEnv: envId put: tmd .
        ].
      ].
      GsCurrentSession currentSession enableSessionMethods: false env: envId . "clears lookup caches"
    ].
    Unicode16 _cacheUsingUnicodeCompares .
 ] ensure:[
   prot _leaveProtectedMode
 ]
%

category: 'Private'
method:
permitSessionMethodFor: aBehavior selector: selector

  | cl thisName |
  cl := aBehavior whichClassIncludesSelector: selector.
  cl ifNotNil: [ (cl compiledMethodAt: selector) _isProtected ifTrue: [ ^false ].  ].
  thisName := aBehavior thisClass name asSymbol .
  (self class restrictedClasses includes: thisName ) ifTrue: [ ^false ].
  externalSymbolList do: [:symDict | 
		| possible |
		possible := symDict at: thisName otherwise: nil.
		(possible isBehavior and: [aBehavior theNonMetaClass isVersionOf: possible theNonMetaClass]) 
			ifTrue: [ ^true ].
  ].
  ^ (aBehavior canWriteMethodsEnv: 0) not
%

category: 'Private'
method:
extractSelectorFor: aBehavior 
source: sourceString 
dictionaries: aSymbolList 
category: categorySymbol 

  "Returns a selector or signlas a CompileError"
  | mDict cDict meth symList sel |
  aSymbolList class == SymbolList
      ifTrue:[ symList := aSymbolList ]
      ifFalse:[
          aSymbolList _validateClass: Array .
          symList := SymbolList withAll: aSymbolList .
        ] .
  mDict := GsMethodDictionary new.
  cDict := GsMethodDictionary new.
  meth := aBehavior _checkCompileResult:( aBehavior
      _primitiveCompileMethod: sourceString
      symbolList: symList 
      category: categorySymbol 
      oldLitVars: nil 
      intoMethodDict: mDict 
      intoCategories: cDict environmentId: 0 ) 
    source: sourceString 
    suppressCompileWarning: true .
  sel :=  meth selector.
  ^ sel 
%
category: 'Methods'
method:
selectorsFor: aBehavior into: anArray

  <primitive: 2001>  "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [
    | mDict  |
    mDict := aBehavior transientMethodDictForEnv: 0 .
    mDict ifNotNil:[
      anArray addAll: mDict keys.
    ].
  ] ensure:[
    prot _leaveProtectedMode
  ].
  ^self
%
category: 'Categories'
method:
selectorsIn: categoryName for: aBehavior into: anArray

  self packages_Do: [ :package |
    package selectorsIn: categoryName for: aBehavior into: anArray.
  ].
%
! renamed packagesDo , packageReverseDo to ensure uses
!  conform to new style of their argument blocks
category: 'Enumerating'
method:
packages_Do: aBlock
  "aBlock must be a one argument block which returns true if iteration
   should be terminated early and false if iteration is to continue .
   This is to avoid return-from-home which in Gs64 v3.0 would
   reset the protected mode to zero."
   
  | symList |
  self enabled ifFalse: [ ^self ].
  symList := self symbolList .
  1 to: symList size do:[ :j| | symDict package |
     symDict := symList at: j .
     package := symDict at: GsPackage globalName otherwise: nil.
     package ifNotNil:[  aBlock value: package  ].
  ].
%
category: 'Enumerating'
method:
packageReverse_Do: aBlock
  "aBlock must be a one argument block which returns true if iteration
   should be terminated early and false if iteration is to continue .
   This is to avoid return-from-home which in Gs64 v3.0 would
   reset the protected mode to zero."
   
  self enabled ifFalse: [ ^self ].
  self _packageReverse_Do: aBlock .
%

category: 'Private'
method:
_packageReverse_Do: aBlock
  | symList |
  symList := self symbolList . 
  symList size _downTo: 1 do:[ :j| | symDict package |
     symDict := symList at: j .
     package := symDict at: GsPackage globalName otherwise: nil.
     package ifNotNil:[  aBlock value: package ].
  ].
%

category: 'Compiling'
method:
setPragmas: pragmaArrayOrNil
forBehavior: aBehavior
forMethod: selector

  | aSym |
  aSym := selector asSymbol.
  self packages_Do: [ :package | 
    (package
      setPragmas: pragmaArrayOrNil
          forBehavior: aBehavior
          forMethod: aSym ) ifNotNil:[ ^ self ].
  ].
  ^ nil

%
category: 'Compiling'
method:
setStamp: aStampOrNil
forBehavior: aBehavior
forMethod: selector
  | aSym |
  aSym := selector asSymbol.
  self packages_Do: [ :package |
    (package
      setStamp: aStampOrNil
            forBehavior: aBehavior
            forMethod: aSym) ifNotNil: [ ^ self ].
  ].
  ^ nil
%
category: 'Accessing'
method:
symbolList

  ^GsSession currentSession symbolList
%
category: 'Methods'
method:
removeMethodAt: aSymbol for: aBehavior
  | meth |
  self packages_Do: [ :package |
    meth := package removeMethodAt: aSymbol for: aBehavior.
    meth ifNotNil:[
      self sessionMethodRemoveMethodAt: aSymbol for: aBehavior.
      ^ meth 
    ].
  ].
  self sessionMethodRemoveMethodAt: aSymbol for: aBehavior.
  ^ nil
%

category: 'Methods'
method:
removeAllMethodsFor: aBehavior
  "self pause ."
  self packages_Do: [ :package |
    package removeAllMethodsFor: aBehavior.
  ].
  "fix up the session method dictionary"
  self sessionMethodRemoveAllMethodsFor: aBehavior.
  ^nil
%

category: 'Methods'
method:
removeAllSubclassCodeFor: aBehavior

"Dereference the code objects of all GsMethods for aBehavior,
 to force recompilation of those methods."
  "self pause."
  self packages_Do: [:package |
    package removeAllSubclassCodeFor: aBehavior.
  ].
  self sessionMethodRemoveAllSubclassCodeFor: aBehavior.
%
category: 'Transaction Boundaries'
method:
sessionMethodChanged

    TransactionBoundaryDefaultPolicy isActive
        ifTrue: [ TransactionBoundaryDefaultPolicy current sessionMethodChanged].
%
category: 'Private'
method:
sessionMethodRemoveMethodAt: aSymbol for: aBehavior

  <primitive: 2001>  "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [
    | mDict oldMeth |
    mDict := aBehavior transientMethodDictForEnv: 0 .
    mDict ifNotNil: [
      oldMeth := mDict removeKey: aSymbol otherwise:  nil .
      oldMeth ifNotNil:[
	aBehavior _refreshLookupCache: aSymbol oldMethod: oldMeth env: 0 .
	self sessionMethodChanged
      ].
    ].
  ] ensure:[
    prot _leaveProtectedMode
  ]
%
category: 'Private'
method:
sessionMethodRemoveAllMethodsFor: aBehavior

  <primitive: 2001>  "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [
    | oldDict smSet envId |
    envId := 0 .
    oldDict := aBehavior transientMethodDictForEnv: envId .
    oldDict ifNotNil:[
      aBehavior transientMethodDictForEnv: envId put: nil .
      aBehavior _clearLookupCaches: envId .
      self sessionMethodChanged .
      (smSet := self sessionMethodsSet) ifNotNil:[ smSet remove: aBehavior] .
    ].
  ] ensure:[
    prot _leaveProtectedMode
  ]
%
category: 'Private'
method:
sessionMethodRemoveAllSubclassCodeFor: aBehavior

  <primitive: 2001>  "enter protected mode"
  | aDict prot |
  prot := System _protectedMode .
  [
    aDict := aBehavior transientMethodDictForEnv: 0 .
    aDict ifNotNil:[
      "All methods in aBehavior have been marked for recompilation, 
       and we have entries in the sessionMethodDictionary for Behavior, 
       so refresh sessionMethodDictionary."
  
      self refreshSessionMethodDictionary .
    ].
  ] ensure:[
    prot _leaveProtectedMode
  ].
%

category: 'Categories'
method:
removeSelector: aSelector fromCategoriesFor: aBehavior
    self packages_Do: [ :package |
      package removeSelector: aSelector fromCategoriesFor: aBehavior.
    ].
%
category: 'Categories'
method:
addSelector: aSelector toCategory: categoryName for: aBehavior
   self packages_Do: [ :package |
      package addSelector: aSelector toCategory: categoryName for: aBehavior .
   ].
%
method:
moveSelector: aSelector toCategory: categoryName for: aBehavior

   self packages_Do: [ :package |
      package moveSelector: aSelector toCategory: categoryName for: aBehavior 
   ].
%
method:
categorysDo: aBlock for: aBehavior
  "evaluates aBlock for each method category of receiver
   in specified environment. Returns the receiver.

   aBlock should be a two argument block
   expecting the args  categoryNameSymbol ,   selectorsSet.
   aBlock may be invoked more than once for each category name.
   The iteration is done directly over the categories in each
   of the receiver's packages."

  self packages_Do:[ :package | package categorysDo: aBlock for: aBehavior ].
%

category: 'Reporting'
method:
_report: includeMethsBool
 "Reports on the methods without regard to whether enabled is true"
 | str symList |
  str := String new .
  symList := self symbolList .
  1 to: symList size do:[ :j| | symDict package |
     symDict := symList at: j .
     package := symDict at: GsPackage globalName otherwise: nil.
     package ifNotNil:[  
       includeMethsBool ifTrue:[ str add:'===== ' ].
       str add: 'GsPackage oop:' ; add: package asOop asString ;
         add: ' name:' ; add: package name asString ;
         add: ' enabled:' ; add: package enabled asString ; lf .
       includeMethsBool ifTrue:[ str add: package methodsReport ].
     ].
  ].
 ^ str
%   
method:
packagesReport
 "Reports without regard to whether enabled is true"
 ^ self _report: false
%
method:
methodsReport
 "Reports without regard to whether enabled is true"
 ^ self _report: true
%

run
GsPackagePolicy current.    "create fresh instance in slowfilein"
true
%

category: 'Private'

classmethod:
_originVersion

"For use in repository upgrade code, before upgradeimage completes, returns a 2 digit
 SmallInteger."
(Globals at:#DbfHistory otherwise: nil) ifNotNil:[:hist |
  #( 35 34 33 32 ) do:[:ver | | num vStr |
    num := ScaledDecimal numerator: ver denominator: 10 scale: 1 .
    vStr := 'v', num asString, '.' .
    (hist matchPattern: { $* . vStr .  $? . ' kernel classes filein' . $* }) ifTrue:[
      ^ ver 
    ]
  ].
].
^ 0
%

! fix 48866
category: 'Private'
classmethod:
_previousVersion
"For use in repository upgrade code, before upgradeimage completes, returns a 2 digit
 SmallInteger."
| prevVer hist |
prevVer := 0 .
(Globals at:#DbfHistory otherwise: nil) ifNotNil:[:h | | ofs |
  hist := h .
  ofs := hist _findLastString: 'upgrade to GemStone' startingAt: hist size
               ignoreCase: true .
  ofs == 0 ifTrue:[ 
     (ImageVersion at: #gsVersion otherwise: nil ) ifNotNil:[:iVer |
        ofs := 1 .
        hist := '  v' , iVer, '  ' .
     ].
  ].
  ofs ~~ 0 ifTrue:[ 
    #( 35 34 33 32 ) do:[:ver | | num vStr |
      num := ScaledDecimal numerator: ver denominator: 10 scale: 1 .
      vStr := 'v', num asString, '.' .
      (hist matchPattern: { $* . vStr . $* }) ifTrue:[ ^ ver ].
      vStr := ' ', num asString, '.' .
      (hist matchPattern: { $* . vStr . $* }) ifTrue:[ ^ ver ].
    ].
  ].
].
prevVer == 0 ifTrue:[ prevVer := self _originVersion ].
^ prevVer
%

