!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   ExecBlock, BlockClosure, Object.
!
!=========================================================================

removeallmethods ExecBlock
removeallclassmethods ExecBlock

set class ExecBlock

category: 'Documentation'
classmethod:
comment
^'ExecBlock is the block class representing all kinds
 of executable blocks in the Gemstone64 virtual machine.
 ExecBlock was introduced in Gemstone64 version 3.0 . 
 Instances of subclasses ExecBlock0..ExecBlockN exist in-memory
 only for use by the VM. In-memory copies of ExecBlocks have 
 their class changed to one of ExecBlock0..ExecBlockN  after 
 validating number of args, etc.  The varying instVars , if any,
 contain copies of the values of outer block(s) args or temps.

--- instVar _method
The instance of GsNMethod containing the code for this ExecBlock.
 In an in-memory ExecBlock, this instVar can contain a reference to
 a GsNativeCode object.

--- instVar iFields1 
A SmallInteger which encodes the unsigned ints 
    nArgs          mask             16rFF,
    lastRubyArgIsStar  mask        16r100
    rubyNoArgsDecl    mask         16r200 
    nTemps mask                16rFFFF000,
    selfOffsetInVC  mask   16rFFFF0000000    (offset is zero-based)
    cost           mask  16r0300000000000 
    isCopyingBlock mask  16r0400000000000
    spare                16r0800000000000
    selfOffsetInSelf    16r0F000000000000
    spare              16rFF0000000000000
    isRubyLambda mask 16r1000000000000000  (the sign bit)

--- instVar staticLink
Set to nil by the method compiler; for use by the VirtualMachine only. 
 If non-nil in a stored or active block ,
 a reference to the VariableContext of the home method.

-- instVar stkSerialNum
Set to nil by the method compiler; for use by the VirtualMachine only,
 in the implementation of continuations.

Constraints:
	_method: GsNMethod
	iFields1: SmallInteger
	staticLink: VariableContext
	stkSerialNum: SmallInteger
	ccallin: CCallin'
%

category: 'Accessing'
method:
staticLink

^ staticLink
%

! added vcOfs > theVc basicSize check,  workaround for 46661 
! fix 48068
category: 'Accessing'
method:
selfValue

"return self in an active copy , or nil if self is not referenced."
| aVc ofs |
aVc := staticLink .
aVc ifNotNil:[ | vcOfs |
  vcOfs := self selfOffsetInVC .
  vcOfs > 2"VAR_CONTEXT_NAMED_SIZE" ifTrue:[
    [ aVc parent ~~ nil ] whileTrue:[ aVc := aVc parent ].
    vcOfs > aVc basicSize ifTrue:[ ^ nil ].
    ^ aVc _primitiveAt: vcOfs 
  ].
].
ofs := self _selfOffsetInSelf .
ofs ~~ 0 ifTrue:[
  ofs := ofs - 1 . 
  ^ self _primitiveAt: ofs . 
].
^ nil
%

category: 'Accessing'
method:
_sourceString

"Returns a String that will create a block similar to the receiver when
 the string is compiled.  References to variables in other contexts or
 to the pseudovariable 'self' will not recompile properly if the source
 string is used to create a new block similar to the receiver."

| result |
result := String new.
result addAll: '"This is source for a block.  " 
' .

result addAll:' ^ ' ;
       addAll: self method _sourceStringForBlock .
^result
%

category: 'Private'
method:
_sourceStringForExecute

"Returns a String that will create a block similar to the receiver when
 the string is compiled.  References to variables in other contexts or
 to the pseudovariable 'self' will not recompile properly if the source
 string is used to create a new block similar to the receiver.

 Used by GsExternalSession which further formats the string
"

^ self method _sourceStringForBlock .
%

category: 'Accessing'
method:
_debugSourceForBlock
  | str meth |
  "used by topaz  LIST @<blockObjId>  , and by RubyBridge class"

  meth := self method .
  str := meth _sourceStringForBlock . 
  ^ str
%

category: 'Repository Conversion'
method:
_iFields1

"Private"

^ iFields1
%

category: 'Repository Conversion'
method:
needsRecompile

"Returns true if the receiver's block needs recompilation, false otherwise."

^  self method needsRecompile
%

category: 'Repository Conversion'
method:
recompile
  "Attempts to recompile the receiver's method assuming it is a simple block
   without references to self."

| src newBlk doitMeth litVars |
self needsRecompile ifFalse:[ ^ self "recompile not needed"].
staticLink ifNotNil:[
  Error signal:'cannot recompile a complex block with reference to home VariableContext'.
].
ccallin ifNotNil:[
  Error signal:'cannot recompile a block for a CCallin'.
].
src := self _sourceString .
litVars := self method _literalVariablesForRecompile .
doitMeth := src _compileInContext: nil 
       symbolList: GsSession currentSession symbolList
       oldLitVars: litVars environmentId: 0 flags: 4"GCI_COMPILE_evaluateWithoutSelf".
newBlk := doitMeth _executeInContext: nil .
newBlk _iFields1 = iFields1 ifFalse:[
  Error signal:'in recompile: iFields1 differs, newBlock ', 
	   newBlk _iFields1 asString , ' old ', iFields1 asString .
].
self _unsafeAt: 1 put: newBlk method .  "_method := newBlk method"
%

category: 'Accessing'
classmethod: 
_cost

"cost is not define at the class level, each instance
 has its own cost ."

self shouldNotImplement: #_cost
%

category: 'Accessing'
method:
_cost

"return 1 for simple, 2 for complex , 3 for complex with return to home"

^ (iFields1 bitShift: -44) bitAnd: 16rF 
%

method:
_isCopyingBlock

^ ((iFields1 bitShift: -46) bitAnd: 1) ~~ 0
%

method:
_selfOffsetInSelf

"result is zero or a positive zero-based absolute offset"

^ (iFields1 bitShift: -48) bitAnd: 16rF
%

method:
_isRubyLambda

^ iFields1 < 0
%

method:
_fileAndLine
  "Return an Array, { fileName . lineNumber } , 
   or nil if method  does not have that information.
   Smalltalk methods return nil."
 
  | mth fl |
  mth := self method .
  (fl := mth homeMethod _fileAndLine) ifNotNil:[ 
    ^ { (fl at: 1) . (fl at: 2) + mth _lineDeltaForBlock }
  ].
  ^ nil  
%

category: 'Accessing'
method:
isSimple

^ self _cost = 1
%

category: 'Accessing'
method:
instVarAt: anIndex

  anIndex == 1 ifTrue:[
    "translate GsNativeCode to GsNMethod if needed"
    ^ self method 
  ].
  ^ super instVarAt: anIndex
%
category: 'Updating'
method: 
at: anIndex put: anObj
  "Disallowed"
  self shouldNotImplement: #at:put:
%
method: 
_basicSize: anInteger
  "Disallowed"
  self shouldNotImplement: #_basicSize:
%
method: 
size: anInteger
  "Disallowed"
  self shouldNotImplement: #size:
%


category: 'Accessing'
method:
method
  "Returns the GsNMethod containing the code for this block.

   A primitive is needed here because the value of the _method instVar 
   can be an instance of GsNativeCode for an in-memory block."

<primitive: 492>
self _primitiveFailed: #method .
%

category: 'Debugging Support'
method:
argsAndTemps

"Returns an Array of Symbols which are the names of arguments and
 temporaries for this block,  not including inner blocks"

^ self method argsAndTemps
%

category: 'Debugging Support'
method:
vcTempNames

"Returns an Array of Symbols which are the names for the temps
 defined in the receiver's VariableContext.  The ordering of names
 in the result matches ordering of temps in the varying instVars
 of the receiver."

| vc allNames allOfs res vcNamedSize myMeth lastTmpOfs | 
vc := staticLink .
vc == nil ifTrue:[ ^ #() ].
vcNamedSize := vc class instSize .
allNames := self argsAndTemps .
myMeth := self method .
allOfs := myMeth _argsAndTempsOffsets .
res := Array new: vc size .
lastTmpOfs := allOfs size - myMeth _numCopyingBlockArgs .
self argumentCount + 1 to: lastTmpOfs do:[:j| | anOfs ofs |
  anOfs := allOfs at: j .
  anOfs > 0 ifTrue:[
    (anOfs bitAnd: 16rFF) == 0 ifTrue:[ "lexLevel == 0"
      ofs := anOfs bitShift: -8 . 
      res at: (ofs + 1 - vcNamedSize) put:( allNames at: j) .
    ] "else temp is defined in a parent block"
  ] "else temp is on stack"
].
^ res
%

! _sourceStringForConversion not supported


! numArgs is Gemstone legacy API but not deprecated.
!  numArgs is implemented as a primitive because it is 
!   used in process scheduler
!  otherwise it would be
!     ^ iFields1 bitAnd: 16rFF
category: 'Accessing'
method:
numArgs

  "Returns a SmallInteger, the number of arguments to the receiver.
   numArgs is GemStone legacy API. "
  <primitive: 458>
  self _primitiveFailed: #numArgs .
  ^ 0
%
! argumentCount is provided for ANSI compliance
category: 'Accessing'
method: ExecBlock
argumentCount

  "Returns a SmallInteger, the number of arguments to the receiver.
   argumentCount is provided to conform to ANSI Smalltalk."
  <primitive: 458>
  self _primitiveFailed: #argumentCount.
  ^0.
%
category: 'Accessing'
method:
numberArgs

  "Returns a SmallInteger, the number of arguments to the receiver."
  self deprecated: 'ExecBlock>>numberArgs deprecated v3.2; use #argumentCount'.
  ^ self argumentCount
%

category: 'Accessing'
method:
numberTemps

^ (iFields1 bitShift: -12) bitAnd: 16rFFFF
%

category: 'Accessing'
method:
lastRubyArgIsStar
  ^ (iFields1 bitAnd: 16r100) ~~ 0
%
category: 'Accessing'
method:
noRubyDeclaredArgs
  ^ (iFields1 bitAnd: 16r200) ~~ 0
%

category: 'Accessing'
method:
selfOffsetInVC

" Return a one-based offset to self for use as arg to primitiveAt:
  If there is no reference to self in the receiver,
  returns  1 , otherwise returns a value > VAR_CONTEXT_NAMED_SIZE"

^ ((iFields1 bitShift: -28) bitAnd: 16rFFFF) + 1

%

category: 'Decompiling without Sources'
method:
_asSource

"return a stripped source representation of the block."
 
"not implemented yet"
self shouldNotImplement: #_asSource
%


! _fixConversionResult not supported

category: 'Block Evaluation'
method:
value

"Return the value of the receiver evaluated with no arguments.
 If the block expects any arguments, an error is generated.

 #value is optimized by the compiler.
 This method is in the image for use by perform: and for
 failure paths from the optimized bytecode "

^ self valueWithArguments: #()
%

category: 'Block Evaluation'
method:
value: anObject

"Return the value of the receiver evaluated with anObject as its argument.  If
 the block expects a different number of arguments, an error is generated.

 #value: is optimized by the compiler.
 This method is in the image for use by perform: and for
 failure paths from the optimized bytecode "


^ self valueWithArguments: { anObject }
%

category: 'Block Evaluation'
method:
value: firstObject value: secondObject

"Return the value of the receiver evaluated with the two objects as its
 arguments.  If the block expects a different number of arguments, an error is
 generated.

 #value:value: is optimized by the compiler.
 This method is in the image for use by perform: and for
 failure paths from the optimized bytecode "

^ self valueWithArguments: {  firstObject . secondObject }
%

category: 'Block Evaluation'
method:
value: firstObject value: secondObject value: thirdObject

"Return the value of the receiver evaluated with the three objects as its
 arguments.  If the block expects a different number of arguments, an error is
 generated.

 #value:value:value: is optimized by the compiler.
 This method is in the image for use by perform: and for
 failure paths from the optimized bytecode "

^ self valueWithArguments: {  firstObject . secondObject . thirdObject }
%

category: 'Block Evaluation'
method:
value: first value: second value: third value: fourth

"Return the value of the receiver evaluated with the four objects as
 its arguments.  If the block expects a different number of arguments,
 an error is generated.

 #value:value:value:value: is optimized by the compiler.
 This method is in the image for use by perform: and for
 failure paths from the optimized bytecode "

^ self valueWithArguments: {  first . second . third . fourth }
%

category: 'Block Evaluation'
method:
value: first value: second value: third value: fourth value: fifth

"Return the value of the receiver evaluated with the five objects as
 its arguments.  If the block expects a different number of arguments,
 an error is generated.

 #value:value:value:value:value: is optimized by the compiler.
 This method is in the image for use by perform: and for
 failure paths from the optimized bytecode "

^ self valueWithArguments: {  first . second . third . fourth . fifth }
%

category: 'Block Evaluation'
method:
valueWithArguments: argList

"Return the value of the receiver evaluated with the elements of the Array
 argList as arguments.  If the block expects a different number of arguments,
 an error is generated."

<primitive: 2003>  "compiler emits special bytecode"
^ self _primitiveFailed: #valueWithArguments: args: { argList }
%


category: 'Private'
method:
_gbsTraversalCallback

"Private.  When GemBuilder Smalltalk traverses an ExecutableBlock, this method
 is called to place the block's source string in the traversal buffer."

^self _sourceString
%

! deleted _gsReturnNoResult
! deleted _valueOnUnwind

category: 'Private'
method:
_valueOnUnwind

"This method should be invoked only from within the virtual machine. Other
 use from a Smalltalk program will corrupt the Smalltalk execution
 stack."

"Used to implement valueNowOrOnUnwind:"

self value .  "execute the block"
self _gsReturnNoResult "exit from method with no result, 
			special selector optimized by compiler"
%

category: 'Deprecated'
method:
valueNowOrOnUnwindDo: aBlock

self deprecated: 'ExecBlock>>valueNowOrOnUnwindDo: deprecated v3.2; use #ensure:'.
^self _valueNowOrOnUnwindDo: aBlock.
%

category: 'Deprecated'
method:
_valueNowOrOnUnwindDo: aBlock

self deprecated: 'ExecBlock>>_valueNowOrOnUnwindDo: deprecated v3.2; use #ensure:'.
^self _valueNowOrOnUnwindDoPrim: aBlock.
%

category: 'Deprecated'
method:
_valueNowOrOnUnwindDoPrim: aBlock
"Obsolete in GemStone/64. Use ensure: instead.

 aBlock must be a zero-arg instance of ExecBlock, otherwise an
 error is generated. "

<primitive: 2017>  "marks frame with ENSURE_Mark_NIL, always fails"
| result |
result := self value .  "execute the block"
aBlock value .       "normal execution of the unwind block"
^ result
%

category: 'Block Evaluation'
method:
ensure: aBlock
"Evaluate the receiver.  
 Evaluate aBlock after evaluating the receiver,
 or before any return from a block that would return to the sender.
 Returns result of evaluating the receiver.

 aBlock must be a zero-arg instance of ExecBlock, otherwise an
 error is generated. "

<primitive: 2017>  "marks frame with ENSURE_Mark_NIL, always fails"
| result |
result := self value .  "execute the receiver"
self _removeEnsure ifNotNil:[:b | b value "normal execution of argument"].
^ result
%

category: 'Private'
method:
_removeEnsure

"The sender's frame is changed from  ensure:[] to  ensure: nil .
 Returns nil or an ensure block that has not been started."
<primitive: 1004>

self _primitiveFailed: #_removeEnsureAtFP: 
%

category: 'Flow of Control'
method:
untilFalse

"(Reserved selector.)  Evaluates the receiver repeatedly until the evaluation's
 result is false.  Return nil.  Generates an error if the receiver is not a
 zero-argument block."

"The following is a control structure optimization, not a recursive send."

^ [self value] untilFalse
%

category: 'Flow of Control'
method:
untilTrue

"(Reserved selector.)  Evaluates the receiver repeatedly until the evaluation's
 result is true.  Return nil.  Generates an error if the receiver is not a
 zero-argument block."

"The following is a control structure optimization, not a recursive send."

^ [self value] untilTrue
%

category: 'Flow of Control'
method:
untilTrue: aBlock

"(Reserved selector.)  Evaluates the receiver once ,
 and then repeats the evaluation while aBlock returns false .
 Returns nil.  Generates an error if the receiver is not a
 zero-argument block."

"The following is a control structure optimization, not a recursive send."

^ [self value] untilTrue:[ aBlock value]
%

category: 'Flow of Control'
method:
untilFalse: aBlock

"(Reserved selector.)  Evaluates the receiver once ,
 and then repeats the evaluation while aBlock returns true .
 Returns nil.  Generates an error if the receiver is not a
 zero-argument block."

"The following is a control structure optimization, not a recursive send."

^ [self value] untilFalse:[ aBlock value]
%

category: 'Flow of Control'
method:
whileFalse: aBlock

"(Reserved selector.)  Evaluates the zero-argument block aBlock repeatedly
 while the receiver evaluates to false.  Return nil.  Generates an error if the
 receiver is not a zero-argument block."

"The following is a control structure optimization, not a recursive send."

^ [self value] whileFalse: [aBlock value]
%

category: 'Flow of Control'
method:
whileTrue: aBlock

"(Reserved selector.)  Evaluates the zero-argument block aBlock repeatedly
 while the receiver evaluates to true.  Return nil.  Generates an error if the
 receiver is not a zero-argument block."

"The following is a control structure optimization, not a recursive send."

^ [self value] whileTrue: [aBlock value]
%

category: 'Flow of Control'
method:
whileTrue
"(Reserved selector.)  Evaluate the receiver once and then repeatedly as long 
 as the value returned by the evaluation is true.

 The following is a control structure optimization, not a recursive send."

^ [ self value] whileTrue
%
category: 'Flow of Control'
method: 
repeat
"(Reserved selector.)  Evaluate the receiver repeatedly, ending only if the 
 block forces some stopping condition.

 The following is a control structure optimization, not a recursive send."

 [ self value] repeat
%

category: 'Flow of Control'
method:
whileFalse
"(Reserved selector.)  Evaluate the receiver once and then repeatedly as long 
 as the value returned by the evaluation is false.

 The following is a control structure optimization, not a recursive send."

^ [ self value] whileFalse
%

category: 'Storing and Loading'
classmethod:
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| src result marker meth o symbolList |

"Returns a new instance of the receiver read from the given PassiveObject"

marker := passiveObj objectPositionMarker.
src := passiveObj readObject.
symbolList := GsCurrentSession currentSession symbolList .
meth := src _compileInContext: (o := Object new) symbolList: symbolList.
result := meth _executeInContext: o.
(result == nil or: [(result _isExecBlock) not]) ifTrue: [
  "error in compiling"
  self _halt: 'Error in recreating a ' , name.
  ^nil
  ]
ifFalse: [
  passiveObj hasRead: result marker: marker.
  ^result
  ]
%

category: 'Storing and Loading'
method:
writeTo: aPassiveObject

"Converts the receiver to its passive form and writes that information on
 aPassiveObject.

 Simple blocks can usually be passivated and then reactivated.  Complex blocks
 can be passivated but may have to be massaged to be reactivated.  References
 to 'self' in complex blocks will resolve to an instance of Object when the
 block is activated, and any arguments or temporaries from enclosing scopes
 will be nil."

aPassiveObject writeClass: self class.
aPassiveObject writeObject: self _sourceString; cr
%

category: 'Modification Tracking'
method:
_setModificationTrackingTo: tracker

"Private.

 No modification tracking is required for blocks,
 even if they are not invariant."

^self
%

!
! Thread Support
!

category: 'Processes - Blue Book'
method:
fork
  "forks the receiver as a new process at the current scheduling priority"
  
  ^ GsProcess _forkBlock: self with: #() env: 0 prio: nil  
%

category: 'Processes - Blue Book'
method:
forkAt: priority
  "forks the receiver as a new process at the given priority"
  
  | proc |
  proc := GsProcess _forkBlock: self with: #() env: 0 prio: priority .
  ^ proc 
%

category: 'Processes'
method:
forkWith: blockArgs
  "forks the receiver as a new process at the current scheduling priority"
  
  ^ GsProcess _forkBlock: self with: blockArgs env: 0 prio: nil 
%

category: 'Processes'
method:
forkAt: priority with: blockArgs
  "forks the receiver as a new process at the given priority"
  
  | proc |
  proc := GsProcess _forkBlock: self with: blockArgs env: 0 prio: priority.
  ^ proc 
%

category: 'Processes - Blue Book'
method:
newProcess
  "creates a new suspended process holding the receiver"

  ^ GsProcess _newForBlock: self with: #() env: 0
%

category: 'Processes - Blue Book'
method:
newProcessWith: argArray
  "creates a new suspended process holding the receiver to be evaluated 
   with the given arguments"

  ^GsProcess _newForBlock: self with: argArray env: 0
%

category: 'Updating'
method:
objectSecurityPolicy: anObjectSecurityPolicy

"Assign the receiver to the given security policy.  
 For complex blocks, also changes the security policy for variable contexts."

| vc |
super objectSecurityPolicy: anObjectSecurityPolicy .
(vc := staticLink) ifNotNil:[ vc objectSecurityPolicy: anObjectSecurityPolicy ].
%


! edited for 44426
category: 'Block Evaluation - Private'
method: 
onException: anException do: handlerBlock
 "An optimized form of on:do: .  anException may not be any
  object which responds to  handles: 

  Private.

  Try to evaluate the receiver, which should be a zero-argument block.
  If an exception occurs which is matched by anExceptionClass,
  evaluate the one argument block handlerBlock , passing it the exception 
  instance as its argument.

  These forms are supported:
    (1) on: anException do: handlerBlock
    (2) on: anArray     do: handlerBlock
    (3) on: anExceptionSet do: handlerBlock
    (4) on: anArray     do: anArrayOfBlocks

   This form is not supported
    (5) on: anObject     do: handlerBlock

  anException must be the class Exception  or a subclass thereof ;
  anExceptionSet must be a kind of ExceptionSet;
  anArray must be a kind of Array ;
  handlerBlock must be an instance of ExecBloc ; if any of these
  is violated, the send of on:do: will fail with an error.

  anArrayOfBlocks must be an Array containing instances of ExecBloc ,
  the elements are not checked during the on:do: send , and are only
  checked if they are attempted to be evaluated during exception handling.
  
  In forms 2,3,4 elements of anArray or anExceptionSet which
  are not a kind of Exception, or an Array containing kinds of Exception,
  will silently have no effect on catching exceptions.

  In form 4, elements of anArray may in turn be Arrays of Exceptions.
  In form 4, the offset of the first element of anArray containing a 
  match to a signaled exception defines the offset in anArrayOfBlocks
  at which to find the handlerBlock to invoke.

  For forms 1,2,3 if handlerBlock expects more than 1 argument,
  an error is generated if exception handling attempts to
  invoke that handlerBlock.  

  For form 4, elements of anArrayOfBlocks must be instances of ExecBlock,
  otherwise the corresponding entry in anArray will silently have no effect 
  on catching exceptions.  If an element of anArrayOfBlocks is an ExecBlock
  taking more than 1 argument, an error is generated if exception 
  handling attempts to invoke that block.

  If handlerBlock is invoked to handle an Exception which occurs during
  execution of the receiver and handlerBlock completes normally , then
  the result of handlerBlock (value of last expression in handlerBlock)
  will be the result of the on:do: send .  Other-than-normal
  completion of handlerBlock is available by use of 
  Exception's instance methods such as
    #return  #return:  #retry #retryUsing: #resume #resume: #pass #outer 
  within handlerBlock 

  When searching for a handlerBlock to handle a signaled Exception, the VM
  uses Behavior>>_subclassOf: semantics . classHistories of the 
  class of the signaled Exception and of anException or elements of anExceptionSet
  are ignored.
"

<primitive: 2018> "marks frame with Exception_Mark_NIL, always fails"
^ self value
%

category: 'Block Evaluation - Private'
method:
onSynchronous: anExceptionClass do: handlerBlock

"Same as onException:do: , except that asynchronous Exceptions 
 (those exceptions which have trappable==1) will never be handled
 by handlerBlock . 

 Allows asynchronous Exceptions to be handled by a higher
 level handler, even if the async Exception would match anExceptionClass.

 Private.
"

<primitive: 2029>
^ self value
%

! fixed 41744 , comments edited for 44745
category: 'Block Evaluation'
method:
on: exceptionSelector do: handlerBlock
 "Try to evaluate the receiver, which should be a zero-argument block.
  If an exception occurs and the expression
      exceptionSelector handles: theExceptionInstance
  returns true, then evaluate the one argument block handlerBlock , 
  passing it the exception instance as its argument.

  Two forms are supported directly by the VM, and the 'fast path code'
  below is used.
    (1) on: anException do: handlerBlock
    (2) on: anExceptionSet do: handlerBlock
  A third form is handled by Smalltalk code in the body of this method,
  and for this form only, #handles is sent to anObject to determine
  whether an exception should be handled .
    (3) on: anObject     do: handlerBlock

  anException must be the class Exception  or a subclass thereof ;
  anExceptionSet must be a kind of ExceptionSet;
  handlerBlock must be an instance of ExecBloc otherwise an error is generated.

  For forms 1,2,3 if handlerBlock expects more than 1 argument,
  an error is generated if exception handling attempts to
  invoke that handlerBlock.  

  If handlerBlock is invoked to handle an Exception which occurs during
  execution of the receiver and handlerBlock completes normally , then
  the result of handlerBlock (value of last expression in handlerBlock)
  will be the result of the on:do: send .  Other-than-normal
  completion of handlerBlock is available by use of 
  Exception's instance methods such as
    #return  #return:  #retry #retryUsing: #resume #resume: #pass #outer 
  within handlerBlock 

  For forms 1 and 2, when searching for a handlerBlock to handle a signaled Exception, 
  the VM uses Behavior>>_subclassOf: semantics . classHistories of the 
  class of the signaled Exception and of anException or elements of anExceptionSet
  are ignored.
"

<primitive: 2030> "always fails"
| fastPath | "fastPath := true by primitive if form 1 or 2 detected"
             "fastPath := nil  by primitive if form 3 detected."
fastPath ifNotNil:[ "fast path code"
 ^ self value
].
"Any changes to this method's code before this line may also
 require changes to code in comgen.c conditional on BcPrim_ENTER_onDo .
"
^ self onException: AbstractException do:[:ex |
    (exceptionSelector handles: ex) ifTrue:[
      handlerBlock argumentCount == 0 
         ifTrue:[ handlerBlock value ]
        ifFalse:[ handlerBlock value: ex ]
    ] ifFalse:[
      ex pass
    ]
  ]
%

category: 'Block Evaluation'
method: 
value: anArg on: anExceptionClass do: handlerBlock
  "Evaluate the one-argument receiver with the specified on:do: .
   Useful for avoiding complex blocks in the sender of value:on:do: "

  ^ [ self value: anArg ] onException: anExceptionClass do: handlerBlock
%

method: 
value: anArg onSynchronous: anExceptionClass do: handlerBlock
  "Evaluate the one-argument receiver with the specified onSynchronous:do: .
   Useful for avoiding complex blocks in the sender of value:onSynchronous:do: "

  ^ [ self value: anArg ] onSynchronous: anExceptionClass do: handlerBlock
%

method:
cull: anArg
  "Return the value of the receiver evaluated with 0 or 1 arguments. If the block
   expects 1 argument pass anArg as the value of the argument"

  ^ self argumentCount == 0
    ifTrue: [ self value ]
    ifFalse: [ self value: anArg ]
%

method:
cull: firstArg cull: secondArg
  "Return the value of the receiver evaluated with between 0 and 2 arguments,
   discarding arguments not needed by the receiver."

  | nargs |
  (nargs := self argumentCount) < 2 ifTrue:[
     nargs == 1 ifTrue:[ ^ self value: firstArg ].  
     ^ self value .
  ].
  ^ self value: firstArg value: secondArg
%

method:
cull: firstArg cull: secondArg cull: thirdArg
  "Return the value of the receiver evaluated with between 0 and 3 arguments,
   discarding arguments not needed by the receiver."

  | nargs |
  (nargs := self argumentCount) < 3 ifTrue:[
     nargs == 2 ifTrue:[ ^ self value: firstArg value: secondArg ].
     nargs == 1 ifTrue:[ ^ self value: firstArg ].
     ^ self value .
  ].
  ^ self value: firstArg value: secondArg value: thirdArg 
%

method:
cull: firstArg cull: secondArg cull: thirdArg cull: fourthArg
  "Return the value of the receiver evaluate with between 0 and 4 arguments,
   discarding arguments not needed by the receiver."

  | nargs |
  (nargs := self argumentCount) <= 2 ifTrue:[
     nargs == 2 ifTrue:[ ^ self value: firstArg value: secondArg ].
     nargs == 1 ifTrue:[ ^ self value: firstArg ].
     ^ self value .
  ].
  nargs == 3 ifTrue:[ 
     ^ self value: firstArg value: secondArg value: thirdArg .
  ].
  ^ self value: firstArg value: secondArg value: thirdArg value: fourthArg.
%

category: 'Private'
method: 
_deepCopyWith: copiedObjDict

"Private.  Used internally to implement deepCopy."

^ self.
%

category: 'Clustering'
method:
clusterDepthFirst

 self cluster ifTrue:[ ^ true ].
 self method cluster .   "cannot use instVarAt to access"
        "other instVars are specials or transient"
 ^ false 
% 
  
!--------------------------------------------------------
removeallmethods ExecBlock0
removeallclassmethods ExecBlock0

category: 'Documentation'
classmethod:
comment
^'An instance of ExecBlock0 is an in-memory instance of ExecBlock which
 the VM already knows should have zero args .
 ExecBlock0 was introduced in Gemstone64 version 3.0

Constraints:
	_method: GsNMethod
	iFields1: SmallInteger
	staticLink: VariableContext
	stkSerialNum: SmallInteger
	ccallin: CCallin'.
%

!--------------------------------------------------------
removeallmethods ExecBlock1
removeallclassmethods ExecBlock1

category: 'Documentation'
classmethod:
comment
^'An instance of ExecBlock1 is an in-memory instance of ExecBlock which
 the VM already knows should have one args .
 ExecBlock1 was introduced in Gemstone64 version 3.0

Constraints:
	_method: GsNMethod
	iFields1: SmallInteger
	staticLink: VariableContext
	stkSerialNum: SmallInteger
	ccallin: CCallin'.
%

!--------------------------------------------------------
removeallmethods ExecBlock2
removeallclassmethods ExecBlock2

category: 'Documentation'
classmethod:
comment
^'An instance of ExecBlock2 is an in-memory instance of ExecBlock which
 the VM already knows should have two args .
 ExecBlock2 was introduced in Gemstone64 version 3.0

Constraints:
	_method: GsNMethod
	iFields1: SmallInteger
	staticLink: VariableContext
	stkSerialNum: SmallInteger
	ccallin: CCallin' .
%

!--------------------------------------------------------
removeallmethods ExecBlock3
removeallclassmethods ExecBlock3

category: 'Documentation'
classmethod:
comment
^'An instance of ExecBlock3 is an in-memory instance of ExecBlock which
 the VM already knows should have three args .
 ExecBlock3 was introduced in Gemstone64 version 3.0

Constraints:
	_method: GsNMethod
	iFields1: SmallInteger
	staticLink: VariableContext
	stkSerialNum: SmallInteger
	ccallin: CCallin' .
%

!--------------------------------------------------------
removeallmethods ExecBlock4
removeallclassmethods ExecBlock4

category: 'Documentation'
classmethod:
comment
^'An instance of ExecBlock4 is an in-memory instance of ExecBlock which
 the VM already knows should have four args .
 ExecBlock4 was introduced in Gemstone64 version 3.0

Constraints:
	_method: GsNMethod
	iFields1: SmallInteger
	staticLink: VariableContext
	stkSerialNum: SmallInteger
	ccallin: CCallin' .
%

!--------------------------------------------------------
removeallmethods ExecBlock5
removeallclassmethods ExecBlock5

category: 'Documentation'
classmethod:
comment
^'An instance of ExecBlock5 is an in-memory instance of ExecBlock which
 the VM already knows should have five args .
 ExecBlock5 was introduced in Gemstone64 version 3.0

Constraints:
	_method: GsNMethod
	iFields1: SmallInteger
	staticLink: VariableContext
	stkSerialNum: SmallInteger
	ccallin: CCallin' .
%

!--------------------------------------------------------
removeallmethods ExecBlockN
removeallclassmethods ExecBlockN

category: 'Documentation'
classmethod:
comment
^'An instance of ExecBlockN is an in-memory instance of ExecBlock which
 is ready to execute for a Ruby #call: .  At each send, excess args may
 need to be deleted or nils added to match number of args required by the 
 block.  ExecBlockN was introduced in Gemstone64 version 3.0

Constraints:
	_method: GsNMethod
	iFields1: SmallInteger
	staticLink: VariableContext
	stkSerialNum: SmallInteger
	ccallin: CCallin' .
%


