!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! ************************************************************************
! ATTENTION!!!! This is NOT the definitive list of errors!
! Any new error messages must be added to gcierr.ht FIRST, and then
! text must be added to all supported language dictionaries, such as
! in this file.
! ************************************************************************
!
!  Description - This generates the English version of errors for the GEM
!   system.  Each error is an Array of Strings and Integers.  The Integers
!   refer to parameters reported with the error by the underlying machine.
! 
!   The virtual image refers to errors by number.  Refer to gcierr.ht
!   for the C mapping of error numbers to mnemonics.
!
!   Errors are accessed through the Globals Dictionary Association with
!   key #errorText.  This leads to a language dictionary.  Each entry
!   in the language dictionary is an Array of 4 Arrays, one for each category
!   of error.  Each of these Arrays has elements 1 - n for the n errors
!   in that category.  The actual error number is found by adding the
!   index into the error Array to 1000 times the index into the 
!   category Array.
!
!========================================================================

commit

! Create script to map error strings from mnemonics in ErrorSymbols
run

| doit |
System currentObjectSecurityPolicy: Globals objectSecurityPolicy .

doit := [:name :errsArray :message | | theOffset |
          theOffset := ErrorSymbols at: name.
          errsArray size < theOffset ifTrue: [ errsArray size: theOffset].
          errsArray at: theOffset put: message 
        ].

Globals at: #doit put: doit .

^true
%
run
| err |
" create error Array of error message descriptors.  "
err := Array new: 4154.
(Globals at: #GemStoneError) at: #English put: err.
^ true
%
commit
run
| err |
" create error strings for compiler category"

err := (Globals at: #GemStoneError) at: #English.

doit value: #compilerErrStDB value: err value:
    'compilation errors -- parameter 1 is error descriptor'.
doit value: #StDBErrMaxPath value: err value: 'path too long'.
doit value: #StDBErrArrayTooLarge value: err value: 'Array constructor too large (exceeds 10000 elements)'.
doit value: #StDBErrEofInStrLit value: err value:
    'missing end of literal mark (prime)'.
doit value: #StDBErrEofInComment value: err value:
    'missing end of comment (")'.
doit value: #StDBErrEofInChrLit value: err value:
    'invalid or missing Character'.
doit value: #StDBErrUnexpectedColon value: err value:
    'invalid colon'.
doit value: #StDBErrUnexpectedPound value: err value: 'invalid pound sign'.
doit value: #StDBErrBadChr value: err value: 'illegal character'.
doit value: #StDBErrLitTooBig value: err value: 'string literal too big (exceeds 5M bytes)'.
doit value: #StDBErrSymTooBig value: err value: 'selector or symbol literal would exceed 1024 bytes' .
doit value: #StDBErrBadRadix value: err value: 'illegal radix'.
doit value: #StDBErrBadChrInRadixNum value: err value:
    'illegal Character in number'.
doit value: #StDBErrExpectedRightParen value: err value:
    'expected a right parenthesis'.
doit value: #StDBErrExpectedPrimary value: err value:
    'expected a primary expression'.
doit value: #StDBErrExpectedVar value: err value:
    'expected a variable name' .
doit value: #StDBErrExpectedSelector value: err value:
    'missing or bad message pattern'.
doit value: #StDBErrMaxArg value: err value: 'too many arguments'.
doit value: #StDBErrExpectedEof value: err value:
    'characters found after end of method'.
doit value: #StDBErrExpectedStatement value: err value: 'expected start of a statement'.
doit value: #StDBErrExpectedPrim value: err value:
    'expected keyword "primitive"'.
doit value: #StDBErrExpectedPrimNum value: err value: 'expected an integer'.
doit value: #StDBErrBadPrimNum value: err value: 'illegal primitive number'.
doit value: #StDBErrExpectedRightBrace value: err value:
    'expected a right brace (}) ' .
doit value: #StDBErrMaxArgsTemps value: err value:
    'too many arguments and temporaries'.
doit value: #StDBbadAtRuby value: err value:
    'second and subsequent keywords for @ruby may only be  _:  or __STAR: ' .
doit value: #StDBnoRuby value: err value:
    'This VM does not support Ruby code generation' .
doit value: #StDBBadLit value: err value: 'invalid numeric literal' .
^ true
%
commit
! edited ExpectedAssignable message for 46455
run
| err |
" create more compiler error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #StDBErrExpectedVerticalBar value: err value:
    'missing end of temporaries mark (|)'.
doit value: #StDBErrExpectedMsgPattern value: err value:
    'invalid message pattern'.
doit value: #StDBErrExpectedGt value: err value:
    'missing end of primitive mark (>)'.
doit value: #StDBErrBadFlt value: err value: 'illegal exponent'.
doit value: #StDBErrExpectedAssignable value: err value:
    'expected an assignable variable' .
doit value: #StDBErrAlreadyDefined value: err value:
    'variable has already been declared'.
doit value: #StDBErrNotDefined value: err value: 'undefined symbol'.
doit value: #StDBErrPredicateTooBig value: err value:
    'predicate too complex'.
doit value: #StDBErrBlocksTooDeep value: err value:
    'blocks or if statements can only be nested 255 levels'.
doit value: #StDBErrUnexpectedToken value: err value: 'unexpected token'.
doit value: #StDBErrExpectedRightBracket value: err value:
    'expected a right bracket (])'.
doit value: #StDBErrStackTooBig value: err value: 'method too complex'.
doit value: #StDBErrStmtNoEffect value: err value: 'statement has no effect'.
doit value: #StDBErrCodeTooBig value: err value: 'method too large'.
doit value: #StDBErrMaxLits value: err value: 'too many literals'.
doit value: #StDBErrMaxSelectors value: err value: 'too many selectors'.
doit value: #StDBErrPrimNotAllowed value: err value:
    'Must be SystemUser or have CompilePrimitives privilege to compile a primitive or protected method.'.
doit value: #StDBErrExpectedBoolOrExpr value: err value:
    'The object was not true, false or a non-literal expression.'.
doit value: #StDBErrExpectedLitOrExpr value: err value:
    'The object was not a literal or a non-literal expression.'.
doit value: #StDBErrExpectedBlockOrExpr value: err value:
    'The object was not a block or a non-literal expression.'.
doit value: #StDBErrExpectedIntOrExpr value: err value:
    'The object was not a kind of Integer or a non-literal expression.'.
doit value: #StDBErrNotPosIntOrExpr value: err value:
   'The object was not a positive kind of Integer or a non-literal expression.'.
doit value: #StDBErrDisallowedSelector value: err value:
    'You may not compile a method for this selector.'.
doit value: #StDBErrBadNumOfArgs value: err value:
    'The block has the wrong number of arguments for this selector.'.
doit value: #StDBNoDigitInAtEnv value: err value:
    'For @rubyN: or @envN: , no digits found for N ' .
doit value: #StDBBadChInAtEnv value: err value:
    '@env or @ruby is followed by character other than 0..9 or : ' .
doit value: #StDBEnvTooBig value: err value:
    'For @envN: or @rubyN: , value of N is > 255  ' .
doit value: #StDBByteArrElemRange value: err value:
    'Element of a ByteArray literal not a SmallInteger within 0..255' .
^ true
%
commit
run
| err |
" create more compiler error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #StDBErrLiteralInvariant value: err value:
    'Attempt to modify an invariant literal.'.

doit value: #StDBErrExpectedOperand value: err value:
    'An operand was missing from the given SelectBlock term.'.
doit value: #StDBErrBadSelectOperator value: err value:
    'An unacceptable operator was given in the SelectBlock ' ,
	'term.  The operator must be one of <, >, <=, >=, =, ~=, ==, ' ,
	' or ~~.'.
doit value: #StDBErrExpectedSelectTerm value: err value:
    'The given SelectBlock must contain a Boolean expression.'.
doit value: #StDBErrTermsNotParen value: err value:
    'The conjoined terms of the given SelectBlock were not ' ,
	'parenthesized.'.
doit value: #StDBErrBadNumOfSelectArgs value: err value:
    'A SelectBlock was encountered that does not have ' ,
	'one argument.'.
doit value: #StDBErrSourceNotEUCFormat value: err value:
    'The bytes of the source string are not in EUC format.'.
doit value: #StDBErrTooManyBlocks value: err value:
    'The maximum number of blocks in a method is 65536.'.
doit value: #StDBErrMaxArgs value: err value:
    'The maximum number of arguments to a method or block is 255.'.
doit value: #StDBErrCodeGenLogic value: err value:
    'Internal logic error in compiler:' .
doit value: #StDBErrFirstTermCannotBeSetValued value: err value:
    'The first term of a path in a SelectBlock cannot indicate ' ,
        'search over a set-valued instance variable (i.e. cannot be *). '.
doit value: #StDBErrIllegalProtectedMethod value: err value:
    'The specified primitive may not be a protected method.' .
doit value: #StDBMissingProtectedToken value: err value:
    'Method requires either <protected> or <unprotected> directive.' .
doit value: #StDBSuperNotAllowed value: err value:
    'super not allowed in instance method for Object or when superClass is nil.'.
doit value: #StDBUnusedTemp value: err value:
    'Unused method or block temporary.'.
doit value: #StDBMbStrOddSize value: err value:
    'Corrupt source string, a Double/QuadByteString has odd basic size.'.
doit value: #StDBPrimitiveInPragma value: err value:
    'primitive:, protected, unprotected, requiresVc not allowed as keywords in pragma.'.
doit value: #StDBPragmaInPrim value: err value:
    'pragma not allowed in a primitive:, protected, unprotected , or requiresVc method.'.
^ true
%
commit
run
| err |

" create execution error strings "
err := (Globals at: #GemStoneError) at: #English.

doit value: #rtErrInvalidTransMode value: err value:
     #( 1 ' is not recognized as a valid transactionMode.' ).
doit value: #rtErrBadSymbolList value: err value:
    #('The user''s symbol list is not a kind of Array containing '
	   'objects that are a kind of SymbolDictionary.').
doit value: #objErrBadOffsetIncomplete value: err value:
    #('An indexable object or NSC ' 1 ' was referenced with an '
	   'index ' 2 ' that was out of range.').
doit value: #rtErrBadSubscript value: err value:
    #('A subscript ' 2 ' that was out of range or not an Integer was used to '
	   'index the object ' 1 '.').
doit value: #gciErrBadNumMsgParts value: err value:
    #( 'GciSendMsg was called with an inconsistent number of '
	   ' message parts.').
doit value: #rtErrAuthArraySize value: err value:
    #('(OBSOLETE) An attempt was made to change GsObjectSecurityPolicy authorization with '
	   'an Array ' 1 ' which should be of size 5.').
doit value: #rtErrShouldNotImplement value: err value:
    #('A method was invoked that has been specifically disallowed '
	   'in a subclass. Receiver:  ' 1 '.  Selector:  ' 2 '.').
doit value: #rtErrSubclassResponsibility value: err value:
    #('A method has been invoked in the abstract superclass ' 1
	   ' that was designed to have been overridden in a concrete '
	   'subclass.  Selector:  '   2 '.').
doit value: #objErrClassVariant value: err value:
    #('An attempt was made to create an instance of the modifiable'
	   ' class ' 1 ' .  Send "immediateInvariant" to the class to terminate'
	   ' class modification and allow instance creation.' ).
doit value: #rtErrDoesNotUnderstand value: err value:
    #('No method was found for the selector ' 2
	   ' when sent to ' 1 ' with arguments contained in ' 3 '.').
doit value: #objErrNotObjectSecurityPolicy value: err value:
    #('Invalid objectSecurityPolicyId ' 1 ' , ' 2 ).
doit value: #objErrNotIndexable value: err value:
    #('An attempt was made to index the object ' 1 ' that is not indexable.').
doit value: #rtErrCantPerform value: err value:
    #(1 ' cannot perform the selector ' 2 ' with the arguments in ' 3  '. '
      'Perform may have been attempted with wrong number of args.').
doit value: #classErrSubclassDisallowed value: err value:
    #('Cannot create a subclass of the class ' 1 '.').
doit value: #objErrNotInColl value: err value:
    #('The object ' 2 ' was not found in the collection ' 1 '.').
doit value: #rtErrBadPattern value: err value:
    #(1 ' is an illegal pattern for string comparisons.').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #repErrMaxExtents value: err value:
    #('Attempt to create new extent failed because ' 1 ) .
doit value: #rtErrBadBlockArgCount value: err value:
    #('An attempt was made to evaluate the block or method ' 1 ' with ' 3
	   ' arguments when ' 2 ' were expected.').
doit value: #objErrCollectionEmpty value: err value:
    #('An attempt was made to access elements of the empty '
	   'collection ' 1 '.').
doit value: #rtErrArgNotChr value: err value:
    #('An attempt was made to store the object ' 1 ' into a string.').
doit value: #rtErrKeyNotFound value: err value:
    #('A reference using the non-existent key ' 2 
       ' was made into the dictionary ' 1 ).
doit value: #rtErrBadDateTimeArgs value: err value:
    #('Invalid arguments given to DateTime instance creation.').
doit value: #genericKernelError value: err value:
    #( 'Error, ' 1 ) .
doit value: #rtErrNoSuchBp value: err value:
    #('The specified breakpoint does not exist.').
doit value: #repErrCantCreateFile value: err value:
    #('The system was unable to create the file ' 1 ', ' 2 ).
doit value: #numErrIntDivisionByZero value: err value:
    #('An attempt was made to divide ' 1 ' by zero.').
doit value: #rtErrSpecialOrNotCommitted value: err value:
    #('A committed object was expected rather than object ' 1 ', which is either special or uncommitted.').
doit value: #repErrPreGrowFailure value: err value:
    #('The extent ' 1 ' could not be created because of an attempt to '
      'initialize the file size when the disk was too full.').
doit value: #rtErrBeginTrans value: err value:
 #('An attempt was made to begin a new transaction when already in a transaction.').
doit value: #rtErrPrimOutsideTrans value: err value:
    #('An attempt was made to execute a primitive that is not allowed when not '
       'inside of a transaction.  Examples are: commit, backup and restore.').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #objErrInvariant value: err value:
    #('Attempt to modify the invariant object ' 1 ).
doit value: #classErrMethCatExists value: err value:
    #('An attempt was made to create the method category ' 2
	   ' which already exists.  Class:  ' 1 '.').
doit value: #classErrSelectorNotFound value: err value:
    #('A reference was made to the selector ' 2
	   ' which could not be found in the class ' 1
	   ' method dictionary.').
doit value: #classErrMethCatNotFound value: err value:
    #('In searching the class ' 1 ' the category name ' 2 ' was not found.').
doit value: #lockErrRemove value: err value:
    #('The user is not allowed to remove a lock on an object ' 1 
      ' that they do not have a lock on.').
doit value: #classErrByteObjInstVars value: err value:
    #('An attempt was made to create a byte subclass with '
	   'instance variables.  Superclass:  ' 1 '.').
doit value: #classErrInvariantSuperClass value: err value:
    #('An attempt was made to create a variant subclass of an '
	   'invariant class.').
doit value: #classErrNscNotIndexable value: err value:
    #('An attempt was made to create an indexable subclass of '
	   'the NSC class ' 1 '.').
doit value: #repErrExtentNotMounted value: err value:
    #('The extent with filename or extentId: ' 1 ' , '
       ' was not part of the Repository.').
doit value: #rtErrPagesNeedScavenging value: err value:
    #('Attempt to schedule pages for scavenging failed because '
       'some other session had the commit token.  Try again later.').
doit value: #classErrNscInstVars value: err value:
    #('An attempt was made to create an NSC subclass ' 1
	   ' with instance variables.').
doit value: #classErrClassVarNameExists value: err value:
    #('An attempt was made to create the new class variable ' 2
	   ' with the same name as an existing class variable in class '
	   1 '.').
doit value: #classErrPoolDictExists value: err value:
    #('An attempt was made to add the dictionary ' 2
	   ' to a shared pool of class ' 1
	   ' in which it was already a member.').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #classErrPoolDictNotFound value: err value:
    #('An attempt was made to remove the dictionary ' 2
	   ' from the shared pool of class ' 1
	   ' in which it was not a member.').
doit value: #clampErrNoSuchInstvar value: err value:
    #('During clamp compilation for class ' 1
	   ' an instance variable clamp was encountered for '
	   ' non-existent instance variable ' 2 '.').
doit value: #clampErrNotAClass value: err value:
    #('An object ' 1 ' was specified for instance '
	   'variable clamping that was not a class object.').
doit value: #clampErrNotAClampspec value: err value:
    #('In an attempt to perform clamped object traversal, the '
	   'specified object ' 1 ' was not a ClampSpecification object.').
doit value: #clampErrBadArg value: err value:
    #('The object ' 1 ' has an implementation or size not allowed'
	   ' in clamp specifications.').
doit value: #repErrBadExtentSize value: err value:
    #('The given maximum extent size (' 1 ') is smaller than the '
	   'minimum size (' 2 ') allowed for an extent.').
doit value: #repErrCantOpenFile value: err value:
    #('The file ' 1 ' could not be opened.').
doit value: #rtErrNoSuchInstVar value: err value:
    #('The instance variable ' 2 ' was not found in evaluating '
	   'a path expression for the object ' 1 '.').
doit value: #rtErrTagNotAllowed value: err value:
    #('An attempt was made to put a tag on the object '
	   1 ' which does not support tags.').
doit value: #rtErrBadTagNum value: err value:
    #('The tag index ' 2 ' requested for object ' 1 ' is not '
	   ' allowed, max tag index is ' 3 ).
doit value: #segErrMaxSegGroups value: err value:
    #('(OBSOLETE) An attempt was made to add the group ' 2
	   ' to the security policy ' 1 ', which already recognizes four groups.').
doit value: #rtErrNoMessage value: err value:
    #('There is no error message for the error ' 1
	   ' in the SymbolDictionary ' 2 '.').
doit value: #segErrBadAuthKind value: err value:
    #('An attempt was made to change the authorization for '
	   'the security policy ' 1 ' to the unrecognized value ' 2 '.').
doit value: #rtUnresolvedFwdRefs value: err value:
    #('Commit failed. GciCreate/GciStore have left unsatisfied'
                 ' forward references to the objectId ' 1 '.'
                 '  Create the object with GciCreate and retry commit.').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #rtErrArgNotPositive value: err value:
    #(1 ' was found where a positive numeric value was expected.').
doit value: #rtErrArgOutOfRange value: err value:
    #('Argument is too large or out of range: ' 1  
       ', allowed range is ' 2 ' to '  3 ).
doit value: #rtErrCannotChgConstraint value: err value:
    #('A constraint cannot be changed in a Dictionary that is not empty.').

doit value: #rtErrStackLimit value: err value:
    #('Smalltalk execution stack error, ' 1 ).
doit value: #rtErrStackLimitRed value: err value:
    #('Smalltalk execution stack overflow, Red Zone.').
doit value: #rtErrNewStackLimit value: err value:
    #('The value ' 1 ' specified for the new stack limit'
 ' is too large or is smaller than the current size of the execution stack.' ).
doit value: #numErrArgNotChr value: err value:
    #('An attempt was made to coerce the integer ' 1
	   ' into a ' 2 ', but its value was not in range.').
doit value: #numErrFltException value: err value:
    #('A floating point exception of type ' 1 ' was raised on the operation '
      2 '.  The default result is ' 3 '.  The first argument is ' 4 '.').
doit value: #numErrArgNotFltException value: err value:
    #('An unrecognized float exception ' 1 ' was specified.').
doit value: #numErrArgNotRoundingMode value: err value:
    #('An unrecognized float rounding mode ' 1 ' was specified.').
doit value: #segErrCantMoveObj value: err value:
    #('An attempt was made to change the security policy of '
	   'the special object or symbol ' 1 '.').
doit value: #rtErrExceptionNotLinked value: err value:
    #('An attempt was made to unlink an exception ' 1
	   ' that is not installed.').
doit value: #numErrArgNotFltStatus value: err value:
    #('An invalid float status ' 1 ' was specified.').
doit value: #lockErrUndefinedLock value: err value:
    #('A request to lock object ' 1 ' using an invalid kind ' 2 '.').

"rtErrHardStackLimit deleted"

doit value: #lockErrIncomplete value: err value:
    #('One or more of the locks you requested was denied or is dirty.').
doit value: #lockErrObjHasChanged value: err value:
    #('A request to lock object ' 1 ' was granted, but the object '
	   ' has been read or written since you started this transaction.').
doit value: #lockErrDenied value: err value:
    #('A request to lock object ' 1 ' with lock kind ' 2 ' was denied.').
doit value: #rtErrMaxClusterId value: err value:
    #('A request to create a new ClusterBucket exceeded the maximum ' 
      'size allowed for a clusterId = ' 1 '.').
doit value: #lockErrDeadlock  value: err value:
    #('A request to lock object ' 1 ' might deadlock.') .
doit value: #lockErrTimeout  value: err value:
    #('A request to lock object ' 1 ' timed out.') .
doit value: #lockErrInvalidObject  value: err value:
    #('Lock queue already bound to a different object, ' 1 ).
doit value: #rtErrLrgIntOverflow value: err value:
    #('Integer overflow, an Integer would exceed 130144 bits.').
doit value: #rtErrCompilerBadIr value: err value:
    #('Invalid IR input to code generator, ' 1 ).
doit value: #errPrintf value: err value:
    #('Error parsing format string, ' 1 ).
doit value: #errBadIvName value: err value:
    #('Invalid instVar name, ' 1 2 ).
doit value: #errRangeErr value: err value:
    #('RangeError,  ' 1 ).
doit value: #errTypeErr value: err value:
    #('TypeError,  ' 1 ).
doit value: #errArgumentErr value: err value:
    #('ArgumentError,  ' 1 ).
doit value: #errRubySystemExitErr value: err value:
    #('SystemExit Error,  ' 1 ).
doit value: #errMemoryRegionFull value: err value:
    #('The memory region ' 1 ',  is full.' ).
doit value: #rtErrNotPowerOfTwo value: err value:
    #('The value ' 1 ' was expected to be a power of 2.' ).
doit value: #rtErrPasswordMissingChar value: err value:
    #('The password must contain at least one character of type: ' 2 ).
doit value: #errDuringMtOperation value: err value:
   #('An error occurred during a multi-threaded operation: ' 1 ).
doit value: #authErrStoreTrav value: err value: 
   #('Attempt to change GsObjectSecurityPolicy via store traversal. ' 1 ) .
^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #rtErrBadErr value: err value:
    #('An attempt to signal error ' 1 ' in dictionary ' 3
       ' was made, but the signal number ' 1 ' is less than one.').
doit value: #rtErrUserIdAlreadyExists value: err value:
    #('An attempt was made to add a UserProfile to the UserProfileSet ' 1
	   ' which already has an entry with the same userId:  ' 2 '.').
doit value: #rtErrCantReturn value: err value:
    #('Execution could not return from the current Activation, ' 1 ).
doit value: #rtErrCantChangeClass value: err value:
    #('An illegal attempt was made to change the class of '
	   'object ' 1 ' to class ' 2 ' ; ' 3 ).
doit value: #rtErrCantBecomeSpecial value: err value:
    #('An attempt was made to use become on the object ' 1
	   ' that has a special implementation.') .
doit value: #rtErrGarbageCollect value: err value:
    #('Attempt to run markForCollection when not the only user on the '
      'system and the concurrency mode is set for NO_RW_CHECKS_NO_READ_SET.').
doit value: #rtErrPrimNotFound value: err value:
    #('Primitive number ' 1 ' does not exist in the virtual machine.') .
doit value: #rtErrNilKey value: err value:
    #('An illegal attempt was made to store nil '
      'as a key in the dictionary ' 1 '.').
doit value: #rtErrNoInstVars value: err value:
    #('An attempt was made to directly access the instance '
	   'variables of the object ' 1
	   ' but the object has no instance variables.').
doit value: #rtErrExpectedBoolean value: err value:
    #('Expected ' 1 ' to be a Boolean.').
doit value: #rtErrDirtyObjsNeedsInit value: err value:
  #('GciDirtyObjsInit must be executed at least once before GciDirtySaveObjs.').

doit value: #rtErrWeakDictNeedsInit value: err value:
  #('GciWeakDictInit must be executed before other weak dictionary calls.').

doit value: #rtErrTrackedObjsNeedsInit value: err value:
  #('GciTrackedObjsInit must be executed before other tracked objs calls.').

doit value: #rtErrCantChangePassword value: err value:
    #('An illegal attempt was made to change the password '
	   'of ' 1 ', which is not the UserProfile '
           'of the current session.').

doit value: #rtErrBadCopyFromTo value: err value:
    #('An invalid index range was specified for a sequenceable collection '
           'starting index ' 1 ', ending index ' 2 '.').

doit value: #rtErrCantBecomeBothIdx value: err value:
    #('Both the receiver and argument of become participate in indexes. '
      'Become is not allowed because they are not of the same class.  '
      1 ' , ' 2   ) .

doit value: #rtErrSymbolTooLarge  value: err value:
    #('Cannot create a Symbol, String exceeds 1024 Characters. ' 1 ).

doit value: #rtErrOmFlushFailed  value: err value:
    #('Cannot commit, ' 1 ).

doit value: #rtErrNoExistingSymbol  value: err value:
    #('Message send failed, there is no Symbol with the value ' 1 ).

doit value: #rtErrObjMustBeCommitted value: err value:
    #('The object ' 1 ' is a temporary object but is required to '
      'be a committed object. ' 2 ).

doit value: #rtErrCantWriteFile value: err value:
    #('Error writing to file ' 1 '.  errno=' 2 ' message=' 3 ).

doit value: #rtErrCantReadFile value: err value:
    #('Error reading file ' 1 '.  errno= ' 2 ' message=' 3 ).

doit value: #rtErrFileCorrupt value: err value:
    #('The file ' 1 ' is corrupt, ' 2).

doit value: #rtErrGcCommitFailure value: err value:
    #('A commit during a garbage collection operation failed.').

doit value: #rtErrSymbolCreateErr value: err value:
    #('Symbol creation not allowed, ' 1 ', symbol value ' 2 ) .

doit value: #rtErrObjIsNp value: err value:
    #('The object ' 1 ' may not be committed, ' 2 ) .

doit value: #rtErrSuperclassIsNP value: err value:
    #('Cannot make instances of ' 1 ' persistent because instances of superClass are non-persistent.' ) .

doit value: #rtErrContinueTransFail value: err value:
    #('continueTransaction failed, ' 1 ).

doit value: #rtErrMethodSrcInvalid value: err value:
    #(' invalid method source string').

doit value: #lgcErrTravBuffSize value: err value:
    #(' Traversal buffer received over the network exceeds destination allocatedBytes ' ).

doit value: #rtErrAbortWouldLoseData value: err value:
    #('A method that requires an abort is being executed, however, an abort would result in lost data.').

doit value: #gciErrCallNotSupported value: err value:
    #('This GCI call is not supported when invoked from a client user action.' ).

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #rtErrNoProcessToContinue value: err value:
    #('The Process ' 1 ' to continue from is invalid,' 2).
doit value: #rtErrBadStreamPosition value: err value:
    #('An attempt was made to set the Stream ' 1
	   ' to position ' 2 ' beyond the limits of the collection.').
doit value: #rtErrBadArgKind value: err value:
    #('The object ' 1 ' was not of the expected class ' 2 '.').
doit value: #classErrClassVarNotFound value: err value:
    #('An attempt was made to remove the non-existent class '
	   'variable ' 2 ' from the class ' 1 '.').
doit value: #assocErrNoElementsDetected value: err value:
    #('detect: was sent to a collection, but no elements of '
	   'the collection ' 1 ' satisfied the block ' 2 '.').
doit value: #classErrNotAVar value: err value:
    #('The symbol ' 2 ' was not resolvable as a variable within the class '
         1 '.').
doit value: #segErrTooManyGroups value: err value:
    #('The collection of groups ' 1
	   ' was specified that held more than four members.').
doit value: #rtErrExpectedByteValue value: err value:
    #('Byte objects store values from 0 to 255, not ' 1 '.').
doit value: #classErrBadFormat value: err value:
    #('Invalid format for new class ' 1 ' , ' 2 ).
doit value: #objErrDoesNotExist value: err value:
    #('The object with object ID ' 1 ' does not exist.').
doit value: #objErrNotOopKind value: err value:
    #('The object ' 1 ' is not implemented as a pointer object.').
doit value: #objErrNotByteKind value: err value:
    #('The object ' 1 ' is not implemented as a byte object.').
doit value: #objErrNotNscKind value: err value:
    #('The object ' 1 ' is not implemented as an NSC object.').
doit value: #objErrAlreadyExists value: err value:
    #('Attempt to create an object with object identifier ' 2 
      ', which already exists as object ' 1 '.' ).
doit value: #objErrOopNotAllocated value: err value:
    #('Attempt to store a forward reference using the object '
	'identifier ' 1 ' which has not been allocated to this session.').
doit value: #objErrConstraintViolation value: err value:
    #('Attempt to store ' 2 ' of class ' 4 ' into an instance '
        'variable of ' 1 ' constrained to be ' 3 '.').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #objErrDictConstraintViolation value: err value:
    #('Attempt to store ' 2 ' of class ' 4 ' into ' 1 ' constrained '
      'to hold only instances which are a kind of ' 3 '.').

doit value: #rtErrExpectedClass value: err value:
    #('The object ' 1 ' was expected to be a class but was not.').
doit value: #objClassNotOopKind value: err value:
    #('The class ' 1  ' does not specify pointer object format.').
doit value: #objClassNotOopOrNscKind value: err value:
    #('The class ' 1 ', used as argument to GciCreateOopObj '
	' does not specify pointer or NSC object format.').
doit value: #rtMaxRecursion value: err value:
    #('Too many levels of recursion from user actions to GemStone Smalltalk or '
      'within object manager.').
doit value: #gciErrOpInProgress value: err value:
    #('GCI operation requested before the current '
	'nonblocking or thread safe call was completed.').

doit value: #repErrBkupNoStart value: err value:
   #('Unable to begin full backup, ' 1 ).

doit value: #rtErrDecodedObjDoesNotExist value: err value:
  #('The decoded object ID ' 1 ' at offset ' 2 ' does not exist. ').

doit value: #rtErrMaxSuperclassDepth value: err value:
  #('Cannot create subclass of ' 1 ' . Subclass would exceed ' 2 
     ' levels below Object when following the superClass chain. ' ).

doit value: #rtErrReclaimAllMissingGcGem value: err value:
  #('A reclaimAll operation was attempted but at least one GC session '
    'is not running.  Ensure all reclaim sessions and the Admin GC session '
    'are running and try the operation again.' ).

^ true
%

commit
run
| err |

err := (Globals at: #GemStoneError) at: #English.

"
! create execution error strings 
! break up addition to errorText; maximum literals per method
! is 255
"

doit value: #objErrBadOffset value: err value:
    #('The object ' 1 ' was indexed using structural access with the index ' 3
       ' that was out of range. The maximum index is ' 2 '.').
doit value: #objErrCantCreateInstance value: err value:
    #('Creating an instance of class ' 1 ' is not allowed.').
doit value: #objClassNotByteKind value: err value:
    #('The class ' 1 ', used as argument to GciCreateByteObj '
	'does not specify byte object format.').
doit value: #lockErrArgSize value: err value:
    #('An argument to a locking primitive was too large.').
doit value: #objErrNotSpecialKind value: err value:
    #('The object ' 1 ' is not implemented as a special object.').
doit value: #authErrSegRead value: err value:
    #('An attempt was made to read the object with ID ' 1
	' in objectSecurityPolicyId ' 2 ' with insufficient authorization. ' 3).
doit value: #authErrSegWrite value: err value:
    #('An attempt was made to modify the object ' 1
	' in objectSecurityPolicyId ' 2 ' with insufficient authorization.').
doit value: #objErrNotOopOrNsc value: err value:
    #('An operation was attempted on the object ' 1
	' that is legal only on pointer or NSC objects.').
doit value: #rtErrObsolete value: err value:
    #(1 ' cannot respond to the selector ' 2 ' because its class is obsolete.').
doit value: #objErrNotFlt value: err value:
    #('The object ' 1 ' is not a float.').
doit value: #rtErrCantBecomeOneIdx value: err value:
    #('Become is not allowed because the object ' 1 ' participates in'
      ' an index and the object ' 2 ' has a different format. '
       ) .
doit value: #rtErrCantBecomeClassKind value: err value:
    #('Become is not allowed because the object ' 1 ' is a kind of ' 2 ).
      
doit value: #classErrByteSubclass value: err value:
    #('You may not create a byte subclass of ' 1 '.').
doit value: #repErrCantCreateRepos value: err value:
    #('The Repository ' 1 ' could not be created.').
doit value: #repErrBadFileSpec value: err value:
    #('The file specification is invalid.').
doit value: #repErrFileAlreadyExists value: err value:
    #('The file ' 1 ' already exists.').
doit value: #rtErrDuplicateKey value: err value:
    #('The key ' 2 ' already exists in dictionary ' 1 ).

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.


doit value: #assocErrBadComparison value: err value:
    #('The evaluation of a SelectBlock resulted '
	'in an illegal comparison being performed.').
doit value: #repErrIncompatibleRepos value: err value:
    #('The Repository version does not match the version '
	'of the GemStone executable.').
doit value: #assocErrClassModifiable value: err value:
    #('Index creation failed because the class ' 1 ' is modifiable.'
	' Send "immediateInvariant" to the class to enable index creation. '  ).
doit value: #classErrConstrInher value: err value:
    #('The constraint ' 1 ' is not a subclass of the inherited '
	' constraint ' 2 ' for offset ' 3 '.').
doit value: #rtErrBadDictConstraint value: err value:
    #('Only Association or one of its subclasses can be used as a constraint'
      ' in ' 1 '.').
doit value: #rtErrCantBecomeIndexedNsc value: err value:
    #('Become is not allowed because the object ' 1 ' is a kind of '
      ' Bag and currently has indexes').
doit value: #rtErrNoIndexForPath value: err value:
    #('An index with the path ' 2 ' was not found for the NSC.').
doit value: #rtErrBadSession value: err value:
    #('The specified session does not exist.').
doit value: #rtErrNotOnlyUser value: err value:
    #('An operation that requires exclusive use of the '
	'system was attempted when ' 1 ' users were logged in.').
doit value: #objErrMaxSize value: err value:
    #('An attempt was made to extend an object to size '
	2 ' when the maximum legal size is ' 3 ', object: ').
doit value: #rtErrInvalidMethod value: err value:
    #('Cannot execute method, ' 1 ). 
doit value: #gciErrNoStartCmd value: err value:
    #('A request was made to complete a non-blocking GemBuilder for C call '
	'when no such call was initiated.').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #repErrMaxOnlineRepos value: err value:
    #('Too many repositories are already attached.').
doit value: #rtErrRcQueueEntriesFound value: err value:
    #('In attempting to reset the maximum session ID for the queue to a '
      'smaller value, an entry was found that was added by a session '
      'whose session ID is larger than the value currently being set. '
      'Remove and save all entries in the queue.  Then change the maximum '
      'session ID and add the saved entries back into the queue.').
doit value: #rtErrFirstPathTermSetValued value: err value:
    #('The first term in an NSC index path expression cannot indicate '
      ' a set-valued instance variable (i.e. cannot be an asterisk).').
doit value: #gciErrParentSession value: err value:
    #('The attempted GemBuilder for C operation is illegal in a user action '
      'when applied to a session owned by a parent GemBuilder for C or '
      'GemStone Smalltalk scope.' ) .
doit value: #gciErrAlreadyLoggedIn value: err value:
    #('The current GemBuilder for C (GCI) session is already logged in.').
doit value: #classErrBadIdentifier value: err value:
    #('An illegal identifier ' 1
	' was used to name an instance variable or class.').
doit value: #classErrConstrPrivateIv value: err value:
    #('Illegal attempt to constrain the private instanceVariable ' 1 ).
doit value: #rtErrNoPriv value: err value:
    #('An attempt was made to do a privileged operation '
	'for which no privilege had been granted.').
doit value: #rtErrInvalidBtreeReadStream value: err value:
    #('The BtreeReadStream is invalid (possibly due to modifications '
      'to objects referenced by the stream).').
doit value: #rtErrDecrNotAllowed value: err value:
    #('The RcPositiveCounter with value ' 3 ' cannot be decremented by ' 2 '.').
doit value: #repErrReposNotAttached value: err value:
    #('The Repository could not be attached.').
doit value: #repErrReposNeedsRecovery value: err value:
    #('The Repository needs recovery.').
doit value: #repErrReposRead value: err value:
    #('A disk read error was detected when reading from the Repository extent '
	2 ' on page number ' 1 '.').
doit value: #repErrReposWrite value: err value:
    #('A disk write error was detected when writing to the Repository extent '
	2 ' on page number ' 1 '.' ).

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #rtErrSelectiveAbort value: err value:
    #('The selectiveAbort primitive was attempted on an object (' 1 ') , ' 2 ).
doit value: #objErrDateTimeOutOfRange value: err value:
    #( 'DateTime argument to GciDateTimeToCTime is out of range of time_t, or '
       'the seconds instance variable is not between 0 and 86399 inclusive.').
doit value: #objErrLongNotSmallInt value: err value:
    #('A C long used in a GemBuilder for C (GCI) call is not representable '
      'as a SmallInteger.').
doit value: #objErrNotLong value: err value:
    #('Argument to GciOopToLong or GciOopToInt64 , ' 1 
  ', is not an Integer or is not representable in the C function result type.').
doit value: #objErrNotChr value: err value:
    #('GciOopToChr was passed an object which was not a Character.').
doit value: #lgcErrCompressFailure value: err value:
    #('compress() in the zlib library failed, status ' 1 ).
doit value: #lgcErrUncompressFailure value: err value:
    #('uncompress() in the zlib library failed.').
doit value: #lgcErrTravswizzFailure value: err value:
    #('travBuffSwizzle() in the linkgc module failed.').
doit value: #errArgTooLarge value: err value:
  #( 'The object ' 1 ' is too large, maximum size/value is ' 2  ) .
doit value: #commitPromoteFailed value: err value:
  #( 'Unable to promote commit to a checkpoint, ' 1 ) .
doit value: #objErrDiffSizeColl value: err value:
  #( 'Receiver and argument collections must be the same size' ).
doit value: #rtErrUalibLoadFailed value: err value:
    #('user action library load failed, ' 1 ).
doit value: #lgcErrObjRepHdrSize value: err value:
    #('Mismatch in size of GciObjRepHdrSType between GCI client and server.').

doit value: #authErrSegLoad value: err value:
    #('Unable to load auth cache, ' 1 ', for object ' 2 ).
doit value: #authErrSegRecursion value: err value:
    #('Infinite recursion when attempting load GsObjectSecurityPolicy with objectSecurityPolicyId ' 1 
      ', the GsObjectSecurityPolicy is not in DataCuratorObjectSecurityPolicy.' ).
doit value: #assocErrPathTooLong value: err value:
    #('The path ' 1 ' has more than 1024 characters.').
doit value: #rtErrSemaphore value: err value:
    #('A semaphore operation failed on semaphore with index ' 1 
	'.  Reason: ' 2  '.').
doit value: #repErrFileNameTooBig value: err value:
    #('The filename ' 1 ' has more than ' 2 ' characters.').
doit value: #rtErrPasswordTooBig value: err value:
    #('A password was specified with more than 1024 characters.').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.


doit value: #errNotSameClassHist value: err value:
    #('Migration is not allowed because the classes '
       1 ' and ' 2 ' do not have identical class histories.').

doit value: #classErrMethDictLimit value: err value:
    #('The class ' 1 ' has more than 1500 methods in its method dictionary.').
doit value: #rtErrShrPcDetach value: err value:
    #('Error occurred during SharedPageCache detach: ' 1).
doit value: #repErrCantDispose value: err value:
    #('Unable to dispose of the file ' 1 '.').
doit value: #rtErrInternal value: err value:
    #('Please report to your GemStone Administrator.  Internal Runtime Error:  '
	1 ' with arguments:  ' 2 ' ' 3 ' ' 4 ' ' 5 ' ' 6 ' ' 7 ' ' 8 ' ' 9
        ' ' 10 '.').
doit value: #rtErrBadStreamColl value: err value:
    #('An attempt was made to create an instance of ' 1
	' from the stream ' 2 ', but the collection in the stream is'
	' not a ' 3 '.'). 
doit value: #rtErrBadFormat value: err value:
    #('An attempt was made to create an instance of ' 1 ' from ' 2
	' but either the format is incorrect or the contents of ' 2
	' are incomplete.').
doit value: #segErrBadGroup value: err value:
    #(1 ' is not currently a group.').
doit value: #rtErrShrpcCompatibility value: err value:
    #('The compatibility level of the SharedPageCache Monitor' 1 
      'does not match that of the executable trying to attach' 2).
doit value: #rtErrBadPriv value: err value:
    #(1 ' is not a valid privilege.').
doit value: #rtErrResolveSymFailed value: err value:
    #('GciResolveSymbol failed; the symbol is not defined').
doit value: #rtErrSymAlreadyDefined value: err value:
    #(2 ' is already defined.').
doit value: #rtErrSymNotFound value: err value:
    #(2 ' could not be found in the symbol list for ' 1 '.').
doit value: #rtErrEofOnReadStream value: err value:
    #('End of stream was encountered in ReadStream:  ' 1 '.').
doit value: #assocErrSortOddLengthArray value: err value:
    #('An illegal sorting was specified:  ' 1 '.').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #assocErrBadDirection value: err value:
    #('A sort direction must be ''ascending'' or ''descending'', not ' 1 '.').
doit value: #rtErrConfigReadOnly value: err value:
    #('GemStone Smalltalk access to the configuration parameter ' 1 ' is read-only.').

doit value: #rtErrBadFormatSpec value: err value:
    #(2 ' is an illegal formatting Array for ' 1 '.').
doit value: #hostErrFileExport value: err value:
    #('GemStone cannot export the string ' 1 ' to the file ' 
	2 ' in the server OS.').
doit value: #hostErrFileImport value: err value:
    #('GemStone cannot import the file ' 1 ' from the server OS.' 2 ).
doit value: #hostErrFileDirectory value: err value:
    #('GemStone cannot fetch the server OS directory ' 1 '.').
^ true
%
commit
run
  " more runtime errors"
| err |
err := (Globals at: #GemStoneError) at: #English.

doit value: #hostErrFileDelete value: err value:
    #('GemStone cannot delete the file ' 1 ' from the server OS.').
doit value: #hostErrPerform value: err value:
    #('GemStone cannot execute "' 1 '" on the server OS shell, ' 2 
       ' errno=' 3 ' rawStatus=' 4 ' childStatus=' 5 ).
doit value: #rtErrSigMsgTooBig value: err value:
    #('Attempt to send a string of size ' 2 ' bytes as argument to a signal. '
      'The maximum allowed size is ' 3 ' bytes.').
doit value: #objErrBadClusterBucket value: err value:
    #('The clusterId ' 1 ' is invalid. Max legal clusterId is ' 2 ' .' ).
doit value: #rtErrEpochGcArraySize value: err value:
    #('The number of elements in the Array used to set the Epoch Garbage '
      'Collector information is incorrect.  The size of ' 1 
      ' should be 4.').
doit value: #objErrResultNscTooBig value: err value:
    #('The NSC operation failed; the size of the result would have exceeded '
      ' the maximum size for an NSC.' ).
doit value: #hostErrMemoryAlloc value: err value:
    #('Host memory allocation failed; there is insufficient '
      'swap space and/or RAM.').
doit value: #gciErrCatchBuffNotFound value: err value:
    #('A non-existent catch buffer was specified.').
doit value: #gciErrCatchBuffLimit value: err value:
    #('Catch buffer level is outside the range 0 to 20.').
doit value: #objErrNotBoolean value: err value:
    #('Argument to GciOopToBool() must be either OOP_TRUE or OOP_FALSE.').
doit value: #rtErrUncompiledMethod value: err value: 
    #(' The method with source ' 2 ' in class ' 3 ' is obsolete after '
      ' schema modification and must be recompiled.' ).
doit value: #rtErrMustBeSystemUser value: err value:
    #('An operation was attempted that may only be performed by SystemUser.' ) .
doit value: #rtErrBadPassword value: err value:
    #('The given password is not the password of ' 1 '.').
doit value: #gciErrTravObjNotFound value: err value:
    #('Object not found in the traversal buffer.').
doit value: #gciErrTravCompleted value: err value:
    #('A continuation of a traversal was attempted when there '
	'was no traversal in progress.').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #gciErrTravBuffTooSmall value: err value:
    #('The traversal buffer was smaller than the minimum size '
      'GCI_MIN_TRAV_BUFF_SIZE (which is 2048)' ).
doit value: #rtErrPathToStrIvname value: err value:
    #('Path to String conversion starting with class ' 1 ' failed at term ' 2
	' because there is no named instance variable at offset ' 3
	' in class ' 4 ' that has ' 5 ' named instance variables.').
doit value: #objErrNegativeCount value: err value:
    #('In GciFetchBytes, GciFetchIdxOops, or GciFetchNamedOops, '
	'a negative count of ' 1 ' was specified.').
doit value: #gciErrResultPathTooLarge value: err value:
    #('GciStrToPath or GciPathToStr, result of size ' 1
	' is larger than specified max size of ' 2  '.').
doit value: #gciErrFetchPathFail value: err value:
    #('GciFetchPaths failed on a path; OOP_ILLEGAL substituted in '
        'result Array.').
doit value: #rtErrStrToPathIvname value: err value:
    #('String to path conversion starting with class ' 1 ' failed on path term '
      2 ' because there is no instance variable named ' 3 ' in class ' 4 '.').

doit value: #errAlmostOomPending value: err value:
    #('AlmostOutOfMemory after 2 attempts to signal as a Notification') .

doit value: #lgcErrTravRead value: err value:
    #('Socket error while reading traversal buffer from gci peer').
doit value: #lgcErrPacketKindBad value: err value:
    #('A packet of the wrong kind was received.  Expected: ' 1 
	', Received: ' 2 '.').
doit value: #lgcErrExpectedContinue value: err value:
    #('A packet kind of ' 1 ' was received rather than '
	'the continue packet that was expected.').
doit value: #lgcErrOopPacketTooLong value: err value:
    #('An OOP Array packet was received over the network that was too long.').
^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #lgcErrPacketKindUnknown value: err value:
    #('An unknown packet kind was received: ' 1 '.').
doit value: #hostErrLogFileNotOpened value: err value:
    #('The log file was not opened before adding messages to it.').
doit value: #classErrMaxInstVars value: err value:
    #('An attempt was made to create a subclass of ' 1 
	 ' with more than 2030 instance variables specified in ' 2 '.').
doit value: #rtErrTooManyErrArgs value: err value:
    #('System | signal:args:signalDictionary: was sent with '
	 'an Array argument containing ' 2 ' arguments, but it '
	 'is limited to 10.').
doit value: #objErrBadSize value: err value:
    #('For class ' 1 ' the size ' 2 ' is illegal.').
doit value: #lgcErrInconsistentSize value: err value:
    #('When using GemBuilder for C (GCI), the specified size ' 1
      'is inconsistent with the size ' 2 ' implied by other arguments.').

doit value: #lgcErrInconsistentObjKind value: err value:
    #('Inconsistent implementation received.  Class = ' 1 
	 ', implementation = ' 2 '.').
doit value: #rtErrNscParticipatesInMultipleTerms value: err value:
    #('The NSC ' 1 ' is not allowed to participate in more than one term '
      'in an index path (' 2 ').' ).

doit value: #rtErrStartGc value: err value:
    #('A GcGem is already running, or a session that is attempting to run '
      'reclaim or epoch garbage collection is not the GcGem.').
doit value: #lgcErrSequenceMismatch value: err value:
    #('Sequence number mismatch for an IPC response packet.').
doit value: #rtErrExceptBlockNumArgs value: err value:
    #('The block ' 1 ' which takes ' 3 ' arguments, cannot be used as'
      ' an Exception block. Exception blocks require ' 2 ' arguments.').

doit value: #rtErrGciStoreFloat value: err value:
    #('An illegal GemBuilder for C (GCI) store into the float object ' 1
      ' was detected.  You must use GciStoreBytesInst with the correct class '
      'argument.') .

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.


doit value: #rtErrGciStoreClassMismatch value: err value:
    #('Mismatch between class in store traversal buffer and class in Repository
       for object ' 1 '.  Class in buffer is ' 2 ' , class in Repository is '
      3). 

doit value: #rtErrBadArgKind2 value: err value:
    #('The object ' 1 ' was neither of class ' 2 ' nor ' 3 ).
doit value: #lgcErrSequenceMismatch value: err value:
    #('Sequence number mismatch for an IPC response packet.').
doit value: #rtErrCommitDbInRestore value: err value:
    #('Commits are not allowed while a restore from backups or transaction'
      ' logs is in progress.').

doit value: #rtErrCommitDisallowed value: err value:
    #('Further commits have been disabled for this session because: ' 1
      '. This session must logout.' ).
doit value: #tranLogIoError value: err value:
    #('I/O error when writing the transaction log. This transaction must be aborted.').
doit value: #lgcErrPacketKind value: err value:
    #('Internal error:  Gem to Client packet kind is unknown or unexpected.').
doit value: #rtErrLoadSaveStack value: err value:
    #('Error ' 1 ' , '  2 ' . Execution cannot continue.').
doit value: #rtErrUnknownBytecode value: err value:
    #('Unknown opcode = ' 1 ' encountered during GemStone Smalltalk execution.').
doit value: #errSesBlockedOnOutput value: err value:
    #('Attempt to send signal to session ' 1 ' failed. Buffer from Stone to '
      ' that session is full.').
doit value: #errPrimNotSupported value: err value:
    #('Primitive failure, primitive for receiver= ' 1 ' selector= ' 2 
      '  is not supported in this executable. You must run "' 3 
     '" in order to use this method.') .
doit value: #authErrSegCurrentSeg value: err value:
    #('(OBSOLETE)No authorization write in your current security policy ' 1 '.' 'Resetting '
      'current security policy to default login security policy.' ).
doit value: #authErrSegSetCurrentSeg value: err value:
    #('No authorization to set the current security policy to ' 1 '.').
doit value: #rtErrPrimFailed value: err value:
    #('Primitive failed, receiver ' 1 ' selector ' 2 ' arguments ' 3 ' .').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.


doit value: #gciErrExecClientUserAct value: err value:
    #('Request to invoke client userAction named ' 1 ', invocation failed.').
doit value: #gciErrActiveSessions value: err value:
    #('You may not install a user action after logging in. Action name:  ' 1).
doit value: #gciErrMaxActionArgs value: err value:
    #('User action invoked with too many arguments. '
      ' action name ' 1 ' actual number ' 2 ' maximum number ' 3 ).
doit value: #gciErrBadNumMethodArgs value: err value:
    #('The method ' 1 ' takes ' 2 ' arguments, not ' 3 ' as passed.').
doit value: #gciErrUserActionPending value: err value:
    #('You attempted an illegal GemBuilder for C (GCI) operation during '
        'a user defined action ' 1 '.').
doit value: #gciErrBadNumUaActionArgs value: err value:
     #( 'You invoked a user action with the wrong number of arguments.'
        ' action name ' 1 ' correct number ' 2 ' actual number ' 3 ).
doit value: #rtErrInstvarAddToNsc value: err value:
     #( 'An attempt was made to add a named instance variable to '
	' the class ' 1 ' which has a format of NSC.' ). 
doit value: #rtErrVaryingConstrBytes value: err value:
     #( 'An attempt was made to constrain the indexable portion'
	' of the class ' 1 ' which has a format of Bytes.' ).
doit value: #rtErrVaryingConstrNonidx value: err value:
     #( 'An attempt was made to constrain the indexable portion'
	' of the class ' 1 ' which is not indexable.' ).
doit value: #rtErrInstvarAddToBytes value: err value:
     #( 'An attempt was made to add a named instance variable to'
	 ' the class ' 1 ' which has a format of Bytes.' ). 
doit value: #rtErrClassNotModifiable value: err value:
     #( 'The class ' 1 ' is not modifiable.' ).
doit value: #rtErrAddDupInstvar value: err value:
     #( 'The name of the new instance variable, ' 2 ', would'
	' duplicate the name of an existing instance variable in class ' 1 '.').
doit value: #rtErrNotASubclassOf value: err value:
     #( 'The class ' 1 ' is neither identical to nor a subclass of'
	' the class ' 2 '.').
doit value: #rtErrConstrNotSubclassOf value: err value:
     #( 'In a class modification operation, the new constraint ' 1
	' was neither identical to, nor a subclass of, the inherited '
        'constraint ' 2 '.' ).

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.


doit value: #rtErrConstrNotAClass value: err value:
     #( 'The new constraint ' 1 ' was not a class.').
doit value: #rtErrObjInvariant value: err value:
     #( 'The object ' 1 ' is invariant.' ).
doit value: #classErrDupVarConstr value: err value:
     #( 'Two constraints, ' 1 ' and ' 2 ', were specified upon the '
	'indexable portion of the class. Only one constraint is allowed.' ).

doit value: #bkupErrLoginsEnabled value: err value:
     #( 'A restore operation requires that logins be disabled.'
        ' Run   System suspendLogins    before doing restores.').
doit value: #classErrRemoveInherIv value: err value:
     #( 'In class ' 1 ', an attempt was made to remove the '
	' instance variable ' 2 ' which is inherited from a superclass.' ).
doit value: #concurErrInvalidMode value: err value:
     #( 1 ' is not recognized as a valid Concurrency Control Mode.' ).
doit value: #classErrSelectorLookup value: err value:
     #( 'The message ' 4 ' sent to object ' 1 ' was found in class '
         2 '. It should have been found in class ' 3 ).
doit value: #rtErrBadEUCFormat value: err value:
     #('The bytes of the EUCString with oop ' -1 ' are not in EUC format.').
doit value: #rtErrGaijiNotSupported value: err value:
     #('A Gaiji Character was encountered in the JapaneseString '
         1 ' but Gaiji is not supported.').
doit value: #rtErrInvalidArgClass value: err value:
     #(1 ' is not one of the class kinds in ' 2 '.').
doit value: #rtErrSizeLimit value: err value:
     #('The object ' 1 ' was referenced with a byte index ' 2
                        ' which is out of range.').
doit value: #rtErrNoEUCRep value: err value:
     #(1 ' cannot be represented in EUC format.').
doit value: #rtErrBadEUCValue value: err value:
     #(1 ' is not a valid EUC value and does not correspond to '
                        'a JISCharacter.').
doit value: #rtErrInvalidLang value: err value:
     #('The compiler language environment ' 1 ' is invalid.').
doit value: #rtErrInvalidIndexPathExpression value: err value:
     #('The following string is an invalid term in a path expression: ' 2 ).
doit value: #rtErrDependencyListTooLarge value: err value:
     #('A dependency list ' 1 ' is too large.').
doit value: #rtErrMaxCommitRetry value: err value:
    #('There were too many attempts to commit after a concurrency conflict '
       'or after failing to resolve RC conflicts.  You must abort before '
       'attempting to commit again.').
doit value: #rtErrInvalidArgument value: err value:
    #( 'The object ' 1 ' was an invalid argument to a method, reason: ' 2 ). 

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #rtErrPathNotTraversable value: err value:
    #( 'The class ' 2 ' does not have an instance variable ' 3
       ' on the index path.').
doit value: #rtErrBtreeReadStreamEndOfStream value: err value:
    #( 'An attempt was made to read beyond the end of the stream. ').
doit value: #rtErrObjectPathTermNotInDependencyList value: err value:
    #( 'The object ' 1 ' did not have the path term ' 2 ' in its dependency list. ').
doit value: #rtErrObjectInvalidOffset value: err value:
    #( 'The object ' 1 ' does not have an instance variable with the given name ' 2).
doit value: #rtErrObjectNoDependencyList value: err value:
    #( 'The object ' 1 ' does not have a dependency list. ').
doit value: #rtErrIndexDictionaryEntryNotInDictionary value: err value:
    #( 'An entry for the key/term/value (' 2 '/' 3 '/' 4 ') was not present in the dictionary ' 1 ).
doit value: #rtErrPathTermObjectNotAnNsc value: err value:
    #( 'The object ' 1 ' traversed along an index path through a set-valued '
       'instance variable was not an NSC. ' ).
doit value: #rtErrIdentityIndexCannotInvokeRangeOperation value: err value:
    #( 'An attempt was made to use a range operation ( <, >, =, <=, or >= ) on a path '
       'expression only supported by an identity index.').
doit value: #rtErrRangeEqualityIndexInvalidClassKindForBtree value: err value:
    #( 'Attempt to insert a key of class ' 1 ' for path term ' 2 
       ' into the B-tree that is an invalid class for which the B-tree was created ( ' 3 '). ').
doit value: #rtErrRangeEqualityIndexObjectNotInBtree value: err value:
    #( 'An entry for the key/value pair (' 2 '/' 3 ') was not present in the index.' ).
doit value: #errNoBackupInProgress value: err value:
    #( 'A backup or restore continuation was attempted before 
        executing either restoreFrom: or fullBackupTo:MBytes: . ').
doit value: #bkupErrOpenFailed value: err value:
    #( ' An attempt to open file ' 1 ' for ' 2 ' failed because ' 3 '.').
doit value: #bkupErrMbyteLimitBadRange value: err value:
    #( 'The byte limit specified ' 1 ' is out of the allowable '
        ' range from ' 2 ' to ' 3 ' MBytes.').
doit value: #bkupErrWriteFailed value: err value:
    #( 'An attempt to write to file ' 1 ' failed because ' 2 '.'
       'This error terminates the backup and the backup file is not usable.').
doit value: #bkupErrInProgress value: err value:
    #( 'An attempt was made to start a full backup, '
       'but a backup is currently in progress by another session.').
doit value: #bkupErrReadFailed value: err value:
    #( 'Restore failed, file read error ' 1 ', file name ' 2 ' record ' 3 ).
doit value: #bkupErrNotInProgress value: err value:
    #( 'An attempt to continue a full backup when no backup is in progress.' ).

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #rtErrBagClassDoesNotSupportRangeOperators value: err value:
    #( ' An attempt was made to create an equality index with a class ' 2
       ' that does not support range operators. ' ).
doit value: #rtErrBagOnlySelectBlockAllowed value: err value:
    #( ' Only select blocks are allowed for selectAsStream. ' ).
doit value: #rtErrBagOnlyOnePredicateAllowed value: err value:
    #( ' Only one predicate is allowed for selectAsStream. ' ).
doit value: #rtErrBagNoRangeIndexOnPathExpression value: err value:
    #( ' The path expression in the predicate for selectAsStream does '
       'not have a range equality index. ' ).
doit value: #rtErrBagInvalidPredicateForStreamSelection value: err value:
    #( ' The predicate for selectAsStream was invalid. ' ).
doit value: #rtErrBagOperationNotSupportedForStreamSelection value: err value:
    #( ' The comparison operation in the predicate for selectAsStream is '
       'not supported. ' ).
doit value: #rtErrBagInvalidSortSpecification value: err value:
    #( ' Unable to sort using the sort specification: ' 2 ).
doit value: #rtErrIndexAuthErrSegRead value: err value:
    #('(OBSOLETE) An attempt was made to read the object using index ' 2
        ' in security policy ' 3 ' with insufficient authorization.').
doit value: #objErrTime_tOutOfRange value: err value:
    #( 'Given time_t is out of range; must be greater '
	   'than or equal to zero.').
doit value: #genericError value: err value:
    #( 'User defined error, ' 2 ) .
doit value: #rtErrMethodProtected value: err value:
    #( 'Illegal attempt to execute a protected method. ' ) .
"rtErrPreventingCommit deleted"
doit value: #rtErrRemoveAllIndexesFailed value: err value:
    #( '  An attempt to remove all indexes on ' 1 ' has failed. Index participation info: ' 2 '.').
doit value: #rtErrCollectionWithIncompleteIndex value: err value:
    #( '  An attempt was made to create an index on a collection that has incomplete indexes.'
       ' The incomplete indexes must be removed before creating new indexes.' ).
doit value: #rtErrCantBecomeSelfOnStack value: err value:
    #( 'Object ' 1 ' is present on the GemStone Smalltalk stack as "self", and '
       'cannot participate in become: or changeClassTo:' ).
doit value: #rtErrObjectProtected value: err value:
    #( 'Illegal attempt to fetch/store into a protected object. ' ) .
doit value: #rtErrNewTranlogDirFail value: err value:
    #( 'Attempt to define new transaction log directory failed, reason: ' 1 ).
doit value: #rtErrNoMoreObjectSecurityPolicies value: err value:
    #( 'No more security policies can be created. The object SystemRepository has reached maximum allowed size.' ).
doit value: #rtErrNotInExportSet value: err value:
    #( 'During GciStoreTravDoTravRefs, OOP ' 1 '  was not found in ExportSet' ).
doit value: #rtErrGciTravNotLicensed value: err value:
    #( 'License does not allow Gci Traversal operations.').
doit value: #rtErrExceptionAlreadyInstalled value: err value:
    #( 'ensure block already installed in the current frame'  ).

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #errCommitWhileBackupInProgress value: err value:
    #( 'A commit or abort was attempted while a multi-file full backup is '
       'in progress.  This may have been caused by mistakenly repeating the '
       'fullBackupTo:MBytes: method instead of using the '
       'continueFullBackupTo:MBytes: method.  '
       'To cancel the backup use the abortFullBackup method. '
       'To continue the backup use the continueFullBackupTo:MBytes: method.').
doit value: #errUnconvertedObject value: err value:
    #( 'Incomplete conversion from previous version of GemStone, object: '
       1 ' reason: ' 2 ).
doit value: #rtErrCommitProhibitingError value: err value:
    #(1 '  This error occurred during object manager recursion to GemStone '
      'Smalltalk.  This transaction must be aborted.').
doit value: #rtErrAttemptToPassivateInvalidObject value: err value:
    #( 'Attempted to write ' 1 ' to a passive object.  '
       'nil was written instead.').
doit value: #rtErrTimeToRestoreToArg value: err value:
    #( 'timeToRestoreTo: failed, reason:  ' 1 ).
doit value: #lockErrAllSymbols value: err value: 
    #('Users are not allowed to lock AllSymbols').
doit value: #gciErrSymbolFwdRef value: err value: 
    #('User attempted to create a forward reference, OOP = ' 1 
      ' to a Symbol or fill in the state of an existing forward reference'
      ' as a Symbol').
doit value: #rtErrChangeSymbol value: err value: 
    #('Changing the class of an object ' 1 
       ' from or to class Symbol is disallowed').
doit value: #rtErrObjVariant value: err value:
     #( 'The object ' 1 ' is not invariant.' ).
doit value: #rtErrAlreadyHasSubclasses value: err value:
     #( 'In disallowSubclasses, ' 1 ' already has subclasses.' ).
doit value: #clientForwarderSend value: err value:
     #( 'Message to forward to client, forwarder: ' 1 ' receiver: ' 2 ' 
          selector: ' 3 ' args: ' 4 ).
doit value: #rtErrBadSize value: err value:
     #( 'Invalid object size, required size ' 2 ' actual size ' 3 
        ' for object ' 1) .
doit value: #rtErrFreeSpaceThreshold value: err value:
     #( 'The Repository is currently running below the freeSpaceThreshold.').
doit value: #rtErrTranlogDirFull value: err value:
     #( 'All transaction log directories are full. The Stone process is '
        'waiting for an operator to make more space available by either '
        'cleaning up the existing files (copying them to archive media and '
        'deleting them) or by adding a new transaction log directory.').
^ true
%
commit
run
  " more runtime errors"
| err |
err := (Globals at: #GemStoneError) at: #English.
doit value: #rtMaxPasswordSize value: err value:
     #( 'The maximum size of a password is ' 2 ' characters.  The string '
        1 ' is too large.' )  .
doit value: #rtMinPasswordSize value: err value:
     #( 'The minimum size of a password is ' 2 ' characters.  The string '
        1 ' is too small.' )  .

doit value: #rtMaxRepeatingChars value: err value:
  #( 'A password may not have more than ' 2 ' repetitions of a Character. '
     ' The substring ' 3 ' is invalid in the password ' 1 ).

doit value: #rtMaxConsecutiveChars value: err value:
  #( 'A password may not have more than ' 2 ' consecutive characters. '
     'The substring ' 3 ' is invalid in the password ' 1 ).

doit value: #rtMaxCharsOfSameType value: err value:
  #( 'A password may not have more than ' 2 ' consecutive ' 3 ' characters. '
     'The substring ' 4 ' is invalid in the password ' 1 ).

doit value: #rtDisallowedPassword value: err value:
  #( 'A password may not have the value ' 1 ) .

doit value: #rtErrWarningPasswordExpire value: err value:
  #( 'The password of the current UserProfile is about to expire.' ).

doit value: #errLogDirNotExist value: err value:
  #( 'Directory or raw device not found, ' 1 ) .

doit value: #errArgTooSmall value: err value:
  #( 'The object ' 1 ' is too small, minimum size/value is ' 2  ) .

doit value: #errNoStructuralUpdate value: err value:
  #( 'GemBuilder for C (GCI) update operation ' 2
     ' is not supported on object ' 1 ) .

doit value: #rtObsoleteClass value: err value:
  #( 'New instances of this obsolete class are not allowed.' ).

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.


doit value: #rtErrLocalSessionFailedCommit value: err value:
  #( 'Local session failed to commit after remote sessions voted to commit.').

doit value: #rtErrRemoteSessionFailedCommit value: err value:
  #('Remote session ' 2 ' failed to commit after voting affirmative.').

doit value: #rtErrNoElemDetected value: err value:
  #('The object ' 2 ' was not detected in ' 1 '.').

doit value: #rtErrDateTimeOutOfRange value: err value:
  #('The resulting DateTime object would be out of range.').

doit value: #rtErrObjNotFound value: err value:
  #('The object ' 2 ' was not found in ' 1 '.').

doit value: #rtErrFailedStnCfgWrite value: err value:
  #( 'The operation should update the Stone configuration file, '
     'but the write failed.  See Stone log for more details.' ).

doit value: #gciErrNoUserAction value: err value:
  #( 'Attempt to call a user action that is not registered with this '
     'virtual machine, user action name: ' 1 ).

doit value: #segErrMaxUserProfileGroups value: err value:
    #('An attempt was made to add the UserProfile ' 1 ' to the group ' 2
	   ' when the UserProfile was already a member of four groups.').

doit value: #rtErrGsCommitListNBTimeout value: err value:
    #('During synchronized commits, session ' 2 ' timed out.').

doit value: #rtErrGcGemStartupFailures value: err value:
    #('Warning: Stone has logged GcGem startup failures since last success').

doit value: #rtErrGsSocket value: err value:
    #('A GsSocket method failed due to a socket error.').

doit value: #rtErrGsSocketInvariant value: err value:
    #('The GsSocket object ' 1 ' that was not changable was sent a '
      'message that would have required it to change.').

doit value: #rtErrLostSessionState value: err value:
    #('The object ' 1 ' that has transient session state has lost that '
      'transient state.') .

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.


doit value: #rtErrDelayInProgress value: err value:
    #('The interval delay object ' 1 ' already has a process waiting on it.').

doit value: #rtErrSchedulerDeadlocked value: err value:
    #('The process scheduler object ' 1 ' is deadlocked.').

doit value: #authErrProcessSwitch  value: err value:
    #('Processor scheduler cannot switch processes while a primitive
      is within a bypass-authorization block' ).

doit value: #rtErrPrimitiveSysErr value: err value:
    #('A primitive failed with the following system error: ' 1).

doit value: #rtErrGsProcessTerminated value: err value:
    #('A main GsProcess was sent the terminate message.').

doit value: #rtErrNoMoreCBuffers value: err value:
    #('All available CBuffers are in use.').

doit value: #rtErrRemoteError value: err value:
"    #('A GemBuilder for C error occurred on the remote host: ' 2). "
    #('Remote Error: ' 2).

doit value: #rtErrRemoteLoginFailed value: err value:
    #('Remote logins to this repository are disabled by the keyfile.').

doit value: #rtErrClientFwd value: err value:
    #('The client encountered an uncaught exception, client error number ' 1
      'when a message was forwarded.  The client error message is: ' 2
      'The stack trace from the client is: ' 3).
doit value: #rtErrBadGsProcessArgCount value: err value:
    #('An attempt was made to create a GsProcess for the block ' 1 ' with ' 3
	   ' arguments when ' 2 ' were expected.').
doit value: #rtErrPasswordNotVariant value: err value:
    #('The password for session parameters must be variant.').
doit value: #rtErrNotEnoughCBuffers value: err value:
    #('There are not any more C buffers available to allocate a new one.').

doit value: #rtErrGsProcessNotDebuggable value: err value:
    #('The gsprocess ' 1 ' is not in a debuggable state.').

doit value: #gciErrArgNotPairs value: err value:
    #('The argument must be an Array with pairs of objects and booleans, of maximum size 20.').

doit value: #rtErrClientActionDisallowed value: err value:
    #('An attempt was made to execute the client user action ' 1
       ' from a session that is not allowed to do this.').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.


doit value: #rtErrRemoveSpecialUser value: err value:
    #('An attempt was made to remove the userProfile with userId ' 1
       ' from AllUsers.').

doit value: #errLostOtHandlingFailed value: err value:
    #('Fatal error during lostOt handling, sharedOt cache may be invalid.').

^ true
%

commit
run
| err |
err := (Globals at: #GemStoneError) at: #English.

"
! create selected aborting error strings 
"

doit value: #rtErrAbortTrans value: err value:
    #('The transaction was aborted by the user.').
"abortErrInternal deleted"

doit value: #abortErrCommitsSuspended value: err value:
   #('Commits are suspended for failover to standby system, your transaction has been aborted.').

doit value: #abortErrGarbageCollection value: err value:
  #('Garbage collection aborted, reason: '
	1  ', conflict code: ' 2 
	'.  Garbage collection was aborted and should be tried again later.').

doit value: #bkupErrRestoreLogSuccess value: err value:
    #( 'Restore from transaction log succeeded.' ).
doit value: #bkupErrRestoreLogFail value: err value:
    #( 'Restore from transaction log failed. Reason: ' 1 ).

doit value: #bkupErrRestoreSuccessful value: err value:
    #( 'Restore from full backup completed with ' 1 ' objects restored and '
       2 ' corrupt objects not restored.' 3 ).

"bkupErrCorruptFile deleted"
"bkupErrWrongFile deleted"

doit value: #gsErrShrpcLostOtTimeout value: err value: 
    #( 'LostOt timeout during shared cache access' ).

"abortErrFinishedMark deleted"

doit value: #abortErrLostOtRoot value: err value:
    #('When running outside of a transaction the session failed to respond '
      'quickly enough to a request from Stone to abort.  All object caches '
      'have been cleared. ' 1 ).

doit value: #errAutoAbort value: err value: 
    #('A sigAbort was serviced automatically by idle Gem.').

doit value: #errAutoLostOt value: err value: 
    #('A LostOt was serviced automatically by idle Gem.' ) .

^ true
%
commit
! messages for fatal errors
run
| err |

err := (Globals at: #GemStoneError) at: #English.

doit value: #gsErrBadRootPage value: err value:
    #('The Repository root page is corrupted or contains a disk media defect.').
doit value: #repErrReposFull value: err value:
    #('The Repository is full and can no longer be expanded.').
doit value: #repErrNoFreePages value: err value:
    #('No free pages were found after growing the Repository ' 1 '.').
doit value: #hostFatalErrOutOfMem value: err value:
    #('Host memory allocation failed. Insufficient '
      'swap space and/or RAM to allocate: ' 1 ).
doit value: #gsErrCorruptObjSize value: err value:
   #('The object ' 1 ' is corrupted and has size ' 2 '. The correct size
      is ' 3 '.' ). 
doit value: #repErrBadBkupSwizzle value: err value:
   #('An invalid swizzle transform was detected in the backup file'
     ' long transform: ' 1 ' short transform: ' 2 ) .
doit value: #repErrBadDbfSwizzle value: err value:
   #('An invalid swizzle transform was detected in an extent'
     ' long transform: ' 1 ' short transform: ' 2 ) .
doit value: #gciErrActionDefined value: err value:
    #('Attempt to install user failed, there is already a user action '
      'installed with the name: ' 1 ).
doit value: #errUserProfileLost value: err value:
    #('The UserProfile with object ID ' 1 ' has been garbage collected '
      'as a result of a successful restore or by a commit of another session').
doit value: #gsErrShrpcConnectionFailure value: err value:
    #('The Gem''s connection to the local shared cache monitor was lost. '
      'Error text: ' 1).
doit value: #gsErrShrpcUnexpectedNetEvent value: err value:
    #('There was an unexpected read or write event ' 1 
      ' from the shared cache monitor.').
doit value: #gsErrShrpcInvalidConfig value: err value:
    #('The process tried to login to a Stone with the configuration file '
      'specifying that the SharedPageCache should NOT be used when '
      'the Stone already has a SharedPageCache active on the host machine').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #repErrBadBkupVersion value: err value:
   #('The backup file is incompatible with this version of GemStone.'
     ' Backup file version is: ' 1 ).

doit value: #rtErrCantSuspendLogins value: err value:
   #('Request to Suspend logins denied. ' 1 ).

doit value: #objErrCorruptObj value: err value:
    #('The object with object ID ' 1 ' is corrupt. Reason: ' 2).

doit value: #gsErrStnNetProtocol value: err value:
    #('A fatal network protocol error occurred on the Gem to Stone '
      'network, failure code = ' 1 ' .').
doit value: #errTranLogOpenFail value: err value:
    #('GemStone is unable to open the next transaction log for writing.').

doit value: #errFinishedObjAuditRepair value: err value:
    #('Repair operation completed: ' 1 'repairs were made.').

doit value: #gsErrStnNetLost value: err value:
    #('The GemStone session has lost its connection to the Stone repository '
      'monitor.').

doit value: #gsErrDisconnectInLogin value: err value:
    #('The session lost its connection to the Stone repository monitor during '
      'login. Logins are disabled or the maximum number of users are already '
      'logged in.').
doit value: #gsErrMaxSessionsLimit value: err value:
    #('Login failed: the maximum number of users is already logged in.').
doit value: #lgcErrIncompatGci value: err value:
    #('The version of GemBuilder for C (GCI) is not compatible with the '
      'version of the Gem').
doit value: #hostErrCantSpawn value: err value:
    #('Unable to create a GemStone session.').
doit value: #gsActiveUserLimitExceeded value: err value:
  #( 'Login failed: Too many sessions already logged in with this userId.').
doit value: #gsErrLoginDenial value: err value:
    #('Login failed:  the GemStone userId/password combination is invalid '
      'or expired.').

doit value: #gsErrLoginsDisabled value: err value:
    #('Login failed:  GemStone logins are currently disabled.').

doit value: #gsErrCacheTooBig value: err value:
    #('Login failed:  GemStone could not start a remote shared page cache because the '
      'requested cache size exceeds the license limit.').

doit value: #gsErrStnShutdown value: err value:
    #('The Stone repository monitor has shut down.').
doit value: #gsErrSessionShutdown value: err value:
    #('Your GemStone session has been forcibly terminated, probably '
      'by DataCurator, ').
doit value: #gsErrGemNormalShutdown value: err value:
    #('The GemStone session is terminating normally.').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #gsErrGemFatalShutdown value: err value:
    #('The GemStone session is terminating abnormally.').
doit value: #gsErrNoMoreOops value: err value:
    #('GemStone system ran out of OOPs.').
doit value: #netErrNoSuchStn value: err value:
    #('The given Stone repository monitor cannot be reached:' 1).
doit value: #gciErrBadSessionId value: err value:
    #('The given GemStone session ID is invalid.').
doit value: #gciErrUnexpectedLnkErr value: err value:
    #('Fatal unexpected error in Linkable GemBuilder for C (GCI) session while '
      'GemBuilder for C call in progress on another RPC session').
doit value: #gciErrInternalRpc value: err value:
    #('A logic error was detected in the implementation of an RPC GemBuilder for C call. '
      'Please report to your GemStone Administrator.').
doit value: #fatalErrInternal value: err value:
    #('Please report to your GemStone Administrator. Fatal Internal Error: ' 1 
	' with arguments:  ' 2 ' ' 3 ' ' 4 ' ' 5 ' ' 6 ' ' 7 ' ' 8 ' ' 9
	' ' 10 '.').
doit value: #netErrConnectionRefused value: err value:
    #('The connection to the Stone repository monitor was refused.').
doit value: #netErr value: err value: #('Network error, text follows:' 1).
doit value: #repErrSystemRepNotAttached value: err value:
    #('The Repository is not attached.').
doit value: #authErrSegLoginSeg value: err value:
    #('You have no authorization to write in your default login security policy, ' 1 ).

doit value: #rtErrPause value: err value:
    #('Execution has been suspended by a "pause" or "halt" message.').
"rtErrStep deleted"
doit value: #rtErrStep value: err value:
    #('Single-step breakpoint encountered.').
doit value: #rtErrSoftBreak value: err value:
    #('A soft break was received. ' ).
doit value: #rtErrHardBreak value: err value:
    #('A hard break was received.').
doit value: #rtErrCodeBreakpoint value: err value:
    #('Method breakpoint encountered.').
doit value: #rtErrStackBreakpoint value: err value:
    #('Stack breakpoint encountered on return from method or block.').
doit value: #rtErrCommitAbortPending value: err value:
    #('A transaction commit or abort is pending.').

^ true
%
commit
run
| err |
" create more error strings "

err := (Globals at: #GemStoneError) at: #English.

doit value: #rtErrSignalCommit value: err value:
    #('This error indicates that a member of the notify set has been '
      'committed and was added to the signaledObjects set.').
doit value: #rtErrSignalAbort value: err value:
    #('This error indicates that when running outside of a '
      'transaction Stone signaled the Gem to request an abort').

doit value: #rtErrSignalGemStoneSession value: err value:
    #('The signal ' 2 ' was received from sessionSerialNumber = ' 1
      ' and the message string associated with the signal is ' 3).

doit value: #rtErrUncontinuable value: err value:
    #('Execution cannot be continued, ' 1 ). 

doit value: #rtErrSignalFinishTransaction value: err value:
    #('Stone requests commit, abort or continueTransaction of the current '
      'transaction.' ) .

doit value: #rtErrSignalAlmostOutOfMemory value: err value:
    #('Session''s temporary object memory is almost full' ) .

doit value: #rtErrProcessStepInNative value: err value:
    #('Breakpoint and single-step not supported in native code' ) .

doit value: #rtErrTimeoutInterrupt value: err value: #('Timeout interrupt' ) .
doit value: #rtErrSignalException value: err value: #('SignalException interrupt' ) .
doit value: #rtErrGcFinalize value: err value: #('GcFinalize interrupt for ephemerons' ) .
doit value: #rtErrTerminateProcess value: err value: #('TerminateProcess interrupt' ) .
doit value: #rtErrSignalTrap value: err value: #('Signal ' 3 ' trapped (by Ruby)').

doit value: #gsErrCopyProtection value: err value:
    #('Copy protection check failed, invalid keyfile.').
doit value: #gsErrNoCapability value: err value:
    #('Attempt to use a feature not licensed by your keyfile.').
doit value: #gsErrBadGemStoneVersion value: err value:
    #('The Gem version ' 1 ' and the Stone version ' 2 ' are incompatible.').
doit value: #gsErrBadGemDbfVersion value: err value:
    #('The Gem version ' 1 ' and the Repository version ' 2 
      ' are incompatible.').
doit value: #rtErrCommitDisallowedUntilAbort value: err value:
    #('Commits are disabled until after the next abort because: ' 1
      '  The session must abort.').
doit value: #rtErrRollbackDlFail value: err value:
    #('Rollback of a specific dirty list not allowed after commit failure,
       session must abort entire transaction.').

doit value: #rtErrExceptionAlreadyInstalled value: err value:
    #('ExecBlock>>ensure: exception already installed in frame, for block ' 1).

doit value: #rtErrNativeCodeGen value: err value:
    #('Native code error, ' 1 ).

doit value: #rtErrNativeCodeNoBreakpts value: err value:
    #('Breakpoints and single-step not supported in native code' ).

doit value: #rtErrDynIvDependencyTag value: err value:
    #('Cannot add or modify a dynamic instance variable in an object with a dependency tag, object: ' 1 ).

doit value: #rtErrDynIvStore  value: err value:
    #('cannot store because ' 2 '; receiver: ' 1  ) .

doit value: #rtErrComExpressionDepth value: err value:
    #('expression nesting too deep in method source, increase GEM_TEMPOBJ_SCOPES_SIZE in gem config file.').

doit value: #ansiNotification value: err value:
    #('Notification signalled, ' ).

doit value: #rtErrReposUpgradeNeeded value: err value:
    #( 'Repository upgrade needed' ) .

doit value: #rtErrCannotConvertStack value: err value:
    #('Cannot convert native stack to portable stack, ' 1 ) .

doit value: #rtErrRcUpdateDisallowed value: err value: 
    #('An attempt was made to update an RC object in a continued transaction. '
      'The transaction must be aborted.').

doit value: #rtErrGetGcLockFailed value: err value:
    #('Request for gcLock timed out. ' 1 ).

doit value: #rtErrCantBecomeGeneric value: err value: 
    #('become is not allowed on ' 1 ', ' 2 ).

doit value: #rtErrOpNotAllowed value: err value: 
    #('The current operation is not allowed at this time, reason: ' 1 ).

doit value: #rtErrInvalidAuthSchemeId value: err value:
    #('Invalid authentication scheme id: ' 1 ).

doit value: #rtErrInvalidAuthSchemeSymbol value: err value:
    #('Invalid authentication scheme symbol: ' 1 ).

doit value: #rtErrInvalidUnixUserId value: err value:
    #('The specified UNIX user ID is invalid: ' 1 ).

doit value: #rtErrIllegalAuthSchemeChange value: err value:
    #('Illegal attempt to modify authentication scheme of a special UserProfile: ' 1 ).

doit value: #rtErrAuthIllegalOperation value: err value:
    #('Attempt to perform an illegal operation under the current authentication scheme. '
      'UserProfile: ' 1   '  AuthScheme: ' 2 ).

doit value: #rtErrAuthIllegalPasswordChange value: err value:
    #('Attempt to change the password of a UserProfile that is not using GemStone authentication. '
      'UserProfile: ' 1 ).

doit value: #rtErrIllegalPasswordExpiration value: err value:
    #('An attempt was made enable password expiration for a special UserProfile. ' 1 ).

doit value: #rtErrIllegalDn value: err value:
    #('The string is not a valid distinguished name or does not contain the %s pattern. ' 1 ).

doit value: #errSslLibLoadFailed value: err value:
    #('GemStone was unable to load the OpenSSL shared libraries.  Reason: ' 1 ).

doit value: #rtErrGroupAlreadyExists value: err value:
    #('An attempt was made to create a UserProfileGroup with a name that already exists.  Name: ' 1 ).

doit value: #rtErrPrincipalAlreadyExists value: err value:
    #('An attempt was made to create a Kerberos principal with a name that already exists.  Name: ' 1 ).

doit value: #rtErrGsSecureSocket value: err value:
    #('A GsSecureSocket method failed due to an error.').

doit value: #gsErrSslShutdown value: err value:
    #('A network error occurred while switching from SSL to non-SSL communication.').

doit value: #errLostOtGci value: err value:
    #('A LostOtRoot signal was received while waiting for a GCI command to complete.').

doit value: #errDuringLostOtService value: err value:
    #('An error occurred while servicing a LostOtRoot signal.').
^ true
%


commit

! Remove "doit"
run
Globals removeKey: #doit .
System currentObjectSecurityPolicy: Object objectSecurityPolicy .
^ true
%

input $upgradeDir/legacyErrNum.gs

run
| set bag map numToReason templates defaultTemplate |
set := IdentitySet new .
bag := IdentityBag new .
ErrorSymbols valuesDo:[ :v |
  set add: v . bag add: v 
].
( bag - set ) size == 0 ifFalse:[ nil error:'duplicate values'] .
numToReason := IdentityKeyValueDictionary new .
ErrorSymbols keysAndValuesDo:[:k :v |
  "convert Error symbols to reason Strings "
  numToReason at: v put: k asString immediateInvariant 
].
templates := (Globals at: #GemStoneError) at: #English.
map := Globals at: #LegacyErrNumMap .
defaultTemplate := #( 'Unknown error number' ) .
1 to: map size do:[:j | | vAry tStr val |
  (val := map at: j ) ifNotNil:[
    tStr := (templates atOrNil: j) ifNil:[ 
      (j >= 2700 and:[ j <= 2799 ]) ifTrue:[ nil ] ifFalse:[ defaultTemplate ].
    ].
    val _isSymbol ifTrue:[
	val == #none ifTrue:[ 
	  vAry := { InternalError . #rtErrNoMessage . defaultTemplate }
	] ifFalse:[ nil error:'unknown symbol' ].
			  ] ifFalse: [
    val _isArray ifTrue:[
       vAry := { val at: 1 . numToReason at: j ifAbsent:[nil] . tStr . val }
			  ] ifFalse:[
    val isBehavior ifTrue:[    
       vAry := { val"a Class" . numToReason at: j ifAbsent:[ nil ] . tStr } .
    ] ifFalse:[
       nil error:'bad value'
    ]]] .
    "vAry is { aClass . aReasonString . legacyTemplateStringOrNil }
	or { firstClass . aReasonString . legacyTemplateStringOrNil. 
		  { list of split classes...} } "
    map at: j put: vAry immediateInvariant
  ] ifNil:[
    (j < 2000 and: [j > 1001]) ifTrue:[ 
       (templates at: j) ifNotNil:[ :compileMsg |  map at: j put: compileMsg ].
    ]
  ].
].
"map immediateInvariant .  future optimization"
Globals at:#LegacyErrNumMap put: map .
"(Globals associationAt:#LegacyErrNumMap) immediateInvariant.  future optimization"
true
%
