!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: Canonicalization.gs ? $
!
!=========================================================================

! ------------------- Class definition for AbstractReferencingObjectPolicy
doit
Object subclass: 'AbstractReferencingObjectPolicy'
  instVarNames: #( manager)
  classVars: #()
  classInstVars: #()
  poolDictionaries: #()
  inDictionary: Globals
  options: #().
true
%
doit
AbstractReferencingObjectPolicy comment: 
'AbstractReferencingObjectPolicy is part of the Canonical Object 
Framework.  This class is the abstract superclass for policy classes 
that are used to do the referencing object cleanup (i.e., replacing the 
duplicate object with the canonical object). Subclasses should implement 
#''canonicalizeReferencesIn:''. See ReferencingObjectPolicy for the
default implementation.'.
true
%
doit
AbstractReferencingObjectPolicy category: 'Canonicalization'.
true
%
! ------------------- Class definition for ReferencingObjectPolicy
doit
AbstractReferencingObjectPolicy subclass: 'ReferencingObjectPolicy'
  instVarNames: #( referencingClassHistories)
  classVars: #()
  classInstVars: #()
  poolDictionaries: #()
  inDictionary: Globals
  options: #().
true
%

doit
ReferencingObjectPolicy comment: 
'ReferencingObjectPolicy is part of the Canonical Object Framework.  
This class is the default policy used to do the referencing object cleanup.
In order to support manager cleanup, referencing (or container) classes must be 
specified to this policy using #''addRefencingClass:''.'.
true
%
doit
ReferencingObjectPolicy category: 'Canonicalization'.
true
%
! ------------------- Class definition for CanonicalObjectManager
doit
Object subclass: 'CanonicalObjectManager'
  instVarNames: #( referencingObjectPolicy registries sessionCacheStatOffset)
  classVars: #()
  classInstVars: #( default securityPolicy)
  poolDictionaries: #()
  inDictionary: Globals
  options: #().
true
%

doit
CanonicalObjectManager comment: 
'The Canonical Object Framework provides an optional way to reduce
duplicate instances of equivalent objects in your application. To accomplish
this goal you register a class as being canonicalizable (#''createRegistryFor:'')
and then you can request a canonical object (#''canonicalObjectFor:''). 

To avoid commit conflicts, if a canonical object is not yet in the registry, the 
framework allows overlapping transactions to see different objects as the 
canonical one, but subsequent transactions will see one of the earlier ones.
To ensure that only one object is used, send #''ultimateCanonicalObjectFor:''
and be prepared to handle commit conflicts.

An alternate way to canonicalize objects is to provide a set of objects to the
manager and have it scan them for references to canonicalizable objects
(#''cleanupAll:''). This would typically be used with objects found using a
repository scan (Repository>>#''listReferencesToInstancesOfClasses:toDirectory:'').

Because some objects should not have references arbitrarily replaced (e.g., 
IdentitySets), the framework uses a pluggable policy to decide which
referencing objects can be modified. The default ReferencingObjectPolicy 
bases the decision on the referencing object class.

A special case of object cleanup is #''cleanupWriteSet'' that scans the write
set for the current transaction and replaces references as indicated by the
policy. Thus, a properly configured manager could do appropriate cleanup 
before a commit.

By default, the default instance of CanonicalObjectManager has world write 
permission so that canonical objects can be created by any session and shared.
You can modify this using CanonicalObjectManager class>>#securityPolicy:'.
true
%
doit
CanonicalObjectManager category: 'Canonicalization'.
true
%
! ------------------- Class definition for CanonicalObjectPolicy
doit
Object subclass: 'CanonicalObjectPolicy'
  instVarNames: #()
  classVars: #()
  classInstVars: #()
  poolDictionaries: #()
  inDictionary: Globals
  options: #().
true
%

doit
CanonicalObjectPolicy comment: 
'CanonicalObjectPolicy is internal to the Canonical Object Framework.
It may be that in the future we will allow pluggable policies for some
canonicalization behavior.'.
true
%
doit
CanonicalObjectPolicy category: 'Canonicalization'.
true
%
! ------------------- Class definition for CanonicalObjectRegistry
doit
Object subclass: 'CanonicalObjectRegistry'
  instVarNames: #( canonicalObjectPolicy manager potentialCanonicalObjects
                    registry)
  classVars: #()
  classInstVars: #()
  poolDictionaries: #()
  inDictionary: Globals
  options: #().
true
%

doit
CanonicalObjectRegistry comment: 
'CanonicalObjectRegistry is internal to the Canonical Object Framework.
There may be one registry for each ClassHistory and it holds a collection 
(a KeyValueDictionary) of the ultimate canonical objects and a collection 
of potentialCanonicalObjects (an RcIdentityBag). If an equivalent object 
is not found in the registry, then the framework scans the potentials for
an equivalent object (which could have been added by any session). If
an equivalent object is found it is returned; otherwise the new object is
added to the potentials and returned.

At some time when conflicts are considered unlikely, one can send 
#''canonicalizePotentialCanonicalObjects'' to the CanonicalObjectManager 
and the potentials will be added to the registry (improving subsequent
lookup performance).'.
true
%
doit
CanonicalObjectRegistry category: 'Canonicalization'.
true
%

! ------------------- Remove existing behavior from AbstractReferencingObjectPolicy
doit
AbstractReferencingObjectPolicy removeAllMethods.
AbstractReferencingObjectPolicy class removeAllMethods.
true
%
! ------------------- Class methods for AbstractReferencingObjectPolicy
set compile_env: 0
category: 'Instance Creation'
classMethod: AbstractReferencingObjectPolicy
new

	self error: 'Use #newForManager:'.
%
set compile_env: 0
category: 'Instance Creation'
classMethod: AbstractReferencingObjectPolicy
newForManager: aManager

	^self basicNew
		initialize: aManager;
		yourself.
%
! ------------------- Instance methods for AbstractReferencingObjectPolicy
set compile_env: 0
category: 'Initialization'
method: AbstractReferencingObjectPolicy
initialize: aManager

	manager := aManager.
	self objectSecurityPolicy: aManager objectSecurityPolicy.
%
set compile_env: 0
category: 'Subclass Responsibilities'
method: AbstractReferencingObjectPolicy
canonicalizeReferencesIn: aReferencingObject

	self subclassResponsibility: #'canonicalizeReferencesIn:'.
%

! ------------------- Remove existing behavior from ReferencingObjectPolicy
doit
ReferencingObjectPolicy removeAllMethods.
ReferencingObjectPolicy class removeAllMethods.
true
%
! ------------------- Class methods for ReferencingObjectPolicy
! ------------------- Instance methods for ReferencingObjectPolicy
set compile_env: 0
category: 'Accessors'
method: ReferencingObjectPolicy
referencingClasses

	^referencingClassHistories
		inject: IdentitySet new
		into: [:sum :each | sum addAll: each. sum].
%
set compile_env: 0
category: 'Cleanup'
method: ReferencingObjectPolicy
canonicalizeReferencesIn: aReferencingObject

	| counter |
	(referencingClassHistories includes: aReferencingObject class classHistory) ifFalse: [^self].
	manager sessionCacheStatAt: 5 incrementBy: 1.		"count of referencing objects scanned during cleanup"
	aReferencingObject isInvariant ifTrue: [
		manager sessionCacheStatAt: 8 incrementBy: 1.	"count of invariant referencing objects scanned during cleanup"
		^self
	].
	counter := aReferencingObject _canonicalizeReferencesUsingPolicy: self.
	counter == 0 ifTrue: [^self].
	manager
		sessionCacheStatAt: 6 incrementBy: 1;				"count of referencing objects changed during cleanup"
		sessionCacheStatAt: 7 incrementBy: counter;		"count of references changed during cleanup"
		yourself.
%
set compile_env: 0
category: 'Cleanup'
method: ReferencingObjectPolicy
canonicalizeReferencesInObject: anObject

	| counter |
	counter := 0.
	1 to: anObject _primitiveSize do: [:i |
		| each |
		each := anObject _primitiveAtNoFault: i otherwise: nil.
		each isSpecial ifFalse: [
			| canonical |
			(canonical := manager canonicalObjectFor: each) ~~ each ifTrue: [
				anObject _primitiveAt: i put: canonical.
				counter := counter + 1.
			].
		].
	].
	^counter
%
set compile_env: 0
category: 'Cleanup'
method: ReferencingObjectPolicy
canonicalizeReferencesInCollection: aCollection

	| counter |
	counter := 0.
	aCollection copy do: [:each | 
		each isSpecial ifFalse: [
			| canonical |
			(canonical := manager canonicalObjectFor: each) ~~ each ifTrue: [
				aCollection removeIdentical: each; add: canonical.
				counter := counter + 1.
			].
		].
	].
	^counter
%
set compile_env: 0
category: 'Cleanup'
method: ReferencingObjectPolicy
canonicalizeReferencesInDictionary: aDictionary

	| counter |
	counter := 0.
	aDictionary copy keysAndValuesDo: [:key :value | 
		| newKey |
		newKey := key.
		key isSpecial ifFalse: [
			| canonical |
			(canonical := manager canonicalObjectFor: key) ~~ key ifTrue: [
				aDictionary removeKey: key; at: canonical put: value.
				newKey := canonical.
				counter := counter + 1.
			].
		].
		value isSpecial ifFalse: [
			| canonical |
			(canonical := manager canonicalObjectFor: value) ~~ value ifTrue: [
				aDictionary removeKey: newKey; at: newKey put: canonical.
				counter := counter + 1.
			].
		].
	].
	^counter
%
set compile_env: 0
category: 'Initialization'
method: ReferencingObjectPolicy
initialize: aManager

	referencingClassHistories := IdentitySet new.
	super initialize: aManager.		"This will set the objectSecurityPolicy"
%
set compile_env: 0
category: 'Public'
method: ReferencingObjectPolicy
addReferencingClass: aClass

	aClass implementationFormat even	"OOP or NSC"
		ifTrue: [referencingClassHistories add: aClass classHistory]
		ifFalse: [self error: aClass name , ' not supported as a referencing class for Canonicalization'].
%
category: 'Public'
method: ReferencingObjectPolicy
addReferencingClasses: aCollection

	aCollection do: [:each |
		referencingClassHistories add: each classHistory.
	].
%
set compile_env: 0
category: 'Security'
method: ReferencingObjectPolicy
objectSecurityPolicy: anObjectSecurityPolicy

	super objectSecurityPolicy: anObjectSecurityPolicy.
	referencingClassHistories objectSecurityPolicy: anObjectSecurityPolicy.
%

! ------------------- Remove existing behavior from CanonicalObjectManager
doit
CanonicalObjectManager removeAllMethods.
CanonicalObjectManager class removeAllMethods.
true
%
! ------------------- Class methods for CanonicalObjectManager
set compile_env: 0
category: 'Instance Creation'
classmethod: CanonicalObjectManager
clearDefault
"
	CanonicalObjectManager clearDefault.
"
	default := nil.
%
category: 'Instance Creation'
classmethod: CanonicalObjectManager
default

	default ifNil: [
		default := Array with: self new.
		default first objectSecurityPolicy: securityPolicy.
	].
	^default first.
%
category: 'Instance Creation'
classmethod: CanonicalObjectManager
new

	^super new
		initialize;
		yourself.
%
category: 'Security'
classmethod: CanonicalObjectManager
securityPolicy

	^securityPolicy
%
category: 'Security'
classmethod: CanonicalObjectManager
securityPolicy: anObjectSecurityPolicy

	securityPolicy := anObjectSecurityPolicy.
	default ifNotNil: [self default objectSecurityPolicy: anObjectSecurityPolicy].
%
! ------------------- Instance methods for CanonicalObjectManager
set compile_env: 0
category: 'Accessors'
method: CanonicalObjectManager
 canonicalClasses

	^registries keys 
		inject: IdentitySet new
		into: [:sum :each | sum addAll: each. sum].
%
category: 'Accessors'
method: CanonicalObjectManager
referencingObjectPolicy

	^referencingObjectPolicy
%
category: 'Accessors'
method: CanonicalObjectManager
_registries

	^registries.
%
set compile_env: 0
category: 'Canonical'
method: CanonicalObjectManager
canonicalObjectFor: anObject
	"returns a possible canonical object"

	| domain |
	domain := self
		_registryFor: anObject
		ifAbsent: [^anObject].
	^domain canonicalObjectFor: anObject.
%
category: 'Canonical'
method: CanonicalObjectManager
hasCanonicalObjectFor: anObject

	| domain |
	domain := self
		_registryFor: anObject
		ifAbsent: [^false].
	^domain hasCanonicalObjectFor: anObject.
%
category: 'Canonical'
method: CanonicalObjectManager
hasUltimateCanonicalObjectFor: anObject

	| domain |
	domain := self
		_registryFor: anObject
		ifAbsent: [^false].
	^domain hasUltimateCanonicalObjectFor: anObject.
%
category: 'Canonical'
method: CanonicalObjectManager
isCanonicalObject: anObject
	"Could answer true for two equivalent objects if both are in potentialCanonicalObjects"

	| domain |
	domain := self
		_registryFor: anObject
		ifAbsent: [^false].
	^domain isCanonicalObject: anObject.
%
category: 'Canonical'
method: CanonicalObjectManager
isUltimateCanonicalObject: anObject

	| domain |
	domain := self
		_registryFor: anObject
		ifAbsent: [^false].
	^domain isUltimateCanonicalObject: anObject.
%
category: 'Canonical'
method: CanonicalObjectManager
ultimateCanonicalObjectFor: anObject
	"returns an actual canonical object (with possible conflicts)"

	| domain |
	domain := self
		_registryFor: anObject
		ifAbsent: [^anObject].
	^domain ultimateCanonicalObjectFor: anObject.
%
category: 'Canonical'
method: CanonicalObjectManager
_registryFor: anObject ifAbsent: aBlock

	^registries
		at: anObject class classHistory
		ifAbsent: aBlock.
%
set compile_env: 0
category: 'Cleanup'
method: CanonicalObjectManager
cleanupAll: aCollection

	| beginTime endTime |
	beginTime := System timeNs.
	aCollection do: [:each | 
		referencingObjectPolicy canonicalizeReferencesIn: each.
	].
	endTime := System timeNs.
	self 
		sessionCacheStatAt: 3 		incrementBy: 1;  						"count of calls to #'cleanupAll:'"
		sessionCacheStatAt: 4 		incrementBy: aCollection size;  	"count of objects scanned during cleanup"
		sessionCacheStatAt: 10	incrementBy: ((endTime - beginTime) // 1000 max: 1);	"microseconds in #'cleanupAll:'"
		yourself.
%
category: 'Cleanup'
method: CanonicalObjectManager
cleanupWriteSet

	| beginTime writeSet endTime |
	beginTime := System timeNs.
	writeSet := self _writeSet.
	endTime := System timeNs.
	self 
		sessionCacheStatAt: 9 incrementBy: ((endTime - beginTime) // 1000 max: 1);	"microseconds in building writeSet"
		cleanupAll: writeSet;
		yourself.
%
category: 'Cleanup'
method: CanonicalObjectManager
_writeSet

	| writeSet queue |
	queue := OrderedCollection withAll: System _writtenObjects.
	writeSet := IdentitySet new.
	[
		queue notEmpty.
	] whileTrue: [
		| next |
		next := queue removeLast.
		next class isBytes ifFalse: [
			writeSet add: next.
			1 to: next _primitiveSize do: [:i | 
				| x |
				x := next _primitiveAtNoFault: i otherwise: nil.
				(x isSpecial or: [x class isBytes or: [x isCommitted or: [writeSet includes: x]]]) not ifTrue: [
					queue add: x.
				].
			].
		].
	].
	^writeSet.
%
category: 'Cleanup'
method: CanonicalObjectManager
_writeSetAdding: newObjects ignoring: oldObjects

	| writeSet queue |
	queue := OrderedCollection withAll: System _writtenObjects.
	writeSet := IdentitySet new.
	[
		queue notEmpty.
	] whileTrue: [
		| next |
		next := queue removeLast.
		next class isBytes ifFalse: [
			writeSet add: next.
			1 to: next _primitiveSize do: [:i | 
				| x |
				x := next _primitiveAt: i.
				(x isSpecial or: [x class isBytes or: [x isCommitted or: [writeSet includes: x]]]) not ifTrue: [
					queue add: x.
				].
			].
		].
	].
	^writeSet.
%
set compile_env: 0
category: 'Initialize'
method: CanonicalObjectManager
initialize

	registries :=  IdentityKeyValueDictionary new.
	referencingObjectPolicy := ReferencingObjectPolicy newForManager: self.
%
set compile_env: 0
category: 'Registry Management'
method: CanonicalObjectManager
canonicalizePotentialCanonicalObjects

	registries do: [:each | each canonicalizePotentialCanonicalObjects].
%
category: 'Registry Management'
method: CanonicalObjectManager
createRegistryFor: aClass

	registries
		at: aClass classHistory
		ifAbsentPut: [CanonicalObjectRegistry newForManager: self].
%
category: 'Registry Management'
method: CanonicalObjectManager
isCanonicalClass: aClass

	^registries includesKey: aClass classHistory.
%
category: 'Security'
method: CanonicalObjectManager
objectSecurityPolicy: anObjectSecurityPolicy

	super objectSecurityPolicy: anObjectSecurityPolicy.
	referencingObjectPolicy objectSecurityPolicy: anObjectSecurityPolicy.
	registries objectSecurityPolicy: anObjectSecurityPolicy.
%
set compile_env: 0
category: 'Statistics'
method: CanonicalObjectManager
resetSessionCacheStats

	| endOffset |
	sessionCacheStatOffset ifNil: [^self].
	endOffset := sessionCacheStatOffset + self sessionCacheStats size - 1.
	sessionCacheStatOffset to: endOffset do: [:i | 
		System sessionCacheStatAt: i put: 0.
	].
%
category: 'Statistics'
method: CanonicalObjectManager
sessionCacheStatAt: anIndex

	sessionCacheStatOffset ifNil: [^0].
	^System 
		sessionCacheStatAt: sessionCacheStatOffset + anIndex - 1.
%
category: 'Statistics'
method: CanonicalObjectManager
sessionCacheStatAt: anIndex incrementBy: aSmallInt

	sessionCacheStatOffset ifNil: [^self].
	System 
		sessionCacheStatAt: sessionCacheStatOffset + anIndex - 1
		incrementBy: aSmallInt.
%
category: 'Statistics'
method: CanonicalObjectManager
sessionCacheStatAt: anIndex put: aSmallInt

	sessionCacheStatOffset ifNil: [^self].
	System 
		sessionCacheStatAt: sessionCacheStatOffset + anIndex - 1
		put: aSmallInt.
%
category: 'Statistics'
method: CanonicalObjectManager
sessionCacheStatsSize

	^self sessionCacheStatDescriptions size
%
category: 'Statistics'
method: CanonicalObjectManager
sessionCacheStatDescriptions

	^#(
	"1"	'count of objects added to global registry'
	"2"	'count of objects added to potentialCanonicalObjects'

	"3"	'count of calls to #''cleanupAll:'''
	"4"    'count of objects scanned during cleanup'
	"5"    'count of referencing objects scanned during cleanup'
	"6"    'count of referencing objects changed during cleanup'
	"7"    'count of references changed during cleanup'
	"8"	'count of invariant referencing objects scanned during cleanup'

	"9"    'microseconds in building writeSet'
	"10"	'microseconds in #''cleanupAll:'''
	)
%
category: 'Statistics'
method: CanonicalObjectManager
sessionCacheStatOffset
	"nil means do not collect any statistics; aSmallInteger means start statistics here"

	^sessionCacheStatOffset
%
category: 'Statistics'
method: CanonicalObjectManager
sessionCacheStatOffset: anIntegerOrNil
	"nil means do not collect any statistics; aSmallInteger means start statistics here"

	sessionCacheStatOffset := anIntegerOrNil.
%
category: 'Statistics'
method: CanonicalObjectManager
sessionCacheStats
"
	CanonicalObjectManager soleInstance sessionCacheStats.
"
	^self sessionCacheStatsForSession: System session.
%
category: 'Statistics'
method: CanonicalObjectManager
sessionCacheStatsForSession: anInteger

	sessionCacheStatOffset ifNil: [^#()].
	^(System sessionCacheStatsForSessionId: anInteger)
		copyFrom: sessionCacheStatOffset + 1
		to: sessionCacheStatOffset + self sessionCacheStatDescriptions size.
%

! ------------------- Remove existing behavior from CanonicalObjectPolicy
doit
CanonicalObjectPolicy removeAllMethods.
CanonicalObjectPolicy class removeAllMethods.
true
%
! ------------------- Class methods for CanonicalObjectPolicy
! ------------------- Instance methods for CanonicalObjectPolicy
set compile_env: 0
category: 'other'
method: CanonicalObjectPolicy
canonicalRepresentationOf: anObject

	^anObject immediateInvariant.
%
category: 'other'
method: CanonicalObjectPolicy
wantsToCanonicalize: anObject

	^anObject size < 256.
%

! ------------------- Remove existing behavior from CanonicalObjectRegistry
doit
CanonicalObjectRegistry removeAllMethods.
CanonicalObjectRegistry class removeAllMethods.
true
%
! ------------------- Class methods for CanonicalObjectRegistry
set compile_env: 0
category: 'Instance Creation'
classmethod: CanonicalObjectRegistry
new

	self shouldNotImplement: #'new'.
%
category: 'Instance Creation'
classmethod: CanonicalObjectRegistry
newForManager: aManager

	^super new
		initializeForManager: aManager;
		yourself.
%
! ------------------- Instance methods for CanonicalObjectRegistry
set compile_env: 0
category: 'Accessors'
method: CanonicalObjectRegistry
_canonicalObjectPolicy

	^canonicalObjectPolicy.
%
category: 'Accessors'
method: CanonicalObjectRegistry
_potentialCanonicalObjects

	^potentialCanonicalObjects.
%
category: 'Accessors'
method: CanonicalObjectRegistry
_registry

	^registry.
%
set compile_env: 0
category: 'Initialize'
method: CanonicalObjectRegistry
initializeForManager: aManager

	canonicalObjectPolicy := CanonicalObjectPolicy new.
	manager := aManager.
	potentialCanonicalObjects := RcIdentityBag new.
	registry := KeyValueDictionary new.
	self objectSecurityPolicy: aManager objectSecurityPolicy.
%
set compile_env: 0
category: 'Lookup'
method: CanonicalObjectRegistry
canonicalObjectFor: anObject
	"returns an actual or potential canonical object (without 
	conflicts, but might need to canonicalize potentials later). 
	Does not yet deal with local registry."

	| canonical |
	canonical := self ultimateCanonicalObjectOrNilFor: anObject.
	canonical ifNotNil: [^canonical].
	potentialCanonicalObjects do: [:each | anObject = each ifTrue: [^each]].
	(canonicalObjectPolicy wantsToCanonicalize: anObject) ifFalse: [^anObject].
	canonical := canonicalObjectPolicy canonicalRepresentationOf: anObject.
	potentialCanonicalObjects add: canonical.
	manager sessionCacheStatAt: 2 incrementBy: 1.
	^canonical.
%
category: 'Lookup'
method: CanonicalObjectRegistry
hasCanonicalObjectFor: anObject

	| canonical |
	canonical := self ultimateCanonicalObjectOrNilFor: anObject.
	canonical ifNotNil: [^true].
	^potentialCanonicalObjects includesValue: anObject.
%
category: 'Lookup'
method: CanonicalObjectRegistry
hasUltimateCanonicalObjectFor: anObject

	"^(registry at: anObject otherwise: nil) notNil."
	^registry includesKey: anObject
%
category: 'Lookup'
method: CanonicalObjectRegistry
isCanonicalObject: anObject
	"Could answer true for two equivalent objects if both are in potentialCanonicalObjects"

	| canonical |
	canonical := self ultimateCanonicalObjectOrNilFor: anObject.
	canonical == anObject ifTrue: [^true].
	^potentialCanonicalObjects includesIdentical: anObject.
%
category: 'Lookup'
method: CanonicalObjectRegistry
isUltimateCanonicalObject: anObject

	^(registry at: anObject otherwise: nil) == anObject.
%
category: 'Lookup'
method: CanonicalObjectRegistry
ultimateCanonicalObjectFor: anObject
	"returns an actual canonical object (with possible conflicts)"

	^registry
		at: anObject
		ifAbsentPut: [
			manager sessionCacheStatAt: 1 incrementBy: 1.
			canonicalObjectPolicy canonicalRepresentationOf: anObject].
%
category: 'Lookup'
method: CanonicalObjectRegistry
ultimateCanonicalObjectOrNilFor: anObject

	^registry
		at: anObject
		ifAbsent: [nil].
%
set compile_env: 0
category: 'Registry Management'
method: CanonicalObjectRegistry
canonicalizePotentialCanonicalObjects

	| list |
	list := potentialCanonicalObjects asArray.
	potentialCanonicalObjects removeAll: list.
	list do: [:each | 
		self ultimateCanonicalObjectFor: each.
	].
%
set compile_env: 0
category: 'Security'
method: CanonicalObjectRegistry
objectSecurityPolicy: anObjectSecurityPolicy

	super objectSecurityPolicy: anObjectSecurityPolicy.
	canonicalObjectPolicy objectSecurityPolicy: anObjectSecurityPolicy.
	potentialCanonicalObjects objectSecurityPolicy: anObjectSecurityPolicy.
	registry objectSecurityPolicy: anObjectSecurityPolicy.
%
run
CanonicalObjectManager default ~~ nil.
%
category: 'Object Canonicalization'
method: Object
_canonicalizeReferencesUsingPolicy: aReferencingObjectPolicy

	^aReferencingObjectPolicy canonicalizeReferencesInObject: self
%
category: 'Object Canonicalization'
method: Collection
_canonicalizeReferencesUsingPolicy: aReferencingObjectPolicy

	self error: 'Unable to canonicalize references in ' , self printString.
%
category: 'Object Canonicalization'
method: AbstractDictionary
_canonicalizeReferencesUsingPolicy: aReferencingObjectPolicy

	^aReferencingObjectPolicy canonicalizeReferencesInDictionary: self
%
category: 'Object Canonicalization'
method: Array
_canonicalizeReferencesUsingPolicy: aReferencingObjectPolicy

	^aReferencingObjectPolicy canonicalizeReferencesInObject: self
%
category: 'Object Canonicalization'
method: OrderedCollection
_canonicalizeReferencesUsingPolicy: aReferencingObjectPolicy

	^aReferencingObjectPolicy canonicalizeReferencesInObject: self
%
category: 'Object Canonicalization'
method: SortedCollection
_canonicalizeReferencesUsingPolicy: aReferencingObjectPolicy

	^aReferencingObjectPolicy canonicalizeReferencesInCollection: self
%
category: 'Object Canonicalization'
method: UnorderedCollection
_canonicalizeReferencesUsingPolicy: aReferencingObjectPolicy

	^aReferencingObjectPolicy canonicalizeReferencesInCollection: self
%
