!=========================================================================
! Copyright (C) GemTalk Systems 1991-2020.  All Rights Reserved
!
! file AppendStream.gs 
!=========================================================================

expectvalue /String
run
(Globals includesKey: #AppendStream) ifFalse:[
  Stream _newKernelSubclass: 'AppendStream'
        instVarNames:#( collection )
        classVars: #()
        classInstVars: #()
        poolDictionaries: #()
        inDictionary: Globals
        options: #() reservedOop: nil .
  ^ 'created'
].
^ 'exists'
%
expectvalue /String
run
(AppendStream classVarAt: #CrLf otherwise: nil) ifNil:[ | arr |
  arr := { 
    #CrLf .  (String new add:(Character codePoint: 13);
                 add: (Character codePoint: 10); immediateInvariant
                ; yourself ) .
    #CrTab .  (String new add:(Character codePoint: 13);
                 add: (Character codePoint: 9); immediateInvariant
                ; yourself ) 
  }.
  1 to: arr size by: 2 do:[:j |
    AppendStream _addInvariantClassVar: (arr at: j) value: (arr at: j + 1)
  ]. 
  ^ 'added'
].
^ 'no change'
%

set class AppendStream
removeallmethods 
removeallclassmethods 

category: 'For Documentation Installation only'
classmethod:
installDocumentation
self comment:
'AppendStream is a replacement for WriteStream, for streams that are are 
created empty,  appended to , and then the contents are retrieved.
There is no stream-specific support for reading, write limit, nor 
position logic.  Because position and write limit are not
supported the nextPut methods are faster than in WriteStream.'
%

category: 'Instance Creation'
classmethod:
on: aCollection
 ^ self basicNew on: aCollection.
%

category: 'Private'
method:
on: aCollection
  collection := aCollection .
%

category: 'Accessing'
method:
contents
  "Returns the contents without making a copy, for efficency, 
   Sets the contents instVar to nil, so subsequent adds to this 
   stream cannot alter the returned contents.  To continue using
   this stream after sending #contents, send #reset ."  
  | res |
  res := collection .
  collection := nil . 
  ^ res
%

category: 'Adding'
method:
nextPut: aCharacter
  ^ collection add: aCharacter .
%
method:
nextPutAll: aString
  ^ collection addAll: aString
%
method:
nextPutCodePoint: aSmallInteger

  ^ collection addCodePoint: aSmallInteger
%
method:
nextPutAllBytes: aCharacterCollection

"Adds the byte contents of aCharacterCollection to the receiver ,
 using big-endian byte ordering of aCharacterCollection . 
 Returns aCharacterCollection.  
 
 The aCharacterCollection argument must be a kind of String or
 MultiByteString."

"Used in implementation of methods to support PassiveObject."

  ^ collection addAllBytes: aCharacterCollection
%
method
print: anObject
  anObject printOn: self
%


category: 'Encoding'
method: 
nextPutAllUtf8: aCharacterOrString
 "Appends the UTF8 encoding of the argument to receiver. 

  The receiver's  collection must be a String or Utf8.  
  This method will signal an MessageNotUnderstood  (#addAllUtf8: not understood)
  if collection has been promoted to a MultiByteString as a side effect
  of appending code points above 255. "

 ^ collection addAllUtf8: aCharacterOrString
%


category: 'Character writing'
method:
cr
"Append a return character to the receiver."
  collection addCodePoint: 13 .
%
method:
crlf
"Append a carriage return character followed by a line feed character to the receiver."

  collection addAll: CrLf .
%
method:
crtab
  "Append a return character, followed by a single tab character, to the
  receiver."

  collection addAll: CrTab 
%
method:
crtab: anInteger
  "Append a return character, followed by anInteger tab characters, to the
  receiver."

  collection addCodePoint: 13 .
  anInteger timesRepeat: [ collection addCodePoint: 9 ]
%
method:
lf
  "Append a line feed character to the receiver."
  collection addCodePoint: 10 

%
method:
space
  "Append a space character to the receiver."
  collection addCodePoint: 32 .
%
method:
tab
  "Append a tab character to the receiver."
  collection addCodePoint: 9  .
%
method:
space: anInteger
  "Append anInteger space characters to the receiver."

  anInteger timesRepeat: [  collection addCodePoint: 32 ]
%
method:
tab: anInteger
"Append anInteger tab characters to the receiver."

anInteger timesRepeat: [ collection addCodePoint: 9  ]
%

category: 'Positioning'
method: 
reset
  collection ifNotNil:[ :col | collection := col class new ]
                ifNil:[ collection := String new ]
%
method:
size
  ^ collection size
%
method:
isEmpty
  ^ collection size == 0
%

! Test optimized character literals
lev 1
expectvalue true
run
  | strm ary exp |
  strm := AppendStream on: String new .
  strm cr ; crlf; crtab ; crtab: 2 ; lf ; space ; tab ; space: 2 ; tab: 2 .
  ary := { } .
  strm contents do:[:c | ary add: c  codePoint ] .
  ary = #( 13 13 10 13 9 13 9 9 10 32 9 32 32 9 9 ) ifFalse:[ ^ ary ].
  true
%
lev 0

