!=========================================================================
! Copyright (C) VMware, Inc. 1986-2011.  All Rights Reserved.
!
! $Id: softcollisionbucket.gs,v 1.6 2008-01-09 22:50:18 stever Exp $
!
! Superclass Hierarchy:
! Object
!   Collection
!     SequenceableCollection
!       Array
!         AbstractCollisionBucket( numElements)
!           CollisionBucket( keyValueDictionary)
!             SoftCollisionBucket(lastCleanupCount) 
!
!=========================================================================

expectvalue %String
run
| res cls oldCls clsName |
clsName := #SoftCollisionBucket .
oldCls := Globals at:clsName otherwise: nil .
oldCls == nil ifTrue:[
  res := CollisionBucket _newKernelSubclass: clsName"SoftCollisionBucket"
    instVarNames: #( #lastCleanupCount )
    classVars: #()
    classInstVars: #()
    poolDictionaries: #[]
    inDictionary: Globals
    constraints: #[ #[ 'lastCleanupCount', SmallInteger ]]
    instancesInvariant: false
    isModifiable: true
    reservedOop: 945 .
  cls := Globals at: clsName .
  cls makeInstancesNonPersistent .
  cls _disallowGciCreateStore .
  cls immediateInvariant .
] ifFalse: [
  res := clsName asString + 'already exists'.
  oldCls instancesNonPersistent ifFalse:[ nil error:'should be NP' ].
].
^ res
%

! Remove existing behavior 
removeallmethods SoftCollisionBucket
removeallclassmethods 

category: 'For Documentation Installation only'
classmethod
installDocumentation

| doc txt |
doc := GsClassDocumentation newForClass: self.

txt := (GsDocText new) details:
'A SoftCollisionBucket is a CollisionBucket intended to
 store key/SoftReference pairs .
 key/SoftReference pairs whose SoftReferences have been cleared by 
 the in-memory garbage collector will be removed from a SoftCollisionBucket 
 during the next at:put: operation following a garbage collection which cleared
 the references.  
 
 Cleared SoftReferences will also be removed when their key matches
 the argument key for a lookup operation .  The lookup code will
 not remove cleared SoftReferences for non-matching keys.

 Instances of SoftCollisionBucket are non-persistent and may
 not be committed to the repository.' .
doc documentClassWith: txt.

txt := (GsDocText new) details:
'Private to the implementation of SoftCollisionBucket, used to manage
 removal of SoftReferences.'.
doc documentInstVar: #lastCleanupCount with: txt.

self description: doc.
%


category: 'Instance creation'
classmethod
new

^ super new _initializeCleanup
%

category: 'Private'
method
_initializeCleanup

  lastCleanupCount := 0 
%


category: 'Private'
method
_referenceAt: idx 

" for the key stored at offset idx ,
  return the SofReference if softRef.value non-nil, otherwise
  remove the soft reference and associated key .
" 
| aSoftRef val | 
aSoftRef := self _at: idx + 1 .
aSoftRef ~~ nil ifTrue:[ 
  val := aSoftRef _value .
  val == nil ifTrue:[ 
    self _at: idx     put: nil . "remove key"
    self _at: idx + 1 put: nil . "remove SoftRef"
    numElements := numElements - 1 .
    numElements <= 0 ifTrue:[ 
      numElements < 0 ifTrue:[ nil error:'underflow' ].
      keyValueDictionary _removedElements: 1 collisions: 0
    ] ifFalse:[
      keyValueDictionary _removedElements: 1 collisions: 1
    ].
    ^ nil .
  ]. 
  ^ aSoftRef
].
^ nil
%

category: 'Accessing'
method
referenceAt: aKey ifAbsent: aBlock

"Returns the non-cleared SoftReference that corresponds to aKey.  
 If no such key/SoftRef pair exists,
 returns the result of evaluating the zero-argument block aBlock."

  | keyIndex index aSoftRef |
  keyIndex := self searchForKey: aKey.
  keyIndex ~~ nil ifTrue: [ 
    index := (keyIndex * 2) - 1 . 
    aSoftRef:=  self _referenceAt: index .
    aSoftRef == nil ifTrue:[ ^ self _reportKeyNotFound: aKey with: aBlock ] .
    ^ aSoftRef 
  ].
  ^ self _reportKeyNotFound: aKey with: aBlock 
%
 
category: 'Accessing'
method
referenceAt: aKey otherwise: aValue

"Returns the non-cleared SoftReference that corresponds to aKey.
 If no such key/SoftRef pair exists, returns aValue ."

  | index keyIndex aSoftRef |
  keyIndex := self searchForKey: aKey.
  keyIndex ~~ nil ifTrue:[
    index := (keyIndex * 2) - 1 .
    aSoftRef :=  self _referenceAt: index .
    aSoftRef == nil ifTrue:[ ^ aValue ].
    ^ aSoftRef 
  ].
  ^ aValue   
%

! inherited from AbstractCollisionBucket,  at: anIndex putKey: aKey
! inherited from AbstractCollisionBucket,  at: anIndex putValue: aValue

category: 'Updating'
method
at: aKey put: aValue  

"disallowed, should use  at:put:keyValDict_coll:  "

self shouldNotImplement: #at:put:
%

category: 'Updating'
method
at: aKey put: aValue keyValDict_coll: aKeyValDict

"Stores the aKey/aValue pair in the receiver.  
 Returns 1 if this at:put: added a new key, 0 if this at:put:
 replaced a SoftReference for an existing  key .

 aValue is expected to be a SoftReference.

 Also removes key/SoftReference pairs whose SoftReference has been
 cleared by the garbage collector."

| startElem |
self cleanupReferences .
startElem := numElements .
super at: aKey put: aValue keyValDict_coll: aKeyValDict .
startElem < numElements ifTrue:[ ^ 1] ifFalse:[ ^ 0 ].
%

category: 'Removing'
method
removeKey: aKey ifAbsent: aBlock

"Removes the key/value pair having the key aKey.  If aKey is not found,
 returns the result of evaluating the zero-argument block aBlock.

 Also removes key/SoftReference pairs whose SoftReference has been
 cleared by the garbage collector."

self cleanupReferences .
^ super removeKey: aKey ifAbsent: aBlock 
%

! added for 36675
category: 'Removing'
method
removeKey: aKey otherwise: notFoundValue

"Removes the key/value pair having the key aKey.  If aKey is not found,
 returns the notFoundValue .

 Also removes key/SoftReference pairs whose SoftReference has been
 cleared by the garbage collector."

self cleanupReferences .
^ super removeKey: aKey otherwise: notFoundValue
%

category: 'Private'
method
_markSweepsThatClearedSoftRefsCount

"Returns OM.markSweepsThatClearedSoftRefsCount as a positive SmallInteger"

<primitive: 553>

self _primitiveFailed: #_markSweepsThatClearedSoftRefsCount
%


category: 'Cleanup'
method
cleanupReferences

"remove key/SoftReference pairs whose SoftReferences have
 been cleared by the garbage collector."
| currCount |
currCount := self _markSweepsThatClearedSoftRefsCount .
currCount = lastCleanupCount ifFalse:[ | startNumElem |
  startNumElem := numElements .
  1 to: self tableSize do: [:keyIdx | | aRef idx aKey |
    idx := (keyIdx * 2) - 1 . 
    aKey := self _at: idx .
    aKey ~~ nil ifTrue:[
      aRef := self _at: idx + 1 .
      aRef _value == nil ifTrue:[
        self _at: idx     put: nil . "remove key"
        self _at: idx + 1 put: nil . "remove SoftRef"
        numElements := numElements - 1 .
        numElements < 0 ifTrue:[ nil error:'underflow1' ].
      ]. 
    ].
  ]. 
  lastCleanupCount := currCount .
  numElements == startNumElem ifFalse:[
    keyValueDictionary _bucketSizeChangeFrom:startNumElem to: numElements .
  ].
].
%
