!=========================================================================
! Copyright (C) VMware, Inc. 1986-2011.  All Rights Reserved.
!
! $Id: jischar.gs,v 1.5 2008-01-09 22:50:12 stever Exp $
!
! Superclass Hierarchy:
!   JISCharacter, AbstractCharacter, Magnitude, Object.
!
!=========================================================================

removeallmethods JISCharacter
removeallclassmethods JISCharacter

category: 'For Documentation Installation only'
classmethod: JISCharacter
installDocumentation

| doc txt |
doc := GsClassDocumentation newForClass: self.

txt := (GsDocText new) details:
'The Japanese Industrial Standards organization (JIS) has defined a standard
 Japanese character set, containing codes for thousands of Characters.  Some
 Characters have both a one-byte and a two-byte representation.  These
 Characters include Roman characters, digits, the space Character, Katakana,
 and some punctuation and special Characters.  Both the one-byte and two-byte
 representations of the same Character can be freely mixed in Japanese text.

 There are 65535 instances of JISCharacter.  You may not create new instances
 of JISCharacter.  All instances of a given JISCharacter are both equal (=)
 and identical (==).'.
doc documentClassWith: txt.

self description: doc.
%

category: 'Printable Characters'
classmethod: JISCharacter
lowercaseGreek

"Returns an Array containing all lower-case Greek JISCharacters in alphabetic
 order."

| result |

result:= Array new.
16#2641 to: 16#2658
        do: [:i | result add: (JISCharacter withValue: i) ].
^ result
%

category: 'Printable Characters'
classmethod: JISCharacter
lowercaseRussian

"Returns an Array containing all lower-case Russian JISCharacters in alphabetic
 order."

| result |

result:= Array new.
16#2751 to: 16#2771
        do: [:i | result add: (JISCharacter withValue: i) ].
^ result
%

category: 'Printable Characters'
classmethod: JISCharacter
oneByteDigits

"Returns an Array containing one-byte JISCharacters representing the
 digits 0 through 9."

| result |

result:= Array new.
16#30 to: 16#39
      do: [:i | result add: (JISCharacter withValue: i) ].
^ result
%

category: 'Printable Characters'
classmethod: JISCharacter
oneByteLowercaseRoman

"Returns an Array containing all one-byte lower-case Roman JISCharacters
 in alphabetic order."

| result |

result:= Array new.
16#61 to: 16#7A
      do: [:i | result add: (JISCharacter withValue: i) ].
223 to: 246
      do: [:i | result add: (JISCharacter withValue: i) ].
248 to: 254
      do: [:i | result add: (JISCharacter withValue: i) ].
^ result
%

category: 'Printable Characters'
classmethod: JISCharacter
oneByteUppercaseRoman

"Returns an Array containing all one-byte upper-case Roman JISCharacters
 in alphabetic order."

| result |

result:= Array new.
16#41 to: 16#5A
      do: [:i | result add: (JISCharacter withValue: i) ].
192 to: 214
      do: [:i | result add: (JISCharacter withValue: i) ].
216 to: 222
      do: [:i | result add: (JISCharacter withValue: i) ].
^ result
%

category: 'Printable Characters'
classmethod: JISCharacter
twoByteDigits

"Returns an Array containing two-byte JISCharacters representing the
 digits 0 through 9."

| result |

result:= Array new.
16#2330 to: 16#2339
        do: [:i | result add: (JISCharacter withValue: i) ].
^ result
%

category: 'Printable Characters'
classmethod: JISCharacter
twoByteLowercaseRoman

"Returns an Array containing all two-byte lower-case Roman JISCharacters
 in alphabetic order."

| result |

result:= Array new.
16#2361 to: 16#237A
        do: [:i | result add: (JISCharacter withValue: i) ].
^ result
%

category: 'Printable Characters'
classmethod: JISCharacter
twoByteUppercaseRoman

"Returns an Array containing all two-byte upper-case Roman JISCharacters
 in alphabetic order."

| result |

result:= Array new.
16#2341 to: 16#235A
        do: [:i | result add: (JISCharacter withValue: i) ].
^ result
%

category: 'Printable Characters'
classmethod: JISCharacter
uppercaseGreek

"Returns an Array containing all upper-case Greek JISCharacters in
 alphabetic order."

| result |

result:= Array new.
16#2621 to: 16#2638
        do: [:i | result add: (JISCharacter withValue: i) ].
^ result
%

category: 'Printable Characters'
classmethod: JISCharacter
uppercaseRussian

"Returns an Array containing all upper-case Russian JISCharacters in
 alphabetic order."

| result |

result:= Array new.
16#2721 to: 16#2741
        do: [:i | result add: (JISCharacter withValue: i) ].
^ result
%

category: 'Non-Printable Characters'
classmethod: JISCharacter
backspace

"Returns the JIS back-space Character."

^ (JISCharacter withValue: 16#8)
%

category: 'Non-Printable Characters'
classmethod: JISCharacter
cr

"Returns the JIS carriage-return Character."

^ (JISCharacter withValue: 16#0D)
%

category: 'Non-Printable Characters'
classmethod: JISCharacter
esc

"Returns the JIS escape Character."

^ (JISCharacter withValue: 16#1B)
%

category: 'Non-Printable Characters'
classmethod: JISCharacter
lf

"Returns the JIS line-feed Character."

^ (JISCharacter withValue: 16#0A)
%

category: 'Non-Printable Characters'
classmethod: JISCharacter
newPage

"Returns the JIS new-page Character."

^ (JISCharacter withValue: 16#0C)
%

category: 'Non-Printable Characters'
classmethod: JISCharacter
space

"Returns the JIS one-byte space Character."

^ (JISCharacter withValue: 16#20)
%

category: 'Non-Printable Characters'
classmethod: JISCharacter
tab

"Returns the JIS tab Character."

^ (JISCharacter withValue: 16#09)
%

category: 'Instance Creation'
classmethod: JISCharacter
fromStream: aStream

"Returns the next JISCharacter in the stream aStream."

| result |
self _checkReadStream: aStream forClass: JapaneseString.
result:= aStream next.
result _validateClass: JISCharacter.
^result
%

category: 'Instance Creation'
classmethod: JISCharacter
fromString: aJapaneseString

"If aJapaneseString is a one-Character JapaneseString, returns the
 Character in aJapaneseString.  Otherwise, generates an error."

| result |
aJapaneseString _validateClass: JapaneseString.
(aJapaneseString size = 1)
ifTrue:
  [ result:= aJapaneseString at: 1.
    result _validateClass: JISCharacter.
    ^ result
  ]
ifFalse:
  [ self _errIncorrectFormat: aJapaneseString ]
%

category: 'Instance Creation'
classmethod: JISCharacter
new

"Disallowed.  You may not create new instances of JISCharacter."

self shouldNotImplement: #new
%

category: 'Instance Creation'
classmethod: JISCharacter
withValue: aJISValue

"Returns the JISCharacter with the specified JIS value.  Generates
 an error if aJISValue is greater than 65535 or less than 0."

<primitive: 252>

aJISValue _validateClass: Integer.
^ aJISValue _error: #numErrArgNotChr args: #[ JISCharacter ].
%

category: 'Instance Creation'
classmethod: JISCharacter
withEUCValue: anEUCValue

"Returns the JISCharacter with the specified EUC value."

<primitive: 82>

anEUCValue _validateClass: Integer.
^ anEUCValue _error: #rtErrBadEUCValue args:  #() 
%

category: 'Converting'
classmethod: JISCharacter
_twoByteEquivalent: aCharacter

"Returns the JIS two-byte Character that is equivalent to the argument
 aCharacter.  If there is no two-byte equivalent, returns the JIS one-byte
 equivalent.  Two Characters are equivalent if they represent the same
 Character regardless of case or internal representation.  For example, a JIS
 one-byte a is equivalent to a one-byte or two-byte upper-case or lower-case a."

<primitive: 254>

aCharacter _validateClasses: #[JISCharacter, Character].
^ self _primitiveFailed: #_twoByteEquivalent:
%

category: 'Accessing'
method: JISCharacter
eucValue

"Returns the EUC code of the receiver as a SmallInteger."

<primitive: 253>

^ self _primitiveFailed: #eucValue
%

category: 'Accessing'
method: JISCharacter
jisValue

"Returns the JIS code of the receiver as a SmallInteger."

<primitive: 247>

^ self _primitiveFailed: #jisValue
%

category: 'Accessing'
method: JISCharacter
firstByte

"Returns a SmallInteger representing the value of the leading byte of the JIS
 value of the receiver.  If the receiver is a one-byte JISCharacter, returns
 zero."

^ (self jisValue) bitShift: -8
%

category: 'Accessing'
method: JISCharacter
secondByte

"Returns a SmallInteger representing the value of the last byte
 of the JIS value of the receiver."

^ (self jisValue) \\ 16#0100
%

category: 'Testing'
method: JISCharacter
hasEUCFormat

"Returns true if the receiver can be represented in EUC format.
 Returns false otherwise."

<primitive: 255>

^ self _primitiveFailed: #hasEUCFormat
%

category: 'Testing'
method: JISCharacter
isDigit

"Returns true if the receiver is a one- or two-byte digit.  Returns
 false otherwise."

^(self isOneByteDigit) _or: [self isTwoByteDigit]
%

category: 'Testing'
method: JISCharacter
isEquivalent: aCharacter

"Returns true if the receiver is the same Character as the argument
 regardless of case or internal representation."

| argClass |
argClass:= aCharacter class.
((argClass == Character) _or:
              [argClass == JISCharacter])
ifFalse: "argument is not a Character and is not a JISCharacter"
   [ ^aCharacter isEquivalent: self ].

^ (JISCharacter _twoByteEquivalent: self asUppercase) ==
  (JISCharacter _twoByteEquivalent: aCharacter asUppercase)
%

category: 'Testing'
method: JISCharacter
isFirstLevelKanji

"Returns true if the receiver is a first level Kanji Character.
 Returns false otherwise."

| byteOne byteTwo |

(self isOneByteCharacter)
ifTrue: [ ^ false].

byteOne:= self firstByte.
byteTwo:= self secondByte.

((byteOne >= 16#30) _and: [byteOne <= 16#4E])
ifTrue:
   [ ^((byteTwo >= 16#21) _and: [byteTwo <= 16#7E]) ]
ifFalse:
   [ ^(byteOne == 16#4F) _and: [ (byteTwo >= 16#21) _and: [byteTwo <= 16#53]]]
%

category: 'Testing'
method: JISCharacter
isGreek

"Returns true if the receiver is a Greek Character.  Returns false otherwise."

^ (self isLowercaseGreek) _or: [self isUppercaseGreek]
%

category: 'Testing'
method: JISCharacter
isJisAscii

"Returns true if the receiver is a JIS-ASCII Character.  Returns false
 otherwise."

^(self jisValue) <= 127
%

category: 'Testing'
method: JISCharacter
isOneByteKatakana

"Returns true if the receiver is a one-byte Katakana Character.
 Returns false otherwise."

| jisValue |

jisValue:= self jisValue.
 ^ (jisValue <= 16#DF) _and: [jisValue >= 16#A1]
%

category: 'Testing'
method: JISCharacter
isHiragana

"Returns true if the receiver is a Hiragana Character.  Returns false
 otherwise."

| jisValue |

jisValue:= self jisValue.
^ (jisValue <= 16#2473) _and: [jisValue >= 16#2421]
%

category: 'Testing'
method: JISCharacter
isKana

"Returns true if the receiver is a Hiragana Character or a one- or
 two-byte Katakana Character.  Returns false otherwise."

^ (self isHiragana) _or: [self isKatakana]
%

category: 'Testing'
method: JISCharacter
isKanji

"Returns true if the receiver is a Kanji Character.  Returns false otherwise."

^ (self isFirstLevelKanji) _or: [self isSecondLevelKanji]
%

category: 'Testing'
method: JISCharacter
isKatakana

"Returns true if the receiver is a one- or two-byte Katakana Character.
 Returns false otherwise."

^ (self isOneByteKatakana) _or: [self isTwoByteKatakana]
%

category: 'Testing'
method: JISCharacter
isLineElement

"Returns true if the receiver is a line element Character.  Returns
 false otherwise."

| jisValue |

jisValue:= self jisValue.
^ (jisValue <= 16#2840) _and: [jisValue >= 16#2821]
%

category: 'Testing'
method: JISCharacter
isLowercase

"Returns true if the receiver is a lower-case Character.  Returns false
 otherwise.  The only JISCharacters that have case distinction are
 Roman, Greek and Russian."

^ (self isOneByteLowercaseRoman) _or:
     [ (self isTwoByteLowercaseRoman) _or:
        [ (self isLowercaseGreek) _or:
          [self isLowercaseRussian]]]
%

category: 'Testing'
method: JISCharacter
isLowercaseGreek

"Returns true if the receiver is a lower-case Greek Character.  Returns
 false otherwise."

| jisValue |

jisValue:= self jisValue.
^ (jisValue <= 16#2658) _and: [jisValue >= 16#2641]
%

category: 'Testing'
method: JISCharacter
isLowercaseRussian

"Returns true if the receiver is a lower-case Russian Character.  Returns
 false otherwise."

| jisValue |

jisValue:= self jisValue.
^ (jisValue <= 16#2771) _and: [jisValue >= 16#2751]
%

category: 'Testing'
method: JISCharacter
isOneByteCharacter

"Returns true if the receiver is a one-byte Character.  Returns false
 otherwise."

^ (self jisValue <= 16#FF)
%

category: 'Testing'
method: JISCharacter
isOneByteDigit

"Returns true if the receiver is a one-byte digit.  Returns false otherwise."

| jisValue |

jisValue:= self jisValue.
^ (jisValue <= 16#39) _and: [jisValue >= 16#30]
%

category: 'Testing'
method: JISCharacter
isOneByteLowercaseRoman

"Returns true if the receiver is a one-byte lower-case Roman Character.
 Returns false otherwise."

| jisValue |

jisValue:= self jisValue.
(jisValue < 97)  ifTrue: [ ^ false ].
(jisValue < 123) ifTrue: [ ^ true  ].
(jisValue < 223) ifTrue: [ ^ false ].
(jisValue < 256) ifTrue: [ ^ (jisValue == 247) not ].
^ false
%

category: 'Testing'
method: JISCharacter
isOneByteRoman

"Returns true if the receiver is a one-byte Roman Character.
 Returns false otherwise."

^ (self isOneByteLowercaseRoman) _or: [self isOneByteUppercaseRoman]
%

category: 'Testing'
method: JISCharacter
isOneByteUppercaseRoman

"Returns true if the receiver is a one-byte upper-case Roman Character.
 Returns false otherwise."

| jisValue |

jisValue:= self jisValue.
(jisValue < 65)  ifTrue: [ ^ false ].
(jisValue < 91)  ifTrue: [ ^ true  ].
(jisValue < 192) ifTrue: [ ^ false ].
(jisValue < 223) ifTrue: [ ^ (jisValue == 215) not ].
^ false
%

category: 'Testing'
method: JISCharacter
isRussian

"Returns true if the receiver is a Russian Character.  Returns false
 otherwise."

^ (self isLowercaseRussian) _or: [self isUppercaseRussian]
%

category: 'Testing'
method: JISCharacter
isSecondLevelKanji

"Returns true if the receiver is a second-level Kanji Character.
 Returns false otherwise."

| byteOne byteTwo |

(self isOneByteCharacter)
  ifTrue: [ ^ false].

byteOne:= self firstByte.
byteTwo:= self secondByte.

((byteOne >= 16#50) _and: [byteOne <= 16#73])
  ifTrue:
    [ ^((byteTwo >= 16#21) _and: [byteTwo <= 16#7E]) ]
  ifFalse:
    [ ^(byteOne == 16#74) _and: [ (byteTwo >= 16#21) _and: [byteTwo <= 16#24]]]
%

category: 'Testing'
method: JISCharacter
isTwoByteCharacter

"Returns true if the receiver is a two-byte Character.  Returns false
 otherwise."

^ (self jisValue > 16#FF)
%

category: 'Testing'
method: JISCharacter
isTwoByteDigit

"Returns true if the receiver is a two-byte digit.  Returns false otherwise."

| jisValue |

jisValue:= self jisValue.
^ (jisValue <= 16#2339) _and: [jisValue >= 16#2330]
%

category: 'Testing'
method: JISCharacter
isTwoByteKatakana

"Returns true if the receiver is a two-byte Katakana Character.
 Returns false otherwise."

| jisValue |

jisValue:= self jisValue.
^ (jisValue <= 16#2576) _and: [jisValue >= 16#2521]
%

category: 'Testing'
method: JISCharacter
isTwoByteLowercaseRoman

"Returns true if the receiver is a two-byte lower-case Roman Character.
 Returns false otherwise."

| jisValue |

jisValue:= self jisValue.
^ (jisValue <= 16#237A) _and: [jisValue >= 16#2361]
%

category: 'Testing'
method: JISCharacter
isTwoByteRoman

"Returns true if the receiver is a two-byte Roman Character.
 Returns false otherwise."

^ (self isTwoByteLowercaseRoman) _or: [self isTwoByteUppercaseRoman]
%

category: 'Testing'
method: JISCharacter
isTwoByteUppercaseRoman

"Returns true if the receiver is a two-byte upper-case Roman Character.
 Returns false otherwise."

| jisValue |

jisValue:= self jisValue.
^ (jisValue <= 16#235A) _and: [jisValue >= 16#2341]
%

category: 'Testing'
method: JISCharacter
isUppercase

"Returns true if the receiver is an upper-case Character.  Returns false
 otherwise.  The only JISCharacters that have case distinction are
 Roman, Greek and Russian."

^ (self isOneByteUppercaseRoman) _or:
     [ (self isTwoByteUppercaseRoman) _or:
        [ (self isUppercaseGreek) _or:
           [self isUppercaseRussian]]]
%

category: 'Testing'
method: JISCharacter
isUppercaseGreek

"Returns true if the receiver is an upper-case Greek Character.  Returns
 false otherwise."

| jisValue |

jisValue:= self jisValue.
^ (jisValue <= 16#2638) _and: [jisValue >= 16#2621]
%

category: 'Testing'
method: JISCharacter
isUppercaseRussian

"Returns true if the receiver is an upper-case Russian Character.  Returns
 false otherwise."

| jisValue |

jisValue:= self jisValue.
^ (jisValue <= 16#2741) _and: [jisValue >= 16#2721]
%

category: 'Testing'
method: JISCharacter
isSpecialChar

"Returns true if the receiver is a special Character.  Returns false
 otherwise."

| jisValue |

jisValue:= self jisValue.
((jisValue <= 47) _and: [jisValue >= 32]) ifTrue: [ ^true ].
((jisValue <= 64) _and: [jisValue >= 58]) ifTrue: [ ^true ].
((jisValue <= 96) _and: [jisValue >= 91]) ifTrue: [ ^true ].
((jisValue <= 191) _and: [jisValue >= 123]) ifTrue: [ ^true ].
(jisValue == 215) ifTrue: [ ^true ].
(jisValue == 247) ifTrue: [ ^true ].
((jisValue <= 16#217E) _and: [jisValue >= 16#2121]) ifTrue: [ ^true ].
((jisValue <= 16#222E) _and: [jisValue >= 16#2221]) ifTrue: [ ^true ].
((jisValue <= 16#2241) _and: [jisValue >= 16#223A]) ifTrue: [ ^true ].
((jisValue <= 16#2250) _and: [jisValue >= 16#224A]) ifTrue: [ ^true ].
((jisValue <= 16#226A) _and: [jisValue >= 16#225C]) ifTrue: [ ^true ].
((jisValue <= 16#2279) _and: [jisValue >= 16#2272]) ifTrue: [ ^true ].
^ (jisValue == 16#227E)
%

category: 'Formatting'
method: JISCharacter
asEUCString

"Returns an EUCString that represents the receiver."

^EUCString new at: 1 put: self; yourself
%

category: 'Formatting'
method: JISCharacter
asString

"Returns a one-Character instance of String containing the receiver."

^ (String new:1) at: 1 put: self; yourself
%

category: 'Formatting'
method: JISCharacter
displayWidth

"Returns the width necessary to display the receiver.  For a JISCharacter, this
 can be 1 or 2."

(self isOneByteCharacter)
  ifTrue: [ ^1 ]
  ifFalse: [ ^2 ]
%

! fix bug 11589
category: 'Formatting'
method: JISCharacter
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

aStream nextPutAll: ((self class name) asString + ' withValue: ' +
                    self asInteger describe) 
%

category: 'Converting'
method: JISCharacter
asCharacter

"Returns the (ASCII) Character corresponding to the receiver."

^ self asInteger asCharacter
%
category: 'Converting'
method: JISCharacter
asciiValue

"Returns the ASCII value (a SmallInteger) corresponding to the receiver."

^ self asCharacter asciiValue
%

category: 'Converting'
method: JISCharacter
asHankaku

"Returns a JISCharacter that is the one-byte representation of a two-byte digit
 or two-byte Roman receiver.  If the receiver is not a two-byte digit or a
 two-byte Roman Character, returns the receiver."

<primitive: 248>

^ self _primitiveFailed: #asHankaku
%

category: 'Converting'
method: JISCharacter
asInteger

"Returns the JIS code of the receiver as a SmallInteger."

^ self jisValue
%

category: 'Converting'
method: JISCharacter
asJapaneseString: aClass

"Returns an instance of class aClass containing only the receiver.
 aClass must be a subclass of JapaneseString."

aClass validateSubclassOf: JapaneseString.
^ aClass withAll: #[ self ]
%

category: 'Converting'
method: JISCharacter
asJISCharacter

"Returns the receiver."

^ self
%

category: 'Converting'
method: JISCharacter
asLowercase

"Returns a JISCharacter that is the lower-case Character corresponding to the
 receiver.  If the receiver is lower-case or has no case, this returns the
 receiver itself.  Returns a one-byte lower-case Roman Character if the
 receiver is a one-byte upper-case Roman Character.  Returns a two-byte
 lower-case Roman Character if the receiver is a two-byte upper-case Roman
 Character.  The only JISCharacters that have case distinction are Roman,
 Greek, and Russian."

<primitive: 249>

^ self _primitiveFailed: #asLowercase
%

category: 'Converting'
method: JISCharacter
asUppercase

"Returns a JISCharacter that is the upper-case Character corresponding
 to the receiver.  If the receiver is upper-case or has no case, this
 returns the receiver itself.  Returns a one-byte upper-case Roman Character
 if the receiver is a one-byte lower-case Roman Character.  Returns a
 two-byte upper-case Roman Character if the receiver is a two-byte lower-case
 Roman Character.  The only JISCharacters that have case distinction are Roman,
 Greek, and Russian."

<primitive: 250>

^ self _primitiveFailed: #asUppercase
%

category: 'Converting'
method: JISCharacter
asZenkaku

"Returns a JISCharacter that is the two-byte representation of a one-byte digit
 or one-byte Roman receiver.  If the receiver is not a one-byte digit or
 one-byte Roman Character, returns the receiver."

<primitive: 251>

^ self _primitiveFailed: #asZenkaku
%

category: 'Converting'
method: JISCharacter
digitValue

"Returns a SmallInteger representing the value of the receiver,
 a digit, or returns nil if the receiver is not a digit."

(self isOneByteDigit)
  ifTrue: [ ^self jisValue - 16#30 ].

(self isTwoByteDigit)
  ifTrue: [ ^self jisValue - 16#2330]
  ifFalse: [ ^nil]
%

category: 'Comparing'
method: JISCharacter
= aCharacter

"Returns true if the receiver is equal to the argument.  Returns false
 otherwise."

<primitive: 81>

(aCharacter isKindOf: AbstractCharacter)
ifTrue: [ ^ aCharacter = self ]
ifFalse: [ ^ false ]
%

category: 'Comparing'
method: JISCharacter
< aCharacter

"Returns true if the receiver is less than the argument.
 Returns false otherwise."

<primitive: 80>

^aCharacter > self
%

category: 'Comparing'
method: JISCharacter
<= aCharacter

"Returns true if the receiver is less than or equal to
 the argument.  Returns false otherwise."

^(self < aCharacter) _or: [self = aCharacter]
%

category: 'Comparing'
method: JISCharacter
> aCharacter

"Returns true if the receiver is greater than the argument.
 Returns false otherwise."

^(self <= aCharacter) not
%

category: 'Comparing'
method: JISCharacter
>= aCharacter

"Returns true if the receiver is greater than or equal to
 the argument.  Returns false otherwise."

^(self < aCharacter) not
%

category: 'Copying'
method: JISCharacter
copy

"Returns the receiver.  (Does not create a new instance.)"

^ self
%

category: 'Storing and Loading'
method: JISCharacter
writeTo: passiveObj

"Converts the receiver to its passive form and writes that information on
 passiveObj."

passiveObj writeClass: self class;
  nextPutAll: self jisValue asString;
  space
%

category: 'Storing and Loading'
classmethod: JISCharacter
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst |

inst := JISCharacter withValue:(Integer fromString:(passiveObj upToSeparator)).
passiveObj hasRead: inst.
^inst
%

category: 'Storing and Loading'
method: JISCharacter
containsIdentity

"Private."

^true
%
category: 'Testing'
method: JISCharacter
isSpecial

"Returns true if the receiver is a special object."

^ true
%

