#! /usr/bin/sh
# set -xv
#=========================================================================
# Copyright (C) VMware, Inc. 1986-2011.  All Rights Reserved..
#
# Name - printlogs.sh
#
# Purpose - Script to print tranlog data.  
#           See "help" text for details.
#
# $Id: printlogs.sh,v 1.2 2008-01-09 22:50:52 stever Exp $
#
#=========================================================================

if [ "x$1" = "x" ]; then
   echo ""
   echo "===================================================================="
   echo "Usage: `basename $0` [(filters)] [full] [all] [<tlogA> ... <tlogZ>]"
   echo ""
   echo "Purpose:"
   echo ""
   echo "  To print out contents of designated tranlogs in current directory."
   echo ""
   echo "  WARNING: This can product vast amounts of output!!"
   echo ""    
   echo "  Make sure \$GEMSTONE is set."
   echo ""
   echo "  Optional: (filters) "
   echo ""
   echo "      Only print records that match the filtering criteria."
   echo "      One or more of the following pairs:"
   echo ""
   echo "      user    username - filter by GS UserProfile name "
   echo "      host    hostname - filter by gem/topaz process host "
   echo "      client  X.X.X.X  - filter by client IP Address "
   echo ""
   echo "  Keyword: full "
   echo ""
   echo "     More detailed logs are produced (VERY LARGE!)."
   echo ""
   echo "  Keyword: all "
   echo ""
   echo "     Print out contents of all tranlogs in this directory."
   echo ""
   echo "  Tranlogs are assumed to be named 'tranlogXXX.dbf'."
   echo "  Override by setting \$GS_TRANLOG_PREFIX to the prefered value."
   echo ""
   echo "  WARNING: a maximum of 256 tranlogs can be analyzed in one set."
   echo ""
   echo "===================================================================="
   echo ""
   exit 0
fi

# check that $GEMSTONE is set
if [ "x$GEMSTONE" = "x" ]; then
  echo "ERROR:  \$GEMSTONE not set" 1>&2
  exit 1
fi

# determine tranlog naming convention
if [ "x$GS_TRANLOG_PREFIX" = "x" ]; then
  GS_TRANLOG_PREFIX="tranlog"
fi

# debug
# echo "Using tranlog prefix $GS_TRANLOG_PREFIX"

# Check for $GEMSTONE/sys/pgsvrslow
if [ -f $GEMSTONE/sys/pgsvrslow ]; then
  pgsvrExe=$GEMSTONE/sys/pgsvrslow
else
  echo "ERROR:  \$GEMSTONE/sys/pgsvrslow not found" 1>&2
  exit 1;
fi

# debug
# echo "Using $pgsvrExe"

# initialize variables
filters=""
tranlogs=""
dumpCmd="dumplog"
tmpfile="/usr/tmp/gsprintlog$$"

# debug
# echo "tmpfile = $tmpfile"

# create temporary file to hold pgsvr commands
touch $tmpfile

# setup to print username info
echo "printusername " >>$tmpfile

# loop over arguments..
while [ "x$1" != "x" ] 
do

# check for filtering keywords

if [ $1 = "user" ]; then
  echo "'$2' username" >>$tmpfile
  filters="$filters user $2"
  shift
  shift

elif [ $1 = "host" ]; then
  echo "'$2' gemhost" >>$tmpfile
  filters="$filters host $2"
  shift
  shift

elif [ $1 = "client" ]; then
  echo "'$2' clientip" >>$tmpfile
  filters="$filters client $2"
  shift
  shift

# check for keyword full
elif [ $1 = "full" ]; then
  dumpCmd=dumplogfull
  shift

# check for keyword all
elif [ $1 = "all" ]; then
  tranlogs=`ls -1rt *$GS_TRANLOG_PREFIX*.dbf`
  shift

# check if it's a tranlog name
elif [ -r $1 ]; then
  tranlogs="$tranlogs $1"
  shift

# otherwise don't know what it is
else
  echo "Unrecognized keyword $1"
  shift
fi

# done with argument processing loop..
done


# construct pgsvr commands
for each in $tranlogs
do
  echo "'$each' openlog 0 100000000 $dumpCmd closedbf" >>$tmpfile
done

# debug: print out contents of temporary file
# echo "PGSVRSLOW command file contents:"
# cat $tmpfile
# exit

# print summary info
echo "Tranlog Analysis Print Log"
echo
echo "Printing out tranlogs:"
echo
for each in $tranlogs
do
  echo "  $each"
done
echo
if [ "$filters" != "" ]; then
  echo "Filtering on: $filters"
fi

# now execute..
$pgsvrExe < $tmpfile
pStatus=$?
if [ $pStatus -ne 0 ]; then
  echo "pgsvr returned $pStatus";
  exit $pStatus
fi

# delete temporary file
rm -f $tmpfile

#
# end of script


