#define GSORAPUBLIC_C TRUE
/* =========================================================================
 * Copyright (C) GemTalk Systems. 1986-2024.  All Rights Reserved.
 *
 * Name - gsorapublic.c
 *
 * Description - GemConnect for Oracle public C module containing tables and
 *               functions that may be augmented or modified by developers.
 *
 * ==========================================================================
 */
 

#define ORACLE

/* INCLUDES ***************************************************************/

/* standard includes */
#include <stdio.h>
#include <string.h>
#if !defined(FLG_APPLE_UNIX)
#include <malloc.h>
#endif
#include <ctype.h>

/* GemStone includes */
#if defined(FLG_LINKED_BUILD)
#include <gcirtl.hf>
#else
#include <gciua.hf>
#endif

#if defined(FLG_OOP64)
#include "gcisend.hf"
#endif

#include "oci.h" /* 9i */

/* GemConnect includes */
#include "gsoraapi.ht"
#include "gsrdbapi.ht"
#include "gsrdbapi.hf"

// 41057: for GS643: handle the re-defined classes
#if defined(FLG_GS643)
#define OOP_CLASS_DOUBLE_BYTE_STRING   OOP_CLASS_DoubleByteString
#define OOP_CLASS_FLOAT                OOP_CLASS_Float
#define OOP_CLASS_SCALED_DECIMAL       OOP_CLASS_ScaledDecimal
#endif

/* USER ACTIONS ***********************************************************/

#ifdef EXAMPLE  /* example code */

/* common */
/* fix 36402 */
OopType GsNewDateTime(OopType cls,
		      int year, int month, int day, 
		      int hours, int minutes, int seconds, long millisecs);

/*==========================================================================
 * Name - GsColumnInfo
 *
 * Returns an array of arrays, each with the following information about a
 * column in the result set of a particular read stream:
 *
 *	1) column name
 *	2) data type (relational database dependent)
 *	3) max-length
 *	4) nullable
 *	5) precision
 *	6) scale
 *      7) class of field when translated to an object (actual datum
 *	   may be a subclass of this class)
 *
 * This user action may be called from Smalltalk by invoking:
 *
 *        values := System userAction: #GsOraColumnInfo
 *			with: (stream connection) with: stream.
 *
 *==========================================================================
 */

OopType GsColumnInfo(
    OopType connectOop,
    OopType readStreamOop)
{
    int i;
    GsConnection *conn;
    GsReadStream *stream;
    GsCmap *cmap;
    OopType elements[7];
    OopType element;
    OopType result;
    char *name;

    /* check to see if we have a valid (i.e., opened) connection object,
       if not return nil */
    for (conn=AllConnections; conn != NULL; conn=conn->next) {
        if (conn->object == connectOop)
            break;
    }
    if (!conn)
        return OOP_NIL;
    
    /* check to see if we have a valid read stream object, if not return nil */
    for (stream=conn->readStreams; stream != NULL; stream=stream->next) {
        if (stream->object == readStreamOop)
            break;
    }
    if (!stream)
        return OOP_NIL;
    
    /* create the array we will be returning the column information in */
    result = GciNewOop(OOP_CLASS_ARRAY);
    
    for (i=0; i<stream->numColumns; i++) {
        cmap = &(stream->columns[i]);
        name = &(cmap->columnName[0]);
        elements[0] = GciNewByteObj(OOP_CLASS_STRING, (ByteType *)name,
                                    strlen(name));
	elements[1] = GciLongToOop(cmap->dataFmt);
	elements[2] = GciLongToOop(cmap->maxlen);
	elements[3] = (cmap->nullok ? OOP_TRUE : OOP_FALSE);
	elements[4] = GciLongToOop(cmap->precision);
	elements[5] = GciLongToOop(cmap->scale);
        elements[6] = cmap->instVarClass;
	if (elements[6] == OOP_CLASS_FLOAT)
	    elements[6] = OOP_CLASS_NUMBER;
        element = GciNewOop(OOP_CLASS_ARRAY);
        GciStoreOops(element, 1, elements, 7);
        GciStoreOop(result, i+1, element);
    }
    
    return result;
}

#endif

/* ==========================================================================
 * Name - GsPublicIVsAndConstraints
 *
 * Returns an array of pairs of instance variable names and constraints for
 * the given result set read-stream.
 *
 * The instance variable names are translated from the column names obtained
 * from Oracle. Since Oracle allows '#', '$', and other symbols in their
 * identifiers, and GemStone does not, the instance variable names returned
 * are not identical to the column names. All symbols that are illegal in
 * GemStone are converted into underscores, i.e., '_'. In addition, if a
 * column name begins with a symbol other than an alphabetic character or
 * underscore, the character 'g' is added to the beginning of the instance
 * variable translation for it.
 *
 * The constraint classes are obtained from those set by GemConnect, either
 * by default or through the GsPublicSetClassAndConversion function below.
 *
 * ==========================================================================
 */

OopType GsPublicIVsAndConstraints(
    OopType connectOop,
    OopType readStreamOop)
{
    int i;
    GsConnection *conn;
    GsReadStream *stream;
    OopType elements[7]; /* 37102: extend to 7 */
    OopType pair;
    OopType result;
    char *name;
    char newName[1024];
    char *newNamePtr;

    /* check to see if we have a valid (i.e., opened) connection object,
       if not return nil */
    for (conn=AllConnections; conn != NULL; conn=conn->next) {
        if (conn->object == connectOop)
            break;
    }
    if (!conn)
        return OOP_NIL;
    
    /* check to see if we have a valid read stream object, if not return nil */
    for (stream=conn->readStreams; stream != NULL; stream=stream->next) {
        if (stream->object == readStreamOop)
            break;
    }
    if (!stream)
        return OOP_NIL;
    
    /* create the array we will be returning the instance variable names and
       constraint classes in */
    result = GciNewOop(OOP_CLASS_ARRAY);
    
    /* iterate over the number of columns we have selected */
    for (i=0; i<stream->numColumns; i++) {
        name = &(stream->columns[i].columnName[0]);
        newNamePtr = &(newName[0]);

        /* make sure first character of name is alphabetic or an underscore */
        if (!isalpha(*name) && (*name != '_')) {
            *newNamePtr++ = 'g';
        }

#ifdef EXAMPLE  /* example code */

        /* remove all illegal identifier characters */
        while (*name) {
            if (isalnum(*name) || (*name == '_')) {
                *newNamePtr++ = *name;
            }
            name++;
        }
        *newNamePtr = '\0';

#else /* actual code */

        /* convert all illegal identifier characters to underscore */
        while (*name) {
            if (isalnum(*name) || (*name == '_')) {
                *newNamePtr++ = *name;
            }
            else {
                *newNamePtr++ = '_';
            }
            name++;
        }
        *newNamePtr = '\0';

#endif

	/* it is important to return mutable names, so use String instead
	 * of Symbol for instance variable names */

        elements[0] = GciNewByteObj(OOP_CLASS_STRING, (ByteType *)newName,
                                    strlen(newName));
        elements[1] = stream->columns[i].instVarClass;
	if (elements[1] == OOP_CLASS_FLOAT)
	    elements[1] = OOP_CLASS_NUMBER;

	/* 37102: extend elements to include additional Oracle info */
	elements[2] = GciLongToOop(stream->columns[i].origFmt);
	elements[3] = GciLongToOop(stream->columns[i].maxlen);
	elements[4] = GciLongToOop(stream->columns[i].origPrecision);
	elements[5] = GciLongToOop(stream->columns[i].scale);
        elements[6] = (stream->columns[i].nullok == 1) ? OOP_TRUE : OOP_FALSE;

        pair = GciNewOop(OOP_CLASS_ARRAY);
        GciStoreOops(pair, 1, elements, 7);

        GciStoreOop(result, i+1, pair);
    }
    
    return result;
}


#define UA GciUserActionFType

/* register all user actions */
static GciUserActionSType allPublicActions[] = {

#ifdef EXAMPLE  /* example code */
  { "GsOraColumnInfo",			2, (UA)GsColumnInfo },
#endif

  { "GsOraInstVarsAndConstraints",	2, (UA)GsPublicIVsAndConstraints },
  };


/* CALLBACK FUNCTIONS *****************************************************/


/* ==========================================================================
 * Name - GsPublicInitialize
 *
 * This is called at user-action registration time to allow additional
 * initializations to be performed.
 *
 * Note: If this user-action library is being used for client-side user-
 * actions, there may not be a valid GemStone session, so GCI calls that
 * deal with objects should not be used.
 *
 * ==========================================================================
 */

void GsPublicInitialize()
{
    int i;
    int size;

    /* install all registered user actions */
    size = sizeof(allPublicActions) / sizeof(GciUserActionSType);
    for (i=0; i<size; i++) {
#ifdef EXAMPLE
        printf("Loading UA %s\n", allPublicActions[i].userActionName);
	fflush(stdout);
#endif
        GciInstallUserAction(&allPublicActions[i]);
    }

#ifdef EXAMPLE  /* example code */

    /* indicate that the GemConnect for Oracle user actions have been loaded */
    printf("GemConnect for Oracle has now been loaded!\n");

    /* display the maximum number of connections allowed */
    printf("Maximum number of Oracle connections is: unlimited\n");

#endif

}

/* ==========================================================================
 * Name - GsPublicShutdown
 *
 * This is called just before the Oracle interface is unloaded.  GemStone
 * is unavailable at this time and the GCI should not be used.
 *
 * Any memory allocated in GsPublicInitialize() should be freed in this
 * function.
 *
 * ==========================================================================
 */

void GsPublicShutdown()
{
}

/* ========================================================================
 * Name - GsPublicConnectAlloc
 *
 * This is called just after a connection has been allocated but before it
 * is actually opened. The parameters passed are those the connection will
 * be opened with.
 *
 * WARNING: An attempt to actually open the given connection should not be
 *          made here because this will be done by GemConnect upon a
 *          succesfull return from this callback.
 *
 * ========================================================================
 */

void GsPublicConnectAlloc(
    GsConnection * conn,
    OopType parameters)
{
}

/* ========================================================================
 * Name - GsPublicConnected
 *
 * This is called just after a connection has been established with the
 * new connection structure (defined in gsrdbapi.ht) and the parameters
 * object used to establish the connection.
 *
 * ========================================================================
 */

void GsPublicConnected(
    GsConnection * conn,
    OopType parameters)
{
    OopType oAutoCommit;
    
    /* 38023: move code to set textLimit to GsOraConnect() */

    /* retrieve and set the autoCommit parameter, that specifies whether
     * transactions are committed automatically after each command is executed
     * on the server. The default is to manage transactions in manual mode.
     */
    oAutoCommit = GciSendMsg(parameters, 1, "autoCommit");
    if (oAutoCommit == OOP_TRUE) {
      conn->autoCommit = TRUE;
    }

}

/* ========================================================================
 * Name - GsPublicDisconnect
 *
 * This is called just before a connection is dropped.  It is possible that
 * this function will be called again with the same connection if disconnect
 * fails the first time, and is retried from Smalltalk.
 *
 * If the flag GsUnloading is set, this function should take care not
 * to call the GCI.
 *
 * ========================================================================
 */

void GsPublicDisconnect(GsConnection * conn)
{
}

/* ========================================================================
 * Name - GsPublicStreamAlloc
 *
 * This is called just after a read-stream command is allocated but before
 * it is actually executed.
 *
 * WARNING: An attempt to actually execute the given stream's command should
 *          not be made here because this will be done by GemConnect upon a
 *          succesfull return from this callback.
 *
 * ========================================================================
 */

void GsPublicStreamAlloc(
    GsConnection * conn,
    GsReadStream * readStream)
{
}

/* ========================================================================
 * Name - GsPublicStreamInit
 *
 * This is called just after a read-stream command has been executed.
 *
 * ========================================================================
 */

void GsPublicStreamInit(
    GsConnection * conn,
    GsReadStream * readStream)
{
}

/* ========================================================================
 * Name - GsPublicStreamDrop
 *
 * This is called just after a read-stream has been unlinked from a
 * connection structure, but before any of its contents have been freed.
 *
 * If the flag GsUnloading is set, this function should take care not
 * to call the GCI.
 *
 * ========================================================================
 */

void GsPublicStreamDrop(
    GsConnection * conn,
    GsReadStream * readStream)
{
}

/* ========================================================================
 * Name - GsPublicWriteStreamFlush
 *
 * This is called just after a write-stream's buffered contents
 * is flushed to the DB.
 *
 * ========================================================================
 */

void GsPublicWriteStreamFlush(
    GsConnection * conn,
    GsReadStream * readStream)
{
}

/* ========================================================================
 * Name - GsPublicSetClassAndConversion
 *
 * This function is called to obtain the class to be used as the constraint
 * for the given relational column.  Normally this function does nothing,
 * allowing the GemStone Oracle interface to decide on an appropriate
 * conversion.
 *
 * To specify a conversion, this function must fill in the instVarClass and
 * value fields of the cmap argument.  The datafmt field may also be modified
 * to specify the format of data delivered by Oracle.  The instVarClass field
 * should hold the class that the column will be converted to,
 * and the value field must point to a buffer large enough to hold the
 * data Oracle will return.  Array variables are used by the interface,
 * so the value field should point to a buffer that can hold stream->maxRows
 * fetched values.
 *
 * For Oracle, this function is called after the OCI is called and the
 * column-map fields origFmt, maxlen, precision, and scale have been 
 * initialized.  The buffers pointed to by valuelen, rcode, and flag 
 * have been allocated.
 *
 * In order for the Oracle interface to function properly, any custom
 * conversions specified here should be handled in the GsPublicConvertToObject
 * function.
 *
 * ========================================================================
 */

BoolType GsPublicSetClassAndConversion(
    GsConnection * conn,
    GsReadStream * stream,
    GsCmap * cmap)
{

#ifdef EXAMPLE  /* example code */

    OopType myclass;

    /* note that this function handles Oracle "internal" datatypes.  Oracle's
     * OCI returns internal datatype indicators and programs
     * can request that a specific column be mapped to an alternative,
     * external, datatype.
     */
    
    switch (cmap->dataFmt) {

        case SQLT_RID:
        case SQLT_RDD:
            /* to have ROW_ID data converted to strings, change
             * the class to OOP_CLASS_STRING
             */
            cmap->dataFmt = SQLT_CHR;
            cmap->maxlen = ORA_ROWID_LENGTH;
            myclass = OOP_CLASS_STRING;
            break;

        case SQLT_DAT: 
            /* to have date data converted to just dates without
             * without times, change the class to OOP_CLASS_DATE.
             */
            cmap->maxlen = ORA_DATE_LENGTH;
            myclass = OOP_CLASS_DATE;
            break;

        default:
	    return TRUE;
    }

    cmap->instVarClass = myclass;

#endif

    return TRUE;
}

/* =========================================================================
 * Name - GsPublicConvertToObject
 *
 * Converts a relational database datum to an object in GemStone.
 * 
 * This function should follow any data conversion/class matching that
 * is specified in GsPublicSetClassAndConversion.  The classes set in
 * that function may be used as constraints in classes created to hold
 * query results.  Converting to a different kind of object in this function
 * could lead to constraint violations while fetching results.
 *
 * Normally this function will do nothing, allowing the GemStone Oracle
 * interface to choose an appropriate conversion for the given column data.
 *
 * ========================================================================
 */

OopType GsPublicConvertToObject(
    GsConnection * conn,
    GsReadStream * stream,
    GsCmap * cmap)
{

#ifdef EXAMPLE  /* example code */

    OopType   result;
    int		row;
    ByteType	*oraBytes;
    OracleDate  *oraDate;
    int century, year, month, day, hours, minutes, seconds, yrMod100;

    row = stream->rowIndex;

    /* check for NULL value */
    if (cmap->flag[row] == -1)
        return OOP_NIL;
    
    /* check for truncated value.  For some datatypes, the value will be the
     * size of the datum before truncation
     */
    if (cmap->flag[row]) {
        if (cmap->flag[row] == -2)
	    conn->maxTruncatedSize = 0xFFFF;
	else if (conn->maxTruncatedSize < cmap->flag[row])
            conn->maxTruncatedSize = cmap->flag[row];
    }

    result = OOP_ILLEGAL;
    
    switch (cmap->dataFmt) {

      /* Developers can redefine how mapping to various GemStone objects
	 occur using GCI calls.  Returning OOP_ILLEGAL defaults to the
         normal GemConnect conversion code.  

	 Example given here for SQLT_CHR and SQLT_DAT
      */

        case SQLT_CHR:
            oraBytes = (ByteType *)cmap->value;
            oraBytes = oraBytes + (stream->rowIndex * cmap->maxlen);
            result = GciNewByteObj(OOP_CLASS_STRING, oraBytes,
                                   cmap->maxlen);
            break;

        case SQLT_DAT:
            oraDate = (OracleDate *)(cmap->value);

            century = oraDate[row].century;
            century -= 100;
            year    = oraDate[row].year;
            year    -= 100;
            month   = (int)oraDate[row].month;
            day     = (int)oraDate[row].day;
            hours   = (int)oraDate[row].hour - 1;
            minutes = (int)oraDate[row].minute - 1;
            seconds = (int)oraDate[row].second - 1;
    
	    result = GciSendMsg(OOP_CLASS_DATE_TIME, 12,
				"newWithYear:", GCI_LONG_TO_OOP(year),
				"month:",       GCI_LONG_TO_OOP(month),
				"day:",         GCI_LONG_TO_OOP(day),
				"hours:",       GCI_LONG_TO_OOP(hours),
				"minutes:",     GCI_LONG_TO_OOP(minutes),
				"seconds:",     GCI_LONG_TO_OOP(seconds));

	    // 48654: check if we want DateAndTime objects
	    if (conn->returnDAT) {
	      result = GciSendMsg(result, 1, "_asDateAndTime");
	    }
            break;

        default:
            break;
    }

    return result;

#else

    return OOP_ILLEGAL;

#endif

}

/* =========================================================================
 * Name - GsPublicConvertToDbData
 *
 * Converts a GemStone object into relational database data
 * 
 * This function should follow any data conversion/class matching that
 * is specified in GsPublicSetClassAndConversion.
 *
 * Normally this function will do nothing, allowing the GemStone Oracle
 * interface to choose an appropriate conversion for the given column data.
 *
 * ========================================================================
 */

BoolType GsPublicConvertToDbData(
    GsConnection * conn,
    GsReadStream * stream,
    GsCmap * cmap,
    OopType  rowObject,    /* 34925 */
    char*    selector,     /* 34925 */
    int      columnNumber, /* 34925 */
    OopType gsObject)
{

  /* Developers can redefine how mapping from various GemStone objects
     occur using GCI calls.  Returning FALSE defaults to the normal
     GemConnect conversion code.
  */

  return FALSE;
}




/* UTILITY FUNCTIONS *****************************************************/


/* =========================================================================
 * Name - GsDbgLogStr
 *
 * Logs debugging information to the current GCI logging stream.
 *
 * Usually this stream prints to stdout. If running with an RPC Gem, output
 * will go to the Gem's log file. This log file is usually deleted by GemStone
 * when the session logs out.
 *
 * When using linked topaz, the stream is redirected so that logged
 * information is captured in Topaz log files established by "output push"
 * commands.
 *
 * ========================================================================
 */

void _GsDbgLogStr(const char * str )
{
    GsRdbLogFuncType *logfn;

    GciSetLoggerFlush(TRUE);
    logfn = (GsRdbLogFuncType *)GsRdbGetLogger();
    if (logfn != (GsRdbLogFuncType *)NULL) {
      (*logfn)(str);
    }
}

/* This verion prefixes the string "[DBG] " to the message
 * Used in the various DbgLog* functions
 */
void GsDbgLogStr(const char * str )
{
      char msg[300];
      snprintf(msg, 300, "[DBG] %s", str);
      _GsDbgLogStr(msg);
}

