!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! GemConnect for Oracle Smalltalk-DB code
!
! This module must be filed in to a GemStone database's SystemUser account
! before GemConnect for Oracle can be used.
!
! Copyright (C) 1986-2025 GemTalk Systems.
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

display oops
display resultcheck
limit bytes 200
level 0

iferror input pop
expectvalue nil
printit
  System myUserProfile userId = 'SystemUser' ifFalse: [
     nil halt: 'This script must be loaded from the SystemUser account'
  ]
%

output push gsoraapifilein25.log only

iferror stack
expectvalue nil
expectvalue %OrderedCollection
doit
(UserGlobals includesKey: #RdbAPIHistory) ifFalse: [
    UserGlobals at: #RdbAPIHistory put: OrderedCollection new
].
%

expectvalue %String
expectvalue %Unicode7
doit
RdbAPIHistory add: 'Load of GemConnect for Oracle started on ', DateTime now printString
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - Errors.gs
!
! ========================================================================
!
! Install error dictionaries
!
!
expectvalue nil
doit

   | rdbErrors rdbSymbols errors langDict |

   rdbErrors := Globals at: #GsRdbErrors put: (LanguageDictionary new).
   rdbErrors at: #English put: (Array new).
   rdbSymbols := Globals at: #GsRdbErrorSymbols put: (SymbolDictionary new).

   " NOTE:  the order of errors determines error numbers.  C code depends
            upon this ordering. "

   errors :=  #(
"1"     #aggColumnName #('Column names of the form aggXX are not allowed in results, connection: ' 1 ' columnName: ' 4)
"2"     #badColumnMap #('A column map entry with too few or non-character collection elements was encountered, entry: "' 4 '"')
"3"     #badConnCacheName #('There is already a connection cached with key "' 4 '"')
"4"     #invalidConnection #('This relational database connection is not valid, connection: ' 1)
"5"     #invalidStream #('This relational database result stream is not valid, stream: ' 2)
"6"     #oracleError #('An unexpected error was encountered during Oracle processing, connection: ' 1 ' stream: ' 2 ' details: ' 4)
"7"     #queryError #('An error was encountered while performing a relational query, connection: ' 1 ' query: ' 3 ' details: ' 4)
"8"     #readError #('An error was encountered while reading relational data, stream: ' 2)
"9"     #streamAtEnd #('End of stream encountered while reading relational data, stream: ' 2)
"10"     #obsoleteError #('[obsolete error (used to be #sybaseError)]')
"11"     #unmappedColumnInResult #('A column map was given but one or more result columns were not represented in it, columnMap: ' 4)
"12"     #failedToInitialize #('The relational database interface could not be initialized: ' 4)
"13"     #noColumnMap #('There is no column map available to perform the requested action with this tuple class, class: "' 4 '"')
"14"     #noPrimaryKeyMap #('There is no primary key map available to perform the requested action with this tuple class, class: "' 4 '"')
"15"     #noTableName #('There is no table name available to perform the requested action with this tuple class, class: "' 4 '"')
"16"     #invalidColumnMap #('A column map was passed which contained elements which were not of the expected class Array, connection: ' 1 ' columnMap: ' 4)
"17"     #columnAllocError #('A buffer could not be allocated to bind the values for a column in the result set, connection: ' 1 ' columnName: ' 4)
"18"     #connectionKilled #('The relational database terminated the connection while processing the string ' 4)
"19"     #noChangeNotification #('Object change notification is not supported on this version of GemStone')
"20"     #invalidTupleInstance #('A tuple was passed to a write stream that is not the expected class, connection: ' 1 ' stream: ' 2 ' tuple: '4)
"21"     #invalidSql #('The SQL statement is invalid, connection: ' 1 ' query: ' 3)
"22"     #invalidTableName #('The table ' 4 ' does not exist in this relational database, connection: ' 1)
"23"     #columnBindingError #('Could not bind all columns in columnMap with relational table, connection: ' 1 ' query: ' 3)
"24"     #flushError #('Problem writing to relational database during write stream flush, connection: ' 1 ' stream: ' 2 ' details: ' 4)
"25"     #typeConversionError #('Cannot convert GS Object or Oracle column data, connection: ' 1 ' stream: ' 2 ' details: ' 4)
"26"     #internalError #('GemConnect internal error, connection: ' 1 ' stream: ' 2 ' details: ' 4)
"27"     #lobAllocationError #('Cannot allocate space for a GsOracleLOB object, connection: ' 1 ' stream: ' 2)
"28"     #lobUsageError #('Incorrect operation on Oracle BLOB/CLOB object, connection: ' 1 ' stream: ' 2)
"29"     #bindingSetupError #('Error in select binding specification, connection: ' 1 ' query: ' 3 ' details: ' 4)
"30"     #bindingValueError #('Error in binding value assignment, connection: ' 1 ' stream: ' 2 ' details: ' 4)
"31"     #selectUnboundError #('Attempting to read from a select cursor that has not yet had values assigned to its bindings, connection: ' 1 ' stream: '2)
"32"     #keyfileError #('Keyfile does not support GemConnect')
"33"     #timeoutError #('Oracle OCI operation timed out, connection: ' 1 ' stream: '2)
"34"     #oralibError #('Oracle library load error, path: ' 1 ' details: ' 2)
"35"     #credentialsError #('Invalid credentials - must be either #RDBMS or #EXT')
    ).

   langDict := rdbErrors at: #English.
   1 to: (errors size) by: 2 do: [:i |
      langDict add: (errors at: i+1).
      rdbSymbols at: (errors at: i) put: (langDict size).
      ].
^ nil
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - commonclasses.gs
!
! ========================================================================
!
! Common GemConnect classes
!
expectvalue nil
doit
(Globals includesKey: #GsRdbConnManager) ifFalse: [
  Object subclass: 'GsRdbConnManager'
    instVarNames: #( 'connNameCache' 'loadedUAArray' 'keepArray' 'other' )
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
! 49507:  Change superclass from Object to Array
expectvalue nil
doit
(Globals includesKey: #GsRdbChangeNotifObj) ifFalse: [
  Array subclass: 'GsRdbChangeNotifObj'
    instVarNames: #()
    classVars: #()
    classInstVars: #( 'soleInstance')
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
expectvalue nil
doit
(Globals includesKey: #GsRdbParameters) ifFalse: [
  Object subclass: 'GsRdbParameters'
    instVarNames: #( 'server' 'userName'
                     'password' 'textLimit' 'other')
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
expectvalue nil
doit
(  (Globals includesKey: #GsRdbConnection) and:
  [(Globals at: #GsRdbConnection) instVarNames includes: #parameters]) 
ifFalse: [
  "41056: if defined as a kernel class, remove from dictionary first.."
  Globals removeKey: #GsRdbConnection ifAbsent: [ ]. 
  AbstractSession subclass: 'GsRdbConnection'
    instVarNames: #( 'parameters' 'queue')
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
!
! New for 2.0: since we need to add new instvar 'sql' to the class,
! modify the includesKey: check so we'll create the class again 
! (making a new version) if we need to.
!
! New for 2.4.4: added 'bindInfo' instvar
!
expectvalue true
expectvalue nil
doit
(((Globals includesKey: #GsRdbReadStream) and:
  [(Globals at: #GsRdbReadStream) instVarNames includes: #sql]) and:
    [(Globals at: #GsRdbReadStream) instVarNames includes: #bindInfo])
ifTrue: [
  Globals at: #GsRdbReadStreamClean put: true ]
ifFalse: [
  Stream subclass: 'GsRdbReadStream'
    instVarNames: #( 'tupleClass' 'connection' 'sql' 'bindInfo')
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
expectvalue nil
doit
(((Globals includesKey: #GsRdbWriteStream) and:
  [ Globals includesKey: #GsRdbReadStreamClean ]) and:
    [(Globals at: #GsRdbWriteStream) instVarNames includes: #dependencyList])
ifFalse: [
  GsRdbReadStream subclass: 'GsRdbWriteStream'
    instVarNames: #('dependencyList')
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
expectvalue nil
doit
((Globals includesKey: #GsRdbSelectStream) and:
  [ Globals includesKey: #GsRdbReadStreamClean ])
ifFalse: [
  GsRdbReadStream subclass: 'GsRdbSelectStream'
    instVarNames: #('bindValues')
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
expectvalue nil
doit
((Globals includesKey: #GsRdbBlockStream) and:
  [ Globals includesKey: #GsRdbReadStreamClean ])
ifFalse: [
  GsRdbWriteStream subclass: 'GsRdbBlockStream'
    instVarNames: #()
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
expectvalue nil
doit
(Globals includesKey: #GsRdbError) ifFalse: [
  Error subclass: 'GsRdbError'
    instVarNames: #(connection stream sql rdbErrDetails)
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
! Code to insure bindInfo directional symbols are created..
expectvalue nil
doit
Globals removeKey: #GsRdbReadStreamClean ifAbsent: [].
Globals at: #GsRdbBindDirections put: (Array with: #IN with: #OUT with: #INOUT).
^ nil
%



! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - orclasses.gs
!
! ========================================================================
!
! GemConnect for Oracle classes
!
! 49926: add 'credentials' instvar
! (includes logic to recreate class if old version doesn't have it)
!
expectvalue nil
doit
| class instVars|
class := Globals at: #GsOracleParameters ifAbsent: [nil].
(class isNil) 
ifFalse: [
  instVars := class instVarNames.
  (instVars includes: #credentials)
  ifFalse: [ class := nil ]].
(class isNil) ifTrue: [
  GsRdbParameters subclass: 'GsOracleParameters'
    instVarNames: #( 'autoCommit' 'credentials')
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
!
! New for 2.0:  since we need to add new instvars 'batchSize' and 
! 'charConversion' to the class, modify the includesKey: check so we 
! will create the class again (making a new version) if we need to.
!
! New for 2.1:  do the same for new instvars 'lobLimit' (34451)
! and 'textLimit' (copied from parameters)
!
! New for 2.2.4: do the same for new instvars [n]charSet and [n]charGS 
! (note: charConversion removed) (42668)
!
! New for 2.4: ditto for new instvars [n]clobSet (48434)
!
! New for 2.4.1: ditto for new instvar returnDateAndTime (48654)
!
! New for 2.5:  ditto for new instvar ociTimeout (49072)
!
! New for 2.5.2: numberAsScaledDecimal (51439)

expectvalue nil
doit
| class instVars|
class := Globals at: #GsOracleConnection ifAbsent: [nil].
"46977: force rebuild if class skew to GsRdbConnection exists"
class isNil
  ifFalse: [
    (class superClass = GsRdbConnection)
      ifFalse: [ class := nil ]
].
class isNil
  ifFalse: [ | newVars |
    newVars := IdentitySet withAll:
      { #textLimit .
        #lobLimit .
        #batchSize .
        #charSet .
        #clobSet .
        #returnDateAndTime .
        #ociTimeout .
        #numberAsScaledDecimal "51439"
      }.    
  instVars := class instVarNames asIdentitySet.
  "set intersection (*): rebuild GsRdbConnection if any inst var above is missing"
  newVars = (instVars * newVars)
    ifFalse:[ class := nil ].    
 ].
class isNil ifTrue: [
  GsRdbConnection subclass: 'GsOracleConnection'
    instVarNames: #('textLimit' 'lobLimit' 'batchSize' 
                    'charSet' 'ncharSet' 'clobSet' 'nclobSet'
                    'charGS' 'ncharGS' 'utfErr' 'returnDateAndTime' 'ociTimeout'
		    'numberAsScaledDecimal' )
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%


! 35541
expectvalue nil
doit
(Globals includesKey: #GsOracleLOB) ifFalse: [
  Object subclass: 'GsOracleLOB'
    instVarNames: #('connection' 'index')
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
expectvalue nil
doit
(Globals includesKey: #GsOracleBLOB) ifFalse: [
  GsOracleLOB subclass: 'GsOracleBLOB'
    instVarNames: #()
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
expectvalue nil
doit
(Globals includesKey: #GsOracleCLOB) ifFalse: [
  GsOracleLOB subclass: 'GsOracleCLOB'
    instVarNames: #()
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
expectvalue nil
doit
(Globals includesKey: #GsOracleNCLOB) ifFalse: [
  GsOracleLOB subclass: 'GsOracleNCLOB'
    instVarNames: #()
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
].
^ nil
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - GsOracleParameters.gs
!
! ========================================================================
!
! Class 'GsOracleParameters'
!

removeallmethods GsOracleParameters
removeallclassmethods GsOracleParameters

!
! Class Category 'Accessing'
!
category: 'Accessing'
classmethod: GsOracleParameters
connectionClass

"Returns the connection class this parameter class is to be used
 with.

ARGUMENTS:

   none

RETURN VALUE:

   <Class>

ERRORS:

   none
"

   ^GsOracleConnection
%
!
! Instance Category 'Updating'
!
category: 'Updating'
method: GsOracleParameters
autoCommit: bool

"Sets the auto commit mode value, which indicates whether the Oracle
 database connection commits automatically after a command is executed.

ARGUMENTS:

   bool      <Boolean> captured

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   autoCommit := bool
%
! 49926
category: 'Updating'
method: GsOracleParameters
credentials: aSymbol

"Sets the credentials value, which indicates how Oracle will authenticate
 the connection.  Possible values include:

 #RDBMS - authenticate using DB username / password
 #EXT - authenticate using some external mechanism

RETURN VALUE:

   unspecified

ERRORS:

   GsRdbError/credentialsError

"
   ((aSymbol == #RDBMS) or: [aSymbol == #EXT]) ifFalse: [
      (GsRdbError newWithNum: (GsRdbErrorSymbols at: #credentialsError)
                  args: { nil . nil . nil . nil }) signal ].
   credentials := aSymbol
%
!
! Instance Category 'Accessing'
!
category: 'Accessing'
method: GsOracleParameters
autoCommit

"Returns the auto commit mode value, which indicates whether the Oracle
 database connection commits automatically after a command is executed.

ARGUMENTS:

   none

RETURN VALUE:

   <Boolean> state

ERRORS:

   none
"

   ^autoCommit
%
! 49926
category: 'Accessing'
method: GsOracleParameters
credentials

"Returns credentials value, which indicates how Oracle will authenticate
 the connection.  Possible values include:

 #RDBMS - authenticate using DB username / password
 #EXT - authenticate using some external mechanism
 default/nil is #RDBMS

ARGUMENTS:

   none

RETURN VALUE:

   Setting of credentials

ERRORS:

   none
"

   ^ credentials
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - GsOracleConnection.gs
!
! ========================================================================
!
! Class 'GsOracleConnection'
!

removeallmethods GsOracleConnection
removeallclassmethods GsOracleConnection

!
! Class Category 'Instance Creation'
!
category: 'Instance Creation'
classmethod: GsOracleConnection
newWithParameters: parameterObj

"Creates a new connection instance and initializes it with a copy of the 
 given parameter object (to avoid unintended consequences if the parameter 
 values are changed).  
 
No attempt is made to connect to the relational database.

ARGUMENTS:

   parameterObj      <GsRdbParameters subclass> shallow copied

RETURN VALUE:

   <GsRdbConnection> new

ERRORS:

   none
"

   | conn |
   conn := super newWithParameters: parameterObj.
   conn textLimit: parameterObj textLimit.
   ^ conn
%
!
! Instance Category 'Synchronized Transaction Management'
!
category: 'Synchronized Transaction Management'
method: GsOracleConnection
_voteToCommit

"Returns an integer value indicating whether the current transaction on this
 connection can commit successfully. Values returned are the following:

   0    The session was read-only and the connection can commmit successfully
   1    The session was read/write and the connection can commmit successfully
   2    The connection cannot commit successfully

 Note: At this time no support has been added to incorporate an actual voting
 mechanism for Oracle transactions, so the value returned is always 1.

ARGUMENTS:

   none

RETURN VALUE:

   <Integer>
   0 (read-only, vote yes)
   1 (write, vote yes)
   2 (vote no).

ERRORS:

   none
"

   ^1
%
!
! Instance Category 'Transaction Management'
!
category: 'Transaction Management'
method: GsOracleConnection
beginTransaction

"Returns false, as Oracle does not support a begin transaction operation.

ARGUMENTS:

   none

RETURN VALUE:

   <Boolean> success indication

ERRORS:

   GsRdbError/invalidConnection
"

   ^false
%
!
! Instance Category 'Stream User Actions'
!
category: 'Stream User Actions'
method: GsOracleConnection
_rsatEndAction

   ^#GsOraStreamAtEnd
%
method: GsOracleConnection
_rsfreeStreamAction

   ^#GsOraFreeStream
%
method: GsOracleConnection
_rsgetAllAction

   ^#GsOraGetAllRows
%
method: GsOracleConnection
_rsgetRowAction

   ^#GsOraGetRow
%
method: GsOracleConnection
_rsputRowAction

   ^#GsOraPutRow
%
method: GsOracleConnection
_rsinstallCmapAction

   ^#GsOraInstallColumnMap
%
method: GsOracleConnection
_rsisValidAction

   ^#GsOraStreamIsValid
%
method: GsOracleConnection
_rsivcAction

   ^#GsOraInstVarsAndConstraints
%
method: GsOracleConnection
_rsnextResultsAction

   ^#GsOraNextResultSet
%
method: GsOracleConnection
_rsposAction

   ^#GsOraStreamPos
%
method: GsOracleConnection
_rssetToEndAction

   ^#GsOraStreamSetToEnd
%
method: GsOracleConnection
_rsskipRowsAction

   ^#GsOraSkipRows
%
method: GsOracleConnection
_rsflushStreamAction

   ^#GsOraFlushWriteStream
%
!
! Instance Category 'Command Execution'
!
category: 'Command Execution'
method: GsOracleConnection
execute: commandString

"Oracle does not support any non-cursored type of execution so this method
 calls its cursored equivalent: #openCursorOn: .

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured

RETURN VALUE:

   <GsRdbReadStream> new

ERRORS:

   GsRdbError/columnAllocError
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
"

   ^self openCursorOn: commandString
%
method: GsOracleConnection
openCursorOn: commandString

"Sends the given command string to the Oracle database for cursored execution.
 The result set is prepared for reading through an instance of GsRdbReadStream.

 The read stream returned by this method has instance variable and constraint
 information and can be instructed to create a tuple class. The default
 behaviour of the read stream, however, is to return result information in
 instances of OrderedCollection.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured

RETURN VALUE:

   <GsRdbReadStream> new

ERRORS:

   GsRdbError/columnAllocError
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
"

   ^self openCursorOn: commandString tupleClass: nil columnMapping: nil
%
method: GsOracleConnection
openCursorOn: commandString tupleClass: tupleClass

"Sends the given command string to the Oracle database for cursored execution.
 The result set is prepared for reading through an instance of GsRdbReadStream
 with the given tuple class.  Note that it is an error to pass a class that
 cannot accept the results of the execution.  Constraint errors or
 doesNotUnderstand errors will be signaled on fetching information with a class
 that cannot accept the results of execution.

 Column mapping information, if present in the tuple class, will be used in
 installing results in instances of the class.  The updating selectors specified
 in the mapping will be used to send column values to the instances of the tuple
 class as information is read from the Oracle database.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   tupleClass         <Class> captured

RETURN VALUE:

   <GsRdbReadStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"

   | columnMap |

   (tupleClass notNil) ifTrue: [
      columnMap := tupleClass rdbColumnMapping
      ].

   ^self openCursorOn: commandString tupleClass: tupleClass
         columnMapping: columnMap
%
method: GsOracleConnection
openCursorOn: commandString tupleClass: tupleClass columnMapping: columnMap

"Sends the given command string to the Oracle database for cursored execution.
 The result set is prepared for reading through an instance of GsRdbReadStream
 with the given tuple class.  Note that it is an error to pass a class that
 cannot accept the results of the execution.  Constraint errors or
 doesNotUnderstand errors will be signaled on fetching information with a class
 that cannot accept the results of execution.

 Column mapping information, provided by the given column map, will be used in
 installing results in instances of the class.  The updating selectors specified
 in the mapping will be used to send column values to the instances of the tuple
 class as information is read from the Oracle database.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   tupleClass         <Class> captured
   columnMap          <Array> copied

RETURN VALUE:

   <GsRdbReadStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"

   | stream sql |

   self class _loadUALib.
   self class _validateColumnMap: columnMap conn: self.
   sql := self prepareSqlString: commandString.
   stream := System userAction: #GsOraInitStream with: self
                    with: sql with: true
                    with: true with: GsRdbReadStream
                    with: columnMap with: false with: nil.
   (tupleClass notNil) ifTrue: [
      stream _tupleClass: tupleClass.
      ].
   stream _sql: commandString.
   ^stream
%
method: GsOracleConnection
execute: commandString tupleClass: tupleClass

"Oracle does not support any non-cursored type of execution so this method
 calls its cursored equivalent: #openCursorOn:tupleClass .

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   tupleClass         <Class> captured

RETURN VALUE:

   <GsRdbReadStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"

   ^self openCursorOn: commandString tupleClass: tupleClass
%
method: GsOracleConnection
execute: commandString tupleClass: tupleClass columnMapping: columnMap

"Oracle does not support any non-cursored type of execution so this method
 calls its cursored equivalent: #openCursorOn:tupleClass:columnMapping: .

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   tupleClass         <Class> captured
   columnMap          <Array> copied

RETURN VALUE:

   <GsRdbReadStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"

   ^self openCursorOn: commandString tupleClass: tupleClass
         columnMapping: columnMap
%
method: GsOracleConnection
execute: commandString tupleClassName: aName inDictionary: symbolDict

"Oracle does not support any non-cursored type of execution so this method
 calls its cursored equivalent: #openCursorOn:tupleClassName:inDictionary: .

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   aName              <Symbol> captured
   symbolDict         <SymbolDictionary> uncaptured

RETURN VALUE:

   <GsRdbReadStream> new

ERRORS:

   GsRdbError/columnAllocError
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
"

   ^self openCursorOn: commandString tupleClassName: aName
         inDictionary: symbolDict
%
method: GsOracleConnection
executeNoResults: commandString

"Sends the given command string to the Oracle database for execution and ignores
 any result set.  Returns true if the execution of the command was successful
 and generates an error if not.

ARGUMENTS:

   commandString   <CharacterCollection> uncaptured

RETURN VALUE:

   <Boolean> success indication

ERRORS:

   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
"

   | sql |
   self class _loadUALib.
   sql := self prepareSqlString: commandString.
   ^System userAction: #GsOraInitStream with: self
           with: sql with: false
           with: false with: GsRdbReadStream
           with: nil with: false with: nil
%
method: GsOracleConnection
executeReturnRowsAffected: commandString

"Sends the given logical command string to the Oracle database for execution.  (
Logical commands are insert, update, or delete commands.)

ARGUMENTS:

   commandString   <CharacterCollection> uncaptured

RETURN VALUE:

   the number of rows affected by the logical command

ERRORS:

   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
"

   | sql |
   self class _loadUALib.
   sql := self prepareSqlString: commandString.
   ^System userAction: #GsOraInitStream with: self
           with: sql with: false
           with: false with: GsRdbReadStream
           with: nil with: true with: nil
%
method: GsOracleConnection
openCursorOn: commandString tupleClassName: aName inDictionary: symbolDict

"Sends the given command string to the Oracle database for cursored execution.
 The result set is prepared for reading through an instance of GsRdbReadStream.
 A new tuple class with the given name is created for reading results,
 installed in the stream, and placed in the given dictionary.

 The tuple class will have constraints created for it that match the type/class
 conversions specified in the C layer of the API.  Instance variable names for
 the class will be translations of the corresponding result set column names, if
 they are returned by the Oracle database.  Otherwise they are named 'iv1',
 'iv2', 'iv3', and so forth. Computed columns that are unnamed are given the
 names 'agg1', 'agg2', 'agg3', and so on.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   aName              <Symbol> captured
   symbolDict         <SymbolDictionary> uncaptured

RETURN VALUE:

   <GsRdbReadStream> new

ERRORS:

   GsRdbError/columnAllocError
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
"

   | stream |

   stream := self openCursorOn: commandString.
   stream createTupleClassNamed: aName inDictionary: symbolDict.
   ^stream
%
! 
! Methods for openWriteCursor variations
!
method: GsOracleConnection
openDeleteCursorOn: tupleClass

"Open a write stream cursor to be used for performing DELETE SQL statements.

ARGUMENTS:

   tupleClass         <Class> captured

RETURN VALUE:

   <GsRdbWriteStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"

   ^self openDeleteCursorOn: tupleClass
		keyMapping: (tupleClass rdbPrimaryKeyMaps)
%
method: GsOracleConnection
openDeleteCursorOn: tupleClass keyMapping: keyMap

"Open a write stream cursor to be used for performing DELETE SQL statements.
 Allows user to override the keyMapping.

ARGUMENTS:

   tupleClass         <Class> captured
   keyMap             <Array> copied

   NOTE: keyMap entries must be in same order as rows of table.

RETURN VALUE:

   <GsRdbWriteStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"

   ^ self openDeleteCursorOn: tupleClass 
		keyMapping: keyMap 
		tableName: (tupleClass rdbTableName)
%
method: GsOracleConnection
openDeleteCursorOn: tupleClass keyMapping: keyMap tableName: tableName

"Open a write stream cursor to be used for performing DELETE SQL statements.
 Allows user to override the keyMapping and tableName.

ARGUMENTS:

   tupleClass      <Class> captured
   keyMap          <Array> copied
   tableName       <String> copied

   NOTE: keyMap entries must be in same order as rows of table.

RETURN VALUE:

   <GsRdbWriteStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"
   | keyNames sql bindInfo |
   self class _validateColumnMap: keyMap conn: self.
   keyNames := keyMap collect: [ :x | x at: 1 ].
   bindInfo := Array with: tableName.
   bindInfo addAll: keyNames.
   sql := self class generateBindSQLDeleteForTable: tableName keys: keyNames.
   ^ self openWriteCursorOn: sql 
		tupleClass: tupleClass 
		columnMapping: keyMap 
		bindInfo: bindInfo
%
method: GsOracleConnection
openInsertCursorOn: tupleClass

"Open a write stream cursor to be used for performing INSERT SQL statements.

ARGUMENTS:

   tupleClass         <Class> captured

RETURN VALUE:

   <GsRdbWriteStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"

   ^self openInsertCursorOn: tupleClass
		columnMapping: (tupleClass rdbColumnMapping)
%
method: GsOracleConnection
openInsertCursorOn: tupleClass columnMapping: columnMap

"Open a write stream cursor to be used for performing INSERT SQL statements.
 Allows user to override the columnMapping.

ARGUMENTS:

   tupleClass         <Class> captured
   columnMap          <Array> copied

   NOTE: columnMap entries must be in same order as rows of table.

RETURN VALUE:

   <GsRdbWriteStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"
   ^ self openInsertCursorOn: tupleClass 
		columnMapping: columnMap 
		tableName: (tupleClass rdbTableName)
%
method: GsOracleConnection
openInsertCursorOn: tupleClass columnMapping: columnMap tableName: tableName

"Open a write stream cursor to be used for performing INSERT SQL statements.
 Allows user to override the columnMapping and tableName.

ARGUMENTS:

   tupleClass         <Class> captured
   columnMap          <Array> copied
   tableName          <String> captured

   NOTE: columnMap entries must be in same order as rows of table.

RETURN VALUE:

   <GsRdbWriteStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"
   | colNames sql bindInfo |
   self class _validateColumnMap: columnMap conn: self.
   colNames := columnMap collect: [ :x | x at: 1 ].
   bindInfo := Array with: tableName.
   bindInfo addAll: colNames.
   sql := self class generateBindSQLInsertForTable: tableName columns: colNames.
   ^ self openWriteCursorOn: sql 
		tupleClass: tupleClass 
		columnMapping: columnMap 
		bindInfo: bindInfo
%
method: GsOracleConnection
openUpdateCursorOn: tupleClass

"Open a write stream cursor to be used for performing Update SQL statements.

ARGUMENTS:

   tupleClass         <Class> captured

RETURN VALUE:

   <GsRdbWriteStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"

   ^self openUpdateCursorOn: tupleClass
         columnMapping: (tupleClass rdbColumnMapping)
%
method: GsOracleConnection
openUpdateCursorOn: tupleClass columnMapping: columnMap

"Open a write stream cursor to be used for performing UPDATE SQL statements.
 Allows user to override the columnMapping.

ARGUMENTS:

   tupleClass         <Class> captured
   columnMap          <Array> copied

   NOTE: columnMap entries must be in same order as rows of table

RETURN VALUE:

   <GsRdbWriteStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"
   | tableName |
   tableName := tupleClass rdbTableName.
   ^ self openUpdateCursorOn: tupleClass 
	columnMapping: columnMap 
	keyMapping: (tupleClass rdbPrimaryKeyMaps)
%
method: GsOracleConnection
openUpdateCursorOn: tupleClass columnMapping: columnMap keyMapping: keyMap

"Open a write stream cursor to be used for performing UPDATE SQL statements.
 Allows user to override the columnMapping and keyMapping.

ARGUMENTS:

   tupleClass         <Class> captured
   columnMap          <Array> copied
   keyMap             <Array> copied

   NOTE: columnMap entries must include keyMap 
   	 and be in same order as rows of table

RETURN VALUE:

   <GsRdbWriteStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"
   ^ self openUpdateCursorOn: tupleClass 
	columnMapping: columnMap 
	keyMapping: keyMap 
	tableName: (tupleClass rdbTableName)
%
method: GsOracleConnection
openUpdateCursorOn: tupleClass columnMapping: columnMap keyMapping: keyMap tableName: tableName

"Open a write stream cursor to be used for performing UPDATE SQL statements.
 Allows user to override the columnMapping, keyMapping, and tableName.

ARGUMENTS:

   tupleClass         <Class> captured
   columnMap          <Array> copied
   keyMap             <Array> copied
   tableName          <String> captured

   NOTE: columnMap entries must include keyMap 
         and be in same order as rows of table

RETURN VALUE:

   <GsRdbWriteStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"
   | colNames keyNames sql bindInfo |
   self class _validateColumnMap: columnMap conn: self.
   self class _validateColumnMap: keyMap conn: self.
   colNames := columnMap collect: [ :x | x at: 1 ].
   keyNames := keyMap collect: [ :x | x at: 1 ].
   bindInfo := Array with: tableName.
   bindInfo addAll: colNames.
   keyNames do: [ :x | colNames remove: x ifAbsent: [ ] ].
   sql := self class generateBindSQLUpdateForTable: tableName columns: colNames keys: keyNames.
   ^ self openWriteCursorOn: sql tupleClass: tupleClass columnMapping: columnMap bindInfo: bindInfo
%
method: GsOracleConnection
openWriteCursorOn: commandString tupleClass: tupleClass columnMapping: columnMap bindInfo: bindInfo

"Open a write stream cursor to be used for executing bound SQL statements.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   tupleClass         <Class> captured
   columnMap          <Array> copied
   bindInfo           <Array> copied
                      element 1 is the DB Table Name
                      elements 2..n are the names of columns to bind

   NOTE: columnMap entries must be in same order as rows of table

RETURN VALUE:

   <GsRdbWriteStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
"

   | stream sql |

   self class _loadUALib.
   self class _validateColumnMap: columnMap conn: self.
   sql := self prepareSqlString: commandString.
   stream := System userAction: #GsOraInitStream with: self
                    with: sql with: tupleClass
                    with: true with: GsRdbWriteStream
                    with: columnMap with: false with: bindInfo.
   (tupleClass notNil) ifTrue: [
      stream _tupleClass: tupleClass.
      ].
   stream _sql: commandString.
   ^stream
%
method: GsOracleConnection
openSelectCursorOn: commandString tupleClass: tupleClass columnMapping: columnMap bindInfo: bindInfo

"Open a select stream cursor to be used for executing bound SELECT statements.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   tupleClass         <Class> captured
   columnMap          <Array> copied
   bindInfo           <Array> copied
                      1:    DB Table Name (String)
                      2..n: Arrays containing details for bind variables:
                            1:  Name, starting with "":"" (String)
                            2:  Type: Integer, Float, or String

   NOTE: columnMap entries must be in same order as rows of table

RETURN VALUE:

   <GsRdbSelectStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
   GsRdbError/bindingSetupError
"

   | stream sql |

   self class _loadUALib.
   self class _validateColumnMap: columnMap conn: self.
   sql := self prepareSqlString: commandString.
   self class _validateBindInfo: bindInfo against: sql isSelect: true.
   stream := System userAction: #GsOraInitStream with: self
                    with: sql with: tupleClass
                    with: true with: GsRdbSelectStream
                    with: columnMap with: false with: bindInfo.
   (tupleClass notNil) ifTrue: [
      stream _tupleClass: tupleClass.
      ].
   stream _sql: commandString.
   ^stream
%
method: GsOracleConnection
openSelectCursorOn: commandString columnMapping: columnMap bindInfo: bindInfo

"Open a select stream cursor to be used for executing bound SELECT statements.
 See #openSelectCursorOn:tupleClass:columnMapping:bindInfo: for details."

^ self openSelectCursorOn: commandString tupleClass: nil columnMapping: columnMap bindInfo: bindInfo
%
method: GsOracleConnection
openSelectCursorOn: commandString tupleClass: tupleClass bindInfo: bindInfo

"Open a select stream cursor to be used for executing bound SELECT statements.
 See #openSelectCursorOn:tupleClass:columnMapping:bindInfo: for details."

^ self openSelectCursorOn: commandString tupleClass: tupleClass columnMapping: nil bindInfo: bindInfo
%
method: GsOracleConnection
openSelectCursorOn: commandString bindInfo: bindInfo

"Open a select stream cursor to be used for executing bound SELECT statements.
 See #openSelectCursorOn:tupleClass:columnMapping:bindInfo: for details."

^ self openSelectCursorOn: commandString tupleClass: nil columnMapping: nil bindInfo: bindInfo
%
method: GsOracleConnection 
openSelectCursorOn: commandString 
              bind: bindName type: bindType

"Open a select stream cursor to be used for executing bound SELECT statements.
 See #openSelectCursorOn:tupleClass:columnMapping:bindInfo: for details."

  | bindInfo |
  bindInfo := Array with: nil
                    with: (Array with: bindName with: bindType).
^ self openSelectCursorOn: commandString 
       tupleClass: nil 
       columnMapping: nil 
       bindInfo: bindInfo
%
method: GsOracleConnection 
openSelectCursorOn: commandString tupleClass: tupleClass 
              bind: bindName type: bindType

"Open a select stream cursor to be used for executing bound SELECT statements.
 See #openSelectCursorOn:tupleClass:columnMapping:bindInfo: for details."

  | bindInfo |
  bindInfo := Array with: nil
                    with: (Array with: bindName with: bindType).
^ self openSelectCursorOn: commandString 
       tupleClass: tupleClass
       columnMapping: nil 
       bindInfo: bindInfo

%
method: GsOracleConnection 
openSelectCursorOn: commandString columnMapping: columnMap
                    bind: bindName type: bindType

"Open a select stream cursor to be used for executing bound SELECT statements.
 See #openSelectCursorOn:tupleClass:columnMapping:bindInfo: for details."

  | bindInfo |
  bindInfo := Array with: nil
                    with: (Array with: bindName with: bindType).
^ self openSelectCursorOn: commandString 
       tupleClass: nil 
       columnMapping: columnMap
       bindInfo: bindInfo

%
method: GsOracleConnection 
openSelectCursorOn: commandString tupleClass: tupleClass columnMapping: columnMap
                    bind: bindName type: bindType

"Open a select stream cursor to be used for executing bound SELECT statements.
 See #openSelectCursorOn:tupleClass:columnMapping:bindInfo: for details."

  | bindInfo |
  bindInfo := Array with: nil
                    with: (Array with: bindName with: bindType).
^ self openSelectCursorOn: commandString 
       tupleClass: tupleClass
       columnMapping: columnMap
       bindInfo: bindInfo
%
method: GsOracleConnection 
openSelectCursorOn: commandString
                    bind: bindName1 type: bindType1
                    bind: bindName2 type: bindType2

"Open a select stream cursor to be used for executing bound SELECT statements.
 See #openSelectCursorOn:tupleClass:columnMapping:bindInfo: for details."

  | bindInfo |
  bindInfo := Array with: nil
                    with: (Array with: bindName1 with: bindType1)
                    with: (Array with: bindName2 with: bindType2).
^ self openSelectCursorOn: commandString 
       tupleClass: nil 
       columnMapping: nil 
       bindInfo: bindInfo
%
method: GsOracleConnection 
openSelectCursorOn: commandString tupleClass: tupleClass
                    bind: bindName1 type: bindType1
                    bind: bindName2 type: bindType2

"Open a select stream cursor to be used for executing bound SELECT statements.
 See #openSelectCursorOn:tupleClass:columnMapping:bindInfo: for details."

  | bindInfo |
  bindInfo := Array with: nil
                    with: (Array with: bindName1 with: bindType1)
                    with: (Array with: bindName2 with: bindType2).
^ self openSelectCursorOn: commandString 
       tupleClass: tupleClass
       columnMapping: nil 
       bindInfo: bindInfo
%
method: GsOracleConnection 
openSelectCursorOn: commandString columnMapping: columnMap
                    bind: bindName1 type: bindType1
                    bind: bindName2 type: bindType2

"Open a select stream cursor to be used for executing bound SELECT statements.
 See #openSelectCursorOn:tupleClass:columnMapping:bindInfo: for details."

  | bindInfo |
  bindInfo := Array with: nil
                    with: (Array with: bindName1 with: bindType1)
                    with: (Array with: bindName2 with: bindType2).
^ self openSelectCursorOn: commandString 
       tupleClass: nil 
       columnMapping: columnMap
       bindInfo: bindInfo
%
method: GsOracleConnection
 openSelectCursorOn: commandString tupleClass: tupleClass columnMapping: columnMap
                    bind: bindName1 type: bindType1
                    bind: bindName2 type: bindType2

"Open a select stream cursor to be used for executing bound SELECT statements.
 See #openSelectCursorOn:tupleClass:columnMapping:bindInfo: for details."

  | bindInfo |
  bindInfo := Array with: nil
                    with: (Array with: bindName1 with: bindType1)
                    with: (Array with: bindName2 with: bindType2).
^ self openSelectCursorOn: commandString 
       tupleClass: tupleClass
       columnMapping: columnMap
       bindInfo: bindInfo
%
method: GsOracleConnection
openBlockCursorOn: commandString bindInfo: bindInfo

"Open a block stream cursor to be used for executing bound sql block 
 statements (starting with BEGIN or DECLARE).

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   bindInfo           <Array> copied
                      1:    DB Table Name (String)
                      2..n: Arrays containing details for bind variables:
                            1:  Name, starting with "":"" (String)
                            2:  Type: Integer, Float, or String
                            3:  (optional) Direction:  #IN, #OUT, or #INOUT
                            4:  (optional) MaxSize (for Strings only)
                                Defaults to connection textLimit (65532)

RETURN VALUE:

   <aGsRdbBlockStream> new

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidConnection
   GsRdbError/queryError
   GsRdbError/readError
   GsRdbError/unmappedColumnInResult
   GsRdbError/bindingSetupError
"

   | result sql |

   self class _loadUALib.
   sql := self prepareSqlString: commandString.
   self class _validateBindInfo: bindInfo against: sql isSelect: false.
   result := System userAction: #GsOraInitStream with: self
                    with: sql with: nil
                    with: true with: GsRdbBlockStream
                    with: nil with: false with: bindInfo.
   result _sql: sql.
   ^result
%
! bugfix 35736
! bugfix 35449
method: GsOracleConnection 
prepareSqlString: sqlString

" Do some pre-processing of a SQL command string 
  to make sure it complies with various Oracle restrictions:

  * No DoubleByteStrings ( try converting to String )
  * No carriage returns  ( replace with spaces )
"

   | str cr sp |
   sqlString _validateClass: String.
   str := sqlString asString copy.
   " At this point, we shouldn't still have a DoubleByteString "
   (str isKindOf: DoubleByteString)
      ifTrue: [ self _errorExpectedClass: String ].
   cr := Character cr.
   sp := Character space.
   1 to: str size do: [ :i | 
      (( str at: i ) = cr ) ifTrue: [ str at: i put: sp ] ].
   ^ str
%
!
! Instance Category 'Accessing'
!
category: 'Accessing'
method: GsOracleConnection
batchSize

"Returns the batch size used for Oracle OCI operations (default is 20)"

    batchSize isNil ifTrue: [ batchSize := 20 ].
    ^batchSize

%
method: GsOracleConnection
textLimit

"Returns the text limit associated with this connection, 
 used to determine the size (in bytes) of internal buffers 
 used for transfering Oracle LONG/LONGRAW and BLOB/CLOB fields.

 If nil, defaults to 64K.
"

  ^ textLimit
%
method: GsOracleConnection
lobLimit

"Returns the maximum number of Oracle BLOB/CLOB objects to be 
 used concurrently by this connection.  Attempting to use more
 than this number will trigger a #lobAllocationError.

 If lobLimit = 0, Oracle BLOB/CLOB fields will be treated like
 LONG/LONGRAW data, limited to a maximum size of 64K (textLimit).

 If nil, defaults to 0.
"

^ lobLimit
%
method: GsOracleConnection
charSet

"Returns the standard character set code for this Oracle DB.
 Possible values are:

   #CHR8:    A character set encoded in 8-bit fields
   #CHR16:   A character set encoded in 16-bit fields
   #UTF8:    UTF8  character set (8-bit  fields but with encoding)
   #UTF16:   UTF16 character set (16-bit fields but with encoding)
   nil:      The default value (#CHR8 for standard)
"

^ charSet
%
method: GsOracleConnection
ncharSet

"Returns the national character set code for this Oracle DB.
 Possible values are:

   #CHR8:    A character set encoded in 8-bit fields
   #CHR16:   A character set encoded in 16-bit fields
   #UTF8:    UTF8  character set (8-bit  fields but with encoding)
   #UTF16:   UTF16 character set (16-bit fields but with encoding)
   nil:      The default value (#CHR16 for national)
"

^ ncharSet
%
! 48434
method: GsOracleConnection
clobSet

"Returns the standard character set code for CLOBS in this Oracle DB.
 Possible values are:

   #CHR8:    A character set encoded in 8-bit fields
   #CHR16:   A character set encoded in 16-bit fields
   #UTF8:    UTF8  character set (8-bit  fields but with encoding)
   #UTF16:   UTF16 character set (16-bit fields but with encoding)
   nil:      The default value (#CHR8 for standard)
"

^ clobSet
%
! 48434
method: GsOracleConnection
nclobSet

"Returns the national character set code for NCLOBS in this Oracle DB.
 Possible values are:

   #CHR8:    A character set encoded in 8-bit fields
   #CHR16:   A character set encoded in 16-bit fields
   #UTF8:    UTF8  character set (8-bit  fields but with encoding)
   #UTF16:   UTF16 character set (16-bit fields but with encoding)
   nil:      The default value (#CHR16 for national)
"

^ nclobSet
%
method: GsOracleConnection
charGS

"Returns the code controlling how charSet Oracle strings are returned
 to GemStone.  Possible values are:

  #STR or nil   : return a String/DoubleByteString/QuadByteString 
                  as appropriate (same as #QBS) - the default behavior
  #DBS          : return a String/DoubleByteString as appropriate
                  Triggers an error if a character is out of range
  #QBS          : return a String/DoubleByteString/QuadByteString 
                  as appropriate (same as #STR)
  #STX          : return only Strings (eXclusive)
                  Triggers an error if a character is out of range
  #DBX          : return only DoubleByteStrings (eXclusive)
                  Triggers an error if a character is out of range
  #QBX          : return only QuadByteStrings (eXclusive)
  #UTF8         : return only Utf8 strings
  #UTF16        : return only Utf16 strings (not yet supported)
  #BYTE         : return an un-decoded ByteArray (charSet ignored)
"

^ charGS
%
method: GsOracleConnection
ncharGS

"Returns the code controlling how ncharSet Oracle strings are returned
 to GemStone.  Possible values are:

  #STR or nil   : return a String/DoubleByteString/QuadByteString 
                  as appropriate (same as #QBS) - the default behavior
  #DBS          : return a String/DoubleByteString as appropriate
                  Triggers an error if a character is out of range
  #QBS          : return a String/DoubleByteString/QuadByteString 
                  as appropriate (same as #STR)
  #STX          : return only Strings (eXclusive)
                  Triggers an error if a character is out of range
  #DBX          : return only DoubleByteStrings (eXclusive)
                  Triggers an error if a character is out of range
  #QBX          : return only QuadByteStrings (eXclusive)
  #UTF8         : return only Utf8 strings
  #UTF16        : return only Utf16 strings (not yet supported)
  #BYTE         : return an un-decoded ByteArray (ncharSet ignored)
"

^ ncharGS
%
method: GsOracleConnection
utfErr

"Returns flag controlling how GemConnect deals with UTF8/UTF16 conversion
 problems with invalid characters.  If set to false, will replace invalid
 characters with the Unicode Replacement Character (16rFFFD), or a question
 mark for single byte strings.  When set to true/nil (the default value),
 will trigger an error on all UTF conversion problems.
"

^utfErr
%
! 48654
method: GsOracleConnection
returnDateAndTime

"Returns boolean flag indicating what type of Date/Time object should be
 returned from Oracle.  False/nil indicates default DateTime objects,
 True indicates DateAndTime objects.
"

^returnDateAndTime
%

! 49072
method: GsOracleConnection
ociTimeout

"Returns integer indicating timeout interval in seconds for OCI operations.
 OCI operations taking longer than this will fail with a #ociTimeout error.

 If nil, the value defaults to $GS_GEMCONNECT_OCI_TIMEOUT if set, 
 otherwise 2 minutes.
"
^ociTimeout
%

method: GsOracleConnection
allStreams

"Returns all read and write streams created using this connection that have 
 not yet been freed.  If there are no read streams still in use, an empty 
 array is returned instead.

ARGUMENTS:

   none

RETURN VALUE:

   <Array> new

ERRORS:

   GsRdbError/invalidConnection
"

   self class _loadUALib.
   ^System userAction: #GsOraAllStreams with: self
%
!
! Instance Category 'Updating'
!
category: 'Updating'
method: GsOracleConnection
batchSize: size

"Sets the batch size used for Oracle OCI operations"

    batchSize := size

%
category: 'Updating'
method: GsOracleConnection
textLimit: size

"Sets the text limit associated with this connection, 
 used to determine the size (in bytes) of internal buffers 
 used for transfering Oracle LONG/LONGRAW fields.

 If nil, defaults to 64K.
"

    textLimit := size
%
category: 'Updating'
method: GsOracleConnection
lobLimit: size

"Sets the maximum number of Oracle BLOB/CLOB objects to be 
 used concurrently by this connection.  Attempting to use more
 than this number will trigger a #lobAllocationError.

 If lobLimit = 0, Oracle BLOB/CLOB fields will be treated like
 LONG/LONGRAW data, limited to a maximum size of 64K (textLimit).

 lobLimit can only be changed while not connected.

 If nil, defaults to 0.
"
    (self connected) ifTrue: [ 
        GsOracleConnection _signalLobUsageError: self ].
    lobLimit := size
%
method: GsOracleConnection
charSet: aSymbol

"Set the standard character set used by this Oracle DB.

 Possible values are:

   #CHAR8   (default value when nil)
   #CHAR16
   #UTF8
   #UTF16
"

charSet := aSymbol
%
method: GsOracleConnection
ncharSet: aSymbol

"Set the national character set used by this Oracle DB.

 Possible values are:

   #CHAR8
   #CHAR16   (default value when nil)
   #UTF8
   #UTF16
"

ncharSet := aSymbol
%
! 48434
method: GsOracleConnection
clobSet: aSymbol

"Set the standard character set used by CLOBS in this Oracle DB.

 Possible values are:

   #CHAR8   (default value when nil)
   #CHAR16
   #UTF8
   #UTF16
"

clobSet := aSymbol
%
! 48434
method: GsOracleConnection
nclobSet: aSymbol

"Set the national character set used by NCLOBS in this Oracle DB.

 Possible values are:

   #CHAR8
   #CHAR16   (default value when nil)
   #UTF8
   #UTF16
"

nclobSet := aSymbol
%
method: GsOracleConnection
charGS: aSymbol

"Set the code controlling how charSet Oracle strings are returned
 to GemStone.  Possible values are:

  #STR or nil   : return a String/DoubleByteString/QuadByteString 
                  as appropriate (same as #QBS) - the default behavior
  #DBS          : return a String/DoubleByteString as appropriate
                  Triggers an error if a character is out of range
  #QBS          : return a String/DoubleByteString/QuadByteString 
                  as appropriate (same as #STR)
  #STX          : return only Strings (eXclusive)
                  Triggers an error if a character is out of range
  #DBX          : return only DoubleByteStrings (eXclusive)
                  Triggers an error if a character is out of range
  #QBX          : return only QuadByteStrings (eXclusive)
  #UTF8         : return only Utf8 strings
  #UTF16        : return only Utf16 strings (not yet supported)
  #BYTE         : return an un-decoded ByteArray (charSet ignored)
"

charGS := aSymbol
%
method: GsOracleConnection
ncharGS: aSymbol

"Set the code controlling how ncharSet Oracle strings are returned
 to GemStone.  Possible values are:

  #STR or nil   : return a String/DoubleByteString/QuadByteString 
                  as appropriate (same as #QBS) - the default behavior
  #DBS          : return a String/DoubleByteString as appropriate
                  Triggers an error if a character is out of range
  #QBS          : return a String/DoubleByteString/QuadByteString 
                  as appropriate (same as #STR)
  #STX          : return only Strings (eXclusive)
                  Triggers an error if a character is out of range
  #DBX          : return only DoubleByteStrings (eXclusive)
                  Triggers an error if a character is out of range
  #QBX          : return only QuadByteStrings (eXclusive)
  #UTF8         : return only Utf8 strings
  #UTF16        : return only Utf16 strings (not yet supported)
  #BYTE         : return an un-decoded ByteArray (ncharSet ignored)
"

ncharGS := aSymbol
%
method: GsOracleConnection
utfErr: aBoolean

"Sets the flag controlling how GemConnect deals with UTF8/UTF16 conversion
 problems with invalid characters.  If set to false, will replace invalid
 characters with the Unicode Replacement Character (16rFFFD), or a question
 mark for single byte strings.  When set to true/nil (the default value),
 will trigger an error on all UTF conversion problems.
"
(aBoolean isKindOf: Boolean) ifFalse: [ 
   self halt: 'utfErr value must be a boolean' ].
utfErr := aBoolean
%
! 48654
method: GsOracleConnection
returnDateAndTime: aBoolean

"Sets the boolean flag indicating what type of Date/Time object should be
 returned from Oracle.  False/nil indicates default DateTime objects,
 True indicates DateAndTime objects.
"
(aBoolean isKindOf: Boolean) ifFalse: [ 
   self halt: 'returnDateAndTime value must be a boolean' ].
returnDateAndTime := aBoolean
%

! 49072
method: GsOracleConnection
ociTimeout: timeout

"Sets timeout interval in seconds for OCI operations.
 OCI operations taking longer than this will fail with a #ociTimeout error.
 If nil, the value defaults to $GS_GEMCONNECT_OCI_TIMEOUT if set, 
 otherwise 2 minutes.

"
(timeout notNil) ifTrue: [
   timeout _validateClass: SmallInteger.
   (timeout < 1) ifTrue: [ timeout _error: #rtErrArgNotPositive ]].
ociTimeout := timeout.
^ociTimeout
%

!
! Instance Category 'Connection Management'
!
category: 'Connection Management'
method: GsOracleConnection
disconnect

"Disconnects from the Oracle database and frees any read streams associated with
 this connection.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GsRdbError/invalidConnection
   GsRdbError/oracleError
"

   self class _loadUALib.
   ^System userAction: #GsOraDisconnect with: self.
%
method: GsOracleConnection
connect

"Attempts to connect to the Oracle database with the current connection
 parameters within this session.  If this connection is already connected, 
 no action is taken. If  the connection fails, this method signals a 
 non-continuable error.  Note that connections are specific to a particular
 session: connecting a persistent GsOracleConnection in one session does
 not affect its status in another.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GsRdbError/oracleError
"

   self class _loadUALib.
   System userAction: #GsOraConnect with: self with: parameters.
%
method: GsOracleConnection
connected

"Returns true if this connection is connected to the Oracle database in
 this GS session and false if not.  Note that connections are specific
 to a particular session: connecting a persistent GsOracleConnection in 
 one session does not affect its status in another.


ARGUMENTS:

   none

RETURN VALUE:

   <Boolean>

ERRORS:

   none
"

   self class _loadUALib.
   ^System userAction: #GsOraIsConnected with: self
%
!
! Class Category 'Error Handling'
!
category: 'Error Handling'
classmethod: GsOracleConnection
messages

"Returns all information about an error that was raised.  Intended to provide
 access to Oracle database error information obtained during command execution.

 Information is returned in an array containing:

   1    - Oracle Call Interface (OCI) return code (usually -1)
   2    - Number of errors
          (usually 1, unless there are multiple errors)
   3    - Oracle error code
   4    - Oracle error message
   5    - GemStone object causing problems
   6..  - Repeat 3-5 for each additional error
   last - Internal diagnostic message 

ARGUMENTS:

   none

RETURN VALUE:

   <Array> new

ERRORS:

   none
"

   self _loadUALib.
   ^System userAction: #GsOraReturnMessages
%
classmethod: GsOracleConnection
_signalOracleError: connectObj stream: streamObj args: messages


   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #oracleError)
               args: { connectObj . streamObj . nil . messages }) signal
%
classmethod: GsOracleConnection
_signalLobUsageError: connectObj

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #lobUsageError)
               args: { connectObj . nil . nil . nil }) signal
%
category: 'Error Handling'
classmethod: GsOracleConnection
_signalBindingSetupError: sql details: msg


   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #bindingSetupError)
               args: { self . nil . sql . msg }) signal
%
! 49915
category: 'Error Handling'
classmethod: GsOracleConnection
_signalOrLibError: path details: msg

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #oralibError)
               args: { path . msg . nil . nil }) signal
%
!
! Class Category 'Connection Manager'
!
category: 'Connection Manager'
classmethod: GsOracleConnection
libName

"Returns the GemConnect user action library name"

| libName |
libName :=  'liboraapi' copy.
libName addAll: (self gcImageVersion).
libName addAll: '-'.
libName addAll: (self gsVersion).
libName addAll: '-'.
libName addAll: (self bitArch).
^ libName
%
! 49915
classmethod: GsOracleConnection
_loadOrLib

"Load the primary Oracle library libclntsh.

 The value of the Globals value #GemConnectOracleLibrary, or if not set, 
 the value of the gem environment variable $GS_GEMCONNECT_ORACLE_LIBRARY, 
 to specify the full path to libclntsh. For example:

 /opt/oracle/lib/libclntsh.so.12.1

"
  | path olib result |
  path := Globals at: #GemConnectOracleLibrary ifAbsent:  nil.
  path isNil ifTrue: [
     path := System gemEnvironmentVariable: 'GS_GEMCONNECT_ORACLE_LIBRARY' ].
  path isNil ifTrue: [
     self _signalOrLibError: '[not set]' details: 'Oracle library not specified'].
  "If path is a smalltalk block, evaluate it"
  (path isKindOf: ExecBlock) ifTrue: [ path := path value ].
  (path isKindOf: String) ifFalse: [
     self _signalOrLibError: path asString  details: 'Oracle library specification is not a String'].
  GsFile gciLogServer: '[Info] Loading GemConnect/Oracle library:'.
  GsFile gciLogServer: '       ' , path.
  olib := CLibrary named: path.
  result := olib load .    "Returns true or a String with error details"
  result == true ifFalse: [ 
      self _signalOrLibError: path details: 'Dlopen failed: ', result asString].
%
classmethod: GsOracleConnection
_loadUALib

"Checks to see if the Oracle user action library has been loaded. If not, the
 library is loaded and the connection manager in the GemStone session state is
 updated.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/cannotLoadUALib
"

   | connManager libName |

   "Make sure the connection manager has been installed in the session state."
   connManager := self _installConnManager.

   "The connection manager has reserved slot #2 for Oracle in its loaded UA
    libs check array."
   (connManager loadedUAArrayAt: 2) ifFalse: [

      "Make sure GsRdbChangeNotificationMode is configured."
      GsRdbChangeNotifObj _checkChangeNotificationMode.

      "49915: load Oracle library libclntsh.so"
      self _loadOrLib.

      "Load the library."
      libName := self libName.
      System loadUserActionLibrary: libName.
      connManager loadedUAArrayAt: 2 put: true.
      System userAction: #GsOraSetKeepArray with: connManager keepArray.
      ].
%
!
! Class Category 'Accessing'
!
category: 'Accessing'
classmethod: GsOracleConnection
allConnections

"Returns all successfully connected Oracle connection instances. If no
 Oracle connection instance has successfully connected during the current
 GemStone session, returns an empty array.

ARGUMENTS:

   none

RETURN VALUE:

   <Array> new

ERRORS:

   none
"

   self _loadUALib.
   ^System userAction: #GsOraAllConnections
%
classmethod: GsOracleConnection
parameterClass

"Returns the parameter class used to instantiate this connection class.

ARGUMENTS:

   none

RETURN VALUE:

   <Class>

ERRORS:

   none
"

   ^GsOracleParameters
%
classmethod: GsOracleConnection
lastConnection

"Returns the last successfully connected Oracle connection instance. If no
 connection has successfully connected during the current GemStone session,
 returns nil.

ARGUMENTS:

   none

RETURN VALUE:

   <GsOracleConnection>

ERRORS:

   none
"

   self _loadUALib.
   ^System userAction: #GsOraLastConnection
%
classmethod: GsOracleConnection
_rdbTypeName

"Returns a string describing what type of connection this class represents. In
 this case the string returned is: 'Oracle'.

ARGUMENTS:

   none

RETURN VALUE:

   <String> new

ERRORS:

   none
"

   ^'Oracle'
%
!
! Class Category 'Datatype Conversion'
!
category: 'Datatype Conversion'
classmethod: GsOracleConnection
getRdbDateTimeFor: aDateTime

"Returns a string representing the given datetime as a valid Oracle date and
 time constant.

ARGUMENTS:

   aDateTime      <DateTime> uncaptured

RETURN VALUE:

   <String> new
   A string of the form: 
   'TO_DATE('DD/MM/YYYY HH24:MI:SS', 'DD/MM/YYYY HH24:MI:SS')'.

ERRORS:

   none
"

   |  dtstring oracleDate |

   aDateTime _validateClass: DateTime.

   dtstring := aDateTime asString.

   oracleDate := String new.
   oracleDate add: 'TO_DATE('''; add: dtstring;
              add: ''', ''DD/MM/YYYY HH24:MI:SS'')'.
   ^oracleDate
%
classmethod: GsOracleConnection
getRdbDateFor: aDate

"Returns a string representing the given date as a valid Oracle date constant.

ARGUMENTS:

   aDate      <Date> uncaptured

RETURN VALUE:

   <String> new
   A string of the form: 
   'TO_DATE('DD/MM/YYYY', 'DD/MM/YYYY')'.

ERRORS:

   none
"

   |  dtstring oracleDate |
   aDate _validateClass: Date.

   dtstring := aDate asString.

   oracleDate := String new.
   oracleDate add: 'TO_DATE('''; add: dtstring;
              add: ''', ''DD/MM/YYYY'')'.
   ^oracleDate
%
classmethod: GsOracleConnection
getRdbBinaryFor: aByteArray

"Returns a string representing the given byte array as a valid Oracle
 binary constant.

ARGUMENTS:

   aByteArray      <ByteArray> uncaptured

RETURN VALUE:

   <String> new
   A string of the form: 'FFFFFFFFFFF'.

ERRORS:

   none
"

   aByteArray _validateClass: ByteArray.

   ^aByteArray asHexString quoted
%
classmethod: GsOracleConnection
getRdbBitFor: aBool

"Returns a string representing the given boolean as a valid Oracle bit
 constant.

ARGUMENTS:

   aBool      <Boolean> uncaptured

RETURN VALUE:

   <String> new
   A string of the form: '0' or '1'.

ERRORS:

   none
"

   aBool _validateClass: Boolean.

   aBool ifTrue: [
      ^'1'
      ]
   ifFalse: [
      ^'0'
      ].
%
!
! Class Category: 'Version - Library'
!
! fix 37621: modified to retrieve version info from C layer
!
category: 'Version - Library'
classmethod: GsOracleConnection
libraryVersionArray
 
"Returns an Array of version information regarding the GemConnect shared library:

1: Version (String)
2: Patch level (SmallInteger)
3: Build kind (Official or private) (String)
4: Build timestamp (String)
5: Build type (slow or fast) (String)
6: git SHA (String)
7: git branch name (String)
8: Oracle version

"
self _loadUALib.
^ System userAction: #GsOraVersion
%

category: 'Version - Library'
classmethod: GsOracleConnection
librarySha
 
"Returns the git SHA of the library"

^ self libraryVersionArray at: 6
%

category: 'Testing'
classmethod: GsOracleConnection
libraryMatchesImage

"Answer true if the GemConnect image version matches the GemConnect shared library version
 or false otherwise."
 
^ self librarySha = self gcImageSha
%

category: 'Version - Deprecated'
classmethod: GsOracleConnection
version
 
"Deprecated, use #libraryVersion instead of this method"

^ self libraryVersion 
%

category: 'Version - Deprecated'
classmethod: GsOracleConnection
gcVersion

"Deprecated, use #gcImageVersion instead."

^ self gcImageVersion
%


category: 'Version - Library'
classmethod: GsOracleConnection
libraryVersion
 
"Returns a string representing the version of the GemConnect shared library.

ARGUMENTS:

   none

RETURN VALUE:

   <String>

ERRORS:

   none

"
^ self libraryVersionArray at: 1
%
category: 'Version - Deprecated'
classmethod: GsOracleConnection
patchLevel
 
"Deprecated, use #libraryPatchLevel instead of this method." 

^ self libraryPatchLevel
%

category: 'Version - Library'
classmethod: GsOracleConnection
libraryPatchLevel
 
"Returns an integer representing the patch level of the GemConnect shared library."

^ self libraryVersionArray at: 2
%
! 49197
category: 'Version - Deprecated'
classmethod: GsOracleConnection
patchInfo
 
"Deprecated. Use #libraryPatchInfo instead of this method."

^ self libraryPatchInfo
%

category: 'Version - Library'
classmethod: GsOracleConnection
libraryPatchInfo
 
"Returns an information String for the patch level of the GemConnect shared library."

^ self libraryVersionArray at: 3
%

category: 'Version - Server'
classmethod: GsOracleConnection
gsVersion

"Returns concise GemStone database server version string"

| vstr index gsv gsv2 |
vstr := System _gemVersion.
index := 1.
[ (vstr at: index) = $, ] whileFalse: [ index := index + 1].
gsv :=  vstr copyFrom: 1 to: (index - 1).
^ gsv
%

category: 'Version - Library'
classmethod: GsOracleConnection
bitArch
^ '64'
%

category: 'Binding Management'
classmethod: GsOracleConnection
bindingVars: sql

" Returns an array of binding variables found in an SQL statement "

| vars newvar prechar |
vars := Set new.
prechar := $ .
1 to: sql size do: [:i | | c |
  c := sql at: i.
  newvar isNil
    ifTrue: [
      ((c = $:) and: [prechar isAlphaNumeric not ]) 
        ifTrue: [ newvar := c asString ]]
    ifFalse: [
      (c isAlphaNumeric) 
        ifTrue: [ newvar add: c ]
        ifFalse: [ 
          (newvar size > 1)
            ifTrue: [
              (vars includes: newvar)
                ifFalse: [ vars add: newvar ]].
          newvar := nil ]].
  prechar := c].
((newvar isNil) and: [newvar size > 1]) ifFalse: [ 
  (vars includes: newvar) ifFalse: [ vars add: newvar ]].
^vars
%
classmethod: GsOracleConnection
_validateBindInfo: bindInfo against: sql isSelect: isSelect

" Check that formatting and contents for bindInfo are consistent with sql."

| bindVars recordedVars size item |
bindVars := self bindingVars: sql.
recordedVars := IdentitySet new.
(bindInfo isKindOf: Array)
  ifFalse: [ self _signalBindingSetupError: sql 
             details: 'Invalid class for bindInfo' ].
(bindInfo isEmpty)
  ifTrue: [ self _signalBindingSetupError: sql
            details: 'Empty bindInfo array' ].
item := bindInfo at: 1.
(item isNil or: [item isKindOf: String])
  ifFalse: [ self _signalBindingSetupError: sql
             details: 'Incorrect RdbTable entry' ].
size := bindInfo size - 1.
1 to: size do: [:i | | bi biSize biv bit bid bis |
  bi := i + 1.
  item := bindInfo at: bi.
  (item isKindOf: Array)
    ifFalse: [ self _signalBindingSetupError: sql
               details: ('Invalid bindInfo at element ' , (bi asString))].
  biSize := item size.
  ((biSize < 2) or: [ biSize > 4])
    ifTrue: [ self _signalBindingSetupError: sql
              details: ('Incorrect number of items at element ', 
                        (bi asString))].
  biv := item at: 1.
  (biv isKindOf: String)
    ifFalse: [ self _signalBindingSetupError: sql
               details: ('Binding variable at item ' ,
                         (bi asString) ,
                         ' is not a String')].
  (recordedVars includes: biv)
    ifTrue: [ self _signalBindingSetupError: sql
              details: ('Binding variable ' ,
                        biv,
                        ' already found in bindInfo')].
  recordedVars add: biv.
  bindVars remove: biv ifAbsent: [
    self _signalBindingSetupError: sql
    details: ('Binding variable ' ,
              biv ,
              ' not found in SQL statement')].
  bit := item at: 2.
  (bit isKindOf: Class)
    ifFalse: [ self _signalBindingSetupError: sql
               details: ('Binding type ' ,
                         bit asString ,
                         ' is not a Class')].
  ({Integer . Float . String . DateTime} includes: bit)
  ifFalse: [ self _signalBindingSetupError: sql
             details: ('Binding type ' ,
                       bit asString ,
                       ' is not a valid Class (Integer,Float,String,DateTime)')].
  (biSize > 2) ifTrue: [
    bid := item at: 3.
    (bid isKindOf: Symbol)
      ifFalse: [ self _signalBindingSetupError: sql
                 details: ('Binding direction ' ,
                           bid asString ,
                           ' is not a Symbol')].
    ({#IN . #OUT . #INOUT} includes: bid)
      ifFalse: [ self _signalBindingSetupError: sql
                 details: ('Binding direction ' ,
                           bid asString ,
                           ' is not a valid Symbol (#IN #OUT #INOUT)')].
    (isSelect and: [bid ~= #IN])
      ifTrue: [ self _signalBindingSetupError: sql
                details: ('Binding direction ' ,
                          bid asString ,
                          ' used in a select statement (only #IN allowed)')]].
  (biSize > 3) ifTrue: [
    bis := item at: 4.
    (bis isKindOf: Integer)
      ifFalse: [ self _signalBindingSetupError: sql
                 details: 'Binding variable size not an Integer' ].
    (bit = String)
      ifFalse: [ self _signalBindingSetupError: sql
                 details: 'Binding variable size only valid for Strings' ]]].
bindVars isEmpty 
  ifFalse: [ 
    | msg |
    msg := 'No bindings found for variables ' copy.
    bindVars do: [:v |
      msg add: v.
      msg add: ' ' ].
    self _signalBindingSetupError: sql
         details: msg ].
%

! FIX 49024
!
category:  'Callback Handling'
classmethod: GsOracleConnection
callbackPerform: selector on: receiver withArguments: args

^ [ receiver perform: selector withArguments: args ]
    on: Exception do: [:ex | ex return: ex]
%

! 51439
category: 'Updating'
method: GsOracleConnection
numberAsScaledDecimal: aBoolean

"If aBoolean == true, values in columns of type NUMBER 
 will always be returned as a SmallScaledDecimal or a ScaledDecimal. 
 If false (default), values will be returned as Integers, Floats or 
 ScaledDecimals depending on serveral factors."

aBoolean _validateClass: Boolean .
numberAsScaledDecimal := aBoolean .
^aBoolean
%

category: 'Accessing'
method: GsOracleConnection
numberAsScaledDecimal

"Answer a Boolean indicating if values in columns of type NUMBER will always be 
 returned as a SmallScaledDecimal or a ScaledDecimal."

^ numberAsScaledDecimal == true
%
! template file, do not edit this file directly!
category: 'Version - Image'
classmethod: GsOracleConnection
gcImageVersion

"Returns concise GemConnect image version string"

^ '252'
%

category: 'Version - Image'
classmethod: GsOracleConnection
gcOracleVersion

"Returns the version of Oracle used to build GemConnect image"

^ '19.14.0.0.0'
%

category: 'Version - Image'
classmethod: GsOracleConnection
gcImageBranch

"Returns the version of Oracle used to build GemConnect image"

^ '2.5.2'
%

category: 'Version - Image'
classmethod: GsOracleConnection
gcImageSha

"Returns the git sha used to build GemConnect"

^ 'f0ef3099363211b5'
%

category: 'Version - Image'
classmethod: GsOracleConnection
gcImageBuildDate

"Returns a String which is the date GemConnect image was built"

^ '2025-05-22T13:06:09-0700'
%

category: 'Version - Image'
classmethod: GsOracleConnection
gcImageGemStoneVersion

"Returns a String which GemStone version used to build GemConnect image"

^ '3.7.1.4'
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - GsOracleLOB.gs
!
! ========================================================================
!
! Class GsOracleLOB / GsOracleBLOB / GsOracleCLOB
!

doit
GsOracleLOB removeAllMethods.
GsOracleLOB class removeAllMethods.
^ true
%

! ------------------- Class methods for GsOracleLOB
! ------------------- Instance methods for GsOracleLOB
category: 'Accessing'
method: GsOracleLOB
connection

"Returns the connection associated with this LOB "

   ^ connection

%
category: 'Operations'
method: GsOracleLOB
free

" Free internal storage used by this lob.
  When you no longer need access to a lob, 
  calling free helps to avoid triggering #lobAllocationError.

   Once called, any further operations on this lob 
   will generate #lobUsageError.
"

^ self _lobOp: 0 with: nil with: nil with: nil

%
category: 'Accessing'
method: GsOracleLOB
index

"Returns the index associated with this LOB "

   ^ index

%
category: 'Operations'
method: GsOracleLOB
lobSize

" Return the size in characters of the associated LOB object "

^ self _lobOp: 1 with: nil with: nil with: nil

%
category: 'Operations'
method: GsOracleLOB
readAll

" Returns the contents of this LOB.
  This is usually a ByteArray for a BLOB,
  or a String/DoubleByteString for a CLOB.

   See #returnAll: to specify a different return class.
"

^ self readAll: String

%
category: 'Operations'
method: GsOracleLOB
readAll: recommendedResultClass

" Returns the contents of a LOB, using the recommendedResultClass if possible.
  Note that the method may override the recommendedResultClass
  (for example, a CLOB may return a DoubleByteString instead of 
  a String if the CLOB contains a character with a value > 255).
  If recommendedResultClass is nil, uses the associated connection's
  charGS/ncharGS setting to determine class.

  Usual values are ByteArray / String / DoubleByteString / QuadByteString
"

^ self _lobOp: 2 with: recommendedResultClass with: nil with: nil

%
category: 'Operations'
method: GsOracleLOB
readAll

" Returns the contents of a LOB, using the associated connection's
  charGS/ncharGS setting to determine class.
"

^ self _lobOp: 2 with: nil with: nil with: nil

%
category: 'Operations'
method: GsOracleLOB
writeAll: byteObject

" Write out the specified object to this lob "

^ self _lobOp: 3 with: byteObject with: nil with: nil

%
category: 'Operations'
method: GsOracleLOB
_lobOp: opcode with: arg1 with: arg2 with: arg3

"Perform primitive operation on an Oracle lob object "

" If connection has been closed, return error "
connection isNil ifTrue: [ 
  GsOracleConnection _signalLobUsageError: connection ].

connection class _loadUALib.
^ System userAction: #GsOraLobOp 
    with: connection
    with: opcode
    with: self
    with: arg1
    with: arg2
    with: arg3

%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - GsRdbConnManager.gs
!
! ========================================================================
!
! Class GsRdbConnManager
!

removeallmethods GsRdbConnManager
removeallclassmethods GsRdbConnManager

!
! Class Category 'Instance Creation'
!
category: 'Instance Creation'
classmethod: GsRdbConnManager
new

"Creates a new connection manager instance and initializes its connection cache
 and relational database user action check array.

ARGUMENTS:

   none

RETURN VALUE:

   <GsRdbConnManager> new

ERRORS:

   none
"

   ^super new _initLoadedUAArray; _initConnNameCache.
%
!
! Instance Category 'Accessing'
!
category: 'Accessing'
method: GsRdbConnManager
connNameCacheAt: key

"Returns the connection associated with the given key in the connection cache.
 If there is no entry in the cache which matches the given key, the value
 returned is nil.

ARGUMENTS:

   key      <Object> uncaptured

RETURN VALUE:

   <GsRdbConnection subclass>

ERRORS:

   none
"

   ^connNameCache at: key ifAbsent: [ nil ]
%
method: GsRdbConnManager
_otherAt: nm

"Returns the other variable with the given name, if any.

ARGUMENTS:

   nm      <Symbol> uncaptured

RETURN VALUE:

   <Object>

ERRORS:

   none
"

   other isNil ifTrue: [ ^nil ].

   ^other at: nm otherwise: nil.
%
method: GsRdbConnManager
_otherAt: nm otherwise: value

"Returns the other variable with the given name, if any, or the value if
 nothing has been saved previously.

ARGUMENTS:

   nm      <Symbol> uncaptured
   value   <Object> uncaptured

RETURN VALUE:

   <Object>

ERRORS:

   none
"

   other isNil ifTrue: [ ^value ].

   ^other at: nm otherwise: value.
%
method: GsRdbConnManager
_otherDict

"Returns the receiver's 'other' instance variable dictionary, creating one
 if necessary.

ARGUMENTS:

   none

RETURN VALUE:

   <SymbolDictionary>

ERRORS:

   none
"
  (other isNil) ifTrue: [
     other := SymbolDictionary new
  ].

  ^other
%
method: GsRdbConnManager
loadedUAArrayAt: anIndex

"Indicates whether the relational database user action library corresponding to
 the given index into the RDB UA load check array has been loaded. The user
 action libraries corresponding to a particular index value are as follows:

   anIndex      assignedUse
   -------      -----------
      1         Obsolete [former Sybase user action library]
      2         Oracle user action library
     3-6        reserved for future use by GemTalk Systems.
     7-10       available for customer use

ARGUMENTS:

   integer      <SmallInteger> uncaptured
                An integer between 1 and 10.

RETURN VALUE:

   <Boolean>    True if the particular user action library has been loaded 
                and false if not.   False is also returned if anIndex is 
                out of range.

ERRORS:

   none
"

   ^loadedUAArray at: anIndex
%
!
! Instance Category 'Updating'
!
category: 'Updating'
method: GsRdbConnManager
connNameCacheAt: key put: connection

"Caches the given connection in the connection cache using the given key.

ARGUMENTS:

   key             <Object> captured
   connection      <GsRdbConnection subclass> captured

RETURN VALUE:

   the argument connection

ERRORS:

   none
"

   connection _validateClass: GsRdbConnection.
   ^connNameCache at: key put: connection
%
method: GsRdbConnManager
connNameCacheRemove: connection

"Removes the given connection value from the connection cache if it is present.
 If the given connection is not in the cache, no action is taken.

ARGUMENTS:

   connection      <GsRdbConnection subclass> uncaptured

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   | aKey |

   connection _validateClass: GsRdbConnection.
   aKey := connNameCache keyAtValue: connection ifAbsent: [ nil ].
   (aKey notNil) ifTrue: [
      connNameCache removeKey: aKey.
      ].
%
method: GsRdbConnManager
connNameCacheRemoveKey: aKey

"Removes the cache entry with the given key from the connection cache if that
 key is present and returns the connection associated with that key. If the
 given key is not in the cache, returns nil.

ARGUMENTS:

   aKey      <Object> uncaptured

RETURN VALUE:

   <GsRdbConnection subclass>

ERRORS:

   none
"

   | conn |

   conn := self connNameCacheAt: aKey.
   (conn notNil) ifTrue: [
      connNameCache removeKey: aKey.
      ].
   ^conn
%
method: GsRdbConnManager
_otherAt: nm put: value

"Sets the other variable with the given name to the given value. If the
 value is nil, removes the other variable with the given name.

ARGUMENTS:

   nm      <Symbol> captured
   value   <Object> captured

RETURN VALUE:

   <Object>

ERRORS:

   none
"

   (value isNil) ifTrue: [
      (other notNil) ifTrue: [
         other removeKey: nm ifAbsent: [].
         ]
      ]
   ifFalse: [
      self _otherDict at: nm put: value.
      ].

   ^value
%
method: GsRdbConnManager
loadedUAArrayAt: anIndex put: bool

"Sets the RDB UA load check array to indicate whether a particular relational
 database user action library has been loaded.  To see what relational database
 user action library corresponds to which index in the array see the
 #loadedUAArrayAt: method.

ARGUMENTS:

   anIndex      <SmallInteger> uncaptured
   bool         <Boolean> captured

RETURN VALUE:

   returns the boolean value bool

ERRORS:

   none
"

   bool _validateClass: Boolean.
   ^loadedUAArray at: anIndex put: bool.
%
!
! Instance Category 'Initialization'
!
category: 'Initialization'
method: GsRdbConnManager
_initConnNameCache

"Creates a key value dictionary to use as the connection cache.

ARGUMENTS:

   none

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"

   connNameCache := KeyValueDictionary new.
%
method: GsRdbConnManager
_initLoadedUAArray

"Creates an array of size 10 to use for the RDB UA load check array. See
 #loadedUAArrayAt: for more information.

ARGUMENTS:

   none

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"

   loadedUAArray := Array withAll:  {false . false . false . false . false .
                                     false . false . false . false . false}.
%

category: 'Accessing'
method: GsRdbConnManager
keepArray
"Answer a collection that can be used by GemConnect primitives to keep
 objects that may otherwise be unreferenced alive.

ARGUMENTS:

   none
 
RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"

   keepArray isNil ifTrue: [ keepArray := OrderedCollection new ].
   ^keepArray
%
 
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - GsRdbChangeNotifObj.gs
!
! ========================================================================
!
! Class 'GsRdbChangeNotifObj'
!

removeallmethods GsRdbChangeNotifObj
removeallclassmethods GsRdbChangeNotifObj

!
! Instance Category 'Clean Up'
!
! 49506: fix code for going into protected mode
! 49507: fix typo on loop index
category: 'Clean Up'
method: GsRdbChangeNotifObj
removeAllNotification

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"Removes this change notification object from all dependency lists. The change
 notification object will no longer be notified when objects are modified.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   <primitive: 2001> "enter protected mode"
   | prot entries depList |

   prot := System _protectedMode.
   [
   entries := SharedDependencyLists removeEntriesContaining: self.
   1 to: entries size do: [  :i |
      depList := entries at: i.
      depList _removeCompletelyPathTerm: self.
      SharedDependencyLists at: depList logging: false.
      ].
   ] ensure: [
      prot _leaveProtectedMode
   ]
%
!
! Instance Category 'Change Notification'
!
category: 'Change Notification'
method: GsRdbChangeNotifObj
invokingBecomeOn: object1 to: object2

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"Sends the aboutToBecome: message to the given first object, with the second
 object as the argument.

 Note: Nothing is done here as notification of object migration through a
 become operation is not supported at this time.

ARGUMENTS:

   object1      <Object>
   object2      <Object>

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"
%
method: GsRdbChangeNotifObj
removing: anObject from: nsc

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"Sends the #aboutToRemove: message to the given non-sequenceable collection, as
 this collection is about to have the given object removed from it.

ARGUMENTS:

   anObject      <Object> unspecified
   nsc           <NonSequenceableCollection subclass> uncaptured

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"

   nsc aboutToRemove: anObject.
%
method: GsRdbChangeNotifObj
modifyingObject: anObject atOffset: offset to: newValue

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"Sends the #aboutToChange:newValue: message to the given object, as this object
 is about to have an instance variable (named or unnamed for sequenceable
 collection objects) at the given offset changed to the given new value.

ARGUMENTS:

   anObject      <Object> uncaptured
   offset        <Integer> unspecified
   newValue      <Object> unspecified

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"

   anObject aboutToChange: offset newValue: newValue.
%
method: GsRdbChangeNotifObj
modifyingByteObject: anObject startingAt: index withNewValue: newValue

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"Sends the #aboutToChange:newValue: message to the given byte indexable object,
 as this object is about to have its value at the given index changed to the
 given new value.

 Note: Nothing is done here as notification of changes to byte sequenceable
 collections is not supported at this time.

ARGUMENTS:

   anObject      <Object> uncaptured
   index         <Integer> unspecified
   newValue      <Object> unspecified

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"
%
method: GsRdbChangeNotifObj
inserting: aColl into: aSeqColl at: index insertSize: sz

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"Sends the #aboutToInsert:index: message to the given sequenceable collection
 (aSeqColl), with the given collection (aColl) and the given index as arguments,
 as the elements in this collection (aColl) are about to be inserted into this
 sequenceable collection (aSeqColl) at this index.

ARGUMENTS:

   aColl         <SequenceableCollection> unspecified
   aSeqColl      <SequenceableCollection> unspecified
   index         <Integer> unspecified
   sz            <Integer> uncaptured

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"

   aSeqColl aboutToInsert: aColl index: index.
%
!
! 38174: add code to check GsRdbChangeNotificationMode 
!        and calculate index as appropriate..
!
method: GsRdbChangeNotifObj
deletingIn: aSeqColl startingAt: offset count: count

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"Sends the #aboutToDelete:index: message to the given sequenceable collection
 (aSeqColl), with a new collection and index as arguments, as the elements in
 this new collection (which have been determined using the given offset and
 given count) are about to be deleted from this sequenceable collection at this
 index.

ARGUMENTS:

   aSeqColl      <SequenceableCollection> unspecified
   offset        <Integer> uncaptured
   count         <Integer> uncaptured

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"

   | mode index |

   mode := GsRdbChangeNotifObj _changeNotificationMode.
   (mode == 1)
   ifTrue: [ index := offset - (aSeqColl class instSize) ]
   ifFalse: [ 
      (mode == 2)
      ifTrue: [ index := offset ]
      ifFalse: [ self halt: 'Invalid GsRdbChangeNotificationMode' ]].

   aSeqColl aboutToDelete: (aSeqColl copyFrom: index to: (index + count - 1))
            index: index.
%
method: GsRdbChangeNotifObj
changingSizeOfObject: aSeqColl to: newSize

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"Sends the #aboutToDelete:index: message to the given sequenceable collection,
 (aSeqColl), with a new collection and index as arguments, as the size of this
 sequenceablecollection is about to change to the given new size. Only when the
 new size is smaller than the current size of the collection will this occur.

 Note: Nothing is done here for byte indexable collections as notification of
 deletions from byte indexable collections is not supported at this time.

ARGUMENTS:

   aSeqColl      <SequenceableCollection> unspecified
   newSize       <Integer> uncaptured

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"

   | index sz |

   sz := aSeqColl size.
   (newSize < sz) ifTrue: [
      index := newSize + 1.
      aSeqColl aboutToDelete: (aSeqColl copyFrom: index to: sz)
               index: index.
      ].
%
method: GsRdbChangeNotifObj
adding: anObject to: nsc

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"Sends the #aboutToAdd: message to the given non-sequenceable collection, as the
 given object is about to be added to it.

ARGUMENTS:

   anObject      <Object> unspecified
   nsc           <UnorderedCollection subclass> uncaptured

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"

   nsc aboutToAdd: anObject.
%
!
! Class Category 'Instance Creation'
!
category: 'Instance Creation'
classmethod: GsRdbChangeNotifObj
new

"Creates a new instance of the RDB change notification object class if there
 isn't already an instance of this class stored in the 'soleInstance' class
 instance variable. Otherwise, returns the 'soleInstance'.

ARGUMENTS:

   none

RETURN VALUE:
   <GsRdbChangeNotifObj>

ERRORS:

   none
"

   (soleInstance isNil) ifTrue: [
      soleInstance := super new.

      "This is usually done during GemConnect installation,
       so lets also set the GsRdbChangeNotificationMode "
      GsRdbChangeNotifObj _resetChangeNotificationMode

      ].
   ^soleInstance
%
!
! Class Category 'Accessing'
!
category: 'Accessing'
classmethod: GsRdbChangeNotifObj
soleInstance

"Returns the sole instance of the RDB change notification object class.

ARGUMENTS:

   none

RETURN VALUE:

   <GsRdbChangeNotifObj>

ERRORS:

   none
"

   ^soleInstance
%
!
! Class Category 'ChangeNotificationMode'
!
category: 'ChangeNotificationMode'
classmethod: GsRdbChangeNotifObj
_changeNotificationMode

"Determine change notification mode for current system.
  Returns:
    1 for GS/S prior to 6.3, GS/64 prior to 2.2.0
    2 for GS/S 6.3 and later, GS/64 2.2.0 and later
"
   | mode version |
   mode := 2.
   version := System _gemVersion copyFrom: 1 to: 3.
    (nil asOop == 10) 
      ifTrue: [ "GS/S"
         (version < '6.3') ifTrue: [ mode := 1 ]]
      ifFalse: [ "GS/64"
         (version < '2.2') ifTrue: [ mode := 1 ]].
   ^ mode
%
classmethod: GsRdbChangeNotifObj
_checkChangeNotificationMode

" Check that the current value of Global #GsRdbChangeNotificationMode 
  is appropriate for this system. "

   | oldvalue newvalue |
   oldvalue := Globals at: #GsRdbChangeNotificationMode ifAbsent: [ nil ].
   ( oldvalue isNil ) ifTrue: [
      GsFile stdout
         log: ' ';
         log: 'WARNING: GsRdbChangeNotificationMode is not set.';
         log: 'This indicates that GemConnect is not configured properly.';
         log: 'You must run GsRdbChangeNotifObj _resetChangeNotificationMode';
         log: 'as SystemUser and commit the transaction to set this.';
         log: ' '.
      self halt: 'GsRdbChangeNotificationMode is not set - see log for details' ].
   newvalue := self _changeNotificationMode.
   ( oldvalue == newvalue ) ifFalse: [ 
      GsFile stdout
         log: ' ';
         log: 'WARNING: GsRdbChangeNotification mode is incorrect for this system.';
         log: 'This probably indicates the GS system has been upgraded';
         log: 'without reconfiguring GemConnect.';
         log: 'You must run GsRdbChangeNotifObj _resetChangeNotificationMode';
         log: 'as SystemUser and commit the transaction to set this.';
         log: ' '.
      self halt: 'GsRdbChangeNotificationMode is invalid - see log for details' ].
%
classmethod: GsRdbChangeNotifObj
_resetChangeNotificationMode

" Set/reset the Global #GsRdbChangeNotificationMode 
  according to current system version."

   | oldvalue newvalue |
   oldvalue := Globals at: #GsRdbChangeNotificationMode ifAbsent: [ nil ].
   newvalue := self _changeNotificationMode.
   ( oldvalue == newvalue ) ifFalse: [ 
      Globals at: #GsRdbChangeNotificationMode put: newvalue.
      GsFile stdout
         log: ' ';
         log: 'WARNING:  GsRdbChangeNotificationMode has been updated!';
         log: 'You must commit this transaction to make it permanent!';
         log: ' ' ].
   ^ true
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - GsRdbParameters.gs
!
! ========================================================================
!
! Class 'GsRdbParameters'
!

removeallmethods GsRdbParameters
removeallclassmethods GsRdbParameters

!
! Class Category 'Accessing'
!
category: 'Accessing'
classmethod: GsRdbParameters
connectionClass

"Returns the connection class this parameter class is to be used
 with.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #connectionClass
%
!
! Instance Category 'Updating'
!
category: 'Updating'
method: GsRdbParameters
password: pswd

"Sets the password used for logging in to the relational database.

ARGUMENTS:

   pswd      <String> captured

RETURN VALUE:

   unspecified

ERRORS:

   none
"
   pswd isNil ifFalse: [ pswd _validateClass: String ].
   password := pswd
%
method: GsRdbParameters
server: serverName

"Sets the server name used for logging in to the relational database.

ARGUMENTS:

   serverName      <String> captured

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   serverName isNil ifFalse: [ serverName _validateClass: String ].
   server := serverName
%
method: GsRdbParameters
userName: usrName

"Sets the user name used for logging in to the relational database.

ARGUMENTS:

   usrName      <String> captured

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   usrName isNil ifFalse: [ usrName _validateClass: String ].
   userName := usrName
%
method: GsRdbParameters
otherAt: paramName put: value

"Associates a value with the given parameter name in the user-defined parameters
 SymbolDictionary.  Intended for use in extending connection parameters in
 conjunction with work done in the public C module.

ARGUMENTS:

   paramName      <Symbol> captured
   value          <Object> captured

RETURN VALUE:

   the parameter value

ERRORS:

   none
"

   (other isNil) ifTrue: [
      other := SymbolDictionary new.
      ].
   ^other at: paramName put: value.
%
method: GsRdbParameters
textLimit: maxTextSize

"Sets the maximum length of a text column that can be transmitted over the
 relational database connection. For Oracle this affects fields whose data 
 type is long or long raw.  The default value is 65K (65532), which is
 also the maximum allowed value.

ARGUMENTS:

   maxTextSize      <SmallInteger> captured

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   maxTextSize isNil ifFalse: [
      maxTextSize _validateClass: SmallInteger.
      (( maxTextSize > 65532 ) _or: [ maxTextSize < 0 ])
         ifTrue: [ maxTextSize _error: #rtErrArgOutOfRange ]].
   textLimit := maxTextSize
%
!
! Instance Category 'Accessing'
!
category: 'Accessing'
method: GsRdbParameters
password

"Returns the password used for logging in to the relational database.

ARGUMENTS:

   none

RETURN VALUE:

   <String> state

ERRORS:

   none
"

   ^password
%
method: GsRdbParameters
server

"Returns the server name used for logging in to the relational database.

ARGUMENTS:

   none

RETURN VALUE:

   <String> state

ERRORS:

   none
"

   ^server
%
method: GsRdbParameters
userName

"Returns the user name used for logging in to the relational database.

ARGUMENTS:

   none

RETURN VALUE:

   <String> state

ERRORS:

   none
"

   ^userName
%
method: GsRdbParameters
otherAt: paramName

"Returns the value associated with the given parameter name in the user-defined
 parameters SymbolDictionary. If there is no key in the dictionary that matches
 the parameter name, the value returned is nil.

ARGUMENTS:

   paramName      <Symbol> uncaptured

RETURN VALUE:

   the value at paramName

ERRORS:

   none
"

   (other notNil) ifTrue: [
      ^other at: paramName ifAbsent: { nil }.
      ].
   ^nil
%
method: GsRdbParameters
textLimit

"Returns the maximum length of a text column that can be transmitted over the
 relational database connection.

ARGUMENTS:

   none

RETURN VALUE:

   <SmallInteger> state

ERRORS:

   none
"

   ^textLimit
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - GsRdbConnection.gs
!
! ========================================================================
!
! Class 'GsRdbConnection'
!

removeallmethods GsRdbConnection
removeallclassmethods GsRdbConnection

!
! Instance Category 'Command Execution'
!
category: 'Command Execution'
method: GsRdbConnection
execute: commandString

"Sends the given command string to the relational database for non-cursored
 execution.  The result set is prepared for reading through an instance of
 GsRdbReadStream.

 The read stream returned by this method has instance variable and constraint
 information and can be instructed to create a tuple class. The default
 behaviour of the read stream, however, is to return result information in
 instances of OrderedCollection.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #execute:
%
method: GsRdbConnection
execute: commandString tupleClass: tupleClass

"Sends the given command string to the relational database for non-cursored
 execution.  The result set is prepared for reading through an instance of
 GsRdbReadStream with the given tuple class.  Note that it is an error to pass a
 class that cannot accept the results of the execution.  Constraint errors or
 doesNotUnderstand errors will be signaled on fetching information with a class
 that cannot accept the results of execution.

 Column mapping information, if present in the tuple class, will be used in
 installing results in instances of the class.  The updating selectors specified
 in the mapping will be used to send column values to the instances of the tuple
 class as information is read from the relational database.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   tupleClass         <Class> captured

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #execute:tupleClass:

%
method: GsRdbConnection
execute: commandString tupleClass: tupleClass columnMapping: columnMap

"Sends the given command string to the relational database for non-cursored
 execution.  The result set is prepared for reading through an instance of
 GsRdbReadStream with the given tuple class.  Note that it is an error to pass a
 class that cannot accept the results of the execution.  Constraint errors or
 doesNotUnderstand errors will be signaled on fetching information with a class
 that cannot accept the results of execution.

 Column mapping information, provided by the given column map, will be used in
 installing results in instances of the class.  The updating selectors specified
 in the mapping will be used to send column values to the instances of the tuple
 class as information is read from the relational database.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   tupleClass         <Class> captured
   columnMap          <Array> copied

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #execute:tupleClass:columnMapping:


%
method: GsRdbConnection
execute: commandString tupleClassName: aName inDictionary: symbolDict

"Sends the given command string to the relational database for non-cursored
 execution.  The result set is prepared for reading through an instance of
 GsRdbReadStream.  A new tuple class with the given name is created for reading
 results, installed in the stream, and placed in the given dictionary.

 The tuple class will have constraints created for it that match the type/class
 conversions specified in the C layer of the API.  Instance variable names for
 the class will be translations of the corresponding result set column names, if
 they are returned by the relational database.  Otherwise they are named 'iv1',
 'iv2', 'iv3', and so forth. Computed columns that are unnamed are given the
 names 'agg1', 'agg2', 'agg3', and so on.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   aName              <Symbol> captured
   symbolDict         <SymbolDictionary> uncaptured

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #execute:tupleClassName:inDictionary:

%
method: GsRdbConnection
openCursorOn: commandString

"Sends the given command string to the relational database for cursored
 execution.  The result set is prepared for reading through an instance of
 GsRdbReadStream.

 The read stream returned by this method has instance variable and constraint
 information and can be instructed to create a tuple class. The default
 behaviour of the read stream, however, is to return result information in
 instances of OrderedCollection.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #openCursorOn:
%
method: GsRdbConnection
openCursorOn: commandString tupleClass: tupleClass

"Sends the given command string to the relational database for cursored
 execution.  The result set is prepared for reading through an instance of
 GsRdbReadStream with the given tuple class.  Note that it is an error to pass a
 class that cannot accept the results of the execution.  Constraint errors or
 doesNotUnderstand errors will be signaled on fetching information with a class
 that cannot accept the results of execution.

 Column mapping information, if present in the tuple class, will be used in
 installing results in instances of the class.  The updating selectors specified
 in the mapping will be used to send column values to the instances of the tuple
 class as information is read from the relational database.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   tupleClass         <Class> captured

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #openCursorOn:tupleClass:
%
method: GsRdbConnection
openCursorOn: commandString tupleClass: tupleClass columnMapping: columnMap

"Sends the given command string to the relational database for cursored
 execution.  The result set is prepared for reading through an instance of
 GsRdbReadStream with the given tuple class.  Note that it is an error to pass a
 class that cannot accept the results of the execution.  Constraint errors or
 doesNotUnderstand errors will be signaled on fetching information with a class
 that cannot accept the results of execution.

 Column mapping information, provided by the given column map, will be used in
 installing results in instances of the class.  The updating selectors specified
 in the mapping will be used to send column values to the instances of the tuple
 class as information is read from the relational database.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   tupleClass         <Class> captured
   columnMap          <Array> copied

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #openCursorOn:tupleClass:columnMapping:
%
method: GsRdbConnection
openCursorOn: commandString tupleClassName: aName inDictionary: symbolDict

"Sends the given command string to the relational database for cursored
 execution.  The result set is prepared for reading through an instance of
 GsRdbReadStream.  A new tuple class with the given name is created for reading
 results, installed in the stream, and placed in the given dictionary.

 The tuple class will have constraints created that match the type/class
 conversions specified in the C layer of the API.  Instance variable names for
 the class will be translations of the corresponding result set column names, if
 they are returned by the relational database.  Otherwise they are named 'iv1',
 'iv2', 'iv3', and so forth. Computed columns that are unnamed are given the
 names 'agg1', 'agg2', 'agg3', and so on.

ARGUMENTS:

   commandString      <CharacterCollection> uncaptured
   aName              <Symbol> captured
   symbolDict         <SymbolDictionary> uncaptured

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #openCursorOn:tupleClassName:inDictionary:
%
method: GsRdbConnection
executeNoResults: commandString

"Sends the given command string to the relational database for execution and
 ignores any result set.

ARGUMENTS:

   commandString   <CharacterCollection> uncaptured

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #executeNoResults:
%
method: GsRdbConnection
executeReturnRowsAffected: commandString

"Sends the given logical command string (insert, update, or delete) to the
 relational database for execution and returns the number of rows affected.

ARGUMENTS:

   commandString   <CharacterCollection> uncaptured

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #executeReturnRowsAffected:
%
!
! Instance Category 'Updating'
!
category: 'Updating'
method: GsRdbConnection
parameters: parameterObj

"Sets the connection parameters object used by this connection.  Note that this
 method does not cause an automatic reconfiguration of the connection to use
 the new parameters object.  The connection must be disconnected and then re-
 connected with the new parameters for them to take affect.

ARGUMENTS:

   parameterObj      <GsRdbParameters subclass> shallow copied

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   parameters := parameterObj
%
method: GsRdbConnection
queue: queueObject

"Installs the given queue object as the 'queue' for this connection.  This queue
 is typically used to collect updates in implementations of change notification
 methods, such as #aboutToChange:newValue:, so that they can be written in batch
 to the relational database.  Usually, the given queue object is an RcQueue or
 an RcBag; however, no constraint is placed on what this object may be.

ARGUMENTS:

   queueObject      <Object> captured

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   queue := queueObject
%
!
! Instance Category 'Miscellaneous'
!
category: 'Miscellaneous'
method: GsRdbConnection
removeFromCache

"Checks the connection cache in the connection manager for an entry whose value
 is this connection and then removes it from that cache. If this connection has
 not been cached, no action is taken.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   self class _removeFromCache: self
%
method: GsRdbConnection
cacheWithName: key

"Stores this connection, using the given key, in the connection cache in the
 connection manager.  An error will occur if there is already an entry with the
 given key in the cache.

ARGUMENTS:

   key      <Object> captured

RETURN VALUE:

   unspecified

ERRORS:

   GsRdbError/badConnCacheName
"

   self class _cacheConnection: self withName: key.
%
method: GsRdbConnection
free

"Frees all open read streams associated with this connection.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   (self connected) ifTrue: [
      (self allStreams) do: [:each |
         each free.
         ].
      ].
%
!
! Instance Category 'Connection Management'
!
category: 'Connection Management'
method: GsRdbConnection
connect

"Attempts to connect to the relational database with the current connection
 parameters.  If this connection is already connected, no action is taken. If
 the connection fails, this method signals a non-continuable error.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #connect
%
method: GsRdbConnection
disconnect

"Disconnects from the relational database and frees any read streams associated
 with this connection.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #disconnect
%
method: GsRdbConnection
connected

"Returns true if this connection is connected to the relational database and
 false if not.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #connected
%
!
! Instance Category 'Synchronized Transaction Management'
!
category: 'Synchronized Transaction Management'
method: GsRdbConnection
addToCommitList

"Adds this relational database connection to the GemStone commit list so that
 its transactions will be synchronized with those of the local GemStone session
 and any other connections/sessions which are also in the commit list.

 Once added to the commit list, when the local GemStone session is asked to
 commit its current transaction, this connection will be asked to vote along
 with the other connections/sessions in the commit list. If all connections
 and sessions vote affirmatively, then each of the connections/sessions will
 be commited in turn.

 If this connection is already in the commit list, no action is taken.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   (System _commitCoordinator isNil) ifTrue: [
      System _commitCoordinator: (GsCommitList new).
      ].

   (System _commitCoordinator includes: self) ifFalse: [
      System _commitCoordinator add: self.
      ].
%
method: GsRdbConnection
removeFromCommitList

"Removes this relational database connection from the GemStone commit list so
 that its transactions will no longer be synchronized with those of the local
 GemStone session and any other connections/sessions which are still in the
 commit list.

 If this connection is not in the commit list, no action is taken.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   (System _commitCoordinator notNil) ifTrue: [
      System _commitCoordinator remove: self ifAbsent: [ nil ].
      ].
%
method: GsRdbConnection
_publicName

"Returns a string to identify the current transaction on this connection for
 synchronized transaction manmagement.

ARGUMENTS:

   none

RETURN VALUE:

   <String> new

ERRORS:

   none
"

   | publicName |

   publicName := String new.
   publicName add: 'Connection type: '.
   publicName add: (self class _rdbTypeName).

   publicName add: ', user: '.
   (parameters userName notNil) ifTrue: [
      publicName add: (parameters userName).
      ].

   publicName add: ', server: '.
   (parameters server notNil) ifTrue: [
      publicName add: (parameters server).
      ].

   ^publicName
%
method: GsRdbConnection
_voteToCommit

"Returns an integer value indicating whether the current transaction on this
 connection can commit successfully. Values returned are the following:

   0    The session was read-only and the connection can commmit successfully
   1    The session was read/write and the connection can commmit successfully
   2    The connection cannot commit successfully

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #_voteToCommit
%
method: GsRdbConnection
_commit: mode

"Attempts to commit the current transaction on this connection, returning true
 if successfull and false if not. The mode argument is used as follows:

   0   Normal commit operation
   1   Commit and release locks
   2   Commit, checkpoint, and release locks
   3   Normal commit invoked from GCI

 Note: At this time the given mode argument is ignored and this method simply
 calls #commitTransaction.

ARGUMENTS:

   mode      <Integer>
             0 (normal)
             1 (release locks)
             2 (checkpoint and release locks)
             3 (normal from GCI)

RETURN VALUE:

   <Boolean> success indication

ERRORS:

   none
"

   ^self commitTransaction
%   
method: GsRdbConnection
_abort

"Aborts the current transaction on this connection. This method simply calls
 #rollbackTransaction.

ARGUMENTS:

   none

RETURN VALUE:

   <Boolean>
   Always true.

ERRORS:

   none
"

   self rollbackTransaction.
   ^true
%
method: GsRdbConnection
_begin

"This method does nothing. It simply returns true as there is no support for
 synchronized begin transactions at this time.

ARGUMENTS:

   none

RETURN VALUE:

   <Boolean> success indication

ERRORS:

   none
"

   ^true
%
method: GsRdbConnection
_isNonBlocking

"Returns true if this connection's transaction management is handled in non-
 blocking (asynchronous) mode and false if not.

 Note: At this time all relational database connections' transactions are
 handled in blocking (synchronous) mode.

ARGUMENTS:

   none

RETURN VALUE:

   <Boolean>

ERRORS:

   none
"

   ^false
%
!
! Instance Category 'Transaction Management'
!
category: 'Transaction Management'
method: GsRdbConnection
beginTransaction

"Begins a transaction on the relational database this connection is opened on.
 Does not affect the current GemStone transaction.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #beginTransaction
%
method: GsRdbConnection
commitTransaction

"Commits the current transaction on the relational database this connection is
 opened on.  Does not affect the current GemStone transaction.

ARGUMENTS:

   none

RETURN VALUE:

   <Boolean> success indication

ERRORS:

   GsRdbError/invalidConnection
"

   " Make sure that buffered contents in write streams get flushed "
   | alreadyFlushed |
   alreadyFlushed := IdentitySet new.
   self allStreams do: [ :stream | stream _flush: alreadyFlushed ].

   ^self executeNoResults: 'commit work'.
%
method: GsRdbConnection
rollbackTransaction

"Aborts the current transaction on the relational database this connection is
 opened on.  Does not affect the current GemStone transaction.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GsRdbError/invalidConnection
"

   " Make sure that buffered contents in write streams get cleared "
   self allStreams do: [ :stream | stream clear ].

   self executeNoResults: 'rollback work'.
%
!
! Instance Category 'Accessing'
!
category: 'Accessing'
method: GsRdbConnection
parameters

"Returns the current connection parameters object used by this connection.

ARGUMENTS:

   none

RETURN VALUE:

   <GsRdbParameters subclass> state

ERRORS:

   none
"

   ^parameters
%
method: GsRdbConnection
queue

"Returns the 'queue' that was installed in this connection.  If no 'queue' was
 installed, nil is returned instead.

ARGUMENTS:

   none

RETURN VALUE:

   <Object> state

ERRORS:

   none
"

   ^queue
%
method: GsRdbConnection
allReadStreams

"OBSOLETE: retained for compatibility."

   ^self allStreams
%
method: GsRdbConnection
allStreams

"Returns all read and write streams created using this connection that have 
 not yet been freed.  If there are no read streams still in use, an empty 
 array is returned instead.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #allStreams
%
!
! Instance Category 'Stream User Actions'
!
category: 'Stream User Actions'
method: GsRdbConnection
_rsatEndAction

   self subclassResponsibility: #_rsatEndAction
%
method: GsRdbConnection
_rsfreeStreamAction

   self subclassResponsibility: #_rsfreeStreamAction
%
method: GsRdbConnection
_rsgetAllAction

   self subclassResponsibility: #_rsgetAllAction
%
method: GsRdbConnection
_rsgetRowAction

   self subclassResponsibility: #_rsgetRowAction
%
method: GsRdbConnection
_rsinstallCmapAction

   self subclassResponsibility: #_rsinstallCmapAction
%
method: GsRdbConnection
_rsisValidAction

   self subclassResponsibility: #_rsisValidAction
%
method: GsRdbConnection
_rsivcAction

   self subclassResponsibility: #_rsivcAction
%
method: GsRdbConnection
_rsnextResultsAction

   self subclassResponsibility: #_rsnextResultsAction
%
method: GsRdbConnection
_rsposAction

   self subclassResponsibility: #_rsposAction
%
method: GsRdbConnection
_rssetToEndAction

   self subclassResponsibility: #_rssetToEndAction
%
method: GsRdbConnection
_rsskipRowsAction

   self subclassResponsibility: #_rsskipRowsAction
%
!
! Class Category 'Connection Manager'
!
category: 'Connection Manager'
classmethod: GsRdbConnection
_installConnManager

"Checks to see if the connection manager has been installed in GemStone's
 current session state.  If not, a new connection manager is created and put in
 the session state.

ARGUMENTS:

   none

RETURN VALUE:

   <GsRdbConnManager>
   This is the connection manager for the GemStone's current session state.

ERRORS:

   none
"

   | connManager |

   (connManager := System sessionStateAt: 10) isNil ifTrue: [
      connManager := GsRdbConnManager new.
      System sessionStateAt: 10 put: connManager.
      ].
   ^connManager
%
classmethod: GsRdbConnection
_removeFromCache: conn

"Checks the connection cache in the connection manager for an entry whose value
 is identical to the given connection and then removes it from the cache. If the
 given connection has not been cached, no action is taken.

ARGUMENTS:

   conn      <GsRdbConnection subclass>

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   self _installConnManager connNameCacheRemove: conn
%
classmethod: GsRdbConnection
removeFromCacheKey: aKey

"Checks the connection cache in the connection manager for an entry with the
 given key, removes it from the cache, and returns the associated connection
 value. If there is no entry with the given key, returns nil.

ARGUMENTS:

   aKey      <Object>

RETURN VALUE:

   <GsRdbConnection subclass>

ERRORS:

   none
"

   ^self _installConnManager connNameCacheRemoveKey: aKey
%
classmethod: GsRdbConnection
_cacheConnection: conn withName: key

"Stores the given connection, using the given key, in the connection cache in
 the connection manager.  An error will occur if there is already an entry with
 the given key in the cache.

ARGUMENTS:

   conn         <GsRdbConnection subclass> captured
   key          <Object> captured

RETURN VALUE:

   unspecified

ERRORS:

   GsRdbError/badConnCacheName
"

   | connManager |

   connManager := self _installConnManager.
   (connManager connNameCacheAt: key) notNil ifTrue: [
      self _signalBadConnCacheName: conn name: key
   ].
   connManager connNameCacheAt: key put: conn
%
classmethod: GsRdbConnection
_loadUALib

"Checks to see if the relational database user action library corresponding to
 this connection class has been loaded.  If not, the library is loaded and the
 connection manager in the GemStone session state is updated.

 Re-implementations of this method should call the #_installConnManager method
 before checking for or loading user action libraries.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #_loadUALib
%
!
! Class Category 'Error Handling'
!
category: 'Error Handling'
classmethod: GsRdbConnection
messages

"Returns all information about an error that was raised.  Intended to provide
 access to relational database error information obtained during command
 execution.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #messages
%
classmethod: GsRdbConnection
_validateColumnMap: columnMap conn: connectObj

   (columnMap notNil) ifTrue: [
      (columnMap isKindOf: Array) ifFalse: [
         self _signalInvalidColumnMap: connectObj map: columnMap
         ].

      columnMap do: [:entry |
         (entry isKindOf: Array) ifFalse: [
            self _signalInvalidColumnMap: connectObj map: columnMap
            ].
         (entry size < 2) ifTrue: [
            self _signalBadColumnMap: connectObj column: entry
            ].
         entry do: [:each |
            (each isKindOf: CharacterCollection) ifFalse: [
               self _signalBadColumnMap: connectObj column: entry
               ].
            ].
         ].
      ].
%
classmethod: GsRdbConnection
_signalAggColumnName: connectObj name: aggColumnName

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #aggColumnName)
          args: { connectObj . nil . nil . aggColumnName }) 
          signal
%
classmethod: GsRdbConnection
_signalBadColumnMap: connectObj column: badColumn

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #badColumnMap)
          args: { connectObj . nil . nil . badColumn })
          signal
%
classmethod: GsRdbConnection
_signalBadConnCacheName: connectObj name: badKey

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #badConnCacheName)
          args: { connectObj . nil . nil . badKey }) 
          signal
%
classmethod: GsRdbConnection
_signalInvalidConnection: connectObj

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #invalidConnection)
          args: { connectObj . nil . nil . nil })
          signal
%
classmethod: GsRdbConnection
_signalInvalidStream: connectObj stream: streamObj

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #invalidStream)
          args: { connectObj . streamObj . nil . nil })
          signal
%
classmethod: GsRdbConnection
_signalInvalidColumnMap: connectObj map: columnMap

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #invalidColumnMap)
          args: { connectObj . nil . nil . columnMap })
          signal
%
classmethod: GsRdbConnection
_signalNoColumnMap: connectObj tupleClass: tupleClass

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #noColumnMap)
          args: { connectObj . nil . nil . tupleClass })
          signal
%
classmethod: GsRdbConnection
_signalNoPrimaryKeyMap: connectObj tupleClass: tupleClass

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #noPrimaryKeyMap)
          args: { connectObj . nil . nil . tupleClass })
          signal
%
classmethod: GsRdbConnection
_signalNoTableName: connectObj tupleClass: tupleClass

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #noTableName)
          args: { connectObj . nil . nil . tupleClass })
          signal
%
classmethod: GsRdbConnection
_signalQueryError: connectObj queryString: query

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #queryError)
          args: { connectObj . nil . query . nil })
          signal
%
classmethod: GsRdbConnection
_signalReadError: connectObj stream: streamObj

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #readError)
          args: { connectObj . streamObj . nil . nil })
          signal
%
classmethod: GsRdbConnection
_signalStreamAtEnd: connectObj stream: streamObj

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #streamAtEnd)
          args: { connectObj . streamObj . nil . nil })
          signal
%
classmethod: GsRdbConnection
_signalUnmappedColumnInResult: connectObj map: columnMap

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #unmappedColumnInResult)
          args: { connectObj . nil . nil . columnMap })
          signal
%
classmethod: GsRdbConnection
_signalInvalidTupleInstance: connectObj stream: streamObj tuple: tupleObj

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #invalidTupleInstance)
          args: { connectObj . streamObj . nil . tupleObj })
          signal
%
!
! Class Category 'Accessing'
!
category: 'Accessing'
classmethod: GsRdbConnection
connectionWithName:  key

"Returns the connection associated with the given key from the connection cache
 in the connection manager.  If there is no entry in the cache for the given
 key, the value returned is nil.

ARGUMENTS:

   key      <Object>

RETURN VALUE:

   <GsRdbConnection subclass>

ERRORS:

none
"

   ^self _installConnManager connNameCacheAt: key
%
classmethod: GsRdbConnection
parameterClass

"Returns the parameter class used to instantiate this connection class.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #parameterClass
%
classmethod: GsRdbConnection
_rdbTypeName

"Returns a string describing what type of connection this class represents.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #_rdbTypeName
%
classmethod: GsRdbConnection
allConnections

"Returns all successfully connected connection instances of this class. If no
 connection of this class has successfully connected during the current GemStone
 session, returns an empty array.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #allConnections
%
classmethod: GsRdbConnection
lastConnection

"Returns the last successfully connected connection instance of this class. If
 no connection of this class has successfully connected during the current
 GemStone session, returns nil.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #lastConnection
%
!
! Class Category 'Instance Creation'
!
category: 'Instance Creation'
classmethod: GsRdbConnection
newWithParameters: parameterObj

"Creates a new connection instance and initializes it with a copy of the 
 given parameter object (to avoid unintended consequences if the parameter 
 values are changed).  

 No attempt is made to connect to the relational database.

ARGUMENTS:

   parameterObj      <GsRdbParameters subclass> shallow copied

RETURN VALUE:

   <GsRdbConnection> new

ERRORS:

   none
"

   ^super new parameters: (parameterObj shallowCopy).
%
classmethod: GsRdbConnection
new

"Re-implemented to signal an error when this method is called for this class or
 one of its subclasses.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #new
%
!
! Class Category 'Datatype Conversion'
!
category: 'Datatype Conversion'
classmethod: GsRdbConnection
getRdbDateTimeFor: aDateTime

"Returns a string representing the given datetime as a valid relational database
 date and time constant.

ARGUMENTS:

   aDateTime      <DateTime> uncaptured

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #getRdbDateTimeFor:
%
classmethod: GsRdbConnection
getRdbDateFor: aDate

"Returns a string representing the given date as a valid relational database
 date constant.

ARGUMENTS:

   aDate      <Date> uncaptured

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #getRdbDateFor:
%
classmethod: GsRdbConnection
getRdbBinaryFor: aByteArray

"Returns a string representing the given byte array as a valid relational
 database binary constant.

ARGUMENTS:

   aByteArray      <ByteArray> uncaptured

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #getRdbBinaryFor:
%
classmethod: GsRdbConnection
getRdbBitFor: aBool

"Returns a string representing the given boolean as a valid relational
 database bit constant.

ARGUMENTS:

   aBool      <Boolean> uncaptured

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #getRdbBitFor:
%
classmethod: GsRdbConnection
getRdbValueFor: anObject

"Returns a string representing the given object as a valid relational database
 constant.

ARGUMENTS:

   anObject      <Object> uncaptured

RETURN VALUE:

   <String> new

ERRORS:

   none
"

   | rdbValue |

   (anObject isKindOf: Date) ifTrue: [
   "Dates converted according to the particular relational database server."
      rdbValue := self getRdbDateFor: anObject.
      ].

   (anObject isKindOf: DateTime) ifTrue: [
   "Date and time values are converted according to the particular relational
    database server."
      rdbValue := self getRdbDateTimeFor: anObject.
      ].

   (anObject isKindOf: ByteArray) ifTrue: [
   "Byte arrays are converted according to the particular relational database
    server."
      rdbValue := self getRdbBinaryFor: anObject.
      ].

   (anObject isKindOf: Boolean) ifTrue: [
   "Booleans are converted according to the particular relational database
    server."
      rdbValue := self getRdbBitFor: anObject.
      ].

   (anObject isKindOf: Number ) ifTrue: [
   "Numeric values are converted to strings and just passed as is."
      rdbValue := anObject asString.
      ].

   (anObject isNil) ifTrue: [
   "Nil values are converted to a 'NULL' representation."
      rdbValue := 'NULL'.
      ].

   (rdbValue isNil) ifTrue: [
   "All other values (including strings and characters) are converted to strings
    and delimited with single quotes."
      rdbValue := (anObject asString) quoted.
      ].

   ^rdbValue
%
!
! Class Category 'SQL Generation'
!
category: 'SQL Generation'
classmethod: GsRdbConnection
generateSQLWhereClause: keysNvals

"Returns a string representing a properly formatted SQL where clause.  The given
 key/value array is used to generate this where clause.

ARGUMENTS:

   keysNvals      <Array> contents copied
                  An array containing key name and value pairs. 
                  Uneven indexed elements are strings representing the key 
                  names. Even indexed elements are objects representing 
                  the key values.

RETURN VALUE:

   <String> new

   A string of the form: 
   'WHERE key-name1 = 'key1' AND key-name2 = 'key2' ...'.

ERRORS:

   none
"

   | whereCl val |

   keysNvals _validateClass: Array.
   whereCl := String new.
   1 to: (keysNvals size) by: 2 do: [:i |
      (whereCl isEmpty) ifTrue: [
         whereCl add: 'WHERE '.
         ]
      ifFalse: [
         whereCl add: ' AND '.
         ].

      whereCl add: (keysNvals at: i).
      val := self getRdbValueFor: (keysNvals at: (i + 1)).
      (val = 'NULL') ifTrue: [
         whereCl add: ' IS '.
         ]
      ifFalse: [
         whereCl add: ' = '.
         ].
      whereCl add: val.
      ].

   ^whereCl
%
classmethod: GsRdbConnection
generateSQLDeleteForTable: tableName keys: keysNvals

"Returns a string representing a properly formatted SQL delete command.  The
 given table name string and key/value array are used to generate this delete
 command.

ARGUMENTS:

   tableName      <String> copied
                  A string representing the name of a relational table. 
                  Used to identify the table from which the deletion 
                  will take place.

   keysNvals      <Array> contents copied
                  An array containing key name and value pairs. 
                  Uneven indexed elements are strings representing the key 
                  names. Even indexed elements are objects representing 
                  the key values. Used to identify the tuple to be deleted.

RETURN VALUE:

   <String> new

   A string of the form: 
   'DELETE FROM table-name 
       WHERE key-name1 = 'key1' AND key-name2 = 'key2' ...'.

ERRORS:

   none
"

   | sqlSt whereCl |

   sqlSt := String new. 
   whereCl := self generateSQLWhereClause: keysNvals.

   (whereCl isEmpty) ifFalse: [
      sqlSt add: 'DELETE FROM '; add: tableName.
      sqlSt add: ' '; add: whereCl.
      ].

   ^sqlSt
%
classmethod: GsRdbConnection
generateSQLUpdateForTable: tableName columns: colsNvals keys: keysNvals

"Returns a string representing a properly formatted SQL update command.  The
 given table name string, column/value array, and key/value array are used to
 generate this update command.

ARGUMENTS:

   tableName      <String> copied
                  A string representing the name of a relational table. 
                  Used to identify the table on which the update will 
                  take place.

   colsNvals      <Array> contents copied
                  An array containing column name and value pairs. 
                  Uneven indexed elements are strings representing the 
                  column names. Even indexed elements are objects 
                  representing the column values. Used to identify the 
                  columns to be updated and the new values for these columns.

   keysNvals      <Array> contents copied
                  An array containing key name and value pairs. 
                  Uneven indexed elements are strings representing the 
                  key names. Even indexed elements are objects
                  representing the key values. Used to identify the tuple 
                  to be updated.

RETURN VALUE:

   <String> new

   A string of the form: 
   'UPDATE table-name SET col-name1 = 'val1', col-name2 = 'val2' ... 
      WHERE key-name1 = 'key1' AND key-name2 = 'key2' ...'.

ERRORS:

   none
"

   | sqlSt setCl whereCl |

   sqlSt := String new.
   setCl := self generateSQLSetClause: colsNvals.
   whereCl := self generateSQLWhereClause: keysNvals.

   (setCl isEmpty or: [whereCl isEmpty]) ifFalse: [
      sqlSt add: 'UPDATE '; add: tableName.
      sqlSt add: ' '; add: setCl; add: ' '; add: whereCl.
      ].

   ^sqlSt
%
classmethod: GsRdbConnection
generateSQLInsertForTable: tableName columns: colsNvals

"Returns a string representing a properly formatted SQL insert command.  The
 given table name string and column/value array are used to generate this insert
 command.

ARGUMENTS:

   tableName      <String> copied
                  A string representing the name of a relational table. 
                  Used to identify the table on which the insert will 
                  take place.

   colsNvals      <Array> contents copied
                  An array containing column name and value pairs. 
                  Uneven indexed elements are strings representing the 
                  column names. Even indexed elements are objects
                  representing the column values. Used to identify the 
                  values and their corresponding columns where the tuple 
                  will be inserted.

RETURN VALUE:

   <String> new

   A string of the form: 
   'INSERT INTO table-name (col-name1, col-name2, ...) 
      VALUES ('val1', 'val2', ...)'.

ERRORS:

   none
"

   | sqlSt valuesCl |

   sqlSt := String new. 
   valuesCl := self generateSQLValuesClause: colsNvals.

   (valuesCl isEmpty) ifFalse: [
      sqlSt add: 'INSERT INTO '; add: tableName.
      sqlSt add: ' '; add: valuesCl.
      ].

   ^sqlSt
%
classmethod: GsRdbConnection
generateSQLValuesClause: colsNvals

"Returns a string representing a properly formatted SQL values clause. The given
 column/value array is used to generate this values clause.

ARGUMENTS:

   colsNvals      <Array> contents copied
                  An array containing column name and value pairs. 
                  Uneven indexed elements are strings representing the 
                  column names. Even indexed elements are objects 
                  representing the column values.

RETURN VALUE:

   <String> new

   A string of the form: 
   '(col-name1, col-name2, ...) VALUES ('val1', 'val2', ...)'.

ERRORS:

   none
"

   | valuesCl vals |

   colsNvals _validateClass: Array.
   valuesCl := String new.
   1 to: (colsNvals size) by: 2 do: [:i |
      (valuesCl isEmpty) ifTrue: [
         valuesCl add: '('.
         vals := String withAll: ') VALUES ('.
         ]
      ifFalse: [
         valuesCl add: ', '.
         vals add: ', '.
         ].

      valuesCl add: (colsNvals at: i).
      vals add: (self getRdbValueFor: (colsNvals at: (i + 1))).
      ].

   (vals notNil) ifTrue: [
      valuesCl add: vals; add: ')'.
      ].
   ^valuesCl
%
classmethod: GsRdbConnection
generateSQLSetClause: colsNvals

"Returns a string representing a properly formatted SQL set clause.  The given
 column/value array is used to generate this set clause.

ARGUMENTS:

   colsNvals      <Array> contents copied
                  An array containing column name and value pairs. 
                  Uneven indexed elements are strings representing the 
                  column names. Even indexed elements are objects
                  representing the column values.

RETURN VALUE:

   <String> new

   A string of the form: 
   'SET col-name1 = 'val1', col-name2 = 'val2' ...'.

ERRORS:

   none
"

   | setCl |

   colsNvals _validateClass: Array.
   setCl := String new.
   1 to: (colsNvals size) by: 2 do: [:i |
      (setCl isEmpty) ifTrue: [
         setCl := String withAll: 'SET '.
         ]
      ifFalse: [
         setCl add: ', '.
         ].

      setCl add: (colsNvals at: i); add: ' = '.
      setCl add: (self getRdbValueFor: (colsNvals at: (i + 1))).
      ].

   ^setCl
%
!
! Class methods for generating SQL statements with bind variables
!
category: 'SQL Generation'
classmethod: GsRdbConnection
generateBindSQLDeleteForTable: tableName keys: keyNames

"Returns a string representing a properly formatted SQL delete statement
 with bind variables.  The given table name string and key name array are 
 used to generate this delete statement.

ARGUMENTS:

   tableName      <String> copied
                  A string representing the name of a relational table. 
                  Used to identify the table from which the deletion will 
                  take place.

   keyNames       <Array> contents copied
                  An array containing key names. 

RETURN VALUE:

   <String> new

   A string of the form: 
   'DELETE FROM table-name 
      WHERE key-name1 = :key-name1 AND key-name2 = :key-name2 ...'.

ERRORS:

   none
"

   | sqlSt whereCl |

   sqlSt := String new. 
   whereCl := self generateBindSQLWhereClause: keyNames.

   (whereCl isEmpty) ifFalse: [
      sqlSt add: 'DELETE FROM '; add: tableName.
      sqlSt add: ' '; add: whereCl.
      ].

   ^sqlSt
%
category: 'SQL Generation'
classmethod: GsRdbConnection
generateBindSQLInsertForTable: tableName columns: colNames

"Returns a string representing a properly formatted SQL insert statement
 with bind variables.  The given table name string and column array are 
 used to generate this insert statement.

ARGUMENTS:

   tableName      <String> copied
                  A string representing the name of a relational table. 
                  Used to identify the table from which the insert will 
                  take place.

   colNames      <Array> contents copied
                 An array containing column names.

RETURN VALUE:

   <String> new

   A string of the form: 
   'INSERT INTO table-name (col-name1, col-name2, ...) 
      VALUES ( :col-name1, :col-name2, ...)'.

ERRORS:

   none
"

   | sqlSt valuesCl |

   sqlSt := String new. 
   valuesCl := self generateBindSQLValuesClause: colNames.

   (valuesCl isEmpty) ifFalse: [
      sqlSt add: 'INSERT INTO '; add: tableName.
      sqlSt add: ' '; add: valuesCl.
      ].

   ^sqlSt
%
category: 'SQL Generation'
classmethod: GsRdbConnection
generateBindSQLSetClause: colNames

"Returns a string representing a properly formatted SQL set clause with
 bind variables.  The given  column name array is used to generate this 
 set clause.

ARGUMENTS:

   colNames      <Array> contents copied
                 An array containing column names.

RETURN VALUE:

   <String> new

   A string of the form: 
   'SET col-name1 = :col-name1, col-name2 = :col-name2'

ERRORS:

   none
"

   | setCl |

   colNames _validateClass: Array.
   setCl := String new.
   1 to: (colNames size) do: [:i |
      (setCl isEmpty) ifTrue: [
         setCl := String withAll: 'SET '.
         ]
      ifFalse: [
         setCl add: ', '.
         ].

      setCl add: (colNames at: i); add: ' = :'.
      setCl add: (colNames at: i).
      ].

   ^setCl
%
category: 'SQL Generation'
classmethod: GsRdbConnection
generateBindSQLUpdateForTable: tableName columns: colNames keys: keyNames

"Returns a string representing a properly formatted SQL update statement
 with bind variables.  The given table name string, column name array, 
 and key name array are used to generate this update statement.

ARGUMENTS:

   tableName      <String> copied
                  A string representing the name of a relational table. 
                  Used to identify the table on which the update will 
                  take place.

   colNames       <Array> contents copied
                  An array containing column names.

   keysNvals      <Array> contents copied
                  An array containing key names.

RETURN VALUE:

   <String> new

   A string of the form: 
   'UPDATE table-name SET col-name1 = :col-name1, col-name2 = :col-name2 ... 
      WHERE key-name1 = :key-name1 AND key-name2 = :key-name2 ...'.

ERRORS:

   none
"

   | sqlSt setCl whereCl |

   sqlSt := String new.
   setCl := self generateBindSQLSetClause: colNames.
   whereCl := self generateBindSQLWhereClause: keyNames.

   (setCl isEmpty or: [whereCl isEmpty]) ifFalse: [
      sqlSt add: 'UPDATE '; add: tableName.
      sqlSt add: ' '; add: setCl; add: ' '; add: whereCl.
      ].

   ^sqlSt
%
category: 'SQL Generation'
classmethod: GsRdbConnection
generateBindSQLValuesClause: colNames

"Returns a string representing a properly formatted SQL values clause 
 with bind variables. The given column name array is used to generate 
 this values clause.

ARGUMENTS:

   colNames      <Array> contents copied
                 An array containing column names.

RETURN VALUE:

   <String> new

   A string of the form: 
   '(col-name1, col-name2, ...) VALUES (:col-name1, :col-name2, ...)'.

ERRORS:

   none
"

   | valuesCl vals |

   colNames _validateClass: Array.
   valuesCl := String new.
   1 to: (colNames size) do: [:i |
      (valuesCl isEmpty) ifTrue: [
         valuesCl add: '('.
         vals := String withAll: ') VALUES ('.
         ]
      ifFalse: [
         valuesCl add: ', '.
         vals add: ', '.
         ].

      valuesCl add: (colNames at: i).
	vals add: $: .
      vals add: (colNames at: i).
      ].

   (vals notNil) ifTrue: [
      valuesCl add: vals; add: ')'.
      ].
   ^valuesCl
%
category: 'SQL Generation'
classmethod: GsRdbConnection
generateBindSQLWhereClause: keyNames

"Returns a string representing a properly formatted SQL where clause
 with bind variables.  The given  key name array is used to generate 
 this where clause.

ARGUMENTS:

   keyNames       <Array> contents copied
                  An array containing key names.

RETURN VALUE:

   <String> new

   A string of the form: 
   'WHERE key-name1 = :key-name1 AND key-name2 = ':key-name2 ...'.

ERRORS:

   none
"

   | whereCl |

   keyNames _validateClass: Array.
   whereCl := String new.
   1 to: (keyNames size) do: [:i |
      (whereCl isEmpty) ifTrue: [
         whereCl add: 'WHERE '.
         ]
      ifFalse: [
         whereCl add: ' AND '.
         ].
      whereCl add: (keyNames at: i).
	whereCl add: ' = :' .
	whereCl add: (keyNames at: i).
      ].

   ^whereCl
%
!
! Class Category 'Version'
!
category: 'Version'
classmethod: GsRdbConnection
version

"Returns a string holding the representation of a floating point number that
 describes the major and minor version of the relational database interface.

ARGUMENTS:
   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/subclassResponsibility
"

   self subclassResponsibility: #version
%
!
! 42295: Error handling
!
category: Error Handling
classmethod: GsRdbConnection
error: errNum arg1: arg1 arg2: arg2 arg3: arg3 arg4: arg4

    ^ GsRdbError newWithNum: errNum 
                 args: (Array with: arg1 with: arg2 with: arg3 with: arg4)
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - GsRdbReadStream.gs
!
! ========================================================================
!
! Class 'GsRdbReadStream'
!

removeallmethods GsRdbReadStream
removeallclassmethods GsRdbReadStream

!
! Instance Category 'Accessing'
!
category: 'Accessing'
method: GsRdbReadStream
connection

"Returns the connection object that created this stream.

ARGUMENTS:

   none

RETURN VALUE:

   <GsRdbConnection subclass> state

ERRORS:

   none
"

   ^connection
%
method: GsRdbReadStream
tupleClass

"Returns this stream's tuple class. If this stream's tuple class is set, 
 returns the class to which it has been set, otherwise returns nil.

ARGUMENTS:

   none

RETURN VALUE:

   <Class> state

ERRORS:

   none
"

   ^tupleClass
%
method: GsRdbReadStream
isValid

"Returns true if this stream is a valid read stream associated to a connection
 and false if not.

ARGUMENTS:

   none

RETURN VALUE:

   <Boolean>

ERRORS:

   none
"

   ^System userAction: (connection _rsisValidAction)
           with: connection with: self
%
method: GsRdbReadStream
isExternal

"Returns true to indicate that the source of this stream's information is
 external to the GemStone image.

ARGUMENTS:

   none

RETURN VALUE:

   <Boolean>

ERRORS:

   none
"

   ^true
%
method: GsRdbReadStream
position

"Returns the current position index of this stream in its current result set.

ARGUMENTS:

   none

RETURN VALUE:

   <Integer>

ERRORS:

   GsRdbError/invalidStream
"

   ^System userAction: (connection _rsposAction)
           with: connection with: self.
%
method: GsRdbReadStream
sql

"Returns the sql statement string associated with this stream.

ARGUMENTS:

   none

RETURN VALUE:

   <String>

ERRORS:

   none
"

   ^ sql
%
method: GsRdbReadStream
bindInfo

"Returns the bindInfo for this stream.

ARGUMENTS:

   none

RETURN VALUE:

   <String>

ERRORS:

   none
"

   ^ bindInfo
%
!
! Instance Category 'Updating'
!
category: 'Updating'
method: GsRdbReadStream
_connection: aConnection

"Sets the connection object that created this stream. This method is intended to
 be called only by the GsRdbReadStream class.

ARGUMENTS:

   aConnection      <GsRdbConnection subclass> captured

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   connection := aConnection
%
category: 'Updating'
method: GsRdbReadStream
_bindInfo: bindings

"Sets the binding info for this stream. This method is intended to
 be called only by the GsRdbReadStream class.

ARGUMENTS:

   aConnection      <GsRdbConnection subclass> captured

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   bindInfo := bindings
%
method: GsRdbReadStream
_sql: sqlStatement

"Sets the sql statement for this stream.  This method is intended to be 
 called only by the GsRdbConnection classes.

ARGUMENTS:

   sqlStatement      <String> captured

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   sql := sqlStatement
%
method: GsRdbReadStream
_tupleClass: newClass

"Sets this stream's tuple class without installing it. This method is intended
 to be called only by the GsRdbReadStream class.

ARGUMENTS:

   newClass      <Class> captured

RETURN VALUE:

   unspecified

ERRORS:

   none
"
   newClass isNil ifFalse: [ newClass _validateClass: Class ].
   tupleClass := newClass.
%
method: GsRdbReadStream
tupleClass: newClass

"Sets this stream's tuple class and installs it. The tuple class must either
 have a column map whose updating selectors can accept the result columns in the
 current result set, or its first N instance variables will be structurally
 stored into by the C layer of the API, where N is the number of columns in the
 result set.

 If the tuple class is not set, instances of OrderedCollection are used to
 return rows from the stream's current result set.

ARGUMENTS:

   newClass      <Class> captured

RETURN VALUE:

   unspecified

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
   GsRdbError/invalidStream
   GsRdbError/unmappedColumnInResult
"

   self tupleClass: newClass columnMapping: (newClass rdbColumnMapping).
%
method: GsRdbReadStream
tupleClass: newClass columnMapping: columnMap

"Sets this stream's tuple class and installs it. The column mapping information
 provided by the given column map will be used in installing results in
 instances of the tuple class.  The updating selectors specified in the map will
 be used to send column values to the instances of the tuple class as the
 information is read from the current result set.

 If the tuple class is not set, instances of OrderedCollection are used to
 return rows from the stream's current result set.

ARGUMENTS:

   newClass       <Class> captured
   columnMap      <Array> copied

RETURN VALUE:

   unspecified

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
   GsRdbError/invalidStream
   GsRdbError/unmappedColumnInResult
"

   connection class _validateColumnMap: columnMap conn: connection.

   self _tupleClass: newClass.
   System userAction: (connection _rsinstallCmapAction)
          with: connection with: self with: columnMap
%
!
! Instance Category 'Miscellaneous'
!
category: 'Miscellaneous'
method: GsRdbReadStream
free

"Frees this stream's resources, cancelling its result sets. Most uses of a
 stream after freeing its resources will result in an invalid stream error.

ARGUMENTS:

   none

RETURN VALUE:

   the reciever (self)

ERRORS:

   none
"

   System userAction: (connection _rsfreeStreamAction)
          with: connection with: self.
%
!
! Instance Category 'Stream Operations'
!
category: 'Stream Operations'
method: GsRdbReadStream
do: aBlock

"Reads each row of this stream's current result set into a new tuple object in
 turn and evaluates the one argument block with the tuple object as the
 argument. See #next for more information.

ARGUMENTS:

   aBlock      <ExecutableBlock>

RETURN VALUE:

   the receiver (self)

ERRORS:

   GsRdbError/invalidStream
   GsRdbError/readError
"

   [ self atEnd ] whileFalse: [
      aBlock value: self next.
      ].
%
method: GsRdbReadStream
_upToEndInto: aCollection

"Returns the remaining rows in this stream's current result set in the given
 collection. See #next for more information. This private version of the
 method adds the resulting rows in the C API using GciAppendOops(), therefore,
 collections which cannot handle insertions of elements in this manner are not
 supported.

ARGUMENTS:

   aCollection      <Collection> uncaptured
                    A collection to which result rows will be added. 
                    Its class must implement the method #do: and work 
                    with GciAppendOops().
       
RETURN VALUE:

   <Collection> argument
   The given collection.

ERRORS:

   GsRdbError/invalidStream
   GsRdbError/readError
"

   System userAction: (connection _rsgetAllAction)
          with: connection with: self
          with: aCollection with: tupleClass.

   aCollection do: [:each |
      ((each notNil) and: [tupleClass notNil]) ifTrue: [
         each rdbPostLoad.
         ].
      ].

   ^aCollection
%
method: GsRdbReadStream
upToEndInto: aCollection

"Returns the remaining rows in this stream's current result set in the given
 collection. See #next for more information.

ARGUMENTS:

   aCollection      <Collection> uncaptured
                    A collection to which result rows will be added. 
                    Its class must implement the method #add:.

RETURN VALUE:

   <Collection> argument
   The given collection.

ERRORS:

   GsRdbError/invalidStream
   GsRdbError/readError
"

   (aCollection isKindOf: SequenceableCollection) ifTrue: [
      ^self _upToEndInto: aCollection
      ].

   self do: [:each |
      (each notNil) ifTrue: [
         aCollection add: each.
         ].
      ].

   ^aCollection
%
method: GsRdbReadStream
upToEnd

"Returns the remaining rows in this stream's current result set in an Ordered-
 Collection. See #next for more information.

ARGUMENTS:

   none

RETURN VALUE:

   <OrderedCollection> new
   An ordered collection to which the remaining result rows will be added.

ERRORS:

   GsRdbError/invalidStream
   GsRdbError/readError
"

   ^self upToEndInto: (OrderedCollection new).
%
method: GsRdbReadStream
skip: integer

"Skips the next integer entries of this stream's current result set. If the end
 of the stream is encountered during skipping, skipping is halted with no error
 signal.

ARGUMENTS:

   integer      <Integer> uncaptured

RETURN VALUE:

   the receiver (self)

ERRORS:

   GsRdbError/invalidStream
   GsRdbError/readError
"
   | smallMax cnt |

   (integer isKindOf: SmallInteger) ifTrue: [
      System userAction: (connection _rsskipRowsAction)
             with: connection with: self with: integer.
      ]
   ifFalse: [
     "it is possible that the integer value is greater than the GciOopToLong
      conversion can handle so break integer value up into chunks sized
      under the GciOopToLong limit (<2147483648). To avoid creating extra
      LongInteger objects, we'll use the GS/S SmallInteger>>maximumValue."
      smallMax := 536870911.
      cnt := integer // smallMax. 
      1 to: cnt do: [:i |
         System userAction: (connection _rsskipRowsAction)
                with: connection with: self with: smallMax.
         ].
      cnt := integer \\ smallMax.
      System userAction: (connection _rsskipRowsAction)   
                with: connection with: self with: cnt.
      ].
%
method: GsRdbReadStream
atEnd

"Indicates whether this read stream is at the end of its current result set.

ARGUMENTS:

   none

RETURN VALUE:

   <Boolean>
   True if the stream is at its end and false if not.

ERRORS:

   GsRdbError/invalidStream
"

   ^System userAction: (connection _rsatEndAction)
           with: connection with: self
%
method: GsRdbReadStream
nextPut: anObject

"Re-implemented to signal an error when this method is called for this class or
 one of its subclasses.

ARGUMENTS:

   anObject      <Object>

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #nextPut:
%
method: GsRdbReadStream
nextPutAll: aCollection

"Re-implemented to signal an error when this method is called for this class or
 one of its subclasses.

ARGUMENTS:

   aiCollection      <Collection>

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #nextPutAll:
%
method: GsRdbReadStream
cr

"Re-implemented to signal an error when this method is called for this class or
 one of its subclasses.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #cr
%
method: GsRdbReadStream
lf

"Re-implemented to signal an error when this method is called for this class or
 one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #lf
%
method: GsRdbReadStream
tab

"Re-implemented to signal an error when this method is called for this class or
 one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #tab
%
method: GsRdbReadStream
space

"Re-implemented to signal an error when this method is called for this class or
 one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #space
%
!
! fix 13220: refactor #next and #next: to use lower-level 
!            #nextUsing: and #nextCount:
!            
!
method: GsRdbReadStream
nextUsing: tupleObject

"Reads the next row of this read stream's current result set into the supplied
 tuple object.  If the tuple class of the read stream has a column mapping 
 defined, the mapping's updating selectors are used to install the values 
 from corresponding result columns. Once the tuple object has been setup, 
 it is sent the message #rdbPostLoad for any user-defined post-load processing
 that needs to be done.  If an instance of OrderedCollection is used for the 
 tuple object, no #rdbPostLoad message is sent.

 If the end of the stream is reached, a stream at end error is signaled. 

ARGUMENTS:

   tupleObject    either: <Object> uncaptured
                          instance of a tupleClass
                  or:     <OrderedCollection> uncaptured

RETURN VALUE:

   <Object>
   the supplied tupleObject argument

ERRORS:

   GsRdbError/streamAtEnd
   GsRdbError/invalidStream
   GsRdbError/readError
   GsRdbError/selectUnboundError (for GsRdbSelectStreams)
"
   | result |

   result := System userAction: (connection _rsgetRowAction) with: connection
                    with: self with: tupleObject.

   (result isNil) ifTrue: [
      connection class _signalStreamAtEnd: connection stream: self.
      ]
   ifFalse: [
      (tupleClass notNil) ifTrue: [
         tupleObject rdbPostLoad.
         ].
      ].

   ^result
%
method: GsRdbReadStream
next

"Reads the next row of this read stream's current result set into a new tuple
 object. If the tuple class of the read stream has a column mapping defined, 
 the mapping's updating selectors are used to install the values from 
 corresponding result columns. Once the tuple object has been created and 
 setup, it is sent the message #rdbPostLoad for any user-defined post-load 
 processing that needs to be done.  If the tuple class is not set, an instance
 of OrderedCollection is used instead of a tuple object to return the next row.
 In this case no #rdbPostLoad message is sent.

 If the end of the stream is reached, a stream at end error is signaled. 


ARGUMENTS:

   none

RETURN VALUE:

   <Object>
   An instance of this stream's tuple class. 
   By default this is OrderedCollection.

ERRORS:

   GsRdbError/streamAtEnd
   GsRdbError/invalidStream
   GsRdbError/readError
   GsRdbError/selectUnboundError (for GsRdbSelectStreams)
"

   | row |

   row := (tupleClass isNil) ifTrue: [ 
             OrderedCollection new 
             ]
          ifFalse: [ 
             tupleClass new 
             ].
   ^ self nextUsing: row
%
method: GsRdbReadStream
nextCount: integer

"Returns the next integer rows of this read stream's current result set in an
 ordered collection. See #next for more information.

ARGUMENTS:

   integer      <Integer> uncaptured

RETURN VALUE:

   <OrderedCollection> new
   An ordered collection to which the integer result rows will be added.

ERRORS:

   GsRdbError/streamAtEnd
   GsRdbError/invalidStream
   GsRdbError/readError
"

   ^self next: integer into: (OrderedCollection new).
%
method: GsRdbReadStream
next: integerOrTuple

"If the argument is an integer, return the next <integerOrTuple> rows
 (see #nextCount:). Otherwise, treat the argument as an instance of the 
 tupleClass for this read stream and return the next result row in that 
 instance (see #nextUsing:).

ARGUMENTS:

   integerOrTuple      either: <Integer> uncaptured
                       or:     <Object> uncaptured
                               instance of a tupleClass

RETURN VALUE:

   Depends on if integerOrTuple is <Integer> or <Object>

   If <Integer>

      <OrderedCollection> new
      An ordered collection to which the integer result rows will be added.

   If <Object>

      <Object>
      The tupleObject passed in as an argument containing result row

ERRORS:

   GsRdbError/streamAtEnd
   GsRdbError/invalidStream
   GsRdbError/readError
"

  ^ (integerOrTuple isKindOf: Integer)
  ifTrue: [ self nextCount: integerOrTuple ]
  ifFalse: [ self nextUsing: integerOrTuple ]
%
method: GsRdbReadStream
next: integer into: aCollection

"Returns the next integer rows of this read stream's current result set in the
 given collection. See #next for more information.

ARGUMENTS:

   integer          <Integer> uncaptured

   aCollection      <Collection> uncaptured
                    A collection to which the integer result rows will be 
                    added. Its class must implement the method #add: .

RETURN VALUE:

   <Collection> argument
   The given collection.

ERRORS:

   GsRdbError/streamAtEnd
   GsRdbError/invalidStream
   GsRdbError/readError
"

   | row |

   integer timesRepeat: [ 
      row := self next.
      aCollection add: row
      ].
   ^aCollection
%
method: GsRdbReadStream
nextResultSet

"Skips to the beginning of the next result set if there is one. The tuple class
 used for instantiating results is the same as for the previous result set. If
 there isn't a next result set false is returned. Otherwise true is returned.

ARGUMENTS:

   none

RETURN VALUE:

   <Boolean>
   True if there is a next result set, false if not.

ERRORS:

   GsRdbError/aggColumnName
   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidStream
   GsRdbError/OracleError
   GsRdbError/unmappedColumnInResult
"

   ^self nextResultSetWith: tupleClass
%
method: GsRdbReadStream
nextResultSetWith: newClass

"Skips to the beginning of the next result set if there is one. The tuple class
 used for instantiating results is set to the given class. The tuple class must
 either have a column map whose updating selectors can accept the result columns
 in the new result set, or its first N instance variables will be structurally
 stored into by the C layer of the API, where N is the number of columns in the
 result set. If there isn't a next result set false is returned. Otherwise true
 is returned.

ARGUMENTS:

   newClass      <Class> captured

RETURN VALUE:

   <Boolean>
   True if there is a next result set, false if not.

ERRORS:

   GsRdbError/aggColumnName
   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidStream
   GsRdbError/oracleError
   GsRdbError/unmappedColumnInResult
"

   | columnMap |

   (newClass isNil) ifTrue: [
      columnMap := nil
      ]
   ifFalse: [
      columnMap := newClass rdbColumnMapping
      ].  

   ^self nextResultSetWith: newClass columnMapping: columnMap
%
method: GsRdbReadStream
nextResultSetWith: newClass columnMapping: columnMap

"Skips to the beginning of the next result set if there is one. The tuple class
 used for instantiating results is set to the given class. The column mapping
 information provided by the given column map parameter will be used in
 installing results in instances of the tuple class.  The updating selectors
 specified in the map will be used to send column values to the instances of the
 tuple class as information is read from the relational database. If there isn't
 a next result set false is returned. Otherwise true is returned.

ARGUMENTS:

   newClass       <Class> captured
   columnMap      <Array> copied

RETURN VALUE:

   <Boolean>

ERRORS:

   GsRdbError/aggColumnName
   GsRdbError/badColumnMap
   GsRdbError/columnAllocError
   GsRdbError/invalidColumnMap
   GsRdbError/invalidStream
   GsRdbError/oracleError
   GsRdbError/unmappedColumnInResult
"

   connection class _validateColumnMap: columnMap conn: connection.

   self _tupleClass: newClass.
   ^System userAction: (connection _rsnextResultsAction)
           with: connection with: self with: columnMap
%
method: GsRdbReadStream
setToEnd

"Sets the position reference of this read stream at the end of the current
 result set and returns the position index.

ARGUMENTS:

   none

RETURN VALUE:

   <Integer>
   The position index.

ERRORS:

   GsRdbError/invalidStream
   GsRdbError/readError
"

   ^System userAction: (connection _rssetToEndAction)
           with: connection with: self
%
method: GsRdbReadStream
flush

"Flush buffered contents of stream to RDB.  
 This is a no-op for read streams."

^ self
%
method: GsRdbReadStream
_flush: alreadyFlushed

"Flush buffered contents of stream to RDB.  
 This is a no-op for read streams."

^ self
%
method: GsRdbReadStream
clear

"Clear buffered contents of stream.
 This is a no-op for read streams."

^ self
%
!
! Instance Category 'Tuple Class Creation'
!
! bugfix 37102: add additional Oracle info fields 
!
category: 'Tuple Class Creation'
method: GsRdbReadStream
instanceVariablesAndConstraints

"Returns the instance variables and constraints that were determined by the C
 layer of the API in analyzing this stream's current result set. The information
 can be used to create a tuple class that may be installed in this stream to
 instantiate result rows.

 Each field contains 7 elements:

 1:  The column name
 2:  The GemStone class this column is constrained to
 3:  The Oracle DataType code for this column
     (Refer to Oracle Call Interface Programmer's Guide for details)
 4:  The maximum field length in bytes for this column
 5:  The Oracle precision for this column
 6:  The Oracle scale for this column
 7:  A boolean flag indicating if null values are accepted

 Instance variable names for the tuple class will be translations of the current
 result set column names, if they are returned by the relational database.
 Otherwise they are named 'iv1', 'iv2', 'iv3', and so forth.  Computed columns
 that are unnamed are given the names 'agg1', 'agg2', 'agg3', and so on.

 The following rules are also applied when translating column names to GemStone
 instance variable names:

   1 - An instance variable name translated from a column name beginning with a
       character which is not alphabetic or an underscore will begin with the
       character 'g'. E.g., the column '1Num' will become the instance variable
       'g1Num'.

   2 - An instance variable name translated from a column name which is composed
       of characters other than alphanumeric characters or underscores will
       contain underscores in place of these characters. E.g., the column 'id#'
       will become the instance variable 'id_'.

   3 - Rule 1 is applied before rule 2, therefore, an instance variable transl-
       ated from a column name which begins with a non-alphanumeric character
       (and which is not an underscore) will begin with the character 'g'. E.g.,
       the column '$price' will become the instance variable 'g_price'.

 Changes to this column name to instance variable name translation can be made
 in the public C module provided with the API.

ARGUMENTS:

   none

RETURN VALUE:

   <Array>     state (invariant)
               An invariant Array of two element arrays. The first element 
               in each two element array is a string representing an instance 
               variable name and the second element is the corresponding 
               constraint class.

ERRORS:

   GsRdbError/invalidStream
"

   | result str1 str2 duplicate dupCounter |

   result := System userAction: (connection _rsivcAction)
           with: connection with: self.

   (result isNil) ifTrue: [
      connection class _signalInvalidStream: connection stream: self.
      ]
   ifFalse: [
      " look for duplicates and tack on numbers to make them unique "
      1 to: result size do: [:i |
         str1 := (result at: i) at: 1.
         dupCounter := 0.
         str2 := str1.
         duplicate := true.
         [ duplicate ] whileTrue: [ 
            duplicate := false.
            1 to: i-1 do: [:j |
               ((result at: j) at: 1) = str1 ifTrue: [
                  duplicate := true.
                  dupCounter := dupCounter + 1.
                  str1 := String withAll: str2.
                  str1 addAll: dupCounter asString.
                  ].
               ].
            ].
         str1 = str2 ifFalse: [
            (result at: i) at: 1 put: str1
            ].
         ].
      ].
   result immediateInvariant.

   ^result
%
method: GsRdbReadStream
createTupleClassNamed: aName inDictionary: symbolDict

"Creates a tuple class with the given name, installs it in this read stream,
 places it in the given dictionary, and then returns it.  The new tuple class
 will have constraints created for it that match the type/class conversions
 specified in the C layer of the API. If the given dictionary is nil, a new
 instance of SymbolDictionary will be created to store the class.

 Instance variable names for the tuple class will be generated from the method
 #instanceVariablesAndConstraints .

ARGUMENTS:

   aName           <Symbol> captured
   symbolDict      <SymbolDictionary> uncaptured

RETURN VALUE:

   <Class> new

ERRORS:

   GsRdbError/invalidStream
"

   | instVarConsArray newClass symDict |

   "Make sure the symbol dictionary given is valid"
   symDict := symbolDict.
   (symDict isNil) ifTrue: [
      symDict := SymbolDictionary new.
      ]
   ifFalse: [
      symDict _validateClass: SymbolDictionary.
      ].

   "Get information about instance variables and contsraints"
   instVarConsArray := self instanceVariablesAndConstraints collect: [:x |
      Array with: (x at: 1) with: (x at: 2)].

   "Create the new tuple class and constrain its instance variables"
   newClass := Object subclass: aName
                      instVarNames: (instVarConsArray collect: [:e | e at: 1])
                      classVars: #()
                      classInstVars: #()
                      poolDictionaries: #()
                      inDictionary: symDict.
   self tupleClass: newClass.
   ^tupleClass
%
!
! Class Category 'Instance Creation'
!
category: 'Instance Creation'
classmethod: GsRdbReadStream
new

"Re-implemented to signal an error when this method is called for this class or
 one of its subclasses.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #new.
%
classmethod: GsRdbReadStream
_newForConnection: connection

"Creates a read stream for the given connection. This method is intended to be
 called only by GsRdbConnection subclasses and their user-actions.

ARGUMENTS:

   connection      <GsRdbConnection subclass> captured

RETURN VALUE:

   <GsRdbReadStream> new

ERRORS:

   none
"

   ^super new _connection: connection.
%
classmethod: GsRdbReadStream
_newForConnection: connection bindInfo: bindings

"Creates a read stream for the given connection and provide bindInfo.

ARGUMENTS:

   connection      <GsRdbConnection subclass> captured

RETURN VALUE:

   <GsRdbReadStream> new

ERRORS:

   none
"

   ^super new _connection: connection; _bindInfo: bindings
%
category: 'ErrorHandling'
method: GsRdbReadStream
_signalBindingValueError: msg

   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #bindingValueError)
          args: { connection . self . sql . msg }) signal
%
category: 'Binding Management'
method: GsRdbReadStream
_validateBindingValues: anArray

"Check that values being passed are appropriate for given bindings"

((anArray size) = (bindInfo size - 1))
   ifFalse: [ self _signalBindingValueError: 'Incorrect number of values' ].
1 to: anArray size do: [:i | | b |
   b := bindInfo at: i + 1.
   ((b size < 3) or: [(b at: 3) ~= #OUT]) ifTrue: [
      ((anArray at: i) isKindOf: (b at: 2))
         ifFalse: [ self _signalBindingValueError:
                         'Value at index ' , 
                         (i asString) , 
                         ' is incorrect type']]].
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - GsRdbWriteStream.gs
!
! ========================================================================
!
! Class 'GsRdbWriteStream'
!
removeallmethods GsRdbWriteStream
removeallclassmethods GsRdbWriteStream
!
! Instance Category 'DependencyList'
!
category: 'DependencyList'
method: GsRdbWriteStream
dependencyList

"Returns the value of the dependencyList instance variable."

^ dependencyList
%
category: 'DependencyList'
method: GsRdbWriteStream
addDependency: aWriteStream

"Add a writeStream to this stream's dependencyList."

   dependencyList isNil ifTrue: [ 
      dependencyList := IdentitySet new ].
   ^ dependencyList add: aWriteStream
%
category: 'DependencyList'
method: GsRdbWriteStream
removeDependency: aWriteStream

"Remove a writeStream from this stream's dependencyList."

   ^ dependencyList isNil ifFalse: [
        dependencyList remove: aWriteStream ]
%
!
! Instance Category 'Stream Operations'
!
category: 'Stream Operations'
method: GsRdbWriteStream
atEnd

"Indicates whether this stream is at the end of its current result set.

ARGUMENTS:

   none

RETURN VALUE:

   <Boolean>
   Always True for a write stream.

ERRORS:

   GsRdbError/invalidReadStream
"
   ^ true

%
method: GsRdbWriteStream
clear

"Clear the buffered contents of this writestream without actually writing 
 contents to DB.  Should be used only in conjunction with a DB rollback. 
 (See GsRdbConnection>>rollbackTransaction)

ARGUMENTS:

  None

RETURN VALUE:

  receiver

ERRORS:

   GsRdbError/invalidReadStream
"

   ^ System userAction: (connection _rsflushStreamAction) 
      with: connection with: self with: false
%
method: GsRdbWriteStream
flush

"Flush the buffered contents of this write stream to the DB.

ARGUMENTS:

  None

RETURN VALUE:

  the receiver

ERRORS:

   GsRdbError/invalidReadStream
   GsRdbError/oracleError
 "

   ^ self _flush: (IdentitySet new).
%
method: GsRdbWriteStream
_flush: alreadyFlushed

   " Flush if we haven't already done so. "

   " Check if we're already on the flushed list.. "
   (alreadyFlushed includes: self) ifFalse: [

      " add self to alreadyFlushed, to avoid loops.."
      alreadyFlushed add: self.

      " first, flush any streams dependent on this one"
      dependencyList isNil ifFalse: [
         dependencyList do: [:stream | 
            stream _flush: alreadyFlushed ]]].

   ^ System userAction: (connection _rsflushStreamAction) 
      with: connection with: self with: nil

%
method: GsRdbWriteStream
next

"Re-implemented to signal an error when this method is called for this 
 class or one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #next
%
method: GsRdbWriteStream
next: integer

"Re-implemented to signal an error when this method is called for this 
 class or one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #next:
%
method: GsRdbWriteStream
next: integer into: aCollection

"Re-implemented to signal an error when this method is called for this 
 class or one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #next:into
%
method: GsRdbWriteStream
nextPut: anObject

"Put this object out to the writestream.  The effect will depend on the 
 type of DB statement associated with this writeStream: either UPDATE, 
 INSERT, or DELETE 

ARGUMENTS:

  anObject	<Object>

RETURN VALUE:

  anObject      <Object>

ERRORS:

   GsRdbError/invalidReadStream
   GsRdbError/oracleError
"

   ( anObject isKindOf: tupleClass ) ifFalse: [ 
      connection class _signalInvalidTupleInstance: connection 
         stream: self tuple: anObject ].
   ^ System userAction: (connection _rsputRowAction) 
      with: connection with: self with: anObject.
%
method: GsRdbWriteStream
nextPutAll: aCollection

"Put a collection of objects out into the writestream.  The effect will 
 depend on the type of Oracle statement associated with this writeStream: 
 either UPDATE, INSERT, or DELETE.

ARGUMENTS:

  aCollection	<Collection>

RETURN VALUE:

  receiver

ERRORS:

   GsRdbError/invalidReadStream
   GsRdbError/oracleError
"

   aCollection do: [ :entry | self nextPut: entry ].
   ^ self
%
method: GsRdbWriteStream
nextResultSet

"Re-implemented to signal an error when this method is called for this 
 class or one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #nextResultSet
%
method: GsRdbWriteStream
nextResultSetWith: newClass

"Re-implemented to signal an error when this method is called for this 
 class or one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #nextResultSetWith:
%
method: GsRdbWriteStream
nextResultSetWith: newClass columnMapping: columnMap

"Re-implemented to signal an error when this method is called for this 
 class or one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #nextResultSetWith:columnMapping:
%
method: GsRdbWriteStream
setToEnd

"Sets the position reference of this write stream at the end of the current
 result set and returns the position index.

 Since this is a write stream, the position reference is ALWAYS at the end,
 so this just returns the position

ARGUMENTS:

   none

RETURN VALUE:

   <Integer>
   The position index.

ERRORS:

   GsRdbError/invalidReadStream
   GsRdbError/oracleError
"

^ self position

%
method: GsRdbWriteStream
skip: integer

"Re-implemented to signal an error when this method is called for this 
 class or one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #skip:
%
method: GsRdbWriteStream
upToEnd

"Re-implemented to signal an error when this method is called for this 
 class or one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #upToEnd
%
method: GsRdbWriteStream
upToEndInto: aCollection

"Re-implemented to signal an error when this method is called for this 
 class or one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #upToEndInto:
%
method: GsRdbWriteStream
_upToEndInto: aCollection

"Re-implemented to signal an error when this method is called for this 
 class or one of its subclasses.

ARGUMENTS:

   none 

RETURN VALUE:

   unspecified

ERRORS:

   GemStoneError/shouldNotImplement
"

   self shouldNotImplement: #_upToEndInto:
%
method: GsRdbWriteStream
free

"Frees this stream's resources, cancelling its result sets. Most uses of a
 stream after freeing its resources will result in an invalid stream error.

ARGUMENTS:

   none

RETURN VALUE:

   the reciever

ERRORS:

   none
"

   "For a write stream, make sure we flush contents first"
   self flush.

   ^ super free.
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - GsRdbSelectStream.gs
!
! ========================================================================
!
! Class 'GsRdbSelectStream'
!
removeallmethods GsRdbSelectStream
removeallclassmethods GsRdbSelectStream
!
! Instance Category 'Stream Operations'
!
method: GsRdbSelectStream
bindVars: anArray

"Use anArray to determing bindings for bound sql select statement.
 You can now read back results on this stream using #next.

 See #nextPut: for details.
"

^ self nextPut: anArray
%
 
method: GsRdbSelectStream
nextPut: anArray

"Use anArray to determing bindings for bound sql select statement.
 You can now read back results on this stream using #next.
 
ARGUMENTS:

  anArray	<Array>

RETURN VALUE:

  aReadStream   <GsRdbReadStream>

ERRORS:

   GsRdbError/invalidReadStream
   GsRdbError/oracleError
   GsRdbError/bindingValueError
"
   self _validateBindingValues: anArray.
   bindValues := anArray.
   ^ System userAction: (connection _rsputRowAction) 
      with: connection with: self with: anArray.
%
method: GsRdbSelectStream
next

"Reads the next row of this read stream's current result set into a new tuple.
 See GsRdbReadStream>>next for more details. "

  bindValues isNil ifTrue: [ 
   (GsRdbError newWithNum: (GsRdbErrorSymbols at: #selectUnboundError)
               args: { connection . self . sql . nil }) signal ].
  ^ super next
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - GsRdbBlockStream.gs
!
! ========================================================================
!
! Class 'GsRdbBlockStream'
!
removeallmethods GsRdbBlockStream
removeallclassmethods GsRdbBlockStream
!
! Instance Category 'Stream Operations'
!
method: GsRdbBlockStream
bindVars: anArray

"Use anArray to determing bindings for a SQL block statement and return an
 array of output values from the execution of the block.

 See #nextPut: for details.
"

^ self nextPut: anArray
%

method: GsRdbBlockStream
nextPut: anArray

"Use anArray to determing bindings for a SQL block statement and return an
 array of output values from the execution of the block.
 
ARGUMENTS:

  anArray	<Array>

RETURN VALUE:

  anArray   <Array>

ERRORS:

   GsRdbError/invalidReadStream
   GsRdbError/oracleError
   GsRdbError/bindingValueError
"
   self _validateBindingValues: anArray.
   ^ System userAction: (connection _rsputRowAction) 
      with: connection with: self with: anArray.
%
method: GsRdbBlockStream
nextPutAll: aCollection

"Repeated nextPut:, with all results returned as an ordered collection.

ARGUMENTS:

  aCollection	<Collection>

RETURN VALUE:

  receiver

ERRORS:

   GsRdbError/invalidReadStream
   GsRdbError/oracleError
   GsRdbError/bindingValueError
"

   | results |
   results := OrderedCollection new.

   aCollection do: [ :entry | results add: (self nextPut: entry) ].
   ^ results
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - GsRdbError.gs
!
! ========================================================================
!
! Class 'GsRdbError'
!

removeallmethods GsRdbError
removeallclassmethods GsRdbError

!
! Class Category 'Operations'
!
category: 'Operations'
classmethod: GsRdbError
newWithNum: aNumber args: args

   ^ super new initializeOn: aNumber args: args
%
!
! Instance Category 'Initialization'
category: 'Initialization'
method: GsRdbError
initializeOn: aNumber args: args

   self _number: aNumber.
   self args: args.
   self reason: (GsRdbErrorSymbols keyAtValue: aNumber).
   self details: ((GsRdbErrors at: #English) at: aNumber).
   connection    := args at: 1.
   stream        := args at: 2.
   sql           := args at: 3.
   rdbErrDetails := args at: 4.
   "fix 48431"
   self _description: (self _legacyDetails: (self details))
%
!
! Instance Category 'Instance initialization'
!
category: 'Example'
method: GsRdbError
initialize

  gsResumable := true.
  gsTrappable := true.

%
!
! Instance Category 'Accessing'
!
category: 'Accessing'
method: GsRdbError
connection

  ^ connection
%
method: GsRdbError
stream

  ^ stream
%
method: GsRdbError
sql

  ^ sql
%
method: GsRdbError
rdbErrDetails

  ^ rdbErrDetails
%
! fix 48431
method: GsRdbError
description

  ^ messageText
%


! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - Object.gs
!
! ========================================================================

!
! Instance Category 'Rdb Methods'
!
category: 'Rdb Methods'
method: Object
rdbPostLoad

"This message should be re-implemented by developers to provide custom behavior
 for their application classes.  The default behavior is to simply return self.

 This message is sent by GemStone to each tuple object after it is loaded with
 information from the relational database. It is not intended to be called by
 methods implemented by application developers.

 Typical re-implementations will include actions such as determining whether the
 object should be tracked for changes, time stamping the object for the purpose
 of tracking coherency between data read into GemStone and the data in the
 relational database, or storing a copy of the object for use in generating
 updates.

 Care should be taken to have re-implementations return self.

ARGUMENTS:

   none

RETURN VALUE:

   unspecified

ERRORS:

   none
"
%
!
! Instance Category 'Rdb SQL Generation'
!
category: 'Rdb SQL Generation'
method: Object
generateSQLDelete: conn

"Returns a string representing a properly formatted SQL delete command.  The
 receiver's class must re-implement the #rdbTableName and #rdbPrimaryKeyMaps
 messages, as they will be used to identify the table and tuple, respectively,
 for which the delete command will be generated. If these messages return the
 default, nil, an error will occur.

 The generated SQL delete command corresponds to the deletion of the receiver's
 tuple representation from a relational database table.

ARGUMENTS:

   conn      <GsRdbConnection subclass>
             A connection instance used to determine the variant of 
             SQL to generate.

RETURN VALUE:

   <String> new
   A string of the form: 
   'DELETE FROM table-name 
      WHERE key-name1 = 'key1' AND key-name2 = 'key2' ...'.

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
   GsRdbError/noPrimaryKeyMap
   GsRdbError/noTableName
"

   ^self generateSQLDelete: conn
         table: (self class rdbTableName) 
         keys: (self class rdbPrimaryKeyMaps)
%
method: Object
generateSQLInsert: conn

"Returns a string representing a properly formatted SQL insert command.  The
 receiver's class must re-implement the #rdbTableName and #rdbColumnMapping
 messages, as they will be used to identify the table and tuple, respectively,
 for which the insert command will be generated. If these messages return the
 default, nil, an error will occur.

 The generated SQL insert command corresponds to the insertion of the receiver's
 tuple representation into a relational database table.

ARGUMENTS:

   conn      <GsRdbConnection subclass>
             A connection instance used to determine the variant of 
             SQL to generate.

RETURN VALUE:

   <String> new
   A string of the form: 
   'INSERT INTO table-name 
      (col-name1, col-name2, ...) VALUES ('val1', 'val2', ...)'.

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
   GsRdbError/noColumnMap
   GsRdbError/noTableName
"

   ^self generateSQLInsert: conn
         table: (self class rdbTableName)
         columns: (self class rdbColumnMapping)
%
method: Object
getNamesAndValuesFor: columnMap

"Uses the given column map to obtain column names and corresponding values from
 the receiver, and returns an array containing these column names and values
 as elements. Intended to be used to generate the information necessary for
 constructing SQL statements. 

 If the given column map is empty or nil, returns an empty array.

ARGUMENTS:

   columnMap      <Array>
                  A properly formatted column mapping.

RETURN VALUE:

   <Array> new    An array containing column name and value pairs. 
                  Uneven indexed elements are strings representing the 
                  column names. Even indexed elements are strings 
                  representing the column values.

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
"

   | colsNvals colEntry entSize accessor |

   GsRdbConnection _validateColumnMap: columnMap conn: nil.

   colsNvals := Array new.
   1 to: (columnMap size) do: [:i |
      colEntry := columnMap at: i.
      entSize := colEntry size.

      (entSize > 2) ifTrue: [
         accessor := colEntry at: 3.
         ]
      ifFalse: [
         accessor := colEntry at: 2.
         ].

      colsNvals add: (colEntry at: 1); add: (self perform: accessor).
      ].

   ^colsNvals
%
method: Object
getNamesAndValuesFor: columnMap scope: aScope

"Uses the given column map to obtain column names and corresponding values, for
 those instance variable names provided in the given scope, from the receiver,
 and returns an array containing these column names and values as elements. 
 Intended to be used to generate the information necessary for constructing SQL
 statements.

 If the instance variable names provided in the given scope do not match any
 instance variable names specified in the given column map, an empty array is
 returned. If the given column map is empty or nil, an empty array is returned.

ARGUMENTS:

   columnMap      <Array>
                  A properly formatted column mapping.

   aScope         <Collection subclass>
                  A collection of symbols corresponding to instance variable 
                  names to be matched with entries in a properly formatted 
                  column mapping. Its class must implement the method #detect:.

RETURN VALUE:

   <Array> new    An array containing column name and value pairs. 
                  Uneven indexed elements are strings representing the 
                  column names. Even indexed elements are strings 
                  representing the column values.

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
"

   | colsNvals colEntry entSize accessor sym |

   GsRdbConnection _validateColumnMap: columnMap conn: nil.

   colsNvals := Array new.
   1 to: (columnMap size) do: [:i |
      colEntry := columnMap at: i.
      entSize := colEntry size.

      sym := colEntry at: 2.
      accessor := aScope detect: [:scopeSym | sym asSymbol == scopeSym asSymbol ]
                         ifNone: [ nil ].

      "If there is a match get the value, using an accessor if there is one."
      (accessor notNil) ifTrue: [
         (entSize > 2) ifTrue: [
            accessor := colEntry at: 3.
            ].
         colsNvals add: (colEntry at: 1); add: (self perform: accessor).
         accessor := nil.
         ].
      ].

   ^colsNvals
%
method: Object
generateSQLDelete: conn table: tableName keys: keyMap

"Returns a string representing a properly formatted SQL delete command.  The
 given table name and key map will be used to identify the table and tuple,
 respectively, for which the delete command will be generated.

 The generated SQL delete command corresponds to the deletion of the receiver's
 tuple representation from a relational database table.

ARGUMENTS:

   conn           <GsRdbConnection subclass>
                   A connection instance used to determine the variant of 
                   SQL to generate.

   tableName      <String> copied
                  A string representing the name of a relational table. 
                  Used to identify the table from which the deletion will 
                  take place.

   keyMap         <Array> contents copied
                  A properly formatted key mapping. Used to identify the 
                  tuple to be deleted.

RETURN VALUE:

   <String> new
   A string of the form: 
   'DELETE FROM table-name 
      WHERE key-name1 = 'key1' AND key-name2 = 'key2' ...'.

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
   GsRdbError/noPrimaryKeyMap
   GsRdbError/noTableName
"

   (tableName isNil) ifTrue: [
       conn class _signalNoTableName: conn tupleClass: (self class)
       ].
   (keyMap isNil) ifTrue: [
       conn class _signalNoPrimaryKeyMap: conn tupleClass: (self class)
       ].

   ^conn class generateSQLDeleteForTable: tableName
               keys: (self getNamesAndValuesFor: keyMap).
%
method: Object
generateSQLInsert: conn table: tableName columns: columnMap

"Returns a string representing a properly formatted SQL insert command.  The
 given table name and column map will be used to identify the table and tuple,
 respectively, for which the insert command will be generated. 

 The generated SQL insert command corresponds to the insertion of the receiver's
 tuple representation into a relational database table.

ARGUMENTS:

   conn           <GsRdbConnection subclass> uncaptured
                  A connection instance used to determine the variant of 
                  SQL to generate.

   tableName      <String> copied
                  A string representing the name of a relational table. 
                  Used to identify the table from which the deletion 
                  will take place.

   columnMap      <Array> contents copied
                  A properly formatted column mapping. Used to identify 
                  the values and their corresponding columns for the 
                  tuple to be inserted.

RETURN VALUE:

   <String> new
   A string of the form: 
   'INSERT INTO table-name 
      (col-name1, col-name2, ...) VALUES ('val1', 'val2', ...)'.

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
   GsRdbError/noColumnMap
   GsRdbError/noTableName
"

   (tableName isNil) ifTrue: [
       conn class _signalNoTableName: conn tupleClass: (self class)
       ].
   (columnMap isNil) ifTrue: [
       conn class _signalNoColumnMap: conn tupleClass: (self class)
       ].

   ^conn class generateSQLInsertForTable: tableName
               columns: (self getNamesAndValuesFor: columnMap)
%
method: Object
getNamesAndValuesFor: columnMap symbol: sym

"Uses the given column map to obtain a column name and corresponding value, for
 the instance variable named by the given symbol, from the receiver, and returns
 an array containing this column name and value as elements. Intended to be used
 to generate the information necessary for constructing SQL statements.

 If the given symbol does not match any instance variable name specified in the
 given column map, an empty array is returned. If the given column map is empty
 or nil, returns an empty array.

ARGUMENTS:

   columnMap      <Array>
                  A properly formatted column mapping.

   sym            <String>
                  A symbol corresponding to an instance variable name 
                  to be matched with entries in a properly formatted 
                  column mapping.

RETURN VALUE:

   <Array> new    A two element array containing strings representing 
                  a column name and value.

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
"

   | i mapSize colsNvals colEntry entSize accessor |

   GsRdbConnection _validateColumnMap: columnMap conn: nil.

   i := 0.
   colsNvals := Array new.
   mapSize := columnMap size.
   [(colsNvals isEmpty) and: [i < mapSize]] whileTrue: [
      i := i + 1.
      colEntry := columnMap at: i.
      entSize := colEntry size.

      accessor := colEntry at: 2.
      (sym asSymbol == accessor asSymbol) ifTrue: [
         (entSize > 2) ifTrue: [
            accessor := colEntry at: 3.
            ].
         colsNvals add: (colEntry at: 1); add: (self perform: accessor).
         ].
      ].

   ^colsNvals
%
method: Object
generateSQLUpdate: conn scope: aScope table: tableName columns: columnMap keys: keyMap

"Returns a string representing a properly formatted SQL update command.  The
 given table name, column map, and key map will be used to identify the table,
 column name and value, and tuple, respectively, for which the update command
 will be generated.

 If the given scope is nil, then all column names specified in the given column
 map will be used in the generated update command. Otherwise, only those columns
 corresponding to accessor names which match the symbols in the scope will be
 used.

 The generated SQL update command corresponds to the update of the receiver's
 tuple representation in a relational database table. Intended for use when
 queuing up changes to be sent to the relational database.

ARGUMENTS:

   conn           <GsRdbConnection subclass>
                  A connection instance used to determine the variant of 
                  SQL to generate.

   aScope         <aCollection> or nil
                  A collection of symbols corresponding to instance variable 
                  names or accessor names to be matched with entries in a 
                  properly formatted column mapping. Its class must implement 
                  the method #detect: . Used to identify the columns to be
                  updated when not nil.

   tableName      <String> copied
                  A string representing the name of a relational table. 
                  Used to identify the table on which the update will 
                  take place.

   columnMap      <Array> contents copied
                  A properly formatted column mapping. Used to identify 
                  the columns to be updated and the new values for these 
                  columns.

   keyMap         <Array> contents copied
                  A properly formatted key mapping. Used to identify the 
                  tuple to be updated.

RETURN VALUE:

   <String> new
   A string of the form: 
   'UPDATE table-name 
      SET col-name1 = 'val1', col-name2 = 'val2' ... 
      WHERE key-name1 = 'key1' AND key-name2 = 'key2' ...'.

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
   GsRdbError/noColumnMap
   GsRdbError/noPrimaryKeyMap
   GsRdbError/noTableName
"

   | colsNvals |

   (tableName isNil) ifTrue: [
      conn class _signalNoTableName: conn tupleClass: (self class)
      ].

   (columnMap isNil) ifTrue: [
      conn class _signalNoColumnMap: conn tupleClass: (self class)
      ].

   (keyMap isNil) ifTrue: [
      conn class _signalNoPrimaryKeyMap: conn tupleClass: (self class)
      ].

   (aScope isNil) ifTrue: [
      colsNvals := self getNamesAndValuesFor: columnMap.
      ].
	
   (colsNvals isNil) ifTrue: [
      colsNvals :=  self getNamesAndValuesFor: columnMap scope: aScope.
      ].

   ^conn class generateSQLUpdateForTable: tableName 
               columns: colsNvals
               keys: (self getNamesAndValuesFor: keyMap)
%
method: Object
generateSQLUpdate: conn instVarName: ivName newValue: anObject

"Returns a string representing a properly formatted SQL update command.  The
 receiver's class must re-implement the #rdbTableName, #rdbColumnMapping, and
 #rdbPrimaryKeyMaps messages, as they will be used to identify the table, column
 name, and tuple, respectively, for which the update command will be generated.
 If these messages return the default, nil, an error will occur.

 Only the column which corresponds to the given instance variable name or
 accessor name will be used in the generated update command. The given new value
 will be used as the column value for the update command.

 The generated SQL update command corresponds to the update of the receiver's
 tuple representation in a relational database table. Intended for use when
 writing through changes to the relational database.

ARGUMENTS:

   conn          <GsRdbConnection subclass>
                 A connection instance used to determine the variant of 
                 SQL to generate.

   ivName        <Symbol>
                 A symbol corresponding to an instance variable name or 
                 accessor name to be matched with entries in a properly 
                 formatted column mapping. Used to identify the column 
                 to be updated.

   anObject      <Object>
                 An object corresponding to the value of the column to 
                 be updated in the update command.

RETURN VALUE:

   <String> new
   A string of the form: 
   'UPDATE table-name SET col-name = 'val' 
      WHERE key-name1 = 'key1' AND key-name2 = 'key2' ...'.

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
   GsRdbError/noColumnMap
   GsRdbError/noPrimaryKeyMap
   GsRdbError/noTableName
"

   ^self generateSQLUpdate: conn instVarName: ivName newValue: anObject 
         table: (self class rdbTableName) 
         columns: (self class rdbColumnMapping)
         keys: (self class rdbPrimaryKeyMaps)
%   
method: Object
generateSQLUpdate: conn instVarName: ivName newValue: anObject table: tableName columns: columnMap keys: keyMap

"Returns a string representing a properly formatted SQL update command.  The
 given table name, column map, and key map will be used to identify the table,
 column name, and tuple, respectively, for which the update command will be
 generated.

 Only the column which corresponds to the given instance variable name or
 accessor name will be used in the generated update command. The given new value
 will be used as the column value for the update command.

 The generated SQL update command corresponds to the update of the receiver's
 tuple representation in a relational database table. Intended for use when
 writing through changes to the relational database.

ARGUMENTS:

   conn           <GsRdbConnection subclass>
                  A connection instance used to determine the variant of 
                  SQL to generate.

   ivName         <Symbol>
                  A symbol corresponding to an instance variable name or 
                  accessor name to be matched with entries in a properly 
                  formatted column mapping. Used to identify the column 
                  to be updated.

   anObject       <Object>
                  An object corresponding to the value of the column to 
                  be updated in the update command.

   tableName      <String> copied
                  A string representing the name of a relational table. 
                  Used to identify the table on which the update will 
                  take place.

   columnMap      <Array> contents copied
                  A properly formatted column mapping. Used to identify 
                  the column to be updated.

   keyMap         <Array> contents copied
                  A properly formatted key mapping. Used to identify the 
                  tuple to be updated.

RETURN VALUE:

   <String> new
   A string of the form: 
   'UPDATE table-name SET col-name = 'val' 
      WHERE key-name1 = 'key1' AND key-name2 = 'key2' ...'.

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
   GsRdbError/noColumnMap
   GsRdbError/noPrimaryKeyMap
   GsRdbError/noTableName
"

   | colNval |

   (tableName isNil) ifTrue: [
      conn class _signalNoTableName: conn tupleClass: (self class)
      ].

   (columnMap isNil) ifTrue: [
      conn class _signalNoColumnMap: conn tupleClass: (self class)
      ].

   (keyMap isNil) ifTrue: [
      conn class _signalNoPrimaryKeyMap: conn tupleClass: (self class)
      ].

   colNval := self getNamesAndValuesFor: columnMap scope: { ivName }.
   colNval at: 2 put: anObject.

   ^conn class generateSQLUpdateForTable: tableName 
               columns: colNval
               keys: (self getNamesAndValuesFor: keyMap)
%
method: Object
generateSQLUpdate: conn scope: aScope

"Returns a string representing a properly formatted SQL update command.  The
 receiver's class must re-implement the #rdbTableName, #rdbColumnMapping, and
 #rdbPrimaryKeyMaps messages, as they will be used to identify the table, column
 name and value, and tuple, respectively, for which the update command will be
 generated. If these messages return the default, nil, an error will occur.

 If the given scope is nil, then all column names specified in the class' column
 map will be used in the generated update command. Otherwise, only those columns
 corresponding to accessor names which match the symbols in the scope will be
 used.

 The generated SQL update command corresponds to the update of the receiver's
 tuple representation in a relational database table. Intended for use when
 queuing up changes to be sent to the relational database.

ARGUMENTS:

   conn        <GsRdbConnection subclass>
               A connection instance used to determine the variant of 
               SQL to generate.

   aScope      <Collection>
               A collection of symbols corresponding to instance variable 
               names or accessor names to be matched with entries in a 
               properly formatted column mapping. Its class must implement 
               the method #detect: . Used to identify the columns to be
               updated when not nil.

RETURN VALUE:

   <String> new
   A string of the form: 
   'UPDATE table-name 
      SET col-name1 = 'val1', col-name2 = 'val2' ... 
      WHERE key-name1 = 'key1' AND key-name2 = 'key2' ...'.

ERRORS:

   GsRdbError/badColumnMap
   GsRdbError/invalidColumnMap
   GsRdbError/noColumnMap
   GsRdbError/noPrimaryKeyMap
   GsRdbError/noTableName
"

   ^self generateSQLUpdate: conn scope: aScope
         table: (self class rdbTableName) 
         columns: (self class rdbColumnMapping)
         keys: (self class rdbPrimaryKeyMaps)
%
!
! Instance Category 'Change Notification'
!
category: 'Change Notification'
method: Object
notifyChange

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"Indicates whether or not a notification message will be received when this
 object is modified.

ARGUMENTS:

   none

RETURN VALUE:

   <Boolean>
   True if this object will be notified upon change and false if not.

ERRORS:

   none
"

   ^self _hasModificationTrackingTo: (GsRdbChangeNotifObj soleInstance).
%
method: Object
notifyChange: bool

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"Sets whether this object will receive a notification message when it is
 modified.

ARGUMENTS:

   bool   <Boolean> captured
   True if this object is to be notified upon change and false if not.

RETURN VALUE:

   unspecified

ERRORS:

   none
"

   | notifObject |

   bool _validateClass: Boolean.
   ((self isInvariant) or: [self class isBytes]) ifFalse: [
      notifObject := GsRdbChangeNotifObj soleInstance.
      bool ifTrue: [
         (self notifyChange) ifFalse: [
            self _setModificationTrackingTo: notifObject.
            ].
         ]
      ifFalse: [
         (self notifyChange) ifTrue: [
            self _clearModificationTrackingTo: notifObject.
            ].
         ].
      ].
%
method: Object
aboutToChange: offset newValue: anObject

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"This message should be re-implemented by developers to provide custom behavior
 for their application classes.  The default behavior is to simply return self.
 It is not intended to be called by methods implemented by application
 developers.

 This message is sent automatically by GemStone when an object (or sequenceable
 collection), which is set to be notified upon changes, is about to have the
 value of its public named (or possibly unnamed for sequenceable collections) 
 instance variable, at the given offset, changed to the new value anObject.
 The given offset incudes both named and unnamed instance variables, thus to
 obtain the index of an unnamed instance variable which is about to have its
 value changed, the instance size of the receiver (i.e., 'self class instSize')
 should be substracted from this offset.

 Re-implementations of this method should take into account the possibility
 that the newValue could be replacing an existing value located at the given
 offset in the receiver. This message is sent before the actual change occurs,
 so the existence of a current value at the given offset can be verified.
 
 Typical re-implementations will include actions such as obtaining locks in
 other databases, writing changes through to other databases, or recording more
 detailed information about changes than GemStone keeps.  For relational
 database applications this would involve generating SQL statements, and either
 executing them on the relational database or queuing them for later execution.

 Extreme care should be taken not to modify the receiver in re-implementations
 of this method.  Modification of the receiver would cause recursion into this
 same method.  In addition, care should be taken to have re-implementations
 return self.

ARGUMENTS:

   offset        <Integer> unspecified.  
                 The integer offset, starting at the first 'named' instance 
                 variable, of the instance variable about to be changed in 
                 the receiver. Could be an 'unname' instance variable if 
                 the receiver is a sequenceable collection.

   anObject      <Object> unspecified.  
                 The new value of the instance variable about to be changed.

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"
%
!
! Class Category 'Rdb Methods'
!
category: 'Rdb Methods'
classmethod: Object
rdbColumnMapping

"This message should be re-implemented by developers to provide custom behavior
 for their application classes.  The default behavior is to return nil.

 This message returns the default column mapping for all tuple objects.  This
 column mapping message is sent by GemStone when a GsRdbReadStream is created
 and when generating SQL commands. If this message has not been re-implemented
 to provide a column mapping for the tuple class, no column mapping will be used
 by the read stream when generating tuples from the relational data.

 Re-implementations should take care to return a properly formatted column
 mapping.

ARGUMENTS:

   none

RETURN VALUE:

   <UndefinedObject>

ERRORS:

   none
"

   ^nil
%
classmethod: Object
rdbPrimaryKeyMaps

"This message should be re-implemented by developers to provide custom behavior
 for their application classes.  The default behavior is to return nil.

 This message returns the default primary key maps for all tuple objects.  This
 primary key maps message is sent by GemStone when generating SQL commands.

 Re-implementations should take care to return a properly formatted column
 mapping.

ARGUMENTS:

   none

RETURN VALUE:

   <UndefinedObject>

ERRORS:

   none
"

   ^nil
%
classmethod: Object
rdbTableName

"This message should be re-implemented by developers to provide custom behavior
 for their application classes.  The default behavior is to return nil.

 This message returns the default table name for all tuple objects.  This table
 name message is sent by GemStone when generating SQL commands.

 Re-implementations should take care to return a string corresponding to a real
 relational database table.

ARGUMENTS:

   none

RETURN VALUE:

   <UndefinedObject>

ERRORS:

   none
"

   ^nil
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - UnorderedCollection.gs
!
! ========================================================================

!
! Instance Category 'Change Notification'
!
category: 'Change Notification'
method: UnorderedCollection
aboutToAdd: anObject

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"This message should be re-implemented by developers to provide custom behavior
 for their application classes.  The default behavior is to simply return self.
 It is not intended to be called by methods implemented by application
 developers.

 This message is sent automatically by GemStone when a non-sequenceable
 collection, which is set to be notified upon changes, is about to have
 anObject added to it.

 Typical re-implementations will include actions such as obtaining locks in
 other databases, writing changes through to other databases, or recording more
 detailed information about changes than GemStone keeps.  For relational
 database applications this would involve generating SQL statements, and either
 executing them on the relational database or queuing them for later execution.

 Extreme care should be taken not to modify the receiver in re-implementations
 of this method.  Modification of the receiver would cause recursion into this
 same method.  In addition, care should be taken to have re-implementations
 return self.

ARGUMENTS:

   anObject      <Object> unspecified. 
                 The object about to be added to the receiver.

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"
%
method: UnorderedCollection
aboutToRemove: anObject

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"This message should be re-implemented by developers to provide custom behavior
 for their application classes.  The default behavior is to simply return self.
 It is not intended to be called by methods implemented by application
 developers.

 This message is sent automatically by GemStone when a non-sequenceable
 collection, which is set to be notified upon changes, is about to have
 anObject removed from it.

 Typical re-implementations will include actions such as obtaining locks in
 other databases, writing changes through to other databases, or recording more 
 detailed information about changes than GemStone keeps.  For relational
 database applications this would involve generating SQL statements, and either
 executing them on the relational database or queuing them for later execution.

 Extreme care should be taken not to modify the receiver in re-implementations
 of this method.  Modification of the receiver would cause recursion into this
 same method.  In addition, care should be taken to have re-implementations 
 return self.

ARGUMENTS:

   anObject      <Object> unspecified. 
                 The object about to be removed from the receiver.

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - SequenceableCollection.gs
!
! ========================================================================

!
! Instance Category 'Change Notification'
!
category: 'Change Notification'
method: SequenceableCollection
aboutToInsert: aSeqColl index: index

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"This message should be re-implemented by developers to provide custom behavior
 for their application classes.  The default behavior is to simply return self.
 It is not intended to be called by methods implemented by application
 developers.

 This message is sent automatically by GemStone when a sequenceable collection,
 which is set to be notified upon changes, is about to have the elements, which
 are in aSeqColl, inserted into it at the given index. The receiver's size
 will not have changed at the time this message is received. Also, as this
 message notifies of an insertion into the receiver, none of the elements which
 are currently in the collection will be deleted because of the notified change,
 although some of the elements may shift positions.

 Typical re-implementations will include actions such as obtaining locks in
 other databases, writing changes through to other databases, or recording more
 detailed information about changes than GemStone keeps.  For relational
 database applications this would involve generating SQL statements, and either
 executing them on the relational database or queuing them for later execution.

 Extreme care should be taken not to modify the receiver in re-implementations
 of this method.  Modification of the receiver would cause recursion into this
 same method.  In addition, care should be taken to have re-implementations
 return self.

ARGUMENTS:

   aSeqColl      <SequenceableCollection> unspecified
                 The collection of elements which are about to be inserted 
                 into the receiver.

   index         <Integer> unspecified
                 The index of the receiver at which the collection of 
                 elements will be inserted.

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"
%
method: SequenceableCollection
aboutToDelete: aSeqColl index: index

"WARNING: Object Change Notification is not supported in GS/S 64 1.X."

"This message should be re-implemented by developers to provide custom behavior
 for their application classes.  The default behavior is to simply return self.
 It is not intended to be called by methods implemented by application
 developers.
 
 This message is sent automatically by GemStone when a sequenceable collection,
 which is set to be notified upon changes, is about to have the elements, which
 are in aSeqColl, deleted from it at the given index. The receiver's size will
 not have changed at the time this message is received, so aSeqColl should be
 the subset of elements which are located at the given index in the receiver.

 Typical re-implementations will include actions such as obtaining locks in
 other databases, writing changes through to other databases, or recording more
 detailed information about changes than GemStone keeps.  For relational
 database applications this would involve generating SQL statements, and either
 executing them on the relational database or queuing them for later execution.

 Extreme care should be taken not to modify the receiver in re-implementations
 of this method.  Modification of the receiver would cause recursion into this
 same method.  In addition, care should be taken to have re-implementations
 return self.

ARGUMENTS:

   aSeqColl      <SequenceableCollection> unspecified
                 The collection of elements which are about to be deleted 
                 from the receiver.

   index         <Integer> unspecified
                 The index of the receiver from which the collection of 
                 elements will be deleted.

RETURN VALUE:

   the receiver (self)

ERRORS:

   none
"
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - Date.gs
!
! ========================================================================
!
! Instance Category 'Rdb Data Conversion'
!
category: 'Rdb Data Conversion'
method: Date
_asDateTime

"Private method used by GemConnect object-to-RDB data conversion logic.

 Returns a DateTime object representing this Date as of midnight."

   ^ DateTime newWithDate: self seconds: 0
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - DateTime.gs
!
! ========================================================================
!
! Instance Category 'Rdb Data Conversion'
!
category: 'Rdb Data Conversion'
method: DateTime
_adjustForLocalOffset

"Private method used by GemConnect object-to-RDB data conversion logic.

 Return a new DateTime that is offset by the number of seconds 
 difference with GMT."

^ self addSeconds: ( self _localOffset negated )
%

! Fix 48654
method: DateTime
_asDateAndTime

"Private method used by GemConnect to handle DateTime/DateAndTime conversion"

   | days secs x |

   x := year - 1901.
   days := x * 365
            + (x // 4)
            - (x // 100)
            + (x + 300 "1901 - 1601" // 400)
            + dayOfYear
            - 36526.                "set base to 2001-01-01"
   secs := days * 86400 + (milliseconds / 1000s3).
   ^ DateAndTime secondsUTC: secs offsetSeconds: nil
%

! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - DateAndTime.gs
!
! ========================================================================
!
! Instance Category 'Rdb Data Conversion'
!
category: 'Rdb Data Conversion'
method: DateAndTime
_asDateTime

   "Private method for GemConnect to handle DateAndTime/DateTime conversion"

   "Adapted from DateAndTimeANSI #asFloatParts / #partsFrom: "
   | x y year msec |
   Float noInexactResultDo:[
      x := self asSeconds.
      msec := (x \\ 86400 * 1000) rounded.
      x := x // 86400 .       "days from base, an Integer"
   ].
   year := 2001.
   y := x // 146097.
   year := y * 400 + year.
   x := x - (y * 146097).  "days since beginning of 400-year cycle"

   y := x // 36524 min: 3.
   year := y * 100 + year.
   x := x - (y * 36524).   "days since beginning of 100-year cycle"

   y := x // 1461 min: 96.
   year := y * 4 + year.
   x := x - (y * 1461).    "days since beginning of 4-year cycle"

   y := x // 365 min: 3.
   year := y + year.
   x := x - (y * 365) + 1. "days since beginning of year"
   ^ DateTime newGmtWithYear: year dayOfYear: x milliseconds: msec
%

! Fix 48654
category: 'Rdb Data Conversion'
method: DateAndTime
asPartsGC

"GemConnect version of asParts - returns microseconds"
| parts result |
parts := self asParts.
result := Array new: 7.
result at: 1 put: (parts at: 1).
result at: 2 put: (parts at: 3).
result at: 3 put: (parts at: 4).
result at: 4 put: (parts at: 5).
result at: 5 put: (parts at: 6).
result at: 6 put: (parts at: 7) floor.
result at: 7 put: ((parts at: 7) fractionPart * 1000000) floor.
^ result
%

! Fix 48654
category: 'Rdb Data Conversion'
classmethod: DateAndTime
newWithYear: y month: m day: d hour: h minute: min second: sec ms: ms

"Standardized GemConnect Date*Time creation method.
 Note: ms is in microseconds
"
| s |
s := ScaledDecimal _mantissa: (sec * 1000000 + ms) scale: 6.
^ self year: y month: m day: d hour: h minute: min second: s.
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - commondocumentation.gs
!
! ========================================================================
!
! Add installDocumentation method for each new common class
!

category: 'For Documentation Installation only'
classmethod: GsRdbChangeNotifObj
installDocumentation

self comment:
'GsRdbChangeNotifObj is a private class,
 for GemStone internal use only'.
%
category: 'For Documentation Installation only'
classmethod: GsRdbConnManager
installDocumentation

self comment:
'GsRdbConnManager is a private class,
 for GemStone internal use only'.
%
category: 'For Documentation Installation only'
classmethod: GsRdbConnection
installDocumentation

self comment:
'GsRdbConnection is an abstract superclass. Instances of its subclasses
 represent connections to a relational database. See GsOracleConnection.
 Currently, there are no separate protocols for GsOracleConnection.
 All behavior is defined here in the superclass'.
%
category: 'For Documentation Installation only'
classmethod: GsRdbParameters
installDocumentation

self comment:
'GsRdbParameters is an abstract superclass of GsOracleParameters.
 which define parameter objects specific
 to a database type. GsRdbParameters has instance variables and
 methods for storing connection information, but system-specific
 parameter information is defined in the subclasses'.
%
category: 'For Documentation Installation only'
classmethod: GsRdbReadStream
installDocumentation

self comment:
'GsRdbReadStream instances are used to read result sets from
 a relational database query. These are non-positionable read-only
 streams which hold a tuple class with which to instantiate
 result tuples. By default, this class is an OrderedCollection.'.
%
category: 'For Documentation Installation only'
classmethod: GsRdbWriteStream
installDocumentation

self comment:
'GsRdbWriteStream instances are used to write result sets to
 a relational database via bound insert/delete/update sql calls.
 These are non-positionable write-only streams which accept instances
 of a specified tuple class from which data is extraced and sent
 to the relational database for the insert/delete/update call.'.
%
category: 'For Documentation Installation only'
classmethod: GsRdbSelectStream
installDocumentation

self comment:
'GsRdbSelectStreams are used to return results from bound sql select 
statements.  Use GsOracleConnection>>openSelectCursorOn:* to generate
them.  GsRdbSelectStreams behave as *both* write streams and read streams.
You first write to the stream using #bindVars:/#nextPut:, supplying an 
array of values to be assigned to the bindings in the order specified in 
the bindInfo: argument of the #openSelectCursorOn:* call.  You can then 
read from the stream using #next to retrieve back the select statement 
results just as you would for a normal select statement executed in a 
#openCursorOn:* call.  Additional #bindVars:/#nextPut: calls can follow to
assign new bindings.'.
%
category: 'For Documentation Installation only'
classmethod: GsRdbBlockStream
installDocumentation

self comment:
'GsRdbBlockStream instances are used to execute and return results from
bound sql block statements that start with BEGIN or DECLARE. Use
GsOracleConnection>>openBlockCursorOn: to generate them.  GsRdbBlockStreams
behave as write streams.  Use #bindVars:/#nextPut: to supply an array of 
values to be assigned to the bindings in the order specified in the bindInfo:
argument of the #openBlockCursorOn:* call (but note that only #IN or #INOUT 
variables will actually use the supplied value).  The sql block statement is 
then executed, and results returned as an array (#IN variables will be nil).
Additional #bindVars:/#nextPut: calls can follow to assign new bindings.'.
%
category: 'For Documentation Installation only'
classmethod: GsRdbError
installDocumentation

self comment:
'GsRdbError defines all errors generated within GemConnect.'.
%

!
! Install documentation for each new common class
!
expectvalue %Array
run
   {  GsRdbConnection .
      GsRdbParameters .
      GsRdbConnManager .
      GsRdbChangeNotifObj .
      GsRdbReadStream .
      GsRdbWriteStream .
      GsRdbBlockStream .
      GsRdbSelectStream .
      GsRdbError } do: [:each |
     each installDocumentation.
     each class removeCategory: 'For Documentation Installation only'
   ]
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - ordocumentation.gs
!
! ========================================================================
!
! Add installDocumentation method for each new Oracle class
!

category: 'For Documentation Installation only'
classmethod: GsOracleConnection
installDocumentation

self comment:
'GsOracleConnection is a subclass of GsRdbConnection,
 used to connect to Oracle databases. See GsRdbConnection
 for the relevant protocols.'.
%
category: 'For Documentation Installation only'
classmethod: GsOracleParameters
installDocumentation

self comment:
'GsOracleParameters is a subclass of GsRdbParameters that
 adds protocols for configuring a connection to an Oracle
 relational database.'.
%
!
! Install documentation for each new Oracle class
!
expectvalue %Array
run
   { GsOracleConnection . GsOracleParameters } do: [:each |
     each installDocumentation.
     each class removeCategory: 'For Documentation Installation only'
   ]
%
! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - commonlast.gs
!
! ========================================================================
!
! initializations for common (base) classes
!

! Create sole instance of change notification tracking object
!
expectvalue %GsRdbChangeNotifObj
run
   GsRdbChangeNotifObj new.
%

! ========================================================================
! Copyright (C) GemTalk Systems. 1986-2025.  All Rights Reserved
!
! Name - orlast.gs
!
! ========================================================================
!
expectvalue 0
send UserGlobals at: #RdbErrorCount put: ErrorCount
expectvalue %String
expectvalue %Unicode7
doit
    RdbAPIHistory add: 'Load of GemConnect for Oracle version ',
        GsOracleConnection version printString, ' completed at ',
	DateTime now printString, ' with ', RdbErrorCount printString, ' errors'
%
obj ErrorCount
commit
output pop

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!  If this file has been loaded with the topaz utility, the
!  following will return zero if there were no errors
!
obj ErrorCount
commit
