! Copyright (C) GemTalk Systems 1986-2025.  All Rights Reserved.
! Class extensions for 'GsComArrayBuilderNode'

!		Class methods for 'GsComArrayBuilderNode'

removeallmethods GsComArrayBuilderNode
removeallclassmethods GsComArrayBuilderNode

category: 'Instance creation'
classmethod: GsComArrayBuilderNode
new
^ self _basicNew initialize
%

category: 'Instance creation'
classmethod: GsComArrayBuilderNode
with: aNode
 | res |
 (res := self new) appendElement: aNode .
  ^ res
%

category: 'Instance creation'
classmethod: GsComArrayBuilderNode
_basicNew
"create an instance registered with VM for finalization of cData"

<primitive: 674>
self _primitiveFailed: #_basicNew
%

!		Instance methods for 'GsComArrayBuilderNode'

category: 'Instance Initialization'
method: GsComArrayBuilderNode
appendElement: aGsCompilerIRNode
  aGsCompilerIRNode ifNil:[ self error:'invalid nil arg to GsComArrayBuilderNode'].
  elements addLast: aGsCompilerIRNode
%

category: 'Instance Initialization'
method: GsComArrayBuilderNode
initialize
  kind := COMPAR_ARRAY_BUILDER_NODE .
  elements := { } .
%

category: 'Printing'
method: GsComArrayBuilderNode
printFormattedOn: aStream

  super printOn: aStream .
  aStream print:' count:' int: elements size ;
      do: elements ;
  nextPut: $) ; cr .
%

category: 'Accessing'
method: GsComArrayBuilderNode
size
  ^ elements size
%

! Class extensions for 'GsComAssignmentNode'

!		Instance methods for 'GsComAssignmentNode'

removeallmethods GsComAssignmentNode
removeallclassmethods GsComAssignmentNode

category: 'Instance Initialization'
method: GsComAssignmentNode
dest: aGsComVarLeaf source: sourceNode
    "returns receiver."
  kind := COMPAR_ASSIGNMENT_NODE .
  dest := aGsComVarLeaf .
  source := sourceNode .
  assignKind := 0 .
  sourceNode ifNil:[ self error:'illegal nil source for assignment'].
  aGsComVarLeaf ifNil:[ self error: 'illegal nil destination for assignment'].
%

category: 'Instance Initialization'
method: GsComAssignmentNode
destSrcOffset: anOffset
  destSrcOffset := anOffset
%

category: 'Printing'
method: GsComAssignmentNode
printFormattedOn: aStream

  super printOn: aStream .
  aStream print:' assignKind:' int: assignKind ; cr ;
  nextPut: $( ; indentMore .
     dest printFormattedOn: aStream .
  aStream  nextPutAll:') := (' .
     source printFormattedOn: aStream .
  aStream indentLess ; nextPutAll: '))'  ; cr .
%

category: 'Instance Initialization'
method: GsComAssignmentNode
setMethodArgDefault
  assignKind := 1
%

! Class extensions for 'GsComBlockNode'

!		Class methods for 'GsComBlockNode'

removeallmethods GsComBlockNode
removeallclassmethods GsComBlockNode

category: 'Instance creation'
classmethod: GsComBlockNode
new
"create an instance registered with VM for finalization of cData"

<primitive: 674>
self _primitiveFailed: #new
%

!		Instance methods for 'GsComBlockNode'

category: 'Instance Initialization'
method: GsComBlockNode
appendArg: aGsComVarLeaf
  | argKnd needSynth |
  needSynth := false .
  (argKnd := aGsComVarLeaf varKind) == COMPAR_BLOCK_ARG_VAR ifTrue:[
    (aGsComVarLeaf lexLevel < lexLevel and:[ COM_RUBY_COMPAT_LEVEL == 18]) ifTrue:[
      "Fix Ticket 113, block arg escaping to outer block arg,
       Synthesize a blockArg node, and add an assignment to the
       outer arg"
      needSynth := true
    ] ifFalse:[
      args addLast: aGsComVarLeaf.      "normal block arg"
      ^ self .
    ].
  ].
  ((argKnd == COMPAR_METHOD_TEMP_VAR
      or:[ argKnd == COMPAR_BLOCK_TEMP_VAR
        or:[ argKnd == COMPAR_METHOD_ARG_VAR] ])
    and:[ aGsComVarLeaf lexLevel < lexLevel
        and:[ COM_RUBY_COMPAT_LEVEL == 18 ]]) ifTrue:[
    "Fix for Ticket 21, ruby block args escaping to outer level.
     Synthesize a blockArg node, and add an assignment to the
     outer temp"
    needSynth := true
  ].
  needSynth ifTrue:[
    | synthArg assnNod ags |
    synthArg := GsComVarLeaf new blockArg: aGsComVarLeaf varName
                argNumber: (ags := args) size + 1
                forBlock: self .
    ags add: synthArg .
    assnNod := GsComAssignmentNode new dest: aGsComVarLeaf
                source: (GsComVariableNode new leaf: synthArg ).
    statements insertObject: assnNod at: 1 .
    ^ self
  ].
  self error: 'VarLeaf.varKind=', (GsComVarLeaf varKindToString: argKnd) ,
        ' illegal for a block arg'
%

category: 'Instance Initialization'
method: GsComBlockNode
appendMasgnDummyArg
  "for a Ruby block of the form  { | x, |  } append a dummy arg
   so the block-args processing will then do the right thing at runtime"
  | ags dummy |
  (ags := args) size == 1 ifFalse:[ self error:'before appendMasgnDummyArg, size not 1'].
  dummy := GsComVarLeaf new blockArg: #_dummyarg argNumber: 2 forBlock: self .
  ags add: dummy
%

category: 'Instance Initialization'
method: GsComBlockNode
appendStatement: aNode
  aNode sourceOffset ifNil:[ | bsiz |
    (bsiz := statements size) ~~ 0 ifTrue:[
      aNode sourceOffset:( (statements at: bsiz) sourceOffset).
    ]
  ].
  statements addLast: aNode . ^
  srcOffset ifNil:[ | ofs |
    ofs := aNode sourceOffset .
    ofs ifNotNil:[ srcOffset := ofs ]
        ifNil:[ lineNumber ifNil:[ lineNumber := aNode lineNumber]].
  ].
%

category: 'Instance Initialization'
method: GsComBlockNode
appendTemp: aGsComVarLeaf
  aGsComVarLeaf varKind == COMPAR_BLOCK_TEMP_VAR ifFalse:[
    self error: 'bad arg kind ', aGsComVarLeaf varKindString
  ].
  temps addLast: aGsComVarLeaf
%

category: 'Accessing'
method: GsComBlockNode
args
  ^ args
%

category: 'Instance Initialization'
method: GsComBlockNode
lastLineNumber: anInt
  | first last |
  last := anInt .
  srcOffset ifNotNil:[ self error:'should use lastSourceOffset:' ].
  (first := lineNumber) ifNil:[
    lineNumber := anInt
  ] ifNotNil:[
    anInt < first ifTrue:[
      statements do:[ :aNode | | num |
        (num := aNode lineNumber) ifNotNil:[
           num < first ifTrue:[ first := num ].
           num > last ifTrue:[ last := num ].
        ]
      ].
      lineNumber := first.
    ]
  ].
  lastSrcOffset := last  "store a line number"
%

category: 'Instance Initialization'
method: GsComBlockNode
lastSourceOffset: anOffset
  lastSrcOffset := anOffset
%

category: 'Accessing'
method: GsComBlockNode
lexLevel
  ^ lexLevel
%

category: 'Instance Initialization'
method: GsComBlockNode
lexLevel: aLevel
"the primary initialization method"
  kind := COMPAR_BLOCK_NODE.
  aLevel < 0 ifTrue:[ aLevel error:'out of range' ].
  lexLevel := aLevel .
  blkKind := 0 .
  args := { } .
  temps := { } .
  statements := { } .
  lastArgInfo := 0 .
  "terms left as nil"
%

category: 'Accessing'
method: GsComBlockNode
numArgs
  ^ args size
%

category: 'Accessing'
method: GsComBlockNode
numStatements
  ^ statements size
%

category: 'Printing'
method: GsComBlockNode
printFormattedOn: aStream
  super printOn: aStream .
  aStream print:' lexLevel:' int: lexLevel ;
    print:' lastArgInfo:' int: lastArgInfo ; cr ;
    nextPutAll:' args:'; do: args ;
    nextPutAll:' temps:'; do: temps ;
    nextPutAll:' statements:'; do: statements ;
  nextPut: $) ; cr .
%

category: 'Instance Initialization'
method: GsComBlockNode
setAmpersandArg
  lastArgInfo := lastArgInfo bitOr: HasBlockArg_mask
%

category: 'Instance Initialization'
method: GsComBlockNode
setLastArgStar
  "filter out anonymous star here,   {|*| }  equivalent to { }  "
  args size ~~ 0 ifTrue:[ lastArgInfo := lastArgInfo bitOr: LastArgStar_mask ].
%

category: 'Instance Initialization'
method: GsComBlockNode
setNoArgsDecl
  lastArgInfo := lastArgInfo bitOr: NoDeclaredArgs_mask  .
%

category: 'Instance Initialization'
method: GsComBlockNode
_appendLastStatement: aNode
  "OBSOLETE"
  self appendStatement: aNode .
  srcOffset ifNotNil:[
    self lastSourceOffset: aNode lastSourceOffset
  ] ifNil: [ | num |
    num := aNode lastLineNumber .
    num ifNotNil:[ self lastLineNumber: num ].
  ]
%

category: 'Instance Initialization'
method: GsComBlockNode
_computeLastLineNumber
  "OBSOLETE"
  | list sz |
  list := statements .
  srcOffset ifNotNil:[
    (sz := list size) _downTo: 1 do:[:n | | num |
      num := (list at: n) lastSourceOffset .
      num ifNotNil:[
         self lastSourceOffset: num .
         ^ self
      ].
    ].
    self lastSourceOffset: srcOffset .
    ^ self
  ].
  (sz := list size) _downTo: 1 do:[:n | | num |
      num := (list at: n) lastLineNumber .
      num ifNotNil:[
         self lastLineNumber: num .
         ^ self
      ].
  ].
  sz > 1 ifTrue:[ self error:'could not find a last source position'].
%

category: 'Instance Initialization'
method: GsComBlockNode
_finishLastStatement
  "OBSOLETE"
  | stmts last |
  stmts := statements .
  last := stmts atOrNil: stmts size .
  last ifNotNil:[
    srcOffset ifNotNil:[
      self lastSourceOffset: last lastSourceOffset
    ] ifNil:[ | num |
      num := last lastLineNumber .
      num ifNotNil:[ self lastLineNumber: num ].
    ]
  ]
%

! Class extensions for 'GsComCascadeNode'

!		Instance methods for 'GsComCascadeNode'

removeallmethods GsComCascadeNode
removeallclassmethods GsComCascadeNode

category: 'Instance Initialization'
method: GsComCascadeNode
appendSend: aNode
  sends addLast: aNode
%

category: 'Printing'
method: GsComCascadeNode
printFormattedOn: aStream
  super printOn: aStream .
  aStream nextPutAll:' rcvr:'. rcvr printFormattedOn: aStream .
  aStream nextPutAll:' sends:'; do: sends ;
    nextPut: $) ; cr .
%

category: 'Instance Initialization'
method: GsComCascadeNode
rcvr: aNode
  rcvr := aNode .
  sends := { } .
  kind := COMPAR_CASCADE_NODE
%

! Class extensions for 'GsComGotoNode'

!		Class methods for 'GsComGotoNode'

removeallmethods GsComGotoNode
removeallclassmethods GsComGotoNode

category: 'Instance creation'
classmethod: GsComGotoNode
new
  ^ self _basicNew initialize
%

category: 'Instance creation'
classmethod: GsComGotoNode
_basicNew
"create an instance registered with VM for finalization of cData"

<primitive: 674>
self _primitiveFailed: #_basicNew
%

!		Instance methods for 'GsComGotoNode'

category: 'Instance Initialization'
method: GsComGotoNode
argNode: anIrNode
  argNode := anIrNode
%

category: 'Instance Initialization'
method: GsComGotoNode
initialize
  kind := COMPAR_GOTO_NODE .
%

category: 'Instance Initialization'
method: GsComGotoNode
localRubyBreak: aLabelNode
  targetKind :=  COM_GOTO_BREAK.
  target := aLabelNode .
  argForValue :=  true
%

category: 'Instance Initialization'
method: GsComGotoNode
localRubyNext: aLabelNode argForValue: aBoolean
  targetKind := COM_GOTO_NEXT .
  target := aLabelNode  .
  argForValue := aBoolean
%

category: 'Instance Initialization'
method: GsComGotoNode
localRubyRedo: aLabelNode
  targetKind := COM_GOTO_REDO .
  target := aLabelNode  .
  argForValue := false
%

category: 'Instance Initialization'
method: GsComGotoNode
nonLocalRubyNext
  "target left as nil"
  targetKind := COM_GOTO_NEXT .
  argForValue := true
%

category: 'Instance Initialization'
method: GsComGotoNode
nonLocalRubyRedo
  "target left as nil"
  targetKind := COM_GOTO_REDO .
  argForValue := false
%

category: 'Printing'
method: GsComGotoNode
printFormattedOn: aStream
  super printOn: aStream .
  aStream print:' target objId:' int: target asOop ; cr ;
    nextPutAll: ' kind:' ; nextPutAll:
         ( #( #next #redo #retry #break ) at: targetKind) ;
    print: ' argForValue:' bool: argForValue ; cr ;
    nextPutAll: '    argNode:' .
  argNode ifNil:[ aStream nextPutAll:'nil' ]
       ifNotNil:[ argNode printFormattedOn: aStream  ].
  aStream nextPut: $) ; cr .
%

! Class extensions for 'GsComLabelNode'

!		Class methods for 'GsComLabelNode'

removeallmethods GsComLabelNode
removeallclassmethods GsComLabelNode

category: 'Instance creation'
classmethod: GsComLabelNode
new
^ self _basicNew initialize
%

category: 'Instance creation'
classmethod: GsComLabelNode
_basicNew
"create an instance registered with VM for finalization of cData"

<primitive: 674>
self _primitiveFailed: #_basicNew
%

!		Instance methods for 'GsComLabelNode'

category: 'Instance Initialization'
method: GsComLabelNode
argForValue
  ^ argForValue
%

category: 'Instance Initialization'
method: GsComLabelNode
initialize
  argForValue := false .
  lexLevel := -1 .
  kind := COMPAR_LABEL_NODE
%

category: 'Instance Initialization'
method: GsComLabelNode
lexLevel
  ^ lexLevel
%

category: 'Instance Initialization'
method: GsComLabelNode
lexLevel: anInt
  lexLevel := anInt
%

category: 'Instance Initialization'
method: GsComLabelNode
lexLevel: anInt argForValue: aBoolean
  lexLevel := anInt .
  argForValue := aBoolean
%

category: 'Printing'
method: GsComLabelNode
printFormattedOn: aStream
  super printOn: aStream  .
  aStream nextPutAll: ' argForValue:', argForValue asString ;
     print:' lexLevel:' int: lexLevel ;
  nextPut: $) ; cr .
%

! Class extensions for 'GsComLiteralNode'

!		Class methods for 'GsComLiteralNode'

removeallmethods GsComLiteralNode
removeallclassmethods GsComLiteralNode

category: 'Instance Creation'
classmethod: GsComLiteralNode
newConstantRef: aRubyConstantRef
 ^ self new leaf: (GsComLitLeaf new  constRefLiteral: aRubyConstantRef)
%

category: 'Instance Creation'
classmethod: GsComLiteralNode
newFalse
  ^ self new leaf: (GsComLitLeaf new specialLiteral: false)
%

category: 'Instance Creation'
classmethod: GsComLiteralNode
newInteger: aSmallInt
 ^ self new leaf: (GsComLitLeaf new  integerLiteral: aSmallInt)
%

category: 'Instance Creation'
classmethod: GsComLiteralNode
newNil
  ^ self new leaf: (GsComLitLeaf new specialLiteral: nil)
%

category: 'Instance Creation'
classmethod: GsComLiteralNode
newObject: anObject
 ^ self new leaf: (GsComLitLeaf new  objectLiteral: anObject)
%

category: 'Instance Creation'
classmethod: GsComLiteralNode
newRemoteNil
  ^ self new leaf: (GsComLitLeaf new specialLiteral: _remoteNil)
%

category: 'Instance Creation'
classmethod: GsComLiteralNode
newString: aString
  | str |
  str := aString .
  str isInvariant ifFalse:[ str := aString copy ].
  ^ self new leaf: (GsComLitLeaf new stringLiteral: str ).
%

category: 'Instance Creation'
classmethod: GsComLiteralNode
newTrue
  ^ self new leaf: (GsComLitLeaf new specialLiteral: true)
%

!		Instance methods for 'GsComLiteralNode'

category: 'Printing'
method: GsComLiteralNode
leaf: aGsComLitLeaf
  leaf := aGsComLitLeaf .
  kind := COMPAR_LIT_NODE .
%

category: 'Printing'
method: GsComLiteralNode
printFormattedOn: aStream
  super printOn: aStream .
  leaf printFormattedOn: aStream .
  aStream nextPut: $) ; cr .
%

category: 'Accessing'
method: GsComLiteralNode
symbolLiteralValue
  | val |
  val := leaf symbolLiteralValue .
  val ifNil:[ self error:'invalid leaf for symbolLiteralValue'].
  ^ val
%

! Class extensions for 'GsComLitLeaf'

!		Class methods for 'GsComLitLeaf'

removeallmethods GsComLitLeaf
removeallclassmethods GsComLitLeaf

category: 'Instance creation'
classmethod: GsComLitLeaf
newNil
  ^ self new specialLiteral: nil
%

!		Instance methods for 'GsComLitLeaf'

category: 'Instance Initialization'
method: GsComLitLeaf
arrayLiteral: anArray

"anArray is a literal such as the Smalltalk literal  #( 1 2 abc )
 which can be constructed without executing any bytecodes.

 In Smalltalk , these arrays are  Invarant , and the  Parser
 canonicalizes them based on comparing substrings from the source code.
"
  self setIRnodeKind .
  "stringForm left as nil for now"
  litKind := COMPAR_ARRAY_LIT .
  litValue := anArray .
  anArray immediateInvariant
%

category: 'Instance Initialization'
method: GsComLitLeaf
characterLiteral: aCharacter
  self setIRnodeKind .
  litValue := aCharacter .
  litKind := COMPAR_CHAR_LIT .
%

category: 'Instance Initialization'
method: GsComLitLeaf
constRefLiteral: aRubyConstantRef

  self setIRnodeKind .
  litKind := COMPAR_ASSOC_LIT .  "fault into old_gen if possible"
  litValue := aRubyConstantRef .
%

category: 'Instance Initialization'
method: GsComLitLeaf
deferredGlobalLiteral: anObject

"used in ruby IR "

  self setIRnodeKind .
  "stringForm left as nil for now"
  litKind := COMPAR_ARRAY_LIT .
  litValue := anObject .   "do not make invariant."
%

category: 'Instance Initialization'
method: GsComLitLeaf
floatLiteral: aFloat
  litValue := aFloat .
  self setIRnodeKind .
  litKind := COMPAR_FLT_LIT .
%

category: 'Instance Initialization'
method: GsComLitLeaf
floatLiteralFromString: aString

  "See Float>>fromString: for details of NaN representations in the input."

  self floatLiteral: (Float fromString: aString)
%

category: 'Instance Initialization'
method: GsComLitLeaf
integerLiteral: anInteger
  litValue := anInteger .
  self setIRnodeKind .
  litKind := COMPAR_INT_LIT
%

category: 'Instance Initialization'
method: GsComLitLeaf
integerLiteralFromString: aString
  stringForm := aString .
  self integerLiteral: (Integer fromString: aString)
%

category: 'Instance Initialization'
method: GsComLitLeaf
methodLiteral: anASTnode

"anASTnode is the root of a graph of the AST for the method,
 need to transform that tree to tree beginning with a GsComMethNode
 at the time we want to compile the method.  So the AST tree is
 sometimes persistent.
"
  self setIRnodeKind .
  "sourceForm left as nil,  method has source in its debug info."
  litValue := anASTnode .
  litKind := COMPAR_METHOD_LIT .
%

category: 'Instance Initialization'
method: GsComLitLeaf
objectLiteral: anObj

"Used for generic non-Assocation non-invariant literals
 in Ruby methods."

  self setIRnodeKind .
  "stringForm left as nil for now"
  litKind := COMPAR_ARRAY_LIT .
  litValue := anObj .   "do not set invariant"
%

category: 'Printing'
method: GsComLitLeaf
printFormattedOn: aStream

  super printOn: aStream .
  aStream print:' litValue:' int: litValue ;
          print:' litKind:' int: litKind ;
      nextPut: $) ; cr .
%

category: 'Instance Initialization'
method: GsComLitLeaf
rubyCopyingStringLiteral: aString

  "This literal will be accessed with a PUSH_COPYOF_LIT bytecode
   see also comments in stringLiteral: . "

  aString _isOneByteString ifFalse:[
    self error:'expected a String'   "detect parser problems, relax when QB string"
  ].
  self setIRnodeKind .
  stringForm := aString .
  litKind := COMPAR_RUBY_COPYING_STR_LIT .
  litValue := aString .
  aString immediateInvariant .
%

category: 'Instance Initialization'
method: GsComLitLeaf
setIRnodeKind
  kind :=  COMPAR_LIT_LEAF
%

category: 'Instance Initialization'
method: GsComLitLeaf
specialLiteral: aValue
  "aValue is expected to be an instance of Boolean or UndefinedObject"
  self setIRnodeKind .
    litValue := aValue .
  litKind := COMPAR_SPECIAL_LIT .
%

category: 'Instance Initialization'
method: GsComLitLeaf
stringLiteral: aString

" aString is a  string literal such as the Smalltalk literal  'abc'
  that can be completely generated by the Parser.

 It is the responsibility of the parser to maintain a dictionary
 of String literals if it is desired to canonicalize Strings  within
 method compilations, or across method compilations .  All String
 literals will be made invariant by the code generator."

  self setIRnodeKind .
  stringForm := aString .
  litKind := COMPAR_STR_LIT .
  litValue := aString .

  aString immediateInvariant .
%

category: 'Instance Initialization'
method: GsComLitLeaf
symbolLiteral: aString

  | sym |
  self setIRnodeKind .
  stringForm := aString .
  sym := aString asSymbol .
  sym class == DoubleByteSymbol ifTrue:[
    self error:'DoubleByteSymbol not supported for Ruby.'
  ].
  litValue :=  sym .
  litKind := COMPAR_SYM_LIT .
%

category: 'Accessing'
method: GsComLitLeaf
symbolLiteralValue
  litKind == COMPAR_SYM_LIT ifFalse:[ self error:'not a symbol leaf' ].
  ^ litValue
%

! Class extensions for 'GsComLoopNode'

!		Class methods for 'GsComLoopNode'

removeallmethods GsComLoopNode
removeallclassmethods GsComLoopNode

category: 'Instance creation'
classmethod: GsComLoopNode
new
  ^ self _basicNew initialize
%

!		Instance methods for 'GsComLoopNode'

category: 'Instance Initialization'
method: GsComLoopNode
breakLabel: aLabelNode
  breakLabel := aLabelNode
%

category: 'Instance Initialization'
method: GsComLoopNode
initialize
  kind := COMPAR_LOOP_NODE .
%

category: 'Instance Initialization'
method: GsComLoopNode
iterResult: aLiteralNode
  iterResult := aLiteralNode
%

category: 'Printing'
method: GsComLoopNode
printFormattedOn: aStream
  super printOn: aStream .
  aStream nextPutAll:' send: '; indentMore ; cr .
     send printFormattedOn: aStream.
  aStream  indentLess ; cr ;
     nextPutAll:' label: ' .
  breakLabel printFormattedOn: aStream.
  iterResult ifNotNil:[
    aStream nextPutAll:' iterResult: ' .
    iterResult printFormattedOn: aStream.
  ].
  aStream nextPut: $) ; cr .
%

category: 'Instance Initialization'
method: GsComLoopNode
send: aSendNode
  send := aSendNode
%

! Class extensions for 'GsComMethNode'

!		Class methods for 'GsComMethNode'

removeallmethods GsComMethNode
removeallclassmethods GsComMethNode

category: 'Instance creation'
classmethod: GsComMethNode
checkRubyInfoMasks
 "check classVars defined by bom.c against constants hardcoded in
  this class."
 | v |
 (v := (IsPrivate_mask bitOr: IsProtected_mask)) == 16r600 ifFalse:[ 
    Error signal:'inconsistent constant, v = 16r', v asHexString .
  ].
 ^ true
%

category: 'Instance creation'
classmethod: GsComMethNode
new

"disallowed , use newSmalltalk"
self shouldNotImplement: #new
%

category: 'Instance creation'
classmethod: GsComMethNode
newSmalltalk
  ^ self _basicNew initialize setSmalltalk
%

!		Instance methods for 'GsComMethNode'

category: 'Instance creation'
method: GsComMethNode
addMethodProtection: anInt
  "used in building IR to transfer default private/protected from
   current class to this method node, if method not automatically
   private (like  initialize method)"
  | prot mask newBits iv |
  mask := 16r600 .
  iv := rubyInfo .
  prot := (iv bitAnd: mask) bitShift: -9 . "inline methodProtection"
  anInt > prot ifTrue:[
    (anInt >= 0 and:[anInt <= 2]) ifFalse:[
       ArgumentError signal:'invalid method protection argument'
    ].
    newBits := anInt bitShift: 9 .
    rubyInfo := (iv bitAnd: (mask bitInvert))  bitOr: newBits
  ].
%

category: 'Instance creation'
method: GsComMethNode
appendArg: aGsComVarLeaf
  aGsComVarLeaf varKind == COMPAR_METHOD_ARG_VAR ifFalse:[
    self error: 'bad arg kind ', aGsComVarLeaf varKindString
  ].
  arguments addLast: aGsComVarLeaf
%

category: 'Instance creation'
method: GsComMethNode
appendStatement: aNode
  aNode sourceOffset ifNil:[ | bsiz |
    (bsiz := body size) > 0 ifTrue:[
      aNode sourceOffset:( (body at: bsiz) sourceOffset).
    ]
  ].
  body addLast: aNode .
  srcOffset ifNil:[ | ofs |
    ofs := aNode sourceOffset .
    ofs ifNotNil:[ srcOffset := ofs ]
        ifNil:[ lineNumber ifNil:[ lineNumber := aNode lineNumber]].
  ].
%

category: 'Instance creation'
method: GsComMethNode
appendTemp: aGsComVarLeaf
  aGsComVarLeaf varKind == COMPAR_METHOD_TEMP_VAR  ifFalse:[
    self error: 'bad arg kind ', aGsComVarLeaf varKindString
  ].
  temps addLast: aGsComVarLeaf
%

category: 'Accessing'
method: GsComMethNode
arguments
  ^ arguments
%

category: 'Instance creation'
method: GsComMethNode
class: aBehavior
  theClass := aBehavior
%

category: 'Instance creation'
method: GsComMethNode
endSourceOffset: anOffset
  endSrcOffset := anOffset
%

category: 'Instance creation'
method: GsComMethNode
envId
  ^ rubyInfo bitAnd: Env_mask
%

category: 'Instance creation'
method: GsComMethNode
environment: anInteger
  (anInteger < 0 or:[ anInteger > 255]) ifTrue:[
     anInteger error:'out of range'
  ].
  rubyInfo := (rubyInfo bitAnd:( Env_mask bitInvert)) bitOr: anInteger
%

category: 'Accessing'
method: GsComMethNode
fileName
  ^ fileName
%

category: 'Instance creation'
method: GsComMethNode
fileName: aString
  fileName := aString
%

category: 'Instance creation'
method: GsComMethNode
fileName: nameString source: srcString
  fileName := nameString .
  source := srcString
%

category: 'Printing'
method: GsComMethNode
fileNameForPrint
  | sz res |
  fileName ifNil:[ ^ 'nil' ].
  (res := String new) add: $' .
  (sz := fileName size) > 25 ifTrue:[
    res addAll:( fileName copyFrom: sz - 25 to: sz )
  ] ifFalse:[
    res addAll: fileName .
  ] .
  res add: $' .
  ^ res
%

category: 'Instance creation'
method: GsComMethNode
forceAllArgsTmpsToVc
  "set requiresVc bits to 2 "
  methInfo := methInfo bitOr: 16r2
%

category: 'Instance creation'
method: GsComMethNode
initialize
  kind := COMPAR_METHOD_NODE .
  arguments :=  { } .
  temps :=  { } .
  body := { } .
  "nonBridgeSelector left as nil"
  rubyInfo := 0 .
  rubyOptArgsBits := 1 . "starting source offset"
%

category: 'Instance creation'
method: GsComMethNode
insertFirstStatement: aNode
  body insertAll: { aNode } at: 1 .
%

category: 'Instance creation'
method: GsComMethNode
methodProtection
  "return the ruby method protection bits,
   result  0==public,  1==protected, 2==private  "

  ^ (rubyInfo bitAnd: 16r600) bitShift: -9
%

category: 'Instance creation'
method: GsComMethNode
nonBridgeSelector: aSymbol
  nonBridgeSelector := aSymbol
%

category: 'Printing'
method: GsComMethNode
printFormattedOn: aStream
  super printOn: aStream .
  aStream nextPutAll: ' file:' ; nextPutAll: self fileNameForPrint ;
    print:' selector: ' symbol: selector ;
    nextPutAll:' theClass: ';
      nextPutAll: (theClass ~~ nil ifTrue:[ theClass name] ifFalse:['nil '])  ;
    nextPutAll:' methInfo:'; nextPutAll: '16r'; nextPutAll: methInfo asHexString ; cr ;
    nextPutAll:' rubyInfo:'; nextPutAll: '16r'; nextPutAll: rubyInfo asHexString ; cr ;
    nextPutAll:'args:' ; do: arguments ;
    nextPutAll:'temps:' ; do: temps ;
    nextPutAll:'body:' ; do: body ;
    nextPut: $) ; cr .
%

category: 'Accessing'
method: GsComMethNode
selector
  ^ selector
%

category: 'Instance creation'
method: GsComMethNode
selector: aSymbol
  "different implementation(s) in .mcz"
  (GsComSelectorLeaf reimplementationAllowed: aSymbol inEnv: self envId) ifFalse:[
      Error signal: 'reimplementation of special selector ' , aSymbol,
                ' not allowed , near line ' , lineNumber asString , ' of ' , fileName asString .
  ].
  selector := aSymbol
%

category: 'Instance creation'
method: GsComMethNode
setPrimitiveNumber: anInt
  "set IR info corresponding to    <primitive: anInt>  in the source"
  | mask newBits |
  mask := 16rFFFF0000 .
  (anInt < 0 or:[ anInt > (mask bitShift: -16)]) ifTrue:[
    ArgumentError signal:'invalid  primitive number'
  ].
  anInt > System _maxPrimitiveNumber ifTrue:[
    ArgumentError signal:'invalid  primitive number for this VM'
  ].
  newBits := anInt bitShift: 16 .
  methInfo := (methInfo bitAnd: (mask bitInvert)) bitOr: newBits .
%

category: 'Instance creation'
method: GsComMethNode
setSmalltalk
  methInfo := 16rFFFF0000 . "primNumber:=-1, protected:=0,reqVc:=0"
  rubyInfo := 0 .
%

category: 'Instance creation'
method: GsComMethNode
source: aString
  source := aString
%

category: 'Instance creation'
method: GsComMethNode
startingSourceOffset: aSmallInt
  aSmallInt _isSmallInteger ifFalse:[ aSmallInt error:'invalid arg'].
  rubyOptArgsBits := aSmallInt
%

category: 'Printing'
method: GsComMethNode
summary
  | str |
  str := String new .
  str addAll:  ' file:' ; addAll: self fileNameForPrint ;
    addAll: ' line ' ; addAll: lineNumber asString ;
    addAll: ' selector: ' ; addAll: selector  .
  ^ str
%

! Class extensions for 'GsComPathNode'

!		Class methods for 'GsComPathNode'

removeallmethods GsComPathNode
removeallclassmethods GsComPathNode

category: 'Instance creation'
classmethod: GsComPathNode
new

"disallowed, instances created only by comparse.c"

self shouldNotImplement: #new
%

!		Instance methods for 'GsComPathNode'

category: 'Printing'
method: GsComPathNode
printOn: aStream
  super printOn: aStream .
  aStream nextPut: $) ; cr .
%

! Class extensions for 'GsCompilerIRNode'

!		Instance methods for 'GsCompilerIRNode'

removeallmethods GsCompilerIRNode
removeallclassmethods GsCompilerIRNode

category: 'Accessing'
method: GsCompilerIRNode
hasPosition
  ^ srcOffset ~~ nil or:[ lineNumber ~~ nil ]
%

category: 'Accessing'
method: GsCompilerIRNode
kind
  ^ kind
%

category: 'Accessing'
method: GsCompilerIRNode
lastLineNumber
  ^ lineNumber
%

category: 'Accessing'
method: GsCompilerIRNode
lastSourceOffset
  ^ srcOffset
%

category: 'Accessing'
method: GsCompilerIRNode
lineNumber
  ^ lineNumber
%

category: 'Updating'
method: GsCompilerIRNode
lineNumber: aSmallInteger
  "to be used only after initializating the node .
   The argument is a positive one-based  line number"

  lineNumber := aSmallInteger
%

category: 'Accessing'
method: GsCompilerIRNode
litVarValueOrNil
  ^ nil
%

category: 'Printing'
method: GsCompilerIRNode
printFormattedOn: aStream
  self printOn: aStream
%

category: 'Printing'
method: GsCompilerIRNode
printOn: aStream

  aStream nextPut: $(; space;  nextPutAll: self class name ;
     nextPutAll:' objId: ' ; nextPutAll: self asOop asString .
  srcOffset ifNotNil:[
     aStream nextPutAll:' srcOfs:' ; nextPutAll: srcOffset asString ; space
   ] ifNil:[
     aStream nextPutAll:' line:' ; nextPutAll: lineNumber asString ; space
   ].
%

category: 'Printing'
method: GsCompilerIRNode
printString
  | strm |
  strm := IndentingStream newPrinting .
  self printFormattedOn: strm .
  ^ strm contents .
%

category: 'Transformation'
method: GsCompilerIRNode
returnNode

^ GsComReturnNode new return: self
%

category: 'Accessing'
method: GsCompilerIRNode
sourceOffset
  ^ srcOffset
%

category: 'Updating'
method: GsCompilerIRNode
sourceOffset: aSmallInteger
  "Argument is a 1-based character offset into the source string."
  srcOffset ifNil:[ srcOffset := aSmallInteger ]
%

category: 'Accessing'
method: GsCompilerIRNode
symbolLiteralValue

  ^ nil "caller should signal a RubyParseError"
%

category: 'Updating'
method: GsCompilerIRNode
validateEnvironment: anInteger
  "range check on an environment identifier for a method or send node"
  (anInteger < 0 or:[ anInteger > 16rFF ]) ifTrue:[
    anInteger error:'out of range' .
    ^ 0
  ].
  ^ anInteger
%

category: 'Accessing'
method: GsCompilerIRNode
varLeaf
  ^ nil
%

! Class extensions for 'GsComReturnNode'

!		Instance methods for 'GsComReturnNode'

removeallmethods GsComReturnNode
removeallclassmethods GsComReturnNode

category: 'Instance Initialization'
method: GsComReturnNode
breakFromRubyBlock: aNode
  kind := COMPAR_RETURN_NODE .
  expr := aNode .
  returnKind := -1 " would be COM_RTN_TWO_LEVELS " .
  self error:'COM_RTN_TWO_LEVELS not implemented in VM yet' .
%

category: 'Printing'
method: GsComReturnNode
printFormattedOn: aStream
  super printOn: aStream .
  aStream print:' returnKind:' int: returnKind ; cr ;
    indentMore .
  expr printFormattedOn: aStream .
  aStream indentLess ; nextPut: $) ; cr .
%

category: 'Instance Initialization'
method: GsComReturnNode
return: aNode
  kind := COMPAR_RETURN_NODE .
  expr := aNode .
  returnKind := COM_RTN_NORMAL .
%

category: 'Instance Initialization'
method: GsComReturnNode
returnFromHome: aNode
  kind := COMPAR_RETURN_NODE .
  expr := aNode .
  returnKind := COM_RTN_FROM_HOME.
%

! Class extensions for 'GsComSelectorLeaf'

!		Class methods for 'GsComSelectorLeaf'

removeallmethods GsComSelectorLeaf
removeallclassmethods GsComSelectorLeaf

category: 'Accessing'
classmethod: GsComSelectorLeaf
classToISAselector: aClass
  "Return the special selector to use in optimizing Ruby coerce_to
   for coercion to aClass .  Return nil if aClass does not have
   a special _is*  selector. "
  | val |
  val := SpecialISAselectors at: aClass otherwise: nil .
  val ifNotNil: [ val := val at: 1 ] .
  ^ val
%

category: 'Accessing'
classmethod: GsComSelectorLeaf
classToRubyClassName: aClass
  "Return the ruby class name to use in optimizing Ruby coerce_to
   for coercion to aClass .  Return nil if aClass does not have
   a special _is*  selector. "
  | val |
  val := SpecialISAselectors at: aClass otherwise: nil .
  ^ val  ifNil: [ aClass name ]
      ifNotNil: [ val at: 2 ] .
%

category: 'Initialization'
classmethod: GsComSelectorLeaf
newSelector: aSymbol env: envId
  "For Smalltalk this should normally be invoked via
   GsComSendNode>>stSelector: and envId should be zero regardless
   of the environment of the GsComMethNode method being generated."
  | entry |
  entry :=  SpecialSendsDict at: aSymbol otherwise: nil .
  entry ifNil:[
    ^ aSymbol " a non-optimized send"
  ] ifNotNil:[ | entryEnv |
    entryEnv := entry atOrNil: 4 .
    (entryEnv ~~ nil and:[ entryEnv ~~ envId]) ifTrue:[
       ^ aSymbol "not optimized in this environment"
    ].
    ^ self _basicNew _init: entry sym: aSymbol
  ]
%

category: 'Initialization'
classmethod: GsComSelectorLeaf
reimplementationAllowed: aSymbol inEnv: anInt
  "returns true if reimplementation is allowed in specified environmentId"
| env |
env := SpecialObjectSends at: aSymbol otherwise: nil .
env ifNotNil:[
  env < 0 ifTrue:[ ^ false "disallowed in all environments"].
  ^ anInt ~~ env
].
^ true
%

category: 'Initialization'
classmethod: GsComSelectorLeaf
selectorForSuper: aSymbol
  "cannot optimize a send to super "
  ^ aSymbol
%

category: 'Initialization'
classmethod: GsComSelectorLeaf
_initializeSpecialSelectors
  | specSendsDict specialISAdict objectSpecSends data |

  " syms are the ruby or Smalltalk selectors that will be sent for
    the specified bytecodes.  comparse.c takes care of
    optimizations for Smalltalk sends of those bytecodes, when compiling
    Smalltalk source. "

  "each triple  in data Array is
       1. selector to be optimized
                   (either a Ruby or Smalltalk selector from AST to IR phase)
       2. special send bytecode to use ,
       3. class on which special send is defined
       4. environment in which selector is a special send on Object,
           nil means all environments.
   special sends not defined on class Object get a send-site cache
   during bytecode generation and may fall back to real method"
  		"ruby_selector_suffix dependent"
  data := {
  { #'+#1__' . Bc_SEND_SPECIAL_PLUS_u1_u32 . SmallInteger . 1 } .   "for Ruby"
  { #'-#1__' . Bc_SEND_SPECIAL_MINUS_u1_u32 . SmallInteger . 1 } .
  { #'*#1__' . Bc_SEND_SPECIAL_MULTIPLY_u1_u32 . SmallInteger . 1 } .
  { #'>=#1__' . Bc_SEND_SPECIAL_GTE_u1_u32 . SmallInteger . 1 } .
  { #'<=#1__' . Bc_SEND_SPECIAL_Lte_u1_u32 . SmallInteger . 1 } .
  { #'<#1__'  . Bc_SEND_SPECIAL_LT_u1_u32 . SmallInteger . 1 } .

  { #'+' . Bc_SEND_SPECIAL_PLUS_u1_u32 . SmallInteger . 0 } .   "for Smalltalk"
  { #'-' . Bc_SEND_SPECIAL_MINUS_u1_u32 . SmallInteger . 0 } .
  { #'*' . Bc_SEND_SPECIAL_MULTIPLY_u1_u32 . SmallInteger . 0 } .
  { #'>=' . Bc_SEND_SPECIAL_GTE_u1_u32 . SmallInteger . 0 } .
  { #'<=' . Bc_SEND_SPECIAL_Lte_u1_u32 . SmallInteger . 0 } .
  { #'<'  . Bc_SEND_SPECIAL_LT_u1_u32 . SmallInteger . 0 } .

  { #'==' . Bc_SEND_SPECIAL_EQEQ . Object .   0 } .
  { #'~~' . Bc_SEND_SPECIAL_NENE . Object .   0  } .

  { #'_equal?#1__' . Bc_SEND_SPECIAL_EQEQ . Object . 1 } .  "Ruby identity compare"
  { #'_not_equal?#1__' . Bc_SEND_SPECIAL_NENE . Object . 1 } .  "Ruby identity compare"
  " any call variation with & does not use  Bc_SEND_CALL (1.8.7) "
  { #'_isInteger#0__' . Bc_SEND_SPECIAL_IS_INTEGER . Object . 1 } .
  { #'_isSmallInteger#0__' . Bc_SEND_SPECIAL_IS_SMALLINT . Object . 1 } .
  { #'_isFixnum#0__' . Bc_SEND_SPECIAL_IS_SMALLINT . Object . 1 } .
  { #'_isNumeric#0__' . Bc_SEND_SPECIAL_IS_NUMBER . Object . 1 } .
  { #'_isFloat#0__' . Bc_SEND_SPECIAL_IS_FLOAT . Object . 1 } .
  { #'_isSymbol#0__' . Bc_SEND_SPECIAL_IS_SYMBOL . Object . 1 } .
  { #'_isExecBlock#0__' . Bc_SEND_SPECIAL_IS_ExecBlock . Object . 1 } .
  { #'_isBlock#0__' . Bc_SEND_SPECIAL_IS_ExecBlock . Object . 1 } .
  { #'_isArray#0__' . Bc_SEND_SPECIAL_IS_Array . Object . 1 } .
  { #'_isStringOrSymbol#0__' . Bc_SEND_SPECIAL_IS_OneByteString . Object . 1 } . "used in Ruby"
  { #'_isRange#0__' . Bc_SEND_SPECIAL_IS_Range . Object } .

  { #_stringCharSize  . Bc_SEND_SPECIAL_stringCharSize . Object  . 0 } . 
  { #_isOneByteString . Bc_SEND_SPECIAL_IS_OneByteString . Object . 0 } . "used in Smalltalk"
  { #_isExceptionClass . Bc_SEND_SPECIAL_IS_ExceptionClass . Object . 0 } .
  { #_isNumber . Bc_SEND_SPECIAL_IS_NUMBER . Object . 0 } .
  { #_isScaledDecimal . Bc_SEND_SPECIAL_IS_ScaledDecimal . Object . 0 } . "Smalltalk"
  { #_isInteger . Bc_SEND_SPECIAL_IS_INTEGER . Object . 0 } .
  { #_isSmallInteger . Bc_SEND_SPECIAL_IS_SMALLINT . Object . 0 } .
  { #_isFloat . Bc_SEND_SPECIAL_IS_FLOAT . Object . 0 } .
  { #_isSymbol . Bc_SEND_SPECIAL_IS_SYMBOL . Object . 0 } .
  { #_isExecBlock  . Bc_SEND_SPECIAL_IS_ExecBlock . Object . 0 } .
  { #_isArray . Bc_SEND_SPECIAL_IS_Array . Object . 0 } .
  { #_isRange . Bc_SEND_SPECIAL_IS_Range . Object } .
  { #isNil . Bc_IS_NIL . Object } .
  { #notNil . Bc_NOT_NIL . Object } .
  { #yourself . Bc_SEND_yourself . Object }

  }.
  objectSpecSends := IdentityKeyValueDictionary new .
  specSendsDict := IdentityKeyValueDictionary new .
  data do:[:triple | | sym bc cls |
    sym := triple at: 1 . bc := triple at: 2 .  cls := triple at: 3 .
    specSendsDict at: sym put: { bc . cls } .
    cls == Object ifTrue:[ | env |
      env := triple atOrNil: 4 .
      env ifNil:[ env := -1 ].
      objectSpecSends at: sym put: env .
    ].
  ].
  objectSpecSends immediateInvariant .
  specSendsDict immediateInvariant .

  specialISAdict := IdentityKeyValueDictionary new .
  specialISAdict   "env 0 selectors used in Ruby coerce_to logic"
        at: Symbol put: #( _isSymbol Symbol ) ;
        at: SmallInteger put: #(  _isSmallInteger  Fixnum ) ;
        at: Float put: #(  _isFloat  Float ) ;
        at: Integer put: #(  _isInteger  Integer ) ;
        at: Number put: #(  _isNumber  Number ) ;
        at: ExecBlock put: #(  _isExecBlock  ExecBlock ) ;
        at: Array put: #(  _isArray  Array ) ;
        at: String put: #(  _isRubyString  String ) ;
        at: Range put: #(  _isRange  Range )  .
  specialISAdict immediateInvariant .

  #( #SpecialSendsDict #SpecialISAselectors #SpecialObjectSends ) do:[:sym|
    self _removeClassVar: sym ifAbsent:[].
  ].
  self  _addInvariantClassVar: #SpecialSendsDict value: specSendsDict ;
    _addInvariantClassVar: #SpecialISAselectors value: specialISAdict ;
    _addInvariantClassVar: #SpecialObjectSends value: objectSpecSends .
%

!		Instance methods for 'GsComSelectorLeaf'

category: 'Printing'
method: GsComSelectorLeaf
printFormattedOn: aStream

  super printOn: aStream .
  aStream nextPutAll: selector printString ;
      print: ' specialOpcode:' int: specialOpcode ;
      nextPutAll: ' specialSendClass:' ;
      nextPutAll:(specialSendClass ~~ nil ifTrue:[ specialSendClass name] ifFalse:['nil']) ;
      nextPut: $) .
%

category: 'Accessing'
method: GsComSelectorLeaf
selector
  ^ selector
%

category: 'Instance Initialization'
method: GsComSelectorLeaf
setIRnodeKind
%

category: 'Initialization'
method: GsComSelectorLeaf
_init: specialSendsEntry sym: aSymbol
  kind := COMPAR_SELECTOR_LEAF .
  selector := aSymbol .
  specialOpcode := specialSendsEntry at: 1 .
  specialSendClass := specialSendsEntry at: 2 .
  ^ self
%

! Class extensions for 'GsComSendNode'

!		Class methods for 'GsComSendNode'

removeallmethods GsComSendNode
removeallclassmethods GsComSendNode

category: 'Class Initialization'
classmethod: GsComSendNode
initialize
  "initialize the control op dictionary; these are the selectors for
   which the generator can generate in-line branching or looping code,
   if the receiver and/or arguments to the send meet certain critiera.

   For example:
     If the argument to ifTrue:  is a block , the block can be inlined.

   See GsComSendNode >> optimizeIfPossible for smalltalk implementation
   of the optimization logic; this logic also in src/comparse.c .
  "

  | dict |
  dict := IdentityKeyValueDictionary new .
  self _removeClassVar: #ControlOpDict  ifAbsent:[].
  self _addInvariantClassVar: #ControlOpDict value: dict .

  "all of the following branch constructs will observe the isRuby attribute
   of the current method node .
   for example:  'ifTrue:'  means  'if neither false nor nil' in a Ruby method.
  "
  		"ruby_selector_suffix dependent"
  dict at: #ifTrue:          put: COMPAR__IF_TRUE ;
       at: #ifFalse:         put: COMPAR__IF_FALSE ;
       at: #ifTrue:ifFalse:  put: COMPAR_IF_TRUE_IF_FALSE  ;
       at: #ifFalse:ifTrue:  put: COMPAR_IF_FALSE_IF_TRUE  ;
       at: #ifNil:ifNotNil:  put: COMPAR_IF_NIL_IF_NOTNIL ;
       at: #ifNotNil:ifNil:  put: COMPAR_IF_NOTNIL_IF_NIL ;
       at: #ifNil:           put: COMPAR_IF_NIL ;
       at: #ifNotNil:        put: COMPAR_IF_NOT_NIL ;
       at: #or:              put: COMPAR_OR_SELECTOR  ;
       at: #and:             put: COMPAR_AND_SELECTOR  ;
       at: #whileFalse:      put: COMPAR_WHILE_FALSE  ;
       at: #'whileFalse#1__'  put: COMPAR_WHILE_FALSE  ;

       at: #whileTrue:       put: COMPAR_WHILE_TRUE  ;
       at: #'whileTrue#1__'   put: COMPAR_WHILE_TRUE  ;

       at: #untilFalse:      put: COMPAR_UNTIL_FALS_COLON ;
       at: #'untilFalse#1__'  put: COMPAR_UNTIL_FALS_COLON  ;

       at: #untilTrue:       put: COMPAR_UNTIL_TRU_COLON ;
       at: #'untilTrue#1__'   put: COMPAR_UNTIL_TRU_COLON  ;

       at: #untilFalse       put: COMPAR_UNTIL_FALSE ;
       at: #untilTrue        put: COMPAR_UNTIL_TRUE ;
       at: #whileFalse       put: COMPAR_UNTIL_TRUE ;
       at: #whileTrue        put: COMPAR_UNTIL_FALSE ;
       at: #repeat      put: COMPAR_FOREVER_repeat ;

       at: #to:do:           put: COMPAR_TO_DO ;
       at: #to:by:do:        put: COMPAR_TO_BY_DO ;
       at: #timesRepeat:     put: COMPAR_TIMES_REPEAT ;
       at: #_downTo:do:      put: COMPAR__DOWNTO_DO ;
       at: #_downTo:by:do:   put: COMPAR__DOWNTO_BY_DO .
%

category: 'Instance Creation'
classmethod: GsComSendNode
new
  ^ self _basicNew initialize
%

!		Instance methods for 'GsComSendNode'

category: 'Instance Initialization'
method: GsComSendNode
appendArgument: aNode
  aNode ifNil:[ self error:'illegal nil argument'].
  arguments addLast: aNode
%

category: 'Instance Initialization'
method: GsComSendNode
environment
  ^ envFlags bitAnd: 16rFF
%

category: 'Instance Initialization'
method: GsComSendNode
environment: anInteger
  envFlags := ((envFlags bitShift: -8) bitShift: 8)
               bitOr: (self validateEnvironment: anInteger)
%

category: 'Instance Initialization'
method: GsComSendNode
initialize
  arguments := { } .
  controlOp := COMPAR_NO_OPTIMIZATION "a normal send" .
  envFlags := 0 .
  kind := COMPAR_SEND_NODE .
%

category: 'Transformation'
method: GsComSendNode
optimizationPossible
  "Return true if optimization to an in-line  branch or loop could be possible,
   otherwise return false .
   To be sent after initializing the selector."

  | sel |
  (sel := selLeaf) _isSymbol ifFalse:[ sel := sel selector ].
  ^ (ControlOpDict at: sel otherwise: 0 ) ~~ 0
%

category: 'Transformation'
method: GsComSendNode
optimize

  "Attempt to optimize the receiver to a special selector.
   Generates an error if receiver is not optimizable .
   Use  optimizationPossible  to determine if  optimize  would succeed."

  | op sel |
  (sel := selLeaf) _isSymbol ifFalse:[ sel := sel selector ].
  op := ControlOpDict at: sel otherwise: 0 .
  op ~~ 0 ifTrue:[
    controlOp := op
  ] ifFalse: [
    self error:'not optimizable'
  ]
%

category: 'Transformation'
method: GsComSendNode
optimizeIfPossible
  "returns true if optimization performed, false otherwise.
   Does not consider any possible Ruby env1 selectors."
  | sel op argSz args |
  (sel := selLeaf) _isSymbol ifFalse:[ sel := sel selector ].
  op := ControlOpDict at: sel otherwise: 0 .
  op ~~ 0 ifTrue:[ | clsBlkNode |
    clsBlkNode := GsComBlockNode .
    argSz := (args := arguments) size .
    argSz <= 1 ifTrue:[
      argSz == 1 ifTrue:[
        (sel == #ifTrue: or:[ sel == #ifFalse:
          or:[ sel == #ifNil: or:[ sel == #ifNotNil:
          or:[ sel == #or:  or:[ sel == #and:
          or:[ sel == #timesRepeat:          ]]]]]]) ifTrue:[
            (args atOrNil: 1) class == clsBlkNode ifTrue:[
              controlOp := op .
              ^ true
            ].
         ].
         (sel == #whileFalse: or:[ sel == #whileTrue:
          or:[ sel == #untilFalse:  or:[ sel == #untilTrue:  ]]]) ifTrue:[
            ((args atOrNil: 1) class == clsBlkNode
             and: [ rcvr class == clsBlkNode ]) ifTrue:[
              controlOp := op .
              ^ true
             ]
         ].
      ].
      "argSize == 0"
      (sel == #untilFalse or:[ sel == #untilTrue
        or:[ sel == #whileFalse or:[ sel == #whileTrue ]]]) ifTrue:[
          rcvr class == clsBlkNode ifTrue:[
            controlOp := op .
            ^ true
          ].
      ].
      sel == #repeat ifTrue:[
         (rcvr class == clsBlkNode and:[ rcvr numStatements > 0]) ifTrue:[
            controlOp := op .
            ^ true
         ].
      ].
      ^ false .
    ].
    argSz == 2 ifTrue:[
      (sel == #ifTrue:ifFalse:  or:[ sel == #ifFalse:ifTrue:  or:[
       sel == #ifNil:ifNotNil:  or:[ sel == #ifNotNil:ifNil:  ]]]) ifTrue:[
         ((args atOrNil: 1) class == clsBlkNode
          and:[ (args atOrNil: 2) class == clsBlkNode ]) ifTrue:[
            controlOp := op .
            ^ true
          ].
       ].
       (sel == #to:do: or:[ sel == #_downTo:do:])  ifTrue:[
          (args atOrNil: 2) class == clsBlkNode ifTrue:[
            controlOp := op .
            ^ true
          ].
       ].
       ^ false
    ].
    (sel == #to:by:do: or:[ sel ==  #_downTo:by:do: ]) ifTrue:[
      (args atOrNil: 3) class == clsBlkNode  ifTrue:[
         controlOp := op .
         ^ true
      ].
    ].
  ].
  ^ false
%

category: 'Printing'
method: GsComSendNode
printFormattedOn: aStream
  | sel |
  super printOn: aStream .
  aStream nextPutAll:' selLeaf: '; indentMore ; cr .
  (sel := selLeaf) _isSymbol ifTrue:[
     aStream nextPut: $# ; nextPut: $' ; nextPutAll: sel; nextPut: $' ; nextPut: $  .
  ] ifFalse:[
     sel printFormattedOn: aStream.
  ].
  aStream  indentLess ; cr ;
     print:' controlOp:' int: controlOp ;
     print:' envFlags:' int: envFlags ; cr;
  nextPutAll:' rcvr:' .
    rcvr ~~ nil ifTrue:[ rcvr printFormattedOn: aStream]
              ifFalse:[ aStream nextPutAll:'nil '].
  aStream nextPutAll: ' args:' ; do: arguments ;
  nextPut: $) ; cr .
%

category: 'Instance Initialization'
method: GsComSendNode
rcvr: aGsCompilerIRNode
  rcvr := aGsCompilerIRNode
%

category: 'Instance Initialization'
method: GsComSendNode
rubySelector: aSymbol
  " for a send NOT to super"

  selLeaf := GsComSelectorLeaf newSelector: aSymbol env: 1 .
  envFlags ~~ 0 ifTrue:[ self error:'should set flags after selector'].
  envFlags := 1 .
%

category: 'Instance Initialization'
method: GsComSendNode
rubySelector: aSymbol toSuper: superBool
  superBool ifTrue:[
    selLeaf := aSymbol  "can't optimize send to super"
  ] ifFalse:[
    selLeaf := GsComSelectorLeaf newSelector: aSymbol env: 1
  ].
  envFlags ~~ 0 ifTrue:[ self error:'should set flags after selector'].
  envFlags := 1 .
%

category: 'Instance Initialization'
method: GsComSendNode
rubySelectorInBridge: aSymbol
  selLeaf := GsComSelectorLeaf newSelector: aSymbol env: 1 .
    " and  comgen.c should generate a SEND_CURRENT"
  envFlags ~~ 0 ifTrue:[ self error:'should set flags after selector'].
  envFlags := 1 .
%

category: 'Instance Initialization'
method: GsComSendNode
setBypassRubyProtection
  envFlags := envFlags bitOr: BypassProtection_MASK
%

category: 'Instance Initialization'
method: GsComSendNode
setEvalLastArgFirst
  envFlags := envFlags bitOr: EvalLastArgFirst_MASK
%

category: 'Instance Initialization'
method: GsComSendNode
stSelector: aSymbol
  selLeaf := GsComSelectorLeaf newSelector: aSymbol env: 0 .
  envFlags := 0 .
  "at this point selLeaf is a Symbol for a non-optimized send
   or a GsComSelectorLeaf for an optimized send of a special selector."
%

! Class extensions for 'GsComStatementsNode'

!		Class methods for 'GsComStatementsNode'

removeallmethods GsComStatementsNode
removeallclassmethods GsComStatementsNode

category: 'Instance creation'
classmethod: GsComStatementsNode
new
  ^ self _basicNew initialize
%

!		Instance methods for 'GsComStatementsNode'

category: 'Instance Initialization'
method: GsComStatementsNode
initialize
  kind := COMPAR_STATEMENTS_NODE .
%

category: 'Printing'
method: GsComStatementsNode
lastLineNumber
  | lst sz |
  lst := list .
  (sz := lst size) _downTo: 1 do:[:n | | num |
    num := (lst at: n) lastLineNumber .
    num ifNotNil:[
       ^ num
    ].
  ].
  ^ nil
%

category: 'Printing'
method: GsComStatementsNode
lastSourceOffset
  | lst sz |
  lst := list .
  (sz := lst size) _downTo: 1 do:[:n | | num |
    num := (lst at: n) lastSourceOffset .
    num ifNotNil:[
       ^ num
    ].
  ].
  ^ nil
%

category: 'Printing'
method: GsComStatementsNode
lineNumber
  | lst |
  (lst := list) size ~~ 0 ifTrue:[ ^ ( lst at: 1) lineNumber ]  .
%

category: 'Instance Initialization'
method: GsComStatementsNode
list: anArray
  list := anArray
%

category: 'Printing'
method: GsComStatementsNode
printFormattedOn: aStream
  super printOn: aStream .
  aStream
    nextPutAll:' list:'; do: list ;
  nextPut: $) ; cr .
%

category: 'Printing'
method: GsComStatementsNode
sourceOffset
  | lst |
  (lst := list) size ~~ 0 ifTrue:[ ^ ( lst at: 1) sourceOffset ]  .
%

! Class extensions for 'GsComTermNode'

!		Class methods for 'GsComTermNode'

removeallmethods GsComTermNode
removeallclassmethods GsComTermNode

category: 'Instance creation'
classmethod: GsComTermNode
new

"disallowed, instances created only by comparse.c"

self shouldNotImplement: #new
%

!		Instance methods for 'GsComTermNode'

category: 'Printing'
method: GsComTermNode
printOn: aStream
  super printOn: aStream .
  aStream nextPut: $) ; cr .
%

! Class extensions for 'GsComVariableNode'

!		Class methods for 'GsComVariableNode'

removeallmethods GsComVariableNode
removeallclassmethods GsComVariableNode

category: 'Instance Creation'
classmethod: GsComVariableNode
globalNamed: aSymbol inDict: aDictionary
| assoc node |
assoc := aDictionary associationAt: aSymbol .
(node := self new)
  leaf: (GsComVarLeaf new literalVariable: assoc).
^ node
%

category: 'Instance Creation'
classmethod: GsComVariableNode
newSelf
| node |
(node := self new) leaf: (GsComVarLeaf new initializeSelf).
^ node
%

!		Instance methods for 'GsComVariableNode'

category: 'Instance Initialization'
method: GsComVariableNode
leaf: aGsComVarLeaf

kind := COMPAR_VAR_NODE .
leaf := aGsComVarLeaf
%

category: 'Accessing'
method: GsComVariableNode
litVarValue
  ^ leaf litVarValue
%

category: 'Accessing'
method: GsComVariableNode
litVarValueOrNil
  ^ leaf litVarValue
%

category: 'Printing'
method: GsComVariableNode
printFormattedOn: aStream
  super printOn: aStream .
  leaf printFormattedOn: aStream .
  aStream   nextPut: $) ; cr .
%

category: 'Accessing'
method: GsComVariableNode
varLeaf
  ^ leaf
%

! Class extensions for 'GsComVarLeaf'

!		Class methods for 'GsComVarLeaf'

removeallmethods GsComVarLeaf
removeallclassmethods GsComVarLeaf

category: 'Instance creation'
classmethod: GsComVarLeaf
new
"create an instance registered with VM for finalization of cData"

<primitive: 674>
self _primitiveFailed: #new
%

category: 'Printing'
classmethod: GsComVarLeaf
varKindToString: aKind
  aKind == COMPAR_BLOCK_ARG_VAR ifTrue:[ ^ 'BLOCK_ARG'].
  aKind == COMPAR_BLOCK_TEMP_VAR ifTrue:[ ^ 'BLOCK_TEMP'].
  aKind == COMPAR__INST_VAR ifTrue:[ ^ 'INST_VAR'].
  aKind == COMPAR_LIT_VAR ifTrue:[ ^ 'LIT_VAR'].
  aKind == COMPAR_METHOD_ARG_VAR ifTrue:[ ^ 'METHOD_ARG'].
  aKind == COMPAR_METHOD_TEMP_VAR ifTrue:[ ^ 'METHOD_TEMP'].
  "aKind == COMPAR_METH_VC_GLOBAL ifTrue:[ ^ 'METH_VC_GLOBAL']. "
  aKind == COMPAR_SELF_VAR ifTrue:[ ^ 'SELF'].
  aKind == COMPAR_SUPER_VAR ifTrue:[ ^ '_SUPER'].
  "COMPAR_LIT_VAR_SPECIAL_LITERAL not legal, from dbf conversion only"
  ^ 'INVALID_VAR'
%

!		Instance methods for 'GsComVarLeaf'

category: 'Instance Initialization'
method: GsComVarLeaf
blockArg: argNameSymbol argNumber: oneBasedArgNum forBlock: aGsComBlockNode
  self setIRnodeKind.
  argNameSymbol _isSymbol ifFalse:[ self error:'expected a symbol'].
  self setVarName: argNameSymbol .
  oneBasedArgNum < 1 ifTrue:[ oneBasedArgNum error:'out of range'].
  varOffset :=  oneBasedArgNum - 1 . "convert to zero based"
  varKind := COMPAR_BLOCK_ARG_VAR .
  lexLevel := aGsComBlockNode lexLevel
%

category: 'Instance Initialization'
method: GsComVarLeaf
blockTemp: tempNameSymbol sourceLexLevel: aLevel
  "code generator will delete block temps from
   the method if there are no IR references other than the definition"
  aLevel < 1 ifTrue:[ aLevel error:'out of range'].
  self setIRnodeKind.
  tempNameSymbol _isSymbol ifFalse:[ self error:'expected a symbol'].
  self setVarName: tempNameSymbol .
  varOffset := 0 . "generator will assign offsets for temps"
  varKind := COMPAR_BLOCK_TEMP_VAR .
  lexLevel := aLevel
%

category: 'Instance Initialization'
method: GsComVarLeaf
initializeSelf
  self setIRnodeKind .
  varName :=  #self .
  varOffset := 0  .
  varKind := COMPAR_SELF_VAR .
  lexLevel := 0 .
%

category: 'Instance Initialization'
method: GsComVarLeaf
initializeSuper
  self setIRnodeKind .
  varName :=  #super .
  varOffset := 0 .
  varKind := COMPAR_SUPER_VAR .
  lexLevel := 0 .
%

category: 'Instance Initialization'
method: GsComVarLeaf
instanceVariable: ivNameSymbol ivOffset: oneBasedIvOfs
  "Parser must lookup instVar names in class definitions to determine the
   oneBasedIvOfs.  Alternatively, additional behavior could be provided here to
   take an ivName and a Class and do the lookup here ..."
  self setIRnodeKind.
  self setVarName: ivNameSymbol .
  varKind := COMPAR__INST_VAR.
  oneBasedIvOfs > 0 ifTrue:[
    "instVar at known offset, convert to zero based"
    varOffset := oneBasedIvOfs - 1.
  ] ifFalse:[
    oneBasedIvOfs == -1 ifTrue:[
      varOffset := -1 "dynamic instVar"
    ] ifFalse:[
      oneBasedIvOfs error:'out of range'
    ]
  ].
%

category: 'Querying'
method: GsComVarLeaf
isArg
  ^ varKind == COMPAR_METHOD_ARG_VAR or:[ varKind == COMPAR_BLOCK_ARG_VAR]
%

category: 'Querying'
method: GsComVarLeaf
isTemp
  "return true if varKind one of
    COMPAR_METHOD_TEMP_VAR COMPAR_BLOCK_TEMP_VAR "

  ^ varKind <= COMPAR_BLOCK_TEMP_VAR
%

category: 'Accessing'
method: GsComVarLeaf
lexLevel
  ^ lexLevel
%

category: 'Instance Initialization'
method: GsComVarLeaf
literalVariable: anAssociation
  "Smalltalk style global variable, class variable, etc .
   Hopefully usable for Ruby globals ? "
  | key |
  self setIRnodeKind.
  key := anAssociation key .
  key _isSymbol ifFalse:[ key error:'bad arg kind'].
  self setVarName:  key .
  litVarAssoc := anAssociation .
  (anAssociation isKindOf: SymbolAssociation) ifFalse:[
    self error:'arg to literalVariable: is not a SymbolAssociation'
  ].
  varOffset := 0 .
  varKind := COMPAR_LIT_VAR .
%

category: 'Accessing'
method: GsComVarLeaf
litVarValue
  varKind == COMPAR_LIT_VAR ifFalse:[ self error:'not a literal variable'].
  ^ litVarAssoc _value
%

category: 'Instance Initialization'
method: GsComVarLeaf
methodArg: argNameSymbol argNumber: oneBasedArgNum
  self setIRnodeKind.
  argNameSymbol _isSymbol ifFalse:[ self error:'expected a symbol'].
  self setVarName: argNameSymbol .
  oneBasedArgNum < 1 ifTrue:[ oneBasedArgNum error:'out of range'].
  varOffset :=  oneBasedArgNum - 1 . "convert to zero based"
  varKind := COMPAR_METHOD_ARG_VAR.
  lexLevel := 0
%

category: 'Instance Initialization'
method: GsComVarLeaf
methodTemp: aSymbol
  "code generator will delete method temps from
   the method if there are no IR references other than the definition"
  self setIRnodeKind .
  self setVarName: aSymbol .
  varOffset := 0 .  "generator will assign offsets for temps"
  varKind := COMPAR_METHOD_TEMP_VAR  .
  lexLevel := 0
%

category: 'Printing'
method: GsComVarLeaf
printFormattedOn: aStream

  super printOn: aStream .
  aStream print:' varName: ' symbol: varName ;
      nextPutAll: ' varKind:' ; nextPutAll: self varKindString ;
      print: ' lexLevel:' int: lexLevel ;
      print: ' varOffset:' int: varOffset ;
      nextPut: $) ; cr .
%

category: 'Instance Initialization'
method: GsComVarLeaf
setIRnodeKind
  kind :=  COMPAR_VAR_LEAF
%

category: 'Instance Initialization'
method: GsComVarLeaf
setVarName: aName
  varName := aName .
%

category: 'Accessing'
method: GsComVarLeaf
varKind
  ^ varKind
%

category: 'Printing'
method: GsComVarLeaf
varKindString
  ^ self class varKindToString: varKind
%

category: 'Accessing'
method: GsComVarLeaf
varName
  ^ varName
%

category: 'Accessing'
method: GsComVarLeaf
varOffset
  ^ varOffset
%

! Class Initialization Excluded by export visitor
!  GsComSendNode initialize.
