! Copyright (C) GemTalk Systems 1986-2025.  All Rights Reserved.
! Class extensions for 'AbstractCollisionBucket'

!		Class methods for 'AbstractCollisionBucket'

removeallmethods AbstractCollisionBucket
removeallclassmethods AbstractCollisionBucket

category: 'Instance Creation'
classmethod: AbstractCollisionBucket
new

"Returns an AbstractCollisionBucket with a default capacity of four key/value
 pairs."

^ self new: 4
%

category: 'Instance Creation'
classmethod: AbstractCollisionBucket
new: aSize

"Returns an AbstractCollisionBucket with the specified size."

|result|
result := super new: (aSize + aSize).
result initialize.
^result
%

!		Instance methods for 'AbstractCollisionBucket'

category: 'Comparing'
method: AbstractCollisionBucket
= anObject

  "Reimplemented to compare by identity."

  ^ self == anObject
%

category: 'Adding'
method: AbstractCollisionBucket
add: anAssociation

self shouldNotImplement: #add:  " use at:put:keyValDict_coll: "
%

category: 'Accessing'
method: AbstractCollisionBucket
at: aKey

"Returns the value that corresponds to aKey."

^self at: aKey
    ifAbsent: nil "optimization, errors reported via _reportKeyNotFound"
%

category: 'Accessing'
method: AbstractCollisionBucket
at: aKey ifAbsent: aBlock

"Returns the value that corresponds aKey.  If no such key/value pair exists,
 returns the result of evaluating the zero-argument block aBlock."

  | index |
  index := self searchForKey: aKey.
  index ifNotNil:[ ^ self valueAt: index ] .
  aBlock ifNil:[ ^ self _errorKeyNotFound: aKey ] .
  ^ aBlock value
%

category: 'Accessing'
method: AbstractCollisionBucket
at: aKey otherwise: aValue

"Returns the value that corresponds to aKey by searching the elements in the
 bucket.  If no such key/value pair exists, returns the given alternate value."

  | index |
  index := self searchForKey: aKey.
  index ifNotNil:[ ^ self valueAt: index ].
   ^aValue .
%

category: 'Updating'
method: AbstractCollisionBucket
at: aKey put: aValue

self shouldNotImplement: #at:put:  " use at:put:keyValDict_coll: "
%

category: 'Updating'
method: AbstractCollisionBucket
at: aKey put: aValue keyValDict_coll: aKeyValDict
 "Stores the aKey/aValue pair in the receiver.
  Returns self size if this at:put: added a new key, 0 if this at:put:
  replaced the value of an existing key."
 | emptySlotIdx startTableSize thisKey numElem |

  startTableSize := self tableSize .
  aKey ifNil:[ ^ self _error: #rtErrNilKey ] .
  (numElem := numElements) == 0 ifTrue:[
    emptySlotIdx := 1
  ] ifFalse:[ | idx |
    "search for aKey, or for the first empty slot "
    idx := 1 .
    1 to: startTableSize do:[:n |
      thisKey := self _at: idx . "inline keyAt:"
      thisKey ifNotNil:[
        (self compareKey: aKey with: thisKey) ifTrue:[ "Key found.  Store given value"
          self _at: idx + 1 put: aValue .  "inline at:putValue:"
          aKeyValDict _markDirty .
          ^ 0
        ].
      ] ifNil:[
        emptySlotIdx ifNil:[ emptySlotIdx := idx ].
      ].
      idx := idx + 2 .
    ] .
    "Key not found so add key and value"
    emptySlotIdx ifNil:[ " bucket is full so grow it "
      emptySlotIdx := self _basicSize + 1 .
      self size: emptySlotIdx + 7  .  "accommodate 4 more key,value pairs"
    ] .
  ].
  numElem := numElem + 1.
  numElements := numElem .
  self _at: emptySlotIdx put: aKey. "inline at:putKey:"
  self _at: emptySlotIdx + 1 put: aValue . "inline at:putValue"
  ^ numElem
%

category: 'Updating'
method: AbstractCollisionBucket
at: anIndex putKey: aKey

"Stores the key aKey into the key part of the key/value pair referenced by
 anIndex.  Note that this method overwrites the key value at the given index.
 Returns aKey."

super at: (anIndex + anIndex) - 1 put: aKey
%

category: 'Updating'
method: AbstractCollisionBucket
at: anIndex putValue: aValue

"Stores the value aValue into the value part of the key/value pair referenced
 by atIndex.  Returns aValue."

super at: (anIndex + anIndex) put: aValue
%

category: 'Private'
method: AbstractCollisionBucket
compareKey: key1 with: key2

"Returns true if key1 is equivalent to key2, and false otherwise."

^ key1 = key2
%

category: 'Enumerating'
method: AbstractCollisionBucket
do: aBlock

"For each key/value pair in the receiver, evaluates the one-argument block
 aBlock with the value as the argument.  Returns the receiver."

	self valuesDo: aBlock
%

category: 'Deprecated'
method: AbstractCollisionBucket
doKeys: aBlock

self deprecated: 'doKeys: Obsolete in GemStone/64.  Use the keysDo: method instead.'.
^ self keysDo: aBlock.
%

category: 'Deprecated'
method: AbstractCollisionBucket
doValues: aBlock

self deprecated: 'doValues: Obsolete in GemStone/64.  Use the valuesDo: method instead.'.
^ self valuesDo: aBlock.
%

category: 'Searching'
method: AbstractCollisionBucket
firstPair

"Returns an Association containing the receiver's first key/value pair.
 If the receiver is empty, returns an Association containing nils."

   | aKey |
   numElements == 0 ifFalse: [
      "Search for the first non-nil index"
      1 to: self tableSize do: [ :i |
         (aKey := self keyAt: i) ifNotNil: [
            ^Association new key: aKey value: (self valueAt: i)
         ]
      ]
   ].
   "No first pair was found"
   ^Association newWithKey: nil value: nil
%

category: 'Comparing'
method: AbstractCollisionBucket
hash
  ^ self identityHash
%

category: 'Searching'
method: AbstractCollisionBucket
includesKey: aKey

"Returns true if the receiver contains a key that is equal to 'aKey'.
 Otherwise, returns false."

^(self searchForKey: aKey) ~~ nil
%

category: 'Initializing'
method: AbstractCollisionBucket
initialize

"Initializes the instance variable of the receiver to be an empty
 CollisionBucket."

numElements := 0.
%

category: 'Accessing'
method: AbstractCollisionBucket
keyAt: index

"Returns the key at the specified index."

^super at: (index + index - 1)
%

category: 'Accessing'
method: AbstractCollisionBucket
keyAt: aKey otherwise: aValue

"Returns the key that corresponds to aKey by searching the elements in the
 bucket.  If no such key/value pair exists, returns the given alternate value."

  | index |
  index := self searchForKey: aKey.
  index ifNotNil:[ ^ self keyAt: index ].
  ^ aValue
%

category: 'Enumerating'
method: AbstractCollisionBucket
keysAndValuesDo: aBlock

"For each key/value pair in the receiver, evaluates the two-argument block
 aBlock with the key and value as the arguments.  Returns the receiver."

| aKey |
1 to: self tableSize do: [ :index |
  (aKey := self keyAt: index) ~~ nil
    ifTrue: [
      aBlock value: aKey value: (self valueAt: index)
      ]
  ].
%

category: 'Enumerating'
method: AbstractCollisionBucket
keysDo: aBlock

"For each key/value pair in the receiver, evaluates the one-argument block
 aBlock with the key as the argument.  Returns the receiver."

| aKey |
1 to: self tableSize do: [ :index |
  (aKey := self keyAt: index) ~~ nil ifTrue: [ aBlock value: aKey ]
  ].
%

category: 'Accessing'
method: AbstractCollisionBucket
keyValueDictionary

"Returns nil.  Only IdentityCollisionBuckets have the keyValueDictionary
 instance variable."

^nil
%

category: 'Updating'
method: AbstractCollisionBucket
keyValueDictionary: aDict

"No-op for AbstractCollisionBuckets.  Used only for IdentityCollisionBuckets."
%

category: 'Accessing'
method: AbstractCollisionBucket
numElements

"Returns value of the numElements instance variable.  (The name numElements
 is provided for compatibility with earlier releases.  The instance method
 size is preferred for new code.)"

^numElements
%

category: 'Formatting'
method: AbstractCollisionBucket
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

"Copy the implementation from Object so we don't inherit it from Collection."

aStream nextPutAll: self asString
%

category: 'Removing'
method: AbstractCollisionBucket
removeKey: aKey ifAbsent: aBlock

"Removes the key/value pair having the key aKey.  If aKey is not found,
 returns the result of evaluating the zero-argument block aBlock."

  | keyIndex aValue |

  keyIndex := self searchForKey: aKey.
  keyIndex ifNotNil:[
      numElements := numElements - 1.
      aValue := self valueAt: keyIndex.
      self at: keyIndex putKey: nil.
      self at: keyIndex putValue: nil.
      ^aValue
  ].
  aBlock ifNil:[ ^ self _errorKeyNotFound: aKey ] .
  ^ aBlock value
%

category: 'Removing'
method: AbstractCollisionBucket
removeKey: aKey otherwise: notFoundValue

"Removes the key/value pair having the key aKey.  If aKey is not found,
 returns the notFoundValue. "

  | keyIndex aValue |

  keyIndex := self searchForKey: aKey.
  keyIndex == nil
    ifFalse: [
      numElements := numElements - 1.
      aValue := self valueAt: keyIndex.
      self at: keyIndex putKey: nil.
      self at: keyIndex putValue: nil.
      ^aValue ]
    ifTrue: [ ^ notFoundValue ]
%

category: 'Searching'
method: AbstractCollisionBucket
searchForKey: argKey

"Returns the index of argKey, or if not found, nil."
   | idx |
   argKey ifNil:[ ^ self _error: #rtErrNilKey ] .
   idx := 1 .
   1 to: self tableSize do: [ :n | | aKey |
      aKey := self _at: idx . "inline keyAt:"
      aKey ifNotNil:[
        (self compareKey: argKey with: aKey) ifTrue:[ ^ n ].
      ].
      idx := idx + 2
   ].
   ^ nil "Key not found"
%

category: 'Accessing'
method: AbstractCollisionBucket
size

"Returns value of the numElements instance variable."

^numElements
%

category: 'Accessing'
method: AbstractCollisionBucket
tableSize

"Returns the number of key/value pairs in the capacity of the receiver."

^self _basicSize // 2
%

category: 'Accessing'
method: AbstractCollisionBucket
valueAt: index

"Returns the value at the specified index."

^super at: (index + index)
%

category: 'Enumerating'
method: AbstractCollisionBucket
valuesDo: aBlock

"For each key/value pair in the receiver, evaluates the one-argument block
 aBlock with the value as the argument.  Returns the receiver."

1 to: self tableSize do: [ :index |
  (self keyAt: index) == nil ifFalse: [ aBlock value: (self valueAt: index) ]
  ].
%

category: 'Error Handling'
method: AbstractCollisionBucket
_errorKeyNotFound: aKey

"No key/value pair with given key, 'aKey', was found."

^ self _error: #rtErrKeyNotFound args: { aKey }.
%

category: 'Private'
method: AbstractCollisionBucket
_removeAll

"Dereferences the receiver from its parent and shrinks the receiver.
 Used while rebuilding a KeyValueDictionary."

numElements := 0.
"Gs64 v3.0, don't send size: 0 "
%

category: 'Removing'
method: AbstractCollisionBucket
_removeKey: aKey

"Removes the key/value pair having the key aKey.  If aKey is not found,
 generates an error."

  | keyIndex |

  keyIndex := self searchForKey: aKey.
  keyIndex == nil
    ifFalse: [
      numElements := numElements - 1.
      self at: keyIndex putKey: nil.
      self at: keyIndex putValue: nil.
    ]
    ifTrue: [ self _errorKeyNotFound: aKey ]
%

category: 'Accessing'
method: AbstractCollisionBucket
_reportKeyNotFound: aKey with: aBlock

"Private."

aBlock == nil ifTrue:[^ self _errorKeyNotFound: aKey ] .
^aBlock value
%

! Class extensions for 'AbstractDictionary'

!		Class methods for 'AbstractDictionary'

removeallmethods AbstractDictionary
removeallclassmethods AbstractDictionary

category: 'Storing and Loading'
classmethod: AbstractDictionary
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| size inst |
size := passiveObj readSize.
inst := self new.
inst loadFrom: passiveObj size: size.
^inst
%

category: 'Instance Creation'
classmethod: AbstractDictionary
new

"(Subclass responsibility.)  Returns a new instance of AbstractDictionary."

^ self subclassResponsibility: #new

%

category: 'Instance Creation'
classmethod: AbstractDictionary
new: count

"(Subclass responsibility.) Returns a new instance of AbstractDictionary. The
 argument count provides a hint of the number of elements the instance should
 be designed to hold."

^ self subclassResponsibility: #new:
%

category: 'Private'
classmethod: AbstractDictionary
_tableSizeFor: targetSize
  | limit |
  limit := 2034 - self instSize - 1 .
  (targetSize >= (limit - 100) and:[ targetSize <= limit ]) ifTrue:[
    "create the object with basicSize close to max small object size."
    ^ Integer _selectedPrimeGreaterThan: (limit min: 2028 )
  ].
  ^ Integer _selectedPrimeGreaterThan: targetSize .
%

!		Instance methods for 'AbstractDictionary'

category: 'Comparing'
method: AbstractDictionary
= anAbstractDictionary
"Returns true if all of the following conditions are true:

 1.  The receiver and anAbstractDictionary are of the same class.
 2.  The two dictionaries are of the same size.
 3.  The corresponding keys and values of the receiver and anAbstractDictionary
     are equal."

| notFound |
(self == anAbstractDictionary) ifTrue:[ ^ true ].

(self class == anAbstractDictionary class) ifFalse:[ ^ false ].

(self size == anAbstractDictionary size) ifFalse:[ ^ false ].
notFound := Object new .
self keysAndValuesDo: [ :aKey :aValue | | otherVal|
  otherVal := anAbstractDictionary at: aKey otherwise: notFound .
  (aValue == self or:[ aValue == anAbstractDictionary]) ifTrue:[
    (otherVal == self or:[ otherVal == anAbstractDictionary]) ifFalse:[ ^ false].
  ] ifFalse:[
    aValue = otherVal ifFalse:[ ^ false ]
  ].
].
^ true.
%

category: 'Enumerating'
method: AbstractDictionary
accompaniedBy: anObj keysAndValuesDo: aBlock

"Iteratively evaluates the threee argument block, aBlock,
 using anObj, each key and each value
 of the receiver as the arguments to the block.  Returns the receiver."

^ self subclassResponsibility: #accompaniedBy:keysAndValuesDo: .
%

category: 'Adding'
method: AbstractDictionary
add: anAssociation

"Adds the Association or the key-value pair contained in anAssociation to the
 receiver.  If the receiver already includes an Association/key-value pair
 whose key is equal to that of anAssociation, then this method redefines the
 value portion of that Association/key-value pair. Returns anAssociation."

self at: anAssociation key put: anAssociation value.
^ anAssociation.
%

category: 'Adding'
method: AbstractDictionary
addAll: aCollection

"Adds to the receiver all the Associations or key-value pairs contained in
 aCollection.  aCollection must be a collection of Associations or a dictionary.
 Returns the argument, aCollection."

(aCollection isKindOf: AbstractDictionary) ifTrue: [
  aCollection accompaniedBy: self
    keysAndValuesDo:[ :rcvr :aKey :aVal| rcvr at: aKey put: aVal ].
] ifFalse: [
  aCollection accompaniedBy: self do: [ :me :anAssociation | me add: anAssociation ]
].
^ aCollection.
%

category: 'Printing'
method: AbstractDictionary
asReportString

"Returns a String that lists the key-value pairs, one on each line."

| result lf tab |
result := String new.
tab := Character tab .
lf := Character lf .
self keysAndValuesDo: [ :key :val |
  result add: key printString; add: tab ; add: val printString; add: lf
].
^ result
%

category: 'Enumerating'
method: AbstractDictionary
associationsAsArray

"Returns an Array containing all of the receiver's Associations"

^ self selectAssociationsAsArray: [ :assoc| true ]
%

category: 'Enumerating'
method: AbstractDictionary
associationsDetect: aBlock

"Evaluates aBlock repeatedly, with the Associations of the receiver as the
 argument.  Returns the first Association for which the block evaluates to true
 when the Association is used as the argument to the block.  If none of the
 receiver's Associations evaluates to true, generates an error.  The argument
 aBlock must be a one-argument block."

^ self associationsDetect: aBlock
       ifNone: [^ self _error: #assocErrNoElementsDetected args: { aBlock }]
%

category: 'Enumerating'
method: AbstractDictionary
associationsDetect: aBlock ifNone: exceptionBlock

"Evaluates aBlock repeatedly, with the Associations of the receiver as the
 argument.  Returns the first Association for which aBlock evaluates to true
 when the Association is used as the argument to the block.  If none of the
 receiver's Associations evaluates to true, this method evaluates the argument
 exceptionBlock and returns its value.  The argument aBlock must be a
 one-argument block, and exceptionBlock must be a zero-argument block."

self associationsDo: [ :anAssoc|
  (aBlock value: anAssoc) ifTrue: [ ^anAssoc ].
  ].

^ exceptionBlock value.
%

category: 'Enumerating'
method: AbstractDictionary
associationsDo: aBlock

"Iteratively evaluates the one argument block, aBlock, using each Association
 in the receiver as the argument to the block. If the receiver stores the
 key-value pairs directly, new Associations are created for the key-value pairs
 and used as the arguments to aBlock.  Returns the receiver."

^ self subclassResponsibility: #associationsDo:
%

category: 'Accessing'
method: AbstractDictionary
at: aKey

"Returns the value of the Association with key aKey.  Generates an error if no
 such key exists."

    | sentinel result |
    sentinel := AbsDictSentinel"created in bom.c"  .
    (result := self at: aKey otherwise: sentinel) == sentinel ifTrue:
        [^self _errorKeyNotFound: aKey].
    ^result
%

category: 'Accessing'
method: AbstractDictionary
at: aKey ifAbsent: aBlock

"Returns the value associated with key aKey.  If no such key/value pair or
 Association exists, returns the result of evaluating the zero-argument block
 aBlock."

^ self subclassResponsibility: #at:ifAbsent.
%

category: 'Accessing'
method: AbstractDictionary
at: aKey ifAbsentPut: aBlock

"Returns the value associated with key aKey. If no such key/value pair or
 Association exists, returns the result of evaluating the zero-argument block
 aBlock. In the latter case, the result of  evaluating the block aBlock is
 also stored in the receiver using the key aKey."

| known existing value |

known := 'a'.
existing := self at: aKey otherwise: known.
existing == known
  ifTrue: [
    value := aBlock value.
    self at: aKey put: value.
    ^value
    ].
^existing
%

category: 'Accessing'
method: AbstractDictionary
at: key ifPresent: aBlock
"Lookup the given key in the receiver. If it is present, answer the value of
 evaluating the given one argument block with the value associated with the key.
 Otherwise, answer nil."

| v |
v := self at: key ifAbsent:[ ^ nil].
^ aBlock value: v
%

category: 'Accessing'
method: AbstractDictionary
at: aKey otherwise: value

"Returns the value that corresponds to aKey.  If no such key/value pair or
 Association exists, returns the given alternate value."

    ^self at: aKey ifAbsent: [value]
%

category: 'Updating'
method: AbstractDictionary
at: aKey put: aValue

"Creates a new Association with the given key and value and adds it to the
 receiver or adds the key-value pair to the receiver depending on the class of
 the receiver.  If the receiver already contains an Association or key-value
 pair with the given key, this method makes aValue the value of that
 Association or key-value pair.  Returns aValue."

^ self subclassResponsibility: #at:put:
%

category: 'Enumerating'
method: AbstractDictionary
collect: aBlock

"Evaluates aBlock with each of the receiver's values as the argument and
 collects the resulting values into the appropriate Dictionary at the
 corresponding key values."

| result |

result := self speciesForCollect new: (self size).
self keysAndValuesDo: [ :aKey :aValue |
  result at: aKey put: (aBlock value: aValue).
  ].

^ result.
%

category: 'Enumerating'
method: AbstractDictionary
collectAssociations: aBlock

"Evaluates aBlock with each of the receiver's Associations (or Associations
 created using the key-value pairs) as the argument and collects the resulting
 values into an Array. Returns the newly created Array."

| anArray index |

anArray := Array new: (self size).
index := 0.
self associationsDo: [ :anAssoc |
  index := index + 1.
  anArray at: index put: (aBlock value: anAssoc).
  ].

^ anArray.
%

category: 'Deprecated'
method: AbstractDictionary
collectValues: aBlock

self deprecated: 'AbstractDictionary>>collectValues: deprecated long before v3.0. Use collect: instead.'.
^ self collect: aBlock.
%

category: 'Enumerating'
method: AbstractDictionary
collectValuesAsArray: aBlock

"Evaluates aBlock with each of the receiver's values as the argument and
 collects the resulting values into an Array. Returns the new Array."

| result |
result := { } .
self valuesDo: [ :aValue |
  result add: (aBlock value: aValue).
  ].
^ result.
%

category: 'Deprecated'
method: AbstractDictionary
detectAssociations: aBlock

self deprecated: 'AbstractDictionary>>detectAssociations: deprecated long before v3.0. Use associationsDetect: instead.'.
^ self associationsDetect: aBlock
%

category: 'Deprecated'
method: AbstractDictionary
detectAssociations: aBlock ifNone: exceptionBlock

self deprecated: 'AbstractDictionary>>detectAssociations:ifNone: deprecated long before v3.0. Use associationsDetect:ifNone: instead.'.
^ self associationsDetect: aBlock ifNone: exceptionBlock
%

category: 'Deprecated'
method: AbstractDictionary
detectValues: aBlock

self deprecated: 'AbstractDictionary>>detectValues: deprecated long before v3.0.  Use keysAndValuesDo: instead.'.
^ self detectValues: aBlock ifNone: [ ^ self errorNoElementDetected: aBlock ].
%

category: 'Deprecated'
method: AbstractDictionary
detectValues: aBlock ifNone: exceptionBlock

self deprecated: 'AbstractDictionary>>detectValues:ifNone: deprecated long before v3.0. Use keysAndValuesDo: instead.'.
self keysAndValuesDo: [ :aKey :aValue |
  (aBlock value: aValue) ifTrue: [ ^ aKey ].
  ].
^ exceptionBlock value.
%

category: 'Enumerating'
method: AbstractDictionary
do: aBlock

"Iteratively evaluates the one argument block, aBlock, using the value part of
 each Association or key-value pair as the argument of the block.  Returns the
 receiver."

^ self subclassResponsibility: #do.
%

category: 'Deprecated'
method: AbstractDictionary
doAssociations: aBlock

self deprecated: 'AbstractDictionary>>doAssociations: deprecated long before v3.0. Use associationsDo: instead.'.
^ self associationsDo: aBlock.
%

category: 'Deprecated'
method: AbstractDictionary
doKeys: aBlock

self deprecated: 'AbstractDictionary>>doKeys: deprecated long before v3.0. Use keysDo: instead.'.
^ self keysDo: aBlock.
%

category: 'Deprecated'
method: AbstractDictionary
doKeysAndValues: aBlock

self deprecated: 'AbstractDictionary>>doKeysAndValues: deprecated long before v3.0. Use keysAndValuesDo: instead.'.
^ self keysAndValuesDo: aBlock.
%

category: 'Deprecated'
method: AbstractDictionary
doValues: aBlock

self deprecated: 'AbstractDictionary>>doValues: deprecated long before v3.0. Use valuesDo: instead.'.
^ self valuesDo: aBlock
%

category: 'Private'
method: AbstractDictionary
errorKeyNotFound: aKey

"No Association or key/value pair with the given key, aKey was found."

^ self _error: #rtErrKeyNotFound args: { aKey }
%

category: 'Private'
method: AbstractDictionary
errorNilKey

"A nil key was provided as an argument."

^ self _error: #rtErrNilKey.
%

category: 'Private'
method: AbstractDictionary
errorNoElementDetected: aBlock

"Private."

^ self _error: #assocErrNoElementsDetected args: { aBlock }
%

category: 'Comparing'
method: AbstractDictionary
hash

"Returns a numeric hash key for the receiver."

| hashValue |

hashValue := 97633 bitXor: (self size).
"For large dictionaries, the hash value is just a function of its size"
(self size > 64) ifTrue: [ ^ hashValue abs ].
self keysDo: [ :aKey |
   "Skip if the key is a dictionary."
   (aKey isKindOf: AbstractDictionary)
     ifFalse: [
       hashValue := hashValue bitXor: aKey hash
       ]
     ].
^ hashValue abs
%

category: 'Hashing'
method: AbstractDictionary
hashFunction: aKey

"The hash function should perform some operation on the value of the
 key (aKey) which returns a value in the range 1..tableSize."

^(aKey hash \\  self tableSize) + 1
%

category: 'Searching'
method: AbstractDictionary
includes: aValue

"Returns true if the receiver contains a value that is equal to aValue.
 Returns false otherwise."

self valuesDo: [ :element | (aValue = element) ifTrue: [ ^true ]].
^ false.
%

category: 'Searching'
method: AbstractDictionary
includesAssociation: anAssociation

"Returns true if anAssociation is equal to one of the Associations of the
 receiver.  Returns false otherwise."

self associationsDo: [ :assoc | (anAssociation = assoc) ifTrue: [ ^true ]].
^false
%

category: 'Searching'
method: AbstractDictionary
includesIdentical: aValue

"Returns true if the receiver contains a value that is identical to aValue.
 Returns false otherwise."

self valuesDo: [ :element | (aValue == element) ifTrue: [ ^true ]].
^ false.
%

category: 'Searching'
method: AbstractDictionary
includesIdenticalAssociation: anAssociation

"Returns true if anAssociation is identical to one of the Associations of the
 receiver.  Returns false otherwise."

self associationsDo: [ :assoc | (anAssociation == assoc) ifTrue: [ ^true ]].
^false
%

category: 'Searching'
method: AbstractDictionary
includesKey: aKey

"Returns true if the receiver contains an Association or a key-value pair whose
 key is equal to aKey.  Returns false otherwise."

| res |
res := true .
self at: aKey ifAbsent:[ res := false ] .
^ res .
%

category: 'Deprecated'
method: AbstractDictionary
includesValue: aValue

self deprecated: 'AbstractDictionary>>includesValue: deprecated long before v3.0. Use includes: instead.'.
^ self includes: aValue.
%

category: 'Accessing'
method: AbstractDictionary
keyAtValue: anObject

"Returns the key of the first value equal to anObject. If no match is found,
 runtime error objErrNotInColl is signaled."

"Note: In some implementations of Smalltalk, nil is returned if a match is
 not found."

^self keyAtValue: anObject
      ifAbsent: [^ self _error: #objErrNotInColl args: { anObject }]
%

category: 'Accessing'
method: AbstractDictionary
keyAtValue: anObject ifAbsent: aBlock

"Returns the key of the first value equal to the given object, anObject.
 If no match is found, evaluates and returns the result of the block aBlock."

^ self subclassResponsibility: #keyAtValue:ifAbsent:
%

category: 'Enumerating'
method: AbstractDictionary
keysAndValuesDo: aBlock

"Iteratively evaluates the two argument block, aBlock, using each key and value
 of the receiver as the argument to the block.  Returns the receiver."

^ self subclassResponsibility: #keysAndValuesDo:.
%

category: 'Enumerating'
method: AbstractDictionary
keysDo: aBlock

"Iteratively evaluates the one argument block, aBlock, using each key of
 the receiver as the argument to the block. Returns the receiver."

^ self subclassResponsibility: #keysDo:.
%

category: 'Storing and Loading'
method: AbstractDictionary
loadFrom: passiveObj size: varyingSize

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into the receiver."

"This method is similar to loadFrom:, but is used for objects whose size
 is not known when they are first instantiated (such as an IdentitySet)."

^self basicLoadFrom: passiveObj size: varyingSize
%

category: 'Instance Migration'
method: AbstractDictionary
migrateIndexable: anotherObject myClass: cls otherClass: othercls

"For dictionaries, we need to reconstruct the indexable component rather
 than just copy it over (in case the new class uses a different structure)."

anotherObject keysAndValuesDo: [ :key :value |
    self at: key put: value ].

%

category: 'Updating'
method: AbstractDictionary
objectSecurityPolicy: anObjectSecurityPolicy

  "Assigns the receiver and all its components to the given security policy.
   Returns the receiver."

  super objectSecurityPolicy: anObjectSecurityPolicy.
  self _nodesObjectSecurityPolicy: anObjectSecurityPolicy.
%

category: 'Searching'
method: AbstractDictionary
occurrencesOf: aValue

"Returns the number of Associations or key-value pairs in the receiver with
 value equal to 'aValue'."

| numOccurrences |

numOccurrences := 0.
self valuesDo: [ :aVal |
  aValue = aVal ifTrue: [ numOccurrences := numOccurrences + 1 ]
  ].

^ numOccurrences
%

category: 'Searching'
method: AbstractDictionary
occurrencesOfIdentical: aValue

"Returns the number of Associations or key-value pairs in the receiver with
 a value that is identical to aValue."

| numOccurrences |

numOccurrences := 0.
self valuesDo: [ :aVal |
  aValue == aVal ifTrue: [ numOccurrences := numOccurrences + 1 ]
  ].

^ numOccurrences
%

category: 'Deprecated'
method: AbstractDictionary
occurrencesOfValue: aValue

self deprecated: 'AbstractDictionary>>occurrencesOfValue: deprecated long before v3.0. Use occurrencesOf: instead.'.
^ self occurrencesOf: aValue.
%

category: 'Json'
method: AbstractDictionary
printJsonOn: aStream

	| delimiter |
	delimiter := ''.
	aStream nextPut: ${.
	self keysAndValuesDo: [:key :value |
		aStream nextPutAll: delimiter.
		key asString printJsonOn: aStream.
		aStream nextPut: $:.
		value printJsonOn: aStream.
		delimiter := ','.
	].
	aStream nextPut: $}.
%

category: 'Formatting'
method: AbstractDictionary
printNonRecursiveRepresentationOn: aStream recursionSet: anIdentitySet
	"Put a displayable representation of the receiver on the given stream
	 while avoiding recursion from object reference loops."

	| count sz myCls |

	myCls := self class .
	aStream nextPutAll: myCls name describeClassName .
	(myCls whichClassIncludesSelector: #associationsDo:) == AbstractDictionary
		ifTrue:[ ^ self "can't safely execute associationsDo: " ].

	aStream nextPutAll: '( ' .
	count := 1 .
	sz := self size .
	self associationsDo: [:anAssoc |
		anAssoc printOn: aStream recursionSet: anIdentitySet.
		aStream isFull ifTrue:[
			"prevent infinite recursion when printing cyclic structures, and
			limit the size of result when printing large collections."
			aStream _nextPut:( (aStream _collection endsWith: '...') ifTrue:[ $) ] ifFalse:[ ' ...)' ]).
			^ self
			] .
		count < sz ifTrue:[ aStream nextPutAll: ', ' ].
		count := count + 1.
		].
	aStream nextPut: $).
%

category: 'Formatting'
method: AbstractDictionary
printOn: aStream
	"Put a displayable representation of the receiver on the given stream."

	self printNonRecursiveOn: aStream
%

category: 'Enumerating'
method: AbstractDictionary
reject: aBlock

"Evaluates aBlock with each of the receiver's values as the argument. Stores
 the key-value pairs for which aBlock is false into a dictionary of the same
 class as the receiver, and returns the new dictionary. The argument aBlock
 must be a one-argument block."

| result |

result := self species new: (self size).
self keysAndValuesDo: [ :aKey :aValue |
  (aBlock value: aValue) ifFalse: [result at: aKey put: aValue]
  ].

^ result.
%

category: 'Enumerating'
method: AbstractDictionary
rejectAssociations: aBlock

"Evaluates aBlock with each of the receiver's elements as the argument.  Stores
 the values for which aBlock is false into a collection of the same class as
 the receiver, and returns the new collection.  The argument aBlock must be a
 one-argument block.  Uses associative access when the argument is a
 SelectionBlock."

| newCollection |

newCollection := self species new.
self associationsDo: [ :element |
  (aBlock value: element) ifFalse: [ newCollection add: element ].
  ].

^ newCollection.
%

category: 'Deprecated'
method: AbstractDictionary
rejectValues: aBlock

self deprecated: 'AbstractDictionary>>rejectValues: deprecated long before v3.0. Use reject: instead.'.
^ self reject: aBlock.
%

category: 'Enumerating'
method: AbstractDictionary
rejectValuesAsArray: aBlock

"Evaluates aBlock with each of the receiver's values as the argument and
 returns an Array containing the values for which aBlock evaluates false."

| result |
result := { } .
self valuesDo: [ :aValue |
  (aBlock value: aValue) ifFalse: [result add: aValue].
  ].

^ result.
%

category: 'Removing'
method: AbstractDictionary
remove: anObject

"Disallowed.  Use #removeKey: instead."

self shouldNotImplement: #remove:
%

category: 'Removing'
method: AbstractDictionary
remove: anObject ifAbsent: anExceptionBlock

"Disallowed.  Use #removeKey:ifAbsent: instead."

self shouldNotImplement: #remove:ifAbsent.
%

category: 'Removing'
method: AbstractDictionary
removeAll: aCollection

"Disallowed.  Use #removeAllKeys: instead."

self shouldNotImplement: #removeAll:
%

category: 'Removing'
method: AbstractDictionary
removeAllIdentical: aCollection

"Disallowed."

self shouldNotImplement: #removeAllIdentical:
%

category: 'Removing'
method: AbstractDictionary
removeAllKeys: keys

"Removes all the keys equal to the given keys from the receiver. An error is not
 generated if keys equal to any of the specified keys are not present. Returns
 the collection keys."

^ keys accompaniedBy: self do: [ :me :aKey | me removeKey: aKey otherwise: nil ].
%

category: 'Removing'
method: AbstractDictionary
removeAllKeys: keys ifAbsent: aBlock

"Removes all the keys equal to the given keys from the receiver and returns the
 collection keys. For any key which is not a valid key of the receiver, aBlock
 is evaluated with the key as the argument."

^ keys accompaniedBy: self do: [:me :aKey | me removeKey: aKey ifAbsent: [aBlock value: aKey] ].
%

category: 'Removing'
method: AbstractDictionary
removeIdentical: anObject

"Disallowed."

self shouldNotImplement: #removeIdentical:
%

category: 'Removing'
method: AbstractDictionary
removeIdentical: anObject ifAbsent: anExceptionBlock

"Disallowed."

self shouldNotImplement: #removeIdentical:ifAbsent.
%

category: 'Removing'
method: AbstractDictionary
removeKey: aKey

"Removes the Association or key-value pair with key equal to aKey from the
 receiver and returns the value portion of that Association or key-value pair
 respectively. If no Association is present with key equal to aKey, reports an
 error."

^ self removeKey: aKey ifAbsent: [self _errorKeyNotFound: aKey ].
%

category: 'Removing'
method: AbstractDictionary
removeKey: aKey ifAbsent: aBlock

"Removes the Association or key-value pair with key equal to aKey from the
 receiver and returns the value associated with the Association or the key-value
 pair.  If no Association or key-value pair is present with key equal to
 aKey, evaluates the zero-argument block aBlock and returns the result of that
 evaluation."

^ self subclassResponsibility: #removeKey:ifAbsent.
%

category: 'Removing'
method: AbstractDictionary
removeKey: aKey otherwise: notFoundValue

"Removes the Association or key-value pair with key equal to aKey from the
 receiver and returns the value associated with the Association or the key-value
 pair.  If no Association or key-value pair is present with key equal to
 aKey, returns notFoundValue .

 KeyValueDictionary has an optimized implementation.
"

^ self removeKey: aKey ifAbsent:[ notFoundValue ]
%

category: 'Deprecated'
method: AbstractDictionary
removeKeys: keys

self deprecated: 'AbstractDictionary>>removeKeys: deprecated long before v3.0. Use removeAllKeys: instead.'.
^ self removeAllKeys: keys
%

category: 'Enumerating'
method: AbstractDictionary
select: aBlock

"Evaluates aBlock with each of the receiver's values as the argument.  Stores
 the values for which aBlock is true into the dictionary of the same class as
 the receiver, and returns the new dictionary.  The argument aBlock must be a
 one-argument block."

|result|

result := self species new.
self keysAndValuesDo: [ :aKey :aValue |
  (aBlock value: aValue) ifTrue: [result at: aKey put: aValue]
  ].

^ result.
%

category: 'Enumerating'
method: AbstractDictionary
selectAssociations: aBlock

"Evaluates aBlock with each of the receiver's elements as the argument.  Stores
 the values for which aBlock is true into a collection of the same class as the
 receiver, and returns the new collection.  The argument aBlock must be a
 one-argument block.  Uses associative access when the argument is a
 SelectionBlock."

| newCollection |

newCollection := self species new.
self associationsDo: [ :element |
  (aBlock value: element) ifTrue: [ newCollection add: element ].
  ].

^ newCollection.
%

category: 'Enumerating'
method: AbstractDictionary
selectAssociationsAsArray: aBlock

"Evaluates aBlock with each of the receiver's associations as the argument.  Stores
 the values for which aBlock is true into a new Array and returns the Array.
 The argument aBlock must be a one-argument block.  "

| newCollection |

newCollection := { }  .
self associationsDo: [ :element |
  (aBlock value: element) ifTrue: [ newCollection add: element ].
  ].

^ newCollection.
%

category: 'Deprecated'
method: AbstractDictionary
selectValues: aBlock

self deprecated: 'AbstractDictionary>>selectValues: deprecated long before v3.0. Use select: instead.'.
^ self select: aBlock.
%

category: 'Deprecated'
method: AbstractDictionary
selectValuesAsArray: aBlock


"Evaluates aBlock with each of the receiver's values as the argument and
 returns an Array containing the values for which aBlock evaluates true."

| result index |
self deprecated: 'AbstractDictionary>>selectValuesAsArray: deprecated long before v3.0.'.
result := Array new: (self size).
index := 1.
self valuesDo: [ :aValue |
  (aBlock value: aValue)
    ifTrue: [result at: index put: aValue].
  index := index + 1.
  ].

^ result.
%

category: 'Accessing'
method: AbstractDictionary
size

"Returns the number of elements (Associations/key-value pairs) contained in the
 receiver."

^ self subclassResponsibility: #size.
%

category: 'Updating'
method: AbstractDictionary
size: newSize

"Disallowed.  You should not change the size of a dictionary explicitly."

self shouldNotImplement: #size: .
%

category: 'Accessing'
method: AbstractDictionary
squeakBasicAt: anIndex
  ^ self _basicAt: anIndex
%

category: 'Updating'
method: AbstractDictionary
squeakBasicAt: anIndex put: aValue

  ^ self _basicAt: anIndex put: aValue
%

category: 'Private'
method: AbstractDictionary
tableSize

"Returns the size of hash table used for storing the entries."

^ self subclassResponsibility: #tableSize.
%

category: 'Accessing'
method: AbstractDictionary
values

"Returns an OrderedCollection containing the receiver's values."

|result|
result:= OrderedCollection new.
self valuesDo: [ :value | result add: value ].
^ result.
%

category: 'Enumerating'
method: AbstractDictionary
valuesDo: aBlock

"Iteratively evaluates the one argument block, aBlock, using each value of
 the receiver as the argument to the block. Returns the receiver."

"Note: This method has the same behavior as #do:."

^ self subclassResponsibility: #valuesDo:.
%

category: 'Private'
method: AbstractDictionary
_asCollectionForSorting

"Creates a new collection that can be easily indexed for sorting."

| result |

result := { } .
self do: [ :aValue | result add: aValue].
^ result.
%

category: 'Accessing'
method: AbstractDictionary
_at: anIndex

"Signals an error if the receiver is not indexable,
 or if anIndex is not a SmallInteger or is out of bounds.

 This method is for internal use in implementation of the kernel classes.

 The methods _at:, _basicAt:, and _basicSize operate on the logically
 indexable contents of the receiver.

 This method must not be overridden."

<primitive: 699>
(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex }.
self _uncontinuableError
%

category: 'Updating'
method: AbstractDictionary
_at: anIndex put: aValue

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Returns aValue.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds."

<primitive: 745>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

self _primitiveFailed: #at:put: args: { anIndex . aValue }.
self _uncontinuableError
%

category: 'Private'
method: AbstractDictionary
_atZ: zOffset
  "return varying instVar specified by zero-based offset zOffset"
<primitive: 858>
| sz |
zOffset _validateClass: SmallInteger .
sz := self _basicSize .
(zOffset < 0 or:[ zOffset >= sz ]) ifTrue:[
  OffsetError new offset: zOffset maximum: sz - 1 ; signal
].
self _primitiveFailed: #_atZ: args: { zOffset }
%

category: 'Private'
method: AbstractDictionary
_atZ: zOffset put: aValue
  "store aValue into varying instVar specified by zero-based offset zOffset.
   will auto-grow up to 6 instVars past current end"
<primitive: 860>
| max |
zOffset _validateClass: SmallInteger .
max := self _basicSize + 6 .
(zOffset < 0 or:[ zOffset >= max ]) ifTrue:[
  OffsetError new offset: zOffset maximum: max ; signal
].
self _primitiveFailed: #_atZ:put: args: { zOffset . aValue }
%

category: 'Private'
method: AbstractDictionary
_atZ: zOffset putKey: aKey value: aValue
  "store aKey into varying instVar specified by zero-based offset zOffset.
   store aValue into varying instVar specified by zero-based offset zOffset + 1.
   will auto-grow up to 7 instVars past current end"
<primitive: 892>
| max |
zOffset _validateClass: SmallInteger .
max := self _basicSize + 6 .
(zOffset < 0 or:[ zOffset >= max ]) ifTrue:[
  OffsetError new offset: zOffset maximum: max ; signal
].
self _primitiveFailed: #_atZ:putKey:value:
     args: { zOffset . aKey . aValue }
%

category: 'Accessing'
method: AbstractDictionary
_basicAt: anIndex

"Signals an error if the receiver is not indexable,
 or if anIndex is not a SmallInteger or is out of bounds.

 This method is for internal use in implementation of the kernel classes.

 The methods _at:, _basicAt:, and _basicSize operate on the logically
 indexable contents of the receiver.

 This method must not be overridden."

<primitive: 699>
(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex }.
self _uncontinuableError
%

category: 'Updating'
method: AbstractDictionary
_basicAt: anIndex put: aValue

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Returns aValue.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds."

<primitive: 745>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

self _primitiveFailed: #at:put: args: { anIndex . aValue }
%

category: 'Private'
method: AbstractDictionary
_deepCopyWith: copiedObjDict

| copy myClass |

copy := copiedObjDict at: self otherwise: nil.
copy ~~ nil ifTrue: [ ^ copy ].

myClass := self class .
copy := myClass new: (self size).
copiedObjDict at: self put: copy.

self _deepCopyNamedIvsWith: copiedObjDict to: copy .

self keysAndValuesDo: [ :aKey :aValue |
  copy at: (aKey _deepCopyWith: copiedObjDict)
       put: (aValue _deepCopyWith: copiedObjDict).
  ].

^ copy.
%

category: 'Private'
method: AbstractDictionary
_deferredGciUpdateWith: valueArray

^ self subclassResponsibility: #_deferredGciUpdateWith:
%

category: 'Private'
method: AbstractDictionary
_errorKeyNotFound: aKey

"No Association or key-value pair with given key, aKey, was found."

^ self _error: #rtErrKeyNotFound args: { aKey }
%

category: 'Private'
method: AbstractDictionary
_nodesObjectSecurityPolicy: anObjectSecurityPolicy
  "Assigns receiver's components to the given security policy. "

  self subclassResponsibility: #_nodesObjectSecurityPolicy:
%

category: 'Private'
method: AbstractDictionary
_reportKeyNotFound: aKey with: aBlock

"Returns the value of aBlock if aBlock is not nil.  Otherwise, raises an error."

aBlock == nil ifTrue:[^ self _errorKeyNotFound: aKey ] .
^aBlock value
%

! Class extensions for 'AbstractException'

!		Class methods for 'AbstractException'

removeallmethods AbstractException
removeallclassmethods AbstractException

category: 'ExceptionSet support'
classmethod: AbstractException
, anExceptionSetOrClass

 "Return an ExceptionSet containing the receiver and argument."

  ^ ExceptionSet with: self with: anExceptionSetOrClass
%

category: 'Default handlers'
classmethod: AbstractException
addDefaultHandler: handlerBlock

"Install the one-argument block handlerBlock as a static GsExceptionHandler
 to field exceptions whose classes are subclasses of the receiver.
 The handlerBlock may send  #resume: , #pass, or #outer to the
 instance of Exception .

 If handlerBlock returns normally, or sends #return:
 to the instance of Exception, an Error will be signalled
 because there is no send of on:do:  which installed handlerBlock .

 Returns a new instance of GsExceptionHandler .

 When searching for a handlerBlock to handle a signaled exception, the VM
 uses Behavior>>_subclassOf: semantics . classHistories of the
 class of the signaled exception and of self are ignored.  "

^ self _installStaticException: handlerBlock class: self
       category: nil number: nil subtype: nil
%

category: 'Legacy Handlers'
classmethod: AbstractException
block: aBlock category: aCategory number: num subtype: atype

"This form of creating a Legacy handler is no longer supported.
 You must use category:number:do:  or  installStaticException:category:number: ."

self shouldNotImplement: #block:category:number:subtype: .
self _uncontinuableError
%

category: 'Legacy Handlers'
classmethod: AbstractException
category: aCategory number: aNumber do: handlerBlock

"Install a legacy exception handler in the frame of the sender of
   category:number:do:  .
The handler is represented as an instance of GsExceptionHandler .

aCategory is ignored; it is assumed to be nil or
the SymbolDictionary GemStoneError .
All exceptions are trapped, subject to gsTrappable instVar semantices and 
subject to the value of aNumber .

handlerBlock must be a 4-argument instance of ExecBloc.

aNumber is an error number of an exception to trap
or nil (to trap all exceptions) .

The subtype of the handler is always nil."

<primitive: 376>
" aCategory ~~ nil ifTrue:[ aCategory _validateClass: SymbolDictionary ] . "
aNumber ~~ nil ifTrue:[ aNumber _validateClass: SmallInteger ] .
handlerBlock _validateClass: ExecBlock .
handlerBlock argumentCount == 4 ifFalse:[
  handlerBlock _error: #rtErrBadBlockArgCount args: { 4 . handlerBlock argumentCount } .
  ] .
^ self _primitiveFailed: #category:number:do:
       args: { aCategory . aNumber . handlerBlock }
%

category: 'Debugging Support'
classmethod: AbstractException
cpuOsKind

"Returns an integer specifying the CPU and OS on which
 this VM is running. The result is one of
   1 - reserved for use within VM
   2 - Arm Linux
   3 - x86_64 Linux
   4 - PowerPc AIX
   5 - x86_64 Apple Unix
   6 - Arm  Apple Unix
"
<primitive: 345>
self _primitiveFailed: #cpuOsKind
%

category: 'Debugging Support'
classmethod: AbstractException
cpuOsKindString

^ #( 'unknown'
     'arm-linux'
     'x86_64-linux'
     'powerpc-AIX'
     'x86_64-darwin'
     'arm-darwin') at: self cpuOsKind
%

category: 'Default handlers'
classmethod: AbstractException
defaultHandlers
  "Returns all currently installed ANSI default handlers which
   handle the receiver. "
  | handler res hcls |
  res := { } .
  handler := self _staticExceptions .
  [ handler ~~ nil ] whileTrue:[
    (hcls := handler exceptionClass) ifNotNil:[
      (self _subclassOf: hcls) ifTrue:[ res add: handler ].
    ].
    handler := handler next
  ].
  ^ res
%

category: 'Errno'
classmethod: AbstractException
errnoTables
 "Returns the class variable ErrnoTables .
  The class variable ErrnoTables is a zero-based invariant Array ,
   to be indexed by the one-based result of  AbstractException(C)>>cpuOsKind .

   Each element of ErrnoTables is an Array of Strings, to be accessed
     by a one-based errno value, translating the errno value to a name.

   On some cpu/OS there are multiple names for some errno values.
   The tables contain only the preferred name, as follows
     Solaris:  EWOULDBLOCK==EAGAIN , EAGAIN is the preferred name .
     Linux: EWOULDBLOCK == EAGAIN, EAGAIN is the preferred name ,
            EDEADLOCK == EDEADLK,  EDEADLK is the preferred name .
     Apple Unix: EWOULDBLOCK==EAGAIN , EAGAIN is the preferred name ,
                 EOPNOTSUPP == ENOTSUP, ENOTSUP is the preferred name .
  "
  ^ ErrnoTables
%

category: 'Errno'
classmethod: AbstractException
errnoToName: aSmallInteger
 "Return the name for specified errno value, or nil if argument
  is out of range.

  See AbstractException>>errnoTables for discussion of preferred names
"

 | arr |
 arr := ErrnoTables at: self cpuOsKind .
 (aSmallInteger >= 1 and:[aSmallInteger <= arr size]) ifTrue:[
   ^ arr at: aSmallInteger .
 ].
 ^ nil
%

category: 'ExceptionSet support'
classmethod: AbstractException
handles: anException

^ anException isKindOf: self
%

category: 'Debugging Support'
classmethod: AbstractException
installDebugBlock: aBlock

"aBlock must be either nil or a one-argument ExecBlock taking an instance
 of AbstractException as the argument.
 aBlock == nil clears any previously installed block from the VM.
 Each GciLogin includes an automatic   AbstractException installDebugBlock: nil
 for the newly created session.

 aBlock ~~ nil installs the specified block into VM transient state,
 for use by AbstractException>>_debugException:  .
 aBlock will be sent   value: anException   prior to invocation of any
 handler block installed with an on:do:  (or prior to _executeGsHandler:).
 If debugger desires to handle the exception normally, aBlock must return
 normally (i.e. do not return from home, or signal any exception within aBlock).

 aBlock will not be invoked for non-trappable exceptions such as Object>>pause.

 aBlock is only executed if a matching handler was found on the stack.
 If there is no matching handler (i.e. no matching on:do:) then the
 exception is signalled directly to the GCI client.
"

<primitive: 798>
aBlock _validateClass: ExecBlock .
aBlock numArgs ~~ 1 ifTrue:[ ArgumentError signal:'must be a 1-arg block' ].
self _primitiveFailed: #installDebugBlock args: { aBlock }
%

category: 'Static legacy handlers'
classmethod: AbstractException
installStaticException: handlerBlock category: category number: num

"Install the 4-argument handlerBlock as a static GsExceptionHandler
 to field errors of the specified category and number.
 Returns a new instance of GsExceptionHandler "

^ self _installStaticException: handlerBlock class: nil
       category: category number: num subtype: nil
%

category: 'Static legacy handlers'
classmethod: AbstractException
installStaticException: handlerBlock category: aCategory number: aNumber subtype: atype

"Install the 4-argument handlerBlock at the head of the static handlers
 list, not associated with a particular stack frame ,
 to field errors of the specified category, number, and subtype.

 The handler is represented as an instance of GsExceptionHandler,
 and will be a part of the session state.
 The new  GsExceptionHandler is returned ."

^ self _installStaticException: handlerBlock class: nil
      category: aCategory number: aNumber subtype: atype
%

category: 'Instance creation'
classmethod: AbstractException
new
  "Return a newly created object initialized to a standard initial state"

  ^ self _basicNew initialize
%

category: 'Legacy Handlers'
classmethod: AbstractException
removeActivationException: aGsExceptionHandler

"Search the current GemStone Smalltalk call stack for a method or block context
 that specified handler installed, and remove it.  The stack is searched by
 starting with the top method or block context and moving down.  Generates an
 error if aGsExceptionHandler is not installed anywhere in the current GemStone
 Smalltalk call stack or if aGsExceptionHandler is not an instance of
 GsExceptionHandler .

 Does not search the static legacy handlers list."

^ GsExceptionHandler removeActivationException: aGsExceptionHandler

%

category: 'Default handlers'
classmethod: AbstractException
removeAllDefaultHandlers

"Removes all ANSI default handlers and static legacy handlers.
 A faster implementation of
  AbstractException _staticExceptions do:[:h | h remove ].
"
<primitive: 380>
^ self _primitiveFailed: #removeAllDefaultHandlers
%

category: 'Legacy Handlers'
classmethod: AbstractException
removeStaticException: aGsExceptionHandler

"Remove a static exception handler (either an ANSI default handler,
 or a static legacy handler.)
 Returns the removed handler, or nil if not found by identity."

<primitive: 379>
aGsExceptionHandler _validateClass: GsExceptionHandler .
^ self _primitiveFailed: #removeStaticException args: { aGsExceptionHandler }
%

category: 'Instance creation'
classmethod: AbstractException
signal

 "An exception of the type associated with the receiver is signaled."

  ^ self new _signal
%

category: 'Instance creation'
classmethod: AbstractException
signal: signalText

 "An exception of the type associated with the receiver is signaled."
  ^ self new signal: signalText.
%

category: 'Instance creation'
classmethod: AbstractException
signalNotTrappable
  ^ self new signalNotTrappable
%

category: 'Instance creation'
classmethod: AbstractException
signalNotTrappable: signalText
  ^ self new details: signalText ;  signalNotTrappable
%

category: 'Instance creation'
classmethod: AbstractException
signalToGci

  ^ self new signalToGci
%

category: 'SUnit'
classmethod: AbstractException
sunitSignalWith: aString
        ^self signal: aString
%

category: 'Private'
classmethod: AbstractException
_installStaticException: handlerBlock
  class: exceptionClass
  category: aCategory number: aNumber subtype: atype

"Install the specified exception block at the head of the static handlers
 list, not associated with a particular stack frame ,
 to field exceptions of the specified class .
 or errors of the specified category, number, and subtype.
 If exceptionClass is not nil   ignores  aCategory,  aNumber, atype  ;
 else assumes aCategory to be GemstoneError .

 The handler is represented as an instance of GsExceptionHandler,
 and will be a part of the session state.
 The new  GsExceptionHandler is returned ."
<primitive: 378>
| nBlockArgs |
handlerBlock _validateClass: ExecBlock .
exceptionClass ifNotNil:[
  nBlockArgs := 1 .
  exceptionClass _validateClass: AbstractException .
] ifNil:[
  nBlockArgs := 4 .
  " aCategory ifNotNil:[ aCategory _validateClass: SymbolDictionary ] . "
  aNumber ifNotNil:[ aNumber _validateClass: SmallInteger ] .
].
handlerBlock argumentCount ~~ nBlockArgs ifTrue: [
  ^ self _error: #rtErrExceptBlockNumArgs args: { nBlockArgs . handlerBlock argumentCount }.
].
^ self _primitiveFailed: #_installStaticException:class:category:number:subtype:
       args: { handlerBlock . exceptionClass . aCategory . aNumber . atype }
%

category: 'Private Legacy'
classmethod: AbstractException
_new: anInteger args: anArray

"return a new exception for the specified error number."

<primitive: 140>
  "If args ok and handler found, this stack frame reused to invoke handler.
   If exception handling succeeds and execution is to resume,
   we return from this method with the resumption value."
anInteger _validateClass: SmallInteger.
anInteger < 1 ifTrue: [
  ^ self _error: #rtErrBadErr args: { anInteger . anArray  }
].
anArray _validateClass: Array.
anArray size > 10 ifTrue: [
 ^ self _error: #rtErrTooManyErrArgs args: { anArray size }
].
self _primitiveFailed: #_new:args: args: { anArray }.
self _uncontinuableError
%

category: 'Private'
classmethod: AbstractException
_staticExceptions

"Returns the head of the static exception list. The list includes
 static legacy handlers, and ANSI default handlers.  Elements of the
 list are instances of GsExceptionHandler."

<primitive: 377>
^ self _primitiveFailed: #_staticExceptions
%

!		Instance methods for 'AbstractException'

category: 'Accessing'
method: AbstractException
addText: aString
  messageText ifNil:[ self asString "build the default messageText" ].
  messageText := (messageText ifNil:[ String new ]), ', ', aString .
%

category: 'Accessing'
method: AbstractException
arg: anObject
  gsArgs ifNotNil:[:a | a _isArray ifTrue:[ a add: anObject ]]
       ifNil:[ gsArgs := { anObject } ]
%

category: 'Accessing'
method: AbstractException
args: anArray
  gsArgs :=  anArray ifNotNil:[:a| a _isArray ifTrue:[ a ]
                                           ifFalse:[ { a } ]]
%

category: 'Compatibility'
method: AbstractException
category
  "Will be deprecated"
  ^ GemStoneError
%

category: 'Handling'
method: AbstractException
defaultAction

"Return an error to the controlling GCI client. Stack is saved
 and available as an argument to the GCI error struct.

 Instance variable of the receiver are enumerated into the GciErrSType.args
 as follows.
 If there are named instVars after AbstractException.args
 the values of those instVars are stored into GciErrSType.args .
 AbstractException.args is then added if space is available.

 If AbstractException.args is the last named instVar and it is an Array
 of size <= GCI_MAX_ERR_ARGS , it is enumerated into GciErrSType.args.
 otherwise the oop of the Array is stored into GciErrSType.args[0] .

 See also ClientForwarderSend >> defaultAction "

 ^ self _signalToDebugger .
%

category: 'Accessing'
method: AbstractException
describe
  "used by topaz and VM's error message building"

  ^ self asString
%

category: 'Accessing'
method: AbstractException
description
  "Returns a String."
 | res |
 gsDetails ifNotNil:[ :dt |
   res := [
     | str |
     (str := String withAll: self class name)
         add: ': ' ; add: dt asString  .
     str
   ] onException: Error do:[:ex |
     ex return: nil "ignore"
   ]
 ].
 ^ res ifNil:[ self asString ]
%

category: 'Accessing'
method: AbstractException
details
  ^ gsDetails ifNil:[ messageText ]
%

category: 'Accessing'
method: AbstractException
details: aStringOrArray
  "If arg is a String it will be appended to the fully formed error
   message.

   If arg is an Array and the receiver does not reimplement buildMessageText ,
   the arg will be used as a legacy style error message template."

  gsDetails := aStringOrArray  "store ANSI messageText"
%

category: 'Accessing'
method: AbstractException
genericDescription
  ^  self asString
%

category: 'Compatibility'
method: AbstractException
gsArguments
  "Will be deprecated"
  "Returns an Array"

  ^ self _legacyHandlerArgs
%

category: 'Compatibility'
method: AbstractException
gsCategory
  "Will be deprecated"
  ^ GemStoneError
%

category: 'Compatibility'
method: AbstractException
gsNumber
  "Will be deprecated"
  ^ gsNumber
%

category: 'Instance initialization'
method: AbstractException
initialize

 "subclasses may override but must send   super initialize
  at start of the reimplementation, or else initialize
  the 3 instVars   gsNumber, gsResumable, gsTrappable . "

  gsNumber := ERR_AbstractException.
  gsResumable := true .
  gsTrappable := true .
%

category: 'Handling'
method: AbstractException
isNested
 "When sent within an ANSI handler's handlerBlock ,
  returns true if there is an enclosing handler below the  on:do:
  of the active handler that could handle the receiver.

  Returns false if sent within a legacy handler block.
  "

 <primitive: 700>
 self _uncontinuableError.
%

category: 'Accessing'
method: AbstractException
isResumable
  ^ gsResumable .
%

category: 'Accessing'
method: AbstractException
isTrappable
  ^ gsTrappable == true or:[ gsTrappable == 1]
%

category: 'Accessing'
method: AbstractException
messageText: aString
  gsDetails := aString   "store ANSI messageText"
%

category: 'Legacy Accessing'
method: AbstractException
next

  "No longer supported.
   Within a legacyHandler block, to get the next legacyHandler after
   the one currently active  use
       anException legacyHandler next
   where anException is the first arg to the legacyHandler block. "

  self shouldNotImplement: #next .
  self _uncontinuableError
%

category: 'Private'
method: AbstractException
number
  ^ gsNumber
%

category: 'Handling'
method: AbstractException
outer
  "When sent within an ANSI handler's handlerBlock ,
   search for an enclosing handler below the active one that will
   handle the receiver.

   If an enclosing handler found,  send #value: to
   the enclosing handler's handlerBlock , with receiver as the argument.
   If the receiver is 'resumable' and the enclosing handler's handlerBlock
   sends #resume: to the receiver,  the resumption value
   will be returned from this message.
   If the receiver is not 'resumable' or if the enclosing handlerBlock
   does not send #resume: , then this message will not return.

   If no such handler found,  returns the result of
   sending #defaultAction to the receiver.

   For exceptions not 'resumable', #outer is equivalent to #pass ."

   ^ self _outer: nil with: nil .
%

category: 'Handling'
method: AbstractException
pass
  "When sent within an ANSI handler's handlerBlock ,
   search for an enclosing handler below the active one that will
   handle the receiver.

   If an enclosing handler found, push a new stack frame and send #value: to
   the enclosing handler's handlerBlock , with receiver as the argument.

   If no such handler found, send #defaultAction to the receiver.
   Control does not return to the currently active handler."

   "Following resume: only happens if the _pass:with: invoked a default handler
    which returned normally. "
   self resume:( self _pass: nil with: nil ) .
   self _uncontinuableError. "should not reach here"
%

category: 'Accessing'
method: AbstractException
reason
  ^ gsReason
%

category: 'Accessing'
method: AbstractException
reason: aString
  "aString must be a Symbol or String
   Only the first 63 bytes of aString are used. "
  | str |
  aString _isSymbol ifTrue:[
    gsReason := aString asString .
  ] ifFalse:[
    (str := aString) _isOneByteString ifFalse:[
       str := aString asString .
       str _isOneByteString ifFalse:[
         str := aString class name  .
         str _isOneByteString ifFalse:[ ^ self "ignore arg" ].
       ].
    ].
    (str size > 64) ifTrue:[
       gsReason := aString copyFrom: 1 to: 63
    ] ifFalse:[
       gsReason := aString
    ].
  ].
%

category: 'Legacy Handlers'
method: AbstractException
remove

"To be sent only within a legacy handler's handlerBlock.

 Remove the receiver's  GsExceptionHandler from
 the GemStone Smalltalk call stack .

 Does not search the static legacy handlers list.
"
  currGsHandler ifNotNil:[ :h | h remove ]
                ifNil:[ self _uncontinuableError ]
%

category: 'Legacy Handlers'
method: AbstractException
resignal: anErrorDict number: anInteger args: anArray

  "When sent within a legacy handler's handlerBlock,
   create a new instance of Exception as specified by the arguments,
   and search for the next handler below this handler.

   If the newly created Exception is handled successfully,
   a normal return of that handler's handlerBlock will
   result in a return from this method.

   anErrorDict is ignored, and may be nil ."

  | ex |
  currGsHandler ifNil:[
     self _uncontinuableError. "illegal in ANSI handler"
  ].
  ex := AbstractException _new: anInteger args: anArray .
  gsResumable ifFalse:[ ex _resumable: false ].
  ^ self _legacyResignal: ex
%

category: 'Handling'
method: AbstractException
resignalAs: replacementException

  "When sent within an ANSI handler's handlerBlock,
   the stack is trimmed back to the point where the receiver was
   orginally signaled, executing any ensure: blocks installed by
   handlerBlock execution .

   Then replacementException is substituted for the originally
   signaled exception and #signal  is sent to it to restart
   exception handling.

   Generates an error if the stack trim would cross the frame
   of a C primitive or user action."

  gsResumable ifFalse:[ replacementException _resumable: false ].
  self _executeEnsuresBelow: 0 .
  self _resignalAs: replacementException  . "does not return"
  self _uncontinuableError.  "should not be here"
%

category: 'Handling'
method: AbstractException
resume

"See  resume:  for documentation ."

 self isResumable ifFalse:[
   "cannot resume from a not-resumable Exception."
   ^ self error:'cannot resume from a not-resumable Exception'
 ].
 ^ self _resume: nil
%

category: 'Handling'
method: AbstractException
resume: resumptionValue

"To be sent from within an ANSI handler's handlerBlock only.

 If the current handler block was invoked by #outer from a previous
 handler block,  #resume will return to that previous handler block,
 with resumptionValue being returned from the send of #outer .
 Otherwise resume execution from the send that signaled the receiver
 with the specified value.

 An Error is signaled if this is send directly or indirectly from
 within a defaultAction method to the receiver of defaultAction."

 self isResumable ifFalse:[
   "cannot resume from a not-resumable Exception."
   ^ self error:'cannot resume from a not-resumable Exception' .
 ].
 ^ self _resume: resumptionValue
%

category: 'Handling'
method: AbstractException
retry
  "When sent within an ANSI handler's handlerBlock ,
   resume execution by sending  #value  to the receiver of the
   #on:do: send which installed the currently active handlerBlock ,
   and using the stack frame of that #on:do: .

   Any ensure: blocks between top of stack and that #on:do: will
   be executed and stack trimmed back before resuming execution.

  Generates an error if sent within a legacy handler's handlerBlock,
  or if stack trim would cross the frame of a C primitive or user action."

^self retryUsing: nil
%

category: 'Handling'
method: AbstractException
retryUsing: alternativeBlock

  "When sent within an ANSI handler's handlerBlock ,
   resume execution in the frame of the #on:do: send which
   installed the currently active handlerBlock.

   Any ensure: blocks between top of stack and that #on:do: will
   be executed and stack trimmed back. If alternativeBlock  is non-nil
   it is substituted for the original receiver of the #on:do: .
   Then #value is sent to the non-nil alternativeBlock, else to the
   original receiver of #on:do: .

   Generates an error if sent within a legacy handler's handlerBlock,
   or if stack trim would cross the frame of a C primitive or user action."

  self _executeEnsuresBelow: 1 .
  self _retryUsing: alternativeBlock
%

category: 'Handling'
method: AbstractException
return

"When sent within an ANSI handler's handlerBlock,
 resume execution by returning nil from the #on:do: send
 which installed the currently active handlerBlock .

 Any ensure: blocks between top of stack and that #on:do:
 will be executed and the stack trimmed back.

 Generates an error if sent within a legacy handler's handlerBlock,
 or if the return would cross the frame of a C primitive or user action.
"
 self _executeEnsuresBelow: 1 .
 self _return: nil .
%

category: 'Handling'
method: AbstractException
return: returnValue

"When sent within an ANSI handler's handlerBlock,
 resume execution by returning returnValue from the #on:do: send
 which installed the currently active handlerBlock .

 Any ensure: blocks between top of stack and that #on:do:
 will be executed and the stack trimmed back.

 If return would cross a C extension , the _executeEnsuresBelow:
 will execute ensure blocks below the C frame, and
 return: will trim frames below the C frame.

 Generates an error if sent within a legacy handler's handlerBlock.
 or if the return would cross the frame of a C primitive or user action.
"
 self _executeEnsuresBelow: 1 .
 self _return: returnValue .
%

category: 'Copying'
method: AbstractException
shallowCopy
  "clears any exception handling information in the copy. Do this
   in shallowCopy  in case a subclass fails to send postCopy.  "
  | res |
  ( res := super shallowCopy ) size: 0 .
  ^ res
%

category: 'Signaling'
method: AbstractException
signal

  "The current stack is searched for an 'exception handler' which matches
  the receiver.  The search proceeds from the top of stack downwards,
  and then checks the static legacy handlers.

  ANSI handlers are installed by using
    ExecBlock>>on:do:    installs an ANSI handler
    ExecBlock>>onException:do:    installs an ANSI handler
  ANSI static handlers may be installed by
    AbstractException(C)>>defaultAction:
  Legacy handlers may be installed by
    AbstractException(C)>>category:number:do:
  Legacy static handlers may be installed by
    AbstractException(C)>>installStaticException:category:number:

  A matching ANSI handler is defined to be one which would return 'true'
  if the message #handles: was sent to its 'exception selector'
  with the 'signaled exception' as the argument.

  If a matching handler is found, its handlerBlock is executed.

    For a legacy handler or ANSI default handler, if the handlerBlock
    returns normally, execution resumes from the point of the signal ,
    with the value returned by the handlerBlock.

    For a handlerBlock installed by on:do: , if the handlerBlock returns normally,
    any ensure: blocks are executed, and execution returns from the send of on:do:
    with the value returned by the handlerBlock .  However if the return from
    the send of on:do: would cross the frame of a C primitive or user action,
    an uncontinuable error is generated.
    Within the handlerBlock, the methods
      outer, pass , resignalAs:, resume, return ,
    can be send to this instance of AbstractException  to alter flow .

  If a matching handler is not found, 'default action' for the receiver
  is performed, as if #defaultAction were sent to the receiver.
  If the receiver is 'resumable' the value returned from the
  #defaultAction method is returned as the value of the #signal message,
  otherwise an error is returned to the GCI client.

  If (System gemConfigurationAt:#GemExceptionSignalCapturesStack) == true ,
  and gsStack==nil when primitive 2022 (AbstractException>>_signal) is invoked,
  then primitive 2022 will fill in gsStack with an Array.
  See AbstractException>>stackReport. "

  ^ self _signal
%

category: 'Signaling'
method: AbstractException
signal: signalText

  "The message text of the receiver is set to signalText,
   and the receiver is signaled.  See also  #signal ."

  gsDetails := signalText .  "store ANSI messageText"
  ^ self _signal
%

category: 'Signaling'
method: AbstractException
signalNotTrappable

  "Receiver is signaled and it will not be trappable by an exception handler.  
   The receiver will send #defaultAction ; the stack will not be searched for on:do: 
   that would otherwise handle the receiver."

  gsTrappable := false .
  ^ self _signal
%

category: 'Signaling'
method: AbstractException
signalToGci

  "Receiver is signaled and it will not be trappable by any exception handler.  
   An Error will be return to the GCI. "

  gsTrappable := 0 .
  ^ self _signal
%

category: 'Accessing'
method: AbstractException
stackReport

"Returns a formatted String derived from gsStack instVar, or nil .

 If (System gemConfigurationAt:#GemExceptionSignalCapturesStack) == true ,
 and gsStack==nil when primitive 2022 (AbstractException>>_signal) is invoked,
 then primitive 2022 will fill in gsStack with an Array .
"
| arr stackKind |
arr := gsStack .
(arr _isArray and:[ (stackKind := arr atOrNil:1 ) _isSmallInteger ]) ifTrue:[
  | report lf nativeStk |
  nativeStk := stackKind >= 2"STK_native" .
  report := String new .  lf := Character lf .
  2 to: arr size by: 3 do:[:j | | meth ip stepPoint rcvr |
    (ip := arr atOrNil: j + 1 ) ifNotNil:[
      meth :=  arr at: j .
      rcvr := arr at: j + 2 .
      ip < 0 ifTrue:[ ip := 0 ].
      report add: (meth _descrForStackPadTo: 0 rcvr: rcvr)  .
      nativeStk ifTrue:[ ip := meth _nativeIpOffsetToPortable: ip asReturn: false].
      stepPoint := meth _stepPointForIp: ip level: 2 useNext: nativeStk .
      report add:' @' ; add: stepPoint asString ;
          add: ' line ';
          add: (meth _lineNumberForStep: stepPoint) asString ;
          add:'  [GsNMethod '; add: meth asOop asString ; add:']';  add: lf .
    ].
  ].
  ^ report
].
^ nil
%

category: 'Legacy Accessing'
method: AbstractException
subtype

   "No longer supported."

   ^ nil
%

category: 'SUnit'
method: AbstractException
sunitExitWith: aValue
        ^self return: aValue
%

category: 'SUnit'
method: AbstractException
sunitSignalWith: aString
        ^self signal: aString
%

category: 'Accessing'
method: AbstractException
tag
  ^ tag ifNil:[ gsDetails ]
%

category: 'Accessing'
method: AbstractException
tag: anObject
  tag := anObject
%

category: 'Private'
method: AbstractException
_debugException: fakeArg

"If a debugger block has been installed with AbstractException(C)>>installDebugBlock:
 execute that block.  Primitive succeeds and returns receiver if no
 debugger block is installed.
 Returns receiver"

<primitive: 799>
"if we get here, a debugger block was found and the argument fakeArg has
 been changed to be that block."

 fakeArg value: self .
 ^ self
%

category: 'Private'
method: AbstractException
_defaultAction
  | res |
  gsTrappable ~~ 0 ifTrue:[
    ^ self defaultAction .
  ].
  res := self _signalGciError .
  ^ res == self ifTrue:[ nil ] ifFalse:[ res ]
%

category: 'Private'
method: AbstractException
_enableEvents

"perform the enable interrupts part of _gsReturnNothingEnableEvents"

<primitive: 141>
self _uncontinuableError
%

category: 'Private'
method: AbstractException
_executeEnsuresBelow: kind
 | ensBlks |
 ensBlks := self _getEnsuresBelow: kind .
 1 to: ensBlks size by: 2 do:[:j|
   self _removeEnsureAtFP: (ensBlks at: j) .
   (ensBlks at: j + 1) value .
 ].
%

category: 'Private'
method: AbstractException
_executeGsHandler: aGsExceptionHandler
  "Execute aGsExceptionHandler's block .  Sent from VM C code only.
   Used for static ANSI handlers and legacy handlers."
  | res err |
  self _debugException: nil .
  currGsHandler := aGsExceptionHandler .
  aGsExceptionHandler _ansiBlock ifNotNil:[ :blk |
    "static ansi handler, normal return from handler disallowed"
    res := blk argumentCount == 0
           ifTrue:[ blk value  ]
           ifFalse:[ blk value: self ].
    (err := UncontinuableError new)
       details:'an ANSI defaultHandler requires explicit resume:'.
    (self isKindOf: UncontinuableError)
       ifTrue:[ err signalNotTrappable  "avoid infinite recursion"]
      ifFalse:[ err signal ].
  ] ifNil:[  "legacy handler , if handler block returns normally
              resume program execution from the point of the signal."
    res := aGsExceptionHandler block value: self value: GemStoneError
		value: gsNumber value: self _legacyHandlerArgs .
  ].
  self isResumable ifFalse:[ | handNum |
    (err := Error new)
       details: 'cannot resume from a not-resumable Exception' .
     handNum := aGsExceptionHandler number .
     (handNum == nil or:[ handNum == err number])
        ifTrue:[ err signalNotTrappable  "avoid infinite recursion" ]
       ifFalse:[ err signal ].
  ].
  currGsHandler := nil .   "clear to allow use of resume:"
  self _resume: res
%

category: 'Private'
method: AbstractException
_executeHandler: aBlock

  "Execute aBlock , which was the second arg of an #on:do: ,
   to handle the receiver.  Sent from VM C code only .
   aBlock must be a block taking zero or one arguments."
  <primitive: 2025> "mark frame with executeHandler_Mark_NIL, always fails"
  | res |
  self _debugException: nil .
  res := aBlock argumentCount == 0
           ifTrue:[ aBlock value  ]
           ifFalse:[ aBlock value: self ].

  "If we get here, then handler block did not send a #resume,
   so continue program execution by returning from the #on:do: send
   which installed the handlerBlock we just executed."

  self return: res .
%

category: 'Private'
method: AbstractException
_executeOuterGsHandler: aGsExceptionHandler
  "Execute aGsExceptionHandler's block .  Sent from image only"
  self _debugException: nil .
  currGsHandler := aGsExceptionHandler .
  aGsExceptionHandler _ansiBlock ifNotNil:[ :blk |
    "static ansi handler, normal return returns to sender of outer "
    ^ blk value: self .
  ] ifNil:[ | res |  "have a legacy handler"
    res := aGsExceptionHandler block value: self value: GemStoneError
                value: gsNumber value: self _legacyHandlerArgs .
    "if legacy handler returns normally, do a resume:"
    self isResumable ifFalse:[ | handNum err |
      (err := Error new)
         details: 'cannot resume from a not-resumable Exception' .
       handNum := aGsExceptionHandler number .
       (handNum == nil or:[ handNum == err number])
          ifTrue:[ err signalNotTrappable  "avoid infinite recursion" ]
         ifFalse:[ err signal ].
    ].
    currGsHandler := nil . "clear to allow use of resume:"
    self _resume: res
  ].
%

category: 'Private'
method: AbstractException
_executeOuterHandler: aBlock
  "Execute aBlock , which was the second arg of an #on:do: ,
   to handle the receiver.   Sent from implementation of #outer only ."

  <primitive: 2025> "mark frame with executeHandler_Mark_NIL, always fails"
  self _debugException: nil .

  ^ aBlock argumentCount == 0
           ifTrue:[ aBlock value  ]
           ifFalse:[ aBlock value: self ].
%

category: 'Private'
method: AbstractException
_getEnsuresBelow: kind

 "kind determines specified frame ,
     kind 0 -->  frame where receiver originally signaled

     kind 1 -->  #on:do: frame for currently active handler,
         and also clears mark_nil from _executeHandler frames
         while searching for the ensure blocks. Stops the
         search at a C FFI or UA  to Smalltalk boundary.

  Returns an Array containing pairs of
    FP offset,  block that was arg to an ensure:
  for all #ensure:  frames  between top of stack and and specified frame

  Does not remove the ensure block from their frames.
  Returns nil if there are none."

 <primitive: 695>
 self _uncontinuableError
%

category: 'Accessing'
method: AbstractException
_gsStack

  " If (System gemConfigurationAt:#GemExceptionSignalCapturesStack) == true ,
   and gsStack==nil when primitive 2022 (AbstractException>>_signal) is invoked,
   then primitive 2022 will fill in gsStack with an Array .
   The Array contains a Boolean (inNativeCode),
   followed by triples of  aGsNMethod , an ipOffset, a receiver
   per the result of GsProcess>>_frameContentsAt: ,
   terminated by the end of the array, or a nil."

  ^ gsStack
%

category: 'Private'
method: AbstractException
_gsStack: anArray
  gsStack := anArray
%

category: 'Private'
method: AbstractException
_handleInCextension

"Private.
 Caller responsible for executing ensure blocks.
 Unwinds Smalltalk stack to the most recent C extension and
 returns to the C code with receiver as the signaled exception."

<primitive: 556>  "primitive fails unless this is Maglev VM"
"should never reach here"
self _primitiveFailed: #_handleInCextension.
self _uncontinuableError .
%

category: 'Private'
method: AbstractException
_handlerActive
  "an exception currently being handled by an on:do: will have
     FP information appended."

  ^ self size ~~ 0
%

category: 'Accessing'
method: AbstractException
_legacyDetails: templateString
  | str theArgs elem |
  theArgs := self _legacyHandlerArgs .
  str := String new .
  1 to: templateString size do:[:j |
    elem := templateString at: j .
    elem _isOneByteString ifTrue:[
      str add: elem
    ] ifFalse:[
      elem _isSmallInteger ifTrue:[
	elem < 0 ifTrue:[ str add: (theArgs atOrNil: 0 - elem) asOop asString ]
	      ifFalse:[ str add: (theArgs atOrNil: elem) describe1K ]
      ]
    ]
  ].
  ^ str
%

category: 'Private'
method: AbstractException
_legacyHandlerArgs
  "Must be reimplemented in subclasses of AbstractException
   which have named instVar beyond 'args' "

  ^ gsArgs ifNil:[ #() ]
%

category: 'Private Legacy'
method: AbstractException
_legacyResignal: newException

"Search for a handler for newException below this handler.
 Current handler should be a legacy handler .

 If the newException is handled successfully,
 a normal return of the handlerBlock found by this resignal will
 result in a return from this method."

<primitive: 2024>  "succeeds or throws error to GCI"
self _uncontinuableError.
%

category: 'Instance initialization'
method: AbstractException
_number: aSmallInt
  gsNumber := aSmallInt
%

category: 'Private'
method: AbstractException
_outer: aHandler with: handlerKind
  "Search for an enclosing handler below the active one that will
   handle the receiver.
   Caller must always pass  nil for aHandler and isAnsi.
   resume:  from within the outer handler will return from this frame."

 <primitive: 697>
 "primitive always fails.
  if handler found, it was substituted in frame for aHandler ,
  and handlerKind was set to a IntHandlerBlockType "

  aHandler ifNotNil:[   "handler found"
    handlerKind == 0 ifTrue: [ "2nd arg to on:do:"
      self return:( self _executeOuterHandler: aHandler ) 
    ] ifFalse:[ 
      "Static ANSI or legacy handler"
      self return:( self _executeOuterGsHandler: aHandler ) 
    ]
  ] ifNil:[ "no enclosing handler found"
    ^ self defaultAction .
  ].
  self _uncontinuableError. "should not reach here"
%

category: 'Private'
method: AbstractException
_pass: aHandler with: handlerKind
  "Search for an enclosing handler below the active one that will
   handle the receiver.
   Caller must always pass  nil for aHandler and isNonStaticAnsi .
   resume:  from within the outer handler will return from this frame."
 <primitive: 697>
 "primitive always fails.
  If handler found, it was substituted in frame for aHandler ,
  and handlerKind was set to a IntHandlerBlockType "

  aHandler ifNotNil:[   "handler found"
    handlerKind == 0 ifTrue: [ "2nd arg to on:do:"
      self return:( self _executeOuterHandler: aHandler ) 
    ] ifFalse:[
     "Static ANSI or legacy handler"
      self return:( self _executeOuterGsHandler: aHandler )
    ]
  ] ifNil:[ "no enclosing handler found"
    self _executeEnsuresBelow: 0 . "ensures down to original signal"
    self _passToDefaultHandler .
    self _uncontinuableError.  "should not be here"
  ].
%

category: 'Private'
method: AbstractException
_passToDefaultHandler

  "Caller has executed ensure: blocks between top of stack and
   point where receiver was originally signalled.
   This primitive will trim stack back to that signal frame and
   restart execution of AbstractException>>signal such that
   the search for a handler will always fail, thus executing
   the defaultAction."

<primitive: 2009>
self _uncontinuableError . "should never reach here"
%

category: 'Private'
method: AbstractException
_removeEnsureAtFP: anOffset

 "Dereference the ensure block in specified frame,
  so it won't be executed more than once.
  Frame changed  from  ensure:[] to  ensure: nil ."

 <primitive: 696>
 self _uncontinuableError
%

category: 'Private'
method: AbstractException
_resignalAs: replacementException

  "Caller has executed ensure: blocks between top of stack and
   point where receiver was originally signalled.
   This primitive will trim stack back to that signal frame.

   Then replacementException is substituted for the receiver
   and execution of the signal method is restarted."

 <primitive: 2023>
 replacementException _validateClass: AbstractException .
  self _uncontinuableError
%

category: 'Instance initialization'
method: AbstractException
_resumable: aBoolean
  gsResumable := aBoolean
%

category: 'Handling'
method: AbstractException
_resume: resumptionValue

"To be sent from within an ANSI handler's handlerBlock only.

 If the current handler block was invoked by #outer from a previous
 handler block,  #resume will return to that previous handler block,
 with resumptionValue being returned from the send of #outer .
 Otherwise resume execution from the send that signaled the receiver
 with the specified value.

 resumptionValue==nil means resume with current stack value, i.e. AbstractException>>resume

 calling methods in image responsible for checking isResumable."

 <primitive: 2021>
 self _uncontinuableError .  "should not be here"
%

category: 'Private'
method: AbstractException
_retryUsing: alternativeBlock

 "When sent within an ANSI handler's handlerBlock ,
  resume execution in the frame of the #on:do: send which
  installed the currently active handlerBlock.

  caller has executed ensure: blocks between top of stack and that #on:do: .
  this primitive will trim stack back to that #on:do: .

  If alternativeBlock  is non-nil , it must be a zero arg
  ExecBlock and it is substituted for the original receiver of the #on:do: .
  Then #value is sent to the non-nil alternativeBlock, else to the
  original receiver of #on:do: .

  Generates an error if sent within a legacy handler's handlerBlock."

  <primitive: 2019>  "does not return"
  self _uncontinuableError
%

category: 'Private'
method: AbstractException
_return: returnValue
 "When sent within an ANSI handler's handlerBlock,
  resume execution by returning returnValue from the #on:do: send
  which installed the currently active handlerBlock .

  Caller has executed ensure: blocks between top of stack and that #on:do: .
  This primitive will trim stack back to that #on:do: .
  Generates an error if sent within a legacy handler's handlerBlock.

  Will return into a C extension exception handling C code, or
  to from the currently active handlerBlock, which ever is closer
  to the top of the stack.
"
<primitive: 2020> "does not return"
self _uncontinuableError
%

category: 'Private'
method: AbstractException
_signal
  <primitive: 2022>

   "The current stack is searched for an 'exception handler' per
    AbstractException>>signal .
    If a handler found, new frame pushed
      to execute  _executeHandler:   or   _executeGsHandler:  .
      and the primitive does not return.
    If exception handling succeeds and execution is to resume,
    either the resume: or the return: primitive will do the 'goto'
    and we don't actually return from this frame .

    inCextensionArg must be nil at entry, and may be modified
    by primitive.

    If a Smalltalk handler not found,
    primitive fails so we can defer to a C extension or send defaultAction here."

  | res |
  res := self _defaultAction .
    "if you change code from beginning of method to here , may need to
     alter C code at  IntSwiReplaceTos_IPOFFSET"

  1 timesRepeat:[ self class ]. "loop to detect/handle termination interrupt"
  self isResumable ifTrue:[
    ^ res .
  ].
  self _signalToDebugger  .
  self _uncontinuableError . "should never reach here"
%

category: 'Private'
method: AbstractException
_signalAsync: saveProtectedMode

  "To be sent only by the virtual machine.
   The receiver is an asynchronous Exception, usually a ControlInterrupt.
   Examples are  such as soft-break, (ErrorSymbols at: #rtErrSoftBreak, error 6003)
   or signalAbort (ErrorSymbols at: #rtErrSignalAbort , error 6009) .

   Handling of an asynchronous Exception happens between bytecodes and
   if successful does not alter the stack.  This method is always
   executed for an asynchronous Exception, regardless of whether a
   handler exists. "

  | ok |
  gsTrappable == true ifTrue:[
    gsTrappable := 1  "prevent onSynchronous:do: from handling this".
  ].
  ok := false .
  [
    self _signal . "if handler not found  synchronous error thrown to GCI"
    ok := true .  "if handled ok we get to here"
  ] ensure:[
    ok ifFalse:[
      self _enableEvents . "reenable interrupts if resume not successful"
    ]
  ].
  self _gsReturnNothingEnableEvents "special selector optimized by compiler"
%

category: 'Private'
method: AbstractException
_signalFromPrimitive

  "Sent from within the VM only.
   An exception is being signalled from within a C primitive.
   The signalled exception is not resumable .
   otherwise same behavior as AbstractException>>_signal . "

  <primitive: 2022>
  | res num |
  res := self _defaultAction .
    "if you change code from beginning of method to here , may need to
     alter C code at  IntSwiReplaceTos_IPOFFSET"

  "allow breakpoints to be resumable from Smalltalk."
  ( #( 6002 6005 6006 6023 6024 6025) includesIdentical: num) ifTrue:[
       self isResumable ifTrue:[
         ^ res
     ]
  ].
  1 timesRepeat:[ self class ]. "loop to detect/handle termination interrupt"
  self _signalToDebugger .
  2 timesRepeat:[ self class ]. "loop to detect/handle termination interrupt"
  self _uncontinuableError . "should never reach here"
%

category: 'Handling'
method: AbstractException
_signalGciError

"Return an error to the controlling GCI client. Stack is saved
 and available as an argument to the GCI error struct."

<primitive: 33>  "this frame may be removed from stack before saving stack."
self _uncontinuableError
%

category: 'Private'
method: AbstractException
_signalTimeout: saveProtectedMode

  "To be sent only by the virtual machine.
   The receiver is an asynchronous Exception, error 6015 "

  | ok |
  ok := false .
  [
    gsNumber == ERR_TIMEOUT_INTERRUPT ifTrue:[
      ProcessorScheduler scheduler _yieldForTimeout .
      ok := true .  "if handled ok we get to here"
    ] ifFalse:[
      self _signal .
    ].
  ] ensure:[
    ok ifFalse:[
      self _enableEvents . "reenable interrupts if resume not successful"
    ]
  ].
  self _gsReturnNothingEnableEvents "special selector optimized by compiler"
%

category: 'Private'
method: AbstractException
_signalToDebugger
  | res |
  gsTrappable ~~ 0 ifTrue:[
    GsProcess _currentOrNil ifNotNil:[:proc |
      proc debugActionBlock ifNotNil:[:blk | ^ blk cull: self ].
    ].
  ].
  res := self _signalGciError .
  ^ res == self ifTrue:[ nil ] ifFalse:[ res ]
%

! Class extensions for 'AbstractFraction'

!		Class methods for 'AbstractFraction'

removeallmethods AbstractFraction
removeallclassmethods AbstractFraction

category: 'Instance Creation'
classmethod: AbstractFraction
fromStream: aStream

"Returns a Fraction from the stream.  The stream must contain two Integers
 separated by a slash.  (There may be blanks around the slash.)  Generates an
 error if the stream contains anything else, or if an attempt is made to read
 beyond the end of the stream."

| n d |
self _checkReadStream: aStream forClass: CharacterCollection.
aStream peek unicodeIsWhitespace ifTrue:[
  [ aStream next unicodeIsWhitespace ] whileTrue.
  aStream skip: -1.
].
n := Integer _fromStream: aStream.
aStream peek unicodeIsWhitespace ifTrue:[
  [ aStream next unicodeIsWhitespace ] whileTrue.
  aStream skip: -1.
].
(aStream next == $/ ) ifFalse:[ ^ self _errIncorrectFormat: aStream ].
aStream peek unicodeIsWhitespace ifTrue:[
  [ aStream next unicodeIsWhitespace ] whileTrue.
  aStream skip: -1.
].
d := Integer _fromStream: aStream.
^ self numerator: n denominator: d
%

category: 'Instance Creation'
classmethod: AbstractFraction
numerator: n denominator: d
"Returns an instance of Fraction with numerator numInt and denominator
 denomInt.  If that Fraction can be reduced, this method returns the
 corresponding Integer instead.  The result is made invariant.

 If either argument (numerator or denominator) is not an Integer, that
 argument is truncated to the corresponding Integer."

 (self _newSmallFraction: n denom: d reduce: true) ifNotNil:[:res |
    ^ res
 ].
 ^ self _reduce: n denom: d
%

category: 'Private'
classmethod: AbstractFraction
_coerce: aNumber
	" Answer a fraction equal to aNumber (as long as aNumber is finite).
	Private, used for numeric comparisons."

	| f |
	f := aNumber asFraction.
	f _isInteger
		ifTrue: [ ^ AbstractFraction _noreduce_numerator: f denominator: 1 ].
	^ f
%

category: 'Private'
classmethod: AbstractFraction
_newFraction: n denominator: d

  "Returns an instance of Fraction , never reduced to a SmallFraction."

  ^ Fraction _basicNew _numerator: n denom: d
%

category: 'Private'
classmethod: AbstractFraction
_newSmallFraction: n denom: d reduce: reduceBoolean

"Returns nil or a reduced instance of SmallFraction ,
 or a Fraction containing reduced SmallInteger values for n and d.
 A SmallFraction is returned if, after moving overall sign to n
 and reducing n and d, n and d satisfy
                0 <  d <= 134217726
   and -536870912 <= n <= 536870911

 Signals an Error if d == 0 ."

<primitive: 996>
reduceBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_newSmallFraction:denom: args: { n . d }
%

category: 'Private'
classmethod: AbstractFraction
_noreduce_numerator: n denominator: d
  (self _newSmallFraction: n denom: d reduce: false) ifNotNil:[:res |
     ^ res
  ].
  ^ Fraction _basicNew _numerator: n denom: d
%

category: 'Private'
classmethod: AbstractFraction
_reduce: numer denom: denom
  | n d gcd |
  n := numer truncated .
  n == 0 ifTrue:[ ^ 0 ].
  d := denom truncated .
  (d == 0) ifTrue: [ ^ n _errorDivideByZero ].
  d < 0 ifTrue:[
    d := 0 - d .
    n := 0 - n .
  ].
  gcd := n gcd: d .
  n := n // gcd.
  d := d // gcd.
  (d == 1) ifTrue:[ ^ n ].
  (self _newSmallFraction: n denom: d reduce: false) ifNotNil:[:res |
     ^ res
  ].
  ^ Fraction _basicNew _numerator: n denom: d
%

!		Instance methods for 'AbstractFraction'

category: 'Arithmetic'
method: AbstractFraction
* aNumber

"Returns the result of multiplying the receiver by aNumber."

(aNumber isKindOf: AbstractFraction) ifTrue: [
  ^ AbstractFraction numerator: (self numerator * aNumber numerator)
           denominator: (self denominator * aNumber denominator)
] ifFalse: [
  (aNumber _isInteger and:[ aNumber = self denominator ]) ifTrue:[
     ^ self numerator
  ].
  ^ self _retry: #* coercing: aNumber
]
%

category: 'Arithmetic'
method: AbstractFraction
+ aFraction

"Returns the sum of the receiver and aFraction."

(aFraction isKindOf: AbstractFraction) ifTrue:[
  | myDenom otherDenom commonDenominator newNumerator
    gcd myDemon_div_gcd |
  (myDenom := self denominator) = (otherDenom := aFraction denominator) ifTrue: [
    | myNumer otherNumer |
    (myNumer := self numerator) = (otherNumer := aFraction numerator) ifTrue:[
      myDenom _bottomBit == 0 ifTrue:[
        ^ AbstractFraction numerator: myNumer denominator: (myDenom bitShift: -1)
      ]
    ].
    ^ AbstractFraction numerator: (myNumer + otherNumer ) denominator: myDenom
  ].
  gcd := myDenom gcd: otherDenom .
  myDemon_div_gcd := myDenom // gcd .
  commonDenominator := myDemon_div_gcd * otherDenom .
  newNumerator := (self numerator * (otherDenom // gcd ))
                 + (aFraction numerator *  myDemon_div_gcd) .
  ^ AbstractFraction numerator: newNumerator denominator: commonDenominator
] ifFalse: [
  ^self _retry: #+ coercing: aFraction
]
%

category: 'Arithmetic'
method: AbstractFraction
- aFraction

"Returns the difference between the receiver and aFraction."
(aFraction isKindOf: AbstractFraction) ifTrue: [
  ^ self + (Fraction _noreduce_numerator:  0 - aFraction numerator
		 denominator: aFraction denominator)
] ifFalse: [
  ^ self _retry: #- coercing: aFraction
]
%

category: 'Arithmetic'
method: AbstractFraction
/ aNumber

"Returns the result of dividing the receiver by aFraction."

aNumber _isInteger ifTrue:[
  aNumber = 0 ifTrue: [ ^ self _errorDivideByZero ].
  ^ AbstractFraction numerator: self numerator denominator: (self denominator * aNumber)
].
(aNumber isKindOf: AbstractFraction) ifTrue:[
  ^ self * aNumber _reciprocal
].
^ self _retry: #/ coercing: aNumber
%

category: 'Comparing'
method: AbstractFraction
< aNumber
	"Returns true if the receiver is less than aNumber; returns false otherwise."

	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].

	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].	"NaN"
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'<' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].	"NaN"
	(aNumber isKindOf: AbstractFraction)
		ifTrue: [ 
			| otherNum num |
			num := self numerator.
			otherNum := aNumber numerator.
			otherNum == 0
				ifTrue: [ ^ num < 0 ].
			^ num * aNumber denominator < (self denominator * otherNum) ].
	ak == 3
		ifTrue: [ ^ aNumber sign == 1 ].	"Inf"
	^ self < (self _coerce: aNumber)
%

category: 'Comparing'
method: AbstractFraction
<= aNumber
	"Returns true if the receiver is less than or equal to aNumber; returns false
 otherwise."

	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].

	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].	"NaN"
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'<=' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].	"NaN"
	(aNumber isKindOf: AbstractFraction)
		ifTrue: [ 
			| otherNum num |
			num := self numerator.
			otherNum := aNumber numerator.
			otherNum == 0
				ifTrue: [ ^ num <= 0 ].
			^ num * aNumber denominator <= (self denominator * otherNum) ].
	ak == 3
		ifTrue: [ ^ aNumber sign == 1 ].	"Inf"
	^ self <= (self _coerce: aNumber)
%

category: 'Comparing'
method: AbstractFraction
= aNumber
	"Returns true if the receiver is equal to aNumber; returns false otherwise."

	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ false ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'=' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].
	(aNumber isKindOf: AbstractFraction)
		ifTrue: [ 
			| otherNum num |
			otherNum := aNumber numerator.
			num := self numerator.
			otherNum == 0
				ifTrue: [ ^ num == 0 ].
			^ otherNum = num and: [ aNumber denominator = self denominator ] ].
	ak == 3
		ifTrue: [ ^ false ].	
	"If we get here, self and arg are both finite."
	^ self = (self _coerce: aNumber)
%

category: 'Comparing'
method: AbstractFraction
> aNumber
	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].

	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].	"NaN"
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'>' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].	"NaN"
	(aNumber isKindOf: AbstractFraction)
		ifTrue: [ 
			| otherNum num |
			num := self numerator.
			otherNum := aNumber numerator.
			otherNum == 0
				ifTrue: [ ^ num > 0 ].
			^ num * aNumber denominator > (self denominator * otherNum) ].
	ak == 3
		ifTrue: [ ^ aNumber sign == -1 ].	"Inf"
	^ self > (self _coerce: aNumber)
%

category: 'Comparing'
method: AbstractFraction
>= aNumber
	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].

	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].	"NaN"
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'>=' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].	"NaN"
	(aNumber isKindOf: AbstractFraction)
		ifTrue: [ 
			| otherNum num |
			num := self numerator.
			otherNum := aNumber numerator.
			otherNum == 0
				ifTrue: [ ^ num >= 0 ].
			^ num * aNumber denominator >= (self denominator * otherNum) ].
	ak == 3
		ifTrue: [ ^ aNumber sign == -1 ].	"Inf"
	^ self >= (self _coerce: aNumber)
%

category: 'Converting'
method: AbstractFraction
asDecimalFloat

"Returns an instance of DecimalFloat that has the value of the receiver."
| num denom |
(num := self numerator) ifNil:[ ^ PlusSignalingNaN ] .
(denom := self denominator) ifNil:[ ^ PlusSignalingNaN ] .
^ num asDecimalFloat / denom asDecimalFloat
%

category: 'Converting'
method: AbstractFraction
asFixedPoint: scale

^ FixedPoint numerator: self numerator denominator: self denominator scale: scale
%

category: 'Converting'
method: AbstractFraction
asFloat
	"Answer a Float that closely approximates the value of the receiver.
	This implementation will answer the closest floating point number to the receiver.
	In case of a tie, it will use the IEEE 754 round to nearest even mode.
	In case of overflow, it will answer +/- Float infinity.
	If numerator or denominator is nil, anwers NaN."

	| num den n d numeratorSize denominatorSize scalePower scaledValue scaledValueBits significantBits ulp halfULP significanceMask subSignificant scaledFloat |
	(num := self numerator) ifNil: [ ^ PlusSignalingNaN ].
	(den := self denominator) ifNil: [ ^ PlusSignalingNaN ].

	n := num abs.
	d := den.	"denominator is never negative"
	numeratorSize := n highBit.
	denominatorSize := d highBit.	

	"If both numerator and denominator can be represented exactly as floats,
	we can use float division and let it do the correct rounding."
	(numeratorSize <= 53 and: [ denominatorSize <=  53 ])
		ifTrue: [ ^ num asFloat / den asFloat ].	

	"Scale the fraction by a power of two scalePower so as to obtain an integer 
	value with 54 or 55 bits, depending on the values."
	scalePower := numeratorSize - denominatorSize - 54.
	scalePower >= 0
		ifTrue: [ d := d bitShift: scalePower ]
		ifFalse: [ n := n bitShift: scalePower negated ].
	scaledValue := n quo: d.
	scaledValueBits := scaledValue highBit.

	"How many high-order bits of scaledValue will end up in the significand?
	For tiny fractions that underflow to zero, this will be negative."
	significantBits := (scalePower + 1074 + scaledValueBits) min: 53.

	"The 54 or 55-bit integer scaledValue will be rounded to become the 
	significand of the float. Normal floats have a 53-bit significand.
	Subnormal floats have a significand with 0 to 52 significant bits.
	The 0-bit case includes very small fractions that round
	to 0.0 or to Float fmin. A negative number of significantBits will round to 0.0.

	Rounding:
	One or more low-order bits of the scaledValue will not be in the significand of the 
	float, and so are sub-significant. For normal floats, with 53-bit significands,
	there will be one or two sub-significant bits. Subnormal non-zero floats will
	have up to 55 sub-significant bits, and tiny fractions that round to 0.0 may 
	have more than that.
	The lowest-order bit of the significand is a ULP, a unit of least precision.
	If the sub-significant bits more than half a ULP, we round up, if less we round
	down. If equal, we consult the remainder to see whether there are any low-
	order one bits that were rounded away in computation of the scaledValue.
	If there are, we round up, but if there are none, we round either up or down,
	whichever makes the significand even (ULP bit zero).

	We increment the significant bits if rounding up, then clear any sub-significant
	bits to make the conversion to an integral float precise.
	We then scale the integral float to the correct magnitude by
	multiplying it by a power of two. For normal floats, the scale correction
	affects only the exponent of the float; the significand remains the same.
	If the exponent overflows the float becomes an infinity.
	If the exponent underflows the float becomes subnormal."

	ulp := 1 bitShift: scaledValueBits - significantBits.
	halfULP := ulp bitShift: -1.
	significanceMask := ulp - 1. 
	subSignificant := scaledValue bitAnd: significanceMask. "The low-order bits that will not end up in the significand."

	"Round up if needed"
	subSignificant >= halfULP 
		ifTrue: [(subSignificant > halfULP or: [n > (scaledValue * d)])
					ifTrue:  [ scaledValue := scaledValue + ulp ]
					ifFalse: ["Round half to even"
								scaledValue := scaledValue + (scaledValue bitAnd: ulp)]].

	"Clear any subSignificant bits so we only have bits that will go in the significand.
	Otherwise converting scaledValue to a float would round, in some cases incorrectly."
	scaledValue := scaledValue - subSignificant.

	scaledFloat := (self positive
		ifTrue: [ scaledValue asFloat ]
		ifFalse: [ scaledValue asFloat negated ]).
	
	"Scale correction"
	"For very negative exponents, must scale in two steps,
	since (2.0 raisedToInteger: -1075) rounds to 0.0, which would cause premature underflow to zero"
	^ scalePower < -1074 
			ifTrue: [scaledFloat * (2.0 raisedToInteger: scalePower + 1074) * (2.0 raisedToInteger: -1074)]
			ifFalse: [scaledFloat * (2.0 raisedToInteger: scalePower)]
%

category: 'Converting'
method: AbstractFraction
asFraction

"Returns the receiver."

^self
%

category: 'Converting'
method: AbstractFraction
asScaledDecimal: scale

"Returns a ScaledDecimal representation of the receiver."

^ ScaledDecimal numerator: self numerator denominator: self denominator scale: scale.
%

category: 'Formatting'
method: AbstractFraction
asString

"Returns a String of the form 'numerator/denominator'."

| result |
result := self numerator asString .
result add: $/ ; addAll: self denominator asString .
^ result
%

category: 'Testing'
method: AbstractFraction
even

"Returns true if the receiver is an even integer, false otherwise."

 self denominator = 1 ifFalse: [ ^ false ].
 ^ self numerator even

%

category: 'Testing'
method: AbstractFraction
isZero

"Returns true if the receiver is zero."

^ self numerator = 0 .
%

category: 'Arithmetic'
method: AbstractFraction
negated

"Returns a Number that is the negation of the receiver."

^ AbstractFraction _noreduce_numerator:  0 - self numerator
			denominator: self denominator
%

category: 'Testing'
method: AbstractFraction
odd

"Returns true if the receiver is an odd integer, false otherwise."

 self denominator = 1 ifFalse: [ ^ false ].
 ^ self numerator odd
%

category: 'Arithmetic'
method: AbstractFraction
reciprocal

"Returns 1 divided by the receiver.  Generates an error if the receiver is 0.
 Result is reduced."
| numer denom |
((numer := self numerator) == 0 or:[ numer == nil]) ifTrue: [ ^ self _errorDivideByZero].
denom := self denominator .
(numer == 1) ifTrue:[ ^ denom ].
numer < 0 ifTrue:[
  (numer == -1) ifTrue:[ ^ 0 - denom ].
  denom := 0 - denom . numer := 0 - numer .
].
^  AbstractFraction numerator: denom denominator: numer
%

category: 'Truncation and Rounding'
method: AbstractFraction
roundAndCoerceTo: aNumber
	"Returns the multiple of aNumber that is nearest in value to the receiver.
 If aNumber is a kind of Integer, ScaledDecimal, or AbstractFraction,
 the result will be an instance of the class of aNumber . "

	| r |
	aNumber = 0
		ifTrue: [ ^ 0 ].
	r := (self / aNumber) rounded * aNumber.
	aNumber _isFloat
		ifTrue: [ ^ r ].
	aNumber _isScaledDecimal
		ifTrue: [ ^ r ].
	^ r asFraction
%

category: 'Truncation and Rounding'
method: AbstractFraction
roundedHalfToEven

"Returns the integer nearest in value to the receiver. If the receiver is
exactly halfway between two integers, return the even one."

| remainder |
remainder := (self numerator abs \\ self denominator).
^ remainder * 2 = self denominator
	ifFalse: [self rounded]
	ifTrue: [| truncated | truncated := self truncated.
			truncated even
				ifTrue: [truncated]
				ifFalse: [truncated + self sign]]
%

category: 'Truncation and Rounding'
method: AbstractFraction
roundTo: aNumber
	"Returns the multiple of aNumber that is nearest in value to the receiver."

	| r |
	aNumber = 0
		ifTrue: [ ^ 0 ].
	r := (self / aNumber) rounded * aNumber.
	aNumber _isFloat
		ifTrue: [ ^ r ].
	^ r asFraction
%

category: 'Accessing'
method: AbstractFraction
sign
"Returns 1 if the receiver is greater than zero, -1 if the receiver is less
 than zero, and zero if the receiver is zero."

  ^ self numerator sign
%

category: 'Truncation and Rounding'
method: AbstractFraction
truncateAndCoerceTo: aNumber
	"Returns the multiple of aNumber that is closest to the receiver, on
 the same side of the receiver as zero is located.  In particular,
 returns the receiver if the receiver is a multiple of aNumber.

 If aNumber is a kind of Integer, ScaledDecimal, or AbstractFraction,
 the result will be an instance of the class of aNumber ."

	| t |
	aNumber = 0
		ifTrue: [ ^ 0 ].
	t := (self quo: aNumber) * aNumber.
	aNumber _isFloat
		ifTrue: [ ^ t ].
	aNumber _isScaledDecimal
		ifTrue: [ ^ t ].
	^ t asFraction
%

category: 'Truncation and Rounding'
method: AbstractFraction
truncated

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as zero is located."

^ self numerator quo: self denominator
%

category: 'Truncation and Rounding'
method: AbstractFraction
truncateTo: aNumber
	"Returns the multiple of aNumber that is closest to the receiver, on
 the same side of the receiver as zero is located.  In particular,
 returns the receiver if the receiver is a multiple of aNumber."

	| t |
	aNumber = 0
		ifTrue: [ ^ 0 ].
	t := (self quo: aNumber) * aNumber.
	aNumber _isFloat
		ifTrue: [ ^ t ].
	^ t asFraction
%

category: 'Converting'
method: AbstractFraction
_coerce: aNumber

"Reimplemented from Number."
| f |
f := aNumber asFraction .
f _isInteger ifTrue:[
  ^ AbstractFraction _noreduce_numerator: f denominator: 1
].
^ f
%

category: 'Converting'
method: AbstractFraction
_generality

"Reimplemented from Number."

^70
%

category: 'Accessing'
method: AbstractFraction
_getKind
	| num denom |
	(num := self numerator) ifNil: [ ^ 5	"nan" ].
	(denom := self denominator) ifNil: [ ^ 5 ].
	denom == 0
		ifTrue: [ 
			num == 0
				ifTrue: [ ^ 5	"NaN" ]
				ifFalse: [ ^ 3	"infinity" ] ].
	num == 0
		ifTrue: [ ^ 4	"zero" ].
	^ 1	"normal"
%

category: 'Indexing Support'
method: AbstractFraction
_isNaN

"Returns whether the receiver is quiet NaN or signaling NaN.
 This method is only to be used by the indexing subsystem."

^ (self numerator == nil) or: [self denominator == nil]
%

category: 'Private'
method: AbstractFraction
_reciprocal

"Returns 1 divided by the receiver.  Generates an error if the receiver is 0.
 Result is not reduced."
| numer denom |
((numer := self numerator) == 0 or:[ numer == nil]) ifTrue: [ ^ self _errorDivideByZero].
denom := self denominator .
numer < 0 ifTrue:[ denom := 0 - denom . numer := 0 - numer  ].

^  AbstractFraction _noreduce_numerator: denom denominator: numer
%

! Class extensions for 'AbstractSession'

!		Instance methods for 'AbstractSession'

removeallmethods AbstractSession
removeallclassmethods AbstractSession

category: 'Commit Processing'
method: AbstractSession
_abort

"(Subclass responsibility.)  Makes the remote session abort any changes to
 permanent state."

AbstractSession subclassResponsibility: #_abort
%

category: 'Commit Processing'
method: AbstractSession
_beginTransaction

"Do nothing for now."

%

category: 'Commit Processing'
method: AbstractSession
_commit: commitMode

"(Subclass responsibility.)  Makes the remote session commit any changes to
 permanent state."

AbstractSession subclassResponsibility: #_commit:
%

category: 'Commit Processing'
method: AbstractSession
_isNonBlocking

"Return true if the synchronized commit algorithm may use non-blocking
 messages and false otherwise."

^ false
%

category: 'Commit Processing'
method: AbstractSession
_nbAbort

"(Subclass responsibility.)  Non-blocking abort."

self _isNonBlocking
  ifTrue: [ AbstractSession subclassResponsibility: #_nbAbort ]
%

category: 'Commit Processing'
method: AbstractSession
_nbAbortResult

"(Subclass responsibility.)  Gets the result of a non-blocking abort."

self _isNonBlocking
  ifTrue: [ AbstractSession subclassResponsibility: #_nbAbortResult ]
%

category: 'Commit Processing'
method: AbstractSession
_nbCancel

"(Subclass responsibility.)  Cancel the execution of a non-blocking call."

self _isNonBlocking
  ifTrue: [ AbstractSession subclassResponsibility: #_nbCancel ]
%

category: 'Commit Processing'
method: AbstractSession
_nbCommit: commitMode

"(Subclass responsibility.)  Non-blocking commit:"

self _isNonBlocking
  ifTrue: [ AbstractSession subclassResponsibility: #_nbCommit: ]
%

category: 'Commit Processing'
method: AbstractSession
_nbCommitResult

"(Subclass responsibility.)  Gets the result of a non-blocking commit."

self _isNonBlocking
  ifTrue: [ AbstractSession subclassResponsibility: #_nbCommitResult ]
%

category: 'Commit Processing'
method: AbstractSession
_nbEnd

"(Subclass responsibility.)  Determine the status of a non-blocking call."

self _isNonBlocking
  ifTrue: [ AbstractSession subclassResponsibility: #_nbEnd ]
%

category: 'Commit Processing'
method: AbstractSession
_nbTimeout

"(Subclass responsibility.)  Returns a SmallInteger, indicating the minimum
 number of seconds that a non-blocking call will be allowed to execute before
 the coordinator considers the non-blocking call to have failed."

AbstractSession subclassResponsibility: #_nbTimeout
%

category: 'Commit Processing'
method: AbstractSession
_nbVoteResult

"(Subclass responsibility.)  Gets the result of a non-blocking vote."

self _isNonBlocking
  ifTrue: [ AbstractSession subclassResponsibility: #_nbVoteResult ]
%

category: 'Commit Processing'
method: AbstractSession
_nbVoteToCommit

"(Subclass responsibility.)  Non-blocking voteToCommit."

self _isNonBlocking
  ifTrue: [ AbstractSession subclassResponsibility: #_nbVoteToCommit ]
%

category: 'Accessing'
method: AbstractSession
_publicName

"(Subclass responsibility.)  Returns a name to use for error messages."

AbstractSession subclassResponsibility: #_publicName
%

category: 'Commit Processing'
method: AbstractSession
_transactionMode: newMode

"Do nothing for now."

%

category: 'Commit Processing'
method: AbstractSession
_voteToCommit

"(Subclass responsibility.)  Returns a SmallInteger indicating the status
 of the vote.  The returned value has the following meaning:

 0: The session was read-only and the vote is affirmative.
 1: The session wrote objects and the vote is affirmative.
 2: The vote is negative; the session cannot commit."

AbstractSession subclassResponsibility: #_voteToCommit
%

! Class extensions for 'AbstractUserProfileSet'

!		Instance methods for 'AbstractUserProfileSet'

removeallmethods AbstractUserProfileSet
removeallclassmethods AbstractUserProfileSet

category: 'Adding'
method: AbstractUserProfileSet
add: aUserProfile

"(Subclass responsibility.)  Adds aUserProfile to the receiver."

AbstractUserProfileSet subclassResponsibility: #add:
%

category: 'Adding'
method: AbstractUserProfileSet
addAll: aCollection

"Reimplemented to maintain KeyValueDictionary on AllUsers."

AbstractUserProfileSet subclassResponsibility: #addAll:
%

category: 'Removing'
method: AbstractUserProfileSet
removeAll: aCollection

"Reimplemented to maintain KeyValueDictionary on AllUsers."

| aColl tmpArr|
aColl := aCollection.
aColl == self ifTrue:[
  aColl == AllUsers ifTrue:[
    self _halt: 'Attempt to remove all elements of AllUsers'  .
    self _uncontinuableError
    ].
  tmpArr := Array withAll: aColl .
  aColl := tmpArr .
  ].
aColl accompaniedBy: self do:[ :me :element | me remove: element ].
^ aCollection
%

category: 'Removing'
method: AbstractUserProfileSet
removeAllPresent: aCollection

"Reimplemented to maintain KeyValueDictionary on AllUsers."

aCollection == self ifTrue:[ ^ self removeAll: aCollection ].

aCollection accompaniedBy: self do:[ :me :element | me removeIfPresent: element ].
^ aCollection
%

category: 'Removing'
method: AbstractUserProfileSet
removeIfPresent: anObject

"Reimplemented to maintain KeyValueDictionary on AllUsers."

^ self remove: anObject ifAbsent: [ nil ]
%

category: 'Accessing'
method: AbstractUserProfileSet
userWithId: aString

"Searches the receiver for a UserProfile whose userId is equal to aString, and
 returns that UserProfile.  Generates an error if no userId is equal to
 aString."

^ self userWithId: aString
         ifAbsent:[ self _error: #objErrNotInColl args: { aString }]
%

category: 'Adding'
method: AbstractUserProfileSet
_add: aUserProfile

AbstractUserProfileSet subclassResponsibility: #_add:
%

category: 'Removing'
method: AbstractUserProfileSet
_removeAll: aCollection errIfAbsent: aBool

"Reimplemented to maintain KeyValueDictionary on AllUsers."

aBool ifTrue:[ self removeAll: aCollection]
     ifFalse:[ self removeAllPresent: aCollection ]
%

! Class extensions for 'Admonition'

!		Instance methods for 'Admonition'

removeallmethods Admonition
removeallclassmethods Admonition

category: 'Instance initialization'
method: Admonition
initialize
  gsNumber := ERR_Admonition.
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'AlmostOutOfMemory'

!		Class methods for 'AlmostOutOfMemory'

removeallmethods AlmostOutOfMemory
removeallclassmethods AlmostOutOfMemory

category: 'Control'
classmethod: AlmostOutOfMemory
disable
  "Disables signalling of AlmostOutOfMemory and resets threshold to 90%"
  System _updateSignalErrorStatus: 5 toState: -1 .
  ^ self
%

category: 'Control'
classmethod: AlmostOutOfMemory
enable
  "Enable signaling of AlmostOutOfMemory with the current threshold.
  If no in-memory garbage collection has run since the last delivery
  of an AlmostOutOfMemory exception, then the actual enable will
  be deferred until after in-memory garbage collection .

  You must install a handler for AlmostOutOfMemory to take action
  or execute
    Notification enableSignalling
  to have AlmostOutOfMemory and other Notifications signalled to the
  application to avoid having AlmostOutOfMemory silently ignored."

  System _updateSignalErrorStatus: 5 toState: 0 .
  ^ self
%

category: 'Control'
classmethod: AlmostOutOfMemory
enableAtThreshold: anInteger

  "anInteger represents a percentage and must be between 1 and 125.
  Sets the threshold for signalling AlmostOutOfMemory notification
  and enables the notification.  
  Threshold is also used by AlmostOutOfMemoryError if enabled.
  Returns the receiver."

  self _validateThreshold: anInteger .
  System _updateSignalErrorStatus: 5 toState: anInteger .
  ^ self
%

category: 'Control'
classmethod: AlmostOutOfMemory
enabled
  "Returns true if signaling of AlmostOutOfMemory is enabled.

   See also AlmostOutOfMemory >> enable .
   If the effect of the previous enable was deferred, will return false."
  ^ (System _signalErrorStatus: 5) > 0
%

category: 'Control'
classmethod: AlmostOutOfMemory
threshold
  "Returns the current threshold without changing enabled status
   of the AlmostOutOfMemory notification.
  Threshold is shared with AlmostOutOfMemoryError . "

  ^ System _updateSignalErrorStatus: 7 toState: -1 .
%

category: 'Control'
classmethod: AlmostOutOfMemory
threshold: anInteger
  "anInteger represents a percentage and must be between 1 and 125.
  Sets the threshold for signalling AlmostOutOfMemory notification
  without changing the enabled status .
  Threshold is shared with AlmostOutOfMemoryError .
  Returns the receiver."

  self _validateThreshold: anInteger .
  System _updateSignalErrorStatus: 7 toState: anInteger .
  ^ self
%

category: 'Control'
classmethod: AlmostOutOfMemory
_validateThreshold: anInteger

  "anInteger represents a percentage and must be between 1 and 125."

  anInteger _isSmallInteger ifFalse:[ anInteger _validateClass: SmallInteger ].
  (anInteger < 1 or:[ anInteger > 125]) ifTrue:[
    OutOfRange new name: 'threshold' min: 1 max: 125 actual: anInteger ;
	signal
  ].
%

!		Instance methods for 'AlmostOutOfMemory'

category: 'Instance initialization'
method: AlmostOutOfMemory
initialize
  gsNumber := ERR_AlmostOutOfMemory.
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'AlmostOutOfMemoryError'

!		Class methods for 'AlmostOutOfMemoryError'

removeallmethods AlmostOutOfMemoryError
removeallclassmethods AlmostOutOfMemoryError

category: 'Control'
classmethod: AlmostOutOfMemoryError
disable
  "Disables signalling of AlmostOutOfMemoryError and resets threshold to 90%"
  System _updateSignalErrorStatus: 9 toState: -1 .
%

category: 'Control'
classmethod: AlmostOutOfMemoryError
enable
  "Enable signaling of AlmostOutOfMemoryError with the current threshold.
  If no in-memory garbage collection has run since the last delivery
  of an AlmostOutOfMemory or AlmostOutOfMemoryError, then the actual enable will
  be deferred until after in-memory garbage collection ."

  System _updateSignalErrorStatus: 9 toState: 0 .
%

category: 'Control'
classmethod: AlmostOutOfMemoryError
enableAtThreshold: anInteger
  "anInteger represents a percentage and must be between 1 and 125.
  Sets the threshold for signalling AlmostOutOfMemoryError  .
  and enables the signalling of AlmostOutOfMemoryError .
  Threshold is also used by AlmostOutOfMemory (a Notification) if enabled.
  Returns the receiver."

  AlmostOutOfMemory _validateThreshold: anInteger .
  System _updateSignalErrorStatus: 9 toState: anInteger .
%

category: 'Control'
classmethod: AlmostOutOfMemoryError
enabled
  "Returns true if signaling of AlmostOutOfMemoryError is enabled.

   See also AlmostOutOfMemoryError >> enable .
   If the effect of the previous enable was deferred, will return false."
  ^ (System _signalErrorStatus: 9 ) > 0
%

category: 'Control'
classmethod: AlmostOutOfMemoryError
threshold
  "Returns the current threshold without changing enabled status
   of AlmostOutOfMemoryError.
   Threshold is shared with AlmostOutOfMemory (a Notification) ."

  ^ AlmostOutOfMemory threshold
%

category: 'Control'
classmethod: AlmostOutOfMemoryError
threshold: anInteger
  "anInteger represents a percentage and must be between 1 and 125.
  Sets the threshold for signalling AlmostOutOfMemoryError 
  without changing the enabled status .
  Threshold is shared with AlmostOutOfMemory (a Notification) .
  Returns the receiver."

  ^ AlmostOutOfMemory threshold: anInteger .
%

!		Instance methods for 'AlmostOutOfMemoryError'

category: 'Instance initialization'
method: AlmostOutOfMemoryError
initialize
  gsNumber := 2744"ERR_AlmostOutOfMemory".
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'AlmostOutOfStack'

!		Instance methods for 'AlmostOutOfStack'

removeallmethods AlmostOutOfStack
removeallclassmethods AlmostOutOfStack

category: 'Instance initialization'
method: AlmostOutOfStack
initialize
  gsNumber := ERR_AlmostOutOfStack.
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'AlmostOutOfStackError'

!		Class methods for 'AlmostOutOfStackError'

removeallmethods AlmostOutOfStackError
removeallclassmethods AlmostOutOfStackError

category: 'Control'
classmethod: AlmostOutOfStackError
disable
  System _updateSignalErrorStatus: 10 toState: false .
%

category: 'Control'
classmethod: AlmostOutOfStackError
enable
  System _updateSignalErrorStatus: 10 toState: true .
%

category: 'Control'
classmethod: AlmostOutOfStackError
enabled
  "returns a Boolean"
  ^ System _signalErrorStatus: 10  
%

!		Instance methods for 'AlmostOutOfStackError'

category: 'Instance initialization'
method: AlmostOutOfStackError
initialize
  gsNumber := ERR_AlmostOutOfStack.
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'ArgumentError'

!		Instance methods for 'ArgumentError'

removeallmethods ArgumentError
removeallclassmethods ArgumentError

category: 'Formatting'
method: ArgumentError
buildMessageText
^ self buildMessageText:(
  [ | str |
    gsDetails ifNotNil:[:det |
      (str := String new ) add: det asString .
      object ifNotNil:[:o|
        str add:', '; add: o  asString
      ] ifNil:[
        gsArgs ifNotNil:[:ga | (ga atOrNil: 1) ifNotNil:[:d | str add: d asString]]
      ]
    ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

category: 'Instance initialization'
method: ArgumentError
initialize
  gsNumber := ERR_ArgumentError.
  gsResumable := true .
  gsTrappable := true .
%

category: 'Instance initialization'
method: ArgumentError
name: aName
  object := aName
%

! Class extensions for 'ArgumentTypeError'

!		Instance methods for 'ArgumentTypeError'

removeallmethods ArgumentTypeError
removeallclassmethods ArgumentTypeError

category: 'Instance initialization'
method: ArgumentTypeError
args: anArray
  anArray _isArray ifTrue:[ | sz |
    object := anArray atOrNil: 1 .
    expectedClass := anArray atOrNil: 2 .
    actualArg := anArray atOrNil: 3  .
    (sz := anArray size) > 3 ifTrue:[
      gsArgs := anArray copyFrom: 4 to: sz
    ].
  ]
%

category: 'Formatting'
method: ArgumentTypeError
buildMessageText
^ self buildMessageText:(
  [ | str |
    gsDetails ifNotNil:[ :d |  "example use: attempt to modify invariant object"
      str := d asString copy .
      object ifNotNil:[:a | str add: ', ' ; add: a asString ].
    ] ifNil:[ | exp act |
      exp := expectedClass . act := actualArg .
      (exp ~~ nil or:[ act ~~ nil]) ifTrue:[
        str := String new .
        object ifNotNil:[:o | str add: 'for '; add: o asString ; add:'  '].
        exp ifNotNil:[ str add: 'expected a ' ; add: exp name ].
        act ifNotNil:[ str add:', arg is a '; add: act class name ].
      ].
    ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

category: 'Instance initialization'
method: ArgumentTypeError
constrainedIv: aName expectedClass: aClass actualArg: anObject
  gsNumber := 2107 .  "legacy error info"
  gsReason := #objErrConstraintViolation .
  object := aName .
  expectedClass := aClass "may be an array of Classes" .
  actualArg := anObject
%

category: 'Instance initialization'
method: ArgumentTypeError
initialize
  gsNumber := ERR_ArgumentTypeError.
  gsResumable := true .
  gsTrappable := true .
%

category: 'Instance initialization'
method: ArgumentTypeError
name: aName expectedClass: aClass actualArg: anObject
  object := aName .
  expectedClass := aClass "may be an array of Classes" .
  actualArg := anObject
%

category: 'Instance initialization'
method: ArgumentTypeError
_legacyHandlerArgs
  | arr |
  arr := { object . expectedClass . actualArg } .
  gsArgs ifNotNil:[:a |  arr addAll: a ].
  ^ arr
%

! Class extensions for 'Array'

!		Class methods for 'Array'

removeallmethods Array
removeallclassmethods Array

category: 'Disallowed'
classmethod: Array
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
inClassHistory: aClassHistory
description: aDescription
isInvariant: invarBoolean

"Disallowed for Array and its subclasses."
^ self _error: #classErrByteSubclass
%

category: 'Subclass Creation'
classmethod: Array
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
newVersionOf: oldClass
description: aDescription
options: options

"Disallowed for Array and its subclasses."
^ self _error: #classErrByteSubclass
%

category: 'Instance Creation'
classmethod: Array
with: aValue

<primitive: 1099>
^ self _primitiveFailed: #with: args: { aValue }
%

category: 'Instance Creation'
classmethod: Array
with: aValue with: val2

<primitive: 1099>
^ self _primitiveFailed: #with:with: args: { aValue . val2 }
%

category: 'Instance Creation'
classmethod: Array
with: aValue with: val2 with: val3

<primitive: 1099>
^ self _primitiveFailed: #with:with:with: args: { aValue . val2 . val3 }
%

category: 'Instance Creation'
classmethod: Array
with: aValue with: val2 with: val3 with: val4

<primitive: 1099>
^ self _primitiveFailed: #with:with:with:with: args: { aValue . val2 . val3 . val4 }
%

category: 'Instance Creation'
classmethod: Array
with: aValue with: val2 with: val3 with: val4 with: val5

<primitive: 1099>
^ self _primitiveFailed: #with:with:with:with:with: args: { aValue . val2 . val3 . val4 . val5 }
%

category: 'Private'
classmethod: Array
_encodingForStringDensity

  "Number of characters encoded per SmallInteger by <primitive: 542>."

^ 7
%

category: 'Private'
classmethod: Array
_insertEncodingForString: aString arraySize: aSmallInteger

  "Encode the given string into SmallIntegers and place them in a new
   instance of Array. Used for user-defined indexing support.

   Note: if there are null characters in the encoded string they will be
      effectively ignored during encoding and the decoded string will
      not match the encoded string."

  <primitive: 542>

  aString  _validateClass: String .
  aSmallInteger _validateClass: SmallInteger .
  ^ self _primitiveFailed: #_insertEncodingForString:arraySize:
         args: { aString . aSmallInteger }

%

category: 'Filein Support'
classmethod: Array
_newKernelByteSubclass: clsName
classVars: classVarArg
poolDictionaries: poolDicts
inDictionary: aDict
options: options
reservedOop: aReservedOop

"Disallowed for Array and its subclasses."

^ self _error: #classErrByteSubclass
%

!		Instance methods for 'Array'

category: 'Adding'
method: Array
add: anObject

"Makes anObject one of the receiver's elements and returns anObject.  The
 new element is actually added as the last element of the receiver.  This
 method is equivalent to #addLast: ."

<primitive: 203>
^ self _primitiveFailed: #add: args: { anObject }
%

category: 'Adding'
method: Array
addAll: aCollection

"Adds all of the elements of aCollection to the receiver and returns
 aCollection."

<primitive: 213>
"primitive handles arg kinds Array, OrderedCollection and IdentityBag."

^ super addAll: aCollection
%

category: 'Adding'
method: Array
addAll: anArray excludingFirst: excludeCount

"If   anArray size > (excludeCount+1)
adds elements of anArray from (excludeCount+1)  to  (anArray size)
to the receiver,  otherwise does nothing.
Returns the receiver .

If excludeCount is less than 0 it is intepreted as 0 ."

<primitive: 231>
anArray _validateClasses: { Array . OrderedCollection }.
excludeCount _validateClass: SmallInteger .
^ self _primitiveFailed: #addAll:excludingFirst:
       args: { anArray . excludeCount }
%

category: 'Adding'
method: Array
addLast: anObject

"Adds anObject as the last element of the receiver and returns anObject."

<primitive: 203>
^ self _primitiveFailed: #addLast: args: { anObject }
%

category: 'Accessing'
method: Array
at: anIndex

"Returns the value of an indexed variable in the receiver.
 The argument anIndex must not be larger than the size of the
 receiver, and must not be less than 1.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds, or if the receiver is not indexable."

<primitive: 699>
(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
^ self _primitiveFailed: #at: args: { anIndex }
%

category: 'Private'
method: Array
at: anIndex caseInsensitiveEquals: encodedStringArray

 "Compares without regard to case, in order starting at anIndex, if the contents of the receiver
  are equal to the encoded contents of <encodedStringArray>. The elements of both
  collections must be SmallIntegers. The SmallIntegers represents strings encoded using
  <primitive: 542>."

  ^self at: anIndex compareEqualNoCaseWithEncodedStringArray: encodedStringArray startingAt: 1.
%

category: 'Private'
method: Array
at: anIndex caseSensitiveEquals: encodedStringArray

 "Compares without regard to case, in order starting at anIndex, if the contents of the receiver
  are equal to the encoded contents of <encodedStringArray>. The elements of both
  collections must be SmallIntegers. The SmallIntegers represents strings encoded using
  <primitive: 542>."

  ^self compareStartingAt: anIndex to: encodedStringArray startingAt: 1
%

category: 'Private'
method: Array
at: anIndex compareEqualNoCaseWithEncodedStringArray: encodedStringArray startingAt: stringOffset

 "Compares without regard to case, in order starting at anIndex, if the contents of the receiver
  are equal to the encoded contents of <encodedStringArray>, starting at stringOffset. The elements of both
  collections must be SmallIntegers. The SmallIntegers represents strings encoded using
  <primitive: 542>."

  <primitive: 632 >
  ^self _primitiveFailed: #at:compareEqualNoCaseWithEncodedStringArray:startingAt:
        args: { anIndex .  encodedStringArray . stringOffset }

%

category: 'Private'
method: Array
at: anIndex compareWithEncodedStringArray: encodedStringArray startingAt: stringOffset opCode: anOpCode

 "Compares, in order starting at anIndex, if the contents of the receiver
  are less than/greater than the contents of anArray, starting at stringOffset.
  The elements of both collections must be SmallIntegers. The SmallIntegers
  represents strings encoded using <primitive: 542>.

    op code == 0 means less than
    op code == 1 means greater than"

  <primitive: 629 >
  ^ self _primitiveFailed: #at:compareWithEncodedStringArray:startingAt:opCode:
         args: { anIndex . encodedStringArray . stringOffset . anOpCode }
%

category: 'Private'
method: Array
at: anIndex greaterThanEncodedStringArray: encodedStringArray startingAt: arrayOffset
  "Compares, in order starting at anIndex, if the contents of the receiver are greater
   than the contents of anArray, starting at stringOffset. The elements of both
   collections must be SmallIntegers. The SmallIntegers represent encoded string for
   index. The string was encoded using <primitive: 542>."

  ^self at: anIndex
        compareWithEncodedStringArray: encodedStringArray
        startingAt: arrayOffset
	opCode: 1.
%

category: 'Private'
method: Array
at: anIndex lessThanEncodedStringArray: encodedStringArray startingAt: arrayOffset
   "Compares, in order starting at anIndex, if the contents of the receiver
    are less than the contents of anArray, starting at stringOffset.
    The elements of both collections must be SmallIntegers. The SmallIntegers represent
    encoded string for index. The string was encoded using <primitive: 542>."

  ^self at: anIndex
	compareWithEncodedStringArray: encodedStringArray
	startingAt: arrayOffset
	opCode: 0.
%

category: 'Updating'
method: Array
at: anIndex put: aValue

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Returns aValue.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds."

<primitive: 745>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

^ self _primitiveFailed: #at:put: args: { anIndex . aValue }
%

category: 'Accessing'
method: Array
atOrNil: anIndex

"Return the value   self at: anIndex ,
 or return nil if anIndex is out of range . "

<primitive: 699>
anIndex _isInteger ifTrue:[
  ^ nil
].
self _errorNonIntegerIndex: anIndex .
^ self _primitiveFailed: #atOrNil: args: { anIndex }
%

category: 'Private'
method: Array
compareStartingAt: anIndex to: anArray startingAt: anotherIndex

  "Compare the elements of the receiver starting at <anIndex> to <anArray>
   starting at <anotherIndex>. Note that it is assumed that the objects
   in each slot of the receiver and <anArray> are SmallIntegers encoded
   from strings produced by <primitive: 542>, however, it is not a
   requirement, since each element is compared using identity."

  <primitive: 628 >
  ^ self _primitiveFailed: #compareStartingAt:to:startingAt:
         args: { anIndex . anArray . anotherIndex }
%

category: 'Copying'
method: Array
copyFrom: startIndex to: stopIndex

"Returns a new Array containing the elements of the receiver
 between startIndex and stopIndex, inclusive.  The result is of the same class
 as the receiver.

 If startIndex > stopIndex then an empty collection is returned.
 Otherwise both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex.
"

<primitive: 817>
(startIndex < 1) ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].

((stopIndex > self size) or: [(stopIndex < 0)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].

self _primitiveFailed: #copyFrom:to:
     args: { startIndex . stopIndex }
%

category: 'Copying'
method: Array
copyNotNilFrom: startIndex to: stopIndex
 "Return an Array containing the non-nil elements 
  that are within the specified offsets of the receiver."
 <primitive: 541>
 "Primitive fails for a large object."
 | res |
 startIndex _validateInstanceOf: SmallInteger .
 stopIndex _validateInstanceOf: SmallInteger .
 (startIndex < 1 or:[ startIndex > self size ]) ifTrue:[
   ^ self _errorIndexOutOfRange: startIndex
 ].
 stopIndex < startIndex ifTrue:[ ^ self _errorIndexOutOfRange: stopIndex].
 res := { } .
 startIndex to: stopIndex do:[:j |
   (self at: j) ifNotNil:[:x| res add: x  ] .
 ].
 ^ res
%

category: 'Obsolete'
method: Array
fillFrom: index1 resizeTo: newSize with: anObject

  self deprecated: 'Array >> fillFrom:resizeTo:with: has primitive support removed v3.7.2 to avoid out of memory issues. Use fillFrom:to:with: after pregrowing the Array'.

  Error signal: 'Array >> fillFrom:resizeTo:with: not supported, see Array >> fillFrom:to:with:' .
%

category: 'Initializing'
method: Array
fillFrom: startIdx to: endIdx with: anObject

"  store anObject into instVars startIdx .. endIdx of the receiver.
   the receiver will be grown if necessary .
   Attempts to grow the receiver beyond 2034 total instVars signals an error"

<primitive: 607>
| namedSize |
startIdx _validateClass: SmallInteger .
endIdx _validateClass: SmallInteger .
(startIdx < 1) ifTrue:[ self _errorIndexOutOfRange: startIdx ].
(startIdx > endIdx) ifTrue:[ startIdx _error: #rtErrBadCopyFromTo args: { endIdx }].
(endIdx + (namedSize := self class instSize)) > 2034 ifTrue:[ 
  OutOfRange new name:'endIdx' max: 2034 - namedSize actual: endIdx ; signal
].
self _primitiveFailed: #fillFrom:to:with: args: { startIdx . endIdx . anObject }
%

category: 'Comparing'
method: Array
hasIdenticalContents: anArray

"Returns true if all of the following conditions are true:

 1.  The receiver and anArray are of the same class.
 2.  The two Arrays are the same size.
 3.  The corresponding elements of the receiver and anArray are identical.

 Returns false otherwise."

<primitive: 612 >

^ self _primitiveFailed: #hasIdenticalContents: args: { anArray }
%

category: 'Searching'
method: Array
includesIdentical: anObject

"Returns true if anObject is identical to one of the elements of the
 receiver.  Returns false otherwise."

<primitive: 59>
self _primitiveFailed: #includesIdentical: args: { anObject }
%

category: 'Searching'
method: Array
includesValue: anObject

"Returns true if anObject is equal to one of the elements of the receiver.
 Returns false otherwise."

^ self includes: anObject
%

category: 'Searching'
method: Array
indexOfIdentical: anObject

"Private.  Returns the index of the first element in the receiver that is
 identical to the argument.  If the receiver does not have any elements that are
 identical to the argument, returns zero."

<primitive: 494>
self _primitiveFailed: #indexOfIdentical: args: { anObject }
%

category: 'Searching'
method: Array
indexOfNotNil: startOffset to: endOffset
"If startOffset <= endOffset, returns the first offset witin in the specified range
 of a non-nil element of the receiver.
 If startOffset > endOffset, returns the last offset within the specified range
 of a non-nil element of the receiver.
 Returns zero if all are nil .
 Intended for use only on Arrays of size <= 2000 .  
 Performance on larger arrays will be slow.
 "
 <primitive: 540>
 "Primitive fails for a large object."
  | sz |
 startOffset _validateInstanceOf: SmallInteger .
 endOffset _validateInstanceOf: SmallInteger .
 sz := self size .
 (startOffset < 1 or:[ startOffset > sz]) ifTrue:[
   ^ self _errorIndexOutOfRange: startOffset
 ].
 (endOffset < 1 or:[ endOffset > sz]) ifTrue:[
   ^ self _errorIndexOutOfRange: startOffset
 ].
 startOffset <= endOffset ifTrue:[
   startOffset to: endOffset do:[:j |
     (self at: j) ifNotNil:[ ^ j ] .
   ].
 ] ifFalse:[
   startOffset _downTo: endOffset do:[:j |
     (self at: j) ifNotNil:[ ^ j ] .
   ].
 ].
 ^ 0
%

category: 'Enumerating'
method: Array
pairsDo: aBlock
  "aBlock must be a two argument block"

  1 to: self size by: 2 do:[:j |
    aBlock value:(self at: j) value:(self at: j+1)
  ]
%

category: 'Copying'
method: Array
replaceFrom: startIndex to: stopIndex with: aCollection startingAt: repIndex

"Replaces the elements of the receiver between the indexes startIndex and
 stopIndex inclusive with the elements of aSeqCollection starting at repIndex.
 If aCollection is identical to the receiver, the source and
 destination blocks may overlap.

 aCollection must be a kind of SequenceableCollection or a kind of IdentityBag.

 Returns the receiver."

<primitive: 608>
startIndex _isSmallInteger ifFalse:[ startIndex _validateClass: SmallInteger ].
stopIndex _isSmallInteger ifFalse:[ stopIndex _validateClass: SmallInteger ].
repIndex _isSmallInteger ifFalse:[ repIndex _validateClass: SmallInteger ].

^ super replaceFrom: startIndex to: stopIndex with: aCollection startingAt: repIndex
%

category: 'Adding'
method: Array
_addAllFromNsc: aBag

""

<primitive: 202>
^ self _primitiveFailed: #_addAllFromNsc: args: { aBag }
%

category: 'Decompiling without Sources'
method: Array
_asSource

""

| result word lineLength LF |
self isInvariant ifFalse:[ self _error: #rtErrObjVariant ] .
result := String new .
LF := Character lf .
result addAll: ' #( ' .
lineLength := result size .
1 to: self size do:[:j |
  word := (self at: j) _asSource .
  result addAll: word .
  result add: $  .
  lineLength := lineLength + word size + 1 .
  lineLength > 80 ifTrue:[
    result add: LF ; add: $  .
    lineLength := 1 .
    ].
  ].
result addAll: ') ' .
^ result .
%

category: 'Accessing'
method: Array
_at: anIndex

"Returns the value of an indexed variable in the receiver.
 The argument anIndex must not be larger than the size of the
 receiver, and must not be less than 1.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds, or if the receiver is not indexable."

<primitive: 699>
(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
^ self _primitiveFailed: #at: args: { anIndex }
%

category: 'Updating'
method: Array
_at: anIndex put: aValue

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Returns aValue.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds."

<primitive: 745>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

^ self _primitiveFailed: #at:put: args: { anIndex . aValue }
%

category: 'Updating'
method: Array
_basicAt: anIndex put: aValue

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Returns aValue.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds."

<primitive: 745>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

^ self _primitiveFailed: #_basicAt:put: args: { anIndex . aValue }
%

category: 'Private'
method: Array
_decodeKeyAt: keyIdx decoding: numToDecode into: aString
  "Decode <numToDecode> SmallIntegers from the receiver into <aString> as
   encoded by <primitive: 542>.

   Note: if there are null characters in the encoded string they will be
      effectively ignored during encoding and the decoded string will
      not match the encoded string."

  <primitive: 631 >
  aString  _validateClass: String .
  numToDecode _validateClass: SmallInteger .
  keyIdx _validateClass: SmallInteger .
  ^ self _primitiveFailed: #_decodeKeyAt:decoding:into:
         args: { keyIdx . numToDecode . aString }
%

category: 'Removing'
method: Array
_deleteNoShrinkFrom: startIndex to: stopIndex

"The elements of the receiver from startIndex to stopIndex are
 deleted from the receiver.  And the tail of the receiver is filled with
 (stopIndex - startIndex + 1) number of nils.

 The size of the receiver is not changed.
 Does space-efficient tranlogging of the deletion.

 Both startIndex and stopIndex must be positive integers not larger
 than the size of the receiver, with startIndex <= stopIndex."

<primitive: 602 >

(stopIndex < startIndex)
ifTrue:
   [ ^ startIndex _error: #rtErrBadCopyFromTo args: { stopIndex }].
((stopIndex > self size) or: [(stopIndex < 1)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].
(startIndex < 1)
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].
^ self _primitiveFailed: #_deleteNoShrinkFrom:to:
       args: { startIndex . stopIndex }
%

category: 'Private'
method: Array
_deleteNoShrinkFrom: startIndex to: stopIndex anchorTailSize: tailSize

"Same as _deleteNoShrinkFrom: startIndex to: stopIndex except the last
tailSize objects remain in the tail of the array.  The size of the receiver does not change
and the gap created by the delete operation is filled with nils.

This method can fail for the following reasons:
-receiver participates in an index.
-receiver is a large object.
-the delete operation requested would delete any part of the tail

 The size of the receiver is not changed.

 Both startIndex and stopIndex must be positive integers not larger
 than the size of the receiver, with startIndex <= stopIndex.  "

<primitive: 617 >

(stopIndex < startIndex)
ifTrue:
   [ ^ startIndex _error: #rtErrBadCopyFromTo args: { stopIndex }].
((stopIndex > self size) or: [(stopIndex < 1)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].
(startIndex < 1)
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].
(stopIndex > (self size - tailSize) )
   ifTrue: [ ^ self _errorIndexOutOfRange: tailSize].

^ self _primitiveFailed: #_deleteNoShrinkFrom:to:anchorTailSize:
       args: { startIndex . stopIndex . tailSize }
%

category: 'Private'
method: Array
_finalize: anObj at: offset

"Finalizes the reference to object anObject by substituting nil."

self at: offset put: nil.
^true
%

category: 'Copying'
method: Array
_insertAt: destOffset
from: anArray
fromStart: startOffset
fromEnd: endOffset
numToMoveDown: numToMove

"Intended for use in manipulating objects which are nodes of B-trees or
 similar structures.  Does not do optimized tranlogging.

 Inserts the portion of anArray from startOffset to endOffset
 into the receiver beginning at destOffset.

 If anArray is nil, then  (endOffset - startOffset + 1) nils are
 inserted into the receiver.

 The indexable instance variables of the receiver from destOffset to
 (destOffset + numToMove) are moved towards the end of the receiver by
 the amount (endOffset - startOffset + 1).
 The receiver must be a small object.

 If the receiver must be grown, it will be grown by 10 times the number
 of OOPs inserted up to a max of 500 oops. The receiver will not be
 grown larger than the max size of a small object. If the number of
 inserted oops alone will cause the object to be larger than the max
 size of a small object, then an error will be generated.

 Generates an error if any of the arguments imply the receiver must be
 grown to be a large object, or if destOffset is beyond the end of
 the receiver."

<primitive: 605 >

| numInserted newSize |

anArray ~~ nil ifTrue:[ anArray _validateClass: Array ].

(destOffset < 1 or:[ destOffset > self _basicSize] )
   ifTrue:[ self _errorIndexOutOfRange: destOffset] .
(startOffset < 1) ifTrue:[ self _errorIndexOutOfRange: startOffset] .
(endOffset < 1)   ifTrue:[ self _errorIndexOutOfRange: endOffset] .
(numToMove < 0)   ifTrue:[ self _errorIndexOutOfRange: numToMove] .
(endOffset < startOffset) ifTrue:[ self _errorIndexOutOfRange: endOffset ].
(anArray ~~ nil and: [startOffset > anArray _basicSize]) ifTrue:[ anArray _errorIndexOutOfRange: startOffset].
(anArray ~~ nil and: [endOffset > anArray _basicSize]) ifTrue:[ anArray _errorIndexOutOfRange: endOffset].

numInserted := endOffset - startOffset + 1 .

newSize := destOffset + numInserted + numToMove .
newSize > 2034 "virtual machine constant OBJ_OOPS_SIZE" ifTrue:[
  self _error: #objErrMaxSize args:{ 2034 "virtual machine constant" . newSize }.
].

self _primitiveFailed: #_insertAt:from:fromStart:fromEnd:numToMoveDown:
     args: { destOffset . anArray . startOffset . endOffset . numToMove }
%

category: 'Copying'
method: Array
_insertAt: destOffset
value: anObject
numToMoveDown: numToMove

"Intended for use in manipulating objects which are nodes of B-trees or
 similar structures.  Does space-efficient tranlogging of the insertion.

 Inserts anObject into receiver at destOffset, with space-efficient tranlogging
 of the insert.

 The indexable instance variables of the receiver from destOffset to
 (destOffset + numToMove) are moved towards the end of the receiver by 1 .
 then does (self at: destOffset put: anObject ).
 The receiver must be a small object;

 Generates an error if any of the arguments imply the receiver must be
 grown to be a large object, or if destOffset is beyond the end of
 the receiver."

<primitive: 1041>
| newSize |
destOffset _validateClass: SmallInteger .
numToMove _validateClass: SmallInteger .
(destOffset < 1)   ifTrue:[ self _errorIndexOutOfRange: destOffset] .
(numToMove < 1)   ifTrue:[ self _errorIndexOutOfRange: destOffset] .
newSize := self namedSize +  destOffset + 1  +  numToMove .
newSize > 2034 "virtual machine constant OBJ_OOPS_SIZE" ifTrue:[
  self _error: #objErrMaxSize args:{ 2034 "virtual machine constant" . newSize }.
].
self _primitiveFailed: #_insertAt:value:numToMoveDown
     args: { destOffset . anObject . numToMove }
%

category: 'Copying'
method: Array
_insertAt: destOffset
value: valOne
value: valTwo
numToMoveDown: numToMove

"Intended for use in manipulating objects which are nodes of B-trees or
 similar structures.  Does space-efficient tranlogging of the insertion.

 Inserts objects into receiver at destOffset, with space-efficient tranlogging
 of the insert.

 The indexable instance variables of the receiver from destOffset to
 (destOffset + numToMove) are moved towards the end of the receiver by 2 .
 then does (self at: destOffset put: valOne ; at: destOffset+1 put: valTwo ).
 The receiver must be a small object;

 Generates an error if any of the arguments imply the receiver must be
 grown to be a large object, or if destOffset is beyond the end of
 the receiver."

<primitive: 1032>
| newSize |
destOffset _validateClass: SmallInteger .
numToMove _validateClass: SmallInteger .
(destOffset < 1)   ifTrue:[ self _errorIndexOutOfRange: destOffset] .
(numToMove < 1)   ifTrue:[ self _errorIndexOutOfRange: destOffset] .
newSize := self namedSize +  destOffset + 2  +  numToMove .
newSize > 2034 "virtual machine constant OBJ_OOPS_SIZE" ifTrue:[
  self _error: #objErrMaxSize args:{ 2034 "virtual machine constant" . newSize }.
].
self _primitiveFailed: #_insertAt:value:value:numToMoveDown
     args: { destOffset . valOne . valTwo . numToMove }
%

category: 'Copying'
method: Array
_insertAt: destOffset
value: valOne
value: valTwo
value: valThree
numToMoveDown: numToMove

"Intended for use in manipulating objects which are nodes of B-trees or
 similar structures.  Does space-efficient tranlogging of the insertion.

 Inserts objects into receiver at destOffset, with space-efficient tranlogging
 of the insert.

 The indexable instance variables of the receiver from destOffset to
 (destOffset + numToMove) are moved towards the end of the receiver by 3 .
 then does (self at: destOffset put: valOne ; at: destOffset+1 put: valTwo ;
                at: destOffset+2 put: valThree )
 The receiver must be a small object;

 Generates an error if any of the arguments imply the receiver must be
 grown to be a large object, or if destOffset is beyond the end of
 the receiver."

<primitive: 1033>
| newSize |
destOffset _validateClass: SmallInteger .
numToMove _validateClass: SmallInteger .
(destOffset < 1)   ifTrue:[ self _errorIndexOutOfRange: destOffset] .
(numToMove < 1)   ifTrue:[ self _errorIndexOutOfRange: destOffset] .
newSize := self namedSize +  destOffset + 3  +  numToMove .
newSize > 2034 "virtual machine constant OBJ_OOPS_SIZE" ifTrue:[
  self _error: #objErrMaxSize args:{ 2034 "virtual machine constant" . newSize }.
].
self _primitiveFailed: #_insertAt:value:value:value:numToMoveDown
     args: { destOffset . valOne . valTwo . valThree . numToMove }
%

category: 'Copying'
method: Array
_insertAt: destOffset
value: valOne
value: valTwo
value: valThree
value: valFour
numToMoveDown: numToMove

"Intended for use in manipulating objects which are nodes of B-trees or
 similar structures.  Does space-efficient tranlogging of the insertion.

 Inserts objects into receiver at destOffset, with space-efficient tranlogging
 of the insert.

 The indexable instance variables of the receiver from destOffset to
 (destOffset + numToMove) are moved towards the end of the receiver by 4 .
 then does (self at: destOffset put: valOne ; at: destOffset+1 put: valTwo ;
         at: destOffset+2 put: valThree ;  at: destOffset+3 put: valFour )
 The receiver must be a small object;

 Generates an error if any of the arguments imply the receiver must be
 grown to be a large object, or if destOffset is beyond the end of
 the receiver."

<primitive: 1034>
| newSize |
destOffset _validateClass: SmallInteger .
numToMove _validateClass: SmallInteger .
(destOffset < 1)   ifTrue:[ self _errorIndexOutOfRange: destOffset] .
(numToMove < 1)   ifTrue:[ self _errorIndexOutOfRange: destOffset] .
newSize := self namedSize +  destOffset + 4  +  numToMove .
newSize > 2034 "virtual machine constant OBJ_OOPS_SIZE" ifTrue:[
  self _error: #objErrMaxSize args:{ 2034 "virtual machine constant" . newSize }.
].
self _primitiveFailed: #_insertAt:value:value:value:value:numToMoveDown
     args: { destOffset . valOne . valTwo . valThree . valFour . numToMove }
%

category: 'Copying'
method: Array
_insertAt: destOffset
value: valOne
value: valTwo
value: valThree
value: valFour
value: valFive
numToMoveDown: numToMove

"Intended for use in manipulating objects which are nodes of B-trees or
 similar structures.  Does space-efficient tranlogging of the insertion.

 Inserts objects into receiver at destOffset, with space-efficient tranlogging
 of the insert.

 The indexable instance variables of the receiver from destOffset to
 (destOffset + numToMove) are moved towards the end of the receiver by 5 .
 then does (self at: destOffset put: valOne ; at: destOffset+1 put: valTwo ;
         at: destOffset+2 put: valThree ;  at: destOffset+3 put: valFour;
         at: destOffset+4 put: valFive  )
 The receiver must be a small object;

 Generates an error if any of the arguments imply the receiver must be
 grown to be a large object, or if destOffset is beyond the end of
 the receiver."

<primitive: 1035>
| newSize |
destOffset _validateClass: SmallInteger .
numToMove _validateClass: SmallInteger .
(destOffset < 1)   ifTrue:[ self _errorIndexOutOfRange: destOffset] .
(numToMove < 1)   ifTrue:[ self _errorIndexOutOfRange: destOffset] .
newSize := self namedSize +  destOffset + 5  +  numToMove .
newSize > 2034 "virtual machine constant OBJ_OOPS_SIZE" ifTrue:[
  self _error: #objErrMaxSize args:{ 2034 "virtual machine constant" . newSize }.
].
self _primitiveFailed: #_insertAt:value:value:value:value:value:numToMoveDown
     args: { destOffset . valOne . valTwo . valThree . valFour . valFive . numToMove }
%

category: 'Copying'
method: Array
_largeInsertAt: destOffset
from: anArray
fromStart: startOffset
fromEnd: endOffset
numToMoveDown: numToMove

"Intended for use in manipulating objects which are nodes of B-trees or
 similar structures. Similar to primitive 605, except that with this
 primitive objects will be grown to large objects if necessary.
 Does not do optimized tranlogging.

 Inserts the portion of anArray from startOffset to endOffset
 into the receiver beginning at destOffset.

 If anArray is nil, then  (endOffset - startOffset + 1) nils are
 inserted into the receiver.

 The indexable instance variables of the receiver from destOffset to
 (destOffset + numToMove) are moved towards the end of the receiver by
 the amount (endOffset - startOffset + 1).
 The receiver may be a small or large object.

 If the receiver must be grown, it will be grown by 10 times the number
 of OOPs inserted up to a max of 500 oops.

 Generates an error if destOffset is beyond the end of the receiver."

<primitive: 552 >

anArray ~~ nil ifTrue:[ anArray _validateClass: Array ].

(destOffset < 1 or:[ destOffset > self _basicSize] )
   ifTrue:[ self _errorIndexOutOfRange: destOffset] .
(startOffset < 1) ifTrue:[ self _errorIndexOutOfRange: startOffset] .
(endOffset < 1)   ifTrue:[ self _errorIndexOutOfRange: endOffset] .
(numToMove < 0)   ifTrue:[ self _errorIndexOutOfRange: numToMove] .
(endOffset < startOffset) ifTrue:[ self _errorIndexOutOfRange: endOffset ].
(anArray ~~ nil and: [startOffset > anArray _basicSize]) ifTrue:[ anArray _errorIndexOutOfRange: startOffset].
(anArray ~~ nil and: [endOffset > anArray _basicSize]) ifTrue:[ anArray _errorIndexOutOfRange: endOffset].

self _primitiveFailed: #_largeInsertAt:from:fromStart:fromEnd:numToMoveDown:
     args: { destOffset . anArray . startOffset . endOffset . numToMove }
%

! Class extensions for 'Association'

!		Class methods for 'Association'

removeallmethods Association
removeallclassmethods Association

category: 'Instance Creation'
classmethod: Association
newWithKey: aKey value: aValue

"Returns a new Association with the argument aKey as its key and with aValue as
 its value."

^ super new key: aKey value: aValue
%

!		Instance methods for 'Association'

category: 'Comparing'
method: Association
< anObject

"Returns true if the key of the receiver collates before anObject.
 Returns false otherwise."

^ key == self
  ifTrue: [ false ]
  ifFalse: [ anObject > self key ]
%

category: 'Comparing'
method: Association
<= anObject

"Returns true if the key of the receiver collates before anObject, or if the
 key of the receiver is equivalent to anObject.  Returns false otherwise."

^ key == self
  ifTrue: [ true ]
  ifFalse: [ anObject >= self key ]
%

category: 'Comparing'
method: Association
= anObject

"Returns true if (a) the receiver and anObject are both kinds of Association,
 (b) the receiver and anObject have equal keys and (c) the receiver and
 anObject have equal values.  Returns false otherwise."

(self == anObject) ifTrue: [^ true].
(anObject isKindOf: Association) ifFalse:[^ false].
^ (key = anObject key) and: [value = anObject value].
%

category: 'Comparing'
method: Association
> anObject

"Returns true if the key of the receiver collates after anObject.
 Returns false otherwise."

^ anObject < self key
%

category: 'Comparing'
method: Association
>= anObject

"Returns true if the key of the receiver collates after anObject, or if the
 key of the receiver is equivalent to anObject.  Returns false otherwise."

^ anObject <= self key
%

category: 'Clustering'
method: Association
clusterDepthFirst

"This method clusters the receiver, its key, and its value in
 depth-first order.  It returns true if the receiver has already
 been clustered during the current transaction, false otherwise."

  self cluster
  ifTrue:
    [ ^ true ]
  ifFalse:
    [ key clusterDepthFirst.
      value clusterDepthFirst.
      ^ false
    ].
%

category: 'Comparing'
method: Association
hash

"Returns an Integer hash code for the receiver."

^ key hash
%

category: 'Testing'
method: Association
isAssociation
  ^ true
%

category: 'Formatting'
method: Association
isInPrintingRecursionSet: anIdentitySet
	"Answer whether we are recursing through the receiver.

	 For associations, there are pathological cases one can create,
	 that we have chosen to portray in their logical representation.
	 Since some dictionaries use associations and others do not,
	 portray the results as if they all do.

	 In addition to the fast identity check, include a slower equality check."

	^(super isInPrintingRecursionSet: anIdentitySet)
		or: [anIdentitySet includesValue: self]
%

category: 'Accessing'
method: Association
key

"Returns the value of the receiver's key."

^key
%

category: 'Updating'
method: Association
key: aKey

"Sets the object aKey as the key of the receiver."

key := aKey
%

category: 'Updating'
method: Association
key: aKey value: aValue

"Sets the object aKey as the key of the receiver, and the object aValue as the
 value of the receiver."

key := aKey.
value := aValue
%

category: 'Formatting'
method: Association
printNonRecursiveRepresentationOn: aStream recursionSet: anIdentitySet
	"Put a displayable representation of the receiver on the given stream
	 while avoiding recursion from object reference loops."

	key printOn: aStream recursionSet: anIdentitySet.
	aStream nextPutAll: '->'.
	value printOn: aStream recursionSet: anIdentitySet
%

category: 'Formatting'
method: Association
printOn: aStream
	"Put a displayable representation of the receiver on the given
	 stream. If the key or the value is identical to the receiver then
	 avoid an infinite recursion. (There is currently no general way to
	 limit these kinds of recursions. See Bug #16259)"

	self printNonRecursiveOn: aStream
%

category: 'Accessing'
method: Association
value

"Returns the value portion of the receiver."

^value
%

category: 'Updating'
method: Association
value: aValue

"Sets the object aValue as the value of the receiver."

value := aValue
%

category: 'Indexing Support'
method: Association
_idxValue

	^self value
%

category: 'Accessing'
method: Association
_value

" Returns the value portion of the receiver.
  Faster than #value  in Gs64 v3.0 since send of #value
  assumes ExecBlock is most likely receiver. "

^value
%

category: 'Updating'
method: Association
_value: aValue

" Updates the value portion of the receiver.
  Faster than #value:  in Gs64 v3.0 since send of #value:
  assumes ExecBlock is most likely receiver. "

value := aValue
%

category: 'Comparing'
method: Association
~= anObject

"Returns true if one or more of the conditions specified in #= method are
 not satisfied.  Returns false otherwise."

^ (self = anObject) == false.
%

! Class extensions for 'Behavior'

!		Class methods for 'Behavior'

removeallmethods Behavior
removeallclassmethods Behavior

category: 'Instance Creation'
classmethod: Behavior
new

"Disallowed.  To create a new Class or metaclass, use
 Class | subclass:instVarNames:... instead."

self shouldNotImplement: #new .
^ nil
%

category: 'Instance Creation'
classmethod: Behavior
new: anInteger

"Disallowed.  To create a new Class or metaclass, use
 Class | subclass:instVarNames:... instead."

self shouldNotImplement: #new: .
^ nil
%

!		Instance methods for 'Behavior'

category: 'Modifying Classes'
method: Behavior
addInstVar: aString

"Adds a new instance variable named aString to the receiver.  The argument
 aString must be a valid GemStone Smalltalk identifier and must be distinct from
 the names of all other instance variables previously defined for the receiver,
 its superclasses, and its subclasses.

 The new instance variable becomes the last named instance variable in the
 receiver, and is inserted at the appropriate position in each of the
 receiver's subclasses, to preserve the rules for inheritance of instance
 variables.  If, for any of the receiver's subclasses, the new instance
 variable is not the last named instance variable, then all instance methods
 for that subclass are recompiled using the symbol list of the current user.
 If an error occurs during recompilation of methods, the new instance
 variable will have been added to the receiver and all of its subclasses, but
 some methods in some subclasses will not have been recompiled.

 To successfully invoke this method, the receiver must meet one of these two
 conditions:

 * The receiver and all of its subclasses must be modifiable.
 * The receiver must disallow subclasses and must have no unnamed instance
   variables."

| newOffset theSymbol |
self _validatePrivilege ifTrue:[
  "3.2, no error check for NSCs"
  (self isBytes) ifTrue:[ ^ self _error: #rtErrInstvarAddToBytes ] .
  theSymbol:= aString asSymbol .
  theSymbol validateIsIdentifier .
  (self _validateNewNamedInstVar: theSymbol ) ifFalse:[ ^ nil ].

  newOffset := self instSize + 1 .  "the offset of the new instance variable"
  self _insertNamedInstVar: theSymbol atOffset: newOffset .
  self _recompileMethodsAfterNewIvOffset: newOffset .
]
%

category: 'Accessing Variables'
method: Behavior
allClassVarNames

"Returns an Array of Symbols, consisting of the names of the
 smalltalk class variables addressable by this class,
 including those inherited from superclasses.  Contrast with classVarNames."

| result currClass |

result:= { } .
currClass:= self.
[ true ] whileTrue:[
  currClass == nil ifTrue:[ ^ result ].
  result insertAll: (currClass classVarNames) at: 1.
  currClass:= currClass superClass .
]
%

category: 'Listing Instances'
method: Behavior
allInstances

"Returns an Array that contains all instances of the receiver.
 Note: This method scans the entire GemStone repository, and may therefore
 take some time to execute.

 Large result sets may cause out-of-memory issues. To avoid problems,
 use Repository >> allInstances:, which returns a GsBitmap that does not
 require keeping objects in temporary object space.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData."

^ (SystemRepository listInstances: { self }) at: 1.
%

category: 'Accessing Variables'
method: Behavior
allInstVarNames

"Returns an Array of Symbols, consisting of the names of all the receiver's
 instance variables, including those inherited from superclasses.  The ordering
 of the names in the Array follows the ordering of the superclass hierarchy;
 that is, instance variable names inherited from Object are listed first, and
 those peculiar to the receiver are last."

^ Array withAll: instVarNames
%

category: 'Modifying Classes'
method: Behavior
allowSubclasses

"Allows creation of subclasses of a class."

format := (format bitOr:32) bitXor: 32 .  "clear bit 32"
%

category: 'Accessing the Method Dictionary'
method: Behavior
allSelectors

^ self allSelectorsForEnvironment: 0
%

category: 'Accessing the Method Dictionary'
method: Behavior
allSelectorsForEnvironment: envId
"Returns an Array of Symbols, consisting of all of the message
 selectors that instances of the receiver can understand, including
 those inherited from superclasses.  For keyword messages, the
 Symbol includes each of the keywords, concatenated together."

 | result currClass |
 result:= IdentitySet new .
 currClass:= self.
 [ currClass == nil ] whileFalse: [
     result _addAll: (currClass selectorsForEnvironment: envId ) forReplay: false .
     currClass:= currClass superclassForEnv: envId .
 ].
 ^ Array withAll: result
%

category: 'Accessing Variables'
method: Behavior
allSharedPools

"Returns an Array of pool dictionaries used by this class and its superclasses.
 Contrast with sharedPools."

| result currClass |

result:= { } .
currClass:= self.
[ true ] whileTrue:[
  currClass == nil ifTrue:[ ^ result ].
  result insertAll: (currClass sharedPools) at: 1.
  currClass:= currClass superClass
].

^ result
%

category: 'Accessing the Class Hierarchy'
method: Behavior
allSuperClasses

"Returns an Array of the environment 0 superclasses of the receiver,
 beginning with the immediate superclass, and excluding the receiver."

| result cls |
result:= { } .
cls := self superClass .
[ true ] whileTrue:[
  cls == nil ifTrue:[ ^ result ].
  result add: cls .
  cls := cls superClass.
]
%

category: 'Enumerating'
method: Behavior
allSuperClassesDo: aBlock

"Evaluates aBlock with each of the receiver's superclasses as
 the argument, beginning with the immediate superclass."

| currClass |

currClass:= self superClass .
[ true ] whileTrue:[
  currClass ifNil:[ ^ self ].
  aBlock value: currClass .
  currClass:= currClass superClass
]
%

category: 'Accessing the Class Hierarchy'
method: Behavior
allSuperClassesForEnv: envId

"Returns an Array of the superclasses of the receiver, in the specified
 environment beginning with the immediate superclass, and excluding the receiver."

| result cls |
result:= { } .
cls := self superclassForEnv: envId .
[ true ] whileTrue:[
  cls == nil ifTrue:[ ^ result ].
  result add: cls .
  cls := cls superclassForEnv: envId .
]
%

category: 'Queries'
method: Behavior
areInstancesSpecial

"Returns whether instances of this class have their state encoded in their
 identities."  "fix 48559"

^ (format bitAnd: 16r3) == 3  "inline _isSpecial"
%

category: 'Formatting'
method: Behavior
asString

"Returns a String that indicates the class of the receiver."
| n |
n := self _name .  "avoid use of squeak Object>>name"
n ifNil:[ ^ '(unNamedClass)' copy ].
n _stringCharSize == 1 ifTrue:[
  n class == Symbol ifTrue:[
    ^ String withAll: n
  ].
  ^ n
].
^ DoubleByteString withAll: n
%

category: 'Formatting'
method: Behavior
asUnicodeString

"Returns a Unicode string that indicates the class of the receiver."
| n |
n := self _name .  "avoid use of squeak Object>>name"
n ifNil:[ ^ Unicode7 withAll: '(unNamedClass)' ].
n _stringCharSize == 1 ifTrue:[
  ^ Unicode7 withAll: n .
].
^ Unicode16 withAll: n  "Unicode representation of DoubleByteSymbol"
%

category: 'Instance Creation'
method: Behavior
basicNew

"Creates a new, uninitialized instance of the receiver."

^self _basicNew
%

category: 'Instance Creation'
method: Behavior
basicNew: anInteger

"Creates a new, uninitialized instance of the receiver with the given
 number of indexed instance variables."

^self _basicNew: anInteger
%

category: 'Indexing Support'
method: Behavior
btreeLeafNodeClass

"Returns the class of BtreeLeafNode to create for an equality index whose last
 object along the path is an instance of the receiver."

| result |
(RangeEqualityIndex isBasicClass: self)
    ifTrue: [ result := self indexManager btreeBasicLeafNodeClass]
    ifFalse: [ result := self indexManager btreeLeafNodeClass ].
^ result
%

category: 'Accessing the Method Dictionary'
method: Behavior
canUnderstand: aSelector

"Returns true if the receiver can respond to the message indicated by
 aSelector, returns false otherwise.  The selector (a String) can be in the
 method dictionary of the receiver or any of the receiver's superclasses."

| aSymbol |
aSymbol := Symbol _existingWithAll: aSelector .
aSymbol ifNil:[ ^ false ].

^ (self whichClassIncludesSelector: aSymbol) ~~ nil
%

category: 'Accessing Categories'
method: Behavior
categoryNames

"Returns an Array of Symbols.  The elements of the Array are the
 receiver's environment 0 category names (excluding names inherited from superclasses)."

 | set |
 set := IdentitySet new .
 self env: 0 categorysDo:[ :categName :selectors | set add: categName ].
 ^ Array withAll: set
%

category: 'Accessing the Method Dictionary'
method: Behavior
categoryOfSelector: aSelector
 ^ self categoryOfSelector: aSelector environmentId: 0
%

category: 'Accessing the Method Dictionary'
method: Behavior
categoryOfSelector: aSelector environmentId: envId

"Returns a Symbol which is the name of the category for the specified
 selector, or nil if the selector was not found in any category."

| aSymbol |
aSymbol := Symbol _existingWithAll: aSelector .
aSymbol ifNil:[ ^ nil ].
self env: envId categorysDo:[ :categName :selectors |
  (selectors includesIdentical: aSymbol) ifTrue:[ ^ categName ]
].
^ nil
%

category: 'Accessing Categories'
method: Behavior
categorysDo: aBlock
  "evaluates aBlock for each category in environment 0.

   aBlock should be a two argument block
   expecting the args  categoryNameSymbol ,   selectorsSet,
   If the package manager is active in the current session,
   aBlock may be invoked more than once for each category name."

  self env: 0 categorysDo: aBlock
%

category: 'Accessing Variables'
method: Behavior
classVarAt: aString

| key |
"Returns the value of the class variable named aString ."

classVars ifNil:[ ^ self _error: #rtErrKeyNotFound args: { aString } ].
key := Symbol _existingWithAll: aString .
key ifNil:[ ^ self _error: #rtErrKeyNotFound args: { aString } ].
^ classVars at: key .
%

category: 'Accessing Variables'
method: Behavior
classVarAt: aString otherwise: defaultValue

| key |
"Returns the value of the class variable named aString ."

classVars ifNil:[ ^ defaultValue ].
key := Symbol _existingWithAll: aString .
key ifNil:[ ^ defaultValue ].
^ classVars at: key otherwise: defaultValue
%

category: 'Accessing Variables'
method: Behavior
classVarNames

"Returns an Array of Symbols naming the class variables defined by this class.
 Inherited class variables are not included; contrast with allClassVarNames."

| cvars |
(cvars := classVars) ifNil:[ ^ { } ].
^ cvars keys asArray
%

category: 'Clustering'
method: Behavior
clusterBehavior
	"This method clusters, in depth-first order, the parts of the receiver required
 for executing GemStone Smalltalk code (the receiver and its method dictionary).
 Returns true if the receiver has already been clustered during the current
 transaction; returns false otherwise.

 Note that methods in the VM's temporary object memory that were
 loaded for execution in this session cannot be clustered by this session.

 It is recommended that when several classes are being clustered in a
 transaction, send clusterBehavior to all classes to be clustered, then send
 clusterDescription."

	self cluster ifTrue: [^true].
	self isMeta
		ifFalse:
			[| arr |
			(arr := instVarNames) ~~ #() ifTrue: [arr cluster]].
	methDicts _isArray ifTrue: [methDicts cluster].
	self persistentMethodDictsDo:
			[:aDict |
			aDict cluster.
			"note GsNmethod>>clusterDepthFirst clusters execution related parts
       to current bucket and other(debug) info info to bucket 5"
			aDict values do: [:meth | meth clusterDepthFirst]].
	self persistentNameSpacesDo: [:aNs | aNs cluster].
	classVars
		ifNotNil:
			[:cvs |
			cvs cluster ifFalse: [cvs associationsDo: [:assoc | assoc cluster]]].
	^false
%

category: 'Clustering'
method: Behavior
clusterBehaviorExceptMethods: aCollectionOfMethodNames

"This method allows you to cluster the receiver more efficiently by omitting
 infrequently-used methods from the clustering.  The methods that you designate
 as aCollectionOfMethodNames will not be clustered with the receiver.  Thus,
 the methods that are frequently used will be packed more densely.  Returns
 true if the receiver has already been clustered during the current
 transaction; returns false otherwise.

 This method works by first clustering all methods named into the max cluster
 bucket, preventing them from showing up in the default cluster bucket.  It
 then uses the standard method to cluster behavior."

| savedBucket otherBucket systm |
systm := System .
savedBucket := systm clusterBucket.
otherBucket := 6 .
systm clusterBucket: otherBucket.
self persistentMethodDictsDo:[ :aDict |
   aDict keysAndValuesDo:[ :aSelector :aMethod |
     (aCollectionOfMethodNames includesValue: aSelector )
       ifTrue: [ aMethod clusterDepthFirst ].
   ].
].
systm clusterBucket: savedBucket.
^ self clusterBehavior
%

category: 'Clustering'
method: Behavior
clusterDescription

"This method clusters, in depth-first order, those instance variables in the
 receiver that are not expected to be accessed heavily during method execution,
 specifically the  poolDictionaries, and categories dictionaries.

 (The receiver itself is not clustered.)  Returns true if the description
 already been clustered during the current transaction; returns
 false otherwise.

 It is recommended that when several classes are being clustered in a
 transaction, send clusterBehavior to all classes to be clustered, then send
 clusterDescription."

| result |
result := false .
self isMeta ifFalse:[
  poolDictionaries ifNotNil:[ poolDictionaries cluster] .
].
result ifFalse:[
  categorys _isArray ifTrue:[
    result := categorys cluster .
    categorys do:[ :aDict | aDict clusterDepthFirst ].
  ] ifFalse:[
    categorys ifNotNil:[ result := categorys clusterDepthFirst ].
  ]
].
^ result
%

category: 'Updating the Method Dictionary'
method: Behavior
compileAccessingMethodsFor: anArrayOfSymbols

"This method create accessing methods for reading and modifying instance and
 other variables in instances of the receiver.  Each element of anArrayOfSymbols
 must be an instance variable, class variable, or pool dictionary variable in
 the receiver.  For each variable 'x' in the Array, two methods are created: 'x'
 reads the variable, in the category 'Accessing', and 'x:newValue' modifies the
 variable, in the category 'Updating'.

 To create class methods to access class variables, class instance variables,
 or pool variables, the message must be sent to the class of the class.

 The new methods have environmentId == 0 .

 Returns the receiver.  Generates an error if any element of anArrayOfSymbols
 is not an instance variable, class variable, or pool variable of the
 receiver."

| allVarNames nvName compileBlk |

self _validatePrivilege ifFalse:[ ^ nil ].
nvName := 'newValue'.
allVarNames := self allInstVarNames collect:[:s | s asString ].
[allVarNames includesValue: nvName ] whileTrue:[ nvName := 'z' , nvName ].
compileBlk := [ :src :cat :varName  |
    [ self compileMethod: src
       dictionaries:  #()
       category: cat
       environmentId: 0
    ] onException: CompileError do:[:ex |
      ^ self _error: #classErrNotAVar args: { varName }
    ]
  ].
anArrayOfSymbols do: [ :var | | getSrc putSrc lf tab |
  lf := Character lf.
  tab := Character tab.
  (getSrc := String new) add: var; add: lf; add: tab; add: '^' ; add: var; add: lf.
  (putSrc := String new) add: var; add: ': ' ; add: nvName; add: lf;
	add: tab; add: var; add: ' := '; add: nvName; add: lf.
  compileBlk value: getSrc value: #Accessing value: var.
  compileBlk value: putSrc value: #Updating  value: var.
]
%

category: 'Accessing the Method Dictionary'
method: Behavior
compiledMethodAt: aSelector
  ^ self compiledMethodAt: aSelector environmentId: 0
%

category: 'Accessing the Method Dictionary'
method: Behavior
compiledMethodAt: aSelector environmentId: envId

"Returns the compiled method associated with the argument aSelector (a String).
 The argument must be a selector in the receiver's method dictionary; if it is
 not, this method generates an error."

| aMeth |
aMeth := self compiledMethodAt: aSelector environmentId: envId otherwise: nil.
aMeth == nil ifTrue:[ self _error: #rtErrKeyNotFound args: { aSelector } ].
^ aMeth
%

category: 'Accessing the Method Dictionary'
method: Behavior
compiledMethodAt: aSelector otherwise: notFoundVal
  ^ self compiledMethodAt: aSelector environmentId: 0 otherwise: notFoundVal
%

category: 'Updating the Method Dictionary'
method: Behavior
compileMethod: sourceString

"If there are no errors, this adds the resulting compiled method to the
 receiver's method dictionary and returns that method,
 otherwise signals a CompileError .
 A CompileWarning may be signaled, after adding the new method
 to a receiver's method dictionary."

^ self compileMethod: sourceString
   dictionaries: GsCurrentSession currentSession symbolList
   category: '(as yet unclassified)'
   environmentId: 0
%

category: 'Updating the Method Dictionary'
method: Behavior
compileMethod: sourceString
dictionaries: aSymbolList
category: aCategoryString

"May be deprecated in the future.
 Returns nil, or an Array of error descriptors"

^ [ self compileMethod: sourceString
       dictionaries: aSymbolList
       category: aCategoryString
       environmentId: 0 .
    nil
  ] onException: CompileError do:[:ex | ex errorDetails ]
%

category: 'Updating the Method Dictionary'
method: Behavior
compileMethod: sourceString
dictionaries: aSymbolList
category: categorySymbol
intoMethodDict: aMethodDict
intoCategories: aCategDict

  ^self
    compileMethod: sourceString
    dictionaries: aSymbolList
    category: categorySymbol
    intoMethodDict: aMethodDict
    intoCategories: aCategDict
    environmentId: 0

%

category: 'Updating the Method Dictionary'
method: Behavior
compileMethod: sourceString
dictionaries: aSymbolList
category: categorySymbol
intoMethodDict: aMethodDict
intoCategories: aCategDict
environmentId: environmentId

"Compiles sourceString as a method for the receiver into the method category
 categorySymbol, using the symbol list aSymbolList. If the compilation succeeds,
 returns the GsNMethod produced by the compilation.
 If the compilation fails, signals a CompileError .
 If the compilation has warnings, signals a CompileWarning, after the GsNMethod
 has been generated and installed in any method dictionaries.

 aMethodDict may be nil, in which case the resulting method is added to the
 receiver's persistent method dictionary and aCategDict is ignored. If
 aMethodDict is not nil it is added to aMethodDict instead of to the receiver's
 method dictionary.  If aMethodDict and aCategDict are both not nil, category
 and selector are added to aCategDict.
 This is used to add methods to per-session dictionaries.

 The caller is responsible for setting the current GsObjectSecurityPolicy to
 match the receiver's GsObjectSecurityPolicy if desired.

 environmentId must be a SmallInteger >= 0 and <= 255 .
 0 denotes the base Smalltalk image.  1 was reserved for use by Ruby .

 You must have code modification privilege to execute this method.
 "

^ self _checkCompileResult:( self _primitiveCompileMethod: sourceString
		symbolList: aSymbolList category: categorySymbol
             oldLitVars: nil intoMethodDict: aMethodDict intoCategories: aCategDict
	     environmentId: environmentId )
        source: sourceString suppressCompileWarning: false
%

category: 'Updating the Method Dictionary'
method: Behavior
compileMethod: sourceString
dictionaries: aSymbolList
category: categorySymbol
intoMethodDict: aMethodDict
intoCategories: aCategDict
intoPragmas: aPragmasArray
environmentId: environmentId

"pragmasArray is ignored.  use GsNMethod>>pragmas or GsNMethod>>_pragmasArray
 to fetch the pragmas for a method.

 Should eventually be Deprecated."

^ self
  compileMethod: sourceString
  dictionaries: aSymbolList
  category: categorySymbol
  intoMethodDict: aMethodDict
  intoCategories: aCategDict
  environmentId: environmentId
%

category: 'Browser Methods'
method: Behavior
compileMissingAccessingMethods

"Creates accessing and updating methods for all instance variables that do not
 already have such methods.  Sent to a class, creates accessor methods for class
 instance variables that do not have them."

| argName newLine tab allVarNames varNames |

self _validatePrivilege ifTrue:[
  argName := 'newValue' .
  allVarNames := self allInstVarNames collect:[:s | s asString ].
  [allVarNames includesValue: argName] whileTrue: [
    argName := 'z' , argName .
  ].
  newLine := Character lf asString.
  tab := Character tab asString.
  varNames := self instVarNames.
  varNames accompaniedBy: self do: [ :me :var |
    (me includesSelector: var ) ifFalse: [
      me compileMethod: (var , newLine , tab , '^' , var , newLine)
	    dictionaries:  #()
	    category: #Accessing environmentId: 0 .
    ].
    (me includesSelector: var , $: ) ifFalse: [
      me compileMethod: (var , ': ' , argName , newLine , tab , var , ' := ' , argName , newLine)
	    dictionaries:  #()
	    category: #Updating environmentId: 0 .
    ]
  ]
]
%

category: 'Copying'
method: Behavior
copy

"Returns the receiver. Copies of classes and metaclasses are not made."

^ self.
%

category: 'Browsing'
method: Behavior
copyMethodsFrom: sourceClass dictionaries: dicts

"Copies all instance and class methods from the sourceClass.  Returns an Array
 of methods in the source class which failed to compile in this class.  Some of
 them might be class methods.  The Array is empty if no methods failed to
 compile."

| failed srccls targcls envId |
self == sourceClass ifTrue:[
  "because iteration is directly over the source's categories dictionaries"
  ^ self error: 'source of copy must not be self'
].
self _validatePrivilege ifFalse:[ ^ nil ].
failed := { } .

"Copy class and instance methods"
envId := 0 .
1 to: 2 do: [ :i |
  i == 1 ifTrue:[ srccls := sourceClass.  targcls := self ]
        ifFalse:[ srccls := sourceClass class.  targcls := self class ].
  srccls categorysDo:[ :cat :sels |
     sels do: [ :sel | | oldMeth |
        [
          targcls compileMethod: (oldMeth := srccls sourceCodeAt: sel environmentId: envId)
                    dictionaries: dicts category: cat environmentId: envId .
        ] onException: CompileError do:[:ex |
          failed add: oldMeth
        ].
     ].
  ].
].

^failed.
%

category: 'Versions'
method: Behavior
currentVersion
  ^ self
%

category: 'Modifying Classes'
method: Behavior
disallowSubclasses

"Disallows creation of subclasses of a class.  If the receiver is not
 modifiable, this method generates an error.  If the receiver is modifiable and
 already has subclasses, this method generates an error."

self _validatePrivilege ifTrue:[
  self isModifiable ifFalse:[ ^ self validateIsModifiable ].
  self _subclasses size ~~ 0 ifTrue:[ ^ self _error: #rtErrAlreadyHasSubclasses ].
  format := format bitOr: 32
]
%

category: 'Accessing Categories'
method: Behavior
env: environmentId baseCategorysDo: aBlock
  "evaluates aBlock for each method category of receiver
   in specified environment. Returns the receiver.

   aBlock should be a two argument block
   expecting the args  categoryNameSymbol ,   selectorsSet.
   aBlock will be invoked once for each category in the receiver.
   The iteration is directly over the receiver's categories."

  | cats |
  cats := self _baseCategorys: environmentId .
  cats ifNotNil:[ cats keysAndValuesDo: aBlock ].
%

category: 'Accessing Categories'
method: Behavior
env: envId methodsDo: aBlock
  "evaluates aBlock for each method in receiver's method dictionaries
   for specified environment. Returns the receiver.

   aBlock should be a two argument block
   expecting the args  selectorSymbol ,   aGsNMethod ."

  (self persistentMethodDictForEnv: envId ) ifNotNil:[:dict | dict keysAndValuesDo: aBlock ]
%

category: 'Accessing Categories'
method: Behavior
env: envId unifiedCategoriesDo: aBlock
  "evaluates aBlock for each method category of receiver
   in specified environment. Returns the receiver.

   aBlock should be a two argument block
   expecting the args  categoryNameSymbol ,   selectorsSet.
   aBlock will be invoked once for each category in the receiver.
   The iteration is done over a copy of the receiver's categories,
   however the selectorSets are not copied ."

  (self _unifiedCategorys: envId ) keysAndValuesDo: aBlock
%

category: 'Updating the Method Dictionary'
method: Behavior
extractSelector: sourceString
  "Extract selector from source of a method and return a Symbol, 
   or signal a CompileError."
  ^ self _checkCompileResult:( self _primExtractSelector: sourceString )
        source: sourceString suppressCompileWarning: true
%

category: 'Listing Instances'
method: Behavior
fastAllInstances

"Returns an Array that contains all instances of the receiver.
 Note: This method scans the entire GemStone repository, and may therefore
 take some time to execute.

 Large result sets may cause out-of-memory issues. To avoid problems,
 use Repository >> allInstances:, which returns a GsBitmap that does not
 require keeping objects in temporary object space.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData."

^ (SystemRepository fastListInstances: { self }) at: 1.
%

category: 'Accessing the Class Format'
method: Behavior
firstPublicInstVar

"Returns the index of the first publicly available instance variable storage
 location, whether or not a public instance variable has actually been
 defined."

"Currently, an instance variable is considered to be public if it is to be
 included when passivating the object through PassiveObject.  Being public has
 no relationship to whether or not accessing or updating methods are defined
 for the instance variable."

<primitive: 960>
self _primitiveFailed: #firstPublicInstVar
%

category: 'Accessing the Class Format'
method: Behavior
format

"Returns the value of the format instance variable."

^ format
%

category: 'Accessing the Class Format'
method: Behavior
hasPublicInstVars

"Returns true if the receiver has publicly-visible instance variables."

^ self instSize >= self firstPublicInstVar
%

category: 'Modifying Classes'
method: Behavior
immediateInvariant
	"If the receiver is not invariant, make it invariant and
 conditionally also make the receiver's instVarNames invariant.
 Also clears the subclasses instVar .

 After this method executes,  addInstVar: and removeInstVar: will
 no longer be allowed for the receiver."

	self isInvariant
		ifFalse:
			[self _validatePrivilege ifFalse: [^nil].
			" Gs64 v3.0 and later,
			We no longer recompile methods here, the assumption is that removeInstVar:
			has recompiled methods as needed or has failed and invoked
      		System _disallowCommitClassModFailure."
			self subclassesDisallowed
				ifFalse:
					["only make the instVarNames collection invariant if we
					are not permitting possible dynamic addition of named instance variables."
					instVarNames immediateInvariant].
			"clear the subclasses class instance variable to prevent concurrency problems"
			self _subclasses ~~ nil ifTrue: [self _subclasses: nil].
			super immediateInvariant.	"make self invariant"
			"_codeChangedForEnv: not needed, since no instances can exist yet"
			self _refreshClassCache: false]
%

category: 'Accessing the Class Format'
method: Behavior
implementationFormat

"Returns the three least-significant bits of the receiver's format instance
 variable.  The values of those bits mean the following:

 0   OOP       non-indexable
 1   Byte      non-indexable
 2   NSC       non-indexable
 3   Special   non-indexable
 4   OOP       indexable
 5   Byte      indexable"

^ format bitAnd: 16r7
%

category: 'Accessing Categories'
method: Behavior
includesCategory: aString

"Returns true if aString is equivalent to the name of a category in the
 receiver, false otherwise."

^ self includesCategory: aString environmentId: 0
%

category: 'Accessing Categories'
method: Behavior
includesCategory: aString environmentId: envId

"Returns true if aString is equivalent to the name of a category in the
 receiver, false otherwise."

| aSym |
aSym := Symbol _existingWithAll: aString.
aSym ifNil:[ ^ false ].
self env: envId categorysDo:[ :name :selectors |
   name == aSym ifTrue:[ ^ true ].
].
^ false
%

category: 'Accessing the Method Dictionary'
method: Behavior
includesSelector: aString
  ^ self includesSelector: aString environmentId: 0
%

category: 'Accessing the Class Hierarchy'
method: Behavior
inheritsFrom: aClass

"Returns true if the argument aClass is on the receiver's
 superclass chain; returns false if it isn't."


(aClass isKindOf: Behavior)
ifFalse:
   [self _error: #rtErrBadArgKind args: { Class }].

(self isSubclassOf: aClass)
   ifTrue: [
     self == aClass
       ifTrue: [^false]
       ifFalse: [^true]
   ]
   ifFalse: [^false] .
   self _uncontinuableError " should never get here"
%

category: 'Accessing the Class Format'
method: Behavior
instancesDbTransient

  "Return true if the class format has the DbTransient bit set.
   See also  makeInstancesDbTransient ."

^ (format bitAnd: 16r1000) ~~ 0
%

category: 'Accessing the Class Format'
method: Behavior
instancesInvariant

"Returns true if instances of the receiver may not change value after they have
 been committed to GemStone.  Otherwise, returns false."

^ (format bitAnd: 16r8) ~~ 0
%

category: 'Accessing the Class Format'
method: Behavior
instancesNonPersistent

 "Returns true if instances may not be committed.
  See also makeInstancesNonPersistent ."

^ (format bitAnd: 16r800) ~~ 0
%

category: 'Accessing the Class Format'
method: Behavior
instSize

"Returns the number of named instance variables in the receiver, including all
 inherited instance variables."

^ instVarsInfo bitAnd: Class_numIvs_mask
%

category: 'Updating'
method: Behavior
instVarAt: anIndex put: aValue

anIndex == 7 ifTrue:[
  "Gs64 v3.0 structural update of the methDicts instVar disallowed"
   OffsetError  new _number: 2350; reason: #errNoStructuralUpdate ;
	  details:'methDicts instVar'; object: self ; signal .
   self _uncontinuableError.
].
anIndex == 9 ifTrue:[
  "Gs64 v3.0 structural update of the categories instVar disallowed"
   OffsetError  new _number: 2350; reason: #errNoStructuralUpdate ;
	details:'categorys instVar'; object: self ; signal .
   self _uncontinuableError.
].
^ super instVarAt: anIndex put: aValue
%

category: 'Accessing Variables'
method: Behavior
instVarNames

"Returns an Array of Symbols naming the instance variables defined by the
 receiver, but not including those inherited from superclasses.  Contrast
 with allInstVarNames."

| inheritedInstVars "an Array of the inherited instance variables"
  size              "the size of inherited instance variables"
  myInstVars        "an Array of all of the instance variables"
  supercls     |

(myInstVars := instVarNames) ifNil:[ ^ { } ].
(supercls := self superClass) ifNil: [ ^ myInstVars ].

inheritedInstVars := supercls _instVarNames.
(size := inheritedInstVars size) == myInstVars size ifTrue:[ ^{ } ].

"Assume that each inherited instance variable is added to the end of
 the Array result of allInstVarNames."

^ myInstVars copyFrom: size + 1 to: myInstVars size .
%

category: 'Testing'
method: Behavior
isBehavior

"Returns true if the receiver is a kind of Behavior, and returns false
 otherwise."

  ^true
%

category: 'Accessing the Class Format'
method: Behavior
isBytes

"Returns true if instances of the receiver are byte objects.  Otherwise,
 returns false."

^ (format bitAnd: 16r3) == 1
%

category: 'Accessing the Class Format'
method: Behavior
isBytesOrSpecial

"Returns whether instances of the receiver are byte objects."

^ (format bitAnd: 1) == 1
%

category: 'Accessing the Class Format'
method: Behavior
isIndexable

"Returns true if instances of the receiver have indexed variables.
 Otherwise, returns false."

^ (format bitAnd: 16r4) ~~ 0 "that is, is indexable"
%

category: 'Modifying Classes'
method: Behavior
isModifiable
	"Returns true if the receiver may be modified (that is, if the receiver and its Array of
 instance variable names are both variant, and the receiver has a 'subclasses' class variable).
 Returns false otherwise."

	| ivn |
	self isInvariant ifTrue: [^false].
	((ivn := instVarNames) size ~~ 0 and: [ivn isInvariant]) ifTrue: [^false].
	^self _subclasses ~~ nil
%

category: 'Accessing the Class Format'
method: Behavior
isNonByteVarying

"Returns true if the instances of the receiver are not byte objects and have
 unnamed instance variables; returns false otherwise."

| bits |

bits := format bitAnd: 7.
^bits == 2 or: [bits == 4].
%

category: 'Accessing the Class Format'
method: Behavior
isNsc

"Returns true if instances of the receiver are non-sequenceable
 collections (UnorderedCollections).  Otherwise, returns false."

^ (format bitAnd: 16r3) == 2
%

category: 'Accessing the Class Format'
method: Behavior
isPointers

"Returns true if instances of the receiver are pointer objects.
 Otherwise, returns false."

^ (format bitAnd: 16r3) == 0
%

category: 'Accessing the Class Format'
method: Behavior
isProtected

"Returns true if instances of the receiver may not be accessed structurally
 through GemBuilder for C. "

^ (format bitAnd: 16r80) ~~ 0
%

category: 'Testing Inheritance'
method: Behavior
isSubclassOf: aClassHistory

"Returns true if the receiver is identical to or is a subclass of any class
 in aClassHistory; otherwise, returns false.

 nil is an allowed value for the argument.
 If argument is nil and the receiver is a subclass of Object, returns false.
 If argument is nil and the receiver is not a subclass of Object, returns true.

 If the aClassHistory argument is actually a class rather than a class history,
 then this method uses the class history of the argument, instead of the class
 itself."

<primitive: 70>
self _primitiveFailed: #isSubclassOf: args: { aClassHistory } .
^ false
%

category: 'Accessing the Class Format'
method: Behavior
isVariable

"Returns true if instances of the receiver have an unnamed part."

self isIndexable ifTrue: [^true].
self isNsc ifTrue: [^true].
^false
%

category: 'Accessing Categories'
method: Behavior
methodsDo: aBlock
  "evaluates aBlock for each method in receiver's method dictionaries
   for environment 0 . Returns the receiver.

   aBlock should be a two argument block
   expecting args  selectorSymbol ,   aGsNMethod ."

  self env: 0 methodsDo: aBlock
%

category: 'Instance Creation'
method: Behavior
migrateNew

"Create a new instance to use in migration.  By default, use #new.
Override in subclasses that can't use #new with #_basicNew. "

^ self new
%

category: 'Private'
method: Behavior
needsRecompileFor30

 "Returns true if the receiver needs to have methods recompiled."

  methDicts _isArray ifTrue:[ ^ false ].
  ^ methDicts valueConstraint ~~ GsNMethod
%

category: 'Private'
method: Behavior
needsRecompileFor33

 "Returns true if the receiver needs to have methods recompiled."

  self needsRecompileFor30 ifTrue:[ ^ true ].
  self env: 0 methodsDo:[ :selector :aMethod |
    aMethod needsRecompile ifTrue:[ ^ true ]
  ].
  ^ false .
%

category: 'Instance Creation'
method: Behavior
new

"Returns an instance of the receiver with no indexed variables."

<primitive: 51>

self _primitiveFailed: #new .
self _uncontinuableError
%

category: 'Instance Creation'
method: Behavior
new: anInteger

"Returns an instance of the receiver with the specified number of indexed
 variables.  Generates an error if the receiver is not indexable or if
 anInteger is not a positive SmallInteger.

 For new byte objects, all indexed variables are set to zero;
 for new pointer objects, all indexed variables are set to nil."

<primitive: 53>

(self isIndexable) ifFalse:[ self _errorNotIndexable. ^ self new ].
(anInteger _isSmallInteger) ifFalse:[
  anInteger _isInteger ifTrue:[
    anInteger _error: #errArgTooLarge args:{ SmallInteger maximumValue }.
    ^ self new
  ].
  anInteger _validateClass: SmallInteger.
  ^ self new
].
(anInteger < 0) ifTrue:[ anInteger _error: #rtErrArgNotPositive. ^ self new].
self _primitiveFailed: #new: args: { anInteger }.
self _uncontinuableError
%

category: 'Accessing Variables'
method: Behavior
offsetOfInstVar: aSymbol

"Returns the integer offset at which the instance variable named aString is
 stored in instances of the receiver.  Returns zero if the instance variable
 is not found."

^ instVarNames indexOfIdentical: aSymbol.
%

category: 'Accessing the Method Dictionary'
method: Behavior
persistentMethodAt: aSelector
  ^ (self persistentMethodAt: aSelector otherwise: nil)
     ifNil:[ self _error: #rtErrKeyNotFound args: { aSelector } ].
%

category: 'Accessing the Method Dictionary'
method: Behavior
persistentMethodAt: aSelector otherwise: notFoundVal
  | sym pmd |
  (sym := Symbol _existingWithAll: aSelector) ifNil:[ ^ notFoundVal ].
  (pmd := self persistentMethodDictForEnv: 0) ifNil:[ ^ notFoundVal ].
  ^ pmd at: sym otherwise: notFoundVal .
%

category: 'Updating'
method: Behavior
persistentMethodDictForEnv: envId
"result will may be nil if no methods exist for specified environmentId."
| mds |
(mds := methDicts) _isArray ifTrue:[
  ^ mds atOrNil: (envId*4 + 1)
].
envId == 0 ifTrue:[ ^ mds ].
^ nil
%

category: 'Updating'
method: Behavior
persistentMethodDictForEnv: envId put: aValue
  "aValue should be a GsMethodDictionary, or nil ,
   caller responsible for _refreshClassCache "

<protected>
| ofs mds |
aValue ifNotNil:[
  aValue class == GsMethodDictionary ifFalse:[ 
   ^ ArgumentTypeError new name: 'methDicts' expectedClass: GsMethodDictionary actualArg: aValue;
     signal.
  ].
].
(mds := methDicts) _isArray ifFalse:[
  envId == 0 ifTrue:[
    methDicts := aValue .
    ^ self
  ].
  mds := { mds }.
  methDicts := mds .
].
ofs := envId*4 + 1 .
mds size < ofs ifTrue:[ mds size: ofs ].
mds at: ofs put: aValue
%

category: 'Enumerating'
method: Behavior
persistentMethodDictsDo: aBlock
| mds |
(mds := methDicts) ifNotNil: [
  mds _isArray ifTrue:[
    1 to: mds size by: 4 do:[:j |  | aDict |
      (aDict := mds at: j) ifNotNil:[ aBlock value: aDict ].
    ].
  ] ifFalse:[
    aBlock value: mds
  ]
]
%

category: 'Enumerating'
method: Behavior
persistentNameSpacesDo: aBlock
| mds |
(mds := methDicts) ~~ nil ifTrue: [
  mds _isArray ifTrue:[
    2 to: mds size by: 4 do:[:j |  | aNs |
      (aNs := mds at: j) ifNotNil:[ aBlock value: aNs ].
    ].
  ]
]
%

category: 'Indexing Support'
method: Behavior
rcBtreeLeafNodeClass

"Returns the class of BtreeLeafNode to create for an equality index whose last
 object along the path is an instance of the receiver."

| result |
(RangeEqualityIndex isBasicClass: self)
    ifTrue: [ result := self indexManager rcBtreeBasicLeafNodeClass]
    ifFalse: [ result := self indexManager rcBtreeLeafNodeClass ].
^ result
%

category: 'Repository Conversion'
method: Behavior
recompileAllMethods

"Recompile all environment 0 methods for execution in a Gs64 v3.0 or later system."

| meths md |
meths := { } .
self env: 0 methodsDo:[ :selector :aMethod | meths add: aMethod ].
1 to: meths size do:[:j | (meths at: j) recompile ].
(md := methDicts) class == GsMethodDictionary ifTrue:[
  md valueConstraint == (ObsoleteClasses at: #GsMethod) ifTrue:[
     md size ~~ 0 ifTrue:[ Error signal:'inconsistent methodDict size'].
     md valueConstraint: GsNMethod
  ]
].
%

category: 'Repository Conversion'
method: Behavior
recompileMethodAt: aSelector
  (self compiledMethodAt: aSelector environmentId: 0) recompile
%

category: 'Analysis'
method: Behavior
referencedStrings

"Returns a Set containing all Strings and InvariantStrings referenced by
 the environment 0 methods in this Class and its metaclass."

| set blk |

set := IdentitySet new.
blk := [ :selector :method | method _referencedStringsInto: set ].
self env: 0 methodsDo: blk .
self class env: 0 methodsDo: blk .
^set.
%

category: 'Updating Categories'
method: Behavior
removeCategory: categoryName
  ^ self removeCategory: categoryName environmentId: 0
%

category: 'Modifying Classes'
method: Behavior
removeInstVar: aString
	"Removes the instance variable named aString from the receiver and from all of
 the receiver's subclasses.  The receiver and all of its subclasses must be
 modifiable.

 All instance methods for the receiver and its subclasses are recompiled using
 the symbol list of the current user.  If an error occurs during recompilation
 of methods, the instance variable will have been removed from the receiver and
 from all of its subclasses, but some methods in some subclasses will not have
 been recompiled, and subsquent commit will be disallowed.

 You may not use this method to remove an inherited instance variable."

	| offset aSymbol success |
	self _validatePrivilege ifFalse: [^nil].

	"validate that the instance variable exists"
	aSymbol := Symbol _existingWithAll: aString.
	aSymbol ifNotNil: [offset := self _ivOffsetOf: aSymbol].
	offset ifNil: [^self _error: #classErrNotAVar args: {aString}].
	(self superClass _ivOffsetOf: aSymbol) == nil
		ifFalse: [^self _error: #classErrRemoveInherIv args: {aSymbol}].

	"validate that self and all subclasses are modifiable"
	self validateSubclassesAreModifiable.
	success := false.

	[self _removeInstVarAtOffset: offset.	"remove from self and all subclasses"
	self recompileAllSubclassMethodsInContext: GsCurrentSession currentSession
				symbolList.
	success := true]
			ensure: [success ifFalse: [System _disallowCommitClassModFailure]]
%

category: 'Updating the Method Dictionary'
method: Behavior
removeSelector: aString

  self removeSelector: aString environmentId: 0
%

category: 'Browsing'
method: Behavior
removeSelector: aString environmentId: envId ifAbsent: aBlock
"Removes the method whose selector is aString from the receiver's
 method dictionary.  If the selector is not in the method
 dictionary, returns the result of evaluating the
 zero-argument block aBlock.  Otherwise, returns the receiver."

| aKey meth |

self _validatePrivilege ifFalse:[ ^ nil ].
aKey := Symbol _existingWithAll: aString .
aKey ifNotNil:[
  meth := self compiledMethodAt: aKey environmentId: envId otherwise: nil .
].
meth ifNil:[ ^ aBlock value ].
self removeSelector: aKey environmentId: envId
%

category: 'Browsing'
method: Behavior
removeSelector: aString ifAbsent: aBlock
  ^ self removeSelector: aString environmentId: 0 ifAbsent: aBlock
%

category: 'Accessing Variables'
method: Behavior
scopeHas: aVariableName
ifTrue: aBlock

"If aVariableName (a String) is specified as a variable in the receiver or one
 of its superclasses, this evaluates the zero-argument block aBlock and returns
 the result of evaluating aBlock.  Otherwise, returns false."

| allSharedPools |

(aVariableName isKindOf: String)
ifFalse:
   [ ^ aVariableName _error: #rtErrBadArgKind args: { String }].

( ((self allInstVarNames ) includesValue: aVariableName) or:
 [((self allClassVarNames) includesValue: aVariableName)])
ifTrue:
   [^ aBlock value]
ifFalse: "now check sharedPools"
[
   allSharedPools:= self allSharedPools.
   allSharedPools do: [:poolDict |
                         (poolDict includesKey: aVariableName)
                         ifTrue:
                            [ ^ aBlock value]
                      ]
].
^ false
%

category: 'Accessing the Method Dictionary'
method: Behavior
selectors
^ self selectorsForEnvironment: 0
%

category: 'Accessing Categories'
method: Behavior
selectorsIn: categoryName
  ^ self selectorsIn: categoryName environmentId: 0
%

category: 'Accessing Categories'
method: Behavior
selectorsIn: categoryName environmentId: envId

"Returns an Array of all selectors in the specified category.  If categoryName
 is not in the receiver's method dictionary, generates an error."

| set count catSym |
catSym := Symbol _existingWithAll: categoryName .
catSym ifNil:[ ^ self _categoryNotFound:  categoryName ].
count := 0 .
set := IdentitySet new .
self env: envId categorysDo:[ :categName :selectors |
  categName == catSym ifTrue:[ set addAll: selectors . count := count + 1 ].
].
count == 0 ifTrue:[ ^ self _categoryNotFound:  categoryName ].
^ Array withAll: set
%

category: 'Accessing the Class Format'
method: Behavior
selfCanBeSpecial

"Return true if the class format has the SelfCanBeSpecial bit set.

Any class for which self can be a special object must be created
using the value  #selfCanBeSpecial  as an element of the argument
to the options: keyword of the subclass creation method .
If instances of any of
   Boolean Character SmallDate SmallDateAndTime SmallDouble
   SmallFraction SmallInteger SmallScaledDecimal SmallTime UndefinedObject
will be able to inherit methods from a class, that class
must be created with  #selfCanBeSpecial .

Example, the Pharo class ProtoObject needs #selfCanBeSpecial  .

#selfCanBeSpecial is not inherited from a superclass by subclass
creation 

#selfCanBeSpecial be present in a superclass to be able to specify it for a subclass."

^ (format bitAnd: 16r2000) ~~ 0
%

category: 'Accessing Variables'
method: Behavior
sharedPools

"Returns an Array of pool dictionaries used by this class.  Superclasses
 are not included; contrast with allSharedPools."

| pd |
^ (pd := poolDictionaries) ifNil:[ { } ]
		      ifNotNil:[ Array withAll: pd ]
%

category: 'Accessing Categories'
method: Behavior
sortedCategoryNames

"Returns an Array of Symbols.  The elements of the collection are the
 receiver's category names (excluding names inherited from superclasses)."

 | set |
 set := IdentitySet new .
 self env: 0 categorysDo:[ :categName :selectors | set add: categName ].
 ^ Array withAll: (SortedCollection withAll: set ) .
%

category: 'Accessing Categories'
method: Behavior
sortedSelectorsIn: categoryName
  ^ self sortedSelectorsIn: categoryName environmentId: 0
%

category: 'Accessing Categories'
method: Behavior
sortedSelectorsIn: categoryName environmentId: envId

"Returns an Array of all selectors in the specified category, sorted
 in ascending order."

 ^ Array withAll:( SortedCollection withAll:(
        self selectorsIn: categoryName environmentId: envId) )
%

category: 'Indexing Support'
method: Behavior
sortNodeClass

"Returns the class of SortNode to create for sorting on instances of the
 receiver."

| result |
(RangeEqualityIndex isBasicClass: self)
    ifTrue: [ result := self indexManager sortNodeClass ]
    ifFalse: [ result := SortNode ].
^ result
%

category: 'Accessing the Method Dictionary'
method: Behavior
sourceCodeAt: aSelector

^ self sourceCodeAt: aSelector environmentId: 0
%

category: 'Accessing the Method Dictionary'
method: Behavior
sourceCodeAt: aSelector environmentId: envId

"Returns a String representing the source code for the argument, aSelector.  If
 aSelector (a String) is not a selector in the receiver's method dictionary,
 this generates an error."

^ (self compiledMethodAt: aSelector environmentId: envId) sourceString
%

category: 'Instance Creation'
method: Behavior
squeakBasicNew: anInteger

"Returns with an instance of the receiver.
 If the result is oop format, anInteger specifies the number of varying instVars.
 For an bytes format result, anInteger specifies the number of words
 where (self _bytesPerWord) is the number of bytes per word.

 Signals an error if the Behavior is not indexable or if anInteger is bad."

<primitive: 1082>
| isz |
(self _isInstanceDisallowed) ifTrue: [
   self _error: #objErrCantCreateInstance args:  #()  .
   self _primitiveFailed: #_basicNew: args: { anInteger }.
   self _uncontinuableError
].
(self isIndexable) ifFalse:[self _errorNotIndexable .  ^ self _basicNew ].
anInteger _isSmallInteger ifFalse:[
  anInteger _validateClass: SmallInteger . ^ self _basicNew
].
anInteger < 0 ifTrue:[
  anInteger _error: #rtErrArgNotPositive. ^ self _basicNew
].
(anInteger + (isz := self instSize)) _isSmallInteger ifFalse: [
  anInteger _error: #errArgTooLarge args:{ SmallInteger maximumValue - isz} .
  ^ self _basicNew
].
self _primitiveFailed: #_basicNew: args: { anInteger }.
self _uncontinuableError
%

category: 'Accessing the Class Format'
method: Behavior
subclassesDisallowed

"Returns true if subclasses of the receiver have been disallowed by means of
 Behavior | disallowSubclasses.  Otherwise, returns false."

^ (format bitAnd: 16r20) ~~ 0
%

category: 'Accessing the Class Hierarchy'
method: Behavior
superClass

"Returns the receiver's superclass."

 ^ superClass
%

category: 'Accessing the Class Hierarchy'
method: Behavior
superclass
 "for squeak compatibility"
 ^ self superClass
%

category: 'Accessing the Class Hierarchy'
method: Behavior
superclassForEnv: envId

 "Return receiver's superclass in environment zero .
  Reimplemented in Module to handle envId ~~ 0 ."

  ^ superClass
%

category: 'Updating the Class Hierarchy'
method: Behavior
superclassForEnv: envId put: aCls
  "Update the receiver's persistent superclass for method lookup
   environment specified by envId.
   envId must be a SmallInteger > 0 .
   aClass must be a Behavior.

   It is necessary to execute
     Behavior _clearLookupCaches: envId ; _clearLookupCaches: 0 .
   after executing this method if you want the change in hierarchy
   to take effect immediately for all method lookups.
"
  <primitive: 2001>
  | prot |
  prot := System _protectedMode .
  [ | mds ofs |
    aCls isBehavior ifFalse:[ ^ ArgumentError signal:'expected a Class' ] .
    (self selfCanBeSpecial and:[ aCls selfCanBeSpecial not]) ifTrue:[ "fix 48261"
      ^ ArgumentError signal:'receiver has selfCanBeSpecial and argument does not'.
    ].
    envId > 0 ifFalse:[ ^ ArgumentError signal:'expected envId > 0' ].
    (mds := methDicts) _isArray ifFalse:[
      self persistentMethodDictForEnv: envId put: nil .
      mds := methDicts .
      ].
    ofs := envId*4 + 3 .
    mds size < ofs ifTrue:[ mds size: ofs ].
    mds at: ofs put: aCls
  ] ensure:[
    prot _leaveProtectedMode
  ].
%

category: 'Modifying Classes'
method: Behavior
validateIsModifiable

"Returns the receiver if the receiver and its
 Array of instance variables are modifiable.  Generates an error if the
 receiver cannot be modified (that is, if the receiver
 or its Array of instance variable names is not variant)."

self isModifiable ifFalse:[ ^ self _error: #rtErrClassNotModifiable ]
%

category: 'Modifying Classes'
method: Behavior
validateSubclassesAreModifiable

"Generates an error if the receiver or any of its subclasses cannot be
 modified."

self validateIsModifiable .
self _subclasses do:[:x| x validateSubclassesAreModifiable ].
^ self
%

category: 'Testing Inheritance'
method: Behavior
validateSubclassOf: aClass

"Returns true if receiver is identical to aClass or is a subclass
 of aClass; otherwise, generates an error."

( self isSubclassOf: aClass) ifFalse:[
     ^ self _error: #rtErrNotASubclassOf args:{ aClass } ].
^ true
%

category: 'Accessing the Method Dictionary'
method: Behavior
whichClassIncludesSelector: aString
  ^ self whichClassIncludesSelector: aString environmentId: 0
%

category: 'Modifying Classes'
method: Behavior
_addInvariantClassVar: aSymbol value: aVal

"Adds the class variable with name aSymbol to the receiver's
 class variables dictionary, and make the resulting SymbolAssocation
 invariant."
| assoc dict |
self _validatePrivilege ifTrue:[
  (dict := classVars) ifNil:[
    dict := self _createClassVarsDict .
  ].
  assoc := dict associationAt: aSymbol otherwise: nil .
  assoc ifNil:[
    assoc := SymbolAssociation new key: aSymbol value: aVal .
    assoc objectSecurityPolicy: self objectSecurityPolicy .
    dict addAssociation: assoc .
    assoc immediateInvariant
  ] ifNotNil:[
    assoc _value = aVal ifFalse:[
      assoc _value: aVal .
      assoc immediateInvariant
    ]
  ].
]
%

category: 'Private'
method: Behavior
_allStrippedMethodSelectors

"Returns an Array of two Arrays, signifying that all methods of the receiver
 are to have their source stripped."

  ^ { self selectors . self class selectors }.
%

category: 'Jade browser support'
method: Behavior
_allSuperList

 ^ self _allSuperList: 0
%

category: 'Accessing the Class Hierarchy'
method: Behavior
_allSuperList: envId

"Returns an Array of the superclasses of the receiver, beginning
 with the most remote superclass, and excluding the receiver itself."

  | result currClass |
  result:= { } .
  currClass := self .
  envId ~~ 0 ifTrue:[
    [ true] whileTrue:[
      currClass:= currClass superclassForEnv: envId .
      currClass == nil ifTrue: [^ result].
      result insertObject: currClass at: 1.
    ].
  ] ifFalse:[
    [ true] whileTrue:[
      currClass:= currClass superClass.
      currClass == nil ifTrue: [^ result].
      result insertObject: currClass at: 1.
    ].
  ]
%

category: 'Private'
method: Behavior
_allSuperList: includeRubyVirtual env: envId
  ^ self _allSuperList: envId 
%

category: 'Enumerating'
method: Behavior
_baseCategorys: envId

  "Returns nil or the category dictionary for specified environment"
  | cats |
  (cats := categorys) _isArray ifTrue:[
    cats := cats atOrNil:  envId + 1 .
  ] ifFalse:[
    envId ~~ 0 ifTrue:[ cats := nil ].
  ].
  ^ cats
%

category: 'Accessing Categories'
method: Behavior
_baseCategorysForStore: envId
  | dict carr ofs |
  (dict := self _baseCategorys: envId ) ifNil:[
     dict := GsMethodDictionary new  .
     dict objectSecurityPolicy: self objectSecurityPolicy .
     ofs := envId + 1 .
     (carr := categorys) _isArray ifTrue:[
       (carr size < ofs) ifTrue:[ carr size: ofs ].
        carr at: ofs put: dict .
     ] ifFalse:[
       envId > 0 ifTrue:[
         carr := Array new: ofs .
         carr at: 1 put: categorys .
         carr at: ofs put: dict .
         categorys := carr .
       ] ifFalse:[
         categorys := dict
       ].
     ].
  ].
  ^ dict
%

category: 'Instance Creation'
method: Behavior
_basicNew

"Returns an instance of the receiver, with no indexed
 variables.  Do not override this method; contrast with Behavior | new."

<primitive: 50>

(self _isInstanceDisallowed)
ifTrue:[ self _error: #objErrCantCreateInstance args:  #()  .
         self _uncontinuableError
       ].
self _primitiveFailed: #_basicNew .
self _uncontinuableError
%

category: 'Instance Creation'
method: Behavior
_basicNew: anInteger

"Returns with an instance of the receiver, with the given
 number of fields.  Generates an error if the Behavior is not indexable or if
 anInteger is bad.  Do not override this method; contrast with Behavior | new:."

<primitive: 52>
| isz |
(self _isInstanceDisallowed) ifTrue: [
   self _error: #objErrCantCreateInstance args:  #()  .
   self _primitiveFailed: #_basicNew: args: { anInteger }.
   self _uncontinuableError
].
(self isIndexable) ifFalse:[self _errorNotIndexable .  ^ self _basicNew ].
anInteger _isSmallInteger ifFalse:[
  anInteger _validateClass: SmallInteger . ^ self _basicNew
].
anInteger < 0 ifTrue:[
  anInteger _error: #rtErrArgNotPositive. ^ self _basicNew
].
(anInteger + (isz := self instSize)) _isSmallInteger ifFalse: [
  anInteger _error: #errArgTooLarge args:{ SmallInteger maximumValue - isz} .
  ^ self _basicNew
].
self _primitiveFailed: #_basicNew: args: { anInteger }.
self _uncontinuableError
%

category: 'Browser Methods'
method: Behavior
_categoriesReport

^ self _categoriesReportEnv: 0
%

category: 'Browser Methods'
method: Behavior
_categoriesReportEnv: envId

"Returns an Array containing key-value pairs from the receiver's categories
 for specified environment.
 The key in each key-value pair is the name of a category; the value in each
 key-value pair is a sorted Array of selectors.

 Used by the Topaz 'list categories' command."

| result k sz sortedCats |
sortedCats := SortedCollection new:[ :a :b | a key <= b key ].
self env: envId unifiedCategoriesDo:[ :categName :selectors |
  sortedCats add: ( Association newWithKey: categName value: selectors)
].
result := Array new: (sz := sortedCats size) * 2  .
k := 1 .
1 to: sz do:[:j | | anAssoc |
   anAssoc := sortedCats at: j .
   result at: k put: anAssoc key .
   result at: k + 1 put: (Array withAll:(SortedCollection withAll: anAssoc _value)).
   k := k + 2 .
].
^ result .
%

category: 'Accessing Categories'
method: Behavior
_categoryNotFound: aString
  self _error: #classErrMethCatNotFound args: { aString } .
  ^ nil
%

category: 'Browsing'
method: Behavior
_categoryOfSelector: selector

"Returns the category of the given selector, or 'unknown' if it isn't found."

 | result |
 (result := self categoryOfSelector: selector) == nil ifTrue:[
    ^ 'unknown'
    ].
 ^ result
%

category: 'Private'
method: Behavior
_checkCompileResult: result source: sourceString suppressCompileWarning: suppressCompileWarning
  "process the result Array from _primitiveCompileMethod:...
   Returns a GsNMethod or signals a CompileError or CompileWarning."

  result _isArray ifFalse:[ 
    ^ result "a GsNMethod or a selector "
  ].
  (result at: 2) ifNotNil:[ :errorArray|
    "Fill in the error message text for each error in the result."
    [ | errDict |
      errDict := GemStoneError at: System myUserProfile nativeLanguage .
      1 to: errorArray size do:[:j | | thisErr |
	thisErr := errorArray at: j .
	(thisErr atOrNil: 3) ifNil:[  | msg |
	  thisErr size < 3 ifTrue:[ thisErr size: 3 ].
	  msg := errDict atOrNil:(thisErr at: 1) .
	  thisErr at: 3 put: ( msg ifNil:[ '(unknown error number)' ]).
	].
      ].
    ] onException: Error do:[:ex | "ignore"].
    ^ CompileError new args: { errorArray . sourceString . self } ; signal
  ].
  (result at: 3) ifNotNil:[ :warnStr | | meth |
    (meth := result at: 1) class == GsNMethod ifTrue:[
       suppressCompileWarning ifFalse: [ CompileWarning signal: warnStr method: meth ].
       ^ meth
    ].
  ].
  InternalError signal:'unrecognized result from _primitiveCompileMethod'
%

category: 'Private'
method: Behavior
_classCategory

"Private."

"The classCategory instance variable is defined in Class, so return nil here."

^ nil
%

category: 'Accessing Variables'
method: Behavior
_classVars

"Returns the classVars instance variable."

^ classVars
%

category: 'Private'
method: Behavior
_clearConstraints
 "Used in repository upgrade only.
 As of GemStone 64bit v3.4, constraints are no longer implemented."

 constraints ifNotNil:[ constraints := nil ]
%

category: 'Virtual Machine Control'
method: Behavior
_clearLookupCaches: envId

"Invalidates method lookup caches for all classes.
 Invalidates all send-site caches for the specified environment.

 envId must be a SmallInteger >= 0 and <= 255 .
"
<primitive: 165>
envId _validateClass: SmallInteger .
self _primitiveFailed:#_clearLookupCaches: args: { envId }
%

category: 'Updating the Method Dictionary'
method: Behavior
_codeChangedForEnv: envId
  "set bit in VM's cbCodeChanged word so it can propagate to
   other VMs upon commit.  Code which adds or removes entries
   in persistent method dictionaries must send this method. 
   envId >= 0   methods changed for specified environment .
   envId == -1  sessions methods changed .
  "
<primitive: 854>
envId _validateClass: SmallInteger .
self _primitiveFailed: #_codeChangedForEnv: args: {envId} "envId out of range"
%

category: 'Private'
method: Behavior
_compileMethod: sourceString
symbolList: aSymbolList
"Compile sourceString as an instance method in the receiver without installing
 in any dictionaries.
 Returns a GsNMethod or signals an Error ."

^ self _compileMethod: sourceString symbolList: aSymbolList environmentId: 0
%

category: 'Private'
method: Behavior
_compileMethod: sourceString
symbolList: aSymbolList
environmentId: environmentId
"Compile sourceString as an instance method in the receiver without installing
 in any dictionaries.
 Returns a GsNMethod or signals an Error ."

| res |
res := self _primCompileMethod: sourceString
   symbolList: aSymbolList
   category: nil
   oldLitVars: nil
   intoMethodDict: false
   intoCategories: nil
   environmentId: environmentId .
^ self _checkCompileResult: res source: sourceString suppressCompileWarning: false
%

category: 'Updating the Method Dictionary'
method: Behavior
_compileMethodTrappingErrors: sourceString
dictionaries: aSymbolList
category: aCategoryString
environmentId: envId

"If there are no errors, add the resulting compiled method to the receiver's
 method dictionary and returns nil.

 If errors occur, returns an Array of pairs.  The first element of each pair is
 the GemStone error number, and the second element is the offset into the
 sourceString where the error occurred.

 This method differs from compileMethod:dictionaries:category:environmentId:
 in that it traps all errors (not just compiler errors).
 Non-compiler errors are reported with a source offset of 0."

  self _validatePrivilege ifFalse:[ ^ nil ].
  ^ [ self compileMethod: sourceString dictionaries: aSymbolList
           category: aCategoryString asSymbol  intoMethodDict: nil 
           intoCategories: nil environmentId: envId  .
      nil
    ] onException: { CompileError . Error } do:
      { [:ex | { ex gsArguments at: 1 }  "a CompileError" ] .
        [:ex | { { ex number . 0 } }  "any other Error" ]
      }
%

category: 'Deprecated'
method: Behavior
_constraintAt: offset
 "As of GemStone 64bit v3.4, constraints are no longer implemented.
 This method is provided for examining classes in repositories upgraded from
 an older version."

  constraints _isArray ifFalse:[ ^ Object].
  ^ (constraints atOrNil: offset) ifNil:[ Object ]
                 ifNotNil:[ :cls | cls currentVersion ]
%

category: 'Deprecated'
method: Behavior
_constraints
	"Returns nil or an Array.

 As of GemStone 64bit v3.4, constraints are no longer implemented.
 This method is provided for examining classes in repositories upgraded from
 an older version."

	self
		deprecated: 'Behavior>>_constraints  deprecated, Constraints are no longer supported'.
	^constraints
%

category: 'Browsing'
method: Behavior
_copyMethodsAndVariablesFrom: sourceClass except: except dictionaries: dicts

"Copies all instance and class methods, pool dictionaries, and references
 to class variables from the given class to ourselves.  Returns an Array of
 methods in the source class which failed to compile.  Some of them
 might be class methods.  The Array will be empty if none failed.

 Method environmentIds are copied from the source methods.

 The except: argument is a list of categories and/or methods that should
 not be copied.  Each exception is two elements in the Array: a Character
 followed by a String or Symbol.
    $C a category of class methods follows
    $c a category of instance methods follows
    $S the selector of a class method follows
    $s the selector of an instance method follows
    $V a list of class variable names follows
    $P a list of pool dictionaries follows"

| failed srccls targcls sel doit pds otherCvs excludeCvs envId |

self _validatePrivilege ifFalse:[ ^ nil ].
self == sourceClass ifTrue:[
  "because iteration is directly over the source's categories dictionaries"
  ^ self error: 'source of copy must not be self'
].
failed := { } .
pds :=  #() .
excludeCvs :=  #() .

except ifNotNil: [ | i |
  i := except indexOf: $P.
  i ~~ 0 ifTrue: [ pds := except at: i+1 ].
  i := except indexOf: $V.
  i ~~ 0 ifTrue: [ excludeCvs := except at: i+1 ].
  ].

"Copy pool dictionaries"
sourceClass _poolDictionaries do: [ :dict | | poolDicts |
  poolDicts := poolDictionaries .
  (poolDicts ~~ nil and:[ poolDicts includesIdentical: dict]) ifFalse: [
    (pds includesIdentical: dict) ifFalse: [
      poolDicts ifNil:[ poolDicts := { } . poolDictionaries := poolDicts ]
          ifNotNil:[ poolDicts isInvariant ifTrue:[
                       poolDicts := Array withAll: poolDicts . poolDictionaries := poolDicts
                     ]].
      poolDicts add: dict
    ].
  ].
].

"Copy class variables"
otherCvs := sourceClass _classVars .
otherCvs ifNotNil:[ | destCvs |
   destCvs := classVars .
   otherCvs associationsDo: [ :assn | | other |
    destCvs ifNotNil:[ other := destCvs associationAt: assn key otherwise: nil ].
    (other == nil or: [other value == nil and: [assn value ~~ nil]]) ifTrue: [
      (excludeCvs includesValue: assn key) ifFalse:[
        destCvs ifNil:[ destCvs := self _createClassVarsDict ].
        destCvs add: assn
      ].
    ].
  ].
].

"Copy class and instance methods"
envId := 0 .  "change for Maglev"
1 to: 2 do: [ :j |
  j == 1 ifTrue:[ srccls := sourceClass.  targcls := self ]
        ifFalse:[ srccls := sourceClass class.  targcls := self class ].
  srccls categorysDo:[ :cat :sels |
      1 to: sels size do: [ :s | | oldMeth |
	sel := sels at: s.
	doit := true.
	1 to: except size by: 2 do: [ :i | | ch exCat |
          ch := except at: i .
          exCat := except at: i + 1 .
	  (((( ch == $C and: [targcls isMeta and: [cat == exCat ]]) or:
	    [ ch == $S and: [targcls isMeta and: [sel == exCat ]]]) or:
	    [ ch == $c and: [targcls isMeta not and: [cat ==  exCat  ]]]) or:
	    [ ch == $s and: [targcls isMeta not and: [sel ==  exCat ]]]) ifTrue: [
	      doit := false
	  ].
	].
	doit ifTrue: [ | methEnvId |
          oldMeth := srccls compiledMethodAt: sel environmentId: envId .
	  methEnvId := oldMeth environmentId .
          methEnvId == envId ifFalse:[ self error:'environmentId mismatch'].
	  ( targcls
	     _compileMethodTrappingErrors: oldMeth sourceString
	     dictionaries: dicts category: cat environmentId: envId ) ifNotNil:[
	    failed add: oldMeth
          ].
        ].
      ].
  ].
].

^failed.
%

category: 'Modifying Classes'
method: Behavior
_createClassVarsDict
  | dict |
  (dict := classVars) ifNil:[
    (dict := SymbolDictionary new ) objectSecurityPolicy: self objectSecurityPolicy .
    self _setClassVars: dict old: nil .
  ].
  ^ dict
%

category: 'Private'
method: Behavior
_deepCopyWith: copiedObjDict

"Private. Used internally to implement deepCopy."

^ self.
%

category: 'Private Methods for Class Modification'
method: Behavior
_disallowGciCreateStore

"Private.

 Sets bit in the format instance variable to cause GemBuilder for C instance
 creation and updates to go through message sends.  Semantics are private to
 GemBuilder for Smalltalk."

self _validatePrivilege ifTrue:[
  format := format bitOr: 16r200 "no update through structural access" .
  self _refreshClassCache: false
]
%

category: 'Private'
method: Behavior
_gciCompileMethod: sourceString dictionaries: aSymbolList category: aCategoryString environmentId: envId
	"used by GCI implementation , when session methods are enabled.
   Returns nil for successful compilation, a warning String,
   or signals a CompileError"

  | warnStr |
  [
    self compileMethod: sourceString dictionaries: aSymbolList
          category: aCategoryString environmentId: envId .
  ] onException: CompileWarning do: [ :ex | "handle CompileWarning"
    warnStr := ex warningString .
    ex resume .
  ].
  ^ warnStr
%

category: 'Stripping Sources'
method: Behavior
_hideSourceCode

"For each environment 0 method defined for this class, hide the source code.  All that
 remains of the source is the method signature and the initial comment if one
 exists."

self _validatePrivilege ifFalse:[ ^ nil ].
self env: 0 methodsDo:[ :selector :method | method _removeAllSourceButFirstComment ]
%

category: 'Indexing Support'
method: Behavior
_idxIvOffsetOf: aSymbol

"Searches the instVarNames instance variable of the receiver for an instance
 variable named aSymbol, and returns the offset for that instance variable.
 Returns nil if no instance variable exists with the name aSymbol.
 Signals an error if the instance variable is dbTransient. "

  | idx |
  idx := instVarNames indexOfIdentical: aSymbol .
  idx == 0 ifTrue:[ ^ nil ] .
  dbTransientMask ifNotNil:[:dbTrMask |
    ((1 bitShift: idx - 1 ) bitAnd: dbTrMask) ~~ 0 ifTrue:[
       Error signal:'dbTransient instVar ', aSymbol printString, ' may not participate in an index'.
    ].
  ].
  ^ idx
%

category: 'Private Methods for Class Modification'
method: Behavior
_incrementInstVars: delta

"Increment instVars by 1 to account for adding a named instance variable."
| mask n info_old |
mask := Class_numIvs_mask .
info_old := instVarsInfo .
delta _isSmallInteger ifFalse:[ delta _validateClass: SmallInteger ].
n := (info_old bitAnd: mask) + delta .
n < 0 ifTrue:[ self error:'numInstVars would go negative'].
n > 2030"virtual machine constant GEN_MAX_INSTANCE_VARS" ifTrue:[
   ArgumentError signal:'number of instance variables  would exceed 2030' .
].
instVarsInfo := (info_old bitAnd:( mask bitInvert)) bitOr:( n bitAnd: mask)
%

category: 'Private Methods for Class Modification'
method: Behavior
_insertNamedInstVar: aSymbol atOffset: offset
	"Receiver and all subclasses must have be modifiable.  aSymbol must be unique
 with respect to existing instance variables."
	" add instance variable to self"

	| mySubclasses ivn |
	self _validatePrivilege
		ifTrue:
			[self _incrementInstVars: 1.
			(ivn := instVarNames) == #()
				ifTrue:
					[ivn := {}.
					instVarNames := ivn].
			ivn insertObject: aSymbol at: offset.
			self _refreshClassCache: false.
			mySubclasses := self _subclasses.
			mySubclasses ~~ nil
				ifTrue:
					[mySubclasses do: [:x | x _insertNamedInstVar: aSymbol atOffset: offset]]].
	^self
%

category: 'Accessing the Class Format'
method: Behavior
_instancesNpDbtransient

  ^ (format bitAnd: 16r1800)  "used by _become:fullChecks:  methods"
%

category: 'Accessing Variables'
method: Behavior
_instVarNames

"Returns the receiver's instance variables list.  Contrast with the public
 method #allInstVarNames."

^ instVarNames
%

category: 'Private'
method: Behavior
_instVarsEqual: anArrayOfInstvarNames

  "Return true if the argument matches the instVarNames
   defined by the receiver (excluding inherited instVars), false otherwise."

  ^ (Array withAll: self instVarNames) = (anArrayOfInstvarNames collect:[:n | n asSymbol ])
%

category: 'Private'
method: Behavior
_instVarsInfo
  ^ instVarsInfo
%

category: 'Error Handling'
method: Behavior
_isInstanceDisallowed

^ (InstancesDisallowed includesValue: self) or:[ self isMeta ]
%

category: 'Private'
method: Behavior
_isKernel

"Private.  Returns true if the given class is a GemStone kernel class."

<primitive: 480>
self _primitiveFailed: #_isKernel
%

category: 'Accessing the Class Format'
method: Behavior
_isSpecial

"Returns true if instances of the receiver are special objects.
 Otherwise, returns false."

^ (format bitAnd: 16r3) == 3
%

category: 'Indexing Support'
method: Behavior
_ivOffsetOf: aSymbol

"Searches the instVarNames instance variable of the receiver for an instance
 variable named aSymbol, and returns the offset for that instance variable.
 Returns nil if no instance variable exists with the name aSymbol."

| idx |
idx := instVarNames indexOfIdentical: aSymbol .
^ idx == 0 ifTrue:[ nil ] ifFalse:[ idx ].
%

category: 'Private Methods for Class Modification'
method: Behavior
_makeProtected

"Make the receiver a protected class by setting a bit in its format.  This
 protection disallows structural access through GemBuilder for C."

self _validatePrivilege ifTrue:[
  format := format bitOr: 128
]
%

category: 'Private'
method: Behavior
_makeTraversalByCallback

"Private.

 Make the receiver place its instances in a traversal buffer by
 invoking the clampSpecification's traversal callback method."

self _validatePrivilege ifTrue:[
  format := format bitOr: 16r400. "travByCallback"
  self _refreshClassCache: false .
]
%

category: 'Accessing the Method Dictionary'
method: Behavior
_methodWithSource: aString

"Returns the environment 0 method from the receivers' method dictionary
 whose source string is equal to aString , or nil if not found."

self env: 0 methodsDo:[ :aSelector :aMethod |
   aMethod sourceString = aString ifTrue:[ ^ aMethod ].
].
^ nil
%

category: 'Private'
method: Behavior
_moveMethod: aSelector toCategory: categoryName
  ^ self moveMethod: aSelector toCategory: categoryName environmentId:0
%

category: 'Accessing the Class Format'
method: Behavior
_name
  ^ 'aBehavior'
%

category: 'Deprecated'
method: Behavior
_namedIvConstraintAt: anInteger
  "Returns the constraint at the specified offset, or Object.

 As of GemStone 64bit v3.4, constraints are no longer implemented.
 This method is provided for examining classes in repositories upgraded from
 an older version."

  (anInteger > self instSize ) ifTrue:[ ^ Object].
  ^ self _constraintAt: anInteger
%

category: 'Private'
method: Behavior
_nameForFileout
  "used by topaz"

^ self theNonMetaClass name
%

category: 'Private'
method: Behavior
_noStrippedMethodSelectors

"Returns an Array of two empty Arrays, signifying that no methods are to have
 their source stripped."

  ^ #( #() #() ).
%

category: 'Enumerating'
method: Behavior
_persistentMethodDicts
  "Returns an Array of the persistent method dictionaries for all environments"

 methDicts
   ifNil:[ ^ #() ]
   ifNotNil:[ :mds |
     mds _isArray ifTrue:[ | arr |
       arr := { } .
       1 to: mds size by: 4 do:[:j |
         (mds at: j) ifNotNil:[:aDict | arr add: aDict ]
       ].
       ^ arr
     ] ifFalse:[
      ^ { mds }
     ]
   ].
%

category: 'Browser Methods'
method: Behavior
_poolDictionaries

"Returns the object containing this instance's pool dictionaries."

^  poolDictionaries ifNil:[ #() ]
%

category: 'Private'
method: Behavior
_primCompileMethod: sourceString
symbolList: aSymbolList
category: categorySymbol
oldLitVars: litVarArray
intoMethodDict: aMethodDict
intoCategories: aCategDict
environmentId: environmentId

"Compiles sourceString as a method for the receiver in category categorySymbol,
 using the symbol list aSymbolList.  If the compilation succeeds, the method
 dictionary of the receiver will have been updated.

 Returns the GsNMethod produced by the compilation if the compilation
 succeeded with no warnings or errors, or an Array of of the form
    { (GsNMethod or nil if the compilation has errors) .
       (nil or an Array of error descriptors as described for
        compileMethod:dictionaries:category: ) .
       (nil or a String describing warnings)
     } .

 If litVarArray is not nil, it must be an Array of Symbol,SymbolAssociation pairs
 and this Array will be searched prior to searching aSymbolList to
 resolve literal variables within the method.

 If the compilation succeeds,
   if aMethodDict == false, the method is added to no dictionaries,
      and aCategDict is ignored
   else if aMethodDict ~~ nil, the method is added to per-session
      method dictionaries,
   else aMethodDict == nil, the method is added to receiver's
      persistent method dictionary.

 If aMethodDict is neither nil nor false, and aCategDict is not nil and
 the compilation succeeds, the resulting method is added to aCategDict
 instead of the receiver's categories.

 If the compilation succeeds, the selector of the new method is
 removed from all method lookup caches for the receiver and all subclasses
 thereof,   independent of the value of aMethodDict argument.

 environmentId must be a SmallInteger >= 0 and <= 255 .
 0 denotes the base Smalltalk image.  1 was reserved for use by Ruby .

 You must have code modification privilege to execute this primitive.

 The IR graph produced by the parser (a GsComMethNode) is available as
   (System __sessionStateAt: 19)
 until being overwritten by the next invocation of this primitive.
 See GsNMethod(C)>>generateFromIR: for example code of printing this IR."

<primitive: 228>
sourceString _validateClasses: { String }.
aSymbolList ~~ nil  ifTrue:[ aSymbolList _validateClass: SymbolList ].
categorySymbol _validateClass: Symbol.
litVarArray ~~ nil ifTrue:[ litVarArray _validateClass: Array].
(aMethodDict ~~ nil and:[ aMethodDict ~~ false]) ifTrue:[ aMethodDict _validateClass: GsMethodDictionary ].
aCategDict ~~ nil ifTrue:[ aCategDict _validateClass: GsMethodDictionary ].
environmentId _validateClass: SmallInteger .

^ self _primitiveFailed:
  #_primCompileMethod:symbolList:category:oldLitVars:intoMethodDict:intoCategories:environmentId:
   args: { sourceString .  aSymbolList .  categorySymbol .  litVarArray .
	   aMethodDict .  aCategDict .  environmentId }
%

category: 'Private'
method: Behavior
_primExtractSelector: sourceString
  "Return a Symbol or an Array containing error details"

  <primitive: 252>
  sourceString _validateClasses: { String }.
  ^ self _primitiveFailed:
  #_primExtractSelector: args: { sourceString }
   
%

category: 'Private'
method: Behavior
_primitiveCompileMethod: sourceString
symbolList: aSymbolList
category: categorySymbol
oldLitVars: litVarArray
intoMethodDict: aMethodDict
intoCategories: aCategDict
environmentId: environmentId

  | src cvtAb |
  cvtAb := System _zeroArgPrim: 132 "fast System gemConfigurationAt:#GemConvertArrayBuilder" .
 cvtAb ifTrue:[ | cvtRes |
   sourceString _isOneByteString ifFalse:[
     sourceString _validateClasses: { String }.
   ].
   cvtRes := GsNMethod _convertArrayBuildersIfNeeded: sourceString .
   cvtRes _isArray ifTrue:[
     "error during ArrayBuilder conversion"
     ^ cvtRes "a compile error descriptor"
   ].
   src := cvtRes .
 ] ifFalse:[
   src := sourceString
 ].
 ^ self _primCompileMethod: src
     symbolList: aSymbolList
     category: categorySymbol
     oldLitVars: litVarArray
     intoMethodDict: aMethodDict
     intoCategories: aCategDict
     environmentId: environmentId
%

category: 'Private'
method: Behavior
_recompileAllMethods
  "unconditionally recompile all env 0 methods."
| meths symList |
meths := { } .
symList := GsSession currentSession symbolList .
self env: 0 methodsDo:[ :selector :aMethod | meths add: aMethod ].
1 to: meths size do:[:j |
  (meths at: j) recompileIntoMethodDict: nil intoCategories: nil symbolList: symList
].
%

category: 'Private Methods for Class Modification'
method: Behavior
_recompileMethodsAfterNewIvOffset: newOffset

| mySymList mySubclasses |
self _validatePrivilege ifTrue:[
  mySymList:= GsCurrentSession currentSession symbolList .
  (newOffset < self instSize) ifTrue:[
      self recompileAllMethodsInContext: mySymList ] .
  mySubclasses := self _subclasses .
  mySubclasses ~~ nil ifTrue:[
    mySubclasses do:[:x | x _recompileMethodsAfterNewIvOffset: newOffset ] .
    ].
].
^ self
%

category: 'Private Methods for Class Modification'
method: Behavior
_refreshClassCache: clearLookupCachesBool

"Refreshes in-memory state of the class .
 If clearLookupCachesBool==true, also clears all method lookup caches.
 Must also be sent immediately after any change to the
 format or named instance variables of the receiver."

<primitive: 227>
self _primitiveFailed:#_refreshClassCache: args: { clearLookupCachesBool } .
self _uncontinuableError
%

category: 'Virtual Machine Control'
method: Behavior
_refreshLookupCache: aSelector oldMethod: aMethod env: envId

"This message must be sent whenever a class's method dictionary
 changes, to keep lookup caches current.

 If not nil, aSelector specifieds a selector added or removed from receiver's
 method dictionary(s) for specified envId .

 If the argument aMethod is not nil, any breakpoints in the obsolete
 method aMethod are cleared.
 If aMethod is a SmallInteger, it is assumed to be a Ruby method-hidden
 value , and breakpoint clearing logic is skipped.

 envId must be a SmallInteger >= 0 and <= 255 .
"

<primitive: 374>
aSelector ifNotNil:[ aSelector _validateClass: Symbol ] .
aMethod ifNotNil:[ | gsMethod |
   gsMethod := ObsoleteClasses at: #GsMethod .
   (aMethod isKindOf: gsMethod) ifTrue: [
     ^ self "ignore unconverted method from Gs64 2.x"
   ].
   aMethod _validateClass: gsMethod
].
envId _validateClass: SmallInteger .
self _primitiveFailed: #_refreshLookupCache:oldMethod:env:
     args: { aSelector . aMethod . envId }.
self _uncontinuableError
%

category: 'Updating Categories'
method: Behavior
_removeBaseCategory: categoryName environmentId: envId

"Removes the specified category and all its methods from the receiver's
 method dictionary.  If categoryName is not in the receiver's categorys
 generates an error.  Any breakpoints in removed methods are cleared."
  | cats catName |
  self _validatePrivilege ifFalse:[ ^ nil ].
  catName := Symbol _existingWithAll: categoryName .
  catName ifNil:[ ^ self _error: #rtErrKeyNotFound args: { categoryName } ].
  cats := self _baseCategorys: envId .
  cats ifNotNil:[ | selectors |
    selectors := cats at: catName .
    selectors do:[ :aSelector |
       self _basicRemoveSelector: aSelector environmentId: envId
    ].
    cats removeKey: catName
  ]  ifNil:[
    self _error: #rtErrKeyNotFound args: { categoryName }
  ]
%

category: 'Modifying Classes'
method: Behavior
_removeClassVar: aSymbol

"Removes the class variable with name aSymbol from the receiver's
 class variables dictionary.  Generates an error if aSymbol is not
 the name of a class variable of the receiver."

  self _validatePrivilege ifTrue:[
    classVars ifNil:[ ^ self _errorKeyNotFound: aSymbol ].
    classVars removeKey: aSymbol
            ifAbsent: [classVars _errorKeyNotFound: aSymbol].
  ]
%

category: 'Modifying Classes'
method: Behavior
_removeClassVar: aSymbol ifAbsent: exceptionBlock

"Removes the class variable with name aSymbol from the receiver's
 class variables dictionary.  Executes exceptionBlock if aSymbol is not
 the name of a class variable of the receiver."
  | dict |
  self _validatePrivilege ifTrue:[
    (dict := classVars) ifNil:[ exceptionBlock value ]
       ifNotNil:[ dict removeKey: aSymbol ifAbsent: exceptionBlock ].
  ]
%

category: 'Private Methods for Class Modification'
method: Behavior
_removeInstVarAtOffset: offset
	"Remove named instance variable at specified offset from self and all
 subclasses assuming that all error checks have been done."

	self _validatePrivilege
		ifTrue:
			[self _incrementInstVars: -1.
			instVarNames removeFrom: offset to: offset.
			self _refreshClassCache: false.
			self _subclasses do: [:x | x _removeInstVarAtOffset: offset]]
%

category: 'Reloading Decompiled Methods'
method: Behavior
_resolveClassOrPoolVar: aSymbol

"Searches the receiver's class variables dictionary and pool dictionaries to
 attempt to resolve aSymbol.  Returns the SymbolAssociation for the variable
 with name aSymbol, or nil if aSymbol could not be found."

| assoc |
assoc := self _resolveClassVar: aSymbol .
assoc ~~ nil ifTrue:[ ^ assoc ].

poolDictionaries size ~~ 0 ifTrue:[
  1 to: poolDictionaries size do:[:j | | aDict |
    aDict := poolDictionaries at: j .
    assoc := aDict associationAt: aSymbol otherwise: nil .
    assoc ~~ nil ifTrue:[ ^ assoc ].
    ].
  ].
^ nil
%

category: 'Reloading Decompiled Methods'
method: Behavior
_resolveClassVar: aSymbol

"Searches the receiver's class variables dictionary, to attempt to resolve
 aSymbol.  Returns the SymbolAssociation for the variable with name aSymbol, or
 nil if aSymbol could not be found."

| assoc |

classVars ifNotNil:[ :cvs |
  assoc := cvs associationAt: aSymbol otherwise: nil.
  ].
^ assoc
%

category: 'Reloading Decompiled Methods'
method: Behavior
_resolveLiteralVar: aSymbol

"Attempts to resolve a literal variable with name aSymbol.  To attempt to
 resolve aSymbol, searches the receiver's class variable dictionary and its pool
 dictionaries, then the superclasses' class variables, then the current default
 SymbolList.

 Returns the SymbolAssociation for the variable or nil if aSymbol could not be
 found."

"Implementation here must agree with comgen.c:searchClassOrPool()"

| assoc aClass |

aClass := self .
"search receiver's class variables and pool variables"
assoc := aClass _resolveClassOrPoolVar: aSymbol .
assoc ~~ nil ifTrue:[ ^ assoc ].

"search for a class variable inherited from superclasses"
[ (aClass := aClass superClass) ~~ nil
 ]
whileTrue: [
  assoc := aClass _resolveClassVar: aSymbol .
  assoc ~~ nil ifTrue:[ ^ assoc ].
] .
"search the symbol list"
^ GsSession currentSession resolveSymbol: aSymbol
%

category: 'Browser Methods'
method: Behavior
_selectorPrefixesReport: envId
  "In base Smalltalk image, no ruby prefix logic"
  ^ self _selectorsReport: envId
%

category: 'Accessing Categories'
method: Behavior
_selectorsInBaseCategory: aSymbol

  ^ (self _baseCategorys: 0) at: aSymbol
%

category: 'Browser Methods'
method: Behavior
_selectorsReport: envId
  ^ self _selectorsReport: envId matching: nil
%

category: 'Browser Methods'
method: Behavior
_selectorsReport: envId matching: aString
 ^ self _selectorsReport: envId matching: aString primitivesOnly: false
%

category: 'Browser Methods'
method: Behavior
_selectorsReport: envId matching: aString primitivesOnly: primsBoolean

^ self _selectorsReport: envId matching: aString primitivesOnly: primsBoolean
       includeDeprecated: true
%

category: 'Browser Methods'
method: Behavior
_selectorsReport: envId matching: aString primitivesOnly: primsBoolean includeDeprecated: inclDeprecBool
 "Used by topaz (and GBS?).
  Result is a sorted SequenceableCollection  of Symbols, plus an optional string 'Omitted .. deprecated methods'.
  aString if not nil restricts the result to only include those selectors containing
  the case-insensitive substring aString .
  primsBoolean if true further restricts the result to only include only
  selectors of methods which are primitives."
 | list res deprecSet numDeprecated |
 numDeprecated := 0 .
 inclDeprecBool ifFalse:[
   deprecSet := Object _selectorsInBaseCategory:#'Deprecated Notification'.
 ].
 list := self selectorsForEnvironment: envId .
 res := SortedCollection new .
 list do:[:sym | | sel |
    (aString == nil or:[ (sym includesString: aString)]) ifTrue:[
       sel := sym .
       (primsBoolean or:[ deprecSet ~~ nil ]) ifTrue:[
         (self compiledMethodAt: sym environmentId: envId otherwise: nil) ifNotNil:[ :meth|
           primsBoolean ifTrue:[ meth _isPrimitive ifFalse:[ sel := nil ]].
           sel ifNotNil:[
             deprecSet ifNotNil:[ (meth _selectorPool * deprecSet) size ~~ 0 ifTrue:[
               sel := nil . numDeprecated := numDeprecated + 1 ]].
           ].
         ]
       ].
       sel ifNotNil:[ res add: sel ].
    ].
 ].
 numDeprecated > 0 ifTrue:[
   res := Array withAll: res .
   res add:'(Omitted ' , numDeprecated asString, ' deprecated methods)'.
 ].
 ^ res
%

category: 'Updating the Method Dictionary'
method: Behavior
_sessionMethodsChanged
  "set bit in VM's cbCodeChanged word so it can propagate to
   other VMs upon commit. 
   System class >> _sessionMethodsChanged*  returns union of bit
   with respect to a view change."
  ^ self _codeChangedForEnv: -1 .
%

category: 'Debugging Support'
method: Behavior
_setMethodBreak: aSelector breakpointLevel: brkLevel

"Returns true to indicate success.  Otherwise returns a string describing the error.
 brkLevel  >=1  means signal to Smalltalk , 0 means to GCI"

^ self _setMethodBreak: aSelector stepPoint: 1 env: 0 breakpointLevel: brkLevel 
%

category: 'Debugging Support'
method: Behavior
_setMethodBreak: aSelector stepPoint: anInt
 "Set a breakpoint that will be signalled to the GCI.
  Returns true to indicate success.  Otherwise returns a string describing the error."

 ^ self _setMethodBreak: aSelector stepPoint: anInt env: 0 breakpointLevel: 0"To GCI"
%

category: 'Debugging Support'
method: Behavior
_setMethodBreak: aSelector stepPoint: anInt env: envId

 "Set a breakpoint that will be signalled to the GCI.
  Returns true to indicate success.  Otherwise returns a string describing the error."

^ self _setMethodBreak: aSelector stepPoint: anInt env: envId breakpointLevel: 0"to GCI"
%

category: 'Debugging Support'
method: Behavior
_setMethodBreak: aSelector stepPoint: anInt env: envId breakpointLevel: brkLevel

"Returns true to indicate success.  Otherwise returns a string describing the error.
 brkLevel  >=1  means signal to Smalltalk , 0 means to GCI"

  | selectorSym x |
  (aSelector isByteKindOf: CharacterCollection) ifFalse: [ ^ 'Illegal selector' ].
  selectorSym := Symbol _existingWithAll: aSelector .
  selectorSym ifNil:[ ^ 'selector is not an existing Symbol'].
  ((x := GsNMethod optimizedSelectors) includesIdentical: selectorSym) ifTrue:[
      ^ 'You may not set a method break on an optimized selector' ].

  (self == SmallInteger
      and: [ #(  #+  #-  #>=  #*  #=  ) includesIdentical: selectorSym])
        ifTrue:[ ^ 'You may not set a method break on an optimized selector' ].

  (anInt _isSmallInteger) ifFalse:[ ^ 'Step point must be a SmallInteger' ].
  (self includesSelector: selectorSym environmentId: envId)
      ifFalse: [ ^ 'Selector does not exist in class' ].

  ((self compiledMethodAt: selectorSym environmentId: envId) 
    setBreakAtStepPoint: anInt breakpointLevel: brkLevel ) 
  ifNil:[ ^ 'Step point does not exist in method' ].

  ^ true
%

category: 'Private'
method: Behavior
_setSelfCanBeSpecial

  "Set the SelfCanBeSpecial bit in the receiver.
   For use only after altering the superclass hierarchy in such a way
   as to insert new classes between a class for which isSpecial is true
   and Object."

format := format bitOr: 16r2000 .
%

category: 'Debugging Support'
method: Behavior
_sourceCodeAndOffsets: aSelector
  ^ self _sourceCodeAndOffsets: aSelector environmentId: 0
%

category: 'Debugging Support'
method: Behavior
_sourceCodeAndOffsets: aSelector environmentId: envId

"Returns an Array with two elements.  The first element is a String
 representing the source code for the argument, aSelector.  The second element
 is an InvariantArray (that holds SmallIntegers) is a list of offsets into
 sourceString, corresponding in order to the step points.  If aSelector (a
 String) is not a selector in the receiver's method dictionary, returns nil."

| method |
method := self compiledMethodAt: aSelector environmentId: envId otherwise: nil .
method ifNotNil:[ ^ { method _sourceString . method _sourceOffsets } ] .
^ nil
%

category: 'Private'
method: Behavior
_stripAllMethodSources

"Returns true if all method sources should be stripped for the receiver, and
 false otherwise."

"Returns true for the list of classes given in the code, false for others."

| classNameList |
" classNameList is an Array of Symbols because some of these classes
are not known at filein "
classNameList := #( ).

1 to: classNameList size do: [ :i |
  ((Globals at: (classNameList at: i)) == self)
    ifTrue: [ ^ true ]
].
^ false
%

category: 'Private'
method: Behavior
_structuralUpdatesDisallowed

"Private.

 Returns true if GemBuilder for C (GCI) direct structural update of instances
 is disallowed, false otherwise.  A result of true means that the deferred
 update mechanism is used (see GciProcessDeferredUpdates in gci.hf) by
 GemBuilder for C store operations on instances of the receiver."

^ (format bitAnd: 16r200 "no update through structural access") ~~ 0
%

category: 'Modifying Classes'
method: Behavior
_subclasses

"Returns the class variable that is the list of subclasses, or
 nil if the receiver does not keep track of subclasses."

^ nil
%

category: 'Testing Inheritance'
method: Behavior
_subclassOf: aClass
"Returns true if the receiver is identical to or is a subclass aClass.

 aClass must be a Class or a Metaclass3 or nil .
 ClassHistories of the receiver and argument are ignored.

 If argument is nil and the receiver is a subclass of Object, returns false.
 If argument is nil and the receiver is not a subclass of Object, returns true.
"

<primitive: 895>
self _primitiveFailed: #_subclassOf: args: { aClass } .
^ false
%

category: 'Private'
method: Behavior
_superclass
  ^ superClass
%

category: 'Accessing the Method Dictionary'
method: Behavior
_topazMethodAt: aString env: envId
  "Returns a GsNMethod, or signals an Error."
  ^ [ self compiledMethodAt: aString
           environmentId: envId
    ] on: LookupError do: [:ex |
      nil
    ]
%

category: 'Private'
method: Behavior
_traversalByCallback

"Private.

 Returns true if GemBuilder for C (GCI) traversal results of instances
 are obtained by message send of aClampSpecification.traversalCallBackSelector,
 false otherwise."

^ (format bitAnd: 16r400"travByCallback") ~~ 0
%

category: 'Private Methods for Class Modification'
method: Behavior
_validateNewNamedInstVar: aSymbol

"Generate an error if the argument is the name of an already existing instance
 variable of the receiver or if the receiver is not modifiable and has not
 disallowed subclassing, otherwise return true. If execution is continued
 from the error, return false."

"reimplementation of self validateIsModifiable ."
self isModifiable ifFalse:[
  self isNsc ifTrue:[
    self _error: #rtErrClassNotModifiable .
    ^ false .
  ].
  self subclassesDisallowed ifFalse:[
    self _error: #rtErrClassNotModifiable .
    ^ false .
  ] .
  self isIndexable ifTrue:[
     self _error: #rtErrClassNotModifiable.
    ^ false
  ].
] .
(instVarNames includesIdentical: aSymbol) ifTrue:[
  self _error: #rtErrAddDupInstvar args:{ aSymbol } .
  ^ false
].
self _subclasses ifNotNil:[ :mySubclasses |
  mySubclasses do:[:aSubCls|
     (aSubCls _validateNewNamedInstVar: aSymbol) ifFalse:[ ^ false ].
  ].
].
^ true
%

category: 'Private'
method: Behavior
_validatePrivilege

^ System myUserProfile _validateCodeModificationPrivilege
%

category: 'Private'
method: Behavior
_varyingConstraint

"Returns the constraint on the unnamed part of the receiver (a kind of Class).
 If the receiver has no constraint on its unnamed part, or if it has no unnamed
 part, this method returns Object.

 As of GemStone 64bit v3.4, constraints are no longer implemented.
 This method is provided for examining classes in repositories upgraded from
 an older version."

 ^ self _constraintAt: self instSize + 1 .
%

category: 'Accessing Categories'
method: Behavior
__categorys
  ^ categorys
%

! Class extensions for 'BitSet'

!		Class methods for 'BitSet'

removeallmethods BitSet
removeallclassmethods BitSet

category: 'Instance creation'
classmethod: BitSet
new: numberOfBits
  "allocated size is always a multiple of 4 bytes,
   all bits in the result are set to zero."

^ super new: ((numberOfBits + 31) // 32) * 4
%

!		Instance methods for 'BitSet'

category: 'Accessing'
method: BitSet
at: offset
"Return the specified bit, result is 0 or 1.
 offset is a zero-based SmallInteger,
 0 specifies the least significant bit.
 offset < 0 generates an error.
 0 is returned for any offset > allocated size. "

<primitive: 748>
offset _validateClass: SmallInteger.
offset < 0 ifTrue:[ offset _error: #errArgTooSmall args:{ 0 } ].
self _primitiveFailed: #at: args: { offset }
%

category: 'Updating'
method: BitSet
at: offset put: aValue

"offset is zero-based, 0 specifies the least significant bit.
 aValue must be 0 or 1 .
 Returns previous state of the specified bit. "

<primitive: 751>
offset _validateClass: SmallInteger.
offset < 0 ifTrue:[ offset _error: #errArgTooSmall args:{ 0 } ].
(aValue == 0 or:[ aValue == 1]) ifFalse:[
  aValue _error: #rtErrArgOutOfRange args:{ 0 . 1 } ].
self _primitiveFailed: #at:put: args: { offset . aValue }
%

category: 'Updating'
method: BitSet
bitAnd: aBitSet

"clear bits in the receiver which are not also set in the argument.
 aBitSet must be an instance of the class of the receiver.
 Returns receiver. "

<primitive: 752>
aBitSet _validateInstanceOf: self class .
self _primitiveFailed: #bitAnd: args: { aBitSet }
%

category: 'Updating'
method: BitSet
bitInvert
"Inverts the  bits of the receiver."

<primitive: 755>
self _primitiveFailed: #bitInvert
%

category: 'Updating'
method: BitSet
bitOr: aBitSet

"Set bits in the receiver that are set in the argument
 Grows receiver as needed.
 aBitSet must be an instance of the class of the receiver.
 Returns receiver. "

<primitive: 753>
aBitSet _validateInstanceOf: self class .
self _primitiveFailed: #bitOr: args: { aBitSet }
%

category: 'Updating'
method: BitSet
bitXor: aBitSet
"Sets receiver to be the exclusive-or of the receiver with argument.
 Grows receiver as needed.
 aBitSet must be an instance of the class of the receiver.
 Returns receiver."
<primitive: 754>
aBitSet _validateInstanceOf: self class .
self _primitiveFailed: #bitXor: args: { aBitSet }
%

category: 'Updating'
method: BitSet
clear
  "set all bits in receiver to zero, returns receiver.
   does not change allocated size of receiver."
<primitive: 750>
self _primitiveFailed: #clear
%

category: 'Accessing'
method: BitSet
countBits
"return the number of bits set in the receiver"

<primitive: 749>
self _primitiveFailed: #countBits
%

category: 'Accessing'
method: BitSet
size

"return the allocated size in bits"

^ super size * 8
%

category: 'Updating'
method: BitSet
size: numberOfBits

 "change the allocated size. allocated size is always a multiple of 4 bytes,
  rounded up from the specified size.
  If the new size is larger than current size,
   newly allocated bits will be set to zero."

super size: ((numberOfBits + 31) // 32) * 4
%

! Class extensions for 'BlockClosure'

!		Class methods for 'BlockClosure'

removeallmethods BlockClosure
removeallclassmethods BlockClosure

category: 'Instance Creation'
classmethod: BlockClosure
new

"Disallowed."

self shouldNotImplement: #new
%

category: 'Reloading Decompiled Methods'
classmethod: BlockClosure
_new

"Private."

^ super new .
%

!		Instance methods for 'BlockClosure'

category: 'Accessing'
method: BlockClosure
argumentCount
  "Answer the number of arguments needed to evaluate the receiver."

  ^self subclassResponsibility: #argumentCount.
%

category: 'Continuations'
method: BlockClosure
callCC
   "Evaluate the receiver, which must be a one-argument block.
   The argument to the block will be a newly created continuation,
   which is an instance of GsProcess that contains a copy of the
   active process. Unless control flow is otherwise interrupted,
   return the result of evaluating the last expression in the block.

   If a reference is retained to the continuation that is created
   during the execution of this method, it may later be evaluated
   using the continuation's value: or value methods (distinct from
   the BlockClosure's value: and value methods). When the continuation
   value: is evaluated, it is effectively a GOTO to the point in
   execution that the original call to callCC returned, (optionally)
   returning a different value. The actual callCC block is only
   executed once; later evaluations of the continuation return from
   the block, but do not re-execute the block."

   ^ self value: (GsProcess continuationFromLevel: 2)
%

category: 'Copying'
method: BlockClosure
copy

"Returns the receiver. Copies of blocks are not made."

^ self.
%

category: 'Flow of Control'
method: BlockClosure
doWhileFalse: conditionBlock
  "Evaluate the receiver once, then again as long the value of conditionBlock is false."
  ^self subclassResponsibility: #doWhileFalse:
%

category: 'Block Evaluation'
method: BlockClosure
ensure: aBlock
  "Evaluate the receiver.
  Evaluate aBlock after evaluating the receiver,
  or before any return from a block that would return to the sender.
  Returns result of evaluating the receiver.

  aBlock must be a zero-arg instance of ExecBlock,
  otherwise an error is generated."

  ^self subclassResponsibility: #ensure:.
%

category: 'Block Evaluation'
method: BlockClosure
ifCurtailed: terminationBlock
  "Evaluate the receiver and return its result.
  If abnormal termination of the receiver occurs, terminationBlock is
  evaluated. The value returned from the evaluation of terminationBlock
  is discarded.

  Activation of an exception handler from within the receiver is not in and
  of itself an abnormal termination. However, if the exception handler for
  an exception that is not resumable results in termination of the receiver
  or if its handler block contains a return statement that results in
  abnormal termination of the receiver, then terminationBlock will be evaluated
  after evaluation of the exception handler.

  If an abnormal termination result in the termination of multiple blocks
  which were evaluated using either #ensure: or #ifCurtailed: the respective
  terminationBlocks will be executed in the reverse of the order in which
  the corresponding receiver blocks were evaluated."

  | wasCurtailed result |
  wasCurtailed := true.
  ^[
     result := self value.
     wasCurtailed := false.
     result
  ] ensure:[
    wasCurtailed ifTrue: terminationBlock
  ].
%

category: 'Updating'
method: BlockClosure
instVarAt: anIndex put: aValue

"Disallowed."

self shouldNotImplement: #instVarAt:put:
%

category: 'Testing'
method: BlockClosure
isSimple

"Returns the default answer, false."

^ false
%

category: 'Block Evaluation'
method: BlockClosure
on: exceptionSelector do: handlerBlock
	"Try to evaluate the receiver, which should be a zero-argument block.
	 If an exception occurs and the expression exceptionSelector handles: theExceptionInstance
	 returns true, then evaluate the one argument block handlerBlock, passing it the exception
	 instance as its argument.

	 These forms are supported directly by the VM
		(1) on: anException do: handlerBlock
		(2) on: anExceptionSet do: handlerBlock
	 This form is handled by Smalltalk code in the body of this method.
		(3) on: anObject do: handlerBlock

	 anException must be the class Exception  or a subclass thereof;
	 anExceptionSet must be a kind of ExceptionSet;
	 handlerBlock must be an instance of ExecBloc otherwise an error is generated.

	 For forms 1, 2, and 3 if handlerBlock is not a one argument block,
	 an error is generated if exception handling attempts to invoke that handlerBlock.

	 If handlerBlock is invoked to handle an Exception which occurs during execution of the
	 receiver and handlerBlock completes normally , then the result of handlerBlock (the value
	 of the last expression in handlerBlock) will be the result of the on:do: send .  Other-than-normal
	 completion of handlerBlock is available by use of Exception's instance methods such as
	 #return, #return:, #retry, #retryUsing:, #resume, #resume:, #pass, and #outer within handlerBlock.

	 For forms 1 and 2, when searching for a handlerBlock to handle a signaled Exception, the VM
	 uses Behavior>>_subclassOf: semantics . classHistories of the class of the signaled Exception
	 and of anException or elements of anExceptionSet are ignored."

	^self subclassResponsibility: #on:do:.
%

category: 'Flow of Control'
method: BlockClosure
untilFalse

  ^self subclassResponsibility: #untilFalse
%

category: 'Flow of Control'
method: BlockClosure
untilFalse: aBlock

  ^self subclassResponsibility: #untilFalse:
%

category: 'Block Evaluation'
method: BlockClosure
value
	"Return the value of the receiver evaluated with no arguments.
	 If the receiver expects any arguments, signal an error."

	^self subclassResponsibility: #value.
%

category: 'Block Evaluation'
method: BlockClosure
value: anObject
	"Return the value of the receiver evaluated with anObject as its argument.  If
	 the receiver expects a different number of arguments, signal an error."

	^self subclassResponsibility: #value:.
%

category: 'Block Evaluation'
method: BlockClosure
value: firstObject value: secondObject
	"Return the value of the receiver evaluated with the specified arguments.  If
	 the receiver expects a different number of arguments, signal an error."

	^self subclassResponsibility: #value:value:.
%

category: 'Block Evaluation'
method: BlockClosure
valueWithArguments: argList
	"Return the value of the receiver evaluated with the specified arguments.  If
	 the receiver expects a different number of arguments, signal an error."

	^self subclassResponsibility: #valueWithArguments:.
%

category: 'Flow of Control'
method: BlockClosure
whileFalse
	"Evaluate the receiver once and then repeatedly as long as the value
	 returned by the evaluation is false."

	^self subclassResponsibility: #whileFalse.
%

category: 'Flow of Control'
method: BlockClosure
whileFalse: aBlock
	"(Reserved selector.)  Evaluates the zero-argument block aBlock repeatedly
	 while the receiver evaluates to false.  Return nil.  Generates an error if the
	 receiver is not a zero-argument block."

	^self subclassResponsibility: #whileFalse:.
%

category: 'Flow of Control'
method: BlockClosure
whileTrue
	"Evaluate the receiver once and then repeatedly as long as the value
	 returned by the evaluation is true."

	^self subclassResponsibility: #whileTrue.
%

category: 'Flow of Control'
method: BlockClosure
whileTrue: aBlock
	"(Reserved selector.)  Evaluates the zero-argument block aBlock repeatedly
	 while the receiver evaluates to true.  Return nil.  Generates an error if the
	 receiver is not a zero-argument block."

	^self subclassResponsibility: #whileTrue:.
%

category: 'Storing and Loading'
method: BlockClosure
writeTo: aPassiveObject

"Instances of BlockClosure cannot be converted to passive form.  This method
 writes nil to aPassiveObject and stops GemStone Smalltalk execution with a
 notifier."

aPassiveObject writeObject: nil.
self _error: #rtErrAttemptToPassivateInvalidObject.
%

category: 'Private'
method: BlockClosure
_deepCopyWith: copiedObjDict

"Private.  Used internally to implement deepCopy."

^ self.
%

! Class extensions for 'Boolean'

!		Class methods for 'Boolean'

removeallmethods Boolean
removeallclassmethods Boolean

category: 'Instance Creation'
classmethod: Boolean
fromStream: aStream

"If the next characters in aStream are true or false (case insensitive, leading
 spaces permitted), this method returns the appropriate Boolean.  Otherwise,
 generates an error."

| char |

self _checkReadStream: aStream forClass: CharacterCollection.

char := aStream next.
[ char == $  ] whileTrue: [ char := aStream next ].
aStream skip: -1.

((char := aStream next) isEquivalent: $T)
ifTrue:
  [ ((aStream next isEquivalent: $R)
    and: [(aStream next isEquivalent: $U)
    and: [aStream next isEquivalent: $E ]])
    ifTrue:
      [ ^ true ]
  ]
ifFalse:
  [ (char isEquivalent: $F)
    ifTrue:
      [ ((aStream next isEquivalent: $A)
        and: [(aStream next isEquivalent: $L)
        and: [(aStream next isEquivalent: $S)
        and: [aStream next isEquivalent: $E ]]])
        ifTrue:
          [ ^ false ]
      ]
  ].

self _errIncorrectFormat: aStream
%

category: 'Instance Creation'
classmethod: Boolean
fromString: aString

"If aString contains true or false, returns the appropriate Boolean.  Leading
 and trailing spaces are permitted in the String.  If aString contains any
 characters other than true or false, this method generates an error."

| s result |

s := ReadStreamPortable on: aString.
result := self fromStream: s.
[ s atEnd ]
whileFalse:
  [ (s next == $  )
    ifFalse:
      [ self _errIncorrectFormat: aString ]
  ].
^ result
%

category: 'Storing and Loading'
classmethod: Boolean
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form and returns an equivalent instance of Boolean."

"This method supports filein of Booleans in the format required by GeODE
 versions before 2.0.1."

|inst|
inst := passiveObj next == $t.
passiveObj hasRead: inst.
^inst
%

category: 'Instance Creation'
classmethod: Boolean
new

"Disallowed.  You cannot create new instances of Boolean."

self shouldNotImplement: #new
%

!		Instance methods for 'Boolean'

category: 'Logical Operations'
method: Boolean
& aBoolean

"Evaluating conjunction (AND).  Returns true if both the receiver and the
 argument aBoolean are true."

<primitive: 42>
aBoolean _validateClass: Boolean.
self _primitiveFailed: #& args: { aBoolean }.
self _uncontinuableError
%

category: 'Flow of Control'
method: Boolean
and: aBlock

"(Reserved selector.) Nonevaluating conjunction.  Returns the value of the
 zero-argument block aBlock if the receiver is true.  Otherwise, returns false
 without evaluating the argument.

 The following is an optimized control structure, not a recursive send."

^ self and: [ 
   aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
   aBlock value
 ]
%

category: 'Converting'
method: Boolean
asBit
  "Returns 1 if self==true, 0 otherwise"

  ^ self ifTrue:[ 1 ] ifFalse:[ 0 ]
%

category: 'Formatting'
method: Boolean
asString

"Returns a String containing true or false, depending on the receiver."

self
  ifTrue:
    [ ^ String withAll: 'true' ]
  ifFalse:
    [ ^ String withAll: 'false' ]
%

category: 'Converting'
method: Boolean
asSymbol
 "Returns #true if self==true, #false otherwise"

  ^ self ifTrue:[ #true ] ifFalse:[ #false ]
%

category: 'Clustering'
method: Boolean
clusterDepthFirst

"Returns true.  (Because Booleans are self-defining objects, this method has no
 effect.)"

^ true
%

category: 'Storing and Loading'
method: Boolean
containsIdentity

"Private."

^true
%

category: 'Copying'
method: Boolean
copy

"Overrides the inherited method to return the receiver.  The pseudo-variables
 true and false are the only instances of Boolean, and must preserve identity."

^ self
%

category: 'Logical Operations'
method: Boolean
eqv: aBoolean

"Returns true if the receiver is identical to aBoolean."

^ self == aBoolean
%

category: 'Flow of Control'
method: Boolean
ifFalse: aBlock

"(Reserved selector.)  Returns the value of the zero-argument block aBlock if
 the receiver is false.  Otherwise, returns nil without evaluating the
 argument."

"The following is an optimized control structure, not a recursive send."

^ self ifFalse: [
    aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
   ^ aBlock value 
 ]
%

category: 'Flow of Control'
method: Boolean
ifFalse: falseBlock ifTrue: trueBlock

"(Reserved selector.)  Returns the value of the zero-argument block falseBlock
 if the receiver is false.  Otherwise, returns the value of the zero-argument
 block trueBlock without evaluating falseBlock."

"The following is an optimized control structure, not a recursive send."

self ifFalse:[
  falseBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
  ^ falseBlock value
] ifTrue: [
  trueBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
  ^ trueBlock value
]
%

category: 'Flow of Control'
method: Boolean
ifTrue: aBlock

"(Reserved selector.)  Returns the value of the zero-argument block aBlock if
 the receiver is true.  Otherwise, returns nil without evaluating the argument."

"The following is an optimized control structure, not a recursive send."

^ self ifTrue: [
   aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
   ^ aBlock value
]
%

category: 'Flow of Control'
method: Boolean
ifTrue: trueBlock ifFalse: falseBlock

"(Reserved selector.)  Returns the value of the zero-argument block falseBlock
 if the receiver is false.  Otherwise, returns the value of the zero-argument
 block trueBlock without evaluating falseBlock."

"The following is an optimized control structure, not a recursive send."

^ self ifTrue: [
   trueBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
   ^ trueBlock value
  ] ifFalse: [ 
   falseBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
   ^ falseBlock value
  ]
%

category: 'Testing'
method: Boolean
isSpecial

"Returns true if the receiver is a special object."

^ true
%

category: 'Logical Operations'
method: Boolean
not

"This implementation is used only by #perform .  
 If the receiver is a Boolean, #not is a special selector that does not do 
 a message send.

 Negation.  Returns true if the receiver is false.  Returns false if the
 receiver is true."

^ self == false
%

category: 'Flow of Control'
method: Boolean
or: aBlock

"(Reserved selector.) Nonevaluating disjunction.  Returns the value of the
 zero-argument block aBlock if the receiver is false.  Otherwise, returns true
 without evaluating the argument.

 The following is an optimized control structure, not a recursive send."

^ self or:[ 
    aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
    aBlock value 
 ]
%

category: 'Json'
method: Boolean
printJsonOn: aStream

	aStream nextPutAll: (self ifTrue: ['true'] ifFalse: ['false']).
%

category: 'Formatting'
method: Boolean
printString

"Returns a String whose contents are a displayable representation of the
 receiver."

"GemStone does not allow the creation of new kinds of Boolean, so there is no
 point in creating a stream and sending printOn:."

^self asString
%

category: 'Storing and Loading'
method: Boolean
writeTo: passiveObj

"Converts the receiver to its passive form and writes that information on
 passiveObj."

passiveObj nextPut: (self ifTrue: [ $* ] ifFalse: [ $~ ])

%

category: 'Logical Operations'
method: Boolean
xor: aBoolean

"Exclusive OR.  Returns true if the receiver is not identical to aBoolean."

<primitive: 45>
aBoolean _validateClass: Boolean.
self _primitiveFailed: #xor: args: { aBoolean }.
self _uncontinuableError
%

category: 'Flow of Control'
method: Boolean
_and: aBlock

"(Reserved selector.)  Nonevaluating conjunction.  Returns the value of the
 zero-argument block aBlock if the receiver is true.  Otherwise, returns false
 without evaluating the argument.

 The following is an optimized control structure, not a recursive send."

^ self and: [ 
    aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
    aBlock value
  ]
%

category: 'Decompiling without Sources'
method: Boolean
_asSource

^ self asString
%

category: 'New Indexing Comparison'
method: Boolean
_classSortOrdinal

^ 20
%

category: 'New Indexing Comparison - for Compare'
method: Boolean
_idxForCompareBooleanGreaterThanOrEqualToSelf: aBoolean

"second half of a double dispatch call from Boolean>>_idxForCompareGreaterThanOrEqualTo:. Note that aBoolean should be the receiver in any >= comparison"

aBoolean = self
  ifTrue: [ ^true].
^aBoolean == true
%

category: 'New Indexing Comparison - for Compare'
method: Boolean
_idxForCompareBooleanGreaterThanSelf: aBoolean

"second half of a double dispatch call from Boolean>>_idxForCompareGreaterThan:. Note that aBoolean should be the receiver in any > comparison"

self = aBoolean
    ifTrue: [ ^ false ].
^aBoolean == true
%

category: 'New Indexing Comparison - for Compare'
method: Boolean
_idxForCompareBooleanLessThanOrEqualToSelf: aBoolean

"second half of a double dispatch call from Boolean>>_idxForCompareLessThanOrEqualTo:. Note that aBoolean should be the receiver in any <= comparison"

aBoolean = self
  ifTrue: [ ^true].
^aBoolean == false
%

category: 'New Indexing Comparison - for Compare'
method: Boolean
_idxForCompareBooleanLessThanSelf: aBoolean

"second half of a double dispatch call from Boolean>>_idxForCompareLessThan:. Note that aBoolean should be the receiver in any < comparison"

self = aBoolean
    ifTrue: [ ^ false ].
^aBoolean == false
%

category: 'New Indexing Comparison'
method: Boolean
_idxForCompareGreaterThan: arg

""

^arg _idxForCompareBooleanGreaterThanSelf: self
%

category: 'New Indexing Comparison'
method: Boolean
_idxForCompareGreaterThanOrEqualTo: arg

""

^arg _idxForCompareBooleanGreaterThanOrEqualToSelf: self
%

category: 'New Indexing Comparison'
method: Boolean
_idxForCompareLessThan: arg

""

^arg _idxForCompareBooleanLessThanSelf: self
%

category: 'New Indexing Comparison'
method: Boolean
_idxForCompareLessThanOrEqualTo: arg

""

^arg _idxForCompareBooleanLessThanOrEqualToSelf: self
%

category: 'New Indexing Comparison - for Sort'
method: Boolean
_idxForSortBooleanGreaterThanOrEqualToSelf: anObject

"second half of a double dispatch call from Boolean>>_idxForSortGreaterThanOrEqualTo:. Note that aBoolean should be the receiver in any >= comparison"

anObject = self
  ifTrue: [ ^ true ].
^anObject
%

category: 'New Indexing Comparison - for Sort'
method: Boolean
_idxForSortBooleanGreaterThanSelf: anObject

"second half of a double dispatch call from Boolean>>_idxForSortGreaterThan:. Note that aBoolean should be the receiver in any > comparison"

anObject = self
  ifTrue: [ ^ false ].
^anObject
%

category: 'New Indexing Comparison - for Sort'
method: Boolean
_idxForSortBooleanLessThanOrEqualToSelf: anObject

"second half of a double dispatch call from Boolean>>_idxForSortLessThanOrEqualTo:. Note that aBoolean should be the receiver in any <= comparison"

anObject = self
  ifTrue: [ ^ true ].
^anObject not
%

category: 'New Indexing Comparison - for Sort'
method: Boolean
_idxForSortBooleanLessThanSelf: anObject

"second half of a double dispatch call from Boolean>>_idxForSortLessThan:. Note that aBoolean
should be the receiver in any < comparison"

anObject = self
  ifTrue: [ ^ false ].
^anObject not
%

category: 'New Indexing Comparison'
method: Boolean
_idxForSortGreaterThan: arg

""

^arg _idxForSortBooleanGreaterThanSelf: self
%

category: 'New Indexing Comparison'
method: Boolean
_idxForSortGreaterThanOrEqualTo: arg

""

^arg _idxForSortBooleanGreaterThanOrEqualToSelf: self
%

category: 'New Indexing Comparison'
method: Boolean
_idxForSortLessThan: arg

""

^arg _idxForSortBooleanLessThanSelf: self
%

category: 'New Indexing Comparison'
method: Boolean
_idxForSortLessThanOrEqualTo: arg

""

^arg _idxForSortBooleanLessThanOrEqualToSelf: self
%

category: 'Flow of Control'
method: Boolean
_or: aBlock

"(Reserved selector.)  Nonevaluating disjunction.  Returns the value of the
 zero-argument block aBlock if the receiver is false.  Otherwise, returns true
 without evaluating the argument.

 The following is an optimized control structure, not a recursive send."

^ self or:[ 
    aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
    aBlock value 
  ]
%

category: 'Logical Operations'
method: Boolean
| aBoolean

"Evaluating disjunction (OR).  Returns true if either the receiver or the
 argument aBoolean is true."

<primitive: 43>
aBoolean _validateClass: Boolean.
self _primitiveFailed: #| args: { aBoolean }.
self _uncontinuableError
%

! Class extensions for 'Break'

!		Instance methods for 'Break'

removeallmethods Break
removeallclassmethods Break

category: 'Instance initialization'
method: Break
initialize
  gsNumber := ERR_Break.
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'Breakpoint'

!		Instance methods for 'Breakpoint'

removeallmethods Breakpoint
removeallclassmethods Breakpoint

category: 'Instance initialization'
method: Breakpoint
initialize
  gsNumber := ERR_Breakpoint.
  gsResumable := true .
  gsTrappable := true .
%

category: 'Instance initialization'
method: Breakpoint
resignalAs: anException
  "not working yet."
  ^ super resignalAs: anException
%

category: 'Instance initialization'
method: Breakpoint
resume
  "Resume execution at the step point where this Breakpoint was signalled
   and attempt to execute the bytecode at that step point."

  ^ self resume: nil
%

category: 'Instance initialization'
method: Breakpoint
resume: aValue
  "Resume execution at the step point where this Breakpoint was signalled
   and attempt to execute the bytecode at that step point."

  ^ super resume: aValue
%

category: 'Instance initialization'
method: Breakpoint
return
  "Return from the on:do: which installed a handler for this Breakpoint.
   Set the current GsProcess' state so that the breakpoint is reenabled."

  GsProcess _current clearLastBreakpointHistory .
  ^ super return
%

category: 'Instance initialization'
method: Breakpoint
return: aValue
  "Return from the on:do: which installed a handler for this Breakpoint.
   Set the current GsProcess' state so that the breakpoint is reenabled."

  GsProcess _current clearLastBreakpointHistory .
  ^ super return: aValue
%

! Class extensions for 'CannotReturn'

!		Instance methods for 'CannotReturn'

removeallmethods CannotReturn
removeallclassmethods CannotReturn

category: 'Instance initialization'
method: CannotReturn
initialize
  gsNumber := ERR_CannotReturn.
  gsResumable := false .
  gsTrappable := true .
%

! Class extensions for 'Character'

!		Class methods for 'Character'

removeallmethods Character
removeallclassmethods Character

category: 'Deprecated Character Data Table'
classmethod: Character
activateCharTablesFromFile: file

"Install #CharacterDataTables from the passivated contents of a file.

Don't forget to commit to make the change permanent.

You must be SystemUser to execute this method.

See passivateCharTablesToFile: for the other side of this mechanism.

Use $GEMSTONE/goodies/CharacterTableUnicode.dat
to enable full 16-bit Unicode support.

See $GEMSTONE/goodies/UnicodeData.gs for
more info on Unicode Standards support.
"
self deprecated:  'Character class>>activateCharTablesFromFile: Deprecated as of GS/64 3.1, use Unicode classes' .

" Check that we're SystemUser "
System myUserProfile userId = 'SystemUser' ifFalse: [
   self _halt: 'Only SystemUser may execute this method' ].

(Globals at: #CharacterDataTables ifAbsent: [
    Globals at: #CharacterDataTables put: nil ] ).
Globals at: #CharacterDataTables put:
    ( PassiveObject fromServerTextFile: file ) activate.
^ self _loadCharTables
%

category: 'Unicode'
classmethod: Character
allUnicodeCodePointsDo: aBlock
  "Executes the one argument block aBlock for each legal Unicode codepoint."
  0 to: 16rD7FF do:[:n | aBlock value: n ].
  16rE000 to: 16r10FFFF do:[:n| aBlock value: n].
%

category: 'Deprecated Character Data Table'
classmethod: Character
categoryId: aSymbol

"Given a character category symbol, return the numeric id.  Note the
 indexes are not the same as used in ICU.

See Character>>categorySymbol: for symbol meanings. "
self deprecated:  'Charater class>>categoryId: Deprecated as of GS/64 3.1, use Unicode classes' .

^ #( #Lu #Ll #Lt #Lm #Lo #Mn #Mc #Me #Nd #Nl #No #Pc #Pd #Ps
   #Pe #Pi #Pf #Po #Sm #Sc #Sk #So #Zs #Zl #Zp #Cc #Cf #Cs #Co #Cn)
   indexOf: aSymbol
%

category: 'Deprecated Character Data Table'
classmethod: Character
categorySymbol: id

"Given a character category id, return the category symbol. Note the ids are
 not the same as used in ICU.

Category symbols are taken from the Unicode Standard:

#Lu  - Letter, Uppercase
#Ll - Letter, Lowercase
#Lt  - Letter, Titlecase
#Lm - Letter, Modifier
#Lo - Letter, Other
#Mn - Mark, Nonspacing
#Mc - Mark, Spacing Combining
#Me - Mark, Enclosing
#Nd - Number, Decimal Digit
#Nl - Number, Letter
#No - Number, Other
#Pc - Punctuation, Connector
#Pd - Punctuation, Dash
#Ps - Punctuation, Open/Start
#Pe - Punctuation, Close/End
#Pi - Punctuation, Initial Quote
#Pf - Punctuation, Final Quote
#Po - Punctuation, Other
#Sm - Symbol, Math
#Sc - Symbol, Currency
#Sk - Symbol, Modifier
#So - Symbol, Other
#Zs - Separator, Space
#Zl - Separator, Line
#Zp - Separator, Paragraph
#Cc - Other, Control
#Cf - Other, Format
#Cs - Other, Surrogate
#Co - Other, Private Use
#Cn - Other, Not Assigned
"
self deprecated:  'Character class>>categorySymbol: Deprecated as of GS/64 3.1, use Unicode classes' .

^ #( #Lu #Ll #Lt #Lm #Lo #Mn #Mc #Me #Nd #Nl #No #Pc #Pd #Ps
   #Pe #Pi #Pf #Po #Sm #Sc #Sk #So #Zs #Zl #Zp #Cc #Cf #Cs #Co #Cn)
   at: id
%

category: 'Deprecated Character Data Table'
classmethod: Character
charTables

" Reconstruct structured character data tables from the raw byte arrays
  stored in Globals at: #CharacterDataTables

Returns:

An Array of elements, arranged according to collate order,
each element an Array of 4 or 5 entries:

1.  The character for this entry.
2.  The symbolic character category code.
3.  Uppercase character ( if a letter ) / Numerator ( if numeric ).
4.  Lowercase character ( if a letter ) / Denominator ( if numeric/fraction ).
5.  (Optional)  Titlecase character ( if a letter )

"

| tables result dispatchBA indexBA categoryBA mainBA titleCaseBA
  titleCaseData titleCaseEntry size entry category id  chCls |
self deprecated:  'Character class>>charTables Deprecated as of GS/64 3.1, use Unicode classes' .

result := Array new.
tables := Globals at: #CharacterDataTables otherwise: nil .
tables ifNil:[ tables := Character _fetchCharTables ].

dispatchBA := tables at: 1.
indexBA := tables at: 2.
categoryBA := tables at: 3.
mainBA := tables at: 4.
titleCaseBA := tables at: 5.

" Construct TitleCase Data for later use.. "
titleCaseData := { } .
0 to: ( titleCaseBA size / 8 - 1 ) do: [ :i |
   entry := { } .
   entry add: ( titleCaseBA unsigned32At: ( i * 8 + 1 )).
   entry add: ( titleCaseBA unsigned32At: ( i * 8 + 5 )).
   titleCaseData add: entry ].

" Now reconstruct character data table "
size := mainBA size / 8.
chCls := Character .
0 to: size - 1 do: [ :i |
   entry := { } .
   id :=  ( indexBA unsigned32At: ( i * 4 + 1 )).
   entry add: ( chCls withValue: id ).
   category := self categorySymbol: ( categoryBA unsigned8At: ( i + 1 )).
   entry add: category.
   ( #( #Lu #Ll #Lt ) includes: category )
      ifTrue: [
         entry add:
            ( chCls withValue: ( mainBA unsigned32At: ( i * 8 + 1 ))).
         entry add:
            ( chCls withValue: ( mainBA unsigned32At: ( i * 8 + 5 ))).
         titleCaseEntry :=
            titleCaseData detect: [ :x | ( x at: 1 ) = id ] ifNone: [ nil ].
         ( titleCaseEntry == nil ) ifFalse: [
            entry add: ( chCls withValue: ( titleCaseEntry at: 2 )) ] ]
      ifFalse: [
         entry add: ( mainBA unsigned32At: ( i * 8 + 1 )).
         entry add: ( mainBA unsigned32At: ( i * 8 + 5 )) ].
   result add: entry ].
^ result
%

category: 'Instance Creation'
classmethod: Character
codePoint: anInteger

"Returns the Character with the specified value.
 Allowable range is 0 <= anInteger <= 16r10FFFF "

<primitive: 72>

anInteger _validateClass: SmallInteger.
OutOfRange new name:'anInteger' min: 0 max: 16r10FFFF actual: anInteger ; signal
%

category: 'Printable Characters'
classmethod: Character
digits

"Returns an InvariantArray containing Characters representing
 digits 0 through 9."

^#($0 $1 $2 $3 $4 $5 $6 $7 $8 $9)
%

category: 'Instance Creation'
classmethod: Character
fromStream: aStream

"Returns the next Character in the stream aStream."

self _checkReadStream: aStream forClass: String.
^ aStream next.
%

category: 'Instance Creation'
classmethod: Character
fromString: aString

"If aString is a one-Character String, returns the Character in aString.
 Otherwise, generates an error."

aString _validateClass: String.
(aString size == 1)
  ifTrue: [ ^ aString at: 1 ]
  ifFalse: [ self _errIncorrectFormat: aString ]
%

category: 'Deprecated Character Data Table'
classmethod: Character
installCharTables: table

"Converts a structured character data table into appropriately formated
byte arrays and then places them into Globals at: #CharacterDataTables
for use in this and subsequent sessions.  This operation *does not*
do a commit -- follow up with a commit if you wish to make this change
valid for subsequent sessions.

WARNINGS:

Installing incorrectly formatted character table data will break
character/string operations, including command line processing to a point
where the system will be impossible to use.  In this case, clear the
installation by doing the following:

1.  From the OS, set the host machine environmental parameter
    GS_DISABLE_CHARACTER_TABLE_LOAD to some value
    (it's the presence of this parameter that enables the mechanism).

2.  Login a new topaz session

3.  Execute Globals at: #CharacterDataTables put: nil.

4.  Commit

Note that changing the collation order of characters in new
CharacterDataTables will break any indexes that are keyed off of
Strings/DoubleByteStrings.  Before changing the tables, remove all such
indexes, install the new tables, and then reconstruct the indexes.

You must be SystemUser to execute this method.

Table Format:

An Array of elements, arranged according to character collate order,
each element an Array of 4 or 5 entries:

1.  The character for this entry.
2.  The symbolic character category code.
    See Character>>categorySymbol: for a list.
3.  Uppercase character ( if a letter ) / Numerator ( if numeric ).
4.  Lowercase character ( if a letter ) / Denominator ( if numeric/fraction ).
5.  (Optional)  Titlecase character

"

| tables dispatchBA indexBA categoryBA mainBA titleCaseBA
  mainSize dispatchSize titleCaseSize titleCaseIndex entry item id  |
self deprecated:  'Character class>>installCharTables:  Deprecated as of GS/64 3.1, use Unicode classes' .

" Check that we're SystemUser "
System myUserProfile userId = 'SystemUser' ifFalse: [
   self _halt:'Only SystemUser may execute this method' ].

" Generate the ByteArrays "
mainSize := table size.
dispatchSize := 0.
titleCaseSize := 0.
indexBA := ByteArray new: ( mainSize * 4 ).
categoryBA := ByteArray new: mainSize.
mainBA := ByteArray new: ( mainSize * 8 ).
" First pass: do most of the work,
  get info on sizes for dispatch and titleCase tables "
0 to: mainSize - 1 do: [ :i |
   entry := table at: i + 1.
   id := ( entry at: 1 ) codePoint.
   ( id > dispatchSize ) ifTrue: [ dispatchSize := id ].
   ( entry size > 4 ) ifTrue: [ titleCaseSize := titleCaseSize + 1 ].
   indexBA unsigned32At: ( i * 4 + 1 ) put: id.
   categoryBA unsigned8At: ( i + 1 ) put:
      ( Character categoryId: ( entry at: 2 ) ).
   ((item := entry at: 3) == nil) ifFalse: [
      ( item isKindOf: Character )
         ifTrue: [ mainBA unsigned32At: ( i * 8 + 1 ) put: item codePoint ]
         ifFalse: [ mainBA signed32At: ( i * 8 + 1 ) put: item ] ].
   ((item := entry at: 4) == nil) ifFalse: [
      ( item isKindOf: Character )
         ifTrue: [ mainBA unsigned32At: ( i * 8 + 5 ) put: item codePoint ]
         ifFalse: [ mainBA signed32At: ( i * 8 + 5 ) put: item ] ] ].

" Second pass: fill in dispatch and titleCase tables "
dispatchSize := dispatchSize + 1.
dispatchBA := ByteArray new: ( dispatchSize * 4 ).
titleCaseBA := ByteArray new: ( titleCaseSize * 8 ).
titleCaseIndex := 0.
1 to: mainSize do: [ :i |
   entry := table at: i.
   id := ( entry at: 1 ) codePoint.
   dispatchBA unsigned32At: ( id * 4 + 1 ) put: ( i - 1 ).
   ( entry size > 4 ) ifTrue: [
      titleCaseBA unsigned32At:
         ( titleCaseIndex * 8 + 1 ) put:  id.
      titleCaseBA unsigned32At:
         ( titleCaseIndex * 8 + 5 ) put: ( entry at: 5 ) codePoint.
      titleCaseIndex := titleCaseIndex + 1 ] ].

" Setup the table array "
tables := Array new: 5.
tables at: 1 put: dispatchBA.
tables at: 2 put: indexBA.
tables at: 3 put: categoryBA.
tables at: 4 put: mainBA.
tables at: 5 put: titleCaseBA.

" Install on #CharacterDataTables "
(Globals at: #CharacterDataTables ifAbsent: [
   Globals at: #CharacterDataTables put: nil ] ).
Globals at: #CharacterDataTables put: tables.

" Now install in this session's internal tables "
self _dispatch: dispatchBA index: indexBA category: categoryBA
   main: mainBA titleCase: titleCaseBA.

^ tables
%

category: 'Non-Printable Characters'
classmethod: Character
lfValue

"Returns value of the ASCII line-feed Character."

^  10
%

category: 'Printable Characters'
classmethod: Character
lowercaseRoman

"Returns an InvariantArray containing all lower-case Roman ASCII
 characters in alphabetic order."

^#($a $b $c $d $e $f $g $h $i $j $k $l $m $n $o $p $q $r $s $t $u
   $v $w $x $y $z)
%

category: 'Instance Creation'
classmethod: Character
maximumCodePoint
  "Returns the maximum code point per Unicode standards"

  ^ 16r10FFFF
%

category: 'Instance Creation'
classmethod: Character
new

"Disallowed.  You may not create new instances of Character."

self shouldNotImplement: #new
%

category: 'Deprecated Character Data Table'
classmethod: Character
passivateCharTablesToFile: file

"Write the passivated contents of the #CharacterDataTables (if present)
to a file.

Useful for efficiently porting the CharacterDataTable to another stone.
See activateCharTablesFromFile: for the other side of this mechanism. "

self deprecated:  'Character class>>passivateCharTablesToFile: Deprecated as of GS/64 3.1, use Unicode classes' .

(Globals at: #CharacterDataTables ifAbsent: [ ^nil ] )
   passivate toServerTextFile: file
%

category: 'Non-Printable Characters'
classmethod: Character
space

"Returns the ASCII space Character."
"recompiled in charact2.gs"

^ $  .
%

category: 'Printable Characters'
classmethod: Character
uppercaseRoman

"Returns an InvariantArray containing all upper-case Roman ASCII
 characters in alphabetic order."

^#($A $B $C $D $E $F $G $H $I $J $K $L $M $N $O $P $Q $R $S $T $U
   $V $W $X $Y $Z)
%

category: 'Instance Creation'
classmethod: Character
withSortValue: anInt

"Will be deprecated.
 Legacy method returning character at a particular point in collation,
 not Unicode compatible"

<primitive: 546>
self _primitiveFailed: #withSortValue: args: { anInt }
%

category: 'Instance Creation'
classmethod: Character
withValue: anInteger

"Returns the Character with the specified value.
 Allowable range is 0 <= anInteger <= 16r10FFFF "

<primitive: 72>

anInteger _validateClass: SmallInteger.
OutOfRange new name:'anInteger' min: 0 max: 16r10FFFF actual: anInteger ; signal
%

category: 'Deprecated Character Data Table'
classmethod: Character
_dispatch: dispatchTable index: indexTable category: categoryTable
    main: mainTable titleCase: titleCaseTable

" Copy byte array tables to internal C tables for String comparisions

  dispatchTable , indexTable
     are ByteArrays containing big-endian 16 bit values

  mainTable, titleCaseTable
     are ByteArrays containing pairs of big-endian 16 bit values

  categoryTable is a ByteArrays containing 8 bit values .

  if dispatchTable == nil,  then other arguments are ignored, and
  the C tables are reset to the hardcoded defaults defined when
  the VM executable/shared library was built .
"

<primitive: 643>
self _primitiveFailed: #_dispatch:index:category:main:titleCase:
     args: { dispatchTable . indexTable .
             categoryTable . mainTable . titleCaseTable }
%

category: 'Deprecated Character Data Table'
classmethod: Character
_dumpCharTables

" Dump contents of character tables to stdout  "

   <primitive: 645>
self _primitiveFailed: #_dumpCharTables
%

category: 'Deprecated Character Data Table'
classmethod: Character
_fetchCharTables

" Fetch contents of character tables "

<primitive: 644>
self _primitiveFailed: #_fetchCharTables
%

category: 'Deprecated Character Data Table'
classmethod: Character
_loadCharTables

"Load the character data tables recorded in
( Globals at: #CharacterDataTables ) into the session's internal memory.

This data should be in the form of an Array containing 5 ByteArrays.

See Character>>installCharTables for information on how to configure this. "

| table dispatchBA indexBA categoryBA mainBA titleCaseBA |
self deprecated:  'Character class>>_loadCharTables Deprecated as of GS/64 3.1, use Unicode classes' .

table :=  Globals at: #CharacterDataTables otherwise: nil .
table ifNil:[ ^ false ].

dispatchBA := table at: 1.
indexBA := table at: 2.
categoryBA := table at: 3.
mainBA := table at: 4.
titleCaseBA := table at: 5.

self _dispatch: dispatchBA index: indexBA category: categoryBA
      main: mainBA titleCase: titleCaseBA.

^ table
%

category: 'Deprecated Character Data Table'
classmethod: Character
_resetCharTablesToCDefaults

" Resets the character data tables in the VM to the hardcoded defaults
  defined when the VM executable/shared library was built .
  This defaults are the same as those used at session initialization if
   (Globals at: #CharacterDataTables otherwise: nil) == nil
"

self deprecated:  'Character class>>_resetCharTablesToCDefaults  Deprecated as of GS/64 3.1, use Unicode classes' .

self _dispatch: nil index: nil category: nil main: nil titleCase: nil
%

!		Instance methods for 'Character'

category: 'Comparisons'
method: Character
< aCharacter

"Legacy method.
 Returns true if the Legacy sort order of the receiver is
 less than that of aCharacter."

<primitive: 61>

aCharacter _validateClass: Character.
^  self _primitiveFailed: #< args: { aCharacter }.
%

category: 'Comparisons'
method: Character
<= aCharacter

"Legacy method.
 Returns true if the Legacy sort order of the receiver is
 less than or equal to that of aCharacter."

<primitive: 181>

aCharacter _validateClass: Character.
^ self _primitiveFailed: #<= args: { aCharacter }.
%

category: 'Comparisons'
method: Character
= aCharacter

"Returns true if the receiver and aCharacter have the same code point."

^ self == aCharacter 
%

category: 'Comparisons'
method: Character
> aCharacter

"Legacy method.
 Returns true if the Legacy sort order of the receiver is
 greater than that of aCharacter. "

<primitive: 182>

aCharacter _validateClass: Character.
^  self _primitiveFailed: #> args: { aCharacter }.
%

category: 'Comparisons'
method: Character
>= aCharacter

"Legacy method.
 Returns true if the Legacy sort order of the receiver is
 greater than or equal to that of aCharacter."

<primitive: 183>

aCharacter _validateClass: Character.
^  self _primitiveFailed: #>= args: { aCharacter }.
%

category: 'Converting'
method: Character
asCharacter

"Returns the receiver."

^ self
%

category: 'Deprecated'
method: Character
asciiLessThan: aChar

"Returns true if the ASCII code of the receiver is less than that of
 aCharacter."

self deprecated: 'asciiLessThan: deprecated in 3.2.'.
^ self codePoint < aChar codePoint
%

category: 'Accessing'
method: Character
asciiValue

"Returns the Unicode value of the receiver (a SmallInteger)."

<primitive: 71>

self _primitiveFailed: #asciiValue .
self _uncontinuableError
%

category: 'Converting'
method: Character
asDigit

"Returns the digit value (0-9) of the receiver.  If the receiver is not
 a digit, this returns 0."

(self < $0 or:[ self > $9]) ifTrue: [^0].
^self codePoint - $0 codePoint
%

category: 'Converting'
method: Character
asFoldcase

"The result of calling u_foldCase(GCI_OOP_TO_CHAR(self), U_FOLD_CASE_DEFAULT)
 in libicu is returned as a Character

 The foldCase is the equivalent to use non-case-sensitive uses of strings or
 characters.
 It is recommended to use the asFoldcase method implemented
 in string classes MultiByteString and String because
 those methods take into account effects of adjacent characters
 on case folding."

^ self _unicodeStatus: 22
%

category: 'Accessing'
method: Character
asInteger

"Returns the Unicode value of the receiver (a SmallInteger)."

<primitive: 71>

self _primitiveFailed: #asInteger .
self _uncontinuableError
%

category: 'Converting'
method: Character
asLowercase
" Returns Unicode lowercase  equivalent of the receiver.
  If the receiver has no lowercase  equivalent, returns the receiver.

  calls u_tolower in libicu .
  same as java.lang.Character.toLowerCase() .

  Differs from the legacy method asLowercaseOld for code points
  for about 1000 code points between 256 and 67000 .

  This method only returns the simple, single-code point case mapping.
  Full case mappings should be used whenever possible because they produce
  better results by working on whole strings.
  They take into account the string context and the language and can map
  to a result string with a different length as appropriate.
  See asLowercaseForLocale: in classes String and MultiByteString .
"

<primitive: 963>
self _primitiveFailed: #asLowercase
%

category: 'Converting-Legacy'
method: Character
asLowercaseOld

"Legacy method. Not reliable for codepoints over 255 unless
 (deprecated) Character Data Tables installed.
 Returns a Character that is the lower-case character corresponding
 to the receiver.  If the receiver is lower-case or has no case, this
 returns the receiver itself."

<primitive: 74>

self _primitiveFailed: #asLowercaseOld .
self _uncontinuableError
%

category: 'Formatting'
method: Character
asString

"Returns a one-Character String or DoubleByteString containing the receiver."

<primitive: 56>
| cp |
cp := self codePoint .
(cp >= 16rD800 and:[ cp <= 16rDFFF]) ifTrue:[
   ^ OutOfRange signal:'codePoint 16r', cp asHexString ,' is illegal'.
].
self _primitiveFailed: #asString .
self _uncontinuableError
%

category: 'Converting'
method: Character
asSymbol

"Returns a one-Character Symbol that represents the receiver."

^ self asString asSymbol
%

category: 'Converting'
method: Character
asTitlecase
" Returns the titlecase of the receiver.  This is either the uppercase, or
  the distinct titlecase character for the (few) characters that have a
  separate titlecase form. If the receiver has no case or is already title
  case, returns the receiver.

  calls u_totitle in libicu .
  same as java.lang.Character.toTitleCase() .

  This method only returns the simple, single-code point case mapping.
  Full case mappings should be used whenever possible because they produce
  better results by working on whole strings.
  They take into account the string context and the language and can map
  to a result string with a different length as appropriate.
  See asTitlecaseForLocale: in classes String and MultiByteString .
"
<primitive: 964>
self _primitiveFailed: #asUnicodeTitlecase
%

category: 'Converting'
method: Character
asUppercase

" Returns Unicode uppercase equivalent of the receiver.
  If the receiver has no uppercase equivalent, returns the receiver.

  calls u_toupper in libicu .
  same as java.lang.Character.toUpperCase() .

  Differs from the legacy method asUppercaseOld for
  code points 181, 255, and about 1000 code points >= 256 .

  This method only returns the simple, single-code point case mapping.
  Full case mappings should be used whenever possible because they produce
  better results by working on whole strings.
  They take into account the string context and the language and can map
  to a result string with a different length as appropriate.
  See asUppercaseForLocale: in classes String and MultiByteString .
"

<primitive: 968>
self _primitiveFailed: #asUppercase
%

category: 'Converting-Legacy'
method: Character
asUppercaseOld

"Legacy method. Not reliable for codepoints over 255 unless
 (deprecated) Character Data Tables installed.
 Returns a Character that is the upper-case character corresponding
 to the receiver.  If the receiver is upper-case or has no case, this
 returns the receiver itself."

<primitive: 73>

self _primitiveFailed: #asUppercaseOld .
self _uncontinuableError
%

category: 'Accessing'
method: Character
codePoint

"Returns the Unicode value of the receiver (a SmallInteger)."

<primitive: 71>

self _primitiveFailed: #codePoint .
self _uncontinuableError
%

category: 'Unicode'
method: Character
compareTo: aCharacter collator: anIcuCollator

"Returns -1, 0, or 1 depending on how a Unicode16 string holding
 the receiver compares to a Unicode16 string holding the argument using
 the specified IcuCollator.
 anIcuCollator == nil is interpreted as   IcuCollator default ."

<primitive: 980>
 "primitive returns nil if aCharacter is not a Character"
  anIcuCollator ifNil:[ (System __sessionStateAt: 20) ifNil:[
     ^ self compareTo: aCharacter collator: IcuCollator default
  ]].
  self _primitiveFailed: #compareTo:collator: args: { aCharacter . anIcuCollator }
%

category: 'Private'
method: Character
containsIdentity

"Private."

^true
%

category: 'Copying'
method: Character
copy

"Returns the receiver.  (Does not create a new Character.)"

^self
%

category: 'Converting'
method: Character
digitValue

"If the result of calling u_charDigitValue(GCI_OOP_TO_CHAR(self)) in libicu
 is -1, returns nil.
 Otherwise the result from u_charDigitValue is returned as a SmallInteger.

 Results differ from legacy method digitValueOld for 450 code points
 between 1632 and 120831 "

^ self _unicodeStatus: 20
%

category: 'Converting'
method: Character
digitValueInRadix: radixSmallInt

"Returns a SmallInteger representing the value of the receiver
 in the given radix.
 Returns nil if if the receiver is not a digit in the given radix.
 If radixSmallInt < 2 or > 36, signals an OutOfRange error .

 The result is computed by calling u_digit(GCI_OOP_TO_CHAR(self), radix)
 in libicu."

<primitive: 966>

radixSmallInt _validateClass: SmallInteger .
(radixSmallInt < 2 or:[ radixSmallInt > 36]) ifTrue:[
   OutOfRange new
     name: 'radixSmallInt' min: 2 max: 36 actual: radixSmallInt ;
     signal
].
ArgumentError signal:'no valid digit translation'.
 ^ self _primitiveFailed: #digitValueInRadix: args: { radixSmallInt }
%

category: 'Converting-Legacy'
method: Character
digitValueInRadixOld: radix

"Legacy method.
 Returns a SmallInteger representing the value of the receiver, a digit, or
 returns nil if the receiver is not a digit in the given radix."

| val up |
radix == 10 ifTrue: [ ^self digitValueOld ].
radix < 10 ifTrue: [
  val := self digitValueOld .
  val >= radix ifTrue: [ ^nil ].
  ^val
].
val := self digitValueOld .
val ~~ nil ifTrue: [ ^val ].
up := self asUppercaseOld.
($A <= up and: [ up <= (self class withValue: ($A codePoint + radix - 11)) ])
ifTrue: [ ^(up codePoint - $A codePoint) + 10 ]
ifFalse: [ ^nil ]
%

category: 'Converting-Legacy'
method: Character
digitValueOld

"Legacy method.
 Returns a SmallInteger representing the value of the receiver,
 a digit, or returns nil if the receiver is not a digit."

(self isDigitOld)
  ifTrue:[ ^ self codePoint - $0 codePoint]
  ifFalse:[ ^ nil]
%

category: 'Formatting'
method: Character
displayWidth

"Returns the width necessary to display the receiver.
 For a Character, this method always returns 1."

^ 1
%

category: 'Case-Insensitive Comparisons'
method: Character
equalsNoCase: aCharacter

"Returns true if the receiver is the same Character as the argument regardless
 of case or internal representation.
 using equivalent of asUppercase before comparing."

<primitive: 979>
"If the primitive fails,
 then aCharacter must not be an instance of Character"

^ aCharacter isEquivalent: self
%

category: 'Case-Insensitive Comparisons'
method: Character
equalsNoCaseOld: aCharacter

"Legacy method. Not reliable for codepoints over 255 unless
 (deprecated) Character Data Tables installed.
 Returns true if the receiver is the same Character as the argument regardless
 of case or internal representation,
 using equivalent of asUppercaseOld before comparing."

<primitive: 75>
"If the primitive fails,
 then aCharacter must not be an instance of Character"

^ aCharacter isEquivalent: self
%

category: 'Grease'
method: Character
greaseInteger
"Returns the Unicode value of the receiver (a SmallInteger)."
<primitive: 71>
self _primitiveFailed: #greaseInteger .
self _uncontinuableError
%

category: 'Comparing'
method: Character
hash

"Returns a numeric hash key for the receiver." 

^ self codePoint
%

category: 'Testing'
method: Character
hasMirror

"Return true if the receiver has a mirrored equivalent, E.g. s either the open
 or close of mirrored pairs such as ${ $}, $< $>, etc.

 Calls u_isMirrored(GCI_OOP_TO_CHAR(self)) in libicu .
 "

^ self _unicodeStatus: 19
%

category: 'Testing'
method: Character
isAlphabetic

"The result of calling u_isUAlphabetic(GCI_OOP_TO_CHAR(self)) in libicu is
 returned as a Boolean.  Result is TRUE if receiver code point has the
 Alphabetic Unicode property."

^ self _unicodeStatus: 1
%

category: 'Testing'
method: Character
isAlphaNumeric

"The result of calling u_isalnum(GCI_OOP_TO_CHAR(self)) in libicu is returned
 as a Boolean.
 Differs subtantially from the legacy method isAlphaNumericOld
 for codePoints >= 256 "

^ self _unicodeStatus: 6
%

category: 'Testing-Legacy'
method: Character
isAlphaNumericOld

"Legacy method. Not reliable for codepoints over 255 unless
 (deprecated) Character Data Tables installed.
 Returns true if the receiver is a Roman letter or digit.  Returns false
 otherwise."

 | cat |
 cat := self _category .
 ^ cat == 9 or:[ cat < 4 ]
%

category: 'Testing'
method: Character
isDigit

"The result of calling u_isdigit(GCI_OOP_TO_CHAR(self)) in libicu is returned
 as a Boolean.
 Returns true for characters with Unicode general category 'Nd'
 (decimal digit numbers).
 Same as java.lang.Character.isDigit().

 Differs from the legacy method isDigitOld for 450 code points between
 1632 and 120831, which are considered digits in the Unicode standard,
 and for which isDigitOld returns false."

^ self _unicodeStatus: 4
%

category: 'Testing-Legacy'
method: Character
isDigitOld

"Legacy method.
 Returns true if the receiver is a digit.  Returns false otherwise."

^ self _category == 9
%

category: 'Case-Insensitive Comparisons'
method: Character
isEquivalent: aCharacter

"Returns true if the receiver is the same Character as the argument regardless
 of case or internal representation.
 using equivalent of asUppercase before comparing."

<primitive: 979>
"If the primitive fails,
 then aCharacter must not be an instance of Character"

^ aCharacter isEquivalent: self
%

category: 'Case-Insensitive Comparisons'
method: Character
isEquivalentOld: aCharacter

"Legacy method. Not reliable for codepoints over 255 unless
 (deprecated) Character Data Tables installed.
 Returns true if the receiver is the same Character as the argument regardless
 of case or internal representation,
 using equivalent of asUppercaseOld before comparing."

<primitive: 75>
"If the primitive fails,
 then aCharacter must not be an instance of Character"

^ aCharacter isEquivalent: self
%

category: 'Unicode'
method: Character
isGraphic

"Returns true if the receiver is a graphic character; a printable character,
 excluding separators. Return false for characters with general categories Cc (control
 codes), Cf (format controls), Cs (surrogates), Cn (unassigned), and Z (separators).

 Calls u_isgraph(GCI_OOP_TO_CHAR(self)) in libicu"

^ self _unicodeStatus: 9
%

category: 'Testing'
method: Character
isHexDigit

"Returns true if the receiver is a hexadecimal digit.

 Returns true for characters with Unicode general category 'Nd' (decimal digit numbers).
 as well as Latin letters a-f and A-F in both ASCII and Fullwidth ASCII.
 (That is, for letters with code points
  16r0041..16r0046, 16r0061..16r0066, 16rFF21..16rFF26, 16rFF41..16rFF46.)

 In order to narrow the definition of hexadecimal digits to only ASCII
 characters, use  (c codePoint <= 16r7F and:[ unicodeIsXDigit ) .

 calls u_isxdigit in libicu .
"

^ self _unicodeStatus: 7
%

category: 'Testing'
method: Character
isLetter

"The result of calling u_isalpha(GCI_OOP_TO_CHAR(self)) in libicu is
 returned as a Boolean.
 Result is true for code points Unicode general categories 'L' (letters).

 Same as isAlphabetic for code points <= 836 .
 Same as java.lang.Character.isLetter() .

 Differs substantially from legacy isLetterOld for code points >= 256"

^ self _unicodeStatus: 5
%

category: 'Testing-Legacy'
method: Character
isLetterOld

"Legacy method. Not reliable for codepoints over 255 unless
 (deprecated) Character Data Tables installed.
 Returns true if the receiver is a Roman letter.  Returns false otherwise."

^ self _category < 4
%

category: 'Testing'
method: Character
isLowercase
"Returns true if the receiver's code point has
 the Unicode general category  'Ll' (lowercase letter).

 Calls u_islower in libicu .
 Same as java.lang.Character.isLowerCase() .

 This misses some characters that are also lowercase but have
 a different general category value.
 In order to include those, use unicodeIsULowercase .
 isLowercase and unicodeIsULowercase differ for code points 170, 186,
 and about 180 code points >= 256 .

 Differs from legacy method isLowercaseOld
 for code points 170, 186, and about 1700 code points >= 256 .
"

^ self _unicodeStatus: 25
%

category: 'Testing-Legacy'
method: Character
isLowercaseOld

"Legacy method. Not reliable for codepoints over 255 unless
 (deprecated) Character Data Tables installed.
 Returns true if the receiver is a lower-case character.  Returns false
 otherwise."

^ self _category == 2
%

category: 'Unicode'
method: Character
isNumeric
  "The result of calling
   u_getIntPropertyValue(GCI_OOP_TO_CHAR(self), UCHAR_NUMERIC_TYPE) in libicu
   is returned as a Boolean .
   Will return true for characters having either Numeric_Type Numeric
   (such as Han characters in Chinese-style number format) or Numeric_Type Decimal."

^ self _unicodeStatus: 27
%

category: 'Testing-Legacy'
method: Character
isNumericOld

"Legacy method.
 Returns true if the receiver contains numeric content ( category is #Nd #Nl #No).
 Returns false otherwise."

| category |
category := self _category.
category < 9 ifTrue: [ ^ false ].
category < 12 ifTrue: [ ^ true ].
^ false
%

category: 'Testing'
method: Character
isPrintable

 "Return true if the receiver is a printable character; this includes characters
  with any general category other than 'C' (controls).

  Calls u_isprint(GCI_OOP_TO_CHAR(self)) in libicu i.
"

^ self _unicodeStatus: 17
%

category: 'Testing'
method: Character
isPunctuation

"Return a boolean, true if the receiver is a punctuation character, false otherwise.

 Calls u_ispunct(GCI_OOP_TO_CHAR(self)) in libicu"

^ self _unicodeStatus: 8
%

category: 'Testing'
method: Character
isSeparator

"Returns true if codepoint of the receiver satisfies
    u_isISOControl(ch) || (ch == 32) || (ch == 160)
 using libicu.

 which is the same as isSeparator in previous releases."

^ self _unicodeStatus: 29
%

category: 'Testing'
method: Character
isSpecial

"Returns true if the receiver is a special object."

^ true
%

category: 'Testing'
method: Character
isTitlecase
"Returns true if the receiver's code point has the Unicode general category
 'Lt' (titlecase letter).

 Note that only titlecase characters return true. While asTitlecase converts
 to uppercase or titlecase, isTitlecase only returns true for titlecase.

 Calls u_istitle in libicu .
 Same as java.lang.Character.isTitleCase() .

 Differs from legacy method isTitlecaseOld for titlecase characters, when
 character data tables not installed; about 30 code points between 400 and 8100.
 The first code point for which isTitlecase returns true is 453.
"

^ self _unicodeStatus: 26
%

category: 'Testing-Legacy'
method: Character
isTitlecaseOld

"Legacy method.
 Returns true if the receiver is a title-case character.  Returns false
 otherwise."

^ self _category == 3
%

category: 'Testing'
method: Character
isUppercase
"Returns true if the receiver's code point has
 the Unicode general category  'Lu' (uppercase letter).

 Calls u_isupper in libicu .
 Same as java.lang.Character.isUpperCase().

 This misses some characters that are also uppercase but
 have a different general category value.  There are about 50 such
 differences between codepoints 8500 and 9500 .
 In order to include those, use unicodeIsUUppercase .

 Differs from legacy method isUppercaseOld
 for about 1400 code points >= 256 .
"

^ self _unicodeStatus: 24
%

category: 'Testing-Legacy'
method: Character
isUppercaseOld

"Legacy method. Not reliable for codepoints over 255 unless
 (deprecated) Character Data Tables installed.
 Returns true if the receiver is an upper-case character.  Returns false
 otherwise."

^ self _category == 1
%

category: 'Testing-Legacy'
method: Character
isVowel

"Legacy method. Vowel is not a unicode property.  Results may not be correct
 for code points over 255.
 Returns true if the receiver is a vowel ('Y' is considered to be a vowel).
 Returns false otherwise.

 This code assumes that the Legacy collation sequence places all uppercase variations
 of a given letter (including various diacritical marks) immediately following the
 plain version of the letter.
"
| ucs arr chCls |
ucs := self asUppercaseOld sortValue.
arr := #( $A $E $I $O $U $Y ) .
chCls := Character .
1 to: arr size do:[:j | | v |
  v := arr at: j .
  ucs < v sortValue  ifTrue: [ ^ false ].
  ucs < ( chCls withValue: v codePoint  + 1 ) sortValue  ifTrue:[^ true ]
].
^ false
%

category: 'Converting'
method: Character
mirror

"If the argument is the open or close of a mirrored pair, such as $( $) or ${ $},
 return the other character, the mirror of the receiver. If the receiver does
 not have a mirror, return the receiver.

 Calls u_charMirror(GCI_OOP_TO_CHAR(self)) in libicu .
 "

^ self _unicodeStatus: 21
%

category: 'Converting'
method: Character
numericValue

 "Calls u_getNumericValue(GCI_OOP_TO_CHAR(self)) in libicu
  and returns an Integer, Fraction, Float, or nil.

  Differs from digitValue in that numericValue  method handles
  characters having Numeric_Type Numeric (such as Han characters in Chinese-style
  number format) in addition to Numeric_Type Decimal."

^ self _unicodeStatus: 28
%

category: 'Formatting'
method: Character
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

aStream nextPut: $$ .
aStream nextPut: self
%

category: 'Formatting'
method: Character
printString

"Returns a String whose contents are a displayable representation of the
 receiver."

"GemStone does not allow the creation of new kinds of Character, so there
 is no point in creating a stream and sending printOn:."

| result |
result := String new: 2 .
result at: 1 put: $$ .
result at: 2 put: self  .
^ result
%

category: 'Accessing'
method: Character
sortValue

"Will be deprecated.
 Legacy method providing index into collation sequence, not Unicode compatible"

<primitive: 545>
self _primitiveFailed: #sortValue
%

category: 'Unicode'
method: Character
unicodeCategory

  "Returns a Symbol , the 2-letter Unicode name of
   general category for the code point.
   Result is based on u_charType(GCI_OOP_TO_CHAR(self)) in libicu"

  ^ self _unicodeStatus: 30
%

category: 'Unicode'
method: Character
unicodeIsBase

"The result of calling u_isbase(GCI_OOP_TO_CHAR(self)) in libicu is
 returned as a Boolean.

 From libicu uchar.h :
   Returns true for general categories 'L' (letters), 'N' (numbers),
   'Mc' (spacing combining marks), and 'Me' (enclosing marks).

   Note that this is different from the Unicode definition in
   chapter 3.5, conformance clause D13,
   which defines base characters to be all characters (not Cn)
   that do not graphically combine with preceding characters (M)
   and that are neither control (Cc) or format (Cf) characters.
"

^ self _unicodeStatus: 18
%

category: 'Unicode'
method: Character
unicodeIsBlank

"The result of calling u_isblank(GCI_OOP_TO_CHAR(self)) in libicu is returned as a Boolean"

^ self _unicodeStatus: 10
%

category: 'Unicode'
method: Character
unicodeIsControl

"The result of calling u_iscntrl(GCI_OOP_TO_CHAR(self)) in libicu is returned as a Boolean"

^ self _unicodeStatus: 15
%

category: 'Unicode'
method: Character
unicodeIsDefined

"The result of calling u_isdefined(GCI_OOP_TO_CHAR(self)) in libicu is returned as a Boolean"

^ self _unicodeStatus: 11
%

category: 'Unicode'
method: Character
unicodeIsISOControl

"The result of calling u_isISOControl(GCI_OOP_TO_CHAR(self)) in libicu is returned as a Boolean"

^ self _unicodeStatus: 16
%

category: 'Unicode'
method: Character
unicodeIsJavaSpace

"The result of calling u_isJavaSpaceChar(GCI_OOP_TO_CHAR(self)) in libicu is returned as a Boolean"

^ self _unicodeStatus: 13
%

category: 'Unicode'
method: Character
unicodeIsSpace

"The result of calling u_isspace(GCI_OOP_TO_CHAR(self)) in libicu is returned as a Boolean"

^ self _unicodeStatus: 12
%

category: 'Unicode'
method: Character
unicodeIsULowercase

"The result of calling u_isULowercase(GCI_OOP_TO_CHAR(self)) in libicu is returned as a Boolean"

^ self _unicodeStatus: 2
%

category: 'Unicode'
method: Character
unicodeIsUUppercase

"The result of calling u_isUUppercase(GCI_OOP_TO_CHAR(self)) in libicu
 is returned as a Boolean.

 Result is true if the receiver has the Uppercase Unicode property.
 Different than isUppercase.
"

^ self _unicodeStatus: 3
%

category: 'Unicode'
method: Character
unicodeIsWhitespace

"The result of calling u_isWhitespace(GCI_OOP_TO_CHAR(self)) in libicu is returned as a Boolean"

^ self _unicodeStatus: 14
%

category: 'Unicode'
method: Character
unicodeType

"Return the general cateory SmallInteger for the receiver, by calling
 u_charType(GCI_OOP_TO_CHAR(self)) in libicui.  Same as java.lang.Character.getType().

 Values returned from this method do not have the same category equivalent as
 GemStone legacy category; do not use result with _categorySymbol:.
"

^ self _unicodeStatus: 23
%

category: 'Storing and Loading'
method: Character
writeTo: passiveObj

"Converts the receiver to its passive form and writes that information on
 passiveObj."

| av ba |
av := self codePoint .
av <= 16rFF ifTrue:[
  (ba := ByteArray new: 1) at: 1 put: av .
  passiveObj nextPut: $$; nextPutAllBytes: ba .
] ifFalse:[  | ccls b3 b2 b1 b0 |
  ccls := Character .
  b0 := av bitAnd: 16rFF .
  av <= 16rFFFF ifTrue:[
    b1 := av bitShift: -8 .
    (ba := ByteArray new: 2) at: 1 put: b1 ; at: 2 put: b0 .
    passiveObj nextPut: $! ; nextPutAllBytes: ba .
  ] ifFalse:[
    av := av bitShift: -8 .
    b1 := av bitAnd: 16rFF .
    av := av bitShift: -8 .
    b2 := av bitAnd: 16rFF .
    av := av bitShift: -8 .
    b3 := av bitAnd: 16rFF .
    (ba := ByteArray new: 4) at: 1 put: b3 ; at: 2 put: b2 ; 
         at: 3 put: b1; at: 4 put: b0 .
    passiveObj nextPut: $& ; nextPutAllBytes: ba .
  ]
]
%

category: 'Private'
method: Character
_asSource

"Private."

^ self printString
%

category: 'Private'
method: Character
_category

"Returns the numeric category id for this character.  This numeric
 index is specific to the GemStone legacy implementation, and is not
 the same as the ICU index from unicodeType. Use with _categoryAsSymbol
 to determine the correct unicode category. Will not return correct
 results for Characters over 255 unless deprecated Character data tables
 are installed."

<primitive: 640>
self _primitiveFailed: #_category
%

category: 'Private'
method: Character
_categoryAsSymbol

" Returns the category symbol for this character.

Category symbols are taken from the Unicode Standard. Note that
the index of these is not the same as ICU.

#Lu  - Letter, Uppercase
#Ll - Letter, Lowercase
#Lt  - Letter, Titlecase
#Lm - Letter, Modifier
#Lo - Letter, Other
#Mn - Mark, Nonspacing
#Mc - Mark, Spacing Combining
#Me - Mark, Enclosing
#Nd - Number, Decimal Digit
#Nl - Number, Letter
#No - Number, Other
#Pc - Punctuation, Connector
#Pd - Punctuation, Dash
#Ps - Punctuation, Open/Start
#Pe - Punctuation, Close/End
#Pi - Punctuation, Initial Quote
#Pf - Punctuation, Final Quote
#Po - Punctuation, Other
#Sm - Symbol, Math
#Sc - Symbol, Currency
#Sk - Symbol, Modifier
#So - Symbol, Other
#Zs - Separator, Space
#Zl - Separator, Line
#Zp - Separator, Paragraph
#Cc - Other, Control
#Cf - Other, Format
#Cs - Other, Surrogate
#Co - Other, Private Use
#Cn - Other, Not Assigned

"

^ #( #Lu #Ll #Lt #Lm #Lo #Mn #Mc #Me #Nd #Nl #No #Pc #Pd #Ps
   #Pe #Pi #Pf #Po #Sm #Sc #Sk #So #Zs #Zl #Zp #Cc #Cf #Cs #Co #Cn)
at: self _category
%

category: 'New Indexing Comparison'
method: Character
_classSortOrdinal

^ 30
%

category: 'Indexing Support'
method: Character
_idxBasicCanCompareWithClass: aClass
  "Returns true if the receiver may be inserted into a basic BtreeNode whose
   #lastElementClass is <aClass> (see RangeEqualityIndex class>>isBasicClass:)."

  (super _idxBasicCanCompareWithClass: aClass)
    ifTrue: [ ^ true ].
  ^ aClass == Character
%

category: 'Private'
method: Character
_isPasswordSymbol

| cat |
cat := self _category .
^ (cat >= 12) and:[ cat <= 21 ]
%

category: 'Private'
method: Character
_numericValue

"Legacy method.
 Returns the numeric value (if one exists) for this character, or nil,
 using the legacy character tables."
self deprecated:  'Character _numericValue  Deprecated as of GS/64 3.2, use numericValue'.
^ self __numericValue
%

category: 'Private'
method: Character
_type

"Returns 1 for alpha, 2 for digit, and 3 for special."

| cat |
cat := self _category .
cat == 9 ifTrue: [ ^ 2].
cat < 4 ifTrue: [ ^ 1].
^ 3
%

category: 'Private'
method: Character
_typeAsSymbol

"Private."

^ #( #alpha #digit #special ) at: self _type
%

category: 'Unicode'
method: Character
_unicodeStatus: opcode
"
    opcode  libicu function
    1      u_isUAlphabetic(ch) returns a Boolean
    2      u_isULowercase(ch)  returns a Boolean
    3      u_isUUppercase(ch)  returns a Boolean
    4      u_isdigit(ch)    returns a Boolean
    5      u_isalpha(ch)    returns a Boolean
    6      u_isalnum(ch)    returns a Boolean
    7      u_isxdigit(ch)   returns a Boolean
    8      u_ispunct(ch)    returns a Boolean
    9      u_isgraph(ch)    returns a Boolean
    10     u_isblank(ch)    returns a Boolean
    11     u_isdefined(ch)  returns a Boolean
    12     u_isspace(ch)    returns a Boolean
    13     u_isJavaSpaceChar(ch) returns a Boolean
    14     u_isWhiteSpace(ch)    returns a Boolean
    15     u_iscntrl(ch)         returns a Boolean
    16     u_isISOControl(ch)    returns a Boolean
    17     u_isprint(ch)         returns a Boolean
    18     u_isbase(ch)          returns a Boolean
    19     u_isMirrored(ch)  returns a Boolean
    20     u_charDigitValue  returns a SmallInteger
    21     u_charMirror      returns a Character
    22     u_foldCase        returns a Character
    23     u_charType        returns a SmallInteger
    24     u_isupper        returns a Boolean
    25     u_islower        returns a Boolean
    26     u_istitle        returns a Boolean
    27     u_getIntPropertyValue(c, UCHAR_NUMERIC_TYPE), returns a Boolean
    28     u_getNumericValue  returns a Float, SmallDouble or nil
    29     u_isISOControl(ch) || (ch == 32) || (ch == 160)   smalltalk isSeparator
    30     symbolic name of u_charType(ch), returns a Symbol
"
<primitive: 965>
opcode _validateClass: SmallInteger .
self _primitiveFailed: #_unicodeStatus: args: { opcode }
%

category: 'Private'
method: Character
__numericValue
"Returns the numeric value (if one exists) for this character, or nil,
 using the legacy character tables."
<primitive: 641>
self _primitiveFailed: #__numericValue
%

category: 'Comparisons'
method: Character
~= aCharacter

"Returns false if the receiver and aCharacter have the same code point.
 Reimplemented as an optimization"

^ self ~~ aCharacter
%

! Class extensions for 'CharacterCollection'

!		Class methods for 'CharacterCollection'

removeallmethods CharacterCollection
removeallclassmethods CharacterCollection

category: 'Formatting'
classmethod: CharacterCollection
charSize

"(Subclass responsibility.)
 Returns number of bytes that make up a character for instances of this class."

^ self subclassResponsibility: #charSize
%

category: 'Session Control'
classmethod: CharacterCollection
disableUnicodeComparisonMode

  "Causes subsequent sessions, after login, to use Legacy methods for
   #< #> #= and various methods in the indexing
   when a receiver is a kind of String or MultiByteString.

   Signals an Error if the current user is not SystemUser .

   Returns a Boolean , the previous state of
     (Globals at: #StringConfiguration) == Unicode16 ."

   | prev |
   System myUserProfile userId = 'SystemUser' ifFalse:[
     Error signal:'Only SystemUser should execute this method'.
   ].
   prev := (Globals at: #StringConfiguration otherwise: nil) == Unicode16 .
   Globals at: #StringConfiguration put: String .
   System commit .
   ^ prev
%

category: 'Session Control'
classmethod: CharacterCollection
enableUnicodeComparisonMode

  "Causes subsequent sessions, after login, to use Unicode methods for
   #< #> #= and various methods in the indexing
   when a receiver is a kind of String or MultiByteString.

   Signals an Error if the current user is not SystemUser .

   Returns a Boolean , the previous state of
     (Globals at: #StringConfiguration) == Unicode16 ."

   | prev |
   System myUserProfile userId = 'SystemUser' ifFalse:[
     Error signal:'Only SystemUser should execute this method'.
   ].
   prev := (Globals at: #StringConfiguration otherwise: nil) == Unicode16 .
   Globals at: #StringConfiguration put: Unicode16 .
   System commit .
   ^ prev
%

category: 'Deprecated'
classmethod: CharacterCollection
fromServerTextFile: aFileSpec

self deprecated: 'CharacterCollection class >> fromServerTextFile: deprecated long before v3.0. Use an instance of GsFile
 to access the file system.'.
 "For multi-byte characters, assumes the byte order of the file
 matches the current session's cpu's in-memory byte order.  "

^ self new _fromServerTextFile: aFileSpec
%

category: 'Instance Creation'
classmethod: CharacterCollection
fromStream: aStream width: anInteger

"Returns a new instance of the receiver's class that contains the next
 anInteger Characters of aStream."

| result |

self _checkReadStream: aStream forClass: CharacterCollection.
result:= self new: anInteger.
1 to: anInteger do: [ :i | result at: i put: aStream next ].
^ result
%

category: 'Session Control'
classmethod: CharacterCollection
isInUnicodeComparisonMode

  "Returns a Boolean, true if Unicode versions of methods
   #< #> #= and various methods in the indexing
   are used when a receiver is a kind of String or MultiByteString"

  ^ Unicode16 usingUnicodeCompares
%

category: 'Instance Creation'
classmethod: CharacterCollection
new
"(Subclass responsibility.)"

^ self subclassResponsibility: #new
%

category: 'Instance Creation'
classmethod: CharacterCollection
new: anInteger
"(Subclass responsibility.)"

^ self subclassResponsibility: #new:
%

category: 'Instance Creation'
classmethod: CharacterCollection
withAll: aSequenceableCollection

"Returns a new instance of the receiver that contains the elements in the
 argument aSequenceableCollection."

| result |
result:= self new.
aSequenceableCollection accompaniedBy: result do: [:res :each | res add: each].
^result
%

category: 'Indexing Support'
classmethod: CharacterCollection
_idxBasicCanCompareWithCharacterCollectionInstance: aCharacterCollection
  "Returns true if <aCharacterCollection> may be inserted into a basic BtreeNode
   whose #lastElementClass is the receiver (see RangeEqualityIndex
   class>>isBasicClass:)."

  "If using Unicode compares, then *String and Unicode* instances may be compared.
   If not using Unicode compares then *String and Unicode* instances may not be compared."

  ^ true
%

category: 'Indexing Support'
classmethod: CharacterCollection
_idxBasicCanCompareWithUnicodeInstance: aUnicodeString
  "Returns true if <aUnicodeString> may be inserted into a basic BtreeNode whose
   #lastElementClass is the receiver (see RangeEqualityIndex class>>isBasicClass:)."

  "If using Unicode compares, then *String and Unicode* instances may be compared.
   If not using Unicode compares then *String and Unicode* instances may not be compared."

  ^ Unicode16 usingUnicodeCompares
%

category: 'Instance Creation'
classmethod: CharacterCollection
_newString

"Returns a new instance of the receiver, or instance of String as appropriate.
 Reimplemented in subclasses as needed to handle canonical symbols."

^ self new
%

category: 'Private'
classmethod: CharacterCollection
_newString: aSize

"Returns a new instance of the receiver, or instance of String as appropriate.
 Reimplemented in subclasses as needed to handle canonical symbols."

^ self new: aSize
%

!		Instance methods for 'CharacterCollection'

category: 'Concatenating'
method: CharacterCollection
, aCharOrCharCollection

"Returns a new instance of the receiver's class that contains the elements of
 the receiver followed by the elements of aCharOrCharCollection.

 Warning: Creating a new instance and copying the receiver take time.  If you
 can safely modify the receiver, it can be much faster to use the addAll:
 method.  See the documentation of the Concatenating category of class
 SequenceableCollection for more details."

^ self copy addAll: aCharOrCharCollection; yourself
%

category: 'Comparing'
method: CharacterCollection
< aCharCollection

"Returns true if the receiver collates before the argument.  Returns false
 otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #<
%

category: 'Comparing'
method: CharacterCollection
<= aCharCollection

"Returns true if the receiver collates before the argument or if all of the
 corresponding Characters in the receiver and argument are equal.
 Returns false otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #<=
%

category: 'Comparing'
method: CharacterCollection
= aCharCollection

"Returns true if all of the corresponding Characters in the receiver and
 argument are equal.  Returns false otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #=
%

category: 'Comparing'
method: CharacterCollection
> aCharCollection

"Returns true if the receiver collates after the argument.  Returns false
 otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #>
%

category: 'Comparing'
method: CharacterCollection
>= aCharCollection

"Returns true if the receiver collates after the argument or if all of the
 corresponding Characters in the receiver and argument are equal.  Returns
 false otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #>=
%

category: 'Adding'
method: CharacterCollection
add: aCharOrCharColl

"Appends all of the elements of aCharOrCharColl to the receiver and returns
 aCharOrCharColl.  Returns aCharOrCharColl"

| index |

(aCharOrCharColl isKindOf: Collection) ifTrue:[
    index := self size.
    aCharOrCharColl accompaniedBy: self do: [:me :aChar |
      index := index + 1.
      me at: index put: aChar.
    ].
    ^ aCharOrCharColl.
].

^ self at: (self size + 1) put: aCharOrCharColl.
%

category: 'Adding'
method: CharacterCollection
addAll: aCharOrCharCollection

"Equivalent to add: aCharOrCharCollection."

^ self add: aCharOrCharCollection
%

category: 'Adding'
method: CharacterCollection
addCodePoint: aSmallInteger

 ^ self add: (Character codePoint: aSmallInteger)
%

category: 'Adding'
method: CharacterCollection
addLast: aCharOrCharCollection

"Equivalent to add: aCharOrCharCollection."

^ self add: aCharOrCharCollection
%

category: 'Updating'
method: CharacterCollection
addLineDelimiters

"Returns a copy of the receiver that contains each occurrence of the backslash
 Character replaced by the line-feed Character."

^ self copyReplaceAll: '\' with: (self class with: Character lf).
%

category: 'Formatting'
method: CharacterCollection
addLineWith: aCharacterCollection centeredToWidth: anInt

"Add a centered line of width anInt to the receiver.
 Returns the receiver."

| numSpaces aSpace|
aSpace := Character space.
numSpaces := anInt - aCharacterCollection size.
numSpaces <= 0 "No room, just do the add"
  ifTrue:[self addAll: aCharacterCollection.]
  ifFalse:[ | leftSpaces |
    leftSpaces := numSpaces // 2.
    leftSpaces timesRepeat:[self add: aSpace].
    self addAll: aCharacterCollection.
    (numSpaces - leftSpaces) timesRepeat:[self add: aSpace].
].
^self
%

category: 'Decrypting'
method: CharacterCollection
aesDecryptWith128BitKey: aKey salt: aSalt
"Decrypts the receiver using 128 bit AES decryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesDecryptWith128BitKey: aKey salt: aSalt into: nil
%

category: 'Decrypting'
method: CharacterCollection
aesDecryptWith128BitKey: aKey salt: aSalt into: destObjOrNil
"Decrypts the receiver using 128 bit AES decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: -1 into: destObjOrNil
%

category: 'Decrypting'
method: CharacterCollection
aesDecryptWith192BitKey: aKey salt: aSalt
"Decrypts the receiver using 192 bit AES decryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesDecryptWith192BitKey: aKey salt: aSalt into: nil
%

category: 'Decrypting'
method: CharacterCollection
aesDecryptWith192BitKey: aKey salt: aSalt into: destObjOrNil
"Decrypts the receiver using 192 bit AES decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: -2 into: destObjOrNil
%

category: 'Decrypting'
method: CharacterCollection
aesDecryptWith256BitKey: aKey salt: aSalt
"Decrypts the receiver using 256 bit AES decryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesDecryptWith256BitKey: aKey salt: aSalt into: nil
%

category: 'Decrypting'
method: CharacterCollection
aesDecryptWith256BitKey: aKey salt: aSalt into: destObjOrNil
"Decrypts the receiver using 256 bit AES decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: -3 into: destObjOrNil
%

category: 'Encrypting'
method: CharacterCollection
aesEncryptWith128BitKey: aKey salt: aSalt
"Encrypts the receiver using 128 bit AES encryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesEncryptWith128BitKey: aKey salt: aSalt into: nil
%

category: 'Encrypting'
method: CharacterCollection
aesEncryptWith128BitKey: aKey salt: aSalt into: destObjOrNil
"Encrypts the receiver using 128 bit AES encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 1  into: destObjOrNil
%

category: 'Encrypting'
method: CharacterCollection
aesEncryptWith192BitKey: aKey salt: aSalt
"Encrypts the receiver using 192 bit AES encryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesEncryptWith192BitKey: aKey salt: aSalt into: nil
%

category: 'Encrypting'
method: CharacterCollection
aesEncryptWith192BitKey: aKey salt: aSalt into: destObjOrNil
"Encrypts the receiver using 192 bit AES encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 2 into: destObjOrNil
%

category: 'Encrypting'
method: CharacterCollection
aesEncryptWith256BitKey: aKey salt: aSalt
"Encrypts the receiver using 256 bit AES encryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesEncryptWith256BitKey: aKey salt: aSalt into: nil
%

category: 'Encrypting'
method: CharacterCollection
aesEncryptWith256BitKey: aKey salt: aSalt into: destObjOrNil
"Encrypts the receiver using 256 bit AES encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 3 into: destObjOrNil
%

category: 'Authenticated Decrypting'
method: CharacterCollection
aesGcmDecryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using 128 bit AES-GCM decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -7
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Decrypting'
method: CharacterCollection
aesGcmDecryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using 192 bit AES-GCM decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -8
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Decrypting'
method: CharacterCollection
aesGcmDecryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using 256 bit AES-GCM decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -9
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Encrypting'
method: CharacterCollection
aesGcmEncryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using 128 bit AES-GCM encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 7 into: destObjOrNil tag: tag extraData: eData
%

category: 'Authenticated Encrypting'
method: CharacterCollection
aesGcmEncryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using 192 bit AES-GCM encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 8
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Encrypting'
method: CharacterCollection
aesGcmEncryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using 256 bit AES-GCM encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 9
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Decrypting'
method: CharacterCollection
aesOcbDecryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using 128 bit AES-OCB decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -4
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Decrypting'
method: CharacterCollection
aesOcbDecryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using 192 bit AES-OCB decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -5
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Decrypting'
method: CharacterCollection
aesOcbDecryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using 256 bit AES-OCB decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -6
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Encrypting'
method: CharacterCollection
aesOcbEncryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using 128 bit AES-OCB encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 4
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Encrypting'
method: CharacterCollection
aesOcbEncryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using 192 bit AES-OCB encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 5
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Encrypting'
method: CharacterCollection
aesOcbEncryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using 256 bit AES-OCB encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 6
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Converting'
method: CharacterCollection
asArrayOf32PathTerms
  "Returns an Array of path substrings held by the receiver.  The receiver
 is assumed to be a period-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 periods, the Array will hold a copy of the receiver.  The $\ Character
 is no longer recognized as an escape character.

 Raises an error if an element is not a valid path term."

  | nextName result |
  result := {}.
  nextName := self speciesForConversion new.
  self
    do: [ :c |
      c == $.
        ifTrue: [
          nextName _isValid32PathTermName
            ifFalse: [ ^ self _error: #'rtErrInvalidIndexPathExpression' args: { nextName } ].
          result add: nextName asSymbol.
          nextName := self speciesForConversion new ]
        ifFalse: [ nextName add: c ] ].
  nextName size ~~ 0
    ifTrue: [
      nextName _isValid32PathTermName
        ifFalse: [ ^ self _error: #'rtErrInvalidIndexPathExpression' args: { nextName } ].
      result add: nextName asSymbol ]
    ifFalse: [
      result size == 0
        ifTrue: [ result add: nextName asSymbol ] ].
  ^ result
%

category: 'Converting'
method: CharacterCollection
asArrayOfKeywords

"Returns an Array of keyword substrings held by the receiver.  The receiver
 is assumed to be a colon-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 colons, the Array will hold a copy of the receiver."

| c nextName result |

result := { }.
nextName := self speciesForConversion new.
1 to: self size do: [ :i |
  c := self at: i.
  nextName add: c.
  c == $: ifTrue: [
    result add: nextName.
    nextName := self speciesForConversion new.
  ].
].
nextName size ~~ 0 ifTrue: [
  result add: nextName
]
ifFalse: [
  result size == 0 ifTrue: [result add: nextName]
].
^result
%

category: 'Converting'
method: CharacterCollection
asArrayOfPathTerms
  "Returns an Array of path substrings held by the receiver.  The receiver
 is assumed to be a period-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 periods, the Array will hold a copy of the receiver.  The $\ Character
 is no longer recognized as an escape character.

 Raises an error if an element is not a valid path term."

  | nextName result |
  result := {}.
  nextName := self speciesForConversion new.
  self
    do: [ :c |
      c == $.
        ifTrue: [
          nextName _isValidPathTermName
            ifFalse: [ ^ self _error: #'rtErrInvalidIndexPathExpression' args: {nextName} ].
          result add: nextName asSymbol.
          nextName := self speciesForConversion new ]
        ifFalse: [ nextName add: c ] ].
  nextName size ~~ 0
    ifTrue: [
      nextName _isValidPathTermName
        ifFalse: [ ^ self _error: #'rtErrInvalidIndexPathExpression' args: {nextName} ].
      result add: nextName asSymbol ]
    ifFalse: [
      result size == 0
        ifTrue: [ result add: nextName asSymbol ] ].
  (result at: 1) = #'*'
    ifTrue: [
      "* not allowed as first term"
      ^ self _error: #'rtErrInvalidIndexPathExpression' args: {(result at: 1)} ].
  ^ result
%

category: 'Converting'
method: CharacterCollection
asArrayOfSubstrings

"Returns an Array of substrings held by the receiver. The receiver
 is assumed to be a separator-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 separators, the Array will hold a copy of the receiver.  Separators not meant
 to separate substrings may be escaped with a $\ Character."

| nextName result esc sz |

result := { } .
(sz := self size) == 0 ifTrue: [
  ^result
].
nextName := self speciesForConversion new.
esc := false.
1 to: sz do: [ :i | | c |
  c := self at: i.
  esc ifTrue: [
    nextName add: c.
    esc := false.
  ] ifFalse: [
    c == $\ ifTrue: [ esc := true ]
    ifFalse: [
	c isSeparator ifTrue: [
          nextName size ~~ 0 ifTrue: [result add: nextName].
          nextName := self speciesForConversion new.
        ] ifFalse: [
          nextName add: c
        ].
    ].
  ].
].

esc ifTrue:[ nextName add: $\ ].

(nextName size ~~ 0 or: [result size == 0]) ifTrue:[ result add: nextName ].

^result
%

category: 'Converting'
method: CharacterCollection
asBase64String
"Return a String which represents the receiver represented in base64 format."

^ self asBase64StringOnOneLine: true
%

category: 'Converting'
method: CharacterCollection
asBase64StringOnOneLine: aBoolean
"Return a String which represents the receiver represented in base64 format.
 If aBoolean is true, the resulting is one long line which does not contain
 newline characters.  If aBoolean is false, newline characters are inserted
 such that each line does not exceed 64 characters."

<primitive: 1062>
aBoolean _validateClass: Boolean .
self _primitiveFailed: #asBase64StringOnOneLine:
%

category: 'Converting'
method: CharacterCollection
asBase64UrlString
"Return a String which represents the receiver represented in base64Url format."

<primitive: 872>
self _primitiveFailed: #asBase64UrlString
%

category: 'Converting'
method: CharacterCollection
asDecimalFloat

"Returns a DecimalFloat whose value is represented by the receiver."

^ DecimalFloat fromString: self
%

category: 'Converting'
method: CharacterCollection
asDoubleByteString

"Returns a DoubleByteString representation of the receiver."

^ DoubleByteString withAll: self asString .       "fix 39372"
%

category: 'Converting'
method: CharacterCollection
asFloat

"Returns a SmallDouble or Float whose value is represented by the receiver."

^ Float fromString: self
%

category: 'Converting'
method: CharacterCollection
asHexString

"Returns a String containing a hexadecimal printed representation of the
 contents of the receiver.  For example, the message 'abc' asHexString
 returns the String '616263'.

 The receiver must be a byte format object."

<primitive: 467>
self _validateByteClass: CharacterCollection .
self _primitiveFailed: #asHexString .
%

category: 'Message Authentication Codes'
method: CharacterCollection
asHmacSumWithDigestKind: opCode key: keyString

"Computes the keyed-hash message authentication code (HMAC) of the receiver using
 the message digest algorithm indicated by opCode and secret key keyString.

 opCode must be one of the following:

 opCode   Digest Algorithm   Result Class
 ========================================
  -1          md5            ByteArray
  -2          sha1           ByteArray
  -3          sha2-256       ByteArray
  -4          sha2-512       ByteArray
  -5          sha3-224       ByteArray
  -6          sha3-256       ByteArray
  -7          sha3-384       ByteArray
  -8          sha3-512       ByteArray
   1          md5            LargeInteger
   2          sha1           LargeInteger
   3          sha2-256       LargeInteger
   4          sha2-512       LargeInteger
   5          sha3-224       LargeInteger
   6          sha3-256       LargeInteger
   7          sha3-384       LargeInteger
   8          sha3-512       LargeInteger
 =========================================

 secretKey must be an instance or subclass of a ByteArray or String and must
 have a character size of one, i.e.: its class must answer 1 when sent the
 message #_bytesPerWord. secretKey must have a size between 1 and 64 bytes.

 Answers the HMAC of the receiver as a LargeInteger or ByteArray per above table."

<primitive: 1060>
opCode _validateClass: SmallInteger .
((opCode == 0) or:[(opCode < -8) or:[ opCode > 8]])
  ifTrue:[ opCode _error: #rtErrArgOutOfRange args:{ -8 . 8 } ].
keyString _validateClasses: { String . ByteArray } .
((keyString _basicSize > 64) or:[keyString _basicSize < 1])
  ifTrue:[ keyString _error: #rtErrBadSize args: { 64 . keyString _basicSize } ] .
self _primitiveFailed: #asHmacSumWithDigestKind:key: args: { opCode . keyString } .
%

category: 'Converting'
method: CharacterCollection
asInteger

"Returns an Integer whose value is represented by the receiver."

^ Integer fromString: self
%

category: 'Converting'
method: CharacterCollection
asLowercase

"Returns a new instance of the receiver's class, with all upper-case
 characters in the receiver changed to lower-case.

 (Subclass responsibility.)"

^ self subclassResponsibility: #asLowercase
%

category: 'Message Authentication Codes'
method: CharacterCollection
asMd5HmacByteArrayWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the MD5 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a ByteArray."

^ self asHmacSumWithDigestKind: -1 key: keyString
%

category: 'Message Authentication Codes'
method: CharacterCollection
asMd5HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the MD5 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asMd5HmacWithKey: keyString) asHexStringWithLength: 32
%

category: 'Message Authentication Codes'
method: CharacterCollection
asMd5HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the MD5 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 1 key: keyString
%

category: 'Message Digests'
method: CharacterCollection
asMd5String

"Compute the 128 bit MD5 message digest for the receiver and return it as
 a 32 character string of hexadecimal characters."

^ self md5sum asHexStringWithLength: 32
%

category: 'Converting'
method: CharacterCollection
asMultiByteString: example

"Returns a Double/QuadByteString representation of the receiver,
 depending on the class of the example string."

^ example class withAll: self
%

category: 'Converting'
method: CharacterCollection
asNumber

"Returns the receiver converted to a kind of number.  If the receiver contains
 all digits (with optional radix notation), returns a kind of Integer.  If the
 receiver has a slash, returns a Fraction.  Otherwise conversion to a Float is
 attempted.  An error may result if the receiver does not contain the proper
 format for a kind of Number."

 | strm |
 self size == 0 ifTrue: [ ^0 ].
 strm := ReadStreamPortable on: self .
 (self indexOf: $/ startingAt: 1) ~~ 0 ifTrue:[
    ^ Fraction fromStream: strm .
 ].
 ^ Number fromStream: strm .
%

category: 'Converting'
method: CharacterCollection
asQuadByteString

"Returns a QuadByteString representation of the receiver."

^ QuadByteString withAll: self.
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha1HmacByteArrayWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha1 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a ByteArray."

^ self asHmacSumWithDigestKind: -2 key: keyString
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha1HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha1 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha1HmacWithKey: keyString) asHexStringWithLength: 40
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha1HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha1 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 2 key: keyString
%

category: 'Message Digests'
method: CharacterCollection
asSha1String

"Compute the 160 bit SHA1 message digest for the receiver and return it as
 a 40 character string of hexadecimal characters."

^ self sha1Sum asHexStringWithLength: 40
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha256HmacByteArrayWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a ByteArray."

^ self asHmacSumWithDigestKind: -3 key: keyString
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha256HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha256HmacWithKey: keyString) asHexStringWithLength: 64
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha256HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 3 key: keyString
%

category: 'Message Digests'
method: CharacterCollection
asSha256String

"Compute the 256 bit SHA-2 message digest for the receiver and return it as
 a 64 character string of hexadecimal characters."

^ self sha256Sum asHexStringWithLength: 64
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha3_224HmacByteArrayWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 224 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a ByteArray."

^ self asHmacSumWithDigestKind: -5 key: keyString
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha3_224HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 224 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_224HmacWithKey: keyString) asHexStringWithLength: 56
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha3_224HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 224 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 5 key: keyString
%

category: 'Message Digests'
method: CharacterCollection
asSha3_224String

"Compute the SHA3 224 bit message digest for the receiver and return it as
 a 56 character string of hexadecimal characters."

^ self sha3_224Sum asHexStringWithLength: 56
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha3_256HmacByteArrayWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a ByteArray."

^ self asHmacSumWithDigestKind: -6 key: keyString
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha3_256HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_256HmacWithKey: keyString) asHexStringWithLength: 64
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha3_256HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 6 key: keyString
%

category: 'Message Digests'
method: CharacterCollection
asSha3_256String

"Compute the SHA3 256 bit message digest for the receiver and return it as
 a 64 character string of hexadecimal characters."

^ self sha3_256Sum asHexStringWithLength: 64
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha3_384HmacByteArrayWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 384 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a ByteArray."

^ self asHmacSumWithDigestKind: -7 key: keyString
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha3_384HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 384 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_384HmacWithKey: keyString) asHexStringWithLength: 96
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha3_384HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 384 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 7 key: keyString
%

category: 'Message Digests'
method: CharacterCollection
asSha3_384String

"Compute the SHA3 384 bit message digest for the receiver and return it as
 a 96 character string of hexadecimal characters."

^ self sha3_384Sum asHexStringWithLength: 96
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha3_512HmacByteArrayWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a ByteArray."

^ self asHmacSumWithDigestKind: -8 key: keyString
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha3_512HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_512HmacWithKey: keyString) asHexStringWithLength: 128
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha3_512HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 8 key: keyString
%

category: 'Message Digests'
method: CharacterCollection
asSha3_512String

"Compute the SHA3 512 bit message digest for the receiver and return it as
 a 128 character string of hexadecimal characters."

^ self sha3_512Sum asHexStringWithLength: 128
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha512HmacByteArrayWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a ByteArray."

^ self asHmacSumWithDigestKind: -4 key: keyString
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha512HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha512HmacWithKey: keyString) asHexStringWithLength: 128
%

category: 'Message Authentication Codes'
method: CharacterCollection
asSha512HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 4 key: keyString
%

category: 'Message Digests'
method: CharacterCollection
asSha512String

"Compute the 512 bit SHA-2 message digest for the receiver and return it as
 a 128 character string of hexadecimal characters."

^ self sha512Sum asHexStringWithLength: 128
%

category: 'Deprecated'
method: CharacterCollection
asSmallFloat

"SmallFloat is deprecated. Return a SmallDouble or Float whose value
 is represented by the receiver."

self deprecated: 'CharacterCollection>>asSmallFloat deprecated v3.0. Use asFloat or Float class>>#fromString: '.
^ (Float fromString: self) asSmallFloat
%

category: 'Converting'
method: CharacterCollection
asString

"Returns a String representation of the receiver."

^ String withAll: self
%

category: 'Converting'
method: CharacterCollection
asSymbolKind

"Returns a canonical symbol containing the same Characters as the receiver."

CharacterCollection subclassResponsibility: #asSymbolKind
%

category: 'Converting'
method: CharacterCollection
asUppercase

"Returns a new instance of the receiver's class, with all lower-case
 characters in the receiver changed to upper-case.

 (Subclass responsibility.)"

^ self subclassResponsibility: #asUppercase
%

category: 'Comparing'
method: CharacterCollection
at: anIndex equals: aCharCollection

"Returns true if aCharCollection is contained in the receiver starting at
 anIndex.  Returns false otherwise.

 Note that this method returns true only if aCharCollection begins exactly at
 the position designated by anIndex.  To locate a pattern beginning on or after
 anIndex, see the method findPattern:startingAt: in category 'Searching'.

 (Subclass responsibility.)"

^ self subclassResponsibility: #at:equals:
%

category: 'Authenticated Encrypting'
method: CharacterCollection
authenticatedEncryptionExample

| key salt enc dec hamlet extraData tag|
key  := ByteArray withRandomBytes: 32 .
salt := ByteArray withRandomBytes: 12 .

hamlet :=
'Alas, poor Yorick! I knew him, Horatio: a fellow
of infinite jest, of most excellent fancy: he hath
borne me on his back a thousand times; and now, how
abhorred in my imagination it is! my gorge rims at
it. Here hung those lips that I have kissed I know
not how oft. Where be your gibes now? your
gambols? your songs? your flashes of merriment,
that were wont to set the table on a roar? Not one
now, to mock your own grinning? quite chap-fallen?
Now get you to my ladys chamber, and tell her, let
her paint an inch thick, to this favour she must
come; make her laugh at that. Prithee, Horatio, tell
me one thing.' .

extraData := 'Hamlet: Act V, Scene i'.

tag := ByteArray new.
enc := hamlet aesOcbEncryptWith256BitKey: key salt: salt
       into: ByteArray new tag: tag extraData: extraData .
dec := enc aesOcbDecryptWith256BitKey: key salt: salt
       into: String new tag: tag extraData: extraData .
^ dec = hamlet
%

category: 'Comparing'
method: CharacterCollection
between: min and: max
	"Answer whether the receiver is less than or equal to the argument max and
	 greater than or equal to the argument min."

	^self >= min and: [self <= max].
%

category: 'Authenticated Decrypting'
method: CharacterCollection
chacha20Poly1305DecryptWithKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using CHACHA20-Poly1305 decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -10
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Encrypting'
method: CharacterCollection
chacha20Poly1305EncryptWithKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using CHACHA20-Poly1305 encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 10
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Formatting'
method: CharacterCollection
charSize

"Returns number of bytes that make up a character for this string class.
 (Subclass responsibility.)"

^ self subclassResponsibility: #charSize
%

category: 'Accessing'
method: CharacterCollection
codePointAt: anIndex

CharacterCollection subclassResponsibility: #'codePointAt:'
%

category: 'Updating'
method: CharacterCollection
codePointAt: anIndex put: aValue

CharacterCollection subclassResponsibility: #'codePointAt:put:'
%

category: 'Comparing'
method: CharacterCollection
compareTo: aString collator: anIcuCollator

"Returns -1, 0 or 1,  when receiver is less than,
 equal to, or greater than argString .
 argString must be a String, MultiByteString, or a Utf8 .
 anIcuCollator == nil is interpreted as   IcuCollator default ."

^ self compareTo: aString collator: anIcuCollator useMinSize: false
%

category: 'Comparing'
method: CharacterCollection
compareTo: argString collator: anIcuCollator useMinSize: aMinSize
  "Returns -1, 0 or 1,  when receiver is less than,
 equal to, or greater than argString .
 argString must be a String, MultiByteString, or a Utf8.
 anIcuCollator == nil is interpreted as   IcuCollator default .

 If aMinSize==true, compare stops at (self size min: argString size),
 which is Squeak semantics for comparison .

 If aMinSize is a SmallInteger >= 1, compare stops at
 aMinSize min: (self size min: argString size) ."

  ^ self subclassResponsibility: #'compareTo:collator:useMinSize:'
%

category: 'Searching'
method: CharacterCollection
containsDigit

"Answers true if the receiver contains at least one digit, otherwise
 answers false."

1 to: self size do:[:n| (self at: n) isDigit ifTrue:[ ^ true ] ] .
^ false
%

category: 'Searching'
method: CharacterCollection
containsLowercase

"Answers true if the receiver contains at least one lower case character,
 otherwise answers false.
 Differs from legacy method containsLowercaseOld for
 code points 170, 186, and about 1700 code points >= 256"

1 to: self size do:[:n| (self at: n) isLowercase ifTrue:[ ^ true ] ] .
^ false
%

category: 'Legacy'
method: CharacterCollection
containsLowercaseOld

"Answers true if the receiver contains at least one lower case character,
 otherwise answers false."

1 to: self size do:[:n| (self at: n) isLowercaseOld ifTrue:[ ^ true ] ] .
^ false
%

category: 'Private'
method: CharacterCollection
containsPasswordSymbol

"Answers true if the receiver contains at least one symbol character, with
 symbol in this case meaning any Character that is neither a letter nor a
 digit; otherwise answers false."

1 to: self size do:[:n| (self at: n) _isPasswordSymbol ifTrue:[ ^ true ] ] .
^ false
%

category: 'Searching'
method: CharacterCollection
containsUppercase

"Answers true if the receiver contains at least one upper case character,
 otherwise answers false."

1 to: self size do:[:n| (self at: n) isUppercase ifTrue:[ ^ true ] ] .
^ false
%

category: 'Repository Conversion'
method: CharacterCollection
convert
"If receiver is a committed object and receiver's class on disk was
 a subclass of one of
   ObsDoubleByteString, ObsDoubleByteSymbol, ObsQuadByteString
 the receiver is added to the writeSet of current transaction,
 otherwise has no effect.
 Returns true if receiver was added to the writeSet, false otherwise."

<primitive: 744>
self _primitiveFailed:#convert
%

category: 'Copying'
method: CharacterCollection
copy
  "optimization, no need for postCopy"
<primitive: 885>
self _primitiveFailed: #copy .
self _uncontinuableError
%

category: 'Copying'
method: CharacterCollection
copyFrom: startIndex to: stopIndex

"Returns a new SequenceableCollection containing the elements of the receiver
 between startIndex and stopIndex, inclusive.  The result is of the same class
 as the receiver, unless the receiver is a Symbol or DoubleByteSymbol,
 in which case the result class is respectively String or DoubleByteString.

 Both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex.
 If startIndex > stopIndex and both are positive, an empty collection is returned.
 "

| result |
(startIndex > stopIndex) ifTrue: [
  stopIndex < 0 ifTrue:[ self _error: #rtErrBadCopyFromTo args: { stopIndex } ].
  ^ self class new
].
(startIndex < 1)
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].

((stopIndex > self size) or: [(stopIndex < 0)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].

result := (self class _newString: (stopIndex - startIndex + 1)).
result replaceFrom: 1 to: 1 + stopIndex - startIndex
	with: self startingAt: startIndex  .
^ result
%

category: 'Copying'
method: CharacterCollection
copyReplaceAll: subString with: newSubString

"Returns a copy of the receiver with all occurrences of the given substring
 replaced with newSubString."

| copy csize matches ssize idx |

copy := self copy.
matches := { } .
ssize := subString size.
ssize == 0 ifTrue: [^copy].
idx := 1 - ssize.
csize := copy size.
[ idx := idx + subString size.
  idx <= csize and: [
    idx := copy findString: subString startingAt: idx.
    idx ~~ 0
  ]
] whileTrue: [
  matches add: idx
].
matches reverseDo: [:p |
  copy removeFrom: p to: p+ssize-1.
  copy insertAll: newSubString at: p
].
^copy
%

category: 'Copying'
method: CharacterCollection
copyReplaceChar: aCharacter with: secondCharacter

"Returns a copy of the receiver with all occurrences of the given
 Character replaced with secondCharacter."

| copy |
secondCharacter _validateClass: Character .
copy := self speciesForCollect withAll: self .
1 to: copy size do:[:n |
 (copy at: n) == aCharacter ifTrue:[ copy at: n put: secondCharacter].
].
^ copy
%

category: 'Copying'
method: CharacterCollection
copyWithout: anObject

"Returns a copy of the receiver that does not contain the given object.
 Comparisons are by equality."

| copy element sz |

copy := self class _newString .

sz := 0.
1 to: self size do: [:i |
  element := self at: i.
  (element = anObject)
    ifFalse: [
      sz := sz + 1.
      copy at: sz put: element.
      ]
  ].

^copy
%

category: 'Converting'
method: CharacterCollection
copyWrappedTo: rightMargin

"Returns a String with the receiver's contents word-wrapped to the given
 right margin."

| res col ch lf |

self size < rightMargin ifTrue: [^self].
lf := Character lf.
res := self species new.
col := 0.
1 to: self size do: [ :i |
  ch := self at: i.
  res add: ch.
  ch == lf ifTrue: [ col := 0 ]
  ifFalse: [
    col := col + 1.
    (col > rightMargin and: [ch isSeparator]) ifTrue: [
      res add: lf.
      col := 1.
    ].
  ].
].
^ res
%

category: 'Formatting'
method: CharacterCollection
describeClassName

"Returns a copy of the receiver with the Character $a prepended to the
 receiver's contents.  This method is used for formatting class names in object
 descriptions, where the receiver is a string containing the name of a class.
 For example, the String 'UserClass', when sent the message describeClassName,
 returns 'aUserClass'."

| result |
result := self speciesForPrint new .
self size ~~ 0 ifTrue:[   "inline simplified isVowel using Unicode asUppercase"
  ( #( $A $E $I $O $U $Y ) includesIdentical: (self at: 1) asUppercase ) ifTrue:[
    result add: 'an' ; addAll: self .
    ^ result
  ]
].
result add: $a ; add: self.
^ result
%

category: 'Encrypting'
method: CharacterCollection
encryptionExample

| key salt enc dec hamlet |
key  := ByteArray withRandomBytes: 32 .
salt := ByteArray withRandomBytes: 16 .

hamlet :=
'Alas, poor Yorick! I knew him, Horatio: a fellow
of infinite jest, of most excellent fancy: he hath
borne me on his back a thousand times; and now, how
abhorred in my imagination it is! my gorge rims at
it. Here hung those lips that I have kissed I know
not how oft. Where be your gibes now? your
gambols? your songs? your flashes of merriment,
that were wont to set the table on a roar? Not one
now, to mock your own grinning? quite chap-fallen?
Now get you to my ladys chamber, and tell her, let
her paint an inch thick, to this favour she must
come; make her laugh at that. Prithee, Horatio, tell
me one thing.' .


enc := hamlet aesEncryptWith256BitKey: key salt: salt .
dec := enc aesDecryptWith256BitKey: key salt: salt into: String new.
^ dec = hamlet
%

category: 'Testing'
method: CharacterCollection
equalsNoCase: aCharCollection
    "Returns true if the receiver is equivalent to aCharCollection.
    The receiver is equivalent to aCharCollection if the receiver
    contains the same Characters as aCharCollection regardless of case
    or internal representation.  For example, if $a is in
    aCharCollection, it is equivalent to any representation of an 'a'
    in the receiver's character set.

    (Subclass responsibility.)"

^ self subclassResponsibility: #equalsNoCase:
%

category: 'Case-Sensitive Searching'
method: CharacterCollection
findLastSubString: subString startingAt: startIndex

"startIndex should be >= 1 and <= self size . Search is backwards
 through the receiver. Returns 0 if no match found. "
^  self _findLastString: subString startingAt: startIndex ignoreCase: false
%

category: 'Searching'
method: CharacterCollection
findPattern: aPattern startingAt: anIndex

"This method searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, this method returns the
 index of the first Character of the substring.  Otherwise, this returns 0.

 The argument aPattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 Performs a case-sensitive search."

^ self _findPattern: aPattern startingAt: anIndex ignoreCase: false
%

category: 'Searching'
method: CharacterCollection
findPatternNoCase: aPattern startingAt: anIndex

"This method searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, this method returns the
 index of the first Character of the substring.  Otherwise, this returns 0.

 The argument aPattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 Performs a case-insensitive search."

^ self _findPattern: aPattern startingAt: anIndex ignoreCase: true
%

category: 'Case-Sensitive Searching'
method: CharacterCollection
findString: subString startingAt: startIndex

"If a receiver contains subString beginning at some point at or after
 startIndex, this returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0.

 The search is case-sensitive."

^ self _findString: subString startingAt: startIndex ignoreCase: false
%

category: 'Case-Insensitive Searching'
method: CharacterCollection
findStringNoCase: subString startingAt: startIndex

"If a receiver contains subString beginning at some point at or after
 startIndex, this returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0.

 The search is case-insensitive."

^ self _findString: subString startingAt: startIndex ignoreCase: true
%

category: 'Hashing'
method: CharacterCollection
hash

"Returns a positive Integer based on a case-sensitive hash of the contents
 of the receiver.  The algorithm implemented is described in:

 [Pearson 90]
 Pearson, Peter K., Fast Hashing of Variable-Length Text Strings,
 Communications of the ACM 33, 6, (June 1990), 677-680.

 This implementation inherited by JapaneseString.  String and
 MultiByteString each reimplement hash."

<primitive: 31>
self _primitiveFailed: #hash .
self _uncontinuableError
%

category: 'LDAP Support'
method: CharacterCollection
hasLdapWildcardPattern

"Answer true if the receiver contains one and only one '%' character,
 and the character after the '%' is 's'."

(self select:[:e| e == $%]) size == 1
  ifFalse:[^ false ].

^(self findPattern: #('%' 's') startingAt: 1) ~~ 0
%

category: 'Case-Insensitive Searching'
method: CharacterCollection
includesString: aString

"Returns true if aString is contained as a subString within the receiver,
 using a case-insensitive search.  Returns false otherwise."

^ (self _findString: aString startingAt: 1 ignoreCase: true) ~~ 0
%

category: 'Searching'
method: CharacterCollection
indexOf: pattern matchCase: flag startingAt: startIndex

"Searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, returns the
 index of the first Character of the substring.  Otherwise, returns 0.

 The argument pattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 If the flag argument is true, a case-sensitive search is performed.  Otherwise,
 a case-insensitive search is performed."

^ self _findPattern: pattern startingAt: startIndex ignoreCase: (flag not).
%

category: 'Searching'
method: CharacterCollection
indexOfSubCollection: aSubColl startingAt: anIndex

"Returns the index of the first element of the receiver where that element and
 the subsequent ones are equal to those in aSubColl. The search is begun in the
 receiver at starting at anIndex. Returns zero if no match is found."

(aSubColl size == 0) ifTrue: [ ^ 0 ].
^ self findString: aSubColl startingAt: anIndex .
%

category: 'Searching'
method: CharacterCollection
indexOfSubCollection: aSubColl startingAt: anIndex ifAbsent: anExceptionBlock

"Returns the index of the first element of the receiver where that element and
 the subsequent ones are equal to those in aSubColl. The search is begun in the
 receiver at starting at anIndex. Returns the value of evaluating
 anExceptionBlock if no match is found."

| idx |
(aSubColl size == 0) ifTrue: [ ^ anExceptionBlock value ].
idx := self findString: aSubColl startingAt: anIndex .
idx == 0 ifTrue:[ ^ anExceptionBlock value ].
^ idx
%

category: 'Deprecated'
method: CharacterCollection
insert: aCharOrCharCollection at: anIndex

self deprecated: 'CharacterCollection>>insert:at: deprecated long before v3.0. Use insertAll:at: instead.'.
^ self insertAll: aCharOrCharCollection at: anIndex.
%

category: 'Adding'
method: CharacterCollection
insertAll: aCharOrCharCollection at: anIndex

"Inserts aCharOrCharCollection into the receiver at the specified index and
 returns aCharOrCharCollection."

(aCharOrCharCollection isKindOf: CharacterCollection)
   ifTrue: [ ^ super insertAll: aCharOrCharCollection at: anIndex ].

(aCharOrCharCollection class == Character)
  ifTrue: [
    self replaceFrom: anIndex + 1 to: 1 + self size with: self startingAt: anIndex .
    self at: anIndex put: aCharOrCharCollection.
    ^aCharOrCharCollection.
    ].

^ aCharOrCharCollection _error: #rtErrInvalidArgClass
                        args: { Character . CharacterCollection }.
%

category: 'Adding'
method: CharacterCollection
insertObject: anObject at: anIndex

"Inserts anObject into the receiver at index anIndex and returns
 aCollection."

"reimplemented to use at:put: instead of _basicAt:put:"
| selfSize |

anIndex <= 0 ifTrue:[ ^ self _errorIndexOutOfRange: anIndex ].
selfSize := self size.
anIndex > selfSize ifTrue:[
  anIndex > (selfSize + 1) ifTrue:[ ^ self _errorIndexOutOfRange: anIndex].
  ^ self at: anIndex put: anObject.
  ].

"Not adding to the end of the receiver. Create a gap for anObject to
 be copied into."
self replaceFrom: anIndex + 1 to: 1 + selfSize with: self startingAt: anIndex .

^ self at: anIndex put: anObject.
%

category: 'Testing'
method: CharacterCollection
isDigits

"Returns true if the receiver contains only digits.  Returns false if the
 receiver contains non-digit Characters."

| sz |
(sz := self size) == 0 ifTrue:[ ^ false ].
1 to: sz do: [:i |
    (self at: i) isDigit ifFalse: [^false]
].
^true
%

category: 'Testing'
method: CharacterCollection
isEquivalent: aCharCollection

"Returns true if the receiver is equivalent to aCharCollection.  The receiver
 is equivalent to aCharCollection if the receiver contains the same Characters
 as aCharCollection regardless of case or internal representation.  For
 example, if $a is in aCharCollection, it is equivalent to any representation
 of an 'a' in the receiver's character set.

 (Subclass responsibility.)"

^ self subclassResponsibility: #isEquivalent:
%

category: 'Testing'
method: CharacterCollection
isInfix

"Returns true if the receiver is an infix (binary) selector.  Returns false
 otherwise."

| binaryChars mySize fch |

binaryChars := '+-\*~<>=|/&@%,?!'.   "fixed bug 14109"

mySize := self size.
mySize == 0 ifTrue: [ ^ false ].

fch := self at: 1 .
(fch == $_  or:[ (binaryChars indexOf: fch startingAt: 1) ~~ 0] ) ifFalse:[
  ^ false
].
2 to: mySize do: [ :i |
  ((binaryChars indexOf: (self at: 1) startingAt: 1) == 0) ifTrue:[
     ^ false
  ].
].
^ true.
%

category: 'Testing'
method: CharacterCollection
isKeyword

"Returns true if the receiver is a keyword; that is, a legal keyword method
 selector. Returns false otherwise."

| mySize limit ch idx |

mySize := self size.
mySize > 1  ifFalse: [ ^ false ].
idx := 1 .
[ idx <= mySize ] whileTrue:[
  ch := self at: idx .  "first char of next keyword within selector"
  ( ch == $_ or:[ ch isLetter ] ) ifFalse: [ ^ false ].
  idx := idx + 1 .
  limit := mySize .
  [ idx <= limit ] whileTrue:[
    ch := self at: idx .
    ch == $:  ifTrue:[
      idx == mySize ifTrue:[ ^ true "hit ending $: " ].
      limit := 0 . "end of a keyword, exit inner loop"
    ] ifFalse:[
      (ch == $_  or:[ ch isAlphaNumeric]) ifFalse:[ ^ false ].
    ] .
    idx := idx + 1
  ]
].
^ false
%

category: 'Testing'
method: CharacterCollection
isMinusAndDigits

"Returns true if the receiver contains a minus sign followed only by digits.
 Returns false if the receiver has any other Characters."

((self at: 1) == $- ) ifFalse:[ ^ false ].
2 to: self size do: [:i |
    (self at: i) isDigit ifFalse:[ ^ false]
].
^true
%

category: 'Testing'
method: CharacterCollection
isValidIdentifier

"Returns true if the receiver is a valid GemStone Smalltalk variable name,
 and false otherwise.
 Returns true if first character is $_  or answers true to   isLetter ,
  and subsequent characters are all $_  or answer true to   isAlphaNumeric,
  and size is > 0 and <= 1024 .  
 Otherwise returns false .
 Note 1024==GCI_MAX_SYMBOL_SIZE "

 <primitive: 37>
 ^ self _primitiveFailed: #isValidIdentifier
%

category: 'Updating'
method: CharacterCollection
lf

"Appends a line-feed to the receiver and returns the receiver."

self addCodePoint: 10
%

category: 'Formatting'
method: CharacterCollection
linesIndentedBy: anInt

"Returns a copy of the receiver in which all lines have been indented
 by anInt spaces."

| c newStr indentStr lfInd lf targEndInd selfCurrInd sz destIdx sp |

indentStr := self class new: anInt.
indentStr atAllPut: (sp := Character space).
lf := Character lf.
lfInd := self indexOf: lf.
lfInd == 0 ifTrue:[ | res |
  (res := indentStr) addAll: self .
  ^ res
].

newStr := self class new.
selfCurrInd := 1.
c := self copy.

[(lfInd := c indexOf: lf) ~~ 0 ] whileTrue: [
  targEndInd := newStr size + indentStr size.
  newStr addAll: indentStr.
 "self copyFrom: selfCurrInd to: lfInd into: newStr startingAt: targEndInd + 1. "
  destIdx := targEndInd + 1 .
  newStr replaceFrom: destIdx to: destIdx + lfInd - selfCurrInd
	  with: self startingAt: selfCurrInd .
  selfCurrInd := lfInd + 1.
  c at: lfInd put: sp .
].
selfCurrInd < (sz := self size) ifTrue: [
  newStr addAll: indentStr.
 "self copyFrom: selfCurrInd to: sz into: newStr startingAt: newStr size + 1."
  destIdx := newStr size + 1 .
  newStr replaceFrom: destIdx to: destIdx + sz - selfCurrInd
         with: self startingAt: selfCurrInd .
].
^newStr
%

category: 'Comparing'
method: CharacterCollection
match: prefix

"Returns true if the argument prefix is a prefix of the receiver, and
 false if not.  The comparison is case-sensitive."

self size == 0 ifTrue: [ ^ prefix size == 0 ].
^ self at: 1 equals: prefix
%

category: 'Comparing'
method: CharacterCollection
matchesAnyOf: aCollectionOfCharacterColls

"Returns true if the receiver returns true to the message match: with any of
 the objects in the given collection; returns false otherwise.  Examples:

   'xyz' matchesAnyOf: #('xyz' 'abc*')
     true
   'xyz' matchesAnyOf: #('ayz' 'abc')
     false
   'x#z' matchesAnyOf: #('x@z' '*')
     false

 The deprecated class JISString does not support this method."

aCollectionOfCharacterColls do: [:coll |
  (self match: coll) ifTrue: [ ^true ]
].
^false
%

category: 'Comparing'
method: CharacterCollection
matchPattern: aPattern

"Returns true if the receiver matches aPattern, false if it doesn't.  An exact
 match is required.  For partial matching, use the 'Searching' method
 findPattern:startingAt: instead.

 The argument aPattern is a kind of Array containing zero or more
 CharacterCollections, plus zero or more occurrences of the special Characters
 $* or $?.  If either $* or $? occurs in aPattern, it acts as a wild card.
 The Character $? matches any single Character in the receiver, and $* matches
 any sequence of zero or more Characters in the receiver.  For example,

 'weimaraner' matchPattern: #('w' $* 'r')

 returns true, because the Character $* is interpreted as a wild card.

 If either of these special Characters occurs in the receiver, it is
 interpreted literally.  For example,

 'w*r' matchPattern: #('weimaraner')

 returns false - because the Character $* occurs in the receiver, it is
 interpreted as a literal asterisk (not as a wild card)."

| match         "indicates if the current pattern matched"
  pattern       "the Array of pattern elements"
  selfSize      "the size of the receiver"
  patternSize   "the number of elements in the pattern Array"
  startIndexArr "an Array of indexes into the receiver; this Array is
                 parallel to the pattern Array (each element in this Array
                 corresponds to the starting index for each pattern element)"
  index         "index into the pattern Array and the startIndexArr Array"
  thisPattern   "the current element of the pattern to match"
  selfIndex     "an index into the receiver"
|

 "The pattern Array must be processed so that there are no *? pairs in it.
 They must all be converted to ?* pairs for the algorithm to work correctly."
 pattern:= Array withAll: aPattern.
 patternSize:= pattern size.
 index := 1.
 [ index < patternSize ]
 whileTrue:
 [ (((pattern at: index) isEquivalent: $* )
    and:[ (pattern at: index+1) isEquivalent: $? ])
   ifTrue:[
     pattern at: index put: $?.
     pattern at: index+1 put: $*.
     index := 1 max: index-1.
   ]
   ifFalse:
   [ index := index + 1 ].
 ].

 "initialize"
 selfSize := self size.
 startIndexArr:= Array new: (patternSize + 1).
                 "last element is set, but not used"
 index := 1.
 startIndexArr at: 1 put: 1.

 "if no patterns to match, exit early"
 (patternSize == 0)
 ifTrue:
    [^ selfSize == 0 ].

 [index <= patternSize]
 whileTrue:
 [ thisPattern := pattern at: index.
   selfIndex := startIndexArr at: index.
   match := true.

   (thisPattern isKindOf: CharacterCollection) "pattern to match is a string"
   ifTrue:
   [ (selfIndex + thisPattern size - 1) > selfSize
     ifTrue: "this pattern too big to match rest of receiver"
        [^ false ].

     index = patternSize "processing the final pattern"
     ifTrue:
     [ ((index > 1) and: [ (pattern at: index - 1) isEquivalent: $* ])
       ifTrue: "is not the first pattern _and_ previous pattern was a $*"
       [ ((selfSize == 0) and: [thisPattern size == 0])
         ifTrue: [^ true].
         ^(self findString: thisPattern
                startingAt: selfSize - thisPattern size + 1) ~~ 0
         "find the pattern far enough back in the string so that only
         the final chars match"
       ]
       ifFalse: "processing first pattern _or_ previous pattern was not $*"
       [ (match:= (selfIndex + thisPattern size - 1 = selfSize))
         ifTrue: "exactly enough chars in self to match thisPattern"
         [ (selfSize == 0)
           ifTrue: [match:= (thisPattern size == 0)]
           ifFalse: [match:= self at: selfIndex equals: thisPattern ].
         ]
       ].
     ]
     ifFalse: "not processing the final pattern"
     [ ((index > 1) and: [ (pattern at: index - 1) isEquivalent: $* ])
       ifTrue: "not first pattern _and_ previous pattern was $*"
       [ (((selfSize == 0) and: [thisPattern size == 0]) or:
         [(selfIndex:= self findString: thisPattern
                           startingAt: selfIndex) ~~ 0])
         ifTrue: "thisPattern was found"
         [ startIndexArr at: index put: selfIndex.
           startIndexArr at: index + 1 put: selfIndex + thisPattern size.
         ]
         ifFalse: "thisPattern was not found"
            [^ false ]
       ]
       ifFalse: "first pattern _or_ previous pattern was not $*"
       [ (((selfSize == 0) and: [thisPattern size == 0]) or:
         [(self at: selfIndex equals: thisPattern)])
         ifTrue:
            [startIndexArr at: index + 1 put: selfIndex+ thisPattern size]
         ifFalse:
            [match := false ].
       ].
     ]
   ]
   ifFalse: "thisPattern is not a string"
   [ (thisPattern isEquivalent: $*)
     ifTrue:
        [startIndexArr at: (index + 1) put: selfIndex]
     ifFalse:
     [ (thisPattern isEquivalent: $?)
       ifTrue:
       [ selfIndex > selfSize
         ifTrue: "no char to match; already at end of self"
            [^ false ].
         startIndexArr at: (index + 1) put: (selfIndex + 1).
         index = patternSize "processing the last pattern"
         ifTrue:
            [match := selfIndex = selfSize.].
       ]
       ifFalse: "next pattern is neither a $* or $?"
       [ ^ aPattern _error: #rtErrBadPattern].
     ].
   ].  "end ifTrue:ifFalse"

   match
   ifTrue:
      [index := index + 1 ] "advance to the next term in the pattern"
   ifFalse:
   [ "If there is a preceding $* term in the pattern, backup to the
      term following it, and advance position in the string by 1."
      [ index := index - 1.
        index < 2 ifTrue:
           [^ false ].
        (pattern at: index - 1) isEquivalent: $*
      ] untilTrue.
      startIndexArr at: index put: ((startIndexArr at: index) + 1).
   ].
].  "end whileTrue:"

^ true
%

category: 'Comparing'
method: CharacterCollection
max: another

"If the receiver is greater than the argument, return the receiver.
 Otherwise return the argument."

	^self > another
		ifTrue: [self]
		ifFalse: [another].
%

category: 'Searching'
method: CharacterCollection
maxConsecutiveSubstring

"Returns the largest substring within the receiver that contains Characters with
 consecutive ASCII values.  For example, the message

   'abxabcdxabc' maxConsecutiveSubstring

 yields the result 'abcd'.

 If there are no such substrings larger than 2 Characters, returns a String that
 contains the first Character in the receiver."

| mySize bigSize bigStart aStart aSize lastVal thisVal |

mySize := self size .
mySize < 2 ifTrue:[ ^ self ].
bigSize := 1 .
bigStart := 1 .
aStart := 1 .
aSize := 1 .
lastVal := (self at: 1) codePoint  .
2 to: mySize do:[:j |
  lastVal := lastVal + 1 .
  thisVal := (self at: j) codePoint .
  thisVal == lastVal
    ifTrue:[ aSize := aSize + 1 ]
    ifFalse:[
      aSize > bigSize ifTrue:[
	bigStart := aStart .
	bigSize := aSize .
	].
      aSize := 1 .
      lastVal := thisVal .
      aStart := j .
      ].
   ].
aSize > bigSize ifTrue:[
  bigStart := aStart.
  bigSize := aSize.
  ].
^ self copyFrom: bigStart to: (bigStart + bigSize - 1) .
%

category: 'Searching'
method: CharacterCollection
maxRepeatingSubstring

"Returns the largest substring within the receiver that contains repetitions of
 a Character, using case-sensitive comparison.  For example, the message

   'aaxbbbBxccc' maxRepeatingSubstring

 yields the result 'bbb'.

 If there are no such substrings larger than 1 Character, returns a String that
 contains the first Character in the receiver."

| mySize bigSize bigStart aStart aSize lastChar |

mySize := self size .
mySize < 2 ifTrue:[ ^ self ].
bigSize := 1 .
bigStart := 1 .
aStart := 1 .
aSize := 1 .
lastChar := self at: 1 .
2 to: mySize do:[:j | | thisChar |
  thisChar := self at: j .
  thisChar == lastChar
    ifTrue:[ aSize := aSize + 1 ]
    ifFalse:[
      aSize > bigSize ifTrue:[
	bigStart := aStart .
	bigSize := aSize .
	].
      aSize := 1 .
      lastChar := thisChar .
      aStart := j .
      ].
   ].
aSize > bigSize ifTrue:[
  bigStart := aStart.
  bigSize := aSize.
  ].
^ self copyFrom: bigStart to: (bigStart + bigSize - 1) .
%

category: 'Searching'
method: CharacterCollection
maxSameTypeSubstring

"Returns the largest substring within the receiver that contains either all
 digits, all alphabetic characters, or all special characters.  For example, the
 message

   'axv2435,.-' maxSameTypeSubstring

 yields the result '2435'.

 If there are no such substrings larger than 1 Character, returns a String that
 contains the first Character in the receiver.

 This method may generate an error if the receiver is a JapaneseString."

| mySize bigSize bigStart aStart aSize lastType |

mySize := self size .
mySize < 2 ifTrue:[ ^ self ].
bigSize := 1 .
bigStart := 1 .
aStart := 1 .
aSize := 1 .
lastType := (self at: 1) _type  .
2 to: mySize do:[:j | | thisType |
  thisType := (self at: j) _type .
  thisType == lastType
    ifTrue:[ aSize := aSize + 1 ]
    ifFalse:[
      aSize > bigSize ifTrue:[
	bigStart := aStart .
	bigSize := aSize .
	].
      aSize := 1 .
      lastType := thisType .
      aStart := j .
      ].
   ].
aSize > bigSize ifTrue:[
  bigStart := aStart.
  bigSize := aSize.
  ].
^ self copyFrom: bigStart to: (bigStart + bigSize - 1) .
%

category: 'Message Digests'
method: CharacterCollection
md5sum

"Return the 128 bit MD5 checksum of the receiver as a LargeInteger.

 Computation is per RFC 1321 , http://www.ietf.org/rfc/rfc1321.txt,
 using L. Peter Deutsch's C implementation from
 http://sourceforge.net/projects/libmd5-rfc/

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 1
%

category: 'Message Digests'
method: CharacterCollection
md5sumBytes

"Return the 128 bit MD5 checksum of the receiver as a ByteArray.

 Computation is per RFC 1321 , http://www.ietf.org/rfc/rfc1321.txt,
 using L. Peter Deutsch's C implementation from
 http://sourceforge.net/projects/libmd5-rfc/

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -1
%

category: 'Comparing'
method: CharacterCollection
min: another

"If the receiver is less than the argument, return the receiver.
 Otherwise return the argument."

	^self < another
		ifTrue: [self]
		ifFalse: [another].
%

category: 'Accessing'
method: CharacterCollection
numArgs

"Returns the number of arguments the receiver would take, were the receiver
 a message selector."

| idx count sz |
(sz := self size) == 0 ifTrue:[ ^ 0 ].
(self at: sz) == $: ifTrue:[
  count := 1 .
  idx := 0 .
  [ idx := self indexOf: $: startingAt: idx + 1 .
    idx ~~ sz
  ] whileTrue:[
    count := count + 1
  ].
  ^ count
] ifFalse:[
  self isInfix ifTrue: [ ^ 1 ].
  ^ 0
]
%

category: 'Json'
method: CharacterCollection
printJsonOn: aStream

	aStream nextPut: $".
	1 to: self size do: [:j |
		| codePoint char |
		codePoint := (char := self at: j) codePoint.
    (32 <= codePoint and: [codePoint <= 127]) ifTrue: [
      char == $" "34" ifTrue: [aStream nextPutAll: '\"' ] ifFalse: [
      char == $\ "92" ifTrue: [aStream nextPutAll: '\\' ] ifFalse: [
        aStream nextPut: char
      ]]
    ] ifFalse:[
      (codePoint <= 13 and:[ codePoint >= 8]) ifTrue:[
		    codePoint == 8 	ifTrue: [aStream nextPutAll: '\b'	. codePoint := nil. ] ifFalse: [
		    codePoint == 9 	ifTrue: [aStream nextPutAll: '\t'	. codePoint := nil. ] ifFalse: [
		    codePoint == 10 	ifTrue: [aStream nextPutAll: '\n'. codePoint := nil. 	] ifFalse: [
		    codePoint == 12 	ifTrue: [aStream nextPutAll: '\f'. codePoint := nil. 	] ifFalse: [
		    codePoint == 13 	ifTrue: [aStream nextPutAll: '\r'. codePoint := nil. 	]
      ]]]]].
      codePoint ifNotNil:[ | hex |
        hex := '0123456789ABCDEF'.
			  aStream nextPutAll: '\u' ;
             nextPut: (hex at: 1 + ((codePoint bitShift: -12) bitAnd: 16rF)) ;
             nextPut: (hex at: 1 + ((codePoint bitShift: -8) bitAnd: 16rF)) ;
             nextPut: (hex at: 1 + ((codePoint bitShift: -4) bitAnd: 16rF)) ;
             nextPut: (hex at: 1 + (codePoint bitAnd: 16rF)) .
      ]
	  ].
  ].
	aStream nextPut: $".
%

category: 'Formatting'
method: CharacterCollection
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

"Should be reimplemented for more efficiency in subclasses."

aStream nextPutAll: self quoted .
%

category: 'Formatting'
method: CharacterCollection
printOn: aStream recursionSet: anIdentitySet
	"Put a displayable representation of the receiver on the given stream
	 since CharacterCollections cannot have recursive references."

	self printOn: aStream
%

category: 'Formatting'
method: CharacterCollection
printString

"Returns a CharacterCollection whose contents are a displayable representation of the
 receiver."

^ self printStringWithMaxSize: 100000
%

category: 'Formatting'
method: CharacterCollection
printStringWithMaxSize: n

"Returns a CharacterCollection whose contents are a displayable representation of the
 receiver, limited to a specified number of characters <n>.

 If the number of characters in the displayable representation  exceeds <n>,
 display only the first <n>-1 characters and then display '. . .'. "

| ws |

ws := PrintStream printingOn: self speciesForPrint new maxSize: n.
self printOn: ws.
^ws contents
%

category: 'Formatting'
method: CharacterCollection
quoted

"Returns a copy of the receiver enclosed in single-quote marks, with contained
 single-quote Characters doubled.  The copy is of the same class as the
 receiver."

| sz result targetIdx lastIdx idx |
sz := self size.
result := self class _newString: sz + 2.
result at: 1 put: $'.
targetIdx := 2.
lastIdx := 1.
[ (idx := self indexOf: $' startingAt: lastIdx) == 0 ] whileFalse: [
 "self copyFrom: lastIdx to: idx into: result startingAt: targetIdx . "
  result replaceFrom: targetIdx to: targetIdx + idx - lastIdx
	 with: self startingAt: lastIdx .

  targetIdx := targetIdx + (idx - lastIdx) + 2.
  result at: targetIdx - 1 put: $'.
  lastIdx := idx + 1
].
lastIdx <= sz ifTrue: [
 "self copyFrom: lastIdx to: sz into: result startingAt: targetIdx "
  result replaceFrom: targetIdx to: targetIdx + sz - lastIdx
         with: self startingAt: lastIdx .
].
result at: targetIdx + (sz - lastIdx) + 1 put: $'.
^result
%

category: 'Comparing'
method: CharacterCollection
sameAs: aCharCollection

"Returns true if the receiver is equivalent to aCharCollection.  The receiver
 is equivalent to aCharCollection if the receiver contains the same Characters
 as aCharCollection regardless of case or internal representation.  For
 example, if $a is in aCharCollection, it is equivalent to any representation
 of an 'a' in the receiver's character set.
 From ANSI.  Used in Seaside. "

(aCharCollection isKindOf: CharacterCollection) ifFalse: [ ^false ].

self size ~~ aCharCollection size ifTrue: [ ^false ].

^ self isEquivalent: aCharCollection
%

category: 'Message Digests'
method: CharacterCollection
sha1Sum

"Return the 160 bit SHA1 checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 2
%

category: 'Message Digests'
method: CharacterCollection
sha1SumBytes

"Return the 160 bit SHA1 checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -2
%

category: 'Message Digests'
method: CharacterCollection
sha256Sum

"Return the 256 bit SHA256 checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 3
%

category: 'Message Digests'
method: CharacterCollection
sha256SumBytes

"Return the 256 bit SHA256 checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -3
%

category: 'Message Digests'
method: CharacterCollection
sha3_224Sum

"Return the SHA3 224 bit checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 5
%

category: 'Message Digests'
method: CharacterCollection
sha3_224SumBytes

"Return the SHA3 224 bit checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -5
%

category: 'Message Digests'
method: CharacterCollection
sha3_256Sum

"Return the SHA3 256 bit checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 6
%

category: 'Message Digests'
method: CharacterCollection
sha3_256SumBytes

"Return the SHA3 256 bit checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -6
%

category: 'Message Digests'
method: CharacterCollection
sha3_384Sum

"Return the SHA3 384 bit checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 7
%

category: 'Message Digests'
method: CharacterCollection
sha3_384SumBytes

"Return the SHA3 384 bit checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -7
%

category: 'Message Digests'
method: CharacterCollection
sha3_512Sum

"Return the SHA3 512 bit checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 8
%

category: 'Message Digests'
method: CharacterCollection
sha3_512SumBytes

"Return the SHA3 512 bit checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -8
%

category: 'Message Digests'
method: CharacterCollection
sha512Sum

"Return the 512 bit SHA512 checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 4
%

category: 'Message Digests'
method: CharacterCollection
sha512SumBytes

"Return the 512 bit SHA512 checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -4
%

category: 'Digital Signature Creation - EC'
method: CharacterCollection
signWithEcPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Signs the receiver with the given elliptic curve private key. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #EC ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 0 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - DSA'
method: CharacterCollection
signWithSha1AndDsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA1 and signs the resulting hash with the given
 DSA private key. Returns a ByteArray containing the resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #DSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 2 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha1AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA1 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the
 resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 2 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha1AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA1 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -2 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - DSA'
method: CharacterCollection
signWithSha256AndDsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA2 256 and signs the resulting hash with the given
 DSA private key. Returns a ByteArray containing the resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #DSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 3 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha256AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA 256 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 3 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha256AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA2 256 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -3 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha3_224AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 224 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 5 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha3_224AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 224 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -5 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha3_256AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 256 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 6 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha3_256AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 256 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -6 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha3_384AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 384 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 7 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha3_384AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 384 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -7 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha3_512AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 512 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 8 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha3_512AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA3 512 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -8 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - DSA'
method: CharacterCollection
signWithSha512AndDsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA2 512 and signs the resulting hash with the given
 DSA private key. Returns a ByteArray containing the resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #DSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 4 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha512AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA2 512 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 4 signature: aByteArrayOrNil
%

category: 'Digital Signature Creation - RSA'
method: CharacterCollection
signWithSha512AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil
"Hashes the receiver using SHA2 512 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -4 signature: aByteArrayOrNil
%

category: 'Updating'
method: CharacterCollection
space

"Appends a space to the receiver and returns the receiver."

self add: $ .
%

category: 'Private'
method: CharacterCollection
speciesForConversion

"Return the class of the receiver.  Subclasses should reimplement this method."

^ self class.
%

category: 'Converting'
method: CharacterCollection
subStrings

"Returns an Array of CharacterCollections where element represents a word in
 the receiver.  A word is a group of Characters separated by one or more
 separators."

^ self asArrayOfSubstrings.
%

category: 'Converting'
method: CharacterCollection
subStrings: separators
	"Returns an Array of CharacterCollections in which each element represents a
	 substring separated by any of the Characters in separators.  For compatibility with
	 existing code, separators may be a single Character or a collection of Characters.
	 The result will include empty substrings when two adjacent separators exist, as well as if
	 a separator is the first or last element."

	^separators size == 0
		ifTrue: [self subStringsDelimitedBy: separators]
		ifFalse: [self subStringsDelimitedByAny: separators].
%

category: 'Converting'
method: CharacterCollection
subStringsDelimitedBy: aCharacter
	"Returns an Array of CharacterCollections in which each element represents a
	 substring separated by aCharacter.  The result will include empty substrings when
	 two adjacent separators exist, as well as if a separator is the first or last element."

	| result startIndex endIndex sz |
	result := {}.
	startIndex := 1.

	[endIndex := self indexOf: aCharacter startingAt: startIndex.
	endIndex == 0]
			whileFalse:
				[endIndex == startIndex ifTrue: [result add: self class _newString].
				endIndex > startIndex
					ifTrue: [result add: (self copyFrom: startIndex to: endIndex - 1)].
				startIndex := endIndex + 1].
	startIndex > (sz := self size)
		ifTrue: [result add: self class _newString]
		ifFalse: [result add: (self copyFrom: startIndex to: sz)].
	^result
%

category: 'Converting'
method: CharacterCollection
subStringsDelimitedByAny: separators
	"Returns an Array of CharacterCollections in which each element represents a
	 substring separated by any of the Characters in separators.
	 The result will include empty substrings when two adjacent separators exist,
	 as well as if a separator is the first or last element."

	| nextName result |
	result := {}.
	nextName := self speciesForConversion new.
	self do:
			[:c |
			(separators includes: c)
				ifTrue:
					[result add: nextName.
					nextName := self speciesForConversion new]
				ifFalse: [nextName add: c]].
	result add: nextName.
	^result
%

category: 'Deprecated'
method: CharacterCollection
toServerTextFile: aFileSpec

self deprecated: 'CharacterCollection >> toServerTextFile: deprecated long before v3.0. Use an instance of GsFile
 to access the file system.'.
self _toServerTextFile: aFileSpec
%

category: 'Converting'
method: CharacterCollection
trimBlanks

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with leading and trailing blanks removed."

self size == 0 ifTrue: [ ^ self ].
^ (self trimLeadingBlanks) trimTrailingBlanks.
%

category: 'Converting'
method: CharacterCollection
trimLeadingBlanks

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with leading blanks removed."

| idx blank sz |
(sz := self size) == 0 ifTrue: [ ^ self ].
blank := Character space.
((self at: 1) == blank) ifFalse: [ ^ self ].
idx := 2.
[ true ] whileTrue:[
  idx <= sz ifFalse:[ ^ '' ].
  (self at: idx) == blank ifFalse:[ ^ self copyFrom: idx to: sz].
  idx := idx + 1
]
%

category: 'Converting'
method: CharacterCollection
trimLeadingSeparators

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with leading separators removed."

| idx sz |
(sz := self size) == 0 ifTrue:[ ^ self ].
(self at: 1) isSeparator ifFalse: [ ^ self ].
idx := 2.
[ true ] whileTrue:[
  idx <= sz ifFalse:[ ^ '' ].
  (self at: idx) isSeparator ifFalse:[ ^ self copyFrom: idx to: sz].
  idx := idx + 1
]
%

category: 'Converting'
method: CharacterCollection
trimSeparators

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with leading and trailing separators removed."

self size == 0 ifTrue: [ ^ self ].
^ (self trimLeadingSeparators) trimTrailingSeparators.
%

category: 'Converting'
method: CharacterCollection
trimTrailingBlanks

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with trailing blanks removed."

| idx blank sz |

(sz := self size) == 0 ifTrue: [ ^ self ].
blank := Character space.
(self at: sz ) == blank ifFalse: [ ^ self ].
idx := sz  - 1.
[ true ] whileTrue:[
  idx < 1 ifTrue:[ ^ '' ] .
  (self at: idx) == blank ifFalse:[ ^ self copyFrom: 1 to: idx ].
  idx := idx - 1
]
%

category: 'Converting'
method: CharacterCollection
trimTrailingSeparators

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with trailing separators removed."

| idx sz |

(sz := self size) == 0 ifTrue: [ ^ self ].
(self at: sz ) isSeparator ifFalse: [ ^ self ].
idx := sz  - 1.
[ true ] whileTrue:[
  idx < 1 ifTrue:[ ^ '' ] .
  (self at: idx) isSeparator ifFalse:[ ^ self copyFrom: 1 to: idx ].
  idx := idx - 1
]
%

category: 'Formatting'
method: CharacterCollection
trimWhiteSpace

"Returns a copy of the receiver with leading and trailing white space removed."

| first limit selfSize |

((selfSize := self size) == 0) ifTrue: [
   ^ self class _newString
].

limit := selfSize + 1.

first := 1 .
(self at: 1) isSeparator ifTrue: [ | j |
  first := nil .
  j := 2.
  [ j == limit ] whileFalse: [
      (self at: j) isSeparator ifTrue: [
         j := j + 1.
      ] ifFalse:[
         first := j.
         j := limit .
       ].
  ].
  first ifNil: [ ^ self class _newString ].
].

(self at: selfSize) isSeparator ifTrue: [ | k |
  k := selfSize - 1.
  [ k == 0 ] whileFalse: [
     (self at: k) isSeparator ifFalse:[
       ^ self copyFrom: first to: k
     ].
     k := k - 1.
  ].
  ^ self class _newString
].

first == 1 ifTrue:[  ^ self copy ].
^ self copyFrom: first to: selfSize .
%

category: 'Digital Signature Creation - EC'
method: CharacterCollection
verifyWithEcPublicKey: aGsTlsPublicKey signature: aByteArray
"Verifies the receiver using the given elliptic curve key and signature.  Returns true
 if the signature is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #EC ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 0 signature: aByteArray
%

category: 'Digital Signature Verification - DSA'
method: CharacterCollection
verifyWithSha1AndDsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA1 and verifies the resulting hash using the
 given DSA public key and signature. Returns true if the signature is correct.
 Otherwise raises an exception.

 See the method:
  # _primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #DSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 2 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha1AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA1 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -2 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha1AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA1 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 2 signature: aByteArray
%

category: 'Digital Signature Verification - DSA'
method: CharacterCollection
verifyWithSha256AndDsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA256 and verifies the resulting hash using the
 given DSA public key and signature. Returns true if the signature is correct.
 Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #DSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 3 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha256AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA2 256 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -3 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha256AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA2 256 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 3 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha3_224AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 224 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -5 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha3_224AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 224 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 5 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha3_256AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 256 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -6 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha3_256AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 256 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 6 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha3_384AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 384 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -7 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha3_384AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 384 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 7 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha3_512AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 512 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -8 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha3_512AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA3 512 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 8 signature: aByteArray
%

category: 'Digital Signature Verification - DSA'
method: CharacterCollection
verifyWithSha512AndDsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA512 and verifies the resulting hash using the
 given DSA public key and signature. Returns true if the signature is correct.
 Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #DSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 4 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha512AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA2 512 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -4 signature: aByteArray
%

category: 'Digital Signature Verification - RSA'
method: CharacterCollection
verifyWithSha512AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray
"Hashes the receiver using SHA2 512 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 4 signature: aByteArray
%

category: 'Formatting'
method: CharacterCollection
width: anInteger

"Pads the receiver with spaces to create an object of size anInteger.
 If anInteger is positive, the spaces are added to the right of the receiver.
 If anInteger is negative, the spaces are added to the left of the receiver.
 If the size of the receiver is already greater than anInteger, the receiver
 is left unchanged."

| onRight s change |

change := anInteger abs - self size.
change > 0
ifTrue:
  [ onRight := anInteger > 0.
    s := (self speciesForConversion) new: change.
    s atAllPut: $ .
    onRight
    ifTrue:
      [ self addAll: s ]
    ifFalse:
      [ self insertAll: s at: 1 ]
  ]
%

category: 'Copying'
method: CharacterCollection
withCRs

"Supplied for Smalltalk-80 compatibility.  This is equivalent to withLFs."

^self withLFs
%

category: 'Formatting'
method: CharacterCollection
wrapTo: col

"Word-wrap the receiver to column col, treating tab Characters as modulo-8."

^ self _wrapTo: col indentingWith: ''
%

category: 'Message Digests'
method: CharacterCollection
_asMessageDigestKind: opCode

"
opCode   Digest Algorithm   Digest Bits
=======================================
  1          md5              128
  2          sha1             160
  3          sha2-256         256
  4          sha2-512         512
  5          sha3-224         224
  6          sha3-256         256
  7          sha3-384         384
  8          sha3-256         512
=======================================

Postive opCode means return result as a LargeInteger.
Negative opCode means return result as a ByteArray.

"
<primitive: 666>
opCode _validateClass: SmallInteger .
(opCode == 0 or:[(opCode < -8) or:[ opCode > 8]])
  ifTrue:[ opCode _error: #rtErrArgOutOfRange args:{ -8 . 8 } ].
^ self _primitiveFailed: #_asMessageDigestKind:
%

category: 'Private'
method: CharacterCollection
_asUtf8WithEol
  "Return a Utf8 copy of the receiver, with a linefeed
   appended if receiver does not end with a linefeed.  Used by topaz"
  | lastCp sz str |
  sz := self size .
  sz ~~ 0 ifTrue:[ lastCp := self codePointAt: sz ].
  str := self .
  lastCp == 10 ifFalse:[
    (str := self copy) codePointAt: sz + 1 put: 10
  ].
  ^ str encodeAsUTF8
%

category: 'Accessing'
method: CharacterCollection
_at: anIndex

"Private.  Reimplemented to return a kind of Character."

^ self at: anIndex
%

category: 'Private'
method: CharacterCollection
_charCollCompare: aCharCollection

"Returns -1 if self < aCharCollection, returns 0 if self = aCharCollection,
 and returns 1 if self > aCharCollection."

| selfSize argSize selfElement argElement |

aCharCollection _validateClass: CharacterCollection.

selfSize:= self size.
argSize:= aCharCollection size.

1 to: (selfSize min: argSize) do: [:i |

  selfElement := self at: i.
  argElement  := aCharCollection at: i.

  (selfElement < argElement) ifTrue: [ ^ -1 ].
  (selfElement > argElement) ifTrue: [ ^ 1 ]
  ].

"All elements of self and argument are equal."

(selfSize < argSize) ifTrue: [ ^ -1 ].
(selfSize > argSize) ifTrue: [ ^ 1 ].

^ 0 "equal"
%

category: 'New Indexing Comparison'
method: CharacterCollection
_classSortOrdinal

^ 10
%

category: 'Comparing'
method: CharacterCollection
_coerceToUnicode
  ^ self asUnicodeString
%

category: 'Private'
method: CharacterCollection
_deepCopyWith: copiedObjDict

| copy |

copy := copiedObjDict at: self otherwise: nil.
copy ifNotNil:[ ^ copy ].

^ self copy.
%

category: 'Compatibility'
method: CharacterCollection
_encodeAsUTF8intoString

^ self encodeAsUTF8IntoString
%

category: 'Private'
method: CharacterCollection
_findLastString: subString startingAt: startIndex ignoreCase: aBoolean

"If a receiver contains subString beginning at some point at or before
 startIndex, this returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0.
 Search is backwards. "

 self  subclassResponsibility: #_findLastString:startingAt:ignoreCase: .

 self _uncontinuableError
%

category: 'Searching'
method: CharacterCollection
_findPattern: aPattern startingAt: anIndex ignoreCase: caseInsens

"This method searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, this method returns the
 index of the first Character of the substring.  Otherwise, this returns 0.

 The argument aPattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 If caseInsens is true, a case-insensitive search is performed.  Otherwise,
 caseInsens should be false and a case-sensitive search is performed."

| i             "loop counter"
  pattern       "the argument aPattern, converted to an Array"
  selfSize      "the size of the receiver"
  patternSize   "the number of elements in the pattern Array"
  startIndex    "an Array that corresponds to the pattern Array, with each
                 element containing the starting index into the receiver of
                 the corresponding pattern"
  index         "the index into both the pattern and the startIndex Arrays"
  next          "the current pattern (element of the pattern Array) to
                 be matched"
  cursor        "the index into the receiver"
|

aPattern _validateClass: Array.
(anIndex <= 0) ifTrue: [ ^ self _errorIndexOutOfRange: anIndex ].

pattern := Array withAll: aPattern.
patternSize := pattern size.

" First, the pattern Array must be processed so that there are no *? pairs
  it in.  They must all be converted to ?* pairs for the algorithm to work
  correctly."
i := 1.
[ i < patternSize ]
whileTrue:
  [ ( ((pattern at: i) isEquivalent: $*) and:[(pattern at: i+1) isEquivalent: $?])
    ifTrue:
      [ pattern at: i put: $?.
        pattern at: i+1 put: $*.
        i := 1 max: i-1.
      ]
    ifFalse:
      [ i := i + 1 ].
  ].

"initialize"
selfSize := self size.
startIndex := Array new: (patternSize + 1)."last element is set, but not used"
index := 1.
anIndex > selfSize
   ifTrue: [startIndex at: 1 put: (selfSize + 1)] "Fix for bug 15038"
   ifFalse: [startIndex at: 1 put: anIndex].

(patternSize == 0) "no pattern to match"
ifTrue:
  [ startIndex <= selfSize
    ifTrue:
      [ ^ startIndex ]
    ifFalse:
      [ ^ 0 ].
  ].

[index <= patternSize]
whileTrue:
  [ next := pattern at: index.
    cursor := startIndex at: index.

    (next isKindOf: CharacterCollection) "pattern element is a string"
    ifTrue:
      [ ((cursor > selfSize) or:
        [(cursor + next size - 1) > selfSize]) "pattern element too big to "
        ifTrue: "match beginning at cursor location"
          [ ^ 0 ].
        ((index == 1) or: [ (pattern at: index - 1) isEquivalent: $* ])
        ifTrue: "is first pattern or end of *; can skip chars to find match"
          [ cursor := self _findString: next startingAt: cursor
				ignoreCase: caseInsens .
            (cursor == 0)
            ifTrue:
              [ ^ 0 ]
            ifFalse:
              [ startIndex at: index put: cursor.
                startIndex at: index + 1 put: cursor + next size.
              ]
          ]
        ifFalse: "can't skip chars to find match"
          [ (self _at: cursor equals: next ignoreCase: caseInsens )
            ifTrue:
              [ startIndex at: index + 1 put: cursor + next size ]
            ifFalse:
              [
                [ (index := index - 1) < 1
                  ifTrue:
                    [ ^ 0 ].
                  ((pattern at: index) isKindOf: CharacterCollection) or:
                  [ (index == 1) or: [ (pattern at: index - 1) isEquivalent: $* ] ]
                ]
                untilTrue.
                startIndex at: index put: ((startIndex at: index) + 1).
                index := index - 1.
              ].
          ].
      ]
    ifFalse: "pattern element not a string"
      [ (next isEquivalent: $*) "pattern element is *"
        ifTrue:
          [ startIndex at: (index + 1) put: cursor
          ]
        ifFalse:
          [ (next isEquivalent: $?)  "pattern element is ?"
            ifTrue:
              [ cursor > selfSize
                ifTrue:
                  [ ^ 0 ].
                startIndex at: (index + 1) put: (cursor + 1)
              ]
            ifFalse: "found a pattern element other than ?, * or string"
              [ ^ aPattern _error: #rtErrBadPattern
              ].
          ].
      ].

    index := index + 1
  ].

^ startIndex at: 1
%

category: 'Private'
method: CharacterCollection
_findString: subString startingAt: startIndex ignoreCase: aBoolean

CharacterCollection
  subclassResponsibility: #_findString:startingAt:ignoreCase: .

self _uncontinuableError
%

category: 'Instance Creation'
method: CharacterCollection
_fromServerTextFile: aFileSpec

"Import the contents of aFileSpec into the receiver.
 For multi-byte characters, assumes the byte order of the file
 matches the current session's cpu's in-memory byte order.
"
<primitive: 302>

aFileSpec _validateByteClass: CharacterCollection .
aFileSpec _error: #hostErrFileImport args: #()
%

category: 'Indexing Support'
method: CharacterCollection
_idxBasicCanCompareWithClass: aClass
  "Returns true if the receiver may be inserted into a basic BtreeNode whose
   #lastElementClass is <aClass> (see RangeEqualityIndex class>>isBasicClass:)."

  "If using Unicode compares, then *String and Unicode* instances may be compared.
   If not using Unicode compares then *String and Unicode* instances may not be compared."

  ^ aClass _idxBasicCanCompareWithCharacterCollectionInstance: self
%

category: 'New Indexing Comparison - for Compare'
method: CharacterCollection
_idxForCompareCharacterCollectionGreaterThanOrEqualToSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForCompareGreaterThanOrEqualTo:. Note that aCharacterCollection should be the receiver in any >= comparison"

^(aCharacterCollection _idxPrimCompareLessThan: self) not
%

category: 'New Indexing Comparison - for Compare'
method: CharacterCollection
_idxForCompareCharacterCollectionGreaterThanSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForCompareGreaterThan:. Note that aCharacterCollection should be the receiver in any > comparison"

^aCharacterCollection _idxPrimCompareGreaterThan: self
%

category: 'New Indexing Comparison - for Compare'
method: CharacterCollection
_idxForCompareCharacterCollectionLessThanOrEqualToSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForCompareLessThanOrEqualTo:. Note that aCharacterCollection should be the receiver in any <= comparison"

^(aCharacterCollection _idxPrimCompareGreaterThan: self) not
%

category: 'New Indexing Comparison - for Compare'
method: CharacterCollection
_idxForCompareCharacterCollectionLessThanSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForCompareLessThan:. Note that aCharacterCollection should be the receiver in any < comparison"

^aCharacterCollection _idxPrimCompareLessThan: self
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForCompareEqualTo: arg
  ""

  ^ arg _idxForCompareEqualToCharacterCollection: self
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForCompareEqualTo: aCharacterCollection collator: anIcuCollator
  "treat receiver and aCharacterCollection as Unicode, since anIcuCollator is supplied"

  ^ aCharacterCollection
    _idxForCompareEqualToUnicode: self
    collator: anIcuCollator
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForCompareGreaterThan: arg

""

^arg _idxForCompareCharacterCollectionGreaterThanSelf: self
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForCompareGreaterThan: aCharacterCollection collator: anIcuCollator
  "treat receiver and aCharacterCollection as Unicode, since anIcuCollator is supplied"

  ^ aCharacterCollection
    _idxForCompareGreaterThanUnicode: self
    collator: anIcuCollator
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForCompareGreaterThanOrEqualTo: arg

""

^arg _idxForCompareCharacterCollectionGreaterThanOrEqualToSelf: self
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForCompareGreaterThanOrEqualTo: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForCompareGreaterThanOrEqualToUnicode: self
    collator: anIcuCollator
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForCompareLessThan: arg

""

^arg _idxForCompareCharacterCollectionLessThanSelf: self
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForCompareLessThan: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForCompareLessThanUnicode: self
    collator: anIcuCollator
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForCompareLessThanOrEqualTo: arg

""

^arg _idxForCompareCharacterCollectionLessThanOrEqualToSelf: self
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForCompareLessThanOrEqualTo: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForCompareLessThanOrEqualToUnicode: self
    collator: anIcuCollator
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForCompareNotEqualTo: arg

""

^ (self _idxForCompareEqualTo: arg) not
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForCompareNotEqualTo: aCharacterCollection collator: anIcuCollator
  "REL_SIG_STRING_SIZE = 900"

  ^ (self _idxForCompareEqualTo: aCharacterCollection collator: anIcuCollator)
    not
%

category: 'New Indexing Comparison - for Sort'
method: CharacterCollection
_idxForSortCharacterCollectionGreaterThanOrEqualToSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForSortGreaterThanOrEqualTo:. Note that aCharacterCollection should be the receiver in any >= comparison"

^(aCharacterCollection _idxPrimCompareLessThan: self) not
%

category: 'New Indexing Comparison - for Sort'
method: CharacterCollection
_idxForSortCharacterCollectionGreaterThanSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForSortGreaterThan:. Note that aCharacterCollection should be the receiver in any > comparison"

^aCharacterCollection _idxPrimCompareGreaterThan: self
%

category: 'New Indexing Comparison - for Sort'
method: CharacterCollection
_idxForSortCharacterCollectionLessThanOrEqualToSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForSortLessThanOrEqualTo:. Note that aCharacterCollection should be the receiver in any <= comparison"

^(aCharacterCollection _idxPrimCompareGreaterThan: self) not
%

category: 'New Indexing Comparison - for Sort'
method: CharacterCollection
_idxForSortCharacterCollectionLessThanSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForSortLessThan:. Note that aCharacterCollection should be the receiver in any < comparison"

^aCharacterCollection _idxPrimCompareLessThan: self
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortEqualTo: arg

""

^arg _idxForSortEqualToCharacterCollection: self
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortEqualTo: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForSortEqualToCharacterCollection: self
    collator: anIcuCollator
%

category: 'New Indexing Comparison - for Sort'
method: CharacterCollection
_idxForSortEqualToCharacterCollection: aCharacterCollection
  "second half of a double dispatch call from CharacterCollection>>_idxForSortEqualTo:."

  ^ self _idxPrimCompareEqualTo: aCharacterCollection
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortGreaterThan: arg

""

^arg _idxForSortCharacterCollectionGreaterThanSelf: self
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortGreaterThan: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForSortGreaterThanCharacterCollection: self
    collator: anIcuCollator
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortGreaterThanOrEqualTo: arg

""

^arg _idxForSortCharacterCollectionGreaterThanOrEqualToSelf: self
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortGreaterThanOrEqualTo: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForSortGreaterThanOrEqualToCharacterCollection: self
    collator: anIcuCollator
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortLessThan: arg

""

^arg _idxForSortCharacterCollectionLessThanSelf: self
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortLessThan: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForSortLessThanCharacterCollection: self
    collator: anIcuCollator
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortLessThanOrEqualTo: arg

""

^arg _idxForSortCharacterCollectionLessThanOrEqualToSelf: self
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortLessThanOrEqualTo: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForSortLessThanOrEqualToCharacterCollection: self
    collator: anIcuCollator
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortNotEqualTo: arg

""

^(arg _idxForSortEqualTo: self) not
%

category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortNotEqualTo: aCharacterCollection collator: anIcuCollator

  ^ (self _idxForSortEqualTo: aCharacterCollection collator: anIcuCollator) not
%

category: 'New Indexing Comparison - for Sort'
method: CharacterCollection
_idxForSortNotEqualToCharacterCollection: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForSortNotEqualTo:."

^ (self _idxPrimCompareEqualTo: aCharacterCollection) not
%

category: 'New Indexing Comparison - prims'
method: CharacterCollection
_idxPrimCompareEqualTo: arg
  "This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

  "The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #= .
 Same primitive as String>>lessThan: "

  self subclassResponsibility: #_idxPrimCompareEqualTo:
%

category: 'New Indexing Comparison - prims'
method: CharacterCollection
_idxPrimCompareGreaterThan: arg
  "This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

  "The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #< .
 Same primitive as String>>lessThan: "

  self subclassResponsibility: #_idxPrimCompareGreaterThan:
%

category: 'New Indexing Comparison - prims'
method: CharacterCollection
_idxPrimCompareLessThan: arg
  "This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

  "The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #< .
 Same primitive as String>>lessThan: "

  self subclassResponsibility: #_idxPrimCompareLessThan:
%

category: 'Converting'
method: CharacterCollection
_isValid32PathTermName
  "Returns true if the receiver is a valid term in a path expression."

  | first sz maxSize |
  self = '*'
    ifTrue: [ ^ true ].
  maxSize := 64.
  ((sz := self size) > maxSize or: [ sz == 0 ])
    ifTrue: [ ^ false ].
  first := self at: 1.
  (first == $# or: [ first == $_ or: [ first isLetter ] ])
    ifFalse: [ ^ false ].
  2 to: sz do: [ :i |
    | c |
    c := self at: i.
    (c == $_ or: [ c == $| or: [ c isAlphaNumeric ] ])
      ifFalse: [ ^ false ] ].
  ^ true
%

category: 'Indexing Support'
method: CharacterCollection
_isValidPathTermName
  "Returns true if the receiver is a valid term in a path expression."

  ^ self _isValid32PathTermName
%

category: 'Private'
method: CharacterCollection
_keySizeInBytesForOpCode: opCode

| code codes128Bit codes192Bit codes256Bit |
code := opCode abs .
codes128Bit := { 1 . 4 . 7 } .
codes192Bit := { 2 . 5 . 8 } .
codes256Bit := { 3 . 6 . 9 . 10 } .

(codes128Bit includesIdentical: code)
  ifTrue:[ ^ 16 ].
(codes192Bit includesIdentical: code)
  ifTrue:[ ^ 24 ].
(codes256Bit includesIdentical: code)
  ifTrue:[ ^ 32 ].
%

category: 'Private'
method: CharacterCollection
_lineNumberFor:  offset
  "Counts LF characters in the receiver up to offset and
   returns an Array of the form { line number,  offset in line } "
  | cnt lf eolPos |
  cnt := 1 . lf := Character lf .
  eolPos := 0 .
  1 to: (offset min: self size) do:[:n|
    (self at: n) == lf ifTrue:[ eolPos := n . cnt := cnt + 1 ].
  ].
  ^ { cnt . offset - eolPos }
%

category: 'Testing'
method: CharacterCollection
_literalEqual: anotherLiteral
  "Petit Parser definition:
     For two literals to be _literalEqual, their class must be identical and
     otherwise equal.
     For CharacterCollections equality is sufficent
   In GemStone we allow Unicode7 to compare to String, etc."

  ^ self _unicodeEqual: anotherLiteral
%

category: 'Private'
method: CharacterCollection
_primAsymSignVerifyWithKey: aKey digestKind: opCode signature: aByteArray

"Signs or verifies the message contained in the receiver using public key
 encryption. aKey must be an instance of GsTlsPrivateKey (which indicates a
 signing operation) or GsTlsPublicKey (which indicates a verify operation).
 For signing keys that require a message digest, anInt indicates one of the
 supported message digests for the specified key type. For signing keys
 that do not require a message digest, anInt must be zero.  See the tables
 below. The same message digest must be used for both signing and
 verifying.

 For signing operations, aByteArray must be a variant instance of ByteArray
 or nil, in which case a new instance of ByteArray will be created to
 contain the signature. For verifying operations, aByteArray must be a
 non-empty instance of ByteArray containing the signature from the signing.

 For secure signing and verifying with RSA keys only, a padding scheme must
 be used. The default RSA padding scheme is RSA_PKCS1_PADDING
 (PKCS #1 v1.5 padding), which is the most common type. However the newer
 PSS (Probabilistic Signature Scheme) is more secure and is recommended
 whenever possible. See RFC 3447 for additional information on PSS padding.
 Other RSA padding schemes, including no padding, are not supported due
 to known security vulnerabilities. The padding scheme selected must be the
 same for the signature and verification else the verification will fail.

 To sign or verify with RSA_PKCS1_PSS_PADDING, negate the digest opCode.
 For example, a message signed with an RSA private key using opCode 2 uses
 SHA1 message digest and (the default) RSA_PKCS1_PADDING. Signing a message
 with an RSA private key using opCode -2 uses SHA1 message digest and
 RSA_PKCS1_PSS_PADDING padding.

 RSA keys of type RSA-PSS may only use RSA_PKCS1_PSS_PADDING. PSS padding
 will be used for such keys in all cases, even if RSA_PKCS1_PADDING is
 requested. RSA-PSS private keys will answer #EVP_PKEY_RSA_PSS when sent
 the #sslAlgorithm message.

 Signing operations return aByteArray containing the signature on success
 and raise a CryptoError exception on error. Verify operations return true
 if the verification succeeds and raise a CryptoError exception if
 verification fails or an error occurs.

 Note that not all private/public key pairs support digital signatures.

 Key Type  Supported Digest Types
 ================================
 DSA       SHA1,SHA2
 ECDSA	   SHA1,SHA2
 RSA	   SHA1,SHA2,SHA3
 EC        None
 Ed25519   None
 Ed448     None
 ==============================


 Digest Type	OpCode
 =====================
 None           0
 SHA1		2
 SHA2-256	3
 SHA2-512	4
 SHA3-224	5
 SHA3-256	6
 SHA3-384	7
 SHA3-512	8
 =====================
"

<primitive: 1089>
opCode _validateClass: SmallInteger .
aKey _validateClasses: { GsTlsPrivateKey . GsTlsPublicKey } .
(aKey isKindOf: GsTlsPrivateKey)
  ifTrue:[ "signing, arg may be nil"
    aByteArray ifNotNil:[ aByteArray _validateClass: ByteArray ].
  ]
  ifFalse:[
     "verifying, arg must be a ByteArray"
    aByteArray _validateClass: ByteArray
].
self _validateKey: aKey withDigestKind: opCode .
^ self _primitiveFailed: #_primAsymSignVerifyWithKey:digestKind:signature:
%

category: 'Private'
method: CharacterCollection
_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode into: destObjOrNil
"Private method for invoking non-AEAD encrypt/decrypt modes. See the method:
    #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
    into: destObjOrNil tag: tag extraData: eData
 for more information."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: opCode
       into: destObjOrNil
       tag: nil
       extraData: nil
%

category: 'Private'
method: CharacterCollection
_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
into: destObjOrNil tag: tag extraData: eData

"Private method for encrypting or decrypting the receiver.
 The following encryption schemes are currently supported:

 ==================================================
                          Key     Salt    Tag
 opCode Cipher   Mode  bits/Bytes Size   Bytes
 ==================================================
   1     AES     CBC     128/16    16     N/A
   2     AES     CBC     192/24    16     N/A
   3     AES     CBC     256/32    16     N/A
   4     AES     OCB     128/16    12     16
   5     AES     OCB     192/24    12     16
   6     AES     OCB     256/32    12     16
   7     AES     GCM     128/16    12     16
   8     AES     GCM     192/24    12     16
   9     AES     GCM     256/32    12     16
  10   CHACHA20 Poly1305 256/32    12     16
 ==================================================

 AES encryption/decryption (Advanced Encryption Standard) is performed
 using the OpenSSL open source package and the AES specification,
 available at:
   http://csrc.nist.gov/publications/fips/fips197/fips-197.pdf

 CBC is an acronym for cipher block chaining. See the referenced AES
 specification document for further details.

 OCB is an acronym for Offset Cookbook Mode. See RFC 7253 for further
 details.

 GCM is an acronym for Galois Counter Mode. See RFC 5288 for further
 details.

 OCB, GCM and Poly1305 are AEAD modes. AEAD is an acronym for
 Authenticated Encryption with Associated Data. AEAD provides data
 authenticity, confidentiality, and integrity. It also supports
 Additional Authenticated Data (AAD). AAD is not encrypted and therefore
 not kept confidential, however AAD authenticity and integrity are
 guaranteed. AAD is not included in the encrypted payload but must be
 provided in order to decrypt the data. AAD is optional and the eData
 argument may be nil if AAD is not required.

 AEAD encryption/decryption is performed using the OpenSSL open source
 package and is implemented to conform to ISO/IEC standard 19772:2009.
 See https://www.iso.org/standard/46345.html for further information.

 opCode must be an instance of SmallInteger and be one of the values from
 the first column of the above table. A positive value indicates the data
 is to be encrypted and a negative value indicates decryption.

 aKey and aSalt must be instances of ByteArray of the correct size
 per the above table, otherwise an error is raised.

 destObjOrNil must be nil or an instance of a non-invariant byte object.
 If destObjOrNil is nil, the result of the operation will be placed into a
 new instance of ByteArray (encryption) or String (decryption). Otherwise
 the result will be placed into the given byte object starting at offset 1.
 The size of destObjOrNil will be modified to correctly contain all
 encrypted or decrypted data, and may differ from the size of the receiver
 due to the automatic addition or removal of padding by the cipher
 algorithm.

 When encrypting a receiver that has a character size greater than one, data
 is placed into big-endian byte order before encryption.

 When decrypting into a destObjOrNil object that a character size greater
 than one, data is converted to big-endian byte order after decryption.

 During AEAD encryption, a tag is generated which is used during decryption
 to ensure data integrity. The tag data will be stored into the tag
 argument, which must an instance of a variant byte object. During AEAD
 decryption, tag must be a byte object containing the tag bytes returned
 during encryption. For non-AEAD modes, tag must be nil.

 During AEAD encryption, eData must be nil or a byte object with a character
 size of one containing additional data to be used in generating the tag
 value. eData is NOT added to the encrypted payload. During decryption,
 eData must be a byte object with a character size of one containing the
 same bytes provided during encryption or nil if no byte object was
 provided. For non-AEAD modes, eData must be nil.

 Successful encryption or decryption returns encrypted or decrypted data,
 which is stored into destObjOrNil if it was provided or a new byte object
 if it was not. An exception is raised if encryption or decryption fails.
"

<primitive: 953>
self _validateEncryptionOpCode: opCode ;
     _validateEncryptionKey: aKey forOpCode: opCode ;
     _validateEncryptionSalt: aSalt forOpCode: opCode ;
     _validateEncryptionExtraData: eData forOpCode: opCode .
destObjOrNil ifNotNil:[ destObjOrNil _validateIsBytes ; validateIsVariant ] .
self _validateEncryptionTag: tag forOpCode: opCode .
^ self _primitiveFailed: #_primEncryptDecryptWithKey:salt:opCode:into:tag:extraData:
%

category: 'Private'
method: CharacterCollection
_saltSizeForOpCode: opCode

^ opCode abs >= 4
    ifTrue:[ 12 ]
   ifFalse:[ 16 ]
%

category: 'Deprecated'
method: CharacterCollection
_toServerTextFile: aFileSpec

 "Writes the receiver to the specified file
 using the current session's cpu's in-memory byte order of any multi-byte characters.

 The argument aFileSpec must be convertable to a Utf8 , otherwise an error is signaled
 by the primitive."

<primitive: 301>
self deprecated: 'CharacterCollection>>_toServerTextFile: deprecated long before v3.0.  Use an instance of GsFile
 to access the file system.'.

aFileSpec _validateByteClass: CharacterCollection .
self _error: #hostErrFileExport args: { aFileSpec }
%

category: 'Private'
method: CharacterCollection
_validate: anObj isSize: expectedSize

anObj _basicSize == expectedSize
  ifFalse:[ anObj _error: #rtErrBadSize args: { expectedSize . anObj _basicSize } ] .
%

category: 'Private'
method: CharacterCollection
_validateEncryptionExtraData: eData forOpCode: opCode
opCode abs > 3  "AEAD encrypt/decrypt"
  ifTrue:[ eData ifNotNil:[ eData _validateIsBytes ]]
 ifFalse:[ eData _validateClass: UndefinedObject ]
%

category: 'Private'
method: CharacterCollection
_validateEncryptionKey: aKey forOpCode: opCode
aKey _validateClass: ByteArray .
self _validate: aKey isSize: (self _keySizeInBytesForOpCode: opCode)
%

category: 'Private'
method: CharacterCollection
_validateEncryptionOpCode: opCode
opCode _validateClass: SmallInteger .
(((opCode < -10) or:[opCode > 10]) or:[opCode == 0])
  ifTrue:[opCode _error: #rtErrArgOutOfRange args:{ -10 . 10 } ] .
^ true
%

category: 'Private'
method: CharacterCollection
_validateEncryptionSalt: salt forOpCode: opCode
salt _validateClass: ByteArray .
self _validate: salt isSize: (self _saltSizeForOpCode: opCode)
%

category: 'Private'
method: CharacterCollection
_validateEncryptionTag: tag forOpCode: opCode
| absCode |
absCode := opCode abs .
absCode > 3 "AEAD encrypt/decrypt"
  ifTrue:[
    tag _validateClass: ByteArray .
    opCode < 3 ifTrue:[ tag validateIsVariant ]			"AEAD Encrypting"
              ifFalse:[ self _validate: tag isSize: 16 ] . 	"AEAD Decrypting"
  ] ifFalse:[ tag _validateClass: UndefinedObject ] "Not AEAD"
%

category: 'Private'
method: CharacterCollection
_validateInteger: anInt inRangeFrom: lowest to: highest
anInt _validateClass: SmallInteger .
((anInt < lowest) or:[anInt > highest])
  ifTrue:[ anInt _error: #rtErrArgOutOfRange args:{ lowest . highest } ] .
^ true
%

category: 'Private'
method: CharacterCollection
_validateKey: aKey withDigestKind: opCode
|alg|
alg := aKey algorithm . "#RSA, #DSA, #DH, #EC, etc"
alg == #RSA "SHA1,2,3 allowed"
  ifTrue:[ ^ self _validateInteger: opCode abs inRangeFrom: 2 to: 8 ].
alg == #DSA "SHA1 and SHA2 only"
  ifTrue:[ ^ self _validateInteger: opCode inRangeFrom: 2 to: 4 ].
alg == #EC "No digest required/allowed"
  ifTrue:[ ^ self _validateInteger: opCode inRangeFrom: 0 to: 0 ].

^ CryptoError signal: 'Invalid key kind for signing or verifying'
%

category: 'Formatting'
method: CharacterCollection
_wrapTo: col indentingWith: indentStr

"Returns a new instance of the class of the receiver.

 Word-wrap the receiver to column col, treating tab Characters as modulo-8.
 Whenever a line-feed is inserted, prepend indentStr to the subsequent line."

| ch curcol linestart wordstart lf tab str sz |

lf := Character lf.
tab := Character tab.
curcol := 1.
wordstart := 0.
linestart := 1.
str := self class _newString.

1 to: (sz := self size) do: [ :i |
  ch := self at: i.
  ch == lf ifTrue: [
    str add: (self copyFrom: linestart to: i).
    linestart := i + 1.
    wordstart := 0.
    curcol := 1.
  ]
  ifFalse: [
    ch isSeparator ifTrue: [
      wordstart := 0
    ]
    ifFalse: [
      wordstart == 0 ifTrue: [
        wordstart := i
      ].
    ].

    ch == tab ifTrue: [
      curcol := (curcol + 8) \\ 8
    ]
    ifFalse: [
      curcol := curcol + 1
    ].

    curcol > col ifTrue: [
      (wordstart == 0 or: [linestart == wordstart]) ifTrue: [
	str add: (self copyFrom: linestart to: i).
	linestart := i + 1.
	curcol := 1.
      ]
      ifFalse: [
	str add: (self copyFrom: linestart to: wordstart - 1).
	linestart := wordstart.
	curcol := i - wordstart + 1.
      ].
      str add: lf.
      str add: indentStr .
      curcol := curcol + indentStr size .
    ].
  ].
].

linestart <= sz ifTrue: [
  str add: (self copyFrom: linestart to: sz )
].

^str.
%

! Class extensions for 'Class'

!		Class methods for 'Class'

removeallmethods Class
removeallclassmethods Class

category: 'Filein Support'
classmethod: Class
_resolveReservedClass: reservedOop name: className

  " returns an Array of size 2.
     result at:1 is already existing class or nil .
     result at: 2 is a String describing Globals modifications made"

  | oldCls result |
  result := Array new: 2 .
  result at: 2 put: String new .
  reservedOop ifNotNil:[ | oop |
    oop := reservedOop .
    oop _isOneByteString ifTrue:[ oop := Integer fromString: oop ].
    oldCls := Object _objectForOop: oop .
  ] ifNil:[
    oldCls := (System myUserProfile resolveSymbol: className asSymbol) ifNotNil:[:assoc | assoc value ]
  ].
  result at:1 put: oldCls .
  ^ result
%

!		Instance methods for 'Class'

category: 'Class Instance Variables'
method: Class
addClassInstanceVariable: civNameString

"Adds the given class instance variable to the receiver's metaclass.
 Generates an error if the receiver is either not modifiable or does
 not disallow subclasses."

self _validatePrivilege ifTrue:[
  self class addInstVarNames: { civNameString }
]
%

category: 'Updating Variables'
method: Class
addClassVarName: aString

"Add aString to the class variable list for the receiver, if the
 class variable is not already defined."

| aSym definingClass |
self _validatePrivilege ifTrue:[
  aSym := aString asSymbol .
  aSym validateIsIdentifier .
  definingClass := self _classDefiningClassVar: aSym .  "fix bug 10480"
  definingClass ifNotNil:[
    definingClass == self ifTrue:[
      "if the receiver already defines the class variable,
         do nothing and return silently    (fix bug 8094) "
      ^ self
      ].
    LookupError new object: definingClass; key: aSym ; reason: #classErrClassVarNameExists ;
		details: 'class variable already exists'; signal .
    ^ self
  ].
  self _addClassVar: aSym value: nil .
]
%

category: 'Updating'
method: Class
addNewVersion: aClass

"Make aClass a new version of the receiver.  That is, add aClass to the
 receiver's history, and set aClass's history to be the same as the
 receiver's history.  The existing history of aClass will have aClass
 removed from it."

self _validatePrivilege ifTrue:[
  aClass classHistory removeVersion: aClass.
  aClass classHistory: classHistory.
  classHistory newVersion: aClass
].
%

category: 'Updating Variables'
method: Class
addSharedPool: aDictionary

"Add aDictionary to the end of the shared pool list for the receiver. "
| poolDicts |
self _validatePrivilege ifTrue:[
  (aDictionary _validateClass:  SymbolDictionary ) ifTrue:[
     poolDicts := poolDictionaries .
     (poolDicts ~~ nil and:[ poolDicts includesIdentical: aDictionary]) ifTrue:[
        ^ self _error: #classErrPoolDictExists args: { aDictionary }
     ].
     poolDicts ifNil:[ poolDicts := { } . poolDictionaries := poolDicts ]
       ifNotNil:[ poolDicts isInvariant ifTrue:[
                    poolDicts := Array withAll: poolDicts . poolDictionaries := poolDicts
                ]].
     poolDicts add: aDictionary
  ].
].
%

category: 'Queries'
method: Class
allSubclasses

	^ClassOrganizer new allSubclassesOf: self.
%

category: 'Queries'
method: Class
allSuperclasses

	^ClassOrganizer new allSuperclassesOf: self.
%

category: 'Class Instance Variables'
method: Class
atClassInstVar: varName

"Returns the value of the given class instance variable in the receiver.
 Generates an error if the argument does not name a class instance variable
 in the receiver.  In general, it is more efficient to implement a direct
 accessing method for a class instance variable."

| idx cls varSym |
varSym := Symbol _existingWithAll: varName .
varSym == nil ifTrue:[
  self _error: #classErrNotAVar args: { varName } .
  ^ nil
  ].
cls := self class.
idx := cls.instVarNames indexOf: varSym.
(idx < 0 or: [idx <= cls class instSize ]) ifTrue: [
  self _error: #classErrNotAVar args: { varName } .
  ^ nil
].
^ self instVarAt: idx
%

category: 'Class Instance Variables'
method: Class
atClassInstVar: varName put: newValue
	"Changes the value of the given class instance variable in the receiver,
 without regard to the variance or invariance of the receiver.  Generates an
 error if the argument does not name a class instance variable in the receiver.
 Returns the argument 'newValue'."

	self _validatePrivilege
		ifTrue:
			[| idx cls varSym |
			varSym := Symbol _existingWithAll: varName.
			varSym == nil
				ifTrue:
					[self _error: #classErrNotAVar args: {varName}.
					^nil].
			cls := self class.
			idx := cls.instVarNames indexOf: varSym.
			(idx < 0 or: [idx <= cls class instSize])
				ifTrue:
					[self _error: #classErrNotAVar args: {varName}.
					^nil].
			self _unsafeAt: idx put: newValue.
			^newValue].
	^nil
%

category: 'Subclass Creation'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary

^ self byteSubclass: aString
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
    description: nil
    options: #()
%

category: 'Deprecated'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
description: aDescription
isInvariant: invarBoolean

 | opts |
self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
 opts := invarBoolean ifTrue:[ { #instancesInvariant } ]
           ifFalse:[ self instancesInvariant ifTrue:[ ^ self _error: #classErrInvariantSuperClass ].
                     #() ] .
^self
  byteSubclass: aString
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: aDescription
  options: opts
%

category: 'Deprecated'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
inClassHistory: aClassHistory
description: aDescription
isInvariant: invarBoolean

	| theFormat opts |
	self
		deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category ('
				, aString , ').'.
	(aDictionary _validateClass: SymbolDictionary) ifFalse: [^nil].
	self instSize ~~ 0 ifTrue: [^self _error: #classErrByteObjInstVars].
	self isNsc
		ifTrue:
			[^aString _error: #classErrBadFormat
				with: 'cannot create byte subclass of Nsc class'].
	theFormat := (format bitAnd: 16r3 bitInvert) bitOr: 16r1 + 16r4.
	opts := invarBoolean
				ifTrue: [{#instancesInvariant}]
				ifFalse:
					[self instancesInvariant
						ifTrue: [^self _error: #classErrInvariantSuperClass].
					#()].
	^self
		_subclass: aString
		instVarNames: #()
		format: theFormat
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: aClassHistory
		description: aDescription
		options: opts
%

category: 'Deprecated'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
instancesInvariant: invarBoolean

 | opts |
 self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods
  are in the Subclass Creation category (' , aString , ').'.

 opts := invarBoolean ifTrue:[ { #instancesInvariant } ]
           ifFalse:[ self instancesInvariant ifTrue:[ ^ self _error: #classErrInvariantSuperClass ].
                     #() ] .
^self
  byteSubclass: aString
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: nil
  options: opts
%

category: 'Deprecated'
method: Class
byteSubclass: aString
	classVars: anArrayOfClassVars
	classInstVars: anArrayOfClassInstVars
	poolDictionaries: anArrayOfPoolDicts
	inDictionary: aDictionary
	newVersionOf: oldClass
	description: aDescription
	isInvariant: invarBoolean

	| hist theFormat opts descr |
	self
		deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category ('
				, aString , ').'.
	aDictionary _validateClass: SymbolDictionary.
	self instSize ~~ 0 ifTrue: [^self _error: #classErrByteObjInstVars].
	self isNsc
		ifTrue:
			[^aString _error: #classErrBadFormat
				with: 'cannot create byte subclass of Nsc class'].
	theFormat := (format bitAnd: 16r3 bitInvert) bitOr: 16r1 + 16r4.
	opts := invarBoolean
				ifTrue: [{#instancesInvariant}]
				ifFalse:
					[self instancesInvariant
						ifTrue: [^self _error: #classErrInvariantSuperClass].
					#()].
	descr := aDescription.
	oldClass
		ifNotNil:
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: opts
				newFormat: theFormat
				newInstVars: #()
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false)
					ifTrue:
						[oldClass _commentOrDescription: aDescription.
						^oldClass	"avoid creation of a new version"].
			hist := oldClass classHistory.
			descr
				ifNil:
					[descr := [oldClass commentForFileout] on: Error
								do: [:ex | 'old comment not available']]].
	^self
		_subclass: aString
		instVarNames: #()
		format: theFormat
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: opts
%

category: 'Subclass Creation'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
newVersionOf: oldClass
description: aDescription
options: optionsArray

	"Creates and returns a new byte subclass of the receiver.  You are not
 permitted to modify the new class after it is created.  If the receiver is not
 some kind of String class, then instances of the new class store and return
 SmallIntegers in the range 0 - 255.

 This method generates an error if instances of the receiver are of special
 storage format, if they are NSCs, or if they have instance variables.

 optionsArray is an Array of Symbols containing zero or more of
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable ,
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 and at most one of the word size definitions for byte format (affects swizzling)
   #'2byteWords' #'4byteWords' #'8byteWords'
   #'signed2byteWords' #'signed4byteWords' #'signed8byteWords'
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.

 Returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

	| hist fmt descr |
	aDictionary
		ifNotNil: [ (aDictionary _validateClass: SymbolDictionary) ifFalse: [^nil] ].
	self isNsc
		ifTrue:
			[^aString _error: #classErrBadFormat
				with: 'cannot create byte subclass of Nsc class'].
	fmt := (format bitAnd: 16r3 bitInvert) bitOr: 16r1 + 16r4.
	descr := aDescription.
	oldClass
		ifNotNil:
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: optionsArray
				newFormat: fmt
				newInstVars: #()
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false )
					ifTrue:
						[oldClass _commentOrDescription: aDescription.
						^oldClass	"avoid creation of a new version" ].
			hist := oldClass classHistory.
			descr ifNil: [descr := [ oldClass commentForFileout ] on: Error do:[:ex | 'old comment not available']]].
	^self
		_subclass: aString
		instVarNames: #()
		format: fmt
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: optionsArray
%

category: 'Deprecated'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
newVersionOf: oldClass
instancesInvariant: invarBoolean

 | opts |
self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
 opts := invarBoolean ifTrue:[ { #instancesInvariant } ]
           ifFalse:[ self instancesInvariant ifTrue:[ ^ self _error: #classErrInvariantSuperClass ].
                     #() ] .
 ^ self byteSubclass: aString
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: oldClass
    description: (oldClass ifNotNil:[ [ oldClass commentForFileout ] on: Error do:[:ex | 'old comment not available']])
    options: opts
%

category: 'Subclass Creation'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
options: optionsArray


"Creates and returns a new byte subclass of the receiver.  You are not
 permitted to modify the new class after it is created.  If the receiver is not
 some kind of String class, then instances of the new class store and return
 SmallIntegers in the range 0 - 255.

 If aString is the name of a Class that is visible to the current user, this
 method creates the new class as a new version of the existing class, and they
 then share the same class history.  However, if no class named aString is
 visible to the user, then the new class is no relation to any existing class,
 and it has a new class history.

 This method generates an error if instances of the receiver are of special
 storage format, if they are NSCs, or if they have instance variables.

 optionsArray is an Array of Symbols containing zero or more of
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable ,
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 and at most one of the word size definitions for byte format (affects swizzling)
   #'2byteWords' #'4byteWords' #'8byteWords'
   #'signed2byteWords' #'signed4byteWords' #'signed8byteWords'
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.
"

^ self byteSubclass: aString
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
    description: nil
    options: optionsArray
%

category: 'Deprecated'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
inClassHistory: aClassHistory
description: aDescription
isInvariant: invarBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self byteSubclass: aString
  classVars: anArrayOfClassVars
  classInstVars: #()
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  inClassHistory: aClassHistory
  description: aDescription
  isInvariant: invarBoolean
%

category: 'Deprecated'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
isInvariant: invarBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self byteSubclass: aString
    classVars: anArrayOfClassVars
    classInstVars: #()
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    instancesInvariant: invarBoolean
%

category: 'Subclass Creation'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
options: optionsArray

^ self byteSubclass: aString
    classVars: anArrayOfClassVars
    classInstVars: #()
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    options: optionsArray
%

category: 'Instance Migration'
method: Class
cancelMigration

"Disables class migration by clearing the migrationDestination instance
 variable."

self migrationDestination: nil.
%

category: 'Category'
method: Class
category: newCategory

"Sets the classCategory variable of the receiver.
 The argument should be a kind of CharacterCollection or nil."

	self _rwCategory: newCategory
%

category: 'Accessing'
method: Class
classHistory

"Returns the classHistory instance variable for this class, which may be nil."

^ classHistory
%

category: 'Updating'
method: Class
classHistory: aClassHistory

"Set the class history of the receiver. Returns the receiver."

self _validatePrivilege ifTrue:[
  aClassHistory ifNotNil:[
    (aClassHistory _validateInstanceOf: ClassHistory) ifFalse:[ ^ nil ].
  ].
  classHistory := aClassHistory .
  self _refreshClassCache: false .
].
%

category: 'Accessing'
method: Class
classHistoryAt: aSmallInteger

"Returns the specified version of this class."

^ classHistory at: aSmallInteger .
%

category: 'Clustering'
method: Class
clusterBehavior

"Clusters elements of the receiver and its metaclass that are used
 for GemStone Smalltalk execution.

 It is recommended that when several classes are being clustered in a
 transaction, send clusterBehavior to all classes to be clustered, then send
 clusterDescription."

super clusterBehavior ifTrue:[ ^ true ].
self class clusterBehavior .
^ false
%

category: 'Clustering'
method: Class
clusterDescription

"Clusters elements of the receiver and its metaclass that are not required
 for GemStone Smalltalk execution.

 It is recommended that when several classes are being clustered in a
 transaction, send clusterBehavior to all classes to be clustered, then send
 clusterDescription."

| result |
super clusterDescription ifTrue:[ ^ true ].
result := false .
extraDict ifNotNil:[
  result := extraDict cluster .
  result ifFalse:[ extraDict associationsDo:[:assoc| assoc cluster]].
].
result ifFalse:[ timeStamp ifNotNil:[ result := timeStamp cluster ]].
userId ifNotNil:[ userId cluster ].
result ifFalse:[ result := self class clusterDescription ].
^ result
%

category: 'Class Comments'
method: Class
comment

" As of GS/64 3.1, comments are now recorded in the class extraDict
  dictionary under the key #comment.  Comment information formerly
  recorded as a GsClassDocumentation under the key #description are
  converted to a string and placed under #comment during DB
  conversion/upgrade. "

  | str |
  (self _extraDictAt: #comment) ifNotNil:[:cmt | ^ cmt ].
  str := 'No class-specific documentation for ' , self name .
  str add: ', hierarchy is:
'; add: (self hierarchy: 0) .
  ^ str
%

category: 'Class Comments'
method: Class
comment: aString

  (aString isKindOf: CharacterCollection) ifFalse: [
    ArgumentTypeError signal: 'Comment must be a String' ].
  self _extraDictAt: #comment put: aString
%

category: 'Class Comments'
method: Class
commentForFileout

"Returns a non-empty class comment or nil."

| str |
str := self _extraDictAt: #comment .
str size = 0 ifTrue:[ ^ nil ].
^ str
%

category: 'Private'
method: Class
copyVariables
| chSize chist priorVersion priorVars civNames priorCivNames toIgnoreCount |
(chSize := (chist := classHistory) size) = 1 ifTrue:[
  ^ self
].
priorVersion := chist at: chSize - 1.
(priorVars := priorVersion _classVars) notNil ifTrue: [
  | cvars |
  (cvars := classVars) notNil ifTrue: [
    priorVars associationsDo: [:anAssociation | |aKey |
      (cvars includesKey: (aKey := anAssociation key)) ifTrue: [
        cvars removeKey: aKey .
        cvars addAssociation: anAssociation.
      ].
    ].
  ].
].
priorCivNames := priorVersion class allInstVarNames.
toIgnoreCount := Class allInstVarNames size.
toIgnoreCount < priorCivNames size ifTrue: [
	priorCivNames := priorCivNames
		copyFrom: toIgnoreCount + 1
		to: priorCivNames size.
	civNames := self class allInstVarNames.
	priorCivNames do: [:each |
		(civNames includesIdentical: each) ifTrue: [
			self atClassInstVar: each put: (priorVersion atClassInstVar: each).
		].
	].
].
%

category: 'Versions'
method: Class
currentVersion

  "return the most recent version of the receiver"
  classHistory ifNotNil:[ :hist | | sz |
    (sz := hist size) ~~ 0 ifTrue:[ ^ hist at: sz ].
  ].
  ^ self
%

category: 'Accessing'
method: Class
dbTransientInstVarNames
  "Returns an Array (possibly empty) of instVarNames that are dbTransient on a per-instVar basis "
  | res word |
  res := { } .
  (word := self dbTransientMask) ~~ 0 ifTrue:[
     | mask names |
     mask := 1 .
     names := instVarNames .
     1 to: self instSize do:[:n |
       (word bitAnd: mask) ~~ 0 ifTrue:[ res add:(names at: n) ].
       mask := mask bitShift: 1 .
     ].
  ].
  ^ res
%

category: 'Accessing'
method: Class
dbTransientMask
  "Returns a SmallInteger"
  ^ dbTransientMask ifNil:[ 0 ] .
%

category: 'Browser Methods'
method: Class
definition
	"Returns a String containing a GemStone Smalltalk definition for the receiver
 (that is, a subclass creation message).  This method uses the UserProfile
 of the owner of the current session as the correct context."
	"For use with the Topaz run command."

	^self _definitionInContext: System myUserProfile
%

category: 'Accessing'
method: Class
description

"Returns the description of this class."

self deprecated: 'Class>>description deprecated v3.1. Replaced by comment.'.
^ self _description
%

category: 'Class Comments'
method: Class
description: aDescription

"Update the description of this Class.  Returns the argument."
self deprecated: 'Deprecated as of GS/64 3.1'.
^ self _description: aDescription
%

category: 'Accessing'
method: Class
extraDict

"Returns the SymbolDictionary held in extraDict that holds miscellaneous
 information about the receiver.  Result may be nil. "

^ extraDict
%

category: 'Updating'
method: Class
extraDict: aSymbolDictionary

"Set the value of the extraDict instance variable."

self _validatePrivilege ifTrue:[
  aSymbolDictionary ifNotNil:[
    (aSymbolDictionary _validateClass: SymbolDictionary) ifFalse:[ ^ self].
  ].
  extraDict := aSymbolDictionary
].
%

category: 'Accessing'
method: Class
extraDictForStore

"Returns the SymbolDictionary held in extraDict that holds miscellaneous
 information about the receiver.  Creates the dictionary if needed."

^ extraDict ifNil:[ extraDict := SymbolDictionary new ].
%

category: 'Subclass Creation'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary

"Executes
   oldClass := aDictionary at:(aString asSymbol) otherwise: nil
 and returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

^ self indexableSubclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
    description: nil
    options: #()
%

category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
description: aDescription
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  indexableSubclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: aDescription
  isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: ignored
instancesInvariant: invarBoolean
inClassHistory: aClassHistory
description: aDescription
isModifiable: modifyBoolean

| opts fmt |
self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
"Any specified constraints are ignored."
(self isBytes) ifTrue: [ ^ aString _error: #classErrBadFormat with: 'cannot create indexable subclass of byte class'].
(self isNsc) ifTrue: [ ^ aString _error: #classErrBadFormat with: 'cannot create indexable subclass of Nsc class'].
opts := { } .
invarBoolean ifTrue:[ opts add: #instancesInvariant ]
           ifFalse:[ self instancesInvariant ifTrue:[ ^ self _error: #classErrInvariantSuperClass ]].
modifyBoolean ifTrue:[ opts add: #modifiable ].
fmt := format bitOr: 16r4 "add indexable bit" .

^ self _subclass: aString
        instVarNames: anArrayOfInstvarNames
        format: fmt
        classVars: anArrayOfClassVars
        classInstVars: anArrayOfClassInstVars
        poolDictionaries: anArrayOfPoolDicts
        inDictionary: aDictionary
        inClassHistory: aClassHistory
        description: aDescription
        options: opts
%

category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  indexableSubclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: nil
  isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: ignored
instancesInvariant: invarBoolean
newVersionOf: oldClass
description: aDescription
isModifiable: modifyBoolean

	| opts fmt hist descr |
	self
		deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category ('
				, aString , ').'.
	"Any specified constraints are ignored."
	self isBytes
		ifTrue:
			[^aString _error: #classErrBadFormat
				with: 'cannot create indexable subclass of byte class'].
	self isNsc
		ifTrue:
			[^aString _error: #classErrBadFormat
				with: 'cannot create indexable subclass of Nsc class'].
	opts := {}.
	invarBoolean
		ifTrue: [opts add: #instancesInvariant]
		ifFalse:
			[self instancesInvariant
				ifTrue: [^self _error: #classErrInvariantSuperClass]].
	modifyBoolean ifTrue: [opts add: #modifiable].
	fmt := format bitOr: 16r4.	"add indexable bit"
	descr := aDescription.
	oldClass
		ifNotNil:
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: opts
				newFormat: fmt
				newInstVars: anArrayOfInstvarNames
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false)
					ifTrue:
						["avoid creation of a new version"
						oldClass _commentOrDescription: aDescription.
						^oldClass].
			hist := oldClass classHistory.
			descr
				ifNil:
					[descr := [oldClass commentForFileout] on: Error
								do: [:ex | 'old comment not available']]].
	^self
		_subclass: aString
		instVarNames: anArrayOfInstvarNames
		format: fmt
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: opts
%

category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
newVersionOf: oldClass
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self indexableSubclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    constraints: aConstraint
    instancesInvariant: invarBoolean
    newVersionOf: oldClass
    description: nil
    isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
instancesInvariant: invarBoolean
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  indexableSubclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  instancesInvariant: invarBoolean
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
instancesInvariant: invarBoolean
newVersionOf: oldClass
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self indexableSubclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    constraints: #()
    instancesInvariant: invarBoolean
    newVersionOf: oldClass
    description: (oldClass ifNotNil:[ [ oldClass commentForFileout ] on: Error do:[:ex | 'old comment not available']])
    isModifiable: modifyBoolean
%

category: 'Subclass Creation'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
newVersionOf: oldClass
description: aDescription
options: optionsArray
	"Creates and returns a new indexable subclass of the receiver.  Instances of the
 new class are represented as pointer objects.

 This method generates an error if instances of the receiver are of special
 storage format or if they are NSCs.

 optionsArray is an Array of Symbols containing zero or more of
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable ,
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.

 Returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

	| hist fmt descr |
	self isBytes
		ifTrue:
			[^aString _error: #classErrBadFormat
				with: 'cannot create indexable subclass of byte class'].
	self isNsc
		ifTrue:
			[^aString _error: #classErrBadFormat
				with: 'cannot create indexable subclass of Nsc class'].
	fmt := format bitOr: 16r4.	"add indexable bit"
	descr := aDescription.
	oldClass
		ifNotNil:
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: optionsArray
				newFormat: fmt
				newInstVars: anArrayOfInstvarNames
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false )
					ifTrue:
						[oldClass _commentOrDescription: aDescription.
						^oldClass	"avoid creation of a new version"].
			hist := oldClass classHistory.
			descr ifNil: [descr := [ oldClass commentForFileout ] on: Error do:[:ex | 'old comment not available']]].
	^self
		_subclass: aString
		instVarNames: anArrayOfInstvarNames
		format: fmt
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: optionsArray
%

category: 'Subclass Creation'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
options: optionsArray

"Creates and returns a new indexable subclass of the receiver.  Instances of the
 new class are represented as pointer objects.

 This method generates an error if instances of the receiver are of special
 storage format or if they are NSCs.

 optionsArray is an Array of Symbols containing zero or more of
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable ,
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.

 Executes
   oldClass := aDictionary at:(aString asSymbol) otherwise: nil
 and returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

^ self indexableSubclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
    description: nil
    options: optionsArray
%

category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
inClassHistory: aClassHistory
description: aDescription
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self indexableSubclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: #()
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  inClassHistory: aClassHistory
  description: aDescription
  isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self indexableSubclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: #()
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    constraints: aConstraint
    instancesInvariant: invarBoolean
    isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
isInvariant: invarBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self indexableSubclass: aString
     instVarNames: anArrayOfInstvarNames
     classVars: anArrayOfClassVars
     classInstVars: #()
     poolDictionaries: anArrayOfPoolDicts
     inDictionary: aDictionary
     constraints: aConstraint
     instancesInvariant: invarBoolean
     isModifiable: false
%

category: 'Queries'
method: Class
instancesInMemory
"Returns an Array that contains all instances of the receiver.
 that are in the temporary object memory of this session."

^ (SystemRepository listInstancesInMemory: { self }) at: 1
%

category: 'Displaying'
method: Class
instanceString

"Returns a symbol that can be used to name an instance of the receiver."

^(('AEIOUYaeiouy' indexOf: (name at: 1) startingAt: 1) ~~ 0
   ifTrue: ['an']
   ifFalse: ['a'])
 , name
%

category: 'Displaying'
method: Class
instanceSymbol

"Returns a symbol that can be used to name an instance of the receiver."

^ ((('AEIOUYaeiouy' indexOf: (name at: 1) startingAt: 1) ~~ 0
  ifTrue: ['An']
  ifFalse: ['A'])
    , name) asSymbol
%

category: 'Class Instance Variables'
method: Class
instVarAt: anIndex put: aValue
  | sz |
  self _validatePrivilege ifTrue:[
    anIndex <= Class instSize ifTrue:[
      ImproperOperation new object: self ;
         signal: 'instVarAt:put: not allowed to a class instance variable defined for Class'
    ].
    anIndex > (sz := self class instSize) ifTrue:[
       OutOfRange new name:#anIndex max: sz actual: anIndex ; signal .
    ].
    self _unsafeAt: anIndex put: aValue .
    ^ aValue
  ].
  ^ nil
%

category: 'Updating'
method: Class
instVarDbTransient: aSymbol value: aBoolean
  "Change the per-instVar dbTransient attribute of instVar specifed by aSymbol to aBoolean.

   A instVar with attribute true will have its value  flushed to disk be nil on commit,
   and will be nil when faulted in from disk.  To preserve the in-memory state of the
   instVar,  ensure the object is kept reachable in memory , such as from  SessionTemps current,
   or from the Smalltalk stack.

   For an indexable class with named instVars, such as a subclass of Array ,
   if an instance has  (self size + self class instSize > 2034) 
   the per-instVar dbTransient attribute will always behave as false on that instance.

   The change will not take effect completely until after this session commits, and will only
   be completely in effect in sessions which login after such commit .

   For instances committed prior to setting dbTransient attribute to true, that instVar,
   may be non-nil on disk, but that non-nil value will not be visible to Smalltalk execution
   in sessions that login after the attribute change.  listReferences and markForCollection
   will see the non-nil value; the next commit that changes some other instVar in
   the instance will set that instVar on disk to nil.

   Instance variables that are dbTransient may not participate any Index on an UnorderedCollection,
   errors will be signaled if any element of a path for an Index evaluates to a dbTransient
   instance variable.

   Only the first 60 instVars may be dbTransient .
   If aSymbol specifies an instVar beyond 60 (i.e. instVarAt:61 or subsequent instVar) ,
   an Error will be signalled."

  | ofs word mask |
  (ofs := instVarNames indexOfIdentical: aSymbol) == 0 ifTrue:[
     ^ ImproperOperation signal:'not a valid instVarName: ', aSymbol printString .
  ].
  ofs > 60"GC_Class_max_dbTrIvOffset" ifTrue:[ 
     ^ ImproperOperation signal:'dbTransient is supported only on the first 60 instVars of a class'
  ].
  "prevent changes if instVar names are still changable"
  self isModifiable ifTrue:[ ImproperOperation signal:'class cannot be modifiable' ].
  (word := dbTransientMask) ifNil:[ 
     aBoolean ifFalse:[ ^ self ].
     word := 0 .
  ]. 
  mask := 1 bitShift: ofs - 1 .
  aBoolean ifTrue:[ word := word bitOr: mask ]
          ifFalse:[ word := word bitAnd: mask bitInvert ].
  dbTransientMask := word .
%

category: 'Versions'
method: Class
isVersionOf: anotherClass

"Returns whether the receiver and the given class share the same class
 history."

| hist |
^ ( hist := classHistory) ~~ nil and:[ anotherClass classHistory == hist ].
%

category: 'Locking'
method: Class
lockableParts

"Returns an Array of the receiver's contents that are locked by browsers
 and folders."

| parts ed |
parts := super lockableParts.
(ed := extraDict) ifNotNil:[
	parts addLast: ed.
	ed do: [:each | parts addLast: each].
].
^ parts reject:[:ea | ea == nil or:[ ea == #() ] ].
%

category: 'Modifying Classes'
method: Class
makeInstancesDbTransient

  "Takes effect immediately and will cause any instances that get
   committed to be DbTransient.

   For a DbTransient object, instVars on disk are always nil.
   The first transaction which causes the object to be reachable
   from committed state will commit the object with all instVars on
   disk set to nil .  Thereafter any stores to instVars of the committed
   object in any session do not cause the object to be written
   by the transaction.

   Clustering a committed DbTransient object will
   cause the object to be rewritten to disk with all instVars nil
   when the transaction commits.  So clustering is the only way
   that concurrency conflict could occur involving a DbTransient object.

   The format of the receiver must be    non-indexable, pointer
   otherwise an error is generated.

   You cannot change a non-modifiable class to or from dbTransient ."

  <primitive: 2001>   "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [
    self validateIsModifiable .
    self _makeInstancesDbTransient: true .
  ] ensure: [
    prot _leaveProtectedMode
  ].
%

category: 'Modifying Classes'
method: Class
makeInstancesNonPersistent

  "Takes effect immediately and will prevent committing new instances
   of the receiver in the current transaction.

   To change a non-modifiable class from persistent to non-persistent ,
   see  ClassOrganizer >> makeInstancesNonPersistent: . "

  <primitive: 2001>   "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [ self validateIsModifiable .
    self _makeInstancesNonPersistent .
  ] ensure:[
    prot _leaveProtectedMode
  ].
%

category: 'Modifying Classes'
method: Class
makeInstancesNotDbTransient

  "Takes effect immediately and cancels any previous makeInstancesDbTransient.

   The receiver must be a modifiable class."

  <primitive: 2001>   "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [
    self validateIsModifiable .
    self _makeInstancesDbTransient: false .
  ] ensure:[
    prot _leaveProtectedMode .
  ].
%

category: 'Modifying Classes'
method: Class
makeInstancesPersistent

  "Takes effect immediately.
   To change a non-modifiable class from non-persistent to persistent,
   see  ClassOrganizer >> makeInstancesPersistent: . "

  <primitive: 2001>   "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [ self isModifiable ifFalse:[ self validateIsModifiable . ^ self ].
    self _makeInstancesPersistent .
  ] ensure:[
    prot _leaveProtectedMode
  ].
%

category: 'Method Timestamps'
method: Class
methodStampDictName
  ^ #GSMethodStampDict
%

category: 'Instance Migration'
method: Class
migrateTo: aClass

"Enables class migration by setting the migrationDestination instance
 variable."

self _validatePrivilege ifTrue:[
  self migrationDestination: aClass.
].
%

category: 'Accessing'
method: Class
migrationDestination

"Returns the migrationDestination instance variable of this class."

^ destClass
%

category: 'Updating'
method: Class
migrationDestination: aClass

"Update the migrationDestination instance variable.  Returns the argument."

self _validatePrivilege ifTrue:[
  aClass ifNotNil:[ aClass _validateClass: Class ].
  destClass := aClass .
  self _refreshClassCache: false .
].
^ aClass
%

category: 'Repository Conversion'
method: Class
needsRecompileFor30

 "Returns true if the receiver needs to have methods recompiled."

 ^ super needsRecompileFor30 or:[ self class needsRecompileFor30]
%

category: 'Repository Conversion'
method: Class
needsRecompileFor33

 "Returns true if the receiver needs to have methods recompiled."

 ^ super needsRecompileFor33 or:[ self class needsRecompileFor33]
%

category: 'Instance Creation'
method: Class
new

"Returns an instance of the receiver with no indexed variables."
<primitive: 51>
self _primitiveFailed: #new .
self _uncontinuableError
%

category: 'Instance Creation'
method: Class
new: anInteger

"Returns an instance of the receiver with the specified number of indexed
 variables.  Generates an error if the receiver is not indexable or if
 anInteger is not a positive SmallInteger.

 For new byte objects, all indexed variables are set to zero;
 for new pointer objects, all indexed variables are set to nil."

<primitive: 53>
(self isIndexable) ifFalse:[ self _errorNotIndexable .  ^ self new ].
(anInteger _isInteger)
  ifFalse:[ anInteger _validateClass: Integer . ^ self new ]
  ifTrue:[
    (anInteger < 0) ifTrue:[ anInteger _error: #rtErrArgNotPositive .
                            ^ self new].
    anInteger _error: #rtErrArgOutOfRange .
    ^ self new
  ].
self _primitiveFailed: #new: args: { anInteger }.
self _uncontinuableError
%

category: 'Pragmas'
method: Class
pragmaDictName

  ^ #GSMethodPragmaDict
%

category: 'Repository Conversion'
method: Class
recompileAllMethods

"Recompile all methods for execution in a Gs64 v3.0 or later system."

super recompileAllMethods .
self class recompileAllMethods
%

category: 'Browser Methods'
method: Class
recompileWithDicts: symbolList

"Recompiles all the receiver's instance and class methods for envId 0.
 Returns the CompiledMethods that fail to compile properly."
| failed |
failed := { }.
self _validatePrivilege ifTrue:[
  | cls envId |
  cls := self .
  envId := 0 .
  2 timesRepeat: [
    cls env: envId unifiedCategoriesDo:[ :categName :selectorList |
      selectorList copy do: [ :aSel| | oldMeth |
        [ oldMeth := cls compiledMethodAt: aSel environmentId: envId .
          cls compileMethod: oldMeth sourceString dictionaries: symbolList
		          category: categName environmentId: envId
        ] onException: CompileError do:[:ex |
          failed add: oldMeth
        ].
      ].
    ].
    cls := self class .
  ].
].
^ failed.
%

category: 'Updating Variables'
method: Class
removeClassVarName: aString

"Remove the class variable named aString from the receiver.
 The value of the removed Association is set to nil.
 Signals an error if there is no such class variable, or if the
 Association for that class variable is invariant."

self _validatePrivilege ifTrue:[
  (Symbol _existingWithAll: aString) ifNotNil:[ :aSym |
    classVars ifNotNil:[ :cvs | | assoc |
      assoc := cvs associationAt: aSym otherwise: nil .
      assoc ifNotNil:[
         assoc isInvariant ifFalse:[ assoc value: nil ].
         cvs removeKey: aSym  .
         ^ self
      ].
    ].
  ].
  ^ LookupError new reason: #classErrClassVarNotFound; key: aString ; object: self;
	signal
].
%

category: 'Updating Variables'
method: Class
removeSharedPool: aDictionary

"Remove aDictionary from the shared pool list for the receiver.  Generates an
 error if aDictionary is not a shared pool for the receiver.

 You may use this method only if, when the receiver was created, the argument
 to poolDictionaries: was an Array rather than a literal Array, which would
 create an InvariantArray.  (See Class >> subclass:.)"

self _validatePrivilege ifTrue:[
  poolDictionaries ifNotNil:[ | idx |
    (idx := poolDictionaries indexOfIdentical: aDictionary) ~~ 0 ifTrue:[
       poolDictionaries removeAtIndex: idx .
       ^ self
    ].
  ].
  ^ LookupError new reason: #classErrPoolDictNotFound;  key: aDictionary  ;
	object: self ; signal .
].
%

category: 'Deprecated'
method: Class
subclass: aString
inDictionary: aDictionary
constraints: constraintSpec

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self subclass: aString
    instVarNames: #()
    classVars:  #()
    classInstVars:  #()
    poolDictionaries: { }
    inDictionary: aDictionary
    constraints: constraintSpec
    instancesInvariant: false
    isModifiable: false
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classInstVars: anArrayOfClassInstVars
inDictionary: aDictionary
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self
    subclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars:  #()
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: { }
    inDictionary: aDictionary
    constraints:  #()
    instancesInvariant: false
    isModifiable: modifyBoolean
%

category: 'Subclass Creation'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary

"Executes
   oldClass := aDictionary at:(aString asSymbol) otherwise: nil
 and returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

^ self subclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
    description: nil
    options: #()
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
description: aDescription
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: aDescription
  isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: ignored
instancesInvariant: invarBoolean
inClassHistory: aClassHistory
description: aDescription
isModifiable: modifyBoolean

| opts |
self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
opts := { } .
invarBoolean ifTrue:[ opts add: #instancesInvariant ]
           ifFalse:[ self instancesInvariant ifTrue:[ ^ self _error: #classErrInvariantSuperClass ]].
modifyBoolean ifTrue:[ opts add: #modifiable ].
^ self _subclass: aString
          instVarNames: anArrayOfInstvarNames
          format: format
          classVars: anArrayOfClassVars
          classInstVars: anArrayOfClassInstVars
          poolDictionaries: anArrayOfPoolDicts
          inDictionary: aDictionary
          inClassHistory: aClassHistory
          description: aDescription
          options: opts
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: nil
  isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: ignored
instancesInvariant: invarBoolean
newVersionOf: oldClass
description: aDescription
isModifiable: modifyBoolean

	| opts hist descr |
	self
		deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category ('
				, aString , ').'.
	"Any specified constraints are ignored."
	opts := {}.
	invarBoolean
		ifTrue: [opts add: #instancesInvariant]
		ifFalse:
			[self instancesInvariant
				ifTrue: [^self _error: #classErrInvariantSuperClass]].
	modifyBoolean ifTrue: [opts add: #modifiable].
	descr := aDescription.
	oldClass
		ifNotNil:
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: opts
				newFormat: self format
				newInstVars: anArrayOfInstvarNames
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false)
					ifTrue:
						[oldClass _commentOrDescription: aDescription.
						^oldClass	"avoid creation of a new version"].
			hist := oldClass classHistory.
			descr
				ifNil:
					[descr := [oldClass commentForFileout] on: Error
								do: [:ex | 'old comment not available']]].
	^self
		_subclass: aString
		instVarNames: anArrayOfInstvarNames
		format: format
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: opts
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
newVersionOf: oldClass
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self subclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    constraints: aConstraint
    instancesInvariant: invarBoolean
    newVersionOf: oldClass
    description: nil
    isModifiable: modifyBoolean

%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: ignored
options: optionsArr

	| hist descr oldClass |
	self
		deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category ('
				, aString , ').'.
	"Any specified constraints are ignored."
	oldClass := self _classNamed: aString inDictionary: aDictionary.
	oldClass
		ifNotNil:
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: optionsArr
				newFormat: self format
				newInstVars: anArrayOfInstvarNames
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false) ifTrue: [^oldClass	"avoid creation of a new version"].
			hist := oldClass classHistory.
			descr := [oldClass commentForFileout] on: Error
						do: [:ex | 'old comment not available']].
	^self
		_subclass: aString
		instVarNames: anArrayOfInstvarNames
		format: format
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: optionsArr
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
instancesInvariant: invarBoolean
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  instancesInvariant: invarBoolean
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
instancesInvariant: invarBoolean
newVersionOf: oldClass
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self subclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    constraints: #()
    instancesInvariant: invarBoolean
    newVersionOf: oldClass
    description: (oldClass ifNotNil:[ [ oldClass commentForFileout ] on: Error do:[:ex | 'old comment not available']])
    isModifiable: modifyBoolean
%

category: 'Subclass Creation'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
options: optionsArray

"Creates and returns a new subclass of the receiver.

 optionsArray is an Array of Symbols containing zero or more of
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable ,
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.

 Executes
   oldClass := aDictionary at:(aString asSymbol) otherwise: nil
 and returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "


^ self subclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
    description: nil
    options: optionsArray
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
inClassHistory: aClassHistory
description: aDescription
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: #()
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  inClassHistory: aClassHistory
  description: aDescription
  isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
   subclass: aString
   instVarNames: anArrayOfInstvarNames
   classVars: anArrayOfClassVars
   classInstVars: #()
   poolDictionaries: anArrayOfPoolDicts
   inDictionary: aDictionary
   constraints: aConstraint
   instancesInvariant: invarBoolean
   isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
isInvariant: invarBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: #()
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  isModifiable: false
%

category: 'Subclass Creation'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
inDictionary: aDictionary

"Creates and returns a new subclass of the receiver.

 This method is a shortcut for convenience only.  It might not be retained in
 future GemStone releases.  Use it interactively or pedagogically, but avoid
 it in production code.

 The new class has no class variables, no class instance variables,
 and no pool dictionaries.
 Instances of the new class are variant, but the new class itself is not
 modifiable.

 Executes
   oldClass := aDictionary at:(aString asSymbol) otherwise: nil
 and returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars:  #()
  classInstVars:  #()
  poolDictionaries: { }
  inDictionary: aDictionary
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: nil
  options: #()
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
inDictionary: aDictionary
constraints: constraintSpec

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars:  #()
  poolDictionaries: { }
  inDictionary: aDictionary
  constraints: constraintSpec
  instancesInvariant: false
  isModifiable: false
%

category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
inDictionary: aDictionary
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars:  #()
  classInstVars:  #()
  poolDictionaries: { }
  inDictionary: aDictionary
  constraints:  #()
  instancesInvariant: false
  isModifiable: modifyBoolean
%

category: 'Subclass Creation'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
inDictionary: aDictionary
options: optionsArray

"Creates and returns a new subclass of the receiver.

 This method is a shortcut for convenience only.  It might not be retained in
 future GemStone releases.  Use it interactively or pedagogically, but avoid
 it in production code.

 The new class has no class variables, no class instance variables,
 and no pool dictionaries.
 Instances of the new class are variant, but the new class itself is not
 modifiable.

 optionsArray is an Array of Symbols containing zero or more of
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable ,
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.

 Executes
   oldClass := aDictionary at:(aString asSymbol) otherwise: nil
 and returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars:  #()
  classInstVars:  #()
  poolDictionaries: { }
  inDictionary: aDictionary
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: nil
  options: optionsArray
%

category: 'Queries'
method: Class
subclasses

	^ClassOrganizer new subclassesOf: self.
%

category: 'Accessing'
method: Class
thisClass
  ^ self
%

category: 'Accessing'
method: Class
timeStamp

"Returns the timestamp instance variable of this class, a DateTime showing when
 the class was created."

^ timeStamp
%

category: 'Updating'
method: Class
timeStamp: aDateTime

"Set the value of the timeStamp instance variable.
 For use only when creating a class, while the class is not yet invariant."

self _validatePrivilege ifTrue:[
  self isInvariant ifFalse:[
    aDateTime ifNotNil:[ (aDateTime _validateClass: DateTime) ifFalse:[ ^ self]].
    timeStamp := aDateTime
  ] ifTrue:[
    self validateIsVariant .
  ].
].
%

category: 'Accessing'
method: Class
userId

"Returns the userId instance variable of this class, the ID of the user who
 created this class."

^ userId
%

category: 'Updating'
method: Class
userId: aString

"Set the value of the userId instance variable.
 For use only when creating a class, while the class is not yet invariant."

(self _validatePrivilege) ifTrue:[
  self isInvariant ifFalse:[
    aString ifNotNil:[ (aString _validateClass: CharacterCollection ) ifFalse:[ ^ self]].
    userId := aString
  ] ifTrue:[
    self validateIsVariant .
  ].
].
%

category: 'Accessing'
method: Class
versionedName
| ofs str |
(classHistory atOrNil: classHistory size) == self ifTrue:[ ^ name ].
(ofs := classHistory indexOfIdentical: self) ~~ 0 ifTrue:[
  (str := String new) addAll: name ; add: $[ ; add: ofs asString; add:$] .
  ^ str
].
Error signal: 'oop ' , self asOop asString, ', not found in classHistory(oop ' ,
    classHistory asOop asString , $) .
%

category: 'Accessing'
method: Class
versionNumber
 "Returns a SmallInteger"
  | ofs |
  (ofs := classHistory indexOfIdentical: self) ~~ 0 ifTrue:[
     ^ ofs
  ].
  ^ 1
%

category: 'Private'
method: Class
_addByteSwizzle: aSymbol toFormat: fmt newClassName: subClsName signed: isSigned
 | swiz newFmt oldSwiz signedMask |
  "uses GC_BEHAV_byteSwizKind_shift"
  aSymbol == #'2byteWords' ifTrue:[ swiz := 1 ].
  aSymbol == #'4byteWords' ifTrue:[ swiz := 2 ].
  aSymbol == #'8byteWords' ifTrue:[ swiz := 3 ].
  aSymbol == #'signed2byteWords' ifTrue:[ swiz := 1 ].
  aSymbol == #'signed4byteWords' ifTrue:[ swiz := 2 ].
  aSymbol == #'signed8byteWords' ifTrue:[ swiz := 3 ].
  swiz ifNil:[ Error signal:'invalid argument ' , aSymbol].

  oldSwiz := (format bitShift: -31"GC_BEHAV_byteSwizKind_shift") bitAnd: 3 .
  oldSwiz > 0 ifTrue:[
    oldSwiz = swiz ifFalse:[
       subClsName _error: #classErrBadFormat
        with: ' option ', aSymbol printString,' incompatible with superclass'.
    ].
  ].
  newFmt := fmt bitOr: (swiz bitShift: 31"GC_BEHAV_byteSwizKind_shift" ) .
  signedMask := 1 bitShift: 33"GC_BEHAV_signedWords_shift" .
  isSigned ifTrue:[ newFmt := newFmt bitOr: signedMask ]
          ifFalse:[ newFmt := newFmt bitAnd: (signedMask bitInvert) ].
  ^ newFmt
%

category: 'Private'
method: Class
_adjustClassVars: anArray
  classVars ifNil:[
    1 to: anArray size do:[:j | self _addClassVar: (anArray at: j) value: nil ]
  ] ifNotNil:[
    | argSet existingSet toRemove toAdd assocs |
    existingSet := IdentitySet new .
    assocs := { } .
    argSet := IdentitySet withAll: anArray  .
    "GsDevKit and Seaside need to be able to override classVarNames for
     implementation of SharedPool ,
     so must use   self classVarNames    here."
    self classVarNames do: [:classVarName | | a |
      a := classVars associationAt: classVarName.
      assocs add: a . existingSet add: a key ].
    toRemove := existingSet - argSet  .
    classHistory size > 1 ifTrue:[  | cvd | "create a new dictionary"
      cvd := SymbolDictionary new objectSecurityPolicy: classVars objectSecurityPolicy ; yourself.
      assocs do:[:a | (toRemove includes: a key) ifFalse:[ cvd addAssociation: a ]].
      self _setClassVars: cvd old: classVars .
    ] ifFalse:[
      toRemove do:[ :aKey | self removeClassVarName: aKey ].
    ].
    toAdd := argSet - existingSet .
    toAdd do:[ :aKey | self _addClassVar: aKey value: nil ].
  ].
%

category: 'Private'
method: Class
_adjustOptions: opts formatArr: fmtArray
  | fmt newFmt modifiableBool fmtToStore |
  fmt := format .
  fmtToStore := fmt .
  newFmt := fmtArray at: 1 .
  modifiableBool := fmtArray at: 2 .
  ((fmt bitAnd: 16r20) ~~ 0 and:[ (newFmt bitAnd: 16r20) == 0 ]) ifTrue:[
     "clear subclassesDisallowed bit."
     fmtToStore := fmtToStore bitAnd: 16r20 bitInvert .
  ].
  ((fmt bitAnd: 16r800) == 0 and:[ (newFmt bitAnd: 16r800) ~~ 0 ]) ifTrue:[
    "set instancesNonPersistent bit"
    fmtToStore := fmtToStore bitOr: 16r800 .
  ].
  (fmt bitAnd: 16r400) ~~ (newFmt bitAnd: 16r400) ifTrue:[
     "change state of traverseByCallback"
     fmtToStore := (fmtToStore bitAnd:( 16r400 bitInvert)) bitOr:( newFmt bitAnd: 16r400) .
  ].
  (fmt bitAnd: 16r200) ~~ (newFmt bitAnd: 16r200) ifTrue:[
     "change state of disallowGciStore"
     fmtToStore := (fmtToStore bitAnd:( 16r200 bitInvert)) bitOr:( newFmt bitAnd: 16r200) .
  ].
  (fmt bitAnd: 16r2000) ~~ (newFmt bitAnd: 16r2000) ifTrue:[
     "change state of selfCanBeSpecial"
     fmtToStore := (fmtToStore bitAnd:( 16r2000 bitInvert)) bitOr:( newFmt bitAnd: 16r2000) .
  ].
  fmtToStore ~~ fmt ifTrue:[
		 self _unsafeAt: 2 "offset of format" put: fmtToStore .
		 (fmtArray at: 3) ifTrue:[ "logBool"
				self _logServer: 'modified format of class ' , self name,' to ', fmtToStore asString .
		 ].
  ].
  "clear modifiable"
  (self isModifiable and:[ modifiableBool == false ]) ifTrue:[
     self immediateInvariant .
  ].
%

category: 'Private'
method: Class
_beVariantWhile: aBlock

"Executes the given block while the receiver is in a variant (modifiable) form.
 The block should not contain a return or the receiver will be left in a
 variant state.

 Great caution should be exercised in using this method as it allows
 modification of classes in uncontrolled ways that can lead to corrupted class
 objects."

self _validatePrivilege ifTrue:[
  | was |
  was := self isInvariant.
  was ifTrue: [ self __makeVariant; _refreshClassCache: false  ].
  [
    [
      aBlock value
    ] onException: Error do:[:ex |
      was ifTrue: [
	super immediateInvariant; _refreshClassCache: false.
	was := false
      ].
      ex outer .
    ]
  ] ensure:[
    was ifTrue:[ super immediateInvariant; _refreshClassCache: false ].
  ]
].
%

category: 'Browser Methods'
method: Class
_bytesPerWord
  "Returns a SmallInteger, the number of bytes per word in instances of the receiver.
   Returns 0 for special format, 8 for oop or nsc format,
   and 1, 2, 4 or 8 for various byte format classes."

  self isBytes ifTrue:[ | swiz |
    swiz := (format bitShift: -31"GC_BEHAV_byteSwizKind_shift") bitAnd: 3"GC_BEHAV_byteSwizKind_mask".
    ^ 1 bitShift: swiz .
  ].
  self _isSpecial ifTrue:[ ^ 0 ].
  ^ 8
%

category: 'Private'
method: Class
_category: newCategory
"Sets the classCategory variable of the receiver.
 The argument should be a kind of CharacterCollection or nil."
"Implemenation shared by base image and Rowan"

newCategory ifNil:[
	classCategory := nil.
	^ self ].

(newCategory _validateClass: CharacterCollection ) ifFalse:[ ^ nil ].

classCategory := newCategory asString
%

category: 'Category'
method: Class
_classCategory

"Returns the classCategory of the receiver."

^ classCategory
%

category: 'Subclass Creation'
method: Class
_classDefiningClassVar: aSymbol

"Returns the receiver or the superclass that defines a class variable with
 name aSymbol, otherwise returns nil."

| aClass |
aClass := self .
[
  aClass _classVars ifNotNil:[ :cvs |
    ( cvs associationAt: aSymbol otherwise: nil ) ifNotNil:[
      ^ aClass
    ].
  ].
  aClass := aClass superClass .
  aClass == nil
] untilTrue .
^ nil
%

category: 'Class Membership'
method: Class
_classHistoryIncludesIdentical: aClass

 ^ (classHistory indexOfIdentical: aClass) ~~ 0
%

category: 'Accessing'
method: Class
_classInstVars

"Returns an Array of the receiver's class instance variables."

| civs |

civs := self class allInstVarNames.
civs removeFrom: 1 to: (self class superClass instSize).
^civs
%

category: 'Private'
method: Class
_classNamed: aString inDictionary: aDictionary
  | v sym |
  aDictionary ifNil:[ ^ nil ].
  (sym := Symbol _existingWithAll: aString) ifNil:[ ^ nil ].
  v := aDictionary at: sym otherwise: nil .
  ^ (v isKindOf: Class) ifTrue:[ v ] ifFalse:[ nil ].
%

category: 'Private'
method: Class
_classVarsChangableTo: anArray
  ^ self _classVarsChangableTo: anArray log: false
%

category: 'Private'
method: Class
_classVarsChangableTo: anArray log: logBool
  | ok |
  ok := true .
  anArray ifNotNil:[ | argSet |
    anArray _isArray ifFalse:[
      (anArray _validateClass: Array) ifFalse:[ ^ nil ].
    ].
    argSet := IdentitySet new .
    1 to: anArray size do: [:index| | aVarName aSym definingClass |
      aVarName := anArray at: index .
      (aVarName _isOneByteString or:[ aVarName _validateClass: CharacterCollection]) ifTrue:[
	aSym := aVarName asSymbol .
	aSym validateIsIdentifier "fix bug 9666" .
	definingClass := self _classDefiningClassVar: aSym .  "fix bug 10480"
	(definingClass ~~ nil and:[ definingClass ~~ self]) ifTrue:[  
           ok := false .
           logBool ifTrue:[ self _logServer: aSym , ' defined in ', definingClass name ].
        ].
        argSet add: aSym .  "exists or ok to add"
        anArray at: index put: aSym .
      ].
    ].
    classVars ifNotNil:[ :cvs | | toRemove |
      toRemove := cvs keys - argSet  .
      toRemove do:[:aKey |
        (classVars associationAt: aKey) isInvariant ifTrue:[ 
          ok := false .
          logBool ifTrue:[ self _logServer: aKey , ' an invariant class variable to remove '].
        ]
      ].
    ].
  ].
  ^ true
%

category: 'Private'
method: Class
_commentOrDescription: newComment
	"For backward compatibility, accept a GsClassDocumentation (although this is deprecated)
	or a string. Copy any GsClassDocumentation to avoid bug 41763."
newComment
	ifNotNil:
		[(newComment isKindOf: GsClassDocumentation)
			ifTrue:
				[| newDesc |
				newDesc := newComment copy.
				newDesc itsClass: self.
				self _description: newDesc]
			ifFalse: [self comment: newComment]].
%

category: 'Private'
method: Class
_definition
	"Returns a String. Sent by topaz "

	^self _definitionInContext: System myUserProfile
%

category: 'Browser Methods'
method: Class
_definitionInContext: aUserProfile

^ self _definitionInContext: aUserProfile withConstraints: false
%

category: 'Browser Methods'
method: Class
_definitionInContext: aUserProfile withConstraints: withConstraintsBool

"Returns a description of the receiver using object names taken from the given
 UserProfile."

| result newByteSubclass anArray lfsp
  firstElement poolDicts civs supercls aSize iVs |
supercls := self superClass .
result := String new.
result addAll: (supercls == nil ifTrue: ['nil'] ifFalse: [supercls name]).
newByteSubclass := false.
(lfsp := Character lf asString) addAll: '  ' .

(self isBytes and: [supercls isBytes not]) ifTrue: [
  result addAll: ' byteSubclass: '''; addAll: name; addLast: $'.
  newByteSubclass := true.
] ifFalse: [
  (self isIndexable and:[ supercls == nil or:[ supercls isIndexable not]]) ifTrue: [
    result addAll: ' indexableSubclass: '''; addAll: name; addLast: $'.
  ] ifFalse: [
    result addAll: ' subclass: '''; addAll: name; addLast: $'.
  ]
].
" instVarNames: #( <list of strings> ) "
iVs := self instVarNames .
(newByteSubclass not or:[ iVs size > 0 ]) ifTrue: [
  result addAll: lfsp;
    addAll: 'instVarNames: #(';
    addAll: (self _instVarNamesWithSeparator: (lfsp , '                 '));
    add: $).
].
" classVars: #( <list of strings> ) "
result addAll: lfsp; addLast: 'classVars: #('.
self _sortedClassVarNames accompaniedBy: result do: [:res :aKey |
  res addLast: $  .
  (aKey includesValue: $')
    ifTrue:[ res addAll: aKey _asSource ]
    ifFalse:[ res addAll: aKey ].
  ].
result addLast: $).

" classInstVars: #( <list of strings> ) "
result addAll: lfsp; addLast: 'classInstVars: #('.
civs := self class allInstVarNames.
civs removeFrom: 1 to: (self class superClass instSize).
civs accompaniedBy: result do: [:res :civName |
  res addLast: $  .
  (civName includesValue: $')
    ifTrue:[ res addAll: civName _asSource ]
    ifFalse:[ res addAll: civName ].
].
result addLast: $).

" poolDictionaries: { <list of dictionary names> } "
result addAll: lfsp; addAll: 'poolDictionaries: ' .
(poolDicts := self sharedPools) size > 0 ifTrue:[
  result addAll: '{ ' .
  firstElement := true.
  poolDicts do: [:each |
    firstElement ifFalse: [ result add: ' . '].
    anArray := aUserProfile dictionariesAndSymbolsOf: each.
    (aSize := anArray size) == 0 ifTrue:[ result add: '"(not named)"' ]
      ifFalse:[ aSize = 1 ifTrue:[ result add:((anArray at:1) at: 2) ]
                      ifFalse:[ result add: '"(multiple names)"' ]].
    firstElement := false.
  ].
  result add: ' }' .
] ifFalse:[
  result add: '#()'
].

" inDictionary: <name of containing dictionary> "
result addAll: lfsp; addAll: 'inDictionary: ' ;
   addAll: (self _dictionaryNameForFileout: aUserProfile) .

withConstraintsBool ifTrue:[
 (newByteSubclass not and: [self _hasConstraints]) ifTrue: [
    result add: self _definitionOfConstraints; lf
  ].
].

"options:"
result add:  lfsp; add: self _optionsStringForDefinition .
result add: Character lf .
^result
%

category: 'Browser Methods'
method: Class
_definitionOfConstraints

" Returns a string of the form
     constraints: { <Array of instance-variable-symbol/class-name pairs> }
 or
     constraints: <class name>

 As of GemStone 64bit v3.4, constraints are no longer implemented.
 This method is provided for examining classes in repositories upgraded from
 an older version."

| result  aConstraint firstElement |
result := String new.
result lf; add: '  constraints: '.
( constraints isKindOf: Array ) ifTrue: [
    result addAll: '{ '.
    firstElement := true.
    1 to: self instSize do: [ :x |
      aConstraint := constraints atOrNil: x .
      ((aConstraint ~~ nil _and: [aConstraint ~~ Object])
          _and:[ superClass == nil
            _or:[ (superClass _namedIvConstraintAt: x) ~~ aConstraint ]] )
      ifTrue: [
        " if not the first constraint, prefix with a period to separate
          from the last constraint "
        firstElement ifFalse: [
          result add: ' . '; lf; add: '                '
        ]
        ifTrue: [
          firstElement := false
        ].
        result add: '{ #'; add: (instVarNames at: x) ;
              add: ' . '; add: aConstraint name; addLast: $} .
      ]
    ].
    aConstraint:= self _varyingConstraint .
    ( (aConstraint ~~ Object) _and:
        [(superClass _varyingConstraint) ~~ aConstraint] )
    ifTrue:[
      firstElement ifFalse: [
          result add: ' . '; lf; add: '                '
      ]
      ifTrue: [
        firstElement := false
      ].
      result add: '   "the elements"  '; add: aConstraint name
    ].
    result add: ' }'.
  ]
  ifFalse: [
    constraints class class == Metaclass3 ifTrue: [
      result add: constraints name.
    ]
    ifFalse: [
      result add: ' nil'
    ].
  ].

^result
%

category: 'Private'
method: Class
_description
	"Returns the description of this class.
	Deprecated as of GS/64 3.1: subsumed by the new #comment field."

	self deprecated: 'Class>>_description deprecated v3.1. Replaced by comment'.
	^self _extraDictAt: #description
%

category: 'Private'
method: Class
_description: aDescription
	"Update the description of this Class.  Returns the argument.
 As of GS/64 3.1, this is deprecated in favor of #comment:"

	self deprecated: 'Deprecated as of GS/64 3.1'.
	self _extraDictAt: #description put: aDescription.
	^aDescription
%

category: 'Browser Methods'
method: Class
_dictionaryNameForFileout: aUserProfile
  | anArray |
  anArray := aUserProfile dictionariesAndSymbolsOf: self.
  (anArray size) == 0 ifTrue: [ | hist j |
    hist := self classHistory .
    anArray := nil .
    j := hist size .
    [ j >= 1 ] whileTrue:[
      anArray := aUserProfile dictionariesAndSymbolsOf: (hist at: j) .
      anArray size ~~ 0 ifTrue:[ j := 0"exit loop" ] .
      j := j - 1 .
    ]
  ].
  (anArray size) ~~ 0 ifTrue:[
    anArray := aUserProfile dictionariesAndSymbolsOf: ((anArray at: 1) at: 1).
    anArray size == 0 ifTrue:[
      ^ '(dictionary not in your dictionaries)'
    ] ifFalse: [ | dName |
      (dName := (anArray at: 1) at: 2) isValidIdentifier ifTrue: [
        ^ dName
      ] ifFalse: [  "this code moved from sessionmethods.topaz to here for v3.0"
        ^ ( '(GsSession currentSession symbolList objectNamed: ' , dName printString )
	       add: $) ; yourself .
      ]
    ]
  ].
  ^ 'UserGlobals' .
%

category: 'Private'
method: Class
_equivalentSubclass: oldClass
	superCls: actualSelf
	name: aString
	newOpts: optionsArray
	newFormat: theFormat
	newInstVars: anArrayOfInstvarNames
	newClassInstVars: anArrayOfClassInstVars
	newPools: anArrayOfPoolDicts
	newClassVars: anArrayOfClassVars
	inDict: aDictionary
	isKernel: isKernelBool

 "oldClass is equivalent to the subclass that would be created using
  the other arguments if:
     instVar names match exactly ,
     and class instVar names match exactly ,
     and the classVars in oldClass can be modified to add/remove Associations
        to match anArrayOfClassVars ,
     and pool dictionaries match exactly

  With respect to options and format, oldClass is equivalent if
    The state of format bits dbTransient, instancesInvariant match exactly ,
    and subclassesDisallowed cannot be set in the new subclass if it not set in oldClass ,
    and modifiable  cannot be set if it is not set in oldClass  ,
    and (SELF_CAN_BE_SPECIAL, NSC_DUPLICATES, INDEXABLE, IMPLEMENTATION, NO_STRUCT_UPDATE bits)
        of the formats must match exactly.

  If all other equivalence tests pass, the following changes to oldClass may be
  made to match the arguments and avoid creating a new subclass:
    instancesNonPersistent bit may be set (but not cleared) in the format of the oldClass
    subclassesDisallowed bit may be cleared (but not set) in format of oldClass
    traverseByCallback bit may be set or cleared in format of oldClass
    oldClass may be changed from modifiable to not modifiable (by sending immediateInvariant)
    classVars may be added to oldClass
    classVars having modifiable Associations may be removed from oldClass
"

	| oldOk fmtArr nam supr opts ivs civs poolds cvars cvarsArray logBool oldNam |
	fmtArr := self _validateOptions: optionsArray withFormat: theFormat newClassName: aString .
  logBool := fmtArr at: 3 .
	(oldClass isKindOf: Class) ifFalse: [oldClass _validateClass: Class].
	nam := (oldNam := oldClass name) asString = aString asString.
  nam ifFalse:[ | oldOop |
    oldOop := oldClass asOop .
    (oldOop <= 165121"OOP_CLASS_Special56bit15" and:[ oldOop >= 161281"OOP_CLASS_Special56bit0"])
      ifTrue:[ nam := true ].
  ].
	supr := oldClass superClass == actualSelf.
	opts := oldClass _optionsChangableTo: fmtArr.
	ivs := oldClass _instVarsEqual: anArrayOfInstvarNames.
	civs := oldClass class _instVarsEqual: anArrayOfClassInstVars.
	poolds := oldClass _poolDictsEqual: anArrayOfPoolDicts.
	cvars := oldClass
				_classVarsChangableTo: (cvarsArray := anArrayOfClassVars copy) log: logBool .
	oldOk := nam
				and: [supr and: [(opts==true) and: [ivs and: [civs and: [poolds and: [cvars]]]]]].
	oldOk ifTrue: [| oldVal newName |
			aDictionary ifNotNil: [
				newName := aString asSymbol.
				oldVal := aDictionary at: newName otherwise: nil.
        "allow nil as a value in aDictionary to satisfy forward refs during loading"
        (oldVal ~~ nil and:[ oldVal ~~ oldClass ]) ifTrue:[
             ImproperOperation
							signal: 'no new subclass needed, but aDictionary at: oldClass name ~~ oldClass'.
						^false
         ]
      ].
			oldClass _adjustOptions: optionsArray formatArr: fmtArr.
			oldClass _adjustClassVars: cvarsArray.
			logBool ifTrue:[self _logServer: 'class ' , aString , ' equivalent to requested class'].
			^true
    ] ifFalse:[ 
       logBool ifTrue:[
         self _logServer: '_equivalentSubclass false, nam:' , nam asString ,
                ' supr:' , supr asString , ' opts:' , opts asString ,
                ' ivs:' , ivs asString , ' civs:' , civs asString ,
                ' poolds:' , poolds asString , ' cvars:' , cvars asString]
    ].
	^false
%

category: 'Private'
method: Class
_extraDictAt: key

  "Return value for key in extraDict.
   Return nil if extraDict or the key are not present. "

  ^ extraDict
      ifNil: [ nil ]
      ifNotNil: [ :ed | ed at: key otherwise: nil ]
%

category: 'Private'
method: Class
_extraDictAt: key put: value

  "Add value for key to extraDict.  Create extraDict if not present. "

  extraDict ifNil: [
    extraDict := SymbolDictionary new.
    extraDict objectSecurityPolicy: self objectSecurityPolicy ].

  ^ extraDict at: key put: value
%

category: 'Private'
method: Class
_extraDictRemoveKey: key

  " Remove key/value from extraDict.
    Dont care if extraDict or the key itself are not present. "

  extraDict ifNotNil:[ :ed | ed removeKey: key ifAbsent: [] ].
%

category: 'Filein Support'
method: Class
_finishNewReservedClass: newClass old: oldCls resolv: resolveRes
  oldCls == nil ifTrue:[
    (((newClass class) superClass) == (self class)) ifFalse:[
       Error signal: 'Inconsistent class hierarchy'
    ].
     GsFile gciLogServer:'created class (reserved oop ', newClass asOop asString , ') : '  , newClass definition 
  ] ifFalse:[
     GsFile gciLogServer: 'class ', oldCls name asString, ' already exists '.
  ].
%

category: 'Private'
method: Class
_gbsTraversalCallback

"Private.  When GemBuilder Smalltalk traverses a Class, this method
 is called to return a description of the class."

^self definition
%

category: 'Browser Methods'
method: Class
_hasConstraints

"Determine if there are any constraints on any instance variable; return true if any
exist, false otherwise.
As of GemStone 64bit v3.4, constraints are no longer implemented.
This method is provided for examining classes in repositories upgraded from
an older version."

| aConstraint constrs |
constrs := constraints .
 constrs _isArray  ifTrue: [
    1 to: self instSize do: [:x |
      aConstraint := constrs atOrNil: x .
      ((aConstraint ~~ nil _and: [aConstraint ~~ Object])
          _and: [ superClass == nil
          _or: [ (superClass _namedIvConstraintAt: x) ~~ aConstraint ]] )
                   ifTrue: [^true]
        ].
    aConstraint := self _varyingConstraint .
    ( (aConstraint ~~ Object) _and:
        [(superClass _varyingConstraint) ~~ aConstraint] )
    ifTrue: [^true].
  ] ifFalse: [
    constraints class class == Metaclass3 ifTrue: [^true]
  ].

^false
%

category: 'Private'
method: Class
_insertCivAt: offset

"insert space for a new class instance variable at the specified offset.
 Each call will cause a markSweep GC of the VM temporary object memory."

<primitive: 486>

self _primitiveFailed: #_insertCivAt: args: { offset }.
self _uncontinuableError
%

category: 'Repository Conversion'
method: Class
_makeClassObsolete: aSymbolDictionary
"remove association from <aSymbolDictionary>
 rename receiver
 if assoc is invariant:
   create new association for obsolete class
 if not:
   reuse the original association for obsolete class
"

 | newName assoc |
 assoc := aSymbolDictionary associationAt: self name.
 newName := ('Obsolete', self name asString) asSymbol.
 aSymbolDictionary removeAssociation: assoc.
 self _unsafeAt: 11 put: newName.
 assoc isInvariant
   ifTrue: [ aSymbolDictionary at: self name asSymbol put: self ]
   ifFalse: [
      assoc key: self name asSymbol.
      aSymbolDictionary addAssociation: assoc ].
 self removeAllMethods.
 self class removeAllMethods.
%

category: 'Modifying Classes'
method: Class
_makeInstancesDbTransient: aBool

<protected>
self _validatePrivilege ifTrue:[
  aBool ifTrue:[
    (self isPointers and:[ self isIndexable not]) ifFalse:[
      ^ ImproperOperation new details:'Only non-indexable pointer objects may be DbTransient';
           object: self ; signal
    ].
    format := format bitOr: 16r1000 .
  ] ifFalse:[
    format := format bitAnd: (16r1000 bitInvert)
  ].
  self _refreshClassCache: false .
].
%

category: 'Private'
method: Class
_makeInstancesNonPersistent

<protected>
self _validatePrivilege ifTrue:[
  format := format bitOr: 16r800 .
  self _refreshClassCache: false .
].
%

category: 'Private'
method: Class
_makeInstancesPersistent

<protected>
self _validatePrivilege ifTrue:[
  self _validateInstancesPersistent ifTrue:[
    self superClass instancesNonPersistent ifTrue:[
      ^ ImproperOperation new reason: #rtErrSuperclassIsNP;
	details: 'superclass is non-persistent';
	object: self ; signal .
    ]
  ]
].
format := (format bitOr: 16r800 ) bitXor: 16r800 .
self _refreshClassCache: false .
%

category: 'Browser Methods'
method: Class
_modifiableDefinitionInDictionary: dict named: dictName

"Returns a description of the receiver that is modifiable,
 and that places the class in the given dictionary name.
 Byte classes will not use the modifiable form of class definition as
 they have no information that can be modified."

  | result newByteSubclass lfsp firstElement poolDicts cat nm resolver supercls |

  supercls := self superClass .
  nm := supercls == nil ifTrue: [ 'nil' ] ifFalse: [
      dict keyAtValue: supercls ifAbsent: [ supercls name ]
      ].
  (result := String new) add: $( ; addAll: nm .
  newByteSubclass := false.
  lfsp := (Character lf asString) addAll: '  '.

  (self isBytes and: [supercls isBytes not]) ifTrue:
    [
    result addAll: ' byteSubclass: '.
    newByteSubclass := true.
    ]
  ifFalse:
    [
    (self isIndexable and: [supercls isIndexable not]) ifTrue:
      [ result addAll: ' indexableSubclass: ' ]
    ifFalse:
      [ result addAll: ' subclass: ' ].
    ].

  nm := dict keyAtValue: self ifAbsent: [ self name ].
  result addAll: (String withAll: nm) quoted.

 " instVarNames: #( <list of strings> ) "
  newByteSubclass ifFalse: [
    result addAll: lfsp;
      addAll: 'instVarNames: #(';
      addAll: (self _instVarNamesWithSeparator: (lfsp , '                 '));
      add: $).
  ].

  " classVars: #( <list of strings> ) "

  result addAll: lfsp; addLast: 'classVars: #('.
  self _sortedClassVarNames accompaniedBy: result do: [:res :aKey |
    res addLast: $  .
    (aKey includesValue: $')
      ifTrue:[ res addAll: aKey _asSource ]
      ifFalse:[ res addAll: aKey ].
    ].

  result addLast: $).

  " classInstVars: #( <list of strings> ) "

  result addAll: lfsp; addLast: 'classInstVars: #('.
  self _classInstVars accompaniedBy: result do:[ :res :civName |
    res addLast: $  .
    (civName includesValue: $')
      ifTrue:[ res addAll: civName _asSource ]
      ifFalse:[ res addAll: civName ].
    ].
  result addLast: $).

  " poolDictionaries: { <list of dictionary names> } "

  result addAll: lfsp; addAll: 'poolDictionaries: '.
  (poolDicts := self sharedPools ) size > 0 ifTrue:[
    firstElement := true.
    resolver := System myUserProfile .
    result add: ' {' .
    poolDicts do: [:each | | anArray aSize |
      firstElement ifFalse: [ result add: ' . '].
      anArray := resolver dictionariesAndSymbolsOf: each.
      (aSize := anArray size) == 0 ifTrue:[ result add: '"(not named)"' ]
        ifFalse:[ aSize = 1 ifTrue:[ result add:((anArray at:1) at: 2) ]
                      ifFalse:[ result add: '"(multiple names)"' ]].
      firstElement := false.
    ].
    result add: ' }' .
  ] ifFalse:[
    result add: '#()'
  ].

  " inDictionary: <name of containing dictionary> "
  result addAll: lfsp; addAll: 'inDictionary: '; addAll: dictName.

  "constraints omitted"

  "options:"
  result add:  lfsp; add: self _optionsStringForDefinition .

  result add: $) .
  (cat := classCategory ) ifNotNil:[
    result addAll: ' category: '; addAll: (String withAll: cat) quoted
  ].
  result addLast: Character lf .
  ^result
%

category: 'Filein Support'
method: Class
_newKernelByteSubclass: clsName
classVars: classVarArg
poolDictionaries: poolDicts
inDictionary: aDict
options: options
reservedOop: reservedOopNum

	"Old (pre-GS64) behavior:
   Created a new class always, to allow upgrade
   scripts to change the definition of a kernel class.  Preserved
   the identity of both the class and the metaClass if the class
   already existed.

 New (Gemstone64) behavior:
   Does not allow redefinition of a
   class in the upgrade case.  Only create class if it does not exist
   by the given name in globals.  The _unsafeSet11Oop primitive does
   not currently allow changing the identity of an already existing
   object.

 Returns the new or existing class .

 If reservedOopNum is ~~ nil, reservedOopNum must be a positive Integer
 that is a legal gs64v1.1 objectId and the methods   Object >> _unsafeSetOop:
 and   Object(C) >> _objectForOop:   must be installed."

  self ifNil:[ Error signal:'superclass not yet defined'].
	self _validatePrivilege ifTrue:
			[| newClass className oldCls resolveRes dictNames result |
      dictNames := #( ObsoleteClasses GsCompilerClasses GemStone_Legacy_Streams).
			(aDict == Globals or:[ dictNames includesIdentical: aDict name])
				ifFalse:[ aDict _error: #rtErrInvalidArgument.  ^nil].
			className := Symbol withAll: clsName.
			resolveRes := Class _resolveReservedClass: reservedOopNum name: className.
			oldCls := resolveRes at: 1.
			oldCls ifNil:[
        GsObjectSecurityPolicy setCurrent: Object objectSecurityPolicy while:[
           newClass := self
										byteSubclass: className
										classVars: classVarArg
										classInstVars: #()
										poolDictionaries: poolDicts
										inDictionary: aDict
										newVersionOf: nil
										description: nil
										options: options.
							reservedOopNum ifNotNil:[
								newClass := newClass _unsafeSetOop: reservedOopNum .
							  (aDict associationAt: className) immediateInvariant ].
         ].
         result := newClass .
       ] ifNotNil:[ | fmt |
					fmt := (format bitAnd: 16r3 bitInvert) bitOr: 16r1 + 16r4.
					(self
						_equivalentSubclass: oldCls
						superCls: self
						name: className
						newOpts: ((options copy)
								add: #logCreation;
								yourself)
						newFormat: fmt
						newInstVars: #()
						newClassInstVars: #()
						newPools: poolDicts
						newClassVars: classVarArg
						inDict: aDict
						isKernel: true)
							ifFalse: [ArgumentError signal: 'existing class not equivalent to arguments'].
          result := oldCls .
      ].
			self _finishNewReservedClass: newClass old: oldCls resolv: resolveRes .
      ^ result
  ].
	^ nil
%

category: 'Filein Support'
method: Class
_newKernelIndexableSubclass: clsName
instVarNames: ivArg
classVars: classVarArg
classInstVars: classInstVarArg
poolDictionaries: poolDicts
inDictionary: aDict
options: options
reservedOop: reservedOopNum

	"Used by filein.
Old (pre-GS64) behavior:
   Created a new class always, to allow upgrade
   scripts to change the definition of a kernel class.  Perserved
   the identity of both the class and the metaClass if the class
   already existed.
 New (Gemstone64) behavior:
   Does not allow redefinition of a
   class in the upgrade case.  Only create class if it does not exist
   by the given name in globals.  The _unsafeSet11Oop primitive does
   not currently allow changing the identity of an already existing
   object.

 Returns the new class, the old class or nil

 If reservedOopNum is ~~ nil, reservedOopNum must be a positive Integer
 that is a legal gs64v1.1 objectId and the methods   Object >> _unsafeSet11Oop:
 and   Object(C) >> _objectForOop:   must be installed."

  self ifNil:[ Error signal:'superclass not yet defined'].
	self _validatePrivilege
		ifTrue:
			[| newClass className oldCls resolveRes fmt dictNames result |
      dictNames := #( ObsoleteClasses GsCompilerClasses GemStone_Legacy_Streams).
			(aDict == Globals or:[ dictNames includesIdentical: aDict name])
					ifFalse:
					[aDict _error: #rtErrInvalidArgument.
					^false].
			self isBytes
				ifTrue:
					[^clsName _error: #classErrBadFormat
						with: 'cannot create indexable subclass of byte class'].
			self isNsc
				ifTrue:
					[^clsName _error: #classErrBadFormat
						with: 'cannot create indexable subclass of Nsc class'].
			fmt := format bitOr: 16r4.	"add indexable bit"
			className := Symbol withAll: clsName.
			resolveRes := Class _resolveReservedClass: reservedOopNum name: className.
			oldCls := resolveRes at: 1.
			oldCls ifNil:[
          aDict at: className put: nil.
					GsObjectSecurityPolicy setCurrent: Object objectSecurityPolicy while:[
            newClass := self
										_subclass: className
										instVarNames: ivArg
										format: fmt
										classVars: classVarArg
										classInstVars: classInstVarArg
										poolDictionaries: poolDicts
										inDictionary: aDict
										inClassHistory: nil
										description: nil
										options: options.
							reservedOopNum ifNotNil:[
								newClass := newClass _unsafeSetOop: reservedOopNum .
							  (aDict associationAt: className) immediateInvariant ].
           ].
           result := newClass .
         ] ifNotNil: [(self
						_equivalentSubclass: oldCls
						superCls: self
						name: className
						newOpts: ((options copy)
								add: #logCreation;
								yourself)
						newFormat: fmt
						newInstVars: ivArg
						newClassInstVars: classInstVarArg
						newPools: poolDicts
						newClassVars: classVarArg
						inDict: aDict
						isKernel: true)
							ifFalse: [ArgumentError signal: 'existing class not equivalent to arguments'].
           result := oldCls
         ].
			self _finishNewReservedClass: newClass old: oldCls resolv: resolveRes .
      ^ result
      ].
	^false
%

category: 'Filein Support'
method: Class
_newKernelSubclass: clsName
instVarNames: ivArg
classVars: classVarArg
classInstVars: anArrayOfClassInstVars
poolDictionaries: poolDicts
inDictionary: aDict
options: options
reservedOop: reservedOopNum

^ self _newKernelSubclass: clsName
    subclassOf: self
    instVarNames: ivArg
    classVars: classVarArg
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: poolDicts
    inDictionary: aDict
    options: options
    reservedOop: reservedOopNum
%

category: 'Filein Support'
method: Class
_newKernelSubclass: clsName
subclassOf: actualSelf
instVarNames: ivArg
classVars: classVarArg
classInstVars: anArrayOfClassInstVars
poolDictionaries: poolDicts
inDictionary: aDict
options: options
reservedOop: reservedOopNum

	"Old (pre-GS64) behavior:
   Created a new class always, to allow upgrade
   scripts to change the definition of a kernel class.  Preserved
   the identity of both the class and the metaClass if the class
   already existed.

 New (Gemstone64) behavior:
   Does not allow redefinition of a
   class in the upgrade case.  Only create class if it does not exist
   by the given name in globals.  The _unsafeSet11Oop primitive does
   not currently allow changing the identity of an already existing
   object.

 Returns a String, either the definition of the new class or a
 message that the class already exists.

 If reservedOopNum is ~~ nil, reservedOopNum must be a positive Integer
 that is a legal gs64v1.1 objectId and the methods   Object >> _unsafeSet11Oop:
 and   Object(C) >> _objectForOop:   must be installed."
  self ifNil:[ Error signal:'superclass not yet defined'].
	self _validatePrivilege ifTrue:[
     | newClass className oldCls resolveRes dictNames result |
      dictNames := #( ObsoleteClasses GsCompilerClasses GemStone_Legacy_Streams).
			(aDict == Globals or:[ dictNames includesIdentical: aDict name])
				ifFalse:
					[aDict _error: #rtErrInvalidArgument.
					^false].
			className := Symbol withAll: clsName.
			resolveRes := Class _resolveReservedClass: reservedOopNum name: className.
			oldCls := resolveRes at: 1.
			oldCls ifNil:[
          aDict at: className put: nil.
					GsObjectSecurityPolicy setCurrent: Object objectSecurityPolicy while:[
            newClass := self
										_subclass: className
										instVarNames: ivArg
										format: format
										classVars: classVarArg
										classInstVars: anArrayOfClassInstVars
										poolDictionaries: poolDicts
										inDictionary: aDict
										inClassHistory: nil
										description: nil
										options: options.
							reservedOopNum ifNotNil:[ "change object identifier"
								newClass := newClass _unsafeSetOop: reservedOopNum .
							  (aDict associationAt: className) immediateInvariant 
              ].
             ].
             result := newClass .
       ] ifNotNil: [
          (self
						_equivalentSubclass: oldCls
						superCls: actualSelf
						name: className
						newOpts: ((options copy)
								add: #logCreation;
								yourself)
						newFormat: oldCls format
						newInstVars: ivArg
						newClassInstVars: anArrayOfClassInstVars
						newPools: poolDicts
						newClassVars: classVarArg
						inDict: aDict
						isKernel: true)
							ifFalse: [ArgumentError signal: 'existing class not equivalent to arguments'].
         result := oldCls .
       ].
			self _finishNewReservedClass: newClass old: oldCls resolv: resolveRes .
      ^ result
   ].
	^ nil
%

category: 'Browser Methods'
method: Class
_optionsArray
  | result optCount swiz |
  result := { } .
  optCount := 0 .
  self instancesDbTransient ifTrue:[ result add: #dbTransient . optCount := optCount + 1 ].
  self _structuralUpdatesDisallowed ifTrue:[ result add: #disallowGciStore  ].
  self instancesInvariant ifTrue:[ result add:  #instancesInvariant  . optCount := optCount + 1 ].
  self instancesNonPersistent ifTrue:[ result add:  #instancesNonPersistent  . optCount := optCount + 1 ].
  optCount > 1 ifTrue:[
    self _error: #classErrBadFormat
        with:'only one of #dbTransient #instancesNonPersistent  #instancesInvariant allowed' .
  ].
  self isModifiable ifTrue:[ result add: #modifiable  ].
  self selfCanBeSpecial ifTrue:[ result add: #selfCanBeSpecial  ].
  self subclassesDisallowed ifTrue:[ result add: #subclassesDisallowed  ].
  self _traversalByCallback ifTrue:[ result add: #traverseByCallback  ].
  swiz := (format bitShift: -31"GC_BEHAV_byteSwizKind_shift") bitAnd: 3"GC_BEHAV_byteSwizKind_mask" .
  swiz > 0 ifTrue:[ | signed |
    signed := (format bitShift: -33"GC_BEHAV_signedWords_shift") bitAnd: 1 .
    signed == 0 ifTrue:[ result add: ( #( #'2byteWords' #'4byteWords' #'8byteWords' ) at: swiz)]
             ifFalse:[ result add: ( #( #'signed2byteWords' #'signed4byteWords' #'signed8byteWords' ) at: swiz) ]
  ].
  "See also  Class >> _rwOptionsArray  in Rowan ."
  ^ result
%

category: 'Browser Methods'
method: Class
_optionsArrayForDefinition
  "Returns an Array like that from Class>>_optionsArray for use in
   recreating the receiver or exporting a definition to Rowan files."
  | inheritable myOpts supOpts myInheritable supInheritable notInher result |
  inheritable := IdentitySet withAll: #( subclassesDisallowed disallowGciStore traverseByCallback
                                         dbTransient instancesNonPersistent instancesInvariant ).
  myOpts := IdentitySet withAll: self _optionsArray .
  supOpts := IdentitySet new .
  self superclass ifNotNil:[:sc | supOpts addAll: sc _optionsArray ].
  myInheritable := myOpts * inheritable .
  supInheritable := supOpts * inheritable .
  notInher := supInheritable - myInheritable .
  notInher size > 0 ifTrue:[
    result := { #noInheritOptions } .
    result addAll: (SortedCollection withAll: myOpts) .
  ] ifFalse:[
    myOpts := myOpts - inheritable + (myInheritable - supInheritable) .
    result := Array withAll: (SortedCollection withAll: myOpts).
  ].
  ^ result
%

category: 'Private'
method: Class
_optionsChangableTo: formatArr

  "formatArr describes format and options that would be produced by
   a new class creation.
   Returns true or a String describing why a new version of the class would be required."
  | fmt newFmt str list |
  fmt := format .
  str := String new .
  newFmt := formatArr at: 1 .
  ((fmt bitAnd: 16r800) ~~ 0 and:[ (newFmt bitAnd: 16r800) == 0 ]) ifTrue:[
     str addAll: 'instancesNonPersistent cannot be cleared,' 
  ].
  list := #( 16r8 instancesInvariant 16r1000 'dbTransient'
             16r40 'nscDuplicates' 16r7 implementationFormat ).
  1 to: list size by: 2 do:[:j | | mask |
    mask := list at: j .
    (fmt bitAnd: mask) ~~ (newFmt bitAnd: mask) ifTrue:[
      str addAll:'difference in ', (list at: j+1),', '.
    ].
  ].
  ((fmt bitAnd: 16r800) == 0 and:[ (newFmt bitAnd: 16r800) ~~ 0 ]) ifTrue:[
     "setting instancesNonPersistent"
     (newFmt bitAnd: 16r1008) ~~ 0 ifTrue:[
       str addAll:
       'instancesNonPersistent not allowed with dbTransient or instancesInvariant,'.
     ].
  ].
  ((fmt bitAnd: 16r20) == 0 and:[ (newFmt bitAnd: 16r20) ~~ 0 ]) ifTrue:[
     str addAll:'cannot set subclassesDisallowed if previously cleared,'.
  ].
  self isModifiable ifFalse:[
    (formatArr at: 2 "modifiableBool") ifTrue:[ str addAll:'cannot reenable modifiable,' ].
  ].
  ((fmt bitOr: newFmt) bitAnd: 16r7FFFC000) ~~ 0 ifTrue:[
     str addAll:'isMetaClass,' 
  ].
  (fmt bitShift: -31) = (newFmt bitShift: -31) ifFalse:[ "GC_BEHAV_byteSwizKind_shift"
     str addAll:'difference in byteSwizKind,'.
  ].
  str size == 0 ifTrue:[ ^ true ].
  ^ str.
%

category: 'Browser Methods'
method: Class
_optionsStringForDefinition
  | result arr |
  result :=  'options: #(' copy .
  arr := self _optionsArrayForDefinition . "fix 48681"
  1 to: arr size do:[:j | | sym |
    result add: $  .
    sym := arr at: j .
    (sym at: 1) isDigit ifTrue:[ result add: $#; add: $' ; add: sym ; add: $' ]
                      ifFalse:[ result add: sym ].
  ].
  result add: $)  .
  ^ result
%

category: 'Private'
method: Class
_poolDictsEqual: anArray
  "pool dictionaries should be identical to ensure that compiled-in Associations can be shared"
  "fix bug 42279"

  | sharedPoolsArray |
  sharedPoolsArray := self sharedPools.
  sharedPoolsArray size ~= anArray size ifTrue: [ ^false ].
  1 to: sharedPoolsArray size do: [:index |
    (sharedPoolsArray at: index) == (anArray at: index) ifFalse: [ ^false ]].
  ^true
%

category: 'Private'
method: Class
_recompileAllMethods

"Unconditionally recompile all env 0 methods."

super _recompileAllMethods .
self class _recompileAllMethods
%

category: 'Private'
method: Class
_removeClassVar: aSymbol

"Remove the class variable named aSymbol from the receiver.
 Signals an error if there is no such class variable."

self _validatePrivilege ifTrue:[
  (Symbol _existingWithAll: aSymbol) ifNotNil:[ :aSym |
    classVars ifNotNil:[ :cvs | | assoc |
      assoc := cvs associationAt: aSym otherwise: nil .
      assoc ifNotNil:[
         cvs removeKey: aSym  .
         ^ self
      ].
    ].
  ].
  ^ LookupError new reason: #classErrClassVarNotFound; key: aSymbol ; object: self;
	signal
].
%

category: 'Modifying Classes'
method: Class
_setClassVars: aDict old: previousDict

  classVars ~~ aDict ifTrue:[
    previousDict ~~ classVars ifTrue:[ self error:'invalid store to classVars'].
    classVars := aDict .
    self class _setClassVars: aDict old: previousDict
  ].
%

category: 'Browser Methods'
method: Class
_signedWords

  "Return true if instances of the receiver are byte format with signed words,
   false otherwise."

  self isBytes ifTrue:[
     ^ ((format bitShift: -33"GC_BEHAV_signedWords_shift") bitAnd: 1) == 1
  ].
  ^ false .
%

category: 'Subclass Creation'
method: Class
_subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
newVersionOf: oldClass
description: aDescription
options: optionsArray

 "optionsArray is an Array of Symbols containing zero or more of
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable ,
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.

 Returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

	| hist descr |
	descr := aDescription.
	oldClass
		ifNotNil:
			[ (self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: optionsArray
				newFormat: oldClass format
				newInstVars: anArrayOfInstvarNames
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false)
					ifTrue:
						[oldClass _commentOrDescription: aDescription.
						^oldClass	"avoid creation of a new version"].
			hist := oldClass classHistory.
			descr
				ifNil:
					[descr := [oldClass commentForFileout] on: Error
								do: [:ex | 'old comment not available']]].
	^self
		_subclass: aString
		instVarNames: anArrayOfInstvarNames
		format: format
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: optionsArray
%

category: 'Modifying Classes'
method: Class
_subclasses

"Returns the class variable that is the list of subclasses, or
 nil if the receiver does not keep track of subclasses."

^ subclasses
%

category: 'Modifying Classes'
method: Class
_subclasses: anIdentitySet

"Modifies the class variable that is the list of subclasses."

self _validatePrivilege ifTrue:[
  anIdentitySet ifNotNil:[ (anIdentitySet _validateClass: IdentitySet) ifFalse:[ ^ nil ]].
  subclasses := anIdentitySet
].
%

category: 'Private'
method: Class
_validateOptions: optionsArray  withFormat: theFormat newClassName: subClsName
  "theFormat is requested format and optionsArray is
   requested options for a new class .
   Returns an Array,
    { (theFormat with possible bits added) .  modifiableBBoolean . logBoolean } .
   signals an Error if the optionsArray contains an unregconized option."
| iOptCount fmt modifiableBool logBool swizSym nSwiz signed |
modifiableBool := false .
logBool := false .
nSwiz := 0 .
fmt := theFormat bitAnd: 16r2000 bitInvert "never inherit selfCanBeSpecial" .
iOptCount := 0 .
1 to: optionsArray size do:[:j | | oSym |
  oSym := optionsArray at: j .
  oSym == #noInheritOptions ifTrue:[
    j == 1 ifFalse:[ subClsName _error: #classErrBadFormat with: '#noInheritOptions must be first element of options'].
    "do not inherit dbTransient,instancesNonPersistent,instancesInvariant,subclassesDisallowed,
                    disallowGciStore,traverseByCallback "
    fmt := fmt bitAnd: (16r1E28  bitInvert) ] ifFalse:[
  oSym == #dbTransient ifTrue:[ iOptCount := iOptCount + 1 . fmt := fmt bitOr: 16r1000 ] ifFalse:[
  oSym == #instancesNonPersistent  ifTrue:[ iOptCount := iOptCount + 1 . fmt := fmt bitOr: 16r800 ] ifFalse:[
  oSym == #instancesInvariant ifTrue:[ iOptCount := iOptCount + 1 . fmt := fmt bitOr: 16r8 ] ifFalse:[
  oSym == #subclassesDisallowed ifTrue:[ fmt := fmt bitOr: 16r20 ] ifFalse:[
  oSym == #disallowGciStore ifTrue:[ fmt := fmt bitOr: 16r200 ] ifFalse:[
  oSym == #traverseByCallback ifTrue:[ fmt := fmt bitOr: 16r400 ] ifFalse:[
  oSym == #selfCanBeSpecial ifTrue:[ fmt := fmt bitOr: 16r2000 ] ifFalse:[
  oSym == #modifiable ifTrue:[ modifiableBool := true ] ifFalse:[
  oSym == #logCreation ifTrue:[ logBool := true ] ifFalse:[
  oSym == #'2byteWords' ifTrue:[ swizSym := oSym  . nSwiz := nSwiz + 1 . signed:=false ] ifFalse:[
  oSym == #'4byteWords' ifTrue:[ swizSym := oSym  . nSwiz := nSwiz + 1 . signed:=false ] ifFalse:[
  oSym == #'8byteWords' ifTrue:[ swizSym := oSym  . nSwiz := nSwiz + 1 . signed:=false ] ifFalse:[
  oSym == #'signed2byteWords' ifTrue:[ swizSym := oSym  . nSwiz := nSwiz + 1 . signed:=true ] ifFalse:[
  oSym == #'signed4byteWords' ifTrue:[ swizSym := oSym  . nSwiz := nSwiz + 1 . signed:=true ] ifFalse:[
  oSym == #'signed8byteWords' ifTrue:[ swizSym := oSym  . nSwiz := nSwiz + 1 . signed:=true ] ifFalse:[
    subClsName _error: #classErrBadFormat with: 'unrecognized option ' , oSym printString
  ]]]]]]]]]]]]]]]].
  "See also  Class >> _rwOptionsArray  in Rowan ."
].
logBool ifFalse:[  "fix 50214"
  (SessionTemps current at:#GsClass_logCreation otherwise: nil) == true ifTrue:[ logBool := true ].
].
iOptCount > 1 ifTrue:[
  subClsName _error: #classErrBadFormat
        with:'only one of #dbTransient #instancesNonPersistent  #instancesInvariant allowed' .
].
nSwiz > 1 ifTrue:[
  subClsName _error: #classErrBadFormat
        with: 'only one of #''2byteWords'' #''4byteWords'' #''8byteWords'' #''signed2byteWords'' #''signed4byteWords'' #''signed8byteWords'' allowed'
].
swizSym ifNotNil:[
  (fmt bitAnd: 3) ~~ 1  ifTrue:[
    subClsName _error: #classErrBadFormat
        with:'A *byteWords options wa specified but new class would not be byte format'.
  ].
  fmt := self _addByteSwizzle: swizSym toFormat: fmt newClassName: subClsName signed: signed .
].
^ { fmt . modifiableBool . logBool  }
%

category: 'Subclass Creation'
method: Class
_validName: str

"Returns whether the given string contains Characters valid for a class name.
 The string need not be a Symbol."

| ch |

(str size == 0 or: [str size > 64]) ifTrue: [
  ^false
].

ch := str at: 1.
(ch == $_  or: [ch isLetter]) ifFalse: [
  ^false
].

2 to: str size do: [ :i |
  ch := str at: i.
  (ch == $_  or: [ch isAlphaNumeric]) ifFalse: [
    ^false
  ].
].

^true.
%

category: 'Browser Methods'
method: Class
_versionedName
"used by topaz"

^ [ self versionedName ] onException: Error do:[:ex| ^ self describe]
%

category: 'Private'
method: Class
__definition
	"Returns a String.
	For insteractive use to show constraints in a class
  in an upgraded repository."

	^self _definitionInContext: System myUserProfile withConstraints: true
%

category: 'Private'
method: Class
__makeVariant

"Makes the receiver variant."

<primitive: 272>
self _primitiveFailed: #__makeVariant .
self _uncontinuableError
%

! Class extensions for 'ClassHistory'

!		Class methods for 'ClassHistory'

removeallmethods ClassHistory
removeallclassmethods ClassHistory

category: 'Instance Creation'
classmethod: ClassHistory
new

"Create a new ClassHistory."

| result |
result := super new .
result _setNoStubbing  .
^ result
%

category: 'Instance Creation'
classmethod: ClassHistory
new: anInt

"Disallowed"

self shouldNotImplement: #new: .
self _uncontinuableError
%

category: 'Instance Creation'
classmethod: ClassHistory
with: anObj

"Disallowed"

self shouldNotImplement: #with: .
self _uncontinuableError
%

category: 'Instance Creation'
classmethod: ClassHistory
with: anObj with: obj2

"Disallowed"

self shouldNotImplement: #with:with: .
self _uncontinuableError
%

category: 'Instance Creation'
classmethod: ClassHistory
with: anObj with: obj2 with: obj3

"Disallowed"

self shouldNotImplement: #with:with:with: .
self _uncontinuableError
%

category: 'Instance Creation'
classmethod: ClassHistory
with: anObj with: obj2 with: obj3 with: obj4

"Disallowed"

self shouldNotImplement: #with:with:with:with: .
self _uncontinuableError
%

category: 'Instance Creation'
classmethod: ClassHistory
withAll: aCollection

"Disallowed"

self shouldNotImplement: #withAll: .
self _uncontinuableError
%

!		Instance methods for 'ClassHistory'

category: 'Accessing'
method: ClassHistory
at: aTimeOrIndex

"Returns the Class that was current at the given time.  The time may be
 specified absolutely using a DateTime, or relatively using an integer.  If a
 DateTime is specified, returns the version of the class that was active at
 that time, or nil if the time is before the earliest version.

 If an Integer is specified, it is used to chronologically select the
 version, with 1 indicating the first version created, 2 the version, and so on.
 If the index is less than one or greater than the number of versions in
 the history, an error is generated."

| candidate aClass|

(aTimeOrIndex class == DateTime)
  ifFalse: [ ^ super at: aTimeOrIndex ]
  ifTrue: [
    "candidate is initially nil.  Iterate through the history from
    the beginning.  If the entry's time is before the specified time,
    it is possibly the class that was active so save it as the best
    candidate.  If the entry's time is after the specified time, it
    cannot be the answer and no subsequent ones can be, so returns the
    current candidate."

    1 to: self size do: [ :j |
      aClass := self at: j.
      (aClass timeStamp <= aTimeOrIndex)
        ifTrue: [ candidate := aClass ]
        ifFalse: [ ^ candidate ].
    ].
    ^ candidate
  ]
%

category: 'Accessing'
method: ClassHistory
current

"Returns the current, or most recent class."

^ super at: self size
%

category: 'Accessing'
method: ClassHistory
currentVersion

"Returns the most recent version in the receiver's collection of versions."

| sz |
(sz := self size) == 0 ifTrue: [ ^nil ].
^self at: sz
%

category: 'Accessing'
method: ClassHistory
description

"Returns the description of this ClassHistory."

^ description
%

category: 'Updating'
method: ClassHistory
description: aString

"Updates the description of this ClassHistory."

description := aString
%

category: 'Accessing'
method: ClassHistory
name

"Returns the name of this ClassHistory."

^ name
%

category: 'Updating'
method: ClassHistory
name: aString

"Updates the name of this ClassHistory."

name := aString asString asSymbol
%

category: 'Updating'
method: ClassHistory
newVersion: aClass

"Installs the given class as the receiver's most current version.  Does not
 install the receiver in the given class as its version history.
 Returns the class object.

 Generates an error if the receiver's basicSize would exceed 2034 "

| currSize idxSize |
currSize := self basicSize .
currSize >= 2034 ifTrue:[   "virtual machine constant"
  "VM restriction, classHistory cannot be a large object"
  idxSize := self size .
  self _error: #objErrMaxSize args:{ idxSize . idxSize + 1 }.
  self _uncontinuableError
  ]
ifFalse:[
  ^ self add: aClass
  ]
%

category: 'Updating'
method: ClassHistory
removeVersion: aClass

"Removes the given class from the receiver's list of versions."

| idx |
idx := self indexOf: aClass.
idx ~~ 0 ifTrue: [
  self removeFrom: idx to: idx
].
%

category: 'Class Membership'
method: ClassHistory
species

"Returns a class similar to, or the same as, the receiver's class which
 can be used for containing derived copies of the receiver."

^ Array
%

! Class extensions for 'ClientForwarderSend'

!		Instance methods for 'ClientForwarderSend'

removeallmethods ClientForwarderSend
removeallclassmethods ClientForwarderSend

category: 'Handling'
method: ClientForwarderSend
defaultAction

"Return an error to the controlling debugger or GCI client. 
 If to GCI, stack is saved and available as an argument to the GCI error struct.

 instVars of the receiver go into the GCI error struct as follows
 self.receiver -->  GciErrSType.args[0]
 self.clientObj --> GciErrSType.args[1]
 self.selector -->  GciErrSType.args[2]
 self.gsArgs -->    GciErrSType.args[3]   (instVar defined in AbstractException)
"

^ self _signalToDebugger 
%

category: 'Instance initialization'
method: ClientForwarderSend
initialize
  gsNumber := ERR_ClientForwarderSend.
  gsResumable := true .
  gsTrappable := false . "goes to the controlling debugger or GCI client"
%

category: 'Instance initialization'
method: ClientForwarderSend
receiver: aSelf clientObj: aClientObj selector: aSymbol args: anArray
  receiver := aSelf .
  clientObj := aClientObj .
  selector := aSymbol .
  gsArgs := anArray
%

! Class extensions for 'ClusterBucket'

!		Class methods for 'ClusterBucket'

removeallmethods ClusterBucket
removeallclassmethods ClusterBucket

category: 'Listing Instances'
classmethod: ClusterBucket
allInstances

"Returns the collection of all instances of the receiver."

^ AllClusterBuckets
%

category: 'Accessing'
classmethod: ClusterBucket
bucketWithId: aPositiveSmallInt

"Returns the instance with the specified ID if one exists.  Generates an error
 if anInt is less than 1 or outside of the range of existing cluster buckets."

^ AllClusterBuckets at: aPositiveSmallInt
%

category: 'Instance Creation'
classmethod: ClusterBucket
new

"Creates an instance of the receiver and adds the new instance to
 AllClusterBuckets."

| newId result max |
max := System _maxClusterId .
(AllClusterBuckets size >= max) ifTrue: [
  self _error: #rtErrMaxClusterId args: { max }.
  ^ AllClusterBuckets at: 1 "returns default bucket"
  ] .
newId := AllClusterBuckets size + 1 .
result := super new .
result _clusterId: newId ;
       keepClusteredOnModify: false ;
  "create an empty string for the description, so description can be
   modified later without putting the ClusterBucket in a write set."
       description: String new .

AllClusterBuckets addLast: result .
^ result
%

category: 'Instance Creation'
classmethod: ClusterBucket
newForExtent: extentId

"DEPRECATED.
 In Gemstone64 v2.0, the relationship between a ClusterBucket and
 an extent is weak.  Clustering an object into a ClusterBucket will
 attempt to use the specified extentId , if the session already has a
 free page within that extent.
 Otherwise  DBF_ALLOCATION_MODE in Stone's config file will
 take precedence when the session asks stone for more free pages.

 Creates an instance of the receiver for clustering objects in the extent
 extentId.  The extentId argument is a positive SmallInteger in the range of
 1 to (SystemRepository numberOfExtents)."

self deprecated: 'ClusterBucket>>newForExtent: deprecated in v3.2, use new instead'.
^ self new extentId: extentId
%

!		Instance methods for 'ClusterBucket'

category: 'Accessing'
method: ClusterBucket
clusterId

"Returns the value of the private instance variable.  This instance variable
 should only be assigned by ClusterBucket | new."

^ _clusterId
%

category: 'Accessing'
method: ClusterBucket
description

"Returns the value of the description instance variable."

^ description
%

category: 'Updating'
method: ClusterBucket
description: anObject

"Assigns anObject (typically some kind of String object) as the description of
 the receiver."

description := anObject
%

category: 'Accessing'
method: ClusterBucket
extentId

"Returns the value of the extentId instance variable.

 In Gemstone64 v2.0, the relationship between a ClusterBucket and
 an extent is weak.  Clustering an object into a ClusterBucket will
 attempt to use the specified extentId , if the session already has a
 free page within that extent.
 Otherwise  DBF_ALLOCATION_MODE in Stone's config file will
 take precedence when the session asks stone for more free pages."


^ extentId
%

category: 'Updating'
method: ClusterBucket
extentId: anExtentId

"DEPRECATED.  In Gemstone64 v2.0, the relationship between a ClusterBucket and
 an extent is weak.  Clustering an object into a ClusterBucket will
 attempt to use the specified extentId , if the session already has a
 free page within that extent.
 Otherwise  DBF_ALLOCATION_MODE in Stone's config file will
 take precedence when the session asks stone for more free pages.

 An argument of nil specifies don't care behavior.  Positive arguments are
 an offset into the result of Repository | fileNames, thus specifying an extent.
 Reference to an extent which does not exist will generate an error at the
 time of executing this method.

 At the time of object modification, a non-nil extentId specifies which extent
 to attempt to put the object in (per DEPRECATED notice above).
 If the extent no longer exists, or is full,
 at the time of clustering or object modification, don't care behavior occurs."


self deprecated: 'ClusterBucket>>extentId: deprecated in v3.2, use new instead'.
SystemRepository validateExtentId: anExtentId .
extentId := anExtentId .
self _clearClusterCache .
%

category: 'Accessing'
method: ClusterBucket
keepClusteredOnModify

"This feature is not implemented in this release."

^ keepClusteredOnModify
%

category: 'Updating'
method: ClusterBucket
keepClusteredOnModify: aBoolean

"This feature is deferred until a future release.  Argument value of true
 is not supported in this release."

keepClusteredOnModify := false .
self _clearClusterCache .
%

category: 'Updating'
method: ClusterBucket
_clearClusterCache

"Private.  Any method that modifies an instance variable of a
 ClusterBucket that affects clustering behavior must send this method.  The
 instance variable description is not included in such instance variables."

<primitive: 235>
self _primitiveFailed: #_clearClusterCache .
self _uncontinuableError
%

category: 'Updating'
method: ClusterBucket
_clusterId: aSmallInteger

"Private.  For use only during instance creation."

_clusterId := aSmallInteger .
self _clearClusterCache .
%

! Class extensions for 'ClusterBucketArray'

!		Instance methods for 'ClusterBucketArray'

removeallmethods ClusterBucketArray
removeallclassmethods ClusterBucketArray

category: 'Clustering'
method: ClusterBucketArray
cluster

"Instances of ClusterBucketArray, especially AllClusterBuckets
 must always be clustered in the default bucket."

^ super clusterInBucket: 1
%

category: 'Clustering'
method: ClusterBucketArray
clusterInBucket: aClusterBucketOrId

"Instances of ClusterBucketArray, especially AllClusterBuckets
 must always be clustered in the default bucket."

^ super clusterInBucket: 1
%

! Class extensions for 'Collection'

!		Class methods for 'Collection'

removeallmethods Collection
removeallclassmethods Collection

category: 'Instance Creation'
classmethod: Collection
with: aValue

"Returns an instance of the receiver containing the argument."

| inst |
inst := self new.
inst add: aValue.
^inst
%

category: 'Instance Creation'
classmethod: Collection
with: aValue with: val2

"Returns an instance of the receiver containing the arguments."

| inst |
inst := self new.
inst add: aValue; add: val2.
^inst
%

category: 'Instance Creation'
classmethod: Collection
with: aValue with: val2 with: val3

"Returns an instance of the receiver containing the arguments."

| inst |
inst := self new.
inst add: aValue; add: val2; add: val3.
^inst
%

category: 'Instance Creation'
classmethod: Collection
with: aValue with: val2 with: val3 with: val4

"Returns an instance of the receiver containing the arguments."

| inst |
inst := self new.
inst add: aValue; add: val2; add: val3; add: val4.
^inst
%

category: 'Instance Creation'
classmethod: Collection
withAll: aCollection

"Returns an instance of the receiver containing the elements of the argument."

| result |

result:= self new.
result addAll: aCollection.
^result
%

!		Instance methods for 'Collection'

category: 'Comparing'
method: Collection
= aCollection

"Returns true if all of the following conditions are true:

 1.  The receiver and aCollection are of the same class.
 2.  The two collections are of the same size.
 3.  The corresponding elements of the receiver and aCollection are equal."

(self == aCollection)
  ifTrue: [ ^ true ].

(self class == aCollection class)
  ifFalse: [ ^ false ].

(self size == aCollection size)
  ifFalse: [ ^ false ].

self do: [ :anElement |
  (aCollection includes: anElement)
    ifFalse: [ ^ false ].
  ].

^ true.
%

category: 'Adding'
method: Collection
add: newObject

"Makes newObject one of the receiver's elements and returns newObject."

Collection subclassResponsibility: #add:
%

category: 'Adding'
method: Collection
addAll: aCollection

"Adds all of the elements of aCollection to the receiver and returns
 aCollection."

(self == aCollection) ifTrue: [ ^ self addAll: (aCollection copy) ].
aCollection accompaniedBy: self do: [ :me :each | me add: each ].
^ aCollection.
%

category: 'Enumerating'
method: Collection
allSatisfy: aBlock

	"Return true if <aBlock> evaluates to true for every element of the receiver.
	Return true if the receiver is empty. Otherwise return false."

	self do: [:each |
		(aBlock value: each) ifFalse: [^false].
	].
	^true.
%

category: 'Enumerating'
method: Collection
any
	"Return an arbitrary element of the receiver. Error if the receiver is empty."

	self do: [:each | ^each].
	self _error: #assocErrNoElementsDetected.
%

category: 'Enumerating'
method: Collection
anySatisfy: aBlock

        "Return true if <aBlock> evaluates to true for any element of the receiver.
        Otherwise return false. Return false if the receiver is empty."

	self do: [:each |
		(aBlock value: each) ifTrue: [^true].
	].
	^false.
%

category: 'Converting'
method: Collection
asArray

"Returns an Array with the contents of the receiver."

^Array withAll: self
%

category: 'Converting'
method: Collection
asBag

"Returns a Bag with the contents of the receiver."

^Bag withAll: self
%

category: 'Converting'
method: Collection
asByteArray

"Returns an Array with the contents of the receiver."

^ByteArray withAll: self
%

category: 'Converting'
method: Collection
asIdentityBag

"Returns an IdentityBag with the contents of the receiver."

^IdentityBag withAll: self
%

category: 'Converting'
method: Collection
asIdentitySet

"Returns an IdentitySet with the contents of the receiver."

^IdentitySet withAll: self
%

category: 'Converting'
method: Collection
asOrderedCollection

"Returns an OrderedCollection with the contents of the receiver."

^OrderedCollection withAll: self
%

category: 'Converting'
method: Collection
asSet

"Returns a Set with the contents of the receiver."

^Set withAll: self
%

category: 'Converting'
method: Collection
asSortedCollection

"Returns a SortedCollection with the contents of the receiver."

^SortedCollection withAll: self
%

category: 'Converting'
method: Collection
asSortedCollection: sortBlock

"Returns a SortedCollection with the contents of the receiver, using the
 given sort block."

| coll |
coll := SortedCollection sortBlock: sortBlock.
coll addAll: self.
^coll
%

category: 'Converting'
method: Collection
asSortedOrderedCollection

"Returns an OrderedCollection that has been sorted with a SortedCollection
 and having the contents of the receiver."

^OrderedCollection withAll: (SortedCollection withAll: self)
%

category: 'Clustering'
method: Collection
clusterDepthFirst

"This method clusters the receiver and its named and unnamed instance variables
 in depth-first order.  Returns true if the receiver has already been clustered
 during the current transaction; returns false otherwise."

self cluster
  ifTrue: [ ^ true ]
  ifFalse:[
    1 to: self namedSize do: [ :i | (self instVarAt: i) clusterDepthFirst ].
    self do: [ :each | each clusterDepthFirst ].
    ^ false
    ].
%

category: 'Enumerating'
method: Collection
collect: aBlock

"Evaluates aBlock with each of the receiver's elements as the argument.
 Collects the resulting values into a collection of class specified by
 sending #speciesForCollect message to the receiver. Returns the new
 collection. The argument aBlock must be a one-argument block.

 For SequenceableCollections, the result preserves the ordering of the
 receiver.  That is, if element a comes before element b in the receiver,
 then element a is guaranteed to come before b in the result."

|result|

result := self speciesForCollect new .
self accompaniedBy: result do: [ :res :each | res add: (aBlock value: each) ].
^ result
%

category: 'Enumerating'
method: Collection
detect: aBlock

"Evaluates aBlock repeatedly, with elements of the receiver as the argument.
 Returns the first element for which aBlock evaluates to true.  If none of the
 receiver's elements evaluates to true, generates an error.  The argument
 aBlock must be a one-argument block."

^ self detect: aBlock
       ifNone: [^ self _error: #assocErrNoElementsDetected args: { aBlock }]
%

category: 'Enumerating'
method: Collection
detect: aBlock ifNone: exceptionBlock

"Evaluates aBlock repeatedly, with elements of the receiver as the argument.
 Returns the first element for which aBlock evaluates to true.  If none of the
 receiver's elements evaluates to true, this evaluates the argument
 exceptionBlock and returns its value.  The argument aBlock must be a
 one-argument block, and exceptionBlock must be a zero-argument block."

self do: [:each| (aBlock value: each) ifTrue: [^each]].
^exceptionBlock value
%

category: 'Enumerating'
method: Collection
do: aBlock

"Evaluates the one-argument block aBlock using each element of the
 receiver in order.  Returns the receiver."

"Must be reimplemented in subclasses such as Dictionary which have different
 format or implementation."

1 to: self size do: [:i |
  aBlock value: (self at: i)
].
^ self
%

category: 'Enumerating'
method: Collection
do: operationBlock separatedBy: aBlock
	"Evaluates the one-argument block operationBlock using each element of the
	 receiver in order.  aBlock is evaluated between successive elements.
	 Returns the receiver. "

	| index |
	index := 0.
	self do:
			[:each |
			(index := index + 1) > 1 ifTrue: [aBlock value].
			operationBlock value: each].
	^self
%

category: 'Error Handling'
method: Collection
errorDifferentSizeCollections

"Reports an error indicating that the size of the receiver collection is
 different from the size of the argument collection."

^ self _error: #objErrDiffSizeColl
%

category: 'Error Handling'
method: Collection
errorInvalidArgClass: argument classes: classArray

"Reports an error indicating that the class of argument is not one of those
 specified in classArray."

^ self _error: #rtErrInvalidArgClass args: { argument . classArray }.
%

category: 'Repository Conversion'
method: Collection
getIndexInfo
  "Returns indexing information for the receiver."

  ^ {}
%

category: 'Comparing'
method: Collection
hash

"Returns a numeric hash key for the receiver."

| hashValue |

hashValue := (97133 bitXor: (self size)) bitXor: (self class asOop).
"For large collections, the hash value is just a function of its size
 and class"
(self size > 64) ifTrue: [ ^ hashValue abs ].

self do: [ :anElement |
  (anElement isKindOf: Collection)
    ifTrue: [ hashValue := hashValue bitXor: anElement size ]
    ifFalse: [ hashValue := hashValue bitXor: anElement hash ].
  ].
^ hashValue abs.
%

category: 'Searching'
method: Collection
identicalOccurrencesOf: anObject

"Returns the number of the receiver's elements that are identical (==) to
 anObject."

| result |

result := 0.
self do: [ :element |
  (anObject == element) ifTrue: [ result := result + 1 ]
  ].
^result
%

category: 'Searching'
method: Collection
includes: anObject

"Returns true if anObject is equal to one of the elements of the receiver.
 Returns false otherwise."

self do: [ :element | (anObject = element) ifTrue: [ ^true ]].
^false
%

category: 'Searching'
method: Collection
includesIdentical: anObject

"Returns true if anObject is identical to one of the elements of the receiver.
 Returns false otherwise."

self do: [ :element | (anObject == element) ifTrue: [ ^true ]].
^false
%

category: 'Searching'
method: Collection
includesValue: anObject

"Returns true if anObject is equal to one of the elements of the receiver.
 Returns false otherwise."

"Note: This method is the same as #includes:"

^ self includes: anObject.
%

category: 'Enumerating'
method: Collection
inject: aValue into: aBlock

"Accumulates a running value associated with evaluating the argument,
 aBlock, with the current value and the each element of the receiver
 as block arguments.  The initial value is the value of the argument, aValue.
 For example: total := #(1 2 3 4) inject: 0 into: [:sum:int | sum + int]"

| val |

val := aValue.
self do: [:element | val := aBlock value: val value: element ].
^val
%

category: 'Testing'
method: Collection
isEmpty

"Returns true if the receiver is empty.  Returns false otherwise."

^self size == 0
%

category: 'Testing'
method: Collection
notEmpty

"Returns true if the receiver is not empty.  Returns false otherwise."

^self size ~~ 0
%

category: 'Searching'
method: Collection
occurrencesOf: anObject

"Returns the number of the receiver's elements that are equal to anObject."

| count |

count := 0.
self do: [ :element| (anObject = element) ifTrue: [count := count + 1]].
^count.
%

category: 'Json'
method: Collection
printJsonOn: aStream

	| delimiter |
	delimiter := ''.
	aStream nextPut: $[.
	self do: [:each |
		aStream nextPutAll: delimiter.
		each printJsonOn: aStream.
		delimiter := ','.
	].
	aStream nextPut: $].
%

category: 'Formatting'
method: Collection
printNonRecursiveRepresentationOn: aStream recursionSet: anIdentitySet
	"Put a displayable representation of the receiver on the given stream
	 while avoiding recursion from object reference loops."

	| count sz |
	super printOn: aStream recursionSet: anIdentitySet.
	aStream nextPutAll: '( ' .
	count := 1 .
	sz := self size .
	self do:[:anElement |
	  anElement printOn: aStream recursionSet: anIdentitySet.
	  aStream isFull ifTrue:[
		  "prevent infinite recursion when printing cyclic structures, and
		   limit the size of result when printing large collections."
		  aStream _nextPut:( (aStream _collection endsWith: '...') ifTrue:[ $) ] ifFalse:[ ' ...)' ]).
		  ^ self
		] .
	  count < sz ifTrue:[ aStream nextPutAll: ', ' ].
	  count := count + 1 .
	].
	aStream nextPut: $) .
%

category: 'Formatting'
method: Collection
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

^ self printNonRecursiveOn: aStream
%

category: 'Formatting'
method: Collection
printString
  | ws str |
  str := String new.
  ws := PrintStream printingOn: str maxSize: 700 . "fix 51281"
  self printOn: ws.
   "contents might have been converted to another subclass of CharacterCollection."
  ^ ws _collection
%

category: 'Hashing'
method: Collection
rehash
	"Re-establish any hash invariants of the receiver."
%

category: 'Enumerating'
method: Collection
reject: aBlock

"Evaluates aBlock with each of the receiver's elements as the argument.  Stores
 the values for which aBlock is false into a collection of the same class as
 the receiver, and returns the new collection.  The argument aBlock must be a
 one-argument block.

 For SequenceableCollections, the result preserves the ordering of the
 receiver.  That is, if element a comes before element b in the receiver,
 then element a is guaranteed to come before b in the result."

|result|
result := self species new.
self do: [:each|
  (aBlock value: each) ifFalse: [result add: each]
  ].
^ result
%

category: 'Removing'
method: Collection
remove: oldObject

"Removes from the receiver an object that is equivalent to oldObject and
 returns oldObject.  If several elements of the receiver are equivalent to
 oldObject, only one instance is removed.  If oldObject has no equivalent
 elements in the receiver, raises an error."

^ self remove: oldObject ifAbsent: [self _errorNotFound: oldObject]
%

category: 'Removing'
method: Collection
remove: oldObject ifAbsent: anExceptionBlock

"Removes from the receiver an object that is equivalent to oldObject and
 returns oldObject.  If several elements of the receiver are equivalent to
 oldObject, only one instance is removed.  If oldObject has no equivalent
 elements in the receiver, anExceptionBlock is evaluated and the result of the
 evaluation is returned."

^ self subclassResponsibility: #remove:ifAbsent:
%

category: 'Removing'
method: Collection
removeAll: aCollection

"For each element in aCollection, removes from the receiver one element that is
 equivalent to the element in aCollection.  Returns aCollection if successful."

aCollection == self ifTrue:[
  self size: 0 .
  ^ aCollection
  ].
aCollection accompaniedBy: self do: [ :me :element | me remove: element ].
^ aCollection.
%

category: 'Removing'
method: Collection
removeAllIdentical: aCollection

"For each element in aCollection, removes from the receiver one element that is
 identical to the element in aCollection.  Returns aCollection if successful."

aCollection accompaniedBy: self do: [:me :element | me removeIdentical: element ].
^ aCollection.
%

category: 'Removing'
method: Collection
removeAllPresent: aCollection

"Removes from the receiver one occurrence of each element of aCollection that is
 also an element of the receiver.  Differs from removeAll: in that, if some
 elements of aCollection are not present in the receiver, no error is generated.
 Returns aCollection."

aCollection do: [ :anElement |
  self removeIfPresent: anElement
  ].

^ aCollection
%

category: 'Removing'
method: Collection
removeAllSuchThat: aBlock
   "Remove all elements of the receiver for which aBlock returns true.
    Answer the removed elements.
    This is a fallback implementation. It is not efficient."

  | toBeRemoved |
  toBeRemoved := self select: aBlock.
  self removeAllIdentical: toBeRemoved.
  ^ toBeRemoved
%

category: 'Removing'
method: Collection
removeIdentical: oldObject

"Removes from the receiver an object that is identical to oldObject, and
 returns oldObject.  If several elements of the receiver are identical to
 oldObject, only one instance is removed.  If oldObject is not present in
 the receiver, raises an error."

^ self removeIdentical: oldObject ifAbsent: [self _errorNotFound: oldObject]
%

category: 'Removing'
method: Collection
removeIdentical: oldObject ifAbsent: anExceptionBlock

"Removes from the receiver an object that is identical to oldObject and
 returns oldObject.  If several elements of the receiver are identical to
 oldObject, only one instance is removed.  If oldObject is not present in
 the receiver, anExceptionBlock is evaluated and the result of the
 evaluation is returned."

^ self subclassResponsibility: #removeIdentical:ifAbsent:
%

category: 'Enumerating'
method: Collection
select: aBlock

"Evaluates aBlock with each of the receiver's elements as the argument.
 Stores the values for which aBlock is true into a collection of the
 same class as the receiver, and returns the new collection.
 The argument aBlock must be a one-argument block.

 For SequenceableCollections, the result preserves the ordering of the
 receiver.  That is, if element a comes before element b in the receiver,
 then element a is guaranteed to come before b in the result.

 The new collection that this method returns does not retain any indexes of the
 receiver.  If you want to perform indexed selections on the new collection,
 you must build all of the necessary indexes.  For more information, see the
 GemStone Programming Guide."

| result |

result:= self species new.
self do: [ :each | (aBlock value: each) ifTrue: [result add: each]].
^result
%

category: 'Sorting'
method: Collection
sortAscending

"Returns an Array containing the elements of the receiver sorted in ascending
 order."

^ self sortAscending: '' persistentRoot: nil.
%

category: 'Sorting'
method: Collection
sortAscending: aSortSpec

"Returns an Array containing the elements of the receiver, sorted in ascending
 order, as determined by the values of the instance variables represented by
 aSortSpec.  The argument aSortSpec must be either a String representing a
 single path, or an Array holding up to 16 such Strings (each representing a
 path).  If aSortSpec is an Array, the first path in the Array is the primary
 sort key, and the remaining paths are taken in order as subordinate keys.

 Each path in aSortSpec must follow the rules for equality indexes.
 In addition, if any path in aSortSpec is an empty path (that is, a zero-length
 String), the sort is performed upon the elements of the receiver itself,
 rather than upon the instance variables of those elements."

^self sortAscending: aSortSpec persistentRoot: nil
%

category: 'Sorting'
method: Collection
sortAscending: aSortSpec persistentRoot: persistentArrayOrNil

"Returns an Array containing the elements of the receiver, sorted in ascending
 order, as determined by the values of the instance variables represented by
 aSortSpec.  The argument aSortSpec must be either a String representing a
 single path, or an Array holding up to 16 such Strings (each representing a
 path).  If aSortSpec is an Array, the first path in the Array is the primary
 sort key, and the remaining paths are taken in order as subordinate keys.

 Each path in aSortSpec must follow the rules for equality indexes.
 In addition, if any path in aSortSpec is an empty path (that is, a zero-length
 String), the sort is performed upon the elements of the receiver itself,
 rather than upon the instance variables of those elements."

| arg |

arg := { } .

(aSortSpec isKindOf: CharacterCollection)
  ifTrue: [ arg add: aSortSpec ; add: #ASCENDING ]
  ifFalse: [
     aSortSpec accompaniedBy: arg do: [ :aArg :each | aArg add: each ; add: #ASCENDING ]
  ].

^ self sortWith: arg persistentRoot: persistentArrayOrNil
%

category: 'Sorting'
method: Collection
sortDescending

"Returns an Array containing the elements of the receiver sorted in descending
 order."

^ self sortDescending: '' persistentRoot: nil.
%

category: 'Sorting'
method: Collection
sortDescending: aSortSpec

"Returns an Array containing the elements of the receiver, sorted in descending
 order, as determined by the values of the instance variables represented by
 aSortSpec.  The argument aSortSpec must be either a String representing a
 single path, or an Array holding up to 16 such Strings (each representing a
 path).  If aSortSpec is an Array, the first path in the Array is the primary
 sort key, and the remaining paths are taken in order as subordinate keys.

 Each path in aSortSpec must follow the rules for equality indexes.
 In addition, if any path in aSortSpec is an empty path (that is, a zero-length
 String), the sort is performed upon the elements of the receiver itself,
 rather than upon the instance variables of those elements."

^ self sortDescending: aSortSpec persistentRoot: nil
%

category: 'Sorting'
method: Collection
sortDescending: aSortSpec persistentRoot: persistentArrayOrNil

"Returns an Array containing the elements of the receiver, sorted in descending
 order, as determined by the values of the instance variables represented by
 aSortSpec.  The argument aSortSpec must be either a String representing a
 single path, or an Array holding up to 16 such Strings (each representing a
 path).  If aSortSpec is an Array, the first path in the Array is the primary
 sort key, and the remaining paths are taken in order as subordinate keys.

 Each path in aSortSpec must follow the rules for equality indexes.
 In addition, if any path in aSortSpec is an empty path (that is, a zero-length
 String), the sort is performed upon the elements of the receiver itself,
 rather than upon the instance variables of those elements."

| arg |

arg := { } .

(aSortSpec isKindOf: CharacterCollection)
  ifTrue: [ arg add: aSortSpec ; add: #DESCENDING ]
  ifFalse: [
    aSortSpec accompaniedBy: arg do: [:aArg :each | aArg add: each ; add: #DESCENDING ]
  ].

^ self sortWith: arg persistentRoot: persistentArrayOrNil
%

category: 'Sorting'
method: Collection
sortWith: aSortPairArray

"Returns an Array containing the elements of the receiver, sorted according to
 the contents of aSortPairArray.  The argument aSortPairArray is an Array of
 Strings that represent path/direction pairs, in the following form:

 aCollection sortWith: #('a.b' 'ASCENDING' 'a.c' 'DESCENDING' ...)

 That Array may contain up to 16 path/direction pairs.  The first path in the
 Array is the primary sort key, and the remaining paths are taken in order as
 subordinate keys.

 In aSortPairArray, each path String must follow the rules for equality
 indexes.  Each direction String must be either 'ASCENDING' or
 'DESCENDING' (case-insensitive); otherwise, an error is generated.

 In addition, if any path in aSortPairArray is an empty path (that is, a
 zero-length String), the sort is performed upon the elements of the receiver
 itself, rather than upon the instance variables of those elements."

^ self sortWith: aSortPairArray persistentRoot: nil
%

category: 'Sorting'
method: Collection
sortWith: aSortPairArray persistentRoot: persistentArrayOrNil

"Returns an Array containing the elements of the receiver, sorted according to
 the contents of aSortPairArray.  The argument aSortPairArray is an Array of
 Strings that represent path/direction pairs, in the following form:

 aCollection sortWith: #('a.b' 'ASCENDING' 'a.c' 'DESCENDING' ...)

 That Array may contain up to 16 path/direction pairs.  The first path in the
 Array is the primary sort key, and the remaining paths are taken in order as
 subordinate keys.

 In aSortPairArray, each path String must follow the rules for equality
 indexes.  Each direction String must be either 'ASCENDING' or
 'DESCENDING' (case-insensitive); otherwise, an error is generated.

 In addition, if any path in aSortPairArray is an empty path (that is, a
 zero-length String), the sort is performed upon the elements of the receiver
 itself, rather than upon the instance variables of those elements.

 If persistentArrayOrNil is notNil, then it is expected to be an empty persistent array and
 the array will be used to persist large temporary data structures created during
 the sorting operation. IndexManager>>autoCommit must be true in order for periodic
 commits to  be made during the sorting operation. When the sort operation is complete
 the persistent array will be emptied and a final commit performed. The persistentArrayOrNil
 and  IndexManager>>autoCommit should be used when a collection is so large that it
 isn't practical to allocate enough temporary memory."

| paths booleans thePath directionStr directionBool result spaSize |

paths := { } .
booleans := { } .

((spaSize := aSortPairArray size) ~~ 0 and:[ spaSize even])
    ifFalse: [ ^ aSortPairArray _error: #assocErrSortOddLengthArray ].

1 to: spaSize by: 2 do: [ :i |
    thePath := aSortPairArray at: i.
    thePath _isOneByteString ifFalse:[
      thePath _validateClasses: { String }.
      ].
    directionStr := (aSortPairArray at: i + 1) .
    directionBool := true .
    directionStr == #ASCENDING
      ifFalse:[
        (directionStr == #DESCENDING )
          ifTrue:[ directionBool := false ]
          ifFalse:[
            (Symbol _existingWithAll: directionStr asUppercase) ifNotNil:[ :directionSym|
               directionSym == #ASCENDING
               ifFalse:[
                 directionSym == #DESCENDING
                   ifTrue: [ directionBool := false ]
                   ifFalse: [ directionStr _error: #assocErrBadDirection ].
	       ].
	    ] ifNil:[  directionStr _error: #assocErrBadDirection ].
	  ].
       ].
    paths add: thePath.
    booleans add: directionBool .
].
result := self _sortPaths: paths directions: booleans persistentRoot: persistentArrayOrNil.
^ result
%

category: 'Private'
method: Collection
_asCollectionForSorting

"Redefine this method for classes that understand _at: or have different
 semantics for the do: method."

| result |

result := { } .
self accompaniedBy: result do: [:res :element | res add: element ].
^ result.
%

category: 'Private'
method: Collection
_asIdentityBag

"Returns the receiver."

^ self asIdentityBag
%

category: 'Private'
method: Collection
_deepCopyWith: copiedObjDict

| copy myClass |

copy := copiedObjDict at: self otherwise: nil.
copy ifNotNil: [ ^ copy ].

myClass := self class.
copy := myClass new.
copiedObjDict at: self put: copy.

self _deepCopyNamedIvsWith: copiedObjDict to: copy .

1 to: self _basicSize do: [ :n | | anElement |
  anElement := self _at: n.
  copy add: (anElement _deepCopyWith: copiedObjDict)
].
^ copy.
%

category: 'Error Handling'
method: Collection
_errorNotFound: anObject

"Sends an error message indicating that the expected object was not found."

^ self _error: #objErrNotInColl args: { anObject }
%

category: 'Private'
method: Collection
_findRangeIndexWithPath: pathArray

"Redefine this method for classes that support indexes."

^ nil
%

category: 'Private'
method: Collection
_indexedPaths

"Returns the indexed paths for the receiver.  This is a private method that
 should only be used for sorting/indexing."

"This method should be reimplemented for classes where _indexedPaths is
 defined."

^ nil
%

category: 'Sorting'
method: Collection
_sortPaths: thePaths directions: theBooleans

""

^ self _sortPaths: thePaths directions: theBooleans  persistentRoot: nil
%

category: 'Sorting'
method: Collection
_sortPaths: thePaths directions: theBooleans persistentRoot: persistentArrayOrNil
  ""

  | thePath pathArrays indexObjs result |
  pathArrays := Array new: thePaths size.	" do some error checking "
  1 to: thePaths size do: [ :i |
    thePath := thePaths at: i.
    thePath size > 1024
      ifTrue: [
        " check the string size "
        ^ thePath _error: #'assocErrPathTooLong' ].
    (thePath includesValue: $*)
      ifTrue: [
        " check if sorting over set-valued instance variable (not allowed) "
        ^ self _error: #'rtErrBagInvalidSortSpecification' args: {thePath} ].
    pathArrays at: i put: thePath asArrayOfPathTerms ].
  indexObjs := self _indexObjectsFor: pathArrays.
  result := (indexObjs at: 1)
    _sortOn: indexObjs
    directions: theBooleans
    persistentRoot: persistentArrayOrNil.
  ^ result
%

category: 'Private'
method: Collection
_validateNotEmpty

^ self isEmpty
    ifTrue:[ self _error: #objErrCollectionEmpty]
    ifFalse:[ true ]
%

! Class extensions for 'CollisionBucket'

!		Instance methods for 'CollisionBucket'

removeallmethods CollisionBucket
removeallclassmethods CollisionBucket

category: 'Accessing'
method: CollisionBucket
keyValueDictionary

"Returns the value of the instance variable."

^keyValueDictionary
%

category: 'Updating'
method: CollisionBucket
keyValueDictionary: aDict

"Updates the value of the keyValueDictionary instance variable."

keyValueDictionary := aDict
%

category: 'Searching'
method: CollisionBucket
_firstPair

"Returns an Array containing the receiver's first key/value pair.
 If the receiver is empty, returns an Array containing nils."

   numElements == 0 ifFalse: [  | idx |
      idx := 1 .
      "Search for the first non-nil key"
      1 to: self tableSize do: [ :n | | aKey |
         (aKey := self _at: idx ) ifNotNil: [    "inline keyAt:"
            ^ { aKey . (self _at: idx + 1) }
         ].
         idx := idx + 2
      ]
   ].
   ^ { nil . nil } "No first pair was found"
%

category: 'Private'
method: CollisionBucket
_removeAll

"Dereferences the receiver from its parent and shrinks the receiver.
 Used while rebuilding a KeyValueDictionary."

keyValueDictionary := nil.
numElements := 0.
"gs64 v3.0 don't send  size: 0"
%

category: 'Private'
method: CollisionBucket
_removePairAt: anOffset

(self _at: anOffset)  ifNil:[
  ^ Error signal:'key has already been removed'.
].
self _at: anOffset put: nil ;
     _at: anOffset + 1 put: nil .
numElements := numElements - 1 .
%

! Class extensions for 'CompileError'

!		Instance methods for 'CompileError'

removeallmethods CompileError
removeallclassmethods CompileError

category: 'Instance initialization'
method: CompileError
initialize
  gsNumber := ERR_CompileError.
  gsResumable := true .
  gsTrappable := true .
%

category: 'Formatting'
method: CompileError
sourceString
^ gsArgs ifNotNil:[:gsa | gsa atOrNil: 2 ]
%

category: 'Formatting'
method: CompileError
_firstCompilerErrorNumber

^ (self errorDetails at: 1) at: 1
%

! Class extensions for 'CompileWarning'

!		Class methods for 'CompileWarning'

removeallmethods CompileWarning
removeallclassmethods CompileWarning

category: 'Instance creation'
classmethod: CompileWarning
signal: aString method: aMethod
  | ex |
  (ex := self new) args: { aString . aMethod } ;
     signal
%

!		Instance methods for 'CompileWarning'

category: 'Instance initialization'
method: CompileWarning
buildMessageText
^ self buildMessageText:(
  [ | str |
    str := String new .
    gsArgs ifNotNil:[:d| | meth |
      meth := d at: 2 .
      str add:( d at: 1) ; add: ' in ' ;
          add: meth inClass name ;
          add: ' >> ' ; 
          add: meth selector .
    ].
    str
  ] onException: Error do:[:ex |
    ex return: nil
  ]
)
%

category: 'Instance initialization'
method: CompileWarning
initialize
  gsNumber := ERR_CompileWarning .
  gsResumable := true .
  gsTrappable := true .
%

category: 'Accessing'
method: CompileWarning
method
  ^ gsArgs at: 2
%

category: 'Accessing'
method: CompileWarning
warningString
  ^ gsArgs at: 1
%

! Class extensions for 'ControlInterrupt'

!		Instance methods for 'ControlInterrupt'

removeallmethods ControlInterrupt
removeallclassmethods ControlInterrupt

category: 'Instance initialization'
method: ControlInterrupt
initialize
  gsNumber := ERR_ControlInterrupt.
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'CryptoError'

!		Instance methods for 'CryptoError'

removeallmethods CryptoError
removeallclassmethods CryptoError

category: 'Instance initialization'
method: CryptoError
initialize
  gsNumber := ERR_CryptoError.
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'Date'

!		Class methods for 'Date'

removeallmethods Date
removeallclassmethods Date

category: 'Inquiries'
classmethod: Date
dayOfWeek: dayName

"Returns a SmallInteger that gives the numeric index of the day of the week
 described by dayName. The index is a number between 1 and 7 inclusive,
 where 1 signifies Sunday. dayName must be the full symbolic name of the
 day. Raises an error if dayName is not a valid week day name."

 "Example: Date dayOfWeek: 'Tuesday'."

| aDay |
aDay := dayName asString asLowercase.
aDay at: 1 put: ((aDay at: 1) asUppercase).
^ WeekDayNames value indexOf: aDay
    ifAbsent: [ dayName _error: #rtErrInvalidArgument
    args: { 'argument is not a week day name' }].
%

category: 'Instance Creation'
classmethod: Date
fromStream: aStream

"Creates and returns an instance of the receiver by reading a String from
 aStream.  The String expresses the date in the default format (DD/MM/YYYY).
 Generates an error if the String does not conform to the format."

^ self fromStream: aStream usingFormat: #(1 2 3 $/ 1 1)
%

category: 'Instance Creation'
classmethod: Date
fromStream: aStream usingFormat: anArray

"Creates and returns an instance of the receiver by reading a String from
 aStream.  The String expresses the date in the format specified by anArray.
 The expression is terminated either by a space Character or by the end of the
 Stream.  Generates an error if the String does not conform to the format,
 or if anArray contains an incorrect formatting specification.

 See the class documentation of Date for a complete description of the
 String-formatting specification Array.

 If the month format (5th element) indicates either an abbreviation (2) or an
 entire name (3), then this method tries to determine the month by decoding a
 Character substring.  That substring may include any number of characters, but
 must exactly match a legal month name (or the initial portion of that month
 name).  If the substring matches more than one month, the first month matched
 is used (the search begins with January)."

| dayInt monthInt yearInt dateDelim result parseField parseBlocks |

"This block returns a string up from the input stream up to the specified
 delimiter.  If also allows an end-of-file if that parameter is set true.
 It then skips over the delimiter if it is found.
"
parseField := [ :delim :allowEof | | str |
  str := aStream contents class new.
  [ ((aStream peek isEquivalent: delim) not) and:[aStream atEnd not] ]
    whileTrue: [ str add: aStream next ].

  (aStream atEnd) ifTrue:[
     allowEof ifFalse:[ Date _error: #rtErrBadFormat args: { aStream } ].
     ]
  ifFalse:[ aStream next "skip over delimiter" ].
  str
  ].

parseBlocks := {
  "parse day"
  [ :delim | | nextField |
    nextField := parseField value: delim value: (delim == $  ).
    dayInt := Integer fromCompleteString: nextField
  ] .
  "parse month"
  [ :delim | | nextField |
    nextField := parseField value: delim value: (delim == $  ).
    (nextField = '' )
    ifTrue:
      [ Date _error: #rtErrBadFormat args: { aStream }].
    (anArray at: 5) == 1
    ifTrue:
      [ monthInt := Integer fromCompleteString: nextField ]
    ifFalse:
      [ monthInt := self _getMonthFrom: nextField ].
    (monthInt < 1 or:[monthInt > 12])
    ifTrue:
      [ Date _error: #rtErrBadFormat args: { aStream } ]
  ] .
  "parse year"
  [ :delim | | nextField |
    nextField := parseField value: delim value: (delim == $  ).
    yearInt := Integer fromCompleteString: nextField.
    (anArray at: 6) == 2
    ifTrue:
      [ (yearInt > 99)
        ifFalse: [yearInt := yearInt + ((Date today year) // 100 * 100) ]
      ]
  ]
 }.

self _checkReadStream: aStream forClass: CharacterCollection.

Date _checkFormat: anArray.

dateDelim := anArray at: 4.

"parse the date, with day, month, year in the specified format order"
(parseBlocks at:( anArray at: 1)) value: dateDelim .
(parseBlocks at:( anArray at: 2)) value: dateDelim .
(parseBlocks at:( anArray at: 3)) value: $  .

result := self newDay: dayInt monthNumber: monthInt year: yearInt .
^ result
%

category: 'Instance Creation'
classmethod: Date
fromString: aString

"Creates and returns an instance of the receiver from the String aString.
 The String expresses the date in the default format (DD/MM/YYYY).
 Generates an error if the String does not conform to the format."

^ self fromString: aString usingFormat: #(1 2 3 $/ 1 1)
%

category: 'Instance Creation'
classmethod: Date
fromString: aString usingFormat: anArray

"Creates and returns an instance of the receiver from the String aString.
 The String expresses the date in the format specified by anArray.  The
 expression is terminated either by a space Character or by the end of the
 String.  Generates an error if the String does not conform to the format,
 or if anArray contains an incorrect formatting specification.

 See the class documentation of Date for a complete description of the
 String-formatting specification Array.

 If the month format (5th element) indicates either an abbreviation (2) or an
 entire name (3), then this method tries to determine the month by decoding a
 character substring.  That substring may include any number of Characters, but
 must exactly match a legal month name (or the initial portion of that month
 name).  If the substring matches more than one month, the first month matched
 is used (the search begins with January)."

| s result |

s := ReadStreamPortable on: aString.
result := self fromStream: s usingFormat: anArray.
[ s atEnd ]
whileFalse:
  [ (s next == $  )
    ifFalse:
      [ self _errIncorrectFormat: aString ]
  ].
^ result
%

category: 'Inquiries'
classmethod: Date
indexOfMonthName: monthName

"Return an Integer corresponding to the month with the name monthName.
 If monthName is not a recognized name, return 0, otherwise return an
 Integer between 1 and 12 inclusive, where 1 signifies January."

| monthArray |
monthArray := MonthNames value.
1 to: 12 do: [:index | (monthArray at: index) = monthName ifTrue: [^ index]] .
^ 0
%

category: 'Inquiries'
classmethod: Date
isLeap: year

"Returns true if year is a leap year; false otherwise."

"a year is a leap year if: (it is evenly divisible by 4 and it is not a
 century year) or (it is a century year and evenly divisible by 400)"

((year \\ 100) == 0)
   ifTrue: [^ ((year \\ 400) == 0)].
^ ((year \\ 4) == 0)
%

category: 'Storing and Loading'
classmethod: Date
loadFrom: passiveObj

"Creates and returns an active instance of the receiver from the passive form
 of the object."
| inst yr dy |
yr := passiveObj readObject .
yr  _isSmallInteger ifFalse:[ Error signal:'unexpected ', yr class name ].
dy := passiveObj readObject .
dy  _isSmallInteger ifFalse:[ Error signal:'unexpected ', dy class name ].
self == SmallDate ifTrue:[
  inst := self newDay: dy year: yr .
  inst isSpecial ifFalse:[ Error signal:'a SmallDate should be special'].
] ifFalse:[
  inst := self _basicNew _year: yr day: dy .
  passiveObj hasRead: inst .
].
^ inst
%

category: 'Instance Creation'
classmethod: Date
migrateNew

"Override default migrateNew behavior with #_basicNew."

^ self _basicNew
%

category: 'Inquiries'
classmethod: Date
nameOfMonth: anIndex

"Returns a String that gives the name, in the user's native language, of the
 month of the year whose numeric index is anIndex.  The index is a number
 between 1 and 12 inclusive, where 1 signifies January."

^ (MonthNames value) at: anIndex.
%

category: 'Instance Creation'
classmethod: Date
new

"Disallowed.  To create a new Date, use another instance creation method."

self shouldNotImplement: #new
%

category: 'Instance Creation'
classmethod: Date
new: anInteger

"Disallowed.  To create a new Date, use another instance creation method."

self shouldNotImplement: #new:
%

category: 'Instance Creation'
classmethod: Date
newDay: dayInt month: monthString year: yearInt

"Creates and returns an instance of the receiver from the specified values.
 Generates an error if any of the values are out of range."

 | monthInt |

 monthInt := self _getMonthFrom: monthString .
 (monthInt < 1 or: [ monthInt > 12 ]) ifTrue:[
   Date _error: #rtErrBadFormat args: { { dayInt . monthString . yearInt } }.
 ].
^ self _newDay: dayInt monthNumber: monthInt year: yearInt .
%

category: 'Instance Creation'
classmethod: Date
newDay: day monthNumber: month year: year

(month < 1 or:[ month > 12]) ifTrue: [
  ArgumentError signal: 'Incorrect specified month: ' , month asString
].
(day < 1 or:[ day > (self numberOfDaysIn: month year: year)]) ifTrue: [
  ArgumentError signal: 'Incorrect specified day: ', day asString.
].
^ self _newDay: day monthNumber: month year: year
%

category: 'Instance Creation'
classmethod: Date
newDay: day year: year

"Creates and returns an instance of the receiver from the specified values.
 Generates an error if any of the values are out of range."

^ self _newDay: day monthNumber: 1 year: year.
%

category: 'Instance Creation'
classmethod: Date
numberOfDaysIn: month year: aYear

| x |
x := #( 31 nil 31 30 31 30 31 31 30 31 30 31 ) atOrNil: month .
x ifNotNil:[ ^ x ] .
month == 2 ifFalse:[ ArgumentError signal: 'Incorrect specified month: ' , month asString ].
(((aYear \\ 100) == 0)
   ifTrue: [ ((aYear \\ 400) == 0)]
   ifFalse: [ ((aYear \\ 4) == 0) ])
  ifTrue: [^ 29].
^ 28
%

category: 'Instance Creation'
classmethod: Date
today

  "Returns an instance of the receiver representing the current Date, taking into
   consideration the current GemStone TimeZone.

   See Date class >>_today for 3.2.x and earlier implementation that gets date
   directly from the OS and does not reflect repository TimeZone."

   | parts |
   parts := DateAndTime now asFloatParts.
     "parts is { year. dayOfYear. monthIndex. dayOfMonth. hour. minute. second } "
   ^self newDay: (parts at: 4) monthNumber: (parts at: 3) year: (parts at: 1).
%

category: 'Instance Creation'
classmethod: Date
_checkFormat: anArray

"Private.  Verifies that anArray is a valid string-formatting specification for
 the receiver.  Generates an error if it is not."

| v |

anArray _validateClass: Array.
(anArray size < 6)
  ifTrue:[ Date _error: #rtErrBadFormatSpec args: { anArray } ].

"Check for a combination of the digits 1, 2, and 3"
((anArray at: 1) + (anArray at: 2) + (anArray at: 3) == 6 and:
        [(anArray at: 1) * (anArray at: 2) * (anArray at: 3) == 6])
  ifFalse:[ Date _error: #rtErrBadFormatSpec args: { anArray } ].

(anArray at: 4) _validateClass: Character.

((v := anArray at: 5) == 1 or: [v == 2 or: [v == 3]])
  ifFalse:[ Date _error: #rtErrBadFormatSpec args: { anArray } ].

((anArray at: 6) == 1 or: [(anArray at: 6) == 2])
  ifFalse:[ Date _error: #rtErrBadFormatSpec args: { anArray } ].
%

category: 'Instance Creation'
classmethod: Date
_getMonthFrom: aCharCollection

"Private.  Returns the SmallInteger that corresponds to the month of the year
 that matches aCharCollection.

 The argument may include any number of characters, but must exactly match a
 legal month name (or the initial portion of that month name).  If the argument
 matches more than one month, the first month matched is used.  The search
 begins with January."

| whichMonth monthArray argSize matchMonth |

monthArray := MonthNames value.
matchMonth:= [:monthStr | | strSize match i |
   i:= 1.
   match:= false.
   strSize:= monthStr size.
   [ ((i <= argSize) and:[i <= strSize]) and:
     [match:= (aCharCollection at: i) isEquivalent: (monthStr at: i)]]
   whileTrue: [
      i:= i + 1.
   ].
   match
].

   argSize:= aCharCollection size.
   whichMonth:= 1.
   [ (whichMonth <= 12) and:
     [(matchMonth value: (monthArray at: whichMonth)) not]
   ]
   whileTrue:
      [whichMonth := whichMonth + 1].

   (whichMonth <= 12)
      ifTrue: [ ^whichMonth].
   ^ 0
%

category: 'Instance Creation'
classmethod: Date
_newDay: day monthNumber: month year: year

"Creates and returns an instance of SmallDate if possible, otherwise
 returns an instance of the receiver from the specified values.
 Generates an error if any of the values are out of range.
 If self == Date or self == SmallDate,  result is a SmallDate if possible."

<primitive: 316>
| blk |
blk := [:x :argName | 
  | min max |
  x _isSmallInteger ifFalse:[ 
    ArgumentTypeError new name: argName expectedClass: SmallInteger actualArg: x; signal
  ].
  (x < (min := SmallInteger minimum32bitInteger) 
      or:[ x > (max := SmallInteger maximum32bitInteger)]) ifTrue:[
    OutOfRange new name: argName min: min max: max actual: x ; signal
  ].
 ].
blk value: day value: 'day' .
blk value: month value: 'month' .
blk value: year value: 'year' .
^ self _primitiveFailed: #_newDay:monthNumber:year:
       args: { day . month . year }
%

category: 'Instance Creation'
classmethod: Date
_today

"Creates and returns an instance of the receiver from the system calendar
 on the machine that is running the Gem process, which is assumed to
 represent the current date. 
 If self == Date or self == SmallDate,  result is a SmallDate if possible."

<primitive: 315>
^ self _primitiveFailed: #today
%

!		Instance methods for 'Date'

category: 'Comparing'
method: Date
< aDate

"Returns true if the receiver represents a date before that of the argument,
 and false if it doesn't.  Generates an error if the argument is not
 a Date."

| argYear myYear |

argYear := aDate year.
(myYear := self year) == argYear
  ifTrue: [ ^ self dayOfYear < aDate dayOfYear ]
  ifFalse: [ ^ myYear < argYear ].
%

category: 'Comparing'
method: Date
= aDate

"Returns true if the receiver represents the same date as that of the
 argument, and false if it doesn't."

(self == aDate) ifTrue: [ ^ true ].
(aDate isKindOf: Date ) ifFalse: [ ^false ].
^ (self year == aDate year) and: [self dayOfYear == aDate dayOfYear]
%

category: 'Comparing'
method: Date
> aDate

"Returns true if the receiver represents a date after that of the argument,
 and false if it doesn't.  Generates an error if the argument is not
 a Date."

| argYear myYear |

argYear := aDate year.
(myYear := self year) == argYear
  ifTrue: [ ^ self dayOfYear > aDate dayOfYear ]
  ifFalse: [^  myYear > argYear ].
%

category: 'Arithmetic'
method: Date
addDays: anInteger

"Returns a Date that describes a date anInteger days later than that
 of the receiver."

^ (self class) newDay: (self dayOfYear + anInteger) year: self year.
%

category: 'Arithmetic'
method: Date
addMonths: anInteger

"Returns a Date that describes a date anInteger months later than that of the
 receiver.

 This method attempts to keep the day of the month the same.  If the
 new month has fewer days than the receiver's original month, then it
 truncates to the last day of the new month."

| yr month day newYear newMonth newDay newDate generatedDay |

yr := self year.
month := self month.
day := self day.

newMonth := month + anInteger.
newYear := yr + ((newMonth - 1) // 12).
newMonth := (newMonth - 1) \\ 12 + 1.
newDate := self class _newDay: day monthNumber: newMonth year: newYear.
generatedDay := newDate day.
(generatedDay ~= day)
  ifTrue: [
    newDay := newDate _daysInMonth: newMonth.
    newDate := self class _newDay: newDay monthNumber: newMonth year: newYear
    ].
^ newDate.
%

category: 'Arithmetic'
method: Date
addYears: anInteger

"Returns a Date that describes a date anInteger years later than that of the
 receiver."

| yr month day newYear newDay newDate generatedDay |

yr := self year.
month := self month.
day := self day.

newYear := yr + anInteger.
newDate := self class _newDay: day monthNumber: month year: newYear.
generatedDay := newDate day.
(generatedDay ~= day)
  ifTrue: [
    newDay := newDate _daysInMonth: month.
    newDate := self class _newDay: newDay monthNumber: month year: newYear
    ].
^ newDate.
%

category: 'Converting'
method: Date
asCanonicalForm
	"Answer self, or, if I am a Date with an equivalent
	SmallDate, answer that SmallDate."

	| cls res |
	(cls := self class) ~~ SmallDate
		ifTrue: [ 
			cls == Date
				ifTrue: [ 
					res := cls _newDay: self day monthNumber: self month year: self year.
					res class == SmallDate
						ifTrue: [ ^ res ] ] ].
	^ self
%

category: 'Converting'
method: Date
asDays

"Returns an Integer that represents the receiver in units of days since
 January 1, 1901."

| numYears numDays |

numYears := self year - 1901.
numDays := (numYears * 365) + (numYears // 4) +
           ((numYears + 300) // 400) - (numYears // 100) + self dayOfYear - 1.
^ numDays.
%

category: 'Formatting'
method: Date
asString

"Returns a String that expresses the receiver in the default format
 (DD/MM/YYYY)."

| t result |

t := self _yearMonthDay.
result := (t at: 3) _digitsAsString .
result addAll: '/';
  addAll: (t at: 2) _digitsAsString;
  addAll: '/';
  addAll: (t at: 1) _digitsAsString.
^ result
%

category: 'Formatting'
method: Date
asStringUsingFormat: anArray

"Returns a String that expresses the receiver in the format defined by anArray.
 Generates an error if anArray contains an incorrect formatting specification.

 See the class documentation of Date for a complete description of the
 String-formatting specification Array."

| t dateSeparator monthName aString day yearNumber |

t := self _yearMonthDay.

Date _checkFormat: anArray.

dateSeparator := (anArray at: 4) asString.

((anArray at: 5) == 2) "get the month name according to the format"
   ifTrue: [monthName := self _monthAbbrev: (t at: 2)]
   ifFalse: [((anArray at: 5) == 3) "month as number is default"
      ifTrue: [monthName := Date nameOfMonth: (t at: 2)]
      ifFalse: [monthName := (t at: 2) _digitsAsString]].

((anArray at: 6) == 2)
   ifTrue: [yearNumber := ((t at: 1) \\ 100) _digitsAsString]
   ifFalse: [yearNumber := (t at: 1) asString].  "YYYY is default"

day := (t at:3) _digitsAsString.
((anArray at: 1) == 2) "month first"
   ifTrue: [aString := monthName , dateSeparator]
   ifFalse: [((anArray at: 1) == 3) "yearNumber first"
      ifTrue: [aString := yearNumber , dateSeparator]
      ifFalse: [aString := day , dateSeparator]].  "day first is default"

((anArray at: 2) == 1) "day second"
   ifTrue: [aString addAll: day; addAll: dateSeparator] "yearNumber second"
   ifFalse: [((anArray at: 2) == 3) "month second is default"
      ifTrue: [aString addAll: yearNumber; addAll: dateSeparator]
      ifFalse: [aString addAll: monthName; addAll: dateSeparator]].

((anArray at: 3) == 1) "day third"
   ifTrue: [aString addAll: day]
   ifFalse: [((anArray at: 3) == 2) "month third"
      ifTrue: [aString addAll: monthName]
      ifFalse: [aString addAll: yearNumber]].  "yearNumber third is default"

^ aString
%

category: 'Accessing'
method: Date
at: anIndex put: aValue

"Disallowed.  You may not change the value of a Date."

self shouldNotImplement: #at:put:
%

category: 'Accessing'
method: Date
day

"Returns a SmallInteger that gives the day of the month described by the
 receiver."

^  (self _yearMonthDay) at: 3
%

category: 'Accessing'
method: Date
dayOfMonth

"Returns a SmallInteger that gives the day of the month described by the
 receiver."

^ self day.
%

category: 'Accessing'
method: Date
dayOfWeek

"Returns a SmallInteger that gives the numeric index of the day of the week
 described by the receiver.  The index is a number between 1 and 7 inclusive,
 where 1 signifies Sunday."

^ self julianDay - 2299295 - 1 \\ 7 + 1
%

category: 'Accessing'
method: Date
dayOfYear

"Returns a SmallInteger that gives the day of the year described by the
 receiver."

^ dayOfYear
%

category: 'Accessing'
method: Date
daysInMonth

"Returns a SmallInteger that gives the number of days in the month
 described by the receiver."

^ self _daysInMonth: self month

%

category: 'Accessing'
method: Date
daysInYear

"Returns a SmallInteger that gives the number of days in the year
 described by the receiver."

(self leap) ifTrue: [^ 366].
^ 365
%

category: 'Comparing'
method: Date
hash

"Returns an Integer hash code for the receiver."

^ ((self year hash) bitShift: -1) bitXor: (self dayOfYear hash)
%

category: 'Accessing'
method: Date
julianDay

"Returns the Julian Day of the receiver, a SmallInteger that gives the number of
 days since January 1, 4713 B.C., as defined in Communications of the ACM,
 algorithm #199."

<primitive: 46 >

^ self _primitiveFailed: #julianDay
%

category: 'Accessing'
method: Date
leap

"Returns true if the receiver describes a leap year and false if it does not."

| yr |
  "a year is a leap year if: (it is evenly divisible by 4 and it is not a
   century year) or (it is a century year and evenly divisible by 400)"

yr := self year .
((yr \\ 100) == 0)
   ifTrue: [^ ((yr \\ 400) == 0)].
^ ((yr \\ 4) == 0)
%

category: 'Accessing'
method: Date
month

"Returns a SmallInteger that gives the numeric index of the month of the year
 described by the receiver.  The index is a number between 1 and 12 inclusive,
 where 1 signifies January."

^ (self _yearMonthDay) at: 2
%

category: 'Accessing'
method: Date
monthIndex

"Returns a SmallInteger that gives the numeric index of the month of the year
 described by the receiver.  The index is a number between 1 and 12 inclusive,
 where 1 signifies January."

^ (self _yearMonthDay) at: 2
%

category: 'Accessing'
method: Date
monthName

"Returns a String that gives the name of the month of the year described by the
 receiver, in the user's native language."

^ MonthNames value at: self month
%

category: 'Accessing'
method: Date
monthOfYear

"Returns a SmallInteger that gives the numeric index of the month of the year
 described by the receiver.  The index is a number between 1 and 12 inclusive,
 where 1 signifies January."

^ self month.
%

category: 'Inquiries'
method: Date
next: dayName
"Returns the next date whose weekday name is dayName"

^ self addDays:
   ((self class dayOfWeek: dayName) - 1 - self weekdayIndex - 1) \\ 7 + 1
%

category: 'Inquiries'
method: Date
nextMonth: monthName
"Returns the next Date which describes a date later than the receiver and has
 month named monthName."

| tmp index |
index := self class indexOfMonthName: monthName asString .
^ self month == index
    ifTrue: [self addYears: 1]
   ifFalse: [tmp := self.
             [tmp month == index]
                whileFalse: [tmp := tmp addMonths: 1].
              tmp]
%

category: 'Inquiries'
method: Date
previous: dayName

"Returns the previous date whose weekday name is dayName."

^ self subtractDays: self dayOfWeek -
       (self class dayOfWeek: dayName) -
       1 \\ 7 + 1.
%

category: 'Inquiries'
method: Date
previousMonth: monthName
"Returns the next Date which describes a date earlier than the receiver and has
 month named monthName."

| tmp index |
index := self class indexOfMonthName: (monthName asString).
^ self month == index
    ifTrue: [self subtractYears: 1]
    ifFalse: [tmp := self.
              [tmp month == index]
                 whileFalse: [tmp := tmp subtractMonths: 1].
               tmp]
%

category: 'Formatting'
method: Date
printJsonOn: aStream
  (self asStringUsingFormat: #( 3 2 1 $- 1 1)) printJsonOn: aStream 
%

category: 'Formatting'
method: Date
printOn: aStream

"Puts a displayable representation of the receiver on aStream."

aStream nextPutAll: self asString .
%

category: 'Accessing'
method: Date
size: anInteger

"Disallowed.  You may not change the size of a Date."

self shouldNotImplement: #size:
%

category: 'Arithmetic'
method: Date
subtractDate: aDate

"Returns a positive Integer that counts the number of times midnight occurs
 between the times described by the receiver and aDate."

^ (self asDays - aDate asDays) abs
%

category: 'Arithmetic'
method: Date
subtractDays: anInteger

"Returns a Date that describes a date anInteger days earlier than that
 of the receiver."

^ self class newDay: self dayOfYear - anInteger year: self year
%

category: 'Arithmetic'
method: Date
subtractMonths: anInteger

"Returns a Date that describes a date anInteger months earlier than that of the
 receiver.

 This method attempts to keep the day of the month the same.  If the
 new month has fewer days than the receiver's original month, then it
 truncates to the last day of the new month."

^ self addMonths: (anInteger negated).
%

category: 'Arithmetic'
method: Date
subtractYears: anInteger

"Returns a Date that describes a date anInteger years earlier than that of the
 receiver."

^ self addYears: (anInteger negated).
%

category: 'Formatting'
method: Date
USDateFormat

"Returns a String that expresses the date of the receiver. The date is in
 United States format, month first (MM/DD/YY)."

^self asStringUsingFormat: #(2 1 3 $/ 1 2 $: false )
%

category: 'Inquiries'
method: Date
weekdayIndex
"Return a SmallInteger between 1 and 7 representing the day of the week of the
 receiver where Monday is 1 and Sunday is 7."

^ (self asDays + 1) \\ 7 + 1
%

category: 'Accessing'
method: Date
weekDayName

"Returns a String that gives the name of the day of the week described by the
 receiver, in the user's native language."

^ WeekDayNames value at: (self dayOfWeek)
%

category: 'Storing and Loading'
method: Date
writeTo: passiveObj

"Writes the passive form of the receiver into passiveObj."

passiveObj writeClass: self class.
self year writeTo: passiveObj.
self dayOfYear writeTo: passiveObj.
passiveObj space
%

category: 'Accessing'
method: Date
year

"Returns a SmallInteger that gives the year described by the receiver."

^ year
%

category: 'New Indexing Comparison'
method: Date
_classSortOrdinal

^ 50
%

category: 'Accessing'
method: Date
_daysInMonth: month

"Returns a SmallInteger that gives the number of days in the month
 specified by the Integer month."

((month == 1) or: [(month == 3) or: [(month == 5) or: [(month == 7) or:
   [(month == 8) or: [(month == 10) or: [(month == 12)]]]]]])
   ifTrue: [^ 31].
((month == 4) or: [(month == 6) or: [(month == 9) or: [(month == 11)]]])
   ifTrue: [^ 30].
(self leap)
   ifTrue: [^ 29].
^ 28
%

category: 'Formatting'
method: Date
_monthAbbrev: anIndex

"Private.  Returns a three-letter String that gives the abbreviation, in the
 user's native language, of the name of the month whose numeric index is
 anIndex.  The index is a number between 1 and 12 inclusive, where 1 signifies
 January."

|theMonth itsAbbrev|

theMonth := Date nameOfMonth: anIndex.  "get its full name"
itsAbbrev := String new.
1 to: 3 do: "take the first three letters"
   [:aChar | itsAbbrev := itsAbbrev , (theMonth at: aChar)].
^ itsAbbrev
%

category: 'Private'
method: Date
_year: yr day: dy
  "used by PassiveObject "
   year := yr .
   dayOfYear := dy
%

category: 'Accessing'
method: Date
_yearMonthDay

"Private.  Returns a three-element Array of SmallIntegers containing the year,
 the index of the month, and the day of the month described by the receiver."

<primitive: 239 >

^ self _primitiveFailed: #_yearMonthDay
%

! Class extensions for 'DateTime'

!		Class methods for 'DateTime'

removeallmethods DateTime
removeallclassmethods DateTime

category: 'Instance Creation'
classmethod: DateTime
basicNew

"Returns a variant instance"

^ (self newGmtWithYear: 1901 dayOfYear: 0 seconds: 0 timeZone: TimeZone current) copy
%

category: 'Instance Creation'
classmethod: DateTime
fromStream: aStream

"Creates and returns an instance of the receiver by reading a String from
 aStream.  The String expresses local time in the default format
 (DD/MM/YYYY HH:MM:SS).  Generates an error if the String does not conform to
 the format."

^ self fromStream: aStream usingFormat: #(1 2 3 $/ 1 1 $: true true false)
%

category: 'Instance Creation'
classmethod: DateTime
fromStream: aStream usingFormat: anArray

"Creates and returns an instance of the receiver by reading a String from
 aStream.  The String expresses local time in the format specified by
 anArray.  The expression is terminated either by a space Character or by the
 end of the Stream.  Generates an error if the String does not conform to the
 format, or if anArray contains an incorrect formatting specification.

 See the class documentation of DateTime for a complete description of the
 String-formatting specification Array.

 If the month format (5th element) indicates either an abbreviation (2) or an
 entire name (3), then this method tries to determine the month by decoding a
 character substring.  That substring may include any number of Characters, but
 must exactly match a legal month name (or the initial portion of that month
 name).  If the substring matches more than one month, the first month matched
 is used (the search begins with January).

 If the specification indicates that seconds should not be included (9th
 element is false), and aString includes seconds, this method generates an
 error."

| aDateTime localTimeZone |

localTimeZone := TimeZone current.
aDateTime := (self fromStreamGmt: aStream usingFormat: anArray)
             subtractSeconds: (localTimeZone secondsFromGmt).
aDateTime isDst
  ifTrue:  [ ^ aDateTime subtractSeconds: (localTimeZone secondsForDst)]
  ifFalse: [ ^ aDateTime ].
%

category: 'Instance Creation'
classmethod: DateTime
fromStreamGmt: aStream

"Creates and returns an instance of the receiver by reading a String from
 aStream.  The String expresses Greenwich Mean Time in the default format
 (DD/MM/YYYY HH:MM:SS).  Generates an error if the String does not conform to
 the format."

^ self fromStreamGmt: aStream usingFormat: #(1 2 3 $/ 1 1 $: true true false)
%

category: 'Instance Creation'
classmethod: DateTime
fromStreamGmt: aStream usingFormat: anArray

"Creates and returns an instance of the receiver by reading a String from
 aStream.  The String expresses Greenwich Mean Time in the format specified by
 anArray.  The expression is terminated either by a space Character or by the
 end of the Stream.  Generates an error if the String does not conform to the
 format, or if anArray contains an incorrect formatting specification.

 See the class documentation of DateTime for a complete description of the
 String-formatting specification Array.

 If the month format (5th element) indicates either an abbreviation (2) or an
 entire name (3), then this method tries to determine the month by decoding a
 character substring.  That substring may include any number of Characters, but
 must exactly match a legal month name (or the initial portion of that month
 name).  If the substring matches more than one month, the first month matched
 is used (the search begins with January).

 If the specification indicates that seconds should not be included (9th
 element is false), and aString includes seconds, this method generates an
 error."

| dayInt monthInt yearInt hourInt minInt secInt timeDelim dateDelim ampm
  ampmPresent secondsPresent timePresent result parseDay
  blkNo parseMonth parseYear parseField |

"This block returns a string up from the input stream up to the specified
 delimiter.  If also allows an end-of-file if that parameter is set true.
 It then skips over the delimiter if it is found.
"
parseField := [ :delim :allowEof | | str |
                str := aStream contents class new.
                [ ((aStream peek isEquivalent: delim) == false ) and:[ aStream atEnd ==false ] ]
                whileTrue:
                  [ str add: aStream next ].
                (aStream atEnd)
                ifTrue:
                  [ allowEof
                    ifFalse:
                      [ self _error: #rtErrBadFormat args: { aStream } ]
                  ]
                ifFalse:
                  [ aStream next "skip over delimiter" ].
                str
             ].

parseDay:= "parse day"
  [ :delim | | nextField |
    nextField := parseField value: delim value: (delim == $  ).
    dayInt := Integer fromCompleteString: nextField
  ].
parseMonth:= "parse month"
  [ :delim | | nextField |
    nextField := parseField value: delim value: (delim == $  ).
    (nextField =  '' )
    ifTrue:
      [ self _error: #rtErrBadFormat args: { aStream }].
    (anArray at: 5) == 1
    ifTrue:
      [ monthInt := Integer fromCompleteString: nextField ]
    ifFalse:
      [ monthInt := Date _getMonthFrom: nextField ].
    (monthInt < 1 or:[ monthInt > 12 ] )
    ifTrue:
      [ self _error: #rtErrBadFormat args: { aStream } ]
  ].
parseYear := "parse year"
  [ :delim | | nextField |
    nextField := parseField value: delim value: (delim == $  ).
    yearInt := Integer fromCompleteString: nextField.
    (anArray at: 6) == 2
    ifTrue:
      [ (yearInt > 99)
        ifFalse: [yearInt := yearInt + ((self now yearGmt) // 100 * 100) ]
      ]
  ].

self _checkReadStream: aStream forClass: CharacterCollection.

self _checkFormat: anArray.

dateDelim := anArray at: 4.
timeDelim := anArray at: 7.
timePresent := anArray at: 8.

"parse the date, with day, month, year in the specified format order"
true ifTrue:[ | delim |
  delim:= {  dateDelim . dateDelim .  $ }.
  1 to: 3 do: [:i | blkNo:= anArray at: i.
            (blkNo == 1) ifTrue: [parseDay value: (delim at: i)].
            (blkNo == 2) ifTrue: [parseMonth value: (delim at: i)].
            (blkNo == 3) ifTrue: [parseYear value: (delim at: i)]
  ].
].
timePresent ifTrue:[ "read the time"
    secondsPresent := anArray at: 9.
    ampmPresent := anArray at: 10.
    hourInt := Integer fromCompleteString: (parseField value: timeDelim value: false).
    minInt := Integer fromCompleteString:
                     (parseField value: (secondsPresent
                                         ifTrue: [timeDelim]
                                         ifFalse: [$ ])
                                 value: (secondsPresent not and:[ampmPresent not]) ).
    secondsPresent ifTrue: [
      secInt := Integer fromCompleteString: (parseField value: $  value: ampmPresent not)]
    ifFalse:[ secInt := 0 ].

    ampmPresent ifTrue: [
        hourInt < 0 ifTrue: [
          self _error: #rtErrBadFormat args: { aStream }].
        hourInt > 12 ifTrue: [
          self _error: #rtErrBadFormat args: { aStream }].
	(ampm := String new) add: (aStream next); add: aStream next.
        (ampm isEquivalent: 'PM') ifTrue: [
	  hourInt := hourInt + 12.
	  hourInt == 24 ifTrue: [
            hourInt := 12].
          ]
        ifFalse: [
	  (ampm isEquivalent: 'AM') ifFalse: [
            self _error: #rtErrBadFormat args: { aStream } ].
	  hourInt == 12 ifTrue: [
            hourInt := 0].
          ].
      ]
] ifFalse:[ "ignore the time"
    hourInt := 0.
    minInt := 0.
    secInt := 0
].
result := self newGmtWithYear: yearInt
                month: monthInt
                day: dayInt
                hours: hourInt
                minutes: minInt
                seconds: secInt.

"This is an easy way to test that all of the values specified were in
 in range.  If any of them were not, the result will be different
 than what we specified."

(result asPartsGmt = { yearInt . monthInt . dayInt . hourInt . minInt . secInt })
ifFalse:
  [ self _error: #rtErrBadFormat args: { aStream } ].

^ result
%

category: 'Instance Creation'
classmethod: DateTime
fromString: aString

"Creates and returns an instance of the receiver from the String aString.
 The String expresses local time in the default format
 (DD/MM/YYYY HH:MM:SS).  Generates an error if the String does not conform to
 the format."

^ self fromString: aString usingFormat: #(1 2 3 $/ 1 1 $: true true false)
%

category: 'Instance Creation'
classmethod: DateTime
fromString: aString usingFormat: anArray

"Creates and returns an instance of the receiver from the String aString.
 The String expresses local time in the format specified by anArray.
 The expression is terminated either by a space Character or by the end of the
 String.  Generates an error if the String does not conform to the format,
 or if anArray contains an incorrect formatting specification.

 See the class documentation of DateTime for a complete description of the
 String-formatting specification Array.

 If the month format (5th element) indicates either an abbreviation (2) or an
 entire name (3), then this method tries to determine the month by decoding a
 character substring.  That substring may include any number of Characters, but
 must exactly match a legal month name (or the initial portion of that month
 name).  If the substring matches more than one month, the first month matched
 is used (the search begins with January).

 If the specification indicates that seconds should not be included (9th
 element is false), and aString includes seconds, this method generates an
 error."

| aDateTime localTimeZone |

localTimeZone := TimeZone current.
aDateTime := (self fromStringGmt: aString usingFormat: anArray)
             subtractSeconds: (localTimeZone secondsFromGmt).
aDateTime isDst
  ifTrue:  [ ^ aDateTime subtractSeconds: (localTimeZone secondsForDst)]
  ifFalse: [ ^ aDateTime ].
%

category: 'Instance Creation'
classmethod: DateTime
fromStringGmt: aString

"Creates and returns an instance of the receiver from the String aString.
 The String expresses Greenwich Mean Time in the default format
 (DD/MM/YYYY HH:MM:SS).  Generates an error if the String does not conform to
 the format."

^ self fromStringGmt: aString usingFormat: #(1 2 3 $/ 1 1 $: true true false)
%

category: 'Instance Creation'
classmethod: DateTime
fromStringGmt: aString usingFormat: anArray

"Creates and returns an instance of the receiver from the String aString.
 The String expresses Greenwich Mean Time in the format specified by anArray.
 The expression is terminated either by a space Character or by the end of the
 String.  Generates an error if the String does not conform to the format,
 or if anArray contains an incorrect formatting specification.

 See the class documentation of DateTime for a complete description of the
 String-formatting specification Array.

 If the month format (5th element) indicates either an abbreviation (2) or an
 entire name (3), then this method tries to determine the month by decoding a
 character substring.  That substring may include any number of Characters, but
 must exactly match a legal month name (or the initial portion of that month
 name).  If the substring matches more than one month, the first month matched
 is used (the search begins with January).

 If the specification indicates that seconds should not be included (9th
 element is false), and aString includes seconds, this method generates an
 error."

| s result |

s := ReadStreamPortable on: aString.
result := self fromStreamGmt: s usingFormat: anArray.
[ s atEnd ]
whileFalse:
  [ (s next == $  )
    ifFalse:
      [ self _errIncorrectFormat: aString ]
  ].
^ result
%

category: 'Storing and Loading'
classmethod: DateTime
loadFrom: passiveObj

"Creates and returns an active instance of the receiver from the passive form
 of the object, which expresses itself in Greenwich Mean Time."

| inst year dayCount ms zone marker |
marker := passiveObj objectPositionMarker.
passiveObj version >= 510
  ifTrue: [
     passiveObj readNamedIV.
     year := passiveObj ivValue.
     passiveObj readNamedIV.
     dayCount := passiveObj ivValue.
     passiveObj readNamedIV.
     ms := passiveObj ivValue.
     passiveObj readNamedIV.
     zone := passiveObj ivValue.
     passiveObj skipNamedInstVars.

     inst := self _newGmtWithYear: year dayOfYear: dayCount milliseconds: ms
               timeZone: zone .
    ]
  ifFalse: [
    inst := self _newGmtWithYear: passiveObj readObject
               month: passiveObj readObject
               day: passiveObj readObject
               seconds: passiveObj readObject
               timeZone: TimeZone current.
    passiveObj version >= 500
      ifFalse: [ "convert 4.1 local time to GMT"
        inst := inst addSeconds: Time gmtOffsetSeconds
        ].
    ].
passiveObj hasRead: inst marker: marker .
^inst.
%

category: 'Instance Creation'
classmethod: DateTime
migrateNew

"Override default migrateNew behavior with #_basicNew."

^ self _basicNew
%

category: 'General Inquiries'
classmethod: DateTime
nameOfMonth: anIndex

"Returns a String that gives the name, in the user's native language, of the
 month of the year whose numeric index is anIndex.  The index is a number
 between 1 and 12 inclusive, where 1 signifies January."

^ (MonthNames value) at: anIndex.
%

category: 'Instance Creation'
classmethod: DateTime
new

"Disallowed. To create a new DateTime, use another instance creation method."

self shouldNotImplement: #new
%

category: 'Instance Creation'
classmethod: DateTime
new: anInteger

"Disallowed. To create a new DateTime, use another instance creation method."

self shouldNotImplement: #new:
%

category: 'Private'
classmethod: DateTime
newGmtWithDate: aDate seconds: secsGmt

"Creates and returns an instance of the receiver from the specified values,
 in the current time zone."

^ self newGmtWithDate: aDate seconds: secsGmt timeZone: (TimeZone current).
%

category: 'Private'
classmethod: DateTime
newGmtWithDate: aDate seconds: secsGmt timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 in the time zone specified."

^ self newGmtWithYear: (aDate year) dayOfYear: (aDate dayOfYear)
  seconds: secsGmt  timeZone: aTimeZone.
%

category: 'Instance Creation'
classmethod: DateTime
newGmtWithDate: aDate time: aTime timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 in the time zone specified."

^ self newGmtWithYear: (aDate year) dayOfYear: (aDate dayOfYear)
  seconds: (aTime asSeconds) timeZone: aTimeZone.
%

category: 'Instance Creation'
classmethod: DateTime
newGmtWithYear: year dayOfYear: dayCount milliseconds: ms

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

^ self newGmtWithYear: year dayOfYear: dayCount milliseconds: ms
  timeZone: (TimeZone current).
%

category: 'Instance Creation'
classmethod: DateTime
newGmtWithYear: year dayOfYear: dayCount milliseconds: ms timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

<primitive: 314>

aTimeZone _validateClass: TimeZone.
(year _isSmallInteger and: [ (dayCount _isSmallInteger) ])
  ifFalse: [
    self _error: #rtErrDateTimeOutOfRange
    ].

(year >= 1901)
  ifFalse: [ ^ year _error: #rtErrBadDateTimeArgs ].
((dayCount <= 366) and: [ dayCount > 0 ])
  ifFalse: [ ^ dayCount _error: #rtErrBadDateTimeArgs ].

(ms _isInteger)
  ifTrue: [ | extraDays newMs |
    extraDays := ms // 86400000.
    newMs := ms \\ 86400000.
    ^ (self newGmtWithYear: year dayOfYear: dayCount milliseconds: newMs
      timeZone: aTimeZone) addDays: extraDays.
    ]
  ifFalse: [ ^ ms _error:  #rtErrBadDateTimeArgs ].

^ self _primitiveFailed: #newGmtWithYear:dayOfYear:milliseconds:timeZone:
       args: { year . dayCount . ms . aTimeZone }
%

category: 'Instance Creation'
classmethod: DateTime
newGmtWithYear: year dayOfYear: dayCount seconds: seconds

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

^ self newGmtWithYear: year dayOfYear: dayCount seconds: seconds
       timeZone: (TimeZone current).
%

category: 'Instance Creation'
classmethod: DateTime
newGmtWithYear: year dayOfYear: dayCount seconds: seconds timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

^ self newGmtWithYear: year dayOfYear: dayCount
       milliseconds: (seconds * 1000) timeZone: aTimeZone.
%

category: 'Instance Creation'
classmethod: DateTime
newGmtWithYear: yearInt month: monthInt day: dayInt hours: hourInt
  minutes: minuteInt seconds: secondInt

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

^ self newGmtWithYear: yearInt month: monthInt day: dayInt hours: hourInt
       minutes: minuteInt seconds: secondInt timeZone: (TimeZone current).
%

category: 'Instance Creation'
classmethod: DateTime
newGmtWithYear: yearInt month: monthInt day: dayInt hours: hourInt
  minutes: minuteInt seconds: secondInt ms: msInt

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

| dt |
dt := self newGmtWithYear: yearInt
           month: monthInt
           day: dayInt
           hours: hourInt
           minutes: minuteInt
           seconds: secondInt
           timeZone: (TimeZone current) .
^dt addMs: msInt
%

category: 'Instance Creation'
classmethod: DateTime
newGmtWithYear: yearInt month: monthInt day: dayInt hours: hourInt
  minutes: minuteInt seconds: secondInt timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

<primitive: 312>

   "Check the class of the arguments"
yearInt _validateClass: SmallInteger.
monthInt _validateClass: SmallInteger.
dayInt _validateClass: SmallInteger.
hourInt _validateClass: SmallInteger.
minuteInt _validateClass: SmallInteger.
secondInt _validateClass: Integer.
aTimeZone _validateClass: TimeZone.

(yearInt >= 1901)
   ifFalse: [yearInt _error: #rtErrBadDateTimeArgs].
((hourInt >= 0) and: [hourInt <= 23])
   ifFalse: [hourInt _error: #rtErrBadDateTimeArgs].
((minuteInt >= 0) and: [minuteInt <= 59])
   ifFalse: [minuteInt _error: #rtErrBadDateTimeArgs].
((secondInt >= 0) and: [secondInt <= 59])
   ifFalse: [secondInt _error: #rtErrBadDateTimeArgs].
((monthInt >=1) and: [monthInt <= 12])
   ifFalse: [monthInt _error: #rtErrBadDateTimeArgs].
dayInt _error: #rtErrBadDateTimeArgs.
^ self _primitiveFailed: #newGmtWithYear:month:day:hours:minutes:seconds:timeZone:
       args: { yearInt . monthInt . dayInt . hourInt .
               minuteInt . secondInt . aTimeZone }
%

category: 'Instance Creation'
classmethod: DateTime
newGmtWithYear: yearInt month: monthInt day: dayInt milliseconds: millisecondsInt

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

^ self _newGmtWithYear: yearInt
       month: monthInt
       day: dayInt
       milliseconds: millisecondsInt
       timeZone: TimeZone current
%

category: 'Private'
classmethod: DateTime
newWithDate: aDate milliseconds: ms

"Creates and returns an instance of the receiver from the specified values,
 in the current time zone."

^ self newWithDate: aDate milliseconds: ms timeZone: (TimeZone current).
%

category: 'Private'
classmethod: DateTime
newWithDate: aDate milliseconds: ms timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 in the time zone specified."

| aDateTime |

aDateTime := (self newGmtWithYear: (aDate year) dayOfYear: (aDate dayOfYear)
                   milliseconds: ms timeZone: aTimeZone  )
             subtractSeconds: (aTimeZone secondsFromGmt).

(aDateTime isDstIn: aTimeZone)
  ifTrue: [ ^ aDateTime subtractSeconds: (aTimeZone secondsForDst) ]
  ifFalse: [ ^ aDateTime ].
%

category: 'Private'
classmethod: DateTime
newWithDate: aDate seconds: secs

"Creates and returns an instance of the receiver from the specified values,
 in the current time zone."

^ self newWithDate: aDate seconds: secs timeZone: (TimeZone current).
%

category: 'Private'
classmethod: DateTime
newWithDate: aDate seconds: secs timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 in the time zone specified."

| aDateTime |

aDateTime := (self newGmtWithYear: (aDate year) dayOfYear: (aDate dayOfYear)
             seconds: secs timeZone: aTimeZone)
             subtractSeconds: (aTimeZone secondsFromGmt).

(aDateTime isDstIn: aTimeZone)
  ifTrue: [ ^ aDateTime subtractSeconds: (aTimeZone secondsForDst) ]
  ifFalse: [ ^ aDateTime ].
%

category: 'Instance Creation'
classmethod: DateTime
newWithDate: aDate time: aTime

"Creates and returns an instance of the receiver from the specified values,
 in the current time zone."

^ self newWithDate: aDate time: aTime timeZone: (TimeZone current).
%

category: 'Instance Creation'
classmethod: DateTime
newWithDate: aDate time: aTime timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 in the time zone specified."

| aDateTime |

aDateTime := (self newGmtWithYear: (aDate year) dayOfYear: (aDate dayOfYear)
             milliseconds: (aTime asMilliseconds rounded) timeZone: aTimeZone)   "fix 51332"
             subtractSeconds: (aTimeZone secondsFromGmt).

(aDateTime isDstIn: aTimeZone)
  ifTrue: [ ^ aDateTime subtractSeconds: (aTimeZone secondsForDst) ]
  ifFalse: [ ^ aDateTime ].
%

category: 'Instance Creation'
classmethod: DateTime
newWithYear: year dayOfYear: dayCount seconds: seconds

"Creates and returns an instance of the receiver from the specified values,
 which express local time."

^ self newWithYear: year dayOfYear: dayCount seconds: seconds
       timeZone: (TimeZone current).
%

category: 'Instance Creation'
classmethod: DateTime
newWithYear: year dayOfYear: dayCount seconds: seconds timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 in the time zone specified."

| aDateTime |

aDateTime := (self newGmtWithYear: year dayOfYear: dayCount
              seconds: seconds timeZone: aTimeZone)
             subtractSeconds: (aTimeZone secondsFromGmt).

(aDateTime isDstIn: aTimeZone)
  ifTrue: [ ^ aDateTime subtractSeconds: (aTimeZone secondsForDst) ]
  ifFalse: [ ^ aDateTime ].
%

category: 'Instance Creation'
classmethod: DateTime
newWithYear: yearInt month: monthInt day: dayInt hours: hourInt
  minutes: minuteInt seconds: secondInt

"Creates and returns an instance of the receiver from the specified values,
 which express local time."

^ self newWithYear: yearInt month: monthInt day: dayInt hours: hourInt
       minutes: minuteInt seconds: secondInt timeZone: (TimeZone current).
%

category: 'Instance Creation'
classmethod: DateTime
newWithYear: yearInt month: monthInt day: dayInt hours: hourInt
  minutes: minuteInt seconds: secondInt timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 as represented in the time zone specified."

| aDateTime |

aDateTime := (self newGmtWithYear: yearInt month: monthInt day: dayInt
                   hours: hourInt minutes: minuteInt seconds: secondInt
                   timeZone: aTimeZone)
             subtractSeconds: (aTimeZone secondsFromGmt).

(aDateTime isDstIn: aTimeZone)
  ifTrue:  [ ^ aDateTime subtractSeconds: (aTimeZone secondsForDst)]
  ifFalse: [ ^ aDateTime ].
%

category: 'Instance Creation'
classmethod: DateTime
now: aTimeZone

"Creates and returns an instance of the receiver representing the current
 time in the time zone specified."

<primitive: 237>
^ self _primitiveFailed: #now
%

category: 'Deprecated'
classmethod: DateTime
today

"Creates and returns an instance of the receiver representing the 00:00 hrs,
 today, local time."

self deprecated: 'DateTime>>today deprecated v3.0, use Date today or DateTime now'.
^ self today: (TimeZone current).
%

category: 'Deprecated'
classmethod: DateTime
today: aTimeZone

"Creates and returns an instance of the receiver representing the 00:00 hrs,
 today, local time."

self deprecated: 'DateTime class>>today: deprecated v3.0, use #now:'.
^ self newWithDate: (Date today) seconds: 0 timeZone: aTimeZone.
%

category: 'Instance Creation'
classmethod: DateTime
whenStoneStarted

"Return a new instance of the receiver which corresponds to when the
 repository was started."

^ self now subtractSeconds: System secondsSinceStoneStarted
%

category: 'Private'
classmethod: DateTime
_checkFormat: anArray

"Private. Verifies that anArray is a valid string-formatting specification for
 the receiver.  Generates an error if it is not."

"anArray is a format Array as documented in DateTime>>asStringUsingFormat:"

| v doTime |

anArray _isArray ifFalse:[ anArray _validateClass: Array ].
(anArray size < 8 or:[ anArray size > 12])
  ifTrue:[ self _error: #rtErrBadFormatSpec args: { anArray } ].

"Check for a combination of the digits 1, 2, and 3"
((anArray at: 1) + (anArray at: 2) + (anArray at: 3) == 6 and:
        [(anArray at: 1) * (anArray at: 2) * (anArray at: 3) == 6])
  ifFalse:[ self _error: #rtErrBadFormatSpec args: { anArray } ].

v := anArray at: 4 .
v class ~~ Character ifTrue:[ v _validateClass: CharacterCollection ].

((v := anArray at: 5) == 1 or: [v == 2 or: [v == 3]])
  ifFalse:[ self _error: #rtErrBadFormatSpec args: { anArray } ].

((anArray at: 6) == 1 or: [(anArray at: 6) == 2])
  ifFalse:[ self _error: #rtErrBadFormatSpec args: { anArray } ].

(doTime := anArray at: 8) _validateClass: Boolean.
doTime ifTrue:[
  anArray size >= 10
    ifFalse:[ self _error: #rtErrBadFormatSpec args: { anArray } ].
  v := anArray at: 7 .
  v class ~~ Character ifTrue:[ v _validateClass: CharacterCollection].
  (anArray at: 9) _validateClass: Boolean.
  (anArray at: 10) _validateClass: Boolean.
  ].

(anArray size > 10)
  ifTrue: [ | doTimeZone |
    (doTimeZone := anArray at: 11) _validateClass: Boolean.
    doTimeZone ifTrue: [
      (anArray size > 11) ifTrue: [ (anArray at: 12) _validateClass: Boolean ]
      ].
    ].
%

category: 'Private'
classmethod: DateTime
_newGmtWithYear: year dayOfYear: dayCount milliseconds: ms

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

^ self _newGmtWithYear: year dayOfYear: dayCount milliseconds: ms
       timeZone: (TimeZone current).
%

category: 'Private'
classmethod: DateTime
_newGmtWithYear: year dayOfYear: dayCount milliseconds: ms timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

| extraDays newMillisecs newDayOfYear |

(ms abs >= 86400000)
  ifTrue: [
    extraDays := ms // 86400000.
    newDayOfYear := dayCount + extraDays.
    newMillisecs := ms \\ 86400000.
    ]
  ifFalse: [
    newDayOfYear := dayCount.
    newMillisecs := ms.
    ].

^ self newGmtWithYear: year dayOfYear: newDayOfYear milliseconds: newMillisecs
       timeZone: aTimeZone.
%

category: 'Private'
classmethod: DateTime
_newGmtWithYear: year dayOfYear: dayCount seconds: seconds

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

^ self _newGmtWithYear: year dayOfYear: dayCount seconds: seconds
       timeZone: (TimeZone current).
%

category: 'Private'
classmethod: DateTime
_newGmtWithYear: year dayOfYear: dayCount seconds: seconds timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

| extraDays newSeconds newDayOfYear |

(seconds abs >= 86400)
  ifTrue: [
    extraDays := seconds // 86400.
    newDayOfYear := dayCount + extraDays.
    newSeconds := seconds \\ 86400.
    ]
  ifFalse: [
    newDayOfYear := dayCount.
    newSeconds := seconds.
    ].

^ self newGmtWithYear: year dayOfYear: newDayOfYear seconds: newSeconds
%

category: 'Private'
classmethod: DateTime
_newGmtWithYear: yearInt month: monthInt day: dayInt
milliseconds: millisecInt timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

<primitive: 238>

aTimeZone _validateClass: TimeZone.
( yearInt _isSmallInteger and:
[ (monthInt _isSmallInteger) and:
[ (dayInt _isSmallInteger) and: [millisecInt _isSmallInteger]]] )
  ifFalse: [
    self _error: #rtErrDateTimeOutOfRange
    ].

(yearInt >= 1901)
  ifFalse: [ yearInt _error: #rtErrBadDateTimeArgs ].
((monthInt >=  1) and: [ monthInt <= 12 ])
  ifFalse: [ ^ monthInt _error: #rtErrBadDateTimeArgs ].
((dayInt >=  1) and: [ dayInt <= 31 ])
  ifFalse: [ ^ dayInt _error: #rtErrBadDateTimeArgs ].
((millisecInt > 0) and: [ millisecInt <= 86399999 ])
  ifFalse: [ ^ millisecInt _error:  #rtErrBadDateTimeArgs ].

^ self _primitiveFailed: #_newGmtWithYear:month:day:milliseconds:timeZone:
       args: { yearInt . monthInt . dayInt . millisecInt . aTimeZone }
%

category: 'Private'
classmethod: DateTime
_newGmtWithYear: yearInt month: monthInt day: dayInt
seconds: secondInt

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

^ self _newGmtWithYear: yearInt month: monthInt day: dayInt
       seconds: secondInt timeZone: (TimeZone current).
%

category: 'Private'
classmethod: DateTime
_newGmtWithYear: yearInt month: monthInt day: dayInt
seconds: secondInt timeZone: aTimeZone

"Creates and returns an instance of the receiver from the specified values,
 which express Greenwich Mean Time."

^ self _newGmtWithYear: yearInt month: monthInt day: dayInt
       milliseconds: (secondInt * 1000) timeZone: aTimeZone.
%

category: 'Private'
classmethod: DateTime
_newJulianDay: julianDay localTime: milliseconds

"Returns a new instance of the receiver based on the specified values. The
 values julianDay and milliseconds represent the local time."

| aTimeZone aDateTime |

aTimeZone := TimeZone current.
aDateTime := (self _newJulianDay: julianDay millisecond: milliseconds
              timeZone: aTimeZone)
             subtractSeconds: (aTimeZone secondsFromGmt).

(aDateTime isDstIn: aTimeZone)
  ifTrue:  [ ^ aDateTime subtractSeconds: (aTimeZone secondsForDst)]
  ifFalse: [ ^ aDateTime ].
%

category: 'Private'
classmethod: DateTime
_newJulianDay: julianDay millisecond: ms

"Returns a new instance of the receiver based on the specified values. The
 values julianDay and milliseconds should represent Greenwich Mean Time."

^ self _newJulianDay: julianDay millisecond: ms timeZone: (TimeZone current).
%

category: 'Private'
classmethod: DateTime
_newJulianDay: julianDay millisecond: ms timeZone: aTimeZone

"Returns a new instance of the receiver based on the specified values. The
 values julianDay and milliseconds should represent Greenwich Mean Time."

<primitive: 236>
julianDay _validateClass: SmallInteger.
ms _validateClass: SmallInteger.
aTimeZone _validateClass: TimeZone.
{ julianDay . ms . aTimeZone } _error: #rtErrBadDateTimeArgs.
^ self _primitiveFailed: #_newJulianDay:milliseconds:timeZone
       args: { julianDay . ms . aTimeZone }
%

category: 'Private'
classmethod: DateTime
_newJulianDay: julianDay second: sec

"Private. Creates and returns an instance of the receiver from the specified
 values, which express Greenwich Mean Time. Generates an error if any of the
 values are out of range."

^ self _newJulianDay: julianDay second: sec timeZone: (TimeZone current).
%

category: 'Private'
classmethod: DateTime
_newJulianDay: julianDay second: sec timeZone: aTimeZone

"Private. Creates and returns an instance of the receiver from the specified
 values, which express Greenwich Mean Time. Generates an error if any of the
 values are out of range."

^ self _newJulianDay: julianDay millisecond: (sec * 1000)
    timeZone: aTimeZone.
%

!		Instance methods for 'DateTime'

category: 'Comparing'
method: DateTime
< aDateTime

"Returns true if the receiver represents a moment in time before that of the
 argument, and false if it doesn't.  Generates an error if the argument is not
 a DateTime."

| argYear argDayOfYear |

argYear := aDateTime yearGmt.
(year < argYear) ifTrue: [ ^true ].
(year > argYear) ifTrue: [ ^false ].

"The years are the same"

argDayOfYear := aDateTime dayOfYearGmt .
(dayOfYear < argDayOfYear) ifTrue: [ ^true ].
(dayOfYear > argDayOfYear) ifTrue: [ ^false ].

"The days are the same"

^ milliseconds < aDateTime millisecondsGmt
%

category: 'Comparing'
method: DateTime
= aDateTime

"Returns true if the receiver represents the same moment in time as that of the
 argument, and false if it does not."

self == aDateTime ifTrue:[ ^ true ].

(aDateTime isKindOf: self class) ifFalse:
    [^false].
^(year = aDateTime yearGmt) and:
     [(dayOfYear = aDateTime dayOfYearGmt) and:
          [milliseconds = aDateTime millisecondsGmt ]]
%

category: 'Comparing'
method: DateTime
> aDateTime

"Returns true if the receiver represents a moment in time after that of the
 argument, and false if it doesn't.  Generates an error if the argument is not
 a DateTime."

| argYear argDayOfYear |

argYear := aDateTime yearGmt.
(year > argYear) ifTrue: [ ^true ].
(year < argYear) ifTrue: [ ^false ].

"The years are the same"

argDayOfYear := aDateTime dayOfYearGmt.
(dayOfYear > argDayOfYear) ifTrue: [ ^true ].
(dayOfYear < argDayOfYear) ifTrue: [ ^false ].

"The days are the same"

^ milliseconds > aDateTime millisecondsGmt
%

category: 'Arithmetic'
method: DateTime
addDays: anInteger

"Returns a DateTime that describes a moment in time anInteger days
 later than that of the receiver."

| newDate  |

newDate := self asDate addDays: anInteger.
^ self class newWithDate: newDate milliseconds: self timeSinceMidnightMs .
%

category: 'Arithmetic'
method: DateTime
addHours: aNumber

"Returns a DateTime that describes a moment in time aNumber hours
 later than that of the receiver."

^ (self class) _newGmtWithYear: year dayOfYear: dayOfYear
                  milliseconds: (milliseconds + (aNumber * 3600000)) asInteger
                  timeZone: timeZone.
%

category: 'Arithmetic'
method: DateTime
addMilliseconds: aNumber

"Returns a DateTime that describes a moment in time aNumber milliseconds
 later than that of the receiver."

^ (self class) _newGmtWithYear: year dayOfYear: dayOfYear
		milliseconds: (milliseconds + aNumber ) asInteger
                timeZone: timeZone.
%

category: 'Arithmetic'
method: DateTime
addMinutes: aNumber

"Returns a DateTime that describes a moment in time aNumber minutes
 later than that of the receiver."

^ (self class) _newGmtWithYear: year dayOfYear: dayOfYear
           milliseconds: (milliseconds + (aNumber * 60000)) asInteger
           timeZone: timeZone.
%

category: 'Arithmetic'
method: DateTime
addMonths: anInteger

"Returns a DateTime that describes a moment in time anInteger months
 later than that of the receiver.

 This method attempts to keep the day of the month the same.  If the
 new month has fewer days than the receiver's original month, then it
 truncates to the last day of the new month."

| newDate |

newDate := self asDate addMonths: anInteger.
^ self class newWithDate: newDate milliseconds: self timeSinceMidnightMs .
%

category: 'Arithmetic'
method: DateTime
addMs: aNumber

"Return a DateTime that describes a moment in time aNumber milliseconds
 later than that of the receiver."

^ self class _newGmtWithYear: year
             dayOfYear: dayOfYear
             milliseconds: (milliseconds + aNumber) asInteger
             timeZone: timeZone
%

category: 'Arithmetic'
method: DateTime
addSeconds: aNumber

"Returns a DateTime that describes a moment in time aNumber seconds
 later than that of the receiver."

^ (self class) _newGmtWithYear: year dayOfYear: dayOfYear
		milliseconds: (milliseconds + (aNumber * 1000)) asInteger
                timeZone: timeZone.
%

category: 'Arithmetic'
method: DateTime
addWeeks: anInteger

"Returns a DateTime that describes a moment in time anInteger weeks
 later than that of the receiver."

^ self addDays: (anInteger * 7).
%

category: 'Arithmetic'
method: DateTime
addYears: anInteger

"Returns a DateTime that describes a moment in time anInteger years
 later than that of the receiver."

| newDate |

newDate := self asDate addYears: anInteger.
^ self class newWithDate: newDate milliseconds: self timeSinceMidnightMs .
%

category: 'Converting'
method: DateTime
asDate

"Returns a Date that represents the receiver, as expressed in local time."

^ self asDateIn: TimeZone current.
%

category: 'Converting'
method: DateTime
asDateIn: aTimeZone
	"Returns a Date that represents the receiver,
	as expressed in the specified time zone.
	Bug 40674 plb 2010.05.20"

	| offset aDateTime |
	offset := self _localOffset: aTimeZone.
	aDateTime := offset = 0
		ifTrue: [self]
		ifFalse: [self addSeconds: offset].
	^self dateClass
		newDay: aDateTime dayOfYearGmt
		year: aDateTime yearGmt.
%

category: 'Converting'
method: DateTime
asDateTime

"Returns the receiver."

^ self.
%

category: 'Converting'
method: DateTime
asDays

"Returns an Integer that represents the receiver in units of days since
 January 1, 1901, local time."

^ self asDate asDays.
%

category: 'Converting'
method: DateTime
asDaysGmt

"Returns an Integer that represents the receiver in units of days since
 January 1, 1901, Greenwich Mean Time."

| numYears numDays |

numYears := year - 1901.
numDays := (numYears * 365) + (numYears // 4) +
           ((numYears + 300) // 400) - (numYears // 100) + dayOfYear - 1.
^ numDays.
%

category: 'Converting'
method: DateTime
asMillisecondsGmt

"Returns an Integer that represents the receiver in units of milliseconds since
 midnight January 1, 1901, Greenwich Mean Time."

^ ((self asDaysGmt) * 86400000) + milliseconds
%

category: 'Converting'
method: DateTime
asParts

"Returns an Array of six SmallIntegers (year month day hours minutes seconds)
 that expresses the receiver in local time."

^ (self addSeconds: (self _localOffset)) asPartsGmt  .
%

category: 'Converting'
method: DateTime
asPartsGmt

"Returns an Array of six SmallIntegers (year month day hours minutes seconds)
 that expresses the receiver in Greenwich Mean Time."

| result |

result := self _yearMonthDayGmt.  "year/month/day"
result addLast: (milliseconds // 3600000).  "hours"
result addLast: (milliseconds \\ 3600000) // 60000.  "minutes"
result addLast: (milliseconds // 1000  \\ 60).  "seconds"
^ result
%

category: 'Converting'
method: DateTime
asPosixSeconds

"Returns a SmallDouble that represents the receiver in units of seconds since
 midnight January 1, 1970, Coordinated Universal Time."

	^self asMillisecondsGmt - 2177452800000 / 1000.0
%

category: 'Converting'
method: DateTime
asSeconds

"Returns an Integer that represents the receiver in units of seconds since
 midnight January 1, 1901, local time."

^ self asSecondsGmt + self _localOffset.
%

category: 'Converting'
method: DateTime
asSecondsGmt

"Returns an Integer that represents the receiver in units of seconds since
 midnight January 1, 1901, Greenwich Mean Time."

^ ((self asDaysGmt) * 86400) + (milliseconds // 1000)
%

category: 'Formatting'
method: DateTime
asString

"Returns a String that expresses the receiver in local time
 in the default format (DD/MM/YYYY HH:MM:SS)."

^self asStringUsingFormat: #(1 2 3 $/ 1 1 $: true true false)
%

category: 'Formatting'
method: DateTime
asStringGmt

"Returns a String that expresses the receiver in Greenwich Mean Time
 in the default format (DD/MM/YYYY HH:MM:SS)."

| t result |

t := self _yearMonthDayGmt.
result := (t at: 3) _digitsAsString .
result addAll: '/';
  addAll: (t at: 2) _digitsAsString;
  addAll: '/';
  addAll: (t at: 1) _digitsAsString;
  addAll: ' ';
  addAll: (milliseconds // 3600000) _digitsAsString;
  addAll: ':';
  addAll: ((milliseconds // 1000 \\ 3600) // 60) _digitsAsString;
  addAll: ':';
  addAll: (milliseconds // 1000 \\ 60 ) truncated _digitsAsString .
^ result
%

category: 'Formatting'
method: DateTime
asStringGmtMs
"Returns a String that expresses the receiver in Greenwich Mean Time
 in the default format (DD/MM/YYYY HH:MM:SS.mmm)."

| result |
result := self asStringGmt .
result add: $. ;
  addAll: (milliseconds \\ 1000) _digitsAsString3  .
^ result
%

category: 'Formatting'
method: DateTime
asStringGmtUsingFormat: anArray

"Returns a String that expresses the receiver in Greenwich Mean Time
 in the format defined by anArray.  Generates an error if anArray
 contains an incorrect formatting specification.

 See the class documentation of DateTime for a complete description of the
 String-formatting specification Array."

| aString |

((anArray size >= 11) and: [ anArray at: 11 ])
  ifTrue: [
    aString := self asStringStdUsingFormat: anArray.
    aString add: Character space.
    aString addAll: 'Greenwich Mean Time'.
    ]
  ifFalse: [
    aString := self asStringStdUsingFormat: anArray.
    ].

^ aString.
%

category: 'Formatting'
method: DateTime
asStringISO8601
  "yyyy-mm-ddThh:mm:ss+zzzz"

  | string offset offsetHours offsetMinutes |
  string := self asStringUsingFormat: #(3 2 1 $- 1 1 $: true true false true true).
  string at: 11 put: $T.
  string := string copyFrom: 1 to: 19.
  offset := timeZone offsetAtLocal: self.
  string add: (offset < 0
    ifTrue: [$-]
    ifFalse: [$+]).
  offset := offset abs // 60.
  offsetHours := offset // 60.
  offsetMinutes := offset \\ 60.
  offsetHours < 10 ifTrue: [string add: $0].
  string addAll: offsetHours printString.
  offsetMinutes < 10 ifTrue: [string add: $0].
  string addAll: offsetMinutes printString.
  ^ string
%

category: 'Formatting'
method: DateTime
asStringMs
"Returns a String that expresses the receiver in local time
 in the default format (DD/MM/YYYY HH:MM:SS.mmm)."

| result |
result := self asString .
result add: $. ;
  addAll: (milliseconds \\ 1000) _digitsAsString3  .
^ result
%

category: 'Private'
method: DateTime
asStringStdUsingFormat: anArray

"Returns a String that expresses the receiver according to its
 contents without regard to the time zone parameter."

|t dateSeparator timeSeparator monthName aString
 hour hourInt min sec day yearNumber |

t := self _yearMonthDayGmt.
self class _checkFormat: anArray.
dateSeparator := (anArray at: 4) asString.

timeSeparator := (anArray at: 7) asString.

((anArray at: 5) == 2) "get the month name according to the format"
   ifTrue: [monthName := self _monthAbbrev: (t at: 2)]
   ifFalse: [((anArray at: 5) == 3) "month as number is default"
      ifTrue: [monthName := Date nameOfMonth: (t at: 2)]
      ifFalse: [monthName := (t at: 2) _digitsAsString]].

((anArray at: 6) == 2)
   ifTrue: [yearNumber := ((t at: 1) \\ 100) _digitsAsString]
   ifFalse: [yearNumber := (t at: 1) asString].  "YYYY is default"

day := (t at:3) _digitsAsString.
((anArray at: 1) == 2) "month first"
   ifTrue: [aString := monthName , dateSeparator]
   ifFalse: [((anArray at: 1) == 3) "yearNumber first"
      ifTrue: [aString := yearNumber , dateSeparator]
      ifFalse: [aString := day , dateSeparator]].  "day first is default"

((anArray at: 2) == 1) "day second"
   ifTrue: [aString addAll: day; addAll: dateSeparator] "yearNumber second"
   ifFalse: [((anArray at: 2) == 3) "month second is default"
      ifTrue: [aString addAll: yearNumber; addAll: dateSeparator]
      ifFalse: [aString addAll: monthName; addAll: dateSeparator]].

((anArray at: 3) == 1) "day third"
   ifTrue: [aString addAll: day]
   ifFalse: [((anArray at: 3) == 2) "month third"
      ifTrue: [aString addAll: monthName]
      ifFalse: [aString addAll: yearNumber]].  "yearNumber third is default"

hourInt := (milliseconds // 3600000) .
hour := hourInt _digitsAsString.
min := (milliseconds // 1000 \\ 3600 // 60) _digitsAsString.
sec := (milliseconds // 1000 \\ 60) _digitsAsString.

(anArray at: 8) ifTrue: [ "print the time"
  aString add: $ .
  (anArray at: 10) ifTrue: [ "12-hour format"
    (hourInt > 12) ifTrue: [
      aString addAll: (hourInt - 12) _digitsAsString;
              addAll: timeSeparator;
              addAll: min.

      (anArray at: 9) ifTrue: [
        aString addAll: timeSeparator;
                addAll: sec
        ].
      ]
    ifFalse: [
      aString addAll: (hourInt == 0 ifTrue: ['12'] ifFalse: [hour]);
              addAll: timeSeparator;
              addAll: min.

      (anArray at: 9) ifTrue: [
        aString addAll: timeSeparator;
                addAll: sec.
        ].
      ].

    aString addAll: (hourInt >= 12 ifTrue: [' PM'] ifFalse: [' AM']).
    ]
  ifFalse: [
    aString addAll: hour;
            addAll: timeSeparator;
            addAll: min.

    (anArray at: 9) ifTrue: [
      aString addAll: timeSeparator;
              addAll: sec.
      ].
    ].
  ].

^ aString
%

category: 'Formatting'
method: DateTime
asStringUsingFormat: anArray

"Returns a String that expresses the receiver in local time
 in the format defined by anArray.  Generates an error if anArray
 contains an incorrect formatting specification.

 See the class documentation of DateTime for a complete description of the
 String-formatting specification Array."

| localDt aString |

((anArray size >= 11) and: [ anArray at: 11 ])
  ifTrue: [
    | aTimeZone |
    ((anArray size >= 12) and: [ anArray at: 12 ])
      ifTrue: [ aTimeZone := self timeZone ]
      ifFalse: [ aTimeZone := TimeZone current ].
    localDt := self addSeconds: (self _localOffset: aTimeZone).
    aString := localDt asStringStdUsingFormat: anArray.
    aString add: Character space.
    (self isDstIn: aTimeZone)
      ifTrue: [ aString addAll: aTimeZone dstPrintString ]
      ifFalse: [ aString addAll: aTimeZone standardPrintString ].
    ]
  ifFalse: [
    localDt := self addSeconds: self _localOffset.
    aString := localDt asStringStdUsingFormat: anArray.
    ].

^ aString.
%

category: 'Converting'
method: DateTime
asTime

"Returns a Time which represents the time since midnight represented by the
 receiver, as expressed in local time."

^ self timeClass fromMilliseconds: self timeSinceMidnightMs
%

category: 'Accessing'
method: DateTime
at: anIndex put: aValue

"Disallowed. Changes to a DateTime a not allowed."

^ self shouldNotImplement: #at:put:
%

category: 'Repository Conversion'
method: DateTime
convertToDateTime

"Returns the receiver."

^ self.
%

category: 'Converting'
method: DateTime
dateClass

"Returns the class of the object to be created when the receiver is being
 converted to a date."

^ Date
%

category: 'Accessing'
method: DateTime
day

"Returns a SmallInteger that gives the day of the month described by the
 receiver, expressed in local time."

^ self asDate day.
%

category: 'Accessing'
method: DateTime
dayGmt

"Returns a SmallInteger that gives the day in the month described by the
 receiver, expressed in Greenwich Mean Time."

^  (self _yearMonthDayGmt) at: 3
%

category: 'Accessing'
method: DateTime
dayOfMonth

"Returns a SmallInteger that gives the day of the month described by the
 receiver, expressed in local time."

^self day
%

category: 'Accessing'
method: DateTime
dayOfMonthGmt

"Returns a SmallInteger that gives the day of the month described by the
 receiver, expressed in Greenwich Mean Time."

^ self dayGmt
%

category: 'Accessing'
method: DateTime
dayOfWeek

"Returns a SmallInteger that gives the numeric index of the day of the week
 described by the receiver, expressed in local time.  The index is a
 number between 1 and 7 inclusive, where 1 signifies Sunday."

^ self asDate dayOfWeek.
%

category: 'Accessing'
method: DateTime
dayOfWeekGmt

"Returns a SmallInteger that gives the numeric index of the day of the week
 described by the receiver, expressed in Greenwich Mean Time.  The index is a
 number between 1 and 7 inclusive, where 1 signifies Sunday."

    ^(DateTime fromString: (self asStringGmt)) dayOfWeek
%

category: 'Accessing'
method: DateTime
dayOfYear

"Returns a SmallInteger that gives the day of the year described by the
 receiver, expressed in local time."

^ self asDate dayOfYear.
%

category: 'Accessing'
method: DateTime
dayOfYearGmt

"Returns a SmallInteger that gives the day of the year described by the
 receiver, expressed in Greenwich Mean Time."

^ dayOfYear
%

category: 'Deprecated'
method: DateTime
daysInMonth

"Returns a SmallInteger that gives the number of days in the month
 described by the receiver, expressed in local time."

self deprecated: 'DateTime>>daysInMonth deprecated v3.0, use Date>>daysInMonth'.
^ self asDate daysInMonth.
%

category: 'Deprecated'
method: DateTime
daysInMonthGmt

"Returns a SmallInteger that gives the number of days in the month described by
 the receiver, expressed in Greenwich Mean Time."

self deprecated: 'DateTime>>daysInMonthGmt deprecated v3.0, use Date protocols'.
^ self _daysInMonth: self monthOfYearGmt
%

category: 'Accessing'
method: DateTime
daysInYear

"Returns a SmallInteger that gives the number of days in the year
 described by the receiver."

(self leap) ifTrue: [^ 366].
^ 365
%

category: 'Comparing'
method: DateTime
hash

"Returns an Integer hash code for the receiver."

^ (((year hash) bitShift: -1) bitXor: (dayOfYear hash)) bitXor: (self secondGmt hash).
%

category: 'Accessing'
method: DateTime
hour

"Returns the hour in the 24-hour clock represented by the receiver in local
 time."

^ self _secondsLocal // 3600
%

category: 'Accessing'
method: DateTime
hourGmt

"Returns the hour in the 24-hour clock represented by the receiver, as
 expressed in Greenwich Mean Time"

^ milliseconds // 3600000
%

category: 'Accessing'
method: DateTime
hours

"Returns a SmallInteger (between zero and 23 inclusive) that gives the number of
 hours represented by the receiver since midnight, local time."

^ self hour
%

category: 'Accessing'
method: DateTime
hoursGmt

"Returns a SmallInteger (between zero and 23 inclusive) that gives the number of
 hours represented by the receiver since midnight, Greenwich Mean Time."

^ self hourGmt
%

category: 'Private'
method: DateTime
isDst

^ self isDstIn: (TimeZone current)
%

category: 'Private'
method: DateTime
isDstIn: aTimeZone

| startOfDst endOfDst aYear|

(aTimeZone secondsForDst == 0)
  ifTrue: [ ^ false "No Dst"].

aYear := self yearGmt.
(aTimeZone yearStartDst > aYear)
  ifTrue: [ ^ false "Dst did not begin until later on"].

(startOfDst := aTimeZone startOfDstFor: aYear) == nil ifTrue: [^false].
(endOfDst   := aTimeZone endOfDstFor: aYear) == nil ifTrue: [^false].

(startOfDst < endOfDst)
  ifTrue: [ "Northern Hemisphere"
    ^ ((self >= startOfDst) and: [ self < endOfDst ])
    ]
  ifFalse: [ "Southern Hemisphere"
    ^ ((self >= startOfDst) or: [ self < endOfDst ])
    ]
%

category: 'Accessing'
method: DateTime
julianDay

"Returns the Julian Day of the receiver, expressed in local time. The julian
 day is a SmallInteger that gives the number of days since January 1, 4713 B.C.,
 as defined in Communications of the ACM, algorithm #199."

^ self asDate julianDay.
%

category: 'Deprecated'
method: DateTime
julianSecond

"Returns a SmallInteger (between zero and 86399 inclusive) that gives the number
 of seconds represented by the receiver since midnight, local time."

self deprecated: 'DateTime>>julianSecond  deprecated v3.0, use #timeSinceMidnight'.
^ self timeSinceMidnight.
%

category: 'Accessing'
method: DateTime
leap

"Returns true if the receiver describes a leap year, expressed in local time,
 and false if it does not."

^ self asDate leap.
%

category: 'Accessing'
method: DateTime
leapGmt

"Returns true if the receiver describes a leap year, expressed in Greenwich Mean
 Time, and false if it does not."

| yr |

"a year is a leap year if: (it is evenly divisible by 4 and it is not a
 century year) or (it is a century year and evenly divisible by 400)"

yr := (self _yearMonthDayGmt) at: 1.
((yr \\ 100) == 0) ifTrue: [^ ((yr \\ 400) == 0)].
^ ((yr \\ 4) == 0)
%

category: 'Accessing'
method: DateTime
millisecondsGmt

"Returns the value of the milliseconds instance variable."

^ milliseconds.
%

category: 'Accessing'
method: DateTime
minute

"Returns a SmallInteger (between zero and 59 inclusive) that gives the minute
 of the time represented by the receiver in local time."

^ (self _secondsLocal \\ 3600) // 60
%

category: 'Accessing'
method: DateTime
minuteGmt

"Returns a SmallInteger (between zero and 59 inclusive) that gives the minute
 of the time represented by the receiver in Greenwich Mean Time."

^ milliseconds \\ 3600000 // 60000
%

category: 'Accessing'
method: DateTime
minutes

"Returns a SmallInteger (between zero and 59 inclusive) that gives the number of
 minutes represented by the receiver since the previous hour, local time."

^ self minute
%

category: 'Accessing'
method: DateTime
minutesGmt

"Returns a SmallInteger (between zero and 59 inclusive) that gives the number of
 minutes represented by the receiver since the previous hour, Greenwich Mean
 Time."

^ self minuteGmt
%

category: 'Accessing'
method: DateTime
month

"Returns a SmallInteger that gives the numeric index of the month of the year
 described by the receiver, expressed in local time.  The index is a number
 between 1 and 12 inclusive, where 1 signifies January."

^ self asDate month
%

category: 'Accessing'
method: DateTime
monthGmt

"Returns a SmallInteger that gives the numeric index of the month of the year
 described by the receiver, expressed in Greenwich Mean Time. The index is a
 number between 1 and 12 inclusive, where 1 signifies January."

^  (self _yearMonthDayGmt) at: 2
%

category: 'Accessing'
method: DateTime
monthName

"Returns a String that gives the name of the month of the year described by the
 receiver, expressed in local time, in the user's native language."

^ self asDate monthName.
%

category: 'Accessing'
method: DateTime
monthNameGmt

"Returns a String that gives the name of the month of the year described by the
 receiver, expressed in Greenwich Mean Time, in the user's native language."

^ MonthNames value at: self monthGmt
%

category: 'Accessing'
method: DateTime
monthOfYear

"Returns a SmallInteger that gives the numeric index of the month of the year
 described by the receiver, expressed in local time.  The index is a number
 between 1 and 12 inclusive, where 1 signifies January."

^ self month
%

category: 'Accessing'
method: DateTime
monthOfYearGmt

"Returns a SmallInteger that gives the numeric index of the month of the year
 described by the receiver, expressed in Greenwich Mean Time.  The index is a
 number between 1 and 12 inclusive, where 1 signifies January."

^ self monthGmt
%

category: 'Formatting'
method: DateTime
printJsonOn: aStream
	"JSON does not specify the format for dates and times, so we use ISO 8601"
	self asStringISO8601 printJsonOn: aStream  "fix 50391"
%

category: 'Formatting'
method: DateTime
printOn: aStream

"Puts a displayable representation of the receiver, expressed in local
 time, on aStream."

aStream nextPutAll: self asString .
%

category: 'Accessing'
method: DateTime
second

"Returns a SmallInteger (between zero and 59 inclusive) that gives the number of
 seconds represented by the receiver since the previous minute."

^ self _secondsLocal \\ 60.
%

category: 'Accessing'
method: DateTime
secondGmt

"Returns a SmallInteger (between zero and 59 inclusive) that gives the number
 of seconds represented by the receiver, as expressed in Greenwich Mean Time,
 since the previous minute."

^ milliseconds // 1000 \\ 60.
%

category: 'Accessing'
method: DateTime
seconds

"Returns a SmallInteger (between zero and 59 inclusive) that gives the number of
 seconds represented by the receiver since the previous minute."

^ self second.
%

category: 'Accessing'
method: DateTime
secondsGmt

"Returns a SmallInteger (between zero and 59 inclusive) that gives the number
 of seconds represented by the receiver, as expressed in Greenwich Mean Time,
 since the previous minute."

^ self secondGmt.
%

category: 'Accessing'
method: DateTime
size: anInteger

"Disallowed.  You may not change the size of a DateTime."

self shouldNotImplement: #size:
%

category: 'Arithmetic'
method: DateTime
subtractDate: aDateTime

"Returns a positive Integer that counts the number of times midnight local time
 occurs between the times described by the receiver and aDateTime."

^ (self asDate) subtractDate: (aDateTime asDate)
%

category: 'Arithmetic'
method: DateTime
subtractDateGmt: aDateTime

"Returns a positive Integer that counts the number of times that midnight
 Greenwich Mean Time occurs between the times described by the receiver and
 aDateTime."

^ (self asDays - aDateTime asDays) abs
%

category: 'Arithmetic'
method: DateTime
subtractDays: anInteger

"Returns a DateTime that describes a moment in time anInteger days
 earlier than that of the receiver."

^ self addDays: (anInteger negated).
%

category: 'Arithmetic'
method: DateTime
subtractHours: aNumber

"Returns a DateTime that describes a moment in time aNumber hours
 earlier than that of the receiver."

^ self addHours: (aNumber negated).
%

category: 'Arithmetic'
method: DateTime
subtractMinutes: aNumber

"Returns a DateTime that describes a moment in time aNumber minutes
 earlier than that of the receiver."

^ self addMinutes: (aNumber negated).
%

category: 'Arithmetic'
method: DateTime
subtractMonths: anInteger

"Returns a DateTime that describes a moment in time anInteger months
 earlier than that of the receiver.

 This method attempts to keep the day of the month the same.  If the
 new month has fewer days than the receiver's original month, then it
 truncates to the last day of the new month."

^ self addMonths: (anInteger negated).
%

category: 'Arithmetic'
method: DateTime
subtractSeconds: aNumber

"Returns a DateTime that describes a moment in time aNumber seconds
 earlier than that of the receiver."

^ self addSeconds: (aNumber negated).
%

category: 'Arithmetic'
method: DateTime
subtractTime: aDateTime

"Returns an Array of three positive Integers that count the hours, minutes, and
 seconds, respectively, between the times of day described by the receiver and
 aDateTime.

 The computation ignores the dates of both the receiver and aDateTime, and
 assumes that the receiver is the later time.  Hence, if the time of day in the
 receiver is less than the time of day in aDateTime, then the receiver's time of
 day is assumed to occur on the day following that of aDateTime."

| parts h m s |

parts := self asParts.
h := parts at: 4.
m := parts at: 5.
s := parts at: 6.

parts := aDateTime asParts.
h < (parts at: 4) ifTrue:[ h := h + 24 ].
h := h - (parts at: 4).
m := m - (parts at: 5).
s := s - (parts at: 6).

s < 0 ifTrue: [
  s := s + 60.
  m := m - 1
].
s > 60 ifTrue: [
  s := s - 60.
  m := m + 1
].
m < 0 ifTrue: [
  m := m + 60.
  h := h - 1
].
m > 60 ifTrue: [
  m := m - 60.
  h := h + 1
].
^{ h . m . s }
%

category: 'Arithmetic'
method: DateTime
subtractWeeks: anInteger

"Returns a DateTime that describes a moment in time anInteger weeks
 earlier than that of the receiver."

^ self subtractDays: (anInteger * 7).
%

category: 'Arithmetic'
method: DateTime
subtractYears: anInteger

"Returns a DateTime that describes a moment in time anInteger years
 earlier than that of the receiver."

^ self addYears: (anInteger negated).
%

category: 'Deprecated'
method: DateTime
timeAsSeconds

"Returns a SmallInteger (between zero and 86399 inclusive) that gives the number
 of seconds represented by the receiver since midnight, local time."

self deprecated: 'DateTime>>timeAsSeconds deprecated v3.0, use #timeSinceMidnight'.
^ self timeSinceMidnight.
%

category: 'Deprecated'
method: DateTime
timeAsSecondsGmt

"Returns a SmallInteger (between zero and 86399 inclusive) that gives the number
 of seconds represented by the receiver since midnight, Greenwich Mean Time."

self deprecated: 'DateTime>>timeAsSecondsGmt deprecated v3.0, use #timeSinceMidnightGmt'.
^ self timeSinceMidnightGmt.
%

category: 'Converting'
method: DateTime
timeClass

^ Time
%

category: 'Deprecated'
method: DateTime
timeCompare: aDateTime
 "Provided for compatibility.

  Answer 0 is the receiver's time component is the same as that of
  aDateTime. Otherwise, -1 is the receiver's is less than, or 1 if
  the receiver's is greater than."

| otherMs |
self deprecated: 'DateTime>>timeCompare: deprecated v3.0, use other comparison methods'.
otherMs := aDateTime millisecondsGmt.
milliseconds = otherMs ifTrue: [^0].
milliseconds < otherMs ifTrue: [^-1].
^1
%

category: 'Converting'
method: DateTime
timeSinceMidnight

"Returns a SmallInteger (between zero and 86399 inclusive) that gives the number
 of seconds represented by the receiver since midnight, local time."

| secs tz |

secs := self timeSinceMidnightGmt + (tz := TimeZone current) secondsFromGmt.
self isDst ifTrue: [ secs := secs + tz secondsForDst ].
secs := (secs + 86400) \\ 86400.
^ secs.
%

category: 'Converting'
method: DateTime
timeSinceMidnightGmt

"Returns a SmallInteger (between zero and 86399 inclusive) that gives the number
 of seconds represented by the receiver since midnight, Greenwich Mean Time."

^ milliseconds // 1000.
%

category: 'Converting'
method: DateTime
timeSinceMidnightMs

"Returns a SmallInteger (between zero and 86399999 inclusive) that gives the number
 of milliseconds represented by the receiver since midnight, local time."

| ms tz|
ms := milliseconds + ((tz := TimeZone current) secondsFromGmt * 1000) .
self isDst ifTrue:[ ms := ms + (tz secondsForDst * 1000) ].
ms := (ms + 86400000) \\ 86400000.
^ ms.
%

category: 'Accessing'
method: DateTime
timeZone

^ timeZone
%

category: 'Converting'
method: DateTime
timeZone: aTimeZone

"Returns a new DateTime with the specified TimeZone. The new DateTime
 represents the same point in time as the receiver. A point in time
 is usually different when viewed as local time in different time zones."

^ self class newGmtWithYear: (self yearGmt) dayOfYear: (self dayOfYearGmt)
       milliseconds: (self millisecondsGmt) timeZone: aTimeZone.
%

category: 'Formatting'
method: DateTime
US12HrFormat

"Returns a String that expresses the receiver in local time.  The date is in
 United States format (month first) and the time of day is based on the 12-hour
 clock (MM/DD/YY HH:MM:SS pm)."

^self asStringUsingFormat: #(2 1 3 $/ 1 2 $: true false true)
%

category: 'Formatting'
method: DateTime
US24HrFormat

"Returns a String that expresses the receiver in local time.  The date is in
 United States format (month first) and the time of day is based on the 24-hour
 clock (MM/DD/YY HH:MM:SS)."

^self asStringUsingFormat: #(2 1 3 $/ 1 2 $: true false false)
%

category: 'Formatting'
method: DateTime
USDateFormat

"Returns a String that expresses the date of the receiver, expressed in local
 time. The date is in United States format, month first (MM/DD/YY)."

^self asStringUsingFormat: #(2 1 3 $/ 1 2 $: false)
%

category: 'Accessing'
method: DateTime
weekDayName

"Returns a String that gives the name of the day of the week described by the
 receiver, expressed in local time."

^ self asDate weekDayName
%

category: 'Accessing'
method: DateTime
weekDayNameGmt

"Returns a String that gives the name of the day of the week described by the
 receiver, expressed in Greenwich Mean Time."

^  WeekDayNames value at: (self dayOfWeekGmt).
%

category: 'Accessing'
method: DateTime
year

"Returns a SmallInteger that gives the year described by the receiver,
 expressed in local time."

^ self asDate year.
%

category: 'Accessing'
method: DateTime
yearGmt

"Returns a SmallInteger that gives the year described by the receiver, expressed
 in Greenwich Mean Time."

^ year
%

category: 'New Indexing Comparison'
method: DateTime
_classSortOrdinal

^ 60
%

category: 'Private'
method: DateTime
_daysInMonth: month

"Returns a SmallInteger that gives the number of days in the month
 specified by the Integer month."

((month == 1) or: [(month == 3) or: [(month == 5) or: [(month == 7) or:
   [(month == 8) or: [(month == 10) or: [(month == 12)]]]]]])
   ifTrue: [^ 31].
((month == 4) or: [(month == 6) or: [(month == 9) or: [(month == 11)]]])
   ifTrue: [^ 30].
(self leap)
   ifTrue: [^ 29].
^ 28
%

category: 'Private'
method: DateTime
_localOffset

^ self _localOffset: TimeZone current.
%

category: 'Private'
method: DateTime
_localOffset: aTimeZone

| offset |

offset := aTimeZone secondsFromGmt.
(self isDstIn: aTimeZone)
  ifTrue: [ offset := offset + aTimeZone secondsForDst ].
^ offset.
%

category: 'Formatting'
method: DateTime
_monthAbbrev: anIndex

"Private.  Returns a three-letter String that gives the abbreviation, in the
 user's native language, of the name of the month whose numeric index is
 anIndex.  The index is a number between 1 and 12 inclusive, where 1 signifies
 January."

|theMonth itsAbbrev|

theMonth := self class nameOfMonth: anIndex.  "get its full name"
itsAbbrev := String new.
1 to: 3 do: "take the first three letters"
   [:aChar | itsAbbrev := itsAbbrev , (theMonth at: aChar)].
^ itsAbbrev
%

category: 'Deprecated'
method: DateTime
_monthName: anIndex

"Private.  Returns a String that gives the name, in the user's native language,
 of the month of the year whose numeric index is anIndex.  The index is a number
 between 1 and 12 inclusive, where 1 signifies January."

self deprecated: 'DateTime>>_monthName: deprecated v3.0, use nameOfMonth:'.
^ (MonthNames value) at: anIndex.
%

category: 'Private'
method: DateTime
_secondsLocal

"Returns the number of seconds since midnight in the receiver, as expressed in
 local time."

^ ((milliseconds // 1000) + self _localOffset) \\ 86400.
%

category: 'Private'
method: DateTime
_topazAsString
 ^ self asStringMs
%

category: 'Private'
method: DateTime
_yearMonthDay

"Private. Returns a three-element Array of SmallIntegers containing the year,
 index of the month, and the day of the month represented by the receiver,
 expressed in local time."

^ (self addSeconds: (self _localOffset)) _yearMonthDayGmt.
%

category: 'Private'
method: DateTime
_yearMonthDayGmt

"Private. Returns a three-element Array of SmallIntegers containing the year,
 index of the month, and the day of the month represented by the receiver,
 expressed in Greenwich Mean Time."

<primitive: 239 >
^ self _primitiveFailed: #_yearMonthDayGmt
%

! Class extensions for 'Deprecated'

!		Instance methods for 'Deprecated'

removeallmethods Deprecated
removeallclassmethods Deprecated

category: 'Instance initialization'
method: Deprecated
initialize
  gsNumber := ERR_Deprecated.
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'DoubleByteString'

!		Class methods for 'DoubleByteString'

removeallmethods DoubleByteString
removeallclassmethods DoubleByteString

category: 'Formatting'
classmethod: DoubleByteString
charSize

"Returns number of bytes that make up a character for this string class"

^ 2
%

category: 'Instance Creation'
classmethod: DoubleByteString
new

"Returns a new instance of the receiver."

^ self _basicNew
%

category: 'Instance Creation'
classmethod: DoubleByteString
new: aSize

"Returns a new instance of the receiver, of sufficient size to hold aSize
 Characters of two bytes each."

^ self _basicNew:  aSize + aSize
%

category: 'Instance Creation'
classmethod: DoubleByteString
withAll: aString

"Returns an instance of the receiver or an instance of QuadByteString,
 using the minimum bytes per character required to represent the argument."

<primitive: 456>
^ QuadByteString withAll: aString
%

!		Instance methods for 'DoubleByteString'

category: 'Concatenating'
method: DoubleByteString
, aCharOrCharColl

"Returns a new instance of the receiver's class that contains the elements of
 the receiver followed by the elements of aCharOrCharColl.  The argument
 must be a String, a DoubleByteString, or a Character.

 The result may not be an instance of the class of the receiver if one of the
 following rules applies:

 1) If the receiver is a DoubleByteSymbol, the result is a DoubleByteString.

 Warning: Creating a new instance and copying the receiver take time.  If you
 can safely modify the receiver, it can be much faster to use the addAll:
 method.  See the documentation of the Concatenating category of class
 SequenceableCollection for more details."

<primitive: 614 >
^ self copy addAll: aCharOrCharColl; yourself
%

category: 'Adding'
method: DoubleByteString
add: aCharOrCharColl

"Appends all of the elements of aCharOrCharColl to the receiver and returns
 aCharOrCharColl."

<primitive: 437>
| cSize |
aCharOrCharColl class == Character ifTrue:[ | cp |
  cp := aCharOrCharColl codePoint .
  (cp >= 16rD800 and:[ cp <= 16rDFFF]) ifTrue:[
    ^ OutOfRange signal:'codePoint 16r', cp asHexString ,' is illegal'.
  ].
  cp > 16rFFFF ifTrue:[
    ^ self _convertToQuadByte add: aCharOrCharColl
  ].
  ^ self _primitiveFailed: #add: args: { aCharOrCharColl }
].
(cSize := aCharOrCharColl stringCharSize) == 4 ifTrue:[
  aCharOrCharColl _asDoubleByteString ifNotNil:[ :aString |
    ^ self addAll: aString
  ].
  ^ self _convertToQuadByte addAll: aCharOrCharColl
].
aCharOrCharColl _validateClass: CharacterCollection .
^ self add: aCharOrCharColl asDoubleByteString
%

category: 'Adding'
method: DoubleByteString
addAll: aCharOrCharColl

"Equivalent to add: aCharOrCharColl."

<primitive: 437>
| cSize |
(cSize := aCharOrCharColl stringCharSize) == 4 ifTrue:[
  aCharOrCharColl _asDoubleByteString ifNotNil:[ :aString |
    ^ self addAll: aString
  ].
  ^ self _convertToQuadByte addAll: aCharOrCharColl
].
aCharOrCharColl class == Character ifTrue:[ | cp |
  cp := aCharOrCharColl codePoint .
  (cp >= 16rD800 and:[ cp <= 16rDFFF]) ifTrue:[
    ^ OutOfRange signal:'codePoint 16r', cp asHexString ,' is illegal'.
  ].
  cp > 16rFFFF ifTrue:[
    ^ self _convertToQuadByte add: aCharOrCharColl
  ].
  ^ self _primitiveFailed: #add: args: { aCharOrCharColl }
].
(aCharOrCharColl isKindOf: CharacterCollection) ifTrue:[
  ^ self add: aCharOrCharColl asDoubleByteString.
].
aCharOrCharColl do: [:each | self add: each].
^aCharOrCharColl.
%

category: 'Adding'
method: DoubleByteString
addCodePoint: aSmallInteger

<primitive: 1048>
^ self add: (Character codePoint: aSmallInteger).
%

category: 'Adding'
method: DoubleByteString
addLast: aCharOrCharColl

"Equivalent to add: aCharOrCharColl."

<primitive: 437>
| cSize |
aCharOrCharColl class == Character ifTrue:[ | cp |
  cp := aCharOrCharColl codePoint .
  (cp >= 16rD800 and:[ cp <= 16rDFFF]) ifTrue:[
    ^ OutOfRange signal:'codePoint 16r', cp asHexString ,' is illegal'.
  ].
  cp > 16rFFFF ifTrue:[
    ^ self _convertToQuadByte add: aCharOrCharColl
  ].
  ^ self _primitiveFailed: #add: args: { aCharOrCharColl }
].
(cSize := aCharOrCharColl stringCharSize) == 4 ifTrue:[
  aCharOrCharColl _asDoubleByteString ifNotNil:[ :aString |
    ^ self addAll: aString
  ].
  ^ self _convertToQuadByte addAll: aCharOrCharColl
].
aCharOrCharColl _validateClass: CharacterCollection .
^ self add: aCharOrCharColl asDoubleByteString

%

category: 'Deprecated'
method: DoubleByteString
asciiLessThan: aString

"Returns true if the receiver collates before the argument using the
 ASCII collating table, which collates AB...Z...ab..z."

self deprecated: 'DoubleByteString>>asciiLessThan: deprecated in 3.2'.

^ self lessThan: aString collatingTable: DoubleByteAsciiCollatingTable
%

category: 'Concatenating'
method: DoubleByteString
asDoubleByteString
 ^ self
%

category: 'Accessing'
method: DoubleByteString
at: anIndex

"Returns the Character at anIndex."

<primitive: 435>

(anIndex _isSmallInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex } .
self _uncontinuableError
%

category: 'Updating'
method: DoubleByteString
at: anIndex put: aChar

"Stores aChar at the specified location and returns aChar."

<primitive: 436>
(aChar class == Character) ifTrue:[ | cp |
  (cp := aChar codePoint) > 16rFFFF ifTrue:[
    ^ self _convertToQuadByte at:anIndex put:aChar
  ].
  (cp >= 16rD800 and:[ cp <= 16rDFFF]) ifTrue:[
    ^ OutOfRange signal:'codePoint 16r', cp asHexString ,' is illegal'.
    ].
] ifFalse:[
  aChar _validateClass: Character .
].
(anIndex _isSmallInteger) ifTrue: [
  ((anIndex > (self size + 1)) or: [anIndex <= 0]) ifTrue: [
    ^ self _errorIndexOutOfRange: anIndex
    ].
] ifFalse:[
  ^ self _errorNonIntegerIndex: anIndex
] .
self _primitiveFailed: #at:put: args: { anIndex . aChar }
%

category: 'Accessing'
method: DoubleByteString
atOrNil: anIndex

"Returns the Character at anIndex, or nil if anIndex is beyond end of the receiver"

<primitive: 435>
(anIndex _isSmallInteger)
  ifTrue: [ ^ nil ]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #atOrNil: args: { anIndex } .
self _uncontinuableError
%

category: 'Formatting'
method: DoubleByteString
charSize

"Returns number of bytes that make up a character for this string class."

^ 2
%

category: 'Accessing'
method: DoubleByteString
codePointAt: anIndex

"Returns the integer value of the Character at anIndex in the receiver."

<primitive: 1072>

(anIndex _isSmallInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #codePointAt: args: { anIndex } .
self _uncontinuableError
%

category: 'Updating'
method: DoubleByteString
codePointAt: anIndex put: aValue

"Stores aValue at the specified location, with autogrow, 
 and with auto-conversion of receiver to  QuadByteString if needed.
 Returns aValue.
"
<primitive: 1074>
aValue _validateClass: SmallInteger .
^ self at: anIndex put: (Character withValue: aValue)
%

category: 'Encoding'
method: DoubleByteString
encodeAsUTF16

"Encode receiver in UTF16 format.  Returns a Utf16 ."

<primitive: 1084>
self _primitiveFailed: #encodeAsUTF16
%

category: 'Encoding'
method: DoubleByteString
encodeAsUTF8

"Encode receiver in UTF8 format. Returns a Utf8 ."

<primitive: 468>
self _primitiveFailed: #encodeAsUTF8
%

category: 'Encoding'
method: DoubleByteString
encodeAsUTF8IntoString
"Encode receiver in UTF8 format.  Returns a String. For Seaside"

<primitive: 994>
self _primitiveFailed: #encodeAsUTF8IntoString
%

category: 'Adding'
method: DoubleByteString
insertAll: aCharOrCharColl at: anIndex

"Inserts aCharOrCharColl at the specified index.  Returns aCharOrCharColl."

<primitive: 454>
| cSize |
anIndex _isSmallInteger ifFalse:[ anIndex _validateClass: Integer ].
((anIndex <= 0) or: [anIndex > (self size + 1)])
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

(cSize := aCharOrCharColl stringCharSize) == 4 ifTrue:[ | aString |
  (aString := aCharOrCharColl _asDoubleByteString )
    ifNil:[ ^ self _convertToQuadByte insertAll: aCharOrCharColl at: anIndex]
    ifNotNil:[ ^ self insertAll: aString at: anIndex ].
].
aCharOrCharColl class == Character ifTrue:[
  ^ self insertAll: aCharOrCharColl asString at: anIndex .
].
aCharOrCharColl _validateClasses: { Character . CharacterCollection }.
^self insertAll: aCharOrCharColl asDoubleByteString at: anIndex.
%

category: 'Copying'
method: DoubleByteString
replaceFrom: startIndex to: stopIndex with: charCollection startingAt: repIndex

"Replaces the elements of the receiver between the indexes startIndex and
 stopIndex inclusive with the elements of aSeqCollection starting at repIndex.
 If aSeqCollection is identical to the receiver, the source and
 destination blocks may overlap.

 Returns the receiver."

<primitive: 439>
| cSize argInfo recInfo src16 |
startIndex _isSmallInteger ifFalse:[ startIndex _validateClass: SmallInteger ].
stopIndex _isSmallInteger ifFalse:[ stopIndex _validateClass: SmallInteger ].
repIndex _isSmallInteger ifFalse:[ repIndex _validateClass: SmallInteger ].
argInfo := charCollection _stringCharSize .
recInfo := self _stringCharSize .
(recInfo bitAnd: 16r8) ~~ 0 ifTrue:[  "receivier is a Utf string"
  (cSize := argInfo bitAnd: 16r7) > 1 ifTrue:[
    src16 := charCollection _asUnicode16 .
    (src16 ~~ nil and:[ src16 ~~ charCollection]) ifTrue:[
      ^ self replaceFrom: startIndex to: stopIndex with: src16 startingAt: repIndex
    ].
    ^ self _convertToQuadByte
     replaceFrom: startIndex to: stopIndex with: charCollection startingAt: repIndex
  ].
].
(cSize := argInfo bitAnd: 16r7) == 4 ifTrue:[
   src16 := charCollection _asDoubleByteString .
   (src16 ~~ nil and:[ src16 ~~ charCollection]) ifTrue:[
      ^ self replaceFrom: startIndex to: stopIndex with: src16 startingAt: repIndex
   ].
   ^ self _convertToQuadByte replaceFrom: startIndex to: stopIndex
                        with: charCollection startingAt: repIndex
].
^ super replaceFrom: startIndex to: stopIndex with: charCollection startingAt: repIndex
%

category: 'Accessing'
method: DoubleByteString
size

"Returns the number of Characters in the receiver."

^ self _basicSize // 2.
%

category: 'Updating'
method: DoubleByteString
size: anInteger

"Changes the size of the receiver to anInteger.

 If anInteger is less than the current size of the receiver, the receiver is
 shrunk accordingly.  If anInteger is greater than the current size of the
 receiver, the receiver is extended and new elements are initialized to
 codePoint zero."

^ super _basicSize: anInteger + anInteger
%

category: 'Private'
method: DoubleByteString
_basicAt: anInteger put: aValue

"Disallowed"
self shouldNotImplement: #_basicAt:put:
%

category: 'Private'
method: DoubleByteString
_basicSize: anInteger

"Disallowed"
self shouldNotImplement: #_basicSize:
%

category: 'Converting'
method: DoubleByteString
_convertToQuadByte

"Change the receiver from a DoubleByteString to a QuadByteString,
 or from a Unicode16 to Unicode32.  Changes the
 class of the receiver and the receiver's storage structure.
 Returns the receiver."

<primitive: 445>
%

! Class extensions for 'DoubleByteSymbol'

!		Class methods for 'DoubleByteSymbol'

removeallmethods DoubleByteSymbol
removeallclassmethods DoubleByteSymbol

category: 'Class Membership'
classmethod: DoubleByteSymbol
isSubclassOf: aClassHistoryOrClass

  (self _classHistoryIncludesIdentical: aClassHistoryOrClass) ifTrue:[ ^ true].
  ^ super isSubclassOf: aClassHistoryOrClass
%

category: 'Storing and Loading'
classmethod: DoubleByteSymbol
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| theSize str marker result |

"since Symbols can't refer to other objects, the 'hasRead:' message
 may be sent after values have been filled in."

theSize := passiveObj readSize .
str := String new .
marker := passiveObj objectPositionMarker .
passiveObj next: theSize bytesTo: str ;
           next .
result :=  (str _changeClassToMultiByte: DoubleByteString) asSymbol .
passiveObj hasRead: result marker: marker .
^ result
%

category: 'Instance Creation'
classmethod: DoubleByteSymbol
migrateNew

"Override default migrateNew behavior with #_basicNew."

^ self _basicNew
%

category: 'Instance Creation'
classmethod: DoubleByteSymbol
new

"Disallowed.  To create a new DoubleByteSymbol, use the class method
 #withAll: instead."

self shouldNotImplement: #new
%

category: 'Instance Creation'
classmethod: DoubleByteSymbol
new: size

"Disallowed.  To create a new DoubleByteSymbol, use the class method #withAll:
 instead."

self shouldNotImplement: #new:
%

category: 'Instance Creation'
classmethod: DoubleByteSymbol
withAll: aString

"Returns a canonical symbol that has the same Characters as aString."

<primitive: 300>
^ Symbol withAll: aString
%

category: 'Instance Creation'
classmethod: DoubleByteSymbol
_basicNew
^ self shouldNotImplement: #_basicNew
%

category: 'Instance Creation'
classmethod: DoubleByteSymbol
_basicNew: aSize
^ self shouldNotImplement: #_basicNew:
%

category: 'Class Membership'
classmethod: DoubleByteSymbol
_classHistoryIncludesIdentical: aClass
  ^ aClass == Symbol or:[ aClass == DoubleByteSymbol or:[ aClass == QuadByteSymbol]]
%

category: 'Instance Creation'
classmethod: DoubleByteSymbol
_existingWithAll: aString

"Return an existing canonical symbol that has the same value as 'aString'.
 If no such Symbol or DoubleByteSymbol already exists, returns nil."

<primitive: 310>
^ Symbol _existingWithAll: aString
%

category: 'Instance Creation'
classmethod: DoubleByteSymbol
_newString

"Return a new instance of DoubleByteString."

^ DoubleByteString new
%

category: 'Instance Creation'
classmethod: DoubleByteSymbol
_newString: aSize

"Return a new instance of DoubleByteString of the specified size."

^ DoubleByteString new: aSize
%

!		Instance methods for 'DoubleByteSymbol'

category: 'Concatenating'
method: DoubleByteSymbol
, aCharOrCharCollection

"Returns a new instance of DoubleByteString that contains the elements of
 the receiver followed by the elements of aCharOrCharCollection.
 A DoubleByteString is returned rather than a DoubleByteSymbol to avoid
 the expense of unnecessary creation and canonicalization of Symbols."

^ ((self _unicodeCompareEnabled ifTrue:[ Unicode16 ] ifFalse:[ DoubleByteString])
    withAll: self) addAll: aCharOrCharCollection ; yourself
%

category: 'Converting'
method: DoubleByteSymbol
-> anObject

"Returns a SymbolAssociation with the receiver as the key and the given object
 as the value."

^ SymbolAssociation newWithKey: self value: anObject
%

category: 'Comparing'
method: DoubleByteSymbol
= anObject

"Returns true if anObject is equal to the receiver.  Since symbols
 are canonicalized, this method does the check based on the
 identities of the receiver and the argument."

^ self == anObject.
%

category: 'Compatiblity'
method: DoubleByteSymbol
argumentCount

 ^ self numArgs
%

category: 'Converting'
method: DoubleByteSymbol
asDoubleByteString

"Returns a copy of the receiver as a DoubleByteString."

^ DoubleByteString withAll: self.
%

category: 'Converting'
method: DoubleByteSymbol
asString

"Returns a copy of the receiver as a String."

^ String withAll: self.
%

category: 'Converting'
method: DoubleByteSymbol
asSymbol

"Returns the receiver.  All Symbols and DoubleByteSymbols are canonical."

^ self
%

category: 'Converting'
method: DoubleByteSymbol
asSymbolKind

"Returns the receiver.  All Symbols and DoubleByteSymbols are canonical."

^ self
%

category: 'Clustering'
method: DoubleByteSymbol
cluster

"Has no effect.  Clustering of DoubleByteSymbols is only performed by
 the clusterAllSymbols method in class System . "

^ true
%

category: 'Clustering'
method: DoubleByteSymbol
clusterInBucket: aClusterBucketOrId

"Has no effect.  Clustering of Symbols is only performed by
 the clusterAllSymbols method in class System . "

^ true
%

category: 'Copying'
method: DoubleByteSymbol
copy

"Returns self.  Copies of (canonical) double byte symbols are not allowed."

^ self
%

category: 'Copying'
method: DoubleByteSymbol
copyReplacing: oldObject withObject: newObject
	"Returns a String comprising a copy of the receiver in which all occurrences
	 of objects equal to oldObject have been replaced by newObject."

	^self asDoubleByteString copyReplacing: oldObject withObject: newObject
%

category: 'Comparing'
method: DoubleByteSymbol
hash

"Returns a numeric hash key for the receiver."

^ self identityHash
%

category: 'Converting'
method: DoubleByteSymbol
keywords

"Returns an Array of the keywords in the receiver, treating
 any colon-delimited segment as if it was a legal keyword."

^ Symbol _keywords: self
%

category: 'Testing'
method: DoubleByteSymbol
precedence

"Returns the precedence of the receiver, were it a message selector, with
 1=unary, 2=binary and 3=keyword."

^ self isInfix
    ifTrue: [ 2 ]
    ifFalse: [ self isKeyword ifTrue: [ 3 ]
                              ifFalse: [ 1 ] ]
%

category: 'Formatting'
method: DoubleByteSymbol
printOn: aStream

"Puts a displayable representation of the receiver on the given stream.
 That representation conforms to GemStone Smalltalk parsing rules."

aStream nextPut: $# .
super printOn: aStream
%

category: 'Copying'
method: DoubleByteSymbol
shallowCopy

"Returns self.  Copies of (canonical) double byte symbols are not allowed."

^ self
%

category: 'Class Membership'
method: DoubleByteSymbol
species

"Returns a class similar to, or the same as, the receiver's class which
 can be used for containing derived copies of the receiver."

^ DoubleByteString
%

category: 'Private'
method: DoubleByteSymbol
speciesForConversion

^ DoubleByteString .
%

category: 'Formatting'
method: DoubleByteSymbol
withNoColons

"Returns a String containing the value of the receiver with all colons removed.

 A String is returned rather than a symbol to avoid the expense of unnecessary
 creation and canonicalization of symbols."

^ self copyWithout: $:   .
%

category: 'Decompiling without Sources'
method: DoubleByteSymbol
_asSource

| result |
result := String new .
result add: $#  .
result addAll: super _asSource .
^ result
%

category: 'New Indexing Comparison'
method: DoubleByteSymbol
_idxForCompareEqualTo: arg

""

^arg _idxForCompareEqualToDoubleByteSymbol: self
%

category: 'New Indexing Comparison'
method: DoubleByteSymbol
_idxForCompareEqualTo: aCharacterCollection collator: anIcuCollator
  ""

  ^ aCharacterCollection _idxForCompareEqualToDoubleByteSymbol: self
%

category: 'New Indexing Comparison - for Compare'
method: DoubleByteSymbol
_idxForCompareEqualToCharacterCollection: aCharacterCollection
  "second half of a double dispatch call from CharacterCollection>>_idxForCompareEqualTo:."

  ^ false
%

category: 'New Indexing Comparison - for Compare'
method: DoubleByteSymbol
_idxForCompareEqualToUnicode: aUnicodeString collator: aCollator
"second half of a double dispatch call from CharacterCollection>>_idxForCompareEqualTo:collator:."

  ^ false
%

category: 'New Indexing Comparison'
method: DoubleByteSymbol
_idxForSortEqualTo: arg
  ""

  ^ arg _idxForSortEqualToSymbol: self
%

category: 'New Indexing Comparison'
method: DoubleByteSymbol
_idxForSortEqualTo: aCharacterCollection collator: anIcuCollator
  ""

  ^ self asString
    _idxForSortEqualTo: aCharacterCollection
    collator: anIcuCollator
%

category: 'New Indexing Comparison'
method: DoubleByteSymbol
_idxForSortEqualToSymbol: aSymbol
  "second half of a double dispatch call from Symbol>>_idxForSortEqualTo:."

  ^ self asString _idxPrimCompareEqualTo: aSymbol asString
%

category: 'New Indexing Comparison'
method: DoubleByteSymbol
_idxForSortNotEqualTo: aCharacterCollection collator: anIcuCollator
  ""

  ^ (self _idxForSortEqualTo: aCharacterCollection collator: anIcuCollator) not
%

category: 'Comparing'
method: DoubleByteSymbol
~= aCharCollection

"This method can be optimized for symbols since they are canonical."

^ self ~~ aCharCollection
%

! Class extensions for 'EndOfStream'

!		Instance methods for 'EndOfStream'

removeallmethods EndOfStream
removeallclassmethods EndOfStream

category: 'Instance initialization'
method: EndOfStream
initialize
  gsNumber := ERR_EndOfStream.
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'EqualityCollisionBucket'

!		Instance methods for 'EqualityCollisionBucket'

removeallmethods EqualityCollisionBucket
removeallclassmethods EqualityCollisionBucket

category: 'Searching'
method: EqualityCollisionBucket
at: aKey put: aValue keyValDict_coll: aKeyValDict
 "Stores the aKey/aValue pair in the receiver.
  Returns self size if this at:put: added a new key, 0 if this at:put:
  replaced the value of an existing key."
 | emptySlotIdx startTableSize thisKey numElem |

  startTableSize := self tableSize .
  aKey ifNil:[ ^ self _error: #rtErrNilKey ] .
  (numElem := numElements) == 0 ifTrue:[
    emptySlotIdx := 1
  ] ifFalse:[ | idx |
    "search for aKey, or for the first empty slot "
    idx := 1 .
    1 to: startTableSize do:[:n |
      thisKey := self _at: idx . "inline keyAt:"
      thisKey ifNotNil:[
        (aKey = thisKey) ifTrue:[ "Key found.  Store given value"
          self _at: idx + 1 put: aValue .  "inline at:putValue:"
          aKeyValDict _markDirty .
          ^ 0
        ].
      ] ifNil:[
        emptySlotIdx ifNil:[ emptySlotIdx := idx ].
      ].
      idx := idx + 2 .
    ] .
    "Key not found so add key and value"
    emptySlotIdx ifNil:[ " bucket is full so grow it "
      emptySlotIdx := self _basicSize + 1 .
      self size: emptySlotIdx + 7  .  "accommodate 4 more key,value pairs"
    ] .
  ].
  numElem := numElem + 1 .
  numElements := numElem .
  self _at: emptySlotIdx put: aKey. "inline at:putKey:"
  self _at: emptySlotIdx + 1 put: aValue . "inline at:putValue"
  ^ numElem
%

category: 'Private'
method: EqualityCollisionBucket
compareKey: key1 with: key2

"Not used by implementation of EqualityCollisionBucket"

self shouldNotImplement: #compareKey:with:
%

category: 'Searching'
method: EqualityCollisionBucket
searchForKey: argKey

"Returns the index of argKey, or if not found, nil."
   | idx |
   argKey ifNil:[ ^ self _error: #rtErrNilKey ] .
   idx := 1 .
   1 to: self tableSize do: [ :n | | aKey |
      aKey := self _at: idx . "inline keyAt:"
      aKey ifNotNil:[
        argKey = aKey ifTrue:[ ^ n ].
      ].
      idx := idx + 2
   ].
   ^ nil "Key not found"
%

! Class extensions for 'Error'

!		Class methods for 'Error'

removeallmethods Error
removeallclassmethods Error

category: 'Signaling'
classmethod: Error
signalFatalSigTerm

  "Terminates the session with a Fatal error 4052.  Allows
   Smalltalk or Ruby handler to implement SIGTERM semantics."

  ^ self new _number: GS_ERR_SIGTERM ; signalNotTrappable
%

!		Instance methods for 'Error'

category: 'Instance initialization'
method: Error
initialize
  gsNumber := ERR_Error.
  gsResumable := false .
  gsTrappable := true .
%

! Class extensions for 'Exception'

!		Instance methods for 'Exception'

removeallmethods Exception
removeallclassmethods Exception

category: 'Instance initialization'
method: Exception
initialize
  gsNumber := ERR_Exception.
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'ExceptionSet'

!		Class methods for 'ExceptionSet'

removeallmethods ExceptionSet
removeallclassmethods ExceptionSet

category: 'instance creation'
classmethod: ExceptionSet
with: selector1 with: selector2
  "Private - Answer a new instance of the receiver containing
  the <exceptionSelector> arguments, selector1 and selector2"

  | res |
  res := self new .
  res , selector1 .
  res , selector2 .
  ^ res
%

!		Instance methods for 'ExceptionSet'

category: 'ANSI - exceptionSelector'
method: ExceptionSet
, anotherExceptionOrSet
  "Return an exception set that contains all of the exception selectors in both
  the receiver and the argument.
  This is commonly used to specify a set of exception selectors for an
  'exception handler.'

  The argument must be a subclass of Exception or a kind of ExceptionSet .
  Returns the receiver."

  anotherExceptionOrSet _isExceptionClass ifTrue:[
    self at: self size + 1 put: anotherExceptionOrSet  . "inline self add:"
    ^ self
  ].
  (anotherExceptionOrSet isKindOf: ExceptionSet) ifTrue:[ | sz |
    sz := self size .
    1 to: anotherExceptionOrSet size do:[:j |
      "inline self add: "
      sz := sz + 1 .
      self at: sz put: (anotherExceptionOrSet at:j)
    ].
  ].
  anotherExceptionOrSet _validateClass: ExceptionSet.
%

category: 'ANSI support'
method: ExceptionSet
add: anException

 ^self at: self size + 1 put: anException
%

category: 'ANSI support'
method: ExceptionSet
at: anOffset put: anException

 anException _isExceptionClass ifFalse:[
   anException _error: #rtErrBadArgKind args: { Error }.
   self _uncontinuableError .
 ].
 ^ super at: anOffset put: anException
%

category: 'ANSI - exceptionSelector'
method: ExceptionSet
handles: anExceptionClass

  1 to: self size do:[:j |
    ((self at: j) handles: anExceptionClass) ifTrue:[ ^ true ] .
  ].
  ^ false
%

category: 'initializing'
method: ExceptionSet
initialize

  ^ self
%

category: 'accessing'
method: ExceptionSet
selectors
  "Return the list of Exceptions contained in the receiver."
  | res |
  res := OrderedCollection new: self size .
  1 to: self size do:[:j |
    res at: j put: (self at: j) .
  ].
  ^ res
%

category: 'Exceptions'
method: ExceptionSet
selectors: anOrderedCollection
  "Store all elements of anOrderedCollection into receiver's varying instVars,
   growing or shrinking receiver to match size of anOrderedCollection."

  self size: anOrderedCollection size .
  1 to: anOrderedCollection size do:[:j |
    self at: j put: (anOrderedCollection at:j)
  ].
%

! Class extensions for 'ExecBlock'

!		Class methods for 'ExecBlock'

removeallmethods ExecBlock
removeallclassmethods ExecBlock

category: 'Storing and Loading'
classmethod: ExecBlock
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| src result marker meth o symbolList |

"Returns a new instance of the receiver read from the given PassiveObject"

marker := passiveObj objectPositionMarker.
src := passiveObj readObject.
symbolList := GsCurrentSession currentSession symbolList .
meth := src _compileInContext: (o := Object new) symbolList: symbolList.
result := meth _executeInContext: o.
(result == nil or: [(result _isExecBlock) not]) ifTrue: [
  "error in compiling"
  self _halt: 'Error in recreating a ' , name.
  ^nil
  ]
ifFalse: [
  passiveObj hasRead: result marker: marker.
  ^result
  ]
%

category: 'Accessing'
classmethod: ExecBlock
_cost

"cost is not define at the class level, each instance
 has its own cost ."

self shouldNotImplement: #_cost
%

!		Instance methods for 'ExecBlock'

category: 'Debugging Support'
method: ExecBlock
argsAndTemps

"Returns an Array of Symbols which are the names of arguments and
 temporaries for this block,  not including inner blocks"

^ self method argsAndTemps
%

category: 'Accessing'
method: ExecBlock
argumentCount

  "Returns a SmallInteger, the number of arguments to the receiver.
   argumentCount is provided to conform to ANSI Smalltalk."
  <primitive: 458>
  self _primitiveFailed: #argumentCount.
  ^0.
%

category: 'Updating'
method: ExecBlock
at: anIndex put: anObj
  "Disallowed"
  self shouldNotImplement: #at:put:
%

category: 'Clustering'
method: ExecBlock
clusterDepthFirst

 self cluster ifTrue:[ ^ true ].
 self method cluster .   "cannot use instVarAt to access"
        "other instVars are specials or transient"
 ^ false
%

category: 'Block Evaluation'
method: ExecBlock
cull: anArg
  "Return the value of the receiver evaluated with 0 or 1 arguments. If the block
   expects 1 argument pass anArg as the value of the argument"

  ^ self argumentCount == 0
    ifTrue: [ self value ]
    ifFalse: [ self value: anArg ]
%

category: 'Block Evaluation'
method: ExecBlock
cull: firstArg cull: secondArg
  "Return the value of the receiver evaluated with between 0 and 2 arguments,
   discarding arguments not needed by the receiver."

  | nargs |
  (nargs := self argumentCount) < 2 ifTrue:[
     nargs == 1 ifTrue:[ ^ self value: firstArg ].
     ^ self value .
  ].
  ^ self value: firstArg value: secondArg
%

category: 'Block Evaluation'
method: ExecBlock
cull: firstArg cull: secondArg cull: thirdArg
  "Return the value of the receiver evaluated with between 0 and 3 arguments,
   discarding arguments not needed by the receiver."

  | nargs |
  (nargs := self argumentCount) < 3 ifTrue:[
     nargs == 2 ifTrue:[ ^ self value: firstArg value: secondArg ].
     nargs == 1 ifTrue:[ ^ self value: firstArg ].
     ^ self value .
  ].
  ^ self value: firstArg value: secondArg value: thirdArg
%

category: 'Block Evaluation'
method: ExecBlock
cull: firstArg cull: secondArg cull: thirdArg cull: fourthArg
  "Return the value of the receiver evaluate with between 0 and 4 arguments,
   discarding arguments not needed by the receiver."

  | nargs |
  (nargs := self argumentCount) <= 2 ifTrue:[
     nargs == 2 ifTrue:[ ^ self value: firstArg value: secondArg ].
     nargs == 1 ifTrue:[ ^ self value: firstArg ].
     ^ self value .
  ].
  nargs == 3 ifTrue:[
     ^ self value: firstArg value: secondArg value: thirdArg .
  ].
  ^ self value: firstArg value: secondArg value: thirdArg value: fourthArg.
%

category: 'Flow of Control'
method: ExecBlock
doWhileFalse: conditionBlock
  "Evaluate the receiver once, then again as long the value of conditionBlock is false."
  | result |
  [ result := self value.
    conditionBlock value
  ] whileFalse.
  ^ result
%

category: 'Block Evaluation'
method: ExecBlock
ensure: terminationBlock
"Evaluate the receiver.
 Evaluate terminationBlock after evaluating the receiver,
 or before any return from a block that would return to the sender.
 Returns result of evaluating the receiver.

 terminationBlock must be a zero-arg instance of ExecBlock, otherwise an
 error is generated. "

<primitive: 2017>  "marks frame with ENSURE_Mark_NIL, always fails"
| result |
result := self value .  "execute the receiver"
   "_removeEnsure returns nil if the terminationBlock has already been executed."
self _removeEnsure ifNotNil:[:b | b _terminationBlockValue ].
^ result
%

category: 'Processes - Blue Book'
method: ExecBlock
fork
  "forks the receiver as a new process at the current scheduling priority"

  ^ GsProcess _forkBlock: self with: #() env: 0 prio: nil
%

category: 'Processes - Blue Book'
method: ExecBlock
forkAt: priority
  "forks the receiver as a new process at the given priority"

  | proc m |
  priority > (m := ProcessorScheduler highestPriority) ifTrue:[
    OutOfRange new name:'priority' max: m actual: priority ; signal.
  ].
  proc := GsProcess _forkBlock: self with: #() env: 0 prio: priority .
  ^ proc
%

category: 'Processes'
method: ExecBlock
forkAt: priority with: blockArgs
  "forks the receiver as a new process at the given priority"

  | proc m |
  priority > (m := ProcessorScheduler highestPriority) ifTrue:[
    OutOfRange new name:'priority' max: m actual: priority ; signal.
  ].
  proc := GsProcess _forkBlock: self with: blockArgs env: 0 prio: priority.
  ^ proc
%

category: 'Processes'
method: ExecBlock
forkWith: blockArgs
  "forks the receiver as a new process at the current scheduling priority"

  ^ GsProcess _forkBlock: self with: blockArgs env: 0 prio: nil
%

category: 'Accessing'
method: ExecBlock
instVarAt: anIndex

  anIndex == 1 ifTrue:[
    "translate GsNativeCode to GsNMethod if needed"
    ^ self method
  ].
  ^ super instVarAt: anIndex
%

category: 'Accessing'
method: ExecBlock
isSimple

^ self _cost = 1
%

category: 'Accessing'
method: ExecBlock
lastRubyArgIsStar
  ^ (iFields1 bitAnd: 16r100) ~~ 0
%

category: 'Accessing'
method: ExecBlock
method
  "Returns the GsNMethod containing the code for this block.

   A primitive is needed here because the value of the _method instVar
   can be an instance of GsNativeCode for an in-memory block."

<primitive: 492>
self _primitiveFailed: #method .
%

category: 'Repository Conversion'
method: ExecBlock
needsRecompile

"Returns true if the receiver's block needs recompilation, false otherwise."

^  self method needsRecompile
%

category: 'Processes - Blue Book'
method: ExecBlock
newProcess
  "creates a new suspended process holding the receiver"

  ^ GsProcess _newForBlock: self with: #() env: 0
%

category: 'Processes - Blue Book'
method: ExecBlock
newProcessWith: argArray
  "creates a new suspended process holding the receiver to be evaluated
   with the given arguments"

  ^GsProcess _newForBlock: self with: argArray env: 0
%

category: 'Accessing'
method: ExecBlock
noRubyDeclaredArgs
  ^ (iFields1 bitAnd: 16r200) ~~ 0
%

category: 'Accessing'
method: ExecBlock
numArgs

  "Returns a SmallInteger, the number of arguments to the receiver.
   numArgs is GemStone legacy API. "
  <primitive: 458>
  self _primitiveFailed: #numArgs .
  ^ 0
%

category: 'Accessing'
method: ExecBlock
numberArgs

  "Returns a SmallInteger, the number of arguments to the receiver."
  self deprecated: 'ExecBlock>>numberArgs deprecated v3.2; use #argumentCount'.
  ^ self argumentCount
%

category: 'Accessing'
method: ExecBlock
numberTemps

^ (iFields1 bitShift: -12) bitAnd: 16rFFFF
%

category: 'Updating'
method: ExecBlock
objectSecurityPolicy: anObjectSecurityPolicy

"Assign the receiver to the given security policy.
 For complex blocks, also changes the security policy for variable contexts."

| vc |
super objectSecurityPolicy: anObjectSecurityPolicy .
(vc := staticLink) ifNotNil:[ vc objectSecurityPolicy: anObjectSecurityPolicy ].
%

category: 'Block Evaluation'
method: ExecBlock
on: exceptionSelector do: handlerBlock
 "Try to evaluate the receiver, which should be a zero-argument block.
  If an exception occurs and the expression
      exceptionSelector handles: theExceptionInstance
  returns true, then evaluate the one argument block handlerBlock ,
  passing it the exception instance as its argument.

  Two forms are supported directly by the VM, and the 'fast path code'
  below is used.
    (1) on: anException do: handlerBlock
    (2) on: anExceptionSet do: handlerBlock
  A third form is handled by Smalltalk code in the body of this method,
  and for this form only, #handles is sent to anObject to determine
  whether an exception should be handled .
    (3) on: anObject     do: handlerBlock

  anException must be the class Exception  or a subclass thereof ;
  anExceptionSet must be a kind of ExceptionSet;
  handlerBlock must be an instance of ExecBloc otherwise an error is generated.

  For forms 1,2,3 if handlerBlock expects more than 1 argument,
  an error is generated if exception handling attempts to
  invoke that handlerBlock.

  If handlerBlock is invoked to handle an Exception which occurs during
  execution of the receiver and handlerBlock completes normally , then
  the result of handlerBlock (value of last expression in handlerBlock)
  will be the result of the on:do: send .  Other-than-normal
  completion of handlerBlock is available by use of
  Exception's instance methods such as
    #return  #return:  #retry #retryUsing: #resume #resume: #pass #outer
  within handlerBlock

  For forms 1 and 2, when searching for a handlerBlock to handle a signaled Exception,
  the VM uses Behavior>>_subclassOf: semantics . classHistories of the
  class of the signaled Exception and of anException or elements of anExceptionSet
  are ignored.
"

<primitive: 2030> "always fails"
| fastPath | "fastPath := true by primitive if form 1 or 2 detected"
             "fastPath := nil  by primitive if form 3 detected."
fastPath ifNotNil:[ "fast path code"
 ^ self value
].
"Any changes to this method's code before this line may also
 require changes to code in comgen.c conditional on BcPrim_ENTER_onDo .
"
^ self onException: AbstractException do:[:ex |
    (exceptionSelector handles: ex) ifTrue:[
      handlerBlock argumentCount == 0
         ifTrue:[ handlerBlock value ]
        ifFalse:[ handlerBlock value: ex ]
    ] ifFalse:[
      ex pass
    ]
  ]
%

category: 'Block Evaluation - Private'
method: ExecBlock
onException: anException do: handlerBlock
 "An optimized form of on:do: .  anException may not be any
  object which responds to  handles:

  Private.

  Try to evaluate the receiver, which should be a zero-argument block.
  If an exception occurs which is matched by anExceptionClass,
  evaluate the one argument block handlerBlock , passing it the exception
  instance as its argument.

  These forms are supported:
    (1) on: anException do: handlerBlock
    (2) on: anArray     do: handlerBlock
    (3) on: anExceptionSet do: handlerBlock
    (4) on: anArray     do: anArrayOfBlocks

   This form is not supported
    (5) on: anObject     do: handlerBlock

  anException must be the class Exception  or a subclass thereof ;
  anExceptionSet must be a kind of ExceptionSet;
  anArray must be a kind of Array ;
  handlerBlock must be an instance of ExecBloc ; if any of these
  is violated, the send of on:do: will fail with an error.

  anArrayOfBlocks must be an Array containing instances of ExecBloc ,
  the elements are not checked during the on:do: send , and are only
  checked if they are attempted to be evaluated during exception handling.

  In forms 2,3,4 elements of anArray or anExceptionSet which
  are not a kind of Exception, or an Array containing kinds of Exception,
  will silently have no effect on catching exceptions.

  In form 4, elements of anArray may in turn be Arrays of Exceptions.
  In form 4, the offset of the first element of anArray containing a
  match to a signaled exception defines the offset in anArrayOfBlocks
  at which to find the handlerBlock to invoke.

  For forms 1,2,3 if handlerBlock expects more than 1 argument,
  an error is generated if exception handling attempts to
  invoke that handlerBlock.

  For form 4, elements of anArrayOfBlocks must be instances of ExecBlock,
  otherwise the corresponding entry in anArray will silently have no effect
  on catching exceptions.  If an element of anArrayOfBlocks is an ExecBlock
  taking more than 1 argument, an error is generated if exception
  handling attempts to invoke that block.

  If handlerBlock is invoked to handle an Exception which occurs during
  execution of the receiver and handlerBlock completes normally , then
  the result of handlerBlock (value of last expression in handlerBlock)
  will be the result of the on:do: send .  Other-than-normal
  completion of handlerBlock is available by use of
  Exception's instance methods such as
    #return  #return:  #retry #retryUsing: #resume #resume: #pass #outer
  within handlerBlock

  When searching for a handlerBlock to handle a signaled Exception, the VM
  uses Behavior>>_subclassOf: semantics . classHistories of the
  class of the signaled Exception and of anException or elements of anExceptionSet
  are ignored.
"

<primitive: 2018> "marks frame with Exception_Mark_NIL, always fails"
^ self value
%

category: 'Block Evaluation - Private'
method: ExecBlock
onSynchronous: anExceptionClass do: handlerBlock

"Same as onException:do: , except that asynchronous Exceptions
 (those exceptions which have trappable==1) will never be handled
 by handlerBlock .

 Allows asynchronous Exceptions to be handled by a higher
 level handler, even if the async Exception would match anExceptionClass.

 Private.
"

<primitive: 2029>
^ self value
%

category: 'Repository Conversion'
method: ExecBlock
recompile
  "Attempts to recompile the receiver's method assuming it is a simple block
   without references to self."

| src newBlk doitMeth litVars mask |
self needsRecompile ifFalse:[ ^ self "recompile not needed"].
staticLink ifNotNil:[
  Error signal:'cannot recompile a complex block with reference to home VariableContext'.
].
ccallin ifNotNil:[
  Error signal:'cannot recompile a block for a CCallin'.
].
src := self _sourceString .
litVars := self method _literalVariablesForRecompile .
doitMeth := src _compileInContext: nil
       symbolList: GsSession currentSession symbolList
       oldLitVars: litVars environmentId: 0 flags: 4"GCI_COMPILE_evaluateWithoutSelf".
newBlk := doitMeth _executeInContext: nil .
mask := 16r40000ffff0ff . "nTemps, nArgs, isCopying"
(newBlk _iFields1 bitAnd: mask) = (iFields1 bitAnd: mask) ifFalse:[
  Error signal:'in recompile: iFields1 differs, newBlock 16r',
     newBlk _iFields1 asHexString , ' old 16r', iFields1 asHexString .
].
self _unsafeAt: 1 put: newBlk method .  "_method := newBlk method"
%

category: 'Flow of Control'
method: ExecBlock
repeat
"(Reserved selector.)  Evaluate the receiver repeatedly, ending only if the
 block forces some stopping condition.

 The following is a control structure optimization, not a recursive send."

 [ self value] repeat
%

category: 'Accessing'
method: ExecBlock
selfOffsetInVC

" Return a one-based offset to self for use as arg to primitiveAt:
  If there is no reference to self in the receiver,
  returns  1 , otherwise returns a value > VAR_CONTEXT_NAMED_SIZE"

^ ((iFields1 bitShift: -28) bitAnd: 16rFFFF) + 1

%

category: 'Accessing'
method: ExecBlock
selfValue

"return self in an active copy , or nil if self is not referenced."
| aVc ofs |
aVc := staticLink .
aVc ifNotNil:[ | vcOfs |
  vcOfs := self selfOffsetInVC .
  vcOfs > 2"VAR_CONTEXT_NAMED_SIZE" ifTrue:[
    [ aVc parent ~~ nil ] whileTrue:[ aVc := aVc parent ].
    vcOfs > aVc basicSize ifTrue:[ ^ nil ].
    ^ aVc _primitiveAt: vcOfs
  ].
].
ofs := self _selfOffsetInSelf .
ofs ~~ 0 ifTrue:[
  ofs := ofs - 1 .
  ^ self _primitiveAt: ofs .
].
^ nil
%

category: 'Updating'
method: ExecBlock
size: anInteger
  "Disallowed"
  self shouldNotImplement: #size:
%

category: 'Accessing'
method: ExecBlock
staticLink

^ staticLink
%

category: 'Flow of Control'
method: ExecBlock
untilFalse

"(Reserved selector.)  Evaluates the receiver repeatedly until the evaluation's
 result is false.  Return nil.  Generates an error if the receiver is not a
 zero-argument block."

"The following is a control structure optimization, not a recursive send."

^ [self value] untilFalse
%

category: 'Flow of Control'
method: ExecBlock
untilFalse: aBlock

"(Reserved selector.)  Evaluates the receiver once ,
 and then repeats the evaluation while aBlock returns true .
 Returns nil.  Generates an error if the receiver is not a
 zero-argument block."

"The following is a control structure optimization, not a recursive send."

^ [self value] untilFalse:[ aBlock value]
%

category: 'Flow of Control'
method: ExecBlock
untilTrue

"(Reserved selector.)  Evaluates the receiver repeatedly until the evaluation's
 result is true.  Return nil.  Generates an error if the receiver is not a
 zero-argument block."

"The following is a control structure optimization, not a recursive send."

^ [self value] untilTrue
%

category: 'Flow of Control'
method: ExecBlock
untilTrue: aBlock

"(Reserved selector.)  Evaluates the receiver once ,
 and then repeats the evaluation while aBlock returns false .
 Returns nil.  Generates an error if the receiver is not a
 zero-argument block."

"The following is a control structure optimization, not a recursive send."

^ [self value] untilTrue:[ aBlock value]
%

category: 'Block Evaluation'
method: ExecBlock
value

"Return the value of the receiver evaluated with no arguments.
 If the block expects any arguments, an error is generated.

 #value is optimized by the compiler.
 This method is in the image for use by perform: and for
 failure paths from the optimized bytecode "

^ self valueWithArguments: #()
%

category: 'Block Evaluation'
method: ExecBlock
value: anObject

"Return the value of the receiver evaluated with anObject as its argument.  If
 the block expects a different number of arguments, an error is generated.

 #value: is optimized by the compiler.
 This method is in the image for use by perform: and for
 failure paths from the optimized bytecode "


^ self valueWithArguments: { anObject }
%

category: 'Block Evaluation'
method: ExecBlock
value: anArg on: anExceptionClass do: handlerBlock
  "Evaluate the one-argument receiver with the specified on:do: .
   Useful for avoiding complex blocks in the sender of value:on:do: "

  ^ [ self value: anArg ] onException: anExceptionClass do: handlerBlock
%

category: 'Block Evaluation'
method: ExecBlock
value: anArg onSynchronous: anExceptionClass do: handlerBlock
  "Evaluate the one-argument receiver with the specified onSynchronous:do: .
   Useful for avoiding complex blocks in the sender of value:onSynchronous:do: "

  ^ [ self value: anArg ] onSynchronous: anExceptionClass do: handlerBlock
%

category: 'Block Evaluation'
method: ExecBlock
value: firstObject value: secondObject

"Return the value of the receiver evaluated with the two objects as its
 arguments.  If the block expects a different number of arguments, an error is
 generated.

 #value:value: is optimized by the compiler.
 This method is in the image for use by perform: and for
 failure paths from the optimized bytecode "

^ self valueWithArguments: {  firstObject . secondObject }
%

category: 'Block Evaluation'
method: ExecBlock
value: firstObject value: secondObject value: thirdObject

"Return the value of the receiver evaluated with the three objects as its
 arguments.  If the block expects a different number of arguments, an error is
 generated.

 #value:value:value: is optimized by the compiler.
 This method is in the image for use by perform: and for
 failure paths from the optimized bytecode "

^ self valueWithArguments: {  firstObject . secondObject . thirdObject }
%

category: 'Block Evaluation'
method: ExecBlock
value: first value: second value: third value: fourth

"Return the value of the receiver evaluated with the four objects as
 its arguments.  If the block expects a different number of arguments,
 an error is generated.

 #value:value:value:value: is optimized by the compiler.
 This method is in the image for use by perform: and for
 failure paths from the optimized bytecode "

^ self valueWithArguments: {  first . second . third . fourth }
%

category: 'Block Evaluation'
method: ExecBlock
value: first value: second value: third value: fourth value: fifth

"Return the value of the receiver evaluated with the five objects as
 its arguments.  If the block expects a different number of arguments,
 an error is generated.

 #value:value:value:value:value: is optimized by the compiler.
 This method is in the image for use by perform: and for
 failure paths from the optimized bytecode "

^ self valueWithArguments: {  first . second . third . fourth . fifth }
%

category: 'Block Evaluation'
method: ExecBlock
valueWithArguments: argList

"Return the value of the receiver evaluated with the elements of the Array
 argList as arguments.  If the block expects a different number of arguments,
 an error is generated."

<primitive: 2003>  "compiler emits special bytecode"
^ self _primitiveFailed: #valueWithArguments: args: { argList }
%

category: 'Debugging Support'
method: ExecBlock
vcTempNames

"Returns an Array of Symbols which are the names for the temps
 defined in the receiver's VariableContext.  The ordering of names
 in the result matches ordering of temps in the varying instVars
 of the receiver."

| vc allNames allOfs res vcNamedSize myMeth lastTmpOfs |
vc := staticLink .
vc == nil ifTrue:[ ^ #() ].
vcNamedSize := vc class instSize .
allNames := self argsAndTemps .
myMeth := self method .
allOfs := myMeth _argsAndTempsOffsets .
res := Array new: vc size .
lastTmpOfs := allOfs size - myMeth _numCopyingBlockArgs .
self argumentCount + 1 to: lastTmpOfs do:[:j| | anOfs ofs |
  anOfs := allOfs at: j .
  anOfs > 0 ifTrue:[
    (anOfs bitAnd: 16rFF) == 0 ifTrue:[ "lexLevel == 0"
      ofs := anOfs bitShift: -8 .
      res at: (ofs + 1 - vcNamedSize) put:( allNames at: j) .
    ] "else temp is defined in a parent block"
  ] "else temp is on stack"
].
^ res
%

category: 'Flow of Control'
method: ExecBlock
whileFalse
"(Reserved selector.)  Evaluate the receiver once and then repeatedly as long
 as the value returned by the evaluation is false.

 The following is a control structure optimization, not a recursive send."

^ [ self value] whileFalse
%

category: 'Flow of Control'
method: ExecBlock
whileFalse: aBlock

"(Reserved selector.)  Evaluates the zero-argument block aBlock repeatedly
 while the receiver evaluates to false.  Return nil.  Generates an error if the
 receiver is not a zero-argument block."

"The following is a control structure optimization, not a recursive send."

^ [self value] whileFalse:[
    aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
    aBlock value
  ]
%

category: 'Flow of Control'
method: ExecBlock
whileTrue
"(Reserved selector.)  Evaluate the receiver once and then repeatedly as long
 as the value returned by the evaluation is true.

 The following is a control structure optimization, not a recursive send."

^ [ self value] whileTrue
%

category: 'Flow of Control'
method: ExecBlock
whileTrue: aBlock

"(Reserved selector.)  Evaluates the zero-argument block aBlock repeatedly
 while the receiver evaluates to true.  Return nil.  Generates an error if the
 receiver is not a zero-argument block."

"The following is a control structure optimization, not a recursive send."

^ [self value] whileTrue: [
    aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
    aBlock value
  ]
%

category: 'Storing and Loading'
method: ExecBlock
writeTo: aPassiveObject

"Converts the receiver to its passive form and writes that information on
 aPassiveObject.

 Simple blocks can usually be passivated and then reactivated.  Complex blocks
 can be passivated but may have to be massaged to be reactivated.  References
 to 'self' in complex blocks will resolve to an instance of Object when the
 block is activated, and any arguments or temporaries from enclosing scopes
 will be nil."

aPassiveObject writeClass: self class.
aPassiveObject writeObject: self _sourceString; cr
%

category: 'Decompiling without Sources'
method: ExecBlock
_asSource

"return a stripped source representation of the block."

"not implemented yet"
self shouldNotImplement: #_asSource
%

category: 'Updating'
method: ExecBlock
_basicSize: anInteger
  "Disallowed"
  self shouldNotImplement: #_basicSize:
%

category: 'Accessing'
method: ExecBlock
_cost

"return 1 for simple, 2 for complex , 3 for complex with return to home"

^ (iFields1 bitShift: -44) bitAnd: 16rF
%

category: 'Accessing'
method: ExecBlock
_debugSourceForBlock
  | str meth |
  "used by topaz  LIST @<blockObjId>"

  meth := self method .
  str := meth _sourceStringForBlock .
  ^ str
%

category: 'Private'
method: ExecBlock
_deepCopyWith: copiedObjDict

"Private.  Used internally to implement deepCopy."

^ self.
%

category: 'Accessing'
method: ExecBlock
_fileAndLine
  "Return an Array, { fileName . lineNumber } ,
   or nil if method  does not have that information.
   Smalltalk methods return nil."

  | mth fl |
  mth := self method .
  (fl := mth homeMethod _fileAndLine) ifNotNil:[
    ^ { (fl at: 1) . (fl at: 2) + mth _lineDeltaForBlock }
  ].
  ^ nil
%

category: 'Private'
method: ExecBlock
_gbsTraversalCallback

"Private.  When GemBuilder Smalltalk traverses an ExecutableBlock, this method
 is called to place the block's source string in the traversal buffer."

^self _sourceString
%

category: 'Repository Conversion'
method: ExecBlock
_iFields1

"Private"

^ iFields1
%

category: 'Accessing'
method: ExecBlock
_isCopyingBlock

^ ((iFields1 bitShift: -46) bitAnd: 1) ~~ 0
%

category: 'Accessing'
method: ExecBlock
_isRubyLambda

^ iFields1 < 0
%

category: 'Private'
method: ExecBlock
_recursiveSize: arr
 | terminationSet |
  ((terminationSet := arr at: 1) includes: self) ifTrue:[ ^ self ].
  terminationSet add: self .
 arr at: 2 put:(arr at:2) + self physicalSizeOnDisk .
%

category: 'Private'
method: ExecBlock
_recursiveSizeInMemory: arr
 | terminationSet |
  self isCommitted ifTrue:[ ^ self ] .
  ((terminationSet := arr at: 1) includes: self) ifTrue:[ ^ self ].
  terminationSet add: self .
 arr at: 2 put:(arr at:2) + self physicalSizeOnDisk .
%

category: 'Private'
method: ExecBlock
_removeEnsure

"The sender's frame is changed from  ensure:[] to  ensure: nil .
 Returns nil or a terminationBlock that has not been started."
<primitive: 1004>

self _primitiveFailed: #_removeEnsureAtFP:
%

category: 'Accessing'
method: ExecBlock
_selfOffsetInSelf

"result is zero or a positive zero-based absolute offset"

^ (iFields1 bitShift: -48) bitAnd: 16rF
%

category: 'Modification Tracking'
method: ExecBlock
_setModificationTrackingTo: tracker

"Private.

 No modification tracking is required for blocks,
 even if they are not invariant."

^self
%

category: 'Accessing'
method: ExecBlock
_sourceString

"Returns a String that will create a block similar to the receiver when
 the string is compiled.  References to variables in other contexts or
 to the pseudovariable 'self' will not recompile properly if the source
 string is used to create a new block similar to the receiver."

| result |
result := String new.
result addAll: '"This is source for a block.  "
' .

result addAll:' ^ ' ;
       addAll: self method _sourceStringForBlock .
^result
%

category: 'Private'
method: ExecBlock
_sourceStringForExecute

"Returns a String that will create a block similar to the receiver when
 the string is compiled.  References to variables in other contexts or
 to the pseudovariable 'self' will not recompile properly if the source
 string is used to create a new block similar to the receiver.

 Used by GsExternalSession which further formats the string
"

^ self method _sourceStringForBlock .
%

category: 'Private'
method: ExecBlock
_terminationBlockValue
  "For use only in the implementation of ensure: "

 <primitive: 2006>  "marks frame with terminationBlockValue_Mark_NIL, always fails"
 | dbgSemaphore |   "installed by GsProcess terminate methods , or nil"
 self value .  
 dbgSemaphore ifNotNil:[:s | s signal ].
 "return value ignored by sender"
%

category: 'Private'
method: ExecBlock
_terminationBlockValue: aSemaphore
  "For use only in the implementation of GsProcess >> terminate* "

 self value .  
 aSemaphore signal .  
 "If we reach here, the process that was terminating current process 
   was probably itself terminated."
 GsProcess current _finishTermination: TerminateProcess new .  "fix 50754"
 self _uncontinuableError "should not reach here"
%

category: 'Private'
method: ExecBlock
_valueOnUnwind

"This method should be invoked only from within the virtual machine. Other
 use from a Smalltalk program will corrupt the Smalltalk execution
 stack."

"Used to execute the terminationBlock of an  ensure:"

self value .  "execute the block"
self _gsReturnNoResult "exit from method with no result,
			special selector optimized by compiler"
%

! Class extensions for 'ExternalError'

!		Instance methods for 'ExternalError'

removeallmethods ExternalError
removeallclassmethods ExternalError

category: 'Instance initialization'
method: ExternalError
initialize
  gsNumber := ERR_ExternalError.
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'FloatingPointError'

!		Class methods for 'FloatingPointError'

removeallmethods FloatingPointError
removeallclassmethods FloatingPointError

category: 'Private'
classmethod: FloatingPointError
enableAllExceptions

^ self enableExceptions: self _exceptionList
%

category: 'Private'
classmethod: FloatingPointError
enabledExceptions
  "Returns an Array containing zero or more of the Symbols
    #divideByZero #overflow #underflow #invalidOperation #inexactResult
  reflecting the most recent call to either of
     FloatingPointError class>>enableAllExceptions
     FloatingPointError class>>enableExceptions:
  "
  ^ self _enabledBitsAsArray: (self _enableExceptions: -1)  .
%

category: 'Accessing'
classmethod: FloatingPointError
enableExceptions: argument

"Argument may be one of the symbols
   #divideByZero #overflow #underflow #invalidOperation #inexactResult
 or an Array containing zero or more of those Symbols ,
 or nil which means enable none of the exceptions.

 The specified exceptions will be checked for
 after each arithmetic primitive in Float and SmallDouble and if
 any of those execptions occurs, a FloatingPointError will be signaled.
 Overrides the settings of a previous call to enableExceptions: .
 Returns a SmallInteger, the previously enabled exception bits."

 | bits |
 argument ifNil:[ ^ self _enableExceptions: 0 ].
 argument _isSymbol ifTrue:[
   ^ self _enableExceptions: (self _symbolToBit: argument)
 ].
 argument _isArray ifTrue:[
   bits := 0 .
   argument do:[:aSym |
     bits := bits bitOr:( self _symbolToBit: aSym)
   ].
   ^ self _enableExceptions: bits
 ].
 argument _validateClasses: { Array . Symbol }.
%

category: 'Private'
classmethod: FloatingPointError
_checkFpStatus
  "If any of the currently enabled exceptions have occurred,
   clear those bits from the VM state and signal an error.
   Currently enabled exceptions are returned by
     FloatingPointError(C) >> enabledExceptions .
   This method is called from primitive failure paths in in each floating
   point primitive."

| flags |
flags := self _getExceptions: false .
flags ~~ 0 ifTrue:[
  self new actual: flags ; signal
].
%

category: 'Private'
classmethod: FloatingPointError
_enabledBitsAsArray: bits
  "The result is in sorted order."
  | res |
  res := { } .
  bits ifNotNil:[
    (bits bitAnd: 1) ~~ 0 ifTrue:[ res add: #divideByZero ].
    (bits bitAnd: 16r10) ~~ 0 ifTrue:[ res add: #inexactResult ].
    (bits bitAnd: 8) ~~ 0 ifTrue:[ res add: #invalidOperation ].
    (bits bitAnd: 2) ~~ 0 ifTrue:[ res add: #overflow ].
    (bits bitAnd: 4) ~~ 0 ifTrue:[ res add: #underflow ].
  ].
  ^ res
%

category: 'Private'
classmethod: FloatingPointError
_enableExceptions: aSmallInteger

"aSmallInteger may have one or more bits in the range 16r1F,
 per the constants in FloatingPointError(C)>>_initializeConstants .
 other bits are ignored.  The specified exceptions will be checked for
 after each arithmetic primitive in Float and SmallDouble and if
 any of those execptions occurs, a FloatingPointError will be signaled.
 Returns the value which was passed in the previous call to this primitive.

 If aSmallInteger == -1 , then returns the positive value
 which was passed in the previous call to this primitive without changing
 the state of the enabled exceptions.
"

<primitive: 122>

aSmallInteger _validateClass: SmallInteger .
self _primitiveFailed: #enableExceptions: args: { aSmallInteger }
%

category: 'Private'
classmethod: FloatingPointError
_exceptionList

^  #( #divideByZero #overflow #underflow #invalidOperation #inexactResult)
%

category: 'Private'
classmethod: FloatingPointError
_getExceptions: aBoolean

"Fetch and clear bits representing floating point exceptions that have occurred
 since last execution of this primitive.  Result is a SmallInteger.

 aBoolean == false  means get bits enabled by the last call to
 FloatingPointError(C)>>enableExceptions: , and clear all bits.
 aBoolean == true means get and clear all bits .

 The bits in the result are in the range 16r1F and
 are defined by the constants in FloatingPointError(C)>>_initializeConstants ."

<primitive: 129>

aBoolean _validateClass: Boolean .
self _primitiveFailed: #_getExceptions: args: { aBoolean }
%

category: 'Private'
classmethod: FloatingPointError
_initializeConstants
   "VM changes needed if you change any of these bit definitions.
    Also fix sends of _enableExceptions: in image."
   self _addInvariantClassVar: #divideByZero value: 1 ;
         _addInvariantClassVar: #overflow value: 2  ;
         _addInvariantClassVar: #underflow value: 4  ;
         _addInvariantClassVar: #invalidOperation  value: 8  ;
         _addInvariantClassVar: #inexactResult  value: 16r10 .
%

category: 'Private'
classmethod: FloatingPointError
_symbolToBit: aSymbol

  | bit |
  bit := classVars at: aSymbol otherwise: nil .
  bit ifNil:[ Error signal:'invalid name of a floating point exception bit, ' , aSymbol].
  ^ bit
%

!		Instance methods for 'FloatingPointError'

category: 'Private'
method: FloatingPointError
actual: aValue
  actual := aValue
%

category: 'Formatting'
method: FloatingPointError
asString
  | str arr cls |
  str := 'a ' copy .
  str add: (cls := self class) name .
  arr := cls _enabledBitsAsArray: actual .
  arr size == 0 ifTrue:[
    gsNumber == ERR_LargeIntegerOverflow
      ifTrue:[ str add:' overflow during LargeInteger arithmetic'].
  ] ifFalse:[
    1 to: arr size do:[:n | str add: $  ; add: (arr at: n) ].
  ].
  ^ str .
%

category: 'Accessing'
method: FloatingPointError
exceptionList
 "Return an Array of Symbols , the list of exceptions
  which produced this instance.  The result will contain one or more of
     #divideByZero #overflow #underflow #invalidOperation #inexactResult  .
  and is sorted alphabetically.
 "
  ^ self class _enabledBitsAsArray: actual
%

category: 'Instance initialization'
method: FloatingPointError
initialize
  gsNumber := ERR_FloatingPointError.
  gsResumable := true .
  gsTrappable := true .
  actual := 0 .
%

! Class extensions for 'FloatingPointException'

!		Instance methods for 'FloatingPointException'

removeallmethods FloatingPointException
removeallclassmethods FloatingPointException

category: 'Instance initialization'
method: FloatingPointException
initialize
  gsNumber := ERR_FloatingPointException.
  gsResumable := true .
  gsTrappable := true .
%

! Class extensions for 'Fraction'

!		Class methods for 'Fraction'

removeallmethods Fraction
removeallclassmethods Fraction

category: 'Storing and Loading'
classmethod: Fraction
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst num den |
inst := Fraction _basicNew .
passiveObj hasRead: inst.
passiveObj readNamedIV.
num := passiveObj ivValue.
passiveObj readNamedIV.
den := passiveObj ivValue.

passiveObj skipNamedInstVars.

inst _numerator: num denom: den .
^inst.
%

!		Instance methods for 'Fraction'

category: 'Converting'
method: Fraction
asCanonicalForm
	"Answer self, or, if I am a Fraction with an equivalent
	SmallFraction, answer that SmallFraction.

	Note: SmallFraction inherits Object >> asCanonicalForm."

	| cls res |
	(cls := self class) == Fraction
		ifTrue: [ 
			res := cls _newSmallFraction: numerator denom: denominator reduce: true.
			res class == SmallFraction
				ifTrue: [ ^ res ] ].
	^ self
%

category: 'Accessing'
method: Fraction
denominator

"Returns the denominator of the receiver."

^ denominator
%

category: 'Accessing'
method: Fraction
instVarAt: anIndex put: aValue

"Disallowed.  You may not change the value of a Fraction."

self shouldNotImplement: #instVarAt:put:
%

category: 'Accessing'
method: Fraction
numerator

"Returns the numerator of the receiver."

^ numerator
%

category: 'Copying'
method: Fraction
postCopy
  ^ self immediateInvariant
%

category: 'Storing and Loading'
method: Fraction
writeTo: passiveObj

"Converts the receiver to its passive form and writes that information on
 passiveObj."

"Reimplemented from Number since the receiver has a non-literal representation."

^super basicWriteTo: passiveObj
%

category: 'Private'
method: Fraction
_numerator: n denom: d
  numerator := n .
  denominator := d .
  ^ self immediateInvariant
%

! Class extensions for 'GciTransportError'

!		Instance methods for 'GciTransportError'

removeallmethods GciTransportError
removeallclassmethods GciTransportError

category: 'Instance initialization'
method: GciTransportError
initialize
  gsNumber := ERR_GciTransportError.
  gsResumable := false .
  gsTrappable := true .
%

! Class extensions for 'GsCurrentSession'

!		Class methods for 'GsCurrentSession'

removeallmethods GsCurrentSession
removeallclassmethods GsCurrentSession

category: 'Instance Creation'
classmethod: GsCurrentSession
currentSession

"Returns the sole instance of GsCurrentSession that represents this login
 session."

<primitive: 311>
self _primitiveFailed: #currentSession
%

category: 'Initialization'
classmethod: GsCurrentSession
initialize
 "If the GciLogin flags contained bit GCI_CLIENT_DOES_SESSION_INIT.
  the VM expects the GCI application to do
    GciPerform(OOP_CLASS_GSCURRENT_SESSION, 'initialize', NULL, 0);
  after a successful login, so that errors in initialize are debuggable.
  Otherwise the VM will do the message send itself during the login.

  topaz passes GCI_CLIENT_DOES_SESSION_INIT. to GciLogin and does the
  GciPerform itself.
 "

 | res |
 (res := self currentSession)
    "No application changes here, they belong in
     the instance method GsCurrentSession >> initialize"
    initialize .
 ^ res
%

category: 'Instance Creation'
classmethod: GsCurrentSession
new

"Disallowed.

 The only instance of GsCurrentSession that is permitted in a session is created
 automatically when a user logs in to GemStone.  Its default SymbolList is a
 copy of the user's SymbolList.  You can obtain that instance by sending the
 message GsSession currentSession."

self shouldNotImplement: #new.
self _uncontinuableError .
%

category: 'Private'
classmethod: GsCurrentSession
_sessionStateSizeReport
  | str | str := String new .
  1 to: 40 do:[:n |
    (System __sessionStateAt: n) ifNotNil:[ :obj |
      str add: n asString, ':  ', obj recursiveSizeInMemoryReport; lf .
    ]
  ].
  ^ str
%

!		Instance methods for 'GsCurrentSession'

category: 'Transaction Control'
method: GsCurrentSession
abortTransaction

"Rolls back all modifications made to committed GemStone objects and provides
 the session with a new view of the most current committed state of the
 repository.

 These operations are performed whether or not the session was previously in a
 transaction.  If the transaction mode is set to #autoBegin, then a new
 transaction is started.  If the transaction mode is set to #manualBegin, then
 a new transaction is not started."

System abortTransaction
%

category: 'Transaction Control'
method: GsCurrentSession
beginTransaction

"Starts a new transaction for the session.  If the session is already
 in a transaction, aborts the current transaction and starts a new transaction.

 If the session changed any permanent objects without committing them, their
 state is aborted."

System beginTransaction
%

category: 'Session Configuration Access'
method: GsCurrentSession
clientVersionAt: aSymbol

"Returns the value of the GsSession's client version information parameter named
 aSymbol.  Returns nil if no version parameter named aSymbol exists."

^ System clientVersionAt: aSymbol
%

category: 'Transaction Control'
method: GsCurrentSession
commitAndReleaseLocks

"Attempts to commit the transaction for the session.

 This method is the same as 'commitTransaction' except for the handling of
 locks.  If the commit succeeds, this method releases all locks for the session
 and returns true.  Otherwise, it returns false and does not release locks.

 This method also clears the commit release locks and commit-or-abort release
 locks sets.  See the 'Releasing Locks' method category in class System for
 more information."

^ System commitAndReleaseLocks
%

category: 'Transaction Control'
method: GsCurrentSession
commitTransaction

"Attempts to update the persistent state of GemStone to include changes made
 by this session.

 If the commit operation succeeds, then this method returns true, and the
 current transaction's changes, if any, become a part of the persistent
 repository.  After the repository update, the session exits the current
 transaction.  If the transaction mode is autoBegin, then the session enters
 a new transaction.  If the transaction mode is #manualBegin, then the session
 remains outside of a transaction.

 If conflicts prevent the repository update, then this method returns false.
 Call the transactionConflicts method to determine the nature of the
 conflicts.  If the session is outside of a transaction, then this method
 raises the error #rtErrPrimOutsideTrans.

 This method also updates the session's view of GemStone.  If the commit
 operation succeeds, then all objects in the session's view are consistent with
 the current state of GemStone.  If the commit fails, then this method retains
 all the changes that were made to objects within the current transaction.
 However, commits made by other sessions are visible to the extent that changes
 in this transaction do not conflict with them."

^ System commitTransaction
%

category: 'Session Configuration Access'
method: GsCurrentSession
configurationAt: aSymbol

"Returns the value of the configuration parameter named aSymbol (for example,
 #GEM_HALT_ON_ERROR).  Raises an error if aSymbol is not a valid parameter
 name."

^ System configurationAt: aSymbol
%

category: 'Session Configuration Access'
method: GsCurrentSession
configurationAt: aSymbol put: anObject

"Sets the value of the configuration parameter named aSymbol to anObject.
 Raises an error if aSymbol is not a valid parameter name or anObject is an
 inappropriate value for the parameter."

^ System configurationAt: aSymbol put: anObject
%

category: 'Session Configuration Access'
method: GsCurrentSession
configurationParameters

"Returns a Set of Symbols containing the names of all valid configuration
 parameters for this GsSession."

^ ConfigurationParameterDict keys
%

category: 'Transaction Control'
method: GsCurrentSession
continueTransaction

"Updates the session's view to the most recently committed state of GemStone
 without rolling back modifications made to committed objects in the session.
 The read and write sets of the session are carried forward and continue to
 accumulate until the session either commits or aborts.  Changes made by this
 session to committed objects are not visible to other sessions until the
 session commits.

 Returns true if accumulated modifications to the committed state of GemStone
 would not cause concurrency conflict as of the new view; otherwise returns
 false.  If it returns false, you can call the transactionConflicts method to
 determine the nature of the conflicts.

 This method can be used whether or not the session is outside of a transaction.
 Of course, the session cannot commit the accumulated changes unless it is
 inside a transaction.

 If transaction mode is #manualBegin, then continueTransaction does not alter
 the inside/outside of transaction state of the session.

 Modifications made by other committed transactions are accumulated for
 retrieval by GciDirtyObjs() and GciDirtySavedObjs() just as they are
 accumulated for commitTransaction or abortTransaction.

 This method has no effect on object locks held by the session.  Locks in the
 release locks sets are not released."

^ System continueTransaction
%

category: 'Method lookup control'
method: GsCurrentSession
enableSessionMethods: aBoolean env: envId
  "sets omPtr->sessionMethodsEnabled := aBoolean  in VM
   and returns previous state ; this controls compilation logic
   but has no effect on method lookup.  Caller responsible for
   using transientMethodDictForEnv:put: to install session methods.
   Clears method lookup caches for all in-memory classes, and
   clears all send-site caches for all in-memory methods. "

<primitive: 650>
aBoolean _validateClass: Boolean .
envId _validateClass: SmallInteger .
self _primitiveFailed:#enableSessionMethods: args: { aBoolean . envId }
%

category: 'Smalltalk Execution'
method: GsCurrentSession
execute: aString

"Executes aString containing GemStone Smalltalk code in the session represented
 by the receiver.  Symbol resolution is from the default symbol list."

^ ( aString _compileInContext: nil symbolList: self symbolList )
     _executeInContext: nil
%

category: 'Smalltalk Execution'
method: GsCurrentSession
execute: aString symbolList: aSymbolList

"Executes aString containing GemStone Smalltalk code in the session represented
 by the receiver.  Symbol resolution is from the given symbol list."

^ ( aString _compileInContext: nil symbolList: aSymbolList )
     _executeInContext: nil
%

category: 'Transaction Control'
method: GsCurrentSession
inTransaction

"Returns true to indicate that the session is in a transaction, false
 otherwise."

^ (System _zeroArgPrim: 5) ~~ 0   "fix 48290"
%

category: 'Session Control'
method: GsCurrentSession
logoutAndGoIdle

"The session performs a relogin as the Nameless user and goes idle by
 setting the transactionMode to #transactionless."

| namelessUserProfile |
namelessUserProfile := AllUsers userWithId: 'Nameless' ifAbsent: [self error: 'There is no Nameless user profile.'].

System currentObjectSecurityPolicy: namelessUserProfile defaultObjectSecurityPolicy.
System abortTransaction.
System _generationScavenge.
self reloginAsUser: 'Nameless' withPassword: ''.
self transactionMode: #transactionless.

%

category: 'Deprecated'
method: GsCurrentSession
nativeLanguage

"Returns the String that designates the language that controls error message
 generation in the current session."

| language |
self deprecated: 'GsCurrentSession>>nativeLanguage deprecated v3.0; base system no longer uses nativeLanguage'.
language := self objectNamed: #'NativeLanguage'.
language ifNil: [^#'English'].
^language.
%

category: 'Accessing the Symbol List'
method: GsCurrentSession
objectNamed: aSymbol

"Returns the first object in the receiver's symbol list that has the given
 name.  If no object with the given name is found, returns nil."

| assn |

assn := (transientSymbolList ifNil:[ symbolList]) resolveSymbol: aSymbol.
assn ifNil: [ ^nil ].
^assn _value
%

category: 'Session Control'
method: GsCurrentSession
reloginAsUser: aUserId withEncryptedPassword: aPassword

"The session aborts its current transaction, if any, and attempts a relogin
 with the specified userid and encrypted password.  Errors in login leave
 the session in the idle state, as if logoutAndGoIdle had been successfully
 executed."

| result |
result := System _reloginAsUser: aUserId password: aPassword encrypted: true.
(System hasUserAction: #_topazReloginCallback)
  ifTrue: [ System userAction: #_topazReloginCallback ].

result ifFalse: [System _error: #rtErrWarningPasswordExpire]
       ifTrue: [System transactionMode: #autoBegin]
%

category: 'Session Control'
method: GsCurrentSession
reloginAsUser: aUserId withPassword: aPassword

"The session aborts its current transaction, if any, and attempts a relogin
 with the specified userid and password.  Errors in login leave the session
 in the idle state, as if logoutAndGoIdle had been successfully executed."

| result |
result := System _reloginAsUser: aUserId password: aPassword encrypted: false.

(System hasUserAction: #_topazReloginCallback)
  ifTrue: [ System userAction: #_topazReloginCallback ].

result ifFalse: [System _error: #rtErrWarningPasswordExpire]
       ifTrue: [System transactionMode: #autoBegin]
%

category: 'Accessing the Symbol List'
method: GsCurrentSession
resolveSymbol: aSymbol

"Searches the receiver's symbol list for an Association whose key is equal to
 aString, and returns that Association.  If no such Association is found in the
 symbol list, returns nil.

 Implemented to use the current session's transient copy of the symbol list.
 This method is the default mechanism for symbol-resolution during
 compilation of GemStone Smalltalk methods."

^ (transientSymbolList ifNil:[ symbolList]) resolveSymbol: aSymbol
%

category: 'Session Configuration Access'
method: GsCurrentSession
serverVersionAt: aSymbol

"Returns the value of the GsSession's Stone version information parameter named
 aSymbol."

^ System stoneVersionAt: aSymbol
%

category: 'Session Configuration Access'
method: GsCurrentSession
sessionVersionAt: aSymbol

"Returns the value of the GsSession's Gem version information parameter
 named aSymbol."

^ System gemVersionAt: aSymbol
%

category: 'Deprecated'
method: GsCurrentSession
signalFromSession

"Deprecated.
 Return a GsInterSessionSignal object containing information about a signal from
 another session, or nil if there is no signal waiting."

| sigArray result |
self deprecated:'GsCurrentSession>>signalFromSession replaced by InterSessionSignal(C)>>poll' .
sigArray := System _signalFromGemStoneSession .
sigArray ifNil:[ ^ nil ].
result := GsInterSessionSignal signal: (sigArray at: 2)
			      message: (sigArray at: 3) .
result session: (GsSession sessionWithSerialNumber: (sigArray at: 1)).
^ result
%

category: 'Accessing the Symbol List'
method: GsCurrentSession
symbolList

"If a transientSymbolList has been installed, return it.
 Otherwise return the symbolList installed at session login, which
 is  System myUserProfile symbolList .

If you have not executed any of
  System class >> refreshTransientSymbolList
  GsCurrentSession >> transientSymbolList:
  GsCurrentSession >> transientSymbolList
in the current session, then
  GsCurrentSession currentSession symbolList == System myUserProfile symbolList
"

^ transientSymbolList ifNil:[ symbolList]
%

category: 'Transaction Control'
method: GsCurrentSession
transactionMode

"Returns the current transaction mode for the current GemStone session, either
 #autoBegin, #manualBegin or #transactionless.  The default is #autoBegin."

^ System _zeroArgPrim: 4
%

category: 'Transaction Control'
method: GsCurrentSession
transactionMode: newMode

"Sets a new transaction mode for the current GemStone session and exits the
 previous mode by aborting the current transaction.  Valid arguments are
 #autoBegin, #manualBegin or #transactionless.
 The mode transactionless is intended primarily for idle sessions. Users
 may scan database objects, but are at risk of obtaining inconsistent views.
 When in transactionless mode, the session is exempt from voting on
 possibleDeadObjs and will not be terminated by STN_GEM_TIMEOUT expiring.
"

System transactionMode: newMode
%

category: 'Accessing the Symbol List'
method: GsCurrentSession
transientSymbolList
  "If transientSymbolList is nil , set it to be a copy of the
      current UserProfile's symbolList .
   Returns the receiver's transientSymbolList .

   transientSymbolList is nil unless one of
		GsCurrentSession >> transientSymbolList
    GsCurrentSession >> transientSymbolList:  
    System class >> refreshTransientSymbolList
  have been executed in the current session."

  ^ transientSymbolList 
		ifNil: [
			"System class refreshTransientSymbolList directly updates the transientSymbolList instance variable"
			System refreshTransientSymbolList.
			transientSymbolList ]
%

category: 'Accessing the Symbol List'
method: GsCurrentSession
transientSymbolList: aSymbolListOrNil
 "Installs aSymbolListOrNil as the transientSymbolList of the receiver.
  Requires CodeModification privilege .

  The transientSymbolList is nil unless one of
		GsCurrentSession >> transientSymbolList
    GsCurrentSession >> transientSymbolList:  
    System class >> refreshTransientSymbolList
 have been executed in the current session."
  (aSymbolListOrNil isNil or: [aSymbolListOrNil _validateClass: SymbolList]) ifTrue:[
    System myUserProfile _validateCodeModificationPrivilege .
    transientSymbolList := aSymbolListOrNil 
  ]
%

category: 'Session Configuration Access'
method: GsCurrentSession
versionParameters

"Returns a Set of Strings containing the names of all valid version parameters
 for this GsSession."

^ VersionParameterDict keys
%

category: 'Reduced Conflict Support'
method: GsCurrentSession
_RcHasConflicts

"Do the hasConflicts check for Rc objects but do not selectively abort."

| scanArray redoObject conflicts|

scanArray := System _getRedoAndConflictObjects.
" if no redo objects were found, cannot resolve conflicts "
scanArray ifNil: [ ^ true ].

1 to: scanArray size by: 2 do: [ :i |
  redoObject := scanArray at: i.
  conflicts := scanArray at: i + 1.

  (redoObject _validateRcConflictsWith: conflicts)
    ifFalse: [ ^ true ]
].
^false
%

category: 'Reduced Conflict Support'
method: GsCurrentSession
_scanRedoLogForConflicts

"Scan the redo log for entries on RC objects that could conflict
on replay.  If we find one, return true, indicating that we do not
know if we can replay successfully without selectively aborting."

| redoLog |
(redoLog := System _redoLog) == nil
  ifTrue: [ ^ false ].

redoLog redoObjects keysAndValuesDo: [ :redoObj :logEntries |
  (redoObj _validateRcConflictsWith: #())
    ifFalse: [ ^ true ]
].
^ false
%

category: 'Accessing the Symbol List'
method: GsCurrentSession
_transientSymbolList
	"Direct access to transentSymbolList .
  The transientSymbolList is nil unless one of
		GsCurrentSession >> transientSymbolList
    GsCurrentSession >> transientSymbolList:  
    System class >> refreshTransientSymbolList
 have been executed in the current session."

  ^ transientSymbolList
%

! Class extensions for 'GsExceptionHandler'

!		Class methods for 'GsExceptionHandler'

removeallmethods GsExceptionHandler
removeallclassmethods GsExceptionHandler

category: 'Disallowed'
classmethod: GsExceptionHandler
basicNew
  "Instances are created only by methods in Exception. See also #new ."

  self shouldNotImplement: #basicNew .
  ^ nil
%

category: 'Disallowed'
classmethod: GsExceptionHandler
new
  "instances may be created only by using
     Exception(C)>>category:number:do:
     Exception(C)>>installStaticException:category:number:
     Exception(C)>>installStaticException:category:number:subtype:
     Exception(C)>>addDefaultHandler:
  "
  self shouldNotImplement: #new .
  ^ nil
%

category: 'Management'
classmethod: GsExceptionHandler
removeActivationException: aGsExceptionHandler

"Search the current GemStone Smalltalk call stack for a method or block context
 that has anException installed, and remove it.  The stack is searched by
 starting with the top method or block context and moving down.

 If aGsExceptionHandler was found on the stack, returns aGsExceptionHandler,
 otherwise signals a RuntimeError.  "

aGsExceptionHandler class == GsExceptionHandler ifTrue:[
  aGsExceptionHandler _remove ifNil:[
    LookupError signal: 'aGsExceptionHandler was not found on the stack'
  ].
  ^ aGsExceptionHandler .
].
aGsExceptionHandler _validateClass: GsExceptionHandler
%

!		Instance methods for 'GsExceptionHandler'

category: 'Accessing'
method: GsExceptionHandler
block

"Returns the value of the instance variable 'theBlock'."

^ theBlock
%

category: 'Accessing'
method: GsExceptionHandler
category

^ GemStoneError
%

category: 'Accessing'
method: GsExceptionHandler
exceptionClass
  ^ exClass
%

category: 'Accessing'
method: GsExceptionHandler
isLegacy
  ^ exClass == nil
%

category: 'Accessing'
method: GsExceptionHandler
next

"Returns the next exception to be invoked (the value of the next instance
 variable)."

^next
%

category: 'Accessing'
method: GsExceptionHandler
number

"Returns the value of the instance variable 'number'."

^number
%

category: 'Management'
method: GsExceptionHandler
remove
"Search the current GemStone Smalltalk call stack for a frame
 that has the receiver installed, and remove it if found.
 If not found on the stack, search the list of static handlers and
 remove it if found.

 Returns receiver if found, nil if not found ."

  ^ self _remove ifNil:[ AbstractException removeStaticException: self ]
%

category: 'Accessing'
method: GsExceptionHandler
subtype

"Returns the value of the receiver's subtype instance variable."

^subtype
%

category: 'Accessing'
method: GsExceptionHandler
_ansiBlock
  "If receiver is an ANSI static handler, return the receivers block,
   else return nil."
  exClass ifNotNil:[ ^ theBlock ].
  ^ nil
%

category: 'Management'
method: GsExceptionHandler
_remove

"Search the current GemStone Smalltalk call stack for a method or block context
 that has the receiver installed, and remove it.  The stack is searched by
 starting with the top method or block context and moving down.

 Returns receiver if found, nil if not found ."

<primitive: 375>
self _primitiveFailed: #remove
%

! Class extensions for 'GsFile'

!		Class methods for 'GsFile'

removeallmethods GsFile
removeallclassmethods GsFile

category: 'Instance Creation'
classmethod: GsFile
basicNew

"disallowed"

self shouldNotImplement: #basicNew
%

category: 'Private'
classmethod: GsFile
classUserAction: actionName onClient: onClient

"Executes a GsFile user action, passing it the following arguments:
    self
    any arguments to the primitive (see other uses of primitive 396)
 Maximum of 6 arguments (6 with: keywords) for this primitive.
 actionName must be a Symbol .
 Checks NoGsFileOnServer or NoGsFileOnClient privilege"

<primitive: 396>

^ self _primitiveFailed: #classUserAction:onClient:
       args: { actionName . onClient }
%

category: 'Private'
classmethod: GsFile
classUserAction: actionName onClient: onClient with: arg1

"See GsFile | classUserAction:onClient: for documentation."

<primitive: 396>

^ self _primitiveFailed: #classUserAction:onClient:with:
       args: { actionName . onClient . arg1 }
%

category: 'Private'
classmethod: GsFile
classUserAction: actionName onClient: onClient with: arg1 with: arg2

"See GsFile | classUserAction:onClient: for documentation."

<primitive: 396>

^ self _primitiveFailed: #classUserAction:onClient:with:
       args: { actionName . onClient . arg1 . arg2 }
%

category: 'Drastic Measures'
classmethod: GsFile
closeAll

"Closes all open files on the client machine except stdin/stdout/stderr; in a 
 linked session, files opened as server files are closed. Returns the receiver 
 if successful, nil if not."

"make sure the sources file has been explicitly closed"

^ self classUserAction: #GsfCloseAll onClient: self _clientIsRemote
%

category: 'Drastic Measures'
classmethod: GsFile
closeAllOnServer

"Closes all open files on the server machine except stdin/stdout/stderr.
 Returns the receiver if successful, nil if not."

^ self classUserAction: #GsfCloseAll onClient: false
%

category: 'Directory Operations'
classmethod: GsFile
contentsAndTypesOfDirectory: dirSpec onClient: onClient

"Returns an Array of objects describing the contents of the given directory.
 The argument bool indicates the location of the directory on the client or
 the server.  Successive pairs of elements of the Array are each the name of an
 entry, and a Boolean - true if the entry is a file, and false if not.
 The order of results is based on the file system.

 Sample:  { 'file.c'  true . 'subdir' . false . ... }

 Returns anArray if successful, nil if not."
| dirContents result |
(onClient and:[ self _clientIsRemote]) ifTrue:[ 
  dirContents := self contentsOfDirectory: dirSpec onClient: true .
  dirContents ifNil:[ ^ nil ].
  result := { }  .
  1 to: dirContents size do:[:n | | str |
    str := dirContents at: n .
    result add: str ;
           add: (self _fileKind: str onClient: true ) == 0  .
  ] .
] ifFalse:[  | unicodeEnabled |
  "use faster primitives available in VM for server file access"
  dirContents := self _contentsOfServerDirectory: dirSpec expandPath: true utf8Results: true .
  dirContents _isSmallInteger ifTrue:[ ^ nil ].
  result := { }  .
  unicodeEnabled := Unicode16 _unicodeCompareEnabled.
  1 to: dirContents size do:[:n | | elem aGsFileStat |
    result add: (elem := self _decodeUtfResult: (dirContents at: n) unicode: unicodeEnabled) .
    (aGsFileStat := self stat: elem isLstat: false) _isSmallInteger ifTrue:[
      result add: false "probably a symlink to a non-existant file"
    ] ifFalse:[
      result add: aGsFileStat isDirectory not
    ]
  ].
].
^ result
%

category: 'Directory Operations'
classmethod: GsFile
contentsOfDirectory: dirSpec onClient: clientBool

"Returns an Array describing the contents of the given directory.
 Elements of result are Unicode strings if the system is in Unicode Comparison
 Mode (String Configuration isInUnicodeComparsonMode = true), otherwise they are
 traditional Strings.

 The argument bool indicates the location of the directory on the client or
 the server.
 The ordering of the result is based on the file system implementation.
 Returns anArray if successful, nil if not."

| unicodeEnabled files uDir res |
unicodeEnabled := Unicode16 _unicodeCompareEnabled.
uDir := self _utfPath: dirSpec forClient: clientBool .
files :=  self classUserAction: #GsfDirectory onClient: clientBool
       with: uDir
       with: true.
files ifNil:[ ^ files ].
res := { } .
1 to: files size do:[:n |
  res add: (self _decodeUtfResult: (files at: n) unicode: unicodeEnabled) .
].
^ res
%

category: 'Directory Operations'
classmethod: GsFile
createClientDirectory: aPathName

"Creates the named directory on the client machine's file system.
 Returns the receiver if the directory was created, nil if an error occurs."

^ self _createDirectory: aPathName onClient: self _clientIsRemote  mode: nil .
%

category: 'Directory Operations'
classmethod: GsFile
createServerDirectory: aPathName

"Creates the named directory on the server machine.
 Returns the receiver if the directory was created, nil if an error occurs."

^ self _createDirectory: aPathName onClient: false mode: nil .
%

category: 'Directory Operations'
classmethod: GsFile
createServerDirectory: aPathName mode: modeInt

"Creates the named directory on the server machine.
 Returns the receiver if the directory was created, nil if an error occurs.
 modeInt should be a SmallInteger, the value of which will be used
 as the second arg to mkdir() .
 If modeInt == nil, a value of 8r770 will be used.
 Note that mkdir() takes the modeInt and removes bits set in the process' umask.
"

^ self _createDirectory: aPathName onClient: false mode: modeInt .
%

category: 'File Operations'
classmethod: GsFile
exists: aPathName

"Returns true if the given path points to a file on the client,
 false if not, and nil if an error occurs trying to find out."

^ self _exists: aPathName onClient: self _clientIsRemote
%

category: 'File Operations'
classmethod: GsFile
existsOnServer: aPathName

"Returns true if the given path points to a file, on the server,
 false if not, and nil if an error occurs trying to find out."

^ self _exists: aPathName onClient: false
%

category: 'Garbage Collection'
classmethod: GsFile
finalizeAll

"Closes all open files on the client machine that were opened by a
 instance of GsFile but that are no longer being used by an instance
 of GsFile. The need for finalization is caused by instances being
 garbage collected before they are sent close.
 Returns nil if an error occurs.
 Returns the total number of GsFile instances that were not finalized
 because they were still in use."

System _generationScavenge.
^ self classUserAction: #GsfFinalizeAll onClient: self _clientIsRemote
%

category: 'Garbage Collection'
classmethod: GsFile
finalizeAllOnServer

"Closes all open files on the server machine that were opened by a
 instance of GsFile but that are no longer being used by an instance
 of GsFile. The need for finalization is caused by instances being
 garbage collected before they are sent close.
 Returns nil if an error occurs.
 Returns the total number of GsFile instances that were not finalized
 because they were still in use."

System _generationScavenge.
^ self classUserAction: #GsfFinalizeAll onClient: false
%

category: 'Directory Operations'
classmethod: GsFile
gciClientIsWindows

"Returns true if GCI client is on a Microsoft Windows operating system,
 false otherwise."

^ self _directoryPrim: 6 with: nil with: nil
%

category: 'Writing'
classmethod: GsFile
gciLog: aString onClient: clientBool

"Passes the contents of the given collection to a GciGetLogger() function.
 The clientBool must be a Boolean , true means use GciGetLogger() on
 the client process, false means server process .
 If the collection has size > 0 and does not end in an Ascii LineFeed,
 a LineFeed is appended to the bytes passed to the logging function.

 If clientBool==true,
 a check for client process disconnected is made at start of the GsfGciLog
 useraction invocation, and if the client has disconnected, this invocation
 will behave as if it were GsFile class >> gciLogServer: .
"

^ clientBool ifTrue:[ self gciLogClient: aString ]
	    ifFalse:[ self gciLogServer: aString]
%

category: 'Writing'
classmethod: GsFile
gciLogClient: aString

"Passes the contents of the given collection to the GciGetLogger() function
 of the client process.

 If the client is a topaz process, the default logger writes to the
 current output file as controlled by topaz OUTPUT PUSH statements,
 else to stdout.

 If the collection has size > 0 and does not end in an Ascii LineFeed,
 a LineFeed is appended to the bytes passed to the logging function.

 A check for client process disconnected is made at start of the GsfGciLog
 useraction invocation, and if the client has disconnected, this invocation
 will behave as if it were GsFile class >>gciLogServer: .

 If this method is invoked from within a non-blocking Gci execution or
 a GciTs(or GsTsExternalSession) execution , it will behave as GsFile class>>gciLogServer: .
"

self _clientIsRemote ifFalse:[ ^ self gciLogServer: aString].
^ self classUserAction: #GsfGciLog  onClient: true with: aString
%

category: 'Writing'
classmethod: GsFile
gciLogServer: aString

"Passes the contents of the given String to the GciGetLogger() function
 of the server process.
 In a gem process, the default logger writes to stdout, i.e. to the gem
 log file.  In the linked login of a topaz -l process,
 the default logger writes to the current output file as controlled by
 topaz OUTPUT PUSH statements, else to stdout.
 If the collection has size > 0 and does not end in an Ascii LineFeed,
 a LineFeed is appended to the bytes passed to the logging function."

<primitive: 1003>
"primitive accepts String , ByteArray, Utf8 args"
aString _validateClass: MultiByteString .
[ self gciLogServer: aString encodeAsUTF8 ] onException: Error do:[:ex |
  self classUserAction: #GsfGciLog  onClient: false with: aString
].
^ self
%

category: 'Reading'
classmethod: GsFile
getContentsOfServerFile: filePathAndName
"Return the contents of the named file in a single operation.
 Use caution when reading a large file as doing so may cause
 an out of memory error."
 
  | gsf result |
  gsf := GsFile openReadOnServer: filePathAndName.
  gsf ifNil:[ IOError signal: GsFile serverErrorString ] .
  result := gsf contents.
  gsf close.
  ^ result
%

category: 'Directory Operations'
classmethod: GsFile
isClientDirectory: aPathName
"Returns true if 'aPathName' names an existing directory on the client
 machine.
 Returns false if it is not a directory.
 Returns nil if an error occurs."

^ self _isDirectory: aPathName onClient: self _clientIsRemote .
%

category: 'Directory Operations'
classmethod: GsFile
isServerDirectory: aPathName
"Returns true if 'aPathName' names an existing directory on the server
 machine.
 Returns false if it exists but is not a directory.
 Returns nil if it does not exist or an error occurs."

^ self _isDirectory: aPathName onClient: false .
%

category: 'File Operations'
classmethod: GsFile
isSymbolicLink: aPathName onClient: clientBool
  | expPath stats |
  expPath := self _expandFilenamePreserveLinks: aPathName isClient: clientBool.
  stats := self stat: expPath isLstat: true .
  ^ (stats mode bitAnd: 8r170000"S_IFMT") = 8r120000"S_IFLNK" 
%

category: 'Error Reporting'
classmethod: GsFile
lastErrorString

"For class operations on the client, returns the current error string, or nil 
 if no error has occurred, and clears the error string. In linked sesssions, 
 returns and clears the error for client and server class operations."

^ self classUserAction: #GsfClassError onClient: self _clientIsRemote with: nil
%

category: 'File Operations'
classmethod: GsFile
lastModificationOfClientFile: aPathName

"Gets the date and time the named file on the client machine was last
 modified. Returns a DateTime if successful, nil if an error occurs."

^ self _fileModTime: aPathName onClient: self _clientIsRemote .
%

category: 'File Operations'
classmethod: GsFile
lastModificationOfServerFile: aPathName

"Gets the date and time the named file on the server machine was last
 modified. Returns a DateTime if successful, nil if an error occurs."

^ self _fileModTime: aPathName onClient: false .
%

category: 'Private'
classmethod: GsFile
new

"disallowed"

self shouldNotImplement: #new
%

category: 'Instance Creation'
classmethod: GsFile
open: aPathName mode: openMode

"Creates an instance of the receiver and opens a file on the client machine; 
 in a linked session, opens as a server file since there is no difference.

 The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+', 'rb', 'wb', 'ab', 'r+b', 'w+b',
 'a+b', 'rb+', 'wb+', 'ab+'.  The mode has the same meaning as it does for the
 C library function, fopen().

 Returns a GsFile if successful, nil if an error occurs."

^ self open: aPathName mode: openMode onClient: self _clientIsRemote
%

category: 'Instance Creation'
classmethod: GsFile
open: aPathName mode: openMode  onClient: clientBool

"Creates an instance of the receiver and opens a file on the client machine
 (if clientBool is true) or the server machine (if clientBool is false).

 The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+', 'rb', 'wb', 'ab', 'r+b', 'w+b',
 'a+b', 'rb+', 'wb+', 'ab+'.  The mode has the same meaning as it does for the
 C library function, fopen().

 Returns a GsFile if successful, nil if an error occurs."

^self open: aPathName mode: openMode  onClient: clientBool withCompression: false
%

category: 'Instance Creation (compression)'
classmethod: GsFile
open: aPathName mode: openMode  onClient: clientBool withCompression: compBool

"Creates an instance of the receiver and opens a file on the client machine
 (if clientBool is true) or the server machine (if clientBool is false).

 The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+', 'rb', 'wb', 'ab', 'r+b', 'w+b',
 'a+b', 'rb+', 'wb+', 'ab+'.  The mode has the same meaning as it does for the
 C library function, fopen().

 The classmethod GsFile>>validFileModes may be used to generate a list
 of all valid file modes.

 If compBool is true, the file is opened using gzip gzopen() call rather than
 fopen().  In this case, the file is assumed to be a compressed file, although
 gzopen() will also succeed when used to open uncompressed files.  When reading
 a compressed file, it is automatically decompressed as it is read.  Similarly,
 data written to a compressed file is automatically compressed before it is
 written to the file.

 For compressed files, all open mode arguments listed above that do not contain
 a plus sign ('+') are valid.  In addition, any valid write mode argument
 (those contaning 'a' or 'w') may have a '1' or '9' appended.
 Appending a '1' means compress using the best possible speed while appending
  a '9' means compress to attain the best possible compression.

 The classmethod GsFile>>validFileModesForCompression may be used to
 generate a list of all valid file modes for compressed files.

 Returns a GsFile if successful, nil if an error occurs."

| inst |
inst := self _basicNew.
inst _open: aPathName mode: openMode onClient: clientBool.
^ compBool ifTrue:[inst gzOpen] ifFalse:[inst open]
%

category: 'Instance Creation'
classmethod: GsFile
openAppend: aPathName

"Opens a text file on the client machine for writing; in a linked session,
 opens as a server file since there is no difference.  If the file does not 
 exist it is created.  The file position indicator is positioned at the end 
 of the file before each write operation.  Read operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'a' onClient: self _clientIsRemote
%

category: 'Instance Creation'
classmethod: GsFile
openAppendOnServer: aPathName

"Opens a text file on the server machine for writing.
 If the file does not exist it is created.
 The file position indicator is positioned at the end of the file
 before each write operation.
 Read operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'a' onClient: false
%

category: 'Instance Creation (compression)'
classmethod: GsFile
openCompressed: aPathName mode: openMode

"Creates an instance of the receiver and opens a file on the client machine
 using the gzip compression routing gzopen(); in a linked session, opens as 
 a server file since there is no difference.

 The openMode argument must be one of the strings return by the method
 GsFile>>validFileModesForCompression.

 Returns a GsFile if successful, nil if an error occurs."

^ self open: aPathName mode: openMode onClient: self _clientIsRemote withCompression: true
%

category: 'Instance Creation (compression)'
classmethod: GsFile
openCompressed: aPathName mode: openMode  onClient: clientBool

"Creates an instance of the receiver and opens a file on the client machine
 (if clientBool is true) or the server machine (if clientBool is false).

 The openMode argument must be one of the strings return by the method
 GsFile>>validFileModesForCompression.

 Returns a GsFile if successful, nil if an error occurs."

^self open: aPathName mode: openMode  onClient: clientBool withCompression: true
%

category: 'Instance Creation'
classmethod: GsFile
openOnServer: aPathName mode: openMode

"Creates an instance of the receiver and opens a file on the server machine.

 The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+', 'rb', 'wb', 'ab', 'r+b', 'w+b',
 'a+b', 'rb+', 'wb+', 'ab+'.  The mode has the same meaning as it does for the
 C library function, fopen().

 Returns a GsFile if successful, nil if an error occurs."

^ self open: aPathName mode: openMode onClient: false
%

category: 'Instance Creation (compression)'
classmethod: GsFile
openOnServerCompressed: aPathName mode: openMode

"Creates an instance of the receiver and opens a file on the server machine.

 The openMode argument must be one of the strings return by the method
 GsFile>>validFileModesForCompression.

 Returns a GsFile if successful, nil if an error occurs."

^ self open: aPathName mode: openMode onClient: false withCompression: true
%

category: 'Instance Creation'
classmethod: GsFile
openRead: aPathName

"Opens an existing text file on the client machine for reading; in a linked 
 session, opens as a server file since there is no difference.
 Write operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'r' onClient: self _clientIsRemote
%

category: 'Instance Creation (compression)'
classmethod: GsFile
openReadCompressed: aPathName

"Opens an existing text file on the client machine for reading; in a linked session,
 opens as a server file since there is no difference.
 Write operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'rb' onClient: self _clientIsRemote withCompression: true
%

category: 'Instance Creation'
classmethod: GsFile
openReadOnServer: aPathName

"Opens an existing text file on the server machine for reading.
 Write operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'r' onClient: false
%

category: 'Instance Creation (compression)'
classmethod: GsFile
openReadOnServerCompressed: aPathName

"Opens an existing text file on the server machine for reading.
 Write operations are not allowed.

 If the file was compressed with gzip, all data will be
 uncompressed as it is read.

 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'rb' onClient: false withCompression: true
%

category: 'Instance Creation'
classmethod: GsFile
openUpdate: aPathName

"Opens an existing text file on the client machine for reading and writing; 
 in a linked session, opens as a server file since there is no difference.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'r+' onClient: self _clientIsRemote
%

category: 'Instance Creation'
classmethod: GsFile
openUpdateOnServer: aPathName

"Opens an existing text file on the server machine for reading and writing.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'r+' onClient: false
%

category: 'Instance Creation'
classmethod: GsFile
openWrite: aPathName

"Opens a text file on the client machine for writing; in a linked session,
 opens as a server file since there is no difference. 
 If the file exists it is truncated to zero length. If the file does not exist 
 it is created. Read operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'w' onClient: self _clientIsRemote
%

category: 'Instance Creation (compression)'
classmethod: GsFile
openWriteCompressed: aPathName

"Opens a text file on the client machine for writing; in a linked session,
 opens as a server file since there is no difference. If the file exists it is 
 truncated to zero length. If the file does not exist it is created.
 Read operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'wb' onClient: self _clientIsRemote withCompression: true
%

category: 'Instance Creation'
classmethod: GsFile
openWriteOnServer: aPathName

"Opens a text file on the server machine for writing.
 If the file exists it is truncated to zero length.
 If the file does not exist it is created.
 Read operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'w' onClient: false
%

category: 'Instance Creation (compression)'
classmethod: GsFile
openWriteOnServerCompressed: aPathName

"Opens a text file on the server machine for writing.
 If the file exists it is truncated to zero length.
 If the file does not exist it is created.
 Read operations are not allowed.
 All data will be compressed in gzip format before being
 written to the file.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'wb' onClient: false withCompression: true
%

category: 'Directory Operations'
classmethod: GsFile
removeClientDirectory: aPathName

"Removes the named directory from the client machine's file system.
 Returns the receiver if the directory was deleted, nil if an error occurs.
 Note that this method will fail if the directory is not empty."

^ self _removeDirectory: aPathName onClient: self _clientIsRemote .
%

category: 'File Operations'
classmethod: GsFile
removeClientFile: aPathName

"Removes the named file from the client machine's file system.
 Returns the receiver if the file was deleted, nil if an error occurs."

^ self _removeFile: aPathName onClient: self _clientIsRemote .
%

category: 'Directory Operations'
classmethod: GsFile
removeServerDirectory: aPathName

"Removes the named directory from the server machine.
 Returns the receiver if the directory was deleted, nil if an error occurs.
 Note that this method will fail if the directory is not empty."

^ self _removeDirectory: aPathName onClient: false .
%

category: 'File Operations'
classmethod: GsFile
removeServerFile: aPathName

"Removes the named file from the server machine.  Returns the receiver if the
 file was deleted, nil if an error occurs."

^ self _removeFile: aPathName onClient: false .
%

category: 'File Operations'
classmethod: GsFile
renameFile: oldName to: newName

"Rename the given file to the new name.  Returns 0 if successful, nil or a non-zero
  SmallInteger if not.  The resulting SmallInteger is the UNIX error number (errno)
  returned by the C function rename().  See the man page for rename on your system
  for errno value definitions."

| oldN newN |
oldN := self _utfPath: oldName forClient: true .
newN := self _utfPath: newName forClient: true .
^ self classUserAction: #GsfRename onClient: self _clientIsRemote with: oldN with: newN
%

category: 'File Operations'
classmethod: GsFile
renameFileOnServer: oldName to: newName

"Rename the given file to the new name.  Returns 0 if successful, nil or a non-zero
  SmallInteger if not.  The resulting SmallInteger is the UNIX error number (errno)
  returned by the C function rename().  See the man page for rename on your system
  for errno value definitions."

| oldN newN |
oldN := self _utfPath: oldName forClient: false .
newN := self _utfPath: newName forClient: false .

^ self classUserAction: #GsfRename onClient: false with: oldN with: newN
%

category: 'Directory Operations'
classmethod: GsFile
serverChangeDirectory: aString

"Changes the directory of the gem or topaz -l process on the server to aString.
Returns true on success or false if the directory change was not successful. 
Raises an error is aString is not a kind of String or Utf8."

^ 0 == (self _directoryPrim: 0 with: aString with: nil)
%

category: 'Directory Operations'
classmethod: GsFile
serverCurrentDirectory

"Returns a String, the current directory of this topaz -l  or gem process."
^ self _directoryPrim: 2 with: nil with: nil
%

category: 'Error Reporting'
classmethod: GsFile
serverErrorString

"For class operations on the server, returns the current error string, or nil 
 if no error has occurred, and clears the error string. In linked sesssions, 
 returns and clears the error for server and client class operations."

^ self classUserAction: #GsfClassError onClient: false with: nil
%

category: 'File Operations'
classmethod: GsFile
serverRealPath: aPathName
  "Return the result of calling the posix function realpath for aPathName."
  | res |
  res := self _directoryPrim: 3 with: aPathName with: nil .
  res _isSmallInteger ifTrue:[  
    SystemCallError new errno: res ; signal .
  ].
  ^ res
%

category: 'File Operations'
classmethod: GsFile
sizeOf: aPathName

"Returns the size in bytes of the given client file, false if
 file does not exist, or nil if any other error occurs."

^ self _sizeOf: aPathName onClient: self _clientIsRemote
%

category: 'File Operations'
classmethod: GsFile
sizeOfOnServer: aPathName

"Returns the size in bytes of the given server file, false if
 file does not exist, or nil if any other error occurs."

^ self _sizeOf: aPathName onClient: false
%

category: 'File Operations'
classmethod: GsFile
stat: aName isLstat: aBoolean

"uses the server file system access.
 Returns a SmallInteger errno value if an error occurs or
 or if aName is not a valid file or directory. Otherwise
 returns a new instance of GsFileStat. "

^ self _stat: (self _utfPath: aName forClient: false) isLstat: aBoolean
%

category: 'Standard Files'
classmethod: GsFile
stderr

"Returns an instance of the receiver that is set up to write to the standard
 error output of the client process, or nil if an error occurs."

^self _getStdFile: 2 onClient: self _clientIsRemote
%

category: 'Standard Files'
classmethod: GsFile
stderrServer

"Returns an instance of the receiver that is set up to write to the standard
 error of the server process, or nil if an error occurs."

^self _getStdFile: 2 onClient: false
%

category: 'Standard Files'
classmethod: GsFile
stdin

"Returns an instance of the receiver that is set up to read the standard input
 of the client process, or nil if an error occurs."

^self _getStdFile: 0 onClient: self _clientIsRemote
%

category: 'Standard Files'
classmethod: GsFile
stdinServer

"Returns an instance of the receiver that is set up to read the standard input
 of the server process, or nil if an error occurs."

^self _getStdFile: 0 onClient: false
%

category: 'Standard Files'
classmethod: GsFile
stdout

"Returns an instance of the receiver that is set up to write to the standard
 output of the client process, or nil if an error occurs."

^self _getStdFile: 1 onClient: self _clientIsRemote
%

category: 'Standard Files'
classmethod: GsFile
stdoutServer

"Returns an instance of the receiver that is set up to write to the standard
 output of the server process, or nil if an error occurs."

^self _getStdFile: 1 onClient: false
%

category: 'File Modes'
classmethod: GsFile
validFileModes

"Answers an array of strings, each of which is a valid mode for
 opening files without compression."

^self classUserAction: #GsfAllValidFileModes onClient: self _clientIsRemote with: false
%

category: 'File Modes'
classmethod: GsFile
validFileModesForCompression

"Answers an array of strings, each of which is a valid mode for
 opening files with compression."

^self classUserAction: #GsfAllValidFileModes onClient: self _clientIsRemote with: true
%

category: 'Private'
classmethod: GsFile
_basicNew

"creates an instance registered with VM for finalization of cData"

<primitive: 674>
self _primitiveFailed: #_basicNew
%

category: 'Private'
classmethod: GsFile
_clientIsRemote
  ^ System clientIsRemote
%

category: 'Private'
classmethod: GsFile
_clientVersionAt: verParamId

"Returns client process version information.  verInfoId should be one of the
 values in the global dictionary VersionParameterDict."

^ self classUserAction: #GsfVersionParam onClient: self _clientIsRemote with: verParamId
%

category: 'Private'
classmethod: GsFile
_contentsOfServerDirectory: aPathName expandPath: aBoolean utf8Results: u8Boolean

"Returns an Array, or a SmallInteger errno.
 aPathName must be a String, Unicode string , Utf8 or Utf16  .
 The result is an Array of file names in the specified directory.
 If aBoolean is true, the result includes the expansion of aPathName
 as a prefix of each element of the result.
 If u8Boolean is true, elements of result not representable with 7 bit code points
 will be Utf8 .
 "

^ self _primContentsOfServerDirectory: (self _utfPath: aPathName forClient: false)
         expandPath: aBoolean utf8Results: u8Boolean
%

category: 'Private'
classmethod: GsFile
_createDirectory: aPathName onClient: clientBool mode: modeInt

"Returns receiver if creation succeeds.  Returns nil if an error occurs,
 in which case the class error buffer contains the error."
| path |
path := self _utfPath: aPathName forClient: clientBool .
^ self classUserAction: #GsfMkdir onClient: clientBool with: path with: modeInt
%

category: 'Private'
classmethod: GsFile
_decodeUtfResult: obj unicode: unicodeCompareEnabled
 | unic |
 unic := unicodeCompareEnabled ifNil:[ Unicode16 _unicodeCompareEnabled ].
 obj ifNil:[ ^ obj ].
 ((obj isKindOfClass: Utf8) or:[ obj isKindOfClass: Utf16]) ifTrue:[
   ^ unic ifTrue:[ obj decodeToUnicode] ifFalse:[ obj decodeToString].
 ].
 ^ unic ifTrue:[ obj asUnicodeString ] ifFalse:[ obj ].
%

category: 'Private'
classmethod: GsFile
_directoryPrim: opcode with: arg1 with: arg2

"opcode 0    chdir   arg1 is a String or Utf8 , arg2 ignored
 opcode 1    rmdir   arg1 is a String or Utf8 , arg2 ignored
 opcode 2    getcwd,   arg1 and arg2 ignored
 opcode 3    realpath   arg1 is a String or Utf8, arg2 ignored.
 opcode 4    mkdir   arg1 is a String or Utf8 , arg2 ignored
 opcode 5    umask   arg1 ignored, arg1 a 32 bit SmallInteger 
                          arg2 < 0 , result = umask(0); umask(result) .
                          arg2 >= 0  result = umask(arg2)
 opcode 6    gciClientIsWindows   arg1 and arg2 ignored "
<primitive: 765>
^ self _primitiveFailed: #_directoryPrim:with:with: args: { opcode. arg1 . arg2 }
%

category: 'Private'
classmethod: GsFile
_exists: aPathName onClient: clientBool

"Returns true if the given path points to a file, false if not, and nil if an
 error other than file-does-not-exist occurs trying to find out."

| result path |
path := self _utfPath: aPathName forClient: clientBool .
"GsfSize returns -1 for non-existant file , nil for other error ,
 >= 0 for a file that exists "
result := self classUserAction: #GsfSize onClient: clientBool with: path .
result == -1 ifTrue:[ ^ false "file does not exist" ].
result == nil ifTrue:[ ^ nil "some other error" ] .
^ true
%

category: 'Private'
classmethod: GsFile
_expandEnvVariable: varName isClient: clientBool

"Expands the environment variable named varName in either the GemBuilder for C
 or Gem process, returning a String. varName should be a kind of String.

 Returns nil if any of the following are true
    varName is not a byte format object.
    there is no environment variable defined with name  varName,
    the value of the environment variable is more than approximately 8000 bytes,
    clientBool is true and the size of varName exceeds approximately 8000 bytes."
| vn res |
vn := self _utfPath: varName forClient: clientBool .
res := self classUserAction: #GsfExpandEnvVar onClient: clientBool with: vn with: 0 .
^ self _decodeUtfResult: res unicode: nil .
%

category: 'Private'
classmethod: GsFile
_expandFilename: fName isClient: clientBool

"Expands fName using HostExpandFileName including following symbolic links.
 in either GCI or Gem process, returning a String. fName should be a kind of String.

 Returns nil if any of the following are true
    fName is not a byte format object.
    For each environment variable name varName within fName ,
      there is no environment variable defined with name  varName,
      the value of the environment variable is more than approximately 8000 bytes,
      clientBool is true and the size of varName exceeds approximately 8000 bytes."

| nam res |
nam := self _utfPath: fName forClient: clientBool .
res := self classUserAction: #GsfExpandEnvVar onClient: clientBool with: nam with: 1 .
^ self _decodeUtfResult: res unicode: nil
%

category: 'Private'
classmethod: GsFile
_expandFilenamePreserveLinks: fName isClient: clientBool

"Expands fName using HostExpandFileName without following symbolic links.
 in either GCI or Gem process, returning a String. fName should be a kind of String.

 Returns nil if any of the following are true
    varName is not a byte format object.
    there is no environment variable defined with name  varName,
    the value of the environment variable is more than approximately 8000 bytes,
    clientBool is true and the size of varName exceeds approximately 8000 bytes."

| nam res |
nam := self _utfPath: fName forClient: clientBool .
res := self classUserAction: #GsfExpandEnvVar onClient: clientBool with: nam with: 2 .
^ self _decodeUtfResult: res unicode: nil
%

category: 'Private'
classmethod: GsFile
_fileKind: aPathName onClient: clientBool

"Returns a SmallInteger representing the kind of the file, or returns nil if an
 error occurs, in which case the class error buffer contains the error.

 The file kinds are enumerated as follows.  Symbolic links are reported as
 the type of the file pointed to.

    0. file
    1. directory
    2. character device
    3. block device
    4. symbolic link (not applicable)
    5. other
    6. error
    7. FIFO
    8. stdin, stdout, or stderr
    9. stdin or stdout connected to a terminal (isatty(fileno(f) == 1)  "
| path |
path := self _utfPath: aPathName forClient: clientBool .
^ self classUserAction: #GsfFileKind onClient: clientBool with: path
%

category: 'Private'
classmethod: GsFile
_fileModTime: aPathName onClient: clientBool

"Returns DateTime of files last modification time if successful.
 Returns nil if an error occurs,
 in which case the class error buffer contains the error."
| path |
path := self _utfPath: aPathName forClient: clientBool .
^ self classUserAction: #GsfModTime onClient: clientBool with: path
%

category: 'Private'
classmethod: GsFile
_fstat: aFileDescriptor isLstat: aBoolean

"uses the server file system access.
 Returns a SmallInteger errno value if an error occurs or
 or if aFileDescriptor is not a valid file descriptor . Otherwise
 returns a new instance of GsFileStat. "
<primitive: 758>
aFileDescriptor _validateClass: SmallInteger .
aBoolean _validateClass: Boolean .
self _primitiveFailed: #_fstat:isLstat: args: { aFileDescriptor . aBoolean }
%

category: 'Private'
classmethod: GsFile
_getStdFile: stdId onClient: clientBool

"Returns an instance of the receiver that is set up to operate on a standard
 file of the client or nil if an error occurs.  stdId can be 0 (stdin),
 1 (stdout), or 2 (stderr)."

|result stdIdCache virtualId |

(stdIdCache := System __sessionStateAt: 18) ifNil:[
  stdIdCache := { nil . nil . nil . nil . nil . nil } .
  System __sessionStateAt: 18 put: stdIdCache.
].
clientBool ifTrue:[   virtualId := stdId + 1 ]
          ifFalse:[  virtualId := stdId + 4 ].
result := stdIdCache at: virtualId .
result == nil ifTrue: [ | status |
  "It has not yet been cached so create it"
  result := self _basicNew .
  result _newStdFile: stdId isClient: clientBool .
  "tell the client about it"
  status := result userAction: #GsfCreateStdFile onClient: clientBool with: stdId.
  status == nil ifFalse:[
    stdIdCache at: virtualId put: result.
    status == 1 ifTrue:[ result _newEmptyClientCData ].
  ] ifTrue: [
    result := nil.
  ].
].

^ result
%

category: 'Private'
classmethod: GsFile
_isDirectory: aPathName onClient: clientBool

"Returns true if 'aPathName' names an existing directory.
 Returns false if it is not a directory.
 Returns nil if an error occurs."
| path |
path := self _utfPath: aPathName forClient: clientBool .
^ self classUserAction: #GsfIsDir onClient: clientBool with: path
%

category: 'Private'
classmethod: GsFile
_log: string
  "Convenience method to print to stdout just like log: does."

  self stdout log: string
%

category: 'Private'
classmethod: GsFile
_primContentsOfServerDirectory: aPathName expandPath: aBoolean utf8Results: u8Boolean

"Returns an Array, or a SmallInteger errno.
 aPathName must be a String,  Utf8 or Utf16.
 The result is an Array of file names in the specified directory.
 If aBoolean is true, the result includes the expansion of aPathName
 as a prefix of each element of the result.
 If u8Boolean is true, elements of result not representable with 7 bit code points
 will be Utf8 ."
<primitive: 431>
aPathName _validateClasses: { String . Utf8 }.
aBoolean _validateClass: Boolean .
u8Boolean _validateClass: Boolean .
self _primitiveFailed: #_primContentsOfServerDirectory:
     args: { aPathName . aBoolean . u8Boolean}
%

category: 'Private'
classmethod: GsFile
_removeDirectory: aPathName onClient: clientBool

"Returns receiver if removal succeeds.  Returns nil if an error occurs,
 in which case the class error buffer contains the error."
| path |
path := self _utfPath: aPathName forClient: clientBool .
^ self classUserAction: #GsfRmdir onClient: clientBool with: path
%

category: 'Private'
classmethod: GsFile
_removeFile: aPathName onClient: clientBool

"Returns receiver if removal succeeds.  Returns nil if an error occurs,
 in which case the class error buffer contains the error."
| path |
path := self _utfPath: aPathName forClient: clientBool .
^ self classUserAction: #GsfUnlink onClient: clientBool with: path
%

category: 'Private'
classmethod: GsFile
_setEnvVariable: varName value: valueStr isClient: clientBool

"Both arguments must be either a String or a MultiByteString representable in
 8000 bytes of Utf8 or Utf16 encoding.
 If valueStr is nil or of size 0 , implements unsetenv semantics.
"

| nam val res |
nam := self _utfPath: varName forClient: clientBool .
valueStr ifNotNil:[
  val := self _utfPath: valueStr forClient: clientBool .
  val size = 0 ifTrue:[ val := nil  "unsetenv"].
].
"check of NoGsFileOnServer or NoGsFileOnClient privilege is done by C code of prim 396"
res := self classUserAction: #GsfSetEnvVar onClient: clientBool with: nam with: val .
res ifNil:[ Error signal: 'GsfSetEnvVar failed' ].
^ self
%

category: 'Error Reporting'
classmethod: GsFile
_setLastError: aString onClient: clientBool

self classUserAction: #GsfClassError onClient: clientBool with: aString
%

category: 'Private'
classmethod: GsFile
_sizeOf: aPathName onClient: clientBool

"Returns size of the file in bytes.  Returns nil if an error occurs,
 in which case the class error buffer contains the error."

| result path |
path := self _utfPath: aPathName forClient: clientBool .

"GsfSize returns -1 for non-existant file , nil for other error ,
 >= 0 for a file that exists "
result := self classUserAction: #GsfSize onClient: clientBool with: path .
result == nil ifTrue:[ ^ nil ] .
result == -1 ifTrue:[ ^ false "file does not exist" ].
^ result
%

category: 'Private'
classmethod: GsFile
_stat: aName isLstat: aBoolean

"uses the server file system access.
 Returns a SmallInteger errno value if an error occurs or
 or if aName is not a valid file or directory. Otherwise
 returns a new instance of GsFileStat. "
<primitive: 757>
aName _validateClasses: { String . Utf8 } .
aBoolean _validateClass: Boolean .
self _primitiveFailed: #_stat:isLstat: args: { aName . aBoolean }
%

category: 'Compressed File Operations'
classmethod: GsFile
_utfPath: aPath forClient: clientBool
  "Returns a String or Unicode7 with 7bit characters; or a Utf8 or Utf16"
  | isClient |
  (clientBool == false or:[ clientBool == 0 or:[ clientBool == 2]]) ifTrue:[
    isClient := false .
  ] ifFalse:[
    (clientBool == true or:[ clientBool == 1 or:[ clientBool == 3]]) ifTrue:[
      isClient := true .
    ] ifFalse:[
      ArgumentError signal:'invalid clientBool argument, a', clientBool class name
    ]
  ].
  (isClient and:[ self gciClientIsWindows]) ifTrue:[
    ^ aPath encodeAsUTF16  "file operation executing in Windows client"
  ].
  aPath _stringCharSize > 0 ifTrue:[
    aPath _asUnicode7 ifNotNil:[:p |
      "all chars are 7 bit"
      aPath isUnicodeString ifFalse:[ ^ String withAll: p ].
    ].
  ].
  ^ aPath encodeAsUTF8  "file operation on Unix client or in gem process"
%

!		Instance methods for 'GsFile'

category: 'Writing'
method: GsFile
+ collection

"Writes the contents of the given collection to the receiver's file at
 the current position. The argument must be a Collection with byte format.
 Returns a count of bytes added, or nil if an error occurs."

self deprecated: 'GsFile>>+ deprecated v3.2. Use the , method instead.'.
^  self nextPutAll: collection
%

category: 'Writing'
method: GsFile
, collection

"Writes the contents of the given collection to the receiver's file at
 the current position. The argument must be a Collection with byte format.
 Returns a count of bytes added, or nil if an error occurs."

^  self nextPutAll: collection
%

category: 'Comparing'
method: GsFile
= aFile

"Returns true if the receiver and aFile represent the same file system file.
 Returns false otherwise.

 Result is always false if either receiver or argument are not open."

(self == aFile) ifTrue:[ ^ true].
aFile == nil ifTrue:[ ^ false].
(aFile isKindOf: self class) ifFalse:[ ^ false].
(self isOpen and:[ aFile isOpen]) ifTrue:[
   self isClient = aFile isClient  ifTrue:[
    pathName = aFile pathName ifFalse:[
      ^ fileDescriptor = aFile fileDescriptor
    ].
    ^ true
  ].
].
^ false
%

category: 'Writing'
method: GsFile
add: char

"Writes the given Character to the receiver's file at the current position.
 Returns true, or nil if an error occurs."

^ self nextPut: char
%

category: 'Writing'
method: GsFile
addAll: collection

"Writes the contents of the given collection to the receiver's file at
 the current position. The argument must be a Collection
 with byte format.  Returns a count of bytes added, or nil if an error occurs."

^ self nextPutAll: collection
%

category: 'Stream Compatibility'
method: GsFile
atBeginning
"Answer true if the stream is positioned at the beginning"

^self position == 0
%

category: 'Positioning'
method: GsFile
atEnd

"Returns true if the receiver is currently positioned at the end of its
 file, false if not, or nil if an error occurs."

<primitive: 724>  "primitive succeeds only if file is on server"
(isClient _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ ^ self _setNotOpenError ].
^ self userAction: #GsfAtEnd onClient: isClient
%

category: 'Positioning'
method: GsFile
beforeEnd
  ^ self atEnd == false  
%

category: 'File Operations'
method: GsFile
close

"Closes the receiver's file.  Returns the receiver, or nil if an error occurs.
 If receiver is a standard file, has no effect and returns receiver."

 | res stdIdCache |
 self isOpen ifFalse: [^ self ].
 stdIdCache := System __sessionStateAt: 18.
 stdIdCache == nil ifFalse: [
   (stdIdCache includesIdentical: self) ifTrue:[
   fileDescriptor := -1 .
     ^ self
   ]
 ].
 fileDescriptor := -1 .
 res := self userAction: #GsfClose onClient: isClient .
 res == nil ifFalse:[ self _setNotOpened ].
 ^ res
%

category: 'Reading'
method: GsFile
contents

"Returns a String containing the contents of the receiver from the current
 position to the end of the file.  Returns nil if an error occurs."

^ self isCompressed
    ifTrue:[self _contentsUncompressed: String new ]
    ifFalse:[self _contents: String new ]
%

category: 'Reading'
method: GsFile
contentsAsUtf8

"Returns a Utf8 containing the contents of the receiver from the current
 position to the end of the file.  Returns nil if an error occurs."

^ self isCompressed
    ifTrue:[self _contentsUncompressed: Utf8 new ]
    ifFalse:[self _contents: Utf8 new ]
%

category: 'Writing'
method: GsFile
cr

"If the receiver is a text file then writes an end-of-line sequence to it.
 If the receiver is a binary file then writes a carriage return to it.
 Returns a count of bytes added, or nil if an error occurs."

(self _isBinary) ifTrue: [ | isCl |
  ((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ ^ self _setNotOpenError ].
  ^ self userAction: #GsfPutC onClient: isCl with: 13 "ASCII carriage return"
] ifFalse: [
  "ANSI-C text files will write eoln when they see a single lf character."
  ^ self lf
].
%

category: 'ANSI Compatibility'
method: GsFile
do: aBlock
"To be compatible with ANSI gettableStream (5.9.2.2):
Each member of the receiver's future sequence values is, in turn, removed from the future
sequence values; appended to the past sequence values; and, passed as the argument to an
evaluation of operand. The argument, operation, is evaluated as if sent the message #value:.
The number of evaluations is equal to the initial size of the receiver's future sequence
values. If there initially are no future sequence values, operation is not evaluated.
The future sequence values are used as arguments in their sequence order. The result is
undefined if any evaluation of operand changes the receiver's future sequence values"

	self positionA to: self fileSize - 1 do: [:i |
		aBlock value: self next.
	].
%

category: 'Writing'
method: GsFile
ff

"Writes a form-feed (page break) to the receiver's file.
 Returns true, or nil if an error occurs."

^ self nextPut:  Character newPage
%

category: 'Accessing'
method: GsFile
filePointer
"Returns a CPointer representing the underlying FILE * in C.
 Valid for open server files only. Returns nil the file is compressed,
 on the client, or if an error occurs. Care must be taken to not access
 the result of this method after the receiver has been closed."

^ self _zeroArgPrim: 3
%

category: 'File Operations'
method: GsFile
fileSize

"Returns the size in bytes of the receiver, or nil if an error occurs.

 Note that the value returned is independent of the open mode used to
 create the receiver."

^ self class _sizeOf: pathName onClient: isClient
%

category: 'File Operations'
method: GsFile
flush

"Flushes all written bytes to the file.  Returns the receiver, or nil if an
 error occurs. On Unix, calls fflush() . "
| isCl |
((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[
   ^ self _setNotOpenError
].
^ self userAction: #GsfFlush onClient: isCl
%

category: 'Locking'
method: GsFile
getStatusForLockKind: kind atOffset: offset forBytes: bytes
"Tests if the receiver could be locked with the given arguments but without
 actually acquiring the lock.  Returns information about a conflicting lock,
 if any.

 The kind argument must be a symbol that indicates the lock kind requested, and
 must be either #readLock or #writeLock.  offset indicates the position within
 the receiver in bytes.  bytes indicates the number of bytes starting at offset
 (0 means to the end of the file).

 Returns an Array of 5 elements as described below.  If the first element is
 false, then elements 2 through 5 contain information about a conflicting
 lock (there may be more than one) which prevented the lock from being available.
 If the first element is true, then element 2 will be nil and elements 3
 through 5 will be -1.

 1 - a Boolean: true if the lock would have been granted, otherwise false.
 2 - a Symbol (#readLock or #writeLock): conflicting lock kind.
 3 - a SmallInteger: start position of the conflicting lock.
 4 - a SmallInteger: number of bytes held by the conflicting lock.
 5 - a SmallInteger: process ID of the owner of the conflicting lock.

 Raises an IOError if an error occurs.

 This method is not supported and will always raise an error if the receiver
 references a client file and the GCI client is running on the Microsoft
 Windows platform."

| result |
result := self userAction: #GsfLockQuery onClient: self isClient
               with: kind with: offset with: bytes .
^ result class == Array
     ifTrue:[ result ]
    ifFalse:[ IOError signal: result ]
%

category: 'Compressed File Operations'
method: GsFile
gzOpen

"If the receiver is not open, open it using the existing mode.
 Returns the receiver, or nil if an error occurs.
 If an error occurs,  there is no error information associated with
 the instance; you must use    GsFile lastErrorString   to retrieve the error. "

^self _openUsingGzip: true
%

category: 'Compressed File Operations'
method: GsFile
gzOpen: aPathName mode: openMode

"Opens the receiver's file with the given mode.  If the file is already open,
 it is closed and reopened.

 The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+', 'rb', 'wb', 'ab', 'r+b', 'w+b',
 'a+b', 'rb+', 'wb+', 'ab+'.  The mode has the same meaning as it does for the
 C library function, fopen().

 Returns the receiver if successful, or nil if an error occurs.
 If an error occurs,  there is no error information associated with
 the instance; you must use   GsFile lastErrorString to retrieve the error. "

self isOpen ifTrue:[ self close ].
mode := openMode.
pathName := aPathName.
^self gzOpen
%

category: 'Comparing'
method: GsFile
hash

"Returns a SmallInteger related to the value of the receiver.  If two instances
 of GsFile are equal (as compared by the = method), then they must have the same
 hash value.
 Gemstone 64 v3.0, hash algorithm has changed for GsFile
 "

^ pathName hash
%

category: 'File Operations'
method: GsFile
head: lineCount

"Returns a String containing the first lineCount lines from the
 receiver's file, or nil if an error occurs."

| result line |

self position: 0.
result := String new.
lineCount timesRepeat: [
  line := self nextLine.
  line ~~ nil ifTrue: [
    result addAll: line.
  ]
  ifFalse: [
    ^result
  ]
].
^result
%

category: 'Testing'
method: GsFile
isClient

"Returns true if the receiver's file is a client file, or nil if an error
 occurs."
| isCl |
(isCl := isClient) _isSmallInteger ifTrue:[ ^ (isCl bitAnd:1) == 1 ].
^ isCl
%

category: 'Testing'
method: GsFile
isCompressed

"Returns true if the file was opened in compressed mode or false
 if it was not or an uncompressed file was opened in compressed mode.

 Returns nil if the receiver is not in a valid open state."

^ (self _zeroArgPrim: 1) ifNil:[
    self userAction: #GsfIsCompressed onClient: isClient
  ]
%

category: 'ANSI Compatibility'
method: GsFile
isEmpty
"To be compatible with ANSI sequencedStream (5.9.1.3):
Returns true if both the set of past and future sequence values of the receiver are empty.
Otherwise returns false."

	^self fileSize == 0.
%

category: 'Testing'
method: GsFile
isExternal

"Is the source for this stream is external to GemStone Smalltalk."

^true
%

category: 'Testing'
method: GsFile
isOpen

"Returns true if the receiver's file is open, false otherwise"
| isCl |
((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[
  ^ false
].
isClient ifNil:[ ^ false "an instance created with _basicNew" ].
^ self userAction: #GsfIsOpen onClient: isCl
%

category: 'Standard Files'
method: GsFile
isTerminal

"Returns true if receiver is stdin or stdout, and is connected to a terminal,
  i.e. in the client process , in C,  istty(fileno(f)) == 1 "

^ (pathName at: 1 equals:'std')
   and:[ (GsFile _fileKind: pathName onClient: self isClient) == 9 ]
%

category: 'Error Reporting'
method: GsFile
lastErrorCode

"Returns the currently posted error code, or zero if no error has occurred.
 A result of -1  means
   'attempt to access a GsFile that was never opened or has been closed' .
 Does not clear the error code or error string.

 If the last error was a failure during reopen of a closed file,
 this method returns zero;  you must use lastErrorString to get information
 on such reopen  failures.   This situation applies to errors produce by
 sending  open   or   open:mode:    to an instance of GsFile. "

 isClient _isSmallInteger ifFalse:[
   "a pre v2.1 committed or open instance, get result from C"
   ^ self userAction: #GsfErrorCode onClient: isClient
 ].
 self isCommitted ifTrue:[
   "v2.1 committed instance, get result from C"
   ^ self userAction: #GsfErrorCode onClient: isClient
 ].
 "at this point we have a v2.1 transient instance."
 isClient >= 2 ifTrue:[
   ^ -1 "attempt to access a GsFile that was never opened or has been closed."
 ].
 ^ 0
%

category: 'Error Reporting'
method: GsFile
lastErrorString

"Returns the currently posted error string, or nil if no error has occurred.
 Clears the error string and error code.
 If the last error was a failure during reopen of a closed file,
 will retrieve and clear the error string from the class."

 isClient _isSmallInteger ifFalse:[
   "a pre v2.1 committed or open instance, get result from C"
   ^ self userAction: #GsfErrorString onClient: isClient
 ].
 self isCommitted ifTrue:[
   "v2.1 committed instance in any state, get result from C"
   ^ self userAction: #GsfErrorString onClient: isClient
 ].
 "at this point we have a v2.1 transient instance."
 isClient >= 2 ifTrue:[
   "change state from 'closed, access error' to 'closed' and return err string."
   isClient := isClient - 2 .
   ^ 'attempt to access a GsFile that was never opened or has been closed'.
 ].
 "get class' error string for last unsuccessful open/reopen."
 ^ self class lastErrorString .
%

category: 'Accessing'
method: GsFile
lastModified
"Returns a DateTime that represents the last time the receiver's file
 was modified. Returns nil if an error occurs."

^ GsFile _fileModTime: (self pathName) onClient: isClient .
%

category: 'Writing'
method: GsFile
lf

"Append an end-of-line to the receiver.
 Returns true if successful, nil if an error occurred."

^ self nextPut: 10"ASCII line-feed Character"
%

category: 'Writing'
method: GsFile
log: string

"Writes the contents of the given collection to the receiver's file at the
 current position.  Appends a newline to the file if the string does not end
 with one.  The argument must be a kind of Collection with byte format.

 Returns the receiver if successful; returns nil otherwise."

| lf result |

result := self nextPutAll: string.
(result ~~ nil and:[ result > 0]) ifTrue:[
  lf := Character lf.
  string last == lf ifFalse: [ result := self nextPut: lf ]
  ].
result == nil ifTrue: [ ^nil ].
self flush.
^ self
%

category: 'Accessing'
method: GsFile
maxSize
  "For compatibility with Stream , return a large value"

  ^ SmallInteger maximumValue
%

category: 'Accessing'
method: GsFile
mode

"Returns the access mode of the receiver's file."

^mode
%

category: 'Accessing'
method: GsFile
name

"Returns the receiver's file path name."

^ pathName
%

category: 'Reading'
method: GsFile
next

"Returns a Character representing the next byte from the receiver's file.
 Returns nil if an error occurs, or if receiver is at EOF."

<primitive: 1014>
"primitive fails if receiver not an open server file, or if EINTR occurs"
| isCl res |
((isCl :=isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[
  ^ self _setNotOpenError
].
[
  res := self userAction: #GsfGetC onClient: isCl .
  res _isSmallInteger   "repeat loop if EINTR , to allow ctl-C detection"
] untilFalse .
^ res
%

category: 'Reading'
method: GsFile
next: numberOfBytes

"Returns a String containing the next numberOfBytes Characters from the
 receiver's file, or nil if an error occurs."

| result count |
result := String new.
count := self _read: numberOfBytes into: result .
(count == 0 or:[ count == nil]) ifTrue:[ ^ nil ].
^ result
%

category: 'Reading'
method: GsFile
next: amount byteStringsInto: byteObj

"Reads bytes written by printBytes: into the given byte object.
 Returns count of bytes read, or nil if an error occurs."
| isCl |
((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[
   ^ self _setNotOpenError
].
^ self userAction: #GsfLoadByteStrings onClient: isCl with: byteObj with: amount
%

category: 'Reading'
method: GsFile
next: numberOfBytes into: aByteObject

"Reads the next numberOfBytes into the given collection object. The
 object's size is truncated to the amount of data actually read.
 Returns a count of bytes read, or nil if EOF or an error occurs."

| cnt |
cnt := self _read: numberOfBytes into: aByteObject .
cnt == 0 ifTrue:[ ^ nil ].
cnt ifNotNil:[
  cnt < aByteObject size ifTrue:[ aByteObject size: cnt ].
].
^cnt
%

category: 'Reading'
method: GsFile
next: numberOfItems ofSize: bytesPerItem into: byteObj

"Reads bytes for the next numberOfItems of the given bytesPerItem into
 the given collection object. The object's size is truncated to the
 amount of data actually read.  bytesPerItem must between 1 and 4096 inclusive.

 Returns a count of bytes read, or nil if an error occurs."
| count |
count := self _read: numberOfItems ofSize: bytesPerItem into: byteObj .
(count == 0 or:[ count == nil]) ifTrue:[ ^ nil ].
count < byteObj size ifTrue:[ byteObj size: count ].
^ count
%

category: 'Reading'
method: GsFile
nextByte

"Returns the next byte (integer) from the receiver's file.
 Returns nil if an error occurs, or if receiver is at EOF."

| c |
c := self next .
c ifNil:[ ^ nil ].
^ c codePoint
%

category: 'Reading'
method: GsFile
nextLine

"Returns a String containing the next line from the receiver's file. The String
 will be terminated with a newline, unless the end of file is reached and there
 is no line terminator.  If receiver is positioned at the end of the file, or
 a read error occurs, returns nil.

 There is no limit on line size."

^ self nextLineTo: Character lfValue prompt: nil
%

category: 'Reading'
method: GsFile
nextLineInto: str startingAt: pos

"Deprecated,  new code should use  nextLineTo: .
 Reads the next line from the receiver's file into the given collection object,
 starting at the given position in the collection. The collection will be
 terminated with a newline, unless the end of file is reached and there is no
 line terminator.  If the receiver is positioned at the end of the file, nothing
 is written.  Returns a count of bytes read, or nil if an error occurs."

"There is no limit on line size."

| res isCl |
self deprecated: 'GsFile>>nextLineInto:startingAt: deprecated in v3.2, use nextLineTo: instead'.
((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[
   ^ self _setNotOpenError
].
res := self userAction: #GsfGetLine onClient: isCl with: str with: pos
		with: Character lfValue with: nil .
res ~~ nil ifTrue:[ ^ res size - pos + 1 ].
^ res
%

category: 'Reading'
method: GsFile
nextLineTo: eolValue
  ^ self nextLineTo: eolValue prompt: nil
%

category: 'Reading'
method: GsFile
nextLineTo: eolValue prompt: promptString
| res |
[
  "if receiver is stdin, GsfGetLine will return one of
    nil (for EOF), a String, or a SmallInteger (if interrupted by a ctl-C) .
   otherwise GsfGetLine returns nil or a String .
   By rerunning this loop if a SmallInteger is obtained, we give a chance
   for the SIGINT handler to signal a soft-break error
  "
  res := self _nextLineTo: eolValue prompt: promptString .
  res _isSmallInteger
] untilFalse .
^ res
%

category: 'Reading'
method: GsFile
nextLineToChar: eolCharacter

^ self nextLineTo: eolCharacter codePoint prompt: nil
%

category: 'ANSI Compatibility'
method: GsFile
nextMatchFor: anObject
"To be compatible with ANSI gettableStream (5.9.2.2):
The first object is removed from the receiver's future sequence value and appended to
the end of the receiver's past sequence values. The value that would result from sending
#= to the object with anObject as the argument is returned.
The results are undefined if there are no future sequence values in the receiver."

	^self next = anObject.
%

category: 'Writing'
method: GsFile
nextPut: aByte

"Writes the given byte to the receiver's file at the current position.
 aByte must be a Character or a SmallInteger in the range 0..255.

 If aByte is a SmallInteger, it will be interpreted logically as

 Character withValue: aByte

 Returns true, or nil if an error occurs."

<primitive: 1011>
"handle file not open on server"
| isCl |
((isCl :=isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[
  ^ self _setNotOpenError
].
^ self userAction: #GsfPutC onClient: isCl with: aByte
%

category: 'Writing'
method: GsFile
nextPutAll: collection

"Writes the contents of the given collection to the receiver's file at the
 current position. The argument must be a kind of Collection with byte
 format.  Returns a count of bytes added, or nil if an error occurs."

^ self write: (collection _basicSize) from: collection
%

category: 'Writing'
method: GsFile
nextPutAllBytes: collection

"Writes the byte contents of the given collection to the receiver's file at the
 current position.  The argument must be a kind of Collection with byte
 format.  Returns a count of bytes added, or nil if an error occurs."

^ self nextPutAll: collection
%

category: 'Writing'
method: GsFile
nextPutAllUtf8: aStringOrCharacter

"Writes the contents of the given String, MultiByteString or Utf8 to the
 receiver's file using UTF8 encoding .
 Returns a count of bytes added, or nil if an error occurs."

| utf cls |
(cls := aStringOrCharacter class) == Character ifTrue:[
  aStringOrCharacter codePoint <= 127 ifTrue:[
    self nextPut: aStringOrCharacter.
    ^ 1
  ].
  utf := Unicode16 with: aStringOrCharacter.
  utf := utf encodeAsUTF8 .
] ifFalse:[
  cls == Unicode7 ifTrue:[ 
    ^ self write: aStringOrCharacter size from: aStringOrCharacter.
  ].
  utf := aStringOrCharacter encodeAsUTF8 .
].
^ self write: utf size from: utf .
%

category: 'Writing'
method: GsFile
nextPutAsUtf8: aString
  "Will be deprecated.   Use nextPutAllUtf8: "
  ^ self nextPutAllUtf8: aString
%

category: 'File Operations'
method: GsFile
open

"If the receiver is not open, open it using the existing mode.
 Returns the receiver, or nil if an error occurs.
 If an error occurs,  there is no error information associated with
 the instance; you must use    GsFile lastErrorString   to retrieve the error. "

^self _openUsingGzip: false
%

category: 'File Operations'
method: GsFile
open: aPathName mode: openMode

"Opens the receiver's file with the given mode.  If the file is already open,
 it is closed and reopened.

 The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+', 'rb', 'wb', 'ab', 'r+b', 'w+b',
 'a+b', 'rb+', 'wb+', 'ab+'.  The mode has the same meaning as it does for the
 C library function, fopen().

 Returns the receiver if successful, or nil if an error occurs.
 If an error occurs,  there is no error information associated with
 the instance; you must use   GsFile lastErrorString to retrieve the error. "

self isOpen ifTrue:[
  self close  .
  self isOpen ifTrue:[
    ^ self "a standard file that cannot be closed"
  ].
].
mode := openMode.
pathName := aPathName .
^self open
%

category: 'Accessing'
method: GsFile
pathName

"Returns the receiver's file path name."

^pathName
%

category: 'Reading'
method: GsFile
peek

"Returns the a Character representing the next byte in the receiver's file,
 without advancing the current pointer.
 Returns nil if an error occurs, or peek hits EOF."

| result |
[
  (result := self _peek: 1 ) == false
] whileTrue . "loop to handle EINTR"
result == true ifTrue:[
  | pos |  "handle client file"
  pos := self position.
  result := self next.
  self position: pos.
].
^ result
%

category: 'Reading'
method: GsFile
peek2

"Returns the next byte plus one in the receiver's file, without advancing the
 current pointer.  Returns nil if an error occurs, or peek hits EOF."
| result |
[
  (result := self _peek: 2 ) == false
] whileTrue . "loop to handle EINTR"
result == true ifTrue:[
  | pos |  "handle client file"
  pos := self position.
  result := self next; next.
  self position: pos.
].
^result
%

category: 'ANSI Compatibility'
method: GsFile
peekFor: expectedObject
"To be compatible with ANSI gettableStream (5.9.2.8):
Returns the result of sending #= to the first object in the receiver's future sequence
values with anObject as the argument. Returns false if the receiver has no future
sequence values."

	| foundObject flag |
	foundObject := self peek.
	(flag := expectedObject = foundObject) ifTrue: [
		self next.
	].
	^flag.
%

category: 'Positioning'
method: GsFile
position

"Returns the current position of the receiver's file,
 or nil if an error occurs.  Result is zero based."
 ^ (self _zeroArgPrim: 2) ifNil:[
   | isCl |
   ((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[
     ^ self _setNotOpenError
   ].
   ^ self userAction: #GsfPosition onClient: isCl
 ].
%

category: 'Positioning'
method: GsFile
position: offset

"Changes the receiver's position in its file to the given offset, which may be
 negative or zero.  Returns the new position, or nil if an error occurs."

^ self _seekTo: offset opcode: 0 .
%

category: 'Positioning'
method: GsFile
positionA
	"For compatibility with PositionableStream"

	^self position.
%

category: 'Positioning'
method: GsFile
positionA: offset
	"For compatibility with PositionableStream"

	^self position: offset.
%

category: 'Positioning'
method: GsFile
positionL
	"For compatibility with PositionableStream"

	^self position.
%

category: 'Positioning'
method: GsFile
positionL: offset
	"For compatibility with PositionableStream"

	^self position: offset.
%

category: 'Positioning'
method: GsFile
positionW
	"For compatibility with PositionableStream"

	^self position.
%

category: 'Positioning'
method: GsFile
positionW: offset
	"For compatibility with PositionableStream"

	^self position: offset.
%

category: 'Private'
method: GsFile
postCopy
	"Do cleanup on new copy. Ensure that it is closed."

super postCopy.
isClient := self isClient ifTrue:[ 1 ] ifFalse:[ 0 ]
%

category: 'Writing'
method: GsFile
print: anObject
       anObject printOn: self
%

category: 'Writing'
method: GsFile
printBytes: byteObj

"Prints the bytes from the given byte object in decimal notation with
 line breaks to keep output lines from being too long.

 Returns the receiver, or nil if an error occurs.

 See also the method next:byteStringsInto:."

(isClient _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ ^ self _setNotOpenError ].
^ self userAction: #GsfPrintBytes onClient: isClient with: byteObj
%

category: 'Reading'
method: GsFile
read: numberOfBytes into: byteObj

"Reads up to the given number of bytes into the given byte object (for
 example, a String) starting at index 1.
 Returns the number of bytes read, or nil if an error occurs,
 or 0 if EOF on the receiver.
 The destination object is grown as needed, but is not shrunk. "

^ self _read: numberOfBytes into: byteObj
%

category: 'Locking'
method: GsFile
readLockAtOffset: offset forBytes: bytes
"Requests a read lock on a portion of the receiver without blocking.  Returns
 true if successful, false if the lock was denied. Raises an IOError if an error
 occurred.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more
 information about file locks."

 ^ self readLockAtOffset: offset forBytes: bytes waitTime: 0
%

category: 'Locking'
method: GsFile
readLockAtOffset: offset forBytes: bytes waitTime: milliseconds
"Requests a read lock on a portion of the receiver and blocks for up to
 milliseconds to acquire the lock.  Returns true if successful, false if the lock
 was denied. Raises an IOError if an error occurred.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more
 information about file locks."

 ^ self _acquireLockKind: #readLock atOffset: offset forBytes: bytes waitTime: milliseconds
%

category: 'Locking'
method: GsFile
readLockContents
"Requests a read lock on the entire contents the receiver without blocking.
 Returns true if successful, false if the lock was denied. Raises an IOError if
 an error occurred.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more
 information about file locks."

 ^ self readLockAtOffset: 0 forBytes: 0
%

category: 'File Operations'
method: GsFile
reopen
  "Reopen the file if the disk file has been deleted or renamed (using mv) by another process.  
   If the underlying disk file no longer present, GsFile creates a new disk file with the same 
   name and path and returns true. reopen is done using freopen per man 3 fopen, using the existing 
   path and mode. If the disk file still exists, nothing is done and this method returns false. 
   If an error occurs, this method returns nil.

   Receiver must be a server file, with mode containing 'a' or 'w', and cannot be stdin, stdout, 
   nor stderr. Note that stdout, the Gem log, is automatically reopened on SIGHUP.

   This is provided to support log rotation. OS-level log rotation of application GsFiles should 
   rename the disk file and send SIGHUP to the Gem. The application should setup a handler for the 
   asynchronous LogRotateNotification signal (using addDefaultHandler:). This handler can then send 
   reopen."

  | status |
  status := self _zeroArgPrim: 4 .
  status ~~ 0 ifTrue:[
    status > 0 ifTrue:[ ^ nil ].
    ^ false .
  ].
  ^ true
%

category: 'ANSI Compatibility'
method: GsFile
reset
"To be compatible with ANSI sequencedStream (5.9.1.6):
Sets the receiver's future sequence values to be the current past sequence values appended
with the current future sequence values. Make the receiver's past sequence values be empty."

	self positionA: 0.
%

category: 'Positioning'
method: GsFile
rewind

"Repositions the receiver's file to the beginning.  Returns 0, or nil if an
 error occurs."

self position: 0
%

category: 'Positioning'
method: GsFile
seekFromBeginning: offset

"Moves the receiver's position in its file to the given offset from the
 beginning of the file, which may be positive or zero, but not negative.
 Returns the new position, or nil if an error occurs."

^ self _seekTo: offset opcode: 0
%

category: 'Positioning'
method: GsFile
seekFromCurrent: offset

"Changes the receiver's position in its file by the given offset, which
 may be negative or zero.  Returns the new position, or nil if an error occurs."

^ self _seekTo: offset opcode: 1
%

category: 'Positioning'
method: GsFile
seekFromEnd: offset

"Moves the receiver's position in its file to the given offset from the
 end of the file, which may be negative or zero, but not positive.
 Returns the new position, or nil if an error occurs.  Not supported
 for compressed files."

^ self _seekTo: offset opcode: 2
%

category: 'ANSI Compatibility'
method: GsFile
setToEnd
"To be compatible with ANSI sequencedStream (5.9.1.7):
All of the receiver's future sequence values are appended, in sequence, to the receiver's
past sequence values. The receiver then has no future sequence values."

	self positionA: self fileSize.
%

category: 'Reading'
method: GsFile
skip: count

"Changes the receiver's position in its file by the given offset, which may be
 zero, or negative.  Returns the new position, or nil if an error occurs."

| pos |
pos := self position.
^ self position: pos + count
%

category: 'ANSI Compatibility'
method: GsFile
skipTo: anObject
"To be compatible with ANSI gettableStream (5.9.2.10):
Each object in the receiver's future sequence values up to and including the first
occurrence of an object that is equivalent to anObject is removed from the future
sequence values and appended to the receiver's past sequence values. If an object that
is equivalent to anObject is not found in the receiver's future sequence values, all
of the objects in future sequence values are removed from future sequence values and
appended to past sequence values. If an object equivalent to anObject is not found
false is returned. Otherwise return true."

	self positionA to: self fileSize - 1 do: [:i |
		self next = anObject ifTrue: [^true].
	].
	^false.
%

category: 'Writing'
method: GsFile
space

"Append an space character to the receiver.
 Returns true if successful, nil if an error occurred."

^ self nextPut: 32 " $   codePoint "
%

category: 'File Operations'
method: GsFile
sync

"Flushes all written bytes to the file's storage device.  Returns the receiver, or nil if an
 error occurs. On Unix, calls fflush() followed by fsync() . "
| isCl |
((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[
   ^ self _setNotOpenError
].
^ self userAction: #GsfSync onClient: isCl
%

category: 'Writing'
method: GsFile
tab

"Append a tab to the receiver.
 Returns true if successful, nil if an error occurred."

^ self nextPut: 9  "ASCII tab"
%

category: 'Locking'
method: GsFile
unlockAtOffset: offset forBytes: bytes

"Releases a previously acquired lock on a portion the file referenced by the
 receiver.  offset indicates the position within the receiver in bytes and bytes
 indicates the number of bytes to unlock (0 means unlock the file from offset to
 the end of the file).

 Please refer the comments in the following method for more details on file
 locks:
   GSFile _acquireLockKind: atOffset: forBytes: waitTime:

 Returns true if the lock was released or if the region was not locked by the process.
 Raises an IOError if an error occurs."

| result |
result := self userAction: #GsfUnlock onClient: self isClient with: offset with: bytes .
^ result class == Boolean
     ifTrue:[ result ]
    ifFalse:[ IOError signal: result ]
%

category: 'Locking'
method: GsFile
unlockContents
"Releases a previously acquired lock on contents of the receiver .

 Returns true if the lock was released successfully or if the contents were not
 locked by the process.  Raises an IOError if an error occurs.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more
 information about file locks."

^ self unlockAtOffset: 0 forBytes: 0
%

category: 'ANSI Compatibility'
method: GsFile
upTo: anObject
"To be compatible with ANSI gettableStream (5.9.2.10):
Each object in the receiver's future sequence values up to and including the first
occurrence of an object that is equivalent to anObject is removed from the future
sequence values and appended to the receiver's past sequence values. A collection,
containing, in order, all of the transferred objects except the object (if any) that
is equivalent to anObject is returned. If the receiver's future sequence values is
initially empty, an empty collection is returned."

	| result |
	result := String new.
	self positionA to: self fileSize - 1 do: [:i |
		| char |
		(char := self next) = anObject ifTrue: [^result].
		result add: char.
	].
	^result.
%

category: 'Private'
method: GsFile
userAction: actionName onClient: onClient

"Executes a GsFile user action, passing it the following arguments:
    self
    any arguments to the primitive (see other uses of primitive 396)
 Maximum of 6 arguments (6 with: keywords) for this primitive.
 actionName must be a Symbol "

<primitive: 396>

^ self _primitiveFailed: #userAction:onClient: args: { actionName . onClient }
%

category: 'Private'
method: GsFile
userAction: actionName onClient: onClient with: arg1

"See GsFile | userAction:onClient: for documentation."

<primitive: 396>

^ self _primitiveFailed: #userAction:onClient:with:
       args: { actionName . onClient . arg1 }
%

category: 'Private'
method: GsFile
userAction: actionName onClient: onClient with: arg1 with: arg2

"See GsFile | userAction:onClient: for documentation."

<primitive: 396>

^ self _primitiveFailed: #userAction:onClient:with:with:
       args: { actionName . onClient . arg1 . arg2 }
%

category: 'Private'
method: GsFile
userAction: actionName onClient: onClient with: arg1 with: arg2 with: arg3

"See GsFile | userAction:onClient: for documentation."

<primitive: 396>

^ self _primitiveFailed: #userAction:onClient:with:with:with:
       args: { actionName . onClient . arg1 . arg2 . arg3 }
%

category: 'Private'
method: GsFile
userAction: actionName onClient: onClient with: arg1 with: arg2 with: arg3 with: arg4

"See GsFile | userAction:onClient: for documentation."

<primitive: 396>

^ self _primitiveFailed: #userAction:onClient:with:with:with:with:
       args: { actionName . onClient . arg1 . arg2 . arg3 . arg4 }
%

category: 'Writing'
method: GsFile
write: amount from: byteObj

"Write the given number of bytes from the given byte object.  Returns the
 number of bytes written, or nil if an error occurs."

<primitive: 875>  "primitive succeeds only if file is open on server"
| isCl |
((isCl:=isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[
  ^ self _setNotOpenError
].
^ self userAction: #GsfWrite onClient: isCl
       with: byteObj with: 1 with: amount
%

category: 'Writing'
method: GsFile
write: byteObj itemCount: itemCount ofSize: bytesPerItem

"Writes itemCount items of size bytesPerItem from the given byte object to
 the receiver's file.  bytesPerItem must be between 1 and 4096 inclusive.
 Returns a count of bytes written, or nil if an error occurs."

^ self write: itemCount * bytesPerItem from: byteObj
%

category: 'Locking'
method: GsFile
writeLockAtOffset: offset forBytes: bytes
"Requests a write lock on a portion of the receiver without blocking.
 Returns true if successful, false if the lock was denied. Raises an IOError if
 an error occurred.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more
 information about file locks."

 ^ self writeLockAtOffset: offset forBytes: bytes waitTime: 0
%

category: 'Locking'
method: GsFile
writeLockAtOffset: offset forBytes: bytes waitTime: milliseconds
"Requests a write lock on a portion of the receiver and blocks for up to
 milliseconds to acquire the lock.  Returns true if successful, false if the
 lock was denied. Raises an IOError if an error occurred.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more
 information about file locks."

 ^ self _acquireLockKind: #writeLock atOffset: offset forBytes: bytes waitTime: milliseconds
%

category: 'Locking'
method: GsFile
writeLockContents
"Requests a write lock on the entire contents the receiver without blocking.
 Returns true if successful, false if the lock was denied. Raises an IOError
 if an error occurred.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more
 information about file locks."

 ^ self writeLockAtOffset: 0 forBytes: 0
%

category: 'Locking'
method: GsFile
_acquireLockKind: kind atOffset: offset forBytes: bytes waitTime: milliseconds

"Requests a lock on a portion the file referenced by the receiver.  kind
 is a symbol that indicates the lock kind and must be either #readLock or
 #writeLock.  A read lock is a shared lock such that a given byte in a file
 may have multiple read locks on it at the same time.  A write lock is an
 exclusive lock which precludes all other read locks and write locks.

 offset indicates the position within the receiver in bytes.
 offset is zero-based; the first by in the file is at offset 0.  bytes
 indicates the number of bytes to lock (0 means lock the file from offset to the
 end of the file).

 waitTime is the number of milliseconds to wait when requesting a lock and
 the lock is unavailable.  0 means return immediately (do not wait).  -1 means
 wait forever.  Specifying a waitTime greater than zero causes the session to
 rapidly poll for the lock to be available.

 For processes running on UNIX based systems, file locks are advisory, which
 means locks are not enforced by the operating system and applications may
 freely ignore them.

 For processes running on Microsoft Windows-based systems, file locks are
 mandatory, which means locks are enforced by the operating system and
 applications cannot ignore them.

 *** Warning ***
 On UNIX-based systems only, closing any handle referencing a given physical
 file forces *all* locks on that file held by the process to be released.  For
 this reason, locks should not be used on any file which the process opens more
 than once.

 This method is highly dependent upon the underlying operating system
 implementation of advisory file locks.  On UNIX-based systems, the fcntl(F_SETLK)
 call is used.  On Microsoft Windows systems, the LockFileEx() call is used.
 Consult the documentation for your host system to learn about the behavior of
 file locking for your system.

 Some systems do not support read locks if the file is opened in write-only
 mode or write locks if the file is open in read-only mode.  Attempting to lock
 files opened in these manners may raise an error indicating a bad file
 descriptor.

 Returns true if the lock was acquired, false if the lock was denied.
 Raises an IOError if an error occurs."

| result |
result := self userAction: #GsfLock onClient: self isClient with: kind with: offset with: bytes with: milliseconds .
^ result class == Boolean
     ifTrue:[ result ]
    ifFalse:[ IOError signal: result ]
%

category: 'Reading'
method: GsFile
_contents: resultObj

"Read the contents of the receiver from the current
 position to the end of the file into resultObj.
 Returns nil if an error occurs, otherwise returns resultObj."

| mySize myPosition |

mySize := self fileSize .
(mySize == nil or:[ mySize == 0]) ifTrue:[
  "a file not reporting size like stdin, or of size zero like /dev/zero"
  | buf |
  buf := resultObj class new .
  [ true ] whileTrue:[ | status |
    status := self _next:1024 into: buf .
    status ifNil:[ ^ resultObj ].
    resultObj addAll: buf
  ]
].
myPosition := self position .
(mySize + myPosition) == 0 ifTrue:[ ^ resultObj ]. "a zero length file"
myPosition == nil ifTrue:[ ^ nil ] .
(self next: ( mySize - myPosition ) into: resultObj ) ifNil:[ ^ nil ].
^ resultObj
%

category: 'Private'
method: GsFile
_contentsUncompressed: resultObj

| buf |
buf := resultObj class new .
[ true ] whileTrue:[ | count |
  count := self _read: 4096 into: buf .
  count == 0 ifTrue:[ ^ resultObj ].
  count ifNil:[ ^ nil ].
  count < 4096 ifTrue:[
    buf size: count  "truncate to match amount read"
  ].
  resultObj addAll: buf
].
%

category: 'Fileout'
method: GsFile
_fileOutAll: aString

  self nextPutAllUtf8: aString 
%

category: 'Private'
method: GsFile
_isBinary

"Returns true if receiver is a binary file. Otherwise returns false."
| m |
(m := mode) ifNil: [ ^ false].
^ m includesValue: $b
%

category: 'Private'
method: GsFile
_isReadable

"Returns true if receiver is open for read, otherwise returns false."
| m |
(m := mode) ifNil:[ ^ false].
^ (m includesValue: $r) or:[ m includesValue: $+ ]
%

category: 'Private'
method: GsFile
_newEmptyClientCData

"initialized cData for a client file, if cData not already
 initialized for a server file."

<primitive: 675>
self _primitiveFailed: #_newEmptyClientCData
%

category: 'Private'
method: GsFile
_newStdFile: stdId isClient: clientBool

  "Initialize the receiver to represent a standard file."

  clientBool ifTrue:[
    isClient := true .
    fileDescriptor := -1 .
  ] ifFalse:[
    isClient := false .
    fileDescriptor := stdId .
  ].
  stdId == 0 ifTrue: [mode := 'r'. pathName := 'stdin']
          ifFalse:[mode := 'w'.
                   stdId == 1 ifTrue: [pathName := 'stdout']
                             ifFalse:[pathName := 'stderr'].
                  ].

%

category: 'Reading'
method: GsFile
_next: numberOfBytes into: resultObj

"Returns a String containing the next numberOfBytes Characters from the
 receiver's file, or nil if an error occurs."

| count |
count := self _read: numberOfBytes into: resultObj .
(count == 0 or:[ count == nil]) ifTrue:[ ^ nil ].
^ resultObj
%

category: 'Reading'
method: GsFile
_nextLineTo: eolValue prompt: promptString

"Reads the next line from the receiver's file into a new String .

 eolValue is either a SmallInteger 0..255, or a String with
 size >= 1 and <= 128 .

 The String will be terminated with the codePoint per eolValue
 unless the end of file is reached and there is no line terminator.
 Returns the new String, or nil if an error occurs.
 Result is nil if file is at EOF.

 If promptString is non-nil, and receiver is stdin, and
   isatty(fileno(stdin)) == 1 , then the topaz line editor will be used
 to read the input
 Returns -2 if EINTR occurred."

<primitive: 723>  "primitive succeeds for server file"
| isCl |
((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[
  ^ self _setNotOpenError
].
^ self userAction: #GsfGetLine onClient: isCl with: nil
                with: 1 with: eolValue with: promptString.
%

category: 'Private'
method: GsFile
_nextPutAllBytes: collection
  "used by PassiveObject"
| str |
(str := String new) addAllBytes: collection . "str is big-endian"
^ self nextPutAll: str .
%

category: 'Private'
method: GsFile
_open: aPath mode: aMode  onClient:  clientBool

  "Initialize the receiver as specified by the arguments."

  "Does not actually open the underlying file."

  clientBool ifTrue:[ isClient := 3 ] ifFalse:[ isClient := 2 ].
  pathName := aPath .
  mode := aMode .
%

category: 'Compressed File Operations'
method: GsFile
_openUsingGzip: aBoolean

"If the receiver is not open, open it using the existing mode.
 If aBoolean is true, the gzip function gzopen() is used to open
 the file.  gzopen() can be used to open an existing file which was
 not compressed with gzip.  In that case, the file will be read
 without decompression.

 Returns the receiver, or nil if an error occurs.
 If an error occurs,  there is no error information associated with
 the instance; you must use:
     GsFile lastErrorString
 to retrieve the error. "


| res path |
(self isOpen) ifTrue:[ ^self ].
pathName ifNil:[ self class _setLastError:'path for open is nil' onClient: isClient. ^ nil ].
mode ifNil:[ self class _setLastError:'mode for open is nil' onClient: isClient. ^ nil ].
mode stringCharSize == 1 ifFalse:[
   ArgumentError signal: 'mode must be a String or Unicode7'. ^nil.
].
path := GsFile _utfPath: pathName forClient: isClient .
res := self userAction: #GsfOpen onClient: isClient
	         with: path with: mode with: aBoolean.
res ifNotNil:[
  self _setIsOpen .
  res == 1 ifTrue:[ self _newEmptyClientCData ].
  self isClient ifTrue:[
    fileDescriptor := -1 .
  ].
  fileDescriptor := self userAction: #GsfGetFileDesc onClient: false .
  " maglev only ifFalse:[
    IO _rememberFileDescriptor: fileDescriptor obj: self .
  ]."
  ^ self .
].
^ res
%

category: 'Private'
method: GsFile
_peek: numBytes

"Returns nil for error on server file, false if EINTR occurred,
 true if receiver is a client file.
 Primitive fails for invalid numBytes argument."
<primitive: 1013>
(numBytes == 1 or:[ numBytes == 2]) ifFalse:[
   ArgumentError signal:'invalid number of bytes for peek'].
^ self _primitiveFailed: #_peek: args: { numBytes }
%

category: 'Reading'
method: GsFile
_read: numberOfBytes into: byteObj

"Reads up to the given number of bytes into the given byte object
 starting at index 1.   byteObj must be a String or ByteArray .
 Returns the number of bytes read, or nil if an error occurs,
 or 0 if EOF on the receiver.
 The destination object is grown as needed, but is not shrunk"

<primitive: 874>
"primitive succeeds only if file is open on server, and no EINTR occurred"
^ self _read: numberOfBytes ofSize: 1 into: byteObj .  "client file or EINTR"
%

category: 'Reading'
method: GsFile
_read: numberOfItems ofSize: bytesPerItem into: byteObj

"Reads bytes for the next numberOfItems of the given bytesPerItem into
 the given collection object. The object's size is truncated to the
 amount of data actually read.  bytesPerItem must between 1 and 4096 inclusive.

 Returns a count of bytes read, 0 if hit EOF, or nil if an error occurs."
| count isCl |
( (isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[
   ^ self _setNotOpenError
].
[ count := self userAction: #GsfRead onClient: isCl
       with: byteObj with: bytesPerItem with: numberOfItems .
  count == -2
] whileTrue . "loop to handle EINTR and soft break"
^ count
%

category: 'Private'
method: GsFile
_seekTo: offset opcode: seekKind

"seekKind  function
   0        seek to  start of file + offset
   1        seek  offset from current position
   2        seek to  end of file - offset

 Returns the new position, or nil if an error occurs."
<primitive: 1010>
"primitive failed, handle client file"
| isCl |
((isCl:=isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[
  ^ self _setNotOpenError
].
^ self userAction: #GsfSeek onClient: isCl with: offset with: seekKind
%

category: 'Private'
method: GsFile
_setIsOpen

  "for a transient instance, change state to 'open' "
  self isCommitted ifFalse:[
    isClient := (isClient bitAnd: 1) == 1  "convert 0/1, 2/3 to false/true"
  ]
%

category: 'Private'
method: GsFile
_setNotOpened

  "for a transient instance,  change state to 'closed' "
  self isCommitted ifFalse:[
    isClient _isSmallInteger ifFalse:[
       isClient ifTrue:[ isClient:= 1 ] ifFalse:[ isClient:=0 ].
    ].
  ].
%

category: 'Private'
method: GsFile
_setNotOpenError

  "set state of a transient instance to 'closed, access error' "

  self isCommitted ifFalse:[
    isClient _isSmallInteger ifFalse:[
      self error:'invalid state' .
      self _uncontinuableError
    ].
    isClient < 2 ifTrue:[ isClient := isClient + 2 ].
  ].
  ^ nil
%

category: 'Private'
method: GsFile
_zeroArgPrim: opcode

"returns nil for a client file or file error.

  opcode 1   isCompressed
  opcode 2   position
  opcode 3   filePointer
  opcode 4   reopen
"
<primitive: 1012>
self _primitiveFailed: #_zeroArgPrim: args: { opcode }
%

category: 'Comparing'
method: GsFile
~= aFile

"Returns false if the receiver and aFile represent the same file system file.
 Returns true otherwise."

^ (self = aFile) not
%

! Class extensions for 'GsMethodDictionary'

!		Instance methods for 'GsMethodDictionary'

removeallmethods GsMethodDictionary
removeallclassmethods GsMethodDictionary

category: 'Enumerating'
method: GsMethodDictionary
accompaniedBy: anObj keysAndValuesDo: aBlock

"Evaluates aBlock with each of the receiver's key/value pairs as the
 2nd and 3rd arguments.
 aBlock must be a 3 argument block, with arguments anObj, key value ."

| aKey tSize |
"process hash slots"
tSize := tableSize * 2 .
1 to: tSize by: 2 do: [ :offset |
  nil == (aKey := self _basicAt: offset) ifFalse:[
    aBlock value: anObj value: aKey value: (self _basicAt: offset + 1) .
    ].
  ].
"process collision chains"
tSize + 1 to: self _basicSize by: 3 do:[:offset |
  (aKey := self _basicAt: offset) == nil ifFalse:[
    aBlock value: anObj value: aKey value: (self _basicAt: offset + 1) .
    ].
  ].
%

category: 'Enumerating'
method: GsMethodDictionary
associationsDo: aBlock

"Evaluates aBlock with each of the receiver's key/value pairs as the argument by
 creating a SymbolAssociation for each key/value pair.  The argument aBlock must
 be a one-argument block.  Returns the receiver."

self keysAndValuesDo: [:aKey :aValue |
  aBlock value: (SymbolAssociation newWithKey: aKey value: aValue)
  ].
%

category: 'Accessing'
method: GsMethodDictionary
at: aKey ifAbsent: aBlock

"Returns the value whose key is identical to aKey.  If no such key/value pair
 exists, returns the result of evaluating the zero-argument block aBlock."

| offset hashKey |

nil == aKey ifTrue:[ ^self _reportKeyNotFound: aKey with: aBlock ].

(hashKey := self _basicAt:
       ( offset := (self hashFunction: aKey) * 2 - 1 )
  ) == aKey
ifTrue:[
  ^ self _basicAt: offset + 1 . "the value"
  ]
ifFalse:[
  nil == hashKey ifTrue:[
    "search collision chain"
    offset := self _basicAt: (offset + 1). "get one based offset to start of chain"
    [ offset == nil ] whileFalse:[
      (self _basicAt: offset )  == aKey ifTrue:[
        ^ self _basicAt: offset + 1
        ].
      offset := self _basicAt: offset + 2 .
      ] .
    ].
  ].
^ self _reportKeyNotFound: aKey with: aBlock .
%

category: 'Accessing'
method: GsMethodDictionary
at: aKey otherwise: aValue

"Returns the value whose key is identical to aKey.  If no such key/value pair
 exists, returns the given alternate value."

<primitive: 857>
keyConstraint ifNotNil:[ aKey _validateClass: keyConstraint ].
valueConstraint ifNotNil:[ aValue _validateClass: valueConstraint ].
self _primitiveFailed: #at:otherwise: args: { aKey . aValue }
%

category: 'Updating'
method: GsMethodDictionary
at: aKey put: aValue

"Stores the aKey/aValue pair in the hash dictionary.  Rebuilds the hash table
 if the addition caused the number of collisions to exceed the limit allowed.

 If aKey is not compatible with the key constraint of the receiver, or aValue is
 not compatible with the value constraint of the receiver, an error is
 generated."

self _validatePrivilege ifTrue:[
  self _atKey: aKey put: aValue .
].
^ aValue
%

category: 'Private'
method: GsMethodDictionary
atHash: hashIndex putKey: aValue

"Disallowed."

self shouldNotImplement: #atHash:putKey:
%

category: 'Private'
method: GsMethodDictionary
atHash: hashIndex putKey: aKey value: aValue

"Disallowed."

self shouldNotImplement: #atHash:putKey:value:
%

category: 'Private'
method: GsMethodDictionary
atHash: hashIndex putValue: aValue

"Disallowed."

self shouldNotImplement: #atHash:putValue:
%

category: 'Clustering'
method: GsMethodDictionary
clusterDepthFirst

"This method clusters the receiver and its values in depth-first order.  The
 keys are not clustered because they are Symbols.

 Has no effect and returns true if the receiver was previously clustered in the
 current transaction."

self cluster
  ifTrue:[ ^ true ]
  ifFalse: [
      "none of the named instance variables should be clustered"

      self valuesDo:[:aMethod | aMethod clusterDepthFirst ].
      ^ false
    ]
%

category: 'Private'
method: GsMethodDictionary
collisionBucketClass

"Returns the class of object to create when keys collide.
 GsMethodDictionary does not use collision buckets."

^ nil
%

category: 'Hashing'
method: GsMethodDictionary
hashFunction: aKey

"The hash function performs an operation on the value of the key aKey and
 returns some Integer between 1 and tableSize, inclusive."

^(aKey basicIdentityHash \\ self tableSize) + 1
%

category: 'Initializing'
method: GsMethodDictionary
initialize: newSize

"Initializes the instance variables of the receiver to be an empty
 IdentityKeyValueDictionary of the specified size."

self _validatePrivilege ifTrue:[
  super initialize: newSize .
]
%

category: 'CodeModification Override'
method: GsMethodDictionary
instVarAt: anIndex put: aValue

self _validatePrivilege ifTrue:[
  ^ super instVarAt: anIndex put: aValue
]
%

category: 'Accessing'
method: GsMethodDictionary
keyAtValue: anObject ifAbsent: aBlock

"Returns the key of the first value identical to anObject.  If no
 match is found, this method evaluates the block aBlock and returns its
 result."

self keysAndValuesDo:[ :aKey :aValue |
  anObject == aValue ifTrue:[ ^ aKey ]
  ].
^ aBlock value
%

category: 'Accessing'
method: GsMethodDictionary
keyConstraint

"Returns the key constraint of the receiver."

^ keyConstraint
%

category: 'Updating'
method: GsMethodDictionary
keyConstraint: aClass

"Sets the key constraint of the receiver to aClass.  Generates an error if
 the receiver is not empty."

self _validatePrivilege ifTrue:[
  numElements ~~ 0 ifTrue:[ ^ self _error: #rtErrCannotChgConstraint ].
  keyConstraint := aClass
]
%

category: 'Enumerating'
method: GsMethodDictionary
keysAndValuesDo: aBlock

"Evaluates aBlock with each of the receiver's key/value pairs as the
 arguments.  The argument aBlock must be a two-argument block.  The
 first argument is the key and the second argument is the value of
 each key/value pair.  Returns the receiver."

| aKey tSize |

"process hash slots"
tSize := tableSize * 2 .
1 to: tSize by: 2 do: [ :offset |
  (aKey := self _basicAt: offset) ifNotNil:[
    aBlock value: aKey value: (self _basicAt: offset + 1) .
    ].
  ].
"process collision chains"
tSize + 1 to: self _basicSize by: 3 do:[:offset |
  (aKey := self _basicAt: offset) ifNotNil:[
    aBlock value: aKey value: (self _basicAt: offset + 1) .
    ].
  ].
%

category: 'Enumerating'
method: GsMethodDictionary
keysDo: aBlock

"Evaluates aBlock with each of the receiver's keys as the
 argument.  The argument aBlock must be a one-argument block.
 Returns the receiver."

| aKey tSize |

"process hash slots"
tSize := tableSize * 2 .
1 to: tSize by: 2 do: [ :offset |
  (aKey := self _basicAt: offset) ifNotNil:[
    aBlock value: aKey .
    ].
  ].
"process collision chains"
tSize + 1 to: self _basicSize by: 3 do:[:offset |
  (aKey := self _basicAt: offset) ifNotNil:[
    aBlock value: aKey
    ].
  ].
%

category: 'Copying'
method: GsMethodDictionary
postCopy

"Because there are no collision buckets to copy, we can ignore inherited behavior."

^self
%

category: 'Formatting'
method: GsMethodDictionary
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

"copy the implementation from Object."

aStream nextPutAll: self asString
%

category: 'Hashing'
method: GsMethodDictionary
rebuildTable: newSize

"Rebuilds the method dictionary by populating a larger method dictionary
 first and doing a (primitive) become:"

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  | newGsMethodDict |

  tableSize = newSize ifTrue:[
    ^ self "no change in table size"
  ].
  collisionLimit == 536870911 ifTrue:[
    ^ self              "avoid recursive rebuild"
  ].

  newGsMethodDict := self class new: (newSize * 2).
  newGsMethodDict valueConstraint: valueConstraint.
  newGsMethodDict keyConstraint: keyConstraint.

  self keysAndValuesDo: [ :aKey :aValue |
    newGsMethodDict at: aKey put: aValue.
    ].
  "receiver is not expected to have a dependencyList, so an unprotected become: should be safe"
  newGsMethodDict _becomeDictionary: self.
] ensure:[
  prot _leaveProtectedMode
]
%

category: 'Removing'
method: GsMethodDictionary
removeAll

"Remove all key/value pairs from the receiver."

| tSize |
self _validatePrivilege ifTrue:[
  self _basicSize: 0 . "dereference all keys and values"
  tableSize :=  (tSize := 29) .
  self _basicSize: tSize + tSize  .  "reinitialize hash table to all nils"
  numElements := 0 .
  numCollisions := 0
]
%

category: 'Removing'
method: GsMethodDictionary
removeKey: aKey ifAbsent: aBlock

"Removes the key/value pair whose key is identical to aKey.  If no such
 key/value pair exists, returns the result of evaluating the zero-argument
 block aBlock."

 | oldVal |

 self _validatePrivilege ifTrue:[
   nil == aKey ifTrue:[ ^self _error: #rtErrNilKey ].
   oldVal := self _atKey: aKey put: nil .
   oldVal == nil ifTrue:[
     ^ self _reportKeyNotFound: aKey with: aBlock
   ].
   ^ oldVal
 ]
%

category: 'Removing'
method: GsMethodDictionary
removeKey: aKey otherwise: notFoundValue

"Removes the key/value pair whose key is identical to aKey.  If no such
 key/value pair exists, returns notFoundValue . "

 | oldVal |

 self _validatePrivilege ifTrue:[
   nil == aKey ifTrue:[ ^self _error: #rtErrNilKey ].
   oldVal := self _atKey: aKey put: nil .
   oldVal == nil ifTrue:[
     ^ notFoundValue
   ].
   ^ oldVal
 ]
%

category: 'CodeModification Override'
method: GsMethodDictionary
squeakBasicAt: anIndex put: aValue

  ^ self _basicAt: anIndex put: aValue
%

category: 'Statistics'
method: GsMethodDictionary
statistics

"A GsMethodDictionary has no collision buckets, so the statistics defined
 for KeyValueDictionary have no meaning."

^ nil
%

category: 'Accessing'
method: GsMethodDictionary
valueConstraint

"Returns the value constraint of the receiver."

^ valueConstraint
%

category: 'Updating'
method: GsMethodDictionary
valueConstraint: aClass

"Sets the value constraint of the receiver to aClass.  Generates an error if
 the receiver is not empty."

self _validatePrivilege ifTrue:[
  numElements ~~ 0 ifTrue:[ ^ self _error: #rtErrCannotChgConstraint ].
  valueConstraint := aClass
]
%

category: 'Private'
method: GsMethodDictionary
_atKey: aKey put: aValue

"aValue == nil means remove KV pair , and result is
 the value removed, or nil if not found.

 aValue ~~ nil means add KV pair, and return aValue.
 aValue must not be nil."

<primitive: 721>
self _primitiveFailed: #_atKey:put: args: { aKey . aValue }
%

category: 'CodeModification Override'
method: GsMethodDictionary
_basicAt: anIndex put: aValue

self _validatePrivilege ifTrue:[
  ^ super _basicAt: anIndex put: aValue
]
%

category: 'CodeModification Override'
method: GsMethodDictionary
_basicSize: anInteger

self _validatePrivilege ifTrue:[
  ^ super _basicSize: anInteger
]
%

category: 'Private'
method: GsMethodDictionary
_collisionBucketsDo: aBlock
  "a GsMethodDictionary has no collision buckets"
  ^ self
%

category: 'Private'
method: GsMethodDictionary
_initializeWithoutClear: newSize

"Private. Initializes the instance variables of the receiver to be an empty
 KeyValueDictionary of the specified size. Does not clear the contents
 of the receiver - assumes they are all nil."

tableSize := newSize.
numElements := 0.
numCollisions := 0.
collisionLimit := newSize // 4 .
^self
%

category: 'Testing'
method: GsMethodDictionary
_isLarge

"Returns true if the object is implemented as a tree of private smaller objects.
 Note that GsMethodDictionary does not use collision buckets.
"
^ ((self _status: false) bitShift: -5) ~~ 0
%

category: 'Private'
method: GsMethodDictionary
_nodesObjectSecurityPolicy: anObjectSecurityPolicy
  "a GsMethodDictionary has no collision buckets"
  ^ self
%

category: 'CodeModification Override'
method: GsMethodDictionary
_primitiveAt: anIndex put: aValue

self _validatePrivilege ifTrue:[
  ^ super _primitiveAt: anIndex put: aValue
]
%

category: 'Private'
method: GsMethodDictionary
_resetParentRef

"Private. After a become:, the parent refs of the collisionBuckets must
 be reset to point to the correct parent."

"GsMethodDictionarys don't use collision buckets"

^self

%

category: 'Canonical Symbol Support'
method: GsMethodDictionary
_selectiveAbort

"Performs an abort operation on the receiver."

"Receiver is just an ordinary small or large object as far as selective abort is
 concerned, so execute the implementation in Object.  Must avoid the
 reimplementation in IdentityKeyValueDictionary, so can't use super."

 <primitive: 274>
 self _primitiveFailed: #_selectiveAbort .
 self _uncontinuableError
%

category: 'CodeModification Override'
method: GsMethodDictionary
_unsafeAt: anIndex put: aValue

self _validatePrivilege ifTrue:[
  ^ super _unsafeAt: anIndex put: aValue
]
%

category: 'CodeModification Override'
method: GsMethodDictionary
_validatePrivilege

( self isMemberOf: GsMethodDictionary ) ifTrue: [
  ^   System myUserProfile _validateCodeModificationPrivilege
].
^ true
%

! Class extensions for 'GsNativeCode'

!		Class methods for 'GsNativeCode'

removeallmethods GsNativeCode
removeallclassmethods GsNativeCode

category: 'Disallowed'
classmethod: GsNativeCode
new

"Disallowed.  You cannot create new instances of GsNativeCode."

self shouldNotImplement: #new
%

category: 'Disallowed'
classmethod: GsNativeCode
new: aSize

"Disallowed.  You cannot create new instances of GsNativeCode."

self shouldNotImplement: #new:
%

!		Instance methods for 'GsNativeCode'

category: 'Disallowed'
method: GsNativeCode
at: anOffset put: aVal

"Disallowed."
self shouldNotImplement:#at:put:
%

category: 'Disallowed'
method: GsNativeCode
basicAt: anOffset put: aValue

"Disallowed."
self shouldNotImplement:#basicAt:put:
%

category: 'Disallowed'
method: GsNativeCode
copy
"Disallowed."
self shouldNotImplement:#copy
%

category: 'Disallowed'
method: GsNativeCode
objectSecurityPolicy: anObjectSecurityPolicy

"Disallowed."
self shouldNotImplement:#objectSecurityPolicy:
%

category: 'Disallowed'
method: GsNativeCode
passivate

"Disallowed."
self shouldNotImplement:#passivate
%

category: 'Disallowed'
method: GsNativeCode
size: aSize

"Disallowed."
self shouldNotImplement:#size:
%

category: 'Disallowed'
method: GsNativeCode
squeakBasicAt: anOffset put: aValue

"Disallowed."
self shouldNotImplement:#squeakBasicAt:put:
%

category: 'Disallowed'
method: GsNativeCode
_at: anOffset put: aVal

"Disallowed."
self shouldNotImplement:#_at:put:
%

category: 'Disallowed'
method: GsNativeCode
_basicAt: anOffset put: aVal

"Disallowed."
self shouldNotImplement:#_basicAt:put:
%

category: 'Disallowed'
method: GsNativeCode
_basicSize: aSize

"Disallowed."
self shouldNotImplement:#_basicSize:
%

category: 'Private'
method: GsNativeCode
_myNmethod

<primitive: 677>
^ self _primitiveFailed: #_myNmethod
%

category: 'Disallowed'
method: GsNativeCode
_objectSecurityPolicy: anObjectSecurityPolicy

"Disallowed."
self shouldNotImplement:#objectSecurityPolicy:
%

category: 'Disallowed'
method: GsNativeCode
_primitiveAt: anOffset put: aVal

"Disallowed."
self shouldNotImplement:#_primitiveAt:put:
%

! Class extensions for 'GsNMethod'

!		Class methods for 'GsNMethod'

removeallmethods GsNMethod
removeallclassmethods GsNMethod

category: 'Debugging Support'
classmethod: GsNMethod
clearAllBreaks

"Clear all method breakpoints that have been set in any methods."

self _setBreakAtIp: -1 operation: 2 frame: nil process: nil breakpointLevel: 0
%

category: 'Debugging Support'
classmethod: GsNMethod
clearBreakInClass: aClass selector: aSelector stepPoint: aStepPoint

"Clear the breakpoint at aStepPoint in method aSelector of class aClass."

(aClass compiledMethodAt: aSelector) clearBreakAtStepPoint: aStepPoint
%

category: 'Class organizer support'
classmethod: GsNMethod
configurableOptimizedSelectors
  "The class variable OptimizedSelectors by default contains 
    #( isNil notNil yourself )
  This array is read by the compiler initialzation that is part of session
  initialization.  Sends of these selectors are optimized by the compiler to a a bytecode.
  Only SystemUser may compile implementations of them, and those implementations
  will only be used by #perform .
  Any elements other than   isNil notNil yourself   are ignored by the compiler.

  OptimizedSelectors is by default invariant.
  If you want to remove any of the elemements to allow reimplementation in an
  application, you will have to (as SystemUser) remove and reinstall the class variable 
  and execute $GEMSTONE/bin/upgradeImage and recompile all application methods ."

  ^ OptimizedSelectors
%

category: 'Converting ArrayBuilders'
classmethod: GsNMethod
convertArrayBuildersIn: directoryName list: listName
  "given a directoryName and a file named 'listName' within that
   directory, assuming listName has one fileName per line,
   convert each file in the list.  within the list,
   leading/trailing whitespace on each line is ignored,
   and lines starting with # are ignored."

| dPath listF nC nNotC fName line fPath |
dPath := directoryName copy .
dPath last == $/ ifFalse:[ dPath add: $/ ].
listF := GsFile openRead:  dPath , listName  .
listF ifNil:[ ArgumentError signal:'file open failed for: ', (dPath , listName)].
nC := 0 . nNotC := 0 .
[ true ] whileTrue:[
  (line := listF nextLine) ifNil:[
    listF close .
    ^ nC asString, ' changed ' , nNotC asString , ' not changed '
  ].
  fName := line trimWhiteSpace .
  (fName at: 1) == $# ifFalse:[
    fPath := dPath , fName .
    [
      (self convertArrayBuildersInTopazScript: fPath)
         ifTrue:[ nC := nC + 1 . GsFile gciLogServer:'  changed ', fName ]
        ifFalse:[ nNotC := nNotC + 1 ].
    ] onException: Error do:[:ex |
      GsFile gciLogServer:'Error during ' , fName , ' ; ', ex description .
      ex pass .
    ]
  ].
].
%

category: 'Converting ArrayBuilders'
classmethod: GsNMethod
convertArrayBuildersInString: aString
  "Convert any ArrayBuilder productions in aString to
   CurlyArrayBuilder per $GEMSTONE/doc/bnf.txt .
   Returns aString if no ArrayBuilder found,
   otherwise returns a modified copy of aString."

  ^ self _convertArrayBuilders: aString stream: (ReadStreamPortable on: aString)
%

category: 'Converting ArrayBuilders'
classmethod: GsNMethod
convertArrayBuildersInTopazScript: aFileName
  "Convert any ArrayBuilder productions in source String within
   specified file to CurlyArrayBuilder per $GEMSTONE/doc/bnf.txt.
   The file is overwritten in place. Return true if file changed,
   false otherwise."
  | f origContents result cmd lineNum line src startLnum |
  f := GsFile openRead: aFileName .
  f ifNil:[ ArgumentError signal:'file open failed for: ', aFileName ].
  origContents := f contents .
  f close .
  f := GsFile openRead: aFileName .
  f ifNil:[ ArgumentError signal:'file open failed for: ', aFileName ].
  result :=  String new .
  lineNum := 0 .
  [ true ] whileTrue:[
    (line := f nextLine) ifNil:[
      f close .
      result = origContents ifFalse:[
        f := GsFile openWrite: aFileName .
        f ifNil:[ ArgumentError signal:'open for write failed for: ', aFileName ].
        (f nextPutAll: result) = result size ifFalse:[
          Error signal:'write failed to ' , aFileName
        ].
        f close .
        ^ true
      ].
      ^ false
    ].
    lineNum := lineNum + 1 .
    result addAll: line .
    (cmd := line trimWhiteSpace) size > 0 ifTrue:[
      "per abbreviations allowed in topaz command decoding in src/tpaux.c"
      (      (cmd at: 1 equalsNoCase:'m')
        or:[ (cmd at: 1 equalsNoCase:'cl')
        or:[ (cmd at: 1 equalsNoCase:'doi')
        or:[ (cmd at: 1 equalsNoCase:'interp')
        or:[ (cmd at: 1 equalsNoCase:'pri')
        or:[ (cmd at: 1 equalsNoCase:'run')
        or:[ (cmd at: 1 equalsNoCase:'rubyc')
        or:[ (cmd at: 1 equalsNoCase:'rubyr')
        or:[ (cmd at: 1 equalsNoCase:'rubym')
        or:[ (cmd at: 1 equalsNoCase:'ru') and:[ (cmd at: 1 equalsNoCase:'rub') not ]
           ]]]]]]]]] ) ifTrue:[
	 src := String new .   startLnum := lineNum .
	 [ line := f nextLine .
           lineNum := lineNum + 1 .
	   line == nil or:[ (line at: 1) == $% ]
	 ] whileFalse:[
	   src addAll: line .
	 ].
         [
	   result addAll:( self convertArrayBuildersInString: src ).
         ] onException: Error do:[:ex |
           GsFile gciLogServer:'Error for source starting at line ', startLnum asString .
           ex pass .
         ].
	 line ifNotNil:[ result addAll: line  "the % line "].
      ].
    ].
  ].
%

category: 'Debugging Support'
classmethod: GsNMethod
disableBreakInClass: aClass selector: aSelector stepPoint: aStepPoint

"Disable the breakpoint previously set at aStepPoint in method aSelector of
 class aClass."

(aClass compiledMethodAt: aSelector) disableBreakAtStepPoint: aStepPoint
%

category: 'Debugging Support'
classmethod: GsNMethod
enableBreakInClass: aClass selector: aSelector stepPoint: aStepPoint

"Set or reenable the breakpoint previously set at aStepPoint in method
 aSelector of class aClass."

(aClass compiledMethodAt: aSelector) setBreakAtStepPoint: aStepPoint
%

category: 'Compiler Access'
classmethod: GsNMethod
generateFromIR: aGsComMethNode

 "Invokes the code generator to generate an instance of GsNMethod from the
  specified IR graph.
  May be used for anonymous methods or for methods to be installed in a class.
  Caller of this method is responsible for installing in the
  method dictionary of an appropriate class.

  Returns a GsNMethod if the compilation
  succeeded with no warnings or errors, or an Array of of the form
    { (GsNMethod , or nil if the compilation has errors) .
       (nil or an Array of error descriptors as described for
        compileMethod:dictionaries:category: ) .
       (nil or a String describing warnings)
     } .

  Each element of the 'Array of error descriptors'
  is an Array of size 3 or 4, containing the elements:
   1. The GemStone error number.
   2. Offset into the source string where the error occurred.
   3. Error message text, if available, or nil.
   4. Internal compiler error text, if the error is internal.
 "
TraceIR ~~ 0 ifTrue:[
  TraceIR == 1 ifTrue:[
    aGsComMethNode fileName ifNotNil:[
      GsFile gciLogServer: '--IR: ' , aGsComMethNode summary
    ].
  ].
  TraceIR > 1 ifTrue:[ | strm |
    strm := (GsCompilerClasses at: #IndentingStream) newPrinting .
    aGsComMethNode printFormattedOn: strm .
    GsFile gciLogServer: strm contents .
    "  following code to be enabled after filein for debugging only,"
    "  SessionTemps not defined at this point in slowfilein  "
    " false ifTrue:[ |  hist |
        hist := SessionTemps current at:#GsNMethod_AllIrs otherwise: nil .
        hist == nil ifTrue:[
           SessionTemps current at:#GsNMethod_AllIrs put:( hist := { } ).
        ].
        hist addLast: aGsComMethNode .
      ].
    "
  ].
].
^ self _generateFromIR: aGsComMethNode
%

category: 'Constants'
classmethod: GsNMethod
maxArgs
  "Returns the maximum number of arguments to a method."

  ^ GEN_MAX_ARGS
%

category: 'Method Lookup Cache Statistics'
classmethod: GsNMethod
methodLookupCachesReport: minSize

| raw arr rpt lf lines line overflows |
(minSize _isSmallInteger not or:[ minSize < 0 ]) ifTrue:[
  ArgumentError signal:'minSize must be a SmallInteger >= 0'.
].
raw := self _mluCachesReport: minSize .
lf := Character lf .
(rpt := '--- selected session stats' copy) add: lf .

arr := raw at: 1 . "selected method lookup stats ( name, value pairs)"
lines := { } .
1 to: arr size by: 2 do:[:j | | val |
  val := arr at: j + 1 .
  (line := '  ' copy) add: (arr at: j); add: '  ' ; add: val asString .
  lines add: line .
].
(SortedCollection withAll: lines)  do:[:s | rpt add: s ; add: lf . ].

rpt add: '--- Class , descending size of per-class method lookup cache '; add: lf .
arr := raw at: 3 . "size of classes' method lookup cache"
lines := { } .
1 to: arr size by: 2 do:[:m | | cls sz |
  sz := arr at: (m + 1 ).
  sz >= minSize ifTrue:[
    line := String new .
    (cls := arr at: m ) isMeta ifTrue:[ line add:'meta' ].
    "ifFalse:[ cls isRubySingletonClass ifTrue:[ line add:'singleton ' ]]."
    line add: cls thisClass name ; add: '  ' ; add: sz asString .
    line := Association newWithKey: line value: sz .
    lines add: line .
  ].
].
(lines sortDescending:'value') do:[:assoc | rpt add: '  '; add: assoc key; add: lf ].

arr := raw at: 2 . "send-site caches details"
lines := { } .
overflows := { } .
arr := raw at: 2 .
1 to: arr size by: 5 do:[:k | | cacheSize |
  cacheSize := arr at: k + 4 .
  (cacheSize < 0 or:[ cacheSize >= minSize]) ifTrue:[
    | cls selEnvId selector ipOfs meth isBlk |
    meth := arr at: k .
    ipOfs := arr at: k + 1 .
    selector := arr at: k + 2 .
    selEnvId := arr at: k + 3 .
    meth isMethodForBlock ifTrue:[  meth := meth homeMethod . isBlk := true ].
    line := String withAll: (cls := meth inClass) thisClass name .
    line add:( meth environmentId > 0
               ifTrue:[ cls isMeta ifTrue:[ $. ] ifFalse:[ $# ] ]
               ifFalse:[ cls isMeta ifTrue:[ '(C) >> '] ifFalse:[ ' >> ']]) .
    line add: meth selector ; add: '  ' .
    isBlk ifNotNil:[ line add: ' (in block) ' ].
    line add:'  IP:'; add: ipOfs asString ; add: '  ';
        add:' '; add: selector ; add: ' , ' .
    cacheSize < 0 ifTrue:[
      line add: 'polymorphic overflow'. overflows add: line
    ] ifFalse:[
      line add: 'send-site size: '; add: cacheSize asString . lines add: line
    ]
  ].
].
rpt add: '--- Overflowed send sites '; add: lf .
(overflows sortAscending:'') do:[:s | rpt add: '  '; add: s ; add: lf ].
rpt add: '--- Send sites with size >= ' ; add: minSize asString ; add: lf .
(lines sortAscending:'') do:[:s | rpt add: '  '; add: s ; add: lf  ].

^ rpt
%

category: 'Instance Creation'
classmethod: GsNMethod
new

"Disallowed.  You cannot create new instances of GsNMethod."

self shouldNotImplement: #new
%

category: 'Instance Creation'
classmethod: GsNMethod
new: anInteger

"Disallowed.  You cannot create new instances of GsNMethod."

self shouldNotImplement: #new:
%

category: 'Class organizer support'
classmethod: GsNMethod
optimizedSelectors
 | res |  
   res := #( #~~
    #==
    #_and:
    #_downTo:by:do:
    #_downTo:do:
    #_isArray
    #_isExceptionClass
    #_isExecBlock
    #_isFloat
    #_isInteger
    #_isNumber
    #_isOneByteString
    #_isRange
    #_isScaledDecimal
    #_isSmallInteger
    #_isSymbol
    #_leaveProtectedMode
    #_or:
    #_stringCharSize
    #and:
    #ifFalse:
    #ifFalse:ifTrue:
    #ifNil:
    #ifNil:ifNotNil:
    #ifNotNil:
    #ifNotNil:ifNil:
    #ifTrue:
    #ifTrue:ifFalse:
    #isKindOf:
    #or:
    #repeat
    #timesRepeat:
    #to:by:do:
    #to:do:
    #untilFalse
    #untilFalse:
    #untilTrue
    #untilTrue:
    #whileFalse
    #whileFalse:
    #whileTrue
    #whileTrue: 
    #__inProtectedMode
    #_gsReturnNoResult
    #_gsReturnNothingEnableEvents
) copy .
  res addAll: self configurableOptimizedSelectors .
  ^ res 
%

category: 'Compiler Access'
classmethod: GsNMethod
traceIR: anInt

 "set value of class variable controlling printing of IR inputs to
  generateFromIR: .  printing is with GsFile gciLogServer:  .

  anInt == 0 means  no tracing ,
        == 1 means  log source line number of each method node
        == 2 means  full print of IR graph for each method
"

  TraceIR := anInt .
%

category: 'Debugging Support'
classmethod: GsNMethod
_addMarkerIds: anArray

""

| placesMarked markerLine markerLineSize space addToEnd markPosition
  aStr neededSize subStr |

placesMarked:= anArray at: 1.
markerLine:= anArray at: 2.
space:= Character space.

"have the source marked at each error with ^; now add marker identifier"
addToEnd:= false.
1 to: (placesMarked size) do: [:i |
   markPosition:= (placesMarked at: i) at: 1.
   aStr:= ((placesMarked at: i) at: 2) asString.
   neededSize:= markPosition + aStr size.
   markerLineSize := markerLine size .
   (markerLineSize < neededSize) ifTrue: [
       markerLine size: neededSize.
       markerLineSize + 1 to: neededSize do: [:k |
          markerLine at: k put: space].
       markerLineSize:= neededSize.
   ].

   (addToEnd) ifFalse: [
      subStr:= markerLine copyFrom: markPosition + 1
                                to: (markPosition + aStr size).
      subStr do: [:each | (each == $ ) ifFalse: [addToEnd := true]].
   ].
   (addToEnd) ifTrue: [
       markerLine add: aStr.
       (i == placesMarked size) ifFalse: [ markerLine add: ',']
   ] ifFalse: [  | destIdx |
      destIdx := markPosition + 1 .
      markerLine replaceFrom: destIdx to: destIdx + aStr size - 1 with: aStr startingAt: 1 .
   ]
].
(68 - markerLine size) timesRepeat:[ markerLine add: $ ].
(75 - markerLine size) timesRepeat:[ markerLine add: $* ] .
markerLine add: Character lf.
^ true
%

category: 'Private'
classmethod: GsNMethod
_allBreakpointsPerform: aSelector

"For all breakpoint in the result of GsNMethod class >> _breakReport:,
 perform  aSelector .  "

| reportArray |
reportArray := (self _breakReport: true ) at: 2 .

1 to: reportArray size do:[:j| | aBrk aMethod stepPoint |
  aBrk := reportArray at: j .
  aMethod := aBrk at: 5 .
  stepPoint := aBrk at: 4 .
  aMethod perform: aSelector with: stepPoint .
].
%

category: 'Debugging Support'
classmethod: GsNMethod
_allMethodBreakpoints

"Returns an Array of Arrays of the form

  { { <breakpointNumber>. <aGsNMethod>. <ipOffset>. <breakpointLevel>
        ...
        <breakpointNumber>. <aGsNMethod>. <ipOffset>. <breakpointLevel> }.
      ...
    { <breakpointNumber>. <aGsNMethod>. <ipOffset> . <breakpointLevel>
         ...
       {
   }

 The interior Arrays are as described for GsNMethod | _allBreakpoints.

 Breakpoints installed by single-step are not reported.
 If no breakpoints are set, returns an Array of size zero."

^ self _oneArgPrim: 0 with: 0
%

category: 'Private'
classmethod: GsNMethod
_breakpoint: aNumber perform: aSelector

"For the breakpoint listed with number aNumber
 in the result of GsNMethod class >> _breakReport:,
 perform  aSelector .
 Return true if the breakpoint found, false otherwise."

| reportArray |
reportArray := (self _breakReport: true ) at: 2 .

1 to: reportArray size do:[:j| | aBrk |
  aBrk := reportArray at: j .
  (aBrk at:1) = aNumber ifTrue:[  | aMethod stepPoint |
    aMethod := aBrk at: 5 .
    stepPoint := aBrk at: 4 .
    aMethod perform: aSelector with: stepPoint .
    ^ true
  ].
].
^ false
%

category: 'Debugging Support'
classmethod: GsNMethod
_breakReport: withLineNumbersBool

"Returns an Array describing all method breakpoints currently set.
 The Array contains a string report of all breakpoints and an Array
 of Arrays, each describing one breakpoint.

 The Array is 
   { <break report string as displayed in topaz, one line per breakpoint> .
      { { <breakNumber>. <class> . <selector>. <stepPoint>. <GsNMethod> . <disabledBoolean> . <breakpointLevel>}.
        ...
        {<breakNumber>. <class> . <selector>. <stepPoint>. <GsNMethod> . <disabledBoolean> . <breakpointLevel>}
      }
   }
 "

| allBreaksRaw sortedBreaks report descriptors |

allBreaksRaw := GsNMethod _allMethodBreakpoints .
allBreaksRaw size == 0 ifTrue:[ ^ { 'No breaks set' , Character lf . { } } ] .

sortedBreaks := SortedCollection new .
allBreaksRaw do:[: methodBreakArray |
  1 to: methodBreakArray size by: 4 do:[ :j |
    | brkNum aMeth ipOfs hmMeth assoc stepPt brkLevel |
    brkNum := methodBreakArray at: j .
    aMeth :=  methodBreakArray at: j + 1 .
    ipOfs :=  methodBreakArray at: j + 2 .
    brkLevel := methodBreakArray at: j + 3 .
    hmMeth  := aMeth homeMethod .
    stepPt := hmMeth _stepPointForMeth: aMeth ip: ipOfs .
    stepPt == nil ifTrue:[ self error:'could not translate IP to a step point'].
    assoc := Association newWithKey: brkNum  
	       value: { brkNum . hmMeth inClass . hmMeth selector . stepPt . hmMeth . brkLevel } .
    sortedBreaks add: assoc .
  ].
].

report := String new .
descriptors := { }  .
sortedBreaks do:[ :assoc |
  | rawArr aBreakArray breakNumber aMethod theClass selector className stepPoint disabled env |
  aBreakArray := (rawArr := assoc value) copy .
  breakNumber := aBreakArray at: 1 .
  aMethod := aBreakArray at: 5 .
  theClass := aBreakArray at: 2 .
  className := theClass ifNil: ['nil'] ifNotNil: [:cls | cls name]. "fix 49650"
  selector := (aBreakArray at: 3) ifNil: ['(executed code)' ] ifNotNil: [:sel | sel].
  stepPoint := aBreakArray at: 4 .
  withLineNumbersBool ifTrue:[
    report addAll: breakNumber asString; addAll:': ' .
  ] .
  report addAll: className; addAll: ' >> ' ;
	 addAll: selector;  addAll: ' @ ' ;
	 addAll: stepPoint asString .
  (env := aMethod environmentId) ~~ 0 ifTrue:[
    report addAll: ' env ' , env asString .
  ].
  report addAll:' breakpointLevel '; addAll: (aBreakArray at: 6) asString.
  (disabled := stepPoint < 0) ifTrue:[
    stepPoint := stepPoint negated .
    report addAll: ' (disabled)'
  ].
  report add: Character lf .
  aBreakArray at: 4 put: stepPoint .
  descriptors add: aBreakArray .
  aBreakArray at: 6 put: disabled .
  aBreakArray at: 7 put: (rawArr at: 6).
].
^ { report . descriptors }
%

category: 'Debugging Support'
classmethod: GsNMethod
_buildMarkedSourceFrom: sourceStrArg sourceSize: aSize markers: markerArray

^ self _buildMarkedSourceFrom: sourceStrArg sourceSize: aSize markers: markerArray tabSize: 8
%

category: 'Debugging Support'
classmethod: GsNMethod
_buildMarkedSourceFrom: sourceStrArg sourceSize: aSize markers: markerArray tabSize: tabSize

"Given a source string, its size (passed in for efficiency), and a marker
 Array, returns an instance of sourceStr's class containing the marked source."

| lineFeed tab space
  placesMarked     "an Array of markers marked on the current line"
  markerLineIndex "index into the current marker line"
  result          "the result of this method"
  markerLine      "the current marker line"
  aChar           "one Character of the source"
  displayWidth    "the number of positions it takes to display aChar"
  lineSz
  sourceStr
|

 "initialize"
 lineFeed := Character lf.
 tab:= Character tab.
 space:= Character space.

 placesMarked:= { } .
 markerLineIndex:= 1.
 sourceStr := sourceStrArg .
 sourceStr == nil ifTrue:[ sourceStr := String withAll:' "source not available" '].
 result:= sourceStr class new .
 result addAll: '   ' .
 lineSz := 0 .
 markerLine:= String new .
 markerLine add: $  .
 1 to: aSize do: [:i |
   aChar:= sourceStr at: i.  "fetch a char"
   displayWidth := 1 .
   "Add the char to the result"
   (aChar == tab) ifTrue: [
      displayWidth:= tabSize - (lineSz \\ tabSize).
      displayWidth timesRepeat: [result add: space].
      lineSz := lineSz + displayWidth .
   ] ifFalse: [
      result add: aChar.
      lineSz := lineSz + displayWidth .
      ((i == aSize) and: [aChar ~~ lineFeed]) ifTrue: [
        result add: lineFeed .
      ].
   ].

   ((markerArray at: i) == nil) ifTrue: [ "no marker at this position"
      displayWidth timesRepeat:[ markerLine add: space].
   ] ifFalse: [ "found an error at this position"
      placesMarked add: { markerLineIndex + 1 . markerArray at: i }.
      markerLine add: $^ .
      displayWidth - 1 timesRepeat: [markerLine add: space].
   ].
   markerLineIndex:= markerLineIndex + displayWidth.

   ((aChar == lineFeed) or: [i == aSize]) ifTrue: [ "we are at end of line"
      "add error identifiers to marker line "
      (placesMarked size ~~ 0) ifTrue: [
         self _addMarkerIds: { placesMarked . markerLine . markerLineIndex }.
         result add: $  ; add: $* .
         result add: markerLine.
       ] .
      (i == aSize) ifFalse: [
         result addAll: '   ' .
         lineSz := 0 .
      ].
      markerLine size: 1.
      markerLineIndex:= 1.
      placesMarked size: 0.
   ]
 ].
 ^result
%

category: 'Debugging Support'
classmethod: GsNMethod
_buildMarkersFrom: sourceOffsets ofSize: sizeArg

"Given an Array of source offsets, build an Array of size sizeArg containing
 the index into anArray at the position corresponding to anArray's element.
 The remainder of the Array contains nil.  Negative offsets denote disabled
 breakpoints."

| markerArray anOffset posOffset aSize |
aSize := 1 max: sizeArg .                          "fix bug 14976"
markerArray:= Array new: aSize.
1 to: sourceOffsets size do: [:i |
  anOffset := sourceOffsets at: i .
  anOffset == nil ifFalse:[
    posOffset:= (anOffset abs max: 1) min: aSize.  "limit within range"
    (markerArray at: posOffset) ifNotNil:[ "one retry for fix 45431"
      posOffset := posOffset + 1 min: aSize .
    ].
    (markerArray at: posOffset) ifNil:[
       anOffset < 0 ifTrue:[ markerArray at: posOffset put: i negated ]
		    ifFalse:[ markerArray at: posOffset put: i ]
       ]
    ]
  ].
^markerArray
%

category: 'Debugging Support'
classmethod: GsNMethod
_clearAllStepBreaks

"Legacy code, no longer used.
 Clear all single step breakpoints from all loaded methods."
self _setBreakAtIp: -1 operation: 3 frame: nil process: nil breakpointLevel: 0 .
%

category: 'Converting ArrayBuilders - Private'
classmethod: GsNMethod
_convertArrayBuilders: aString stream: stream
  | result ch |
  result := aString class new .
  [ true ] whileTrue:[
    ch := stream nextOrNil .
    ch ifNil:[ ^ aString = result ifTrue:[ aString ] ifFalse:[ result]].
    self _parseChar: ch stream: stream to: result
  ].
%

category: 'Converting ArrayBuilders - Private'
classmethod: GsNMethod
_convertArrayBuildersIfNeeded: aString
 "called from within VM when parsing a doit, or
  when compiling a method and session methods are not activated.
  Invoked from code in Behavior when session methods in use.
  VM has already found GemConvertArrayBuilder==true  "

  | stream |
^ [
    stream := ReadStreamPortable on: aString .
    self _convertArrayBuilders: aString stream: stream .
  ] onException: Error do:[:ex | | pos msg comErrInfo |
    pos := 1 .
    msg := 'unknown ArrayBuilder conversion error' .
    [ | txt |
      pos := stream ifNotNil:[ stream position ].
      (txt := ex messageText) ifNotNil:[ msg := 'ArrayBuilder conversion, ', txt ].
    ] onException: Error do:[:exx |  "ignore"].
    comErrInfo := { { 1071 . pos . msg } } .
    { nil . comErrInfo . nil }
  ].
%

category: 'Converting ArrayBuilders - Private'
classmethod: GsNMethod
_convertArrayBuildersIn: directoryName list: listName refDir: refDirPath
  "refDirPath is to a separate checkout previously converted ;
   its listName files control the operation.
   Compare results of conversion of files in directoryName
   with files in refDirPath , and raise an error if files differ.
"

| dPath listF nC nNotC line fName fPath |
dPath := directoryName copy .
dPath last == $/ ifFalse:[ dPath add: $/ ].
listF := GsFile openRead:  refDirPath , listName  .
listF ifNil:[ ArgumentError signal:'file open failed for: ', (dPath , listName)].
nC := 0 . nNotC := 0 .
[ true ] whileTrue:[
  (line := listF nextLine) ifNil:[
    listF close .
    ^ nC asString, ' changed ' , nNotC asString , ' not changed '
  ].
  fName := line trimWhiteSpace .
  (fName at: 1) == $# ifFalse:[
    fPath := dPath , fName .
    [
      (self convertArrayBuildersInTopazScript: fPath) ifTrue:[  | refFil newFil |
        nC := nC + 1 . GsFile gciLogServer:'  changed ', fName  .
        refFil := GsFile openRead: refDirPath , fName .
        newFil := GsFile openRead: fPath .
        refFil contents = newFil contents ifFalse:[
          GsFile gciLogServer:'difference in ', fPath
        ].
        refFil close .  newFil close .
      ] ifFalse:[
         nNotC := nNotC + 1
      ].
    ] onException: Error do:[:ex |
      GsFile gciLogServer:'Error during ' , fName , ' ; ', ex description .
      ex pass
    ]
  ].
].
%

category: 'Debugging Support'
classmethod: GsNMethod
_deleteAllBreaks

"Delete all breakpoints listed in the result of GsNMethod class >> _breakReport:."

self _allBreakpointsPerform: #clearBreakAtStepPoint:
%

category: 'Debugging Support'
classmethod: GsNMethod
_deleteBreakNumber: aNumber

"Delete the breakpoint listed with number aNumber in the result of
 GsNMethod class >> _breakReport:.
 Return true if breakpoint deleted, false if not found.
 Used by Topaz"

^ self _breakpoint: aNumber perform: #clearBreakAtStepPoint:
%

category: 'Debugging Support'
classmethod: GsNMethod
_disableAllBreaks

"Disable all breakpoints listed in the result of GsNMethod class >> _breakReport:."

self _allBreakpointsPerform: #disableBreakAtStepPoint:
%

category: 'Debugging Support'
classmethod: GsNMethod
_disableBreakNumber: aNumber

"disable the breakpoint listed with number aNumber in the result of
 GsNMethod class >> _breakReport: .
 Return true if breakpoint disabled, false if not found.
 Used by Topaz"

^ self _breakpoint: aNumber perform: #disableBreakAtStepPoint:
%

category: 'Debugging Support'
classmethod: GsNMethod
_enableAllBreaks

"Enable all breakpoints listed in the result of GsNMethod class >> _breakReport:."

self _allBreakpointsPerform: #setBreakAtStepPoint:
%

category: 'Debugging Support'
classmethod: GsNMethod
_enableBreakNumber: aNumber

"Enable the breakpoint listed with number aNumber in the result of
 GsNMethod class >> _breakReport: .
 Return true if breakpoint enabled, false if not found.
 Used by Topaz"

^ self _breakpoint: aNumber perform: #setBreakAtStepPoint:
%

category: 'Private'
classmethod: GsNMethod
_findImplementation: aSelector dir: directoryName list: listName
  "Return a report String describing where in topaz scripts
   aSelector is implemented. aSelector may be either a Symbol
   or a message pattern String like   'kw1: arg1 kw2: arg2' .
   Both directoryName and listName are Strings.
   directoryName is a path to the file listName which
   contains a list of files to search.  Lines in the file beginning
   with $# are comment lines."
| dPath listF rpt line fName fPath|
rpt := String new .
dPath := directoryName copy .
dPath last == $/ ifFalse:[ dPath add: $/ ].
listF := GsFile openRead:  dPath , listName  .
listF ifNil:[ ArgumentError signal:'file open failed for: ', (dPath , listName)].
[ true ] whileTrue:[
  (line := listF nextLine) ifNil:[
    listF close .
    ^ rpt .
  ].
  fName := line trimWhiteSpace .
  (fName at: 1) == $# ifFalse:[
    fPath := dPath , fName .
    [
      rpt addAll:(self _findImplementation: aSelector inTopazScript: fPath fileName: fName)
    ] onException: Error do:[:ex |
      GsFile gciLogServer: 'ERROR during ' , fName  .
      ex pass
    ].
  ].
].
%

category: 'Private'
classmethod: GsNMethod
_findImplementation: aSelector inTopazScript: aPath fileName: fName
  "Return a report String describing where in topaz script aPath
   aSelector is implemented. aSelector may be either a Symbol
   or a message pattern String like   'kw1: arg1 kw2: arg2'  "
  | f keyWords numColons result lineNum selSym lf tab prevSrc |
  f := GsFile openRead: aPath .   lf := Character lf . tab := Character tab .
  f ifNil:[ ArgumentError signal:'file open failed for: ', fName ].
  aSelector _isSymbol ifFalse:[ | pat idx nxt lim sel |
    pat := String new .  "extract selector from message pattern"
    sel := aSelector trimWhiteSpace .
    lim := sel size .
    idx := 1 .
    [ (nxt := sel indexOf: $:  startingAt: idx)  ~~ 0 ] whileTrue:[
      pat addAll:(sel copyFrom: idx to: nxt) .
      idx := nxt + 1 .
      [ idx <= lim and:[ (sel at: idx) isSeparator] ] whileTrue:[
         idx := idx + 1   "skip whitespace after colon"
      ].
      [ idx <= lim and:[ (sel at: idx) isSeparator not ] ] whileTrue:[
         idx := idx + 1   "skip until end of argument name"
      ].
      [ idx <= lim and:[ (sel at: idx) isSeparator] ] whileTrue:[
         idx := idx + 1   "skip whitespace after argumentName"
      ].
    ].
    selSym := pat asSymbol .
  ] ifTrue:[
    selSym := aSelector
  ].
  (selSym includesValue: $: )
       ifTrue:[ keyWords := selSym keywords . numColons := keyWords size ]
      ifFalse:[ keyWords := { selSym } . numColons := 0 ].
  result :=  String new .
  lineNum := 0 .
  [ true ] whileTrue:[ | line methSrc cmd |
    (line := f nextLine) ifNil:[ f close .  ^ result ].
    lineNum := lineNum + 1 .
    cmd := line trimWhiteSpace .
    cmd size > 0 ifTrue:[
      "per abbreviations allowed in topaz command decoding in src/tpaux.c"
      ((     cmd at: 1 equalsNoCase:'m')
       or:[ (cmd at: 1 equalsNoCase:'cl')
       or:[ (cmd at: 1 equalsNoCase:'doi')
       or:[ (cmd at: 1 equalsNoCase:'interp')
       or:[ (cmd at: 1 equalsNoCase:'pri')
       or:[ (cmd at: 1 equalsNoCase:'run')
       or:[ (cmd at: 1 equalsNoCase:'rubyc')
       or:[ (cmd at: 1 equalsNoCase:'rubyr')
       or:[ (cmd at: 1 equalsNoCase:'rubym')
       or:[ (cmd at: 1 equalsNoCase:'ru') and:[ (cmd at: 1 equalsNoCase:'rub') not ]
           ]]]]]]]]] ) ifTrue:[  "have doit or method of some kind"

         ((cmd at: 1 equalsNoCase:'m')
         or:[ cmd at: 1 equalsNoCase:'cl']) ifTrue:[ "have method: or classmethod:" | keywd |
           [
             (line := f nextLine) ifNil:[ f close .  ^ result ].
	     lineNum := lineNum + 1 .
             (line at: 1) == $% ifFalse:[ line := line trimWhiteSpace ].
             line size == 0
           ] whileTrue .
           keywd := keyWords at: 1 .
	   (line at:1 equals: keywd) ifTrue:[  | found methStartLine |
             methSrc := line copy .   found := false .
	     methStartLine := lineNum .
             numColons == 0 ifTrue:[
               found :=  line = keywd asString .
             ] ifFalse:[ | ofs |
	       ofs := keywd size + 1 .
	       [ | pos endOfSel nextL |
		 pos := f position .
		 (nextL := f nextLine) ifNil:[ f close .  ^ result ].
		 methSrc addAll: nextL .
		 lineNum := lineNum + 1 .
		 (nextL at: 1) == $% ifTrue:[ f position: pos ].
		 (endOfSel := (nextL occurrencesOf: $: ) == 0) ifFalse:[
		   line addAll: nextL trimWhiteSpace .
		 ].
		 endOfSel
	       ] whileFalse .
	       (line occurrencesOf: $: ) == numColons ifTrue:[ | j kwlim |
		 found := true .
		 j := 2 . kwlim := numColons .
		 [ j <= kwlim ] whileTrue:[ | idx |
		   keywd := keyWords at: j .
		   (idx := line findString: keywd startingAt: ofs) == 0 ifTrue:[
		     kwlim := 0 . found := false
		   ] ifFalse:[  "line contains keywd"
		     j := j + 1 .  ofs := idx + keywd size .
		   ]
		 ].
               ].
             ].
	     found ifTrue:[
	       result size == 0 ifTrue:[ result add: lf ].
	       result add: fName , ':' , methStartLine asString , '  ', selSym , lf .
	     ] ifFalse:[
               methSrc := nil .
             ].
	   ].
         ].
         [(line at:1) == $% ] whileFalse:[
           (line := f nextLine) ifNil:[ f close . ^ result ].
           methSrc ifNotNil:[ methSrc addAll: line ].
           lineNum := lineNum + 1 .
         ].
         methSrc ifNotNil:[
           (prevSrc ~~ nil and:[ (methSrc = prevSrc) not]) ifTrue:[
	      result addAll:'     <different source>', lf
           ].
           prevSrc := methSrc .
           methSrc := nil .
         ].
      ].
    ].
  ]
%

category: 'Private'
classmethod: GsNMethod
_findImplementations: aSelector
  | insts |
  insts := self allInstances .
  ^ insts select:[:x | x selector == aSelector ].
%

category: 'Private'
classmethod: GsNMethod
_firstPortableIpoffset

  "returns the VM constant OC_GSNMETHOD_FIRST_INSTR_OFFSET"
  ^ self _oneArgPrim: 3 with: 0
%

category: 'Compiler Access'
classmethod: GsNMethod
_generateFromIR: aGsComMethNode

 "Invokes the code generator to generate an instance of GsNMethod from the
  specified IR graph.
  May be used for anonymous methods or for methods to be installed in a class.
  Caller of this method is responsible for installing in the
  method dictionary of an appropriate class.

  Returns a GsNMethod if the compilation
  succeeded with no warnings or errors, or an Array of of the form
    { (GsNMethod , or nil if the compilation has errors) .
       (nil or an Array of error descriptors as described for
        compileMethod:dictionaries:category: ) ,
       (nil or a String describing warnings)
    }  .

  Each element of the 'Array of error descriptors'
  is an Array of size 3 or 4, containing the elements:
   1. The GemStone error number.
   2. Offset into the source string where the error occurred.
   3. Error message text, if available, or nil.
   4. Internal compiler error text, if the error is internal.
 "

<primitive: 679>
self _primitiveFailed: #generateFromIR: args: { aGsComMethNode }
%

category: 'Private'
classmethod: GsNMethod
_gsReturnToC

"Return from Smalltalk execution to C code.
 The C code may be a primitive within the VM,
 a user action or FFI function which invoked Smalltalk ,
 or a GCI client process which made a GCI call to run Smalltalk.

 To be sent only from within the virtual machine. Do not do a send or perform
 of this method in any Smalltalk or GCI code. "

<primitive: 2013>
"If primitive succeeds control returns to the GCI, useraction/FFI, or the VM.
 If primitive fails, the scheduler needs to regain control
 for process termination."

self _terminateNonForkedProcess .
%

category: 'Debugging Support'
classmethod: GsNMethod
_hasBreakpoints
  "Returns a Boolean. "
  ^ self _oneArgPrim: 4 with: 0
%

category: 'Method Lookup Cache Statistics'
classmethod: GsNMethod
_mluCachesReport: minSize

"Returns an Array of the form
   { { String, value , ... }     (various stats)
     { Class , lookupCacheSize , ... )
     { GsNMethod, IP , selector, envId, send_site_cache_size , ... }
   }"

^ self _oneArgPrim: 1 with:  minSize
%

category: 'Private'
classmethod: GsNMethod
_noopReturnTos

"Used in implementation of partial continuation evaluation only."
<primitive: 2032>
"Execution returns to the calling frame"
%

category: 'Debugging Support'
classmethod: GsNMethod
_oneArgPrim: opcode with: aSmallInteger
"opcode == 0 , _allMethodBreakpoints
 opcode == 1 , send-site caches report
 opcode == 2   unused
 opcode == 3 , _firstPortableIpoffset , arg is 0, result SmallInteger. 
 opcode == 4,  _hasBreakpoints, arg not used, result Boolean"

<primitive: 194>
self _primitiveFailed: #_oneArgPrim args: { opcode . aSmallInteger }

%

category: 'Converting ArrayBuilders - Private'
classmethod: GsNMethod
_parseChar: ch stream: stream to: result
    ch == $# ifTrue:[
      stream peek == $[  ifTrue:[
        "don't put $# in result"
        stream next .  "  throw away  $[ "
        self _parseOldArrayBuilder: stream to: result
      ] ifFalse:[
        result add: ch .
        self _parseSymbolLiteral: stream to: result
      ]
    ] ifFalse:[
      result add: ch .
      ch == $$ ifTrue:[ | nxt |  "character literal"
        (nxt := stream nextOrNil) ifNotNil:[ result add: nxt ]. ] ifFalse:[
      ch == $( ifTrue:[  "parenthesized expression"
        self _parseExpression: stream to: result end: $) ] ifFalse:[
      ch == $[ ifTrue:[  "block"
        self _parseExpression: stream to: result end: $] ] ifFalse:[
      ch == ${ ifTrue:[  "select block or CurlyArrayBuilder "
        self _parseExpression: stream to: result end: $} ] ifFalse:[
      ch == $'  ifTrue:[
        self _parseStringLiteral: stream to: result   ] ifFalse: [
      ch == $"  ifTrue:[
        self _parseComment: stream to: result
      ]]]]]]
    ].
%

category: 'Converting ArrayBuilders - Private'
classmethod: GsNMethod
_parseComment: stream to: result
  | ch |
  [ true ] whileTrue:[
    ch := stream nextOrNil .
    ch ifNil:[ ArgumentError signal:'premature end of stream in _parseComment' ].
    result add: ch .
    ch == $"  ifTrue:[ ^ self "end of comment" ]
  ]
%

category: 'Converting ArrayBuilders - Private'
classmethod: GsNMethod
_parseExpression: stream to: result end: endCh
  | ch |
  [ true ] whileTrue:[
    ch := stream nextOrNil .
    ch ifNil:[ ArgumentError signal:'premature end of stream in _parseExpression' ].
    ch == endCh ifTrue:[
      result add: ch .
      ^ self
    ].
    self _parseChar: ch stream: stream to: result
  ]
%

category: 'Converting ArrayBuilders - Private'
classmethod: GsNMethod
_parseOldArrayBuilder: stream to: result
  | ch |
  result add: ${  .
  stream peek == $  ifFalse:[ result add: $  "space after { for readability"].
  [ true ] whileTrue:[
    ch := stream nextOrNil .
    ch ifNil:[ ArgumentError signal:'premature end of stream in _parseOldArrayBuilder' ].
    ch == $,  ifTrue:[  "ensure space before/after .  per syntax"
       result last == $  ifFalse:[ result add: $  ].
       result add: $.  .
       stream peek == $  ifFalse:[ result add: $  ].
    ] ifFalse:[
      ch == $]  ifTrue:[
        result last == $  ifFalse:[ result add: $  "space before } for readability"].
        result add: $} .
        ^ self
      ] ifFalse:[
        self _parseChar: ch stream: stream to: result
      ]
    ]
  ]
%

category: 'Converting ArrayBuilders - Private'
classmethod: GsNMethod
_parseStringLiteral: stream to: result
  | ch |
  [ true ] whileTrue:[
    ch := stream nextOrNil .
    ch ifNil:[ ArgumentError signal:'premature end of stream in _parseStringLiteral' ].
    result add: ch .
    ch == $' ifTrue:[
      stream peek == $' ifFalse:[ ^ self  "end of literal"].
      stream next .  "handle  adjacent single quotes"
      result add: $' .
    ].
  ].
%

category: 'Converting ArrayBuilders - Private'
classmethod: GsNMethod
_parseSymbolLiteral: stream to: result
  "Caller has just consumed a # .
   Just parse enough to distinguish commas within a binary selector Symbol
    literal from commas that are separators or selectors on their own"
  | ch selectorChars |
  ch := stream peek .
  ch ifNil:[ ^ self ].
  selectorChars := '+-\*~<>=|/&@%,?!' .
  (selectorChars indexOf: ch startingAt:1) ~~ 0 ifTrue:[
    "we have a binary selector Symbol literal"
    ch := stream next .
    result add: ch .
    ch := stream peek .
    ch ifNil:[ ^ self ].
    (selectorChars indexOf: ch startingAt:1) ~~ 0 ifTrue:[
      "ch is part of the binary selector"
      ch := stream next .
      result add: ch .
    ].
  ].
%

category: 'Private'
classmethod: GsNMethod
_reportImplementations: aSelector
  | insts list str |
  insts := self allInstances .
  list := insts select:[:x | x selector == aSelector ].
  str := String new .
  list do:[:m | | cls | 
    str add: (cls := m inClass) name , '(class oop ', cls asOop asString,') >> ', m selector ; lf ].
  ^ str  
%

category: 'Debugging Support'
classmethod: GsNMethod
_setBreakAtIp: ipOffset operation: opcode frame: fpOffset process: aGsProcess breakpointLevel: brkLevel

"Apply the following action to all methods that contain breakpoints.

 opcode  action
   2     delete all method breakpoints and disabled method breakpoints
   3     delete single step breakpoints
   (note function of prim 190 varies depending on receiver)

 Other opcodes that are defined for the instance method version of this
 primitive are illegal when used as a class method.

 ipOffset must be == -1 

 brkLevel must be a SmallInteger >= 0."

<primitive: 190>
self _primitiveFailed: #_setBreakAtIp:operation:frame:process:breakpointLevel: 
  args: { ipOffset . opcode . fpOffset . aGsProcess . brkLevel }
%

category: 'Debugging Support'
classmethod: GsNMethod
_sourceWithErrors: compilerError fromString: aString

 ^ self _sourceWithErrors: compilerError fromString: aString tabSize: 8
%

category: 'Debugging Support'
classmethod: GsNMethod
_sourceWithErrors: compilerError fromString: aString tabSize: tabSize

"This method returns an instance of aString's class containing the text in a
 string with compiler errors marked, plus the error text for each error.

 The argument compilerError is the result Array from either the
 Behavior | compileMethod:dictionaries:category:  method or the
 GsNMethod | _recompileWithSource: method.

 The argument aString is the source string which was an input to either the
 Behavior | compileMethod:dictionaries:category: method or the
 GsNMethod | _recompileWithSource: method."

| lineFeed result aStringSize offsets errNumbers thisErr pos
  markerArray errDict errMsgs auxMsgs errsz |

"initialize"
lineFeed := Character lf.
offsets := Array new: (errsz:= compilerError size) .
errNumbers := Array new: errsz  .
errMsgs := Array new: errsz  .
auxMsgs := Array new: errsz  .
aString == nil ifTrue:[
  result := String new  .
  result addAll:' "method source not available" '; add: lineFeed .
].

"get an Array of source offsets with errors, and an Array of error numbers"
1 to: errsz   do: [:i | | errNum |
   thisErr := compilerError at: i.
   offsets at: i put: (thisErr at: 2 "source offset").
   errNumbers at: i put: (errNum := thisErr at: 1 "error number") .
   result ~~ nil ifTrue:[
     result addAll:'error '; add: errNum asString; addAll: ', '
   ].
   thisErr size >= 3 ifTrue:[ | eMsg |
     errMsgs at: i put: (eMsg := thisErr at: 3"error message String") .
     (result ~~ nil and:[ eMsg ~~ nil]) ifTrue:[
        result addAll: eMsg asString ; add: $,
     ].
     thisErr size >= 4 ifTrue:[
       auxMsgs at: i put: (eMsg := thisErr at: 4 "additional message text").
       (result ~~ nil and:[ eMsg ~~ nil]) ifTrue:[
          result addAll: eMsg asString ; add: $,
       ].
     ].
   ].
   result ~~ nil ifTrue:[ result add: lineFeed ].
].
result ~~ nil ifTrue:[ ^ result ].

aStringSize:= aString size.

"build an Array parallel to the source that contains nil if no error at
 that source position, and an index into offsets if there is an error at
 that source position"
markerArray:= self _buildMarkersFrom: offsets ofSize: aStringSize.

result:= self _buildMarkedSourceFrom: aString
                          sourceSize: aStringSize
                             markers: markerArray tabSize: tabSize .

"add error strings"
errDict := GemStoneError at: System myUserProfile nativeLanguage.
1 to: errNumbers size do: [:i | | msg |
  result add: lineFeed.
  result addAll: i asString.
  result addAll: ': ['.
  pos := errNumbers at: i.
  result addAll: pos asString.
  result addAll: '] '.
  msg := errMsgs at: i .
  msg == nil ifTrue:[
    pos > errDict size
      ifTrue: [ msg := '(unknown error number)']
      ifFalse: [ msg := (errDict at: pos) asString].
    ].
  result addAll: msg .
  (auxMsgs at: i) ~~ nil ifTrue:[ result addAll: (auxMsgs at: i) ].
  ].
result add: lineFeed.

^result
%

category: 'Private'
classmethod: GsNMethod
_terminateNonForkedProcess
  2 timesRepeat:[ self _class ]. "loop to detect/handle termination interrupt"
  GsProcess current _setTerminated .
  TerminateProcess new details:'Cannot continue execution'; signalNotTrappable
%

!		Instance methods for 'GsNMethod'

category: 'Accessing'
method: GsNMethod
argsAndTemps

"Returns an Array of Symbols which are the names of arguments and
 temporaries for this method,  not including inner blocks"

 | offset numArgsTmps fi |
 numArgsTmps := self _numArgsTempsCblkargs .
 numArgsTmps < 1 ifTrue:[ ^  #()  ].

 offset := self isMethodForBlock ifTrue:[ DEBUGINFO_BLK_HDRSIZE ] ifFalse:[ 
   (fi := self _debugInfoFileInfo) == 1 ifTrue:[ 
     DEBUGINFO_RowanMTH_HDRSIZE 
   ] ifFalse:[
     fi == 0 ifTrue:[ 
        self _hasPragmaInfo ifTrue:[ DEBUGINFO_BLK_HDRSIZE ] 
                           ifFalse:[ DEBUGINFO_MTH_HDRSIZE ]
     ] ifFalse:[ 
       Error signal:' _debugInfoFileInfo value ', fi asString,' unsupported'. 
       0
     ]   
   ]
 ].
 offset := offset + 1 .
 ^ debugInfo copyFrom: offset to: (offset + numArgsTmps - 1)
%

category: 'Accessing'
method: GsNMethod
at: anIndex

"Returns value of specified varying instVar of receiver.

 Attempt to fetch from the send-site caches area of a method that
 has been loaded for execution will return nil.
 Use GsNMethod>>literals to get the complete literals pool.
"

anIndex < 1 ifTrue:[ self _errorIndexOutOfRange: anIndex ].
^ self _primitiveAt: (anIndex + GsNMethod_InstSize)
%

category: 'Accessing'
method: GsNMethod
at: anOffset put: aValue

"For use only on methods not yet invariant, such as during compilation
 of a CCallout for Ruby.  Most instances of GsNMethod are produced
 as invariant by primitives 679 or 228 .  Stores aValue into specified
 varying instVar of the receiver."

<primitive: 572>
anOffset _validateClass: SmallInteger .
self _errorIndexOutOfRange: anOffset .
self _primitiveFailed: #at:put:  args: { anOffset . aValue }
%

category: 'Accessing'
method: GsNMethod
basicSize

"Returns varying size of receiver."

^ self _primitiveSize - GsNMethod_InstSize
%

category: 'Disassembly'
method: GsNMethod
blockLiterals

"Return an Array of Block Literals that the receiver contains,
 including all inner blocks. Returns nil if the receiver
 contains no blocks.
 In-line blocks never have any block literals."

| arr |
self _numBlockLiterals ~~ 0 ifTrue:[
  arr := { } .
  self _addBlockLiteralsTo: arr .
] ifFalse:[
  arr := #() .
].
^ arr
%

category: 'Debugging Support'
method: GsNMethod
clearAllBreaks

"Clear all method breakpoints in the receiver."

self _allBreaksOp: 2 frame: nil process: nil
%

category: 'Debugging Support'
method: GsNMethod
clearBreakAtStepPoint: aStepPoint

"Clear method breakpoint at specified step point.
 Returns nil if aStepPoint is not legal otherwise returns receiver."

self _breakOperation: 2 forStepPoint: aStepPoint breakpointLevel: 0 
%

category: 'Clustering'
method: GsNMethod
clusterDepthFirst

"This method clusters the receiver, its bytecodes, its selector pool, and its
 selector in depth-first order.  Returns true if the receiver has already been
 clustered during the current transaction; returns false otherwise."

self _clusterDepthFirst ifFalse:[ | blks |
  blks := self blockLiterals .
  blks do:[ :aBlk |
      aBlk clusterDepthFirst
  ].
  ^ false
].
^ true
%

category: 'Copying'
method: GsNMethod
copy

"Disallowed.  You may not create new instances of GsNMethod."

self shouldNotImplement: #copy
%

category: 'Debugging Support'
method: GsNMethod
disableAllBreaks

"Disable all method breakpoints in the receiver."


self _allBreaksOp: 4 frame: nil process: nil
%

category: 'Debugging Support'
method: GsNMethod
disableBreakAtStepPoint: aStepPoint

"Disable method breakpoint at specified step point.
 Returns nil if aStepPoint is not legal otherwise returns receiver."

self _breakOperation: 4 forStepPoint: aStepPoint breakpointLevel: 0
%

category: 'Stripping Sources'
method: GsNMethod
emptySource

"Returns nil in place of the source string.  The #emptySource selector may be
 used as an argument to the stripWith: keyword of the method
 GsNMethod>>decompileForCategory:classRef:stripWith:classMethod:, where it causes
 the string 'source not available...' to be used as the source string when
 reloading the decompiled method."

^ nil
%

category: 'Accessing'
method: GsNMethod
environmentId

"Return a SmallInteger,
 the 8 bit unsigned compilation environment identifier of this method."

 ^ (selector bitShift: 0 - SI_SELECTORID_ENV_shift) bitAnd: SELECTORID_ENV_mask
%

category: 'Stripping Sources'
method: GsNMethod
fullSource

"Returns the complete source string.  The #fullSource selector may be used as an
 argument to the stripWith: keyword of the method
 GsNMethod>>decompileForCategory:classRef:stripWith:classMethod:."

^ self sourceString
%

category: 'Disassembly'
method: GsNMethod
hasBlockLiteralsOfCost: aClassOrCost

"Return true if the receiver contains a block as costly as, or
 more costly than aClassOrCost, return false otherwise.

 For the purposes of this analysis, we have:
   simple block  cost == 1  (old class SimpleBlock)
   complex block cost == 2  (old class ComplexBlock)
   complex block with return to home, cost == 3 (ComplexVCBlock)

 Note that SimpleBlock, ComplexBlock, ComplexVCBlock
 are supported for compatiblity only . New code should use
 the integer costs since there is now only one block class, ExecBlock .
 In-line blocks never have an actual ExecBlock and are never included
 in this analysis.
 "

| blockLits |
blockLits := self _blockLiterals .
blockLits size ~~ 0 ifTrue:[  | argCost |
  aClassOrCost _isSmallInteger ifTrue:[
    argCost := aClassOrCost
  ] ifFalse:[
    argCost := aClassOrCost _cost .
  ].
  1 to: blockLits size do:[:k| | aBlk |
    aBlk := blockLits at: k .
    aBlk _cost >= argCost ifTrue:[ ^ true ] .
  ].
  1 to: blockLits size do:[:k| | aBlk |
    aBlk := blockLits at: k .
    "recursively check inner blocks"
    (aBlk method hasBlockLiteralsOfCost: argCost) ifTrue:[ ^ true].
  ].
].
^ false
%

category: 'Accessing'
method: GsNMethod
homeMethod

"Returns the home method if the receiver is method for a block,
 otherwise returns self."

^ self isMethodForBlock ifTrue:[ inClass ] ifFalse:[ self ]
%

category: 'Accessing'
method: GsNMethod
inClass

"Returns the class in which the receiver was compiled.
 Returns nil for an anonymous method."

self isMethodForBlock ifTrue:[
  ^ inClass"the home method"   inClass
] ifFalse:[
  ^ inClass   "result is nil for anonymous method"
]
%

category: 'Accessing'
method: GsNMethod
instVarAt: anIndex

"If the receiver has an instance variable at anIndex, returns
 its value.  Generates an error if anIndex is not a SmallInteger or
 is out of bounds, or if the receiver has no instance variables."

| numInstVars |
(anIndex _isSmallInteger) ifTrue: [
  numInstVars:= self class instSize .
  ((anIndex >= 1) and:[anIndex <= numInstVars]) ifFalse:[
        ^ self _errorIndexOutOfRange: anIndex "out of bounds"
  ].
  ^ self _primitiveAt: anIndex
] ifFalse: [
  ^ self _errorNonIntegerIndex: anIndex
].
%

category: 'Disassembly'
method: GsNMethod
instVarsAccessed

"Returns an IdentitySet of instVarNames that the method accesses,
 including accesses by all inner blocks."

^ self _allInstVarsAccessed: -2 into: IdentitySet new
%

category: 'Disassembly'
method: GsNMethod
instVarsRead

"Returns an IdentitySet of instVarNames that the method accesses,
 including accesses by all inner blocks "

^ self _allInstVarsAccessed: 0 into: IdentitySet new
%

category: 'Disassembly'
method: GsNMethod
instVarsWritten

"Returns an IdentitySet of instVarNames that the method accesses,
 including accesses by all inner blocks "

^ self _allInstVarsAccessed: -1 into: IdentitySet new
%

category: 'Accessing'
method: GsNMethod
invocationCount

"no longer supported"
^ 0
%

category: 'Accessing'
method: GsNMethod
isMethodForBlock

"Returns true if the receiver is the method for an ExecBloc."

^ ((iFields1 bitShift: 0 - IsMethodForBlock_shift) bitAnd: 1) == 1
%

category: 'Reporting'
method: GsNMethod
isSenderOf: aSymbol

"Returns true if the receiver sends the message aSymbol.  Returns false
 otherwise."

| ofs |
ofs := self _sourceOffsetOfFirstSendOf: aSymbol .
^ ofs ~~ nil
%

category: 'Stripping Sources'
method: GsNMethod
isSourceStripped

    "Answer true if the source code has been stripped for the
    receiver. Otherwise answer false. Determine this by asking the
    class if the method's selector is one of the known stripped
    selectors."

 "Stripping not implemented yet for GsNMethod"
 ^ false
%

category: 'Accessing'
method: GsNMethod
literals

"Returns an Array containing the literal pool of the receiver.,
 including literal pools of all inner blocks. "

| res blks |
res := self _literals .
blks := self blockLiterals .
1 to: blks size do:[:k | | aBlk|
  aBlk := blks at: k .
  res addAll: aBlk method _literals
].
^ res
%

category: 'Accessing'
method: GsNMethod
literalsOffset

"Returns the one-based offset to the start of the non-selector literals.
 Returns zero if there are no literals other than those in the selectorPool."

| ofs high |
ofs := (iFields1 bitShift: 0 - LiteralsOffset_shift) bitAnd: LiteralsOffset_mask .
ofs ~~ 0 ifTrue:[
  (ofs bitAnd: LiteralsOffset_lrgBit) ~~ 0 ifTrue:[
    ofs := ofs bitXor: LiteralsOffset_lrgBit .
    high := debugInfo at: DebugInfo_Fields2_offset .
    high := (high bitShift: 0 - HighLiteralsOffset_shift ) bitAnd: HighLiteralsOffset_mask .
    high := high bitShift: LiteralsOffset_smallBits .
    ofs := ofs bitOr: high .
  ].
  ofs := ofs + 1  "convert to one-based"
].
^ ofs
%

category: 'Accessing'
method: GsNMethod
loadedSizeBytes

 "Return the number of bytes of code generation memory that
  the receiver will occupy when loaded for execution, not
  including the native code."

  "header + namedIvs = 3 + 4 = 7 words"

  ^ (7 + self size + (self numSends * 2)) * 8
%

category: 'Accessing'
method: GsNMethod
methodCompilerVersion

"Returns the method compiler version.
 4 indicates a method compiled by Gs64 v3.3 or above method compiler,
 3 indicates a method compiled by Gs64 v3.0 or above method compiler,
 2 indicates a method compiled by Gs64 v2.0 or above method compiler,
 1 indicates a method compiled in a previous version and processed by
   repository conversion.

 Any other value indicates a method from a previous version that
 did not get converted. "

^ (iFields1 bitShift: MethCompilerVers_shift negated) bitAnd: MethCompilerVers_mask
%

category: 'Repository Conversion'
method: GsNMethod
needsRecompile

"Returns true if the receiver needs recompilation, false otherwise."

 ^ self methodCompilerVersion < 4
%

category: 'Accessing'
method: GsNMethod
numArgs

"Returns the number of arguments expected by the method."

^ iFields1 bitAnd: NArgs_mask
%

category: 'Accessing'
method: GsNMethod
numSends

"Returns the number of non-special sends in the method."

^ (iFields1 bitShift: 0 - NumSends_shift) bitAnd: NumSends_mask
%

category: 'Pragmas'
method: GsNMethod
pragmas
  "Returns a possibly empty Array of Pragmas."
  (self _debugInfoAccess: 11 at: -1) ifNotNil:[:parray |
    parray == #() ifTrue:[ ^ #() ].
    ^ self class createPragmaFrom: parray for: self
  ].
  self environmentId ~~ 0 ifTrue:[ ^ #() ].
  self inClass ifNotNil:[:cls | ^ cls _pragmasForMethod: self ]
%

category: 'Formatting'
method: GsNMethod
printOn: aStream
  | classOrNil selectorOrNil |
  aStream nextPutAll: self class name; nextPut: $  .
  self isMethodForBlock ifTrue: [aStream nextPutAll: '[] in '].
  classOrNil := self inClass.
  selectorOrNil := self homeMethod selector.
  aStream nextPutAll: (classOrNil ifNil:[ '<nil>' ] ifNotNil:[ classOrNil name]);
          nextPutAll: '>>';
          nextPutAll: (selectorOrNil ifNil:[ '<anonymous>'] ifNotNil:[ selectorOrNil]).
%

category: 'Repository Conversion'
method: GsNMethod
recompile
  "Returns self if recompile not needed, otherwise
   returns a new instance of GsNMethod using session's default symbolList"

  self needsRecompile ifFalse:[ ^ self "recompile not needed"].
  ^ self recompileIntoMethodDict: nil intoCategories: nil .
%

category: 'Repository Conversion'
method: GsNMethod
recompileFromSource
	"Recompiles the receivier using session's default symbolList, and without
   doing any special processing of literal variables."

	^ self _rwRecompileFromSourceIfUnpackagedDo: [ 
      self recompileIntoMethodDict: nil intoCategories: nil 
    ]
%

category: 'Repository Conversion'
method: GsNMethod
recompileIntoMethodDict: aMethodDict intoCategories: aCategDict
  ^ self recompileIntoMethodDict: aMethodDict intoCategories: aCategDict symbolList: nil
%

category: 'Repository Conversion'
method: GsNMethod
recompileIntoMethodDict: aMethodDict intoCategories: aCategDict symbolList: aSymbolList

"Recompiles the method for execution in a Gs64 v3.0 or later system.

 Literal variables whose key is in ObsoleteClasses
 are replaced by the appropriate association from the ObsoleteClassesDict .
 Other literal variables are looked up in the literal pool of the receiver,
 before searching class variables, class pool dictionaries, or
 the current symbolList.  Thus recompilation should work without knowing
 what symbolList was used when the receiver was created.

 The result is a GsNMethod if compilation succeeds, otherwise
 an error is generated.  environmentId zero is used for all compilations.

 If aMethodDict is not nil, and the compilation succeeds,
 the resulting method is added to aMethodDict instead of to
 the receiver's method dictionary.  This is used to add methods
 to per-session dictionaries.

 If aMethodDict is not nil and aCategDict is not nil and
 the compilation succeeds, the resulting method is added aCategDict
 instead of the receiver's categories.

 If the receiver is an anonymous method, the sender of this method is
 reponsible for saving the result."

| cls litVars newSrc anOrigin |
litVars := self _literalVariablesForRecompile .
newSrc := self sourceString .
anOrigin := self _origin .
(System gemConfigurationAt:#GemConvertArrayBuilder) ifTrue:[
  newSrc := GsNMethod convertArrayBuildersInString: newSrc .
].
cls := self inClass .
cls ifNil:[
  ^ newSrc _compileInContext: nil symbolList: aSymbolList oldLitVars: litVars
		environmentId: 0 flags: 0
] ifNotNil:[ | categ res |
 categ := cls categoryOfSelector: self selector .
 res := cls _checkCompileResult:
     ( cls _primitiveCompileMethod: newSrc symbolList: aSymbolList
         category: categ oldLitVars: litVars
         intoMethodDict: aMethodDict intoCategories: aCategDict
         environmentId: 0 )
     source: newSrc suppressCompileWarning: false .
 anOrigin ifNotNil:[ res _origin: anOrigin ].
 ^ res
]
%

category: 'Stripping Sources'
method: GsNMethod
removeAllSourceButFirstComment

"Installs a new source string for the receiver so that only the method signature
 and the first comment (if it exists) are left.  For use in stripping a method
 in place in GemStone.  Bypasses the invariance of the receiver."

self _validatePrivilege ifTrue:[
  "v3.0, debuginfo ok to modify if committed."
  self _debugInfo _unsafeAt: DebugInfo_source_offset put: self _sourceToFirstComment
].
%

category: 'Accessing'
method: GsNMethod
selector

"Returns the Symbol which is the selector of this method."

| selIv selectorObjId |
" in blocks and anonymous methods, selector instVar contains envId and #'' "
self isMethodForBlock ifTrue:[ ^ nil ].
inClass == nil ifTrue:[ ^ nil "anonymous method"].
selIv := selector .
"note bit operations are on integer value of SmallInteger, not on a OopType"
selectorObjId := ((selIv bitAnd: 16r1fffffffffe0 ) bitShift:3) bitOr: 1 .
^ Object _objectForOop: selectorObjId .
%

category: 'Debugging Support'
method: GsNMethod
setBreakAtStepPoint: aStepPoint
  ^ self setBreakAtStepPoint: aStepPoint breakpointLevel: 0"signal to GCI"
%

category: 'Debugging Support'
method: GsNMethod
setBreakAtStepPoint: aStepPoint breakpointLevel: brkLevel

"Set method breakpoint at specified step point.
 Returns nil if aStepPoint is not legal otherwise returns receiver.

 brkLevel must be a SmallInteger .
 brkLevel == 0 means breakpoint will be signalled to GCI .
 brkLevel >= 1 means breakpoint will be signalled to Smalltalk .
 See breakpointLevel instVar in GsProcess .  "

 brkLevel _validateClass: SmallInteger .
^ self _breakOperation: 0 forStepPoint: aStepPoint breakpointLevel: brkLevel
%

category: 'Accessing'
method: GsNMethod
size
"Returns varying size of receiver."

^ self _primitiveSize - GsNMethod_InstSize
%

category: 'Disassembly'
method: GsNMethod
sourceOffsetsOfInstVar: aString

"Returns an Array of sourceOffsets of access to the specified instVar.
 Returns nil if there is no instVar named aString in the receiver's class."
| ofs sym |
(sym := Symbol _existingWithAll: aString ) ifNil:[ ^ nil ].
ofs := self inClass offsetOfInstVar: sym .
ofs == 0 ifTrue:[ ^ nil ].
^ Array withAll:(self _allInstVarsAccessed: ofs into: IdentitySet new)
%

category: 'Accessing'
method: GsNMethod
sourceString

"Returns a CharacterCollection that contains the source code of the receiver."

^ debugInfo at: DebugInfo_source_offset
%

category: 'Stripping Sources'
method: GsNMethod
sourceToFirstComment

"Returns a new source string for the receiver that contains only the method
 signature and the first comment (if it exists).  Does not modify the
 receiver.  The #sourceToFirstComment selector may be used as an argument to
 the stripWith: keyword of the method
 GsNMethod>>decompileForCategory:classRef:stripWith:classMethod:."

| i tmpString sz srcStr nArgs selectorSym |

i := 0.
srcStr := self sourceString .
sz := srcStr size.
selectorSym := self selector .
(selectorSym occurrencesOf: $:) timesRepeat: [
    i := srcStr indexOf: $: startingAt: i + 1.
].
" check if it's a binary selector (it has an argument but no colons) "
nArgs := self numArgs .
(i == 0 and: [ nArgs = 1 ])
  ifTrue: [ i := selectorSym size ].
" i is the offset of the last colon in the signature, or the last character
of a binary selector, or zero "
i := i + 1.

" scan past any white space "
[ (srcStr at: i) isSeparator ] whileTrue: [ i := i + 1 ].

i > sz
  ifTrue: [ ^ srcStr copy ].

" scan past any non-white space to get to end of argument to last keyword"
[ i <= sz and: [ (srcStr at: i) isSeparator not ]] whileTrue: [ i := i + 1 ].
" i is now the offset of the first white space past the signature "

i > sz
  ifTrue: [ ^ srcStr copy ].

" scan past any white space "
[ i <= sz and: [ (srcStr at: i) isSeparator ] ] whileTrue: [ i := i + 1 ].
" i is now the offset of the initial comment or first line of code "

i > sz
  ifTrue: [ ^ srcStr copy ].

" if i is the offset of the initial comment, jump to the end of the comment "
(srcStr at: i) == $"
  ifTrue: [ i := srcStr indexOf: $" startingAt: i + 1 ]
  ifFalse: [ i := i - 1 ].

" create the string to replace the original source with "
tmpString := srcStr copyFrom: 1 to: i .
tmpString addAll: '

< source code not available >'.

^ tmpString
%

category: 'Accessing'
method: GsNMethod
squeakBasicAt: anIndex
 ^ self _basicAt: anIndex
%

category: 'CodeModification Override'
method: GsNMethod
squeakBasicAt: anIndex put: aValue

self shouldNotImplement:#squeakBasicAt:put:
%

category: 'Accessing'
method: GsNMethod
squeakBasicSize
^ self basicSize
%

category: 'Storing and Loading'
method: GsNMethod
writeTo: aPassiveObject

"Instances of GsNMethod cannot be converted to passive form.  This method writes
 nil to aPassiveObject and stops GemStone Smalltalk execution with a notifier."

aPassiveObject writeObject: nil.
self _error: #rtErrAttemptToPassivateInvalidObject.

%

category: 'Disassembly'
method: GsNMethod
_addBlockLiteralsTo: anArray
  "Add all block literals of receiver plus inner blocks to anArray"
  | lastBlkIdx numBlks |
  numBlks := self _numBlockLiterals .
  numBlks ~~ 0 ifTrue:[
    lastBlkIdx := self _lastBlockLiteralOffset .
    lastBlkIdx - numBlks + 1 to: lastBlkIdx do:[:n| | aBlk | "recursively process blocks"
      aBlk := self at: n .
      anArray add: aBlk .
      aBlk method _addBlockLiteralsTo: anArray
    ].
  ].
%

category: 'Debugging Support'
method: GsNMethod
_allBreakpoints

"Returns an Array of the form

       { breakpointNumber1 . method . ipOffset1 . breakpointLevel
          ...
          breakpointNumberN . method . ipOffsetN . breakpointLevel
       }

 The ipOffsets are instruction offsets.
 Negative ipOffsets denote disabled breakpoints.
 breakpointLevel == 0 means the breakpoint will be signalled to GCI ,
 breakpointLevel >= 1 means signalled to Smalltalk .

 Returns nil if no method breakpoints set in the receiver.
 Single step breakpoints managed by GciStep() are not reported."

| res blks |
res := self __allBreakpoints .
blks := self blockLiterals .
blks do:[ :aBlk | | blkRes |
    blkRes := aBlk method __allBreakpoints .
    blkRes size ~~ 0 ifTrue:[
      res == nil ifTrue:[ res := { }  ].
      res addAll: blkRes
    ].
].
^ res
%

category: 'Debugging Support'
method: GsNMethod
_allBreakpointsSourceOffsets

| brksArr result |
brksArr := self _allBreakpoints .         "includes methods of inner blocks"
brksArr == nil ifTrue:[ ^ nil ].
result := { }  .
1 to: brksArr size by: 4 do:[:k | | meth allIps stepPoint anIp |
  meth := brksArr at: k + 1 .
  anIp := brksArr at: k + 2 .
  allIps := meth _debugInfoAccess: 2 at: -1 .
  stepPoint := allIps indexOf: anIp .
  stepPoint ~~ 0 ifTrue:[  "convert to source offset"
    result add:( meth _debugInfoAccess: 1 at: stepPoint )
  ] ifFalse:[
    result add: 1 . "unexpected path"
  ].
 ].
 ^ result
%

category: 'Debugging Support'
method: GsNMethod
_allBreaksOp: anInt frame: fpOffset process: aGsProcess

"Used to clear or disable breakpoints , or to set single step breakpoints."
| blks brkLevel |
brkLevel := aGsProcess ifNotNil:[ aGsProcess breakpointLevel ] ifNil:[ 0 ] .
self _setBreakAtIp: -1 operation: anInt frame: fpOffset process: aGsProcess breakpointLevel: brkLevel .
blks := self blockLiterals .
"Don't use do:  here,  we might be setting single step breakpoints
  within a do: implementation..."
1 to: blks size do:[ :j |
  (blks at: j) method 
     _setBreakAtIp: -1 operation: anInt frame: fpOffset process: aGsProcess 
         breakpointLevel: brkLevel 
].
%

category: 'Debugging Support'
method: GsNMethod
_allDebugInfo: kind

| res blks |
res := self _debugInfoAccess: kind  at: -1 .
blks := self blockLiterals .
blks do:[ :aBlk | | blkRes |
    blkRes := aBlk method _debugInfoAccess: kind  at: -1 .
    blkRes size ~~ 0 ifTrue:[ res addAll: blkRes ].
].
^ res
%

category: 'Debugging Support'
method: GsNMethod
_allDebugInfoWithMeths: kind

| res arr blks |
res := { }  .
arr := self _debugInfoAccess: kind  at: -1 .
1 to: arr size do:[:j | res addLast: { self . (arr at: j) } ].
blks := self blockLiterals .
blks do:[ :aBlk | | blkRes aMeth |
    aMeth := aBlk method .
    blkRes := aMeth _debugInfoAccess: kind  at: -1 .
    1 to: blkRes size do:[:j | res addLast: { aMeth . (blkRes at: j) }].
].
^ res
%

category: 'Disassembly'
method: GsNMethod
_allInstVarsAccessed: kind into: anIdentitySet

"Adds to result the instVarNames that the method accesses,
 including accesses by all inner blocks , as specified by kind.
 Returns anIdentitySet .

  kind == 0 --> read ,   result is an IdentitySet of instVar names
  kind == -1 --> written,  result is an IdentitySet of instVar names
  kind == -2 --> read or written, result is an IdentitySet of instVar names
  kind is one based offset of an instVar,  result is IdentitySet of source offsets of access to that instVar"

| blks |
self _instVarsAccessed: kind into: anIdentitySet .
blks := self blockLiterals .
blks do:[ :aBlk |
  aBlk method _instVarsAccessed: kind into: anIdentitySet .
].
^ anIdentitySet
%

category: 'Accessing'
method: GsNMethod
_argsAndTempsOffsets

"Returns an Array of SmallIntegers .
 Each SmallInteger has bits
     16rFF , lexical level , number of VC.parent refs to follow to the defining VC
     high order bits are a signed   offset*256 .
 The Array represents
   numArgs positive zero-based offsets with respect to FP ( level == 0)
   numTemps offsets ,
      negative offsets are zero-based with respect to FP (level == 0)
      positive offsets are zero based wrt.  instVar 0 of a VC
        the bits 16r000FF are number of VC.parent refs to follow to VC
   Does not include any info for  inner blocks"

 ^ self _debugInfoAccess: 8 at: -1
%

category: 'Accessing'
method: GsNMethod
_at: anIndex

"Returns value of specified varying instVar of receiver.
 Attempt to fetch from the send-site caches area of a method that
 has been loaded for execution will return nil.
 Use GsNMethod>>literals to get the complete literals pool.
"

anIndex < 1 ifTrue:[ self _errorIndexOutOfRange: anIndex ].
^ self _primitiveAt: (anIndex + GsNMethod_InstSize)
%

category: 'CodeModification Override'
method: GsNMethod
_at: anIndex put: aValue

self shouldNotImplement:#_at:put:
%

category: 'Accessing'
method: GsNMethod
_basicAt: anIndex

"Returns value of specified varying instVar of receiver.
 Attempt to fetch from the send-site caches area of a method that
 has been loaded for execution will return nil.
 Use GsNMethod>>literals to get the complete literals pool.
"

anIndex < 1 ifTrue:[ self _errorIndexOutOfRange: anIndex ].
^ self _primitiveAt: (anIndex + GsNMethod_InstSize)
%

category: 'CodeModification Override'
method: GsNMethod
_basicAt: anIndex put: aValue

self shouldNotImplement:#_basicAt:put:
%

category: 'Disassembly'
method: GsNMethod
_blockLiterals

"Return an Array of block literals that the receiver contains,
 or nil if the receiver contains no block literals.
 In-line blocks never have any block literals.
 Literals for inner blocks are not included."

| arr lastBlkIdx numBlks |
numBlks := self _numBlockLiterals .
numBlks ~~ 0 ifTrue:[  | resIdx |
  arr := Array new: numBlks .
  lastBlkIdx := self _lastBlockLiteralOffset .
  resIdx := 1 .
  lastBlkIdx - numBlks + 1 to: lastBlkIdx do:[:n|
    arr at: resIdx put:( self at: n ) .
    resIdx := resIdx + 1 .
  ].
].
^ arr
%

category: 'Debugging Support'
method: GsNMethod
_blockSourceOffsets
  "If receiver is a home method, return an array containing source offsets
   of all step point in receiver and its blocks,
   otherwise return an Array of size equal all source offsets in the home method,
   and containing source offsets of step points within the receiver,
   and nils for all other step points of the home method and its blocks."
| homeMth res blks |
homeMth := self homeMethod .
homeMth == self ifTrue:[ ^ self _sourceOffsets ].
res := homeMth _debugInfoAccess: 1  at: -1 .
res := Array new:(res size) . "convert to nils"
blks := homeMth blockLiterals .
blks do:[ :aBlk | | blkMth blkRes |
    blkMth := aBlk method .
    blkRes := blkMth _debugInfoAccess: 1  at: -1 .
    blkMth == self ifTrue:[ res addAll: blkRes "just these step points"]
        ifFalse:[ res size: (res size + blkRes size) "append nils"].
].
^ res
%

category: 'Debugging Support'
method: GsNMethod
_breakOperation: anInt forStepPoint: aStepPoint breakpointLevel: brkLevel

"Returns nil if aStepPoint is not legal otherwise returns receiver.
 anInt is an opcode as defined for second arg to
 _setBreakAtIp:operation:frame:process:
"

| info |
info := self _meth_ip_ForStepPoint: aStepPoint abs .
info == nil ifTrue:[ ^ nil ].
(info at:1) _setBreakAtIp: (info at: 2)  operation: anInt frame: nil process: nil 
             breakpointLevel: brkLevel
%

category: 'Debugging Support'
method: GsNMethod
_breakPointKind: anIp

"This method infers the kind of action associated with a given bytecode."

self error:'GsNMethod>>_breakPointKind:  not implemented'.
%

category: 'Debugging Support'
method: GsNMethod
_buildIpMarkerArray

"This method builds a marker Array for the receiver's source code string.
 containing IPs of all step points, not including step points in inner blocks.

 The result Array is the same size as the source string and
 contains IP numbers at offsets corresponding to the source string."

| srcOffsets ipsArr srcSize mrkSize markerArray |

srcOffsets := self _debugInfoAccess: 1 at: -1 .
ipsArr := self _debugInfoAccess:2 at: -1 .
srcSize := self sourceString size .

mrkSize := 1 max: srcSize .                          "fix bug 14976"
markerArray:= Array new: mrkSize .
1 to: srcOffsets size do: [:i | | anOffset anIp posOffset |
  anOffset := srcOffsets at: i .
  anIp := (ipsArr at: i ) .
  posOffset := (anOffset abs max: 1) min: mrkSize.  "limit within range"
  (markerArray at: posOffset) ifNotNil:[  "one retry for fix 45431"
     posOffset := posOffset + 1 min: mrkSize
  ].
  (markerArray at: posOffset) ifNil:[
     markerArray at: posOffset put: anIp
  ]
].
^ markerArray
%

category: 'Debugging Support'
method: GsNMethod
_buildMarkerArray: allSteps ofSize: aSize

"This method returns a marker Array for the receiver's source code string,
 each element of the result is the source offset of a step point.

 allSteps == true , show all steps
          == false, show steps where a breakpoint currently exists
          a SmallInteger, show just that step point

 The result Array is the same size as the source string and
 contains step numbers at offsets corresponding to the source string."

| srcOffsets |

srcOffsets := self _blockSourceOffsets  .
(allSteps _isSmallInteger ) ifTrue:[  | numSteps stepToDisplay |
  stepToDisplay := allSteps .
  numSteps := srcOffsets size .
  stepToDisplay < 1 ifTrue:[ stepToDisplay := 1 ].
  stepToDisplay > numSteps ifTrue:[ stepToDisplay:= numSteps ].
  1 to: numSteps do:[ :j |
     j == stepToDisplay ifFalse:[ srcOffsets at: j put: nil ].
  ].
] ifFalse:[
  allSteps ifFalse:[ self _setBreakpointsInSourceOffsets: srcOffsets ].
].

^ GsNMethod _buildMarkersFrom: srcOffsets ofSize: aSize
%

category: 'Reporting'
method: GsNMethod
_classAndSelectorNameWidth: anInt

"Return a String of the form className | selector with the className substring
 padded to width anInt.
 Used by ProfMonitor"

|text sel|
self isMethodForBlock ifTrue:[ | homeMeth homeCls |
  homeMeth := self homeMethod .
  homeCls := homeMeth inClass .
  homeCls ifNotNil:[
    text := 'block in ', homeCls name .
  ] ifNil:[
    text := 'block in executed code' copy .
  ].
  sel := homeMeth selector .
] ifFalse:[  | inCls |
  (inCls := 