"
Hierarchy is: 
Object
  HostAgent

The association named #ObjectFilter in HostAgentUser's UserGlobals
is created if needed in $GEMSTONE/upgradeDir/HostAgent.gs.

  iAmMidCacheAgent is a Boolean, true in a hostagent on a midcache host

  legalCacheIps is an Array of Strings obtained from stone, 
    System _otherCacheAddresses   in a hostagent on a midcache host.

  validatedMidCacheIps is a dictionary , keys are  peerAddrString ,
     value is true  if we have validated that a peer address has
     exactly the same  objFilterByteArray contents as the one
     installed in this hostagent's pgsvr threads.

  
"
Class {
	#name : 'HostAgent',
	#superclass : 'Object',
	#instVars : [
		'configuration',
		'lastAbortGmt',
		'lastLckRefreshGmt',
		'listeningPortNum',
		'listeningSocket',
		'leafNetldiSocket',
		'logger',
		'remoteHostIp',
		'remoteHostIPString',
		'remoteHostName',
		'tlsActor',
		'versionString',
		'iAmMidCacheAgent',
		'iAmBigEndian',
		'logLevel',
		'cacheStatusCount',
		'legalCacheIps',
		'filterMap',
		'objFilterByteArray',
		'validatedMidCacheIps',
		'peerIsLocal',
		'peerIsMyLeafHost',
		'versionBuildStr',
		'pushPagesToMidCache',
		'warmerConfigDict',
		'acceptTimeoutMs',
		'repliedToStoneNetldi'
	],
	#category : 'X509-HostAgent'
}

{ #category : 'primitives' }
HostAgent class >> _replyToStoneNetldi: status [
   "Returns true , false or an error String.
    if status arg ==true, and remote cache is connected
      replys to the stone netldi with  'SUCCESS stoneCacheString' and returns true
    else returns false.

    otherwise status arg should be a String which is delivered to the netldi as 'ERROR ...'"

  <primitive: 1052>
  status == true ifFalse:[ status _validateKindOfClass: String ].
	self _primitiveFailed: #_replyToStoneNetldi: args: { status }.
]

{ #category : 'private' }
HostAgent class >> replyToStoneNetldi: arg [
  "arg is true if a sucess reply is to be sent, or arg is an error String.
   Returns true if reply was sent (always returns true if arg is a String ).
   Returns false if reply should be deferred because remote cache still in startup."
  | res |
  res := self _replyToStoneNetldi: arg .
  res _isOneByteString ifTrue:[ Error signal: 'ERROR, replyToStoneNetldi failed, ', res].
  ^ res .
]

{ #category : 'startup' }
HostAgent class >> start [
	(self new)
		initialize: false ;
		startup
]

{ #category : 'startup' }
HostAgent class >> startMidCacheAgent: warmerCfg [
	(self new)
		initialize: true ;
                warmerConfig: warmerCfg;
		startup
]

{ #category : 'primitives' }
HostAgent >> _createNamedSocket: serverName port:aPortNumber queueLength: queueLength [
  "Returns a SmallInteger fileDescriptor, or nil .
   aPortNumber == -1 causes creation of LCK file only without a listening socket, 
      and returns 0 for success, nil for failure.
   serverName is String from HostAgent>>serverLckName.
   aPortNumber and queueLength are args to the bind and listen .
   Calls  NetSCreateNamedSocket in the VM C code, which creates the
   lock file  /opt/gemstone/locks/<self serverLckName>..LCK . "
  <primitive: 1065>
  serverName _validateKindOfClass: String.
  self validatePortNumber: aPortNumber .
  self validateQueueLength: queueLength .
  self _primitiveFailed: #_createNamedSocket:queueLength: args: { aPortNumber . queueLength }
]

{ #category : 'primitives' }
HostAgent >> _createSessionInStone: cloudHostId gemIp: ipAddressString userId: userIdString [
	"create a session in stone on behalf of a gem which has presented
 a valid login certificate.  Gem C code for X509Login will
 complete the login with GSC_FINISH_CLOUD_LOGIN to register its processId etc.
 Must call this before startPgsvrThread.
 Returns an Array of 3 integers
    { stoneSessionId . remoteShrpcMonProcessId. stnStartupId }  
   or an error String"
	<primitive: 1054>

	self validateHostId: cloudHostId.
	ipAddressString _validateKindOfClass: String.
	userIdString _validateKindOfClass: String.
	self _primitiveFailed: #_createSessionInStone:gemIp:userId:
		args: 
			{cloudHostId.
			ipAddressString.
			userIdString}
]

{ #category : 'primitives' }
HostAgent >> _installObjectFilterArray: aByteArray [
  "Install a copy of the argument to control filtering of objects exported
   to the remote host.
   The argument must be a ByteArray of size >= 8192 bytes, bytes beyond 8192 are ignored.
   For an object in a dataPage , zero based byte  (object.securityPolicyId bitShift:-3) ,
   and  zero based bit (object.securityPolicyId & 7)  within that byte is the bit.
   Bit zero is the least signifigant bit of a byte .
   See ByteArray >> bitAtZ:put:  for use in building   aByteArray .

   If the bit is 1 , an UnauthorizedObjectStub is substituted for the object
   before any data page containing a copy of the object is transmitted to a remote
   gem or to a hostagent on a mid level cache.

   The primitive always sets bitAtZ: 5 in the copy of the argument , thus disallowing 
   transmit of objects in the SecurityDataObjectSecurityPolicy .
"

   <primitive: 1067>
   aByteArray _validateClass: ByteArray .
   aByteArray size < 8192 ifTrue:[ Error signal:'ByteArray too small'].
   self _primitiveFailed: #_installObjectFilterArray: args: { aByteArray } .
]

{ #category : 'primitives' }
HostAgent >> _netInfoFind: serverName [

  "serverName is a String from HostAgent>>serverLckName. 
   Returns a SmallInteger 0..9 which is a NetInfoResultEnum per src/netinfo.ht "
 <primitive: 1064>
 serverName _validateKindOfClass: String.
 self _primitiveFailed: #_netInfoFind: args: { serverName }
 
]

{ #category : 'primitives' }
HostAgent >> _refreshLckFile: deleteBoolean [
  "Recreates or deletes the .LCK file that was created by _createNamedSocket... .
   Returns a SmallInteger, a NetInfoResultEnum value "
  <primitive: 1066>
  deleteBoolean _validateClass: Boolean .
  self _primitiveFailed: #_refreshLckFile: args: { deleteBoolean }.
]

{ #category : 'primitives' }
HostAgent >> _registerHostAgentWithStone: cloudHostId isMidCache: midBoolean listeningPort: portNum hostIp: aHostIp hostName: aHostName [
	"Returns a Boolean (pushToMidCaches) or an error String "
	<primitive: 1055>

	self validateHostId: cloudHostId.
  midBoolean _validateClass: Boolean .
  self validatePortNumber: portNum .
  aHostIp _validateKindOfClass: String .   " of the leaf host"
  aHostName _validateKindOfClass: String .
	self _primitiveFailed: #registerHostAgentWithStone:isMidCache:listeningPort:hostIp:hostName:
     args: {cloudHostId . midBoolean . portNum . aHostIp . aHostName }
]

{ #category : 'private' }
HostAgent >> _startLeafCache: relayTimeout startTimeout: startTimeout [
  "Timeout args are in seconds.
   Requests the leaf netldi to fork the leaf cache,
   Starts the cache control relay threads. When cloud side of
   relay receives GSC_CACHE_PGSVR_LOGIN, local side of relay connects to
   stone listening socket and forwards the GSC_CACHE_PGSVR_LOGIN. Then
   relay goes into relay mode.
   The primitive does not return until the cache page server is ready.

   returns an Array  { leafIsMidCacheBoolean . leafIsBigEndianBoolean }
   or an error String .
  "
  <primitive: 1070>

  self _primitiveFailed: #_startLeafCache:startTimeout: 
       args: { relayTimeout . startTimeout }
]

{ #category : 'primitives' }
HostAgent >> _startMidcachePgsvrThread: stoneSessionId threadNum: thrNum socket: socketFdInteger sslState: aCPointer bigEndian: bigEndInt pushRcvr: pushReceiverBool [
  "Used in hostagent on mid cache, after allocateMidPgsvrThread: executed,
   and after connection accepted from hostagent on stone host.
   Returns nil or a String with error details.  "

  <primitive: 1069>

  stoneSessionId _validateClass: SmallInteger.
  self validateFd: socketFdInteger.
  aCPointer _validateClass: CPointer .
  thrNum _validateClass: SmallInteger .
  self validateBoolInt: bigEndInt .
  pushReceiverBool _validateClass: Boolean .
  self _primitiveFailed: #_startMidcachePgsvrThread:threadNum:socket:sslState:bigEndian:pushRcvr: 
    args: { stoneSessionId . thrNum . socketFdInteger . aCPointer . bigEndInt . pushReceiverBool}
]

{ #category : 'private' }
HostAgent >> _startPagePusher: myPort [
  "Executes in a mid-cache hostagent on the mid host.
   Use midHa session's rdbf connection to hostagent on stone host to
   request a page pusher be started, that pusher will connect to me
   and I will start a page receiver thread.
   Returns true or an error String .
   Returns when page pusher start has been received by hostagent on stone host
   and before threads are started so this green thread in this mid cache hostagent
   can receive the request.
  "
  <primitive: 1091>
  self validatePortNumber: myPort .
  self _primitiveFailed: #_startPagePusher: args: { myPort }
]

{ #category : 'primitives' }
HostAgent >> _startStonePgsvrThread: socketFdInteger sslState: aCPointer session: stoneSessionId reply: aString [
   "Create and start a pgsvr thread which will attach to shared cache on localhost.
    Used in hostagent on stone host. 
    Returns nil or an Error String ."

  <primitive: 1053>
	self validateFd: socketFdInteger.
	stoneSessionId _validateClass: SmallInteger.
  aCPointer _validateClass: CPointer .
  aString _validateClass: String .
	self _primitiveFailed: #_startStonePgsvrThread:sslState:session:reply:
		args: { socketFdInteger.  aCPointer.  stoneSessionId  . aString }
]

{ #category : 'private' }
HostAgent >> _startWarmerPusherThread: destIp arg: extraArg [
  "Start a warmer pusher thread which will connect to destination mid cache at destIp,destPort.
   nThreads and thrIdx  let the new thread figure which part of this cache to scan.
   Returns nil or an error String .  "
  <primitive: 1105>
 
  destIp _validateInstanceOf: String .
  extraArg _validateClass: SmallInteger .
  self _primitiveFailed: #_startWarmerPusherThread:arg: 
       args: { destIp . extraArg }
]

{ #category : 'private' }
HostAgent >> _warmMidLevelCache: fromHost port: aPort threads: numThreads includeData: aBoolean myPort: myPort [

  "fromHost and port specify IP and port of the source mid cache hostagent .
   numThreads specifies number of threads.
   fromHost == nil means warm from the stone cache .
   aBoolean == true means include data pages .
   Starts a startup thread which will start specified number of threads.
   Returns as soon as the startup thread are started.
   numTheads must be a SmallInteger in the range of 1 to 20.
   Each thread does  PomConnectToHostagent  'StartWarmerPusher'  ."

  <primitive: 1092>
  fromHost _validateInstanceOf: String .
  numThreads _validateClass: SmallInteger .
  aBoolean _validateClass: Boolean .
  aPort _validateClass: SmallInteger .
  (aPort < 1 or:[ aPort > 65535])
    ifTrue:[ ^ aPort _error: #rtErrArgOutOfRange args:{ 1 . 65535 }] .
  myPort _validateClass: SmallInteger .
  (myPort < 1 or:[ myPort > 65535])
    ifTrue:[ ^ myPort _error: #rtErrArgOutOfRange args:{ 1 . 65535 }] .

  (numThreads < 1 or:[ numThreads > 20 ])
    ifTrue:[ ^ numThreads _error: #rtErrArgOutOfRange args:{ 1 . 20 }] .
  self _primitiveFailed: #_warmMidLevelCache:port:threads:includeData: args:
    { fromHost . aPort . numThreads . aBoolean }
]

{ #category : 'private' }
HostAgent >> abort [
  "explicit abort"
  System abortTransaction.
	lastAbortGmt := System timeGmt.
  TransactionBacklog enableSignalling .
]

{ #category : 'private' }
HostAgent >> abortIfNecessary [

	| maxSecondsBetweenAborts now maxSecondsBetweenRefresh |
	maxSecondsBetweenAborts := 30. "We are enabling sigAbort also"
  maxSecondsBetweenRefresh := 30 .
	now := System timeGmt.
	now - lastAbortGmt > maxSecondsBetweenAborts ifTrue:[
    self abort .
  ].
  iAmMidCacheAgent ifTrue:[
    now - lastLckRefreshGmt > maxSecondsBetweenRefresh ifTrue:[
      self refreshLckFile .
      lastLckRefreshGmt := now .
    ].
  ].
]

{ #category : 'private' }
HostAgent >> addrToString: aByteArray [
  "Assume IPv4 form "
  | str sz |
  str := String new .
  1 to: (sz := aByteArray size) do:[:n | 
    str addAll: (aByteArray at: n) asString .
    n < sz ifTrue:[ str add: $. ].
  ].               
  ^ str
]

{ #category : 'primitives' }
HostAgent >> allocateMidPgsvrThread: socketFdInteger sslState: aCPointer session: stoneSessionId [
  "Used in a hostagent on mid cache to allocate thread to service a
   session accessing this cache.  Allocates a thread , installs
   the socket,sslState as command connection, does not start the thread.
   Returns  SmallInteger threadNum or a String with error details."

  <primitive: 1078>

  self validateFd: socketFdInteger.
  aCPointer _validateClass: CPointer .
  stoneSessionId _validateClass: SmallInteger.

  self _primitiveFailed: #allocateMidPgsvrThread:sslState:session:
     args: { socketFdInteger . aCPointer . stoneSessionId }
]

{ #category : 'private' }
HostAgent >> bigEndianArgString [
  ^ ':isBigEndian=' , (iAmBigEndian ifTrue:[ $1 ] ifFalse:[ $0 ])
]

{ #category : 'private' }
HostAgent >> checkObjectFilterForMidCache: peerAddrString [
  "Returns true or false"
  | ba val |
  validatedMidCacheIps ifNil:[
    validatedMidCacheIps := StringKeyValueDictionary new .
  ] ifNotNil:[
    val := validatedMidCacheIps at: peerAddrString otherwise: nil .
  ].
  val ifNil:[ 
    filterMap ifNil:[
      val := true .  "running with all default filters"
      validatedMidCacheIps at: peerAddrString put: val .
    ] ifNotNil:[
      ba := filterMap byteArrayForIP: peerAddrString .  
      val :=  ba = objFilterByteArray .
      validatedMidCacheIps at: peerAddrString put: val .
    ].
  ].
  ^ val 
]

{ #category : 'private' }
HostAgent >> checkUserProfile: userName [
  "Returns a boolean   isReadOnly  or denys the login.
   Check various restrictions on the UserProfile after authenticating the Certs."
  | userPro status |
  userPro := AllUsers userWithId: userName ifAbsent:[ nil ].
  userPro ifNil:[ 
      self abort . "might have been a newly created UserProfile"
      userPro := AllUsers userWithId: userName ifAbsent:[
        self denyLogin:'UserProfile does not exist for ' , userName ]
  ].

  status := userPro x509loginStatus .
  (status bitAnd: 16r1) ~~ 0 ifTrue:[
     self denyLogin:'UserProfile is disabled for ' , userName].

  ^ (status bitAnd: 16r2) ~~ 0  "isReadOnly "

  "TODO xxx  implement any of password age expiration , etc
    AllUsers.passwordAgeLimit vs userPro.securityData.lastPasswordChange ??
    expiration because too long since last successful login ?
    expire based on   loginsAllowedBeforeExpiration ?
    update lastLoginTime for the userPro ?
  "
]

{ #category : 'private' }
HostAgent >> cleanup [

	self log: 'Cleaning up and shutting down.'.
	listeningSocket ifNotNil:[ 
    listeningSocket close.
		listeningSocket := nil. 
  ].
  leafNetldiSocket ifNotNil:[
    leafNetldiSocket close .
    leafNetldiSocket := nil.
  ].
  self deleteLckFile .
	self stopAllPgsvrThreads
]

{ #category : 'private' }
HostAgent >> createSessionInStoneForUser: userName [
	"returns an Array  { sessionId . remoteShrpmonProcessId }  ,
   or denies the login. "
	| gemIp result printed sleptMs |
	gemIp := tlsActor peerIpString.
	sleptMs := 0.
	[sleptMs < 20000] whileTrue: 
			[result := self
						_createSessionInStone: configuration remoteHostId
						gemIp: gemIp
						userId: userName.
			result _isArray
				ifTrue: 
					[^result	" return { sessionId . remoteShrpmonProcessId. stnStartupId }  "].
			result _isOneByteString
				ifFalse: 
					[self denyLogin: 'createSessionInStone, invalid result: ' , result asString].
			"string constant also in hostAgentNewSession in stndocall.c"
			result = 'Remote cache is in startup'
				ifTrue: 
					[printed
						ifNil: 
							[self log: 'Waiting for remote cache to startup on ' , gemIp.
							printed := true].
					Delay waitForMilliseconds: 50.
					sleptMs := sleptMs + 50]
				ifFalse: [self denyLogin: 'createSessionInStone failed, ' , result]].
	self denyLogin: 'createSessionInStone remote cache startup timed out'
]

{ #category : 'private' }
HostAgent >> deleteLckFile [
  | status |
  status := self _refreshLckFile: true .
  (status == 1"NIRES_DOESNOTEXIST" or:[ status == 9"NIRES_OK"]) ifFalse:[
     self log: 'deleteLckFile failed, status ', status asString level: #error. 
  ].
]

{ #category : 'error handling' }
HostAgent >> denyLogin: reasonString [
	LoginDeniedError signal: reasonString
]

{ #category : 'private' }
HostAgent >> doAdditionalValidation [
	"Assumed already validated by the handshake: Chain of signing to our trust anchor, valid date.
	To be validated here: certificate type, stone name, IP address."

	self
		validatePeerType;
		validatePeerStoneName;
		validatePeerIpAddress
]

{ #category : 'private' }
HostAgent >> doTlsAcceptAndValidateOn: tcpSocket [
	"Do the TLS handshake and validate all parameters. 
	If successful, answer a CPointer to the SSL struct corresponding to the validated connection.
	If the validation fails at any point, signal a LoginDeniedError with what is known of the reason."
	tlsActor 
    tcpSocket: tcpSocket ;
		doTlsAccept: acceptTimeoutMs .
  [ self doAdditionalValidation  .
  ] onException: Error do:[:ex | tcpSocket secureClose . ex pass ]
]

{ #category : 'private' }
HostAgent >> doTlsConnectAndValidateOn: tcpSocket [
  "Do the TLS handshake and validate all parameters. 
	If successful, answer a CPointer to the SSL struct corresponding to the validated connection.
	If the validation fails at any point, signal a LoginDeniedError with what is known of the reason."

	tlsActor
		tcpSocket: tcpSocket;
		doTlsConnect: acceptTimeoutMs .
  [ self doAdditionalValidation  .
  ] onException: Error do:[:ex | tcpSocket secureClose . ex pass ]
]

{ #category : 'error handling' }
HostAgent >> error: errorDescription [
	Error signal: errorDescription
]

{ #category : 'private' }
HostAgent >> getPreTlsLineFrom: tcpSocket timeoutMs: timeMs [
	"Get characters from the given socket until a newline is received, and answer the characters 
	up to but not including the newline.  Limit the number of characters before newline to 
	something reasonable to prevent DOS attack by very long string without newline.
	Not for general use; the error reporting is specific to the situation during pre-TLS 
	phases of a login handshake."

	| cmd limit newline loopCount cmdSize |
	cmd := String new.
	limit := 512.
	newline := Character lf.
  loopCount := 0 .	
	[ 
    [  | numRead sz notDone lastCh |
       self waitForReadReadyOn: tcpSocket timeoutMs: timeMs .
			 numRead := tcpSocket _rawRead: limit - cmd size into: cmd startingAt: cmd size + 1 .
       numRead _isSmallInteger ifFalse:[ self denyLogin: 'peer not read-ready' ].
       numRead == 0 ifTrue:[ self denyLogin: 'EOF on socket to peer' ].
       loopCount := loopCount + 1 .
       sz := cmd size .
       notDone := sz < limit and: [ (lastCh:= cmd at: sz) ~~ newline and: [ loopCount <= 10 ]].
       notDone ifTrue:[
         self log:'waitForReadReady cmd size ', sz asString,
             ' lastCp:', lastCh codePoint asString, ' : ', cmd asString .
       ].
       notDone 
    ] whileTrue  .
  ] onException: Error do: [:ex | 
    tcpSocket close .
    self denyLogin: 'In getPreTlsLineFrom: ' , ex description
  ].
  cmdSize := cmd size .
	cmdSize == 0
		ifTrue: [self denyLogin: 'Peer disconnected or read timed out.'].
	cmdSize == limit
		ifTrue: 
			[self denyLogin: 'Unexpected long line received starting with: '
						, (cmd copyFrom: 1 to: 80)].
	(cmd at: cmdSize) ~~ newline
		ifTrue: 
			[| msg |
			msg := 'Incomplete line from peer after receiving '
						, cmdSize printString , ' characters: '.
			cmdSize > 80
				ifTrue: 
					[msg
						add: (cmd copyFrom: 1 to: 80);
						add: '...']
				ifFalse: [msg add: cmd].
			self denyLogin: msg].
	cmd removeLast.
  logLevel == #debug ifTrue:[
	  self logDebug: 'Received line ' , cmd printString.
  ].
	^ cmd .
]

{ #category : 'startup' }
HostAgent >> getRemoteHostInformation [
	"Get the IP address (and later perhaps hostname?) of the remote host I'm servicing."

  iAmMidCacheAgent ifTrue:[
	  remoteHostName := '<servicing mid cache>' .
    remoteHostIp := nil . "allow connections from any host for now. TODO xxx configurable?"
  ] ifFalse:[
	  remoteHostIp := tlsActor remoteIpOfSsl: configuration leafLdiSsl .
    remoteHostIPString := self remoteHostIpString .
    remoteHostName := (GsSocket getHostNameByAddress: remoteHostIPString) ifNil:[ 'unknownHost'].
  ].
]

{ #category : 'startup' }
HostAgent >> getTlsParameters [
	"For TLS connections, we use the ssl context used by the netldi to 
	make the initial connection."
  | ssl |
  ssl := configuration leafLdiSsl ifNil:[ configuration inheritedSsl ].
	tlsActor := HostAgentTlsActor forSsl: ssl .
	tlsActor logger: logger
]

{ #category : 'error handling' }
HostAgent >> handleError: anError [
	"Production error handler -- log it."

	logger
		ifNotNil: 
			[logger
				log: anError description level: #error;
				log: 'Stack:' copy lf , (GsProcess stackReportToLevel: 1000) level: #error]
]

{ #category : 'initialization' }
HostAgent >> initialize: isMidCacheBool [
  | onStoneHost |
  "save logLevel in an instVar,  so WE can make a simple decision in places."
  logLevel := #routine . "normal plus error ; omits debug."
  logger := GemLogger logLevel: logLevel .
  versionString := System gemVersionAt: 'gsRelease'.
  lastAbortGmt := System timeGmt .
  lastLckRefreshGmt := lastAbortGmt .
  iAmMidCacheAgent := isMidCacheBool .
  iAmBigEndian :=  System gemIsBigEndian .
  acceptTimeoutMs := UserGlobals at: #AcceptTimeoutMs otherwise: 5000 .
  cacheStatusCount := -1 .
  versionBuildStr := System _hostAgentVersionString .
  repliedToStoneNetldi := isMidCacheBool . "no reply needed if mid cache agent"
  self log: 'System _hostAgentVersionString = ', versionBuildStr .
  (onStoneHost := System sessionIsOnStoneHost) == iAmMidCacheAgent not ifFalse:[
    Error signal:'isMidCacheAgent=' , iAmMidCacheAgent asString ,
       ' disagrees with  sessionIsOnStoneHost=', onStoneHost asString .
  ].
]

{ #category : 'initialization' }
HostAgent >> initializeConfigurationFromSessionState [

	configuration := HostAgentConfig retrieveFromSessionState: iAmMidCacheAgent withDebugLevel: logger logLevel
]

{ #category : 'startup' }
HostAgent >> installObjectFilter [
  | fp |
  filterMap := UserGlobals at: #ObjectFilter otherwise: nil .
  (filterMap isKindOf: ObjectFilteringPolicyMap) ifFalse:[ 
    Error signal:'ObjectFilter is not a ObjectFilteringPolicyMap, got ' , filterMap asString .
  ].
  fp := filterMap policyForIP: remoteHostIPString .
  self log:'Installing ObjectFilteringPolicy for ' , remoteHostIPString .
  self log: fp mappingReport . 
  objFilterByteArray := fp asByteArray .

  "to disallow transmitting objects in a GsObjectSecurityPolicy to the remote cache
   ba bitAtZ: aGsObjectSecurityPolicy objectSecurityPolicyId) put: 1  .
   The primitive always sets bitAtZ:5 put:1  in the copy of the argument , 
   thus disallowing transmit of objects in the SecurityDataObjectSecurityPolicy .
  "
  self _installObjectFilterArray: objFilterByteArray . 
]

{ #category : 'private' }
HostAgent >> listen [
  | lckName lckStatus fd aPort pRange |
  lckName := self serverLckName .
  iAmMidCacheAgent ifTrue:[
    lckStatus := self _netInfoFind: lckName .
    lckStatus == 9"NIRES_OK" ifTrue:[ 
      self error: 'hostagent already running per /opt/gemstone/locks/', lckName ,'.LCK'].
    lckStatus == 8"NIRES_EXE_DEL" ifTrue:[
      self error: 'hostagent running from deleted executable per /opt/gemstone/locks/', lckName ,'.LCK'].
    lckStatus ~~ 1 ifTrue:[
      self log:'Found stale /opt/gemstone/locks/', lckName ,'.LCK' ].
	  pRange := configuration portRange .
    self log: 'Using PortRange: ',  pRange asString .
    self log: 'Using LCK Name: ',  lckName asString .

	  pRange detect: [:portnum |  aPort := portnum .
      fd := self _createNamedSocket: lckName port: portnum queueLength: 128.
      fd  ~~ nil 
    ] ifNone: [self error: 'All ports in range are in use.'].
	  listeningSocket := GsSecureSocket fromFileHandle: fd .
    listeningPortNum := aPort .
	  self log: 'Listening on port ' , listeningPortNum printString
  ] ifFalse:[ 
    "a hostagent on stone host has no listening socket, just a socket connection to
     the leaf netldi for life of the hostagent.
     Create LCK file with no listening socket."
    (self _createNamedSocket: lckName port: -1 queueLength: 1 ) ifNil:[
      self error: 'Unable to create LCK file'.
    ].
    "setup smalltalk state for the connection produced in C "
    leafNetldiSocket := GsSecureSocket fromFileHandle: (fd := configuration leafLdiFd) .
    leafNetldiSocket makeNonBlocking .
    leafNetldiSocket _installSsl: configuration leafLdiSsl .
	  self log: 'Reading from leaf netldi on fileDescriptor ' , fd asString .
  ]
]

{ #category : 'logging' }
HostAgent >> log: aString [

	logger log: aString level: #routine
]

{ #category : 'logging' }
HostAgent >> logDebug: aString [
  logger log: aString level: #debug 
]

{ #category : 'logging' }
HostAgent >> logStartupInformation [
  | str |
  str := 'Host Agent for Stone ' , System stoneName printString .
  iAmMidCacheAgent ifTrue:[
    str addAll: ' for mid cache on ' , GsSignalingSocket getLocalHostName ,
         ' with Id ' , configuration cacheHostId asString .
  ] ifFalse:[
		str addAll: ' remote host '  "TODO  remoteHostName printString , "
				, ' at IP address ' , remoteHostIPString
				, '  with ID ' , configuration remoteHostId printString
  ].
  self log: str .
]

{ #category : 'private' }
HostAgent >> makeConnection [
  | tcpSocket authKind |
  listeningSocket ifNotNil:[
	  "In mid cache host agent,
     Wait 5 seconds accept to complete on the ready socket.
	   Returns an aGsSecureSocket  
     or nil if the accept timed out, or failed.
    "
		tcpSocket := listeningSocket acceptTimeoutMs: acceptTimeoutMs
                                  errorOnTimeout: false.
		"tcpSocket is a GsSecureSocket with no SSL state nor context, non blocking."
		tcpSocket == false ifTrue: [ 
				^ nil "timed out" 
		].
    self validatePeerAddress: tcpSocket .
		"Leave the underlying fd non-blocking until we are ready to give it to the SSL library."
		 authKind := self preTlsAcceptHandshakeOn: tcpSocket .
     authKind == #MutualAuth ifTrue:[
			 self doTlsAcceptAndValidateOn: tcpSocket .
       ^ tcpSocket
     ].
     authKind == #ReverseAuth ifTrue:[
			 self doTlsConnectAndValidateOn: tcpSocket .
       ^ tcpSocket
     ].
     tcpSocket close . "gslist or other"
  ] ifNil:[
    "in a hostagent on stone host, handle requests to connect to an x509 gem"
    (leafNetldiSocket readWillNotBlockWithin: 1000"ms") ifTrue:[ 
      (leafNetldiSocket _readLine: 255 maxWaitMs: 2000"ms") ifNotNil:[:cmd |
        tcpSocket := self processRequestFromLeafLdi: cmd .
        ^ tcpSocket
      ].
    ]
  ].
  ^ nil
]

{ #category : 'startup' }
HostAgent >> midLevelCacheWarmFrom: hostName threads: numThreads includeData: aBoolean [
  "Connects to the 'source' cache on hostName and
   starts the specified number of threads to push pages already in the
   source cache to this mid cache .
   Starts a C starter thread which triggers startup of specified number of 
   pusher threads  in source mid hostagent; each pusher thread connects to
   us to start a page push receiver thread.
   Returns as soon as starter thread is started.
  
   The source shared cache (either stone's cache or a mid cache)
   must be configured with SHR_PUSH_TO_MIDCACHES_THREADS >= numThreads
   in the stone config file or in the startnetldi -E config file  .  "
   | arr |
   hostName ifNil:[ arr := { nil . nil } ]
      ifNotNil:[ 
        arr := System _getMidCachePort: hostName .  "get midCacheIp, midCachePort from stone"
        arr _isArray ifFalse:[
          Error signal:'mid cache connect failed, ' , arr asString
        ].
      ].
   ^ self _warmMidLevelCache: (arr atOrNil: 1) port: (arr atOrNil: 2)
           threads: numThreads includeData: aBoolean myPort: listeningPortNum 
]

{ #category : 'private' }
HostAgent >> myStoneName [

	^GsNetworkResourceString currentStoneName
]

{ #category : 'private' }
HostAgent >> preTlsAcceptHandshakeOn: tcpSocket [
  "return one of #MutualAuth #ReverseAuth #GsList"
	| request |
	request := self getPreTlsLineFrom: tcpSocket timeoutMs: acceptTimeoutMs .
  (request at: 1 equals: 'GsListPlainText:') ifTrue:[
    self sendGsListReply: tcpSocket .
    tcpSocket close .
    ^ #GsList 
  ].
	(request at: 1 equals: 'StartTls-MutualAuth:hostagent') ifTrue:[ | kind response |
    kind := #MutualAuth .
    response := kind , '=Ready' .
	  self sendPreTlsLine: response to: tcpSocket .
    ^ kind
  ].
	(request at: 1 equals: 'StartTls-ReverseAuth:hostagent') ifTrue:[
    "no response, we do sslConnect and other end does sslAccept"
    ^ #ReverseAuth
  ].
	self denyLogin: 'Request for unsupported communication method: ' , request  .
  ^ nil 
]

{ #category : 'private' }
HostAgent >> preTlsConnectHandshakeOn: tcpSocket [
  "return  true if ok, or signal a login denied error"
  self sendPreTlsLine: 'StartTls-ReverseAuth:x509gem' to: tcpSocket .
  "No reply expected , hostagent just does sslAccept"
  ^ true 
]

{ #category : 'private' }
HostAgent >> processAuthenticatedLogin [

	| loginRequest userNames userName sessionId reply sessArray remotePcmonPid 
    stnStartupid readOnlyBool kind pgsErr |
	loginRequest := tlsActor readLine.
	kind := self validateLoginRequest: loginRequest.
  (kind == #'Login:' and:[ peerIsMyLeafHost == true ]) ifFalse:[
    ^ self denyLogin: 'Invalid login request kind ' , kind asString , ' myLeafHost ', peerIsMyLeafHost asString.
  ].
	userNames := tlsActor peerUserNames.
	userNames size == 1
		ifFalse: 
			[self
				denyLogin: 'User certificate should contain exactly 1 user name, but contains '
						, userNames size printString].
	userName := userNames first.
  readOnlyBool := self checkUserProfile: userName .
	sessArray := self createSessionInStoneForUser: userName.
	sessionId := sessArray at: 1.
	remotePcmonPid := sessArray at: 2.
	stnStartupid := sessArray at: 3.
	sessionId _isSmallInteger
		ifFalse: [Error signal: 'invalid sessionId from createSession'].
	remotePcmonPid _isSmallInteger
		ifFalse: [Error signal: 'invalid remotePcmonPid from createSession'].
 
  "compose the reply which will be sent by the pgsvrThread"
	(reply := 'Login=OK:stoneHostId=' copy)  
		add: System hostId printString;
		add: ':sessionId=';
		add: sessionId printString;
		add: ':pcmonPid=';
		add: remotePcmonPid printString;
		add: self bigEndianArgString ;
		add: ':stnStartupId=';
		add: stnStartupid printString ;
    add: ':readOnly='; 
    add: ( readOnlyBool ifTrue:[ $1 ] ifFalse:[ $0 ]) ; lf .

  "startPgsvrThread prim will start a thread,  the ssl becomes owned by the thread
   or will be freed by the failure to start the thread."
	pgsErr := self startStonePgsvrThread: tlsActor tcpSocket sslState: tlsActor ssl
		           session: sessionId reply: reply .
  pgsErr ifNil:[ | msg |
    self log: 'processAuthenticatedLogin sent reply: ' , reply .
		 (msg := 'Logged in session ' copy) add: sessionId printString; 
              add: ' for user '; add: userName .
     self log: msg .
  ] ifNotNil:[ 
    "C code has closed the socket"
    self log: 'ERROR, startPgsvrThread failed, ' , pgsErr asString
  ].
]

{ #category : 'private' }
HostAgent >> processAuthenticatedMidCacheRequest [
  "executes in a hostagent on a mid cache host"
  | loginRequest ofs sessId reply getIntBlk sock kind errStr |
  loginRequest := tlsActor readLine.
  kind := self validatePgsvrThreadRequest: loginRequest.
  sock := tlsActor tcpSocket .

  getIntBlk := [:prefix | 
    ofs := loginRequest findString: prefix startingAt: ofs .
    ofs < 1 ifTrue:[ ^ self denyLogin: 'request has no ', prefix ].
    ofs := ofs + prefix size .
    Integer fromString: (loginRequest copyFrom: ofs to: loginRequest size) .
  ].
  ofs := 1 .
  kind == #'AllocateMidThread:' ifTrue:[ | thrNum |
    "sock is from accept of a connection from an x509 gem on a leaf host,
     the connection is from x509 gem to commandSocket of thread in mid hostagent."
    sessId :=  getIntBlk value: ':sessionId=' .
    sock makeBlocking .
    thrNum := self allocateMidPgsvrThread: sock fileDescriptor
                   sslState: tlsActor ssl session: sessId .
    thrNum _isSmallInteger ifTrue:[ 
      (reply := 'MidCache=OK' copy) 
         add: self bigEndianArgString ;
         add:':threadNum=' ; add: thrNum asString; lf .
      self sendLine: reply to: sock .
      sock _noFreeSslOnGc .  "C thread will close socket"
      ^ self
    ].
    errStr := thrNum
  ] ifFalse:[
  kind == #'StartMidThread:' ifTrue:[ | bigEnd res thrNum |
    "sock is from accept of connect from x509 gem's hostagent on stone host.
     The connection is from a thread in mid hostagent to secondCommand thread in
     session's hostagent on stone host."
    sessId :=  getIntBlk value: ':sessionId=' .
    thrNum := getIntBlk value: ':threadNum=' .
    bigEnd := getIntBlk value: ':isBigEndian=' .
    sock makeBlocking .
    res := self _startMidcachePgsvrThread: sessId threadNum: thrNum 
        socket: sock fileDescriptor sslState: tlsActor ssl bigEndian: bigEnd 
        pushRcvr: false .
    res ifNil:[
      (reply := 'MidCache=OK' copy) lf .
      self sendLine: reply to: sock .
      sock _noFreeSslOnGc .  "C thread will close socket"
      ^ self
    ].
    errStr := res .
  ] ifFalse:[
  kind == #'StartPagePushRcvr:' ifTrue:[ | res bigEnd |
    "also used for mid cache warmer receiver"
    bigEnd := (getIntBlk value: ':isBigEndian=') == 1 .
    bigEnd == iAmBigEndian ifTrue:[
    sock makeBlocking .
      res := self _startMidcachePgsvrThread: 0 threadNum: 0 
          socket: sock fileDescriptor sslState: tlsActor ssl 
          bigEndian: (bigEnd ifTrue:[ 1 ] ifFalse:[ 0 ]) 
          pushRcvr: true .
      res ifNil:[
        (reply := 'MidCache=OK' copy) lf .
        self sendLine: reply to: sock .
        sock _noFreeSslOnGc .  "C thread will close socket"
        ^ self
      ].
      errStr := res .
    ] ifFalse:[
      errStr := 'StartPagePushRcvr illegal mixed byteorder'.
    ].
  ] ifFalse:[
  kind == #'StartWarmerPusher:' ifTrue:[ | res extraArg bigEnd |
    bigEnd := (getIntBlk value: ':isBigEndian=') == 1 .
    bigEnd == iAmBigEndian ifTrue:[
      extraArg := getIntBlk value: ':Arg=' .
      res := self _startWarmerPusherThread:  tlsActor peerIpString arg: extraArg .
      res ifNil:[
        (reply := 'MidCache=OK' copy) lf .
        self sendLine: reply to: sock .
        sock _noFreeSslOnGc .  "C thread will close socket"
        ^ self
      ].
      errStr := res .
    ] ifFalse:[
      errStr := 'StartPagePushRcvr illegal mixed byteorder'.
    ].
  ] ifFalse:[ errStr := kind asString ]
  ]]].
  reply := 'MidCache=Fail,' , errStr asString .
  self sendLine: reply to: sock .
  sock close .
  self log: 'ERROR, startPgsvrThread ' , kind asString, 
            ' failed session:', sessId asString, ', ', errStr asString .
]

{ #category : 'private' }
HostAgent >> processRequestFromLeafLdi: request [
  "netldi on a leaf host is requesting that a hostagent on
   stone host to connect to a listening x509 gem that was forked by that netldi.
   Sends a reply to the netldi on leaf host.
   Returns a GsSecureSocket (a connection to the x509 gem) or nil ."
  | getIntBlk ofs pat sock portNum isShutdown | 
  pat := 'GemListening:' .
  (request at: 1 equals: pat) ifFalse:[
    pat := 'Shutdown:' . 
    (request at: 1 equals: pat) ifTrue:[
      isShutdown := true.
    ] ifFalse:[
      self log: 'netldi request has invalid start: ', request.
      ^ self sendFailureToLeafLdi 
    ].
  ].
  ofs := pat size + 1 .
  (request at: ofs equals: versionBuildStr) ifFalse:[ 
    self log: 'netldi request has invalid version: ', request.
    ^ self sendFailureToLeafLdi 
  ].
  isShutdown == true ifTrue:[
    Error signal:'Shutdown: lost connection to netldi on our leaf host'.
  ].
  getIntBlk := [:prefix | 
    ofs := request findString: prefix startingAt: ofs .
    ofs < 1 ifTrue:[ self log: 'netldi request has no ', prefix . 
                     ^ nil ].
    ofs := ofs + prefix size .
    Integer fromString: (request copyFrom: ofs to: request size) .
  ].
  portNum := getIntBlk value: 'port=' . 
  portNum == 0 ifTrue:[  
    "netldi testing connection ; do not do a connect."
    self sendLine: 'Gem=OK' to: leafNetldiSocket .
    ^ nil  
  ].
  sock := GsSecureSocket _newClientNoSslState  . 
  [ 
    self log: 'attempting connect to port ', portNum asString,
               ' on ', remoteHostIPString asString.
    sock connectTo: portNum on: remoteHostIPString timeoutMs: acceptTimeoutMs  .
  ] onException: Error do:[:ex |
    self log:'connect to x509 gem failed, ' , ex asString .
    sock close .
    ^ self sendFailureToLeafLdi
  ].
  [ 
    self validatePeerAddress: sock .
    self preTlsConnectHandshakeOn: sock .
	  self doTlsAcceptAndValidateOn: sock . "installs SSL into the socket"
  ] onException: Error do:[:ex |
    sock close .
    self log: 'rejected a x509 gem, ' , ex asString .
    ^ self sendFailureToLeafLdi .
  ].
  self sendLine: 'Gem=OK' to: leafNetldiSocket. "success"
  ^ sock
]

{ #category : 'private' }
HostAgent >> refreshLckFile [
  | status |
  status := self _refreshLckFile: false .
  (status == 9"NIRES_OK" or:[ status == 2"NIRES_EXISTS"]) ifFalse:[
     Error signal:'unable to refresh LCK file, status ' , status asString.
  ].
]

{ #category : 'private' }
HostAgent >> registerHostAgentWithStone: cloudHostId [
	"returns self or signals an Error."

	| res |
	res := self _registerHostAgentWithStone: cloudHostId isMidCache: iAmMidCacheAgent 
                 listeningPort: listeningPortNum 
                 hostIp: remoteHostIPString hostName: remoteHostName  .
  res class == Boolean ifFalse:[
		self error: 'registerHostAgentWithStone failed, ' , res asString
  ].
  pushPagesToMidCache := res .  "from stone's config"
  iAmMidCacheAgent ifTrue:[
	  self log: 'Registered as midcache hostagent on host ' , cloudHostId printString .
    pushPagesToMidCache ifTrue:[
       self startPagePusher
    ].
  ] ifFalse:[
	  self log: 'Registered as hostagent for host ' , cloudHostId printString
  ].
]

{ #category : 'unknown' }
HostAgent >> remoteHostIpInteger [
	"remoteHostIp is a byte array. Convert it to an integer."

	| ip |
	ip := 0.
	1 to: remoteHostIp size do: [:n | ip := ip * 16r100 + (remoteHostIp at: n)].
	^ip
]

{ #category : 'logging' }
HostAgent >> remoteHostIpString [

	| string size |
	string := String new.
	size := 4.
	1 to: size
		do: 
			[:i |
			string addAll: (remoteHostIp at: i) printString.
			i = size ifFalse: [string add: $. ] ].
	^string
]

{ #category : 'startup' }
HostAgent >> runConfigured [
	"I already have a configuration. Do everything else."
	| interactive listeningForDebug abortBlock |
	interactive := GsFile stdin isTerminal.
  listeningForDebug := (System gemConfigurationAt:'GEM_LISTEN_FOR_DEBUG') == true.
  abortBlock := [:ex | 
    "self log: ex class name , ' starting abort at ' , System _timeMs asString, ' ms' ."
    self abort .
    ex resume .
  ].
	[ 
    System transactionMode: #manualBegin .
    TransactionBacklog enableSignalling .
    self
		  getTlsParameters;
		  getRemoteHostInformation;
		  logStartupInformation .
	  [ 
      self listen.  "Possibly set up a listening port."
		  self registerHostAgentWithStone: configuration cacheHostId  .
      iAmMidCacheAgent ifTrue:[ 
        self warmMidCache .
      ] ifFalse:[
        self startLeafCache . "before creating .LCK file"
        self installObjectFilter .
      ].
      self serviceLoginRequests .
    ] ensure: [
      self cleanup "cleanup inside of onException so errors are logged"
    ] .
    TransactionBacklog disableSignalling . 
  ] onException: { TransactionBacklog . RepositoryViewLost . Error . Break } 
		do: { abortBlock . abortBlock .
         [:ex | | eStr |
           eStr := ex asString .
           ((eStr includesString:'Shutdown:') or:[ eStr includesString:'cache failed'])  ifFalse:[
             listeningForDebug ifTrue:[ 
               GsFile gciLogServer:'Error ' asUppercase, ex asString .
               self waitForDebug 
             ].
           ].
				   interactive ifFalse:[ self handleError: ex.  self cleanup].
				   ex pass	"let topaz give additional stack dump"
         ] .
          [:ex | | num |
            (num := ex gsNumber) == 6020 ifTrue:[ "request from Stone to logout"
              self log:'HostAgent shutdown request(Break oob) received from stone.'. "fix 47603"
              self cleanup .  
              ex return: nil .
            ].
            listeningForDebug ifTrue:[ self waitForDebug ].
            interactive ifTrue:[ self pause "topaz gets control for debugging"] .
            self log:'Ignoring Break ' , num asString .
            ex resume . 
          ]
        }
]

{ #category : 'private' }
HostAgent >> sendFailureToLeafLdi [
  "must return nil."

  self sendLine: 'Gem=Fail' to: leafNetldiSocket .
  ^ nil "no tcpSocket"
]

{ #category : 'private' }
HostAgent >> sendGsListReply: aSocket [
  | reply |
  "compose reply per netsocket.c  sendReplyToClient"
  reply := 'Text=Ready
'.  
  aSocket _rawNbWrite: reply size from: reply startingAt: 1 . "one attempt to write it." 
  "caller closes socket"
  ^ true .
]

{ #category : 'private' }
HostAgent >> sendLine: aString to: tcpSocket [
  "Sends data with ssl write."
	| data dataSize numWrote sleptMs |
	data := aString copy lf.
	sleptMs := 0.
	dataSize := data size.
	numWrote := tcpSocket nbwrite: dataSize from: data startingAt: 1 .
	numWrote ~~ dataSize ifTrue:[
    (tcpSocket writeWillNotBlockWithin: 2000) ifTrue:[
      numWrote := numWrote + tcpSocket nbwrite: dataSize - numWrote from: data startingAt: numWrote + 1.
    ].
    numWrote ~~ dataSize ifTrue:[ 
      self denyLogin: 'sendLine incomplete'
    ].
  ]
]

{ #category : 'private' }
HostAgent >> sendPreTlsLine: aString to: tcpSocket [
  "Sends data as clear text."
	| data dataSize numWrote sleptMs |
	data := aString copy lf.
	sleptMs := 0.
	dataSize := data size.
	numWrote := tcpSocket _rawNbWrite: dataSize from: data startingAt: 1 .
	numWrote ~~ dataSize ifTrue: [self denyLogin: 'sendPreTlsLine incomplete']
]

{ #category : 'startup' }
HostAgent >> serverLckName [

  | stnName str |
  "Returns a String, the name of the hostagent.LCK file minus the .LCK suffix.

   Format of result must agree with findRunningHostAgent() and
   mlcHaExists() in src/nldicmn.c. "

  stnName := GsNetworkResourceString currentStoneName . 
  str := 'hostagent-', stnName, '-' .
  iAmMidCacheAgent ifTrue:[ str add: 'midcache-', GsSignalingSocket getLocalHostName ]
                  ifFalse:[ str add: remoteHostIPString ].
  ^ str 
]

{ #category : 'private' }
HostAgent >> serviceLoginRequests [
	"For now, service them forever."
	  [ repliedToStoneNetldi ifFalse:[
        repliedToStoneNetldi := self class replyToStoneNetldi: true .
      ].  
		  self serviceOneLoginRequest .
		  self abortIfNecessary .
    ] repeat
]

{ #category : 'private' }
HostAgent >> serviceOneLoginRequest [
	"Waits for an incoming connection, and processes it if there is one."

	| tcpSocket validated |
 [ 
   tcpSocket := self makeConnection . 
   tcpSocket ifNotNil:[
       validated := true .
			 iAmMidCacheAgent ifTrue:[
				 self processAuthenticatedMidCacheRequest
			 ] ifFalse:[
			     self processAuthenticatedLogin
       ].
     ]. 
 ] onException: LoginDeniedError do: [:ex | 
   validated ifNotNil:[  
     "Avoid client ssl read hanging until TCP detects socket close."
     tcpSocket ifNotNil:[  
       [
         tcpSocket close .
       ] onException: SocketError do:[:sEx | "ignore"].
     ]. 
   ].
	 self log: 'Login denied, ' , ex description.
   "Do not execute expensive stackReportToLevel unless logLevel is #debug"
   logLevel == #debug ifTrue:[ 
	   self logDebug: 'Stack:' copy lf , (GsProcess stackReportToLevel: 1000).
   ].
	 tcpSocket ifNotNil: [tcpSocket close].
	 ^self
 ]
]

{ #category : 'startup' }
HostAgent >> startLeafCache [
  | relayTimeout res isMid leafIsBigEndian |
  relayTimeout := System stoneConfigurationAt: #StnRemoteCachePgsvrTimeout .
  res := self _startLeafCache: relayTimeout 
        startTimeout: 60"secs xxx configurable ?" .
  res _isArray ifFalse:[ "res is an error String."
    self class replyToStoneNetldi: res asString .
    Error signal:'start leaf cache failed, ' , res asString .
  ].
  isMid := res at: 1 .
  leafIsBigEndian := res at: 2 .
]

{ #category : 'private' }
HostAgent >> startPagePusher [
  "executes in a mid cache hostagent."
  | status |
  status := self _startPagePusher: listeningPortNum  .
  status == true ifFalse:[ 
    self error: 'startPagePusher failed, ', status asString
  ].
  self log:'begin page pusher startup, ok'.
]

{ #category : 'private' }
HostAgent >> startStonePgsvrThread: aGsSocket sslState: aCPointer session: stoneSessionId reply: aString [
	"To be called after createSessionInStone:gemIp:  
   stoneSessionId must be result of createSessionInStone:gemIp: in the hostagent on stone host .

   After this method returns, HostAgent Smalltalk main program
   must no longer reference  aGsSocket nor aCPointer.
   Returns nil or a String describing an error.
  "
  aGsSocket makeBlocking .
  aGsSocket _noFreeSslOnGc .
	^ self _startStonePgsvrThread: aGsSocket fileDescriptor
			sslState: aCPointer
			session: stoneSessionId  reply: aString 
]

{ #category : 'startup' }
HostAgent >> startup [

	self
		log: 'Host agent starting.';
		initializeConfigurationFromSessionState;
		runConfigured
]

{ #category : 'private' }
HostAgent >> stopAllPgsvrThreads [
 "Attempt to get all threads to cleanly detach from stone cache.
  Should be sent before exiting main program, prior to hostagent logout.
  Returns self."


 <primitive: 1059>
 self _primitiveFailed: #stopAllPgsvrThreads


]

{ #category : 'validation' }
HostAgent >> validateBoolInt: anInt [
	anInt _validateClass: SmallInteger.
  (anInt == 0 or:[ anInt == 1]) ifFalse:[
			(OutOfRange new)
				name: 'boolInt'
					min: 0
					max: 1
					actual: anInt;
				signal
  ]
]

{ #category : 'validation' }
HostAgent >> validateFd: anInt [
	| max |
	anInt _validateClass: SmallInteger.
	max := (1 bitShift: 31) - 1.
	(anInt between: 4 and: max)
		ifFalse: 
			[(OutOfRange new)
				name: 'fileDescriptor'
					min: 4
					max: max
					actual: anInt;
				signal]
]

{ #category : 'validation' }
HostAgent >> validateHostId: hostId [
	| max |
	hostId _validateClass: Integer.
	max := (1 bitShift: 64) - 1.
	(hostId between: 1 and: max)
		ifFalse: 
			[(OutOfRange new)
				name: 'hostId'
					min: 1
					max: max
					actual: hostId;
				signal]
]

{ #category : 'private' }
HostAgent >> validateLoginRequest: loginRequest [
  "Returns #'Login:' as the request type"
  | prefix ofs |
  prefix := #'Login:' .
  (loginRequest at: 1 equals: prefix) ifTrue:[ 
    ofs := prefix size + 1 .
    (loginRequest at: ofs equals: versionBuildStr) ifTrue:[
      ^ prefix
    ].
    ^ self denyLogin: 'Invalid version in Login request ', loginRequest printString.
  ].
  self denyLogin: 'Expected login request. Instead, received '
						, loginRequest printString
]

{ #category : 'private' }
HostAgent >> validatePeerAddress: tcpSocket [
  tcpSocket ifNotNil:[ | peerAddress peerStr |
		peerStr := tcpSocket peerAddress .
		(peerStr = '::1' or:[ peerStr = '127.0.0.1']) ifTrue:[
			^ tcpSocket .  "handle possible gslist"
		]. 
		peerAddress := tlsActor remoteIpOfFd: tcpSocket id.
		peerIsMyLeafHost := self validatePeerAddrForAccept: peerAddress .
		peerIsMyLeafHost class == Boolean ifTrue:[
			^ tcpSocket 
		].
		tcpSocket close.
		self denyLogin: 'Rejected connection from ', (self addrToString: peerAddress),
													'  ', peerIsMyLeafHost asString .
  ]
]

{ #category : 'private' }
HostAgent >> validatePeerAddrForAccept:  peerAddress [
 "Validate the peer address with respect to the address of the remote
  cache we are servicing, and with respect to known midlevel caches. 
  See also validatePeerIpAddress which validates peer against the certs presented.

  Returns isMyLeafHostBoolean or a String describing the failure ."
  | peerAddrString updateIpsBlk |
  iAmMidCacheAgent ifTrue:[ 
    updateIpsBlk := [  | res |
       res := System _otherCacheAddresses . 
       res add: configuration stoneIpString .
       res
    ].
  ] ifFalse:[
    remoteHostIp = peerAddress ifTrue:[
      ^ true "ok, the configured leaf host we are servicing"
    ].
  ].
  peerAddrString := self addrToString: peerAddress .
  1 to: 2 do: [:loopCount | | cnt |
    1 to: legalCacheIps size do:[:n |
      peerAddrString = (legalCacheIps at: n) ifTrue:[ 
        iAmMidCacheAgent ifTrue:[ ^ false "ok" ] .
        (self checkObjectFilterForMidCache: peerAddrString) ifTrue:[
          ^ false "ok, but not my leaf host"
        ].
        ^ 'different objectFilterByteArray'
      ]
    ].
    cnt := System cacheStatusCount .
    cnt == cacheStatusCount ifTrue:[
      ^ 'invalid peer address' 
    ] ifFalse:[
      legalCacheIps := updateIpsBlk value .
      cacheStatusCount := cnt .
      legalCacheIps do:[:str | GsFile gciLogServer:'legal peer ' , str asString ].
    ]
  ].
  ^ 'invalid peer address after list refresh from stone' 
]

{ #category : 'private' }
HostAgent >> validatePeerIpAddress [
  "validates the peer address against the certificates "
	| certSubnets certSubnet cidr cidrErr |
	certSubnets := tlsActor peerSubnets.
	certSubnets size == 1
		ifFalse: 
			[self
				denyLogin: 'User certificate should contain exactly one subnet specification, but contains '
						, certSubnets size printString].
	certSubnet := certSubnets first.
  [
	  cidr := CidrParser parse: certSubnet.
  ] onException: Error do:[:ex | cidrErr := ex ].
	(cidr == nil or:[ cidr isPetitFailure]) ifTrue:[ | eStr |
    eStr := cidr ifNil:[ cidrErr asString ] ifNotNil:[ cidr printString ]. 
    self
				denyLogin: 'Subnet specification has an invalid format, should be in IPv4 CIDR format, but parser says: ', eStr ].
	(remoteHostIp == nil or:[cidr containsIp: self remoteHostIpInteger])
		ifFalse: 
			[self
				denyLogin: 'Attempt to login from IP address not allowed by the user certificate, which is restricted to subnet '
						, certSubnet asString ]
]

{ #category : 'private' }
HostAgent >> validatePeerStoneName [

	| peerStoneNames peerStoneName |
	peerStoneNames := tlsActor peerStoneNames.
	peerStoneNames size == 1
		ifFalse: 
			[self
				denyLogin: 'User certificate should contain exactly 1 stone name, but contains '
						, peerStoneNames size printString].
	peerStoneName := peerStoneNames first.
	peerStoneName = self myStoneName
		ifFalse: 
			[self
				denyLogin: 'Peer should have presented a certificate for stone '
						, self myStoneName printString , ' but presented a certificate for stone '
						, peerStoneName printString]
]

{ #category : 'private' }
HostAgent >> validatePeerType [
	| certTypes certType expTypes |
	certTypes := tlsActor peerCertificateTypes.
	certTypes size == 1
		ifFalse: 
			[self
				denyLogin: 'User certificate should contain exactly 1 certificate types, but contains '
						, certTypes size printString].
	certType := certTypes first.
  expTypes := iAmMidCacheAgent ifTrue:[ #( 'host' 'user' ) ] ifFalse:[ #( 'user' ) ] .
  (expTypes includes: certType) ifFalse:[
      self
				denyLogin: 'Peer certificate of invalid type ', certType asString 
  ]
]

{ #category : 'private' }
HostAgent >> validatePgsvrThreadRequest: loginRequest [
  "executes in hostagent on mid cache.
   Returns request kind (a Symbol) or an error String. "
  | prefix ofs |
  prefix := #'AllocateMidThread:' .
	(loginRequest at: 1 equals: prefix) ifFalse:[
    prefix := #'StartMidThread:' .
	  (loginRequest at: 1 equals: prefix) ifFalse:[
      prefix := #'StartPagePushRcvr:' .
	    (loginRequest at: 1 equals: prefix) ifFalse:[
        prefix := #'StartWarmerPusher:' .
	      (loginRequest at: 1 equals: prefix) ifFalse:[
          ^ 'Invalid request kind'  
  ]]]].
  ofs := prefix size + 1 .
  (loginRequest at: ofs equals: versionBuildStr) ifTrue:[
    ^ prefix
  ].
  self log:'Invalid version: ' , loginRequest printString.
  ^ 'Invalid version'
]

{ #category : 'validation' }
HostAgent >> validatePortNumber: anInt [
	| max |
	anInt _validateClass: SmallInteger.
	max := (1 bitShift: 16) - 1.
	(anInt between: 0 and: max)
		ifFalse: 
			[(OutOfRange new)
				name: 'portNumber'
					min: 0
					max: max
					actual: anInt;
				signal]
]

{ #category : 'validation' }
HostAgent >> validateQueueLength: anInt [
	| max min |
	anInt _validateClass: SmallInteger.
	max := 256 . min := 5 .  "see limits in hostagent.c prim 1061"
	(anInt between: min and: max)
		ifFalse: 
			[(OutOfRange new)
				name: 'queueLength'
					min: min
					max: max
					actual: anInt;
				signal]
]

{ #category : 'startup' }
HostAgent >> waitForDebug [
  GsFile gciLogServer: (GsProcess stackReportToLevel: 200) .
  System waitForDebug
]

{ #category : 'private' }
HostAgent >> waitForReadReadyOn: tcpSocket timeoutMs: timeMs [
	"Wait for characters to be ready to read from the given socket.
	Deny login on timeout.
	Not for general use; the error reporting is specific to the situation during pre-TLS 
	phases of a login handshake."

	| result |
	[ result := tcpSocket _rawReadWillNotBlockWithin: timeMs .
  ] onException: Error do:[:ex | 
    tcpSocket close  .
    ^ self denyLogin: ex description .
  ].
	result ifNil:[
    tcpSocket close .
    self denyLogin: 'Unknown error waiting for handshake from remote gem.'
  ].
	result ifFalse: [
    tcpSocket close .
    self denyLogin: 'Timeout waiting for handshake from remote gem.'
  ]
]

{ #category : 'private' }
HostAgent >> warmerConfig: aString [
  aString ifNotNil:[ 
    self log:'Using NETLDI_WARMER_ARGS=''' , aString asString , '''' .
    warmerConfigDict := System parseWarmerConfig: aString . 
  ].
]

{ #category : 'startup' }
HostAgent >> warmMidCache [
  warmerConfigDict ifNotNil:[ | otherMidHost nThreads includeData |
    otherMidHost := warmerConfigDict at: #midHost otherwise: nil .  "nil means from stone host"
    otherMidHost ifNotNil:[ 
      (GsSocket hostIsLocalhost: otherMidHost) ifTrue:[ 
        self log:'In NETLDI_WARMER_ARGS , ''-M ', otherMidHost asString,
          '''  specifies localhost,  warming from stone cache.' .
        otherMidHost := nil "cannot warm from our leaf cache, warm from stone cache"
      ]. 
    ].
    nThreads := warmerConfigDict at: #n ifAbsent:[ 2 ].
    includeData := (warmerConfigDict at: #d otherwise: nil) ifNil:[ false ] ifNotNil:[ true ]. 
    
    self midLevelCacheWarmFrom: otherMidHost threads: nThreads 
              includeData: includeData .
  ]
]
