"
Object
   ObjectFilteringPolicy ( name default policyZero allowed prohibited   )

An ObjectFilteringPolicy specifies, for each possible objectSecurityPolicy,
whether objects having that policy should be filtered.
 
Used in object filtering for x509 sessions. An ObjectFilteringPolicy has no
effect unless it is applied by using it in an ObjectFilteringPolicyMap, and
installing that map.

 name       is a String  .
 default    is one of  #allow, #prohibit , and controls transmission
            of objects whose objectSecurityPolicy is not nil and
            is in neither of the allowed or prohibited instVars .
 policyZero is one of  #allow, #prohibit , and controls transmission
            of objects for which (anObject objectSecurityPolicy == nil)  .
 allowed    is an IdentitySet containing instances of GsObjectSecurityPolicy .
 prohibited is an IdentitySet containing instances of GsObjectSecurityPolicy .

 #allow  means objects in that policy may be transmitted to X509 remote caches.
 #prohibit  means objects in that policy will be converted to instances of
     UnAuthorizedObjectStub before transmission to X509 remote caches.

 A default instance allows policies  
   nil , SystemObjectSecurityPolicy, DataCuratorObjectSecurityPolicy ,
   HostAgentUserObjectSecurityPolicy , PublishedObjectSecurityPolicy ,
   GsIndexingObjectSecurityPolicy  
 and all other policies are prohibited by default.
 ObjectFiltersSecurityPolicy and  SecurityDataObjectSecurityPolicy are
 always prohibited in the result of asByteArray .
"
Class {
	#name : 'ObjectFilteringPolicy',
	#superclass : 'Object',
	#instVars : [
		'name',
		'default',
		'policyZero',
		'allowed',
		'prohibited'
	],
	#category : 'X509-Core'
}

{ #category : 'instance creation' }
ObjectFilteringPolicy class >> new [
  "The default new instance has
     policies  nil , SystemObjectSecurityPolicy, DataCuratorObjectSecurityPolicy
     are allowed , and all others are prohibited by default."
  ^ super new initialize
]

{ #category : 'updating' }
ObjectFilteringPolicy >> allow: aGsObjectSecurityPolicy [

  "If aGsObjectSecurityPolicy == nil, it specifies that
   objects for which (anObject objectSecurityPolicy == nil) are allowed"

  aGsObjectSecurityPolicy ifNil:[
    policyZero := #allow . 
  ] ifNotNil:[
    aGsObjectSecurityPolicy class == GsObjectSecurityPolicy ifFalse:[
      ArgumentError new name: 'arg' expectedClass: GsObjectSecurityPolicy 
                actualArg: aGsObjectSecurityPolicy ; signal
    ].
    allowed add: aGsObjectSecurityPolicy .
    prohibited remove: aGsObjectSecurityPolicy otherwise: nil .
  ].
]

{ #category : 'updating' }
ObjectFilteringPolicy >> allowAll: aCollection [
  aCollection do:[:each | self allow: each ].
]

{ #category : 'updating' }
ObjectFilteringPolicy >> allowByDefault [
  "Changes the default value to #allow ."

  default := #allow .
]

{ #category : 'accessing' }
ObjectFilteringPolicy >> asByteArray [
  "Return a ByteArray to be used as argument to HostAgent>>_installObjectFilterArray: .
   Bit values of 1 in the result disallows transmitting objects in the corresponding
   GsObjectSecurityPolicy from stone to a remote cache. 
   Transmission of objects in SecurityDataObjectSecurityPolicy and
   ObjectFiltersSecurityPolicy are always disallowed."
   | ba |
   ba := ByteArray new: 8192 .
   self default == #prohibit ifTrue:[
      1 to: ba size by: 4 do:[:j | ba unsigned32At: j put: 16rFFFFFFFF ].
   ].
   self policyZero == #prohibit ifTrue:[ ba bitAtZ: 0 put: 1 ]
                               ifFalse:[ ba bitAtZ: 0 put: 0 ].
   ba bitAtZ: SecurityDataObjectSecurityPolicy objectSecurityPolicyId put: 1 .
   ba bitAtZ: ObjectFiltersSecurityPolicy objectSecurityPolicyId put: 1 .
   allowed do:[:policy | ba bitAtZ: policy objectSecurityPolicyId put: 0 ].
   prohibited do:[:policy | ba bitAtZ: policy objectSecurityPolicyId put: 1 ].
   ^ ba .
]

{ #category : 'accessing' }
ObjectFilteringPolicy >> default [
  (default == #allow or:[ default == #prohibit]) ifTrue:[ ^ default]. 
  Error signal:'invalid value for default ' , default asString .
]

{ #category : 'initialization' }
ObjectFilteringPolicy >> initialize [
  "The default in a new instances is that
     policies  nil , SystemObjectSecurityPolicy, DataCuratorObjectSecurityPolicy
     are allowed, and all others are prohibited by default."
  
  name := nil .
  allowed := IdentitySet new .
  prohibited := IdentitySet new .
  default := #prohibit .
  policyZero := #allow . "objects for which anObject objectSecurityPolicy == nil "
  self allow: SystemObjectSecurityPolicy "base classes like Object" ; 
    allow: DataCuratorObjectSecurityPolicy "Symbols in AllSymbols";
    allow: HostAgentUserObjectSecurityPolicy "HostAgent UserGlobals, for mid cache ha";
    allow: PublishedObjectSecurityPolicy "the Published dictionary in default symbolLists" ;
    allow: GsIndexingObjectSecurityPolicy "objs implementing indexes" ;
    prohibit: ObjectFiltersSecurityPolicy .
  self objectSecurityPolicy: ObjectFiltersSecurityPolicy .
  ^ self
]

{ #category : 'printing' }
ObjectFilteringPolicy >> mappingReport [
  | str repoSize ba repos res defaultBit |
  repoSize := (repos := SystemRepository) size .
  str := String new .
  ba := self asByteArray .
  self default == #allow ifTrue:[
    res := 'AllowingAll' copy .
    defaultBit := 0 .
  ] ifFalse:[
    res := 'ProhibitAll' copy .
    defaultBit := 1 .
  ].
  (ba bitAtZ: 0) ~~ defaultBit ifTrue:[ str add:' 0:NilPolicy' ].
  1 to: repoSize do:[:n | (ba bitAtZ: n) ~~ defaultBit ifTrue:[ 
      str add: ' ' ; addAll: n asString; add: $: ; addAll:(repos at: n) name 
    ].
  ].
  str size > 0 ifTrue:[ res addAll:' except:( ' ; addAll: str ; add: ' )' ].
  ^ res
]

{ #category : 'accessing' }
ObjectFilteringPolicy >> name [
  ^ name 
]

{ #category : 'updating' }
ObjectFilteringPolicy >> name: aString [
  name := aString .
]

{ #category : 'accessing' }
ObjectFilteringPolicy >> policyZero [
  "Return the value controlling transmission of objects 
    for which (anObject objectSecurityPolicy == nil). "

  (policyZero == #allow or:[ policyZero == #prohibit]) ifTrue:[ ^ policyZero]. 
  Error signal:'invalid value for policyZero ' , policyZero asString .
]

{ #category : 'updating' }
ObjectFilteringPolicy >> prohibit: aGsObjectSecurityPolicy [

  "If aGsObjectSecurityPolicy == nil, it specifies that
   objects for which (anObject objectSecurityPolicy == nil) are prohibited"

  aGsObjectSecurityPolicy ifNil:[
    policyZero := #prohibit .
  ] ifNotNil:[
    aGsObjectSecurityPolicy class == GsObjectSecurityPolicy ifFalse:[
      ArgumentError new name: 'arg' expectedClass: GsObjectSecurityPolicy 
                  actualArg: aGsObjectSecurityPolicy ; signal
    ].
    prohibited add: aGsObjectSecurityPolicy .
    allowed remove: aGsObjectSecurityPolicy otherwise: nil .
  ].
]

{ #category : 'updating' }
ObjectFilteringPolicy >> prohibitAll: aCollection [
  aCollection do:[:each | self prohibit: each ].
]

{ #category : 'updating' }
ObjectFilteringPolicy >> prohibitByDefault [
  "Changes the default value to #prohibit ."

  default := #prohibit .
]
