Class {
	#name : 'GsRowanImageTool',
	#superclass : 'GemStoneRowanTool',
	#category : 'GemStone-Rowan-Tools'
}

{ #category : 'generate gs files' }
GsRowanImageTool >> _generateHostAgentBootstrapGsFromSt: repositoryRoot gsSrcRoot: gsSrcRoot topazFileHeader: topazFileHeader [
	| projectSetDefinition projectSetModification specUrl loadSpec resolvedProject topazFileNameMap visitor doCompile |
	doCompile := false.	"not reading in host agent code in host agent environment, so no compilation"

	specUrl := repositoryRoot / 'GemStone64' / 'rowan' / 'specs'
		/ 'gemstoneHostAgentUser.ston'.
	(loadSpec := RwSpecification fromUrl: 'file:' , specUrl pathString)
		projectsHome: repositoryRoot;
		diskUrl: 'file:' , (repositoryRoot / 'GemStone64') pathString;
		yourself.
	resolvedProject := loadSpec resolve
		compileWhileReading: doCompile;
		yourself.

	[ projectSetDefinition := resolvedProject readProjectSet ]
		on: CompileWarning
		do: [ :ex | 
			| str |
			((str := ex asString) includesString: 'not optimized')
				ifTrue: [ ex resume ]
				ifFalse: [ 
					GsFile gciLogServer: str.
					ex pass ] ].

	projectSetModification := projectSetDefinition
		compareAgainstBase: RwProjectSetDefinition new.
	topazFileNameMap := Dictionary new.
	resolvedProject packageNames
		do: [ :packageName | (topazFileNameMap at: packageName ifAbsentPut: [ {} ]) add: packageName ].
	visitor := GsModificationTopazWriterVisitor new
		logCreation: true;
		repositoryRootPath: gsSrcRoot;
		topazFilenamePackageNamesMap: topazFileNameMap;
		topazFileHeader: topazFileHeader;
		yourself.
	visitor visit: projectSetModification.
	^ true
]

{ #category : 'generate gs files' }
GsRowanImageTool >> _generateSystemUserBootstrapGsFromSt: repositoryRoot doCompile: doCompile gsSrcRoot: gsSrcRoot topazFileHeader: topazFileHeader [
	| platformConfigurationAttributes projectSetModification resolvedProject topazFileNameMap visitor gemStoneRowanPackageNames filein4PackageNames loadSpec specUrl projectSetDefinition calculatedGsFilenames expectedGsFilenames combinedPackageNames |
	platformConfigurationAttributes := {'common'.
	'gemstone'.
	'bootstraponly'.
	'compilerClasses'.
	'filein3'.
	'filein4'.
	'obsoleteClasses'}.

	specUrl := repositoryRoot / 'rowan' / 'specs' / 'gemstoneBaseImage.ston'.
	(loadSpec := RwSpecification fromUrl: 'file:' , specUrl pathString)
		projectsHome: repositoryRoot;
		diskUrl: 'file:' , repositoryRoot pathString;
		yourself.

	resolvedProject := loadSpec resolve
		compileWhileReading: doCompile;
		yourself.

	[ 
	projectSetDefinition := resolvedProject
		readProjectSet: platformConfigurationAttributes ]
		on: CompileWarning
		do: [ :ex | 
			| str |
			((str := ex asString) includesString: 'not optimized')
				ifTrue: [ ex resume ]
				ifFalse: [ 
					GsFile gciLogServer: str.
					ex pass ] ].

	projectSetModification := projectSetDefinition
		compareAgainstBase: RwProjectSetDefinition new.

	gemStoneRowanPackageNames := #('GemStone-Rowan-Extensions-Tools' 'GemStone-Rowan-Tools').
	filein4PackageNames := #('Filein4-CompilerClasses' 'Filein4-ObsoleteClasses' 'Filein4Rowan').
	combinedPackageNames := gemStoneRowanPackageNames, filein4PackageNames.
	topazFileNameMap := Dictionary new
		at: 'Filein4' put: filein4PackageNames asSet;
		yourself.
	resolvedProject packageNames
		do: [ :packageName | 
			(combinedPackageNames includes: packageName)
				ifFalse: [ 
					"write one package per file, except for the GemStone-Rowan and Filein4 packages. NOTE: GemStone-Rowan written out separately below"
					(topazFileNameMap at: packageName ifAbsentPut: [ {} ]) add: packageName ] ].
	calculatedGsFilenames := topazFileNameMap keys sort asArray.
	expectedGsFilenames := self _systemUserBootstrapFilenamesOrdered sort asArray.
	calculatedGsFilenames = expectedGsFilenames
		ifFalse: [ 
			self
				error:
					'There is a mismatch between the calculated bootstrap gs file names '
						, calculatedGsFilenames printString
						, ' and the expected bootstrap .gs file names '
						, expectedGsFilenames printString
						,
							'. Edit the list in GsRowanImageTool>>_systemUserBootstrapFilenamesOrdered.' ].
	visitor := GsModificationTopazWriterVisitor new
		fileNamesInFileInOrder: self _systemUserBootstrapFilenamesOrdered;
		logCreation: true;
		repositoryRootPath: gsSrcRoot;
		topazFilenamePackageNamesMap: topazFileNameMap;
		excludeClassInitializers: true;
		topazFileHeader: topazFileHeader;
		yourself.
	visitor visit: projectSetModification.

	"GemStone-Rowan .gs file should only put removeall* commands after class definitions - no upgrade of extent0.rowan.dbf ... yet"
	topazFileNameMap := Dictionary new
		at: 'GemStone-Rowan' put: gemStoneRowanPackageNames asSet;
		yourself.
	visitor := RwGsModificationTopazWriterVisitorV2 new
		logCreation: true;
		repositoryRootPath: gsSrcRoot;
		topazFilenamePackageNamesMap: topazFileNameMap;
		topazFileHeader: topazFileHeader;
		yourself.
	visitor visit: projectSetModification.

	^ true
]

{ #category : 'generate gs files' }
GsRowanImageTool >> _generateX509CoreBootstrapGsFromSt: repositoryRoot doCompile: doCompile gsSrcRoot: gsSrcRoot topazFileHeader: topazFileHeader [
	| platformConfigurationAttributes projectSetDefinition projectSetModification specUrl loadSpec resolvedProject topazFileNameMap visitor |
	platformConfigurationAttributes := {'common'.
	'x509'}.

	specUrl := repositoryRoot / 'rowan' / 'specs' / 'gemstoneBaseImage.ston'.
	(loadSpec := RwSpecification fromUrl: 'file:' , specUrl pathString)
		projectsHome: repositoryRoot;
		diskUrl: 'file:' , repositoryRoot pathString;
		yourself.
	resolvedProject := loadSpec resolve
		compileWhileReading: doCompile;
		yourself.

	[ 
	projectSetDefinition := resolvedProject
		readProjectSet: platformConfigurationAttributes ]
		on: CompileWarning
		do: [ :ex | 
			| str |
			((str := ex asString) includesString: 'not optimized')
				ifTrue: [ ex resume ]
				ifFalse: [ 
					GsFile gciLogServer: str.
					ex pass ] ].

	projectSetModification := projectSetDefinition
		compareAgainstBase: RwProjectSetDefinition new.
	topazFileNameMap := Dictionary new.
	((Rowan projectNamed: 'gemstoneBaseImage') _loadedProject components components
		at: 'x509/Kernel') packageNames
		do: [ :packageName | 
			"write one package per file"
			(topazFileNameMap at: packageName ifAbsentPut: [ {} ]) add: packageName ].
	visitor := GsModificationTopazWriterVisitor new
		logCreation: true;
		repositoryRootPath: gsSrcRoot;
		topazFilenamePackageNamesMap: topazFileNameMap;
		topazFileHeader: topazFileHeader;
		yourself.
	visitor visit: projectSetModification.
	^ true
]

{ #category : 'generate gs files' }
GsRowanImageTool >> _systemUserBootstrapFilenamesOrdered [
	"The order that bootstrap .gs files are loaded needs to be 
		known so that the placement of the remove all methods 
		commands is correct ... remove all methods commands 
		are inserted before the first occurence of a class in the 
		bootstrap .gs files, whether a method definition or class 
		definition"

	^ {'Filein1A'.
	'Filein1A-BootstrapOnly'.
	'Filein1B'.
	'Filein1C'.
	'Filein1C-BootstrapOnly'.
	'Filein1D-ObsoleteClasses'.
	'Filein2A'.
	'Filein2CInit'.
	'Filein2Traits'.
	'Filein2Streams'.
	'Filein3A'.
	'Filein3B'.
	'Filein3B-BootstrapOnly'.
	'Filein3C-ObsoleteClasses'.
	'Filein3D-CompilerClasses'.
	'Filein3E-CompilerClasses'.
	'Filein3Init'.
	'Filein4'.
	}
]

{ #category : 'bootstrap' }
GsRowanImageTool >> adoptGemStone64: specUrl diskUrl: diskUrl projectsHome: projectsHome [

	self adoptGemStone64: specUrl diskUrl: diskUrl projectsHome: projectsHome forUpgrade: false

]

{ #category : 'bootstrap' }
GsRowanImageTool >> adoptGemStone64: specUrl diskUrl: diskUrl projectsHome: projectsHome forUpgrade: forUpgrade [
	"
	Create loaded project (if needed), traverse the package definitions and 
		create loaded packages for each.
	"

	| tracer wasTracing |
	tracer := Rowan projectTools trace.
	wasTracing := tracer isTracing.
	tracer startTracing.

	[ 
	Rowan projectTools adopt
		adoptProjectFromUrl: specUrl
		diskUrl: diskUrl
		projectsHome: projectsHome ]
		on: RwAuditMethodErrorNotification
		do: [ :ex | 
			(ex description = 'Missing loaded method' and: [ forUpgrade ])
				ifTrue: [ 
					| theBehavior |
					"missing loaded method during upgrade, means the method is no longer
						part of image, or was added by end user. Remove the method, for now"
					tracer trace: 'Removing method ' , ex methodPrintString.
					theBehavior := Rowan globalNamed: ex className.
					ex isMetaclass
						ifTrue: [ theBehavior := theBehavior class ].
					theBehavior removeSelector: ex selector.
					ex resume: false	"no audit error" ]
				ifFalse: [ 
					"issue audit error"
					ex resume: true ] ].
	System commit
]

{ #category : 'bootstrap' }
GsRowanImageTool >> createAndPopulateUnPackagedPackage: forUpgrade [
	"
	Then package the unpackaged classes an methods into an unpackaged
		package so that the ENTIRE image is packaged. The UnPackaged 
		should be empty at the end of slow filein ... upgradeImage will be
		expected to manage the UnPackaged package differently.
	"

	| project packagePrefix componentName loadedProject unpackagedName adoptTool userName tracer wasTracing symbolList ignoredSymbolDicts |
	tracer := Rowan projectTools trace.
	wasTracing := tracer isTracing.
	tracer startTracing.

	project := Rowan newProjectNamed: Rowan unpackagedProjectName.
	componentName := Rowan unpackagedProjectName.
	project
		packageConvention: 'Rowan';
		addNewComponentNamed: componentName.
	packagePrefix := Rowan unpackagedPackagePrefix.
	userName := System myUserProfile userId.
	symbolList := Rowan image symbolList.
	ignoredSymbolDicts := (UserGlobals at: #PACKAGE_OBSOLETE_COMPILER_CLASSES ifAbsent: [ false ])
		ifTrue: [ #(GemStone_Portable_Streams GemStone_Legacy_Streams) ]
		ifFalse: [ #(GemStone_Portable_Streams GemStone_Legacy_Streams ObsoleteClasses GsCompilerClasses) ].
	symbolList
		do: [ :symbolDictionary | 
			(ignoredSymbolDicts includes: symbolDictionary name)
				ifTrue: [
					tracer
						trace: '---Ignoring Unpackaged symbol dictionary ' , symbolDictionary name ]
				ifFalse: [ 
					| thePackageName |
					"create unpackaged packages for each symbol dictionary"
					thePackageName := packagePrefix , symbolDictionary name asString.
					tracer
						trace: '---Creating Unpackaged package ' , thePackageName printString.
					project
						packageNamed: thePackageName
						ifAbsent: [ 
							project
								addPackageNamed: thePackageName
								toComponentNamed: componentName
								gemstoneDefaultSymbolDictionaryForUser:
									userName -> symbolDictionary name asString ] ] ].
	System commit.
	loadedProject := project load first.	"load the projec"
	System commit.
	unpackagedName := Rowan unpackagedName.
	adoptTool := Rowan packageTools adopt.
	Rowan image symbolList
		do: [ :symbolDictionary | 
			(ignoredSymbolDicts includes: symbolDictionary name)
				ifFalse: [ 
					| thePackage thePackageName |
					thePackageName := packagePrefix , symbolDictionary name asString.
					thePackage := project packageNamed: thePackageName.
					tracer
						trace:
							'---Adopting Unpackaged classes and methods for package '
								, thePackageName printString.
					self
						_classesIn: symbolDictionary
						do: [ :aClass | 
							aClass rowanProjectName = unpackagedName
								ifTrue: [ 
									tracer trace: '	Unpackaged Class ' , aClass name asString printString.
									adoptTool adoptClass: aClass intoPackageNamed: thePackageName ]
								ifFalse: [ 
									| instanceSelectors classSelectors unpackageMethods |
									instanceSelectors := Set new.
									classSelectors := Set new.
									unpackageMethods := false.
									aClass
										methodsDo: [ :selector :method | 
											method rowanProjectName = unpackagedName
												ifTrue: [ 
													tracer
														trace:
															'	Unpackaged method ' , aClass name asString , ' >> ' , selector printString.
													instanceSelectors add: selector.
													unpackageMethods := true ] ].
									aClass class
										methodsDo: [ :selector :method | 
											method rowanProjectName = unpackagedName
												ifTrue: [ 
													tracer
														trace:
															'	Unpackaged method ' , aClass name asString , ' class >> '
																, selector printString.
													classSelectors add: selector.
													unpackageMethods := true ] ].
									unpackageMethods
										ifTrue: [ 
											adoptTool
												adoptClassExtension: aClass
												instanceSelectors: instanceSelectors
												classSelectors: classSelectors
												intoPackageNamed: thePackageName ] ] ].
					System commit ] ].
	wasTracing
		ifFalse: [ tracer stopTracing ]
]

{ #category : 'generate gs files' }
GsRowanImageTool >> generateBootstrapGsFromSt [
	| doCompile |
	doCompile := SessionTemps current at: #'ROWAN_COMPILE_WHILE_READING'
		otherwise: true.
	^ self generateBootstrapGsFromSt: doCompile
]

{ #category : 'generate gs files' }
GsRowanImageTool >> generateBootstrapGsFromSt: doCompile [
	| archBase objBase mtype repositoryRoot gsSrcRoot topazFileHeader  sess symbolList |
	SessionTemps current at: #'ROWAN_TRACE' put: nil.	"#gciLogServer "
	(archBase := System gemEnvironmentVariable: 'ARCHBASE')
		ifNil: [ Error signal: 'ARCHBASE not defined in environment' ].
	(objBase := System gemEnvironmentVariable: 'OBJBASE')
		ifNil: [ Error signal: 'OBJBASE not defined in environment' ].
	(mtype := System gemEnvironmentVariable: 'MTYPE')
		ifNil: [ Error signal: 'MTYPE not defined in environment' ].
	repositoryRoot := (archBase , '/image') asFileReference.
	gsSrcRoot := objBase , '/slow' , mtype , '/bootstrap'.	"where .gs files are written"
	topazFileHeader := '! Copyright (C) GemTalk Systems 1986-', Date today year printString, '.  All Rights Reserved.
'.
	"add ObsoleteClasses and GsCompilerClasses symbol dicts to transient symbol list
		in order to compile all of the methods in the image"
	sess := GsCurrentSession currentSession.
	sess _transientSymbolList ifNil: [ System refreshTransientSymbolList ].
	symbolList := sess transientSymbolList.
	symbolList
		insertObject: (Globals at: #'ObsoleteClasses') at: 1;
		insertObject: (Globals at: #'GsCompilerClasses') at: 1;
		yourself.
	[ 
	GsModificationTopazWriterVisitor startTrackingClassNames.
	self
		_generateSystemUserBootstrapGsFromSt: repositoryRoot
			doCompile: doCompile
			gsSrcRoot: gsSrcRoot
			topazFileHeader: topazFileHeader;
		_generateX509CoreBootstrapGsFromSt: repositoryRoot
			doCompile: doCompile
			gsSrcRoot: gsSrcRoot
			topazFileHeader: topazFileHeader;
		_generateHostAgentBootstrapGsFromSt: repositoryRoot
			gsSrcRoot: gsSrcRoot
			topazFileHeader: topazFileHeader ]
		ensure: [ GsModificationTopazWriterVisitor stopTrackingClassNames ]
]

{ #category : 'generateGs command line' }
GsRowanImageTool >> generateGsCommandLine [
	| stdout bootstrapDir |
	stdout := GsFile stdout.
	stdout
		lf;
		nextPutAll: 'using ARCHBASE=' , '$ARCHBASE' asFileReference pathString;
		lf;
		lf.
	(self generateGsCommandLine_processArguments: stdout)
		ifFalse: [ 
			"help displayed or unknown argument"
			^ self ].
	bootstrapDir := '$ARCHBASE/image/bootstrap' asFileReference.
	bootstrapDir deleteAllChildren.	"do the dirty work and generate the .gs files"
	Rowan gemstoneTools image generateBootstrapGsFromSt.

	stdout
		nextPutAll: 'Generated files';
		lf;
		nextPutAll: bootstrapDir pathString;
		lf;
		yourself.
	stdout
		nextPutAll:
				(System
						performOnServer:
							'cd ' , bootstrapDir pathString , '; ls -l ' , bootstrapDir pathString);
		lf
]

{ #category : 'generateGs command line' }
GsRowanImageTool >> generateGsCommandLine_processArguments: stdout [
	"process command line arguments display help ... answer false if help has been displayed (i.e., exit without generating .gs files)"

	| args scriptArgStart |
	args := System commandLineArguments.
	scriptArgStart := args indexOf: '--'.
	scriptArgStart > 0
		ifTrue: [ 
			| descriptionBlock usageBlock argIndex argsSize |
			descriptionBlock := [
					stdout
						nextPutAll: '  Regenerate $ARCHBASE/image/bootstrap/*.gs files from the tonel files *.st.'; lf;
						nextPutAll: '    Compile each method before writing the .gs files, if a compile error'; lf;
						nextPutAll: '    occurs, you will be able to use the debugger and get a description of'; lf;
						nextPutAll: '    the error.'; lf;
						nextPutAll: '  If you hit a syntax error, you have no choice but to manually edit the'; lf;
						nextPutAll: '    file and fix the error.'; lf;
						nextPutAll: '  If you hit an undefined variable error, you can choose to skip the method'; lf;
						nextPutAll: '    compile altogether (using the -n or --no-compile command line options).'; lf;
						nextPutAll: '    A good choice if you are certain that the variable is defined, i.e, you''ve'; lf;
						nextPutAll: '    already compiled the methods using topaz or jadeite.'; lf;
						nextPutAll: '  If you have manually edited tonel files, it is prudent to compile all of the'; lf;
						nextPutAll: '    methods. Add the list of undefined variables to the command line, where'; lf;
						nextPutAll: '    they will be added to UserGlobals and allow the methods to compile.'; lf;
						lf;
						yourself ].
				usageBlock := [
					stdout
						nextPutAll: 'SYNOPSIS:'; lf;
						nextPutAll: '	generateGs.sh [-n | --no-compile] [-h | --help]'; lf;
						nextPutAll: '	generateGs.sh [UNDEFINED-VARIABLE]...'; lf;
						lf;
						nextPutAll: 'DESCRIPTION';lf;
						nextPutAll: '	Add UNDEFINED-VARIABLE to UserGlobals before compiling methods.'; lf;
						lf;
						nextPutAll: '	-h, --help'; lf;
						nextPutAll: '		Display help and exit'; lf;
						lf;
						nextPutAll: '	-n, --no-compile'; lf;
						nextPutAll: '		Write method source directly to disk without compiling methods.'; lf;
						lf;
						nextPutAll: 'EXAMPLES:'; lf;
						nextPutAll: '	generateGs.sh                     # regenerate .gs files compiling all methods'; lf;
						nextPutAll: '	generateGs.sh --no-compile        # regenerate .gs files skipping method compile'; lf;
						nextPutAll: '	generateGs.sh newIvar NewClass    # add variables to UserGlobals and compile methods'; lf;
						nextPutAll: '	generateGs.sh -h                  # print help text'; lf;
						yourself.
					false ].
			argIndex := scriptArgStart + 1.	"arg after initial --"
			argsSize := args size.
			[ argIndex <= argsSize ] whileTrue: [ 
				| arg |
				arg := args at: argIndex.
				argIndex := argIndex + 1.
				(arg = '--help') | (arg = '-h')
					ifTrue: [ 
						descriptionBlock value.
						^ usageBlock value ].
				(arg = '--no-compile') | (arg = '-n')
					ifTrue: [ SessionTemps current at: #ROWAN_COMPILE_WHILE_READING put: false ]
					ifFalse: [
						(arg beginsWith: '-')
							ifTrue: [
								stdout lf; nextPutAll: 'UNKNOWN argument ', arg printString; lf; lf.
								^ usageBlock value ]
							ifFalse: [
								"add arg to UserGlobals"
								UserGlobals at: arg asSymbol put: nil ] ] ] ].
	"proceed with .gs file generation"
	^ true
]

{ #category : 'repository' }
GsRowanImageTool >> newRepositoryRoot: repositoryRoot forProjectNamed: projectName [
	"change the repositoryRoot and then load from disk, includes enbedded projects"

	| project |
	project := Rowan projectNamed: projectName.
	^ project repositoryRoot: repositoryRoot
]

{ #category : 'repository' }
GsRowanImageTool >> newRepositoryRoot: repositoryRoot platformConditionalAttributes: platformConditionalAttributes forProjectNamed: projectName [
	"change the repositoryRoot and then load from disk, includes enbedded projects"

	| project |
	project := Rowan projectNamed: projectName.
	^ project
		repositoryRoot: repositoryRoot
		platformConditionalAttributes: platformConditionalAttributes
]

{ #category : 'repository' }
GsRowanImageTool >> newRepositoryRoot: repositoryRoot platformConditionalAttributes: platformConditionalAttributes instanceMigrator: instanceMigrator forProjectNamed: projectName [
	"change the repositoryRoot and then load from disk, includes enbedded projects"

	| project |
	project := Rowan projectNamed: projectName.
	^ project
		repositoryRoot: repositoryRoot
		platformConditionalAttributes: platformConditionalAttributes
		instanceMigrator: instanceMigrator
]

{ #category : 'packages' }
GsRowanImageTool >> readRewriteGemStone64Packages: archBase [
	"
		Rowan gemstoneTools image readRewriteGemStone64Packages: '/home/dhenrich/work/j_36x/'
	"

	| repositoryRoot platformConfigurationAttributes specUrl loadSpec resolvedProject |
	SessionTemps current at: #'ROWAN_TRACE' put: nil.	"#gciLogServer "
	repositoryRoot := archBase , '/image'.

	platformConfigurationAttributes := {'common'.
	'gemstone'.
	'bootstraponly'}.

	specUrl := repositoryRoot asFileReference / 'rowan' / 'specs'
		/ 'GemStone64.ston'.
	(loadSpec := RwSpecification fromUrl: 'file:' , specUrl pathString)
		projectsHome: repositoryRoot;
		diskUrl: 'file:' , repositoryRoot;
		yourself.
	resolvedProject := loadSpec resolve.

	[ resolvedProject read: platformConfigurationAttributes ]
		on: CompileWarning
		do: [ :ex | 
			| str |
			((str := ex asString) includesString: 'not optimized')
				ifTrue: [ ex resume ]
				ifFalse: [ 
					GsFile gciLogServer: str.
					ex pass ] ].

	resolvedProject packages
		do: [ :packageDef | 
			| classExtensions |
			"merge class extensions into class definitions and remove class dextension"
			classExtensions := packageDef classExtensions.
			packageDef classDefinitions
				do: [ :classDef | 
					classExtensions
						at: classDef name
						ifPresent: [ :classExtension | 
							classExtension instanceMethodDefinitions
								do: [ :meth | classDef addInstanceMethodDefinition: meth ].
							classExtension classMethodDefinitions
								do: [ :meth | classDef addClassMethodDefinition: meth ].
							packageDef removeClassExtensionDefinition: classExtension ] ] ].

	resolvedProject exportPackages
]
