Extension { #name : 'ObsoleteDateTime' }

{ #category : 'Instance Creation' }
ObsoleteDateTime class >> _checkFormat: anArray [

""

"anArray is a format Array as documented in ObsoleteDateTime>>asStringUsingFormat:"

| v doTime |

anArray _validateClass: Array.
(anArray size < 8 or:[ anArray size > 10])
  ifTrue:[ ObsoleteDateTime _error: #rtErrBadFormatSpec args: { anArray } ].

"Check for a combination of the digits 1, 2, and 3"
((anArray at: 1) + (anArray at: 2) + (anArray at: 3) = 6 and:
        [(anArray at: 1) * (anArray at: 2) * (anArray at: 3) = 6])
  ifFalse:[ ObsoleteDateTime _error: #rtErrBadFormatSpec args: { anArray } ].

(anArray at: 4) _validateClass: Character.

((v := anArray at: 5) = 1 or: [v = 2 or: [v = 3]])
  ifFalse:[ ObsoleteDateTime _error: #rtErrBadFormatSpec args: { anArray } ].

((anArray at: 6) = 1 or: [(anArray at: 6) = 2])
  ifFalse:[ ObsoleteDateTime _error: #rtErrBadFormatSpec args: { anArray } ].

(doTime := anArray at: 8) _validateClass: Boolean.
doTime ifTrue:[
  anArray size = 10
    ifFalse:[ ObsoleteDateTime _error: #rtErrBadFormatSpec args: { anArray } ].
  (anArray at: 7) _validateClass: Character.
  (anArray at: 9) _validateClass: Boolean.
  (anArray at: 10) _validateClass: Boolean.
  ]

]

{ #category : 'Repository Conversion' }
ObsoleteDateTime class >> _correspondingNewClass [

"The class all instances of receiver are converted to during conversion."

(self == ObsoleteDateTime)
  ifTrue: [ ^ DateTime ].

^ self.

]

{ #category : 'Instance Creation' }
ObsoleteDateTime class >> _getMonthFrom: aCharCollection [

"Reads a month name or abbreviation from aCharCollection and returns the
 integer corresponding to the month of the year."

| whichMonth monthArray argSize matchMonth |

monthArray := MonthNames value.
matchMonth:= [:monthStr | | strSize match i |
   i:= 1.
   match:= false.
   strSize:= monthStr size.
   [ ((i <= argSize) and:[i <= strSize]) and:
     [match:= (aCharCollection at: i) isEquivalent: (monthStr at: i)]]
   whileTrue: [
      i:= i + 1.
   ].
   match
].

   argSize:= aCharCollection size.
   whichMonth:= 1.
   [ (whichMonth <= 12) and:
     [(matchMonth value: (monthArray at: whichMonth)) not]
   ]
   whileTrue:
      [whichMonth := whichMonth + 1].

   (whichMonth <= 12)
      ifTrue: [ ^whichMonth].
   ^ 0

]

{ #category : 'Instance Creation' }
ObsoleteDateTime class >> _newJulianDay: anInteger second: anotherInteger [

"New instances cannot be created for obsolete classes."

^ self _error: #rtObsoleteClass.

]

{ #category : 'Instance Creation' }
ObsoleteDateTime class >> _newWithYear: yearInt month: monthInt day: dayInt seconds: secondInt [

"New instances cannot be created for obsolete classes."

^ self _error: #rtObsoleteClass.

]

{ #category : 'Instance Creation' }
ObsoleteDateTime class >> fromStream: aStream [

"New instances cannot be created for obsolete classes."

^ self _error: #rtObsoleteClass.

]

{ #category : 'Instance Creation' }
ObsoleteDateTime class >> fromStream: aStream usingFormat: anArray [

"New instances cannot be created for obsolete classes."

^ self _error: #rtObsoleteClass.

]

{ #category : 'Instance Creation' }
ObsoleteDateTime class >> fromString: aString [

"New instances cannot be created for obsolete classes."

^ self _error: #rtObsoleteClass.

]

{ #category : 'Instance Creation' }
ObsoleteDateTime class >> fromString: aString usingFormat: anArray [

"New instances cannot be created for obsolete classes."

^ self _error: #rtObsoleteClass.

]

{ #category : 'Storing and Loading' }
ObsoleteDateTime class >> loadFrom: passiveObj [

"Reads from passiveObj the passive form of an object with named instance
 variable format.  Converts the object to its active form by loading the
 information into a new instance of the receiver.  Returns the new instance."

^ DateTime loadFrom: passiveObj

]

{ #category : 'Instance Creation' }
ObsoleteDateTime class >> new [

"Disallowed.  To create a new ObsoleteDateTime, use one of the other instance creation
 methods listed here."

self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
ObsoleteDateTime class >> new: anInteger [

"Disallowed.  To create a new ObsoleteDateTime, use one of the other instance creation
 methods listed here."

self shouldNotImplement: #new:

]

{ #category : 'Instance Creation' }
ObsoleteDateTime class >> newWithYear: yearInt month: monthInt day: dayInt hours: hourInt
  minutes: minuteInt seconds: secondInt [

"New instances cannot be created for obsolete classes."

^ self _error: #rtObsoleteClass.

]

{ #category : 'Instance Creation' }
ObsoleteDateTime class >> now [

"New instances cannot be created for obsolete classes."

^ self _error: #rtObsoleteClass.

]

{ #category : 'New Indexing Comparison' }
ObsoleteDateTime >> _classSortOrdinal [

^ 59

]

{ #category : 'Formatting' }
ObsoleteDateTime >> _monthAbbrev: aMonthInt [

"Given the month of the year as a SmallInteger, returns a three-letter string
 that is its abbreviation."

|theMonth itsAbbrev|

theMonth := self _monthName: aMonthInt.  "get its full name"
itsAbbrev := String new.
1 to: 3 do: "take the first three letters"
   [:aChar | itsAbbrev := itsAbbrev + (theMonth at: aChar)].
^ itsAbbrev

]

{ #category : 'Formatting' }
ObsoleteDateTime >> _monthName: aMonthInt [

"Given the month of the year as a SmallInteger, returns a String that is its
 full name."

^ (MonthNames value) at: aMonthInt

]

{ #category : 'Accessing' }
ObsoleteDateTime >> _yearMonthDay [

"Returns a three-element Array of SmallIntegers containing the year, index of
 the month, and the day of the month of the Gregorian calendar calculated from
 the Julian day of the receiver."

^ self asDateTime _yearMonthDay

]

{ #category : 'Comparing' }
ObsoleteDateTime >> < aObsoleteDateTime [

"Returns true if the receiver represents a ObsoleteDateTime before the argument, false
 if it doesn't.  Generates an error if the argument is not a ObsoleteDateTime."

| t |

(self julianDay = (t := aObsoleteDateTime julianDay))
   ifTrue: [ ^ self julianSecond < aObsoleteDateTime julianSecond ].
^ self julianDay < t

]

{ #category : 'Comparing' }
ObsoleteDateTime >> = aObsoleteDateTime [

"Returns true if the receiver represents the same ObsoleteDateTime as the argument,
 false if it doesn't."

aObsoleteDateTime class == self class ifFalse: [ ^false ].
^ (self julianDay = aObsoleteDateTime julianDay) and:
   [self julianSecond = aObsoleteDateTime julianSecond ]

]

{ #category : 'Comparing' }
ObsoleteDateTime >> > aObsoleteDateTime [

"Returns true if the receiver represents a ObsoleteDateTime after the argument, false
 if it doesn't.  Generates an error if the argument is not a ObsoleteDateTime."

| t |

(self julianDay = (t := aObsoleteDateTime julianDay))
   ifTrue: [ ^self julianSecond > aObsoleteDateTime julianSecond ].
^self julianDay > t

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> addDays: anInteger [

"Returns a ObsoleteDateTime anInteger days after the receiver."

^ (self class) _newJulianDay: (self julianDay + anInteger)
                      second: (self julianSecond)

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> addHours: anInteger [

"Returns a ObsoleteDateTime anInteger hours after the receiver."

^ self addSeconds: anInteger * 3600

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> addMinutes: anInteger [

"Returns a ObsoleteDateTime anInteger minutes after the receiver."

^ self addSeconds: anInteger * 60

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> addMonths: anInteger [

"Returns a ObsoleteDateTime anInteger months after the receiver.  This method attempts
 to keep the day of the month the same.  If the new month has fewer days than
 the receiver's original month, the method will truncate to the last day of the
 new month."

| t newMonth newYear |

t := self _yearMonthDay.
newMonth := ((t at: 2) + anInteger) .
newYear := (t at: 1) .
(newMonth == 0) ifTrue:[ newYear := newYear - 1 ].
^ (self class) _newWithYear: newYear
               month: newMonth
               day: (t at: 3)
               seconds: (self julianSecond)

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> addSeconds: anInteger [

"Returns a ObsoleteDateTime anInteger seconds after the receiver."

^ (self class) _newJulianDay: (self julianDay)
                      second: (self julianSecond + anInteger)

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> addWeeks: anInteger [

"Returns a ObsoleteDateTime anInteger weeks after the receiver."

^ self addDays: anInteger * 7

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> addYears: anInteger [

"Returns a ObsoleteDateTime anInteger years after the receiver."

| t |

t := self _yearMonthDay.
^ (self class) _newWithYear: ((t at: 1) + anInteger)
                  month: (t at: 2)
                    day: (t at: 3)
                seconds: (self julianSecond)

]

{ #category : 'Converting' }
ObsoleteDateTime >> asDateTime [

"Returns an instance of DateTime with the date and time contained in
 the receiver."

| aDateTime aTimeZone |

aTimeZone := TimeZone current.
aDateTime := (self transformIntoDateTime) subtractSeconds: (aTimeZone secondsFromGmt).

(aDateTime isDstIn: aTimeZone)
  ifTrue:  [ ^ aDateTime subtractSeconds: (aTimeZone secondsForDst)]
  ifFalse: [ ^ aDateTime ].

]

{ #category : 'Converting' }
ObsoleteDateTime >> asParts [

"Returns an Array of SmallIntegers (year month day hours minutes
 seconds) corresponding to the receiver."

| t |

t := self _yearMonthDay.  "year/month/day"
t addLast: self julianSecond // 3600.  "hours"
t addLast: self julianSecond \\ 3600 // 60.  "minutes"
t addLast: self julianSecond \\ 60.  "seconds"
^ t

]

{ #category : 'Converting' }
ObsoleteDateTime >> asSeconds [

"Returns an Integer representing the receiver in units of seconds
 since 1 March 1583."

^ self julianDay - 2299298 * 86400 + self julianSecond

  "86400 = seconds/24-hour day"
  "converts to March 1, 1583, Gregorian"

]

{ #category : 'Formatting' }
ObsoleteDateTime >> asString [

"Returns a String representing the receiver.  The String is in
 the format DD/MM/YYYY HH:MM:SS, with hours in military time."

| t julSec |

t := self _yearMonthDay.
julSec := self julianSecond.
^ (t at: 3) _digitsAsString
  addAll: '/';
  addAll: (t at: 2) _digitsAsString;
  addAll: '/';
  addAll: (t at: 1) _digitsAsString;
  addAll: ' ';
  addAll: (julSec // 3600) _digitsAsString;
  addAll: ':';
  addAll: (julSec \\ 3600 // 60) _digitsAsString;
  addAll: ':';
  addAll: (julSec \\ 60) _digitsAsString

]

{ #category : 'Formatting' }
ObsoleteDateTime >> asStringUsingFormat: anArray [

"Returns a String representing the receiver.  The format of the String is
 determined by anArray, whose elements are described in the following table.

 Generates an error if anArray does not contain all required elements or
 if the value of any Array element does not conform to the requirements set
 forth in the following table.  anArray must contain at least 8 elements if
 time is not printed, and must contain 10 elements otherwise.

 ObsoleteDateTime String Format

 Element       Acceptable Value        Explanation

 1st,2nd,3rd   The integers 1, 2, 3,   Determines the position of day,
               in any order            month, year.  1 corresponds
                                       to day, 2=month, 3=year.

 4th           A separator Character   Separates year, month, and day.
               (preceded by $)

 5th           Month format            With asStringUsingFormat:
                                       1 = print as number
                                       2 = print 3-letter abbreviation
                                       3 = print entire name

                                       With fromStreamUsingFormat: or
                                       fromString:usingFormat:
                                       1 = decode a number to determine
                                           the month
                                       2 or 3 = decode a character string to
                                           determine the month

 6th           Year format             With asStringUsingFormat:
                                       1 = print entire year
                                       2 = print last 2 digits (year mod 100)

                                       With fromStreamUsingFormat: or
                                       fromString:usingFormat:
                                       1 = read entire year
                                       2 = read only the last 2 digits

 7th           A separator Character   Separates hours, minutes, and seconds.
               (preceded by $)

 8th           true or false           If true, print or read the time as
                                       indicated by the 7th, 9th and 10th
                                       elements.  If false, ignore those
                                       elements, and the 9th and 10th elements
                                       are optional.

 9th           true or false           Whether or not to print or read seconds.

 10th          true or false           true = print (read) time in 12-hour
                                       format, with am or pm (such as
                                       1:30:55 pm).  The space is required
                                       preceding the am or pm indicator.

                                       false = print (read) time in 24-hour
                                       format (such as 13:30:55)"

|t julSec dateSeparator timeSeparator monthName aString
 hour hourInt min sec day year|

t := self _yearMonthDay.
julSec := self julianSecond.

ObsoleteDateTime _checkFormat: anArray.

dateSeparator := (anArray at: 4) asString.

timeSeparator := (anArray at: 7) asString.

((anArray at: 5) = 2) "get the month name according to the format"
   ifTrue: [monthName := self _monthAbbrev: (t at: 2)]
   ifFalse: [((anArray at: 5) = 3) "month as number is default"
      ifTrue: [monthName := self _monthName: (t at: 2)]
      ifFalse: [monthName := (t at: 2) _digitsAsString]].

((anArray at: 6) = 2)
   ifTrue: [year := ((t at: 1) \\ 100) _digitsAsString]
   ifFalse: [year := (t at: 1) asString].  "YYYY is default"

day := (t at:3) _digitsAsString.
((anArray at: 1) = 2) "month first"
   ifTrue: [aString := monthName + dateSeparator]
   ifFalse: [((anArray at: 1) = 3) "year first"
      ifTrue: [aString := year + dateSeparator]
      ifFalse: [aString := day + dateSeparator]].  "day first is default"

((anArray at: 2) = 1) "day second"
   ifTrue: [aString addAll: day; addAll: dateSeparator] "year second"
   ifFalse: [((anArray at: 2) = 3) "month second is default"
      ifTrue: [aString addAll: year; addAll: dateSeparator]
      ifFalse: [aString addAll: monthName; addAll: dateSeparator]].

((anArray at: 3) = 1) "day third"
   ifTrue: [aString addAll: day]
   ifFalse: [((anArray at: 3) = 2) "month third"
      ifTrue: [aString addAll: monthName]
      ifFalse: [aString addAll: year]].  "year third is default"

hourInt := julSec // 3600.
hour := hourInt _digitsAsString.
min := (julSec \\ 3600 // 60) _digitsAsString.
sec := (julSec \\ 60) _digitsAsString.

(anArray at: 8) ifTrue: [ "print the time"
  aString add: $ .
  (anArray at: 10) ifTrue: [ "12-hour format"
    (hourInt > 12) ifTrue: [
      aString addAll: (hourInt - 12) _digitsAsString;
              addAll: timeSeparator;
              addAll: min.

      (anArray at: 9) ifTrue: [
        aString addAll: timeSeparator;
                addAll: sec
        ].
      ]
    ifFalse: [
      aString addAll: (hourInt = 0 ifTrue: ['12'] ifFalse: [hour]);
              addAll: timeSeparator;
              addAll: min.

      (anArray at: 9) ifTrue: [
        aString addAll: timeSeparator;
                addAll: sec.
        ].
      ].

    aString addAll: (hourInt >= 12 ifTrue: [' pm'] ifFalse: [' am']).
    ]
  ifFalse: [
    aString addAll: hour;
            addAll: timeSeparator;
            addAll: min.

    (anArray at: 9) ifTrue: [
      aString addAll: timeSeparator;
              addAll: sec.
      ].
    ].
  ].

^ aString

]

{ #category : 'Accessing' }
ObsoleteDateTime >> at: anIndex put: aValue [

"Disallowed.  You may not change the value of a ObsoleteDateTime."

self shouldNotImplement: #at:put:

]

{ #category : 'Repository Conversion' }
ObsoleteDateTime >> convertToDateTime [

"Converts an instance of ObsoleteDateTime (GemStone 4.1 and earlier) to an
 instance of current DateTime.  The object identity is left the same.
 Returns the receiver.

 The receiver is assumed to represent a date and time in the local time
 zone and is converted to a DateTime in Greenwich Mean Time."

| newDt |

newDt := self asDateTime.
newDt become: self.
^ self.

]

{ #category : 'Accessing' }
ObsoleteDateTime >> dayOfMonth [

"Returns the day of the month as a SmallInteger."

^  (self _yearMonthDay) at: 3

]

{ #category : 'Accessing' }
ObsoleteDateTime >> dayOfWeek [

"Returns the numeric index of the day of the week (1-7), where 1 = Sunday."

^ self julianDay - 2299295 - 1 \\ 7 + 1

  "the Julian day 2299298 is converted to the Gregorian"
  "date of March 1, 1583 by Communications of the ACM #199 algorithm"
  "was March 1, 1583 a Thursday?"

]

{ #category : 'Accessing' }
ObsoleteDateTime >> dayOfYear [

"Returns the day of the year as a SmallInteger."

|t y |

t := self _yearMonthDay.
y := 0.
1 to: ((t at: 2) - 1) do: [:month |
   ((month = 1) or: [(month = 3) or: [(month = 5) or: [(month = 7) or:
   [(month = 8) or: [(month = 10) or: [(month = 12)]]]]]])
      ifTrue: [y := y + 31].
   ((month = 4) or: [(month = 6) or: [(month = 9) or: [(month = 11)]]])
      ifTrue: [y := y + 30].
   (month = 2)
      ifTrue:[(self leap)
         ifTrue: [y := y + 29]
         ifFalse: [y := y + 28]]].
^ y + (t at: 3)

]

{ #category : 'Accessing' }
ObsoleteDateTime >> daysInMonth [

"Returns the number of days in the month represented by the receiver."

|t month|

t := self _yearMonthDay.
month := t at: 2.
((month = 1) or: [(month = 3) or: [(month = 5) or: [(month = 7) or:
   [(month = 8) or: [(month = 10) or: [(month = 12)]]]]]])
   ifTrue: [^ 31].
((month = 4) or: [(month = 6) or: [(month = 9) or: [(month = 11)]]])
   ifTrue: [^ 30].
(self leap)
   ifTrue: [^ 29].
^ 28

]

{ #category : 'Accessing' }
ObsoleteDateTime >> daysInYear [

"Returns the number of days in the year represented by the receiver."

(self leap)
   ifTrue: [^ 366].
^ 365

]

{ #category : 'Comparing' }
ObsoleteDateTime >> hash [

"Returns an Integer hash code for the receiver."

^ ((self julianDay hash) bitShift: -1) bitXor: (self julianSecond hash)

]

{ #category : 'Accessing' }
ObsoleteDateTime >> hours [

"Returns the hours since midnight (0-23)."

^ self julianSecond // 3600

]

{ #category : 'Accessing' }
ObsoleteDateTime >> julianDay [

"Returns the Julian Day, a SmallInteger representing the number of days since
 January 1, 4713 BC (as defined in Communications of the ACM algorithm #199)."

^ self asDateTime julianDay.

]

{ #category : 'Accessing' }
ObsoleteDateTime >> julianSecond [

"Returns a SmallInteger representing the number of seconds since midnight."

^ self asDateTime seconds

]

{ #category : 'Accessing' }
ObsoleteDateTime >> leap [

"Returns true if the receiver represents a leap year and false if it does not."

| yr |
  "a year is a leap year if: (it is evenly divisible by 4 and it is not a
   century year) or (it is a century year and evenly divisible by 400)"

yr := (self _yearMonthDay) at: 1.
((yr \\ 100) = 0)
   ifTrue: [^ ((yr \\ 400) = 0)].
^ ((yr \\ 4) = 0)

]

{ #category : 'Accessing' }
ObsoleteDateTime >> minutes [

"Returns the minutes since the hour (0-59)."

^ self julianSecond \\ 3600 // 60

]

{ #category : 'Accessing' }
ObsoleteDateTime >> monthName [

"Returns the name of the month (a String) in the user's native language."

^ MonthNames value at: ((self _yearMonthDay) at: 2)

]

{ #category : 'Accessing' }
ObsoleteDateTime >> monthOfYear [

"Returns the numeric index of the month (1-12)."

^ (self _yearMonthDay) at: 2

]

{ #category : 'Formatting' }
ObsoleteDateTime >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

aStream nextPut: $' .
aStream nextPutAll: self asString .
aStream nextPut: $'

]

{ #category : 'Accessing' }
ObsoleteDateTime >> seconds [

"Returns the seconds since the minute (0-59)."

^ self julianSecond \\ 60

]

{ #category : 'Accessing' }
ObsoleteDateTime >> size: anInteger [

"Disallowed.  You may not change the size of a ObsoleteDateTime."

self shouldNotImplement: #size:

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> subtractDate: aObsoleteDateTime [

"Returns the number of days between the receiver and aObsoleteDateTime."

^ (self julianDay - aObsoleteDateTime julianDay) abs

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> subtractDays: anInteger [

"Returns a ObsoleteDateTime anInteger days before the receiver."

^ self addDays: anInteger negated

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> subtractHours: anInteger [

"Returns a ObsoleteDateTime anInteger hours before the receiver."

^ self addSeconds: anInteger * 3600 negated

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> subtractMinutes: anInteger [

"Returns a ObsoleteDateTime anInteger minutes before the receiver."

^ self addSeconds: anInteger * 60 negated

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> subtractMonths: anInteger [

"Returns a ObsoleteDateTime anInteger months before the receiver.  As with addMonths:,
 this method attempts to keep the day of the month the same.  If the new month
 has fewer days than the receiver's original month, the method will truncate to
 the last day of the new month."

^ self addMonths: anInteger negated

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> subtractSeconds: anInteger [

"Returns a ObsoleteDateTime anInteger seconds before the receiver."

^ self addSeconds: anInteger negated

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> subtractTime: adate [

"Subtracts the given ObsoleteDateTime's hours/minutes/seconds from the
 receiver's and returns the result in an Array of {hours. minutes. seconds}."

| parts h m s |
parts := self asParts.
h := parts at: 4.
m := parts at: 5.
s := parts at: 6.
parts := adate asParts.
s := s - (parts at: 6).
m := m - (parts at: 5).
h := h - (parts at: 4).
s < 0 ifTrue: [
  s := s + 60.
  m := m - 1
].
s > 60 ifTrue: [
  s := s - 60.
  m := m + 1
].
m < 0 ifTrue: [
  m := m + 60.
  h := h - 1
].
m > 60 ifTrue: [
  m := m - 60.
  h := h + 1
].
^{ h . m . s }

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> subtractWeeks: anInteger [

"Returns a ObsoleteDateTime anInteger weeks before the receiver."

^ self addDays: anInteger * 7 negated

]

{ #category : 'Arithmetic' }
ObsoleteDateTime >> subtractYears: anInteger [

"Returns a ObsoleteDateTime anInteger years before the receiver."

^ self addYears: anInteger negated

]

{ #category : 'Accessing' }
ObsoleteDateTime >> timeAsSeconds [

"Returns the number of seconds since midnight represented by the receiver."

^ self julianSecond

]

{ #category : 'Private' }
ObsoleteDateTime >> transformIntoDateTime [

"Returns an instance of DateTime with the date and time contained in
 the receiver."

^ self transformIntoDateTime: (TimeZone current).

]

{ #category : 'Formatting' }
ObsoleteDateTime >> US12HrFormat [

"Returns a string representation of the receiver in the form `m/d/y hh:mm'
 with 12-hour time representation."

^self asStringUsingFormat: #(2 1 3 $/ 1 2 $: true false true)

]

{ #category : 'Formatting' }
ObsoleteDateTime >> US24HrFormat [

"Returns a string representation of the receiver in the form `m/d/y hh:mm'
 with 24-hour time representation."

^self asStringUsingFormat: #(2 1 3 $/ 1 2 $: true false false)

]

{ #category : 'Formatting' }
ObsoleteDateTime >> USDateFormat [

"Returns a string representation of the receiver in the form `m/d/y'."

^self asStringUsingFormat: #(2 1 3 $/ 1 2 $: false )

]

{ #category : 'Accessing' }
ObsoleteDateTime >> weekDayName [

"Returns the name of the day of the week (a String) in the user's native
 language."

^ WeekDayNames value at: (self dayOfWeek)

]

{ #category : 'Storing and Loading' }
ObsoleteDateTime >> writeTo: passiveObj [

"Converts the receiver to its passive form and writes that information on
 passiveObj."

passiveObj writeClass: self class.
self _yearMonthDay do: [:each |  each writeTo: passiveObj].
self julianSecond writeTo: passiveObj.
passiveObj space

]

{ #category : 'Accessing' }
ObsoleteDateTime >> year [

"Returns the calendar year as an integer."

^ (self _yearMonthDay) at: 1

]
