Extension { #name : 'GsMethod' }

{ #category : 'Debugging Support' }
GsMethod class >> _addMarkerIds: anArray [

""

| placesMarked markerLine markerLineSize space addToEnd markPosition
  aStr neededSize subStr |

placesMarked:= anArray at: 1.
markerLine:= anArray at: 2.
space:= Character space.

"have the source marked at each error with ^; now add marker identifier"
addToEnd:= false.
1 to: (placesMarked size) do: [:i |
   markPosition:= (placesMarked at: i) at: 1.
   aStr:= ((placesMarked at: i) at: 2) asString.
   neededSize:= markPosition + aStr size.
   markerLineSize := markerLine size .
   (markerLineSize < neededSize) ifTrue: [
       markerLine size: neededSize.
       markerLineSize + 1 to: neededSize do: [:k | markerLine at: k put: space].
       markerLineSize:= neededSize.
   ].

   (addToEnd) ifFalse: [
      subStr:= markerLine copyFrom: markPosition + 1
                                to: (markPosition + aStr size).
      subStr do: [:each | (each == $ ) ifFalse: [addToEnd := true]].
   ].
   (addToEnd) ifTrue: [
       markerLine add: aStr.
       (i == placesMarked size) ifFalse: [ markerLine add: ',']
   ] ifFalse: [  | destIdx |
      destIdx := markPosition + 1 .
      markerLine replaceFrom: destIdx to: destIdx + aStr size - 1 with: aStr startingAt: 1 .
   ]
].
(68 - markerLine size) timesRepeat:[ markerLine add: $ ].
(75 - markerLine size) timesRepeat:[ markerLine add: $* ] .
markerLine add: Character lf.
^ true

]

{ #category : 'Debugging Support' }
GsMethod class >> _buildMarkedSourceFrom: sourceStr sourceSize: aSize markers: markerArray [

"Given a source string, its size (passed in for efficiency), and a marker
 Array, returns an instance of sourceStr's class containing the marked source."

| lineFeed tab space
  placesMarked     "an Array of markers marked on the current line"
  markerLineIndex "index into the current marker line"
  result          "the result of this method"
  markerLine      "the current marker line"
  aChar           "one Character of the source"
  displayWidth    "the number of positions it takes to display aChar"
  lineSz
|

"initialize"
lineFeed := Character lf.
tab:= Character tab.
space:= Character space.

placesMarked:= { } .
markerLineIndex:= 1.
result:= sourceStr class new .
result addAll: '   ' .
lineSz := 0 .
markerLine:= String new .
markerLine add: $  .
1 to: aSize do: [:i |
   aChar:= sourceStr at: i.  "fetch a char"
   displayWidth:= aChar displayWidth.

   "Add the char to the result"
   (aChar == tab)
   ifTrue: [
      displayWidth:= 8 - (lineSz \\ 8).
      displayWidth timesRepeat: [result add: space].
      lineSz := lineSz + displayWidth .
   ]
   ifFalse: [
      result add: aChar.
      lineSz := lineSz + displayWidth .
      ((i == aSize) and: [aChar ~~ lineFeed]) ifTrue: [
        result add: lineFeed .
      ].
   ].

   ((markerArray at: i) == nil) "no marker at this position"
   ifTrue: [
      displayWidth timesRepeat: [markerLine add: space].
   ]
   ifFalse: [ "found an error at this position"
      placesMarked add: { markerLineIndex + 1 . markerArray at: i }.
      markerLine add: $^ .
      displayWidth - 1 timesRepeat: [markerLine add: space].
   ].
   markerLineIndex:= markerLineIndex + displayWidth.

   ((aChar == lineFeed) or: [i == aSize])
   ifTrue: [ "we are at end of line"
      "add error identifiers to marker line "
      (placesMarked size ~~ 0) ifTrue: [
         self _addMarkerIds: { placesMarked . markerLine . markerLineIndex }.
         result add: $  ; add: $* .
         result add: markerLine.
         ] .

      (i == aSize) ifFalse: [
         result addAll: '   ' .
         lineSz := 0 .
      ].
      markerLine size: 1.
      markerLineIndex:= 1.
      placesMarked size: 0.
   ]
].
^result

]

{ #category : 'Debugging Support' }
GsMethod class >> _buildMarkersFrom: sourceOffsets ofSize: sizeArg [

"Given an Array of source offsets, build an Array of size sizeArg containing
 the index into anArray at the position corresponding to anArray's element.
 The remainder of the Array contains nil.  Negative offsets denote disabled
 breakpoints."

| markerArray anOffset posOffset aSize |
aSize := 1 max: sizeArg .                          "fix bug 14976"
markerArray:= Array new: aSize.
1 to: sourceOffsets size do: [:i |
  anOffset := sourceOffsets at: i .
  anOffset == nil ifFalse:[
    posOffset:= (anOffset abs max: 1) min: aSize.  "limit within range"
    (markerArray at: posOffset) == nil ifTrue:[
       anOffset < 0 ifTrue:[ markerArray at: posOffset put: i negated ]
		    ifFalse:[ markerArray at: posOffset put: i ]
       ]
    ]
  ].
^markerArray

]

{ #category : 'Debugging Support' }
GsMethod class >> _sourceWithErrors: compilerError fromString: aString [

"This method returns an instance of aString's class containing the text in a
 string with compiler errors marked, plus the error text for each error.

 The argument compilerError is the result Array from either the
 Behavior | compileMethod:dictionaries:category:  method or the
 GsMethod | _recompileWithSource: method.

 The argument aString is the source string which was an input to either the
 Behavior | compileMethod:dictionaries:category: method or the
 GsMethod | _recompileWithSource: method."

| lineFeed result aStringSize offsets errNumbers thisErr pos
  markerArray errDict errMsgs auxMsgs errsz |

"initialize"
lineFeed := Character lf.
"only ask for source size once for efficiency for Japanese"
aStringSize:= aString size.
offsets := Array new: (errsz := compilerError size) .
errNumbers := Array new: errsz .
errMsgs := Array new: errsz .
auxMsgs := Array new: errsz .

"get an Array of source offsets with errors, and an Array of error numbers"
1 to: errsz do: [:i |
   thisErr := compilerError at: i.
   offsets at: i put: (thisErr at: 2 "source offset").
   errNumbers at: i put: (thisErr at: 1 "error number") .
   thisErr size >= 3 ifTrue:[
     errMsgs at: i put: (thisErr at: 3"error message String") .
     thisErr size >= 4 ifTrue:[
       auxMsgs at: i put: (thisErr at: 4 "additional message text")
       ].
     ].
   ].

"build an Array parallel to the source that contains nil if no error at
 that source position, and an index into offsets if there is an error at
 that source position"
markerArray:= self _buildMarkersFrom: offsets ofSize: aStringSize.

result:= self _buildMarkedSourceFrom: aString
                          sourceSize: aStringSize
                             markers: markerArray.

"add error strings"
errDict := GemStoneError at: System myUserProfile nativeLanguage.
1 to: errNumbers size do: [:i | | msg |
  result add: lineFeed.
  result addAll: i asString.
  result addAll: ': ['.
  pos := errNumbers at: i.
  result addAll: pos asString.
  result addAll: '] '.
  msg := errMsgs at: i .
  msg == nil ifTrue:[
    pos > errDict size
      ifTrue: [ msg := '(unknown error number)']
      ifFalse: [ msg := (errDict at: pos) asString].
    ].
  result addAll: msg .
  (auxMsgs at: i) ~~ nil ifTrue:[ result addAll: (auxMsgs at: i) ].
  ].
result add: lineFeed.

^result

]

{ #category : 'Instance Creation' }
GsMethod class >> new [

"Disallowed.  You cannot create new instances of GsMethod."

self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
GsMethod class >> new: anInteger [

"Disallowed.  You cannot create new instances of GsMethod."

self shouldNotImplement: #new:

]

{ #category : 'Debugging Support' }
GsMethod >> _allBreakpoints [

 "Returns nil if no method breakpoints set in the receiver."

^ nil

]

{ #category : 'CodeModification Override' }
GsMethod >> _at: anIndex put: aValue [

self _validatePrivilege.
^ super _at: anIndex put: aValue

]

{ #category : 'CodeModification Override' }
GsMethod >> _basicAt: anIndex put: aValue [

self _validatePrivilege.
^ super _basicAt: anIndex put: aValue

]

{ #category : 'Debugging Support' }
GsMethod >> _breakPointKind: anIp [

"This method infers the kind of action associated with a given bytecode."

self error:'not implemented'

]

{ #category : 'Debugging Support' }
GsMethod >> _buildIpMarkerArray [

"This method builds a marker Array for the receiver's source code string.
 containing IPs of all step points.

 The result Array is the same size as the source string and
 contains IP numbers at offsets corresponding to the source string."

| srcOffsets ipsArr srcSize mrkSize markerArray bias |

srcOffsets := self _sourceOffsets  .  "source offsets of the step points"
ipsArr := self _ipSteps .             "relative IPs of each step point"
srcSize := self sourceString size .

mrkSize := 1 max: srcSize .                          "fix bug 14976"
markerArray:= Array new: mrkSize .
bias := self class instSize .
1 to: srcOffsets size do: [:i | | anOffset anIp posOffset|
  anOffset := srcOffsets at: i .
  anIp := (ipsArr at: i ) + bias .
  posOffset := (anOffset abs max: 1) min: mrkSize.  "limit within range"
  (markerArray at: posOffset) == nil ifTrue:[
     markerArray at: posOffset put: anIp
  ]
].
^markerArray

]

{ #category : 'Debugging Support' }
GsMethod >> _buildMarkerArray: allSteps ofSize: aSize [

"This method builds a marker Array for the receiver's source code string.

 allSteps == true , show all steps
          == false, show steps where a breakpoint currently exists
          a SmallInteger, show just that step point

 The result Array is the same size as the source string and
 contains step numbers at offsets corresponding to the source string."

| srcOffsets stepToDisplay |

srcOffsets := self _sourceOffsets  .
(allSteps _isSmallInteger ) ifTrue:[
    stepToDisplay := allSteps .
    1 to: srcOffsets size do:[ :j |
       j == stepToDisplay ifFalse:[ srcOffsets at: j put: nil ].
    ]
  ]
  ifFalse:[
    allSteps ifFalse:[ self _setBreakpointsInSourceOffsets: srcOffsets ].
  ].

^ self class _buildMarkersFrom: srcOffsets ofSize: aSize

]

{ #category : 'Reporting' }
GsMethod >> _classAndSelectorNameWidth: anInt [

"Return a String of the form className | selector with the className substring
 padded to width anInt."

"Used by ProfMonitor"

|text sel|
selector == nil
  ifTrue:[sel := 'unbound method']
  ifFalse:[ sel := selector].
inClass == nil
  ifTrue: [ text := String withAll: 'executed code'.  ]
  ifFalse: [ text := String withAll: inClass name  ].
text width: anInt; addAll: ' >> '; addAll: sel .
^ text

]

{ #category : 'Debugging Support' }
GsMethod >> _debugInfoHeaderSize [

""

^ 4

]

{ #category : 'Accessing' }
GsMethod >> _inClass [

"Returns the class in which this method was compiled."

^ inClass

]

{ #category : 'Debugging Support' }
GsMethod >> _ipForStepPoint: aStepPoint [

"Return zero-based offset of instruction relative to the first indexable
 instance variable in the portable code."

| offset aStep |

aStepPoint == 0 ifTrue:[ ^ 0 "method entry"] .
aStep := aStepPoint abs .
aStepPoint > self _numIpSteps ifTrue:[ ^ nil ].

offset := self _numArgsAndTemps + self _debugInfoHeaderSize .
^ debugInfo at: (offset + aStepPoint)

]

{ #category : 'Accessing' }
GsMethod >> _ipSteps [

"Returns an Array containing the step points for the portable code for the
 method."

| offset |
offset := self _debugInfoHeaderSize + 1 + self _numArgsAndTemps  .
^ debugInfo copyFrom: offset to: (offset + self _numIpSteps - 1)

]

{ #category : 'Private' }
GsMethod >> _isProtected [

^ self _dnuError: #_isProtected args: #() reason: 1

]

{ #category : 'Debugging Support' }
GsMethod >> _lineNumberForIp: targetIp [

"Returns the line number in the receiver's source string for the
 specified IP value within the receiver.
 Assumes that the IP is from a frame that is Not at top of Stack."

| stepPoint |
stepPoint := self _previousStepPointForIp: targetIp quick: false .
^ self _lineNumberForStep: stepPoint

]

{ #category : 'Debugging Support' }
GsMethod >> _lineNumberForStep: aStepPoint [

"Returns the line number in the receiver's source string for the
 step point with number aStepPoint.
 Returns 1 if aStepPoint is out of range."

| sourceOffset offset lf  lineNumber|
sourceOffset := self _sourceOffsetsAt: aStepPoint .
sourceOffset == nil ifTrue:[ ^ 1 ].

" find the first end-of-line which is at or after the sourceOffset of the step point"
lf := Character lf .
offset := 1 .
lineNumber := 0 .
[ offset <= sourceOffset ] whileTrue:[
  lineNumber := lineNumber + 1 .
  offset := sourceString indexOf: lf startingAt: offset .
  offset == 0 ifTrue:[ ^ lineNumber ].
  offset := offset + 1 .
  ].
^ lineNumber .

]

{ #category : 'Debugging Support' }
GsMethod >> _lineNumberForTosIp: targetIp [

"Returns the line number in the receiver's source string for the
 specified IP value within the receiver.
 Assumes that the IP is from a frame that IS at top of Stack ."

| stepPoint |
stepPoint := self _nextStepPointForIp: targetIp quick: false .
^ self _lineNumberForStep: stepPoint

]

{ #category : 'Decompiling without Sources' }
GsMethod >> _litArrayString [

"Returns a String containing an Array-builder production describing the literal
 pool of the receiver."

| result lastSize aLit j literalsSize |
literalsSize := self size - literalsOffset .
literalsSize <= 0 ifTrue:[ ^ ' #() ' ] .

result := String new .
result addAll: '{ ' .
lastSize := result size .
j := 1 + literalsOffset .
literalsSize timesRepeat:[
  aLit := self at: j .
  (aLit isKindOf: SymbolAssociation) ifTrue:[
       result addAll: true _asSource ; "flag next as key of a literal variable"
              addAll: ' . ' ;
              addAll: aLit key _asSource.
  ] ifFalse:[
       "a block or other kind of literal"
       aLit == true ifTrue:[ self _halt:'true not expected as a literal' ].
       result addAll: aLit _asSource
  ] .
  lastSize := result size .
  result addAll: ' . ' ; add: Character lf ; addAll: '   '.
  j := j + 1 .
].
result size: lastSize .  "remove the last $. "
result addAll:' } ' .
^ result

]

{ #category : 'Class organizer support' }
GsMethod >> _literalsIncludesValue: anObject [

^ self literals includesIdentical: anObject

]

{ #category : 'Accessing' }
GsMethod >> _nArgs [

"Returns the number of arguments expected by the method."

^ numArgs

]

{ #category : 'Debugging Support' }
GsMethod >> _nextStepPointForIp: anIp quick: isQuick [

"Returns the offset in the ipSteps Array of the receiver which represents
 the step point at or just after the given instruction pointer offset.  If
 anIp is after the last step point, returns an integer one greater than
 the last step point.

 The anIp argument is a zero-based offset relative to first named instance
 variable in the receiver."

"This method is useful when doing low-level debugging of the virtual machine."

| numIpSteps aStep offset j  bias posIp |
numIpSteps := self _numIpSteps .
offset := self _numArgsAndTemps + self _debugInfoHeaderSize .

"search backwards so that if anIp points at the last element of an in-line
 send cache, we get the right answer.

 steps in debugInfo are zero based relative to first instruction in portable
 code ."

bias := self class instSize .
posIp := anIp abs .
(isQuick and:[ numIpSteps > 50]) ifTrue:[
  aStep := (debugInfo at: (offset + numIpSteps - 50)) + bias .
  posIp < aStep ifTrue:[ ^ nil ].
  ].
j := numIpSteps.
[j >= 1] whileTrue:[
  aStep := (debugInfo at: (offset + j )) + bias .
  aStep <= posIp ifTrue:[
     aStep == posIp ifTrue:[ ^ j ].
     ^ j + 1 .
     ] .
  j := j - 1 .
  ].
^ 1 .

]

{ #category : 'Debugging Support' }
GsMethod >> _numArgs [

""

^ debugInfo at: 2

]

{ #category : 'Debugging Support' }
GsMethod >> _numArgsAndTemps [

""

^ debugInfo at: 1

]

{ #category : 'Debugging Support' }
GsMethod >> _numIpSteps [

""

^ debugInfo at: 3

]

{ #category : 'Debugging Support' }
GsMethod >> _numSourceOffsets [

""

^ debugInfo at: 4

]

{ #category : 'Disassembly' }
GsMethod >> _opcodeInfo: instrWord [

"instrWord must be a SmallInteger from the body of A GsMethod, containing
 a valid opcode.  If the instrWord represents a currently installed breakpoint,
 the result is for the instruction that the breakpoint is replacing.

 returns a SmallInteger with the following bit fields
           16rFF  instruction size in words , including the opcode word and any
	  	  in-line literal words , an unsigned 8-bit int
       16rFFFF00  instVar offset for instVarAccess
      16r1000000  boolean, 1 means opcode is an instVarAccess
      16r2000000  boolean, 1 means opcode is a pushBlock
   16rFFF0000000  opcode
 "
"primitive 536 no longer in VM"
Error signal:'_opcodeInfo:  not implemented'.

]

{ #category : 'Debugging Support' }
GsMethod >> _previousStepPointForIp: anIp quick: isQuick [

"Returns the offset in the ipSteps Array of the receiver which represents
 the step point preceding the given instruction pointer offset.

 The anIp argument is a zero-based offset relative to first named instance
 variable in the receiver."

"This method is useful when doing low-level debugging of the virtual machine."

| numIpSteps aStep offset bias posIp |
numIpSteps := self _numIpSteps .
offset := self _numArgsAndTemps + self _debugInfoHeaderSize .

"search forwards.
 steps in debugInfo are zero based relative to first instruction in portable
 code ."

bias := self class instSize .
posIp := anIp abs .
(isQuick and:[ numIpSteps > 50]) ifTrue:[
  aStep := (debugInfo at: (offset + 50 )) + bias .
  posIp > aStep ifTrue:[ ^ nil ].
  ].

1 to: numIpSteps do:[ :j |
  aStep := (debugInfo at: (offset + j )) + bias .
  aStep >= posIp ifTrue:[
     j > 1 ifTrue:[ ^ j - 1 ].
     ^ j
     ] .
  ].
^ numIpSteps

]

{ #category : 'CodeModification Override' }
GsMethod >> _primitiveAt: anIndex put: aValue [

self _validatePrivilege.
^ super _primitiveAt: anIndex put: aValue

]

{ #category : 'Accessing' }
GsMethod >> _selector [

"Returns the value of the instance variable named selector."

^ selector

]

{ #category : 'Reporting' }
GsMethod >> _selectorPool [

"Return a SymbolSet containing the selectors sent by the receiver."

| instr result |

result := SymbolSet new .
1 to: literalsOffset - 1 do:[:j |
  (instr := self at: j) _isSmallInteger ifFalse:[
     instr _isSymbol ifTrue:[ result add: instr ].
     ].
  ].
^ result

]

{ #category : 'Debugging Support' }
GsMethod >> _sendCount: anIp [

"This method returns the number of arguments (excluding receiver)
 associated with the message send at this offset."

self error:'not implemented'

]

{ #category : 'Debugging Support' }
GsMethod >> _sourceAtIp: anIp [

"Return the source string with marker for step point closest to
 the specified IP .
 Assumes that the IP is from a frame that is Not at top of Stack."

| aStep |
aStep := self _previousStepPointForIp: anIp quick: false .
^ self _sourceWithSteps: aStep

]

{ #category : 'Debugging Support' }
GsMethod >> _sourceAtTosIp: anIp [

"Return the source string with marker for step point closest to
 the specified IP .
 Assumes that the IP is from a frame that IS at top of Stack."

| aStep |
aStep := self _nextStepPointForIp: anIp quick: false .
^ self _sourceWithSteps: aStep

]

{ #category : 'Reporting' }
GsMethod >> _sourceOffsetOfFirstSendOf: aSymbol [

"Returns the source offset of the step point for the first send in the receiver
 that sends aSymbol.  If the receiver is not a sender of aSymbol, or if aSymbol
 is not a Symbol, returns nil."

1 to: literalsOffset - 1 do:[:j |
  (self at: j) == aSymbol ifTrue:[
     ^ self _sourceOffsetOfSendAt: j .
     ].
  ].
^ nil

]

{ #category : 'Debugging Support' }
GsMethod >> _sourceOffsetOfSendAt: instrOffset [

"Returns the source offset of the step point for the send using the selector
 found at instrOffset.

 The instrOffset argument is one-based relative to first indexable instance
 variable."

| ipOffset stepPoint |

ipOffset := instrOffset + self class instSize "convert to zero-based" .
stepPoint := self _previousStepPointForIp: ipOffset quick: false .
^ self _sourceOffsetsAt: stepPoint

]

{ #category : 'Debugging Support' }
GsMethod >> _sourceOffsets [

"Returns an InvariantArray (that holds SmallIntegers) which is a list
 of offsets into sourceString, corresponding in order to the step points."

| offset numIpSteps numSrcOffsets |
numIpSteps := self _numIpSteps .
numSrcOffsets := self _numSourceOffsets .
(numIpSteps == 0 or:[ numSrcOffsets == 0])
   ifTrue:[ ^ #( 1 ) ].  "handle certain primitive methods"

offset :=
  self _debugInfoHeaderSize + 1 + self _numArgsAndTemps + numIpSteps .
^ debugInfo copyFrom: offset to: (offset + numSrcOffsets - 1)

]

{ #category : 'Debugging Support' }
GsMethod >> _sourceOffsetsAt: aStepPoint [

"Returns the source offset for the step point with number aStepPoint.
 Returns nil if aStepPoint is out of range."

| numIpSteps numSrcOffsets |
numIpSteps := self _numIpSteps .
numSrcOffsets := self _numSourceOffsets .
(numIpSteps == 0 or:[ numSrcOffsets == 0]) ifTrue:[
  "handle certain primitive methods, whose source offsets is #( 1 ) "
  aStepPoint == 1 ifFalse:[ ^ nil ].
  ^ 1
  ].

(aStepPoint < 1 or:[ aStepPoint > numSrcOffsets ]) ifTrue:[ ^ nil ].

^ debugInfo at:
  self _debugInfoHeaderSize + aStepPoint + self _numArgsAndTemps + numIpSteps .

]

{ #category : 'Accessing' }
GsMethod >> _sourceString [

"Returns the value of the instance variable sourceString."

^ sourceString

]

{ #category : 'Accessing' }
GsMethod >> _sourceStringWithFileName [

^ self sourceString

]

{ #category : 'Debugging Support' }
GsMethod >> _sourceWithStepIps [

"This method returns the source string with intermixed control information
 indicating IP values of each step point.
 "

| markerArray "A parallel Array to the source string.  It is filled with nils
               except for locations pointed to by the sourceOffsets Array."
  aSize "the size of the source"
  result |

aSize:= sourceString size.
  "only ask for the source size once for efficiency for Japanese"
markerArray := self _buildIpMarkerArray .
result := self class _buildMarkedSourceFrom: sourceString
                             sourceSize: aSize
                                markers: markerArray .
^ result

]

{ #category : 'Debugging Support' }
GsMethod >> _sourceWithSteps: allSteps [

"This method returns the source string with intermixed control information
 indicating where step points are.

 allSteps == true , show all steps
          == false, show steps where a breakpoint currently exists
          a SmallInteger, show just that step point
 "

| markerArray "A parallel Array to the source string.  It is filled with nils
               except for locations pointed to by the sourceOffsets Array."
  aSize "the size of the source"
  result |

aSize:= sourceString size.
  "only ask for the source size once for efficiency for Japanese"
markerArray := self _buildMarkerArray: allSteps ofSize: aSize.
result := self class _buildMarkedSourceFrom: sourceString
                             sourceSize: aSize
                                markers: markerArray .
				"deleted   'reduce garbage' code"
^ result

]

{ #category : 'Debugging Support' }
GsMethod >> _stepPointForIp: ipOffset level: aLevel quick: isQuick [

""
"ipOffset is zero-based relative to first named instance variable."

aLevel == 1 ifTrue:[  "top of stack"
   ^ self _nextStepPointForIp: ipOffset quick: isQuick
    ]
 ifFalse:[
   ^ self _previousStepPointForIp: ipOffset quick: isQuick
    ]

]

{ #category : 'CodeModification Override' }
GsMethod >> _unsafeAt: anIndex put: aValue [

self _validatePrivilege.
^ super _unsafeAt: anIndex put: aValue

]

{ #category : 'CodeModification Override' }
GsMethod >> _validatePrivilege [

System myUserProfile _validateCodeModificationPrivilege


]

{ #category : 'Accessing' }
GsMethod >> argsAndTemps [

"Returns an Array of Symbols which are the names of arguments and
 temporaries for this method."

| offset numArgsTmps |
numArgsTmps := self _numArgsAndTemps .
numArgsTmps < 1 ifTrue:[ ^  #()  ].

offset := self _debugInfoHeaderSize + 1.
^ debugInfo copyFrom: offset to: (offset + numArgsTmps - 1)

]

{ #category : 'Disassembly' }
GsMethod >> blockLiterals [

"return an Array of Block Literals that the receiver contains,
 or nil if the receiver contains no block literals.
 literals of for select blocks are not included."

| pc blockLits |

"virtual machine constants"
pc := 1 .

"iterate over the instructions in the method. pc here is 1 based,
 literalsOffset is zero based."
[pc <= literalsOffset ] whileTrue:[ | aWord info |
  aWord := self at: pc .
  info := self _opcodeInfo: aWord .
  (info bitAnd: 16r2000000) ~~ 0 ifTrue:[
    blockLits == nil ifTrue:[ blockLits := { }  ].
    blockLits addLast: ( self at: pc + 1 ).  "a block literal"
    ].

  "advance to the next instruction."
  pc := pc + (info bitAnd: 16rFF)
  ].
^ blockLits

]

{ #category : 'Copying' }
GsMethod >> copy [

"Disallowed.  You may not create new instances of GsMethod."

self shouldNotImplement: #copy

]

{ #category : 'Accessing' }
GsMethod >> environmentId [
  "Return a SmallInteger,
   the 8 bit unsigned compilation environment identifier of this method."

  ^ 0

]

{ #category : 'Repository Conversion' }
GsMethod >> fixRefsAfterConversion [

"Default method for fixing references to ObsLargePositiveInteger and
 ObsLargeNegativeInteger instances that can now be represented as
 a SmallInteger and Floats and SmallFloats which can now be represented
 as a SmallDouble.

 Has no effect in this release since all GsMethod's must be recompiled
  to produce GsNMethod's ."

^true

]

{ #category : 'Disassembly' }
GsMethod >> hasBlockLiteralsOfCost: aBlockClass [

"Return true if the receiver contains a block as costly as, or
 more costly than aBlockClass, return false otherwise.

 For the purposes of this analysis, the block classes are considered
 to have this hierarchy:
    ComplexVCBlock   cost == 3 .
      ComplexBlock   cost == 2 .
        SimpleBlock  cost == 1 .

 Note that the actual implementation hierarchy of the block classes is
    ExecutableBlock
    ComplexBlock
      ComplexVCBlock
    SimpleBlock  "

| pc argCost |

"virtual machine constants"
pc := 1 .
argCost := aBlockClass _cost .

"iterate over the instructions in the method. pc here is 1 based,
 literalsOffset is zero based."
[pc <= literalsOffset ] whileTrue:[ | info aWord |
  aWord := self at: pc .
  info := self _opcodeInfo: aWord .
  (info bitAnd: 16r2000000) ~~ 0 ifTrue:[ | aBlockLit |
    aBlockLit :=  self at: pc + 1 .  "a block literal"
    aBlockLit class _cost >= argCost ifTrue:[ ^ true ].
    ].

  "advance to the next instruction."
  pc := pc + (info bitAnd: 16rFF)
  ].
^ false

]

{ #category : 'Accessing' }
GsMethod >> inClass [

"Returns the class in which the receiver was compiled."

selector == nil ifTrue:[ ^ nil "anonymous method, as from GciExecute" ].
^ inClass

]

{ #category : 'Disassembly' }
GsMethod >> instVarsAccessed [

"return a Set of instVarNames that the method accesses."

| pc report  |

"virtual machine constants"
pc := 1 .
report := SymbolSet new .

"iterate over the instructions in the method. pc here is 1 based,
 literalsOffset is zero based."
[pc <= literalsOffset ] whileTrue:[ | aWord info ivOffset |
  aWord := self at: pc .
  info := self _opcodeInfo: aWord .
  (info bitAnd: 16r1000000) ~~ 0 ifTrue:[
    "get zero-based instance variable offset"
    ivOffset := (info bitAnd: 16rFFFF00) bitShift: -8 .
    report add: (inClass _instVarNames at: (ivOffset + 1) ).
    ].
  "advance to the next instruction."
  pc := pc + (info bitAnd: 16rFF) .
  ].
^ report

]

{ #category : 'Reporting' }
GsMethod >> isSenderOf: aSymbol [

"Returns true if the receiver sends the message aSymbol.  Returns false
 otherwise."

1 to: literalsOffset - 1 do:[:j |
  (self at: j) == aSymbol ifTrue:[ ^ true ].
  ].
^ false

]

{ #category : 'Accessing' }
GsMethod >> literals [

"Returns an Array containing the literal pool of the receiver."

| y |

y := { } .
literalsOffset to: self size do: [:i |
  y add: (self at: i)
  ].
^y

]

{ #category : 'Accessing' }
GsMethod >> literalsOffset [

"Returns the value of the instance variable named literalsOffset."

^ literalsOffset

]

{ #category : 'Accessing' }
GsMethod >> loadedSizeBytes [

 "Return the number of bytes of memory that the receiver will
  occupy when loaded in memory."

  "header + namedIvs = 3 + 9 = 12 words"
  "don't count send-caches, GsMethod is obsolete and not executable."

  ^ (12 + self size) * 8

]

{ #category : 'Accessing' }
GsMethod >> methodCompilerVersion [

"Returns the method compiler version.
 2 indicates a method compiled by Gs64 v2.0 or above method compiler,
 1 indicates a method compiled in a previous version and processed by
   repository conversion.

 Any other value indicates a method from a previous version that
 did not get converted.  However the faulting-in of such a method
 will cause an error 2261, so normally you can't send this message
 to such unconverted methods."

"extract bottom 8 bits as a unsigned value"
^ invocationCount bitAnd: 16rFF

]

{ #category : 'Accessing' }
GsMethod >> numArgs [

"Returns the value of the instance variable named numArgs."

^ numArgs

]

{ #category : 'Pragmas' }
GsMethod >> pragmas [
    self inClass == nil ifTrue: [ ^#() ].
    ^self inClass pragmasForMethod: self selector

]

{ #category : 'Repository Conversion' }
GsMethod >> recompile [

"Recompiles the method for execution in a Gs64 v3.0 or later system.
 Returns a GsNMethod, or signals a CompileError or CompileWarning .

 See also  recompileIntoMethodDict:intoCategories: "

^ self recompileIntoMethodDict: nil intoCategories: nil

]

{ #category : 'Repository Conversion' }
GsMethod >> recompileIntoMethodDict: aMethodDict intoCategories: aCategDict [

"Recompiles the method for execution in a Gs64 v3.0 or later system.

 Literal variables whose key is in the GsMethod class variable ObsoleteClassesDict
 are replaced by the appropriate association from the ObsoleteClassesDict .
 Other literal variables are looked up in the literal pool of the receiver,
 before searching class variables, class pool dictionaries, or
 the current symbolList.  Thus recompilation should work without knowing
 what symbolList was used when the receiver was created.

 The result is a GsNMethod if compilation succeeds, otherwise
 an error is generated.  environmentId zero is used for all compilations.

 If aMethodDict is not nil, and the compilation succeeds,
 the resulting method is added to aMethodDict instead of to
 the receiver's method dictionary.  This is used to add methods
 to per-session dictionaries.

 If aMethodDict is not nil and aCategDict is not nil and
 the compilation succeeds, the resulting method is added aCategDict
 instead of the receiver's categories.

 If the receiver is an anonymous method, the sender of this method is
 reponsible for saving the result."

| cls litVars symAssocCls obsDict newSrc |
litVars := { }  .
symAssocCls := SymbolAssociation .
obsDict := ObsoleteClassesDict .
literalsOffset to: self size do: [:j | | aLit |
  aLit := self at: j .
  (aLit isKindOf: symAssocCls) ifTrue:[ | litName aVal newLit srcName lookupKey obsArr |
    litName := aLit key .
    (obsArr := obsDict at: litName otherwise: nil) ifNotNil:[
      aVal := aLit value .
      (aVal ~~ nil and:[ aVal == (obsArr at:1)]) ifTrue:[
         srcName := obsArr at: 2 .
         lookupKey := obsArr at: 3 .
         newLit := Globals associationAt: lookupKey otherwise: nil .
      ].
    ].
    newLit ifNotNil:[ litVars add: srcName ; add: newLit ]
           ifNil:[    litVars add: litName ; add: aLit ].
  ]
].
SessionTemps current at:#OldLitVars put: litVars .
newSrc := GsNMethod convertArrayBuildersInString: sourceString  .
cls := self inClass .
cls ifNil:[
  ^ newSrc _compileInContext: nil symbolList: nil oldLitVars: litVars
		environmentId: 0 flags: 0
] ifNotNil:[ | categ |
 categ := cls categoryOfSelector: selector .
  ^ cls _checkCompileResult: ( cls _primitiveCompileMethod: newSrc symbolList: nil
         category: categ oldLitVars: litVars
         intoMethodDict: aMethodDict intoCategories: aCategDict
         environmentId: 0 ) 
       source: newSrc suppressCompileWarning: false
]

]

{ #category : 'Accessing' }
GsMethod >> selector [

"Returns the value of the instance variable named selector."

^ selector

]

{ #category : 'Accessing' }
GsMethod >> sourceString [

"Returns a CharacterCollection that contains the source code of the receiver."

^ sourceString

]

{ #category : 'Storing and Loading' }
GsMethod >> writeTo: aPassiveObject [

"Instances of GsMethod cannot be converted to passive form.  This method writes
 nil to aPassiveObject and stops GemStone Smalltalk execution with a notifier."

aPassiveObject writeObject: nil.
self _error: #rtErrAttemptToPassivateInvalidObject.

]
