"
This class is the logical representation of a group of UserProfiles.
 It contains the following instance variable:
   groupName         - A symbol representing the name of the group.

"
Class {
	#name : 'UserProfileGroup',
	#superclass : 'IdentitySet',
	#instVars : [
		'groupName'
	],
	#gs_reservedoop : '157441',
	#category : 'System-Users'
}

{ #category : 'Private' }
UserProfileGroup class >> _primNewWithName: aStringOrSymbol [
"Creates a new instance of the receiver, but does not add the new instance
 to the list to AllGroups and does not check for duplicate names."

| result |
result := super new .
result _groupName: aStringOrSymbol .
^ result

]

{ #category : 'Private' }
UserProfileGroup class >> _validateGroupString: aString [

"Look up the group with name aString and raise an exception if it does not
 exist."
^ self groupWithName: aString

]

{ #category : 'Accessing' }
UserProfileGroup class >> allGroups [
  ^ AllGroups

]

{ #category : 'Audit' }
UserProfileGroup class >> auditGroups [
  self allGroups do: [:aGroup |
     (aGroup auditGroup) ifFalse: [^false]].
  ^true

]

{ #category : 'Private' }
UserProfileGroup class >> basicNewWithName: aStringOrSymbol [
"Creates a new instance of the receiver, but does not add the new instance
 to the list to AllGroups.  It is an error if a group with the same name
 already exists."

((self groupWithName: aStringOrSymbol otherwise: nil) notNil)
  ifTrue:[ ^ self _error: #rtErrGroupAlreadyExists args: { aStringOrSymbol } ] .

^ self _primNewWithName: aStringOrSymbol

]

{ #category : 'Deleting' }
UserProfileGroup class >> deleteGroup: aGroup [
"Remove the group from the following places:
  -The groups instance variable of any UserProfile.
  -The loginUserProfileGroups instance variable of any KerberosPrincipal.
  -The global collection AllGroups.

aGroup must be in AllGroups. Returns the receiver."

| gName |
gName := aGroup groupName .
self allGroups at: gName .  "ensure group exists."
SystemRepository do:[:securityPolicy | 
  securityPolicy ifNotNil:[:policy | policy group: aGroup authorization: #none].  "fix 51518"
].
AllUsers removeGroup: gName . "AllUsers needs the group name"
KerberosPrincipal removeGroup: aGroup  . "KerberosPrincipal needs the group object."
self allGroups removeKey: gName .
^ self

]

{ #category : 'Deleting' }
UserProfileGroup class >> deleteGroupWithName: aStringOrSymbol [
"Remove the group with name aStringOrSymbol from the following places:
  -The groups instance variable of any UserProfile.
  -The loginUserProfileGroups instance variable of any KerberosPrincipal.
  -The global collection AllGroups.

It is an error if the group named aStringOrSymbol does not exists.  It is not an error
if the group is not present in one or more UserProfiles or KeberosPrincipals.
Returns the receiver."

^ self deleteGroup: (self groupWithName: aStringOrSymbol)

]

{ #category : 'Deleting' }
UserProfileGroup class >> deleteGroupWithName: aStringOrSymbol ifAbsent: aBlock [
"Remove the group with name aStringOrSymbol from the following places:
  -The groups instance variable of any UserProfile.
  -The loginUserProfileGroups instance variable of any KerberosPrincipal.
  -The global collection AllGroups.

Execute aBlock if the group does not exist.  It is not an error
if the group is not present in one or more UserProfiles or KeberosPrincipals.
Returns the receiver."

| nam |
nam := self groupWithName: aStringOrSymbol otherwise: nil .
nam ifNotNil:[ self deleteGroup: nam ]
    ifNil:[ aBlock value ]

]

{ #category : 'Accessing' }
UserProfileGroup class >> groupWithName: aStringOrSymbol [
"Answer the UserProfileGroup that has the given name.  Raises an exception if no
 such UserProfileGroup exists."

  ^ self allGroups at: aStringOrSymbol ifAbsent:[ aStringOrSymbol  _error: #segErrBadGroup. ^ nil ]

]

{ #category : 'Accessing' }
UserProfileGroup class >> groupWithName: aStringOrSymbol ifAbsent: aBlock [
"Answer the UserProfileGroup that has the given name or evaluate aBlock if it does
 not exist."

  ^ self allGroups at: aStringOrSymbol ifAbsent: aBlock

]

{ #category : 'Accessing' }
UserProfileGroup class >> groupWithName: aStringOrSymbol otherwise: aValue [
"Answer the UserProfileGroup that has the given name or answer aValue if it does
 not exist."

  ^ self allGroups at: aStringOrSymbol otherwise: aValue

]

{ #category : 'Illegal Operations' }
UserProfileGroup class >> new [

"Disallowed.  To create a new UserProfileGroup, use newWithName: instead "

self shouldNotImplement: #new

]

{ #category : 'Illegal Operations' }
UserProfileGroup class >> new: anInt [

"Disallowed.  To create a new UserProfileGroup, use newWithName: instead."
self shouldNotImplement: #new:

]

{ #category : 'Updating' }
UserProfileGroup class >> newGroupWithName: aStringOrSymbol [
"Creates a new instance of UserProfileGroup and adds it to the AllGroups collection.
 It is an error if a group with the same name already exits.

 Requires write access to the GsObjectSecurityPolicy for DataCurator."

| result |
result := self basicNewWithName: aStringOrSymbol .
self allGroups at: result groupName put: result .  "fix 48234"
^ result

]

{ #category : 'Private' }
UserProfileGroup >> _groupName: aStringOrSymbol [
"Private.  Sets the name of the receiver. Used only at instance creation time."

groupName := aStringOrSymbol asSymbol .
^ self

]

{ #category : 'Comparison' }
UserProfileGroup >> <= aUserProfileGroup [
  ^ groupName <= aUserProfileGroup groupName

]

{ #category : 'Managing' }
UserProfileGroup >> addUser: aUserProfile [
"Add the given UserProfile to the receiver.
 It is an error if the current user does not have the appropriate privilege.
 If <aUserProfile> is already in this group, there is no change.
 return the receiver."

aUserProfile addToUserProfileGroup: self.
^self

]

{ #category : 'Accessing' }
UserProfileGroup >> asString [
  ^ groupName asString

]

{ #category : 'Audit' }
UserProfileGroup >> auditGroup [
  self users do: [:aUser |
    (aUser groups includes: self)
	ifFalse: [self error: 'mismatch in group membership'.  ^false].
    ].
  ^true

]

{ #category : 'Accessing' }
UserProfileGroup >> groupName [
  ^ groupName

]

{ #category : 'Formatting' }
UserProfileGroup >> printOn: aStream [

aStream nextPutAll: self printString

]

{ #category : 'Formatting' }
UserProfileGroup >> printString [
|s|
s := String withAll: 'a'  .
s addAll: self class name asString ;
  addAll: ' (' ;
  addAll: groupName asString;
  add: $).
^ s

]

{ #category : 'Managing' }
UserProfileGroup >> removeUser: aUserProfile [
"Remove the given UserProfile from the receiver.
 It is an error if the current user does not have the appropriate privilege.
 If <aUserProfile> is not in this group, there is no change.
 return the receiver."
aUserProfile removeFromUserProfileGroup: self.
^self

]

{ #category : 'Audit' }
UserProfileGroup >> species [
  ^ IdentitySet

]

{ #category : 'Audit' }
UserProfileGroup >> speciesForCollect [
 ^ Array

]

{ #category : 'Accessing' }
UserProfileGroup >> userIds [
"Answer an IdentitySet of User Ids which are all members of the receiver."
| result |
result := IdentitySet new .
self do:[:e| result add: e userId ] .
^ result

]

{ #category : 'Accessing' }
UserProfileGroup >> users [
"Answer an IdentitySet of UserProfiles which are all members of the receiver."

  ^ IdentitySet withAll: self

]
