"
A SharedQueue is a thread-safe queue.  An instance may have 
one or more GsProcesses as producers and one or more GsProcesses
as consumers.

Constraints:
	valueAvailable: Semaphore
	contents: Array
"
Class {
	#name : 'SharedQueue',
	#superclass : 'Object',
	#instVars : [
		'valueAvailable',
		'contents'
	],
	#gs_reservedoop : '116993',
	#category : 'Kernel-Processes'
}

{ #category : 'Instance Creation' }
SharedQueue class >> new [
  "Answer a new SharedQueue"
  ^self basicNew initialize

]

{ #category : 'Instance Creation' }
SharedQueue class >> new: size [
  "Answer a new SharedQueue.  The argument is ignored "
  ^self new

]

{ #category : 'Private' }
SharedQueue >> _canWaitOnSocket [

  ^ true

]

{ #category : 'Private' }
SharedQueue >> _changePriority: aGsProcess from: oldPriority [

  ^ self  "do nothing"

]

{ #category : 'Testing' }
SharedQueue >> _isLarge [

"Returns true if the object is implemented as a tree of private smaller objects"

^ true

]

{ #category : 'Private' }
SharedQueue >> _reapSignal: signalSource [
  "Put signalSource on the receiver."

  self nextPut: signalSource

]

{ #category : 'Private' }
SharedQueue >> initialize [

  valueAvailable := Semaphore new.
  contents := { } .

]

{ #category : 'Testing' }
SharedQueue >> isEmpty [
  "Return true if the receiver does not contains any objects."

  ^(self size == 0)

]

{ #category : 'Removing' }
SharedQueue >> next [
  "Returns with the first object added to the receiver that has not
   yet been removed. If the receiver is empty, the active
   GsProcess waits until an object is added to it.
   If more than one GsProcess is waiting then the one with the
   highest priority that has been waiting the longest will get the
   next item added."

  |value|

  valueAvailable wait.
  value := contents removeFirst.
  ^value

]

{ #category : 'Adding' }
SharedQueue >> nextPut: value [
  "Add value to the contents of the receiver. If a GsProcess has been
   waiting for an object, allow it to proceed.
   Returns the added value."

  contents addLast: value.
  valueAvailable signal.
  ^value.

]

{ #category : 'Accessing' }
SharedQueue >> peek [
  "Returns the result of next without removing the object. If the
   receiver is empty return nil."

  |value actualSize |

  actualSize := self size.
  (actualSize == 0) ifTrue: [
    value := nil.
  ] ifFalse: [
    | nextIdx |
    "Some of the first elements may belong to guys waiting on the semaphore
     that have already been signalled but have not yet had a chance to run."
    nextIdx := (contents size) - (actualSize - 1).
    value := contents at: nextIdx.
  ].
  ^value

]

{ #category : 'Formatting' }
SharedQueue >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

| startIdx sz |
super printOn: aStream .
aStream nextPutAll: '( ' .
sz := contents size.
(self isEmpty) ifFalse: [
  startIdx := sz - (self size - 1).
  contents from: startIdx to: sz doWithIndex: [ :anElement :count |
    aStream isFull ifTrue:[
      "prevent infinite recursion when printing cyclic structures, and
       limit the size of result when printing large collections."
      aStream nextPutAll: '...)' .
      ^ self
      ] .
    anElement printOn: aStream .
    count < sz ifTrue:[ aStream nextPutAll: ', ' ].
  ].
].
aStream nextPut: $) .

]

{ #category : 'Formatting' }
SharedQueue >> printString [
  | ws str |
  str := String new.
  ws := PrintStream printingOn: str maxSize: 700 . "fix 51281"
  self printOn: ws.
   "contents might have been converted to another subclass of CharacterCollection."
  ^ ws _collection
]

{ #category : 'Accessing' }
SharedQueue >> size [
  "Return the number of object that are still in the queue."

  ^valueAvailable excessSignals

]
