Extension { #name : 'Semaphore' }

{ #category : 'Instance Creation' }
Semaphore class >> forMutualExclusion [
  "Answer a new semaphore that has an initial pending
   signal, that can be used for mutual exclusion processing."

  ^ self _basicNew _initializeMutex

]

{ #category : 'Instance Creation' }
Semaphore class >> new [
  "Answer a new semaphore"
  ^ self _basicNew initialize

]

{ #category : 'Instance Creation' }
Semaphore class >> new: size [
  "Answer a new semaphore"
  ^ self _basicNew initialize

]

{ #category : 'Private' }
Semaphore >> _canWaitOnSocket [

  ^ true

]

{ #category : 'Private' }
Semaphore >> _changePriority: aGsProcess from: oldPriority [
  "Used by GsProcess to change the priority of a GsProcess in the receiver."

  | removed |
  removed := self removeIdentical: aGsProcess otherwise: nil .
  removed == nil ifFalse:[
    self add: aGsProcess.
  ]

]

{ #category : 'Communication' }
Semaphore >> _exitCritical [
  "exit critical region"
<primitive: 737>

self _primitiveFailed: #_exitCritical

]

{ #category : 'Private' }
Semaphore >> _initializeMutex [
  "initialize a new instance that has an initial pending
   signal, that can be used for mutual exclusion processing.
   Returns receiver."

  signalCount := 1

]

{ #category : 'Private' }
Semaphore >> _reapSignal: signalSource [
  "Signal the receiver "

  self signal .

]

{ #category : 'Private' }
Semaphore >> _scheduler [

"Returns   ProcessorScheduler scheduler "
<primitive: 457>

self _primitiveFailed: #_scheduler

]

{ #category : 'Private' }
Semaphore >> _signalAll [
  "Wake up the receiver"

  self signalAll.

]

{ #category : 'Private' }
Semaphore >> _unscheduleProcess: aGsProcess [
  "Used by ProcessorScheduler"
  self removeIdentical: aGsProcess otherwise: nil .

]

{ #category : 'Private' }
Semaphore >> _wait [

  self wait.

]

{ #category : 'Adding' }
Semaphore >> add: newObject [

" Makes newObject one of the receiver's elements and returns newObject.
  Reimplemented so that it will add based on the priority of newObject."

  | count |
  count := self size.
  (count == 0) ifTrue: [
    super add: newObject.
  ] ifFalse: [
    | newPriority |
    newPriority := newObject priority.
    count _downTo: 1 do: [ :i |
      (newPriority <= ((self at: i) priority)) ifTrue: [
        self insertObject: newObject at: (i + 1).
        ^ newObject.
      ].
    ].
    self insertObject: newObject at: 1.
  ].
  ^newObject.

]

{ #category : 'Mutual Exclusion' }
Semaphore >> critical: aBlock [
  "execute aBlock when no other critical blocks are executing"

  | value |
  self wait.
  value := aBlock ensure: [self signal].
  ^value

]

{ #category : 'Private' }
Semaphore >> excessSignals [

  ^ signalCount

]

{ #category : 'Communication' }
Semaphore >> highPriorityWaitForMilliseconds: millisecondCount [

  "If signalCount > 0, this merely decrements the count.  Otherwise
   this will suspend the active process and add it to the end of
   the receiver's list of suspended processes.
   It will wait for at least millisecondCount milliseconds for the receiver to
   be signalled.
   The active process' priority is raised for the duration of the
   wait so that expiration of the wait will interrupt any infinite loop
   or infinite wait for socket activity in other processes.
   Returns true if the semaphore was signalled for the caller.
   Returns false if the timeout expires without a signal for the caller."

  | count |
  (count := signalCount) > 0 ifTrue: [
    signalCount := count - 1.
    ^true
  ] ifFalse: [ | currProc ofs sched oldPrio |
    (sched := self _scheduler) _enterCritical .
    currProc := sched activeProcess .
    oldPrio :=  currProc _raisePriority .
    self add: currProc .
    sched _waitForMilliseconds:  millisecondCount ."exits critical"
    sched activeProcess == currProc ifFalse:[
      Error signal:'current process changed in highPriorityWaitForMilliseconds:'
    ].
    sched _enterCritical .
    currProc _setPriority: oldPrio .
    ofs := self indexOfIdentical: currProc .
    ofs == 0 ifTrue:[
      "not in the semaphore so must have been signalled"
      sched _exitCritical .
      ^ true
    ].
    self removeFrom: ofs to: ofs .
    sched _exitCritical .
    ^ false
  ].

]

{ #category : 'Communication' }
Semaphore >> highPriorityWaitForSeconds: secondCount [
  "If signalCount > 0, this merely decrements the count.  Otherwise
   this will suspend the active process and add it to the end of
   the receiver's list of suspended processes.
   It will wait for at least secondCount seconds for the receiver to
   be signalled.
   The active process' priority is raised for the duration of the
   wait so that expiration of the wait will interrupt any infinite loop
   or infinite wait for socket activity in other processes.
   Returns true if the semaphore was signalled for the caller.
   Returns false if the timeout expires without a signal for the caller."

  ^ self highPriorityWaitForMilliseconds: secondCount * 1000

]

{ #category : 'Initialization' }
Semaphore >> initialize [

  signalCount := 0.

]

{ #category : 'Communication' }
Semaphore >> isLocked [
   ^ signalCount == 0

]

{ #category : 'Formatting' }
Semaphore >> printOn: aStream [
  aStream
    nextPutAll: self class name;
    nextPut: $(.
  self asOop printOn: aStream.
  self size ~~ 0 ifTrue: [
    aStream nextPutAll: ',p='.
    1 to: self size do: [:i | (self at: i) printOn: aStream] ].
  aStream nextPut: $)

]

{ #category : 'Copying' }
Semaphore >> replaceFrom: startIndex to: stopIndex with: aSeqCollection [

"Disallowed."
^ self shouldNotImplement: #replaceFrom:to:with:

]

{ #category : 'Copying' }
Semaphore >> replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex [

"Disallowed."
^ self shouldNotImplement: #replaceFrom:to:with:startingAt:

]

{ #category : 'Copying' }
Semaphore >> replaceFrom: startIndex to: stopIndex withObject: anObject [

"Disallowed."
^ self shouldNotImplement: #replaceFrom:to:withObject:

]

{ #category : 'Communication' }
Semaphore >> signal [
  "Send a signal through the receiver. If one or more GsProcesses have
   been suspended trying to receive a signal, make the one with the
   highest priority that has been waiting the longest ready to run.
   If no GsProcess is waiting, remember the excess signal.
   Returns the receiver."
  | wasInCrit sched |
  sched := self _scheduler .
  wasInCrit := sched _enterCritical .
    "ProcessorScheduler scheduler dbglog: [ 'signalling semaphore ', self printString ]."
  (self size ~~ 0) ifTrue: [ | waitingProc |
    (waitingProc := self removeFirst )  _reapSignal: self . "puts waiter on a queue"
    waitingProc priority >  sched  activeProcess priority ifTrue:[
      sched yield   "scheduler will exit critical region"
    ] ifFalse:[
      wasInCrit == 0 ifTrue:[ self _exitCritical].
    ]
  ] ifFalse: [
    signalCount := signalCount + 1  .
    wasInCrit == 0 ifTrue:[ self _exitCritical].
  ]

]

{ #category : 'Communication' }
Semaphore >> signalAll [
  "Just like signal except all of the GsProcesses waiting on the receiver
   are made ready to run.
   Returns the receiver."

  | highPrio sched wasInCrit |
    "ProcessorScheduler scheduler dbglog: [ 'signalling all semaphore ', self printString ]."
  sched := self _scheduler.
  wasInCrit := sched _enterCritical .
  highPrio := sched lowestPriority .
  1 to: self size do: [:j | | aProc  aPrio |
    (aProc := self at:j)  _reapSignal: self.
    (aPrio := aProc priority) > highPrio ifTrue:[  highPrio := aPrio ].
  ].
  self size: 0.
  highPrio > sched activeProcess priority ifTrue:[
    sched yield  "scheduler will exit critical region"
  ] ifFalse:[
    wasInCrit == 0 ifTrue:[ self _exitCritical ].
  ]
]

{ #category : 'Communication' }
Semaphore >> tryLock [
  "If signalCount > 0, decrements the count and returns true,
   otherwise returns false"

  | count |
  ^ (count := signalCount) > 0
        ifTrue:[ signalCount := count - 1 . true ]
       ifFalse:[ false ]

]

{ #category : 'Communication' }
Semaphore >> wait [
  "if signalCount > 0, this merely decrements the count.  Otherwise
   this will suspend the active process and add it to the end of
   the receiver's list of suspended processes"

  | count wasInCrit sched |
    "ProcessorScheduler scheduler dbglog: [ ('wait for ', self printString) ]."
  sched := self _scheduler .
  wasInCrit := sched _enterCritical .
  (count := signalCount) > 0 ifTrue: [
    signalCount := count - 1 .
    wasInCrit == 0 ifTrue:[ self _exitCritical].
  ] ifFalse: [ 
    "sched dbglog: [ '  semaphore suspending process' ]."
    self add: (sched := self _scheduler) activeProcess .
    sched _waitOnSema: self   "scheduler exits critical region"
  ].

]

{ #category : 'Communication' }
Semaphore >> waitForMilliseconds: millisecondCount [
  "if signalCount > 0, this merely decrements the count.  Otherwise
   this will suspend the active process and add it to the end of
   the receiver's list of suspended processes.
   It will wait for at least millisecondCount milliseconds for the receiver to
   be signalled.
   Returns true if the semaphore was signalled for the caller.
   Returns false if the timeout expires without a signal for the caller."

   | count |
  (count := signalCount) > 0 ifTrue: [
    signalCount := count - 1.
    ^true
  ] ifFalse: [ | currProc ofs sched |
    (sched := self _scheduler) _enterCritical .
    currProc := sched activeProcess .
    self add: currProc .
    sched _waitForMilliseconds:  millisecondCount ."exits critical"
    sched _enterCritical.
    ofs := self indexOfIdentical: currProc .
    ofs == 0 ifTrue:[
      "not in the semaphore so must have been signalled"
      sched _exitCritical .
      ^ true
    ].
    self removeFrom: ofs to: ofs .
    sched _exitCritical .
    ^ false
  ].

]

{ #category : 'Communication' }
Semaphore >> waitForSeconds: secondCount [
  "If signalCount > 0, this merely decrements the count.  Otherwise
   this will suspend the active process and add it to the end of
   the receiver's list of suspended processes.
   It will wait for at least secondCount seconds for the receiver to
   be signalled.
   Returns true if the semaphore was signalled for the caller.
   Returns false if the timeout expires without a signal for the caller."

  ^ self waitForMilliseconds: secondCount * 1000

]
