Extension { #name : 'Repository' }

{ #category : 'Private' }
Repository class >> _scaledDecimalFromFileId: aFileId blockId: aBlockId [
  "Return a tranlog position as a ScaledDecimal with scale 10 for convenience of printing"
   | scale denom |   
   scale := 10 .
   denom := 10000000000 "10 raisedTo: scale" .
   ^ ScaledDecimal numerator: (aFileId * denom) + aBlockId denominator: denom scale: scale
]

{ #category : 'Private' }
Repository class >> _validatePercentage: anInt [
anInt _validateClass: SmallInteger .
((anInt < 1) _or:[ anInt > 100 ])
  ifTrue:[anInt _error: #rtErrArgOutOfRange args:{ 1 . 100 } ] .

]

{ #category : 'Instance Creation' }
Repository class >> new [

"Disallowed."
self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
Repository class >> new: anInteger [

"Disallowed."

self shouldNotImplement: #new:

]

{ #category : 'Private' }
Repository >> __shrinkExtents [

<primitive: 427>

self _primitiveFailed: #__shrinkExtents .
self _uncontinuableError

]

{ #category : 'Private' }
Repository >> _aggressiveMaxThreadCount [

"Answer the number of threads to start for an aggressive,
 multithreaded repository scan.  The calculation below is really just a
 guess.  We used twice the number of CPU cores on the host since it is expected
 that threads will often be in I/O wait when scanning large repositories."

^ 2 max: ((System maxSessionId - 7) min: ((System hostCpuCount * 2) min: 64))

]

{ #category : 'Private' }
Repository >> _allInstances: anArray toDirectory: aString fast: isFast [
| allInst result |

anArray _validateInstanceOf: Array.
aString _validateKindOfClass: String.

isFast ifTrue: [ allInst := self fastAllInstances: anArray ]
       ifFalse: [ allInst := self allInstances: anArray ].

result := Array new.
1 to: allInst size do: [ :i |  | each fileName eachClass |
  each := allInst at: i.
  eachClass := each at: 1.
  aString isNil ifFalse: [
    fileName := aString , '/' , eachClass asString , '-' , eachClass asOop asString, '-instances.bm'.
    GsFile removeServerFile: fileName.
    (each at: 2) writeToFile: fileName.
  ].
  result add: eachClass.
  result add: (each at: 2) size.
].

^ result

]

{ #category : 'Private' }
Repository >> _arrayOfClassesAsSet: anArray [

1 to: anArray size do:[:n| | each |
  each := anArray at: n.
  each _validateClass: Behavior.
].
^ IdentitySet withAll: anArray .

]

{ #category : 'Updating' }
Repository >> _at: offset put: aValue [
"Disallowed"
self shouldNotImplement: #_at:put:

]

{ #category : 'Updating' }
Repository >> _basicAt: offset put: aValue [
"Disallowed"
self shouldNotImplement: #_basicAt:put:

]

{ #category : 'Private' }
Repository >> _basicMigrateWithMaxThreads: maxThreads waitForLock: lockWaitTime pageBufSize: aBufSize percentCpuActiveLimit: percentCpu mappingArrays: arrayOfMappings testArray: testObjs [
" This method is a wrapper for the migrate primitive.
  It checks that the session doesn't have any modified persistent objects, begins a transaction,
  generates the mapping arrays required by the primitive. Requires MigrateObjects privilege.
"

| arr  bm ckArr |
System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

System beginTransaction.

arr := Array new.
ckArr := Array new.

1 to: arrayOfMappings size do: [ :each | | oldCl |
  oldCl := (arrayOfMappings at: each) oldClass.
  (ckArr includesIdentical: oldCl) ifTrue: 
     [ Error signal: 'The array of mappings contains mappings for the same oldClass ' , oldCl printString ]  
  ifFalse: [ ckArr add: oldCl ].
  arr add: ((arrayOfMappings at: each) convertForMigrateMt)
  ].

  bm := self _migrateWithMaxThreads: maxThreads waitForLock: lockWaitTime
               pageBufSize: aBufSize percentCpuActiveLimit: percentCpu
	       mappingArrays: arr testArray: testObjs.
  (bm size == 0) ifFalse: [ bm  _error: #rtErrOpNotAllowed args:
       { 'multithreaded detected objects that it could not migrate'}
  ].
  ^self

]

{ #category : 'Updating' }
Repository >> _basicSize: aSize [
"Disallowed"
self shouldNotImplement: #_basicSize:

]

{ #category : 'Private' }
Repository >> _buildAllRefsResult: allRefsResult forInput: inputArr [

 | resultArr |
 resultArr := Array new: (inputArr size).
 1 to: (allRefsResult size) do: [ :i |  | element |
   element := allRefsResult at: i.
   resultArr at: (inputArr indexOfIdentical: (element at: 1)) put: ((element at: 2) asArray).
 ].
 ^ resultArr.

]

{ #category : 'Private' }
Repository >> _buildAllRefsResult: allRefsResult forInput: inputArr withLimit: aSmallInt [

 | resultArr |
 resultArr := Array new: (inputArr size).
 1 to: (allRefsResult size) do: [ :i |  | element |
   element := allRefsResult at: i.
   resultArr at: (inputArr indexOfIdentical: (element at: 1))
             put: ((element at: 2) enumerateWithLimit: aSmallInt startingAfter: 0)
 ].
 ^ resultArr.

]

{ #category : 'Private' }
Repository >> _buildListInstResult: allInstResult [

 | gsBitmap arrSize element resultArr |

 gsBitmap := GsBitmap newForHiddenSet: #ListInstances.
 gsBitmap removeAll.

 arrSize := allInstResult size.
 arrSize == 1 ifTrue: [
   element := allInstResult at: 1.
   gsBitmap addAll: (element at: 2).
   ^ (element at: 2) size
 ].

 resultArr := Array new.
 1 to: arrSize do: [ :i |
   element := allInstResult at: i.
   resultArr add: (element at: 1).
   resultArr add: ((element at: 2) size).
   gsBitmap addAll: (element at: 2).
 ].
 ^ resultArr.

]

{ #category : 'Private' }
Repository >> _checkMbyteLimit: aVal [

  ^ aVal class == SmallInteger and:[ aVal >= 0 and:[ aVal <= (4000 * 1024) ]]

]

{ #category : 'Private' }
Repository >> _checkObjectSecurityPolicyArg: inputArg [

(inputArg class == Array)
  ifTrue: [
      1 to: inputArg size do:[ :k |
      self _validateObjectSecurityPolicyId: (inputArg at: k)] ]
  ifFalse: [self _validateObjectSecurityPolicyId: inputArg]

]

{ #category : 'Private' }
Repository >> _checkPointInTime: aDateTime [
  "aDateTime may be a DateTime or a DateAndTime "
  | aTimeT |
  aTimeT := aDateTime asPosixSeconds truncated .  "fix 51350"
  self restoreStatusPosixTimeRestoredTo ifNotNil:[ :rTime |
    aTimeT < rTime  ifTrue: [
      ImproperOperation signal: ( 'aDateTime ' , aDateTime asString ,
        ' is less than restore position ' , self restoreStatusDateAndTimeRestoredTo asString )
    ].
  ].
  ^ aTimeT
]

{ #category : 'Private' }
Repository >> _checkRestoreActive [

self restoreActive ifFalse: [
  ImproperOperation signal:
     'You cannot restoreLogs without first executing restoreFromBackup'].

]

{ #category : 'Private' }
Repository >> _contentsOfPage: aPage [

"If aPage is an invalid page ID, then the primitive fails.  Returns an Array of
 size at least 5 for all kinds of pages.  For data pages, the Array size is at
 least 15 and varies depending upon the number of objects in the page.

 The aPage argument should be a positive Integer specifying a valid page ID.

 The contents of the returned Array are as follows:

    index       description
     1.         pageKind
     2.         sessionId for user that wrote the page.
     3.         low beginId  (transaction begin ID)
     4.         high beginId
     5.         bytesUsed
                     (elements 6 and above are only present for data pages )
     6.         nil  (was approx page creation time (a DateTime) in gs64v1.0)
     7.         clusterId
     8.         freeBytes
     9.         numFreeChunks
    10.         sizeLargestFree
    11.         numUsedBytes
    12.         numberOfObjects
    13.         maxObjSize
    14.         minObjSize
    15.         numberOfValidObjects
    16.         numberOfInvalidObjects
    (17.. 17 + numberOfValidObjects - 1) list of valid objects
    (17 + numberOfValidObjects ...)  list of OOPs of invalid objects.  "

<primitive: 399>
self _primitiveFailed: #_contentsOfPage: args: { aPage }

]

{ #category : 'Private' }
Repository >> _createFdcResults [
  (Globals associationAt: #FdcResults otherwise: nil) ifNil:[
    Globals at: #FdcResults put: nil .
  ].
  self cleanupFdcResults .
  ^ (Globals at: #FdcResults)

]

{ #category : 'Transaction Logging' }
Repository >> _currentLogInfo [

"Returns the result specified for Repository | _logInfo: for the
 currently active transaction log, or nil if no tranlog is open for writing"

 | numDirs logInfo dirs |
 numDirs := 1 .
 dirs := System configurationAt: #STN_TRAN_LOG_DIRECTORIES .
 dirs _isArray ifTrue:[ numDirs := dirs size ]  .
 2 timesRepeat:[ "cope with changing to the next log"
   1 to: numDirs do:[ :j |
     logInfo := self _logInfo: j .
     (logInfo at: 3) == true ifTrue:[ ^ logInfo ].
	 ].
 ].
 ^ nil

]

{ #category : 'Repository Usage Reporting' }
Repository >> _dbfStatistics: extentFilename [

"Returns an Array of size 2 containing the number of bytes the given extent
 requires for physical size and the number of bytes of free space in the
 extent."

<primitive: 283>

extentFilename _validateKindOfClass: String .
^ self _primitiveFailed: #_dbfStatistics: args: { extentFilename }

]

{ #category : 'Private' }
Repository >> _disableRestoreFromLogsInt [
  SessionTemps current at: #GsDisableRestoreFromLogs
    ifAbsent: [ ^ 0 ].
  ^ 2
]

{ #category : 'Private' }
Repository >> _doListInstancesFrom: inputSet with: diskBlock memory: memoryOptInt limit: aSmallInt [
  "memoryOptInt 0 disk only 
                1 disk plus all of memory, 
                2 memory only 
                3 memory only, exclude pom_gen . "
| inputSetSize result limit |
limit := aSmallInt.
limit == 0 ifTrue: [ limit := SmallInteger maximum32bitInteger ].
inputSetSize := inputSet size .
result := { }  .  "pairs  size, Array of instances, in order of input set"
diskBlock ifNotNil:[
  result addAll: (diskBlock value: inputSet ).
 ].

memoryOptInt > 0  ifTrue:[ | memResult idx |
  idx := 1.
  memResult := self _listInstancesInMemory: (Array withAll: inputSet) limit: limit option: memoryOptInt.
  1 to: memResult size do:[:j | | elem memRes |
    memRes := memResult at: j .
    (elem := result atOrNil: idx) ifNil:[
       result at: idx put: memRes size ;
              at: idx + 1 put: memRes .
    ] ifNotNil:[
       elem := result at: idx + 1.
       (elem := IdentitySet withAll: elem) addAll: memRes .
       elem := elem asArray.
       elem size > limit ifTrue: [ elem size: limit ].
       result at: idx put: elem size ;
              at: idx + 1 put: elem.

    ].
    idx := idx + 2
  ].
].
^ result

]

{ #category : 'Private' }
Repository >> _doScan: kind fast: isFast with: inputArgument [

" This method does the appropriate tests, and calls the _scan primitive
  which does the bulk of the work.

  If the SessionTemps current at: #GsOverrideNumThreads is defined, then that value is
  used to determine the number of threads to use.

      scanKind                      Function
   ==========================================
   OP_ALL_INSTANCES              allInstances
   OP_ALL_REFERENCES             allReferences
   OP_ALL_REFS_TO_INSTANCES      allReferencesToInstancesOfClasses
   OP_ALL_REFS_BY_PARENT_CLASS   allReferencesByParentClass
   OP_ALL_OBJS_IN_SEC_POLICY     allObjectsInObjectSecurityPolicies
   OP_ALL_OBJS_LARGER_THAN       allObjectsLargerThan

"
| maxThreads percentCpu argClass result inputArg |
inputArg := (inputArgument _isArray and: [inputArgument size > 1])
                ifTrue:[ Array withAll:(IdentitySet withAll: inputArgument)]
                ifFalse:[ inputArgument ].
System needsCommit ifTrue: [ System _error: #rtErrAbortWouldLoseData ] .

isFast ifTrue: [ maxThreads := self _aggressiveMaxThreadCount.
                 percentCpu := 95 ]
       ifFalse: [  maxThreads := self getDefaultNumThreads. percentCpu := 90 ].
result := self _scanWithMaxThreads: maxThreads waitForLock: 60 pageBufSize: 8
               percentCpuActiveLimit: percentCpu scanKind: kind with: inputArg .
argClass := inputArg class.
(kind ~= OP_ALL_REFS_BY_PARENT_CLASS) ifTrue: [
  ((argClass == Array) or: [argClass == GsBitmap]) ifFalse: [
       result := (result at: 1) at: 2. ] .
  ].
^result

]

{ #category : 'Repository Usage Reporting' }
Repository >> _extentsReport: unitsArg [

"Returns a String which reports on the name, size, and amount of free space
 for each extent and the size and free space of the entire logical Repository.
 unitsArg should a Float power of two that is either MB or GB; or nil, in 
 which case use GB for total size over 2GB, otherwise MB."

| result stats fileNames totalSize totalFree units |
result := String new.
stats := self _extentStatistics: -1 .
totalSize := 0.0 . totalFree := 0.0 .
1 to: stats size do:[:j |
  totalSize := totalSize + ((stats at: j) at: 1) .
  totalFree := totalFree + ((stats at: j) at: 2) .
].
units := unitsArg .
units ifNil:[ | gb |
  gb := 1073741824.0 .
  units := totalSize > (gb * 2) ifTrue:[ gb ] ifFalse:[ 1048576.0 ].
].
fileNames := self fileNames.
1 to: stats size do: [:j |
    result addAll: (fileNames at: j) ; lf ;
           addAll: '   File size   ';
           addAll: (self _numToString:((stats at:j) at: 1) units: units) ;
           addAll: '   free space  ';
           addAll: (self _numToString:((stats at:j) at: 2) units: units) ; lf.
    ].
result addAll: 'Total'; lf ;
           addAll: '   File size   ';
       addAll: (self _numToString: totalSize units: units) ;
           addAll: '   free space  ';
       addAll: (self _numToString: totalFree units: units) ; lf .
^ result

]

{ #category : 'Repository Usage Reporting' }
Repository >> _extentStatistics: extentIndex [

"If extentIndex > 0 ,
 Returns an Array of size 2 containing the number of bytes the
 given extent requires for physical size and the number of bytes of
 free space in the extent.

 If extentIndex == -1,
 Returns an Array of Arrays. The outer Array has size equal to the
 number of extents; the inner Arrays each have size 2 and contain
 extent size and extent free space.
"

<primitive: 286>

(extentIndex _isSmallInteger)
  ifTrue:[ ^ self _errorIndexOutOfRange: extentIndex]
  ifFalse:[ extentIndex _validateClass: SmallInteger ].
^ self _primitiveFailed: #_extentStatistics: args: { extentIndex }

]

{ #category : 'Garbage Collection' }
Repository >> _fastMarkForCollection [

"Returns an Array { numLiveObjects . numDeadObjects . approxSizeOfDeadObjects .
                     numPossibleDeadSymbols } .
"
^ self _mfcWithMaxThreads: self _aggressiveMaxThreadCount
    waitForLock: 90"seconds"
    pageBufSize: 128 percentCpuActiveLimit: 95.

]

{ #category : 'Private' }
Repository >> _fdcWithMaxThreads: maxThreads waitForLock: lockWaitTime pageBufSize: aBufSize percentCpuActiveLimit: cpuPercent toFile: aFileNameString resultArray: anArray [

 "Run a multithreaded algorithm to determine the disconnected (dead) objects
  in the repository.  The dead objects identifiers (oops) are written
  to a file, stored in anArray, or both.

  Returns anArray .

  This is method never results in the garbage collection of any objects.

  See _mfcWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:
  for documentation of the common arguments.

  If the toFile argument is not nil, a file is created and a list of the
  dead object identifiers found is written to the file in a binary format.
  The file must not exist when the method is called and the argument
  must be an instance or subclass of String.
  The size of the file in bytes is approximately 8 times the number
  of dead objects found.

  Be careful to ensure there is sufficient disk space to write the
  entire list of dead objects when using this method with large repositories.

  The resultArray is expected to be a committed Array object of size 4.
  If successful, this method stores the following values in the resultArray:

    1 - anInteger representing the number of live objects found.
    2 - anInteger representing the number of dead objects found.
    3 - anArray containing the dead objects, false if a commit failed,
        or nil if the list of dead objects is not stored in the repository.
    4 - nil if aFileNameString was nil, otherwise a boolean indicating
        whether the dead object ids were successfully written to the file.

  This method requires the GarbageCollection privilege and aborts the
  current transaction.

  Pre-existing dead objects, i.e., objects which reside in the stone's
  deadNotReclaimed set, are ignored and are not returned by this method.

  The output file is deleted if this method encounters an error.

  See the description of
    #_mfcWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:
  for a description of the memory costs and performance optimizations.

  See the following methods in class System for details of accessing
  and updating the mt statistics:
     mtThreadsLimit: sessionId
     mtThreadsLimit: sessionId setValue: newVal
     mtPercentCpuActiveLimit: sessionId
     mtPercentCpuActiveLimit: sessionId setValue: newVal
     mtMaxThreads: sessionId"

<primitive: 877>
| maxInt |
maxInt := SmallInteger maximum32bitInteger .
maxThreads _validateClass: SmallInteger ; _validateMin: 1 max: maxInt .
lockWaitTime _validateClass: SmallInteger; _validateMin: -1 max: maxInt .
aBufSize _validateClass: SmallInteger ; _validateIsPowerOf2 .
self _validatePercentage: cpuPercent .
aFileNameString ifNil:[ (anArray at: 3) _validateInstanceOf: Array ]
             ifNotNil:[ aFileNameString _validateKindOfClass: String ].
anArray _validateInstanceOf: Array.
(anArray size == 4) ifFalse:[anArray _error: #rtErrArgOutOfRange ].
(anArray at: 3) ifNil:[ aFileNameString _validateKindOfClass: String].
anArray isCommitted ifFalse:[ ArgumentError signal:'result array anArray is not a committed object'].
self _primitiveFailed:
     #_fdcWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:toFile:resultArray:
     args: {  maxThreads . lockWaitTime . aBufSize .
              cpuPercent . aFileNameString . anArray} .
^ self _uncontinuableError

]

{ #category : 'Repository Usage Reporting' }
Repository >> _fileSizeReport: unitsArg [

"Returns a String which reports on the name, size, and amount of free space
 for each extent and the size and free space of the entire logical Repository.
 unitsArg should a Float power of two that is either MB or GB; or nil, in 
 which case use GB for total size over 2GB, otherwise MB."

| result stats repositoryFiles extentFile totalSize totalFree units |
result := String new.
stats := self _extentStatistics: -1 .
totalSize := 0.0 . totalFree := 0.0 .
1 to: stats size do:[:j |
  totalSize := totalSize + ((stats at: j) at: 1) .
  totalFree := totalFree + ((stats at: j) at: 2) .
].
repositoryFiles := self fileNames.
units := unitsArg .
units ifNil:[ | gb |
  gb := 1073741824.0 .
  units := totalSize > (gb * 2) ifTrue:[ gb ] ifFalse:[ 1048576.0 ].
].
1 to: stats size do: [:i |
    extentFile := repositoryFiles at: i.
    i > 1 ifTrue:[ result addAll: '-----------'; lf. ].
    result addAll: 'Extent #'; addAll: i asString; lf.
    result addAll: '   Filename = '; addAll: extentFile; lf ;
           addAll: '   File size =       ';
           addAll: (self _numToString:((stats at:i) at: 1) units: units) ; lf;
           addAll: '   Space available = ';
           addAll: (self _numToString:((stats at:i) at: 2) units: units) ; lf.
    ].
result addAll: '------'; lf.
result addAll: 'Totals'; lf.
result addAll: '   Repository size = ';
       addAll: (self _numToString: totalSize units: units) ; lf;
       addAll: '   Free Space =      ';
       addAll: (self _numToString: totalFree units: units) ; lf .
^ result

]

{ #category : 'Private' }
Repository >> _findConnectedObjs: anArray [

"Find all objects reachable from the oops in anArray.
 This method aborts the current transaction."

<primitive: 560>
anArray _validateInstanceOf: Array .
(anArray size) < 1 ifTrue:[ Error signal:'anArray is empty'].
^ self _primitiveFailed: #_findConnectedObjs

]

{ #category : 'Private' }
Repository >> _findOopsOnPages: anArrayOfPageIds [
"Scan the object table and return an Array containing all OOPs that reference any page
 in the input array of page IDs.
"
<primitive: 561>
self _primitiveFailed: #_findOopsOnPages: args: { anArrayOfPageIds }

]

{ #category : 'Private' }
Repository >> _findPagesContainingOops: anArrayOfInts [
"Returns an Array of page ID's that contain the oops in the input
 Array.  The input Array must contain the oop as a SmallInteger or
 LargeInteger, not the object itself.

 This method does not look at the object table, rather is scans
 the data pages directly from the extents.  Note that the object
 may appear on more than 1 page if it is shadowed."
<primitive: 563>
anArrayOfInts _validateInstanceOf: Array .
anArrayOfInts size == 0 ifTrue:[ anArrayOfInts _error: #errArgTooSmall args:{ 1 } ].
self _primitiveFailed: #_findPagesContainingOops: args: { anArrayOfInts }

]

{ #category : 'Backup and Restore' }
Repository >> _fullBackupTo: fileNames MBytes: limitArray compressKind: cKind bufSize: count numThreads: threadCount [

"Private.  Provides implementation of backup methods.

 Valid compressKinds:
   0 - no compression
   1 - zlib (aka gzip) compression, default compression level 6
   2 - lz4 compression, default compression level 0

 In addition, the following cKind values may be used to specify both the compression
 algorithm and the compression level:

   Zlib:
   101 - fastest performance, lowest compression ratio
   102 - 108 - intermediate performance and compression ratios
   109 - slowest performance, highest compression ratio.

   Lz4
   200 - fastest performance, lowest compression ratio.
   201 - 211 - intermediate performance and compression ratios
   212 - slowest performance, highest compression ratio.

 The bufSize argument controls the size of the buffer used to write records to the file.
 The count argument specifies the number of 128KB backup records contained in the buffer.
 The values allowed are 1 (128KB) through 8196 (1GB).

 Returns true if the backup was completed."

  <primitive: 394>
  | a b |
  fileNames _validateInstanceOf: Array.
  limitArray _validateInstanceOf: Array.
  cKind _validateClass: SmallInteger .
  threadCount _validateClass: SmallInteger .
  self _validateCompressionKind: cKind .
  count _validateClass: SmallInteger .
  (a := fileNames size) == (b := limitArray size) ifFalse:[
    ArgumentError signal: 'number of fileNames (' , a asString,
') not equal to number of limits (', b asString , ')'
  ].
  self _primitiveFailed: #_fullBackupTo:MBytes:compressKind:bufSize: .

]

{ #category : 'Private' }
Repository >> _getNumThreads [
  ^ SessionTemps current at: #GsOverrideNumThreads
      ifAbsent: [ (SystemRepository numberOfExtents) * 2 ]

]

{ #category : 'Private' }
Repository >> _getRestoreOptions [
 ^ self _shrinkRepositoryInt bitOr: self _disableRestoreFromLogsInt

]

{ #category : 'Private' }
Repository >> _listInstancesInMemory: anArray limit: aSmallInt option: memOptionInt [

"anArray may not contain duplicates, and the elements must be sorted in OOP order.
 Returns an Array, each element is an Array of instances for
 corresponding element of anArray.
 memOptionInt  <= 2 all of memory
               3 exclude pom_gen . 
               4  (with anArray == nil) first 1000 objects in memory larger than aSmallInt."

<primitive: 389>
anArray ifNotNil:[ anArray _validateInstanceOf: Array ].
memOptionInt _validateClass: SmallInteger .
aSmallInt _validateClass: SmallInteger; _validateMin: 0 max: SmallInteger maximum32bitInteger .
^ self _primitiveFailed: #_listInstancesInMemory: args: { anArray . aSmallInt }

]

{ #category : 'Private' }
Repository >> _listObjectsInObjectSecurityPolicies: anArray limit: aLimit scanKind: kind toDirectory: aStringOrNil withMaxThreads: maxThreads maxCpuUsage: aPercentage [

 "aLimit is a SmallInteger, max number of objects to report, or 0 for unlimited.

 anArray must be an Array of objectSecurityPolicyIds.  The maximum size
 of anArray is no longer restricted to 2034 elements."

| sortedSegIds inputSegIds primResult primResultOfs result |

System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

sortedSegIds := SortedCollection new .
inputSegIds := { }  .
anArray _validateInstanceOf: Array .
1 to: anArray size do:[ :k | |segId |
  segId := anArray at: k .
  self _validateObjectSecurityPolicyId:  segId .
  inputSegIds add: segId .
  sortedSegIds add: segId
].
primResult := self _scanPomWithMaxThreads: maxThreads waitForLock: 90 pageBufSize: 8
                percentCpuActiveLimit: aPercentage
                identSet: (IdentitySet withAll: sortedSegIds)
                limit: aLimit scanKind: kind toDirectory: aStringOrNil .

kind == 4 ifFalse:[ ^ primResult ].
result := Array new: anArray size * 2 .

primResultOfs := 1 .
1 to: sortedSegIds size do:[:segIdOfs | | inOfs |
  inOfs := inputSegIds indexOfIdentical: (sortedSegIds at: segIdOfs) .
  result at: (inOfs * 2) -1  put: (primResult at: primResultOfs ) .
  result at: (inOfs * 2)   put: (primResult at: primResultOfs + 1) .
  primResultOfs := primResultOfs + 2 .
].
^ result

]

{ #category : 'Private' }
Repository >> _listObjectsWithOneReferenceWithMaxThreads: maxThreads waitForLock: lockWaitTime percentCpuActiveLimit: percentCpu [

"Private, do not call this method directly.

 Results are placed in the #ListInstances GsBitmap.
 Primitive returns a SmallInteger which is the size of
 this object."

<primitive: 536>
 | maxInt |
 maxInt := SmallInteger maximum32bitInteger .
 maxThreads _validateClass: SmallInteger; _validateMin: 1 max: maxInt .
 lockWaitTime _validateClass: SmallInteger; _validateMin: -1 max: maxInt .
 self _validatePercentage: percentCpu .
 ^ self _primitiveFailed:
     #_listObjectsWithOneReferenceWithMaxThreads:waitForLock:percentCpuActiveLimit:
     args: { maxThreads . lockWaitTime . percentCpu } 
]

{ #category : 'Listing Instances' }
Repository >> _listReferencesFromBms: anArray numThreads: numThreads toDirectory: aDirectoryString [

"An array must be an Array of instances of GsBitmap .
 Returns an Array of pairs
     classOop, number of objects containing references
 The pairs are not unique within the array , there will be a pair for each array per thread."

<primitive: 722>

^ self _primitiveFailed: #_listReferencesFromBmFiles:numThreads:toDirectory:
  args: { anArray . numThreads .aDirectoryString }

]

{ #category : 'Private' }
Repository >> _listReferencesInMemory: anArray nonStubbedOnly: aBoolean [

"An error is signaled if anArray contains duplicates.
 Returns an Array, each element is an Array of instances for
 corresponding element of anArray."

<primitive: 393>
anArray _validateInstanceOf: Array.
aBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_listReferencesInMemory:nonStubbedOnly: args: { anArray . aBoolean }

]

{ #category : 'Private' }
Repository >> _listReferencesToInstancesOfClasses: anArray toDirectory: aString withMaxThreads: maxThreads maxCpuUsage: aPercent [

"Similar to fastListReferencesToInstancesOfClasses:toDirectory: except that
 the user can also specify the maxThreads and cpuUseage."

| primResult sortedArray result sortedCollection |
System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

"Sort by oop order."
sortedCollection := SortedCollection sortBlock:[:a :b| a asOop < b asOop].
sortedCollection addAll: anArray.
sortedArray := Array withAll: sortedCollection .

primResult :=  self _listRefsToInstOfClasses: sortedArray  toDirectory: aString
                       withMaxThreads: maxThreads maxCpuUsage: aPercent.

"nil means the primitive detected a duplicate entry.  Raise an error and bail."
primResult ifNil:[  ArgumentError new object: anArray ;
    signal:'Array of references to search for contains invalid class'
].

"build the result Array to match the order of the argument Array."
result := Array new.
1 to: anArray size do: [:j| | cls i |
  cls := anArray at: j .
  i := primResult indexOfIdentical: cls .
  i == 0 ifTrue:[
    ArgumentError signal:'class ' , cls name ,
	 ' not found in results from primitive _listRefsToInstOfClasses' .
    result add: cls ; add: 0; add: 0 .
  ] ifFalse:[
    result add: (primResult at: i); add: (primResult at: i + 1);
         add: (primResult at: i + 2).
  ]
].
^ result

]

{ #category : 'Private' }
Repository >> _listReferencesWithMaxThreads: maxThreads waitForLock: lockWaitTime pageBufSize: aBufSize percentCpuActiveLimit: percentCpu targets: anArray withLimit: aSmallInt [

"Private. Returns a list of instances in the Repository that have a
 reference to one of the objects specified in anArray .  The
 result of this method is an  Array of Arrays, where the contents of
 each inner array consists of instances that have a reference to the
 corresponding element in anArray.

 This method aborts the current transaction.

 The number in inner array is limited to approximately aSmallInt,
 and may exceed aSmallInt by the maximum number of objects in a data page.
 If aSmallInt is <= 0, the result size is unlimited.

 The result contains only permanent objects.

 This primitive method performs a multithreaded scan of the Repository.

 See _scanPomWithMaxThreads primitive for a description of the multithreaded args.

 anArray must be sorted in OOP order and not contain any duplicate entries.

 The maximum size of anArray is no longer restricted to 2034 elements.
"

<primitive: 899>
 | maxInt |
 maxInt := SmallInteger maximum32bitInteger .
 maxThreads _validateClass: SmallInteger; _validateMin: 1 max: maxInt .
 lockWaitTime _validateClass: SmallInteger; _validateMin: -1 max: maxInt .
 aBufSize _validateClass: SmallInteger ; _validateIsPowerOf2 .
 self _validatePercentage: percentCpu .
 aSmallInt _validateClass: SmallInteger ; _validateMin: 0 max: SmallInteger maximumValue .
 anArray _validateInstanceOf: Array.
 ^ self _primitiveFailed:
    #_listReferencesWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:targets:withLimit:
    args: {  maxThreads . lockWaitTime . aBufSize .
             percentCpu . anArray . aSmallInt }

]

{ #category : 'Private' }
Repository >> _listRefsToInstOfClasses: anArray  toDirectory: aString withMaxThreads: maxThreads maxCpuUsage: cpuPercent [

"Primitive implementation of listReferencesToInstancesOfClasses"

<primitive: 504>
 anArray _validateInstanceOf: Array.
 anArray size == 0 ifTrue:[ anArray _error: #errArgTooSmall args:{ 1 } ].
 aString _validateKindOfClass: String.
 maxThreads _validateClass: SmallInteger; _validateMin: 1 max: SmallInteger maximum32bitInteger .
 self _validatePercentage: cpuPercent .
 ^ self _primitiveFailed:
    #_listReferencesToInstancesOfClasses:toDirectory:withMaxThreads:maxCpuUsage:
    args: {  anArray . aString . maxThreads . cpuPercent }

]

{ #category : 'Class Management' }
Repository >> _loadedClasses: includeModulesBoolean [

"Returns an Array of all Classes currently loaded in memory.
 If includeModulesBoolean==true, the result also includes instances of Module.
 Result does not include meta classes, meta modules, or Ruby virtual classes."

<primitive: 869>
includeModulesBoolean _validateClass: Boolean .
self _primitiveFailed: #_loadedClasses: args: { includeModulesBoolean }

]

{ #category : 'Transaction Logging' }
Repository >> _logInfo: logDirId [

"If logDirId is within the range of 1 to number of log directories, returns an
 Array containing the following instance variables.  Otherwise returns nil.

 1: logDirId (SmallInteger)
 2: numLogDirs (SmallInteger)
 3: isActiveLog (Boolean)
 4: fileId (SmallInteger) (if active log )
 5: fileSize (SmallInteger) in units of MBytes (if active log)
 6: maxFileSize (SmallInteger) in units of MBytes
 7: logOriginTime (Integer)  (if active log)
 8: fileName (String) (fileOrDevice if active log, else directoryOrDevice)
 9: replicateName (no longer used, always nil)
10: fileId of begin record of oldest transaction in current checkpoint,
	 if active log, otherwise nil
11: fileSize (SmallInteger) in units of 512byte blocks for active log 
12: fileId of the last tranlog record written  (independent of logDirId)
13: blockId of last tranlog record written  (independent of logDirId)
14: fileId of the last checkpoint record written  (independent of logDirId)
15: blockId of last checkpoint record written  (independent of logDirId)"

<primitive: 397>

logDirId _validateClass: SmallInteger.
^ self _primitiveFailed: #_logInfo: args: { logDirId }

]

{ #category : 'Private Object Migration' }
Repository >> _mapObjects: classMapArray ivMap: ivMapArray sessionNumber: sessionInt objsPerCommit: numObjs [

"Uses the files written by Repository >> _writeAllDataPageIdsToFiles: "

<primitive: 684>

self _primitiveFailed: #_mapObjects:ivMap:sessionNumber 
  args: { classMapArray . ivMapArray . sessionInt . numObjs }

]

{ #category : 'Private Object Migration' }
Repository >> _mapObjectsWithNumSessions: numSessions objsPerCommit: objsPerCommit password: pwString classMapName: classMapSymbol ivMapName: ivMapSymbol [
" (UserGlobals at: classMapSymbol) must be a committed Array of pairs (oldVersionOfClass, newVersionOfClass).
  All of the classes share a common super class .
  (UserGlobals at: ivMapSymbol) is an Array describing the instVars of the old super class, elements are
     aSymbol - instVar is deleted,
     positive SmallInteger - new instVar offset of a surviving instAvar ,
     -1 -  instVar becomes a dynamic instVar if non-nil in an instance .
  Returns a SmallInteger , the total number of data pages processed by all sessions"
| sessions lib done numPages doitString |
numPages := 0.
doitString := 
  'System removeGemLogOnExit: false .
   System gemConfigurationAt:#GemExceptionSignalCapturesStack put: true .
   SystemRepository _mapObjects: (UserGlobals at:#', classMapSymbol, ') 
     ivMap: (UserGlobals at:#', ivMapSymbol, ') sessionNumber: sessionIdx 
     objsPerCommit: ', objsPerCommit asString.

numSessions == 1 ifTrue:[
  "run in this session for easier debugging"
  numPages := (doitString copyReplaceAll:'sessionIdx' with: '1') evaluate .
] ifFalse:[  | tocKb |
  tocKb := 1000000 .
  (objsPerCommit * 1000) > (tocKb *1024) ifTrue:[
     tocKb := objsPerCommit * 1000 / 1024 .
  ].
  sessions := Array new .
  lib := GciTsLibrary newDefault .
  numSessions timesRepeat:[ | sess params nrs |
    nrs := GsNetworkResourceString defaultGemNRSFromCurrent .
    nrs dir: GsFile serverCurrentDirectory .  "to be able to read the datapageIds files"
    nrs temporaryObjectCacheSize: tocKb .
    sess := GsTsExternalSession newDefault: lib .
    params := sess parameters  .
    params password: pwString copy ; gemService: nrs asString  .
    sess login .
    sessions add: sess .
    GsFile gciLogServer:'logged in worker gem process ' , sess gemProcessId asString .
  ].
  1 to: sessions size do:[:n | | aSess |
    aSess := sessions at: n .
    aSess forkString: (doitString copyReplaceAll:'sessionIdx' with: n asString) .
  ].
  done := 0 .
  [ done < numSessions ] whileTrue:[
    1 to: sessions size do:[:n | 
      (sessions at:n) ifNotNil:[:aSess |
        aSess isResultAvailable ifTrue:[ | res |
          res := aSess lastResult .
          res _isSmallInteger ifFalse:[ Error signal:'bad result' ].
          numPages := numPages + res .
          done := done + 1 .
          aSess logout .
          sessions at: n put: nil .
        ].
      ].
    ].
    Delay waitForSeconds: 1 .
    System abortTransaction .
   ].
 ].
^ numPages 

]

{ #category : 'Private' }
Repository >> _mfcWithMaxThreads: maxThreads waitForLock: lockWaitTime pageBufSize: aBufSize percentCpuActiveLimit: cpuPercent [
 "Perform a multithreaded markForCollection operation using at most maxThreads.

  Returns an Array { numLiveObjects . numDeadObjects . approxSizeOfDeadObjects } .
  The numPossibleDeadSymbols element of the result will be nil if
  STN_SYMBOL_GC_ENABLED is false in the stone configuartion.

  The maxThreads argument specifies the maximum number of threads
  that can be used during the operation.  The actual number of active threads
  can be adjusted to a lower value at runtime automatically by the main
  thread based on the percentCpuActiveLimit and dynamically by the setting
  the mtThreadLimit (see details below).

  The lockWaitTime argument is used to specify how many seconds method should
  wait while attempting to acquire the gcLock.  No other garbage collection
  operations may be started or in progress while this method is running.
  There also must be no outstanding possible dead objects in the system for
  the GC lock to be granted.

  The pageBufSize, which must be a power of two, specifies the number
  of pages to buffer per thread. This parameter in conjunction with the
  maxThreads largely determines the memory footprint needed to perform
  this operation.

  The cpuActiveLimit specifies a level of total cpu activity at which the algorithm
  automatically inactivates threads to prevent overload of system resources.

  This algorithm makes use of additional sessions (threads) to achieve
  significant performance improvements.  It also makes space/time trade offs
  so that heap memory in addition to the TemporaryObjectCache (TOC)
  resources are used.  In fact, this algorithm doesn't require any TOC, so
  configuring this process for the smallest TOC space is advantageous.
  The memory space that is needed is variable and depends upon the current
  oopHighWater value, the number of sessions and the pageBufSize specified.

  The overhead associated with the oopHighWater value can be computed:
    oopHighWater in bytes = (stnOopHighWater + 10M) / 2

  The memory cost per thread is 50K + (180K * pageBufSize)

  To give some idea of how this scales, a system that had an oopHighWaterMark
  of 500M running 8 sessions with a pageBufSize of 512 would require about 1GB
  of free memory to start up.

  Other tuning considerations:

  During the operation the repository is scanned repeatedly from low pageId
  to high pageId within each extent.  Performance can be optimized by arranging
  the extents so that the scan causes the fewest head movements on each drive
  and the threads can simultaneously initiate I/O operations on multiple extents.

  For example:   With a single disk controller and one extent per drive the
    following would provide an optimal configuration:
       Disk1: extent0
      Disk2: extent1
       ...
  With multiple controllers:
     Controller1:  Disk 1 extent0
                   Disk 2 extent2
                   Disk 3 extent4
     Controller2:  Disk 4 extent1
                   Disk 5 extent3
                   Disk 6 extent5

  For use with a RAID or SAN system, the device should be configured for 16K read
  operations from sequentially increasing blocks within an extent.

  This operation requires almost continuous access to the entire object table.
  Running on a system that can hold all of the object table in the cache is will
  allow it to perform optimally.  The object table size is approximately
  stnOopHighWater * 12 bytes.

  If the machine can be dedicated to performing this operation, a good rule of thumb
  is to set the maxThreads to numCpus + numControllers and the percentCpuActiveLimit
  should be increased to 100.

  If the operation is being performed on a system that is running a live application
  the maxThreads can still be set to a high value and the cpuActiveLimit can be
  used to control the cpu resources used by this method.  If this operation pushes the
  cpu utilization above the specified limit the main thread automatically deactivates
  threads until the load is again below the specified limit.  While threads are
  not actively working on the operation, they still respond to the commit record
  backlog and abort as needed to prevent the backlog from exceeding its specified limit.

  The pageBufSize specified must be a power of 2. Larger values can improve
  performance.  For example, in a test case where the machine was dedicated to
  performing this operation, increasing the pageBufSize from 64 to 512 reduced
  the elapsed time by about 20 percent.  On the other hand, if the operation is being
  run on a live system then smaller values may be better depending on how often the algorithm
  must abort (and clear the buffers) to prevent the commit record backlog from growing.

  Caution should be taken when setting the numSessions and pageBufSize to prevent
  the memory footprint for this process from being so large that the system starts
  swapping.

  The values of the arguments percentCpuActiveLimit and maxThreads are used as initial
  values for two statistics associated with this process, the mtPercentCpuActiveLimit
  and mtThreadLimit.  The main thread monitors these statistics and uses them to control
  the number of threads that are actively working, which is displayed in the mtActiveThreads
  statistic.  Thus the operation can be tuned dynamically by setting either the
  mtPercentCpuActiveLimit or mtThreadLimit for the process.  When the system is not performing
  adequately because the operation is consuming too much of the cpu resources, the
  mtPercentCpuActiveLimit: method can be used to decrease the value of the corresponding
  statistic and thus remove some of the load on the system caused by this operation.
  In situations where I/O is the limiting factor on the system and the operation is
  using too much I/O bandwidth, the number of active threads can be changed by using
  the mtThreadsLimit: method.  The mtPercentCpuActiveLimit can be set to any value
  between 0 and 100, mtThreadLimit can be set to values between 0 and maxThreads."

<primitive: 876>
| maxInt |
maxInt := SmallInteger maximum32bitInteger .
maxThreads _validateClass: SmallInteger; _validateMin: 1 max: maxInt .
lockWaitTime _validateClass: SmallInteger; _validateMin: -1 max: maxInt .
aBufSize _validateClass: SmallInteger ; _validateIsPowerOf2 .
self _validatePercentage: cpuPercent .
^ self _primitiveFailed: #_mfcWithMaxThreads:
       args: { maxThreads . lockWaitTime . aBufSize . cpuPercent}

]

{ #category : 'Repository Conversion' }
Repository >> _migrateGroups [

"Migrates group collections in all the ObjectSecurityPolicies of the receiver."

self do:[ :anObjectSecurityPolicy | anObjectSecurityPolicy ~~ nil
  ifTrue:[ anObjectSecurityPolicy _migrateGroups] ]

]

{ #category : 'Repository Conversion' }
Repository >> _migrateGroups: aBlock [

"Migrates group collections in all the ObjectSecurityPolicies of the receiver."

self do:[ :anObjectSecurityPolicy | anObjectSecurityPolicy ~~ nil
  ifTrue:[ anObjectSecurityPolicy _migrateGroups: aBlock ] ]

]

{ #category : 'Private' }
Repository >> _migrateWithMaxThreads: maxThreads waitForLock: lockWaitTime pageBufSize: aBufSize percentCpuActiveLimit: percentCpu mappingArrays: arrayOfMappingArrays testArray: testObjs [

" This primitive method performs a scan of the repository, migrating 
  instances of the an oldClass to a newClass.  Up to 2000 classes
  can be migrated in the same operation.  The classes must be committed. 
  An error is generated if there are any modified persistent objects in the 
  temporary object memory at the time it is executed.

  This primitive can only handle the case where new instVars are set via:
    1. set to nil.
    2. direct assignment from an old instVar
    3. it can preserve dynamic instVars if the preserveDynamic instVar in the mapping array is true.

  The primitive can handle migrating indexable objects as long as they are not already large
  objects or would have to become large in order to do the operation.  Currently the primitive 
  returns a GsBitmap containing references to any objects that it rejects.
 
  It is expected that in future releases the method may be able to handle indexable objects as well as:
     1. direct assignment from a dynamic instVar,
     2. direct assignment to a constant value (special oop or a committed object).

  This primitive cannot prerform any migration that requires the execution of a block.

  The scan only takes into account committed objects; i.e., this method
  does not operate on objects in the Temporary Object Cache (TOC).

  This primitive uses a multithreaded algorithm to sweep the active
  data pages in the repository to gather the instances of the oldClasses
  and migrate it to the newClass specification.

  See description of _mfcWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:
  for details of standard multi threaded method arguments.

  Because this transaction must commit the results, the entire operation must be 
  performed within a single transaction.  It does not use the temporary object
  cach (TOC), but the migration could result in a large commit and will likely have 
  concurrency conflicts if run while other operations on the database are in progress.
"

<primitive: 172>
 | maxInt |
  maxInt := SmallInteger maximum32bitInteger.
  maxThreads
    _validateClass: SmallInteger;
    _validateMin: 1 max: maxInt.
  lockWaitTime
    _validateClass: SmallInteger;
    _validateMin: -1 max: maxInt.
  aBufSize
    _validateClass: SmallInteger;
    _validateIsPowerOf2.
  self _validatePercentage: percentCpu.
  arrayOfMappingArrays _validateClass: Array .
  (arrayOfMappingArrays size < 2000) ifFalse: [ ArgumentError signal:
     'cannot migrate more than 2000 classes in a single operation' ] .
  testObjs isNil ifFalse: [ testObjs _validateClass: Array ].
  
^ self _primitiveFailed:
  #_migrateWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:oldClasses:toNewClasses:withInstVarMapArray:
    args: { maxThreads . lockWaitTime . aBufSize . percentCpu . arrayOfMappingArrays . testObjs }

]

{ #category : 'Private' }
Repository >> _mtStat: sesId at: statCode [

"Returns the value of the specified stat.
 statCode values are:
    0 - mtMaxThreads
    1 - mtThreadsLimit
    2 - mtPercentCpuActiveLimit

 Requires the SessionAccess privilege.

 Raises an error if no session with the specified ID was found on this
 host."

<primitive: 878>
sesId _validateClass: SmallInteger .
statCode _validateClass: SmallInteger ; _validateMin: 0 max: 2 .
self _primitiveFailed: #_mtStat:at: args: { sesId . statCode }

]

{ #category : 'Private' }
Repository >> _mtStat: sesId at: statCode put: newValue [

"Sets the value of the specified stat.
 statCode values are:
    1 - mtThreadsLimit
    2 - mtPercentCpuActiveLimit

 Requires the SessionAccess and SystemControl privileges.

 Raises an error if no session with the specified ID was found on this
 host or if the session is not performing a multithreaded repository
 scan."

<primitive: 879>

sesId _validateClass: SmallInteger .
statCode _validateClass: SmallInteger ; _validateMin: 0 max: 2 .
newValue _validateClass: SmallInteger .
self _primitiveFailed: #_mtStat:at:put: args: { sesId . statCode . newValue }

]

{ #category : 'Private Object Migration' }
Repository >> _nextDynamicIvName: aSymbol [
  "Invoked from prim 684 via IntRecurFromPrim"
^ (aSymbol asString , $x ) asSymbol

]

{ #category : 'Repository Usage Reporting' }
Repository >> _numToString: aNumber units: units [

"Convert a number representing a file size in bytes to a formatted
 String reporting the size in megabytes."

| val fmt |
val := aNumber asFloat / units .
units == 1048576.0 ifTrue:[
  ^ (val asStringUsingFormat: #( -5 0 false )), ' MB'.
].
units = 1073741824.0 ifFalse:[ Error signal:'expected divisor of G or M'].
fmt := #( -6 0 false ).
val < 10.0 ifTrue:[  fmt := #( -8 2 false) ].
val < 100.0 ifTrue:[ fmt := #( -7 1 false) ].
^ (val asStringUsingFormat: fmt) , ' GB'

]

{ #category : 'Audit and Repair' }
Repository >> _objectAuditWithMaxThreads: maxThreads kind: kindInt [
  "kindInt   0 partial audit, 1 full audit in transaction , 2 attempt repair"
| errCount |
System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

errCount := self _objectAuditWithMaxThreads: maxThreads waitForLock: 60 pageBufSize: 8
                      percentCpuActiveLimit: 100
                      csvFile: nil kind: kindInt .
errCount == 0 ifFalse:[
  RepositoryError new
     details: 'Object Audit Errors , ' , errCount asString , ' errors found' ;
     signalNotTrappable
  ].
^ true.

]

{ #category : 'Audit and Repair' }
Repository >> _objectAuditWithMaxThreads: maxThreads waitForLock: waitTimeSeconds pageBufSize: aBufSize percentCpuActiveLimit: percentCpu csvFile: aFileName kind: kindInt [

"Performs a multi-threaded object audit on all objects in the repository.

 kindInt = 0  partial audit, commit record is released after finish the OT scan,
                not a complete audit of the OT
 kindInt = 1  full audit including all OT lookups, stays in transaction for complete audit
 kindInt = 2  full audit and attempt repair.

 Requires SystemControl privileges.

 See _scanPomWithMaxThreads for definitions of maxThreads, waitTimeSeconds,
 aBufSize, cpuPersent.

 The following errors can be generated:
   BadClassId     - object references classId which does not exist.
   BadClass       - object class reference is not a class.
   BadFormat      - the object format disagrees with the class format.
   BadSecPolicy   - the object security policy is invalid.
   BadPhysSize    - the object physical size is invalid.
   BadLogicalSize - the object logical size is greater than the physical size.
   BadLength      - the object length > bodySize.
   BadBody        - the object body contains a reference to oop illegal.
   BadReference   - the object references a non existent object at offset.
   BadOt          - is in page but has an invalid OT entry.
   InFreeOops     - the object is an a page and in the FreeOop list.
   FreeOop        - does not exist but is neither in dead nor in the FreeOop list.
   BadObjId       - invalid objectId found in object.
   BadDependency  - the OT indicates the object has a dependency but it is not in depMap.
   BadDepMapKey   - the objectId is in the depMapKeys but is not a valid object.
   BadDepMapValue - the objectId is in the depMapValues but is not a valid object.
   BadDepTag      - the object is in the depMap, but the OT entry is not tagged .
   InFreeOopsTemp - object is in both object table and in stone's freeOopsTemp.

 These are always printed to the GciErrorLog.  If a valid file name is provided for the
 csvFile (comma separated values) argument then a line is written to the file for each
 error encountered.  The lines contain the following information:
   1. ErrorKind - the name associated with each of the errors defined above.
   2. ObjectId  - the objectId for the object with an error.
   3. ClassName  - the name of the object's class.
   4. Offset    - the offset (or other integer value, e.g. size)
   5. Reference - the reference that does not exist.
 If no errors are found, the file is deleted.

 If kindInt == 2 , the repairs are performed after the entire scan of the repository
 is complete.  It is possible that if a very large number of errors are found the
 system may run out of memory to record the information needed to fix the errors.
 In this case the following warning message will be generated:
    WARNING: Ran out of memory to record errors, remaining errors will not be fixed.
 If this is found in the output, then the errors before this message will be fixed, but
 any that are printed after this message will not be fixed and the audit will need to be
 rerun to correct any additional errors.  In addition an additional warning is generated at
 the end of the repair output.

 During the repair phase, each error will be printed out along with a description of the
 action taken to repair the problem.  The system will commit after every 2000 repair
 operations to avoid overflowing temporary memory space.

 The only errors that cannot be corrected by the audit code are the InFreeOops and FreeOop
 error kinds.  To fix these you will need to rebuild the FreeOop list - see admin guide for
 instructions on how to rebuild the FreeOop list."

<primitive: 392>
| maxInt |
maxInt := SmallInteger maximum32bitInteger .
maxThreads _validateClass: SmallInteger; _validateMin: 1 max: maxInt .
waitTimeSeconds _validateClass: SmallInteger ; _validateMin: -1 max: maxInt .
aBufSize _validateClass: SmallInteger ; _validateIsPowerOf2 .
self _validatePercentage: percentCpu .
aFileName _validateKindOfClass: String.
kindInt _validateClass: SmallInteger .
^ self _primitiveFailed:
       #_objectAuditWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:csvFile:kind:
       args: { maxThreads . waitTimeSeconds . aBufSize .
               percentCpu . aFileName . kindInt }
]

{ #category : 'Updating' }
Repository >> _objectSecurityPolicy: aSeg [
"Disallowed"
self shouldNotImplement: #_objectSecurityPolicy:

]

{ #category : 'Private' }
Repository >> _oopIsDead: anInteger [
"Returns a boolean indicating if the given oop is present in the
 stone's dead object set.  
 Primitive returns false if anInteger is the oop of a special object

 Expensive, this primitive makes a call to stone."

<primitive: 564>
self _primitiveFailed: #_oopIsDead: args: { anInteger }

]

{ #category : 'Private' }
Repository >> _pageForOop: anInteger [
"Returns the page ID for the given object ID by looking in the
 object table.  A result of -1 indicates the object ID is a
 free oop, or the object ID is not in the object table."
<primitive: 562>
self _primitiveFailed: #_pageForOop: args: { anInteger }

]

{ #category : 'Private' }
Repository >> _pageForOopIsTagged: anInt [
"Lookup anInt in the shared object table and return a Boolean indicating
 if the page ID for the object is tagged, indicating it is in the shared
 dependency map.  Returns nil if anInt is not in the object table or it is
 a free oop."

<primitive: 570>
self _primitiveFailed: #_pageForOopIsTagged: args: { anInt }

]

{ #category : 'Deprecated' }
Repository >> _pagesWithPercentFree: aPercent doScavenge: aBoolean [

" Deprecated, use pagesWithPercentFree:. Also see the comments for the primitive:
    _pagesWithPercentFree:withMaxThreads:maxCpuUsage:doScavenge:"

self deprecated: 'Repository>>_pagesWithPercentFree:doScavenge: deprecated v3.4 or earlier ',
 ' use #pagesWithPercentFree:'.

^ self pagesWithPercentFree: aPercent

]

{ #category : 'Repository Usage Reporting' }
Repository >> _pagesWithPercentFree: aPercent withMaxThreads: maxThreads maxCpuUsage: percentCpu doScavenge: aBoolean [

"This primitive implements pagesWithPercentFree.

 In order to execute this method you must have the GarbageCollection
 privilege.

 If the doScavenge argument is true, you must also have System Control
 privilege and the method resets the stone's scavengable pages list
 to the results computed, possibly repairing any corruption in the
 scavengable pages (see stat 7 above).

 Returns an array of size 7 with the following contents:
   1. numDataPagesProcessed
   2. totalBytesFree
   3. CFG_PAGE_SIZE_BYTES
   4. numPagesWithPercentFree
   5. pages containing oneObjWithPercentFree
   6. pages containing only one object
   7. numPagesNeedScavenging

 Use of the doScavenge option is discouraged without explicit recomendations
 from GemTalk Systems support."

<primitive: 390>
self _validatePercentage: aPercent .
maxThreads _validateClass: SmallInteger ;
  _validateMin: 1 max: SmallInteger maximum32bitInteger .
self _validatePercentage: percentCpu .
aBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_pagesWithPercentFree:withMaxThreads:maxCpuUsage:doScavenge:
       args: { aPercent . maxThreads . percentCpu . aBoolean }

]

{ #category : 'Private' }
Repository >> _primCreateExtent: extentFilename withMaxSize: aSize startNewReclaimGem: aBool [

"Creates a new Extent with the given extentFilename (aString) and sets the
 maximum size of that Extent to the given size.
 The parameter for startNewReclaimGem is obsolete in v3.2."

<primitive: 284>

extentFilename _validateKindOfClass: String.
aSize _validateClass: SmallInteger.
aBool _validateClass: Boolean .
" aSize out of range errors generated in the primitive "
^ self _primitiveFailed: #createExtent:withMaxSize:startNewReclaimGem:
       args: { extentFilename . aSize . aBool }

]

{ #category : 'Updating' }
Repository >> _primitiveAt: offset put: aValue [
"Disallowed"
self shouldNotImplement: #_primitiveAt:put:

]

{ #category : 'Instance migration' }
Repository >> _primitiveBecome: anObject forDict: aBoolean [
"Disallowed"
self shouldNotImplement: #_primitiveBecome:forDict:

]

{ #category : 'Private' }
Repository >> _primReadObjectsFromFileWithId: aSmallInt startingAt: startIndex upTo: endIndex into: anArray [

<primitive: 831>
aSmallInt _validateClass: SmallInteger .
startIndex ifNil:[ "all args nil means return the number of oops in the file"
 endIndex _validateClass: UndefinedObject .
 anArray _validateClass: UndefinedObject .
] ifNotNil:[ "normal case: request to read the file failed."
  startIndex _validateClass: SmallInteger .
  endIndex _validateClass: SmallInteger .
  anArray _validateInstanceOf: Array .
  anArray size == 0 ifFalse:[ anArray _error: #errArgTooSmall args:{ 1 } ] .
  (startIndex < 1 or:[ startIndex > endIndex]) ifTrue:[
     startIndex _error: #rtErrArgOutOfRange args:{ 1 . endIndex } ] .
].
^ self _primitiveFailed:
       #_primReadObjectsFromFileWithId:startingAt:upTo:into:
       args: { aSmallInt . startIndex . endIndex . anArray }

]

{ #category : 'Backup and Restore' }
Repository >> _primRestoreSecureBackups: arrayOfFileNames scavPercentFree: aPercent bufSize: count privateDecryptionKey: aKey passphrase: aPassphrase numThreads: threadCount options: options newSystemUserPassword: aString [

"Private.  Provides implementation of restoreSecureBackups.
  options arg controls shrink of repository 
              and immediate commit restore (restore without subsquent restore from tranlogs).
"

<primitive: 1040>
arrayOfFileNames _validateInstanceOf: Array.
count _validateClass: SmallInteger .
threadCount _validateClass: SmallInteger .
options _validateClass: SmallInteger.
self  _validatePercentage: aPercent .
aKey ifNotNil:[ aKey _validateClass: String ] .
aPassphrase ifNotNil:[ aPassphrase _validateClass: String ] .
aString ifNotNil:[ aString _validateKindOfClass: String ].
System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .
self _primitiveFailed: #_primRestoreSecureBackups:scavPercentFree:bufSize:privateDecryptionKey:passphrase:numThreads:options:newSystemUserPassword:
  args: { arrayOfFileNames . aPercent . count . aKey . aPassphrase . threadCount . options . aString } .
self _uncontinuableError

]

{ #category : 'Secure Backup and Restore' }
Repository >> _primSecureFullBackupTo: fileNames MBytes: limitArray compressKind: compKind bufSize: count encryptKind: encKind publicKeyCerts: anArrayOrString signatureHashKind: hashKind signingKey: signingKeyFn signingKeyPassphrase: aPassphrase numThreads: threadCount [

"Private.  Provides the implementation of secure full backups."

<primitive: 1039>
| a b |
fileNames _validateInstanceOf: Array.
limitArray _validateInstanceOf: Array.
compKind _validateClass: SmallInteger .
hashKind _validateClass: SmallInteger .
encKind _validateClass: SmallInteger .
threadCount _validateClass: SmallInteger .
((compKind < 0) or:[ compKind > 2])
  ifTrue:[ compKind _error: #rtErrArgOutOfRange args:{ 0 . 2 } ] .
((encKind < 0) or:[ encKind > 3])
  ifTrue:[ encKind _error: #rtErrArgOutOfRange args:{ 0 . 3 } ] .
encKind == 0  ifTrue:[ anArrayOrString _validateClass: UndefinedObject
  ] ifFalse:[
    anArrayOrString _validateKindOfClasses: { String . Array } .
    anArrayOrString _isArray ifTrue:[ | sz |
      sz := anArrayOrString size .
      (sz > 0 and:[ sz <= 24 ]) ifFalse:[
        OutOfRange new name: 'number of public keys' min: 1 max: 24 actual: sz ;
          signal.
      ]
    ]
  ] .
((hashKind < 1) or:[hashKind > 3])
  ifTrue:[ hashKind _error: #rtErrArgOutOfRange args:{ 1 . 3 } ] .
signingKeyFn _validateKindOfClass: String .
aPassphrase _validateKindOfClasses: { String . UndefinedObject } .
count _validateClass: SmallInteger .
(a := fileNames size) == (b := limitArray size) ifFalse:[
  ArgumentError signal: 'number of fileNames (' , a asString,
  ') not equal to number of limits (', b asString , ')'
].
^ self _primitiveFailed:
  #_primSecureFullBackupTo:compressKind:bufSize:encryptKind:publicKeyCerts:signatureHashKind:signingKey:signingKeyPassphrase: .

]

{ #category : 'Private' }
Repository >> _primSetArchiveLogDirectories: logDirs withPrefix: prefixString [

"Send archive log directories specification to Stone.  Sender has
 done class kind checks of all the arguments.  Stone checks for
 existence of each directory.  arrayOfDirectorySpecs should be non-empty.
 If the prefixString is nil, the STN_TRAN_LOG_PREFIX configuration
 parameter is used."

<primitive: 469>
logDirs _validateClass: Array .
self _primitiveFailed: #_primSetArchiveLogDirectories:withPrefix:
     args: { logDirs . prefixString }

]

{ #category : 'Backup and Restore' }
Repository >> _restoreBackups: arrayOfFileNames scavPercentFree: aPercent bufSize: count numThreads: threadCount options: options newSystemUserPassword: aString [

"Private.  Provides implementation of restoreBackups.
  options arg controls shrink of repository 
              and immediate commit restore (restore without subsquent restore from tranlogs). "

<primitive: 822>
arrayOfFileNames _validateInstanceOf: Array.
count _validateClass: SmallInteger .
threadCount _validateClass: SmallInteger .
options _validateClass: SmallInteger.
aString ifNotNil:[ aString _validateKindOfClass: String ].
self _validatePercentage: aPercent .
System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .
self _primitiveFailed: #_restoreBackups:scavPercentFree:bufSize:numThreads:options:newSystemUserPassword:
  args: { arrayOfFileNames . aPercent . count . threadCount . options . aString } .
self _uncontinuableError

]

{ #category : 'Backup and Restore' }
Repository >> _restoreLogs: fileId time: timeArg archiveLogs: useArchiveInt timeout: timeoutSeconds [

"Private.  Provides implementation of restoreFromLogs.
 If _primSetArchiveLogDirectories was called and
   useArchiveInt == 1, archive logs will be used.
   useArchiveInt == 2, continuous restore entered using archive logs.
   useArchiveInt == 3, stop continuous restore.

  timeoutSeconds applys only to starting continuous restore. The max time that the
  prim will wait is 1200 seconds.
  timeArg is a delay in seconds for continuousRestoreFromArchiveLogs:withDelay:
    or time in gmtSeconds for  restoreFromArchiveLogs:toPointInTime:
 "

<primitive: 395>
fileId _validateClass: SmallInteger; _validateMin: -1 max: SmallInteger maximum32bitInteger .
timeArg _validateClass: SmallInteger; _validateMin: 0 max: SmallInteger maximumValue  .
timeoutSeconds _validateClass: SmallInteger; _validateMin: 10 max: SmallInteger maximum32bitInteger .
useArchiveInt _validateClass: SmallInteger; _validateMin: 0 max: 3 .
useArchiveInt >= 2 ifTrue:[
  fileId == -1 ifFalse:[
    ArgumentError signal: 'fileId must be -1 for continuous restore'.
  ].
].
self _primitiveFailed: #_restoreLogs:time:archiveLogs:timeout: 
   args: { fileId . timeArg. useArchiveInt. timeoutSeconds }.
^self _uncontinuableError

]

{ #category : 'Private' }
Repository >> _scanPomWithMaxThreads: maxThreads waitForLock: lockWaitTime pageBufSize: aBufSize percentCpuActiveLimit: percentCpu identSet: anIdentitySet limit: aLimit objMaxSize: sizeLimit scanKind: kind toDirectory: aStringOrNil [

" This primitive method performs various scans of the Repository.
  The scans only take into account committed objects; i.e., this method
  does not operate on objects in the Temporary Object Cache (TOC).

  This primitive uses a multithreaded algorithm to sweep the active
  data pages in the repository to gather the information requested.

  The maxThreads argument specifies the maximum number of threads
  that will be used during the operation.  The actual number of
  active threads can be adjusted to a lower value automatically at runtime
  by the main thread based on the percentCpuActiveLimit and by the user
  by interactively setting the mtThreadLimit (see details below).

  The lockWaitTime argument is used to specify how many seconds method should
  wait while attempting to acquire the garbage collection (GC) lock.
  No other garbage collection operations may be started or in progress while
  this method is running.  Objects in the possibleDead or deadNotReclaimed
  sets at the start of the scan are ignored by the scan.
  A lockWaitTime of -1 or 0 means wait forever.

  The pageBufSize, which must be a power of two, specifies the number
  of pages to buffer per thread. This parameter in conjunction with the
  maxThreads largely determines the memory footprint needed to perform
  this operation.  The pageBufSize doesn't have much impact on the
  performance of the scan, so a default size of 8 is probably sufficient.

  The percentCpu specifies a level of total cpu activity at which the
  algorithm automatically inactivates threads to prevent overload
  of system resources.

  This algorithm makes use of additional sessions (threads) to achieve
  significant performance improvements.  It also makes space/time trade offs
  so that both heap memory and TemporaryObjectCache (TOC)
  resources are used.  In fact, this algorithm doesn't require much TOC,
  except for the operations that return array results, so configuring this
  process for a smaller TOC space is advantageous.

  The memory space that is needed is variable and depends upon number of
  instances being searched for, the number found, the number of sessions
  requested and the pageBufSize specified for each.

  The anIdentitySet argument should be initialized to contain the
  classes whose instances are to be located.

  The aLimit argument specifies the maximum number of instances to report
  for any class or for the objsLargerThan scan.  A value of 0 specifies
  that all instances are returned (warning - unlimited searches on a large
  database can run out of TOC space).

  The sizeLimit arg specifies only return objects with size <= sizeLimit,
  only supported by  OP_LIST_INST_PAGE_ORDER .

  The behavior of this method is dependent on the scanKind as follows.
  The scanKinds are defined by class variables.

   scanKind  Function       Result
   ===========================================================================
   OP_LIST_INSTANCES     Returns an Array.  For each element of
                          anIdentitySet, the result array contains 2
                          elements: the total number of instances and
                          an Array of the instances (possibly limited in size).

   OP_LIST_INST_TO_HIDDEN The input set must contain a single class.
                          The instances found are stored into the hidden
                          set: ListInstancesResult, and the number of
                          elements found is the primitive return value.

    OP_LIST_INST_TO_FILES Returns an Array of pairs where the odd numbered
                          elements are the classes specified in the IdentitySet
                          (in oop order) and the even numbered elements are
                          the number of instances of the preceding class
                          that were found.

                          If the toDirectory argument is nil, then only
                          array described above is returned.  If not nil,
                          then  Bitmaps containing the object ID's of
                          the instances are written to binary bitmap
                          files in the specified directory.  The resulting
                          bitmap files are named:
                             <ClassName>-<classOop>-instances.bm
                          where className is the name of the class and
                          classOop is the object ID of the class.

     OP_LIST_INST_PAGE_ORDER This operation requires a file specification
                          (passed in the toDirOrFile argument).  It is an
                          error if the file already exists.  The oops of the
                          objects in the repository that are instances
                          of the classes specified are written to the file
                          in page order and take 5 bytes each.   Returns a
                          SmallInteger representing the total number of
                          objects found.

     OP_LIST_OBJS_IN_SEC_POLICY Returns an Array.  For each element of
                          anIdentitySet, the result array contains 2
                          elements: the total number of instances and
                          an Array of the instances (possibly limited in size)
                          that have the corresponding objectSecurityPolicyId.

     OP_LIST_OBJS_IN_SEC_POLICY_TO_HIDDEN  The input set must contain a single securityPolicy.
                          The instances found are stored into the hidden
                          set: ListInstancesResult, and the number of
                          elements found is the primitive return value.

     OP_LIST_OBJS_IN_SEC_POLICY_TO_FILES  Returns an Array of pairs where the odd numbered
                          elements are the security policy ids specified
                          in the IdentitySet (in order) and the even numbered
                          elements are the number of instances in the preceding
                          security policy that were found.

                          If the toDirectory argument is nil, then only
                          array described above is returned.  If not nil,
                          then  Bitmaps containing the object ID's of
                          the instances are written to binary bitmap
                          files in the specified directory.  The resulting
                          bitmap files are named:
                           segment<objectSecurityPolicyId>-objects.bm
                          where objectSecurityPolicyId is an element of anArray.

     OP_OBJECTS_LARGER    The input set must contain a single SmallInteger
                          which is the target size for the scan. An array
                          of the objects which are larger than the target
                          size is returned. Result is limited by the
                          aLimit argument.

     OP_COUNT_INSTANCES   Returns an Array of pairs where the odd numbered
                          elements are the classes specified in the IdentitySet
                          (in oop order) and the even numbered elements are
                          the number of instances of the preceding class
                          that were found.

   ===========================================================================

   This method aborts the current transaction.  Because the view can change,
   the results of the scan can vary depending on which view was used when
   a given object was scanned.

   Scans the entire Repository once."

<primitive: 896>
| maxInt |
maxInt := SmallInteger maximum32bitInteger .
maxThreads _validateClass: SmallInteger ; _validateMin: 1 max: maxInt .
lockWaitTime _validateClass: SmallInteger ; _validateMin: -1 max: maxInt .
aBufSize _validateClass: SmallInteger;  _validateIsPowerOf2 .
self _validatePercentage: percentCpu  .
anIdentitySet _validateClass: IdentitySet .
aLimit _validateClass: SmallInteger ; _validateMin: 0 max: SmallInteger maximumValue .
sizeLimit _validateClass: SmallInteger ; _validateMin: 0 max: SmallInteger maximumValue .
kind _validateClass: SmallInteger .
aStringOrNil ifNotNil:[ aStringOrNil _validateKindOfClass: String ].
^ self _primitiveFailed:
       #_scanPomWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:identSet:limit:objMaxSize:scanKind:toDirectory:
       args: { maxThreads . lockWaitTime . aBufSize . percentCpu .
               anIdentitySet . aLimit . sizeLimit . kind . aStringOrNil }

]

{ #category : 'Private' }
Repository >> _scanPomWithMaxThreads: maxThreads waitForLock: lockWaitTime pageBufSize: aBufSize percentCpuActiveLimit: percentCpu identSet: anIdentitySet limit: aLimit scanKind: kind toDirectory: aStringOrNil [
  "synthesize objMaxSize arg"

^ self _scanPomWithMaxThreads: maxThreads waitForLock: lockWaitTime 
      pageBufSize: aBufSize percentCpuActiveLimit: percentCpu 
      identSet: anIdentitySet limit: aLimit objMaxSize: SmallInteger maximumValue
      scanKind: kind toDirectory: aStringOrNil

]

{ #category : 'Private' }
Repository >> _scanWithMaxThreads: maxThreads waitForLock: lockWaitTime pageBufSize: aBufSize percentCpuActiveLimit: percentCpu scanKind: kind with: inputArg [

" This primitive method performs various scans of the Repository.
  The scans only take into account committed objects; i.e., this method
  does not operate on objects in the Temporary Object Cache (TOC).

  This primitive uses a multithreaded algorithm to sweep the active
  data pages in the repository to gather the information requested.

  The maxThreads argument specifies the maximum number of threads
  that will be used during the operation.  The actual number of
  active threads can be adjusted to a lower value automatically at runtime
  by the main thread based on the percentCpuActiveLimit and by the user
  by interactively setting the mtThreadLimit (see details below).

  The lockWaitTime argument is used to specify how many seconds method should
  wait while attempting to acquire the garbage collection (GC) lock.
  No other garbage collection operations may be started or in progress while
  this method is running.  Objects in the possibleDead or deadNotReclaimed
  sets at the start of the scan are ignored by the scan.
  A lockWaitTime of -1 or 0 means wait forever.

  The pageBufSize, which must be a power of two, specifies the number
  of pages to buffer per thread. This parameter in conjunction with the
  maxThreads largely determines the memory footprint needed to perform
  this operation.  The pageBufSize doesn't have much impact on the
  performance of the scan, so a default size of 8 is probably sufficient.

  The percentCpu specifies a level of total cpu activity at which the
  algorithm automatically inactivates threads to prevent overload
  of system resources.

  This algorithm makes use of additional sessions (threads) to achieve
  significant performance improvements.  It also makes space/time trade offs
  so that both heap memory and TemporaryObjectCache (TOC)
  resources are used.  In fact, this algorithm doesn't require much TOC,
  except for the operations that return array results, so configuring this
  process for a smaller TOC space is advantageous.

  The memory space that is needed is variable and depends upon number of
  objects being searched for, the number found, the number of sessions
  requested and the pageBufSize specified for each.

  The behavior of this method is dependent on the scanKind as follows:

      scanKind                   Function
   ==========================================
   OP_ALL_INSTANCES              allInstances
   OP_ALL_REFERENCES             allReferences
   OP_ALL_REFS_TO_INSTANCES      allReferencesToInstancesOfClasses
   OP_ALL_REFS_BY_PARENT_CLASS   allReferencesByParentClass
   OP_ALL_OBJS_IN_SEC_POLICY     allObjectsInObjectSecurityPolicies
   OP_ALL_OBJS_LARGER_THAN       allObjectsLargerThan

"

<primitive: 1030>
 | maxInt |
  maxInt := SmallInteger maximum32bitInteger.
  maxThreads
    _validateClass: SmallInteger;
    _validateMin: 1 max: maxInt.
  lockWaitTime
    _validateClass: SmallInteger;
    _validateMin: -1 max: maxInt.
  aBufSize
    _validateClass: SmallInteger;
    _validateIsPowerOf2.
  self _validatePercentage: percentCpu.
  kind _validateClass: SmallInteger .
^ self _primitiveFailed:
  #_scanWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:scanKind:with:
    args: { maxThreads . lockWaitTime . aBufSize . percentCpu . kind . inputArg }

]

{ #category : 'Private' }
Repository >> _setArchiveLogs: dirArg withPrefix: prefixString [
  "dirArg may be a String or an Array of Strings"
| logDirs |
dirArg _isOneByteString
   ifTrue: [ logDirs := { dirArg } ]
   ifFalse: [ logDirs := dirArg ].
logDirs _validateClass: Array.
logDirs size < 1 ifTrue:[
  logDirs _error: #errArgTooSmall args:{ 1 }.
].
logDirs do:[ :aString | aString _validateKindOfClass: String ].
prefixString ifNotNil:[  prefixString _validateKindOfClass: String ].
^ self _primSetArchiveLogDirectories: logDirs withPrefix: prefixString

]

{ #category : 'Garbage Collection' }
Repository >> _setGcConfigAt: aSymbol put: aValue [

"try 10 times to change the value of the specified GC configuration
 parameter.  If cannot successfully commit within 10 tries, raises an error.
 Aborts current transaction . Returns the previous
 value of the parameter.  aSymbol must be resolvable in the UserGlobals
 for GcUser. "

| triesLeft commitResult oldVal gcUserUg |

triesLeft := 10 .
commitResult := false .
gcUserUg := ((AllUsers userWithId:'GcUser' ) resolveSymbol:#UserGlobals ) value.
[triesLeft > 0 and:[commitResult == false] ] whileTrue:[
  System beginTransaction .
  oldVal := gcUserUg at: aSymbol .
  gcUserUg at: aSymbol put: aValue .

  commitResult := System commitTransaction  .
  triesLeft := triesLeft - 1.
  commitResult
    ifFalse:[ System abortTransaction .  System sleep: 2 . ]
    ifTrue:[ GsFile gciLogServer:'--setGcConfig: set ' , aSymbol , ' to ' ,
		aValue asString
    ].
].
commitResult ifFalse:[ ^ self error:'Unable to modify Gc configuration'].
^ oldVal

]

{ #category : 'Hot Standby' }
Repository >> _setupContinuousRestore: archiveLogDir [

  | ready printed count |
  ready := false .
  printed := false .
  count := 0 .
  [ ready ] whileFalse:[ | info |
    info := self restoreStatusInfo .
    (info at: 2) == 0 ifTrue:[ self _checkRestoreActive ].
    (info at: 12) == 1 ifTrue:[
      ready := true
    ] ifFalse:[
      printed ifFalse:[ printed := true .
         GsFile gciLogServer:'  [Info:] Waiting for a log receiver to connect to stone.' ].
      count := count + 1 .
      count > 15 ifTrue:[ ImproperOperation signal:
  	'continuousRestore cannot start, no log receiver connected to stone.' ].
      System sleep: 1 .
    ]
  ].
  self _setArchiveLogs: archiveLogDir withPrefix: nil.

]

{ #category : 'Updating' }
Repository >> _setVersion [

"This method writes the Stone executable version string to the receiver
 and does a synchronous checkpoint."

<primitive: 289>

self _primitiveFailed: #_setVersion .
self _uncontinuableError

]

{ #category : 'Private' }
Repository >> _shadowPagesByExtent [

"This method aborts the current transaction.
 Returns an Array ; each element is a SmallInteger , the number of pages needing
 reclaim in that extent."

<primitive: 526>
self _primitiveFailed: #_shadowPagesByExtent

]

{ #category : 'Private' }
Repository >> _sharedDepMapLookupForOop: anInt [
"Lookup anInt in the shared dependency map and return an integer which indicates
 the value in the map for the given OOP (which is the DependencyList for the
 given OOP).

 Returns nil if anInt is not present in the shared dependency map."

<primitive: 571>
self _primitiveFailed: #_sharedDepMapLookupForOop: args: { anInt }

]

{ #category : 'Private' }
Repository >> _shrinkExtents [
"Truncate all Extents of the Repository to remove internal free space between
 the last used page in each extent and the end of the file containing the
 extent.  Has no effect for extents on a raw disk partition.
 May only be executed by SystemUser .

 WARNING, you should take a full backup of your extents before
 executing _shrinkExtents.  If the machine crashes (such as a power failure)
 while shrinkExtents is executing, the repository may be unusable.

 Requires that no other user sessions be logged in.
 Stops and restarts the symbol creation session and all gc sessions .
 Because it stops the symbol creation session, this method can only
 be run as SystemUser .

 If DBF_PRE_GROW is enabled in the configuration file then this the extents
 will be grown again the next time Stone is restarted, thus cancelling the
 effect of this method.

 Returns receiver if successful, otherwise returns a String describing
 sessions that prevented the shrink."

  | result errCount finallyBlock oldRemoveThresh pagesNeedRemoveOfs removalThreshSym finallyDone |
  System myUserProfile isSystemUserProfile ifFalse:[
    Error signal:'Must be logged in as SystemUser to run _shrinkExtents'.
  ].
  errCount := 0 .
  removalThreshSym := #StnPageMgrRemoveMinPages .
  oldRemoveThresh := System stoneConfigurationAt: removalThreshSym .
  System stoneConfigurationAt: removalThreshSym put: 0 .
  pagesNeedRemoveOfs := System cacheStatisticsDescriptionForStone indexOf:'PagesWaitingForRemoval'.
  finallyDone := false .
  finallyBlock := [
      finallyDone ifFalse:[
        System startSymbolGem .
        System ensureGcRunning .
        finallyDone := true .
        System stoneConfigurationAt: removalThreshSym put: oldRemoveThresh .
      ].
  ].
  [
    [ | done sesList count otherSes |
      GsFile gciLogServer:'--shrink: doing reclaimAll'.
      SystemRepository reclaimAllWait: 300.
      GsFile gciLogServer:'--shrink: stopping symbol creation session and gc sessions'.
      System stopSymbolCreationSession .   "stop first to fix 31606"
      System stopAllGcGems .
      done := false .
      count := 1 .
      [ done ] whileFalse:[
	sesList := System currentSessions .
	sesList size == 1 ifTrue:[
	  GsFile gciLogServer:'--shrink: sesList size == 1'.
	  done := true .
	] ifFalse:[
	  count <= 10 ifTrue:[
	    count := count + 1 .
	    System sleep: 1 .
	  ] ifFalse:[
	    otherSes := System otherSessionNames .
	    done := true
	  ].
	].
      ].
      otherSes ifNil:[ | needRemove limit |
	GsFile gciLogServer:'--shrink: doing simple commits'.
	3 timesRepeat:[
	  System _simpleCommitForReclaim: false .
	  System abortTransaction .
	  System sleep: 3 .
	  System abortTransaction .
	  System sleep: 3 .
	].
	GsFile gciLogServer:'--shrink: attempting startCheckpointSync '.
	System startCheckpointSync ifFalse:[
	  result := 'ERROR, false from startCheckpointSync. ' .
	].
	System abortTransaction .
	System sleep: 15 .
	needRemove := 1 .
	count := 0 .
        limit := 20 .
	GsFile gciLogServer:'--shrink: waiting for pagesNeedRemove <=20 ' .
	[ needRemove > limit  and:[count < 30]] whileTrue:[
	  System sleep: 1 .
	  System abortTransaction .
	  needRemove := (System cacheStatistics: 1) at: pagesNeedRemoveOfs .
	  count := count + 1
	].
	needRemove > limit ifTrue:[
	  result == nil ifTrue:[ result := String new ].
	  result addAll: ' ERROR, pagesNeedRemove = ' , needRemove asString .
	].
	GsFile gciLogServer:'--shrink: needRemove = ' , needRemove asString.
	self __shrinkExtents .
	result ifNil:[ result := self ].
      ] ifNotNil: [
        GsFile gciLogServer: '--- sessions: ', System currentSessionsReport .
	result := 'ERROR, sessions preventing shrink:' , otherSes .
      ].
    ] ensure: finallyBlock .
  ] onSynchronous: Error do:[:ex |
    errCount := errCount + 1 .
    " GsFile gciLogServer:'--shrink: error count ' , errCount asString . "
    errCount == 1 ifTrue:[ finallyBlock value ] .
    ex outer
  ].
  ^ result

]

{ #category : 'Private' }
Repository >> _shrinkRepositoryInt [
  "SessionTemps current at: #GsShrinkRepository put: 1      
   is done during slowfilein when building $GEMSTONE/bin/extent0.dbf , 
   to minimize the size of the resulting extent."
  SessionTemps current at: #GsShrinkRepository
    ifAbsent: [ ^ 0 ].
  ^ 1
]

{ #category : 'Private' }
Repository >> _signalMfcWarning: mfcPrimResult [
  | ex msg arr |
  arr := mfcPrimResult .
  (ex := Warning new) _number: 2515"WARNING_FINISHED_MFC" ; args: arr .
  msg := 'markForCollection found ' ,
        (arr atOrNil: 1) asString , ' live objects, ',
        (arr atOrNil: 2) asString , ' dead objects(occupying approx ' ,
        (arr atOrNil: 3) asString , ' bytes)' .
  (arr atOrNil: 4) ifNotNil:[:n |
    msg addAll: ', ', n asString , ' possibleDeadSymbols'
  ].
  ex signal: msg .
  ^ ex

]

{ #category : 'Updating' }
Repository >> _unsafeAt: offset put: aValue [
"Disallowed"
self shouldNotImplement: #_unsafeAt:put:

]

{ #category : 'Backup and Restore' }
Repository >> _validateCompressionKind: anInt [

((anInt >= 0) and:[ anInt <= 2 ])
  ifTrue:[ ^ true ].
((anInt >= 101) and:[ anInt <= 109 ] )
  ifTrue:[ ^ true ].
((anInt >= 200) and:[anInt < 212])
  ifTrue:[ ^ true ].

^ ArgumentError signal: 'invalid compression kind ', anInt asString

]

{ #category : 'Private' }
Repository >> _validateFileNames: fileNames limits: mByteLimit [
  | limitArray |
  System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ].
  fileNames _validateInstanceOf: Array.
  fileNames size == 0 ifTrue: [ fileNames _error: #rtErrArgOutOfRange args:{ 1 . 256 }].
  1 to: fileNames size do: [ :i | (fileNames at: i) _validateKindOfClass: String].
  (mByteLimit isKindOfClass: SmallInteger)  ifTrue:[
     ( self _checkMbyteLimit: mByteLimit ) ifFalse:[
           mByteLimit _error: #rtErrArgOutOfRange args: {0 . 4000 * 1024} ].
      limitArray := Array new: fileNames size.
      1 to: limitArray size do: [ :i |
        limitArray at: i put: mByteLimit
      ]
   ] ifFalse: [
      (mByteLimit isKindOfClass: Array)
         ifTrue:  [ limitArray := mByteLimit copy ]
         ifFalse: [ ( ArgumentTypeError new name: 'mByteLimit' expectedClass:
                    { Array . SmallInteger } actualArg: mByteLimit ) signal].
      1 to: (fileNames size min: limitArray size) do:[:j | | v |
        (self _checkMbyteLimit: (v := limitArray at: j)) ifFalse:[
          OutOfRange signal: 'element ', j asString , ' of limits Array is invalid'
        ].
      ].
      limitArray size + 1 to: fileNames size do:[:k|
         limitArray at: k put: 0.
      ].
   ].
  ^limitArray

]

{ #category : 'Private' }
Repository >> _validateObjectSecurityPolicyId: anObjectSecurityPolicyId [

"Generates an error if anObjectSecurityPolicyId is not a SmallInteger
 >= 0 and <= self size .
 Note that objectSecurityPolicyId 0 specifies world write,
 and does not have an associated GsObjectSecurityPolicy . "
| sz |
anObjectSecurityPolicyId _validateClass: SmallInteger .
(anObjectSecurityPolicyId < 0 or:[ anObjectSecurityPolicyId > (sz := self size) ])
  ifTrue:[anObjectSecurityPolicyId _error: #rtErrArgOutOfRange args:{ 0 . sz }
].

]

{ #category : 'Private' }
Repository >> _validatePercentage: anInt [
  ^ self class _validatePercentage: anInt

]

{ #category : 'Deprecated' }
Repository >> _validateSegmentId: anObjectSecurityPolicyId [
"Deprecrated, use #_validateObjectSecurityPolicyId: instead of this method."

self deprecated: 'Repository>>_validateSegmentId: deprecated v3.4 or earlier ',
 ' use #_validateObjectSecurityPolicyId:'.
^ self _validateObjectSecurityPolicyId: anObjectSecurityPolicyId

]

{ #category : 'Private' }
Repository >> _validateString: aString [

aString == nil ifFalse: [
  (aString class isSubclassOf: String) ifFalse:[
    aString _errorExpectedClass: String ].
  aString size > 1023 ifTrue:[ aString _error: #errArgTooLarge args:{ 1023 } ].
  ].

]

{ #category : 'Private Object Migration' }
Repository >> _writeAllDataPageIdsToFiles: numFiles threads: numThreads [
  "Create files dataPages1..N.txt in current directory.  
   Signals an error if number of scavengable pages is > 0  .
   Distributes pageIds across the files like this:
     For N = 3 and a list of pageIds  1 2 3 4 5 6 7 8 9 ,
     the files would contain 1 4 7 ,  2 5 8  ,  3 6 9
   Uses numThreads thread to build the list of pageIds.

   Returns a SmallInteger, the total number of data pages
  "
<primitive: 683>
self _primitiveFailed: #_writeAllDataPageIdsToFiles:threads: args: {  numFiles . numThreads }

]

{ #category : 'Private' }
Repository >> _writeFdcArray: anArray toFile: aFileNameString [
"Given an Array of objects produced by the one of the findDisconnectedObjects*
 methods, write the array of objects to a file which can be loaded by
   System (C) >> readHiddenSet:fromSortedFile:
 for later analysis of disconnected objects.

 anArray must be an Array and contain objects sorted by object ID.
 aFileNameString must be an instance of String that refers to a file which
 does not yet exist."

<primitive: 557>
anArray _validateInstanceOf: Array.
anArray isCommitted ifFalse:[ ArgumentError signal: 'anArray is not a committed object'].
aFileNameString _validateKindOfClass: String.
self _primitiveFailed: #_writeFdcArray:toFile:
     args: { anArray . aFileNameString } .
self _uncontinuableError.

]

{ #category : 'Copying' }
Repository >> add: anObj [
"Disallowed"
self shouldNotImplement: #add:

]

{ #category : 'Copying' }
Repository >> addAll: aCollection [
"Disallowed"
self shouldNotImplement: #addAll:

]

{ #category : 'Deprecated' }
Repository >> addTransactionLog: deviceOrDirectory  replicate: replicateSpec size: aSize [

self deprecated: 'Repository>>addTransactionLog:replicate:size: is Obsolete, replicate arguments is ignored'.
^ self addTransactionLog: deviceOrDirectory size: aSize

]

{ #category : 'Transaction Logging' }
Repository >> addTransactionLog: deviceOrDirectory  size: aSizeMB [

"Add deviceOrDirectory to the configuration parameter STN_TRAN_LOG_DIRECTORIES.

 The aSizeMB argument must be a positive SmallInteger; it is added to the value
 of the STN_TRAN_LOG_SIZES configuration parameter. Units are M bytes.

 Any environment variables in deviceOrDirectory are expanded using the environment
 of the stone process.  An error is signalled if deviceOrDirectory does not exist
 on the stone's host, or if deviceOrDirectory specifies /dev/null .
 /dev/null can only be specified as a single entry in STN_TRAN_LOG_DIRECTORIES
 read by stone at stone startup .

 This method requires the FileControl privilege."

<primitive: 337>

deviceOrDirectory _validateKindOfClass: String .
aSizeMB _validateClass: SmallInteger .
aSizeMB > 0 ifFalse:[ aSizeMB _error: #errArgTooSmall args:{ 1 } ] .
^ self _primitiveFailed: #addTransactionLog:size:
       args: { deviceOrDirectory . aSizeMB }

]

{ #category : 'Listing Instances' }
Repository >> allInstances: inputArg [

" This method scans the repository to find all instances of each class in the 
  argument. The argument may be a Class, an Array of Classes, or a GsBitmap of 
  Classes.  The result is returned as either a GsBitmap, or an Array of (Class, 
  GsBitmaps) pairs.  

  If <inputArg> is a single class, this method returns a GsBitmap containing 
  instances of that class.

  If <inputArg> is an Array or a GsBitmap of classes, single class, this method 
  returns an Array containing sub Arrays, where each sub Array consists of:
     <aClass> , <aGsBitmap with the instances of the class>

  Note that the order in which classes appear in the result Array may differ
  from the order which they appear in the inputArg if it is an Array.

  For special classes (Character, SmallInteger, SmallDouble, etc.), no instances 
  are returned. 

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData.

  The entire repository is scanned once and the result contains only committed
  objects that were present in the view of the data at that time and for
  which the user executing this method has read authorization to the object.

  The scan is done in an non-aggressive manner and uses only 2 threads.  To
  complete the operation in less time, see the version of this method with
  the fast prefix.

  An error is raised if the argument is or contains an object that is not a
  class. For special classes (Character, SmallInteger, SmallDouble, etc.), no 
  instances are returned.
"
^ self _doScan: OP_ALL_INSTANCES  fast: false with: inputArg

]

{ #category : 'Listing Instances' }
Repository >> allInstances: inputArg threads: numThreads [

" Same as allInstances: except that it allows specifying the number of threads to use for the scan.
 The number of threads can be overridden for any other listing instances method by executing:
    setDefaultNumThreads: numThreads
 before executing the method.
"

^self setDefaultNumThreads: numThreads during: [
   self _doScan: OP_ALL_INSTANCES fast: false with: inputArg
	]

]

{ #category : 'Listing By Security Policy' }
Repository >> allObjectsInObjectSecurityPolicies: inputArg [

" This method scans the repository for objects that have the specified
  ObjectSecurityPolicyId.

  If the argument is a single ObjectSecurityPolicyId this method returns a
  GsBitmap with the objects that have the specified ObjectSecurityPolicyId.
  If the input argument is an Array or GsBitmap of ObjectSecurityPolicyIds
  then this method returns an Array of sub Arrays, where each sub Array contains:
    <anObjectSecurityPolicyId> <aGsBitmap with the objects whose SecurityPolicyId
                                matches anObjectSecurityPolicyId>
  Note that the order in which classes appear in the result array may differ
  from the order which they appear in the inputArg if it is an Array.

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData.

  The entire repository is scanned once and the result contains only committed
  objects that were present in the view of the data at that time and for
  which the user executing this method has read authorization to the object.

  The scan is done in an non-aggressive manner and uses only 2 threads.  To
  complete the operation in less time, see the version of this method with
  the fast prefix.
"
self _checkObjectSecurityPolicyArg: inputArg.
^ self _doScan: OP_ALL_OBJS_IN_SEC_POLICY fast: false with: inputArg

]

{ #category : 'Listing By Security Policy' }
Repository >> allObjectsInObjectSecurityPolicies: inputArg threads: numThreads [


" Same as allObjectsInObjectSecurityPolicies: except that it allows specifying the
 number of threads to use for the scan.
 The number of threads can be overridden for any other listing references method by executing:
    setDefaultNumThreads: numThreads
 before executing the method.
"

self _checkObjectSecurityPolicyArg: inputArg.
^self setDefaultNumThreads: numThreads during: [
    self _doScan: OP_ALL_OBJS_IN_SEC_POLICY fast: false with: inputArg
	]

]

{ #category : 'Listing By Size' }
Repository >> allObjectsLargerThan: aSize [

" Returns a GsBitmap with the objects that are larger than aSize.
  The argument aSize must be a SmallInteger.

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData.

  The entire repository is scanned once and the result contains only committed
  objects that were present in the view of the data at that time and for
  which the user executing this method has read authorization to the object.

  The scan is done in an non-aggressive manner and uses only 2 threads.  To
  complete the operation in less time, see the version of this method with
  the fast prefix.
"

^ self _doScan: OP_ALL_OBJS_LARGER_THAN fast: false with: aSize

]

{ #category : 'Listing By Size' }
Repository >> allObjectsLargerThan: aSize limit: aLimit [
 | arr |
 arr := self _scanPomWithMaxThreads: 8 waitForLock: 3 pageBufSize: 8
          percentCpuActiveLimit: 90 identSet: (IdentitySet with: aSize)
          limit: aLimit scanKind: OP_OBJECTS_LARGER  toDirectory: nil.
 ^ arr at: 2

]

{ #category : 'Listing By Size' }
Repository >> allObjectsLargerThan: aSize threads: numThreads [

" Same as allObjectsLargerThan: except that it allows specifying the number of threads to use for the scan.
 The number of threads can be overridden for any other listing references method by executing:
    setDefaultNumThreads: numThreads
 before executing the method.
"

^self setDefaultNumThreads: numThreads during: [
    self _doScan: OP_ALL_OBJS_LARGER_THAN fast: false with: aSize
	]

]

{ #category : 'Listing References' }
Repository >> allReferences: inputArg [

" This method scans the repository for references to specific objects.

  If the argument is a single object this method returns a GsBitmap containing
  the objects that reference it.  If the input argument is an Array or GsBitmap
  of objects then this method returns an Array of sub Arrays, where each
  sub Array contains:
     <anInstance> , <aGsBitmap with the objects that reference it>
  Note that the order in which classes appear in the result array may differ
  from the order which they appear in the inputArg if it is an Array.

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData.

  The entire repository is scanned once and the result contains only committed
  objects that were present in the view of the data at that time and for
  which the user executing this method has read authorization to the object.

  The scan is done in an non-aggressive manner and uses only 2 threads.  To
  complete the operation in less time, see the version of this method with
  the fast prefix.
"

^ self _doScan: OP_ALL_REFERENCES fast: false with: inputArg

]

{ #category : 'Listing References' }
Repository >> allReferences: inputArg threads: numThreads [

" Same as allreferences: except that it allows specifying the number of threads to use for the scan.
 The number of threads can be overridden for any other listing references method by executing:
    setDefaultNumThreads: numThreads
 before executing the method.
"

^self setDefaultNumThreads: numThreads during: [
    self _doScan: OP_ALL_REFERENCES fast: false with: inputArg
	]

]

{ #category : 'Listing References' }
Repository >> allReferencesByParentClass: inputArg [

" This method scans the repository for references to objects in the inputArg.

  The result is an Array of sub Arrays, where each sub Array contains:
     <aClass> , <aGsBitmap> containing the objects that reference any of the objects 
                            in the inputArg whose parent object is an instance of <aClass>


  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData.

  The entire repository is scanned once and the result contains only committed
  objects that were present in the view of the data at that time.

  The scan is done in an non-aggressive manner and uses only 2 threads.  To
  complete the operation in less time, see the version of this method with
  the fast prefix.
"

^ self _doScan: OP_ALL_REFS_BY_PARENT_CLASS fast: false with: inputArg

]

{ #category : 'Listing References' }
Repository >> allReferencesToInstancesOfClasses: inputArg [

" This method scans the repository for objects that reference instances of a class.

  If the argument is a single class this method returns a GsBitmap containing
  objects that reference instances of that class.  If the input argument is
  an Array or GsBitmap of classes then this method returns an Array of sub Arrays,
  where each sub Array contains:
     <aClass>, <aGsBitmap containing the objects that reference instances of the class>
  Note that the order in which classes appear in the result array may differ
  from the order which they appear in the inputArg if it is an Array.

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData.

  This operation requires two scans of the repository, the first scan finds
  instances of the class and the second scan finds the references to these instances.
  The result contains only committed objects that were present in the view of
  the data at that time and for which the user executing this method has read
  authorization to the object.

  The scan is done in an non-aggressive manner and uses only 2 threads.  To
  complete the operation in less time, see the version of this method with
  the fast prefix.
"

^ self _doScan: OP_ALL_REFS_TO_INSTANCES fast: false with: inputArg

]

{ #category : 'Transaction Logging' }
Repository >> allTranlogDirectories [

"Returns an Array of Strings, each of which represents 1 element of the
 STN_TRAN_LOG_DIRECTORIES configuration parameter."

| result |
result := System stoneConfigurationAt: #STN_TRAN_LOG_DIRECTORIES .
^ result class == Array ifTrue:[ result ] ifFalse:[ { result } ].

]

{ #category : 'Transaction Logging' }
Repository >> allTranlogSizes [

"Returns an Array of SmallIntegers, each of which represents 1 element of the
 STN_TRAN_LOG_SIZES  configuration parameter.  Each element has units of M bytes"

| result |
result := System stoneConfigurationAt: #STN_TRAN_LOG_SIZES .
^ result class == Array ifTrue:[ result ] ifFalse:[ { result } ].

]

{ #category : 'Updating' }
Repository >> at: offset put: aValue [
  "Disallowed, installed after Upgrade2C class >> _patchSecurityPolicy15 has run"
  self shouldNotImplement: #at:put:

]

{ #category : 'Deprecated Page-Order Operations' }
Repository >> auditPageOrderOopFileWithId: aSmallInt [

"Deprecated, use
   | arr fileInfo |
   fileInfo := GsBitmap fileInfo: <bitmapFilePath>.
   arr := Array new.
   arr add: fileInfo at: 1.
   arr add: fileInfo at: 4.
   ^ arr

 Reads the page order oop file with the ID returned by the
 Repository>>openPageOrderOopFile: method and counts the number of
 objects in the file which are no longer valid.  Objects are
 considered invalid if they no longer exist (i.e., are not
 present in the shared object table) or present in the dead objects
 set maintained by stone.

 The session must be in a transaction when this method is invoked.  If the
 session is not in a transaction, an #rtErrPrimOutsideTrans error is raised.

 Returns an Array containing 2 elements 
    1 - a SmallInteger, total number of oops in the file.
    2 - a SmallInteger, number of invalid oops in the file.

 Returns an Array containing 2 nils if aSmallInt is not a valid file
 ID returned by the Repository>>openPageOrderOopFile: method."

<primitive: 833>
aSmallInt _validateClass: SmallInteger .
^ self _primitiveFailed: #auditPageOrderOopFileWithId: args: { aSmallInt }

]

{ #category : 'Deprecated' }
Repository >> auditWithLimit: sizeLimit [

" Deprecated, use #objectAudit instead of this method "

self deprecated: 'Repository>>auditWithLimit: deprecated in v3.2. Use objectAudit instead.'.
^ self objectAudit

]

{ #category : 'Updating' }
Repository >> basicAt: offset put: aValue [
"Disallowed"
self shouldNotImplement: #basicAt:put:

]

{ #category : 'Disconnected Objects' }
Repository >> basicFindDisconnectedObjectsAndWriteToFile: aFileNameString pageBufferSize: pageBufSize saveToRepository: saveToRep withMaxThreads: maxThreads maxCpuUsage: aPercentage [

"Perform an FDC on the system and write the list of
 dead objects to given file.  If saveToRep is true, also store the results
 in the array under the key #FdcResults in Globals.

 To save the dead objects to an Array but not write them to a file, set
 aFileNameString to nil and saveToRep to true.  It is illegal for
 aFileNameString to be nil and for saveToRep to be false.

 This method aborts the current transaction.

 pageBufSize must be a power of two and specifies the number of pages to buffer per thread.

 This method will return an error if the repository is in restore mode.

 See also the comments in the method
  Repository>> _fdcWithMaxThreads:...

 for a desciption of the result array and a more detailed description of
 this method."

| fdcResults arrayOfDead |

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

(System transactionMode == #autoBegin)
  ifFalse:[System transactionMode: #autoBegin].

(SystemRepository restoreActive)
  ifTrue:[^ self _error: #rtErrCommitDbInRestore].

System abortTransaction.
System commitTransaction
  ifFalse:[^ self _error: #rtErrGcCommitFailure].
fdcResults := self _createFdcResults .

arrayOfDead := fdcResults at: 3.
saveToRep ifTrue:[ arrayOfDead ifNil:[ fdcResults at: 3 put: Array new ]]
         ifFalse:[ fdcResults at: 3 put: nil ].

System commitTransaction
  ifFalse:[^ self _error: #rtErrGcCommitFailure].

^ self _fdcWithMaxThreads: maxThreads
      waitForLock: 90 pageBufSize: pageBufSize percentCpuActiveLimit: aPercentage
      toFile: aFileNameString resultArray: fdcResults

]

{ #category : 'Storing and Loading' }
Repository >> basicLoadFrom: passiveObj [
"Disallowed."
self shouldNotImplement: #basicLoadFrom:

]

{ #category : 'Storing and Loading' }
Repository >> basicLoadFrom: passiveObj size: aSize [
"Disallowed"
self shouldNotImplement: #basicLoadFrom:size:

]

{ #category : 'Storing and Loading' }
Repository >> basicLoadFromNoRead: passiveObj size: aSize [
"Disallowed"
self shouldNotImplement: #basicLoadFromNoRead:size:

]

{ #category : 'Storing and Loading' }
Repository >> basicLoadFromOld: passiveObj [
"Disallowed"
self shouldNotImplement: #basicLoadFromOld:

]

{ #category : 'Instance migration' }
Repository >> become: anObj [
"Disallowed"
self shouldNotImplement: #become:

]

{ #category : 'Reference Paths' }
Repository >> buildLimitSetForRefPathScan [

"Build a list of objects to serve as the limit set for a reference path scan.
 The reference path scan ends when a reference from any objects
 returned by this method are found during the scan.
 Returns an Array containing all of the objectSecurityPolicies,
 SharedDependencyLists, and the AllSymbols dictionary.

 See also GsSingleRefPathFinder class >> buildLimitSetForRefsToClass "

|set symUser dictBlock symListBlock|

set := IdentitySet new.
set addAll: SystemRepository. "Add all objectSecurityPolicies"
"Get all symbol list dictionaries and all versions of all classes from all user profiles.
 Using a set prevents duplicates."

"Handles SymbolDictionary and SymbolListDictionary"
dictBlock := [:k :v|
      (set includes: v)
	ifFalse:[
          v isBehavior ifTrue:[
            set add: v .
            set add: v class .
	    1 to: v classHistory size do:[:p| |class|
	      class := v classHistory at: p.
              class ~~ v ifTrue:[
	        set add: class.
	        set add: class class
              ]
           ]
         ].
     ].
  ].

symListBlock := [:eachDict|
    (set includes: eachDict) ifFalse:[
      set add: eachDict.
      eachDict keysAndValuesDo: dictBlock .
    ].
].

AllUsers do:[:aUserPro|
  set add: aUserPro .
  aUserPro symbolList do: symListBlock
].

set add: (Globals at: #SharedDependencyLists).

	"gemstone64, AllSymbols is now in SymbolUser's UserGlobals . "
symUser := AllUsers userWithId:'SymbolUser' .
set add: (symUser resolveSymbol: #AllSymbols) value .

^set asArray

]

{ #category : 'Instance migration' }
Repository >> changeClassTo: aCls [
"Disallowed"
self shouldNotImplement: #changeClassTo:

]

{ #category : 'Private' }
Repository >> cleanupFdcResults [
"Delete the results produced by a previous findDisconnectedObjects operation."

(Globals associationAt: #FdcResults otherwise: nil) ifNotNil:[:assoc | | arr |
  arr := assoc value .
  arr ifNil:[ arr := Array new: 4 . assoc value: arr ].
  arr at: 1 put: 0.
  arr at: 2 put: 0.
  (arr at: 3) ifNotNil:[:a | a size: 0 ].
  arr at: 4 put: false.
].
^true

]

{ #category : 'Private' }
Repository >> cleanupFdcResultsForMgc: forMgc [
 "Deprecated.
  Delete the results produced by a previous findDisconnectedObjects operation.
  The forMgc argument is ignored."

 self deprecated: 'Repository>>cleanupFdcResultsForMgc:: deprecated in v3.3. Use cleanupFdcResults .'.
 ^ self cleanupFdcResults

]

{ #category : 'Deprecated Page-Order Operations' }
Repository >> closePageOrderOopFileWithId: aSmallInt [

"Deprecated. Page order operations done using GsBitmap don't require close,
 so there is no replacement.
 Closes a file previously opened with the Repository>>openPageOrderOopFile:
 method and releases the garbage collection lock.

 Returns true if successful; false if the file was not open or was already
 closed."

<primitive: 832>
aSmallInt _validateClass: SmallInteger .
^ self _primitiveFailed: #closePageOrderOopFileWithId: args: { aSmallInt }

]

{ #category : 'Backup and Restore' }
Repository >> commitRestore [

"Terminates a restore operation and puts the system back into a normal
 operating mode in which commits are allowed and reenables logins.

 The repository must be in the ""restore from logs"" state, otherwise this
 method generates an error. As part of the transition from restoreFromLogs
 to normal operation, a new transaction log is opened with fileId N,
 where N := (fileId of last restored transaction log + 1 )  .
 If the directories or partitions specified by STN_TRAN_LOG_DIRECTORIES
 contain any tranlog(s) with fileId >= N , the commitRestore will fail
 and repository will be left in ""restore from logs"" .
 If you want to terminate a restore from current logs before the last current log,
 the remaining tranlogs which were not restored must be deleted or moved
 out of STN_TRAN_LOG_DIRECTORIES .
 A successful transition writes a checkpoint to both the newly opened tranlog
 and the extents.

 If the logs have not been restored to the end of the known logs, i.e.,
 the last restore operation was a restoreToEndOfLog: or restoreToPointInTime:
 then a warning is issued and the restore is stopped at the current
 location.  This use of commitRestore can result in failure to restore
 all previously committed transactions.  However, this allows a Repository
 to be restored as far as practical when some log files are lost or corrupted.

 Upon successful completion, the session is automatically logged out and
 the RestoreLogSuccess error (4048) is generated.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser.

 This method stops continuous restore if it was active as if the
 #stopContinuousRestore method had been executed before this method.

 Because any session that logs in or performs a restore operation
 is prevented from doing normal commits, this session is automatically
 logged out when the restore operation completes."

self restoreActive ifFalse: [
  ImproperOperation signal:
     'You cannot commitRestore without first executing restoreFromBackup'].
^ System _zeroArgPrim: 141

]

{ #category : 'Backup and Restore' }
Repository >> commitRestoreForFailoverAfterWaitingUpTo: anInt [

"On a slave stone, waits up to seconds for the fail over record from the master
stone to be replayed. If seconds is 0, this method checks for the fail over 
record once and does not block. If seconds is -1, this method waits forever 
for the fail over record.

If the fail over record is detected, then the logreceiver process is killed 
(if necessary), a commitRestore is executed, and the session is terminated
with a RestoreLogSuccess error (4048). If the fail over record is not detected 
within the timeout, false is returned.

Raises an exception if the repository is not in restore from log state."

| endMs sys |
sys := System .
endMs := sys _timeMs + (anInt * 1000).
(self waitForFailoverFromMasterUpToSeconds: anInt)
  ifTrue:[
    sys logreceiverSessionId > 0 ifTrue:[
      sys killLogReceiver ; _sleepMs: 1000 .
      [sys logreceiverSessionId > 0] whileTrue:[
         sys _sleepMs: 500.
         sys _timeMs > endMs
           ifTrue:[ RepositoryError signal: 'Timeout waiting for logreceiver to die' ].
       ]] .
  self commitRestore
]. "session gets logged out"

^ false "fail over record not detected"

]

{ #category : 'Backup and Restore' }
Repository >> commitRestoreWithLostData [
"A special case of commitRestore that allows the commit to occur when
 the backup file was corrupted and there was lost data.

 Terminates a restore operation and puts the system back into a normal
 operating mode in which commits are allowed and reenables logins.

 The repository must be in the restoreFromLogLostData state, otherwise this
 method generates an error.

 If the logs have not been restored to the end of the known logs, i.e.,
 the last restore operation was a restoreToEndOfLog: or restoreToPointInTime:
 then the commitRestoreWithLostData should be used to allow the repository
 to be restored as far as practical when some log files are lost or corrupted.

 Upon successful completion, the session is automatically logged out and
 the RestoreLogSuccess error (4048) is generated.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser.

 Because any session that logs in or performs a restore operation
 is prevented from doing normal commits, this session is automatically
 logged out when the restore operation completes."

self restoreActive ifFalse: [
  ImproperOperation signal:
     'You cannot commitRestore without first executing restoreFromBackup'].
^ System _zeroArgPrim: 144

]

{ #category : 'Hot Standby' }
Repository >> continuousRestoreFromArchiveLogs: archiveLogDir [
  ^ self continuousRestoreFromArchiveLogs: archiveLogDir timeout: 60 
  

]

{ #category : 'Hot Standby' }
Repository >> continuousRestoreFromArchiveLogs: archiveLogDir timeout: timeoutSeconds [

"Similar to restoreFromArchiveLogs: except that stone enters
 continuous restore mode, and control is returned to this session
 executing this method. Intended for use on a hot standby stone
 where the archive logs are being written by a log receiver process
 which is receiving data from the master stone's log feeder process.

 Requires that at least one tranlog file already exists in directory(s) 
 specified in archiveLogDir. The logsender must be running; it is 
 recommended to start the logreceiver before executing this method.

 The tranlog containing the checkpoint from which restore will start is
 shown in the result of Repository >> restoreStatus.  If this tranlog
 does not exist in the directory(s) specified,  an attempt is made to open
 it using the directories defined in 
   System stoneConfigurationAt: #STN_TRAN_LOG_DIRECTORIES 

 Waits up to timeoutSeconds, to a maximum of 1200 seconds, for a log 
 receiver to connect to stone, and the first tranlog to be readable, else 
 signals an error. timeoutSeconds must be >= 10 . 

 The restore can be terminated by  Repository >> stopContinuousRestore.

 Repository >> restoreStatus is usable while the continuous restore
 is running.
"

  self _setupContinuousRestore: archiveLogDir.
  self _restoreLogs: -1 time: 0 archiveLogs: 2 timeout: timeoutSeconds .

]

{ #category : 'Hot Standby' }
Repository >> continuousRestoreFromArchiveLogs: archiveLogDir withDelay: seconds [
  ^ self continuousRestoreFromArchiveLogs: archiveLogDir withDelay: seconds timeout: 60 

]

{ #category : 'Hot Standby' }
Repository >> continuousRestoreFromArchiveLogs: archiveLogDir withDelay: seconds timeout: timeoutSeconds [

"Similar to continuousRestoreFromArchiveLogs except that stone enters
 continuous restore mode with the specified delay.

 Records are transferred immediately to the standby system but the
 processing of commits  and checkpoints are delayed by the specified
 number of seconds to allow the system to be stopped to avoid
 restoring an operation on the database that is not desired.
"
  self _setupContinuousRestore: archiveLogDir.
  self _restoreLogs: -1 time: seconds archiveLogs: 2 timeout: timeoutSeconds .

]

{ #category : 'Copying' }
Repository >> copy [
"Disallowed"
self shouldNotImplement: #copy

]

{ #category : 'Instance Counting' }
Repository >> countInstances: anArray [

"Scans the entire Repository and generates an Array of instance counts of the
 the classes contained in anArray.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To
 complete the operation in less time, see the #fastCountInstances:
 method."

^ self countInstances: anArray withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90

]

{ #category : 'Instance Counting' }
Repository >> countInstances: anArray numThreads: maxThreads objMaxSize: aSize [

"Scans the entire Repository and generates an array of instance counts of the
 the classes contained in anArray, counting only objects with size <= aSize .
 Does not include in-memory objects."

| inputSet resultInSetOrder result inputArraySize anObj |
System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

inputSet := self _arrayOfClassesAsSet: anArray .
inputSet size < 1 ifTrue:[ ^ { } ] .

resultInSetOrder := self _scanPomWithMaxThreads: maxThreads
       waitForLock: 90
       pageBufSize: 8
       percentCpuActiveLimit: 95
       identSet: inputSet
       limit: 0  objMaxSize: aSize
       scanKind: 8 "OP_COUNT_INSTANCES - OP_FIRST_SCAN_KIND"
       toDirectory: nil .

inputArraySize := anArray size .
result := Array new: inputArraySize .
1 to: inputArraySize do:[:j|
  anObj := anArray at: j .
  result at: j put: ( resultInSetOrder at: (inputSet _offsetOf: anObj) * 2)
  ].
^ result

]

{ #category : 'Instance Counting' }
Repository >> countInstances: anArray withMaxThreads: maxThreads maxCpuUsage: aPercentage [

"Scans the entire Repository and generates an Array of instance counts of the
 the classes contained in anArray. Does not include in-memory objects.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData. The entire
 Repository is scanned once."

| inputSet resultInSetOrder result inputArraySize anObj |

System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

inputSet := self _arrayOfClassesAsSet: anArray .
inputSet size < 1 ifTrue:[ ^ { } ] .

resultInSetOrder := self _scanPomWithMaxThreads: maxThreads
       waitForLock: 90
       pageBufSize: 8
       percentCpuActiveLimit: aPercentage
       identSet: inputSet
       limit: 0
       scanKind: OP_COUNT_INSTANCES
       toDirectory: nil .

inputArraySize := anArray size .
result := Array new: inputArraySize .
1 to: inputArraySize do:[:j|
  anObj := anArray at: j .
  result at: j put: ( resultInSetOrder at: (inputSet _offsetOf: anObj) * 2)
  ].
^ result

]

{ #category : 'Extent Operations' }
Repository >> createExtent: extentFilename [

"Creates a new Extent with a file named extentFilename (a String).  The new
 Extent has no maximum size.

 This method updates the DBF_EXTENT_NAMES Stone option.  It does not require
 the system to be in single-user mode.

 If the given file already exists, then this method generates an error and the
 given Extent is not added to the logical Repository.

 Creating an extent with this method bypasses any setting the user may have
 specified for the DBF_PRE_GROW option at system startup.  As extents created
 with this method have no maximum size, their files cannot be grown to an
 initial size.

 If GemStone is being run using weighted disk resource allocation, then
 the new Extent will be given a weight equal to the average weight of all
 other extents.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser."

^ self createExtent: extentFilename withMaxSize: 0

]

{ #category : 'Extent Operations' }
Repository >> createExtent: extentFilename withMaxSize: aSize [

"Creates a new Extent with the given extentFilename (aString) and sets the
 maximum size of that Extent to the given size.

 If aSize == 0 , then the maximum physical size of the extent is limited
 to disk space or 33554G bytes , whichever comes first . If aSize > 0 ,
 it specifies the maximum physical size of the file in megabytes.  The
 minimum size of a new extent is 16 MB.  Attempts to create an extent of
 a smaller size will raise an error.  The actual maximum size of the new
 extent will be equal to aSize if and only if aSize is a multiple of 16.
 Otherwise the actual maximum size will be the next lowest multiple of 16.
 For example, an extent created with a maximum size of 260 MB really has a
 maximum size of only 256 MB.

 This method updates the DBF_EXTENT_NAMES and DBF_EXTENT_SIZES Stone options.
 It does not require the system to be in single-user mode.

                                  Note:
   The extent is created with the default ownership and permissions of the
   Stone process.  If this is not the same as the ownership and permissions of
   the other extents, then Unix utilities must be used to change the ownership
   or permissions of the new file; such changes may be made without stopping
   the Stone, and should be made a soon as possible, to avoid other sessions
    encountering authorization errors.

 If the given file already exists, then this method generates an error and the
 given Extent is not added to the logical Repository.

 If the Stone option DBF_PRE_GROW is set to true, then this method will cause
 the newly created extent's file to be grown to the given size.  If the grow
 fails, then this method returns an error and the new Extent is not added to the
 logical Repository.

 If GemStone is being run using weighted disk resource allocation, then
 the new Extent will be given a weight equal to the average weight of all
 other Extents.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser."

 ^self _primCreateExtent: extentFilename withMaxSize: aSize startNewReclaimGem: false

]

{ #category : 'Deprecated' }
Repository >> createExtent: extentFilename withMaxSize: aSize startNewReclaimGem: aBool [

"Creates a new Extent with the given extentFilename (aString) and sets the
 maximum size of that Extent to the given size.

 If aSize == 0 , then the maximum physical size of the extent is limited
 to disk space or 33554G bytes , whichever comes first . If aSize > 0,
 it specifies the maximum physical size of the file in megabytes.  The
 minimum size of a new extent is 16 MB.  Attempts to create an extent of
 a smaller size will raise an error.  The actual maximum size of the new
 extent will be equal to aSize if and only if aSize is a multiple of 16.
 Otherwise the actual maximum size will be the next lowest multiple of 16.
 For example, an extent created with a maximum size of 260 MB really has a
 maximum size of only 256 MB.

 This method updates the DBF_EXTENT_NAMES and DBF_EXTENT_SIZES Stone options.
 It does not require the system to be in single-user mode.

                                  Note:
   The extent is created with the default ownership and permissions of the
   Stone process.  If this is not the same as the ownership and permissions of
   the other extents, then Unix utilities must be used to change the ownership
   or permissions of the new file; such changes may be made without stopping
   the Stone, and should be made a soon as possible, to avoid other sessions
    encountering authorization errors.

 If the given file already exists, then this method generates an error and the
 given Extent is not added to the logical Repository.

 If the Stone option DBF_PRE_GROW is set to true, then this method will cause
 the newly created extent's file to be grown to the given size.  If the grow
 fails, then this method returns an error and the new Extent is not added to the
 logical Repository.

 If GemStone is being run using weighted disk resource allocation, then
 the new Extent will be given a weight equal to the average weight of all
 other Extents.

 If aBool is true, then a new reclaim gem is started to reclaim pages from the
 newly created extent.  If aBool is false, then no reclaim gem is started.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser."

self deprecated: 'Repository>>createExtent:withMaxSize:startNewReclaimGem:
    deprecated in v3.2, reclaimgems no longer extent-specific; use createExtent:withMaxSize:'.
 ^self _primCreateExtent: extentFilename withMaxSize: aSize startNewReclaimGem: aBool

]

{ #category : 'Transaction Logging' }
Repository >> currentLogDirectoryId [

"Returns a positive SmallInteger, which is the one-based offset specifying the
 element of the configuration list STN_TRAN_LOG_DIRECTORIES for the current
 transaction log.  (See also the currentLogFile method.)"

^ (System stoneConfigurationAt:#StnCurrentTranLogDirId) + 1

]

{ #category : 'Transaction Logging' }
Repository >> currentLogFile [

"Returns a String containing the file name of the transaction log file to which
 log records are being appended.  If the result is of size 0, then a
 log file creation has failed and stone is waiting for more tranlog space
 to be made available. 
 If stone is in restore from logs, signals an error."

 (self restoreStatusInfo at: 2) >= 2 ifTrue:[
    RepositoryError signal:'Repository is in restore from logs, no tranlogs open for writing.'
 ].
 ^ System stoneConfigurationAt:#StnCurrentTranLogNames

]

{ #category : 'Transaction Logging' }
Repository >> currentLogFileId [

"Returns a positive SmallInteger, which is the internal fileId of
 the current transaction log."

^ (self _logInfo: (self currentLogDirectoryId)) at: 4

]

{ #category : 'Transaction Logging' }
Repository >> currentTranlogSizeMB [

"Returns an Integer that is the size of the currently active transaction log in
 units of Megabytes. Returns zero of no tranlog is open for writing."

 ^ self _currentLogInfo ifNotNil:[:info | info at: 5 ] ifNil:[ 0 ].

]

{ #category : 'Accessing' }
Repository >> dataDictionary [

"Accesses the user-defined data dictionary. (Reserved)"

^ dataDictionary

]

{ #category : 'Copying' }
Repository >> deepCopy [
"Disallowed"
self shouldNotImplement: #deepCopy

]

{ #category : 'Repository Usage Reporting' }
Repository >> defaultFileExtension [
"Answer a string which is the default file extension for database extents
 and tranlogs of this repository."

^ System repositoryIsEncrypted ifTrue:[ '.sdbf' ] ifFalse:[ '.dbf' ]

]

{ #category : 'Backup and Restore' }
Repository >> disableRestoreFromLogs [
"Causes the next restoreFromBackup operation (normal or secure) by the session 
 to immediately commit the restore returning the repository to normal mode.
 Restoring from transaction logs after the restore is not possible when this
 option is used and executing the method #commitRestore is unnecessary."

  SessionTemps current at: #GsDisableRestoreFromLogs put: 2
]

{ #category : 'Clustering' }
Repository >> extentForPage: aPageId [

"Returns a SmallInteger specifying an offset into the result from the
 fileNames method.

 The argument aPageId is an Integer, such as the result from the Object | page
 method, specifying a page in the receiver."

^ (aPageId bitShift: -31 ) + 1

]

{ #category : 'Repository Usage Reporting' }
Repository >> extentsReport [

"Returns a String which reports on the name, size, and amount of free space
 for each extent and the size and free space of the entire logical Repository.
 Report is in units of GBytes or MBytes, depending on repository size."

^ self _extentsReport: nil

]

{ #category : 'Repository Usage Reporting' }
Repository >> extentsReportMB [

"Returns a String which reports on the name, size, and amount of free space
 for each extent and the size and free space of the entire logical Repository.
 Report is in units of MBytes. "

^ self _extentsReport: 1048576.0

]

{ #category : 'Backup and Restore' }
Repository >> failoverFromMasterFinished [

"On a slave stone, returns a boolean indicating if the fail over record from the master has
been replayed. Returns true if the fail over record has been replayed, false otherwise. 
Raises an exception if the repository is not in restore from log state."

^ self waitForFailoverFromMasterUpToSeconds: 0

]

{ #category : 'Hot Standby' }
Repository >> failOverStatus [
  | arr str val |
  arr := self failOverStatusInfo .
  (arr at: 1) ifNotNil:[:aString | ^ aString ].

  (str := 'This repository last tranlog commit file ', (arr at: 9) asString) 
      add: ' record '; add: (arr at: 10) asString; lf ;
  add:'  last checkpoint file '; add: (arr at: 2) asString ; 
       add: ' record '; add: (arr at:3) asString ; lf ;
  add: 'Other repository restored to commit file '; add:(arr at:6) asString;
     add:  ' record '; add:(arr at:7) asString; lf ;
  add: '  last restored checkpoint file '; add:(arr at:4) asString;
     add:' record '; add: (arr at: 5) asString ; lf ;
  add: 'This repository ' .
  val := arr at: 8 .
  str add:( val == 2 ifTrue:[ 'in restore from logs.']
             ifFalse:[ val == 1 ifTrue:[ 'commits suspended.']
                              ifFalse:[ 'commits allowed.' ]]).
  val := arr at: 11 .
  str lf; add: (val > 0 ifTrue:[ ' Other repository logreceiver connected.'] ifFalse:[' Other repository logreceiver not connected.']).
  ^ str

]

{ #category : 'Hot Standby' }
Repository >> failOverStatusInfo [
  "Returns an Array describing the tranlog position of this repository
   and the restore status of the slave system to which this repository's
   logsender is connected. The Array contains the following elements:

    1: an error String or nil
    2: a SmallInteger, tranlog fileId of last checkpoint written by this stone .
    3: a SmallInteger, tranlog blockId  of last checkpoint written by this stone .

    4: a SmallInteger, tranlog fileId of last checkpoint restored by slave stone .
    5: a SmallInteger, tranlog blockId of last checkpoint restored by slave stone .
    6: a SmallInteger, tranlog fileId of last commit restored by slave stone.
    7: a SmallInteger, tranlog blockId of last commit restored by slave stone.
    8: a SmallInteger  commitsSuspended (0=false or 1=true 2= in restore from logs)
    9: a SmallInteger, tranlog fileId of last tranlog write by this stone .
   10: a SmallInteger, tranlog recordId of last tranlog write by this stone .
   11: a SmallInteger  logreceiverConnected (0 = false, 1 = true)
"
  | str |
  "Allow executing even if this repository is in restore, so it can be executed
   after failOverToSlave."
  System sessionIsOnStoneHost ifTrue:[
    System logsenderSessionId == 0 ifTrue:[
      str := 'logsender not running on this repository, failOverStatus not available'.
    ].
  ] ifFalse:[
    str := 'this session must be on stone host (on stone shared cache)' .
  ] .
  str ifNil:[ | fs | 
    fs :=  System _zeroArgPrim: 80 .
    fs _isArray ifTrue:[ ^ fs ] 
               ifFalse:[ str :=  fs asString ].
  ].
  ^ { str . 0 . 0 . 0 . 0 . 0 . 0 . 0 . 0 . 0 . 0}

]

{ #category : 'Hot Standby' }
Repository >> failOverToSlave [
   "For use on the master system of a hotstandby pair.
    Suspends commits, starts a checkpoint,
    and waits for completion of the checkpoint, and puts this
    repository into restore mode.

    When continuous restore reads a checkpoint containing a failover timestamp
    and the failover timestamp represents a time after the time at which
    the slave stone last started,  continuous restore will stop.
    If the failover timestamp is before the time at which
    the slave stone last started, continuous restore will continue replay
    of the transaction logs.

    After waiting for restore in the slave system to finish restoring the checkpoint
    just written by failOverToSlave , execute
       Repository >> commitRestore
    on the slave system.  
    The slave system is now ready to be the master.
    To do this, start an inverted pair of logsender/logreceiver, and on this former-master 
    system, execute continuousRestoreFromArchiveLogs: , using the tranlog directory
    specified in the startlogreceiver just executed. 

    Requires SystemControl privilege.
    If the repository is in restore mode, signals an error, and has no effect.

    Returns a failover time ( a DateAndTime ).
    "
  | ts |
  ts := System _zeroArgPrim: 140 .
  ^ DateAndTime posixSeconds: ts offset: nil

]

{ #category : 'Listing Instances' }
Repository >> fastAllInstances: inputArg [

" Same as allInstances: except that the scan is performed more agressively,
  using more of the system resouces for both cpu and I/O.
"

^ self _doScan: OP_ALL_INSTANCES fast: true with: inputArg

]

{ #category : 'Listing By Security Policy' }
Repository >> fastAllObjectsInObjectSecurityPolicies: inputArg [


" Same as allObjectsInObjectSecurityPolicies: except that the scan is performed more agressively,
  using more of the system resouces for both cpu and I/O.
"

self _checkObjectSecurityPolicyArg: inputArg.
^ self _doScan: OP_ALL_OBJS_IN_SEC_POLICY fast: true with: inputArg

]

{ #category : 'Listing By Size' }
Repository >> fastAllObjectsLargerThan: aSize [

" Same as allObjectsLargerThan: except that the scan is performed more agressively,
  using more of the system resouces for both cpu and I/O.
"

^ self _doScan: OP_ALL_OBJS_LARGER_THAN fast: true with: aSize

]

{ #category : 'Listing References' }
Repository >> fastAllReferences: inputArg [

" Same as allreferences: except that the scan is performed more agressively,
  using more of the system resouces for both cpu and I/O.
"

^ self _doScan: OP_ALL_REFERENCES fast: true with: inputArg

]

{ #category : 'Listing References' }
Repository >> fastAllReferencesByParentClass: inputArg [

" Same as allReferencesByParentClass: except that the scan is performed more agressively,
  using more of the system resouces for both cpu and I/O.
"

^ self _doScan: OP_ALL_REFS_BY_PARENT_CLASS fast: true with: inputArg

]

{ #category : 'Listing References' }
Repository >> fastAllReferencesToInstancesOfClasses: inputArg [

" Same as allReferencesToInstancesOfClasses: except that the scan is performed more agressively,
  using more of the system resouces for both cpu and I/O.
"

^ self _doScan: OP_ALL_REFS_TO_INSTANCES fast: true with: inputArg

]

{ #category : 'Instance Counting' }
Repository >> fastCountInstances: anArray [

"Same as the #countInstances: method except the scan is performed
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is in progress."

^ self countInstances: anArray withMaxThreads: self _aggressiveMaxThreadCount
       maxCpuUsage: 95

]

{ #category : 'Listing Instances' }
Repository >> fastListInstances: anArray [

"Same as the #listInstances: method except the scan is performed
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

| reportArr res |
reportArr := self fastListInstances: anArray limit: 0 .
res := Array new: anArray size .
1 to: anArray size do:[:j |
  res at:j put: (reportArr at: j * 2 ) . "array of instances"
].
^ res

]

{ #category : 'Listing Instances' }
Repository >> fastListInstances: anArray limit: aSmallInt [

"Same as the #listInstances:limit: method except the scan is performed
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is in progress."

^ self listInstances: anArray limit: aSmallInt toDirectory: nil
      withMaxThreads: self _aggressiveMaxThreadCount maxCpuUsage: 95
	memory: 1

]

{ #category : 'Listing Instances' }
Repository >> fastListInstances: anArray toDirectory: aString [

"Same as the #listInstances:toDirectory: method except the scan is performed
 aggressively in order to complete in as little time as possible,
 and does not include in-memory objects.

 This method may consume most or all host system resources while it
 is in progress."

^ self _allInstances: anArray toDirectory: aString fast: true

]

{ #category : 'Listing Instances' }
Repository >> fastListInstancesInPageOrder: anArrayOfClasses toFile: aString [
"Same as the #listInstancesInPageOrder:toFile: method except the scan is performed
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is in progress."

^ self listInstancesInPageOrder: anArrayOfClasses toFile: aString
       withMaxThreads: self _aggressiveMaxThreadCount maxCpuUsage: 95


]

{ #category : 'Listing Instances' }
Repository >> fastListInstancesToHiddenSet: aClassOrArray [

"Same as the #listInstancesToHiddenSet: method except the scan is performed
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

| arr |
arr := aClassOrArray.
arr isBehavior ifTrue: [ arr := { arr } ].

^ self _buildListInstResult: (self fastAllInstances: arr)

]

{ #category : 'Listing By Security Policy' }
Repository >> fastListObjectsInObjectSecurityPolicies: anArray [

"Same as the #listObjectsInObjectSecurityPolicies: method except the scan
 is performed aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is
 in progress."

^ self fastListObjectsInObjectSecurityPolicies: anArray withLimit: 0

]

{ #category : 'Listing By Security Policy' }
Repository >> fastListObjectsInObjectSecurityPolicies: anArray toDirectory: aString [

"Same as the #listObjectsInObjectSecurityPolicies:toDirectory: method
 except the scan is performed aggressively in order to complete in as
 little time as possible.

 This method may consume most or all host system resources while it is in progress."

aString _validateKindOfClass: String .
^ self _listObjectsInObjectSecurityPolicies: anArray limit: 0 
       scanKind: OP_LIST_OBJS_IN_SEC_POLICY_TO_FILES
       toDirectory: aString withMaxThreads: self _aggressiveMaxThreadCount
       maxCpuUsage: 95

]

{ #category : 'Listing By Security Policy' }
Repository >> fastListObjectsInObjectSecurityPolicies: anArray withLimit:  aSmallInt [

"Same as the #listObjectsInObjectSecurityPolicies:withLimit: method except
 the scan is performed aggressively in order to complete in as little time
 as possible.

 This method may consume most or all host system resources while it is in progress."

^ self _listObjectsInObjectSecurityPolicies: anArray limit: aSmallInt
      scanKind: OP_LIST_OBJS_IN_SEC_POLICY
      toDirectory: nil withMaxThreads: self _aggressiveMaxThreadCount
      maxCpuUsage: 95

]

{ #category : 'Listing By Security Policy' }
Repository >> fastListObjectsInObjectSecurityPolicyToHiddenSet: anObjectSecurityPolicyId [

"Same as the #listObjectsInObjectSecurityPolicyToHiddenSet: method except the scan is
 performed aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is in progress."

self _validateObjectSecurityPolicyId: anObjectSecurityPolicyId .
^ self _listObjectsInObjectSecurityPolicies: { anObjectSecurityPolicyId }
       limit: 0 scanKind: OP_LIST_OBJS_IN_SEC_POLICY_TO_HIDDEN 
       toDirectory: nil
       withMaxThreads: self _aggressiveMaxThreadCount maxCpuUsage: 95

]

{ #category : 'Listing References' }
Repository >> fastListReferences: anArray [

"Same as the #listReferences: method except the scan is performed
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is in progress."

^ self _buildAllRefsResult: (self fastAllReferences: anArray) forInput: anArray

]

{ #category : 'Listing References' }
Repository >> fastListReferencesToInstancesOfClasses: anArray toDirectory: aString [

"Similar to listReferencesToInstancesOfClasses:toDirectory: except that
 the scan is performed aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it is in progress."

^ self _listReferencesToInstancesOfClasses: anArray  toDirectory: aString
       withMaxThreads: self _aggressiveMaxThreadCount maxCpuUsage: 95

]

{ #category : 'Garbage Collection' }
Repository >> fastMarkForCollection [

"Same as the #markForCollection method except the scan is performed
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

^ self markForCollectionWithMaxThreads: self _aggressiveMaxThreadCount
  waitForLock: 90 pageBufSize: 128 percentCpuActiveLimit: 95

]

{ #category : 'Garbage Collection' }
Repository >> fastMarkForCollectionWithPageBufSize: pageBufSize [

"Same as the #fastMarkForCollection method except you can specify
 a larger pageBufSize than the default 128.

 This method may consume most or all host system resources while it
 is in progress."

^ self markForCollectionWithMaxThreads: self _aggressiveMaxThreadCount
  waitForLock: 90 pageBufSize: pageBufSize percentCpuActiveLimit: 95

]

{ #category : 'Migrating Objects' }
Repository >> fastMigrateMt: arrayOfMappings [
"Same as the #migrateMt: method except the migrate is performed
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

  ^ self _basicMigrateWithMaxThreads: self _aggressiveMaxThreadCount waitForLock: 3
          pageBufSize: 8 percentCpuActiveLimit: 95 mappingArrays: arrayOfMappings testArray: nil

]

{ #category : 'Audit and Repair' }
Repository >> fastObjectAudit [
"Same as the #objectAudit method except the audit is performed
 aggressively in order to complete in as little time as possible.

 This method may consume most or all host system resources while it
 is in progress."

^ self objectAuditWithMaxThreads: self _aggressiveMaxThreadCount
  percentCpuActiveLimit: 95

]

{ #category : 'Audit and Repair' }
Repository >> fastObjectAuditPartial [

"Similar to objectAuditPartial except executes as fast as possible."

^self _objectAuditWithMaxThreads: self _aggressiveMaxThreadCount kind: 0 .

]

{ #category : 'Repository Usage Reporting' }
Repository >> fastPagesWithPercentFree: aPercent [

"Same as pagesWithPercentFree: except that the scan is performed aggressively
 in order to complete in as little time as possible."

^ self _pagesWithPercentFree: aPercent withMaxThreads: self _aggressiveMaxThreadCount
                              maxCpuUsage: 95 doScavenge: false

]

{ #category : 'Listing References' }
Repository >> fastPrivateReferences: anArray [

"Same as fastAllReferences, but includes references from 
 internal nodes of large objects and Nscs."

^ self _buildAllRefsResult: 
   (self _doScan: OP_PRIVATE_REFERENCES fast: true with: anArray)
  forInput: anArray 

]

{ #category : 'Accessing' }
Repository >> fileNames [

  <primitive: 429>

"Returns an Array containing the filenames for the extents
 of the receiver.

 Each element within the returned Array contains a String representing
 the filename of the Nth extent (where N is the index into the
 returned Array)."

  self _primitiveFailed: #fileNames .
  self _uncontinuableError

]

{ #category : 'Repository Usage Reporting' }
Repository >> fileSize [

"Returns an integer giving the total physical size, in bytes, of all
 of the physical extents that compose the logical Repository."

| total stats |

total := 0.
stats := self _extentStatistics: -1.
1 to: stats size do:[:j |
  total := total + ((stats at: j) at: 1)
].
^ total

]

{ #category : 'Repository Usage Reporting' }
Repository >> fileSizeOfExtent: extentFilename [

"Returns the physical size, in bytes, of the extent with the given name.

 If the given file is not an active member of the logical Repository
 represented by the receiver, then this method generates an error."

^(self _dbfStatistics: extentFilename) at: 1

]

{ #category : 'Repository Usage Reporting' }
Repository >> fileSizeReport [
"Returns a String which reports on the name, size, and amount of free space
 for each extent and the size and free space of the entire logical Repository.
 Report is in units of GBytes or MBytes, depending on repository size."

^ self _fileSizeReport: nil

]

{ #category : 'Repository Usage Reporting' }
Repository >> fileSizeReportMB [

"Returns a String which reports on the name, size, and amount of free space
 for each extent and the size and free space of the entire logical Repository.
 Report is in units of MBytes. "

^ self _fileSizeReport: 1048576.0

]

{ #category : 'Disconnected Objects' }
Repository >> findDisconnectedObjectsAndWriteToFile: aFileNameString pageBufferSize: anInt saveToRepository: saveToRep [
"Perform an FDC operation on the system and write the list of dead objects
 to the given file, and if saveToRep is true, also store the results in the
 array under the key #FdcResults in Globals.

 The file or (Globals at:#FdcResults) can be used to analyze the objects
 that would be collectable by a markForCollection, or to manually reconnect
 objects that have been dereferenced by application coding errors.

 To save the dead objects to an Array but not write them to a file, set
 aFileNameString to nil and saveToRep to true.  It is illegal for
 aFileNameString to be nil and for saveToRep to be false.

 The markGcCandidatesFromFile operation is no longer supported, use
 markForCollection to actually perform garbage collection.
 If a previous FDC created FdcResults , you must execute
    (Globals associationAt: :#FdcResults) value: nil . System commit
 before the markForCollection , to make the garbage collectable.

 The pageBufSize, which must be a power of two, specifies the number
 of pages to buffer per thread.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.

 This method will return an error if the repository is in restore mode.

 *** WARNING *** WARNING *** WARNING *** WARNING *** WARNING ***
 This method is intended to complete the FDC operation as quickly as possible
 and will start 2 threads per CPU core.  Therefore it may consume all available
 CPU and disk I/O host system resources.

 The FDC operation can be run less aggressively by calling the
 #basicFindDisconnectedObjectsAndWriteToFile: method with a smaller number
 of threads.
 *** WARNING *** WARNING *** WARNING *** WARNING *** WARNING ***

 See comments in the method:
   Repository>>_fdcWithMaxThreads:waitForLock:pageBufSize:
               percentCpuActiveLimit:toFile:resultArray:
 for a desciption of the result array and a more detailed description of
 this method."

^ self basicFindDisconnectedObjectsAndWriteToFile: aFileNameString
       pageBufferSize: anInt
       saveToRepository: saveToRep
       withMaxThreads: self _aggressiveMaxThreadCount
       maxCpuUsage: 95

]

{ #category : 'Disconnected Objects' }
Repository >> findDisconnectedObjectsAndWriteToFile: aFileNameString withMaxThreads: maxThreads maxCpuUsage: aPercentage [

"Start an FDC operation where the results will be save to the given file but
 not stored in the repository.

 The file can be loaded and used to analyze the objects that would be collectable
 by a markForCollection, or to manually reconnect objects that have been dereferenced
 by application coding errors.

 Starts maxThreads on the host system and allows the host to run up to aPercentage
 percent CPU usage.  A page buffer of 128 pages (2 MB) is allocated per thread.

 See the comments in method:
   Repository>>_fdcWithMaxThreads:waitForLock:pageBufSize:
               percentCpuActiveLimit:toFile:resultArray:
 for a desciption of the result array and a more detailed description of
 this method."

^ self basicFindDisconnectedObjectsAndWriteToFile: aFileNameString
       pageBufferSize: 128 "must be a power of 2"
       saveToRepository: false
       withMaxThreads: maxThreads
       maxCpuUsage: aPercentage

]

{ #category : 'Repository Analysis' }
Repository >> findObjectsConnectedTo: anArray [
"Return an Array of all objects reachable from the objects in anArray .
 Objects implicitly referenced from the header like the class are
 not included in the sweep.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData."

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ self _findConnectedObjs: anArray.

]

{ #category : 'Private' }
Repository >> findObjsConnectedTo: anArray [
"Return an Array of all objects reachable from the objects in anArray.
 Objects implicitly referenced from the header like the class are
 not included in the sweep.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData."

self deprecated: 'Repository>>findObjsConnectedTo: private method deprecated in v3.2, use public findObjectsConnectedTo:'.

^self findObjectsConnectedTo: anArray

]

{ #category : 'Deprecated (Single Ref Path)' }
Repository >> findReferencePathToObject: anObject [

"Finds a reference path to an object.  Returns an instance of GsSingleRefPathResult,
 for which each element represents a connection leading between the search
 objects and the reference in the repository.
 The first element will usually be one of the objects in the limit set, and the last
 element will be the argument.

 The searchObj must be a committed non-special object. This method is not to be used
 for classes or metaclasses. Objects in the stone's list of dead objects are excluded from
 the scan.  It is an error if the object is in the stone's dead objects set.

 Only a single reference path to each object is returned. It is possible for any object
 to be connected to the repository through multiple reference paths.

 It is possible for an object to be disonnected from the repository and therefore have no
 reference path. In this case, the result GsSingleRefPathResult isDead will be true.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData.

 A reference path scan for an object is terminated and considered completed if any class
 object or GsObjectSecurityPolicy is found in the reference path.  Normally all classes are
 connected to the repository and will be included in the limit set.  However, for garbage
 collection purposes, disconnected classes are considered live objects if any other object
 references them.  In other words, a class is a live object if any live instances of that
 class exist in the repository.
"

self deprecated: 'Repository>>findReferencePathToObject: deprecated v3.4. ',
 'Use (GsSingleRefPathFinder newForSearchObjects: {anObject}) runScan buildResultObjects first'.

System needsCommit ifTrue: [ System _error: #rtErrAbortWouldLoseData ] .

^ (GsSingleRefPathFinder newForSearchObjects: { anObject }) runScan
      buildResultObjects first

]

{ #category : 'Deprecated (Single Ref Path)' }
Repository >> findReferencePathToObjects: anArray findAllRefs: findAllRefs printToLog: printToLog [

"Finds a reference path to an array of objects.  Returns an array of GsSingleRefPathResult
 showing how each object is connected to the repository by reference.

 Note that arguments to this method other than <anArray> are not used in v3.4.

 See the comments in findReferencePathToObject: for more information.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData."

self deprecated: 'Repository>>findReferencePathToObjects:findAllRefs:printToLog: deprecated v3.4. ',
 'Use (GsSingleRefPathFinder newForSearchObjects: anArray) runScan buildResultObjects'.

System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ (GsSingleRefPathFinder newForSearchObjects: anArray) runScan
      buildResultObjects

]

{ #category : 'Deprecated (Single Ref Path)' }
Repository >> findReferencePathToObjs: anArray limitObjArray: limitArray findAllRefs: findAllRefs printToLog: printToLog [

"Finds a reference path to an array of objects.  Returns an array of GsSingleRefPathResult
 showing how each object is connected to the repository by reference.

 Note that arguments to this method other than <anArray> are not used in v3.4.

 See the comments in findReferencePathToObject: for more information.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData."

self deprecated: 'Repository>>findReferencePathToObjs:limitObjArray:findAllRefs:printToLog: deprecated v3.4. ',
 'Use (GsSingleRefPathFinder newForSearchObjects: anArray) runScan buildResultObjects'.

System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ (GsSingleRefPathFinder newForSearchObjects: anArray) runScan
      buildResultObjects

]

{ #category : 'Repository Usage Reporting' }
Repository >> freeSpace [

"Returns an integer that gives the number of bytes of free space in the logical
 Repository.  This quantity is equal to the number of free pages in the
 Repository times the size of a page."

| total stats |

total := 0.
stats := self _extentStatistics: -1.
1 to: stats size do:[:j |
  total := total + ((stats at: j) at: 2)
].
^ total

]

{ #category : 'Repository Usage Reporting' }
Repository >> freeSpaceInExtent: extentFilename [

"Returns the number of bytes of free space in the extent with the given name.
 This quantity is equal to the number of free pages in the extent times the
 size of a page.

 If the given file is not an active member of the logical Repository
 represented by the receiver, then this method generates an error."

^(self _dbfStatistics: extentFilename) at: 2

]

{ #category : 'Deprecated' }
Repository >> fullBackupCompressedTo: fileName [

"The same as fullBackupTo: except that the backup file is compressed
 using gzip as it is written.

 If the fileName is does not end in '.gz' then a '.gz' suffix is added.

 Deprecated.  Use the method:
    fullBackupGzCompressedTo: fileName
 instead of this one."

  self deprecated: 'Repository>>fullBackupCompressedTo: deprecated in v3.4.'.
  ^ self fullBackupGzCompressedTo: fileName

]

{ #category : 'Deprecated' }
Repository >> fullBackupCompressedTo: fileNames MBytes: mByteLimit [

"The same as fullBackupTo:MBytes: except that the backup file
 is compressed using gzip as it is written.

 If the fileName is does not end in '.gz' then a '.gz' suffix is added.

 Deprecated.  Use the method:
   fullBackupGzCompressedTo:MBytes:
 instead."

  self deprecated: 'Repository>>fullBackupCompressedTo:MBytes: deprecated in v3.4.' .
  ^ self fullBackupGzCompressedTo: fileNames MBytes: mByteLimit

]

{ #category : 'Backup and Restore' }
Repository >> fullBackupGzCompressedTo: fileNameOrArrayOfNames [

"The same as fullBackupTo: except that the backup file is compressed
 using gzip as it is written.

 If the fileName is does not end in '.gz' then a '.gz' suffix is added."

  ^ self fullBackupGzCompressedTo: fileNameOrArrayOfNames MBytes: 0

]

{ #category : 'Backup and Restore' }
Repository >> fullBackupGzCompressedTo: fileNames MBytes: mByteLimit [

"The same as fullBackupTo:MBytes: except that the backup file
 is compressed using gzip as it is written.

 If the fileName is does not end in '.gz' then a '.gz' suffix is added."

^ self fullBackupTo: fileNames MBytes: mByteLimit compressKind: 1 bufSize: 1

]

{ #category : 'Backup and Restore' }
Repository >> fullBackupGzCompressedTo: fileNames MBytes: mByteLimit compressLevel: cLevel [

"Creates a full backup compressed using zlib with the specified compression level cLevel, which
 must be a SmallInteger between 1 and 9. A value of 1 uses the fastest compression but has
 the worst compression ratio which a value of 9 uses the slowest compression but has the
 best compression ratio."

 ((cLevel < 1) or:[cLevel > 9])
   ifTrue:[self _error: #rtErrArgOutOfRange args:{ 1 . 9 } ] .
   
^ self fullBackupTo: fileNames MBytes: mByteLimit compressKind: (100 + cLevel) bufSize: 16

]

{ #category : 'Backup and Restore' }
Repository >> fullBackupGzCompressedTo: fileNames MBytes: mByteLimit threads: numThreads [
  ^ self setDefaultNumThreads: numThreads during: [
     self fullBackupGzCompressedTo: fileNames MBytes: mByteLimit 
    ].
]

{ #category : 'Backup and Restore' }
Repository >> fullBackupGzCompressedTo: fileNameOrArrayOfNames threads: numThreads [

"The same as fullBackupGzCompressedTo: except that it specifies the number of
 threads to use in the backup. The number of threads can be overridden for any other
 backup method by executing:
   SessionTemps current at: #GsOverrideNumThreads put: numThreads
 before executing the method.
"

^ self setDefaultNumThreads: numThreads during: [
    self fullBackupGzCompressedTo: fileNameOrArrayOfNames
	]

]

{ #category : 'Backup and Restore' }
Repository >> fullBackupLz4CompressedTo: fileNameOrArrayOfNames [

"The same as fullBackupTo: except that the backup file is compressed
 using LZ4 as it is written.

 If the fileName is does not end in '.lz4' then a '.lz4' suffix is added."

  ^ self fullBackupLz4CompressedTo: fileNameOrArrayOfNames MBytes: 0

]

{ #category : 'Backup and Restore' }
Repository >> fullBackupLz4CompressedTo: fileNames MBytes: mByteLimit [

"The same as fullBackupTo:MBytes: except that the backup file
 is compressed using lz4 compression as it is written.

 If the fileName is does not end in '.lz4' then an '.lz4' suffix is added."

^ self fullBackupTo: fileNames MBytes: mByteLimit compressKind: 2 bufSize: 16

]

{ #category : 'Backup and Restore' }
Repository >> fullBackupLz4CompressedTo: fileNames MBytes: mByteLimit compressLevel: cLevel [

"Creates a full backup compressed using lz4 with the specified compression level cLevel, which
 must be a SmallInteger between 0 and 12. A value of 0 uses the fastest compression but has
 the worst compression ratio which a value of 12 uses the slowest compression but has the
 best compression ratio."

 ((cLevel < 0) or:[cLevel > 12])
   ifTrue:[self _error: #rtErrArgOutOfRange args:{ 0 . 12 } ] .
   
^ self fullBackupTo: fileNames MBytes: mByteLimit compressKind: (200 + cLevel) bufSize: 16

]

{ #category : 'Backup and Restore' }
Repository >> fullBackupLz4CompressedTo: fileNames MBytes: mByteLimit threads: numThreads [
  ^ self setDefaultNumThreads: numThreads during: [
     self fullBackupLz4CompressedTo: fileNames MBytes: mByteLimit 
    ].
]

{ #category : 'Backup and Restore' }
Repository >> fullBackupLz4CompressedTo: fileNameOrArrayOfNames threads: numThreads [

"The same as fullBackupLz4CompressedTo: except that it specifies the number of
 threads to use in the backup. The number of threads can be overridden for any other
 backup method by executing:
   SessionTemps current at: #GsOverrideNumThreads put: numThreads
 before executing the method.
"

^ self setDefaultNumThreads: numThreads during: [
    self fullBackupLz4CompressedTo: fileNameOrArrayOfNames
	]

]

{ #category : 'Backup and Restore' }
Repository >> fullBackupTo: fileNameOrArrayOfNames [

"Writes a full backup to an individual file or an Array of files containing
 the most recently committed version of the receiver as of the time the
 method is executed.  If an Array of files is specified, the backup
 will partition the data equally across the files.

 This method always writes the file without compression. To compress
 the backup files use fullBackupGzCompressedTo: or fullBackupLz4CompressedTo

 The fileName argument may use GemStone Network Resource String syntax.
 For example, this may be used to access a file on another machine, provided
 a GemStone NetLDI process is running on the remote machine. fileName cannot
 specify a raw partition.

 If the device containing the file runs out of space, then the backup
 terminates with a system I/O error and the partially written backup
 file is deleted.

 If the session contains uncommitted changes to the repository, the method
 signals a error: #rtErrAbortWouldLoseData, to indicate that data could
 be lost.  Otherwise, it puts the session in autoBegin mode and performs
 the backup operation.  The session may be aborted a number of times during
 the backup to prevent a commit record backlog.

 When the backup completes, the session is set to manualBegin mode
 mode so that it does not reference a commit record  that would cause
 the repository to grow.

 Returns true if the backup was completed.

 This method requires the FileControl privilege.

 This method performs the backup using multiple threads.
 The number of threads is automatically set to 2 times the
 number of extents in the repository.  This can be overridden by
 executing:
     SessionTemps current at: #GsOverrideNumThreads put: numThreads.
 where numThreads can be any value between 1 and 4 * numCpus before
 executing the backup method.
 The performance can be modified during the run by
 updating the Multithreaded Scan Tuning methods.

 A GciHardBreak during this method will terminate the session."

  ^ self fullBackupTo: fileNameOrArrayOfNames MBytes: 0

]

{ #category : 'Backup and Restore' }
Repository >> fullBackupTo: fileNames MBytes: mByteLimit [

"The same as fullBackupTo: except that it accepts an Array of filenames
 as the first argument so that the backup may be spread across multiple
 files on different devices.  In addition, the mByteLimit argument
 specifies the maximum size of the files.  The mByteLimit may be a single
 integer which is applied to each file in the fileNamesArray or
 an Array of integer values, where the array must be the same size
 as the fileNamesArray.

 The size of the file produced for the last element of fileNames 
 may be larger than the specified limit. In order to avoid backup failures, 
 the last file does not limit the size, so backup will complete unless the 
 device used for the last backup file becomes full.

 A zero value for the megabyte limit means that there is no limit on the
 size of the resulting file(s) and is an easy way to specify that the backup
 should partition the data equally across the files.  The maximum value
 of a backup file even with no limit is 4TB. A mByteLimit value less
 than zero generates an error."

^ self fullBackupTo: fileNames MBytes: mByteLimit compressKind: 0 bufSize: 8

]

{ #category : 'Backup and Restore' }
Repository >> fullBackupTo: fileNames MBytes: mByteLimit compressKind: cKind bufSize: count [

"The same as fullBackupTo:MBytes: except that the backup file is compressed
 based on the value of cKind (see primitive for definition of compressionKinds).

 Compression level may now also be specified by using special values for cKind.
 See the method:
    _fullBackupTo:MBytes:compressKind:bufSize:numThreads: 
 for details.
"

  | limitArray fnArray |
  (fileNames isKindOfClass: Array)
    ifTrue: [ fnArray := fileNames ]
    ifFalse: [ fnArray := { fileNames } ].
  limitArray := self _validateFileNames: fnArray limits: mByteLimit.
  ^ self _fullBackupTo: fnArray MBytes: limitArray compressKind: cKind
                        bufSize: count numThreads: self _getNumThreads

]

{ #category : 'Backup and Restore' }
Repository >> fullBackupTo: fileNameOrArrayOfNames threads: numThreads [

"The same as fullBackupTo: except that it specifies the number of
 threads to use in the backup. The number of threads can be overridden for any other
 backup method by executing:
   SessionTemps current at: #GsOverrideNumThreads put: numThreads
 before executing the method.
"

^ self setDefaultNumThreads: numThreads during: [
    self fullBackupTo: fileNameOrArrayOfNames
	]

]

{ #category : 'Default numThreads' }
Repository >> getDefaultNumThreads [

"Returns the default number of threads to use for multi threaded operations"

 ^ SessionTemps current at: #GsOverrideNumThreads
     ifAbsent: [ 2 ]

]

{ #category : 'Multithreaded Scan Tuning' }
Repository >> getMultiThreadedScanMaxThreads [

"Answer the maximum number of threads that the current multithreaded scan
 is configured to use.  Answers 0 if no multithreaded scan is in progress.

 Raises an error if the session holding the GC lock is not running on this
 host or if the GC lock session is not performing a multithreaded repository
 scan.

 Requires the SessionAccess privilege."

| sessions nThr |
sessions := System sessionsHoldingGcLock .
sessions size == 0 ifTrue:[ RepositoryError signal: 'No scan in progress' ].
nThr := 0 .
sessions do:[ :sesId | | v |
  v := self mtMaxThreads: sesId .
  v > nThr ifTrue:[ nThr := v ].
  ].
^ nThr

]

{ #category : 'Searching' }
Repository >> indexOf: anObject [

"Returns the index of the first element in the receiver that is equivalent to
 the argument. If the receiver does not have any elements equivalent to the
 argument, returns, zero.

 Since ObjectSecurityPolicies are canonicalized by the GsObjectSecurityPolicy creation methods, and
 only referenced from SystemRepository, this is implemented with
 indexOfIdentical:
"

^ self indexOfIdentical: anObject

]

{ #category : 'Searching' }
Repository >> indexOf: anObject ifAbsent: anExceptionBlock [

"Returns the index of the first element in the receiver that is equivalent to
 the argument. If the receiver does not have any elements equivalent to the
 argument, returns the value of evaluating anExceptionBlock."

| ofs |
ofs := self indexOfIdentical: anObject .
ofs > 0 ifTrue:[ ^ ofs ].

^ anExceptionBlock value.

]

{ #category : 'Searching' }
Repository >> indexOfIdentical: anObject [

"Returns the index of the first element in the receiver that is identical to
 the argument. If the receiver does not have any elements identical to the
 argument, returns, zero."

<primitive: 494>
self _primitiveFailed: #indexOfIdentical: args: { anObject }

]

{ #category : 'Updating' }
Repository >> instVarAt: offset put: aValue [
"Disallowed"
self shouldNotImplement: #instVarAt:put:

]

{ #category : 'Transaction Logging' }
Repository >> lastCheckpointPosition [
  "Returns a ScaledDecimal representation of the last checkpoint position within the tranlogs"
  | info |
  info := self _logInfo:  1 .
  ^ self class _scaledDecimalFromFileId: (info at: 14) blockId:(info at: 15 ).

]

{ #category : 'Backup and Restore' }
Repository >> lastFailoverTime [

"If system is in restore from transaction logs, returns the
 last failOver timestamp detected while reading transaction logs,
 or nil if no failOver timestamp has been processed since restore started.
 If system is not in restore, returns the time from the most
 recent suspendCommitsForFailover, or nil if no suspendCommitsForFailover
 has been executed.

 Result is a DateAndTime or nil."

 | info tm suspendTime |
 ((info := self restoreStatusInfo) at: 2) == 2 ifTrue:[
   (tm := info at: 13) == 0 ifTrue:[ ^ nil ].
   ^ DateAndTime posixSeconds: tm offset: nil
 ].
 (suspendTime := info at: 14) == 0 ifTrue:[ ^ nil ].
 ^ DateAndTime posixSeconds: suspendTime offset: nil

]

{ #category : 'Transaction Logging' }
Repository >> lastTranlogPosition [
  "Returns a ScaledDecimal representation of the tranlog position"
  | info |
  info := self _logInfo:  1 .
  ^ self class _scaledDecimalFromFileId: (info at: 12) blockId:(info at: 13 ).

]

{ #category : 'Listing Instances' }
Repository >> listInstances: anArray [

" This method finds all instances of each class in the argument. The argument is 
  an Array of classes. The result is returned as an Array of Arrays.  

  <anArray> should be a collection of one or more classes. For each of these 
  classes, the result will contain, at the corresponding index, an Array holding 
  all instances whose class is equal to that class. The results include both 
  committed and in-memory instances. 

  Large result sets may cause out-of-memory issues. To avoid problems,
  use Repository >> allInstances:, which returns GsBitmaps that do not
  require keeping objects in temporary object space.

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData.

  If anArray contains multiple occurrences of a class, then the result
  will contain corresponding multiple occurrences of the same Array that lists
  the instances of that class.

  If anArray contains an element that is not a kind of Behavior, an error
  is generated. For special classes (Character, SmallInteger, SmallDouble, etc.), 
  the returned sub array will be empty.

  Scans the entire Repository at least once. The scan is done in an non-aggressive 
  manner and uses only 2 threads. To complete the operation in less time, see 
  the #fastListInstances: method. 
"

| reportArr res |
reportArr := self listInstances: anArray limit: 0 .
res := Array new: anArray size .
1 to: anArray size do:[:j |
  res at:j put: (reportArr at: j * 2 ) . "array of instances"
].
^ res

]

{ #category : 'Listing Instances' }
Repository >> listInstances: anArray limit: aSmallInt [

" This method finds instances of each class in the argument. The arguments 
  are an Array of classes, and the maximum number of instances to return,
  or 0 for unlimited. The result is returned as an Array containing counts 
  and Array of instances. 

  For each of the classes in the argument, the result Array will contain two 
  entries. These are the number of instances returned (an integer between 0 
  and aSmallInt), and a list of up to aSmallInt instances of that class. 
  The results include both committed and in-memory instances.

  Large result sets may cause out-of-memory issues. To avoid problems,
  use Repository >> allInstances:, which returns a GsBitmap that does not
  require keeping objects in temporary object space.

  This method aborts the current transaction; if an abort would cause unsaved
  changes to be lost it signals an error, #rtErrAbortWouldLoseData.  The entire
  Repository is scanned at least once.

  Scans the entire Repository at least once. The scan is done in an non-aggressive 
  manner and uses only 2 threads. To complete the operation in less time, see 
  the #fastListInstances:limit: method.
"

^ self listInstances: anArray limit: aSmallInt toDirectory: nil
      withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90 memory: 1

]

{ #category : 'Listing Instances' }
Repository >> listInstances: anArray limit: aSmallInt toDirectory: directoryString withMaxThreads: maxThreads maxCpuUsage: aPercentage [

 ^ self listInstances: anArray limit: aSmallInt toDirectory: directoryString
   withMaxThreads: maxThreads maxCpuUsage: aPercentage memory: 1

]

{ #category : 'Listing Instances' }
Repository >> listInstances: anArray limit: aSmallInt toDirectory: directoryString withMaxThreads: maxThreads maxCpuUsage: aPercentage memory: memOptionInt [

"If directoryString == nil, result includes in-memory objects.
  and maxThreads and aPercentage  are ignored.
 memOptionInt  0 disk only 
               1 disk plus all of memory, 
               2 memory only 
               3 memory only, exclude pom_gen . "

| inputSet resultInSetOrder result scanBlk |

(memOptionInt <= 1 or:[ directoryString ~~ nil]) ifTrue:[
  System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ]
].
inputSet := self _arrayOfClassesAsSet: anArray .
inputSet size < 1 ifTrue:[ ^ { } ] .

memOptionInt <= 1 ifTrue:[
 scanBlk := [ :scanSetThisTime | | scanKind |
     scanKind := directoryString ifNotNil:[ OP_LIST_INST_TO_FILES  ] 
                                    ifNil:[ OP_LIST_INSTANCES  ] .
     self _scanPomWithMaxThreads: maxThreads waitForLock: 60 pageBufSize: 8
                             percentCpuActiveLimit: aPercentage
                             identSet: scanSetThisTime limit: aSmallInt
                             scanKind: scanKind 
                             toDirectory: directoryString
  ].
].
inputSet := IdentitySet withAll: anArray .

resultInSetOrder:= self _doListInstancesFrom: inputSet with: scanBlk
			memory:  memOptionInt limit: aSmallInt.

directoryString ifNotNil:[
  result := resultInSetOrder "primitive wrote the files, so we are done"
] ifNil:[ | inputArraySize |
  inputArraySize := anArray size .
  result := Array new: inputArraySize * 2.
  1 to: inputArraySize do:[:j| | soIdx resIdx anObj |
      anObj := anArray at: j .
      soIdx := (inputSet _offsetOf: anObj) * 2 .
      resIdx := j * 2 .
      result at: resIdx - 1 put: ( resultInSetOrder at: soIdx - 1 ) ."totalCount"
      result at: resIdx put: ( resultInSetOrder at: soIdx) . "array of instances"
  ].
].
^ result

]

{ #category : 'Listing Instances' }
Repository >> listInstances: anArray limit: aSmallInt toDirectory: directoryString withMaxThreads: maxThreads maxCpuUsage: aPercentage memoryOnly: memOnlyBool [

  ^ self listInstances: anArray limit: aSmallInt toDirectory: directoryString
     withMaxThreads: maxThreads maxCpuUsage: aPercentage 
     memory: (memOnlyBool ifTrue:[ 2 ] ifFalse:[ 1 ])

]

{ #category : 'Listing Instances' }
Repository >> listInstances: anArray toDirectory: aString [

"Scan the repository for the instances of classes in anArray and write
 the results to binary bitmap files in the directory specified by
 aString.  Binary bitmap files have an extension of .bm and may be
 loaded into hidden sets using class methods in class System or
 GsBitmap readFromFile: method.

 aString must specify a path to a writable directory.

 Does not include objects in VM's temporary object memory.

 Bitmap files are named as follows:

 <ClassName>-<classOop>-instances.bm

 where className is the name of the class and classOop is the object
 ID of the class.

 The result is an Array of pairs.  For each element of the argument anArray,
 the result array contains  <aClass>, <numberOfInstancesFound>.
 The numberOfInstances is the total number written to the output bitmap file

 Note: the order of classes in the result array may differ from that of
 the argument anArray.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.  The entire
 Repository is scanned at least once.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To
 complete the operation in less time, see the #fastListInstances:toDirectory:
 method."

^ self _allInstances: anArray toDirectory: aString fast: false

]

{ #category : 'Listing Instances' }
Repository >> listInstancesInMemory: anArray [

"Returns a list of instances of the classes specified in anArray
 that are found in temporary object memory.

 The result of this method is an Array of Arrays, where the
 contents of each inner array consists of all instances
 of the corresponding element in anArray.

 The result contains in-memory objects only.  Objects which are not
 currently in memory are not analyzed.

 Does not abort the current transaction."

| pairs res |
pairs := self listInstances: anArray limit: 0 toDirectory: nil
      withMaxThreads: 1 maxCpuUsage: 95 memory: 2 .
res := Array new: anArray size .
1 to: anArray size do:[:j |
  res at:j put: (pairs at: j * 2 ) . "array of instances"
].
^ res

]

{ #category : 'Listing Instances' }
Repository >> listInstancesInMemoryNoPomgen: anArray [

"Returns a list of instances of the classes specified in anArray
 that are found in temporary object memory.

 The result of this method is an Array of Arrays, where the
 contents of each inner array consists of all instances
 of the corresponding element in anArray.

 The result contains in-memory objects only.  Objects which are not
 currently in memory are not analyzed.

 Does not abort the current transaction."

| pairs res |
pairs := self listInstances: anArray limit: 0 toDirectory: nil
      withMaxThreads: 1 maxCpuUsage: 95 memory: 3 .
res := Array new: anArray size .
1 to: anArray size do:[:j |
  res at:j put: (pairs at: j * 2 ) . "array of instances"
].
^ res

]

{ #category : 'Listing Instances' }
Repository >> listInstancesInPageOrder: anArrayOfClasses toFile: aString [
"See the method #listInstancesInPageOrder:toFile:withMaxThreads:maxCpuUsage:
 for more information on this method.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To
 complete the operation in less time, see the #fastListInstancesInPageOrder:toFile:
 method."

^ self listInstancesInPageOrder: anArrayOfClasses toFile: aString
       withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90
]

{ #category : 'Listing Instances' }
Repository >> listInstancesInPageOrder: anArrayOfClasses toFile: aString withMaxThreads: maxThreads maxCpuUsage: aPercentage [

"Scans the repository for all committed instances of the classes
 contained in anArray and stores the results into a new file
 specified in aString.
 Does not include objects in VM's temporary object memory.

 anArrayOfClasses must be an instance of Array which contains at
 least 1 and not more than 2034 classes.

 aString must be the full path to a new file on the gem's host system
 which will be created by this method.  It is an error if the file
 already exists.

 All results of the scan (all instances of the classes in anArray) are
 stored in the file in page order (i.e., the same order as the  objects
 are stored on disk) as of the time of the scan.   Note that unlike other
 #listInstances methods,  the file contains the complete list of all
 instances of all classes contained in anArray.

 The ultimate file size will be approximately 5 bytes for each object
 found by the scan, plus 24 bytes.  For example, a scan which locates 10
 million objects will generate a file approximately 50 megabytes in size.

 Objects created and committed after starting this method are not included
 in the result file.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.

 This method runs in a transaction and is therefore not recommended for
 production systems.

 Operations which process a large volume of objects (e.g.: instance migration,
 etc) often run significantly faster if the objects are processed in page order
 rather than in object ID order.  Other #listInstances: methods always return
 results in object ID order.

 The objects written to the file are left in hidden set 1 when this method
 completes successfully.

 Objects are routinely moved from old pages to new pages whenever modified.
 The Reclaim GC gems also move objects to new pages.  Therefore the actual
 order the objects appear on disk will diverge over time from the order of
 the objects in the result file.

 Equivalent to executing the following methods:

   SystemRepository listInstancesToHiddenSet: anArrayOfClasses
                    withMaxThreads: maxThreads maxCpuUsage: aPercentage .
   System writeHiddenSet: 1 toPageOrderFile: aString maxBufferSizeMb: 0 .

 except that this method creates the result file before starting the scan.

 Returns a SmallInteger representing the total number of objects written to the
 file."

System needsCommit
  ifTrue: [ ^ self _error: #rtErrAbortWouldLoseData ] .

^ self _scanPomWithMaxThreads: maxThreads waitForLock: 90 pageBufSize: 8
                  percentCpuActiveLimit: aPercentage
                  identSet: (IdentitySet withAll: anArrayOfClasses)
                  limit: 0 scanKind: OP_LIST_INST_PAGE_ORDER
                  toDirectory: aString

]

{ #category : 'Listing Instances' }
Repository >> listInstancesToHiddenSet: aClassOrArray [

"Scans the entire Repository and generates a list of instances into hidden set 1.
 The previous state of the hidden set 1 is cleared.

 aClassOrArray is expected to be either an Array of classes or a Class.
 Does not include objects in VM's temporary object memory.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData. The entire
 Repository is scanned once.

 If aClassOrArray is a Class, this method returns the number of instances of
 aClassOrArray which were added to hidden set 1.

 If aClassOrArray is an Array of classes, this method returns an Array of pairs with
 a Class at the odd numbered elements and a count at the even numbered elements.
 The count is the number of instances of the class that were added to hidden set 1.
 Note that the order in which classes appear in the result array may differ
 from the order which they appear in aClassOrArray.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To
 complete the operation in less time, see the #fastListInstancesToHiddenSet:
 method."

| arr |
arr := aClassOrArray.
arr isBehavior ifTrue: [ arr := { arr } ].

^ self _buildListInstResult: (self allInstances: arr)

]

{ #category : 'Listing Instances' }
Repository >> listInstancesToHiddenSet: aClassOrArray withMaxThreads: maxThreads maxCpuUsage: aPercentage [

"Scans the entire Repository and generates a list of instances into hidden set 1.
 The previous state of the hidden set 1 is cleared.

 aClassOrArray is expected to be either an Array of classes or a Class.
 Does not include objects in VM's temporary object memory.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData. The entire
 repository is scanned once.

 If aClassOrArray is a Class, this method returns the number of instances of
 aClassOrArray which were added to hidden set 1.

 If aClassOrArray is an Array of classes, this method returns an Array of pairs with
 a Class at the odd numbered elements and a count at the even numbered elements.
 The count is the number of instances of the class that were added to hidden set 1.
 All instances of all classes present in aClassOrArray are combined and added to
 hidden set 1.

 Note that the order in which classes appear in the result array may differ
 from the order which they appear in aClassOrArray.

 This method is kept for compatibility, ignores withMaxThreads: and maxCpuUsage: args
 and executes fastListInstancesToHiddenSet: .
"

^ self fastListInstancesToHiddenSet: aClassOrArray.

]

{ #category : 'Listing By Security Policy' }
Repository >> listObjectsInObjectSecurityPolicies: anArray [

"Returns a list of objects in the receiver that have the specified
 objectSecurityPolicyIds.  The result of this method is an Array
 of Arrays, where the contents of each inner array consists of
 objects whose objectSecurityPolicyId is the objectSecurityPolicyId of
 of the corresponding element in anArray.

 anArray must be an Array of objectSecurityPolicyIds.  The maximum size
 of anArray is no longer restricted to 2034 elements.

 The value zero can be used to indicate you wish to get the objects that
 have no objectSecurityPolicyId.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.

 The result contains both permanent and temporary objects.  The temporary
 objects found may vary from run to run.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To
 complete the operation in less time, see the #fastListInstances:toDirectory:
 method."

^ self listObjectsInObjectSecurityPolicies: anArray withLimit: 0

]

{ #category : 'Listing By Security Policy' }
Repository >> listObjectsInObjectSecurityPolicies: anArray toDirectory: aString [

"Scan the repository for objects that have the specified objectSecurityPolicyIds
 and writes the results to binary bitmap files in the directory specified by
 aString.  Binary bitmap files have an extension of .bm and may be
 loaded into hidden sets using class methods in class System.
 aString must specify a path to a writable directory.

 anArray must be an Array of objectSecurityPolicyIds.  The maximum size
 of anArray is no longer restricted to 2034 elements.

 The value zero can be used to indicate you wish to get the objects that
 have no objectSecurityPolicyId.

 Bitmap files are named as follows:
   segment<objectSecurityPolicyId>-objects.bm
 where objectSecurityPolicyId is an element of anArray .

 The result is an Array of pairs.  For each element of the argument anArray,
 the result array contains  <objectSecurityPolicyId>, <numberOfInstancesFound>.
 The numberOfInstances is the total number written to the output bitmap file.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData."

aString _validateKindOfClass: String .
^ self _listObjectsInObjectSecurityPolicies: anArray limit: 0 
       scanKind: OP_LIST_OBJS_IN_SEC_POLICY_TO_FILES
       toDirectory: aString withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90

]

{ #category : 'Listing By Security Policy' }
Repository >> listObjectsInObjectSecurityPolicies: anArray withLimit:  aSmallInt [

"Returns a list of objects in the receiver that have the specified
 objectSecurityPolicyIds.  The result of this method is an Array
 of Arrays, where the contents of each inner array consists of
 objects whose objectSecurityPolicyId is the objectSecurityPolicyId of
 of the corresponding element in anArray.  The number in each inner
 array is limited to approximately aSmallInt.

 anArray must be an Array of objectSecurityPolicyIds.  The maximum size
 of anArray is no longer restricted to 2034 elements.

 The value zero can be used to indicate you wish to get the objects that
 have no objectSecurityPolicyId.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.

 The result contains both permanent and temporary objects.  The temporary
 objects found may vary from run to run.

 This method is very expensive.  It scans the entire Repository and looks
 at the object header of every object."

^ self _listObjectsInObjectSecurityPolicies: anArray limit: aSmallInt
      scanKind: OP_LIST_OBJS_IN_SEC_POLICY
      toDirectory: nil withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90

]

{ #category : 'Listing By Security Policy' }
Repository >> listObjectsInObjectSecurityPolicyToHiddenSet: anObjectSecurityPolicyId [

"Scans the entire Repository and generates a list of objects that have specified
 objectSecurityPolicyId into hidden set 1 .  The previous state of hidden set 1
 is cleared.

 The value zero can be used to indicate you wish to get the objects that
 have no objectSecurityPolicyId.

 Returns the number of instances found.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData. The entire
 repository is scanned once.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To
 complete the operation in less time, see the
 #fastListObjectsInObjectSecurityPolicyToHiddenSet: method."

self _validateObjectSecurityPolicyId: anObjectSecurityPolicyId .
^ self _listObjectsInObjectSecurityPolicies: { anObjectSecurityPolicyId }
       limit: 0 scanKind: OP_LIST_OBJS_IN_SEC_POLICY_TO_HIDDEN
       toDirectory: nil withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90

]

{ #category : 'Deprecated' }
Repository >> listObjectsInSegments: anArray [
"Deprecated"

self deprecated: 'Repository>>listObjectsInSegments: deprecated v3.0.
Use #listObjectsInObjectSecurityPolicies: instead.'.
^ self _buildListInstResult: (self allObjectsInObjectSecurityPolicies: anArray)

]

{ #category : 'Deprecated' }
Repository >> listObjectsInSegments: anArray toDirectory: aString [
"Deprecated"

self deprecated: 'Repository>>listObjectsInSegment:toDirectory: deprecated v3.0.
Use #listObjectsInObjectSecurityPolicies:toDirectory: instead.'.

^ self listObjectsInObjectSecurityPolicies: anArray toDirectory: aString.

]

{ #category : 'Deprecated' }
Repository >> listObjectsInSegments: anArray withLimit:  aSmallInt [
"Deprecrated"

  self deprecated: 'Repository>>listObjectsInSegments:withLimit: deprecated v3.0.
Use #listObjectsInObjectSecurityPolicies:withLimit: instead.'.
^ self listObjectsInObjectSecurityPolicies: anArray withLimit:  aSmallInt

]

{ #category : 'Deprecated' }
Repository >> listObjectsInSegmentToHiddenSet: anObjectSecurityPolicyId [
"Deprecated"

self deprecated: 'Repository>>listObjectsInSegmentToHiddenSet: deprecated v3.0.
Use #listObjectsInObjectSecurityPolicyToHiddenSet: instead.'.

^ self listObjectsInObjectSecurityPolicyToHiddenSet: anObjectSecurityPolicyId

]

{ #category : 'Listing Instances' }
Repository >> listObjectsWithOneReferenceWithMaxThreads: maxThreads waitForLock: lockWaitTime percentCpuActiveLimit: percentCpu [

"Scans the entire repository for objects referenced only once and only once
 by any other object. Multiple references from a parent object to a child 
 object are considered a single references and the the child object will
 be included in the result. Objects which reference themselves will be
 included in the result if that reference is the only one.

 Returns an instance of GsBitmap.

 Note that objects in the result set may be disconnected from the repository 
 (unreachable) and therefore could disappear after the next garbage collection
 cycle.

 Raises an error if the session has modified persistent objects.

 The lockWaitTime argument is used to specify how many seconds method should
 wait while attempting to acquire the gcLock.  No other garbage collection
 operations may be started or in progress while this method is running.
 There also must be no outstanding possible dead objects in the system for
 the GC lock to be granted.

 Starts maxThreads on the host system and allows the host to run up to percentCpu
 percent CPU usage.  A page buffer of 16 pages (256 KB) is allocated per thread.

 This method begins a transaction and runs in this transaction for its duration. 
 It should therefore not be used in production systems due to commit record
 backlogs which may cause excessive repository growth.

 Raises an error if and any garbage collection operation is in progress
 with the repository vote state is not zero (see System(C) voteState).

 Uncommitted objects and dead not reclaimed objects are excluded from 
 the result."

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

System beginTransaction.
self _listObjectsWithOneReferenceWithMaxThreads: maxThreads waitForLock: lockWaitTime percentCpuActiveLimit: percentCpu .

^ GsBitmap newForHiddenSet: #ListInstances
]

{ #category : 'Listing References' }
Repository >> listReferences: anArray [

"Returns a list of instances in the Repository that have a reference to one of
 the objects specified in anArray.  The result of this method is an Array of
 Arrays, where the contents of each inner array contains a target object at offset 1 and
 a GsBitmap that contains the references to that object at offset 2.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.

 The result contains only committed objects.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To
 complete the operation in less time, see the #fastListReferences
 method."

^ self _buildAllRefsResult: (self allReferences: anArray) forInput: anArray

]

{ #category : 'Listing References' }
Repository >> listReferences: anArray withLimit: aSmallInt [

"Returns a list of instances in the Repository that have a reference to one of
 the objects specified in anArray.  The result of this method is an Array of
 Arrays, where the contents of each inner array contains a target object at offset 1 and
 a GsBitmap that contains the references to that object at offset 2.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.

 The result contains only committed objects.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To
 complete the operation in less time, see the #fastListReferences
 method."

^ self _buildAllRefsResult: (self allReferences: anArray) forInput: anArray withLimit: aSmallInt

]

{ #category : 'Listing References' }
Repository >> listReferences: anArray withLimit: aSmallInt withMaxThreads: maxThreads maxCpuUsage: aPercentage [

"Returns a list of instances in the Repository that have a reference to one of
 the objects specified in anArray.  The result of this method is an Array of
 Arrays, where the contents of each inner array contains a target object at offset 1 and
 a GsBitmap that contains the references to that object at offset 2.
 The number in each inner array is limited to approximately aSmallInt.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.

 The result contains both permanent and temporary objects.  The temporary
 objects found may vary from run to run.

 anArray is expected to contain no duplicates.  An ArgumentError is raised if
 duplicate entries are detected.

 This method is very expensive.  It scans the entire Repository and looks
 at every instance variable of every object."

| outerArr sortedArray resultArray sortedCollection |

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

"Sort by oop order."
sortedCollection := SortedCollection sortBlock:[:a :b| a asOop < b asOop].
sortedCollection addAll: anArray.
sortedArray := Array withAll: sortedCollection .

outerArr := self _listReferencesWithMaxThreads: maxThreads waitForLock: 60
      pageBufSize: 8 percentCpuActiveLimit: aPercentage
      targets: sortedArray withLimit: aSmallInt.

"nil means the primitive detected a duplicate entry.  Raise an error and bail."
outerArr ifNil:[  ArgumentError new object: anArray ;
    signal:'Array of references to search for contains 1 or more duplicates'
].

"Now truncate the inner result arrays as needed and build the outer result array
 to match the order of the argument anArray."
resultArray := Array new: outerArr size.
1 to: outerArr size  do:[ :j| | innerArr index |
    innerArr := outerArr at: j .
    (aSmallInt > 0 and:[ innerArr size > aSmallInt ])
      ifTrue:[ innerArr size: aSmallInt ].
    index := anArray indexOfIdentical: (sortedArray at: j).
    resultArray at: index put: innerArr.
].
^ resultArray

]

{ #category : 'Listing References' }
Repository >> listReferencesFromBmFiles: anArray numThreads: numThreads toDirectory: aDirectoryString [
 "Reads bitmap files of instances whose paths are listed in anArray.
  Writes files, one per thread to directory specified by aDirectoryString.
  Files are   references-N.txt  where N is thread number,
  containing one line per object containing a reference, with format
     <objId> <1-based iv offset>:<objId of value> <ivOffset>:<valueId> ...
  Excludes references to OOP_EMPTY_INVARIANT_ARRAY, OOP_EMPTY_INVARIANT_STRING, OOP_EMPTY_SYMBOL
    which have oops 233217, 233473, 233729 .
  Excludes references from instances of VariableContext, GsNMethod, GsProcess .
  Returns a report of classes and number of instances of each class containing references
  to objects specified by anArray ."

| bms countsArray dict sc rpt total globals |
bms := { } .
anArray do:[:aPath | | aBm |
  bms add:(aBm := GsBitmap new). aBm readFromFile: aPath .
  GsFile gciLogServer:'read ' , aPath ,' size ', aBm size asString .
].
(GsFile isServerDirectory: aDirectoryString) ifFalse:[
  Error signal:'not a directory: ', aDirectoryString asString .
].
countsArray := self _listReferencesFromBms: bms numThreads: numThreads toDirectory: aDirectoryString .
dict := IdentityDictionary new .
1 to: countsArray size by: 2 do:[:n | | cls count |
  cls := countsArray at: n . count := countsArray at: n + 1 .
  (dict associationAt: cls otherwise: nil)
    ifNotNil:[ :assoc | assoc _value: (assoc _value + count) ]
    ifNil:[ dict at: cls put: count ].
].
sc := SortedCollection sortBlock: [:a :b | a _value > b _value ]. "descending sort on counts"
dict associationsDo:[:assoc | sc add: assoc ].
rpt := String new .  "sorted by descending number of instances
                      and segregated into non-Globals and Globals classes."
globals := String new .
total := 0 .
sc do:[:assoc | | cls cnt str |
  cls := assoc key . cnt := assoc _value .
  str := (Globals includesKey: cls name) ifTrue:[ globals ] ifFalse:[ rpt ].
  str add: cls name, ' oop ', cls asOop asString, ' count ', cnt asString ; lf .
  total := total + cnt.
].
rpt lf ; addAll:'------- Globals classes'; lf ; addAll: globals .
rpt lf ; add:'Total ', total asString,' objects containing references' ; lf .
^ rpt

]

{ #category : 'Listing References' }
Repository >> listReferencesInMemory: anArray [

"Returns a list of instances in temporary object memory that
 have a reference to one of the objects specified in anArray.
 The result of this method is an Array of Arrays, where the
 contents of each inner array consists of all instances
 that have a reference to the corresponding element in anArray.

 The result contains in-memory objects only.  Objects which are not
 currently in memory are not analyzed.

 An error is raised if anArray contains the same object more than once.

 Does not abort the current transaction."

| resultArray |

resultArray := self _listReferencesInMemory: anArray nonStubbedOnly: false .

"nil means the primitive detected at least one duplicate entry.  Raise an error and bail."
resultArray ifNil:[  ArgumentError new object: anArray ;
    signal:'Array of references to search for contains 1 or more duplicates'
].
^ resultArray

]

{ #category : 'Listing References' }
Repository >> listReferencesToInstancesOfClasses: anArray toDirectory: aString [

"First scan the repository for the instances of classes in anArray,
 then rescan to produce the sets of references to these instances.

 If anArray contains multiple occurrences of a class or an object that
 is not a kind of Behavior, then an error is generated.

 Returns an Array of triplets which contains meta information about the 
 results of the scan.  For each element of the argument anArray, 
 the result Array contains:
    <aClass>, <numberOfInstancesFound>  <numberOfReferencesFound>

 The special classes 
   Boolean Character SmallDate SmallDateAndTime SmallDouble
   SmallFraction SmallInteger SmallScaledDecimal SmallTime UndefinedObject
 may be used as elements of the input anArray.
 The second value in the triplet for these classes is always zero.

 The instances which contain references to the classes are written to
 bitmap files in the directory specified by aString, which must specify
 a path to a writable directory.

 For each class in the anArray, a bitmap file is created with the
 filename <className>-<seqNum>-references.bm and this bitmap contains
 the objects that have references to instances of this class.
 The seqNum is incremented for each file written by this method so that multiple runs
 can be made without deleting the files or getting errors for trying to overwrite a file.

 The binary bitmap files may be loaded into GsBitmap (see GsBitmap>>readFromFile:) to analyze the results.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.

 The scan is done in an non-aggressive manner and uses only 2 threads.  To
 complete the operation in less time, see the #fastListReferencesToInstancesOfClasses
 method."

^ self _listReferencesToInstancesOfClasses: anArray toDirectory: aString
       withMaxThreads: self getDefaultNumThreads maxCpuUsage: 90

]

{ #category : 'Storing and Loading' }
Repository >> loadFrom: passiveObj [
"Disallowed"
self shouldNotImplement: #loadFrom:

]

{ #category : 'Repository Analysis' }
Repository >> loadGcCandidatesFromFile: aString intoHiddenSet: hiddenSetId [

"Given a file which is binary list of dead objects (produced by one of the
 #findDisconnectedObjects: methods), load the objects into the hidden set
 with the given ID.  NOTE: in 3.4 the hiddenSetId changed from a SmallInteger to
 one of the symbols in GsBitmap>>hiddenSetSpecifiers.

 Only objects which exist and are not in the stone's dead object
 set are added to the hidden set.

 Used by GsObjectInventory to profile garbage in the repository.

 Returns an Array containing 3 SmallIntegers:
  1 - Total number of objects in the file.
  2 - Number of valid objects loaded into the hidden set.
  3 - Number of invalid objects not loaded.

 Requires the GarbageCollection privilege.

 Signals an error if hiddenSetId specifies a hidden set only writable
 by SystemUser and the session is not logged in as SystemUser.
"

| gsBitmap numInvalid result bmSize |

gsBitmap := GsBitmap newForHiddenSet: hiddenSetId.
numInvalid := gsBitmap readFromFile: aString.
bmSize := gsBitmap size.
result := Array with: (bmSize + numInvalid)
                with: bmSize
                with: numInvalid.
^result

]

{ #category : 'Storing and Loading' }
Repository >> loadNamedIVsFrom: passiveObj [
"Disallowed"
self shouldNotImplement: #loadNamedIVsFrom:

]

{ #category : 'Storing and Loading' }
Repository >> loadVaryingFrom: passiveObj [
"Disallowed"
self shouldNotImplement: #loadVaryingFrom:

]

{ #category : 'Storing and Loading' }
Repository >> loadVaryingFrom: passiveObj size: aSize [
"Disallowed"
self shouldNotImplement: #loadVaryingFrom:size:

]

{ #category : 'Transaction Logging' }
Repository >> logOriginTime [

"Returns the log origin time of the receiver. This is the time when a Stone
 started a new sequence of log files for the receiver. A new sequence of
 logs is started if one of the following occurs:

 * Stone is started using extents that were cleanly shutdown, and without
   any log files being present.

 * Stone is started using extents and no existing logs
   using the 'startstone -N' command.

 * The commitRestore method is executed, and during preceding
   restore operations we restored a log file with fileId greater
   than the fileId of the log file being written to during the restore."

^ System stoneConfigurationAt:#StnTranLogOriginTime

]

{ #category : 'Garbage Collection' }
Repository >> markForCollection [
"Performs a garbage collection analysis of all permanent objects on disk.
 Every object in the receiver that cannot be reached from AllUsers is marked
 for subsequent reclaiming of storage space.

 This method aborts the current transaction, empties the GcCandidates queue
 and commits, runs the analysis while outside of a transaction and then reenters
 a transaction if the session was in a transaction at the start of this method.
 If an abort would cause unsaved changes to be lost, it does not execute and
 signals an error, #rtErrAbortWouldLoseData.

 When this method completes successfully, signals a Warning and then
 returns the Warning.  The Warning has error text of the form
   Successful completion of Garbage Collection
     found <anInt> live objects,
     found <anInt> dead objects, occupying <anInt> bytes
 This specific Warning will satisfy a topaz
     expectvalue true
 expected result.

 After this method completes, the GcGem process automatically reclaims the space
 occupied by the dead objects.  Space is not reclaimed until other sessions have
 either committed or aborted the transactions that were concurrent with this
 method.

 This method may fail with an error if another garbage collection is
 in progress.  This method waits up to 2 minutes for the other garbage
 collection to finish before issuing the error.
 Another garbage collection can be in progress because:
   1) an epoch garbage collection is in progress in a gc gem.
   2) a markForCollection is in progress in another session
   3) a previous epoch, or markForCollection completed the
     mark phase, but the voting on the possibly dead objects has
     not completed. Note that an admin GcGem must be running for the
     voting to complete.  Also, long running sessions that are idle and
     never aborting will prevent voting from completing.

 This method requires the GarbageCollection privilege.
 A GciHardBreak during this method terminates the session.

 This method performs the MFC in a non-aggressive manner and uses only 2
 threads.  To complete the operation in less time, see the
 #fastMarkForCollection method.

 See also additional comments in the method:
 #_mfcWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:"

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ self markForCollectionWithMaxThreads: self getDefaultNumThreads waitForLock: 120

]

{ #category : 'Garbage Collection' }
Repository >> markForCollectionWait: waitTimeSeconds [

"Perform a markForCollection, waiting up to waitTimeSeconds for any
 garbage collection in progress in another session to complete.

 To wait as long as necessary for the other gc to finish, pass the argument -1.
 This should be done with caution, however, as under certain conditions
 the session could appear to wait forever.  To avoid this you need to:

 1.  Make sure that other sessions are committing/aborting to allow
     voting on possible dead to complete.

 2.  Make sure that the Admin Gc gem and at least one Reclaim Gc gem are
     is running to complete processing of dead objects once the vote is
     completed.

 This method performs the MFC in a non-aggressive manner and uses only 2
 threads.  To complete the operation in less time, see the
 #fastMarkForCollection method.

 For further documentation see markForCollection. "

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ self markForCollectionWithMaxThreads: self getDefaultNumThreads waitForLock: waitTimeSeconds

]

{ #category : 'Garbage Collection' }
Repository >> markForCollectionWithMaxThreads: numThreads [

"Similar to markForCollection: except that it allows specifying the
 number of threads to use for the scan.
  The number of threads can be overridden for any other markForCollection method by executing:
      setDefaultNumThreads: numThreads
  before executing the method.
"

^self setDefaultNumThreads: numThreads during: [ self markForCollection]

]

{ #category : 'Garbage Collection' }
Repository >> markForCollectionWithMaxThreads: maxThreads waitForLock: waitTimeSeconds [
"See the markForCollection method for comments."

^ self markForCollectionWithMaxThreads: maxThreads
                           waitForLock: waitTimeSeconds
                           pageBufSize: 128
                 percentCpuActiveLimit: 90

]

{ #category : 'Garbage Collection' }
Repository >> markForCollectionWithMaxThreads: maxThreads waitForLock: waitTimeSeconds pageBufSize: pageBufSize percentCpuActiveLimit: aPercent [

"See the markForCollection method for comments."

| arr |
arr := self _mfcWithMaxThreads: maxThreads waitForLock: waitTimeSeconds
    pageBufSize: pageBufSize percentCpuActiveLimit: aPercent.
^ self _signalMfcWarning: arr .

]

{ #category : 'Garbage Collection' }
Repository >> markForCollectionWithMaxThreads: maxThreads waitForLock: waitTimeSeconds percentCpuActiveLimit: aPercent [

"See the markForCollection method for comments."

| arr |
arr := self _mfcWithMaxThreads: maxThreads waitForLock: waitTimeSeconds
    pageBufSize: 128 percentCpuActiveLimit: aPercent.
^ self _signalMfcWarning: arr .

]

{ #category : 'Instance migration' }
Repository >> migrate [
"Disallowed"
self shouldNotImplement: #migrate

]

{ #category : 'Instance migration' }
Repository >> migrateFrom: anObj [
"Disallowed"
self shouldNotImplement: #migrateFrom:

]

{ #category : 'Instance migration' }
Repository >> migrateFrom: anObj instVarMap: aDict [
"Disallowed"
self shouldNotImplement: #migrateFrom:instVarMap:

]

{ #category : 'Instance migration' }
Repository >> migrateIndexable: anObj myClass: aCls otherClass: secondCls [
"Disallowed"
self shouldNotImplement: #migrateIndexable:myClass:otherClass:

]

{ #category : 'Migrating Objects' }
Repository >> migrateMt: arrayOfMappings [

" Performs a multi threaded scan of the repository migrating instances of the classes found 
  in the arrayOfMappings to their corresponding new classes.  Up to 2000 classes can
  be migrated in a single operation.  The classes must be committed. An error is generated if there
  are any modified persistent objects in the temporary object memory at the time it is executed.
  The method will commit all of the migrated objects before it returns.  It is possible that
  the commit may fail due to concurrency conflicts, so to avoid that the operation should be
  performed while there is no other activity on the system.

  The arrayOfMappings is Array of instances of InstVarMappingArray which defines 
  the classes that are to be migrated.  For example
    SystemRepository migrateMt: { (InstVarMappingArray mappingFrom: oldClass to: newClass) }

  Before executing this method you may wish to execute testMigrageMt:with: to validate that 
  the migration does what is intended.

  This method requires the MigrateObjects privilege.
  "
  
  ^ self _basicMigrateWithMaxThreads: self getDefaultNumThreads waitForLock: 3 pageBufSize: 8
              percentCpuActiveLimit: 90 mappingArrays: arrayOfMappings testArray: nil

]

{ #category : 'Multithreaded Scan Tuning' }
Repository >> mtMaxThreads: sessionId [

"Returns the current value of the mtMaxThreads for the specified sessionId.
 The process executing this method must be on the same machine as the specified session.

 Requires the SessionAccess privilege."

^ self _mtStat: sessionId at: 0

]

{ #category : 'Multithreaded Scan Tuning' }
Repository >> mtPercentCpuActiveLimit: sessionId [

"Returns the current value of the mtPercentCpuActiveLimit for the specified sessionId.
 The process executing this method must be on the same machine as the specified session.

 Requires the SessionAccess privilege."

^ self _mtStat: sessionId at: 2

]

{ #category : 'Multithreaded Scan Tuning' }
Repository >> mtPercentCpuActiveLimit: sessionId setValue: newVal [

"Sets a new value for the mtThreadsLimit for the specified sessionId.
 The process executing this method must be on the same machine as the specified session.
 The newVal may be in the range from 0 to 100.

 Requires the SessionAccess and SystemControl privileges."

^ self _mtStat: sessionId at: 2 put: newVal

]

{ #category : 'Multithreaded Scan Tuning' }
Repository >> mtThreadsLimit: sessionId [

"Returns the current value of the mtThreadsLimit for the specified sessionId.
 The process executing this method must be on the same machine as the specified session.

 Requires the SessionAccess privilege."

^ self _mtStat: sessionId at: 1

]

{ #category : 'Multithreaded Scan Tuning' }
Repository >> mtThreadsLimit: sessionId setValue: newVal [

"Sets a new value for the mtThreadsLimit for the specified sessionId.
 The process executing this method must be on the same machine as the specified session.
 The newVal may be 0, which can be used to temporarily pause the operation in progress.
 The maximum newVal allowed is determined by the current value of mtMaxSessions.

 Requires the SessionAccess and SystemControl privileges."

^ self _mtStat: sessionId at: 1 put: newVal

]

{ #category : 'Accessing' }
Repository >> name [

"Returns the logical name of the receiver (a Symbol)."

^ name

]

{ #category : 'Updating' }
Repository >> nilFields [
"Disallowed"
self shouldNotImplement: #nilFields

]

{ #category : 'Extent Operations' }
Repository >> numberOfExtents [

  <primitive: 428>

"Returns the number of active extents."

  self _primitiveFailed: #numberOfExtents .
  self _uncontinuableError

]

{ #category : 'Deprecated Page-Order Operations' }
Repository >> numberOfObjectsInPageOrderOopFileWithId: aSmallInt [

"Deprecated, use (GsBitmap fileInfo: <bitmapFilePath>) at: 1

 Answer the number of objects in a page order oop file previously opened
 with the Repository>>openPageOrderOopFile: method.
 aSmallInt must be the result of the Repository>>openPageOrderOopFile: method.
 Does not close or otherwise change the state of the file."

  self deprecated: 'Repository>>numberOfObjectsInPageOrderOopFileWithId: deprecated v3.4. ',
     'Use (GsBitmap fileInfo: <bitmapFilePath>) at: 1'.

^ self _primReadObjectsFromFileWithId: aSmallInt startingAt: nil upTo: nil into: nil

]

{ #category : 'Repository Usage Reporting' }
Repository >> numToMByteString: aNumber [

^ self _numToString: aNumber units: 1048576.0

]

{ #category : 'Audit and Repair' }
Repository >> objectAudit [
 "See comments in objectAuditWithMaxThreads:percentCpuActiveLimit:"

^ self objectAuditWithMaxThreads: self getDefaultNumThreads percentCpuActiveLimit: 90

]

{ #category : 'Audit and Repair' }
Repository >> objectAuditPartial [

"Similar to objectAudit, except that it aborts after processing the scavengable pages
and audits the rest of the repository aborting as needed to avoid causing a commit record
backlog.   Does not check all object table references for validity, but will detect
most references to non-existant objects."

^self _objectAuditWithMaxThreads: self getDefaultNumThreads  kind: 0 .

]

{ #category : 'Audit and Repair' }
Repository >> objectAuditPartialWithMaxThreads: numThreads [

"Similar to objectAuditPartial except that it allows specifying the number of threads to use for the scan.  "
^self _objectAuditWithMaxThreads: numThreads kind: 0 .

]

{ #category : 'Audit and Repair' }
Repository >> objectAuditToCsvFile: aFileName [
 "Similar to the basic objectAudit except that if a valid file name is provided for the
  csvFile (comma separated values) argument then a line is written to the file for each
  error encountered.  The lines contain the following information:
    1. ErrorKind - the name associated with each of the errors defined above.
    2. ObjectId  - the objectId for the object with an error.
    3. ClassName  - the name of the object's class.
    4. Offset    - the offset (or other integer value, e.g. size)
    5. Reference - the reference that does not exist.
  If no errors are found, the file is deleted. "

| errCount |

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

errCount := self _objectAuditWithMaxThreads: self getDefaultNumThreads
                      waitForLock: 60 pageBufSize: 8
                      percentCpuActiveLimit: 90
                      csvFile: aFileName kind: 1"full audit in transaction" .
errCount == 0 ifFalse:[
  RepositoryError new
     details: 'Object Audit Errors , ' , errCount asString , ' errors found' ;
     signalNotTrappable
  ].
^ true.

]

{ #category : 'Audit and Repair' }
Repository >> objectAuditWithMaxThreads: numThreads [

"Similar to objectAudit except that it allows specifying the number of threads to use for the scan.
 The number of threads can be overridden for any other listing references method by executing:
    SessionTemps current at: #GsOverrideNumThreads put: numThreads
     before executing the method.
     "
^self _objectAuditWithMaxThreads: numThreads kind: 1 .

]

{ #category : 'Audit and Repair' }
Repository >> objectAuditWithMaxThreads: maxThreads percentCpuActiveLimit: aPercent [

"Checks all objects in GemStone for consistency.

 This method should be executed from topaz -l (the linked version of Topaz).

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.
 After the initial abort, this method stays in transaction holding a reference
 to the starting commit record for the duration of the audit.

 This method requires SystemControl privileges.

 A GciHardBreak during this method will terminate the session.

 This method raises the error #3022 if audit detects problems, otherwise
 returns true."

| errCount |

System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

errCount := self _objectAuditWithMaxThreads: maxThreads waitForLock: 60 pageBufSize: 8
                      percentCpuActiveLimit: aPercent
                      csvFile: nil kind: 1"full audit in transaction" .
errCount == 0 ifFalse:[
  RepositoryError new
     details: 'Object Audit Errors , ' , errCount asString , ' errors found' ;
     signalNotTrappable
  ].
^ true.

]

{ #category : 'Updating' }
Repository >> objectSecurityPolicy: aSeg [
"Disallowed"
self shouldNotImplement: #objectSecurityPolicy:

]

{ #category : 'Listing By Size' }
Repository >> objectsInMemoryLargerThan: aSize [
  "Return an Array of up to 1000 objects that are in memory and larger than aSize."

  ^  self _listInstancesInMemory: nil limit: aSize option: 4 

]

{ #category : 'Transaction Logging' }
Repository >> oldestLogFileIdForRecovery [

"Returns a positive SmallInteger, which is the internal fileId of the oldest
 transaction log needed to recover from the most recent checkpoint, if the
 Stone were to fail right now.  Returns nil of no tranlog is open for writing."

^ (self _currentLogInfo) ifNotNil:[ :info | info at: 10 ] ifNil:[ nil ].

]

{ #category : 'Deprecated Page-Order Operations' }
Repository >> openPageOrderOopFile: aString [

"Deprecated, See GsBitmap category 'File Operations', these methods
    don't keep the file open, so no need to open/close with an Id.

 Opens a binary page-ordered file of object ID's created by the
 Repository>>listInstancesInPageOrder:toFile: method.

 Raises an error if the file could not be opened for reading, the file
 is corrupt, or aString is not an instance of String.

 Returns a SmallInteger which is the ID of the opened file.  This ID must
 be used to read the contents of the file (see the method
 readObjectsFromFileWithId:startingAt:upTo:into: in this class).

 This method acquires and retains the garbage collection lock.   No garbage
 collection operations can be started until the
 Repository>>closePageOrderOopFileWithId: method is executed or the session
 logs out."

<primitive: 830>
aString _validateKindOfClass: String .
aString size == 0 ifTrue:[ aString _error: #errArgTooSmall args:{ 1 } ].
^ self _primitiveFailed: #openPageOrderOopFile: args: { aString }

]

{ #category : 'Repository Usage Reporting' }
Repository >> pageSize [

"Returns size in bytes of a disk page in the Repository."

^ System _zeroArgPrim:50

]

{ #category : 'Repository Usage Reporting' }
Repository >> pagesWithPercentFree: aPercent [

"This method returns an Array containing the following statistics:

 1.  The total number of data pages processed.
 2.  The sum in bytes of unused space in all data pages.  This quantity is a
     measure of data fragmentation in the receiver.
 3.  The number of bytes in a page.
 4.  The number of data pages that have at least the specified percentage of
     unused space.
 5.  The number of data pages that have at least the specified percentage of
     unused space and contain only 1 object.
 6.  The number of data pages that contain only 1 object.
 7.  The number of pages that should be in the scavengablePages that are not.

 Do not confuse unused space on a page with free space (unused pages) in a
 Repository or Extent.  See the freeSpace and freeSpaceInExtent: methods
 for more information.

 This method requires the GarbageCollection privilege."

^ self _pagesWithPercentFree: aPercent withMaxThreads:
              self getDefaultNumThreads maxCpuUsage: 90 doScavenge: false

]

{ #category : 'Multithreaded Scan Tuning' }
Repository >> pauseMultiThreadedScan [

"Suspends any multithreaded scan in progress by setting the number of threads
 available to the scan to 0.  The session running the scan is not stopped or
 killed, it is in a suspended state waiting to be resumed.

 Returns true if successful, false if the session running the multithreaded
 scan could not be determined.

 Raises an error if the session holding the GC lock is not running on this
 host or if the GC lock session is not performing a multithreaded repository
 scan.

 Examples of multithreaded scan operations include:
   markForCollection
   findDisconnectedObjects
   listInstances
   listReferences
   objectAudit

 Multithreaded scan operations always hold the GC lock.  However not all
 operations which hold the GC lock are multithreaded scans.

 The scan may be resumed by executing one of the setMultiThreadedScan* methods
 in this class.

 Requires the SessionAccess and SystemControl privileges."

^ self setMultiThreadedScanThreads: 0  maxCpu: nil "nil means no change"

]

{ #category : 'Formatting' }
Repository >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

"Copy the implementation from Object so we don't inherit it from Collection."

aStream nextPutAll: self asString

]

{ #category : 'Deprecated' }
Repository >> quickObjectAuditWithLevel: anInt [

" Deprecated, use #fastObjectAudit instead of this method."

self deprecated: 'quickObjectAuditWithLevel: deprecated in v3.2, use fastObjectAudit instead'.
^ self fastObjectAudit

]

{ #category : 'Deprecated Page-Order Operations' }
Repository >> readObjectsFromFileWithId: aSmallInt startingAt: startIndex upTo: endIndex into: anArray [

"Deprecated: use GsBitmap>>readFromFile:withLimit:startingAt:

 Reads and validates objects from a page-ordered file which was previously
 opened with the Repository>>openPageOrderOopFile: method.

 aSmallInt must be the result of the Repository>>openPageOrderOopFile: method.

 startIndex is the index of the first object in the file to read.  The first
 object in the file has an index of 1.  It is an error if startIndex is less than 1
 or greater than endIndex.

 endIndex is the index of the last object to read from the file.  endIndex
 must be greater than or equal to startIndex.  endIndex may exceed the index of
 the last object in the file.  In that case, all objects from startIndex to the
 end of the file will be returned.

 anArray must be an instance of Array when the resulting objects will be
 stored.  It is an error if anArray is not empty.

 Returns a SmallInteger which is the size of anArray.  This will be the lesser
 of the number (endingIndex - startIndex + 1) or, all remaining objects in the
 file.

 It is possible that one or more object identifiers contained in the file
 are no longer valid due to garbage collection.  Objects which are no longer
 valid have nil stored in their place in anArray.  Objects that have been
 garbage collected and are in the free oop list or the dead not reclaimed set
 are considered to be invalid.

 The session must be in a transaction when this method is invoked.  If the
 session is not in a transaction, an #rtErrPrimOutsideTrans error is raised.

 For best performance, it is recommended that no more than 2034 objects
 be returned by any single invocation of this method.  Returning a high
 number of objects will cause anArray to grow very large and may result in
 a fatal out-of-memory error."

 self deprecated: 'Repository>>readObjectsFromFileWithId:startingAt:upTo:into: ',
  'deprecated v3.4. Use GsBitmap>>readFromFile:withLimit:startingAt:'.

^ self _primReadObjectsFromFileWithId: aSmallInt startingAt: startIndex upTo: endIndex into: anArray

]

{ #category : 'Garbage Collection' }
Repository >> reclaimAll [

"Explicitly triggers the reclamation of all shadowed and dead objects if
 the Gc configuration parameter #reclaimDeadEnabled.

 The need for running reclaimAll is reduced by the newer multi-threaded
 implementations of backup, restore and objectAudit.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.

 This method requires the GarbageCollection and SystemControl privileges.

 Signals an error if the timeout expires before the reclaim completes.

 This method waits for voting to complete and then the reclaim of any shadowed pages or dead objects.

 It waits forever, but throws an error if it detects that there is no progress for 5 minutes.
 "

 self reclaimAllWait: -1

]

{ #category : 'Garbage Collection' }
Repository >> reclaimAllWait: timeLimitSeconds [

"Explicitly triggers the reclamation of all shadowed objects.
 If the Gc configuration parameter #reclaimDeadEnabled is true, then it waits
 until the dead objects are ready to reclaim (voteStateIdle) before it starts the reclaim.

 If the timeLimitSeconds value is -1, then it will wait indefinitely for the reclaim to complete.
 This should be done with caution, however, as under certain conditions
 the session could appear to wait forever.  To avoid this you need to:

     1.  Make sure that other sessions are committing/aborting to allow
         voting on possible dead to complete.

 This method signals an error if:
     1.  The voteState is not idle and AdminGem is not running.
     2.  ReclaimGem is not running.
     3.  The timeout expires before the reclaim completes.
          Explicit errors are returned to indicate whether it was waiting for voteStateIdle or the reclaim.
     4.  No progress is detected for 5 minutes.

 A warning is logged if a session takes more than 20 seconds to vote.
 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.

 This method requires the GarbageCollection and SystemControl privileges.
"

| voteState gcUserUg reclaimDead sleepCount sleepTime timeLimit pagesNeedReclaiming prevPagesNeedReclaiming
  deadNotReclaimed prevDeadNotReclaimed noProgressCount tStart |
  System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .
  System myUserProfile _validatePrivilegeName:#SystemControl ; _validatePrivilegeName:#GarbageCollection .
  tStart := DateAndTime now .
  System startAllGcGems.
  System waitForAllGcGemsToStartForUpToSeconds: 30 .
  System hasMissingGcGems ifTrue:[
    System _adminGemMissing ifTrue:[self _error: #rtErrReclaimAllMissingGcGem].
    (System reclaimGemSessionId == 0) ifTrue:[self _error: #rtErrReclaimAllMissingGcGem].
  ].
  timeLimit := timeLimitSeconds.
  timeLimit = -1 ifTrue: [ timeLimit := SmallInteger maximumValue ].
" Wait for voteState idle if needed"
  gcUserUg := ((AllUsers userWithId:'GcUser' ) resolveSymbol:#UserGlobals ) value.
  reclaimDead := gcUserUg at:#reclaimDeadEnabled .
  sleepCount := 0.
  voteState := System voteState.
  (voteState == 0) ifFalse: [
    reclaimDead ifTrue: [ "wait for vote state idle"  | prevSessVoting currSessVoting voteCount |
      [(System voteState) > 0] whileTrue: [
        System voteState == 2 ifTrue: [  "voting"
          currSessVoting := System stoneCacheStatisticWithName: 'WaitingForSessionToVote'.
          currSessVoting == prevSessVoting ifTrue: [
	      sleepCount - voteCount > 20 ifTrue: [
	         GsFile gciLogServer:'WARNING -- session ', currSessVoting asString ,
		                     ' taking too long to vote'] ]
            ifFalse: [ prevSessVoting := currSessVoting. voteCount := sleepCount]
        ].
        Delay waitForSeconds: 1. System abortTransaction.
        sleepCount := sleepCount + 1.
        (sleepCount > timeLimit) ifTrue: [ 
          UserDefinedError new _number: 2318;
          reason: #fail ; details: 'Timeout of ' , timeLimit asString , 
                                   ' seconds expired while waiting for voteState idle' ; signal.  ]
       ]
     ] ifFalse: [
       ((System deadNotReclaimedCount > 0) or: [System possibleDeadSize > 0]) ifTrue: [
           GsFile gciLogServer:'WARNING --reclaimAll: ',
             'vote state is not idle and reclaimDeadEnabled is false, so ',
             'dead objects will not be reclaimed at this time.' ]
     ]
   ].

"do simple commits to ensure CRs are disposed and reclaim completes"
  sleepTime := 1.
  noProgressCount := 0.
  prevPagesNeedReclaiming := 0.
  prevDeadNotReclaimed := 0.
  System abortTransaction .
  [(System _simpleCommitForReclaim: reclaimDead)] whileFalse: [
    Delay waitForSeconds: sleepTime .
    sleepCount := sleepCount + 1 .
    sleepCount > timeLimit ifTrue: [
      UserDefinedError new _number: 2318; reason: #fail ;
         details: 'Waited too long for reclaimAll with timeLimit ' , timeLimit asString; signal.
    ].
    deadNotReclaimed := System deadNotReclaimedCount.
    pagesNeedReclaiming := System scavengablePagesCount.

    (sleepCount \\ 10) == 0  ifTrue:[
      GsFile gciLogServer:
       '--reclaimAll: ', DateTime now asStringMs,' simpleCommit loopCount ' , sleepCount asString ,
       ' crBacklog ' , ((System commitRecordBacklog) asString) ,
       ' deadNotReclamed ' , deadNotReclaimed asString,
       ' pagesNeedReclaiming ' , pagesNeedReclaiming asString, 
       ' sessionsReferencingOldestCr ', System sessionsReferencingOldestCr printString . 
    ].
    ((reclaimDead and: [deadNotReclaimed < prevDeadNotReclaimed])
         or: [pagesNeedReclaiming < prevPagesNeedReclaiming]) ifTrue: [ 
           noProgressCount := 0.
       ] ifFalse: [ 
         noProgressCount := noProgressCount + sleepTime.
       ].
    prevPagesNeedReclaiming := pagesNeedReclaiming.
    prevDeadNotReclaimed := deadNotReclaimed.
    (noProgressCount > 600) ifTrue: [  | tError |
      tError := DateAndTime now .
      GsFile gciLogServer:'--( ', tError asString;
         gciLogServer: System currentSessionsReport , ' )--'.
      RepositoryError signal:'Waited too long in reclaimAll, not making progress'.
    ].
    System abortTransaction.
  ].

]

{ #category : 'Copying' }
Repository >> remove: anObj [
"Disallowed"
self shouldNotImplement: #remove:

]

{ #category : 'Copying' }
Repository >> remove: anObj ifAbsent: aBlock [
"Disallowed"
self shouldNotImplement: #remove:ifAbsent:

]

{ #category : 'Copying' }
Repository >> removeAll: aCollection [
"Disallowed"
self shouldNotImplement: #removeAll:

]

{ #category : 'Copying' }
Repository >> removeAllIdentical: aCollection [
"Disallowed"
self shouldNotImplement: #removeAllIdentical:

]

{ #category : 'Copying' }
Repository >> removeIdentical: anObj [
"Disallowed"
self shouldNotImplement: #removeIdentical:

]

{ #category : 'Copying' }
Repository >> removeIdentical: anObj ifAbsent: aBlock [
"Disallowed"
self shouldNotImplement: #removeIdentical:ifAbsent:

]

{ #category : 'Audit and Repair' }
Repository >> repair [
" This method requires SystemControl privileges.

  See the comments in:
    _objectAuditWithMaxThreads:waitForLock:pageBufSize:percentCpuActiveLimit:csvFile:kind:
  for the kinds of repository errors that can be repaired."


System needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^self _objectAuditWithMaxThreads: self getDefaultNumThreads
                      waitForLock: 60 pageBufSize: 8
                      percentCpuActiveLimit: 90
                      csvFile: nil kind: 2.

]

{ #category : 'Deprecated' }
Repository >> repairWithLimit: sizeLimit [

" Deprecated, use #repair instead of this method."
self deprecated: 'Repository>>repairWithLimit: deprecated in v3.2. Use repair instead.'.
^self repair

]

{ #category : 'Backup and Restore' }
Repository >> restoreActive [

"Returns true if the restore is currently active, false if not."

^(self restoreStatusInfo at: 2) ~= 0

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromArchiveLogs: archiveLogDir [

"Similar to restoreFromLogs except that the restore operation looks
 for the tranlogs to restore in the directory(s) specified by archiveLogDir
 instead of the currently configured STN_TRAN_LOG_DIRECTORIES.

 archiveLogDir is either a String specifying a directory on the stone host,
 or an Array of Strings specifying multiple directories on the stone host.
 It is an error if any of the directories specified does not exist.   
 Any of the directories may be empty.

 When opening a log file on a file system, if the filename ends in '.gz',
 the file is expected to be in gzip compressed format.  If the filename
 ends in '.lz4', the file is expected to be in lz4 compressed format.

 Environment variables , if any, in the directory specifications
 are expanded using the environment of the gem  or topaz -l) process
 running this session, before the directory specifications are passed to
 stone for execution of the restore operation.

 Terminates when all of the logs in the specified directories have
 been restored or if an error is discovered while processing the logs."

self restoreFromArchiveLogs: archiveLogDir withPrefix: nil

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromArchiveLogs: archiveLogDir toEndOfLog: fileId [

"Similar to restoreFromArchiveLogs: except that the restore stops
 when the end of the log with fileId is reached."

self restoreFromArchiveLogs: archiveLogDir toEndOfLog: fileId withPrefix: nil

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromArchiveLogs: archiveLogDir toEndOfLog: fileId withPrefix: prefixString [

"Similar to restoreFromArchiveLogs:toEndOfLog: except that the specified
 prefixString is used instead of the value of the configuration parameter
 specified by STN_TRAN_LOG_PREFIX."

self _checkRestoreActive.
self _setArchiveLogs: archiveLogDir withPrefix: prefixString.
self _restoreLogs: fileId time: 0 archiveLogs: 1 timeout: 60

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromArchiveLogs: archiveLogDir toPointInTime: aDateTime [

"Similar to restoreFromArchiveLogs: except that the restore stops at
 the first checkpoint which originally occurred at or after aDateTime.

 An error is generated if aDateTime precedes the time of the last restored
 checkpoint."

 self restoreFromArchiveLogs: archiveLogDir toPointInTime: aDateTime withPrefix: nil

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromArchiveLogs: archiveLogDir toPointInTime: aDateTime withPrefix: prefixString [

"Similar to restoreFromArchiveLogs:toPointInTime: except that the specified
 prefixString is used instead of the value of the configuration parameter
 specified by STN_TRAN_LOG_PREFIX.
 aDateTime may be a DateTime or a DateAndTime 
"

| aTime |
self _checkRestoreActive.
aTime := self _checkPointInTime: aDateTime.
self _setArchiveLogs: archiveLogDir withPrefix: prefixString.
^ self _restoreLogs: -1 time: aTime archiveLogs: 1 timeout: 60

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromArchiveLogs: archiveLogDir withPrefix: aPrefix [

"Similar to restoreFromLogs except that the restore operation looks
 for the tranlogs in locations specified by archiveLogDir 
 instead of the currently configured STN_TRAN_LOG_DIRECTORIES.

 archiveLogDir is either a String specifying a directory on the stone host,
 or an Array of Strings specifying multiple directories on the stone host.
 It is an error if any of the directories specified does not exist.  
 Any directory specified may be empty.

 aPrefix specifies file name pattern in sequence instead of standard tranlog.
 aPrefix must be astring if nil default tranlog<n> is used

 Terminates when all of the logs in the specified directories have
 been restored or if an error is discovered while processing the logs."

self _checkRestoreActive.
self _setArchiveLogs: archiveLogDir withPrefix: aPrefix.
self _restoreLogs: -1 time: 0 archiveLogs: 1 timeout: 60 .

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromBackup: fileNameOrArrayOfNames [

"Disables logins and starts a full restore of the repository based
 on the contents of the specified backup file.

 Restored objects are clustered in a manner that is similar, but not
 necessarily identical to the clustering at the time the  backup file
 was created.  If the Repository being restored into has the same
 number of extents as the system had when the backup file was created,
 then distribution of objects within extents is preserved unless one
 of the extents becomes full.  If the number of extents is different
 than the number when the backup was created, then the current
 DBF_ALLOCATION_MODE configuration controls the distribution of
 objects across the extents.

 If the backup file was made when in partial-logging mode, then logins
 are reenabled and the system is fully operational when the restore
 completes.  If the backup was made when in full-logging mode, then
 the system is not fully operational until tranlogs have been
 restored and the commitRestore method is executed.

 If the file has a .gz or .lz4 suffix, then the file can be specified
 with or without the suffix.  The restore checks for files that match
 the filename.  If it doesn't find a match and the filename doesn't
 contain a suffix, then the restore attempts to find a match first
 with the .gz suffix and then with the .lz4 suffix.

 To minimize the size of the resulting repository the stone should be
 restarted on a copy of the initial repository (copied from
 $GEMSTONE/bin/extent0.dbf).

 To optimize the time to restore the backup, the exents in the new
 repository should be pregrown to the minimum expected size of
 the restored repository.

 Upon successful completion, the session is automatically logged out and
 the RestoreBackupSuccess error (4046) is generated. Also, stone 
 closes the current tranlog, and then writes a checkpoint to the extents
 which puts the extents into ""restore from logs"" state .
 While in ""restore from logs"", no tranlogs are written, extents
 are checkpointed when a checkpoint is replayed during restore from tranlogs,
 and extents are checkpointed if stone is shutdown.

 A GciHardBreak during this method will terminate the session.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser.

 This method performs the restore using multiple threads.
 The number of threads is automatically set to 2 times the
 number of extents in the repository.  This can be overridden by
 executing:
     SessionTemps current at: #GsOverrideNumThreads put: numThreads.
 where numThreads can be any value between 1 and 4 * numCpus before
 executing the restore method.
 The performance can be modified during the run by
 updating the Multithreaded Scan Tuning methods.

 The restore must be run from a process on the same machine as the stone.
 If it is attempted from a remote machine, a repository error (2734) is
 reported.  "

  ^ self restoreFromBackup: fileNameOrArrayOfNames newSystemUserPassword: nil 

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromBackup: fileNameOrArrayOfNames newSystemUserPassword: aString [

 "aString , if not nil, is a password to be installed for SystemUser.

  If argument to the newSystemUserPassword: keyword is non-nil,
  You must be logged in as SystemUser to do this restore.  The restore stores the new password
  in encrypted form in the repository root page.  The first login as SystemUser after the
  restore validates against the password, and if successful commits the SystemUser's password
  information to the security data objects in the repository.

  Otherwise the same as Repository >> restoreFromBackup: "

  | arr | 
  arr := fileNameOrArrayOfNames.
  fileNameOrArrayOfNames _isArray ifFalse: [ arr := { fileNameOrArrayOfNames } ].
  ^ self restoreFromBackups: arr newSystemUserPassword: aString

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromBackups: arrayOfFileNames [

"Similar to restoreFromBackup: except that it accepts an Array of fileNames."

^ self _restoreBackups: arrayOfFileNames scavPercentFree: 100
                        bufSize: 8 numThreads: self _getNumThreads
			options: (self _getRestoreOptions) newSystemUserPassword: nil

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromBackups: arrayOfFileNames newSystemUserPassword: aString [

"Similar to restoreFromBackup: except that it accepts an Array of fileNames.

 aString , if not nil, is a password to be installed for SystemUser.
 You must be logged in as SystemUser to do a restore with a non nil argument
 to the newSystemUserPassword: keyword ."

^ self _restoreBackups: arrayOfFileNames scavPercentFree: 100
                        bufSize: 8 numThreads: self _getNumThreads
			options: (self _getRestoreOptions)
      newSystemUserPassword: aString 

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromBackups: arrayOfFileNames scavengePagesWithPercentFree: aPercent [

"Same as restoreFromBackups, except that restored pages that have greater
 than aPercent free space are added to the scavengable pages at the end
 of the restore.  WARNING: a small percentage, less than 5 percent can cause
 the reclaim gems to be very busy after the restore."

^ self _restoreBackups: arrayOfFileNames scavPercentFree: aPercent
                        bufSize: 8 numThreads: self _getNumThreads
			options: (self _getRestoreOptions) newSystemUserPassword: nil

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromBackups: fileNameOrArrayOfNames threads: numThreads [
"The same as restoreFromBackups: except that it specifies the number of
 threads to use in the restore. The number of threads can be overridden for any other
 restore method by executing:
   SessionTemps current at: #GsOverrideNumThreads put: numThreads
 before executing the method.
"

  | arr | 
  arr := fileNameOrArrayOfNames.
  fileNameOrArrayOfNames _isArray ifFalse: [ arr := { fileNameOrArrayOfNames } ].
  ^ self setDefaultNumThreads: numThreads during: [
     self restoreFromBackups: arr
  ]

]

{ #category : 'Backup and Restore' }
Repository >> restoreFromCurrentLogs [

"After a restoreFromBackup: successfully completes this method may be
 executed to replay transactions which occurred since the backup was made.
 If the restorestatus of the repository is not activeFromLog then an error
 is generated.

 This method attempts to sequentially replay all transaction logs from
 the first transaction log required to the end of the current logs.

 If previous restore operation was a variant of restoreFromArchiveLogs,
 the last archive log will be reopened and any data written since
 the last EOF postion restored, before restoring the first current log.

 The restoreStatus or restoreStatusNextFileId methods can be used to determine
 the next file required for a restoreLogs operation.

 If some of the tranlogs have been archived and are not in the locations
 configured with the STN_TRAN_LOG_DIRECTORIES, then the restoreFromArchiveLogs:
 methods should be used to restore up until the next files are in the
 current tranlog directories.

 When opening a log file on a file system, if the filename ends in '.gz',
 the file is expected to be in gzip compressed format.  If the filename
 ends in '.lz4', the file is expected to be in lz4 compressed format.

 Restoring a compressed tranlog from a raw partition is not supported.

 Upon successful completion, the session is automatically logged out and
 the RestoreLogSuccess error (4048) is generated.

 If an error occurs, such as a corrupted tranlog or a missing tranlog in
 the sequence then an error is generated and the session is not loggged out.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser."

self _checkRestoreActive.
^ self _restoreLogs: -1 time: 0 archiveLogs: 0 timeout: 60

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackup: aFileName privateDecryptionKey: aKey passphrase: aPassphrase [

^ self restoreFromSecureBackup: aFileName privateDecryptionKey: aKey passphrase: aPassphrase
   newSystemUserPassword: nil

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackup: aFileName privateDecryptionKey: aKey passphrase: aPassphrase  newSystemUserPassword: aString [

^ self restoreFromSecureBackups: { aFileName } scavengePagesWithPercentFree: 100
   privateDecryptionKey: aKey passphrase: aPassphrase newSystemUserPassword: aString

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackup: aFileName privateDecryptionKey: aKey passphraseFile: pfFileName [
  ^ self restoreFromSecureBackup: aFileName privateDecryptionKey: aKey passphraseFile: pfFileName
        newSystemUserPassword: nil

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackup: aFileName privateDecryptionKey: aKey passphraseFile: pfFileName  newSystemUserPassword: aString [

|pf|
pfFileName ifNotNil:[ pf := (GsFile getContentsOfServerFile: pfFileName) trimWhiteSpace ] .
^ self restoreFromSecureBackups: { aFileName } 
       scavengePagesWithPercentFree: 100
       privateDecryptionKey: aKey 
       passphrase: pf newSystemUserPassword: aString

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackup: aFileName scavengePagesWithPercentFree: aPercent privateDecryptionKey: aKey passphrase: aPassphrase [

"Same as restoreFromSecureBackups except this method accepts a single file name."

^ self restoreFromSecureBackups: { aFileName } scavengePagesWithPercentFree: aPercent
privateDecryptionKey: aKey passphrase: aPassphrase newSystemUserPassword: nil

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackup: aFileName scavengePagesWithPercentFree: aPercent privateDecryptionKey: aKey passphraseFile: pfFileName [

"Same as restoreFromSecureBackups except this method accepts a single file name
 and a file name on the server which contains the
passphrase for the decryption key.

Raises an IOError if the passphrase file could not be opened or read.
If aFileName is nil, then the decryption key is assumed to have no passphrase."

|pf|
pfFileName ifNotNil:[ pf := (GsFile getContentsOfServerFile: pfFileName) trimWhiteSpace ] .
^ self restoreFromSecureBackups: { aFileName } 
       scavengePagesWithPercentFree: aPercent
       privateDecryptionKey: aKey 
       passphrase: pf newSystemUserPassword: nil

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackups: anArray privateDecryptionKey: aKey passphrase: aPassphrase [

^ self restoreFromSecureBackups: anArray scavengePagesWithPercentFree: 100
     privateDecryptionKey: aKey passphrase: aPassphrase newSystemUserPassword: nil

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackups: anArray privateDecryptionKey: aKey passphrase: aPassphrase newSystemUserPassword: aString [

^ self restoreFromSecureBackups: anArray scavengePagesWithPercentFree: 100
     privateDecryptionKey: aKey passphrase: aPassphrase newSystemUserPassword: aString

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackups: anArray privateDecryptionKey: aKey passphraseFile: pfFileName [
  ^ self restoreFromSecureBackups: anArray privateDecryptionKey: aKey passphraseFile: pfFileName
      newSystemUserPassword: nil 

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackups: anArray privateDecryptionKey: aKey passphraseFile: pfFileName newSystemUserPassword: aString [

|pf|
pfFileName ifNotNil:[ pf := (GsFile getContentsOfServerFile: pfFileName) trimWhiteSpace ] .
^ self restoreFromSecureBackups: anArray scavengePagesWithPercentFree: 100
   privateDecryptionKey: aKey passphrase: pf newSystemUserPassword: aString

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackups: arrayOfFileNames scavengePagesWithPercentFree: aPercent privateDecryptionKey: aKey passphrase: aPassphrase [
  ^ self restoreFromSecureBackups: arrayOfFileNames scavengePagesWithPercentFree: aPercent
  privateDecryptionKey: aKey passphrase: aPassphrase newSystemUserPassword: nil

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackups: arrayOfFileNames scavengePagesWithPercentFree: aPercent privateDecryptionKey: aKey passphrase: aPassphrase newSystemUserPassword: aString [

"Disables logins and starts a full restore of the repository based
 on the contents of the specified secure backup file(s).  Behaves the
 same as the restoreFromBackup: method except for extra security
 and integrity checks as described below.

 Secure backup files must have a file extension of .sdbf. All
 file names in arrayOfFileNames will have the .sdbf suffix appended
 if it is not present.

 All secure backups have been signed with a private signing key.
 Before the backup is restored, the integrity of each backup file
 is checked by reading the file and computing the signature.  If the
 computed signature does not match the signature stored in the file,
 then an error is raised indicating the backup file has been either
 been corrupted or tampered with. If the signature is correct, the
 restore proceeds.

 All certificates and keys must be in one of the directories specified in
 the key ring, which is the list of directories specified in the
 GEM_KEYRING_DIRS configuration option.  Certificate and key file names
 must be specified without the path.  GemStone will automatically search
 the key ring for the matching file.

 A certificate containing the public key which matches the private
 key used to sign the backup must be present in the key ring. GemStone
 automatically searches all files in the key ring for a certificate which
 contains a matching public key. The restore will fail and an error will be
 raised if the key ring does not contain the appropriate certificate.

 If the backup was encrypted, aKey is a String representing the file name
 containing a private key in PEM format.  The private key must match one
 of the public keys specified when creating the backup. The file must be
 present in the key ring as described above. If the backup is not encrypted
 then aKey must be nil.

 aPassphrase is the passphrase to access the private key. If the backup is
 not encrypted or aKey is not protected by a passphrase (not recommended)
 then aPassphrase must be nil.

 aString , if not nil, is a password to be installed for SystemUser.
 You must be logged in as SystemUser to do a restore with a non nil argument
 to the newSystemUserPassword: keyword .

 Upon successful completion, the session is automatically logged out and
 the RestoreBackupSuccess error (4046) is generated.

 A GciHardBreak during this method will terminate the session.

 This method requires the FileControl privilege.  It is recommended
 that it be run by either DataCurator or SystemUser.

 This method performs the restore using multiple threads.
 The number of threads is automatically set to 2 times the
 number of extents in the repository.  This can be overridden by
 executing:
     SessionTemps current at: #GsOverrideNumThreads put: numThreads.
 where numThreads can be any value between 1 and 4 * numCpus before
 executing the restore method.
 The performance can be modified during the run by
 updating the Multithreaded Scan Tuning methods.

 WARNING: a small percentage, less than 5 percent can cause
 the reclaim gems to be very busy after the restore."

aString ifNotNil:[ (AllUsers userWithId:'SystemUser') _validate__password: aString ].

^ self _primRestoreSecureBackups: arrayOfFileNames scavPercentFree: aPercent bufSize: 8
  privateDecryptionKey: aKey passphrase: aPassphrase numThreads: self _getNumThreads
  options: (self _getRestoreOptions) newSystemUserPassword: aString 

]

{ #category : 'Secure Backup and Restore' }
Repository >> restoreFromSecureBackups: arrayOfFileNames scavengePagesWithPercentFree: aPercent privateDecryptionKey: aKey passphraseFile: aFileName [

"Same as the method
  restoreFromSecureBackups:scavengePagesWithPercentFree:privateDecryptionKey:passphrase:
except this method specifies a file name on the server which contains the
passphrase for the decryption key.
Raises an IOError if the passphrase file could not be opened or read.

If aFileName is nil, then the decryption key is assumed to have no passphrase.

 Upon successful completion, the session is automatically logged out and
 the RestoreBackupSuccess error (4046) is generated."

|pf|
aFileName ifNotNil:[ pf := (GsFile getContentsOfServerFile: aFileName) trimWhiteSpace ] .
^ self restoreFromSecureBackups: arrayOfFileNames 
       scavengePagesWithPercentFree: aPercent
       privateDecryptionKey: aKey 
       passphrase: pf

]

{ #category : 'Backup and Restore' }
Repository >> restorePosition [
  "Return a tranlog position as a ScaledDecimal with scale 10"
  | info |
  info := self restoreStatusInfo .
  ^ self class _scaledDecimalFromFileId: (info at: 9) blockId:(info at: 10 ).
]

{ #category : 'Backup and Restore' }
Repository >> restoreStatus [

"Returns a String describing the current restore status of the Repository,
 including the next transaction log file required to continue
 the restore, and the oldest transaction log file that must be present
 for a restore from log."

^ self restoreStatusInfo at: 1

]

{ #category : 'Backup and Restore' }
Repository >> restoreStatusDateAndTimeRestoredTo [

"Returns a DateAndTime which represents time of the last checkpoint
 that the repository was restored to or nil if restore not active,
 or the restored-to time is not available.

 The result will be nil if restore is active but has not yet
 replayed any checkpoints from tranlogs."

  | s |
  s := self restoreStatusInfo at: 17 .  "fix 51350"
  s = 0 ifTrue:[ ^ nil ].
  ^ DateAndTime posixSeconds: s offset: nil 
]

{ #category : 'Backup and Restore' }
Repository >> restoreStatusInfo [

"Returns an Array describing the current restore status of the Repository,
 including the next transaction log file or backup file required to continue
 the restore.  The Array contains the following elements:

   1: a String describing the restore status.
   2: a SmallInteger,
      0 = restore not active,
      1 = restoreFromBackup withErrors
      2 = active from log
      3 = restoreFromLog withErrors
   3: a SmallInteger, next fileId to restore
   4: a SmallInteger, -1 (unused element)
   5: an Integer,  firstStartupTime of the repository
   6: an Integer,  commitSequence.high of the repository
   7: an Integer,  commitSequence.low of the repository
   8: a SmallInteger, fileId of oldest tranlog that must be present for restore.
   9: a SmallInteger, fileId of current restore position (last commit replayed)
   10: a SmallInteger, blockId of current restore position (last commit replayed)
   11: a String, the time of the last checkpoint restored to. (legacy)
   12: a SmallInteger, 0 no log receiver, 1 log receiver is connected to stone.
   13: a SmallInteger, 0 or posix time in seconds of last failover timestamp replayed.
   14: a SmallInteger, 0 or the posix time in seconds of the time returned
                    from the last execution of Repository >> failOverToSlave.
   15: a SmallInteger, fileId of last checkpoint replayed by restore.
   16: a SmallInteger, blockId of last checkpoint replayed by restore.
   17: a SmallInteger, posix time in seconds of time of last checkpoint restored to.

 If restore is not active, the elements 3 to 16 are nil.
 Element 11 may be nil if restore has not yet replayed any checkpoints
 from tranlogs.

 Restore status is an attribute of the Repository, not of the session.
 The restore position in the tranlogs persists across logout/login
 and stopstone/startstone.

 Element 13 of the result persists across logout/login only"

^ System _zeroArgPrim: 143

]

{ #category : 'Backup and Restore' }
Repository >> restoreStatusNextFileId [

"Returns a SmallInteger, the fileId of the next transaction log or backup that
 should be restored, or nil if restore not active."

^ self restoreStatusInfo at: 3

]

{ #category : 'Backup and Restore' }
Repository >> restoreStatusOldestFileId [

"Returns a SmallInteger, the fileId of the oldest transaction log
 that needs to be present for the next restore from log operation,
 or nil if restore not active."

^ self restoreStatusInfo at: 8

]

{ #category : 'Backup and Restore' }
Repository >> restoreStatusPosixTimeRestoredTo [
  "Returns an Integer, or nil if restore is not active."
  | s |
  s := self restoreStatusInfo at: 17 .  "fix 51350"
  s = 0 ifTrue:[ ^ nil ].
  ^ s
]

{ #category : 'Backup and Restore' }
Repository >> restoreStatusTimeRestoredTo [

"Returns a String which represents time of the last checkpoint
 that the repository was restored to or nil if restore not active,
 or the restored-to time is not available.

 The result will be nil if restore is active but has not yet
 replayed any checkpoints from tranlogs."

^ self restoreStatusInfo at: 11

]

{ #category : 'Backup and Restore' }
Repository >> restoreToEndOfLog: fileId [

"Similar to restoreFromCurrentLogs except that the restore stops when it has
 completed replaying the transactions in the file specified by fileId.

 The fileId argument must be a positive SmallInteger.

 A subsequent restore operation may be used to continue restoring logs
 past the log location at which this operation stopped."

self _checkRestoreActive.
fileId _validateClass: SmallInteger.
fileId > 0 ifFalse:[ fileId _error: #rtErrArgNotPositive ].
^ self _restoreLogs: fileId time: 0 archiveLogs: 0 timeout: 60

]

{ #category : 'Backup and Restore' }
Repository >> restoreToPointInTime: aDateTime [

"Similar to restoreToEndOfLog except that the restore stops the restore at
 the first checkpoint which originally occurred at or after aDateTime.

 An error is generated if aDateTime precedes the time of the last restored
 checkpoint.

 aDateTime may be a DateTime or a DateAndTime"

| aTimeT |
self _checkRestoreActive.
aTimeT := self _checkPointInTime: aDateTime.
self _restoreLogs: -1 time: aTimeT archiveLogs: 0 timeout: 60

]

{ #category : 'Hot Standby' }
Repository >> resumeCommits [

  ^ System resumeCommits

]

{ #category : 'Secure Backup and Restore' }
Repository >> secureFullBackupTo: fileNames MBytes: mByteLimit compressKind: compKind bufSize: count encryptKind: encKind publicKeyCerts: anArrayOrString signatureHashKind: hashKind signingKey: signingKeyFn signingKeyPassphrase: aPassphrase [

"Writes a secure full backup file containing the most recently committed
 version of  the receiver as of the time the method is executed.

 Secure backups support compression, encryption and digital signatures.
 Compression and encryption are optional.  Digital signatures are mandatory.

 fileNames must either be a String which is the backup file name or an Array of
 Strings containing a list of backup file names.

 Secure backups have a file extension of .sdbf, which will be appended to all
 backup file names if necessary.

 The following compression kinds are supported.  Lz4 is recommended since it is
 faster than zlib, but does not compress as much.

 compKind argument meanings:
   0 - no compression
   1 - zlib (aka gzip) compression. (default is level 6)
   2 - lz4 compression. (default is level 0)

 In addition, the following compKind values may be used to specify both the compression
 algorithm and the compression level:

   Zlib:
   101 - fastest performance, lowest compression ratio
   102 - 108 - intermediate performance and compression ratios
   109 - slowest performance, highest compression ratio.

   Lz4
   200 - fastest performance, lowest compression ratio.
   201 - 211 - intermediate performance and compression ratios
   212 - slowest performance, highest compression ratio.

 encKind argument meanings:
   An encKind of zero indicates the backup is not encrypted.
   An encKind greater than zero indicates the backup is encrypted using AES-CTR
   mode using the key size shown below:

   0 - no encryption
   1 - AES-CTR-128
   2 - AES-CTR-192
   3 - AES-CTR-256

 hashKind argument meanings:
   hashKind specifies what message digest (hash) algorithm will be used to
   hash the signature before it is encrypted and stored in the backup
   file.  The following message digests are supported:

   1 - SHA256
   2 - SHA384
   3 - SHA512

 Unless otherwise noted, all certificates and keys must be in one
 of the directories specified in the key ring, which is the list
 of directories specified in the GEM_KEYRING_DIRS configuration option.
 Certificate and key file names must be specified without the path.
 GemStone will automatically search the key ring for the matching file.

 If encKind is greater than zero, the publicKeyCerts argument anArrayOrString must
 either be a String or an Array of Strings where each element is the filename
 of an X509 certificate in PEM format that contains an RSA public key
 without a passphrase. The RSA public key(s) may be any valid length
 (1024, 2048, 3072, bits etc). A key length of at least 2048 bits
 is recommended. anArrayOrString must be nil if encKind is zero. A maximum
 of 24 public key certificates may be specified. The corresponding
 private key(s) for these certificate(s) need not be present in the key ring
 at the time the secure backup is created.  However at least one private key
 will be required in order to restore the backup.

 When encryption is enabled, each backup file is encrypted with a randomly
 generated session key, which is then encrypted with the provided public
 key(s) and stored in the backup file.  One of the corresponding private keys
 must be available in order to restore the backup, but are not required when
 the backup is created.

 signingKeyFn is the name of a private key file in PEM format used to sign the
 backup.  The file must be in the key ring. aPassphrase is a String which is
 the passphrase for the private key. If the private key does not use a
 passphrase, then aPassphrase must be nil. A certificate containing the public
 key which matches the private key specified in privateKeyFn is stored in the
 backup and must also be present in the key ring.  GemStone automatically
 searches the keyring for the matching certificate.

 For encrypting backups, only RSA keys are supported.
 For signing backups, both RSA and DSA private keys are supported.

 The bufSize argument controls the size of the buffer used to write records to
 the file. The count argument specifies the number of 128KB backup records are
 contained in the buffer. The values allowed are 1 (128KB) through 8192 (1GB).

 This method performs the backup using multiple threads.
 The number of threads is automatically set to 2 times the
 number of extents in the repository.  This can be overridden by
 executing:
     SessionTemps current at: #GsOverrideNumThreads put: numThreads.
 where numThreads can be any value between 1 and 4 * numCpus before
 executing the backup method.
 The performance can be modified during the run by
 updating the Multithreaded Scan Tuning methods.

Returns true if the backup was completed."

| limitArray fnArray |
(fileNames isKindOfClass: Array)
  ifTrue: [ fnArray := fileNames ]
  ifFalse: [ fnArray := { fileNames } ] .
limitArray := self _validateFileNames: fnArray limits: mByteLimit.
^ self _primSecureFullBackupTo: fnArray MBytes: limitArray compressKind: compKind
       bufSize: count encryptKind: encKind publicKeyCerts: anArrayOrString
       signatureHashKind: hashKind signingKey: signingKeyFn
       signingKeyPassphrase: aPassphrase numThreads: self _getNumThreads

]

{ #category : 'Secure Backup and Restore' }
Repository >> secureFullBackupTo: fileNames MBytes: mByteLimit compressKind: compKind bufSize: count encryptKind: encKind publicKeyCerts: anArrayOrString signatureHashKind: hashKind signingKey: signingKeyFn signingKeyPassphraseFile: aFileName [

"Same as the method
  secureFullBackupTo:MBytes:compressKind:bufSize:encryptKind:publicKeyCerts:signatureHashKind:signingKey:signingKeyPassphrase:
except this method specifies a file name on the server which contains the
passphrase for the signing key.

Raises an IOError if the passphrase file could not be opened or read.

If aFileName is nil, then the signing key is assumed to have no passphrase.

Returns true if the secure backup was completed."

|pf|
aFileName ifNotNil:[ pf := (GsFile getContentsOfServerFile: aFileName) trimWhiteSpace ] .
^ self secureFullBackupTo: fileNames 
       MBytes: mByteLimit 
       compressKind: compKind 
       bufSize: count 
       encryptKind: encKind 
       publicKeyCerts: anArrayOrString 
       signatureHashKind: hashKind
       signingKey: signingKeyFn 
       signingKeyPassphrase: pf

]

{ #category : 'Secure Backup and Restore' }
Repository >> secureGzFullBackupTo: fileNames MBytes: mByteLimit compressLevel: cLevel bufSize: count encryptKind: encKind publicKeyCerts: anArrayOrString signatureHashKind: hashKind signingKey: signingKeyFn signingKeyPassphrase: aPassphrase numThreads: numThreads [

"Same as the secureFullBackupTo: method except the zlib compression level cLevel 
 and number of threads numThreads are be specified directly.
 cLevel must be between 1 and 9. See the comments in the secureFullBackupTo:
 method for more information."
 
| limitArray fnArray |
(fileNames isKindOfClass: Array)
  ifTrue: [ fnArray := fileNames ]
  ifFalse: [ fnArray := { fileNames } ] .
((cLevel < 1) or:[cLevel > 9])
   ifTrue:[self _error: #rtErrArgOutOfRange args:{ 1 . 9 } ] .
limitArray := self _validateFileNames: fnArray limits: mByteLimit.
^ self _primSecureFullBackupTo: fnArray MBytes: limitArray compressKind: (100 + cLevel)
       bufSize: count encryptKind: encKind publicKeyCerts: anArrayOrString
       signatureHashKind: hashKind signingKey: signingKeyFn
       signingKeyPassphrase: aPassphrase numThreads: numThreads

]

{ #category : 'Secure Backup and Restore' }
Repository >> secureLz4FullBackupTo: fileNames MBytes: mByteLimit compressLevel: cLevel bufSize: count encryptKind: encKind publicKeyCerts: anArrayOrString signatureHashKind: hashKind signingKey: signingKeyFn signingKeyPassphrase: aPassphrase numThreads: numThreads [

"Same as the secureFullBackupTo: method except the lz4 compression level cLevel 
 and number of threads numThreads are be specified directly.
 cLevel must be between 0 and 12. See the comments in the secureFullBackupTo:
 method for more information."

| limitArray fnArray |
(fileNames isKindOfClass: Array)
  ifTrue: [ fnArray := fileNames ]
  ifFalse: [ fnArray := { fileNames } ] .
((cLevel < 0) or:[cLevel > 12])
   ifTrue:[self _error: #rtErrArgOutOfRange args:{ 0 . 12 } ] .
limitArray := self _validateFileNames: fnArray limits: mByteLimit.
^ self _primSecureFullBackupTo: fnArray MBytes: limitArray compressKind: (200 + cLevel)
       bufSize: count encryptKind: encKind publicKeyCerts: anArrayOrString
       signatureHashKind: hashKind signingKey: signingKeyFn
       signingKeyPassphrase: aPassphrase numThreads: numThreads

]

{ #category : 'Default numThreads' }
Repository >> setDefaultNumThreads: numThreads [

"Sets default number of threads to use for multi threaded operations"

 SessionTemps current at: #GsOverrideNumThreads put: numThreads

]

{ #category : 'Default numThreads' }
Repository >> setDefaultNumThreads: numThreads during: aBlock [
"Sets default number of threads to use for the execution of aBlock containing 
 a multi threaded operation and then resets it after the operation completes."

| prevDefault |
 prevDefault := SessionTemps current at: #GsOverrideNumThreads otherwise: nil.
 SessionTemps current at: #GsOverrideNumThreads put: numThreads.
 ^ aBlock ensure:
     [prevDefault 
        ifNil: [ SessionTemps current removeKey: #GsOverrideNumThreads ]
        ifNotNil: [  SessionTemps current at: #GsOverrideNumThreads put: prevDefault ]
     ].

]

{ #category : 'Multithreaded Scan Tuning' }
Repository >> setMultiThreadedScanMaxAggressive [

"Makes any multithreaded scan in progress as aggressive as possible.  This
 will cause the scan to complete as quickly as possible at the expense of
 consuming more system resources.

 Returns true on success, false if the multithreaded scan session could not
 be found on this host.

 Raises an error if the session holding the GC lock is not running on this
 host."

^ self setMultiThreadedScanThreads:
       self getMultiThreadedScanMaxThreads maxCpu: 95 .

]

{ #category : 'Multithreaded Scan Tuning' }
Repository >> setMultiThreadedScanMediumAggressive [

"Makes any multithreaded scan in progress moderately aggressive.
 This is a compromise between completing as quickly as possible and the
 overall impact on the system.

 Returns true on success, false if the multithreaded scan session could not
 be found on this host.

 Raises an error if the session holding the GC lock is not running on this
 host."

| maxThds |
maxThds := self getMultiThreadedScanMaxThreads .

"Use 50% of the maximum number of threads configured."
maxThds := 1 min: (maxThds // 2) .

^ self setMultiThreadedScanThreads: maxThds maxCpu: 80 .

]

{ #category : 'Multithreaded Scan Tuning' }
Repository >> setMultiThreadedScanMinAggressive [

"Makes any multithreaded scan in progress run as slowly as possible without
 suspending the scan.  This will cause the scan to take much more time to
 complete and will greatly reduce the system resources used by the scan.

 Requires the SessionAccess and SystemControl privileges.

 Returns true on success, false if the multithreaded scan session could not
 be found on this host.

 Raises an error if the session holding the GC lock is not running on this
 host."

  ^ self setMultiThreadedScanThreads: 1 maxCpu: 60 .

]

{ #category : 'Multithreaded Scan Tuning' }
Repository >> setMultiThreadedScanThreads: numThreads maxCpu: aPercent [

"Change the number of threads active for a multithreaded scan.
 Also changes the maximum CPU usage parameter if aPercent is not nil.

 Returns true if successful, false if the session running the multithreaded
 scan could not be determined.

 Raises an error if the session holding the GC lock is not running on this
 host.

 Raises an error if numThreads is greater than the maximum number of threads for
 which the scan was configured (i.e., the value of the maxThreads argument
 passed to the multithreaded scan method).

 Requires the SessionAccess and SystemControl privileges."

| sessions |
sessions := System sessionsHoldingGcLock .
sessions size == 0 ifTrue:[ RepositoryError signal: 'No scan in progress' ].

sessions do:[:sesId | 
  self mtThreadsLimit: sesId setValue: numThreads .
  aPercent ifNotNil:[ self mtPercentCpuActiveLimit: sesId setValue: aPercent ] .
  ].
^ true "success if we get here"

]

{ #category : 'Backup and Restore' }
Repository >> setTranlogPrivateKeysForRestore: anArray [

"Specifies a list of private keys to be used to read encrypted tranlogs
 during a restoreFromArchiveLogs operation. Each element of anArray must
 be an instance of GsTlsPrivateKey, and is added to the list of keys in
 the stone process that is used to decrypt tranlogs.

 The private key used to start the stone is always available to the stone
 and does not need to be specified here.

 During a restoreFromArchiveLogs operation, the stone searches the 
 stone's list of keys for a match to the public key stored in the
 encrypted tranlog.  An error is raised if no match is found and the
 restore operation fails.

 This method requires the FileControl privilege.
 
 Returns true on success."

<primitive: 1109>
anArray _validateInstanceOf: Array.
self _primitiveFailed: #setTranlogPrivateKeysForRestore: args: { anArray }

]

{ #category : 'Copying' }
Repository >> shallowCopy [
"Disallowed"
self shouldNotImplement: #shallowCopy

]

{ #category : 'Updating' }
Repository >> size: aSize [
"Disallowed"
self shouldNotImplement: #size:

]

{ #category : 'Class Membership' }
Repository >> species [

"Returns a class similar to, or the same as, the receiver's class which
 can be used for containing derived copies of the receiver."

^ Array

]

{ #category : 'Updating' }
Repository >> squeakBasicAt: anIndex put: aValue [
  "Disallowed"
  self shouldNotImplement: #squeakBasicAt:put:

]

{ #category : 'Transaction Logging' }
Repository >> startNewLog [

"Causes the most current transaction log to be closed and a new transaction log
 to be opened for writing.  The location of the new log is controlled by the
 STN_TRAN_LOG_DIRECTORIES configuration file parameter.

 If GemStone is running in partial logging mode, then a preceding transaction
 log may be deleted.  See documentation on the STN_TRAN_FULL_LOGGING
 configuration parameter for more details.

 If a checkpoint is in progress, or the repository is in restore mode,
 the operation is NOT performed and a -1 is returned.
 If it is successful the SmallInteger fileId of the new log is returned.

 This method requires the FileControl privilege."

<primitive: 432>

self _primitiveFailed: #startNewLog .
self _uncontinuableError

]

{ #category : 'Hot Standby' }
Repository >> stopContinuousRestore [

"Terminate a previous continuousRestoreFromArchiveLogs:.
 If a delay was specified, then stop immediately, otherwise
 processes the tranlogs found in the archive logs directory
 to be restored, but does not wait for more logs to be
 received by the logreceiver process.
 Has no effect if no such restore is active."

  self _checkRestoreActive.
  self _restoreLogs: -1 time: 0 archiveLogs: 3 timeout: 60

]

{ #category : 'Hot Standby' }
Repository >> suspendCommitsForFailover [

 self deprecated: 'Repository>>suspendCommitsForFailover  deprecated in v3.7.0.  Use failOverToSlave instead.'.
  ^ self failOverToSlave

]

{ #category : 'Migrating Objects' }
Repository >> testMigrateMt: arrayOfMappings with: testObjs [

" This method is similar to migrateMt: and can be used to test the migrateMt operation 
  on a number of objects before performing a full repository scan.

  It performs a multi threaded scan of only the objects listed in the test array, migrating the
  objects in the same way that migrateMt: would, but it does NOT commit them. The session is 
  left in a state with commits disallowed. Once the objects have been inspected and have been
  verified that the migration performed as expected the user must abort the current transaction.
  The abort must be done in order to execute the migrateMt: that will scan the entire database 
  and perform the operation on all of the objects matching the oldClass in the InstVarMappingArrays.

  This method requires the MigrateObjects privilege.
  "

  1 to: testObjs size do: [ :i | | obj |
    obj := testObjs at: i.
    obj isCommitted ifFalse: [ obj _error: #rtErrOpNotAllowed args:
       { 'testMigateMt does not support testing with non committed objects' }
    ]
  ].
       
  ^ self _basicMigrateWithMaxThreads: self getDefaultNumThreads waitForLock: 3 pageBufSize: 8
              percentCpuActiveLimit: 90 mappingArrays: arrayOfMappings testArray: testObjs

]

{ #category : 'Extent Operations' }
Repository >> validateExtentId: anExtentId [

"Returns the argument if it is valid, otherwise generates an error."

| maxExtentId |
anExtentId == nil ifTrue:[ ^ anExtentId ] .
anExtentId _validateClass: SmallInteger.
maxExtentId := self numberOfExtents .
(anExtentId < 1 or:[ anExtentId > maxExtentId]) ifTrue:[
   anExtentId _error: #rtErrArgOutOfRange args:{ 1 . maxExtentId }.
   anExtentId < 1 ifTrue:[ ^ 1 ].
   anExtentId > maxExtentId ifTrue:[ ^ maxExtentId].
].
^ anExtentId .

]

{ #category : 'Backup and Restore' }
Repository >> waitForFailoverFromMasterUpToSeconds: seconds [

"On a slave stone, waits up to seconds for the fail over record from the master stone to be replayed.
If seconds is 0, this method returns a status immediately and does not block.
If seconds is -1, this method waits forever for the fail over record.
Returns true if the fail over record has been replayed or false if the time out expires.
Raises an exception if the repository is not in restore from log state."

| startTime rep waitTimeMs sys |
sys := System.
rep := SystemRepository.
waitTimeMs := seconds * 1000.
startTime := sys _timeMs.
[true] whileTrue:[ | failOverTs info|
  info := rep restoreStatusInfo.
 ( info at: 2) == 2
   ifFalse:[ ImproperOperation signal: 'Repository is not in restoreFromLogs state' ].
  failOverTs := info at: 13.
  failOverTs > 0 ifTrue:[ ^ true ].
  waitTimeMs == 0 ifTrue:[ ^ false ].
  waitTimeMs > 0 ifTrue:[ | now |
	now := sys _timeMs.
	now - startTime > waitTimeMs
		ifTrue:[ ^ false ].
  ].
  sys _sleepMs: 500 . "Delay methods do not work during upgrade"
]. "forever"
    

]

{ #category : 'Disconnected Objects' }
Repository >> writeFdcArrayToFile: aFileNameString [
"Write a bitmap with the array of dead objects found by the last findDisconnectedObjects
 (FDC) to the specified file. This can be loaded into a hidden set and used to analyze
 garbage.

 The file is created by this method and must not exist when it is called.

 anArray must be an Array object and contain objects sorted by object ID.
 The primitive will fail if the objects are not in order.

 aFileNameString must be an instance of String that refers to a file which
 does not yet exist.

 Returns true if successful, false if the entire file could not be written,
 or nil if the results of the last FDC could not be found."

 |array theSize result|
 array := Globals at: #FdcResults otherwise: nil.
 (array == nil)
  ifTrue:[^nil].
 (array size < 3)
  ifTrue:[^nil].
 array := array at: 3.
 (array class == Array)
    ifFalse:[^nil].
 theSize := array size.
 (theSize == 0)
   ifTrue:[^nil].
 result := self _writeFdcArray: array toFile: aFileNameString.
 ^ result == theSize.

]

{ #category : 'Storing and Loading' }
Repository >> writeTo: aPassiveObject [

"Instances of Repository cannot be converted to passive form.  This method
 writes nil to aPassiveObject and stops GemStone Smalltalk execution with a
 notifier."

aPassiveObject writeObject: nil.
self _error: #rtErrAttemptToPassivateInvalidObject

]
