Extension { #name : 'RcQueue' }

{ #category : 'Instance Creation' }
RcQueue class >> new [

"Returns a new RcQueue."

^ (super new: 20) _initialize

]

{ #category : 'Instance Creation' }
RcQueue class >> new: initialNumberOfUsers [

"Returns a new RcQueue with a size that supports initialNumberOfUsers.  The
 new RcQueue will handle more users, but will have subcomponents created
 for initialNumberOfUsers."

| newOne |
newOne := (super new: initialNumberOfUsers) _initialize.
^ newOne initialize

]

{ #category : 'Obsolete' }
RcQueue >> _addAll: anArray [
  ^ self addAll: anArray
]

{ #category : 'Accessing' }
RcQueue >> _calculateSize [

"Returns the number of valid entries in the RcQueue."

| count firstValid |
count := 0.
1 to: super size do: [:i |
  firstValid := self _findFirstValidQueueElementForSession: i.
  firstValid ~~ nil
    ifTrue: [ count := count + ((self at: i) size - firstValid + 1) ]
  ].
System rcValueCacheAt: #size put: count for: self.

^count

]

{ #category : 'Enumerating' }
RcQueue >> _collectElems [

"This private method is added to support C++ access to the valid elements
 in the RcQueue (the C++ equivalent of do:)."

| resultArray |

resultArray := { } .
self do: [:each | resultArray add: each].
^resultArray

]

{ #category : 'Private' }
RcQueue >> _deepCopyWith: copiedObjDict [

| copy |
copy := copiedObjDict at: self otherwise: nil.
copy ~~ nil ifTrue: [ ^ copy ].

copy := self class new: (self size).
copiedObjDict at: self put: copy.

self _deepCopyNamedIvsWith: copiedObjDict to: copy .

self _timeSortedComponents do: [:ea |
   copy add: (ea value _deepCopyWith: copiedObjDict)
   ].
^ copy.

]

{ #category : 'Private' }
RcQueue >> _doComponents: aBlock [
    "Evaluates aBlock with each of the current element components of
    the RcQueue as the argument. The argument aBlock must be a
    one-argument block. This method does not traverse the queue
    elements in order."

    | sessionComponent firstValid |
    1 to: self maxSessionId do:
        [ :i |
         sessionComponent := self at: i.
         sessionComponent ~~ nil
             ifTrue:
                 [firstValid := self _findFirstValidQueueElementForSession: i.
                  firstValid to: sessionComponent size do:
                      [ :j |
                       aBlock value: (sessionComponent at: j)]]]

]

{ #category : 'Error Handling' }
RcQueue >> _errorIndexOutOfRange: anIndex [

"Sends an error message indicating that anIndex was outside legal limits
 for the receiver."

^ self _error: #objErrBadOffsetIncomplete args: { anIndex - 1 }

]

{ #category : 'Error Handling' }
RcQueue >> _errorRcQueueEntriesFound [

"In attempting to reset the maximum session ID for the queue to a smaller value,
 an entry was found that was added by a session whose session ID is larger than
 the value currently being set.  Remove and save all entries in the queue.  Then
 change the maximum session ID and add the saved entries back into the queue."

^ self _error: #rtErrRcQueueEntriesFound

]

{ #category : 'Obsolete' }
RcQueue >> _fastAddAll: anArray [
  ^ self addAll: anArray
]

{ #category : 'Private' }
RcQueue >> _findFirstValidQueueElementForSession: aSessionId [

"Returns the index of the first valid queue element for a session.

 The first valid queue element is the one whose sequence number is the smallest
 value that is greater than the removal sequence number of session.

 The argument aSessionId is an instance of SmallInteger that has a value that
 is one greater than the session number (which is returned by the
 System | session method)."

<primitive: 638>
aSessionId _validateClass: SmallInteger .
^self _primitiveFailed: #_findFirstValidQueueElementForSession:
      args: { aSessionId } .

]

{ #category : 'Private' }
RcQueue >> _getSessionComponentFor: offset [

"Create an RcQueueSessionComponent for the session with the given offset."

| sessionComponent |
offset > self maxSessionId ifTrue: [ self changeMaxSessionId: offset ].

sessionComponent := self _rcAt: offset.
sessionComponent ifNil:[ 
  sessionComponent := RcQueueSessionComponent new.
  sessionComponent objectSecurityPolicy: self objectSecurityPolicy .
  self at: offset put: sessionComponent.
  System redoLog addConflictObject: self for: self ;
          addConflictObject: removalSeqNumbers for: self.
].
^ sessionComponent

]

{ #category : 'Initializing' }
RcQueue >> _initialize [

"Creates and initializes the components of the RcQueue.  The new method
 automatically initializes a new instance of RcQueue, so there is rarely any
 need for a user application to call initialize. Not Rc"
| seg |
seg := self objectSecurityPolicy .
removalSeqNumbers := RcQueueRemovalSeqNumbers new: super size.
removalSeqNumbers objectSecurityPolicy: seg.
1 to: super size do: [:i | | newOne |
  removalSeqNumbers at: i put: 0.
  newOne := RcQueueSessionComponent new.
  newOne objectSecurityPolicy: seg.
  self at: i put: newOne.
].

]

{ #category : 'Testing' }
RcQueue >> _isLarge [

"Returns true if the object is implemented as a tree of private smaller objects"

^ true

]

{ #category : 'Locking Support' }
RcQueue >> _lockableValues [

"Returns a kind of object usable as an argument to _lockAll: primitives.
 This method returns all the elements logically contained in the receiver."

^ self _collectElems

]

{ #category : 'Private' }
RcQueue >> _logRemoval [
  "Log the fact that a removal occurred (used in _resolveRcConflicts:)."
  | redoLog  redoObjs |
  (redoLog := System redoLog) addConflictObject: removalSeqNumbers for: self.

  ((redoObjs := redoLog redoObjects) at: self otherwise: nil) ifNil:[
    redoObjs  at: self put: removalSeqNumbers copy 
  ]
]

{ #category : 'Private' }
RcQueue >> _notPerformedRemoval [

"Returns whether the receiver did not perform a removal during the current
 transaction."

^ (System redoLog redoObjects at: self otherwise: nil) == nil

]

{ #category : 'private - extensions' }
RcQueue >> _numberInvalidForMySession [

"Returns the number of entries in the RcQueue which have been removed
but
 not yet reclaimed in for my session.
The intent is to determine if it is worth invoking cleanup using
cleanupMySession."

| firstValid |

firstValid := self _findFirstValidQueueElementForSession: (System session + 1).
firstValid == nil
  ifTrue:[ ^0 ]
  ifFalse:[ ^firstValid - 1 ]

]

{ #category : 'Private' }
RcQueue >> _performedRemoval [

"Returns whether the receiver performed a removal during the current
 transaction."

^ (System redoLog redoObjects at: self otherwise: nil) ~~ nil

]

{ #category : 'Private' }
RcQueue >> _refreshAfterCommitFailure [

"Returns whether the receiver should be selectively aborted when there is a
 failed attempt to commit.  Only special Reduced Objects should answer true."

^ true

]

{ #category : 'Private' }
RcQueue >> _removalSeqNumbersAtBeginningOfTransaction [

"Returns a copy of removalSeqNumbers that was stored before the first
 removal occurred."

^ System redoLog redoObjects at: self otherwise: nil

]

{ #category : 'private - extensions' }
RcQueue >> _removeAllInto: anArray [

"Removes all entries from the RcQueue and put them into anArray in the
correct order."

| resultArray validSessions firstValidInSession sz
 minElement minSession firstValid sessionComponent
 anElement systm |


" log the fact that a removal occurred (used in _resolveRcConflicts:) "
self _logRemoval.

anArray _validateClass: Array.

sz := self size.
anArray size: 0.
resultArray := anArray size: sz.
sz == 0
  ifTrue: [ ^ resultArray ].

validSessions := { } .
firstValidInSession := { } .
1 to: self maxSessionId do: [:i |
  sessionComponent := self at: i.
  sessionComponent ~~ nil
    ifTrue: [
      firstValid := self _findFirstValidQueueElementForSession: i.
      (firstValid <= sessionComponent size) ifTrue: [
         validSessions add: sessionComponent.
         firstValidInSession add: firstValid.
         removalSeqNumbers at: i put:
            ((sessionComponent at: (sessionComponent size)) sequenceNumber)
         ]
    ]
  ].
1 to: resultArray size do: [:j |
  minSession := 0.
  minElement := nil.
  1 to: validSessions size do: [:i |
    sessionComponent := validSessions at: i.
    firstValid := firstValidInSession at: i.
    (firstValid <= sessionComponent size) ifTrue: [
      anElement := sessionComponent at: firstValid.
     (anElement isOlderThan: minElement) ifTrue: [
           minElement := anElement.
           minSession := i.
         ].
      ]
    ].
  minElement ~~ nil
    ifTrue: [
      resultArray at: j put: minElement value.
      minElement value: nil.
      firstValidInSession at: minSession put:
                      ((firstValidInSession at: minSession) + 1).
    ]
  ].

systm := System .
(systm rcValueCacheAt: #size for: self otherwise: nil) ifNotNil:[:rcz|
   systm rcValueCacheAt: #size put: 0 for: self ].

^resultArray


]

{ #category : 'private - extensions' }
RcQueue >> _removeCount: maxToRemove into: anArray [

"Removes entries from the RcQueue, and returns the argument, anArray,
 which contains the  minimum of maxToRemove or the queue size entries,
 in order.  It is more efficient to remove multiple entries than to send the
 message remove repeatedly."

| resultArray validSessions validSessionIds firstValidInSession
  theSize minElement minSession firstValid sessionComponent
  anElement systm |

" log the fact that a removal occurred (used in _resolveRcConflicts:) "
self _logRemoval.

anArray _validateClass: Array.

anArray size: 0.
theSize := self size min: (maxToRemove max: 0).
resultArray := anArray size: theSize.

validSessions := { } .
validSessionIds := { } .
firstValidInSession := { } .

1 to: self maxSessionId do: [:i |
  sessionComponent := self at: i.
  sessionComponent ~~ nil
    ifTrue: [
      firstValid := self _findFirstValidQueueElementForSession: i.
      (firstValid <= sessionComponent size) ifTrue: [
         validSessions add: sessionComponent.
         validSessionIds add: i.
         firstValidInSession add: firstValid.
         ]
    ]
  ].
1 to: resultArray size do: [:j |
  minSession := 0.
  minElement := nil.
  1 to: validSessions size do: [:i |
    sessionComponent := validSessions at: i.
    firstValid := firstValidInSession at: i.
    (firstValid <= sessionComponent size) ifTrue: [
      anElement := sessionComponent at: firstValid.
      (anElement isOlderThan: minElement)
          ifTrue: [
             minElement := anElement.
             minSession := i.
           ].
      ]
    ].
  minElement ~~ nil
    ifTrue: [
      resultArray at: j put: minElement value.
      minElement value: nil.
      removalSeqNumbers at: (validSessionIds at: minSession)
                        put: (minElement sequenceNumber).
      firstValidInSession at: minSession
                        put: ((firstValidInSession at: minSession) + 1).
    ]
  ].

systm := System .
(systm rcValueCacheAt: #size for: self otherwise: nil) ifNotNil:[:rcz|
  systm rcValueCacheAt: #size put: (rcz - resultArray size) for: self
].

^resultArray


]

{ #category : 'Private' }
RcQueue >> _resolveRcConflictsWith: conflictObjects [

"A logical write-write conflict has occurred on the receiver.  The objects that
 had the actual physical write-write conflicts are in the conflictObjects
 Array.  Returns whether the conflicts could be successfully resolved."

| offset sessionComponent beginX endX sz |
" If no objects experienced physical conflict, then just returns "
conflictObjects isEmpty
  ifTrue: [ ^ true ].

" determine if this transaction is a consumer and a transaction that
committed in the interim was a consumer "

self _performedRemoval
  ifTrue: [ " this session is a consumer "

    " get a copy before the first removal in the transaction occurred "
    beginX := self _removalSeqNumbersAtBeginningOfTransaction.

    " save a copy of the state at the end of the transaction "
    endX := removalSeqNumbers copy.
    " refresh the view to see other transaction's changes "
    removalSeqNumbers _selectiveAbort.

    " compare removal numbers to see if any changed "
    1 to: (removalSeqNumbers size min: beginX size) do: [ :i |
      (beginX at: i) == (removalSeqNumbers at: i)
        ifFalse: [
          " replace removalSeqNumbers to get a consistent state of the queue "
          removalSeqNumbers := endX.
          ^ false
        ]
    ]
  ].

" at this point, we've failed if two consumers conflicted "

" see if just conflicted on removalSeqNumbers "
( (conflictObjects size == 1) and:[ (conflictObjects at: 1) == removalSeqNumbers ] )
  ifTrue: [
    " if endX is nil, we did not abort removalSeqNumbers above "
    endX ifNil:[
        " save a copy of the state at the end of the transaction "
        endX := removalSeqNumbers copy.
        " refresh the view to see other transaction's changes "
        removalSeqNumbers _selectiveAbort.
      ].

    " compare the sizes of removalSeqNumbers before and after the abort "

    endX size > removalSeqNumbers size
      ifTrue: [
        " other session consumed and committed,
          this session is a producer who grew the root object "

        sz := removalSeqNumbers size.
        " grow the removalSeqNumbers Array and initialize with zeros "
        removalSeqNumbers size: endX size.
        sz + 1 to: endX size do: [ :i | removalSeqNumbers at: i put: 0 ].

        ^ true
      ].

    endX size < removalSeqNumbers size
      ifTrue: [
        " other session was a producer who grew the root and committed,
          this session consumed "

        " overlay my changes to removalSeqNumbers "
        1 to: endX size do: [ :i | removalSeqNumbers at: i put: (endX at: i) ].

        ^ true
      ]
  ].

" if had a physical conflict on the root "
(conflictObjects includes: self) ifTrue: [
        " keep reference to this session's components "
        offset := System session + 1.
        sessionComponent := self _rcAt: offset.

        " abort the root and the removalSeqNumbers "
        super _selectiveAbort.
        removalSeqNumbers _selectiveAbort.

        " check if need to grow the root object "
        offset > self maxSessionId
            ifTrue: [ self changeMaxSessionId: offset ].

        " now re-insert the session element "
        self at: offset put: sessionComponent.

        ^ true
    ].

^ true

]

{ #category : 'Private' }
RcQueue >> _selectiveAbort [

"In addition to aborting the RcQueue, selectively aborts the session components
 and removal sequence numbers.  Returns the receiver."

| sessionComponent |

super _selectiveAbort.
removalSeqNumbers _selectiveAbort.
1 to: self maxSessionId do: [:i |
  sessionComponent := self at: i.
  sessionComponent ~~ nil
    ifTrue: [
      sessionComponent _selectiveAbort.
      1 to: sessionComponent size do: [ :j |
        (sessionComponent at: j) _selectiveAbort
      ]
    ].
].

]

{ #category : 'private - extensions' }
RcQueue >> _statistics [
"Return a SymbolKeyValueDictionary containing statistics
on the RcQueue as follows:
	#largestSessionComponent - the largest session component
	#largestSessionComponentSize - size of the largest session component
	#largestSessionComponentPhysicalSize - physical size of the largest session component, in bytes
	#totalNumberInvalid - total number of invalid entries, including those belonging to active sessions
	#numberInvalid - number of invalid entries which belong to inactive sessions.
	#numberOfSessions - number of concurrent sessions for which the queue is configured
	#queueSize - number of entries in the queue"

	| largest largestSize result invalid firstValid |
	largest := nil.
	largestSize := 0.
	invalid := 0.
	1 to: super size do: [:i | |s e|
  		e := self _rcAt: i.
		s := e size.
		s > largestSize
			ifTrue:[largest := e.
					largestSize := s.
			].
		firstValid := self _findFirstValidQueueElementForSession: i.
		(firstValid > 1) ifTrue: [
  			invalid :=  invalid + firstValid - 1 .
  		 ].

	].
	largest == nil
		ifTrue:[^nil].
	result := SymbolKeyValueDictionary new.
	result add: (#largestSessionComponent -> largest);
		   add: (#largestSessionComponentSize -> largestSize);
		   add: (#largestSessionComponentPhysicalSize -> largest physicalSize);
		   add: (#totalNumberInvalid -> invalid);
		   add: (#numberInvalid -> self numberInvalid);
		   add: (#numberOfSessions -> self maxSessionId);
		   add: (#queueSize -> self size).
	^ result


]

{ #category : 'Private' }
RcQueue >> _timeSortedComponents [
    "Answer a SortedCollection of the receiver's internal queue
    element objects sorted by increasing time."

    | collection |
    collection := SortedCollection sortBlock: [ :a :b | a isOlderThan: b].
    self _doComponents:
        [ :each |
         collection add: each].
    ^collection

]

{ #category : 'private - extensions' }
RcQueue >> _totalNumberInvalid [

"Returns the total number of entries in the RcQueue which have been
removed but not yet reclaimed in all session components.  Unlike the numberInvalid
method, this method includes invalid entries for both active and inactive
sessions.  The intent is to determine if it is worth invoking cleanup."

| sz count firstValid |

sz := super size.
count := 0.
1 to: sz do: [:i |
    firstValid := self _findFirstValidQueueElementForSession: i.
    firstValid ~~ nil
      ifTrue: [ count := count + (firstValid - 1) ]

  ].
^count

]

{ #category : 'private - extensions' }
RcQueue >> _unsafeCleanupQueue [

"Removes all obsolete entries.  Unlike the cleanupQueue method, this method
removes all obsolete entries, including those belonging to active
sessions.  This method will cause concurrency conflicts with the consumer and any
feeders  and should only be used while no sessions are adding to or removing from
the queue."

| maxSessionId  firstValid sessionComponent sessionComponentSize |

maxSessionId := self maxSessionId.
1 to: maxSessionId do: [:i |
    sessionComponent := self at: i.
    sessionComponent ~~ nil
      ifTrue: [
        firstValid := self _findFirstValidQueueElementForSession: i.
        sessionComponentSize := sessionComponent size.
        removalSeqNumbers at: i put: 0.
        (firstValid <= sessionComponentSize)
           ifTrue: [
             (firstValid > 1) ifTrue:[
               sessionComponent removeFrom: 1 to: firstValid - 1 .
               ].
             ]
            ifFalse: [ sessionComponent size: 0]
      ]

  ].
^self


]

{ #category : 'private - extensions' }
RcQueue >> _unsafeResetQueue [
	"Reinitialize the internal RcQueue structures.
	WARNING: this method removes all elements from the queue!
	USE AT YOUR OWN PERIL!"

	self _initialize.
	self initialize.
	^self.

]

{ #category : 'Private' }
RcQueue >> _validateRcConflictsWith: conflictObjects [

" There are four possible cases that we can analyze by looking
the write sets:

1. this transaction writes the removalSeqNum
   another transaction writes the removalSeqNum and the root
In this case we only conflict on removalSeqNum

2. this transaction writes the removalSeqNum
   another transaction writes the removalSeqNum
In this case we only conflict on removalSeqNum

3. this transaction writes the removalSeqNum and the root
   another transaction writes the removalSeqNum and the root
In this case we conflict on both

4. this transaction writes the removalSeqNum and the root
   another transaction writes the removalSeqNum
In this case we only conflict on removalSeqNum

When this transaction writes the removalSeqNum and the root,
we performed an add:, and we can also tell if we performed a removal.

When the other transaction writes the removalSeqNum and the root,
he either performed just an add:, or an add: and a removal:,
and we cannot determine which without doing a selective abort.
"

" If no objects experienced physical conflict, then just returns "
conflictObjects isEmpty
  ifTrue: [ ^ true ].

" if conflicted on both the removalSeqNum and the root (case 3) or
we wrote the root (case 4) "
(conflictObjects size > 1 or:
[ ((GsBitmap newForHiddenSet: #PomWriteSet) includes: self) ])
  ifTrue: [ " see if we also performed a removal "
    ^ self _notPerformedRemoval
  ].

" case 1 and 2 return false (we just cannot tell) "
^ false

]

{ #category : 'Comparing' }
RcQueue >> = anRcQueue [

"Returns true if all of the following conditions are true:

 1.  The receiver and anRcQueue are of the same class.
 2.  The two collections are of the same size.
 3.  The corresponding elements of the receiver and anRcQueue are equal."

| selfSorted otherSorted |

self == anRcQueue
  ifTrue: [ ^ true ].

self class == anRcQueue class
  ifFalse: [ ^ false ].

self size == anRcQueue size
  ifFalse: [ ^ false ].

self size == 0 ifTrue:
    [^true].

selfSorted := self _timeSortedComponents.
otherSorted := anRcQueue _timeSortedComponents.

1 to: selfSorted size do:
    [ :i |
     (selfSorted at: i) value = (otherSorted at: i) value
         ifFalse: [^false]].
^true

]

{ #category : 'Adding' }
RcQueue >> add: aValue [

"Adds aValue to the RcQueue, returns the receiver."

| sessIdx queueElement sessionComponent sessCompSize firstValid lastSeqNumber systm |
systm := System .
sessIdx := systm session + 1.
sessionComponent := self _getSessionComponentFor: sessIdx .
sessCompSize := sessionComponent size.
(sessCompSize > 0) ifTrue: [
  queueElement := sessionComponent _rcAt: sessCompSize.  "add right path to rcReadSet"
  lastSeqNumber := queueElement sequenceNumber.
] ifFalse: [lastSeqNumber := removalSeqNumbers at: sessIdx ].

queueElement := RcQueueEntry
                newWithValue: aValue sequenceNumber: lastSeqNumber + 1 inObjectSecurityPolicyOf: self .
sessionComponent add: queueElement.
sessionComponent _rcAt: sessCompSize + 1. "put right path in rcReadSet"

"clean up entries that have been removed"
firstValid := self _findFirstValidQueueElementForSession: sessIdx.
firstValid > 1 ifTrue: [
  sessionComponent removeFrom: 1 to: firstValid - 1 .
  systm _addToRcReadSet: sessionComponent includingAllNodes: true.
].

(systm rcValueCacheAt: #size for: self otherwise: nil) ifNotNil:[ :rcz |
   systm rcValueCacheAt: #size put: (rcz + 1) for: self
].
^aValue

]

{ #category : 'Adding' }
RcQueue >> addAll: aCollection [

"Adds an array of objects to the RcQueue, returns the receiver.
It is much more efficient to add many elements with this method
than calling add: multiple times."

| sessIdx  queueElement sessionComponent sessCompSize 
  lastSeqNumber clss systm firstValid |

systm := System .
sessIdx  := systm session + 1.
sessionComponent := self _getSessionComponentFor: sessIdx .
sessCompSize := sessionComponent size.
(sessCompSize > 0) ifTrue: [
  queueElement := sessionComponent _rcAt: sessCompSize. "add right path to rcReadSet"
  lastSeqNumber := queueElement sequenceNumber.
] ifFalse:[
  lastSeqNumber := removalSeqNumbers at: sessIdx 
].
clss := RcQueueEntry .
aCollection _isArray ifTrue:[
  1 to: aCollection size do:[:n | | val | 
    val := aCollection at: n .
    queueElement := clss newWithValue: val sequenceNumber: lastSeqNumber + 1
	               inObjectSecurityPolicyOf: self .
    sessionComponent add: queueElement.
    lastSeqNumber := lastSeqNumber + 1.
  ]
] ifFalse:[
  aCollection do:[:val|
    queueElement := clss newWithValue: val sequenceNumber: lastSeqNumber + 1
	               inObjectSecurityPolicyOf: self .
    sessionComponent add: queueElement.
    lastSeqNumber := lastSeqNumber + 1.
  ]
].
"clean up entries that have been removed"
firstValid := self _findFirstValidQueueElementForSession: sessIdx.
firstValid > 1 ifTrue: [
  sessionComponent removeFrom: 1 to: firstValid - 1 .
  systm _addToRcReadSet: sessionComponent includingAllNodes: true.
].

(systm rcValueCacheAt: #size for: self otherwise: nil) ifNotNil:[:rcz|
  systm rcValueCacheAt: #size put: (rcz + aCollection size) for: self 
].
^aCollection

]

{ #category : 'Updating' }
RcQueue >> at: offset put: value [
  value _validateClass: RcQueueSessionComponent.
  ^ super at: offset put: value

]

{ #category : 'Updating' }
RcQueue >> changeMaxSessionId: newMaxSessionId [

"Changes the maximum number of sessions for which the RcQueue is configured.
 Modifying the capacity of an RcQueue in this way may cause concurrency conflicts
 with the consumer session, if one is active."

| oldSize firstValid|

oldSize := self maxSessionId.
(oldSize == newMaxSessionId) ifTrue:[ ^self ].

(newMaxSessionId > oldSize) ifTrue:[ | seg |
    seg := self objectSecurityPolicy .
    removalSeqNumbers size: newMaxSessionId.
    super size: newMaxSessionId.
    oldSize + 1 to: newMaxSessionId do: [:i | | newOne |
      removalSeqNumbers at: i put: 0.
      "Bug 42719 - finish initializing the queue"
      newOne := RcQueueSessionComponent new.
      newOne objectSecurityPolicy: seg.
      self at: i put: newOne.
   ]
] ifFalse: [
    newMaxSessionId + 1 to: oldSize do: [:i |
      firstValid := self _findFirstValidQueueElementForSession: i.
      firstValid ifNotNil:[
        (firstValid <= (self at: i) size) ifTrue:[ ^ self _errorRcQueueEntriesFound].
      ]
    ].
    super size: newMaxSessionId
].
^self

]

{ #category : 'Performance Enhancement' }
RcQueue >> cleanupMySession [

"Cleans up the entries for my session that have already been removed by
 the consumer.  This method is only needed if a producer adds entries to
 the queue faster than the consumer removes them and then no longer adds
 to the queue.  In this situation, the producer can enhance the performance
 of the consumer by either logging out or periodically executing this
 method and committing."

| sessIdx sessionComponent firstValid systm |
systm := System.
sessIdx := systm session + 1.
( sessIdx > self maxSessionId or:[ (sessionComponent := self at: sessIdx) == nil ] )
  ifTrue:[ ^ self ].

"clean up entries that have been removed"
firstValid := self _findFirstValidQueueElementForSession: sessIdx .
(firstValid > 1) ifTrue: [
   sessionComponent removeFrom: 1 to: firstValid - 1 .
   systm _addToRcReadSet: sessionComponent includingAllNodes: true.
].
^self

]

{ #category : 'Updating' }
RcQueue >> cleanupQueue [

"Removes obsolete entries belonging to inactive sessions.  Can cause
 concurrency conflicts with the consumer."

| maxSessionId invalidSessions firstValid
  sessionComponent systm |

maxSessionId := self maxSessionId.
invalidSessions := Array new: maxSessionId.
systm := System.
1 to: maxSessionId do: [:i | invalidSessions at: i put: true].
systm currentSessions do: [:each |
  (each < maxSessionId) ifTrue: [
     invalidSessions at: each + 1 put: false]
  ].

1 to: maxSessionId do: [:i | | componentSz |
  (invalidSessions at: i) ifTrue: [
    sessionComponent := self at: i.
    (componentSz := sessionComponent size) ~~ 0 ifTrue:[
      firstValid := self _findFirstValidQueueElementForSession: i.
      removalSeqNumbers at: i put: 0.
      (firstValid <= componentSz) ifTrue: [
        (firstValid > 1) ifTrue:[
          sessionComponent removeFrom: 1 to: firstValid - 1
        ].
      ] ifFalse: [ sessionComponent size: 0 ]
    ].
  ].
].
^self

]

{ #category : 'Clustering' }
RcQueue >> clusterDepthFirst [

"Performs no action, as clustering defeats the conflict-reduction scheme."

^true

]

{ #category : 'Enumerating' }
RcQueue >> do: aBlock [

"Evaluates aBlock with each of the current elements of the RcQueue as the
 argument.  The argument aBlock must be a one-argument block.  This method does
 not traverse the queue elements in order."

| sessionComponent firstValid |

1 to: self maxSessionId do: [:i |
  sessionComponent := self at: i.
  sessionComponent ~~ nil
    ifTrue: [
      firstValid := self _findFirstValidQueueElementForSession: i.
      firstValid to: sessionComponent size do: [:j |
        aBlock value: (sessionComponent at: j) value].
      ]
    ]

]

{ #category : 'Initialization' }
RcQueue >> initialize [

"Create subcomponents for all available session IDs.  This can avoid
 initial concurrency conflict when many sessions add an object to the
 RcQueue for the first time."

1 to: self maxSessionId do: [ :i |
    self _getSessionComponentFor: i
]

]

{ #category : 'Testing' }
RcQueue >> isEmpty [

"Returns true if the queue is empty, and false otherwise."
^self size == 0

]

{ #category : 'Accessing' }
RcQueue >> maxSessionId [

"Returns the max sessionId that can be used with this RcQueue.
 Equivalent to _basicSize
"
<primitive: 0>   
self _primitiveFailed: #maxSessionId .
self _uncontinuableError

]

{ #category : 'Accessing' }
RcQueue >> numberInvalid [

"Returns the number of entries in the RcQueue which have been removed but
 not yet reclaimed in the session components for sessions that are not active.
 The intent is to determine if it is worth invoking cleanup."

| sz invalidSessions count firstValid |

sz := super size.
invalidSessions := Array new: sz.
1 to: sz do: [:i | invalidSessions at: i put: true].
System currentSessions do: [:each |
  (each < sz) ifTrue: [
     invalidSessions at: each + 1 put: false]
  ].

count := 0.
1 to: sz do: [:i |
  (invalidSessions at: i) ifTrue: [
    firstValid := self _findFirstValidQueueElementForSession: i.
    firstValid ~~ nil
      ifTrue: [ count := count + (firstValid - 1) ]
    ]
  ].
^count

]

{ #category : 'Updating' }
RcQueue >> objectSecurityPolicy: anObjectSecurityPolicy [

"Assigns the receiver and subcomponents to the given ObjectSecurityPolicy."

super objectSecurityPolicy: anObjectSecurityPolicy .
removalSeqNumbers objectSecurityPolicy: anObjectSecurityPolicy .
1 to: self maxSessionId do: [:i | | sessionComponent |
  (sessionComponent := self at: i) ifNotNil:[
      sessionComponent objectSecurityPolicy: anObjectSecurityPolicy .
      1 to: sessionComponent size do: [ :j |
        (sessionComponent at: j) objectSecurityPolicy: anObjectSecurityPolicy
      ]
    ]
].

]

{ #category : 'Removing' }
RcQueue >> peek [

"Returns the leading element from the receiver without removing it.
 If the receiver is empty, returns nil."

| minElement minSession firstValid sessionComponent anElement |


minSession := 0.
minElement := nil.
1 to: self maxSessionId do: [:i |
  sessionComponent := self at: i.
  sessionComponent ~~ nil
    ifTrue: [
      firstValid := self _findFirstValidQueueElementForSession: i.
      (firstValid <= sessionComponent size) ifTrue: [
        anElement := sessionComponent at: firstValid.
        (anElement isOlderThan: minElement) ifTrue: [
         minElement := anElement.
         minSession := i.
         ].
        ]
    ]
  ].
(minSession > 0)
   ifTrue: [
     ^ minElement value.
   ]
   ifFalse: [^nil]

]

{ #category : 'Copying' }
RcQueue >> postCopy [

"Cleanup new copy."

super postCopy.
self removalSeqNumbers: removalSeqNumbers copy.
1 to: self _basicSize do: [:index |
	self _at: index put: (self _at: index) copy.
].

]

{ #category : 'Accessing' }
RcQueue >> removalSeqNumbers [

"Return the value of the instance variable 'removalSeqNumbers'."
^removalSeqNumbers

]

{ #category : 'Updating' }
RcQueue >> removalSeqNumbers: newValue [

"Modify the value of the instance variable 'removalSeqNumbers'."
removalSeqNumbers := newValue

]

{ #category : 'Removing' }
RcQueue >> remove [

"Removes the leading element from the receiver and returns that element.
 If the receiver is empty, returns nil."

| minElement minSession firstValid sessionComponent anElement |

" log the fact that a removal occurred (used in _resolveRcConflicts:) "
self _logRemoval.

minSession := 0.
minElement := nil.
1 to: self maxSessionId do: [:i |
  sessionComponent := self at: i.
  sessionComponent ifNotNil:[ | sz |
    sz := sessionComponent size .
    sz ~~ 0 ifTrue:[
      firstValid := self _findFirstValidQueueElementForSession: i.
      (firstValid <= sz ) ifTrue: [
        anElement := sessionComponent at: firstValid.
        (anElement isOlderThan: minElement) ifTrue: [
	   minElement := anElement.
	   minSession := i.
	].
      ]
    ]
  ]
].
( minSession ~~ 0 and: [ minElement ~~ nil ] )
   ifTrue: [ | systm val |
     removalSeqNumbers at: minSession put: (minElement sequenceNumber).
     val := minElement value.
     minElement value: nil.
     systm := System .
     (systm rcValueCacheAt: #size for: self otherwise: nil) ifNotNil:[:rsz |
        systm rcValueCacheAt: #size put: (rsz - 1) for: self
     ].
     ^ val
   ]
   ifFalse: [^nil]

]

{ #category : 'Removing' }
RcQueue >> remove: oldObject ifAbsent: anExceptionBlock [
  ^ self shouldNotImplement: #remove:ifAbsent: 
]

{ #category : 'Removing' }
RcQueue >> removeAll [

"Removes all entries from the RcQueue, and returns an Array that contains those
 entries, in order.  It is more efficient to use removeAll than to send the
 message remove repeatedly."

| resultArray validSessions firstValidInSession sz
 minElement minSession firstValid sessionComponent anElement |

" log the fact that a removal occurred (used in _resolveRcConflicts:) "
self _logRemoval.

sz := self size.
resultArray := Array new: sz.
sz == 0
  ifTrue: [ ^ resultArray ].

validSessions := { } .
firstValidInSession := { } .
1 to: self maxSessionId do: [:i |
  sessionComponent := self at: i.
  sessionComponent ~~ nil
    ifTrue: [
      firstValid := self _findFirstValidQueueElementForSession: i.
      (firstValid <= sessionComponent size) ifTrue: [
         validSessions add: sessionComponent.
         firstValidInSession add: firstValid.
         removalSeqNumbers at: i put:
            ((sessionComponent at: (sessionComponent size)) sequenceNumber)
         ]
    ]
  ].
1 to: resultArray size do: [:j |
  minSession := 0.
  minElement := nil.
  1 to: validSessions size do: [:i |
    sessionComponent := validSessions at: i.
    firstValid := firstValidInSession at: i.
    (firstValid <= sessionComponent size) ifTrue: [
      anElement := sessionComponent at: firstValid.
    (anElement isOlderThan: minElement) ifTrue: [
           minElement := anElement.
           minSession := i.
         ].
      ]
    ].
  minElement ~~ nil
    ifTrue: [
      resultArray at: j put: minElement value.
      minElement value: nil.
      firstValidInSession at: minSession put:
                      ((firstValidInSession at: minSession) + 1).
    ]
  ].

(System rcValueCacheAt: #size for: self otherwise: nil) ifNotNil:[:rcz|
    System rcValueCacheAt: #size put: 0 for: self ].

^resultArray

]

{ #category : 'Removing' }
RcQueue >> removeAllAsPairsInto: anArray [

"Removes all entries from the RcQueue, and returns anArray which contains
 entries in pairs.  The odd numbered entries are the objects removed from the queue,
 in order.  The even numbered entries are the session IDs of the sessions that added
 the preceding entry."

| validSessions firstValidInSession sz minElement minSessionIdx minSession
  firstValid sessionComponent anElement validSessionIds systm |

sz := self size.
anArray size: (sz * 2).
sz == 0 ifTrue: [ ^ anArray ].

" log the fact that a removal occurred (used in _resolveRcConflicts:) "
self _logRemoval.

validSessions := { } .
validSessionIds := { } .
firstValidInSession := { } .
1 to: self maxSessionId do: [:i |
  sessionComponent := self at: i.
  sessionComponent ~~ nil
    ifTrue: [
      firstValid := self _findFirstValidQueueElementForSession: i.
      (firstValid <= sessionComponent size) ifTrue: [
         validSessions add: sessionComponent.
         firstValidInSession add: firstValid.
         validSessionIds add: i.
         removalSeqNumbers at: i put:
            ((sessionComponent at: (sessionComponent size)) sequenceNumber)
         ]
    ]
  ].
1 to: anArray size by: 2 do: [:j |
  minSessionIdx := 0.
  minElement := nil.
  1 to: validSessions size do: [:i |
    sessionComponent := validSessions at: i.
    firstValid := firstValidInSession at: i.
    (firstValid <= sessionComponent size) ifTrue: [
      anElement := sessionComponent at: firstValid.
    (anElement isOlderThan: minElement) ifTrue: [
           minElement := anElement.
           minSessionIdx := i.
           minSession := validSessionIds at: i.
         ].
      ]
    ].
  minElement ~~ nil
    ifTrue: [
      anArray at: j put: minElement value.
      anArray at: (j + 1) put: (minSession - 1).
      minElement value: nil.
      firstValidInSession at: minSessionIdx put:
                      ((firstValidInSession at: minSessionIdx) + 1).
    ]
  ].

systm := System .
(systm rcValueCacheAt: #size for: self otherwise: nil) ifNotNil:[
    systm rcValueCacheAt: #size put: 0 for: self
].
^anArray

]

{ #category : 'Removing' }
RcQueue >> removeCount: maxToRemove [

"Removes entries from the RcQueue, and returns an Array that contains the
 minimum of maxToRemove or the queue size entries, in order.
 It is more efficient to remove multiple entries than to send the
 message remove repeatedly."

| resultArray validSessions validSessionIds firstValidInSession
  minElement minSession firstValid sessionComponent anElement |

" log the fact that a removal occurred (used in _resolveRcConflicts:) "
self _logRemoval.

resultArray := Array new: ((self size) min: (maxToRemove max: 0)).
validSessions := { } .
validSessionIds := { } .
firstValidInSession := { } .

1 to: self maxSessionId do: [:i |
  sessionComponent := self at: i.
  sessionComponent ~~ nil
    ifTrue: [
      firstValid := self _findFirstValidQueueElementForSession: i.
      (firstValid <= sessionComponent size) ifTrue: [
         validSessions add: sessionComponent.
         validSessionIds add: i.
         firstValidInSession add: firstValid.
         ]
    ]
  ].
1 to: resultArray size do: [:j |
  minSession := 0.
  minElement := nil.
  1 to: validSessions size do: [:i |
    sessionComponent := validSessions at: i.
    firstValid := firstValidInSession at: i.
    (firstValid <= sessionComponent size) ifTrue: [
      anElement := sessionComponent at: firstValid.
      (anElement isOlderThan: minElement) ifTrue: [
         minElement := anElement.
         minSession := i.
         ].
      ]
    ].
  minElement ~~ nil
    ifTrue: [
      resultArray at: j put: minElement value.
      minElement value: nil.
      removalSeqNumbers at: (validSessionIds at: minSession)
                        put: (minElement sequenceNumber).
      firstValidInSession at: minSession
                        put: ((firstValidInSession at: minSession) + 1).
    ]
  ].

(System rcValueCacheAt: #size for: self otherwise: nil) ifNotNil:[:rcz|
  System rcValueCacheAt: #size put: (rcz - resultArray size) for: self
].

^resultArray

]

{ #category : 'Removing' }
RcQueue >> removeCountAsPairs: maxToRemove intoArray: anArray [

"Removes entries from the RcQueue, and returns an Array that contains 2 elements
 per entry removed:  the object removed and the session ID of the session that
 added the element."


| validSessions validSessionIds firstValidInSession numRemoved
  minElement minSessionIdx firstValid sessionComponent anElement systm |

" log the fact that a removal occurred (used in _resolveRcConflicts:) "
self _logRemoval.

anArray size: (2 * ((self size) min: (maxToRemove max: 0))).
validSessions := { } .
validSessionIds := { } .
firstValidInSession := { } .
numRemoved := 0.

1 to: self maxSessionId do: [:i |
  sessionComponent := self at: i.
  sessionComponent ~~ nil
    ifTrue: [
      firstValid := self _findFirstValidQueueElementForSession: i.
      (firstValid <= sessionComponent size) ifTrue: [
         validSessions add: sessionComponent.
         validSessionIds add: i.
         firstValidInSession add: firstValid.
         ]
    ]
  ].
1 to: anArray size by: 2 do: [:j |
  minSessionIdx := 0.
  minElement := nil.
  1 to: validSessions size do: [:i |
    sessionComponent := validSessions at: i.
    firstValid := firstValidInSession at: i.
    (firstValid <= sessionComponent size) ifTrue: [
      anElement := sessionComponent at: firstValid.
      (anElement isOlderThan: minElement) ifTrue: [
         minElement := anElement.
         minSessionIdx := i.
         ].
      ]
    ].
  minElement ~~ nil
    ifTrue: [
      | minSessionId |
      minSessionId := validSessionIds at: minSessionIdx.
      anArray at: j put: minElement value.
      anArray at: (j + 1) put: minSessionId - 1.
      numRemoved := numRemoved + 1.
      minElement value: nil.
      removalSeqNumbers at: minSessionId
                        put: (minElement sequenceNumber).
      firstValidInSession at: minSessionIdx
                        put: ((firstValidInSession at: minSessionIdx) + 1).
    ]
  ].
systm := System .
(systm rcValueCacheAt: #size for: self otherwise: nil) ifNotNil:[:rcz|
  systm rcValueCacheAt: #size put: (rcz - numRemoved) for: self
].

^anArray

]

{ #category : 'Removing' }
RcQueue >> removeIdentical: oldObject ifAbsent: anExceptionBlock [
  ^ self shouldNotImplement: #removeIdentical:ifAbsent: 
]

{ #category : 'Removing' }
RcQueue >> removeIntoArray [

"Removes the leading element from the receiver and stores the element in
 a new Array at index 1.  A SmallInteger representing the session ID of the
 session that added the element to the queue is also stored in the new Array
 at index 2.

 If the receiver is empty, returns nil."

^ self removeIntoArray: { nil . nil }

]

{ #category : 'Removing' }
RcQueue >> removeIntoArray: anArray [

"Removes the leading element from the receiver and stores the element in
 anArray at index 1.  A SmallInteger representing the session ID of the
 session that added the element to the queue is also stored in anArray
 at index 2.

 If the receiver is empty, returns nil."

| minElement minSession firstValid sessionComponent anElement val |

" log the fact that a removal occurred (used in _resolveRcConflicts:) "
self _logRemoval.

minSession := 0.
minElement := nil.
1 to: self maxSessionId do: [:i |
  sessionComponent := self at: i.
  sessionComponent ~~ nil
    ifTrue: [
      firstValid := self _findFirstValidQueueElementForSession: i.
      (firstValid <= sessionComponent size) ifTrue: [
        anElement := sessionComponent at: firstValid.
       (anElement isOlderThan: minElement)
          ifTrue: [
             minElement := anElement.
             minSession := i.
           ].
        ]
    ]
  ].
( minSession > 0 and: [ minElement ~~ nil ] )
   ifTrue: [ | systm |
     removalSeqNumbers at: minSession put: (minElement sequenceNumber).
     val := minElement value.
     minElement value: nil.
     anArray at: 1 put: val.
     anArray at: 2 put: (minSession - 1).
     systm := System .
     (systm rcValueCacheAt: #size for: self otherwise: nil) ifNotNil:[:rcz|
         systm rcValueCacheAt: #size put: (rcz - 1) for: self ].

     ^anArray .
   ]
   ifFalse: [^nil]

]

{ #category : 'Removing' }
RcQueue >> removeObject: anObject addedBySessionId: sessionId [

"Removes the given object from the receiver assuming it is the oldest
 object added by the given session Id.  Returns the object or nil if
 the object was not found or the receiver is empty."

| firstValid sessionComponent anElement val sessIdx systm |

 " adjust sessionId, check range"
 sessIdx := sessionId + 1.
 ( (sessIdx < 1 ) or: [ sessIdx > self _basicSize ] )
   ifTrue: [ ^ nil ].

 " fetch sessionComponent "
 sessionComponent := self at: sessIdx .
 ( sessionComponent == nil )
   ifTrue:[ ^ nil ].

 " find appropriate element, and from that
   the value, with appropriate range checks "
 firstValid := self _findFirstValidQueueElementForSession: sessIdx .
 ( firstValid > sessionComponent size )
   ifTrue: [ ^ nil ].
 anElement := sessionComponent at: firstValid.
 ( anElement == nil )
   ifTrue:[ ^ nil ].
 val := anElement value.
 ( val == anObject )
   ifFalse: [ ^ nil ].

" log the fact that a removal occurred (used in _resolveRcConflicts:) "
 self _logRemoval.

 " handle details of removing element from rcqueue "
 removalSeqNumbers at: sessIdx put: anElement sequenceNumber.
 anElement value: nil.
 systm := System .
 (systm rcValueCacheAt: #size for: self otherwise: nil) ifNotNil:[:rcz|
   systm rcValueCacheAt: #size put: (rcz - 1) for: self
 ].
 ^ val

]

{ #category : 'Accessing' }
RcQueue >> size [

"Returns the number of valid entries in the RcQueue."

| sz |
sz := System rcValueCacheAt: #size for: self otherwise: nil.
sz ifNil:[ sz := self _calculateSize ].
^ sz

]

{ #category : 'Updating' }
RcQueue >> size: anInteger [

"Disallowed.  You cannot change the size of anRcQueue other than to add or
 remove elements.  To change the maximum sessionId that can be used with
 this queue see changeMaxSessionId:."

^self shouldNotImplement: #size:

]

{ #category : 'Class Membership' }
RcQueue >> species [

"Returns a class similar to, or the same as, the receiver's class which
 can be used for containing derived copies of the receiver."

^ Array

]
