Extension { #name : 'RcKeyValueDictionary' }

{ #category : 'Storing and Loading' }
RcKeyValueDictionary class >> loadFrom: passiveObj [

"Reads from passiveObj the passive form of an object with named instance
 variable format.  Converts the object to its active form by loading the
 information into a new instance of the receiver.  Returns the new instance."

| newDict size |
(passiveObj version >= 500) ifTrue:[
  size := passiveObj readSize.
  newDict := self new.
  newDict loadFrom: passiveObj size: size.
  ]
ifFalse:[
  self halt: 'no support for  activation of objects written by 4.1' ].
^ newDict

]

{ #category : 'Instance Creation' }
RcKeyValueDictionary class >> new [

"Returns a RcKeyValueDictionary with the default table size."

^ self new: 53

]

{ #category : 'Instance Creation' }
RcKeyValueDictionary class >> new: tableSize [

"Returns a RcKeyValueDictionary with the specified table size."

^ super _basicNew initialize: (self _tableSizeFor: tableSize)


]

{ #category : 'Private' }
RcKeyValueDictionary >> _abortAndReplay: conflictObjects [
  Error signal:'should not be in RcKeyValueDictionary >> _abortAndReplay: '.  
]

{ #category : 'Updating' }
RcKeyValueDictionary >> _at: aKey put: aValue oldValue: prevValue logging: aBoolean [

"Stores the aKey/aValue pair in the key-value dictionary.  Rebuilds the hash
 table if the addition caused the number of collisions to exceed the limit
 allowed.  The argument aBoolean determines if an entry is placed in the redo
 log for this operation."

| hash collisionBkt num oldValue |
aKey ifNil: [ ^ self _error: #rtErrNilKey ].
hash := self hashFunction: aKey.
collisionBkt := self _at: hash.  
num := collisionBkt numElements.
oldValue := collisionBkt at: aKey put: aValue.  "adds bucket to rcReadSet"
aBoolean ifTrue: [ | systm |
  oldValue == aValue ifTrue:[ ^ true ]. "did not write, no need to log"
  systm := System .
  systm _addRootObjectToRcReadSet: self .  "add self to rcReadSet, in case of rebuildTable"
  num < collisionBkt numElements ifTrue: [   "a key/value pair was added"
    systm rcValueCacheIncrement: #size for: self by: 1 .
    (num + 1) > collisionLimitPerBucket ifTrue: [
         self rebuildTable: (self class _tableSizeFor: self _basicSize * 2)
    ].
  ].
  self _logAt: aKey put: aValue oldValue: oldValue inCollisionBucket: collisionBkt.
] ifFalse: [ "in replay"
   oldValue ~~ prevValue ifTrue: [ ^ false ]. "collision on same key, not RC"
].
^ true

]

{ #category : 'Private' }
RcKeyValueDictionary >> _bucketClass [
  "Subclasses may override this."

  ^RcCollisionBucket

]

{ #category : 'Accessing' }
RcKeyValueDictionary >> _calculateSize [

"Calculates the number of key/value pairs in the receiver and places it in the
 RC value cache.  Returns the total."

| num collisionBkt |
num := 0.
1 to: self _basicSize do: [ :i |
    collisionBkt := self basicAt: i.
    num := num + collisionBkt size
].
System rcValueCacheAt: #size put: num for: self.
^ num

]

{ #category : 'Updating' }
RcKeyValueDictionary >> _collisionLimitPerBucket: aLimit [

"Sets the collisionLimitPerBucket for the receiver."

collisionLimitPerBucket := aLimit

]

{ #category : 'Private' }
RcKeyValueDictionary >> _deferredGciUpdateWith: valueArray [

"Private."

1 to: valueArray size by: 2 do:[:j |
  self at: (valueArray at: j) put: (valueArray at: j + 1)
  ].

]

{ #category : 'Enumerating' }
RcKeyValueDictionary >> _doCollisionBuckets: aBlock [

"Evaluates aBlock with each of the receiver's collision buckets as the
 argument.  The argument aBlock must be a one-argument block."

| collisionBkt |
1 to: self _basicSize do: [ :tableIndex |
    collisionBkt := self basicAt: tableIndex.
    aBlock value: collisionBkt
]

]

{ #category : 'Private' }
RcKeyValueDictionary >> _gciInitialize [

"Private."

| cache |

self _basicSize == 0
  ifTrue: [ self initialize: 53 ].
1 to: self _basicSize do: [ :i | (self basicAt: i) reset ].

" remove receiver from rc value cache "
( (cache := System _rcValueCache) ~~ nil and:[ (cache at: self otherwise: nil) ~~ nil ])
  ifTrue: [ cache removeKey: self ].
^ true

]

{ #category : 'Testing' }
RcKeyValueDictionary >> _isLarge [

"Returns true if the object is implemented as a tree of private smaller objects"

^ true

]

{ #category : 'Locking Support' }
RcKeyValueDictionary >> _lockableValues [

"Returns an Array of the receiver's keys and values."

| result |
result := { }  .
self keysAndValuesDo: [:aKey :aValue |
  result add: aKey .
  result add: aValue .
] .
^ result

]

{ #category : 'Private' }
RcKeyValueDictionary >> _logAt: aKey put: aValue oldValue: oldValue inCollisionBucket: aCollisionBkt [

"Create a log entry for adding the given key and value.  The collision bucket
 is the object on which a conflict may occur."

| logEntry |
logEntry := LogEntry new.
logEntry receiver: self selector: #_at:put:oldValue:logging: argArray: { aKey . aValue . oldValue . false }.
"GsFile gciLogServer:'Rc  _at:put: to bucket ', aCollisionBkt asOop asString ,
  ' tableSize ', self tableSize asString." 
System redoLog addLogEntry: logEntry for: self forConflictObject: aCollisionBkt

]

{ #category : 'Private' }
RcKeyValueDictionary >> _logRemoveKey: aKey inCollisionBucket: aCollisionBkt [
  "Create a log entry for removing the given key.  The collision bucket is the
   object on which a conflict may occur."
  | logEntry |
  logEntry := LogEntry new.
  logEntry receiver: self selector: #_replayRemoveKey:  argArray: { aKey }.
  System redoLog addLogEntry: logEntry for: self forConflictObject: aCollisionBkt
]

{ #category : 'Private' }
RcKeyValueDictionary >> _nodesObjectSecurityPolicy: anObjectSecurityPolicy [

"Assigns any collision buckets to the given anObjectSecurityPolicy."

self _doCollisionBuckets: [ :cb | cb objectSecurityPolicy: anObjectSecurityPolicy ]

]

{ #category : 'Hashing' }
RcKeyValueDictionary >> _rebuildTable: newSize oldSize: oldSizeArg logging: loggingBool [

"Rebuilds the hash table by saving the current state, initializing and changing
 the size of the table, and adding the key value pairs saved back to the hash
 dictionary.  This method is intended to be used internally.  If it is invoked
 directly by an application, concurrency conflicts may result."

| origSize saveArray thisKey collBkt securPolicy redoLog bktClass systm "keyCnt" |
(newSize <= 0) ifTrue: [
  newSize _error: #rtErrArgNotPositive  .
  ^ self
].

origSize := self _basicSize . 
loggingBool ifFalse:[ oldSizeArg ~~ origSize ifTrue:[ ^ true "skip replay"] ].
  
securPolicy := self objectSecurityPolicy  .

saveArray := Array new: origSize.

" save all old buckets in an Array and create new ones."
bktClass := self _bucketClass .
systm := System .
1 to: origSize do: [ :i | | oldBkt |
  loggingBool ifTrue:[
    oldBkt := self _rcAt: i .    "add nodes of self to rcReadSet"
    systm _addRootObjectToRcReadSet: oldBkt .  "add old bucket to rcReadSet"
  ] ifFalse:[
    oldBkt := self _at: i .
  ].
  saveArray at: i put: oldBkt .

  collBkt := bktClass new .
  collBkt objectSecurityPolicy: securPolicy .
  collBkt keyValueDictionary: self.
  self _at: i put: collBkt.
].
super _basicSize: newSize.
newSize > origSize ifTrue: [ " growing the RcKeyValueDictionary "
    origSize + 1 to: newSize do: [ :i |
      collBkt := self _bucketClass new.
      collBkt objectSecurityPolicy: securPolicy .
      collBkt keyValueDictionary: self.
      self _at: i put: collBkt.
    ]
].

collisionLimitPerBucket := newSize // 4.

redoLog := System redoLog.
"keyCnt := 0."
1 to: origSize do: [ :i |
  collBkt := saveArray _at: i.
  loggingBool ifTrue: [ redoLog addConflictObject: collBkt for: self ].
  1 to: collBkt _basicSize by: 2 do: [ :j |
    thisKey := collBkt _at: j. "make sure collBkt is in rc read set"
    thisKey ifNotNil:[
        self _at: thisKey put: (collBkt _at: j+1) oldValue: nil logging: false 
    ].
    "keyCnt := keyCnt + 1 ."
  ].
  collBkt ifNotNil:[ collBkt keyValueDictionary: nil].  "dereference parent, writes old bucket"
].
"GsFile gciLogServer:'rebuildTable keyCnt ', keyCnt asString, ' replay ', loggingBool not asString. "

loggingBool ifTrue: [ | logEntry |
	logEntry := LogEntry new.
	logEntry receiver: self selector: #_rebuildTable:oldSize:logging:  argArray: { newSize . origSize. false }.
	redoLog addFirstLogEntry: logEntry for: self forConflictObject: self.
  "GsFile gciLogServer:'Rc  _rebuildTable '."
].
^true

]

{ #category : 'Removing' }
RcKeyValueDictionary >> _replayRemoveKey: aKey [
  "If a key/value pair with key aKey is present, the key/value pair is removed
   from the receiver and this method returns true.  If it is not present, return
   false. The argument aBoolean determines if an entry is placed in the redo
   log for this operation."
  | hash collisionBkt |
  hash := self hashFunction: aKey.
  collisionBkt := self _at: hash.
  (collisionBkt _replayRemoveKey: aKey) ifFalse:[ ^ false ].
  ^ true
]

{ #category : 'Private' }
RcKeyValueDictionary >> _resolveRcConflictsWith: conflictObjects [
  "A logical write-write conflict has occurred on the receiver.  The objects that
   had the actual physical write-write conflicts are in the conflictObjects
   Array.  Selectively abort the receiver and then attempt to replay the
   operations maintained in the System redo log.  Returns true if all the
   operations could be replayed; otherwise returns false."
  | redoLog logEntries numEnt |
  (redoLog := System _redoLog) ifNil:[^ false ].
  (logEntries := redoLog getLogEntriesFor: self) ifNil:[ ^ false ].
  self _hasDependencyList ifTrue: [ ^ false ].

" fix 33731, explicitly abort buckets that have conflicts to handle rebuild
  in case old bucket is no longer referenced from the parent."
1 to: conflictObjects size do:[ :j | (conflictObjects at: j) _selectiveAbort ].

"abort all of the nodes involved in the transaction"
redoLog conflictObjects keysAndValuesDo:[ :conflictObject :redoObject |
  redoObject == self ifTrue: [ conflictObject _selectiveAbort ].
].

self _selectiveAbort .

" replay all operations on the receiver"
1 to: (numEnt := logEntries size) do:[:j | | logEnt |
  logEnt := logEntries at: j .
  logEnt redo ifFalse:[
    ^ false  "redo failed, cannot commit"
  ].
].
^ true
]

{ #category : 'Private' }
RcKeyValueDictionary >> _validateRcConflictsWith: conflictObjects [

"If the only conflict on this object is for the root, then we return true.
 Otherwise return false (to fail the transaction) if the redo log contains
 a removeKey: operation since we cannot guarantee success if the transaction
 did a removal from the RcKeyValueDictionary."

(self _checkLogEntriesForNoSelector: #_at:put:oldValue:logging: )
  ifFalse: [ ^ false ].

^ self _checkLogEntriesForNoSelector: #_removeKey:logging:

]

{ #category : 'Comparing' }
RcKeyValueDictionary >> = aKeyValueDictionary [

"Compares two RcKeyValueDictionaries for equality.  Returns true if all of
 the following conditions are true:

 1.  the receiver and aKeyValueDictionary are of the same class,
 2.  the two RcKeyValueDictionaries have the same number of elements
 3.  all of the keys in one dictionary return the same value in
     both dictionaries.

 Returns false otherwise."

| sentinel |
self == aKeyValueDictionary ifTrue:[ ^ true ].
( self class == aKeyValueDictionary class )
    ifFalse: [ ^ false ].

( self size == aKeyValueDictionary size )
    ifFalse: [ ^ false ].

sentinel := Object new .
self keysDo: [ :aKey |
    (self at: aKey)  == (aKeyValueDictionary at: aKey otherwise: sentinel)
        ifFalse: [ ^ false ]
].
^ true

]

{ #category : 'Enumerating' }
RcKeyValueDictionary >> accompaniedBy: anObj keysAndValuesDo: aBlock [

"Evaluates aBlock with each of the receiver's key/value pairs as the
 2nd and 3rd arguments.
 aBlock must be a 3 argument block, with arguments anObj, key value ."

self _doCollisionBuckets: [ :collisionBkt |
    1 to: collisionBkt tableSize do: [ :j | | aKey |
        aKey := collisionBkt keyAt: j.
        nil == aKey
            ifFalse: [ | aValue |
                aValue := collisionBkt valueAt: j.
                aBlock value: anObj value: aKey value: aValue.
            ]
    ]
].

]

{ #category : 'Enumerating' }
RcKeyValueDictionary >> associationsDo: aBlock [

"Iteratively evaluates the one argument block, aBlock, using an Association
 created by using each key-value pair successively.  Returns the receiver."

self keysAndValuesDo: [:aKey :aValue |
   aBlock value: (Association newWithKey: aKey value: aValue)
].

]

{ #category : 'Accessing' }
RcKeyValueDictionary >> at: aKey ifAbsent: aBlock [

"Returns the value that corresponds to aKey.  If no such key/value pair exists,
 returns the result of evaluating the zero-argument block aBlock."

| hash |
aKey ifNil:[ ^ aBlock value ].
hash := self hashFunction: aKey.
^ (self _at: hash) at: aKey ifAbsent: aBlock

]

{ #category : 'Accessing' }
RcKeyValueDictionary >> at: aKey otherwise: aValue [

"Returns the value that corresponds to aKey.  If no such key/value pair exists,
 returns the given alternate value."

| hash |
aKey ifNil: [ ^ aValue ].
hash := self hashFunction: aKey.
^ (self _at: hash) at: aKey otherwise: aValue

]

{ #category : 'Updating' }
RcKeyValueDictionary >> at: aKey put: aValue [

"Adds the key-value pair to the receiver. If the receiver already contains
 a key-value pair with the given key, this method makes aValue the value of that
 Association or key-value pair.  Returns aValue."

self _at: aKey put: aValue oldValue: nil logging: true.
^ aValue

]

{ #category : 'Private' }
RcKeyValueDictionary >> atHash: hashIndex put: aCollisionBucket [

"Updates the hash table by storing aKey at the specified hashIndex.  This
 method is intended to be used internally.  If it is invoked directly by an
 application, concurrency conflicts may result (as well as destroying the
 consistency of the dictionary)."
| sz |
(hashIndex <= 0 or:[ hashIndex > (sz := self _basicSize)])
  ifTrue: [ hashIndex _error: #rtErrArgOutOfRange args:{ 1 . sz }  .
            self _uncontinuableError
          ].
self basicAt: hashIndex put: aCollisionBucket

]

{ #category : 'Storing and Loading' }
RcKeyValueDictionary >> basicWriteTo: passiveObj [

"Converts the receiver to its passive form and writes that information on
 passiveObj."

| s cls c |
  cls := self class.
  passiveObj writeClass: cls.

  passiveObj writeSize: (s := self size) .

  passiveObj writeNamedIvsFrom: self class: cls .
  passiveObj endNamedInstVars.

  c := 0.
  self keysAndValuesDo:[ :aKey :aValue |
    passiveObj writeObject: aKey; writeObject: aValue .
    c := c + 1.
    c > 49 ifTrue: [
      passiveObj lf.
      c := 0.
      ].
    ].

  passiveObj cr

]

{ #category : 'Clustering' }
RcKeyValueDictionary >> clusterDepthFirst [

"This method clusters the receiver and its named instance variables and the
 key/value pairs in depth-first order.  Returns true if the receiver has
 already been clustered during the current transaction; returns false
 otherwise."

self cluster
  ifTrue: [ ^ true ]
  ifFalse: [
    " cluster collision buckets "
    1 to: self _basicSize do: [ :i | (self basicAt: i) cluster ].

    " cluster instance variables "
    1 to: self namedSize do: [ :i | (self instVarAt: i) clusterDepthFirst ].

    " cluster keys and values "
    self keysAndValuesDo: [ :key :value |
      key clusterDepthFirst.
      value clusterDepthFirst
    ].
    ^ false
  ].

]

{ #category : 'Copying' }
RcKeyValueDictionary >> copy [

^ super copy

]

{ #category : 'Enumerating' }
RcKeyValueDictionary >> do: aBlock [

"Iteratively evaluates the one argument block aBlock, using the value
 of each Association as the argument of the block."

self keysAndValuesDo: [ :key :value | aBlock value: value ].

]

{ #category : 'Comparing' }
RcKeyValueDictionary >> hash [

"Since RcKeyValueDictionary | = is based on identity of elements, hash is based
 on identityHash of elements.  Returns the numerical hash value."

| hashValue |
hashValue := 97633.
self keysDo: [ :aKey |
    hashValue := hashValue bitXor: aKey hash
].
^ hashValue

]

{ #category : 'Hashing' }
RcKeyValueDictionary >> hashFunction: aKey [

"The hash function should perform some operation on the value of the key
 aKey which returns a value in the range 1..self _basicSize."

^ (aKey hash \\ self _basicSize) + 1

]

{ #category : 'Initializing' }
RcKeyValueDictionary >> initialize: itsSize [

"Initializes the instance variables of the receiver to be an empty
 RcKeyValueDictionary of the specified size.  This is intended to be used
 internally at the time of instance creation.  If used on an existing,
 populated RcKeyValueDictionary, concurrency conflicts may result."

"Returns the receiver."

| theSize bucket |
theSize := itsSize .
(theSize <= 0)
    ifTrue: [ theSize _error: #rtErrArgNotPositive  .
              theSize := 1
    ].
collisionLimitPerBucket := theSize // 4.
self _basicSize: theSize.
1 to: theSize do: [ :i |
    bucket := self _bucketClass new.
    bucket objectSecurityPolicy: self objectSecurityPolicy .
    bucket keyValueDictionary: self.
    self basicAt: i put: bucket.
].
System rcValueCacheIfPresent: #size put: 0 for: self 

]

{ #category : 'Accessing' }
RcKeyValueDictionary >> keyAtValue: anObject ifAbsent: aBlock [

"Returns the key of the first value equal to the given object, anObject.
 If no match is found, evaluates and returns the result of the block aBlock."

self keysAndValuesDo: [ :aKey :aValue |
  (anObject = aValue) ifTrue: [ ^ aKey ].
  ].
^ aBlock value.

]

{ #category : 'Accessing' }
RcKeyValueDictionary >> keys [

"Returns a Set containing the receiver's keys."

| keys |
keys := Set new.
self keysDo: [ :aKey | keys add: aKey ].
^ keys

]

{ #category : 'Enumerating' }
RcKeyValueDictionary >> keysAndValuesDo: aBlock [

"Iteratively evaluates the two argument block, aBlock, using each key and value
 of the receiver as the argument to the block. Returns the receiver."

self _doCollisionBuckets: [ :collisionBkt |
    1 to: collisionBkt tableSize do: [ :j | | aKey |
        aKey := collisionBkt keyAt: j.
        nil == aKey
            ifFalse: [ | aValue |
                aValue := collisionBkt valueAt: j.
                aBlock value: aKey value: aValue.
            ]
    ]
].


]

{ #category : 'Enumerating' }
RcKeyValueDictionary >> keysDo: aBlock [

"Iteratively evaluates the two argument block aBlock, using each key and
 value of the receiver as the argument to the block."

self _doCollisionBuckets: [ :collisionBkt | collisionBkt keysDo: aBlock ]

]

{ #category : 'Storing and Loading' }
RcKeyValueDictionary >> loadVaryingFrom: passiveObj size: varyingSize [

"Reads the varying part of the receiver from the given passive object.
 Does not record the receiver as having been read.  Does not read the
 receiver's named instance variables, if any."

| oldBkt newBkt |

passiveObj version >= 500 ifTrue:[
  1 to: varyingSize do: [:i |
    self at: passiveObj readObject"aKey" put: passiveObj readObject"aValue"
    ].
  ]
ifFalse:[
  1 to: varyingSize do: [:i |
    oldBkt := passiveObj readObject .
    (oldBkt isKindOf: RcCollisionBucket)
       ifTrue:[ newBkt:= oldBkt ]
       ifFalse:[
         (oldBkt isKindOf: (ObsoleteClasses at: #ObsoleteRcCollisionBucket)) ifFalse:[
            self _halt:'invalid kind of RcCollisionBucket'.
            ].
         newBkt := self _bucketClass new .
         newBkt keyValueDictionary: self .
         oldBkt keysAndValuesDo:[:aKey :aValue | newBkt at: aKey put: aValue ] .
         oldBkt _removeAll .
         ].
    self _basicAt: i put: newBkt
    ].
  ].

]

{ #category : 'Accessing' }
RcKeyValueDictionary >> numElements [

"Same as size."

^ self size

]

{ #category : 'Copying' }
RcKeyValueDictionary >> postCopy [

"Cleanup new copy."

super postCopy.
1 to: self _basicSize do: [ :index |
    self atHash: index put: (self basicAt: index) copy
].

]

{ #category : 'Hashing' }
RcKeyValueDictionary >> rebuildTable: newSize [

"Rebuilds the hash table by saving the current state, initializing and changing
 the size of the table, and adding the key value pairs saved back to the hash
 dictionary.  This method is intended to be used internally.  If it is invoked
 directly by an application, concurrency conflicts may result."

self _rebuildTable: newSize oldSize: self _basicSize logging: true

]

{ #category : 'Hashing' }
RcKeyValueDictionary >> rehash [
	"Re-establish any hash invariants of the receiver."

	self rebuildTable: self tableSize

]

{ #category : 'Removing' }
RcKeyValueDictionary >> removeKey: aKey ifAbsent: aBlock [
  "Removes the key/value pair with key aKey from the receiver and returns the
   value.  If no key/value pair is present with key aKey, evaluates the
   zero-argument block aBlock and returns the result of that evaluation."
  | hash collisionBkt aValue systm |
  hash := self hashFunction: aKey.
  collisionBkt := self _at: hash.
  aValue := collisionBkt removeKey: aKey ifAbsent: [ ^ aBlock value ].

  self _logRemoveKey: aKey inCollisionBucket: collisionBkt. "adds bucket to rcReadSet"
  systm := System .
  systm _addRootObjectToRcReadSet: self .  "add self to rcReadSet, in case of rebuildTable"
  systm rcValueCacheIncrement: #size for: self by: -1 .
  ^ aValue
]

{ #category : 'Accessing' }
RcKeyValueDictionary >> size [

"Returns the number of key/value pairs in the receiver."

| sz |
sz := System rcValueCacheAt: #size for: self otherwise: nil.
sz ifNil:[ sz := self _calculateSize ].
^ sz

]

{ #category : 'Class Membership' }
RcKeyValueDictionary >> species [

"Returns a class, an instance of which should be used as the result of
 select: or other projections applied to the receiver."

^ KeyValueDictionary

]

{ #category : 'Statistics' }
RcKeyValueDictionary >> statistics [

"Returns a Dictionary containing statistics that can be useful in determining
 the performance of a key-value dictionary, including the following information:

 * TotalCollisionBuckets: The number of collision buckets required to
                          implement the key-value dictionary.

 * AvgPairsPerBucket:     The average number of key/value pairs in each bucket.

 * LargestBucket:         The bucket having the most key/value pairs.  This
                          bucket contains the most keys for which the hash
                          function did not provide a good distribution over the
                          range of values in the table.

 Since RcKeyValueDictionaries are implemented differently than
 KeyValueDictionaries, the statistics are calculated as if the implementation
 were the same as KeyValueDictionaries.  To be specific, RcKeyValueDictionaries
 are implemented with all collision buckets.  Therefore, the calculations don't
 count CollisionBuckets with a size of 1 as a collision."

| bktCount pairCount maxBkt maxBktSize result |
bktCount := 0.
pairCount :=0.
maxBktSize := 0.
self _doCollisionBuckets: [ :collisionBkt |
    " if bucket size = 1, this would not be a collision in an ordinary key-value dictionary "
    collisionBkt size > 1
        ifTrue: [
            bktCount := bktCount + 1.
            pairCount := pairCount + collisionBkt size - 1.
            (collisionBkt size > maxBktSize)
                ifTrue: [
                    maxBktSize := collisionBkt size.
                    maxBkt := collisionBkt
                ]
        ]
].

result := SymbolDictionary new.
result at: #TotalCollisionBuckets put: bktCount.
(bktCount ~~ 0)
    ifTrue: [ result at: #AvgPairsPerBucket put: (pairCount / bktCount) asFloat ]
    ifFalse: [ result at: #AvgPairsPerBucket put: 0].
result at: #LargestBucket put: maxBkt.
^ result

]

{ #category : 'Private' }
RcKeyValueDictionary >> tableSize [

"Returns the size of hash table used for storing the entries."

^ self _basicSize

]

{ #category : 'Initializing' }
RcKeyValueDictionary >> tableSize: aSize [

"Sets the size of the receiver to a new value.  Each collision bucket is reset
 to contain no elements.  This method is intended to be used internally.  If it
 is invoked directly by an application, concurrency conflicts may result."

| origSize collBkt securPolicy  |
(aSize <= 0) ifTrue: [
  aSize _error: #rtErrArgNotPositive  .
  ^ self
].

origSize := self _basicSize.
securPolicy  := self objectSecurityPolicy .
" reset each collision bucket to be empty "
1 to: origSize do: [ :i | (self basicAt: i) reset ].

super _basicSize: aSize.

aSize > origSize
    ifTrue: [ " growing the RcKeyValueDictionary "
         (origSize + 1) to: aSize do: [ :i |
            collBkt := self _bucketClass new.
            collBkt objectSecurityPolicy: securPolicy .
            collBkt keyValueDictionary: self.
            self basicAt: i put: collBkt.
        ].
    ].

collisionLimitPerBucket := aSize // 4.

]

{ #category : 'Accessing' }
RcKeyValueDictionary >> values [

"Returns an OrderedCollection containing the receiver's values."

| values |
values := OrderedCollection new.
self valuesDo: [ :aValue | values add: aValue ].
^ values

]

{ #category : 'Enumerating' }
RcKeyValueDictionary >> valuesDo: aBlock [

"Iteratively evaluates the one argument block aBlock, using the value
 of each Association as the argument of the block."

self do: aBlock

]
