Extension { #name : 'RcIdentityBag' }

{ #category : 'Instance Creation' }
RcIdentityBag class >> new [

"Returns a new RcIdentityBag that can handle
 at least 10 user sessions, plus the default system gems (1 admin gc,
 1 reclaim gc, 1 page manager, 1 symbol creation gem) ,
 plus the global components."

^ self new: 14

]

{ #category : 'Instance Creation' }
RcIdentityBag class >> new: maxSessionId [

"Returns a new RcIdentityBag with a size that supports the specified maximum sessionId.
 Maximum number of concurrent sessions is approximately maxSessionId - 4 ,
 assumming no session has sessoinId > maxSessionId .
 The new RcIdentityBag will handle more users, but will only have
 subcomponents initially created up to maxSessionId."

 | newOne |
  newOne := self basicNew initialize: (maxSessionId + 1) * 2.
  ^ newOne initializeComponents

]

{ #category : 'Private' }
RcIdentityBag >> _abortAndReplay: conflictObjects [
  Error signal:'should not be in RcIdentityBag >> _abortAndReplay: '.  
]

{ #category : 'Private' }
RcIdentityBag >> _asCollectionForSorting [

"Returns an IdentityBag that can be used for sorting."

^ self _asIdentityBag

]

{ #category : 'Converting' }
RcIdentityBag >> _asIdentityBag [

"Returns a Bag that consists of objects in this RcIdentityBag.  First check the
 RC value cache and if not there, then create the Bag.  Note that each
 invocation of this message returns the same Bag."
 | bag |
 (bag := System rcValueCacheAt: #asBag for: self otherwise: nil) ifNil:[
    bag := self _createAsBag 
 ].
 ^ bag

]

{ #category : 'Private' }
RcIdentityBag >> _at: offset [

^ self at: offset

]

{ #category : 'Accessing' }
RcIdentityBag >> _basicSize [

"Returns the logical size of the receiver."

^ self size

]

{ #category : 'Accessing' }
RcIdentityBag >> _calculateSize [

"Calculate the number of elements contained in the RcIdentityBag and place it in
 the RC value cache.  Returns the size."

| sz addBag removeBag |
sz := 0.
1 to: components size by: 2 do: [ :i |
    (addBag := components at: i) ifNotNil:[
            removeBag := components at: i + 1.
            sz := sz + addBag size - removeBag size.
    ]
].
System rcValueCacheAt: #size put: sz for: self.
^ sz

]

{ #category : 'Private' }
RcIdentityBag >> _componentsIndex [

"Returns the index of the current session addition Bag."

^ self _idxForSessionId: System session

]

{ #category : 'Converting' }
RcIdentityBag >> _createAsBag [

"Creates a Bag that consists of objects that have been added minus the objects
 that have been removed, and place it in the RC value cache.  Returns the Bag."

| aBag |
aBag := self species new.
self _doSessionBags: [ :addBag :removeBag |
    removeBag size == 0 
       ifTrue:[ aBag addAll: addBag ]
      ifFalse:[ aBag addAll: (addBag _primDifference: removeBag) ].
].
System rcValueCacheAt: #asBag put: aBag for: self.
^ aBag


]

{ #category : 'Private' }
RcIdentityBag >> _createComponentBagsFor: addIndex [

"Create the additionBag and removalBag for the session whose components begin
 at the given addIndex.  Returns the additionBag."

| addBag remBag securityPolicy |

addBag := IdentityBag new.
addBag objectSecurityPolicy: (securityPolicy := self objectSecurityPolicy ) .
components at: addIndex put: addBag.

remBag := IdentityBag new.
remBag objectSecurityPolicy: securityPolicy .
components at: addIndex + 1 put: remBag.

^ addBag

]

{ #category : 'Private' }
RcIdentityBag >> _deferredGciUpdateWith: valueArray [

"Private."

components ifNil:[ self initialize: 10 ].
1 to: valueArray size do:[:j |
  self add: (valueArray at: j)
  ].

]

{ #category : 'Private' }
RcIdentityBag >> _doRemove: anObject removalBag: removalBag logging: aBoolean [

  "Adds anObject to the given removal Bag, performing any logging and updating
    the caches if necessary."
  aBoolean ifTrue:[ | systm |
    _indexedPaths ifNotNil:[
       self _updateIndexesForRemovalOf: anObject.  "update indexes if not in replay"
    ].
    self _logRemovalOf: anObject inRemovalBag: removalBag .
    removalBag _rcAdd: anObject withOccurrences: 1.
    ((systm := System) rcValueCacheFor: self) ifNotNil:[:valueArray |
      (systm _rcv: valueArray at: #asBag) ifNotNil:[:bag |
        bag remove: anObject ifAbsent: [ System clearRcValueCache ] 
      ].
      systm _rcv: valueArray increment: #size by:  -1  .
    ]
  ] ifFalse:[ "in replay" 
    removalBag _rcAdd: anObject withOccurrences: 1.
    "no rcValueCache update during replay"
  ].
]

{ #category : 'Private' }
RcIdentityBag >> _doSessionBags: aBlock [

"Enumerate over all session components, executing aBlock with each
 session's addition Bag and removal Bag as arguments."

| addBag |
1 to: components size by: 2 do: [ :i |
    (addBag := components at: i) 
        ifNotNil: [ aBlock value: addBag value: (components at: i + 1) ]
]

]

{ #category : 'Searching' }
RcIdentityBag >> _find: anObject [
  "Returns a String report."
  | str |
  str := String new .
  1 to: components size by: 2 do:[:j | | addBag removeBag cnt sessId |
    addBag := components atOrNil: j .
    removeBag := components atOrNil: j + 1 .
    sessId := j // 2 .
    addBag ifNotNil:[
      (cnt := addBag occurrencesOf: anObject) ~~ 0 ifTrue:[
         str add: 'session ', sessId asString, ' additions ', cnt asString ; lf .
      ].
    ]. 
    removeBag ifNotNil:[
      (cnt := removeBag occurrencesOf: anObject) ~~ 0 ifTrue:[
         str add: 'session ', sessId asString, ' removals ', cnt asString ; lf .
      ].
    ].
  ].
  ^ str
]

{ #category : 'Private' }
RcIdentityBag >> _gbsTraversalCallback [
  "Needed for initial mapping case. See bug 31058."
  ^'RcIdentityBag>>_gbsTraversalCallback: This string should be ignored on the client.'

]

{ #category : 'Private' }
RcIdentityBag >> _gciInitialize [
 components ifNil:[ self _initializeForCurrentSession ].
 ^ true
]

{ #category : 'Support' }
RcIdentityBag >> _getInactiveSessionIndexes [

"Returns a set of integer indexes into the RcIdentityBag session components for
 sessions that are not active.  The current session nor the global session
 components are considered active."

| allSessions compSize |
compSize := components size.
allSessions := IdentitySet new.
3 to: compSize by: 2 do: [ :i | allSessions add: i ].

" get the current sessions and calculate the index "
System currentSessions do: [ :id | | index |
    index := self _idxForSessionId: id.
    (index <= compSize) ifTrue: [ allSessions remove: index ]
].

^ allSessions

]

{ #category : 'Private' }
RcIdentityBag >> _globalAddBag [

"Returns the global addition Bag for the receiver"

 ^ (components at: 1) ifNil:[ Error signal:'first component missing'].
]

{ #category : 'Private' }
RcIdentityBag >> _idxForSessionId: id [

 "Returns the index of the given session ID's addition Bag.
  components at: 1  is the global additions bag"
  ^ (id + id) + 1  
]

{ #category : 'Private' }
RcIdentityBag >> _initializeForCurrentSession [
  | ofs |
  ofs := self _idxForSessionId: System session .
  components ifNil:[ 
    components := Array new: ofs 
  ] ifNotNil:[ 
    components size >= ofs ifTrue:[ ^ self "no changes"]
                          ifFalse:[ components size: ofs ].
  ].
  self initializeComponents .
]

{ #category : 'Testing' }
RcIdentityBag >> _isRcIdentityBag [

"Returns true; the receiver is an RcIdentityBag."

^ true

]

{ #category : 'Locking Support' }
RcIdentityBag >> _lockableValues [

"Returns a kind of object usable as an argument to _lockAll: primitives."

^ self _asIdentityBag

]

{ #category : 'Private' }
RcIdentityBag >> _logAddAllOf: aCollection [
 "Creates a log entry for adding all of the given collection."
  | logEntry |
  logEntry := LogEntry new.
  logEntry receiver: self selector: #addAll:logging: argArray: { aCollection . false }.
  System redoLog addLogEntry: logEntry
]

{ #category : 'Private' }
RcIdentityBag >> _logAddAllOf: aCollection conflictObj: obj1 conflictObj: obj2 [
 "Creates a log entry for adding all of the given collection."
  | logEntry |
  logEntry := LogEntry new.
  logEntry receiver: self selector: #addAll:logging: argArray: { aCollection . false }.
  System redoLog addLogEntry: logEntry for: self conflictObject: obj1 conflictObject: obj2 
]

{ #category : 'Private' }
RcIdentityBag >> _logAdditionOf: anObject [
  "Create a log entry for adding the given object."
  | logEntry |
  (logEntry := LogEntry new)
     receiver: self selector: #add:logging: argArray: { anObject . false }.
  System redoLog addLogEntry: logEntry
]

{ #category : 'Private' }
RcIdentityBag >> _logAdditionOf: anObject conflictObj: obj1 conflictObj: obj2 [
  "Create a log entry for adding the given object."
  | logEntry |
  (logEntry := LogEntry new)
     receiver: self selector: #add:logging: argArray: { anObject . false }.
  System redoLog addLogEntry: logEntry for: self conflictObject: obj1 conflictObject: obj2 .
]

{ #category : 'Private' }
RcIdentityBag >> _logRemovalOf: anObject inRemovalBag: aRemovalBag [

  "Creates a log entry for removing the given object.  The removal Bag is the
   object on which a conflict may occur.  Returns the receiver."
  | logEntry |
  logEntry := LogEntry new.
  logEntry receiver: self selector: #'_privateRemove:logging:' argArray: { anObject . false }.
    "selective abort on an nsc takes care of all written nodes"
    "caller of _doRemove:  adds additionsBag as a conflict object if needed"
  System redoLog addLogEntry: logEntry for: self forConflictObject: aRemovalBag
]

{ #category : 'Private' }
RcIdentityBag >> _logRemoveAllOf: aCollection [
  | logEntry |
  logEntry := LogEntry new.
  logEntry receiver: self selector: #_removeAll:logging: argArray: { aCollection . false }.
  System redoLog addLogEntry: logEntry .
]

{ #category : 'Private' }
RcIdentityBag >> _privateRemove: anObject logging: aBoolean [

  "Returns true if the object was removed; false if the object was not present.
   If in replay (aBoolean == false) do not update rcvalue cache"
  | addIdx |
  anObject ifNil:[ ^ false ].
  addIdx := self _componentsIndex .
  (components atOrNil: addIdx) ifNotNil:[:additionBag | | removalBag |
     removalBag := components at: addIdx + 1 .
     " first see if the object is in the current session's addition Bag "
     ((additionBag includes: anObject) and:[
       (additionBag occurrencesOf: anObject) > (removalBag occurrencesOf: anObject)]) ifTrue:[
         "GsFile gciLogServer:'_privateRemove A removed ', anObject asString ."
         self _doRemove: anObject removalBag: removalBag logging: aBoolean  .
         self _rcProcessRemovalBagFor: addIdx logging: aBoolean. "process removals on addition bag"
         ^ true
      ].
   ].
   self _doSessionBags: [ :addBag :removeBag | " check other sessions' addition Bags "
      ((addBag includes: anObject) 
           and:[ (addBag occurrencesOf: anObject) > (removeBag occurrencesOf: anObject) ]) ifTrue:[
             " more have been added than removed ... "
             "GsFile gciLogServer:'_privateRemove B removed ', anObject asString ."
             self _doRemove: anObject removalBag: removeBag logging: aBoolean .
             aBoolean ifTrue:[ System redoLog addConflictObject: addBag for: self ].
             ^ true 
        ] 
   ].
   aBoolean ifFalse: [ "retry failure"
      System rcValueCacheAt: #'Rc-Retry-Failure-Reason' put: #'objErrNotInColl' for: System;
          rcValueCacheAt: #'Rc-Retry-Failure-Description'
          put: 'The object [' , anObject asOop printString, '] has already been removed from the collection [', 
            self asOop printString, ']'
        for: System 
    ].
   "GsFile gciLogServer:'_privateRemove not found: ', anObject asString ."
   ^ false "not found"
]

{ #category : 'Private' }
RcIdentityBag >> _processRemovalBagFor: sessionId [

"Apply any removals that were performed on the removal Bag of the given
 session.  Returns the receiver. Not Rc"

| additionBag removalBag |
(additionBag := components at: sessionId) ifNotNil:[
  removalBag := components at: sessionId + 1.
  removalBag size == 0 ifTrue: [
    ^ self
  ].
  additionBag _removeAll: removalBag errIfAbsent: false forReplay: false rc: false .
  removalBag _removeAll: removalBag errIfAbsent: false forReplay: false rc: false .
]
]

{ #category : 'Private' }
RcIdentityBag >> _rcCreateComponentBagsFor: addIndex logging: aBoolean [
  | oldSize |
  oldSize := components size .
  (oldSize < (addIndex + 1)) ifTrue:[
    aBoolean ifTrue:[
      components _rcAt: oldSize  .  " add components right path to rcReadSet"
      System redoLog addConflictObject: components for: self.
    ].
    components size: addIndex + 1 .
    "GsFile gciLogServer: DateAndTime now asStringMs, '  grew components oop ', components asOop asString, 
                 ' new size ', components size asString . "
    oldSize + 1 to: addIndex by: 2 do:[:j |
       self _createComponentBagsFor: j 
    ].
  ] ifFalse:[
    aBoolean ifTrue:[
      components _rcAt: addIndex ; _rcAt: addIndex + 1 .
      System redoLog addConflictObject: components for: self.
    ].
    self _createComponentBagsFor: addIndex .
  ].
  ^ components at: addIndex .
]

{ #category : 'Private' }
RcIdentityBag >> _rcProcessRemovalBagFor: sessionId logging: aBoolean [

"Apply any removals that were performed on the removal Bag of the given
 session.  Returns the receiver."

  | additionBag removalBag |
  removalBag := components at: sessionId + 1.
  removalBag size == 0 ifTrue: [
    ^ self
  ].
  additionBag := components at: sessionId .
  aBoolean ifTrue:[ | systm redoLog |
    (systm := System) _addEntireObjectToRcReadSet: additionBag.
    systm _addEntireObjectToRcReadSet: removalBag.

    (redoLog := systm redoLog) addConflictObject: removalBag for: self.
    redoLog addConflictObject: additionBag for: self.
  ].
  additionBag _removeAll: removalBag errIfAbsent: false forReplay: aBoolean not rc: true .
  removalBag _removeAll: removalBag errIfAbsent: false forReplay: aBoolean not rc: true .
]

{ #category : 'Private' }
RcIdentityBag >> _removeAll: aCollection logging: aBoolean [

"Removes one occurrence of each element of aCollection from the receiver and
 returns the receiver.  Generates an error if any element of aCollection is not
 present in the receiver."

  | array indexes |
  array := aCollection .
  array _isArray ifFalse:[ array := Array withAll: aCollection ].
  indexes := _indexedPaths .
  aBoolean ifTrue:[ | systm argSize |
    1 to: (argSize := array size) do: [ :i | | anObj |
      " do not update rcValueCache in this loop"
      anObj := array at: i .
      indexes ifNotNil:[ self _updateIndexesForRemovalOf: anObj ].
      "addition and removal bags added to rcReadSet and as conflict objects via  _privateRemove"
      (self _privateRemove: anObj logging: false) ifFalse:[ ^ false ].
    ].
    self _logRemoveAllOf: array.
    ((systm := System) rcValueCacheFor: self ) ifNotNil:[:valueArray |
      (systm _rcv: valueArray at: #asBag ) ifNotNil:[:bag | bag removeAll: aCollection ].
      systm _rcv: valueArray increment: #size by: 0 - argSize .
    ].
  ] ifFalse:[
    1 to: array size do: [ :i | 
      " do not update rcValueCache nor indexes in this loop"
      "addition and removal bags added to rcReadSet and as conflict objects via  _privateRemove"
      (self _privateRemove: (array at: i) logging: false) ifFalse:[ ^ false ]
    ].
  ].
  ^ true
]

{ #category : 'Private' }
RcIdentityBag >> _removeAny: numberToRemove from: aBag [

"Remove any numberToRemove elements from the given Bag.  Returns the elements
 that are removed."

| removed |
removed := IdentityBag new.
1 to: numberToRemove do: [ :i | removed add: (aBag _at: i) ].
1 to: removed size do: [ :i | aBag remove: (removed at: i) ].
^ removed

]

{ #category : 'Private' }
RcIdentityBag >> _resolveRcConflictsWith: conflictObjects [

  "A logical write-write conflict has occurred on the receiver.  The objects that
   had the actual physical write-write conflicts are in the conflictObjects
   Array.  Returns whether the conflicts could be successfully resolved."
  | redoLog logEntries systm |
  (_indexedPaths ~~ nil and:[ _indexedPaths _anyIndexIsLegacy]) ifTrue:[
     "abort and replay not supported when legacy index is involved"
      System rcValueCacheAt: #'Rc-Retry-Failure-Reason' put: #'noReplayForLegacyIndex' for: System;
            rcValueCacheAt: #'Rc-Retry-Failure-Description'
            put: 'Rc replay not supported when legacy index is involved' for: System.
      ^false 
  ].
  self _hasDependencyList ifTrue: [
    systm rcValueCacheAt: #'Rc-Retry-Failure-Reason' put: #'objHadDependency' for: systm ;
        rcValueCacheAt: #'Rc-Retry-Failure-Description'
        put: 'The object [', self asOop printString, '] has a dependency list. Replay not allowed.'
        for: systm .
    ^ false 
  ].
  redoLog := (systm := System) _redoLog.
  redoLog ifNil:[ ^ false ].
  (logEntries := redoLog getLogEntriesFor: self) ifNil:[ ^ false ].
  1 to: conflictObjects size do:[ :j | 
     (conflictObjects at: j ) _selectiveAbort
  ].
  redoLog conflictObjects keysAndValuesDo:[ :conflictObject :redoObject |
    "handle additions or removals bag in redoLog conflict objs, but not having a conflict"
    redoObject == self ifTrue: [ conflictObject _selectiveAbort ].
  ].
  ^ redoLog _redoOperationsForEntries: logEntries
]

{ #category : 'Private' }
RcIdentityBag >> _selectiveAbort [

"In addition to aborting the RcIdentityBag, selectively abort the session
 components.  Returns the receiver."

  Error signal:'should not be in RcIdentityBag >> _selectiveAbort'.

]

{ #category : 'Support' }
RcIdentityBag >> _unsafeCleanupBag [

"Process the removalBags to clean up the receiver.
Unlike cleanupBag, this method handles the removalBags for all
sessions, not just inactive ones.
This method will cause concurrency conflicts and should only be used
while no sessions are adding to or removing from the RcIdentityBag .
Not Rc"

1 to: components size by: 2 do: [ :i |
  " apply any removals that have occurred for that session's elements "
  self _processRemovalBagFor: i .
]

]

{ #category : 'Private' }
RcIdentityBag >> _validateRcConflictsWith: conflictObjects [

"If the only conflict on this object is for the root, then we return true.
 Otherwise return false (to fail the transaction) if the redo log contains
 the selector _privateRemove:logging: since we cannot guarantee success
 if the transaction did a removal from the RcIdentityBag."

" if only had a physical conflict on the root "
( conflictObjects size == 1 and: [ (conflictObjects at: 1) == self ] )
  ifTrue: [ ^ true ].

^ self _checkLogEntriesForNoSelector: #_privateRemove:logging:

]

{ #category : 'Set Arithmetic' }
RcIdentityBag >> - anIdentityBag [

"Difference.  Returns an IdentityBag containing exactly those elements of the
 receiver that have a greater number of occurrences in the receiver than in the
 argument.  If an element occurs m times in the receiver and n times in the
 argument (where m >= n), then the result will contain m - n occurrences of that
 element."

(anIdentityBag _isRcIdentityBag or: [ anIdentityBag _isIdentityBag not ])
  ifTrue: [ ^ self asIdentityBag _primDifference: anIdentityBag _asIdentityBag ]
  ifFalse: [ ^ self asIdentityBag _primDifference: anIdentityBag ]

]

{ #category : 'Set Arithmetic' }
RcIdentityBag >> * anIdentityBag [

"Intersection.  Returns a kind of IdentityBag containing only the
 elements that are present in both the receiver and the argument.

 The class of the result is the class of the argument. If the argument is a
 kind of RcIdentityBag, the result is an IdentityBag.

 If the result is a kind of IdentitySet, then each element that occurs in both
 the receiver and the argument occurs exactly once in the result.  If the result
 is an IdentityBag and if an element occurs m times in the receiver and n
 times in the argument, then the result contains the lesser of m or n
 occurrences of that element."

(anIdentityBag _isRcIdentityBag or: [ anIdentityBag _isIdentityBag not ])
  ifTrue: [ ^ self _asIdentityBag _primIntersect: anIdentityBag _asIdentityBag ]
  ifFalse: [ ^ self _asIdentityBag _primIntersect: anIdentityBag ]

]

{ #category : 'Set Arithmetic' }
RcIdentityBag >> + anIdentityBag [

"Union.  Returns a kind of IdentityBag containing exactly the elements
 that are present in either the receiver or the argument.

 The class of the result is the class of the argument. If the argument is a
 kind of RcIdentityBag, the result is an IdentityBag.

 If the result is a kind of IdentitySet, then each element that occurs in
 either the receiver or the argument occurs exactly once in the result. If the
 result is a kind of IdentityBag, and if an element occurs m times in the
 receiver and n times in the argument, then the result contains m + n
 occurrences of that element."

(anIdentityBag _isRcIdentityBag or: [ anIdentityBag _isIdentityBag not ])
  ifTrue: [ ^ self _asIdentityBag _primUnion: anIdentityBag _asIdentityBag ]
  ifFalse: [ ^ self _asIdentityBag _primUnion: anIdentityBag ]

]

{ #category : 'Comparing' }
RcIdentityBag >> = anRcIdentityBag [

"Verifies that the receiver and anRcIdentityBag are of the same class, then uses
 the semantics of IdentityBag comparison for the elements in the RcIdentityBag."

self == anRcIdentityBag ifTrue:[ ^ true ].
anRcIdentityBag class == self class
    ifFalse: [ ^ false ].
^ self _asIdentityBag = anRcIdentityBag _asIdentityBag

]

{ #category : 'Adding' }
RcIdentityBag >> add: newObject [

"Add the object to the RcIdentityBag.  Returns newObject."

newObject == nil ifTrue:[ ^ self "ignore nils" ].
self add: newObject logging: true.
^ newObject.

]

{ #category : 'Adding' }
RcIdentityBag >> add: anObject logging: aBoolean [

"Adds anObject to the current session's addition Bag in the RcIdentityBag.  If
 aBoolean is true, logs the addition to the redo log. "

  | addIndex additionBag persistentAdditionBag |
  " update any indexes if necessary "
  anObject ifNil:[ ^ self ].
  (aBoolean and: [ _indexedPaths ~~ nil ]) ifTrue: [
      self _updateIndexesForAdditionOf: anObject logging: aBoolean.
  ].

  addIndex := self _componentsIndex .
  persistentAdditionBag := components atOrNil: addIndex.
  (additionBag := persistentAdditionBag) ifNil:[ 
     additionBag := self _rcCreateComponentBagsFor: addIndex logging: aBoolean 
  ].
  additionBag _rcAdd: anObject withOccurrences: 1 . "add additionBag to rcReadSet"
  "GsFile gciLogServer: 'RcIdentityBag ', self asOop asString, ' additionBag ', bag asOop asString ."

  self _rcProcessRemovalBagFor: addIndex logging: aBoolean.
  aBoolean ifTrue: [  | systm|
    persistentAdditionBag ifNotNil:[
      self _logAdditionOf: anObject conflictObj: persistentAdditionBag 
                                 conflictObj: (components at: addIndex + 1 ).
    ] ifNil:[
      self _logAdditionOf: anObject .
    ].
    ((systm := System) rcValueCacheFor: self) ifNotNil:[:valueArray |
      (systm _rcv: valueArray at: #asBag) ifNotNil:[:aBag | aBag add: anObject ].
      systm _rcv: valueArray increment: #size by: 1 .
    ].
  ].
  ^ true
]

{ #category : 'Adding' }
RcIdentityBag >> add: anObject withOccurrences: anInteger [

"Add the object to the RcIdentityBag the given number of times.  Returns the
 receiver."

anObject ifNil:[ ^ self "ignore nils" ].
1 to: anInteger do: [ :i | self add: anObject logging: true ]

]

{ #category : 'Adding' }
RcIdentityBag >> addAll: aCollection [

"Adds the collection of objects to the RcIdentityBag.
 Returns aCollection ."

| arr |
(arr := { } ) addAll: aCollection .
self addAll: arr logging: true.
^ aCollection

]

{ #category : 'Adding' }
RcIdentityBag >> addAll: aCollection logging: aBoolean [

"Adds the collection of objects to the current session's addition Bag.  If
 aBoolean is true, logs the addition to the redo log.
 Returns true if successful .
 Note that aCollection is always an Array copy of the original collection that
 was used in RcIdentityBag>>addAll: ."

  | additionBag persistentAdditionBag addIndex systm |
  (aBoolean and: [ _indexedPaths ~~ nil ]) ifTrue: [
    1 to: aCollection size do: [ :i |
      self _updateIndexesForAdditionOf: (aCollection _at: i) logging: aBoolean.
    ]
  ].
  addIndex := self _componentsIndex .
  persistentAdditionBag := components atOrNil: addIndex.
  (additionBag := persistentAdditionBag) ifNil:[
     additionBag := self _rcCreateComponentBagsFor: addIndex logging: aBoolean
  ].
  additionBag addAll: aCollection.
  (systm := System) _addEntireObjectToRcReadSet: additionBag.

  self _rcProcessRemovalBagFor: addIndex logging: aBoolean.
  aBoolean ifTrue:[ 
    persistentAdditionBag ifNil:[ 
      self _logAddAllOf: aCollection .
    ] ifNotNil:[
      self _logAddAllOf: aCollection conflictObj: persistentAdditionBag
                                 conflictObj: (components at: addIndex + 1 ).
    ].
    (systm rcValueCacheFor: self) ifNotNil:[:valueArray |
      (systm _rcv: valueArray at: #asBag) ifNotNil:[:aBag | aBag addAll: aCollection ].
      systm _rcv: valueArray increment: #size by: aCollection size .
    ].
  ].
  ^ true
]

{ #category : 'Converting' }
RcIdentityBag >> asArray [

"Returns an Array with the contents of the receiver."

^ Array withAll: self

]

{ #category : 'Converting' }
RcIdentityBag >> asIdentityBag [

"Returns a Bag that consists of objects in this RcIdentityBag.  Note that each
 invocation of this message returns a new Bag."

^ self _asIdentityBag copy

]

{ #category : 'Accessing' }
RcIdentityBag >> at: offset [

"Returns the element of the receiver that is currently located at logical
 position offset.

 The elements of an RcIdentityBag are inherently unordered, and can change
 position (offset) when the RcIdentityBag is altered.  Thus, after an
 RcIdentityBag is altered, a given element may reside at a different offset than
 before, and a given offset may house a different element.  You should not infer
 an ordering for an RcIdentityBag's elements when you access them by offset.

 This method is useful primarily as a code optimizer for iterating over all the
 elements of an RcIdentityBag (using a loop that runs the offset from 1 to the
 size of the RcIdentityBag).  The RcIdentityBag must not change during the
 iteration.  But the iteration may run faster than it would if you use other
 alternatives, such as the do: method."

| addBag remBag diffSz cumulativeSz prevSz currOffset obj |
cumulativeSz := 0.
1 to: components size by: 2 do: [ :i |
  (addBag := components at: i) ifNotNil:[
      remBag := components at: i + 1.
      diffSz := addBag size - remBag size.
      prevSz := cumulativeSz.
      cumulativeSz := cumulativeSz + diffSz.
      " see if this pair contains the object at the logical offset "
      cumulativeSz >= offset ifTrue: [
          currOffset := offset - prevSz.
          1 to: addBag size do: [ :j |
            obj := addBag at: j.
            (addBag occurrencesOf: obj) > (remBag occurrencesOf: obj) ifTrue: [
                currOffset == 1 ifTrue: [ ^ obj ].
                currOffset := currOffset - 1
              ]
          ]
        ]
    ]
].
self _error: #objErrBadOffsetIncomplete args: { offset }

]

{ #category : 'Support' }
RcIdentityBag >> centralizeSessionElements [

"Place the elements of inactive session components in the global Bag.  
 This may cause concurrency conflicts. Not Rc."

| inactiveSessions globalAddBag |
inactiveSessions := self _getInactiveSessionIndexes .
inactiveSessions add: self _componentsIndex .  " treat this session as inactive "

globalAddBag := self _globalAddBag.
inactiveSessions do: [ :i|  | addBag |
  " apply any removals that have occurred for that session's elements "
  self _processRemovalBagFor: i .
  addBag := components at: i.

  globalAddBag addAll: addBag.
  addBag removeAll: addBag.
].

]

{ #category : 'Support' }
RcIdentityBag >> cleanupBag [

"Iterate through all inactive session bags and process their removal Bags.
 This may cause conflicts . Not Rc"

| thisIdx inactiveSessions |
inactiveSessions := self _getInactiveSessionIndexes.

" treat this session as inactive if there are enough component Bags "
thisIdx := self _componentsIndex .
thisIdx < components size ifTrue:[ inactiveSessions add: thisIdx].

inactiveSessions do: [ :i |
    " apply any removals that have occurred for that session's elements "
    self _processRemovalBagFor: i .
]

]

{ #category : 'Clustering' }
RcIdentityBag >> cluster [

"Clusters an object using the current default ClusterBucket.  
 May cause concurrency conflicts.  Not Rc."

super cluster ifTrue: [ ^ true ].

components cluster.
self _doSessionBags: [ :addBag :removeBag |
  addBag cluster.
  removeBag cluster.
].
^ false

]

{ #category : 'Clustering' }
RcIdentityBag >> clusterDepthFirst [

"Clusters the receiver and its contents in depth-first order.  Returns true if
 the receiver has already been clustered during the current transaction; returns
 false otherwise.  This operation may cause concurrency conflicts . Not Rc"

super cluster ifTrue: [ ^ true ].

self clusterIndexes.

components cluster.
self _doSessionBags: [ :addBag :removeBag |
  addBag clusterDepthFirst.
  removeBag clusterDepthFirst.
].
^ false

]

{ #category : 'Support' }
RcIdentityBag >> distributeSessionElements [

"Distributes the elements of inactive session components across all inactive
 session components.  This could lessen the chance of conflicts that must be
 resolved at commit time.  This may cause concurrency conflict if another
 session performs this operation or if a new session modifies the receiver."

| inactiveSessions numberInactive numPerBag thisIdx
  remainder totalToDistribute extras deficientSessions |

inactiveSessions := self _getInactiveSessionIndexes .
" treat this session as inactive "
thisIdx := self _componentsIndex .
thisIdx < components size ifTrue:[ inactiveSessions add: thisIdx].
" treat the global session as inactive "
inactiveSessions add: 1.

deficientSessions := IdentityBag new.
extras := IdentityBag new.

numberInactive := inactiveSessions size.
totalToDistribute := 0.

" pass1: for each inactive session, process its removal Bag and
  calculate total number that will be distributed "
inactiveSessions do: [ :i | | addBag |
  addBag := components at: i.
  self _processRemovalBagFor: i .
  totalToDistribute := totalToDistribute + addBag size
].
numPerBag := totalToDistribute // numberInactive.
remainder := totalToDistribute \\ numberInactive.

" pass2: perform removals to inactive addition Bags
  that have more than numPerBag "
inactiveSessions do: [ :i | | delta addBag |
    delta := (addBag := components at: i) size - numPerBag.
    (delta > 0) ifTrue:[ "session contains more than numPerBag " | amount|
        remainder > 0 ifTrue: [
            amount := delta - 1.
            remainder := remainder - 1
        ] ifFalse: [ amount := delta ].

        extras addAll: (self _removeAny: amount from: addBag)
     ] ifFalse: [ deficientSessions add: i ]
].

" pass3: perform additions to inactive addition Bags that were
  previously deficient in the number of elements "
deficientSessions do: [ :i | | amount addBag |
    addBag := components at: i.
    remainder > 0 ifTrue: [
       amount := numPerBag + 1.
       remainder := remainder - 1
     ] ifFalse: [ amount := numPerBag ].
    addBag addAll: (self _removeAny: amount - addBag size from: extras)
]

]

{ #category : 'Enumerating' }
RcIdentityBag >> do: aBlock [

"Enumerates over all elements in the RcIdentityBag, executing aBlock with each
 element as the argument.  Returns the receiver."

 (System rcValueCacheAt: #asBag for: self otherwise: nil) ifNotNil:[:bag |
   bag do: aBlock . 
 ] ifNil:[
   self _doSessionBags: [ :addBag :removeBag |
     removeBag size == 0 
        ifTrue:[ addBag do: aBlock ]
       ifFalse:[ (addBag _primDifference: removeBag) do: aBlock ].
   ].
 ].

]

{ #category : 'Comparing' }
RcIdentityBag >> hash [

"Returns an Integer hash code for the receiver.

 Warning:  This is a computationally expensive operation."

^ self _asIdentityBag hash

]

{ #category : 'Searching' }
RcIdentityBag >> includes: anObject [

"Returns true if anObject is present in any session component of the
 RcIdentityBag."

^self includesIdentical: anObject


]

{ #category : 'Searching' }
RcIdentityBag >> includesIdentical: anObject [
  "Returns true if anObject is present in any session component of the
   RcIdentityBag."
  self _doSessionBags: [ :addBag :removeBag |
   (addBag includesIdentical: anObject) ifTrue: [
     removeBag size == 0 ifTrue: [ ^ true ].
     (addBag occurrencesOf: anObject) > (removeBag occurrencesOf: anObject)
         ifTrue: [ ^ true ]
   ]
  ].
  ^ false
]

{ #category : 'Searching' }
RcIdentityBag >> includesValue: anObject [

"Returns true if anObject is present in any session component of the
 RcIdentityBag.  Uses equality comparison."

self _doSessionBags: [ :addBag :removeBag |
    (addBag includesValue: anObject) ifTrue: [
       removeBag size == 0 ifTrue: [ ^ true ].
       ((addBag _primDifference: removeBag) includesValue: anObject) ifTrue: [ ^ true ]
    ]
].
^ false

]

{ #category : 'Initialization' }
RcIdentityBag >> initialize: aSize [

"Initializes a new instance."
aSize < 2 ifTrue:[ OutOfRange new name: 'size' min: 2 actual: aSize ; signal].
(aSize \\ 2) == 0 ifFalse:[ ArgumentError new name:'aSize' ; signal:'must be even' ].
components := Array new: aSize.
self initializeComponents .

]

{ #category : 'Initialization' }
RcIdentityBag >> initializeComponents [

"Create subcomponents for all sessionIds per current size of the components array.  
  This avoids Rc replay when many sessions add an object to the
  RcIdentityBag for the first time, and is executed by new: and maxSessionId: ."

1 to: components size by: 2 do: [ :i |
  (components at: i) ifNil:[ self _createComponentBagsFor: i ].
]

]

{ #category : 'Accessing' }
RcIdentityBag >> maxSessionId [

  "Returns maximum sessionId that can be used without incurring a grow of the components Array"
  ^ (components size // 2) - 1
]

{ #category : 'Accessing' }
RcIdentityBag >> maxSessionId: aSessionId [

  "creates component bags as needed, Not Rc."
  ^ components size // 2 < aSessionId ifTrue:[
     components size: aSessionId * 2 .
     self initializeComponents .
  ]
]

{ #category : 'Accessing' }
RcIdentityBag >> namedSize [
  ^ self class instSize 
]

{ #category : 'Updating' }
RcIdentityBag >> objectSecurityPolicy: anObjectSecurityPolicy [

"Assigns the receiver and its subcomponents to the given ObjectSecurityPolicy."

super objectSecurityPolicy: anObjectSecurityPolicy .
components objectSecurityPolicy: anObjectSecurityPolicy .

1 to: components size do: [ :i | | elem |
  (elem := components at: i) ifNotNil:[
    elem objectSecurityPolicy: anObjectSecurityPolicy
  ]
]

]

{ #category : 'Searching' }
RcIdentityBag >> occurrencesOf: anObject [

"Returns the number of occurrences of anObject in all session components of
 the RcIdentityBag."

^ self _asIdentityBag occurrencesOf: anObject

]

{ #category : 'Copying' }
RcIdentityBag >> postCopy [

"Cleanup new copy."

super postCopy.
components := components copy.
1 to: components size do: [ :i | 
  components at: i put: (components at: i) copy
].

]

{ #category : 'Formatting' }
RcIdentityBag >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

| count sz myCls anObj |

myCls := self class .
aStream nextPutAll: myCls name describeClassName .
aStream nextPutAll: '( ' .
count := 1 .
sz := self size .
1 to: sz do:[:idx |
  aStream isFull ifTrue:[
    "prevent infinite recursion when printing cyclic structures, and
     limit the size of result when printing large collections."
    aStream nextPutAll: '...)' .
    ^ self
    ] .
  anObj := self at: idx .
  anObj printOn: aStream .
  count < sz ifTrue:[ aStream nextPutAll: ', ' ].
  count := count + 1 .
  ].
aStream nextPut: $) .

]

{ #category : 'Removing' }
RcIdentityBag >> remove: anObject [

"Removes anObject from the receiver.  If anObject is present several times in
 the receiver, only one occurrence is removed.  Generates an error if anObject
 is not in the receiver.  Returns anObject. "

^(self _privateRemove: anObject logging: true)
    ifTrue: [ anObject ]
    ifFalse: [ self _errorNotFound: anObject ]

]

{ #category : 'Removing' }
RcIdentityBag >> remove: anObject ifAbsent: aBlock [

"Removes anObject from the receiver.  If anObject is present several times in
 the receiver, only one occurrence is removed.  If anObject is not in the
 receiver, this method evaluates aBlock and returns its value.  The argument
 aBlock must be a zero-argument block.  Returns anObject. "

^(self _privateRemove: anObject logging: true)
    ifTrue: [ anObject ]
    ifFalse: [ aBlock value ]

]

{ #category : 'Removing' }
RcIdentityBag >> remove: anObject otherwise: defaultValue [

^(self _privateRemove: anObject logging: true)
    ifTrue: [ anObject ]
    ifFalse: [ defaultValue ]

]

{ #category : 'Removing' }
RcIdentityBag >> removeAll: aCollection [

"Removes one occurrence of each element of aCollection from the receiver and
 returns the receiver.  Generates an error if any element of aCollection is not
 present in the receiver."

aCollection size ~~ 0 ifTrue:[
  (self _removeAll: aCollection logging: true) ifFalse:[
    LookupError new args: { self } ;
	signal: 'an element of aCollection was not found during removal '
  ]
].
^aCollection

]

{ #category : 'Removing' }
RcIdentityBag >> removeAllPresent: aCollection [

"Removes from the receiver one occurrence of each element of aCollection that
 is also an element of the receiver.  Differs from removeAll: in that, if some
 elements of aCollection are not present in the receiver, no error is
 generated.  Returns aCollection."

aCollection == self
    ifTrue: [ aCollection asBag do: [ :obj | self removeIfPresent: obj ] ]
    ifFalse: [ aCollection do: [ :obj | self removeIfPresent: obj ] ].
^aCollection

]

{ #category : 'Removing' }
RcIdentityBag >> removeIdentical: anObject ifAbsent: aBlock [

"Removes anObject from the receiver.  If anObject is present several times in
 the receiver, only one occurrence is removed.  If anObject is not in the
 receiver, this method evaluates aBlock and returns its value.  The argument
 aBlock must be a zero-argument block."

(self _privateRemove: anObject logging: true)
    ifFalse: [ ^ aBlock value ]

]

{ #category : 'Removing' }
RcIdentityBag >> removeIfPresent: anObject [

"Remove anObject from the receiver.  If anObject is present several times in
 the receiver, only one occurrence is removed.  Returns nil if anObject is
 missing from the receiver."

(self _privateRemove: anObject logging: true)
    ifFalse: [ ^ nil ]

]

{ #category : 'Removing' }
RcIdentityBag >> removeOneObject [
  "Returns nil if receiver is empty" 
  | addIdx |
  addIdx := self _componentsIndex .
  (components atOrNil: addIdx) ifNotNil:[:additionBag | | removalBag |
     removalBag := components at: addIdx + 1 .
     " first see if the object is in the current session's addition Bag "
     additionBag do:[:anObject | 
       (additionBag occurrencesOf: anObject) > (removalBag occurrencesOf: anObject) ifTrue:[
         self _doRemove: anObject removalBag: removalBag logging: true .
         self _rcProcessRemovalBagFor: addIdx logging: true. "process removals on addition bag"
         ^ anObject
       ].
     ].
   ].
   self _doSessionBags: [ :addBag :removeBag | " check other sessions' addition Bags "
      addBag do:[:anObject |
         (addBag occurrencesOf: anObject) > (removeBag occurrencesOf: anObject) ifTrue:[
            " more have been added than removed ... "
            self _doRemove: anObject removalBag: removeBag logging: true .
            System redoLog addConflictObject: addBag for: self .
            ^ anObject 
         ] 
     ].
   ].
   ^ nil "none  found"
]

{ #category : 'Accessing' }
RcIdentityBag >> size [

"Returns the number of elements contained in the RcIdentityBag.  First checks
 the RC value cache, and if not there, calculate it."

| sz |
sz := System rcValueCacheAt: #size for: self otherwise: nil.
sz == nil
    ifTrue: [ sz := self _calculateSize ].
^ sz

]

{ #category : 'Sorting' }
RcIdentityBag >> sortAscending: aSortSpec [

"Returns an Array containing the elements of the receiver, sorted in ascending
 order, as determined by the values of the instance variables represented by
 aSortSpec."

 | bag |
^ [ | result |
    " get the cached Bag "
    bag := self _asIdentityBag.
    " let the cached Bag take advantage of indexes if they exist "
    bag _indexedPaths: self _indexedPaths.
    " now do the sort "
    result := bag sortAscending: aSortSpec.
    " reset the cached Bag's index list "
    bag _indexedPaths: nil.
    result
  ] onException: Error do:[:ex |
    "ensure the cached Bag's index list is made nil"
    bag _indexedPaths: nil.
    ex outer
  ]

]

{ #category : 'Sorting' }
RcIdentityBag >> sortDescending: aSortSpec [

"Returns an Array containing the elements of the receiver, sorted in descending
 order, as determined by the values of the instance variables represented by
 aSortSpec."

 | bag |
^ [ | result |
    " get the cached Bag "
    bag := self _asIdentityBag.
    " let the cached Bag take advantage of indexes if they exist "
    bag _indexedPaths: self _indexedPaths.
    " now do the sort "
    result := bag sortDescending: aSortSpec.
    " reset the cached Bag's index list "
    bag _indexedPaths: nil.
    result
  ] onException: Error do:[:ex |
    "ensure the cached Bag's index list is made nil"
    bag _indexedPaths: nil.
    ex outer
  ]


]

{ #category : 'Sorting' }
RcIdentityBag >> sortWith: aSortPairArray [

"Returns an Array containing the elements of the receiver, sorted according to
 the contents of aSortPairArray."

 | bag |
^ [ | result |
    " get the cached Bag "
    bag := self _asIdentityBag.
    " let the cached Bag take advantage of indexes if they exist "
    bag _indexedPaths: self _indexedPaths.
    " now do the sort "
    result := bag sortWith: aSortPairArray.
    " reset the cached Bag's index list "
    bag _indexedPaths: nil.
    result
  ] onException: Error do:[:ex |
    "ensure the cached Bag's index list is made nil"
    bag _indexedPaths: nil.
    ex outer
  ]

]

{ #category : 'Class Membership' }
RcIdentityBag >> species [

"Returns the class to use to select and reject queries."

^ IdentityBag

]
